// SPDX-License-Identifier: GPL-2.0
/*
 * vehicle sensor nvp6324
 *
 * Copyright (C) 2020 Rockchip Electronics Co.Ltd
 * Authors:
 *      wpzz <randy.wang@rock-chips.com>
 *
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/sysctl.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/proc_fs.h>
#include <linux/suspend.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/uaccess.h>
#include <linux/of_gpio.h>
#include <linux/of_irq.h>
#include "vehicle_cfg.h"
#include "vehicle_main.h"
#include "vehicle_ad.h"
#include "vehicle_ad_nvp6324.h"

enum {
	CVSTD_720P60 = 0,
	CVSTD_720P50,
	CVSTD_1080P30,
	CVSTD_1080P25,
	CVSTD_720P30,
	CVSTD_720P25,
	CVSTD_SVGAP30,
	CVSTD_SD,
	CVSTD_NTSC,
	CVSTD_PAL
};

enum {
	FORCE_PAL_WIDTH = 960,
	FORCE_PAL_HEIGHT = 576,
	FORCE_NTSC_WIDTH = 960,
	FORCE_NTSC_HEIGHT = 480,
	FORCE_SVGA_WIDTH = 800,
	FORCE_SVGA_HEIGHT = 600,
	FORCE_720P_WIDTH = 1280,
	FORCE_720P_HEIGHT = 720,
	FORCE_1080P_WIDTH = 1920,
	FORCE_1080P_HEIGHT = 1080,
	FORCE_CIF_OUTPUT_FORMAT = CIF_OUTPUT_FORMAT_420,
};

enum {
	VIDEO_UNPLUG,
	VIDEO_IN,
	VIDEO_LOCKED,
	VIDEO_UNLOCK
};

#define JAGUAR1_LINK_FREQ_320M			320000000UL
#define JAGUAR1_LINK_FREQ_640M			640000000UL

static struct vehicle_ad_dev *nvp6324_g_addev;
static int cvstd_mode = CVSTD_1080P25;
//static int cvstd_old = CVSTD_720P25;
static int cvstd_old = CVSTD_NTSC;

//static int cvstd_sd = CVSTD_NTSC;
static int cvstd_state = VIDEO_UNPLUG;
static int cvstd_old_state = VIDEO_UNLOCK;
static int video_mode;
static int video_old;

static bool g_nvp6324_streaming;

#define SENSOR_REGISTER_LEN	1	/* sensor register address bytes*/
#define SENSOR_VALUE_LEN	1	/* sensor register value bytes*/
#define JAGUAR1_CHIP_ID		0xB0

struct rk_sensor_reg {
	unsigned int reg;
	unsigned int val;
};

#define SENSOR_CHANNEL_REG		0x41

#define SEQCMD_END  0xFF000000
#define SensorEnd   {SEQCMD_END, 0x00}

#define SENSOR_ID(_msb, _lsb)		((_msb) << 8 | (_lsb))

/* 720p Preview resolution setting*/
static struct rk_sensor_reg sensor_preview_data_720p_25hz[] = {
	{0xff, 0x04},
	{0xa0, 0x24},
	{0xa1, 0x24},
	{0xa2, 0x24},
	{0xa3, 0x24},
	{0xa4, 0x24},
	{0xa5, 0x24},
	{0xa6, 0x24},
	{0xa7, 0x24},
	{0xa8, 0x24},
	{0xa9, 0x24},
	{0xaa, 0x24},
	{0xab, 0x24},
	{0xac, 0x24},
	{0xad, 0x24},
	{0xae, 0x24},
	{0xaf, 0x24},
	{0xb0, 0x24},
	{0xb1, 0x24},
	{0xb2, 0x24},
	{0xb3, 0x24},
	{0xb4, 0x24},
	{0xb5, 0x24},
	{0xb6, 0x24},
	{0xb7, 0x24},
	{0xb8, 0x24},
	{0xb9, 0x24},
	{0xba, 0x24},
	{0xbb, 0x24},
	{0xbc, 0x24},
	{0xbd, 0x24},
	{0xbe, 0x24},
	{0xbf, 0x24},
	{0xc0, 0x24},
	{0xc1, 0x24},
	{0xc2, 0x24},
	{0xc3, 0x24},
	{0xff, 0x21},
	{0x07, 0x80},
	{0x07, 0x00},
	{0xff, 0x0A},
	{0x77, 0x8F},
	{0xF7, 0x8F},
	{0xff, 0x0B},
	{0x77, 0x8F},
	{0xF7, 0x8F},
	{0xFF, 0x21},
	{0x40, 0xAC},
	{0x41, 0x10},
	{0x42, 0x03},
	{0x43, 0x43},
	{0x11, 0x04},
	{0x10, 0x0A},
	{0x12, 0x06},
	{0x13, 0x09},
	{0x17, 0x01},
	{0x18, 0x0D},
	{0x15, 0x04},
	{0x14, 0x16},
	{0x16, 0x05},
	{0x19, 0x05},
	{0x1A, 0x0A},
	{0x1B, 0x08},
	{0x1C, 0x07},
	{0x44, 0x00},
	{0x49, 0xF3},
	{0x49, 0xF0},
	{0x44, 0x02},
	{0x08, 0x40}, //0x40:non-continue;0x48:continuous
	{0x0F, 0x01},
	{0x38, 0x1E},
	{0x39, 0x1E},
	{0x3A, 0x1E},
	{0x3B, 0x1E},
	{0x07, 0x0f}, //0x07:2lane;0x0f:4lane
	{0x2D, 0x01}, //0x00:2lane;0x01:4lane
	{0x45, 0x02},
	{0xFF, 0x13},
	{0x30, 0x00},
	{0x31, 0x00},
	{0x32, 0x00},
	{0xFF, 0x00},
	{0x00, 0x00},
	{0x01, 0x00},
	{0x02, 0x00},
	{0x03, 0x00},
	{0x04, 0x00}, //sd_mode
	{0x05, 0x00},
	{0x06, 0x00},
	{0x07, 0x00},
	{0x08, 0x0d}, //ahd_mode
	{0x09, 0x0d},
	{0x0a, 0x0d},
	{0x0b, 0x0d},
	{0x0c, 0x00},
	{0x0d, 0x00},
	{0x0e, 0x00},
	{0x0f, 0x00},
	{0x10, 0x20}, //video_format
	{0x11, 0x20},
	{0x12, 0x20},
	{0x13, 0x20},
	{0x14, 0x00},
	{0x15, 0x00},
	{0x16, 0x00},
	{0x17, 0x00},
	{0x18, 0x13},
	{0x19, 0x13},
	{0x1a, 0x13},
	{0x1b, 0x13},
	{0x1c, 0x1a},
	{0x1d, 0x1a},
	{0x1e, 0x1a},
	{0x1f, 0x1a},
	{0x20, 0x00},
	{0x21, 0x00},
	{0x22, 0x00},
	{0x23, 0x00},
	{0x24, 0x88}, //contrast
	{0x25, 0x88},
	{0x26, 0x88},
	{0x27, 0x88},
	{0x28, 0x84}, //black_level
	{0x29, 0x84},
	{0x2a, 0x84},
	{0x2b, 0x84},
	{0x30, 0x03}, //y_peaking_mode
	{0x31, 0x03},
	{0x32, 0x03},
	{0x33, 0x03},
	{0x34, 0x0f}, //y_fir_mode
	{0x35, 0x0f},
	{0x36, 0x0f},
	{0x37, 0x0f},
	{0x40, 0x00},
	{0x41, 0x00},
	{0x42, 0x00},
	{0x43, 0x00},
	{0x44, 0x00},
	{0x45, 0x00},
	{0x46, 0x00},
	{0x47, 0x00},
	{0x48, 0x00},
	{0x49, 0x00},
	{0x4a, 0x00},
	{0x4b, 0x00},
	{0x4c, 0x00},
	{0x4d, 0x00},
	{0x4e, 0x00},
	{0x4f, 0x00},
	{0x50, 0x00},
	{0x51, 0x00},
	{0x52, 0x00},
	{0x53, 0x00},
	{0x58, 0x80},
	{0x59, 0x80},
	{0x5a, 0x80},
	{0x5b, 0x80},
	{0x5c, 0x82}, //pal_cm_off
	{0x5d, 0x82},
	{0x5e, 0x82},
	{0x5f, 0x82},
	{0x60, 0x10},
	{0x61, 0x10},
	{0x62, 0x10},
	{0x63, 0x10},
	{0x64, 0x05}, //y_delay
	{0x65, 0x05},
	{0x66, 0x05},
	{0x67, 0x05},
	{0x68, 0x43}, //h_delay_a //h_delay_lsb
	{0x69, 0x43},
	{0x6a, 0x43},
	{0x6b, 0x43},
	{0x6c, 0x00},
	{0x6d, 0x00},
	{0x6e, 0x00},
	{0x6f, 0x00},
	{0x78, 0x21},
	{0x79, 0x21},
	{0x7a, 0x21},
	{0x7b, 0x21},
	{0xFF, 0x01},
	{0x7C, 0x00},
	{0x84, 0x04},
	{0x85, 0x04},
	{0x86, 0x04},
	{0x87, 0x04},
	{0x88, 0x01},
	{0x89, 0x01},
	{0x8a, 0x01},
	{0x8b, 0x01},
	{0x8c, 0x02},
	{0x8d, 0x02},
	{0x8e, 0x02},
	{0x8f, 0x02},
	{0xEC, 0x00},
	{0xED, 0x00},
	{0xEE, 0x00},
	{0xEF, 0x00},
	{0xFF, 0x05},
	{0x00, 0xd0},
	{0x01, 0x2c},
	{0x05, 0x24}, //d_agc_option
	{0x1d, 0x0c},
	{0x24, 0x2a},
	{0x25, 0xdc}, //fsc_lock_mode
	{0x26, 0x40},
	{0x27, 0x57},
	{0x28, 0x80}, //s_point
	{0x2b, 0xa8}, //saturation_b
	{0x31, 0x82},
	{0x32, 0x10},
	{0x38, 0x00}, //burst_dec_b
	{0x47, 0xEE},
	{0x50, 0xc6},
	{0x53, 0x00},
	{0x57, 0x00},
	{0x58, 0x77},
	{0x59, 0x00},
	{0x5C, 0x78},
	{0x5F, 0x00},
	{0x62, 0x20},
	{0x64, 0x00},
	{0x65, 0x00},
	{0x69, 0x00},
	{0x6E, 0x00}, //VBLK_EXT_EN
	{0x6F, 0x00}, //VBLK_EXT_[7:0]
	{0x90, 0x01}, //comb_mode
	{0x92, 0x00},
	{0x94, 0x00},
	{0x95, 0x00},
	{0xa9, 0x00},
	{0xb5, 0x80},
	{0xb7, 0xfc},
	{0xb8, 0x39},
	{0xb9, 0x72},
	{0xbb, 0x0f},
	{0xd1, 0x30}, //burst_dec_c
	{0xd5, 0x80},
	{0xFF, 0x09},
	{0x40, 0x00},
	{0x41, 0x00},
	{0x42, 0x00},
	{0x43, 0x00},
	{0x44, 0x00},
	{0x45, 0x00},
	{0x46, 0x00},
	{0x47, 0x00},
	{0x50, 0x30},
	{0x51, 0x6f},
	{0x52, 0x67},
	{0x53, 0x48},
	{0x54, 0x30},
	{0x55, 0x6f},
	{0x56, 0x67},
	{0x57, 0x48},
	{0x58, 0x30},
	{0x59, 0x6f},
	{0x5a, 0x67},
	{0x5b, 0x48},
	{0x5c, 0x30},
	{0x5d, 0x6f},
	{0x5e, 0x67},
	{0x5f, 0x48},
	{0x96, 0x00},
	{0x97, 0x00},
	{0x98, 0x00},
	{0x99, 0x00},
	{0x9a, 0x00},
	{0x9b, 0x00},
	{0x9c, 0x00},
	{0x9d, 0x00},
	{0x9e, 0x00},
	{0xb6, 0x00},
	{0xb7, 0x00},
	{0xb8, 0x00},
	{0xb9, 0x00},
	{0xba, 0x00},
	{0xbb, 0x00},
	{0xbc, 0x00},
	{0xbd, 0x00},
	{0xbe, 0x00},
	{0xd6, 0x00},
	{0xd7, 0x00},
	{0xd8, 0x00},
	{0xd9, 0x00},
	{0xda, 0x00},
	{0xdb, 0x00},
	{0xdc, 0x00},
	{0xdd, 0x00},
	{0xde, 0x00},
	{0xf6, 0x00},
	{0xf7, 0x00},
	{0xf8, 0x00},
	{0xf9, 0x00},
	{0xfa, 0x00},
	{0xfb, 0x00},
	{0xfc, 0x00},
	{0xfd, 0x00},
	{0xfe, 0x00},
	{0xff, 0x0a},
	{0x3d, 0x00},
	{0x3c, 0x00},
	{0x30, 0xac},
	{0x31, 0x78},
	{0x32, 0x17},
	{0x33, 0xc1},
	{0x34, 0x40},
	{0x35, 0x00},
	{0x36, 0xc3},
	{0x37, 0x0a},
	{0x38, 0x00},
	{0x39, 0x02},
	{0x3a, 0x00},
	{0x3b, 0xb2},
	{0x25, 0x10},
	{0x27, 0x1e},
	{0xbd, 0x00},
	{0xbc, 0x00},
	{0xb0, 0xac},
	{0xb1, 0x78},
	{0xb2, 0x17},
	{0xb3, 0xc1},
	{0xb4, 0x40},
	{0xb5, 0x00},
	{0xb6, 0xc3},
	{0xb7, 0x0a},
	{0xb8, 0x00},
	{0xb9, 0x02},
	{0xba, 0x00},
	{0xbb, 0xb2},
	{0xa5, 0x10},
	{0xa7, 0x1e},
	{0xff, 0x0b},
	{0x3d, 0x00},
	{0x3c, 0x00},
	{0x30, 0xac},
	{0x31, 0x78},
	{0x32, 0x17},
	{0x33, 0xc1},
	{0x34, 0x40},
	{0x35, 0x00},
	{0x36, 0xc3},
	{0x37, 0x0a},
	{0x38, 0x00},
	{0x39, 0x02},
	{0x3a, 0x00},
	{0x3b, 0xb2},
	{0x25, 0x10},
	{0x27, 0x1e},
	{0xbd, 0x00},
	{0xbc, 0x00},
	{0xb0, 0xac},
	{0xb1, 0x78},
	{0xb2, 0x17},
	{0xb3, 0xc1},
	{0xb4, 0x40},
	{0xb5, 0x00},
	{0xb6, 0xc3},
	{0xb7, 0x0a},
	{0xb8, 0x00},
	{0xb9, 0x02},
	{0xba, 0x00},
	{0xbb, 0xb2},
	{0xa5, 0x10},
	{0xa7, 0x1e},
	{0xFF, 0x21},
	{0x3E, 0x00},
	{0x3F, 0x00},
	{0xFF, 0x20},
	{0x01, 0x55},
	{0x00, 0x00},
	{0x40, 0x01},
	{0x0F, 0x00},
	{0x0D, 0x01},  //0x01:4lane;0x00:2lane
	{0x40, 0x00},
	{0x00, 0xFF},  //ch1/2/3/4 enabled
	//{0x00, 0x33},  //ch1/2 enabled
	//{0x00, 0x11},  //ch1 enabled
	{0xFF, 0x01},
	{0xC8, 0x00},
	{0xC9, 0x00},
	{0xCA, 0x00},
	{0xCB, 0x00},
	//pattern enabled
	{0xFF, 0x00},
	{0x1C, 0x1A},
	{0x1D, 0x1A},
	{0x1E, 0x1A},
	{0x1F, 0x1A},
	{0xFF, 0x05},
	{0x6A, 0x80},
	{0xFF, 0x06},
	{0x6A, 0x80},
	{0xFF, 0x07},
	{0x6A, 0x80},
	{0xFF, 0x08},
	{0x6A, 0x80},
	SensorEnd
};

/* 1080p Preview resolution setting*/
static struct rk_sensor_reg sensor_preview_data_1080p_25hz[] = {
	{0xff, 0x04},
	{0xa0, 0x24},
	{0xa1, 0x24},
	{0xa2, 0x24},
	{0xa3, 0x24},
	{0xa4, 0x24},
	{0xa5, 0x24},
	{0xa6, 0x24},
	{0xa7, 0x24},
	{0xa8, 0x24},
	{0xa9, 0x24},
	{0xaa, 0x24},
	{0xab, 0x24},
	{0xac, 0x24},
	{0xad, 0x24},
	{0xae, 0x24},
	{0xaf, 0x24},
	{0xb0, 0x24},
	{0xb1, 0x24},
	{0xb2, 0x24},
	{0xb3, 0x24},
	{0xb4, 0x24},
	{0xb5, 0x24},
	{0xb6, 0x24},
	{0xb7, 0x24},
	{0xb8, 0x24},
	{0xb9, 0x24},
	{0xba, 0x24},
	{0xbb, 0x24},
	{0xbc, 0x24},
	{0xbd, 0x24},
	{0xbe, 0x24},
	{0xbf, 0x24},
	{0xc0, 0x24},
	{0xc1, 0x24},
	{0xc2, 0x24},
	{0xc3, 0x24},
	{0xff, 0x21},
	{0x07, 0x80},
	{0x07, 0x00},
	{0xff, 0x0A},
	{0x77, 0x8F},
	{0xF7, 0x8F},
	{0xff, 0x0B},
	{0x77, 0x8F},
	{0xF7, 0x8F},
	{0xFF, 0x21},
	{0x40, 0xB4},
	{0x41, 0x00},
	{0x42, 0x03},
	{0x43, 0x43},
	{0x11, 0x08},
	{0x10, 0x13},
	{0x12, 0x0B},
	{0x13, 0x12},
	{0x17, 0x02},
	{0x18, 0x12},
	{0x15, 0x07},
	{0x14, 0x2D},
	{0x16, 0x0B},
	{0x19, 0x09},
	{0x1A, 0x15},
	{0x1B, 0x11},
	{0x1C, 0x0E},
	{0x44, 0x00},
	{0x49, 0xF3},
	{0x49, 0xF0},
	{0x44, 0x02},
	{0x08, 0x40}, //0x40:non-continue;0x48:continuous
	{0x0F, 0x01},
	{0x38, 0x1E},
	{0x39, 0x1E},
	{0x3A, 0x1E},
	{0x3B, 0x1E},
	{0x07, 0x0f}, //0x07:2lane;0x0f:4lane
	{0x2D, 0x01}, //0x00:2lane;0x01:4lane
	{0x45, 0x02},
	{0xFF, 0x13},
	{0x30, 0x00},
	{0x31, 0x00},
	{0x32, 0x00},
	{0xFF, 0x00},
	{0x00, 0x00},
	{0x01, 0x00},
	{0x02, 0x00},
	{0x03, 0x00},
	{0x04, 0x00}, //sd_mode
	{0x05, 0x00},
	{0x06, 0x00},
	{0x07, 0x00},
	{0x08, 0x03}, //ahd_mode
	{0x09, 0x03},
	{0x0a, 0x03},
	{0x0b, 0x03},
	{0x0c, 0x00},
	{0x0d, 0x00},
	{0x0e, 0x00},
	{0x0f, 0x00},
	{0x10, 0x20}, //video_format
	{0x11, 0x20},
	{0x12, 0x20},
	{0x13, 0x20},
	{0x14, 0x00},
	{0x15, 0x00},
	{0x16, 0x00},
	{0x17, 0x00},
	{0x18, 0x13},
	{0x19, 0x13},
	{0x1a, 0x13},
	{0x1b, 0x13},
	{0x1c, 0x1a},
	{0x1d, 0x1a},
	{0x1e, 0x1a},
	{0x1f, 0x1a},
	{0x20, 0x00},
	{0x21, 0x00},
	{0x22, 0x00},
	{0x23, 0x00},
	{0x24, 0x86}, //contrast
	{0x25, 0x86},
	{0x26, 0x86},
	{0x27, 0x86},
	{0x28, 0x80}, //black_level
	{0x29, 0x80},
	{0x2a, 0x80},
	{0x2b, 0x80},
	{0x30, 0x00}, //y_peaking_mode
	{0x31, 0x00},
	{0x32, 0x00},
	{0x33, 0x00},
	{0x34, 0x00}, //y_fir_mode
	{0x35, 0x00},
	{0x36, 0x00},
	{0x37, 0x00},
	{0x40, 0x00},
	{0x41, 0x00},
	{0x42, 0x00},
	{0x43, 0x00},
	{0x44, 0x00},
	{0x45, 0x00},
	{0x46, 0x00},
	{0x47, 0x00},
	{0x48, 0x00},
	{0x49, 0x00},
	{0x4a, 0x00},
	{0x4b, 0x00},
	{0x4c, 0xfe},
	{0x4d, 0xfe},
	{0x4e, 0xfe},
	{0x4f, 0xfe},
	{0x50, 0xfb},
	{0x51, 0xfb},
	{0x52, 0xfb},
	{0x53, 0xfb},
	{0x58, 0x80},
	{0x59, 0x80},
	{0x5a, 0x80},
	{0x5b, 0x80},
	{0x5c, 0x82}, //pal_cm_off
	{0x5d, 0x82},
	{0x5e, 0x82},
	{0x5f, 0x82},
	{0x60, 0x10},
	{0x61, 0x10},
	{0x62, 0x10},
	{0x63, 0x10},
	{0x64, 0x05}, //y_delay
	{0x65, 0x05},
	{0x66, 0x05},
	{0x67, 0x05},
	{0x68, 0x48}, //h_delay_a //h_delay_lsb
	{0x69, 0x48},
	{0x6a, 0x48},
	{0x6b, 0x48},
	{0x6c, 0x00},
	{0x6d, 0x00},
	{0x6e, 0x00},
	{0x6f, 0x00},
//	{0x78, 0x21},
//	{0x79, 0x21},
//	{0x7a, 0x21},
//	{0x7b, 0x21},
	{0x78, 0x22},
	{0x79, 0x22},
	{0x7a, 0x22},
	{0x7b, 0x22},
	{0xFF, 0x01},
	{0x7C, 0x00},
	{0x84, 0x04},
	{0x85, 0x04},
	{0x86, 0x04},
	{0x87, 0x04},
	{0x88, 0x01},
	{0x89, 0x01},
	{0x8a, 0x01},
	{0x8b, 0x01},
	{0x8c, 0x02},
	{0x8d, 0x02},
	{0x8e, 0x02},
	{0x8f, 0x02},
	{0xEC, 0x00},
	{0xED, 0x00},
	{0xEE, 0x00},
	{0xEF, 0x00},
	{0xFF, 0x05},
	{0x00, 0xd0},
	{0x01, 0x2c},
	{0x05, 0x24}, //d_agc_option
	{0x1d, 0x0c},
	{0x24, 0x2a},
	{0x25, 0xdc}, //fsc_lock_mode
	{0x26, 0x40},
	{0x27, 0x57},
	{0x28, 0x80}, //s_point
	{0x2b, 0xa8}, //saturation_b
	{0x31, 0x82},
	{0x32, 0x10},
	{0x38, 0x13},
	{0x47, 0xEE},
	{0x50, 0xc6},
	{0x53, 0x00},
	{0x57, 0x00},
	{0x58, 0x77},
	{0x59, 0x00},
	{0x5C, 0x78},
	{0x5F, 0x00},
	{0x62, 0x20},
	{0x64, 0x00},
	{0x65, 0x00},
	{0x69, 0x00},
	{0x6E, 0x00}, //VBLK_EXT_EN
	{0x6F, 0x00}, //VBLK_EXT_[7:0]
	{0x90, 0x01}, //comb_mode
	{0x92, 0x00},
	{0x94, 0x00},
	{0x95, 0x00},
	{0xa9, 0x00},
	{0xb5, 0x80},
	{0xb7, 0xfc},
	{0xb8, 0x39},
	{0xb9, 0x72},
	{0xbb, 0x0f},
	{0xd1, 0x30}, //burst_dec_c
	{0xd5, 0x80},
	{0xFF, 0x09},
	{0x40, 0x00},
	{0x41, 0x00},
	{0x42, 0x00},
	{0x43, 0x00},
	{0x44, 0x00},
	{0x45, 0x00},
	{0x46, 0x00},
	{0x47, 0x00},
	{0x50, 0x30},
	{0x51, 0x6f},
	{0x52, 0x67},
	{0x53, 0x48},
	{0x54, 0x30},
	{0x55, 0x6f},
	{0x56, 0x67},
	{0x57, 0x48},
	{0x58, 0x30},
	{0x59, 0x6f},
	{0x5a, 0x67},
	{0x5b, 0x48},
	{0x5c, 0x30},
	{0x5d, 0x6f},
	{0x5e, 0x67},
	{0x5f, 0x48},
	{0x96, 0x00},
	{0x97, 0x00},
	{0x98, 0x00},
	{0x99, 0x00},
	{0x9a, 0x00},
	{0x9b, 0x00},
	{0x9c, 0x00},
	{0x9d, 0x00},
	{0x9e, 0x00},
	{0xb6, 0x00},
	{0xb7, 0x00},
	{0xb8, 0x00},
	{0xb9, 0x00},
	{0xba, 0x00},
	{0xbb, 0x00},
	{0xbc, 0x00},
	{0xbd, 0x00},
	{0xbe, 0x00},
	{0xd6, 0x00},
	{0xd7, 0x00},
	{0xd8, 0x00},
	{0xd9, 0x00},
	{0xda, 0x00},
	{0xdb, 0x00},
	{0xdc, 0x00},
	{0xdd, 0x00},
	{0xde, 0x00},
	{0xf6, 0x00},
	{0xf7, 0x00},
	{0xf8, 0x00},
	{0xf9, 0x00},
	{0xfa, 0x00},
	{0xfb, 0x00},
	{0xfc, 0x00},
	{0xfd, 0x00},
	{0xfe, 0x00},
	{0xff, 0x0a},
	{0x3d, 0x00},
	{0x3c, 0x00},
	{0x30, 0xac},
	{0x31, 0x78},
	{0x32, 0x17},
	{0x33, 0xc1},
	{0x34, 0x40},
	{0x35, 0x00},
	{0x36, 0xc3},
	{0x37, 0x0a},
	{0x38, 0x00},
	{0x39, 0x02},
	{0x3a, 0x00},
	{0x3b, 0xb2},
	{0x25, 0x10},
	{0x27, 0x1e},
	{0xbd, 0x00},
	{0xbc, 0x00},
	{0xb0, 0xac},
	{0xb1, 0x78},
	{0xb2, 0x17},
	{0xb3, 0xc1},
	{0xb4, 0x40},
	{0xb5, 0x00},
	{0xb6, 0xc3},
	{0xb7, 0x0a},
	{0xb8, 0x00},
	{0xb9, 0x02},
	{0xba, 0x00},
	{0xbb, 0xb2},
	{0xa5, 0x10},
	{0xa7, 0x1e},
	{0xff, 0x0b},
	{0x3d, 0x00},
	{0x3c, 0x00},
	{0x30, 0xac},
	{0x31, 0x78},
	{0x32, 0x17},
	{0x33, 0xc1},
	{0x34, 0x40},
	{0x35, 0x00},
	{0x36, 0xc3},
	{0x37, 0x0a},
	{0x38, 0x00},
	{0x39, 0x02},
	{0x3a, 0x00},
	{0x3b, 0xb2},
	{0x25, 0x10},
	{0x27, 0x1e},
	{0xbd, 0x00},
	{0xbc, 0x00},
	{0xb0, 0xac},
	{0xb1, 0x78},
	{0xb2, 0x17},
	{0xb3, 0xc1},
	{0xb4, 0x40},
	{0xb5, 0x00},
	{0xb6, 0xc3},
	{0xb7, 0x0a},
	{0xb8, 0x00},
	{0xb9, 0x02},
	{0xba, 0x00},
	{0xbb, 0xb2},
	{0xa5, 0x10},
	{0xa7, 0x1e},
	{0xFF, 0x21},
	{0x3E, 0x00},
	{0x3F, 0x00},
	{0xFF, 0x20},
	{0x01, 0x00},
	{0x00, 0x00},
	{0x40, 0x01},
	{0x0F, 0x00},
	{0x0D, 0x01}, //0x01:4lane;0x00:2lane
	{0x40, 0x00},
	{0x00, 0xFF}, //ch1/2/3/4 enabled
	//{0x00, 0x33}, //ch1/2 enabled
	//{0x00, 0x11}, //ch1 enabled
	{0xFF, 0x01},
	{0xC8, 0x00},
	{0xC9, 0x00},
	{0xCA, 0x00},
	{0xCB, 0x00},
	//pattern enabled
	{0xFF, 0x00},
	{0x1C, 0x1A},
	{0x1D, 0x1A},
	{0x1E, 0x1A},
	{0x1F, 0x1A},
	{0xFF, 0x05},
	{0x6A, 0x80},
	{0xFF, 0x06},
	{0x6A, 0x80},
	{0xFF, 0x07},
	{0x6A, 0x80},
	{0xFF, 0x08},
	{0x6A, 0x80},
	SensorEnd
};

/* format detect open*/
static struct rk_sensor_reg sensor_open_format_detect[] = {
	{0xff, 0x13},
	{0x1f, 0x23},
	{0x30, 0xff},
	{0x31, 0xff},
	{0x32, 0xff},
	SensorEnd
};

static void nvp6324_reinit_parameter(struct vehicle_ad_dev *ad, unsigned char cvstd)
{
	int i = 0;

	switch (cvstd) {
	case CVSTD_720P25:
		ad->cfg.width = 1280;
		ad->cfg.height = 720;
		ad->cfg.start_x = 0;
		ad->cfg.start_y = 0;
		ad->cfg.input_format = CIF_INPUT_FORMAT_YUV;
		ad->cfg.output_format = FORCE_CIF_OUTPUT_FORMAT;
		ad->cfg.field_order = 0;
		ad->cfg.yuv_order = 0;/*00 - UYVY*/
		ad->cfg.href = 0;
		ad->cfg.vsync = 0;
		ad->cfg.frame_rate = 25;
		ad->cfg.mipi_freq = JAGUAR1_LINK_FREQ_320M;
		break;

	case CVSTD_1080P25:
		ad->cfg.width = 1920;
		ad->cfg.height = 1080;
		ad->cfg.start_x = 0;
		ad->cfg.start_y = 0;
		ad->cfg.input_format = CIF_INPUT_FORMAT_YUV;
		ad->cfg.output_format = FORCE_CIF_OUTPUT_FORMAT;
		ad->cfg.field_order = 0;
		ad->cfg.yuv_order = 0;/*00 - UYVY*/
		ad->cfg.href = 0;
		ad->cfg.vsync = 0;
		ad->cfg.frame_rate = 25;
		ad->cfg.mipi_freq = JAGUAR1_LINK_FREQ_640M;
		break;

	default:
		ad->cfg.width = 1920;
		ad->cfg.height = 1080;
		ad->cfg.start_x = 0;
		ad->cfg.start_y = 0;
		ad->cfg.input_format = CIF_INPUT_FORMAT_YUV;
		ad->cfg.output_format = FORCE_CIF_OUTPUT_FORMAT;
		ad->cfg.field_order = 0;
		ad->cfg.yuv_order = 0;/*00 - UYVY*/
		ad->cfg.href = 0;
		ad->cfg.vsync = 0;
		ad->cfg.frame_rate = 25;
		ad->cfg.mipi_freq = JAGUAR1_LINK_FREQ_640M;
		break;
	}
	ad->cfg.type = V4L2_MBUS_CSI2_DPHY;
	ad->cfg.mbus_flags = V4L2_MBUS_CSI2_4_LANE |
			 V4L2_MBUS_CSI2_CHANNELS;
	ad->cfg.mbus_code = MEDIA_BUS_FMT_UYVY8_2X8;

	switch (ad->cfg.mbus_flags & V4L2_MBUS_CSI2_LANES) {
	case V4L2_MBUS_CSI2_1_LANE:
		ad->cfg.lanes = 1;
		break;
	case V4L2_MBUS_CSI2_2_LANE:
		ad->cfg.lanes = 2;
		break;
	case V4L2_MBUS_CSI2_3_LANE:
		ad->cfg.lanes = 3;
		break;
	case V4L2_MBUS_CSI2_4_LANE:
		ad->cfg.lanes = 4;
		break;
	default:
		ad->cfg.lanes = 1;
		break;
	}

	/* fix crop info from dts config */
	for (i = 0; i < 4; i++) {
		if ((ad->defrects[i].width == ad->cfg.width) &&
		    (ad->defrects[i].height == ad->cfg.height)) {
			ad->cfg.start_x = ad->defrects[i].crop_x;
			ad->cfg.start_y = ad->defrects[i].crop_y;
			ad->cfg.width = ad->defrects[i].crop_width;
			ad->cfg.height = ad->defrects[i].crop_height;
		}
	}
}

static void nvp6324_reg_init(struct vehicle_ad_dev *ad, unsigned char cvstd)
{
	struct rk_sensor_reg *sensor;
	int i;

	switch (cvstd) {
	case CVSTD_720P25:
		VEHICLE_DG("%s, init CVSTD_720P25 mode)", __func__);
		sensor = sensor_preview_data_720p_25hz;
		break;
	case CVSTD_1080P25:
		VEHICLE_DG("%s, init CVSTD_1080P25 mode", __func__);
		sensor = sensor_preview_data_1080p_25hz;
		break;
	default:
		VEHICLE_DG("%s, init CVSTD_1080P25 mode", __func__);
		sensor = sensor_preview_data_1080p_25hz;
		break;
	}
	i = 0;
	while ((sensor[i].reg != SEQCMD_END) && (sensor[i].reg != 0xFC000000)) {
		vehicle_sensor_write(ad, sensor[i].reg, sensor[i].val);
		i++;
	}
	/* open format detect*/
	sensor = sensor_open_format_detect;
	i = 0;
	while ((sensor[i].reg != SEQCMD_END) && (sensor[i].reg != 0xFC000000)) {
		vehicle_sensor_write(ad, sensor[i].reg, sensor[i].val);
		i++;
	}

	vehicle_sensor_write(ad, 0xff, 0x05 + ad->ad_chl);
	vehicle_sensor_write(ad, 0x82, 0xff);
	vehicle_sensor_write(ad, 0xb8, 0xb9);
}

void nvp6324_channel_set(struct vehicle_ad_dev *ad, int channel)
{
	unsigned int reg;
	unsigned char val = 0x00;

	//detect interesting channel
	reg = channel;
	ad->ad_chl = channel;
	VEHICLE_DG("%s, channel set(%d)", __func__, ad->ad_chl);
	vehicle_sensor_write(ad, 0xff, 0x00);
	vehicle_sensor_write(ad, reg, val);
}

int nvp6324_ad_get_cfg(struct vehicle_cfg **cfg)
{

	if (!nvp6324_g_addev)
		return -ENODEV;

	switch (cvstd_state) {
	case VIDEO_UNPLUG:
		nvp6324_g_addev->cfg.ad_ready = false;
		break;
	case VIDEO_LOCKED:
		nvp6324_g_addev->cfg.ad_ready = true;
		break;
	case VIDEO_IN:
		nvp6324_g_addev->cfg.ad_ready = false;
		break;
	}

	nvp6324_g_addev->cfg.ad_ready = true;

	*cfg = &nvp6324_g_addev->cfg;

	return 0;
}

void nvp6324_ad_check_cif_error(struct vehicle_ad_dev *ad, int last_line)
{
	VEHICLE_DG("%s, last_line %d\n", __func__, last_line);

	if (last_line < 1)
		return;

	ad->cif_error_last_line = last_line;
	if (cvstd_mode == CVSTD_PAL) {
		if (last_line == FORCE_NTSC_HEIGHT) {
			if (ad->state_check_work.state_check_wq)
				queue_delayed_work(
					ad->state_check_work.state_check_wq,
					&ad->state_check_work.work,
					msecs_to_jiffies(0));
		}
	} else if (cvstd_mode == CVSTD_NTSC) {
		if (last_line == FORCE_PAL_HEIGHT) {
			if (ad->state_check_work.state_check_wq)
				queue_delayed_work(
					ad->state_check_work.state_check_wq,
					&ad->state_check_work.work,
					msecs_to_jiffies(0));
		}
	} else if (cvstd_mode == CVSTD_1080P25) {
		if (last_line == FORCE_1080P_HEIGHT) {
			if (ad->state_check_work.state_check_wq)
				queue_delayed_work(
					ad->state_check_work.state_check_wq,
					&ad->state_check_work.work,
					msecs_to_jiffies(0));
		}
	} else if (cvstd_mode == CVSTD_720P25) {
		if (last_line == FORCE_720P_HEIGHT) {
			if (ad->state_check_work.state_check_wq)
				queue_delayed_work(
					ad->state_check_work.state_check_wq,
					&ad->state_check_work.work,
					msecs_to_jiffies(0));
		}
	}
}

int nvp6324_check_id(struct vehicle_ad_dev *ad)
{
	int ret = 0;
	u8 pid;

	ret = vehicle_sensor_write(ad, 0xFF, 0x00);
	ret |= vehicle_sensor_read(ad, 0xf4, &pid);
	if (ret)
		return ret;
	if (pid != JAGUAR1_CHIP_ID) {
		VEHICLE_DGERR("%s: expected 0xB0, detected: 0x%02x !",
		    ad->ad_name, pid);
		ret = -EINVAL;
	} else {
		VEHICLE_INFO("%s Found NVP6324 sensor: id(0x%2x) !\n", __func__, pid);
	}

	return ret;
}

static int nvp6324_check_cvstd(struct vehicle_ad_dev *ad, bool activate_check)
{

	u8 videoloss = 0;
	int ret = 0;
	unsigned char cvstd = 0;

	ret = vehicle_sensor_write(ad, 0xFF, 0x00);
	ret |= vehicle_sensor_read(ad, 0xa4 + ad->ad_chl, &videoloss);

	video_mode = videoloss;

	ret |= vehicle_sensor_write(ad, 0xFF, 0x01);
	ret |= vehicle_sensor_read(ad, 0x10 + (0x20 * (ad->ad_chl%4)), &cvstd);

	if (ret)
		return ret;

	if (cvstd == 0x21) {
		cvstd_mode = CVSTD_720P25;
		VEHICLE_DG("%s(%d): 720P25\n", __func__, __LINE__);
	} else if (cvstd == 0x31) {
		cvstd_mode = CVSTD_1080P25;
		VEHICLE_DG("%s(%d): 1080P25", __func__, __LINE__);
	} else if (cvstd == 0x00) {
		cvstd_mode = CVSTD_NTSC;
		VEHICLE_DG("%s(%d): 720H NTSC\n", __func__, __LINE__);
	} else if (cvstd == 0x10) {
		cvstd_mode = CVSTD_PAL;
		VEHICLE_DG("%s(%d): 720H PAL\n", __func__, __LINE__);
	} else if (cvstd == 0xff) {
		cvstd_mode = cvstd_old;
		VEHICLE_DG("%s(%d): no ahd plugin!\n", __func__, __LINE__);
	} else {
		cvstd_mode = cvstd_old;
		VEHICLE_DG("%s(%d): not support ahd mode!\n", __func__, __LINE__);
	}

	return 0;
}

int nvp6324_stream(struct vehicle_ad_dev *ad, int enable)
{
	VEHICLE_DG("%s on(%d)\n", __func__, enable);

	g_nvp6324_streaming = (enable != 0);
	if (g_nvp6324_streaming) {
		vehicle_sensor_write(ad, 0xff, 0x21);
		vehicle_sensor_write(ad, 0x07, 0x0f);
		if (ad->state_check_work.state_check_wq)
			queue_delayed_work(ad->state_check_work.state_check_wq,
				&ad->state_check_work.work, msecs_to_jiffies(200));
	} else {
		vehicle_sensor_write(ad, 0xff, 0x21);
		vehicle_sensor_write(ad, 0x07, 0x8f);
		if (ad->state_check_work.state_check_wq)
			cancel_delayed_work_sync(&ad->state_check_work.work);
	}

	return 0;
}

static void nvp6324_power_on(struct vehicle_ad_dev *ad)
{
	/* gpio_direction_output(ad->power, ad->pwr_active); */
	VEHICLE_DG("gpio: power(%d), powerdown(%d), reset(%d)",
		ad->power, ad->powerdown, ad->reset);
	if (gpio_is_valid(ad->power)) {
		gpio_request(ad->power, "nvp6324_power");
		gpio_direction_output(ad->power, ad->pwr_active);
		/* gpio_set_value(ad->power, ad->pwr_active); */
	}

	if (gpio_is_valid(ad->powerdown)) {
		gpio_request(ad->powerdown, "nvp6324_pwd");
		gpio_direction_output(ad->powerdown, 1);
		/* gpio_set_value(ad->powerdown, !ad->pwdn_active); */
	}

	if (gpio_is_valid(ad->reset)) {
		gpio_request(ad->reset, "nvp6324_rst");
		gpio_direction_output(ad->reset, 0);
		usleep_range(1500, 2000);
		gpio_direction_output(ad->reset, 1);
	}
}

static void nvp6324_power_deinit(struct vehicle_ad_dev *ad)
{
	if (gpio_is_valid(ad->reset))
		gpio_free(ad->reset);
	if (gpio_is_valid(ad->power))
		gpio_free(ad->power);
	if (gpio_is_valid(ad->powerdown))
		gpio_free(ad->powerdown);
}

static void nvp6324_check_state_work(struct work_struct *work)
{
	struct vehicle_ad_dev *ad;

	ad = nvp6324_g_addev;

	if (ad->cif_error_last_line > 0) {
		nvp6324_check_cvstd(ad, true);
		ad->cif_error_last_line = 0;
	} else {
		nvp6324_check_cvstd(ad, false);
	}

	if (cvstd_old != cvstd_mode ||
		cvstd_old_state != cvstd_state || (video_old != video_mode)) {
		VEHICLE_INFO("%s:ad sensor std mode change, cvstd_old(%d), cvstd_mode(%d)\n",
				 __func__, cvstd_old, cvstd_mode);
		cvstd_old = cvstd_mode;
		cvstd_old_state = cvstd_state;
		video_old = video_mode;

		nvp6324_reinit_parameter(ad, cvstd_mode);
		nvp6324_reg_init(ad, cvstd_mode);
		vehicle_ad_stat_change_notify();
	}
	if (g_nvp6324_streaming) {
		queue_delayed_work(ad->state_check_work.state_check_wq,
				&ad->state_check_work.work, msecs_to_jiffies(100));
	}
}

int nvp6324_ad_deinit(void)
{
	struct vehicle_ad_dev *ad;

	ad = nvp6324_g_addev;

	if (!ad)
		return -ENODEV;

	if (ad->state_check_work.state_check_wq) {
		cancel_delayed_work_sync(&ad->state_check_work.work);
		flush_delayed_work(&ad->state_check_work.work);
		flush_workqueue(ad->state_check_work.state_check_wq);
		destroy_workqueue(ad->state_check_work.state_check_wq);
	}
	if (ad->irq)
		free_irq(ad->irq, ad);
	nvp6324_power_deinit(ad);

	return 0;
}

static __maybe_unused int get_ad_mode_from_fix_format(int fix_format)
{
	int mode = -1;

	switch (fix_format) {
	case AD_FIX_FORMAT_PAL:
	case AD_FIX_FORMAT_NTSC:
	case AD_FIX_FORMAT_720P_50FPS:
	case AD_FIX_FORMAT_720P_30FPS:
	case AD_FIX_FORMAT_720P_25FPS:
		mode = CVSTD_720P25;
		break;
	case AD_FIX_FORMAT_1080P_30FPS:
	case AD_FIX_FORMAT_1080P_25FPS:

	default:
		mode = CVSTD_720P25;
		break;
	}

	return mode;
}

int nvp6324_ad_init(struct vehicle_ad_dev *ad)
{
	int val;
	int i = 0;

	nvp6324_g_addev = ad;

	/*  1. i2c init */
	while (ad->adapter == NULL) {
		ad->adapter = i2c_get_adapter(ad->i2c_chl);
		usleep_range(10000, 12000);
	}
	if (ad->adapter == NULL)
		return -ENODEV;

	if (!i2c_check_functionality(ad->adapter, I2C_FUNC_I2C))
		return -EIO;

	nvp6324_power_on(ad);

	while (++i < 5) {
		usleep_range(10000, 12000);
		val = vehicle_generic_sensor_read(ad, 0xf0);
		if (val != 0xff)
			break;
		VEHICLE_DGERR("nvp6324_init i2c_reg_read fail\n");
	}

	nvp6324_reg_init(ad, cvstd_mode);

	nvp6324_reinit_parameter(ad, cvstd_mode);

	INIT_DELAYED_WORK(&ad->state_check_work.work, nvp6324_check_state_work);
	ad->state_check_work.state_check_wq =
		create_singlethread_workqueue("vehicle-ad-nvp6324");

	/* nvp6324_check_cvstd(ad, true); */

	queue_delayed_work(ad->state_check_work.state_check_wq,
			   &ad->state_check_work.work, msecs_to_jiffies(100));

	return 0;
}
