## 11.3 使用

本节将通过一个具体的 Web 应用案例，介绍 Docker Compose 的基本概念和使用方法。

### 11.3.1 术语

首先介绍几个术语。

* 服务 (`service`)：一个应用容器，实际上可以运行多个相同镜像的实例。

* 项目 (`project`)：由一组关联的应用容器组成的一个完整业务单元。

可见，一个项目可以由多个服务 (容器) 关联而成，`Compose` 面向项目进行管理。

### 11.3.2 场景

最常见的项目是 web 网站，该项目应该包含 web 应用和缓存。

下面我们用 `Python` 来建立一个能够记录页面访问次数的 web 网站。

#### web 应用

新建文件夹，在该目录中编写 `app.py` 文件

```python
from flask import Flask
from redis import Redis

app = Flask(__name__)
redis = Redis(host='redis', port=6379)

@app.route('/')
def hello():
    count = redis.incr('hits')
    return 'Hello World! 该页面已被访问 {} 次。\n'.format(count)

if __name__ == "__main__":
    app.run(host="0.0.0.0", debug=True)
```

#### Dockerfile

编写 `Dockerfile` 文件，内容为

```docker
FROM python:3.12-alpine
ADD . /code
WORKDIR /code
RUN pip install redis flask
CMD ["python", "app.py"]
```

#### compose.yaml

编写 `compose.yaml` 文件，这是 Compose 推荐使用的主模板文件 (也兼容 `docker-compose.yml` 等历史文件名)。

```yaml
services:
  web:
    build: .
    ports:
     - "5000:5000"

  redis:
    image: "redis:alpine"
```

#### 运行 compose 项目

```bash
$ docker compose up
```

此时访问本地 `5000` 端口，每次刷新页面，计数就会加 1。


按下 `Ctrl-C` 停止项目。

#### 后台运行

```bash
$ docker compose up -d
```

#### 停止

```bash
$ docker compose stop
```

#### 进入服务

```bash
$ docker compose exec redis sh
/data # redis-cli
127.0.0.1:6379> get hits
"9"
```

#### 查看日志

```bash
$ docker compose logs -f
```

#### 构建镜像

```bash
$ docker compose build
```

#### 启动服务

```bash
$ docker compose start
```

#### 运行一次性命令

```bash
$ docker compose run web python app.py
```

#### 验证 Compose 文件

```bash
$ docker compose config
```

#### 删除项目

```bash
$ docker compose down
```
