/*
 * This file is part of Cleanflight and Betaflight.
 *
 * Cleanflight and Betaflight are free software. You can redistribute
 * this software and/or modify this software under the terms of the
 * GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option)
 * any later version.
 *
 * Cleanflight and Betaflight are distributed in the hope that they
 * will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software.
 *
 * If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Author: 4712
 * for info about Hagens AVRootloader:
 * http://www.mikrocontroller.net/topic/avr-bootloader-mit-verschluesselung
*/

#include <stdbool.h>
#include <stdint.h>
#include <string.h>

#include "platform.h"

#ifdef  USE_SERIAL_4WAY_BLHELI_INTERFACE

#include "drivers/io.h"
#include "drivers/serial.h"
#include "drivers/time.h"
#include "drivers/timer.h"

#include "io/serial.h"
#include "io/serial_4way.h"
#include "io/serial_4way_impl.h"
#include "io/serial_4way_avrootloader.h"

#if defined(USE_SERIAL_4WAY_BLHELI_BOOTLOADER) && !defined(USE_VIRTUAL_ESC)

// Bootloader commands
// RunCmd
#define RestartBootloader   0
#define ExitBootloader      1

#define CMD_RUN             0x00
#define CMD_PROG_FLASH      0x01
#define CMD_ERASE_FLASH     0x02
#define CMD_READ_FLASH_SIL  0x03
#define CMD_VERIFY_FLASH    0x03
#define CMD_VERIFY_FLASH_ARM 0x04
#define CMD_READ_EEPROM     0x04
#define CMD_PROG_EEPROM     0x05
#define CMD_READ_SRAM       0x06
#define CMD_READ_FLASH_ATM  0x07
#define CMD_KEEP_ALIVE      0xFD
#define CMD_SET_ADDRESS     0xFF
#define CMD_SET_BUFFER      0xFE

#define CMD_BOOTINIT        0x07
#define CMD_BOOTSIGN        0x08

#define START_BIT_TIMEOUT_MS 2

#define BIT_TIME (52)       // 52uS
#define BIT_TIME_HALVE      (BIT_TIME >> 1) // 26uS
#define BIT_TIME_3_4        (BIT_TIME_HALVE + (BIT_TIME_HALVE >> 1))   // 39uS
#define START_BIT_TIME      (BIT_TIME_3_4)
//#define STOP_BIT_TIME     ((BIT_TIME * 9) + BIT_TIME_HALVE)

static uint8_t suart_getc_(uint8_t *bt)
{
    uint32_t btime;
    uint32_t start_time;

    uint32_t wait_time = millis() + START_BIT_TIMEOUT_MS;
    while (ESC_IS_HI) {
        // check for startbit begin
        if (millis() >= wait_time) {
            return 0;
        }
    }
    // start bit
    start_time = micros();
    btime = start_time + START_BIT_TIME;
    uint16_t bitmask = 0;
    uint8_t bit = 0;
    while (micros() < btime);
    while (1) {
        if (ESC_IS_HI)
        {
            bitmask |= (1 << bit);
        }
        btime = btime + BIT_TIME;
        bit++;
        if (bit == 10) break;
        while (micros() < btime);
    }
    // check start bit and stop bit
    if ((bitmask & 1) || (!(bitmask & (1 << 9)))) {
        return 0;
    }
    *bt = bitmask >> 1;
    return 1;
}

static void suart_putc_(uint8_t *tx_b)
{
    // shift out stopbit first
    uint16_t bitmask = (*tx_b << 2) | 1 | (1 << 10);
    uint32_t btime = micros();
    while (1) {
        if (bitmask & 1) {
            ESC_SET_HI; // 1
        }
        else {
            ESC_SET_LO; // 0
        }
        btime = btime + BIT_TIME;
        bitmask = (bitmask >> 1);
        if (bitmask == 0) break; // stopbit shifted out - but don't wait
        while (micros() < btime);
    }
}

static uint8_16_u CRC_16;
static uint8_16_u LastCRC_16;

static void ByteCrc(uint8_t *bt)
{
    uint8_t xb = *bt;
    for (uint8_t i = 0; i < 8; i++)
    {
        if (((xb & 0x01) ^ (CRC_16.word & 0x0001)) !=0 ) {
            CRC_16.word = CRC_16.word >> 1;
            CRC_16.word = CRC_16.word ^ 0xA001;
        } else {
            CRC_16.word = CRC_16.word >> 1;
        }
        xb = xb >> 1;
    }
}

static uint8_t BL_ReadBuf(uint8_t *pstring, uint8_t len)
{
    // len 0 means 256
    CRC_16.word = 0;
    LastCRC_16.word = 0;
    uint8_t  LastACK = brNONE;
    do {
        if (!suart_getc_(pstring)) goto timeout;
        ByteCrc(pstring);
        pstring++;
        len--;
    } while (len > 0);

    if (isMcuConnected()) {
        //With CRC read 3 more
        if (!suart_getc_(&LastCRC_16.bytes[0])) goto timeout;
        if (!suart_getc_(&LastCRC_16.bytes[1])) goto timeout;
        if (!suart_getc_(&LastACK)) goto timeout;
        if (CRC_16.word != LastCRC_16.word) {
            LastACK = brERRORCRC;
        }
    } else {
        if (!suart_getc_(&LastACK)) goto timeout;
    }
timeout:
    return (LastACK == brSUCCESS);
}

static void BL_SendBuf(uint8_t *pstring, uint8_t len)
{
    ESC_OUTPUT;
    CRC_16.word=0;
    do {
        suart_putc_(pstring);
        ByteCrc(pstring);
        pstring++;
        len--;
    } while (len > 0);

    if (isMcuConnected()) {
        suart_putc_(&CRC_16.bytes[0]);
        suart_putc_(&CRC_16.bytes[1]);
    }
    ESC_INPUT;
}

uint8_t BL_ConnectEx(uint8_32_u *pDeviceInfo)
{
    #define BootMsgLen 4
    #define DevSignHi (BootMsgLen)
    #define DevSignLo (BootMsgLen+1)

    //DeviceInfo.dword=0; is set before
    uint8_t BootInfo[9];
    uint8_t BootMsg[BootMsgLen-1] = "471";
    // x * 0 + 9
#if defined(USE_SERIAL_4WAY_SK_BOOTLOADER)
    uint8_t BootInit[] = {0,0,0,0,0,0,0,0,0,0,0,0,0x0D,'B','L','H','e','l','i',0xF4,0x7D};
    BL_SendBuf(BootInit, 21);
#else
    uint8_t BootInit[] = {0,0,0,0,0,0,0,0,0x0D,'B','L','H','e','l','i',0xF4,0x7D};
    BL_SendBuf(BootInit, 17);
#endif
    if (!BL_ReadBuf(BootInfo, BootMsgLen + 4)) {
        return 0;
    }
    // BootInfo has no CRC  (ACK byte already analyzed... )
    // Format = BootMsg("471c") SIGNATURE_001, SIGNATURE_002, BootVersion (always 6), BootPages (,ACK)
    for (uint8_t i = 0; i < (BootMsgLen - 1); i++) { // Check only the first 3 letters -> 471x OK
        if (BootInfo[i] != BootMsg[i]) {
            return (0);
        }
    }

    //only 2 bytes used $1E9307 -> 0x9307
    pDeviceInfo->bytes[2] = BootInfo[BootMsgLen - 1];
    pDeviceInfo->bytes[1] = BootInfo[DevSignHi];
    pDeviceInfo->bytes[0] = BootInfo[DevSignLo];
    return (1);
}

static uint8_t BL_GetACK(uint32_t Timeout)
{
    uint8_t LastACK = brNONE;
    while (!(suart_getc_(&LastACK)) && (Timeout)) {
        Timeout--;
    } ;
    return (LastACK);
}

uint8_t BL_SendCMDKeepAlive(void)
{
    uint8_t sCMD[] = {CMD_KEEP_ALIVE, 0};
    BL_SendBuf(sCMD, 2);
    if (BL_GetACK(1) != brERRORCOMMAND) {
        return 0;
    }
    return 1;
}

void BL_SendCMDRunRestartBootloader(uint8_32_u *pDeviceInfo)
{
    uint8_t sCMD[] = {RestartBootloader, 0};
    pDeviceInfo->bytes[0] = 1;
    BL_SendBuf(sCMD, 2); //sends simply 4 x 0x00 (CRC =00)
    return;
}

static uint8_t BL_SendCMDSetAddress(ioMem_t *pMem) //supports only 16 bit Adr
{
    // skip if adr == 0xFFFF
    if ((pMem->D_FLASH_ADDR_H == 0xFF) && (pMem->D_FLASH_ADDR_L == 0xFF)) return 1;
    uint8_t sCMD[] = {CMD_SET_ADDRESS, 0, pMem->D_FLASH_ADDR_H, pMem->D_FLASH_ADDR_L };
    BL_SendBuf(sCMD, 4);
    return (BL_GetACK(2) == brSUCCESS);
}

static uint8_t BL_SendCMDSetBuffer(ioMem_t *pMem)
{
    uint8_t sCMD[] = {CMD_SET_BUFFER, 0, 0, pMem->D_NUM_BYTES};
    if (pMem->D_NUM_BYTES == 0) {
        // set high byte
        sCMD[2] = 1;
    }
    BL_SendBuf(sCMD, 4);
    if (BL_GetACK(2) != brNONE) return 0;
    BL_SendBuf(pMem->D_PTR_I, pMem->D_NUM_BYTES);
    return (BL_GetACK(40) == brSUCCESS);
}

static uint8_t BL_ReadA(uint8_t cmd, ioMem_t *pMem)
{
    if (BL_SendCMDSetAddress(pMem)) {
        uint8_t sCMD[] = {cmd, pMem->D_NUM_BYTES};
        BL_SendBuf(sCMD, 2);
        return (BL_ReadBuf(pMem->D_PTR_I, pMem->D_NUM_BYTES ));
    }
    return 0;
}

static uint8_t BL_WriteA(uint8_t cmd, ioMem_t *pMem, uint32_t timeout)
{
    if (BL_SendCMDSetAddress(pMem)) {
        if (!BL_SendCMDSetBuffer(pMem)) return 0;
        uint8_t sCMD[] = {cmd, 0x01};
        BL_SendBuf(sCMD, 2);
        return (BL_GetACK(timeout) == brSUCCESS);
    }
    return 0;
}

uint8_t BL_ReadFlash(uint8_t interface_mode, ioMem_t *pMem)
{
    if (interface_mode == imATM_BLB) {
        return BL_ReadA(CMD_READ_FLASH_ATM, pMem);
    } else {
        return BL_ReadA(CMD_READ_FLASH_SIL, pMem);
    }
}

uint8_t BL_ReadEEprom(ioMem_t *pMem)
{
    return BL_ReadA(CMD_READ_EEPROM, pMem);
}

uint8_t BL_PageErase(ioMem_t *pMem)
{
    if (BL_SendCMDSetAddress(pMem)) {
        uint8_t sCMD[] = {CMD_ERASE_FLASH, 0x01};
        BL_SendBuf(sCMD, 2);
        return (BL_GetACK((3000 / START_BIT_TIMEOUT_MS)) == brSUCCESS);
    }
    return 0;
}

uint8_t BL_WriteEEprom(ioMem_t *pMem)
{
    return BL_WriteA(CMD_PROG_EEPROM, pMem, (3000 / START_BIT_TIMEOUT_MS));
}

uint8_t BL_WriteFlash(ioMem_t *pMem)
{
    return BL_WriteA(CMD_PROG_FLASH, pMem, (500 / START_BIT_TIMEOUT_MS));
}

uint8_t BL_VerifyFlash(ioMem_t *pMem)
{
    if (BL_SendCMDSetAddress(pMem)) {
        if (!BL_SendCMDSetBuffer(pMem)) return 0;
        uint8_t sCMD[] = {CMD_VERIFY_FLASH_ARM, 0x01};
        BL_SendBuf(sCMD, 2);
        return (BL_GetACK(40 / START_BIT_TIMEOUT_MS));
    }
    return 0;
}

#endif
#if defined(USE_SERIAL_4WAY_BLHELI_BOOTLOADER) && defined(USE_VIRTUAL_ESC)

#define VIRTUAL_PAGE_SIZE 512
#define VIRTUAL_FLASH_SIZE 16385

static uint8_t virtualFlash[VIRTUAL_FLASH_SIZE] =
{
    0x02, 0x19, 0xFD, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x02, 0x00, 0xAA, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x02, 0x01, 0x87, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x02, 0x03, 0x44, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0x02, 0x03, 0x31, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0x02, 0x03, 0x04, 0x06, 0x08, 0x0C, 0x10, 0x18, 0x20, 0x30, 0x40, 0x60, 0x80, 0x04, 0x06, 0x08,
    0x0C, 0x10, 0x18, 0x20, 0x30, 0x40, 0x60, 0x80, 0xA0, 0xC0, 0x00, 0x03, 0x07, 0x0F, 0x1F, 0x0D,
    0x0D, 0x04, 0x05, 0x0D, 0x05, 0x03, 0x05, 0x03, 0x03, 0x02, 0xC2, 0xAF, 0x30, 0x63, 0x08, 0xC2,
    0x63, 0x85, 0x79, 0x8A, 0xD2, 0xAF, 0x32, 0xC0, 0xD0, 0xC0, 0xE0, 0x20, 0x62, 0x21, 0xE5, 0x25,
    0x60, 0x16, 0xE4, 0x73, 0xE5, 0x25, 0xF4, 0xC3, 0x33, 0x40, 0x09, 0x75, 0x8A, 0x00, 0xD2, 0x63,
    0xF5, 0x79, 0x01, 0xD6, 0xF5, 0x8A, 0xD2, 0x62, 0xD0, 0xE0, 0xD0, 0xD0, 0xD2, 0xAF, 0x32, 0xC3,
    0xE5, 0x26, 0x33, 0x40, 0x09, 0x75, 0x8A, 0x00, 0xD2, 0x63, 0xF5, 0x79, 0x01, 0xF0, 0xF5, 0x8A,
    0xC2, 0x62, 0xE5, 0x26, 0xF4, 0x60, 0x2F, 0x30, 0x68, 0x25, 0x20, 0x72, 0x0D, 0xD0, 0xE0, 0xD0,
    0xD0, 0xC2, 0x93, 0xC2, 0x97, 0xC2, 0x94, 0xD2, 0xAF, 0x32, 0xC2, 0x93, 0xC2, 0x97, 0xC2, 0x94,
    0x20, 0x6F, 0x0C, 0x20, 0x66, 0x09, 0x74, 0x06, 0xD5, 0xE0, 0xFD, 0xE5, 0x7B, 0x42, 0x90, 0xD0,
    0xE0, 0xD0, 0xD0, 0xD2, 0xAF, 0x32, 0x75, 0x8A, 0x00, 0xD2, 0x63, 0x75, 0x79, 0x00, 0xC2, 0x8D,
    0xD2, 0x62, 0x21, 0x1F, 0x01, 0xC4, 0x30, 0x68, 0x05, 0x20, 0x66, 0x02, 0xD2, 0x93, 0x01, 0xC4,
    0x30, 0x68, 0x05, 0x20, 0x66, 0x02, 0xD2, 0x94, 0x01, 0xC4, 0x30, 0x68, 0x05, 0x20, 0x66, 0x02,
    0xD2, 0x97, 0x01, 0xC4, 0xC2, 0x92, 0x30, 0x68, 0x0A, 0x20, 0x66, 0x07, 0x74, 0x06, 0xD5, 0xE0,
    0xFD, 0xD2, 0x93, 0x01, 0xC4, 0xC2, 0x95, 0x30, 0x68, 0x0A, 0x20, 0x66, 0x07, 0x74, 0x06, 0xD5,
    0xE0, 0xFD, 0xD2, 0x94, 0x01, 0xC4, 0xC2, 0x96, 0x30, 0x68, 0x0A, 0x20, 0x66, 0x07, 0x74, 0x06,
    0xD5, 0xE0, 0xFD, 0xD2, 0x97, 0x01, 0xC4, 0xC2, 0xAF, 0xC2, 0xAD, 0x53, 0xE6, 0xEF, 0xC0, 0xD0,
    0xC0, 0xE0, 0xD2, 0xD3, 0xD2, 0xAF, 0xE5, 0x7A, 0x60, 0x09, 0xE5, 0x77, 0x60, 0x05, 0x75, 0x77,
    0x00, 0x41, 0xDE, 0x75, 0x77, 0x01, 0xC2, 0xCE, 0xE5, 0x2A, 0x60, 0x07, 0x20, 0x74, 0x49, 0x15,
    0x2A, 0x21, 0xF8, 0x78, 0x00, 0x79, 0x00, 0x20, 0x74, 0x2E, 0xE5, 0x80, 0x30, 0x7E, 0x01, 0xF4,
    0x30, 0xE5, 0x02, 0x78, 0xFF, 0x53, 0xDA, 0xCF, 0x20, 0x7E, 0x03, 0x43, 0xDA, 0x20, 0x30, 0x7E,
    0x03, 0x43, 0xDA, 0x10, 0xC2, 0xD8, 0xC2, 0x71, 0xE5, 0x80, 0x30, 0x7E, 0x01, 0xF4, 0x30, 0xE5,
    0x02, 0x79, 0xFF, 0xC3, 0xE8, 0x99, 0x70, 0xCB, 0x30, 0x61, 0x03, 0x75, 0x2A, 0x18, 0x20, 0x74,
    0x03, 0x75, 0x2A, 0x18, 0x88, 0x5C, 0xD2, 0x70, 0x20, 0x74, 0x0D, 0xE5, 0x2B, 0x60, 0x04, 0x15,
    0x2B, 0x41, 0x08, 0x43, 0xDA, 0x01, 0xC2, 0xD8, 0x20, 0x70, 0x02, 0x41, 0x77, 0xA8, 0x5C, 0x20,
    0x61, 0x02, 0xC2, 0x70, 0x20, 0x74, 0x36, 0x79, 0x82, 0xB7, 0x04, 0x31, 0xC3, 0xE8, 0x94, 0xF0,
    0x40, 0x03, 0x74, 0xF0, 0xF8, 0xE8, 0xC4, 0x54, 0x0F, 0x28, 0xF8, 0x79, 0x84, 0xB7, 0x03, 0x02,
    0x41, 0x4D, 0xC3, 0x13, 0xC3, 0x13, 0x87, 0x21, 0x20, 0x08, 0x02, 0xC3, 0x13, 0x20, 0x0A, 0x06,
    0xC3, 0xC8, 0x98, 0xF8, 0x41, 0x4D, 0x28, 0xF8, 0x50, 0x03, 0x74, 0xFF, 0xF8, 0x88, 0x22, 0xE5,
    0x2D, 0x54, 0x06, 0x60, 0x22, 0x20, 0x6B, 0x1F, 0xE5, 0x36, 0xC3, 0x33, 0xC3, 0x33, 0xF8, 0xE5,
    0x64, 0xC3, 0x13, 0xC3, 0x13, 0xF9, 0xC3, 0x95, 0x22, 0x40, 0x02, 0x89, 0x22, 0xE8, 0x25, 0x22,
    0xF5, 0x22, 0x50, 0x03, 0x75, 0x22, 0xFF, 0x78, 0x82, 0xB6, 0x04, 0x59, 0x85, 0x22, 0x24, 0xA8,
    0x24, 0xC3, 0xE5, 0x24, 0x95, 0x61, 0x40, 0x02, 0xA8, 0x61, 0xC3, 0xE8, 0x95, 0x63, 0x40, 0x02,
    0xA8, 0x63, 0x88, 0x25, 0xE5, 0x66, 0x70, 0x02, 0x41, 0xC8, 0xC3, 0xE8, 0xAA, 0x66, 0x9A, 0x50,
    0x03, 0xE8, 0xFA, 0xE4, 0xF9, 0xEA, 0xC3, 0x33, 0xFB, 0xE5, 0x68, 0xF4, 0x5B, 0x29, 0x40, 0x0D,
    0x25, 0x67, 0xF8, 0xE5, 0x67, 0x60, 0x02, 0x15, 0x67, 0x40, 0x0B, 0x41, 0xC8, 0x0A, 0xC3, 0xE5,
    0x67, 0x9A, 0x50, 0x02, 0x05, 0x67, 0x78, 0xFF, 0x88, 0x26, 0xC2, 0x6F, 0xC3, 0xE5, 0x26, 0x94,
    0xF8, 0x40, 0x02, 0xD2, 0x6F, 0xC3, 0xE5, 0x25, 0x94, 0x40, 0x40, 0x02, 0xD2, 0x64, 0x20, 0xCF,
    0x0A, 0xD0, 0xE0, 0xD0, 0xD0, 0x43, 0xE6, 0x10, 0xD2, 0xAD, 0x32, 0xC2, 0xCF, 0x05, 0x3A, 0xE5,
    0x7A, 0x60, 0x09, 0xE5, 0x78, 0x60, 0x05, 0x75, 0x78, 0x00, 0x61, 0x27, 0x75, 0x78, 0x01, 0x78,
    0x01, 0xE5, 0x2A, 0x60, 0x05, 0x30, 0x74, 0x02, 0x15, 0x2A, 0xC3, 0xE5, 0x5C, 0x94, 0x01, 0x40,
    0x05, 0x75, 0x5F, 0x00, 0x61, 0x27, 0x75, 0x60, 0x00, 0x75, 0x69, 0x00, 0xE5, 0x5F, 0x24, 0x01,
    0xF5, 0x5F, 0x50, 0x03, 0x75, 0x5F, 0xFF, 0xD0, 0xE0, 0xD0, 0xD0, 0x43, 0xE6, 0x10, 0xD2, 0xAD,
    0x32, 0xC2, 0xAF, 0x53, 0xE6, 0x7F, 0x75, 0x92, 0xFA, 0x75, 0x93, 0xFF, 0xC2, 0x60, 0x75, 0x91,
    0x04, 0xD2, 0xAF, 0x32, 0xC2, 0xAF, 0x53, 0xE6, 0xEF, 0xC2, 0xAD, 0xC0, 0xD0, 0xC0, 0xE0, 0xC0,
    0xF0, 0xD2, 0xD3, 0xD2, 0xAF, 0xA8, 0xFB, 0xA9, 0xFC, 0xE5, 0x7A, 0x60, 0x07, 0xC3, 0xE9, 0x13,
    0xF9, 0xE8, 0x13, 0xF8, 0xC2, 0xD8, 0x30, 0x71, 0x02, 0x61, 0xB6, 0x53, 0xDA, 0xCF, 0x20, 0x7E,
    0x03, 0x43, 0xDA, 0x10, 0x30, 0x7E, 0x03, 0x43, 0xDA, 0x20, 0xD2, 0x71, 0xE5, 0x80, 0x30, 0x7E,
    0x01, 0xF4, 0x20, 0xE5, 0x02, 0x61, 0x8D, 0x88, 0x27, 0x89, 0x28, 0xA1, 0xE3, 0x53, 0xDA, 0xCF,
    0x20, 0x7E, 0x03, 0x43, 0xDA, 0x20, 0x30, 0x7E, 0x03, 0x43, 0xDA, 0x10, 0xC2, 0xD8, 0xC2, 0x71,
    0x30, 0x74, 0x02, 0xA1, 0xCF, 0x78, 0x00, 0xE5, 0x80, 0x30, 0x7E, 0x01, 0xF4, 0x30, 0xE5, 0x02,
    0xA1, 0xCF, 0x88, 0x5C, 0xA1, 0xB7, 0x53, 0xDA, 0xCF, 0x20, 0x7E, 0x03, 0x43, 0xDA, 0x20, 0x30,
    0x7E, 0x03, 0x43, 0xDA, 0x10, 0xC2, 0x71, 0x20, 0x61, 0x02, 0x81, 0x8F, 0x53, 0xDA, 0xCF, 0x20,
    0x7E, 0x03, 0x43, 0xDA, 0x10, 0x30, 0x7E, 0x03, 0x43, 0xDA, 0x20, 0xC2, 0xD8, 0xD2, 0x71, 0x88,
    0x57, 0x89, 0x58, 0xC3, 0xE8, 0x95, 0x55, 0xF8, 0xE9, 0x95, 0x56, 0xF9, 0x7B, 0x00, 0x7E, 0x08,
    0x7A, 0x00, 0xC3, 0xE8, 0x94, 0x8C, 0xE9, 0x94, 0x00, 0x50, 0x05, 0x75, 0x5B, 0x00, 0x81, 0x81,
    0x88, 0x21, 0x78, 0xA2, 0xE6, 0xA8, 0x21, 0x60, 0x55, 0xC3, 0xE8, 0x94, 0xC8, 0xE9, 0x94, 0x00,
    0x50, 0x08, 0xE4, 0xD2, 0xE4, 0xFB, 0x7A, 0x0A, 0x81, 0x5C, 0xC3, 0xE8, 0x94, 0x68, 0xE9, 0x94,
    0x01, 0x50, 0x08, 0xE4, 0xD2, 0xE3, 0xFB, 0x7A, 0x0F, 0x81, 0x5C, 0xC3, 0xE8, 0x94, 0xD0, 0xE9,
    0x94, 0x02, 0x50, 0x08, 0xE4, 0xD2, 0xE2, 0xFB, 0x7A, 0x1E, 0x81, 0x5C, 0xC3, 0xE8, 0x94, 0xA0,
    0xE9, 0x94, 0x05, 0x50, 0x08, 0xE4, 0xD2, 0xE1, 0xFB, 0x7A, 0x3C, 0x81, 0x5C, 0xC3, 0xE8, 0x94,
    0x98, 0xE9, 0x94, 0x08, 0x50, 0x08, 0xE4, 0xD2, 0xE0, 0xFB, 0x7A, 0x78, 0x7E, 0x00, 0xC3, 0xE8,
    0x95, 0x59, 0xFC, 0xE9, 0x95, 0x5A, 0xFD, 0x30, 0xE7, 0x0A, 0xEC, 0xF4, 0x24, 0x01, 0xFC, 0xED,
    0xF4, 0x34, 0x00, 0xFD, 0x75, 0x5B, 0x00, 0xC3, 0xEC, 0x9A, 0xED, 0x9E, 0x50, 0x03, 0x75, 0x5B,
    0x01, 0x88, 0x59, 0x89, 0x5A, 0x85, 0x57, 0x55, 0x85, 0x58, 0x56, 0x78, 0x02, 0xA1, 0xB7, 0xC3,
    0xE8, 0x95, 0x27, 0xF8, 0xE9, 0x95, 0x28, 0xF9, 0x30, 0x7C, 0x02, 0xA1, 0x9B, 0x30, 0x7B, 0x02,
    0xA1, 0x9B, 0x30, 0x7A, 0x02, 0xA1, 0x94, 0x20, 0x75, 0x02, 0x81, 0xB2, 0xE9, 0xFD, 0xE8, 0xFC,
    0x81, 0xD1, 0xE9, 0xC3, 0x13, 0xF9, 0xE8, 0x13, 0xF8, 0x30, 0x79, 0x02, 0xA1, 0x94, 0xE9, 0xC3,
    0x13, 0xF9, 0xE8, 0x13, 0xF8, 0x30, 0x78, 0x02, 0xA1, 0x94, 0xE9, 0xC3, 0x13, 0xFD, 0xE8, 0x13,
    0xFC, 0x20, 0x61, 0x2C, 0xC3, 0xEC, 0x94, 0x1C, 0xED, 0x94, 0x02, 0x40, 0x02, 0x81, 0xE8, 0xED,
    0x70, 0x1E, 0xC3, 0xEC, 0x94, 0xC8, 0x50, 0x18, 0x05, 0x29, 0xE5, 0x29, 0x70, 0x02, 0x15, 0x29,
    0xC3, 0xE5, 0x29, 0x94, 0x0A, 0x50, 0x02, 0xA1, 0xCF, 0x75, 0x5C, 0x00, 0xD2, 0x70, 0xA1, 0xCF,
    0xE5, 0x29, 0x60, 0x02, 0x15, 0x29, 0x78, 0x88, 0xE6, 0xF9, 0x74, 0x00, 0x20, 0x7F, 0x08, 0x78,
    0x96, 0xB9, 0x03, 0x02, 0x78, 0x9E, 0xE6, 0x24, 0xFA, 0xFE, 0xE4, 0x34, 0x00, 0xFF, 0xC3, 0xEC,
    0x9E, 0xFC, 0xED, 0x9F, 0xFD, 0x92, 0x08, 0xB9, 0x03, 0x10, 0xA2, 0x08, 0x50, 0x07, 0x20, 0x76,
    0x09, 0xD2, 0x76, 0xA1, 0x3A, 0x30, 0x76, 0x02, 0xC2, 0x76, 0xA2, 0x08, 0x50, 0x16, 0xB9, 0x03,
    0x0D, 0xEC, 0xF4, 0x24, 0x01, 0xFC, 0xED, 0xF4, 0x34, 0x00, 0xFD, 0x02, 0x05, 0x54, 0x78, 0x00,
    0x79, 0x00, 0xA1, 0x9B, 0xB9, 0x03, 0x15, 0xEC, 0x33, 0xFC, 0xED, 0x33, 0xFD, 0xC3, 0xEC, 0x94,
    0x0A, 0xFC, 0xED, 0x94, 0x00, 0xFD, 0x50, 0x04, 0x7C, 0x00, 0x7D, 0x00, 0xC3, 0xEC, 0x94, 0xFF,
    0xED, 0x94, 0x00, 0x40, 0x06, 0x78, 0xFF, 0x79, 0x00, 0xA1, 0x9B, 0xEC, 0x85, 0x72, 0xF0, 0xA4,
    0xC5, 0xF0, 0xA2, 0xF7, 0x33, 0xF8, 0x79, 0x00, 0x40, 0x03, 0x02, 0x05, 0xB7, 0x78, 0xFF, 0x79,
    0x00, 0x02, 0x05, 0xB7, 0xE9, 0xC3, 0x13, 0xF9, 0xE8, 0x13, 0xF8, 0x30, 0x7C, 0x0E, 0xE9, 0x60,
    0x02, 0x78, 0xFF, 0xC3, 0xE8, 0x13, 0x38, 0xF8, 0xE4, 0x34, 0x00, 0xF9, 0xC3, 0xE8, 0x94, 0xFF,
    0xE9, 0x94, 0x00, 0x40, 0x02, 0x78, 0xFF, 0x88, 0x5C, 0xD2, 0x70, 0x20, 0x61, 0x02, 0xA1, 0xCF,
    0x74, 0x1F, 0xF4, 0x55, 0x2F, 0x4B, 0xF5, 0x2F, 0xC2, 0x74, 0xEB, 0x70, 0x02, 0xD2, 0x74, 0x75,
    0x2A, 0x18, 0x30, 0x74, 0x03, 0x75, 0x2A, 0x0A, 0x30, 0x61, 0x02, 0xA1, 0xE3, 0x20, 0x74, 0x03,
    0x53, 0xDA, 0xFE, 0x20, 0x74, 0x03, 0x75, 0x2B, 0x06, 0xD0, 0xF0, 0xD0, 0xE0, 0xD0, 0xD0, 0xD2,
    0xAD, 0x43, 0xE6, 0x10, 0x32, 0x79, 0x01, 0x02, 0x06, 0x13, 0x79, 0x03, 0x02, 0x06, 0x13, 0x79,
    0x0A, 0x02, 0x06, 0x13, 0x79, 0x1E, 0x02, 0x06, 0x13, 0x79, 0x64, 0x02, 0x06, 0x13, 0x79, 0xC8,
    0x02, 0x06, 0x13, 0x78, 0x17, 0xE4, 0xD5, 0xE0, 0xFD, 0xD8, 0xFA, 0xD9, 0xF6, 0x22, 0x7A, 0x14,
    0x7B, 0x78, 0x02, 0x06, 0x3A, 0x7A, 0x10, 0x7B, 0x8C, 0x02, 0x06, 0x3A, 0x7A, 0x0D, 0x7B, 0xB4,
    0x02, 0x06, 0x3A, 0x7A, 0x0B, 0x7B, 0xC8, 0x02, 0x06, 0x3A, 0x79, 0x02, 0xE4, 0xC2, 0x95, 0xD5,
    0xE0, 0xFD, 0xD2, 0x94, 0xD5, 0xE0, 0xFD, 0xC2, 0x94, 0xD5, 0xE0, 0xFD, 0xD2, 0x95, 0xD5, 0xE0,
    0xFD, 0xE9, 0x20, 0xE0, 0x02, 0xD2, 0x93, 0x30, 0xE0, 0x02, 0xD2, 0x97, 0xE5, 0x73, 0xD5, 0xE0,
    0xFD, 0xE9, 0x20, 0xE0, 0x02, 0xC2, 0x93, 0x30, 0xE0, 0x02, 0xC2, 0x97, 0x74, 0x96, 0xD5, 0xE0,
    0xFD, 0xD9, 0xC9, 0xEA, 0xF8, 0xD5, 0xE0, 0xFD, 0xD8, 0xFB, 0xDB, 0xBE, 0xC2, 0x95, 0x22, 0xC3,
    0x7C, 0x00, 0x7D, 0x00, 0x75, 0xF0, 0x00, 0x05, 0xF0, 0xEA, 0x33, 0xFA, 0xEB, 0x33, 0xFB, 0x50,
    0xF6, 0xEB, 0x13, 0xFB, 0xEA, 0x13, 0xFA, 0xC3, 0xE9, 0xFF, 0xE8, 0xFE, 0xE8, 0x9A, 0xF8, 0xE9,
    0x9B, 0xF9, 0x50, 0x04, 0xEF, 0xF9, 0xEE, 0xF8, 0xB3, 0xEC, 0x33, 0xFC, 0xED, 0x33, 0xFD, 0xD5,
    0xF0, 0xDF, 0xED, 0xF9, 0xEC, 0xF8, 0x22, 0xE8, 0x89, 0xF0, 0x8A, 0x20, 0xD2, 0xD4, 0xF8, 0xA9,
    0xF0, 0x7B, 0x00, 0x30, 0xF7, 0x0B, 0x7B, 0xFF, 0xF4, 0x24, 0x01, 0xF8, 0xE9, 0xF4, 0x34, 0x00,
    0xF9, 0xE8, 0x85, 0x20, 0xF0, 0xA4, 0xAD, 0xF0, 0xF8, 0xE9, 0x85, 0x20, 0xF0, 0xA4, 0xAF, 0xF0,
    0xFE, 0xED, 0x2E, 0xF9, 0x74, 0x00, 0x3F, 0xFA, 0x7C, 0x04, 0xC3, 0xEA, 0x13, 0xFA, 0xE9, 0x13,
    0xF9, 0xE8, 0x13, 0xF8, 0xDC, 0xF4, 0x8B, 0xF0, 0x30, 0xF7, 0x0A, 0xE8, 0xF4, 0x24, 0x01, 0xF8,
    0xE9, 0xF4, 0x34, 0x00, 0xF9, 0xE8, 0x89, 0xF0, 0xC2, 0xD4, 0xF8, 0xA9, 0xF0, 0x22, 0x78, 0x82,
    0xB6, 0x04, 0x03, 0x02, 0x07, 0x69, 0xC3, 0xE5, 0x5C, 0x94, 0x01, 0x40, 0x03, 0x02, 0x07, 0x33,
    0x85, 0x22, 0x24, 0xE4, 0xF5, 0x44, 0xF5, 0x45, 0xF5, 0x46, 0xF5, 0x47, 0xF5, 0x48, 0xC2, 0x6E,
    0x02, 0x07, 0x69, 0x78, 0x82, 0xE6, 0xFC, 0xD2, 0x6E, 0xE5, 0x22, 0xF5, 0x23, 0x78, 0x38, 0x79,
    0xC7, 0xAA, 0x40, 0xAB, 0x41, 0xC3, 0xEB, 0x13, 0xFB, 0xEA, 0x13, 0xFA, 0xEC, 0x14, 0x60, 0x13,
    0xC3, 0xEB, 0x13, 0xFB, 0xEA, 0x13, 0xFA, 0xEC, 0x14, 0x14, 0x60, 0x07, 0xC3, 0xEB, 0x13, 0xFB,
    0xEA, 0x13, 0xFA, 0xD1, 0x7F, 0x88, 0x44, 0x89, 0x45, 0x22, 0xC3, 0xE5, 0x44, 0x95, 0x23, 0xF8,
    0xE5, 0x45, 0x94, 0x00, 0xF9, 0x50, 0x0C, 0xC3, 0xE8, 0x94, 0x80, 0xE9, 0x94, 0xFF, 0x40, 0x16,
    0x02, 0x07, 0x9A, 0xC3, 0xE8, 0x94, 0x7F, 0xE9, 0x94, 0x00, 0x50, 0x03, 0x02, 0x07, 0x9A, 0x78,
    0x7F, 0x79, 0x00, 0x02, 0x07, 0x9A, 0x78, 0x80, 0x79, 0xFF, 0x88, 0x49, 0x89, 0x4A, 0x22, 0xE5,
    0x49, 0x25, 0x46, 0xF8, 0xE5, 0x4A, 0x35, 0x47, 0xF9, 0x85, 0x4A, 0x20, 0xE4, 0x30, 0x07, 0x01,
    0xF4, 0x35, 0x48, 0xFA, 0x30, 0xE7, 0x09, 0xC3, 0xEA, 0x94, 0xF0, 0x40, 0x15, 0x02, 0x07, 0xD8,
    0xC3, 0xEA, 0x94, 0x0F, 0x50, 0x03, 0x02, 0x07, 0xD8, 0x78, 0xFF, 0x79, 0xFF, 0x7A, 0x0F, 0x02,
    0x07, 0xD8, 0x78, 0x00, 0x79, 0x00, 0x7A, 0xF0, 0xC3, 0xE5, 0x24, 0x95, 0x61, 0x50, 0x0A, 0xC3,
    0x74, 0x01, 0x95, 0x24, 0x50, 0x0B, 0x02, 0x07, 0xF6, 0xE5, 0x4A, 0x20, 0xE7, 0x0E, 0x02, 0x07,
    0xF6, 0xE5, 0x4A, 0x30, 0xE7, 0x06, 0x88, 0x46, 0x89, 0x47, 0x8A, 0x48, 0x22, 0x78, 0xA5, 0xE6,
    0xFA, 0xC3, 0xE5, 0x49, 0x33, 0xF8, 0xE5, 0x4A, 0x33, 0xF9, 0x12, 0x06, 0xB7, 0xE9, 0x30, 0xE7,
    0x0B, 0xC3, 0xE8, 0x94, 0x80, 0xE9, 0x94, 0xFF, 0x40, 0x13, 0x01, 0x31, 0xC3, 0xE8, 0x94, 0x7F,
    0xE9, 0x94, 0x00, 0x50, 0x02, 0x01, 0x31, 0x78, 0x7F, 0x79, 0x00, 0x01, 0x31, 0x78, 0x80, 0x79,
    0xFF, 0xE8, 0x20, 0xE7, 0x15, 0xC3, 0xE5, 0x23, 0x98, 0xF8, 0x40, 0x09, 0xC3, 0xE8, 0x94, 0x01,
    0x40, 0x03, 0x02, 0x08, 0x58, 0x78, 0x01, 0x02, 0x08, 0x58, 0xE8, 0xF4, 0x24, 0x01, 0x25, 0x23,
    0xF8, 0x40, 0x03, 0x02, 0x08, 0x58, 0x78, 0xFF, 0x88, 0x4B, 0x22, 0x78, 0xA6, 0xE6, 0xFA, 0xA8,
    0x47, 0xA9, 0x48, 0x12, 0x06, 0xB7, 0xE9, 0x30, 0xE7, 0x0C, 0xC3, 0xE8, 0x94, 0x01, 0xE9, 0x94,
    0xFF, 0x40, 0x16, 0x02, 0x08, 0x8D, 0xC3, 0xE8, 0x94, 0xFF, 0xE9, 0x94, 0x00, 0x50, 0x03, 0x02,
    0x08, 0x8D, 0x78, 0xFF, 0x79, 0x00, 0x02, 0x08, 0x8D, 0x78, 0x01, 0x79, 0xFF, 0xE9, 0x20, 0xE7,
    0x15, 0xC3, 0xE5, 0x4B, 0x98, 0xF8, 0x40, 0x09, 0xC3, 0xE8, 0x94, 0x01, 0x40, 0x03, 0x02, 0x08,
    0xB4, 0x78, 0x01, 0x02, 0x08, 0xB4, 0xE8, 0xF4, 0x24, 0x01, 0x25, 0x4B, 0xF8, 0x40, 0x03, 0x02,
    0x08, 0xB4, 0x78, 0xFF, 0x88, 0x24, 0x22, 0x78, 0xFF, 0xC2, 0x64, 0x20, 0x69, 0x3D, 0x20, 0x6A,
    0x12, 0xD2, 0x64, 0xC3, 0xE5, 0x41, 0x94, 0x0A, 0x40, 0x09, 0xC3, 0xE5, 0x25, 0x94, 0x40, 0x50,
    0x02, 0xC2, 0x64, 0x79, 0xA1, 0xE7, 0x60, 0x23, 0xE5, 0x41, 0x60, 0x1F, 0x74, 0xFF, 0x85, 0x41,
    0xF0, 0x84, 0x85, 0x39, 0xF0, 0x30, 0x6A, 0x03, 0x75, 0xF0, 0x05, 0xA4, 0xF8, 0xC5, 0xF0, 0x60,
    0x02, 0x78, 0xFF, 0xC3, 0xE8, 0x95, 0x64, 0x50, 0x02, 0xA8, 0x64, 0x88, 0x63, 0x22, 0xC3, 0xE5,
    0x40, 0x94, 0xC8, 0xE5, 0x41, 0x94, 0x00, 0xE5, 0x63, 0x50, 0x03, 0x14, 0x21, 0x0F, 0x04, 0x60,
    0x02, 0xF5, 0x63, 0x22, 0x02, 0x09, 0x17, 0x22, 0x75, 0xE8, 0x90, 0x22, 0x78, 0x83, 0xE6, 0xFF,
    0xE5, 0xE8, 0x20, 0xEC, 0xF7, 0xA8, 0xBD, 0xA9, 0xBE, 0x05, 0x70, 0xC3, 0xE5, 0x70, 0x94, 0x08,
    0x40, 0x52, 0x75, 0x70, 0x00, 0xE9, 0xFA, 0x79, 0xA0, 0xE7, 0x60, 0x44, 0xEA, 0x70, 0x07, 0xE5,
    0x71, 0x60, 0x1B, 0x02, 0x09, 0x52, 0xC3, 0xE8, 0x95, 0x71, 0x60, 0x10, 0xE5, 0x71, 0x50, 0x06,
    0x60, 0x0C, 0x14, 0x02, 0x09, 0x5E, 0x04, 0x60, 0xF9, 0x02, 0x09, 0x5E, 0xE5, 0x71, 0xF5, 0x71,
    0xC3, 0x94, 0x72, 0x40, 0x1B, 0x75, 0x61, 0xC0, 0xC3, 0x94, 0x04, 0x40, 0x13, 0x75, 0x61, 0x80,
    0xC3, 0x94, 0x04, 0x40, 0x0B, 0x75, 0x61, 0x40, 0xC3, 0x94, 0x04, 0x40, 0x03, 0x75, 0x61, 0x00,
    0x75, 0xBB, 0x09, 0x22, 0xE5, 0x61, 0x24, 0x10, 0x50, 0x02, 0x74, 0xFF, 0xF5, 0x61, 0x79, 0x82,
    0xB7, 0x04, 0x02, 0x21, 0xAB, 0xC3, 0xA8, 0x61, 0xE5, 0x24, 0x98, 0x50, 0x02, 0xA8, 0x24, 0xC3,
    0xE8, 0x95, 0x63, 0x40, 0x02, 0xA8, 0x63, 0x88, 0x25, 0x88, 0x26, 0xE5, 0x70, 0xB4, 0x07, 0x03,
    0x75, 0xBB, 0x10, 0x22, 0x74, 0x32, 0x79, 0xA7, 0x87, 0xF0, 0xA4, 0xC5, 0xF0, 0xA2, 0xF7, 0x33,
    0xF8, 0xC3, 0xE8, 0x95, 0x61, 0x40, 0x02, 0xA8, 0x61, 0x88, 0x22, 0x88, 0x24, 0x88, 0x25, 0x88,
    0x26, 0x88, 0x64, 0x22, 0x75, 0x40, 0x00, 0x75, 0x41, 0xF0, 0x22, 0xC2, 0xAF, 0x75, 0xC8, 0x20,
    0xA8, 0xCC, 0xA9, 0xCD, 0xAA, 0x3A, 0x30, 0xCF, 0x01, 0x0A, 0x75, 0xC8, 0x24, 0xD2, 0xAF, 0xC3,
    0xEA, 0x13, 0xFA, 0xE9, 0x13, 0xF9, 0xE8, 0x13, 0xF8, 0xAB, 0x3B, 0xAC, 0x3C, 0x88, 0x3B, 0x89,
    0x3C, 0xC3, 0xE8, 0x9B, 0xF8, 0xE9, 0x9C, 0x20, 0x69, 0x0A, 0x54, 0x7F, 0xF9, 0x30, 0x67, 0x02,
    0x41, 0xF5, 0x41, 0x52, 0xAD, 0x3D, 0x8A, 0x3D, 0xF9, 0xEA, 0x9D, 0x54, 0x7F, 0xFA, 0x60, 0x04,
    0x78, 0xFF, 0x79, 0xFF, 0xAE, 0x3E, 0xAF, 0x3F, 0x8B, 0x3E, 0x8C, 0x3F, 0xA8, 0x3B, 0xA9, 0x3C,
    0xC3, 0xE8, 0x9E, 0xF8, 0xE9, 0x9F, 0xF9, 0xC3, 0xE5, 0x41, 0x13, 0xFB, 0xE5, 0x40, 0x13, 0xFA,
    0xE8, 0x2A, 0xF5, 0x40, 0xE9, 0x3B, 0xF5, 0x41, 0x50, 0x06, 0x75, 0x40, 0xFF, 0x75, 0x41, 0xFF,
    0x41, 0x9C, 0xAA, 0x40, 0xAB, 0x41, 0xAC, 0x40, 0xAD, 0x41, 0x7E, 0x04, 0x7F, 0x02, 0xC3, 0xEB,
    0x94, 0x04, 0x40, 0x02, 0x1E, 0x1F, 0xC3, 0xEB, 0x94, 0x08, 0x40, 0x02, 0x1E, 0x1F, 0xC3, 0xED,
    0x13, 0xFD, 0xEC, 0x13, 0xFC, 0xDE, 0xF7, 0xC3, 0xEA, 0x9C, 0xFA, 0xEB, 0x9D, 0xFB, 0xEF, 0x60,
    0x09, 0xC3, 0xE9, 0x13, 0xF9, 0xE8, 0x13, 0xF8, 0xDF, 0xF7, 0xEA, 0x28, 0xFA, 0xEB, 0x39, 0xFB,
    0x8A, 0x40, 0x8B, 0x41, 0x50, 0x06, 0x7B, 0xFF, 0x8B, 0x40, 0x8B, 0x41, 0xC3, 0xEB, 0x94, 0x02,
    0x50, 0x02, 0xD2, 0x67, 0x30, 0x69, 0x04, 0x7F, 0x03, 0x41, 0xC7, 0x78, 0x92, 0xE6, 0xFF, 0xC3,
    0xE5, 0x37, 0x94, 0x82, 0x40, 0x11, 0x0F, 0xC3, 0xE5, 0x37, 0x94, 0xA0, 0x40, 0x01, 0x0F, 0xC3,
    0xEF, 0x94, 0x06, 0x40, 0x02, 0x7F, 0x05, 0x7E, 0x02, 0xE5, 0x41, 0xC4, 0x54, 0x0F, 0xF9, 0xE5,
    0x41, 0xC4, 0x54, 0xF0, 0xF8, 0xE5, 0x40, 0xC4, 0x54, 0x0F, 0x28, 0xF8, 0xC3, 0xE8, 0x9E, 0xFA,
    0xE9, 0x94, 0x00, 0xFB, 0x40, 0x09, 0xC3, 0xEA, 0x94, 0x02, 0xEB, 0x94, 0x00, 0x50, 0x04, 0x7A,
    0x02, 0xE4, 0xFB, 0x61, 0x41, 0xAA, 0x40, 0xAB, 0x41, 0xEB, 0xC4, 0xFE, 0xEA, 0xC4, 0x54, 0x0F,
    0x4E, 0xFC, 0xC3, 0xEA, 0x9C, 0xFA, 0xEB, 0x94, 0x00, 0xFB, 0xC3, 0xE8, 0x13, 0xC3, 0x13, 0xF8,
    0xEA, 0x28, 0xFA, 0xEB, 0x34, 0x00, 0xFB, 0x8A, 0x40, 0x8B, 0x41, 0xC3, 0xEB, 0x94, 0x02, 0x40,
    0x02, 0xC2, 0x67, 0x78, 0x02, 0xEB, 0xC4, 0xFE, 0x7B, 0x00, 0xEA, 0xC4, 0x54, 0x0F, 0x4E, 0xFA,
    0xC3, 0xEA, 0x98, 0xFA, 0x40, 0x05, 0xC3, 0x94, 0x02, 0x50, 0x02, 0x7A, 0x02, 0x78, 0x92, 0xE6,
    0xFF, 0x30, 0x60, 0x02, 0x61, 0x41, 0x85, 0x51, 0x92, 0x85, 0x52, 0x93, 0xD2, 0x60, 0x43, 0xE6,
    0x80, 0xC3, 0xE4, 0x9A, 0xF8, 0xE4, 0x9B, 0xF9, 0xC3, 0xE8, 0x33, 0xF8, 0xE9, 0x33, 0xF9, 0x30,
    0x67, 0x02, 0x61, 0xD4, 0xE8, 0xFA, 0xE9, 0xFB, 0xD3, 0xE9, 0x13, 0xFD, 0xE8, 0x13, 0xFC, 0x88,
    0x51, 0x89, 0x52, 0xC3, 0xEF, 0x94, 0x03, 0x60, 0x38, 0xEF, 0x20, 0xE0, 0x0D, 0xE8, 0x2C, 0xF8,
    0xE9, 0x3D, 0xF9, 0xEC, 0xFA, 0xED, 0xFB, 0x02, 0x0B, 0x9D, 0xE8, 0x28, 0xF8, 0xE9, 0x39, 0xF9,
    0xC3, 0xE8, 0x24, 0x02, 0xF8, 0xE9, 0x34, 0x00, 0xF9, 0x7A, 0xFE, 0x7B, 0xFF, 0xC3, 0xEF, 0x94,
    0x03, 0x40, 0x0E, 0x8A, 0x53, 0x8B, 0x54, 0x88, 0x4D, 0x89, 0x4E, 0x8C, 0x4F, 0x8D, 0x50, 0x81,
    0x08, 0x88, 0x53, 0x89, 0x54, 0x8A, 0x4D, 0x8B, 0x4E, 0x8C, 0x4F, 0x8D, 0x50, 0x30, 0x69, 0x12,
    0x75, 0x53, 0xF0, 0x75, 0x54, 0xFF, 0x75, 0x4F, 0xF0, 0x75, 0x50, 0xFF, 0x75, 0x51, 0xF0, 0x75,
    0x52, 0xFF, 0x81, 0x08, 0xE8, 0xFA, 0xD3, 0xE8, 0x13, 0xFC, 0x88, 0x51, 0xC3, 0xEF, 0x94, 0x03,
    0x60, 0x20, 0xEF, 0x20, 0xE0, 0x07, 0xE8, 0x2C, 0xF8, 0xEC, 0xFA, 0x61, 0xF4, 0xE8, 0x28, 0x24,
    0x02, 0xF8, 0x7A, 0xFE, 0xC3, 0xEF, 0x94, 0x03, 0x40, 0x08, 0x8A, 0x53, 0x88, 0x4D, 0x8C, 0x4F,
    0x81, 0x08, 0x88, 0x53, 0x8A, 0x4D, 0x8C, 0x4F, 0xE5, 0x68, 0xC3, 0x33, 0x50, 0x02, 0x64, 0x6B,
    0xF5, 0x68, 0x30, 0x60, 0x02, 0x81, 0x12, 0x75, 0x34, 0x02, 0xD2, 0x60, 0x43, 0xE6, 0x80, 0xE5,
    0x2D, 0x54, 0x06, 0x60, 0x2D, 0xA8, 0x40, 0xA9, 0x41, 0xC3, 0xE9, 0x13, 0xF9, 0xE8, 0x13, 0xF8,
    0x30, 0x69, 0x04, 0xE9, 0x24, 0x40, 0xF9, 0xC2, 0xAF, 0x53, 0xE6, 0x7F, 0x75, 0x91, 0x00, 0xC3,
    0xE4, 0x98, 0xF5, 0x94, 0xE4, 0x99, 0xF5, 0x95, 0x75, 0x91, 0x04, 0xD2, 0x60, 0x43, 0xE6, 0x80,
    0xD2, 0xAF, 0x22, 0xD2, 0x65, 0x75, 0x43, 0x00, 0x75, 0x20, 0x00, 0x30, 0x6C, 0x03, 0x75, 0x20,
    0x40, 0x02, 0x0C, 0x72, 0xD2, 0x65, 0x75, 0x43, 0x00, 0x75, 0x20, 0x40, 0x30, 0x6C, 0x03, 0x75,
    0x20, 0x00, 0x78, 0x01, 0x79, 0x01, 0x20, 0x67, 0x20, 0xE5, 0x2D, 0x54, 0x06, 0x60, 0x02, 0xC2,
    0x65, 0x79, 0x14, 0xE5, 0x41, 0xC3, 0x13, 0x70, 0x01, 0x04, 0xF8, 0xC3, 0x94, 0x14, 0x40, 0x02,
    0x78, 0x14, 0x30, 0x69, 0x04, 0x78, 0x1B, 0x79, 0x1B, 0xC3, 0xE8, 0x33, 0xF8, 0xC3, 0xE9, 0x33,
    0xF9, 0x20, 0x60, 0x10, 0xE5, 0x43, 0x60, 0x0C, 0x30, 0x69, 0x03, 0xD5, 0x34, 0x04, 0xD2, 0x6D,
    0xA1, 0x2A, 0x91, 0x1A, 0x05, 0x43, 0xE5, 0x9B, 0xF4, 0x54, 0x40, 0xB5, 0x20, 0x02, 0xA1, 0x14,
    0x30, 0x69, 0x09, 0x08, 0xC3, 0xE8, 0x99, 0x40, 0x01, 0x18, 0x81, 0xA1, 0x20, 0x65, 0x0A, 0x08,
    0xC3, 0xE8, 0x99, 0x40, 0x02, 0x81, 0x72, 0x81, 0xA1, 0xC2, 0x65, 0xC2, 0xAF, 0x53, 0xE6, 0x7F,
    0x75, 0x91, 0x00, 0x30, 0x67, 0x12, 0x75, 0x94, 0x00, 0x75, 0x95, 0xFC, 0x75, 0x91, 0x04, 0xD2,
    0x60, 0x43, 0xE6, 0x80, 0xD2, 0xAF, 0x81, 0x72, 0xAE, 0x40, 0xAF, 0x41, 0xC3, 0xEE, 0x33, 0xFE,
    0xEF, 0x33, 0xFF, 0x50, 0x04, 0x7E, 0xFF, 0x7F, 0xFF, 0xC3, 0xE4, 0x9E, 0xF5, 0x94, 0xE4, 0x9F,
    0xF5, 0x95, 0x81, 0xEC, 0xC3, 0xE5, 0x33, 0x94, 0x01, 0x50, 0x02, 0x81, 0x72, 0x30, 0x65, 0x02,
    0x81, 0x72, 0xD8, 0x02, 0xA1, 0x28, 0x81, 0xA1, 0xC2, 0x6D, 0xC2, 0xAF, 0x53, 0xE6, 0x7F, 0x75,
    0x91, 0x00, 0x85, 0x53, 0x94, 0x85, 0x54, 0x95, 0x75, 0x91, 0x04, 0x85, 0x4D, 0x92, 0x85, 0x4E,
    0x93, 0xD2, 0x60, 0x43, 0xE6, 0x80, 0xD2, 0xAF, 0xE5, 0x2D, 0x54, 0x06, 0x60, 0x08, 0x20, 0x6A,
    0x02, 0x05, 0x33, 0x02, 0x0D, 0x66, 0x30, 0x6D, 0x0D, 0x20, 0x6C, 0x0A, 0x20, 0x65, 0x07, 0x15,
    0x81, 0x15, 0x81, 0x02, 0x16, 0x03, 0x22, 0x78, 0x00, 0x30, 0x64, 0x05, 0x30, 0x65, 0x02, 0x78,
    0x01, 0xE5, 0x37, 0x75, 0xF0, 0x07, 0xA4, 0xF9, 0xE5, 0xF0, 0x28, 0xF5, 0xF0, 0xE9, 0xA2, 0xF0,
    0x13, 0xA2, 0xF1, 0x13, 0xA2, 0xF2, 0x13, 0xF5, 0x37, 0xC3, 0x94, 0x78, 0x50, 0x03, 0x75, 0x37,
    0x78, 0xC3, 0xE5, 0x37, 0x95, 0x38, 0x40, 0x08, 0xD2, 0x66, 0xC2, 0x93, 0xC2, 0x97, 0xC2, 0x94,
    0x30, 0x60, 0x02, 0xA1, 0xA0, 0x85, 0x4F, 0x92, 0x85, 0x50, 0x93, 0xD2, 0x60, 0x43, 0xE6, 0x80,
    0x22, 0x20, 0x7D, 0x16, 0xC2, 0xAF, 0x75, 0x42, 0x02, 0xC2, 0x95, 0xD2, 0x92, 0x30, 0x62, 0x02,
    0xD2, 0x97, 0xD2, 0xAF, 0x75, 0x9F, 0x80, 0x02, 0x0F, 0xA0, 0xC2, 0xAF, 0x75, 0x42, 0x02, 0xC2,
    0x95, 0xD2, 0x96, 0x30, 0x62, 0x02, 0xD2, 0x93, 0xD2, 0xAF, 0x75, 0x9F, 0x80, 0x02, 0x0F, 0xA0,
    0x30, 0x72, 0x43, 0x20, 0x7D, 0x20, 0xC2, 0xAF, 0x75, 0x42, 0x03, 0x90, 0x01, 0x65, 0x75, 0x7B,
    0x20, 0xC2, 0x97, 0xC2, 0x96, 0x30, 0x62, 0x04, 0xD2, 0x94, 0xA1, 0xFE, 0xD2, 0x95, 0xD2, 0xAF,
    0x75, 0x9F, 0x89, 0x02, 0x0F, 0xA0, 0xC2, 0xAF, 0x75, 0x42, 0x03, 0x90, 0x01, 0x65, 0x75, 0x7B,
    0x20, 0xC2, 0x93, 0xC2, 0x92, 0x30, 0x62, 0x04, 0xD2, 0x94, 0xC1, 0x1E, 0xD2, 0x95, 0xD2, 0xAF,
    0x75, 0x9F, 0x81, 0x02, 0x0F, 0xA0, 0x20, 0x7D, 0x17, 0xC2, 0xAF, 0x75, 0x42, 0x03, 0x90, 0x01,
    0x40, 0xC2, 0x97, 0x30, 0x62, 0x02, 0xD2, 0x94, 0xD2, 0xAF, 0x75, 0x9F, 0x89, 0x02, 0x0F, 0xA0,
    0xC2, 0xAF, 0x75, 0x42, 0x03, 0x90, 0x01, 0x40, 0xC2, 0x93, 0x30, 0x62, 0x02, 0xD2, 0x94, 0xD2,
    0xAF, 0x75, 0x9F, 0x81, 0x02, 0x0F, 0xA0, 0x20, 0x7D, 0x16, 0xC2, 0xAF, 0x75, 0x42, 0x04, 0xC2,
    0x92, 0xD2, 0x96, 0x30, 0x62, 0x02, 0xD2, 0x94, 0xD2, 0xAF, 0x75, 0x9F, 0x81, 0x02, 0x0F, 0xA0,
    0xC2, 0xAF, 0x75, 0x42, 0x04, 0xC2, 0x96, 0xD2, 0x92, 0x30, 0x62, 0x02, 0xD2, 0x94, 0xD2, 0xAF,
    0x75, 0x9F, 0x89, 0x02, 0x0F, 0xA0, 0x30, 0x72, 0x43, 0x20, 0x7D, 0x20, 0xC2, 0xAF, 0x75, 0x42,
    0x05, 0x90, 0x01, 0x54, 0x75, 0x7B, 0x04, 0xC2, 0x94, 0xC2, 0x95, 0x30, 0x62, 0x04, 0xD2, 0x93,
    0xC1, 0xA4, 0xD2, 0x92, 0xD2, 0xAF, 0x75, 0x9F, 0x80, 0x02, 0x0F, 0xA0, 0xC2, 0xAF, 0x75, 0x42,
    0x05, 0x90, 0x01, 0x76, 0x75, 0x7B, 0x40, 0xC2, 0x94, 0xC2, 0x95, 0x30, 0x62, 0x04, 0xD2, 0x97,
    0xC1, 0xC4, 0xD2, 0x96, 0xD2, 0xAF, 0x75, 0x9F, 0x80, 0x02, 0x0F, 0xA0, 0x20, 0x7D, 0x17, 0xC2,
    0xAF, 0x75, 0x42, 0x05, 0x90, 0x01, 0x36, 0xC2, 0x94, 0x30, 0x62, 0x02, 0xD2, 0x93, 0xD2, 0xAF,
    0x75, 0x9F, 0x80, 0x02, 0x0F, 0xA0, 0xC2, 0xAF, 0x75, 0x42, 0x05, 0x90, 0x01, 0x4A, 0xC2, 0x94,
    0x30, 0x62, 0xEB, 0xD2, 0x97, 0xD2, 0xAF, 0x75, 0x9F, 0x80, 0x02, 0x0F, 0xA0, 0x20, 0x7D, 0x16,
    0xC2, 0xAF, 0x75, 0x42, 0x06, 0xC2, 0x96, 0xD2, 0x95, 0x30, 0x62, 0x02, 0xD2, 0x93, 0xD2, 0xAF,
    0x75, 0x9F, 0x89, 0x02, 0x0F, 0xA0, 0xC2, 0xAF, 0x75, 0x42, 0x06, 0xC2, 0x92, 0xD2, 0x95, 0x30,
    0x62, 0x02, 0xD2, 0x97, 0xD2, 0xAF, 0x75, 0x9F, 0x81, 0x02, 0x0F, 0xA0, 0x30, 0x72, 0x43, 0x20,
    0x7D, 0x20, 0xC2, 0xAF, 0x75, 0x42, 0x01, 0x90, 0x01, 0x76, 0x75, 0x7B, 0x40, 0xC2, 0x93, 0xC2,
    0x92, 0x30, 0x62, 0x04, 0xD2, 0x97, 0xE1, 0x4A, 0xD2, 0x96, 0xD2, 0xAF, 0x75, 0x9F, 0x81, 0x02,
    0x0F, 0xA0, 0xC2, 0xAF, 0x75, 0x42, 0x01, 0x90, 0x01, 0x54, 0x75, 0x7B, 0x04, 0xC2, 0x97, 0xC2,
    0x96, 0x30, 0x62, 0x04, 0xD2, 0x93, 0xE1, 0x6A, 0xD2, 0x92, 0xD2, 0xAF, 0x75, 0x9F, 0x89, 0x02,
    0x0F, 0xA0, 0x20, 0x7D, 0x17, 0xC2, 0xAF, 0x75, 0x42, 0x01, 0x90, 0x01, 0x4A, 0xC2, 0x93, 0x30,
    0x62, 0x02, 0xD2, 0x97, 0xD2, 0xAF, 0x75, 0x9F, 0x81, 0x02, 0x0F, 0xA0, 0xC2, 0xAF, 0x75, 0x42,
    0x01, 0x90, 0x01, 0x36, 0xC2, 0x97, 0x30, 0x62, 0x02, 0xD2, 0x93, 0xD2, 0xAF, 0x75, 0x9F, 0x89,
    0xC2, 0x66, 0x22, 0x90, 0x01, 0x34, 0x75, 0x7B, 0x00, 0xC2, 0x93, 0xC2, 0x97, 0xC2, 0x94, 0xC2,
    0x92, 0xC2, 0x96, 0xC2, 0x95, 0xC2, 0x62, 0x22, 0x78, 0x80, 0x76, 0x09, 0x08, 0x76, 0x09, 0x08,
    0x76, 0x04, 0x08, 0x76, 0xFF, 0x08, 0x76, 0x03, 0x08, 0x76, 0xFF, 0x08, 0x76, 0x09, 0x08, 0x76,
    0x03, 0x08, 0x76, 0x01, 0x08, 0x76, 0x01, 0x78, 0x8C, 0x76, 0x01, 0x08, 0x76, 0xFF, 0x08, 0x76,
    0xFF, 0x08, 0x76, 0xFF, 0x08, 0x76, 0xFF, 0x08, 0x76, 0xFF, 0x08, 0x76, 0x03, 0x08, 0x76, 0xFF,
    0x08, 0x76, 0xFF, 0x08, 0x76, 0xFF, 0x08, 0x76, 0x25, 0x08, 0x76, 0xD0, 0x08, 0x76, 0x28, 0x08,
    0x76, 0x50, 0x08, 0x76, 0x04, 0x08, 0x76, 0xFF, 0x08, 0x76, 0x02, 0x08, 0x76, 0x00, 0x08, 0x76,
    0x7A, 0x08, 0x76, 0xFF, 0x08, 0x76, 0x01, 0x08, 0x76, 0x01, 0x08, 0x76, 0x00, 0x08, 0x76, 0x03,
    0x08, 0x76, 0x00, 0x22, 0x78, 0x87, 0xE6, 0xFF, 0xC2, 0x72, 0xBF, 0x03, 0x02, 0xD2, 0x72, 0x78,
    0x88, 0xE6, 0xC3, 0x94, 0x03, 0x60, 0x08, 0xC2, 0x7D, 0xE6, 0x30, 0xE1, 0x02, 0xD2, 0x7D, 0xC2,
    0x7E, 0x78, 0x89, 0xE6, 0x30, 0xE1, 0x02, 0xD2, 0x7E, 0xC3, 0xEF, 0x94, 0x02, 0x60, 0x08, 0x75,
    0x8E, 0x01, 0xD2, 0x73, 0x02, 0x10, 0x5C, 0x75, 0x8E, 0x00, 0xC2, 0x73, 0x22, 0x78, 0x80, 0xE6,
    0x14, 0x90, 0x00, 0x80, 0x93, 0x78, 0xA5, 0xF6, 0x78, 0x81, 0xE6, 0x14, 0x90, 0x00, 0x80, 0x93,
    0x78, 0xA6, 0xF6, 0x78, 0x86, 0xE6, 0x14, 0x90, 0x00, 0x8D, 0x93, 0x78, 0xA7, 0xF6, 0x78, 0x86,
    0xE6, 0xF5, 0x39, 0xC3, 0x94, 0x02, 0x50, 0x03, 0x75, 0x39, 0x02, 0x78, 0x9C, 0xE6, 0x75, 0x38,
    0xFF, 0xB4, 0x02, 0x03, 0x75, 0x38, 0xA0, 0xB4, 0x03, 0x03, 0x75, 0x38, 0x82, 0x78, 0xA3, 0xE6,
    0x14, 0x90, 0x00, 0x9A, 0x93, 0xF5, 0x66, 0x12, 0x0F, 0xA3, 0x22, 0x22, 0x78, 0x96, 0xE6, 0xFA,
    0x78, 0x97, 0xE6, 0xFB, 0x78, 0x88, 0xE6, 0xB4, 0x03, 0x05, 0xC3, 0xEB, 0x94, 0x0E, 0xFB, 0x30,
    0x7F, 0x04, 0x7A, 0x00, 0x7B, 0xFF, 0xC3, 0xEB, 0x9A, 0xFC, 0xC3, 0x94, 0x82, 0x50, 0x02, 0x7C,
    0x82, 0x75, 0x72, 0x00, 0x05, 0x72, 0xEC, 0x85, 0x72, 0xF0, 0xA4, 0xC3, 0xE5, 0xF0, 0x94, 0x7D,
    0x40, 0xF2, 0x22, 0xD2, 0x7F, 0x11, 0xAC, 0x12, 0x06, 0x04, 0x7A, 0x00, 0x7B, 0x00, 0x7C, 0x10,
    0x12, 0x05, 0xFA, 0xE5, 0x5C, 0x2A, 0xFA, 0x74, 0x00, 0x3B, 0xFB, 0xDC, 0xF3, 0x7C, 0x04, 0xC3,
    0xEB, 0x13, 0xFB, 0xEA, 0x13, 0xFA, 0xDC, 0xF7, 0xFE, 0xC2, 0x7F, 0x11, 0xAC, 0x22, 0xE5, 0xEF,
    0x20, 0xE6, 0x03, 0x75, 0x20, 0x00, 0x05, 0x20, 0x90, 0x3F, 0xFF, 0xE5, 0x20, 0x14, 0x60, 0x06,
    0x90, 0x1F, 0xFF, 0x14, 0x60, 0x00, 0x93, 0x04, 0x60, 0x03, 0x75, 0xEF, 0x12, 0x53, 0xD9, 0xBF,
    0x75, 0x81, 0xC0, 0x43, 0xFF, 0x80, 0x12, 0x05, 0xF5, 0x75, 0xEF, 0x02, 0x43, 0xB2, 0x03, 0xE5,
    0xB3, 0x24, 0x02, 0x20, 0xE7, 0x0D, 0xF5, 0x21, 0xE5, 0xE3, 0x20, 0xE0, 0x06, 0x85, 0x21, 0xB3,
    0x43, 0xE3, 0x01, 0x12, 0x0F, 0xA3, 0x75, 0x80, 0xFF, 0x75, 0xA4, 0x00, 0x75, 0xF1, 0xF0, 0x75,
    0xD4, 0xDF, 0x75, 0x90, 0x02, 0x75, 0xA5, 0xFC, 0x75, 0xF2, 0xFD, 0x75, 0xD5, 0x02, 0x75, 0xA6,
    0x10, 0x75, 0xA0, 0xFF, 0x75, 0xF3, 0xF1, 0x75, 0xE2, 0x41, 0x12, 0x0F, 0xA3, 0xE4, 0xF8, 0xF6,
    0xD8, 0xFD, 0x75, 0x68, 0x01, 0x12, 0x0F, 0xB8, 0x12, 0x16, 0x5C, 0x78, 0x98, 0x86, 0x73, 0x75,
    0x30, 0x01, 0xC2, 0xAF, 0x12, 0x06, 0x0E, 0x12, 0x06, 0x1E, 0x12, 0x06, 0x04, 0x12, 0x06, 0x25,
    0x12, 0x06, 0x04, 0x12, 0x06, 0x2C, 0x12, 0x06, 0x04, 0xC2, 0xAF, 0x78, 0xFA, 0x79, 0xFA, 0x30,
    0x85, 0x07, 0xD9, 0xFB, 0xD8, 0xF7, 0x02, 0x1C, 0x00, 0x11, 0x24, 0x11, 0x5D, 0x11, 0xAB, 0x11,
    0xAC, 0x78, 0x98, 0x86, 0x73, 0x12, 0x0F, 0xA3, 0x75, 0xA9, 0x00, 0x75, 0xB6, 0x80, 0x75, 0x7A,
    0x00, 0x75, 0x88, 0x10, 0x75, 0x89, 0x02, 0x75, 0xC8, 0x24, 0x75, 0x91, 0x04, 0x75, 0xD8, 0x40,
    0x75, 0xA8, 0x22, 0x75, 0xB8, 0x02, 0x75, 0xE6, 0x90, 0x75, 0x9B, 0x80, 0x75, 0x9D, 0x00, 0x75,
    0xD1, 0x0E, 0x75, 0xBC, 0xC0, 0x75, 0xBB, 0x09, 0x75, 0xBA, 0x11, 0x75, 0xE8, 0x80, 0x12, 0x05,
    0xF5, 0xD2, 0xAF, 0x12, 0x09, 0x14, 0x75, 0x36, 0x00, 0x53, 0xDA, 0xCF, 0x20, 0x7E, 0x03, 0x43,
    0xDA, 0x20, 0x30, 0x7E, 0x03, 0x43, 0xDA, 0x10, 0x43, 0xDA, 0x01, 0xC2, 0xD8, 0xC2, 0x71, 0x12,
    0x06, 0x0E, 0xD2, 0x61, 0x7B, 0x03, 0x7A, 0x0C, 0xE5, 0x5B, 0x70, 0x07, 0x7A, 0x0C, 0xDB, 0x03,
    0x02, 0x11, 0xA9, 0x12, 0x06, 0x04, 0x20, 0x70, 0x03, 0x02, 0x11, 0xA9, 0xC2, 0x70, 0xE5, 0x5C,
    0xC3, 0x94, 0x02, 0x40, 0xE1, 0xE5, 0x2F, 0x54, 0x1F, 0x85, 0x5E, 0x5D, 0xF5, 0x5E, 0xB5, 0x5D,
    0xD5, 0xDA, 0xD5, 0xC2, 0x61, 0x53, 0xDA, 0xCF, 0x20, 0x7E, 0x03, 0x43, 0xDA, 0x20, 0x30, 0x7E,
    0x03, 0x43, 0xDA, 0x10, 0xC2, 0xD8, 0xC2, 0x71, 0x78, 0xA2, 0xE6, 0x70, 0x08, 0xD2, 0x74, 0xE5,
    0x2F, 0x54, 0xE0, 0xF5, 0x2F, 0xC2, 0x75, 0x75, 0x29, 0x00, 0x12, 0x06, 0x09, 0x30, 0x74, 0x09,
    0xC3, 0xE5, 0x29, 0x94, 0x0A, 0x40, 0x02, 0xD2, 0x75, 0x12, 0x05, 0xFA, 0x78, 0x02, 0x30, 0x74,
    0x02, 0x78, 0x00, 0xC3, 0xE5, 0x5C, 0x98, 0x40, 0xF0, 0xC2, 0xAF, 0x12, 0x06, 0x1E, 0x12, 0x06,
    0x1E, 0x12, 0x06, 0x1E, 0xD2, 0xAF, 0x12, 0x06, 0x0E, 0x75, 0x4C, 0x00, 0x78, 0x88, 0xE6, 0xB4,
    0x03, 0x02, 0x61, 0x88, 0x12, 0x05, 0xFA, 0x78, 0x8C, 0xE6, 0xC3, 0x94, 0x01, 0x50, 0x03, 0x02,
    0x13, 0x88, 0xE5, 0x30, 0xC3, 0x94, 0x01, 0x50, 0x03, 0x02, 0x13, 0x88, 0x20, 0x74, 0x35, 0xC3,
    0xE5, 0x5C, 0x94, 0xFF, 0x50, 0x03, 0x02, 0x13, 0x88, 0xC2, 0xAF, 0x12, 0x06, 0x33, 0xD2, 0xAF,
    0x12, 0x06, 0x09, 0xC3, 0xE5, 0x5C, 0x94, 0x01, 0x50, 0xEF, 0xC2, 0xAF, 0x12, 0x06, 0x1E, 0x12,
    0x05, 0xFF, 0x12, 0x06, 0x1E, 0xD2, 0xAF, 0x12, 0x06, 0x09, 0xC3, 0xE5, 0x5C, 0x94, 0xFF, 0x40,
    0xE9, 0x02, 0x18, 0x04, 0x7F, 0x02, 0xD2, 0x7F, 0x11, 0xAC, 0x12, 0x06, 0x09, 0xC2, 0xAF, 0xC2,
    0x7F, 0x11, 0xAC, 0xAE, 0x5C, 0xC3, 0xE5, 0x5C, 0x94, 0x7F, 0xD2, 0xAF, 0x40, 0x74, 0x12, 0x05,
    0xF5, 0xC2, 0xAF, 0x12, 0x06, 0x33, 0xD2, 0xAF, 0xDF, 0xDC, 0x11, 0xE3, 0xC3, 0xEE, 0x78, 0x97,
    0xF6, 0x12, 0x06, 0x0E, 0x12, 0x16, 0x9E, 0x12, 0x17, 0x87, 0x7F, 0x0A, 0xD2, 0x7F, 0x11, 0xAC,
    0x12, 0x06, 0x09, 0xC2, 0xAF, 0xC2, 0x7F, 0x11, 0xAC, 0xAE, 0x5C, 0xC3, 0xE5, 0x5C, 0x94, 0x7F,
    0xD2, 0xAF, 0x50, 0xE6, 0x12, 0x05, 0xF5, 0xC2, 0xAF, 0x12, 0x06, 0x1E, 0x12, 0x05, 0xFF, 0x12,
    0x06, 0x1E, 0xD2, 0xAF, 0xDF, 0xD6, 0x11, 0xE3, 0xEE, 0x24, 0x03, 0x78, 0x96, 0xF6, 0x12, 0x06,
    0x0E, 0x12, 0x16, 0x9E, 0x12, 0x17, 0xB6, 0x12, 0x06, 0x09, 0x11, 0xAC, 0xC3, 0xE5, 0x5C, 0x94,
    0xFF, 0x50, 0x02, 0x41, 0xCC, 0x02, 0x18, 0x04, 0xC3, 0xE5, 0x5C, 0x95, 0x4C, 0x40, 0x03, 0x85,
    0x5C, 0x4C, 0x12, 0x06, 0x09, 0x78, 0x01, 0x79, 0x88, 0xE7, 0xB4, 0x03, 0x02, 0x78, 0x05, 0xC3,
    0xE5, 0x5C, 0x98, 0x40, 0x02, 0x41, 0xAC, 0xC2, 0xAF, 0x12, 0x06, 0x33, 0x12, 0x06, 0x33, 0x12,
    0x06, 0x33, 0xD2, 0xAF, 0x12, 0x06, 0x0E, 0x75, 0x30, 0x00, 0xE4, 0xF5, 0x31, 0xF5, 0x32, 0x05,
    0x31, 0xE5, 0x31, 0xF4, 0x70, 0x3F, 0x05, 0x32, 0x78, 0x9A, 0xE6, 0x78, 0x19, 0x14, 0x60, 0x12,
    0x78, 0x32, 0x14, 0x60, 0x0D, 0x78, 0x7D, 0x14, 0x60, 0x08, 0x78, 0xFA, 0x14, 0x60, 0x03, 0x75,
    0x32, 0x00, 0xC3, 0xE5, 0x32, 0x98, 0x40, 0x1D, 0x12, 0x0F, 0xA3, 0x12, 0x05, 0xF5, 0x15, 0x32,
    0x75, 0x31, 0x00, 0x78, 0x99, 0x86, 0x73, 0xC2, 0xAF, 0x12, 0x06, 0x33, 0xD2, 0xAF, 0x78, 0x98,
    0x86, 0x73, 0x12, 0x06, 0x09, 0x12, 0x05, 0xFF, 0xE5, 0x2A, 0x70, 0x05, 0x30, 0x74, 0x02, 0x21,
    0xA9, 0x78, 0x01, 0x20, 0x74, 0x02, 0x78, 0x06, 0xC3, 0xE5, 0x5C, 0x98, 0x40, 0xA1, 0x78, 0x88,
    0xE6, 0xC3, 0x94, 0x03, 0x60, 0x03, 0x12, 0x06, 0x09, 0xE5, 0x2A, 0x70, 0x03, 0x02, 0x11, 0xA9,
    0xC2, 0xAF, 0x12, 0x0F, 0xA3, 0xE4, 0xF5, 0x22, 0xF5, 0x23, 0xF5, 0x24, 0xF5, 0x25, 0xF5, 0x26,
    0xF5, 0x67, 0xD2, 0xAF, 0x78, 0x85, 0xE6, 0xC3, 0x33, 0xF5, 0x65, 0xE4, 0xF5, 0x44, 0xF5, 0x45,
    0xF5, 0x46, 0xF5, 0x47, 0xF5, 0x48, 0xF5, 0x70, 0xF5, 0x2C, 0xF5, 0x2D, 0xF5, 0x37, 0x75, 0x70,
    0x08, 0x75, 0xBB, 0x10, 0x12, 0x05, 0xF5, 0x12, 0x09, 0x18, 0xE5, 0xE8, 0x20, 0xEC, 0xFB, 0xA8,
    0xBD, 0xA9, 0xBE, 0xE9, 0x70, 0x01, 0xF8, 0x88, 0x71, 0x12, 0x09, 0x1C, 0x75, 0x70, 0x08, 0x75,
    0xBB, 0x10, 0x11, 0x24, 0xC2, 0xAF, 0x75, 0x61, 0xFF, 0x12, 0x09, 0xB4, 0x85, 0x22, 0x61, 0x85,
    0x22, 0x62, 0x85, 0x22, 0x63, 0xD2, 0xAF, 0x75, 0x22, 0x01, 0x75, 0x24, 0x01, 0x75, 0x25, 0x01,
    0x75, 0x26, 0x01, 0x85, 0x60, 0x69, 0x75, 0x6A, 0x01, 0x75, 0xB6, 0x90, 0x75, 0xA9, 0x03, 0x75,
    0x7A, 0x01, 0x78, 0x88, 0xE6, 0xB4, 0x03, 0x07, 0xC2, 0x7D, 0x30, 0x76, 0x02, 0xD2, 0x7D, 0xD2,
    0x68, 0xD2, 0x69, 0x75, 0x33, 0x00, 0x12, 0x0E, 0xFD, 0x12, 0x0F, 0x2C, 0x12, 0x09, 0xD4, 0x12,
    0x09, 0xDB, 0x12, 0x09, 0xD4, 0x12, 0x09, 0xDB, 0x12, 0x09, 0xD4, 0x12, 0x0C, 0x64, 0x12, 0x07,
    0x0E, 0x12, 0x0D, 0x67, 0x12, 0x0D, 0xB1, 0x12, 0x09, 0xDB, 0x12, 0x0C, 0x53, 0x30, 0x6E, 0x03,
    0x12, 0x07, 0x6A, 0x20, 0x67, 0x03, 0x12, 0x08, 0xB7, 0x30, 0x67, 0x03, 0x12, 0x08, 0xFE, 0x12,
    0x0D, 0x67, 0x12, 0x0D, 0xE0, 0x12, 0x09, 0xDB, 0x12, 0x0C, 0x64, 0x30, 0x6E, 0x03, 0x12, 0x07,
    0x9F, 0x12, 0x0D, 0x67, 0x12, 0x0E, 0x57, 0x12, 0x09, 0xDB, 0x12, 0x0C, 0x53, 0x30, 0x6E, 0x03,
    0x12, 0x07, 0xFD, 0x12, 0x0D, 0x67, 0x12, 0x0E, 0x86, 0x12, 0x09, 0xDB, 0x12, 0x0C, 0x64, 0x30,
    0x6E, 0x03, 0x12, 0x08, 0x5B, 0x12, 0x0D, 0x67, 0x12, 0x0E, 0xFD, 0x12, 0x09, 0xDB, 0x12, 0x09,
    0x18, 0x12, 0x0C, 0x53, 0x12, 0x0D, 0x67, 0x12, 0x0F, 0x2C, 0x12, 0x09, 0x1C, 0x12, 0x09, 0xDB,
    0x30, 0x69, 0x32, 0x85, 0x64, 0x61, 0x85, 0x64, 0x62, 0x85, 0x60, 0x69, 0x75, 0x6A, 0x01, 0x79,
    0x18, 0x7A, 0x0C, 0xC3, 0xE5, 0x33, 0x99, 0x40, 0x0F, 0xC2, 0x69, 0xD2, 0x6A, 0x8A, 0x35, 0x85,
    0x64, 0x61, 0x85, 0x64, 0x63, 0x02, 0x15, 0x85, 0xC3, 0xE5, 0x5C, 0x94, 0x01, 0x40, 0x03, 0x02,
    0x14, 0xDB, 0x02, 0x16, 0x0B, 0x30, 0x6A, 0x1D, 0x20, 0x6C, 0x1A, 0xE5, 0x35, 0x14, 0x70, 0x06,
    0xC2, 0x6A, 0xD2, 0x6B, 0x81, 0xDB, 0xF5, 0x35, 0xC3, 0xE5, 0x5C, 0x94, 0x01, 0x40, 0x03, 0x02,
    0x14, 0xDB, 0x02, 0x16, 0x0B, 0x75, 0x36, 0x00, 0x78, 0xFA, 0x79, 0xA4, 0xE7, 0x60, 0x02, 0x78,
    0x03, 0xC3, 0xE5, 0x5F, 0x98, 0x50, 0x54, 0x30, 0x74, 0x04, 0xE5, 0x2A, 0x60, 0x4D, 0x78, 0x88,
    0xE6, 0xB4, 0x03, 0x17, 0x20, 0x7D, 0x05, 0x20, 0x76, 0x07, 0xA1, 0xDB, 0x30, 0x76, 0x02, 0xA1,
    0xDB, 0x20, 0x6C, 0x07, 0xD2, 0x6C, 0x85, 0x64, 0x61, 0xA1, 0x1A, 0x78, 0xF0, 0x30, 0x6C, 0x05,
    0x85, 0x64, 0x61, 0x78, 0x20, 0xC3, 0xE5, 0x41, 0x98, 0x50, 0x02, 0x81, 0xDB, 0x30, 0x6C, 0x1B,
    0xC2, 0x6C, 0xC2, 0x7D, 0x30, 0x76, 0x02, 0xD2, 0x7D, 0xD2, 0x6A, 0x75, 0x35, 0x12, 0x85, 0x64,
    0x61, 0x81, 0xDB, 0x05, 0x36, 0xE5, 0x5C, 0x60, 0x02, 0xC1, 0x0E, 0x75, 0x36, 0x00, 0xC2, 0xAF,
    0x12, 0x0F, 0xA3, 0x78, 0x87, 0xE6, 0xFE, 0x76, 0x02, 0x11, 0x24, 0x78, 0x87, 0xEE, 0xF6, 0xE4,
    0xF5, 0x22, 0xF5, 0x23, 0xF5, 0x24, 0xF5, 0x25, 0xF5, 0x26, 0xF5, 0x65, 0x75, 0x2C, 0x00, 0x75,
    0x2D, 0x00, 0x75, 0xA9, 0x00, 0x75, 0xB6, 0x80, 0x75, 0x7A, 0x00, 0xD2, 0xAF, 0x12, 0x06, 0x09,
    0x12, 0x0F, 0xA3, 0x78, 0xA4, 0xE6, 0x60, 0x06, 0xD2, 0x92, 0xD2, 0x95, 0xD2, 0x96, 0x30, 0x74,
    0x09, 0xC3, 0xE5, 0x36, 0x94, 0x04, 0x40, 0x02, 0x21, 0xA9, 0x61, 0xBA, 0x90, 0x1A, 0x0D, 0x78,
    0x20, 0x12, 0x16, 0xE0, 0xE5, 0x20, 0xB4, 0x55, 0x0C, 0xA3, 0x12, 0x16, 0xE0, 0xE5, 0x20, 0xB4,
    0xAA, 0x03, 0x02, 0x16, 0x7E, 0x12, 0x0F, 0xB8, 0x12, 0x16, 0x9E, 0x02, 0x16, 0x9D, 0x90, 0x1A,
    0x03, 0x78, 0x80, 0x7B, 0x0A, 0x12, 0x16, 0xE0, 0xA3, 0x08, 0xDB, 0xF9, 0x90, 0x1A, 0x0F, 0x78,
    0x8C, 0x7B, 0x19, 0x12, 0x16, 0xE0, 0xA3, 0x08, 0xDB, 0xF9, 0x90, 0x1A, 0x28, 0x22, 0xC2, 0xAF,
    0x12, 0x17, 0x1F, 0x12, 0x16, 0xF9, 0x90, 0x1A, 0x00, 0x74, 0x0E, 0x12, 0x16, 0xE5, 0xA3, 0x74,
    0x06, 0x12, 0x16, 0xE5, 0xA3, 0x74, 0x15, 0x12, 0x16, 0xE5, 0x90, 0x1A, 0x03, 0x78, 0x80, 0x7B,
    0x0A, 0x12, 0x16, 0xE4, 0xA3, 0x08, 0xDB, 0xF9, 0x90, 0x1A, 0x0F, 0x78, 0x8C, 0x7B, 0x19, 0x12,
    0x16, 0xE4, 0xA3, 0x08, 0xDB, 0xF9, 0x12, 0x17, 0x32, 0x12, 0x17, 0x10, 0x90, 0x1A, 0x28, 0x22,
    0xE4, 0x93, 0xF6, 0x22, 0xE6, 0x43, 0x8F, 0x01, 0x53, 0x8F, 0xFD, 0x75, 0xEF, 0x02, 0x75, 0xB7,
    0xA5, 0x75, 0xB7, 0xF1, 0xF0, 0x53, 0x8F, 0xFE, 0x22, 0x43, 0x8F, 0x02, 0x43, 0x8F, 0x01, 0x75,
    0xEF, 0x02, 0x75, 0xB7, 0xA5, 0x75, 0xB7, 0xF1, 0x90, 0x1A, 0x0D, 0xF0, 0x53, 0x8F, 0xFC, 0x22,
    0x90, 0x1A, 0x0D, 0x74, 0x55, 0xD1, 0xE5, 0x90, 0x1A, 0x0E, 0x74, 0xAA, 0xD1, 0xE5, 0x22, 0x7A,
    0x30, 0x79, 0xD0, 0x78, 0x20, 0x90, 0x1A, 0x40, 0xD1, 0xE0, 0xE5, 0x20, 0xF7, 0x09, 0xA3, 0xDA,
    0xF7, 0x22, 0x7A, 0x30, 0x79, 0xD0, 0x90, 0x1A, 0x40, 0xE7, 0xD1, 0xE5, 0x09, 0xA3, 0xDA, 0xF9,
    0x22, 0xAB, 0x74, 0xA8, 0x75, 0xBB, 0x01, 0x02, 0x79, 0x80, 0xBB, 0x02, 0x02, 0x79, 0x81, 0xBB,
    0x03, 0x02, 0x79, 0x82, 0xBB, 0x04, 0x02, 0x79, 0x84, 0xBB, 0x05, 0x02, 0x79, 0x86, 0xBB, 0x06,
    0x02, 0x79, 0x92, 0xBB, 0x07, 0x02, 0x79, 0x87, 0xBB, 0x08, 0x02, 0x79, 0xA3, 0xBB, 0x09, 0x02,
    0x79, 0x9C, 0xBB, 0x0A, 0x02, 0x79, 0x88, 0xBB, 0x0B, 0x02, 0x79, 0x89, 0xE8, 0xF7, 0x22, 0x7C,
    0x05, 0x12, 0x06, 0x0E, 0xDC, 0xFB, 0x22, 0xC2, 0xAF, 0x12, 0x06, 0x1E, 0x12, 0x06, 0x25, 0x12,
    0x06, 0x2C, 0x12, 0x06, 0x33, 0x12, 0x05, 0xFF, 0x12, 0x06, 0x1E, 0x12, 0x06, 0x25, 0x12, 0x06,
    0x2C, 0x12, 0x06, 0x33, 0x12, 0x05, 0xFF, 0x12, 0x06, 0x1E, 0x12, 0x06, 0x25, 0x12, 0x06, 0x2C,
    0x12, 0x06, 0x33, 0xD2, 0xAF, 0x22, 0xC2, 0xAF, 0x12, 0x06, 0x33, 0x12, 0x06, 0x2C, 0x12, 0x06,
    0x25, 0x12, 0x06, 0x1E, 0x12, 0x05, 0xFF, 0x12, 0x06, 0x33, 0x12, 0x06, 0x2C, 0x12, 0x06, 0x25,
    0x12, 0x06, 0x1E, 0x12, 0x05, 0xFF, 0x12, 0x06, 0x33, 0x12, 0x06, 0x2C, 0x12, 0x06, 0x25, 0x12,
    0x06, 0x1E, 0xD2, 0xAF, 0x22, 0xAE, 0x74, 0xAF, 0x75, 0xC2, 0xAF, 0x12, 0x06, 0x1E, 0x12, 0x06,
    0x1E, 0x12, 0x06, 0x1E, 0x12, 0x05, 0xFF, 0xDE, 0xF2, 0x12, 0x06, 0x33, 0x12, 0x05, 0xFF, 0xDF,
    0xF8, 0xD2, 0xAF, 0x22, 0x12, 0x17, 0x87, 0x12, 0x17, 0x7F, 0x12, 0x17, 0x7F, 0x75, 0x74, 0x01,
    0x75, 0x75, 0x01, 0x75, 0x76, 0x00, 0x12, 0x17, 0xE5, 0x7C, 0x05, 0xAD, 0x5C, 0x12, 0x06, 0x0E,
    0xC3, 0xED, 0x95, 0x5C, 0x70, 0xF5, 0xC3, 0xE5, 0x5C, 0x94, 0x01, 0x40, 0x0A, 0xC3, 0xE5, 0x5C,
    0x94, 0xFF, 0x40, 0x37, 0x02, 0x18, 0x48, 0x12, 0x17, 0x41, 0x12, 0x16, 0x9E, 0x12, 0x17, 0x87,
    0xC2, 0xAF, 0x75, 0xEF, 0x12, 0x12, 0x17, 0x7F, 0xDC, 0xD1, 0x05, 0x76, 0xC3, 0xE5, 0x76, 0x94,
    0x03, 0x50, 0x02, 0x80, 0xC1, 0x12, 0x17, 0x7F, 0x05, 0x75, 0xE5, 0x74, 0x14, 0x90, 0x00, 0x9F,
    0x93, 0xF8, 0x08, 0xC3, 0xE5, 0x75, 0x98, 0x50, 0x02, 0x80, 0xA8, 0x12, 0x17, 0x7F, 0x12, 0x17,
    0x7F, 0x05, 0x74, 0xC3, 0xE5, 0x74, 0x94, 0x0C, 0x50, 0x02, 0x80, 0x94, 0x12, 0x0F, 0xB8, 0x12,
    0x16, 0x9E, 0xC2, 0xAF, 0x75, 0xEF, 0x12, 0x12, 0x17, 0x7F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x02, 0x11, 0x0E,
    0x0E, 0x06, 0x15, 0x09, 0x09, 0x04, 0xFF, 0x03, 0xFF, 0x09, 0x03, 0x01, 0x01, 0x55, 0xAA, 0x01,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x03, 0xFF, 0xFF, 0xFF, 0x25, 0xD0, 0x28, 0x50, 0x04, 0xFF, 0x02,
    0x00, 0x7A, 0xFF, 0x01, 0x01, 0x00, 0x03, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0x23, 0x46, 0x56, 0x54, 0x4C, 0x69, 0x62, 0x65, 0x65, 0x33, 0x30, 0x41, 0x23, 0x20, 0x20, 0x20,
    0x23, 0x42, 0x4C, 0x48, 0x45, 0x4C, 0x49, 0x23, 0x46, 0x33, 0x39, 0x30, 0x23, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xC2, 0xAF, 0xC2, 0xD3, 0x53, 0xD9, 0xBF, 0x75, 0x81, 0xC0, 0x43, 0xB2, 0x03, 0x43, 0xFF, 0x80,
    0x75, 0x24, 0x38, 0x75, 0x25, 0x03, 0xB1, 0xC9, 0x75, 0xEF, 0x02, 0x75, 0xE2, 0x40, 0x43, 0xF1,
    0x20, 0x53, 0xA4, 0xDF, 0x75, 0xD4, 0xFF, 0xD2, 0x85, 0x7D, 0xFA, 0x78, 0x03, 0x79, 0xE6, 0x7C,
    0x10, 0x75, 0x22, 0x00, 0x75, 0x23, 0x00, 0x90, 0x1D, 0xD6, 0x7B, 0x06, 0x75, 0x24, 0x98, 0x75,
    0x25, 0x01, 0x30, 0x85, 0x02, 0x81, 0x42, 0x20, 0x85, 0x08, 0xD8, 0xFB, 0xD9, 0xF9, 0xDC, 0xF7,
    0x81, 0xB4, 0x30, 0x85, 0x08, 0xD8, 0xFB, 0xD9, 0xF9, 0xDC, 0xF7, 0x81, 0xB4, 0xB1, 0x89, 0xE4,
    0x93, 0xA3, 0xC3, 0x9A, 0x60, 0x04, 0xDD, 0xC3, 0x81, 0xB4, 0xDB, 0xDB, 0xB1, 0x7B, 0x60, 0x02,
    0x81, 0x2B, 0x7B, 0x08, 0xE4, 0x93, 0xFA, 0xA3, 0xB1, 0x5C, 0xDB, 0xF8, 0x7A, 0x30, 0xB1, 0x5C,
    0x75, 0x22, 0x00, 0x75, 0x23, 0x00, 0xB1, 0x7B, 0xEA, 0xFC, 0xEB, 0xFD, 0xC3, 0xED, 0x94, 0xFE,
    0x40, 0x0E, 0xB1, 0x7B, 0x8A, 0x27, 0x8B, 0x28, 0xED, 0x30, 0xE0, 0x04, 0x8A, 0x82, 0x8B, 0x83,
    0xB1, 0x7B, 0x7A, 0xC2, 0x70, 0xD8, 0xC3, 0xED, 0x94, 0xFE, 0x60, 0x13, 0x50, 0xCE, 0xBD, 0x00,
    0x25, 0xEC, 0x60, 0x09, 0x75, 0x20, 0x00, 0x75, 0x21, 0xFF, 0x02, 0x00, 0x00, 0x81, 0x00, 0xA8,
    0x27, 0xA9, 0x28, 0x08, 0x09, 0xD8, 0x04, 0xD9, 0x02, 0x81, 0xD3, 0xB1, 0x7F, 0xEA, 0x89, 0xAA,
    0xF2, 0x81, 0xC5, 0x0D, 0x81, 0xA0, 0xC3, 0xED, 0x94, 0x03, 0x50, 0x49, 0xED, 0xA2, 0xE0, 0x92,
    0x00, 0x7A, 0xC5, 0xC3, 0xE5, 0x82, 0x94, 0x00, 0xE5, 0x83, 0x94, 0x1C, 0x50, 0x90, 0x20, 0x00,
    0x10, 0x43, 0x8F, 0x02, 0x43, 0x8F, 0x01, 0x75, 0xB7, 0xA5, 0x75, 0xB7, 0xF1, 0xF0, 0x30, 0x00,
    0x1F, 0xA8, 0x27, 0xA9, 0x28, 0x08, 0x09, 0x43, 0x8F, 0x01, 0x53, 0x8F, 0xFD, 0xD8, 0x04, 0xD9,
    0x02, 0xA1, 0x20, 0x89, 0xAA, 0xE2, 0x75, 0xB7, 0xA5, 0x75, 0xB7, 0xF1, 0xF0, 0xA3, 0xA1, 0x0D,
    0x53, 0x8F, 0xFC, 0x81, 0x7C, 0x7A, 0xC1, 0xBD, 0x03, 0x0C, 0xE4, 0x93, 0xA3, 0xFA, 0xB1, 0x40,
    0xDC, 0xF8, 0xB1, 0x38, 0x81, 0x7C, 0x81, 0x7E, 0xAA, 0x22, 0xAB, 0x23, 0xB1, 0x5C, 0xEB, 0xFA,
    0xEA, 0x62, 0x22, 0x75, 0x26, 0x08, 0xC3, 0xE5, 0x23, 0x13, 0xF5, 0x23, 0xE5, 0x22, 0x13, 0xF5,
    0x22, 0x50, 0x06, 0x63, 0x23, 0xA0, 0x63, 0x22, 0x01, 0xD5, 0x26, 0xEA, 0xB1, 0xC9, 0xB1, 0xC9,
    0x75, 0x26, 0x0A, 0xEA, 0xF4, 0x20, 0x01, 0x02, 0xD2, 0x85, 0x30, 0x01, 0x02, 0xC2, 0x85, 0xB1,
    0xC9, 0xC3, 0x13, 0x40, 0x02, 0xC2, 0x01, 0xD5, 0x26, 0xEB, 0x22, 0xB1, 0x7F, 0xEA, 0xFB, 0x20,
    0x85, 0x02, 0xA1, 0x7F, 0x30, 0x85, 0x02, 0xA1, 0x84, 0x75, 0x26, 0x08, 0xAE, 0x24, 0xAF, 0x25,
    0xC3, 0xEF, 0x13, 0xFF, 0xEE, 0x13, 0xFE, 0xB1, 0xCD, 0xB1, 0xC9, 0xC3, 0xEA, 0x13, 0x30, 0x85,
    0x02, 0x44, 0x80, 0xFA, 0x30, 0xE7, 0x03, 0x63, 0x22, 0x01, 0xC3, 0xE5, 0x23, 0x13, 0xF5, 0x23,
    0xE5, 0x22, 0x13, 0xF5, 0x22, 0x50, 0x06, 0x63, 0x23, 0xA0, 0x63, 0x22, 0x01, 0xD5, 0x26, 0xD9,
    0xE5, 0x22, 0x65, 0x23, 0x65, 0x23, 0xF5, 0x22, 0x22, 0xAE, 0x24, 0xAF, 0x25, 0x0E, 0x0F, 0xDE,
    0xFE, 0xDF, 0xFC, 0xD2, 0x01, 0x22, 0x42, 0x4C, 0x48, 0x65, 0x6C, 0x69, 0x34, 0x37, 0x31, 0x63,
    0xF3, 0x90, 0x06, 0x01
    };

uint8_t BL_ConnectEx(uint8_32_u *pDeviceInfo)
{
    //only 2 bytes used $1E9307 -> 0x9307
    pDeviceInfo->dword = 0x163F390;
    return true;
}

uint8_t BL_SendCMDKeepAlive(void)
{
    return true;
}

void BL_SendCMDRunRestartBootloader(uint8_32_u *pDeviceInfo)
{
    pDeviceInfo->bytes[0] = 1;
    return;
}

static uint8_t BL_ReadA(uint8_t cmd, ioMem_t *pMem)
{
    UNUSED(cmd);
    uint16_t address = pMem->D_FLASH_ADDR_H << 8 | pMem->D_FLASH_ADDR_L;

    uint16_t bytes = pMem->D_NUM_BYTES;
    if (bytes == 0) bytes = 256;

    memcpy(pMem->D_PTR_I, &virtualFlash[address], bytes);
    return true;
}

static uint8_t BL_WriteA(uint8_t cmd, ioMem_t *pMem, uint32_t timeout)
{
    UNUSED(cmd);
    UNUSED(timeout);
    uint16_t address = pMem->D_FLASH_ADDR_H << 8 | pMem->D_FLASH_ADDR_L;

    uint16_t bytes = pMem->D_NUM_BYTES;
    if (bytes == 0) bytes = 256;
    memcpy(&virtualFlash[address], pMem->D_PTR_I, bytes);
    return true;
}

uint8_t BL_ReadFlash(uint8_t interface_mode, ioMem_t *pMem)
{
    UNUSED(interface_mode);
    return BL_ReadA(0, pMem);
}

uint8_t BL_ReadEEprom(ioMem_t *pMem)
{
    return BL_ReadA(0, pMem);
}

uint8_t BL_PageErase(ioMem_t *pMem)
{
    uint16_t address = pMem->D_FLASH_ADDR_H << 8 | pMem->D_FLASH_ADDR_L;
    if (address + VIRTUAL_PAGE_SIZE > VIRTUAL_FLASH_SIZE)
        return false;
    memset(&virtualFlash[address], 0xFF, VIRTUAL_PAGE_SIZE);
    return true;
}

uint8_t BL_WriteEEprom(ioMem_t *pMem)
{
    return BL_WriteA(0, pMem, 0);
}

uint8_t BL_WriteFlash(ioMem_t *pMem)
{
    return BL_WriteA(0, pMem, 0);
}

#endif
#endif
