/**************************************************************************//**
 * @file     rtl8195bhp_ssi.h
 * @brief    The header file of rtl8195bhp_ssi.c.
 * @version  1.00
 * @date     2017-08-22
 *
 * @note
 *
 ******************************************************************************
 *
 * Copyright(c) 2007 - 2017 Realtek Corporation. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ******************************************************************************/

#ifndef RTL8735B_SSI_H
#define RTL8735B_SSI_H

#include "rtl8735b_ssi_type.h"

/**

        \addtogroup hal_ssi SPI
        \ingroup 8735b_hal
        \brief The Serial Peripheral Interface bus(SPI) module of the HS platform.
        @{
*/

/* Parameters of DW_apb_ssi for RTL8195A */
#define SSI_TX_FIFO_DEPTH  64           //!< Define SPI transmit FIFO depth
#define SSI_RX_FIFO_DEPTH  64           //!< Define SPI receive FIFO depth
#define SSI_DUMMY_DATA     0x00         //!< Define dummy data for master mode, we need to push a dummy data to TX FIFO for read
#define MAX_SPI_NUM        0x4          //!< Define maximum SPI device numbers, 4 devices in our platform at most.
#define RTK_SLAVE0_INDEX   0x2          //!< Define high speed SPI slave index
#define RTK_SLAVE1_INDEX   0x3          //!< Define high speed SPI slave index

/**
  \brief  Enumeration to define transfer mode
*/
enum spi_ctrlr0_tmod_e {
	TmodTr      = 0,                    //!< Transmit and Receive mode
	TmodTo      = 1,                    //!< Transmit only mode
	TmodRo      = 2,                    //!< Receive only mode
	TmodEEPROMR = 3                     //!< EEPROM read, unused.
};
typedef uint8_t spi_ctrlr0_tmod_t;

/**
  \brief  Enumeration to define clock polarity of SPI mode
*/
enum spi_ctrlr0_scpol_e {
	ScpolInactiveIsLow  = 0,            //!< Clock is low when it is idle
	ScpolInactiveIsHigh = 1             //!< Clock is high when it is idle
};
typedef uint8_t spi_ctrlr0_scpol_t;

/**
  \brief  Enumeration to define clock phase of SPI mode
*/
enum spi_ctrlr0_scph_e {
	ScphToggleInMiddle = 0,             //!< Clock starts to toggle in the middle of data frame
	ScphToggleAtStart  = 1              //!< Clock starts to toggle at the start of data frame
};
typedef uint8_t spi_ctrlr0_scph_t;

/**
  \brief  Enumeration to define how chip select toggles between successive frames
*/
enum spi_scph0_toggle_e {
	ScphToggleFirstFrame = 0,           //!< Chip select line does not toggle between data frames
	ScphToggleEveryFrame = 1            //!< Chip select line toggles between every data frames
};
typedef uint8_t spi_scph0_toggle_t;

/**
  \brief  Enumeration to define data frame size
*/
enum spi_ctrlr0_dfs_e {
	DfsEightBits          = 7,          //!< The size of data frame is 8 bits
	DfsSixteenBits        = 15          //!< The size of data frame is 16 bits
};
typedef uint8_t spi_ctrlr0_dfs_t;

/**
  \brief  Enumeration to define the length of the control word for the Microwire frame format
*/
enum spi_ctrlr0_cfs_e {
	CfsOneBit        = 0,               //!< 1 bit control word
	CfsTwoBits       = 1,               //!< 2 bit control word
	CfsThreeBits     = 2,               //!< 3 bit control word
	CfsFourBits      = 3,               //!< 4 bit control word
	CfsFiveBits      = 4,               //!< 5 bit control word
	CfsSixBits       = 5,               //!< 6 bit control word
	CfsSevenBits     = 6,               //!< 7 bit control word
	CfsEightBits     = 7,               //!< 8 bit control word
	CfsNineBits      = 8,               //!< 9 bit control word
	CfsTenBits       = 9,               //!< 10 bit control word
	CfsElevenBits    = 10,              //!< 11 bit control word
	CfsTwelveBits    = 11,              //!< 12 bit control word
	CfsThirteen      = 12,              //!< 13 bit control word
	CfsFourteen      = 13,              //!< 14 bit control word
	CfsFifteen       = 14,              //!< 15 bit control word
	CfsSixteen       = 15               //!< 16 bit control word
};
typedef uint8_t spi_ctrlr0_cfs_t;

/**
  \brief  Enumeration to define slave output enable option
*/
enum spi_ctrlr0_slv_oe_e {
	SlvTxdEnable  = 0,                  //!< Slave output enable
	SlvTxdDisable = 1                   //!< Slave output disable
};
typedef uint8_t spi_ctrlr0_slv_oe_t;

/**
  \brief  Enumeration to define device role
*/
enum spi_role_select_e {
	SsiSlave =  0,                      //!< The SPI device is slave
	SsiMaster = 1                       //!< The SPI device is master
};
typedef uint8_t spi_role_select_t;

/**
  \brief  Enumeration to define SPI frame formats
*/
enum spi_frame_format_e {
	FrfMotorolaSpi  = 0,                //!< Motorola SPI
	FrfTiSsp        = 1,                //!< Texas Instruments SSP
	FrfNsMicrowire  = 2,                //!< National Semiconductor Microwire
	FrfRsvd         = 3                 //!< Reserved
};
typedef uint8_t spi_frame_format_t;

/**
  \brief  Enumeration to define DMA enable option
*/
enum spi_dmacr_enable_e {
	SsiNoDma       = 0,                 //!< SPI does not use DMA transfer
	SsiRxDmaEnable = 1,                 //!< SPI enable hanshaking with GDMA in receive mode
	SsiTxDmaEnable = 2,                 //!< SPI enable handshake with GDMA in transmit mode
	SsiTrDmaEnable = 3                  //!< SPI hanshakes with GDMA in transmit & receive mode
};
typedef uint8_t spi_dmacr_enable_t;

/**
  \brief  Enumeration to define microwire handshake enable
*/
enum spi_mwcr_handshake_e {
	MwHandShakeDisable = 0,             //!< SPI microwrie handshake is disabled
	MwHandShakeEnable  = 1              //!< SPI microwire handshake is enabled
};
typedef uint8_t spi_mwcr_handshake_t;

/**
  \brief  Enumeration to define microwire transfer direction
*/
enum spi_mwcr_direction_e {
	MwDirectionSlaveToMaster = 0,       //!< From slave to master
	MwDirectionMasterToSlave = 1        //!< From master to slave
};
typedef uint8_t spi_mwcr_direction_t;

/**
  \brief  Enumeration to define whether the transfer is seqential
*/
enum spi_mwcr_tmod_e {
	MwTmodNonSequential = 0,            //!< The transfer is non-sequential
	MwTmodSequential = 1                //!< The transfer is sequential
};
typedef uint8_t spi_mwcr_tmod_t;

/**
  \brief  The data struct of SPI pins
*/
typedef struct spi_pin_sel_s {
	u32 spi_cs_pin;
	u32 spi_clk_pin;
	u32 spi_mosi_pin;
	u32 spi_miso_pin;
} spi_pin_sel_t, *pspi_pin_sel_t;

/**
  \brief  The data struct of SPI adaptor
*/
typedef struct hal_ssi_adaptor_s {
	irq_config_t irq_handle;                                            //!< The IRQ handler
	SSI_TypeDef *spi_dev;                                                 //!< The base address of SPI device
	phal_gdma_adaptor_t ptx_gdma_adaptor;                               //!< The pointers of GDMA adaptor in transmit mode
	phal_gdma_adaptor_t prx_gdma_adaptor;                               //!< The pointers of GDMA adaptor in receive mode
	void (*rx_done_callback)(void *para);                               //!< The receive done callback function
	void *rx_done_cb_para;                                              //!< The parameter of the receive done callback function
	void *rx_data;                                                      //!< The destination address in memory in receive mode
	void (*tx_done_callback)(void *para);                               //!< The transmit done callback funtion
	void *tx_done_cb_para;                                              //!< The parameter of the transmit done callback function
	void *tx_data;                                                      //!< The source address in memory in transmit mode
	void (*tx_idle_callback)(void *para);                               //!< The transmit idle call back function
	void *tx_idle_cb_para;                                              //!< The parameter of the transmit idle callback function
	void (*dcache_invalidate_by_addr)(uint32_t *addr, int32_t dsize);   //!< callback function to do the D-cache invalidate
	void (*dcache_clean_by_addr)(uint32_t *addr, int32_t dsize);        //!< callback function to do the D-cache clean
	u32  dma_rx_data_level;                                             //!< The receive FIFO threshold to trigger DMA requests
	u32  dma_tx_data_level;                                             //!< The transmit FIFO threshold to tigger DMA requests
	u32  rx_length;                                                     //!< The total transfer length in receive mode
	u32  rx_threshold_level;                                            //!< The receive FIFO threshold to trigger receive FIFO full interrupt
	u32  tx_length;                                                     //!< The total transfer length in transmit mode
	u32  tx_threshold_level;                                            //!< The transmit FIFO threshold to tigger transmit FIFO empty interrupt
	u32  slave_select_enable;                                           //!< The index of slave in multi-slave mode, should always be 0
	u16  clock_divider;                                                 //!< The divider to determine SPI operting frequency for SPI master
	u16  data_frame_number;                                             //!< The total data length expected to receive in SPI master receive mode
	uint32_t *cache_invalidate_addr;                                    //!< The starting address to perform D-cache invalidate
	int32_t  cache_invalidate_len;                                      //!< The total length to perform D-cache invalidate
	spi_ctrlr0_cfs_t                control_frame_size;                 //!< The control frame size for Microwire
	spi_frame_format_t              data_frame_format;                  //!< The data frame format, SPI, SSI and Microwire are supported
	spi_ctrlr0_dfs_t                data_frame_size;                    //!< The data frame size, only 8 bits and 16 bits modes are supported
	spi_dmacr_enable_t              dma_control;                        //!< Enable DMA transfer by the handshaking interface
	spi_mwcr_direction_t            microwire_direction;                //!< The transfer direction for Microwire
	spi_mwcr_handshake_t            microwire_handshaking;              //!< Enable or disable transfer for Microwire
	spi_mwcr_tmod_t                 microwire_transfer_mode;            //!< Set sequential or non-sequential transfer for Microwire
	spi_role_select_t               role;                               //!< The SPI device role
	spi_ctrlr0_scph_t               sclk_phase;                         //!< The clock phase of SPI mode
	spi_ctrlr0_scpol_t              sclk_polarity;                      //!< The clock polarity of SPI mode
	spi_ctrlr0_slv_oe_t             slave_output_enable;                //!< The SPI slave output control
	spi_ctrlr0_tmod_t               transfer_mode;                      //!< The transfer mode for SPI, transmit only, receive only or transmit and receive mode
	spi_pin_sel_t                   spi_pin;                            //!< The pin selection of the SPI device
	u8   index;                                                         //!< The index of the SPI device
	u8   interrupt_mask;                                                //!< The interrupt mask of the SPI device
	u8   irq_en;                                                        //!< The flag to show if irq is enabled
	u8   unfix_read_en;                                                 //!< The flag to indicate unfix stream transfer is used
} hal_ssi_adaptor_t, *phal_ssi_adaptor_t;

/**
  \brief  The data struct of SPI group, this group is managing IRQ handles and the interrupt signal
*/
typedef struct _hal_ssi_group_s {
	phal_ssi_adaptor_t phal_ssi_adaptor[MAX_SPI_NUM];                   //!< The pointer stores pointers of each SPI device adaptor
	u8 spi_in_use;                                                      //!< A flag shows if any SPI device is used
	u8 irq_en;                                                          //!< A flag indicates if IRQ is enabled
} hal_ssi_group_t, *phal_ssi_group_t;

/**
  \brief  The data struct of SPI stub functions. RAM codes access SPI ROM codes via these stub functions
*/
typedef struct hal_ssi_func_stubs_s {
	phal_ssi_group_t phal_ssi_group;
	hal_status_t (*hal_ssi_clock_ctl)(phal_ssi_adaptor_t phal_ssi_adaptor, u8 ctl);
	hal_status_t (*hal_ssi_enable)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_disable)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_init_setting)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_deinit_setting)(phal_ssi_adaptor_t phal_ssi_adaptor);
	void (*hal_ssi_read_interrupt)(phal_ssi_adaptor_t phal_ssi_adaptor);
	void (*hal_ssi_read_unfix_size_interrupt)(phal_ssi_adaptor_t phal_ssi_adaptor);
	void (*hal_ssi_write_interrupt)(phal_ssi_adaptor_t phal_ssi_adaptor);
	void (*hal_ssi_irq_handle)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_interrupt_enable)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_interrupt_disable)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_interrupt_init_read)(phal_ssi_adaptor_t phal_ssi_adaptor, void *rx_data, u32 length);
	hal_status_t (*hal_ssi_interrupt_init_write)(phal_ssi_adaptor_t phal_ssi_adaptor, void *tx_data, u32 length);
	hal_status_t (*hal_ssi_set_sclk)(phal_ssi_adaptor_t phal_ssi_adaptor, u32 clk_rate);
	hal_status_t (*hal_ssi_set_format)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_set_microwire)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_set_sclk_polarity)(phal_ssi_adaptor_t phal_ssi_adaptor, u8 sclk_polarity);
	hal_status_t (*hal_ssi_set_sclk_phase)(phal_ssi_adaptor_t phal_ssi_adaptor, u8 sclk_phase);
	hal_status_t (*hal_ssi_set_data_frame_number)(phal_ssi_adaptor_t phal_ssi_adaptor, u32 length);
	hal_status_t (*hal_ssi_set_interrupt_mask)(phal_ssi_adaptor_t phal_ssi_adaptor, u8 imr_value);
	hal_status_t (*hal_ssi_set_device_role)(phal_ssi_adaptor_t phal_ssi_adaptor, u32 role);
	hal_status_t (*hal_ssi_set_txfifo_threshold)(phal_ssi_adaptor_t phal_ssi_adaptor, u32 txftlr_value);
	hal_status_t (*hal_ssi_set_rxfifo_threshold)(phal_ssi_adaptor_t phal_ssi_adaptor, u32 rxftlr_value);
	hal_status_t (*hal_ssi_set_slave_enable)(phal_ssi_adaptor_t phal_ssi_adaptor, u32 slave_index);
	u32(*hal_ssi_get_rxfifo_level)(phal_ssi_adaptor_t phal_ssi_adaptor);
	u32(*hal_ssi_get_txfifo_level)(phal_ssi_adaptor_t phal_ssi_adaptor);
	u32(*hal_ssi_get_status)(phal_ssi_adaptor_t phal_ssi_adaptor);
	u32(*hal_ssi_get_busy)(phal_ssi_adaptor_t phal_ssi_adaptor);
	u32(*hal_ssi_get_interrupt_mask)(phal_ssi_adaptor_t phal_ssi_adaptor);
	u32(*hal_ssi_get_interrupt_status)(phal_ssi_adaptor_t phal_ssi_adaptor);
	u32(*hal_ssi_get_raw_interrupt_status)(phal_ssi_adaptor_t phal_ssi_adaptor);
	u32(*hal_ssi_get_slave_enable_register)(phal_ssi_adaptor_t phal_ssi_adaptor);
	u32(*hal_ssi_writable)(phal_ssi_adaptor_t phal_ssi_adaptor);
	u32(*hal_ssi_readable)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_write)(phal_ssi_adaptor_t phal_ssi_adaptor, u32 value);
	u32(*hal_ssi_read)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_stop_recv)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_enter_critical)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_exit_critical)(phal_ssi_adaptor_t phal_ssi_adaptor);
	void (*hal_ssi_tx_gdma_irq_handle)(phal_ssi_adaptor_t phal_ssi_adaptor);
	void (*hal_ssi_rx_gdma_irq_handle)(phal_ssi_adaptor_t phal_ssi_adaptor);
	hal_status_t (*hal_ssi_tx_gdma_init_setting)(phal_ssi_adaptor_t phal_ssi_adaptor, phal_gdma_adaptor_t phal_gdma_adaptor);
	hal_status_t (*hal_ssi_rx_gdma_init_setting)(phal_ssi_adaptor_t phal_ssi_adaptor, phal_gdma_adaptor_t phal_gdma_adaptor);
	hal_status_t (*hal_ssi_dma_send_init)(phal_ssi_adaptor_t phal_ssi_adaptor, u8 *ptx_data, u32 length);
	hal_status_t (*hal_ssi_dma_recv_init)(phal_ssi_adaptor_t phal_ssi_adaptor, u8 *prx_data, u32 length);
} hal_ssi_func_stubs_t;

/// @cond DOXYGEN_ROM_HAL_API

/**

        \addtogroup hal_ssi_rom_func SPI HAL ROM APIs
        @{
*/


hal_status_t hal_rtl_ssi_clock_ctl(phal_ssi_adaptor_t phal_ssi_adaptor, u8 ctl);
hal_status_t hal_rtl_ssi_pinmux_disable(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_enable(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_disable(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_init_setting(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_deinit_setting(phal_ssi_adaptor_t phal_ssi_adaptor);
void hal_rtl_ssi_read_interrupt(phal_ssi_adaptor_t phal_ssi_adaptor);
void hal_rtl_ssi_read_unfix_size_interrupt(phal_ssi_adaptor_t phal_ssi_adaptor);
void hal_rtl_ssi_write_interrupt(phal_ssi_adaptor_t phal_ssi_adaptor);
void hal_rtl_ssi_irq_handle(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_interrupt_enable(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_interrupt_disable(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_interrupt_init_read(phal_ssi_adaptor_t phal_ssi_adaptor, void *rx_data, u32 length);
hal_status_t hal_rtl_ssi_interrupt_init_write(phal_ssi_adaptor_t phal_ssi_adaptor, void *tx_data, u32 length);
hal_status_t hal_rtl_ssi_set_sclk(phal_ssi_adaptor_t phal_ssi_adaptor, u32 clk_rate);
hal_status_t hal_rtl_ssi_set_format(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_set_microwire(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_set_sclk_polarity(phal_ssi_adaptor_t phal_ssi_adaptor, u8 sclk_polarity);
hal_status_t hal_rtl_ssi_set_sclk_phase(phal_ssi_adaptor_t phal_ssi_adaptor, u8 sclk_phase);
hal_status_t hal_rtl_ssi_set_data_frame_number(phal_ssi_adaptor_t phal_ssi_adaptor, u32 length);
hal_status_t hal_rtl_ssi_set_interrupt_mask(phal_ssi_adaptor_t phal_ssi_adaptor, u8 imr_value);
hal_status_t hal_rtl_ssi_set_device_role(phal_ssi_adaptor_t phal_ssi_adaptor, u32 role);
hal_status_t hal_rtl_ssi_set_txfifo_threshold(phal_ssi_adaptor_t phal_ssi_adaptor, u32 txftlr_value);
hal_status_t hal_rtl_ssi_set_rxfifo_threshold(phal_ssi_adaptor_t phal_ssi_adaptor, u32 rxftlr_value);
hal_status_t hal_rtl_ssi_set_slave_enable(phal_ssi_adaptor_t phal_ssi_adaptor, u32 slave_index);
u32 hal_rtl_ssi_get_rxfifo_level(phal_ssi_adaptor_t phal_ssi_adaptor);
u32 hal_rtl_ssi_get_txfifo_level(phal_ssi_adaptor_t phal_ssi_adaptor);
u32 hal_rtl_ssi_get_status(phal_ssi_adaptor_t phal_ssi_adaptor);
u32 hal_rtl_ssi_get_busy(phal_ssi_adaptor_t phal_ssi_adaptor);
u32 hal_rtl_ssi_get_interrupt_mask(phal_ssi_adaptor_t phal_ssi_adaptor);
u32 hal_rtl_ssi_get_interrupt_status(phal_ssi_adaptor_t phal_ssi_adaptor);
u32 hal_rtl_ssi_get_raw_interrupt_status(phal_ssi_adaptor_t phal_ssi_adaptor);
u32 hal_rtl_ssi_get_slave_enable_register(phal_ssi_adaptor_t phal_ssi_adaptor);
u32 hal_rtl_ssi_writable(phal_ssi_adaptor_t phal_ssi_adaptor);
u32 hal_rtl_ssi_readable(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_write(phal_ssi_adaptor_t phal_ssi_adaptor, u32 value);
u32 hal_rtl_ssi_read(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_is_timeout(u32 start_count, u32 timeout_count);
hal_status_t hal_rtl_ssi_stop_recv(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_enter_critical(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_exit_critical(phal_ssi_adaptor_t phal_ssi_adaptor);
void hal_rtl_ssi_tx_gdma_irq_handle(phal_ssi_adaptor_t phal_ssi_adaptor);
void hal_rtl_ssi_rx_gdma_irq_handle(phal_ssi_adaptor_t phal_ssi_adaptor);
hal_status_t hal_rtl_ssi_tx_gdma_init_setting(phal_ssi_adaptor_t phal_ssi_adaptor, phal_gdma_adaptor_t phal_gdma_adaptor);
hal_status_t hal_rtl_ssi_rx_gdma_init_setting(phal_ssi_adaptor_t phal_ssi_adaptor, phal_gdma_adaptor_t phal_gdma_adaptor);
hal_status_t hal_rtl_ssi_dma_send_init(phal_ssi_adaptor_t phal_ssi_adaptor, u8 *ptx_data, u32 length);
hal_status_t hal_rtl_ssi_dma_recv_init(phal_ssi_adaptor_t phal_ssi_adaptor, u8 *prx_data, u32 length);

/** *@} */ /* End of group hal_ssi_rom_func */

/// @endcond /* End of condition DOXYGEN_ROM_HAL_API */


/** *@} */ /* End of group hal_ssi */


#endif /* RTL8735B_SSI_H */
