/* ----------------------------------------------------------------------
 * Project:      CMSIS DSP Library
 * Title:        arm_dct4_init_q31.c
 * Description:  Initialization function of DCT-4 & IDCT4 Q31
 *
 * $Date:        27. January 2017
 * $Revision:    V.1.5.1
 *
 * Target Processor: Cortex-M cores
 * -------------------------------------------------------------------- */
/*
 * Copyright (C) 2010-2017 ARM Limited or its affiliates. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "arm_math.h"

/**
 * @ingroup DCT4_IDCT4
 */

/**
 * @addtogroup DCT4_IDCT4_Table DCT Type IV Tables
 * @{
 */

/*
* @brief  Weights Table
*/

/**
 * \par
 * Weights tables are generated using the formula : <pre>weights[n] = e^(-j*n*pi/(2*N))</pre>
 * \par
 * C command to generate the table
 * <pre>
 * for(i = 0; i< N; i++)
 * {
 *   weights[2*i]= cos(i*c);
 *   weights[(2*i)+1]= -sin(i * c);
 * } </pre>
 * \par
 * where <code>N</code> is the Number of weights to be calculated and <code>c</code> is <code>pi/(2*N)</code>
 * \par
 * Convert the output to q31 format by multiplying with 2^31 and saturated if required.
 * \par
 * In the tables below the real and imaginary values are placed alternatively, hence the
 * array length is <code>2*N</code>.
 */

static const q31_t WeightsQ31_128[256] = {
	(q31_t)0x7fffffff, (q31_t)0x00000000, (q31_t)0x7ffd885a, (q31_t)0xfe6de2e0, (q31_t)0x7ff62182, (q31_t)0xfcdbd541, (q31_t)0x7fe9cbc0, (q31_t)0xfb49e6a3,
	(q31_t)0x7fd8878e, (q31_t)0xf9b82684, (q31_t)0x7fc25596, (q31_t)0xf826a462, (q31_t)0x7fa736b4, (q31_t)0xf6956fb7, (q31_t)0x7f872bf3, (q31_t)0xf50497fb,
	(q31_t)0x7f62368f, (q31_t)0xf3742ca2, (q31_t)0x7f3857f6, (q31_t)0xf1e43d1c, (q31_t)0x7f0991c4, (q31_t)0xf054d8d5, (q31_t)0x7ed5e5c6, (q31_t)0xeec60f31,
	(q31_t)0x7e9d55fc, (q31_t)0xed37ef91, (q31_t)0x7e5fe493, (q31_t)0xebaa894f, (q31_t)0x7e1d93ea, (q31_t)0xea1debbb, (q31_t)0x7dd6668f, (q31_t)0xe8922622,
	(q31_t)0x7d8a5f40, (q31_t)0xe70747c4, (q31_t)0x7d3980ec, (q31_t)0xe57d5fda, (q31_t)0x7ce3ceb2, (q31_t)0xe3f47d96, (q31_t)0x7c894bde, (q31_t)0xe26cb01b,
	(q31_t)0x7c29fbee, (q31_t)0xe0e60685, (q31_t)0x7bc5e290, (q31_t)0xdf608fe4, (q31_t)0x7b5d039e, (q31_t)0xdddc5b3b, (q31_t)0x7aef6323, (q31_t)0xdc597781,
	(q31_t)0x7a7d055b, (q31_t)0xdad7f3a2, (q31_t)0x7a05eead, (q31_t)0xd957de7a, (q31_t)0x798a23b1, (q31_t)0xd7d946d8, (q31_t)0x7909a92d, (q31_t)0xd65c3b7b,
	(q31_t)0x78848414, (q31_t)0xd4e0cb15, (q31_t)0x77fab989, (q31_t)0xd3670446, (q31_t)0x776c4edb, (q31_t)0xd1eef59e, (q31_t)0x76d94989, (q31_t)0xd078ad9e,
	(q31_t)0x7641af3d, (q31_t)0xcf043ab3, (q31_t)0x75a585cf, (q31_t)0xcd91ab39, (q31_t)0x7504d345, (q31_t)0xcc210d79, (q31_t)0x745f9dd1, (q31_t)0xcab26fa9,
	(q31_t)0x73b5ebd1, (q31_t)0xc945dfec, (q31_t)0x7307c3d0, (q31_t)0xc7db6c50, (q31_t)0x72552c85, (q31_t)0xc67322ce, (q31_t)0x719e2cd2, (q31_t)0xc50d1149,
	(q31_t)0x70e2cbc6, (q31_t)0xc3a94590, (q31_t)0x7023109a, (q31_t)0xc247cd5a, (q31_t)0x6f5f02b2, (q31_t)0xc0e8b648, (q31_t)0x6e96a99d, (q31_t)0xbf8c0de3,
	(q31_t)0x6dca0d14, (q31_t)0xbe31e19b, (q31_t)0x6cf934fc, (q31_t)0xbcda3ecb, (q31_t)0x6c242960, (q31_t)0xbb8532b0, (q31_t)0x6b4af279, (q31_t)0xba32ca71,
	(q31_t)0x6a6d98a4, (q31_t)0xb8e31319, (q31_t)0x698c246c, (q31_t)0xb796199b, (q31_t)0x68a69e81, (q31_t)0xb64beacd, (q31_t)0x67bd0fbd, (q31_t)0xb5049368,
	(q31_t)0x66cf8120, (q31_t)0xb3c0200c, (q31_t)0x65ddfbd3, (q31_t)0xb27e9d3c, (q31_t)0x64e88926, (q31_t)0xb140175b, (q31_t)0x63ef3290, (q31_t)0xb0049ab3,
	(q31_t)0x62f201ac, (q31_t)0xaecc336c, (q31_t)0x61f1003f, (q31_t)0xad96ed92, (q31_t)0x60ec3830, (q31_t)0xac64d510, (q31_t)0x5fe3b38d, (q31_t)0xab35f5b5,
	(q31_t)0x5ed77c8a, (q31_t)0xaa0a5b2e, (q31_t)0x5dc79d7c, (q31_t)0xa8e21106, (q31_t)0x5cb420e0, (q31_t)0xa7bd22ac, (q31_t)0x5b9d1154, (q31_t)0xa69b9b68,
	(q31_t)0x5a82799a, (q31_t)0xa57d8666, (q31_t)0x59646498, (q31_t)0xa462eeac, (q31_t)0x5842dd54, (q31_t)0xa34bdf20, (q31_t)0x571deefa, (q31_t)0xa2386284,
	(q31_t)0x55f5a4d2, (q31_t)0xa1288376, (q31_t)0x54ca0a4b, (q31_t)0xa01c4c73, (q31_t)0x539b2af0, (q31_t)0x9f13c7d0, (q31_t)0x5269126e, (q31_t)0x9e0effc1,
	(q31_t)0x5133cc94, (q31_t)0x9d0dfe54, (q31_t)0x4ffb654d, (q31_t)0x9c10cd70, (q31_t)0x4ebfe8a5, (q31_t)0x9b1776da, (q31_t)0x4d8162c4, (q31_t)0x9a22042d,
	(q31_t)0x4c3fdff4, (q31_t)0x99307ee0, (q31_t)0x4afb6c98, (q31_t)0x9842f043, (q31_t)0x49b41533, (q31_t)0x9759617f, (q31_t)0x4869e665, (q31_t)0x9673db94,
	(q31_t)0x471cece7, (q31_t)0x9592675c, (q31_t)0x45cd358f, (q31_t)0x94b50d87, (q31_t)0x447acd50, (q31_t)0x93dbd6a0, (q31_t)0x4325c135, (q31_t)0x9306cb04,
	(q31_t)0x41ce1e65, (q31_t)0x9235f2ec, (q31_t)0x4073f21d, (q31_t)0x91695663, (q31_t)0x3f1749b8, (q31_t)0x90a0fd4e, (q31_t)0x3db832a6, (q31_t)0x8fdcef66,
	(q31_t)0x3c56ba70, (q31_t)0x8f1d343a, (q31_t)0x3af2eeb7, (q31_t)0x8e61d32e, (q31_t)0x398cdd32, (q31_t)0x8daad37b, (q31_t)0x382493b0, (q31_t)0x8cf83c30,
	(q31_t)0x36ba2014, (q31_t)0x8c4a142f, (q31_t)0x354d9057, (q31_t)0x8ba0622f, (q31_t)0x33def287, (q31_t)0x8afb2cbb, (q31_t)0x326e54c7, (q31_t)0x8a5a7a31,
	(q31_t)0x30fbc54d, (q31_t)0x89be50c3, (q31_t)0x2f875262, (q31_t)0x8926b677, (q31_t)0x2e110a62, (q31_t)0x8893b125, (q31_t)0x2c98fbba, (q31_t)0x88054677,
	(q31_t)0x2b1f34eb, (q31_t)0x877b7bec, (q31_t)0x29a3c485, (q31_t)0x86f656d3, (q31_t)0x2826b928, (q31_t)0x8675dc4f, (q31_t)0x26a82186, (q31_t)0x85fa1153,
	(q31_t)0x25280c5e, (q31_t)0x8582faa5, (q31_t)0x23a6887f, (q31_t)0x85109cdd, (q31_t)0x2223a4c5, (q31_t)0x84a2fc62, (q31_t)0x209f701c, (q31_t)0x843a1d70,
	(q31_t)0x1f19f97b, (q31_t)0x83d60412, (q31_t)0x1d934fe5, (q31_t)0x8376b422, (q31_t)0x1c0b826a, (q31_t)0x831c314e, (q31_t)0x1a82a026, (q31_t)0x82c67f14,
	(q31_t)0x18f8b83c, (q31_t)0x8275a0c0, (q31_t)0x176dd9de, (q31_t)0x82299971, (q31_t)0x15e21445, (q31_t)0x81e26c16, (q31_t)0x145576b1, (q31_t)0x81a01b6d,
	(q31_t)0x12c8106f, (q31_t)0x8162aa04, (q31_t)0x1139f0cf, (q31_t)0x812a1a3a, (q31_t)0xfab272b, (q31_t)0x80f66e3c, (q31_t)0xe1bc2e4, (q31_t)0x80c7a80a,
	(q31_t)0xc8bd35e, (q31_t)0x809dc971, (q31_t)0xafb6805, (q31_t)0x8078d40d, (q31_t)0x96a9049, (q31_t)0x8058c94c, (q31_t)0x7d95b9e, (q31_t)0x803daa6a,
	(q31_t)0x647d97c, (q31_t)0x80277872, (q31_t)0x4b6195d, (q31_t)0x80163440, (q31_t)0x3242abf, (q31_t)0x8009de7e, (q31_t)0x1921d20, (q31_t)0x800277a6
};

static const q31_t WeightsQ31_512[1024] = {
	(q31_t)0x7fffffff, (q31_t)0x00000000, (q31_t)0x7fffd886, (q31_t)0xff9b781d, (q31_t)0x7fff6216, (q31_t)0xff36f078, (q31_t)0x7ffe9cb2, (q31_t)0xfed2694f,
	(q31_t)0x7ffd885a, (q31_t)0xfe6de2e0, (q31_t)0x7ffc250f, (q31_t)0xfe095d69, (q31_t)0x7ffa72d1, (q31_t)0xfda4d929, (q31_t)0x7ff871a2, (q31_t)0xfd40565c,
	(q31_t)0x7ff62182, (q31_t)0xfcdbd541, (q31_t)0x7ff38274, (q31_t)0xfc775616, (q31_t)0x7ff09478, (q31_t)0xfc12d91a, (q31_t)0x7fed5791, (q31_t)0xfbae5e89,
	(q31_t)0x7fe9cbc0, (q31_t)0xfb49e6a3, (q31_t)0x7fe5f108, (q31_t)0xfae571a4, (q31_t)0x7fe1c76b, (q31_t)0xfa80ffcb, (q31_t)0x7fdd4eec, (q31_t)0xfa1c9157,
	(q31_t)0x7fd8878e, (q31_t)0xf9b82684, (q31_t)0x7fd37153, (q31_t)0xf953bf91, (q31_t)0x7fce0c3e, (q31_t)0xf8ef5cbb, (q31_t)0x7fc85854, (q31_t)0xf88afe42,
	(q31_t)0x7fc25596, (q31_t)0xf826a462, (q31_t)0x7fbc040a, (q31_t)0xf7c24f59, (q31_t)0x7fb563b3, (q31_t)0xf75dff66, (q31_t)0x7fae7495, (q31_t)0xf6f9b4c6,
	(q31_t)0x7fa736b4, (q31_t)0xf6956fb7, (q31_t)0x7f9faa15, (q31_t)0xf6313077, (q31_t)0x7f97cebd, (q31_t)0xf5ccf743, (q31_t)0x7f8fa4b0, (q31_t)0xf568c45b,
	(q31_t)0x7f872bf3, (q31_t)0xf50497fb, (q31_t)0x7f7e648c, (q31_t)0xf4a07261, (q31_t)0x7f754e80, (q31_t)0xf43c53cb, (q31_t)0x7f6be9d4, (q31_t)0xf3d83c77,
	(q31_t)0x7f62368f, (q31_t)0xf3742ca2, (q31_t)0x7f5834b7, (q31_t)0xf310248a, (q31_t)0x7f4de451, (q31_t)0xf2ac246e, (q31_t)0x7f434563, (q31_t)0xf2482c8a,
	(q31_t)0x7f3857f6, (q31_t)0xf1e43d1c, (q31_t)0x7f2d1c0e, (q31_t)0xf1805662, (q31_t)0x7f2191b4, (q31_t)0xf11c789a, (q31_t)0x7f15b8ee, (q31_t)0xf0b8a401,
	(q31_t)0x7f0991c4, (q31_t)0xf054d8d5, (q31_t)0x7efd1c3c, (q31_t)0xeff11753, (q31_t)0x7ef05860, (q31_t)0xef8d5fb8, (q31_t)0x7ee34636, (q31_t)0xef29b243,
	(q31_t)0x7ed5e5c6, (q31_t)0xeec60f31, (q31_t)0x7ec8371a, (q31_t)0xee6276bf, (q31_t)0x7eba3a39, (q31_t)0xedfee92b, (q31_t)0x7eabef2c, (q31_t)0xed9b66b2,
	(q31_t)0x7e9d55fc, (q31_t)0xed37ef91, (q31_t)0x7e8e6eb2, (q31_t)0xecd48407, (q31_t)0x7e7f3957, (q31_t)0xec71244f, (q31_t)0x7e6fb5f4, (q31_t)0xec0dd0a8,
	(q31_t)0x7e5fe493, (q31_t)0xebaa894f, (q31_t)0x7e4fc53e, (q31_t)0xeb474e81, (q31_t)0x7e3f57ff, (q31_t)0xeae4207a, (q31_t)0x7e2e9cdf, (q31_t)0xea80ff7a,
	(q31_t)0x7e1d93ea, (q31_t)0xea1debbb, (q31_t)0x7e0c3d29, (q31_t)0xe9bae57d, (q31_t)0x7dfa98a8, (q31_t)0xe957ecfb, (q31_t)0x7de8a670, (q31_t)0xe8f50273,
	(q31_t)0x7dd6668f, (q31_t)0xe8922622, (q31_t)0x7dc3d90d, (q31_t)0xe82f5844, (q31_t)0x7db0fdf8, (q31_t)0xe7cc9917, (q31_t)0x7d9dd55a, (q31_t)0xe769e8d8,
	(q31_t)0x7d8a5f40, (q31_t)0xe70747c4, (q31_t)0x7d769bb5, (q31_t)0xe6a4b616, (q31_t)0x7d628ac6, (q31_t)0xe642340d, (q31_t)0x7d4e2c7f, (q31_t)0xe5dfc1e5,
	(q31_t)0x7d3980ec, (q31_t)0xe57d5fda, (q31_t)0x7d24881b, (q31_t)0xe51b0e2a, (q31_t)0x7d0f4218, (q31_t)0xe4b8cd11, (q31_t)0x7cf9aef0, (q31_t)0xe4569ccb,
	(q31_t)0x7ce3ceb2, (q31_t)0xe3f47d96, (q31_t)0x7ccda169, (q31_t)0xe3926fad, (q31_t)0x7cb72724, (q31_t)0xe330734d, (q31_t)0x7ca05ff1, (q31_t)0xe2ce88b3,
	(q31_t)0x7c894bde, (q31_t)0xe26cb01b, (q31_t)0x7c71eaf9, (q31_t)0xe20ae9c1, (q31_t)0x7c5a3d50, (q31_t)0xe1a935e2, (q31_t)0x7c4242f2, (q31_t)0xe14794ba,
	(q31_t)0x7c29fbee, (q31_t)0xe0e60685, (q31_t)0x7c116853, (q31_t)0xe0848b7f, (q31_t)0x7bf88830, (q31_t)0xe02323e5, (q31_t)0x7bdf5b94, (q31_t)0xdfc1cff3,
	(q31_t)0x7bc5e290, (q31_t)0xdf608fe4, (q31_t)0x7bac1d31, (q31_t)0xdeff63f4, (q31_t)0x7b920b89, (q31_t)0xde9e4c60, (q31_t)0x7b77ada8, (q31_t)0xde3d4964,
	(q31_t)0x7b5d039e, (q31_t)0xdddc5b3b, (q31_t)0x7b420d7a, (q31_t)0xdd7b8220, (q31_t)0x7b26cb4f, (q31_t)0xdd1abe51, (q31_t)0x7b0b3d2c, (q31_t)0xdcba1008,
	(q31_t)0x7aef6323, (q31_t)0xdc597781, (q31_t)0x7ad33d45, (q31_t)0xdbf8f4f8, (q31_t)0x7ab6cba4, (q31_t)0xdb9888a8, (q31_t)0x7a9a0e50, (q31_t)0xdb3832cd,
	(q31_t)0x7a7d055b, (q31_t)0xdad7f3a2, (q31_t)0x7a5fb0d8, (q31_t)0xda77cb63, (q31_t)0x7a4210d8, (q31_t)0xda17ba4a, (q31_t)0x7a24256f, (q31_t)0xd9b7c094,
	(q31_t)0x7a05eead, (q31_t)0xd957de7a, (q31_t)0x79e76ca7, (q31_t)0xd8f81439, (q31_t)0x79c89f6e, (q31_t)0xd898620c, (q31_t)0x79a98715, (q31_t)0xd838c82d,
	(q31_t)0x798a23b1, (q31_t)0xd7d946d8, (q31_t)0x796a7554, (q31_t)0xd779de47, (q31_t)0x794a7c12, (q31_t)0xd71a8eb5, (q31_t)0x792a37fe, (q31_t)0xd6bb585e,
	(q31_t)0x7909a92d, (q31_t)0xd65c3b7b, (q31_t)0x78e8cfb2, (q31_t)0xd5fd3848, (q31_t)0x78c7aba2, (q31_t)0xd59e4eff, (q31_t)0x78a63d11, (q31_t)0xd53f7fda,
	(q31_t)0x78848414, (q31_t)0xd4e0cb15, (q31_t)0x786280bf, (q31_t)0xd48230e9, (q31_t)0x78403329, (q31_t)0xd423b191, (q31_t)0x781d9b65, (q31_t)0xd3c54d47,
	(q31_t)0x77fab989, (q31_t)0xd3670446, (q31_t)0x77d78daa, (q31_t)0xd308d6c7, (q31_t)0x77b417df, (q31_t)0xd2aac504, (q31_t)0x7790583e, (q31_t)0xd24ccf39,
	(q31_t)0x776c4edb, (q31_t)0xd1eef59e, (q31_t)0x7747fbce, (q31_t)0xd191386e, (q31_t)0x77235f2d, (q31_t)0xd13397e2, (q31_t)0x76fe790e, (q31_t)0xd0d61434,
	(q31_t)0x76d94989, (q31_t)0xd078ad9e, (q31_t)0x76b3d0b4, (q31_t)0xd01b6459, (q31_t)0x768e0ea6, (q31_t)0xcfbe389f, (q31_t)0x76680376, (q31_t)0xcf612aaa,
	(q31_t)0x7641af3d, (q31_t)0xcf043ab3, (q31_t)0x761b1211, (q31_t)0xcea768f2, (q31_t)0x75f42c0b, (q31_t)0xce4ab5a2, (q31_t)0x75ccfd42, (q31_t)0xcdee20fc,
	(q31_t)0x75a585cf, (q31_t)0xcd91ab39, (q31_t)0x757dc5ca, (q31_t)0xcd355491, (q31_t)0x7555bd4c, (q31_t)0xccd91d3d, (q31_t)0x752d6c6c, (q31_t)0xcc7d0578,
	(q31_t)0x7504d345, (q31_t)0xcc210d79, (q31_t)0x74dbf1ef, (q31_t)0xcbc53579, (q31_t)0x74b2c884, (q31_t)0xcb697db0, (q31_t)0x7489571c, (q31_t)0xcb0de658,
	(q31_t)0x745f9dd1, (q31_t)0xcab26fa9, (q31_t)0x74359cbd, (q31_t)0xca5719db, (q31_t)0x740b53fb, (q31_t)0xc9fbe527, (q31_t)0x73e0c3a3, (q31_t)0xc9a0d1c5,
	(q31_t)0x73b5ebd1, (q31_t)0xc945dfec, (q31_t)0x738acc9e, (q31_t)0xc8eb0fd6, (q31_t)0x735f6626, (q31_t)0xc89061ba, (q31_t)0x7333b883, (q31_t)0xc835d5d0,
	(q31_t)0x7307c3d0, (q31_t)0xc7db6c50, (q31_t)0x72db8828, (q31_t)0xc7812572, (q31_t)0x72af05a7, (q31_t)0xc727016d, (q31_t)0x72823c67, (q31_t)0xc6cd0079,
	(q31_t)0x72552c85, (q31_t)0xc67322ce, (q31_t)0x7227d61c, (q31_t)0xc61968a2, (q31_t)0x71fa3949, (q31_t)0xc5bfd22e, (q31_t)0x71cc5626, (q31_t)0xc5665fa9,
	(q31_t)0x719e2cd2, (q31_t)0xc50d1149, (q31_t)0x716fbd68, (q31_t)0xc4b3e746, (q31_t)0x71410805, (q31_t)0xc45ae1d7, (q31_t)0x71120cc5, (q31_t)0xc4020133,
	(q31_t)0x70e2cbc6, (q31_t)0xc3a94590, (q31_t)0x70b34525, (q31_t)0xc350af26, (q31_t)0x708378ff, (q31_t)0xc2f83e2a, (q31_t)0x70536771, (q31_t)0xc29ff2d4,
	(q31_t)0x7023109a, (q31_t)0xc247cd5a, (q31_t)0x6ff27497, (q31_t)0xc1efcdf3, (q31_t)0x6fc19385, (q31_t)0xc197f4d4, (q31_t)0x6f906d84, (q31_t)0xc1404233,
	(q31_t)0x6f5f02b2, (q31_t)0xc0e8b648, (q31_t)0x6f2d532c, (q31_t)0xc0915148, (q31_t)0x6efb5f12, (q31_t)0xc03a1368, (q31_t)0x6ec92683, (q31_t)0xbfe2fcdf,
	(q31_t)0x6e96a99d, (q31_t)0xbf8c0de3, (q31_t)0x6e63e87f, (q31_t)0xbf3546a8, (q31_t)0x6e30e34a, (q31_t)0xbedea765, (q31_t)0x6dfd9a1c, (q31_t)0xbe88304f,
	(q31_t)0x6dca0d14, (q31_t)0xbe31e19b, (q31_t)0x6d963c54, (q31_t)0xbddbbb7f, (q31_t)0x6d6227fa, (q31_t)0xbd85be30, (q31_t)0x6d2dd027, (q31_t)0xbd2fe9e2,
	(q31_t)0x6cf934fc, (q31_t)0xbcda3ecb, (q31_t)0x6cc45698, (q31_t)0xbc84bd1f, (q31_t)0x6c8f351c, (q31_t)0xbc2f6513, (q31_t)0x6c59d0a9, (q31_t)0xbbda36dd,
	(q31_t)0x6c242960, (q31_t)0xbb8532b0, (q31_t)0x6bee3f62, (q31_t)0xbb3058c0, (q31_t)0x6bb812d1, (q31_t)0xbadba943, (q31_t)0x6b81a3cd, (q31_t)0xba87246d,
	(q31_t)0x6b4af279, (q31_t)0xba32ca71, (q31_t)0x6b13fef5, (q31_t)0xb9de9b83, (q31_t)0x6adcc964, (q31_t)0xb98a97d8, (q31_t)0x6aa551e9, (q31_t)0xb936bfa4,
	(q31_t)0x6a6d98a4, (q31_t)0xb8e31319, (q31_t)0x6a359db9, (q31_t)0xb88f926d, (q31_t)0x69fd614a, (q31_t)0xb83c3dd1, (q31_t)0x69c4e37a, (q31_t)0xb7e9157a,
	(q31_t)0x698c246c, (q31_t)0xb796199b, (q31_t)0x69532442, (q31_t)0xb7434a67, (q31_t)0x6919e320, (q31_t)0xb6f0a812, (q31_t)0x68e06129, (q31_t)0xb69e32cd,
	(q31_t)0x68a69e81, (q31_t)0xb64beacd, (q31_t)0x686c9b4b, (q31_t)0xb5f9d043, (q31_t)0x683257ab, (q31_t)0xb5a7e362, (q31_t)0x67f7d3c5, (q31_t)0xb556245e,
	(q31_t)0x67bd0fbd, (q31_t)0xb5049368, (q31_t)0x67820bb7, (q31_t)0xb4b330b3, (q31_t)0x6746c7d8, (q31_t)0xb461fc70, (q31_t)0x670b4444, (q31_t)0xb410f6d3,
	(q31_t)0x66cf8120, (q31_t)0xb3c0200c, (q31_t)0x66937e91, (q31_t)0xb36f784f, (q31_t)0x66573cbb, (q31_t)0xb31effcc, (q31_t)0x661abbc5, (q31_t)0xb2ceb6b5,
	(q31_t)0x65ddfbd3, (q31_t)0xb27e9d3c, (q31_t)0x65a0fd0b, (q31_t)0xb22eb392, (q31_t)0x6563bf92, (q31_t)0xb1def9e9, (q31_t)0x6526438f, (q31_t)0xb18f7071,
	(q31_t)0x64e88926, (q31_t)0xb140175b, (q31_t)0x64aa907f, (q31_t)0xb0f0eeda, (q31_t)0x646c59bf, (q31_t)0xb0a1f71d, (q31_t)0x642de50d, (q31_t)0xb0533055,
	(q31_t)0x63ef3290, (q31_t)0xb0049ab3, (q31_t)0x63b0426d, (q31_t)0xafb63667, (q31_t)0x637114cc, (q31_t)0xaf6803a2, (q31_t)0x6331a9d4, (q31_t)0xaf1a0293,
	(q31_t)0x62f201ac, (q31_t)0xaecc336c, (q31_t)0x62b21c7b, (q31_t)0xae7e965b, (q31_t)0x6271fa69, (q31_t)0xae312b92, (q31_t)0x62319b9d, (q31_t)0xade3f33e,
	(q31_t)0x61f1003f, (q31_t)0xad96ed92, (q31_t)0x61b02876, (q31_t)0xad4a1aba, (q31_t)0x616f146c, (q31_t)0xacfd7ae8, (q31_t)0x612dc447, (q31_t)0xacb10e4b,
	(q31_t)0x60ec3830, (q31_t)0xac64d510, (q31_t)0x60aa7050, (q31_t)0xac18cf69, (q31_t)0x60686ccf, (q31_t)0xabccfd83, (q31_t)0x60262dd6, (q31_t)0xab815f8d,
	(q31_t)0x5fe3b38d, (q31_t)0xab35f5b5, (q31_t)0x5fa0fe1f, (q31_t)0xaaeac02c, (q31_t)0x5f5e0db3, (q31_t)0xaa9fbf1e, (q31_t)0x5f1ae274, (q31_t)0xaa54f2ba,
	(q31_t)0x5ed77c8a, (q31_t)0xaa0a5b2e, (q31_t)0x5e93dc1f, (q31_t)0xa9bff8a8, (q31_t)0x5e50015d, (q31_t)0xa975cb57, (q31_t)0x5e0bec6e, (q31_t)0xa92bd367,
	(q31_t)0x5dc79d7c, (q31_t)0xa8e21106, (q31_t)0x5d8314b1, (q31_t)0xa8988463, (q31_t)0x5d3e5237, (q31_t)0xa84f2daa, (q31_t)0x5cf95638, (q31_t)0xa8060d08,
	(q31_t)0x5cb420e0, (q31_t)0xa7bd22ac, (q31_t)0x5c6eb258, (q31_t)0xa7746ec0, (q31_t)0x5c290acc, (q31_t)0xa72bf174, (q31_t)0x5be32a67, (q31_t)0xa6e3aaf2,
	(q31_t)0x5b9d1154, (q31_t)0xa69b9b68, (q31_t)0x5b56bfbd, (q31_t)0xa653c303, (q31_t)0x5b1035cf, (q31_t)0xa60c21ee, (q31_t)0x5ac973b5, (q31_t)0xa5c4b855,
	(q31_t)0x5a82799a, (q31_t)0xa57d8666, (q31_t)0x5a3b47ab, (q31_t)0xa5368c4b, (q31_t)0x59f3de12, (q31_t)0xa4efca31, (q31_t)0x59ac3cfd, (q31_t)0xa4a94043,
	(q31_t)0x59646498, (q31_t)0xa462eeac, (q31_t)0x591c550e, (q31_t)0xa41cd599, (q31_t)0x58d40e8c, (q31_t)0xa3d6f534, (q31_t)0x588b9140, (q31_t)0xa3914da8,
	(q31_t)0x5842dd54, (q31_t)0xa34bdf20, (q31_t)0x57f9f2f8, (q31_t)0xa306a9c8, (q31_t)0x57b0d256, (q31_t)0xa2c1adc9, (q31_t)0x57677b9d, (q31_t)0xa27ceb4f,
	(q31_t)0x571deefa, (q31_t)0xa2386284, (q31_t)0x56d42c99, (q31_t)0xa1f41392, (q31_t)0x568a34a9, (q31_t)0xa1affea3, (q31_t)0x56400758, (q31_t)0xa16c23e1,
	(q31_t)0x55f5a4d2, (q31_t)0xa1288376, (q31_t)0x55ab0d46, (q31_t)0xa0e51d8c, (q31_t)0x556040e2, (q31_t)0xa0a1f24d, (q31_t)0x55153fd4, (q31_t)0xa05f01e1,
	(q31_t)0x54ca0a4b, (q31_t)0xa01c4c73, (q31_t)0x547ea073, (q31_t)0x9fd9d22a, (q31_t)0x5433027d, (q31_t)0x9f979331, (q31_t)0x53e73097, (q31_t)0x9f558fb0,
	(q31_t)0x539b2af0, (q31_t)0x9f13c7d0, (q31_t)0x534ef1b5, (q31_t)0x9ed23bb9, (q31_t)0x53028518, (q31_t)0x9e90eb94, (q31_t)0x52b5e546, (q31_t)0x9e4fd78a,
	(q31_t)0x5269126e, (q31_t)0x9e0effc1, (q31_t)0x521c0cc2, (q31_t)0x9dce6463, (q31_t)0x51ced46e, (q31_t)0x9d8e0597, (q31_t)0x518169a5, (q31_t)0x9d4de385,
	(q31_t)0x5133cc94, (q31_t)0x9d0dfe54, (q31_t)0x50e5fd6d, (q31_t)0x9cce562c, (q31_t)0x5097fc5e, (q31_t)0x9c8eeb34, (q31_t)0x5049c999, (q31_t)0x9c4fbd93,
	(q31_t)0x4ffb654d, (q31_t)0x9c10cd70, (q31_t)0x4faccfab, (q31_t)0x9bd21af3, (q31_t)0x4f5e08e3, (q31_t)0x9b93a641, (q31_t)0x4f0f1126, (q31_t)0x9b556f81,
	(q31_t)0x4ebfe8a5, (q31_t)0x9b1776da, (q31_t)0x4e708f8f, (q31_t)0x9ad9bc71, (q31_t)0x4e210617, (q31_t)0x9a9c406e, (q31_t)0x4dd14c6e, (q31_t)0x9a5f02f5,
	(q31_t)0x4d8162c4, (q31_t)0x9a22042d, (q31_t)0x4d31494b, (q31_t)0x99e5443b, (q31_t)0x4ce10034, (q31_t)0x99a8c345, (q31_t)0x4c9087b1, (q31_t)0x996c816f,
	(q31_t)0x4c3fdff4, (q31_t)0x99307ee0, (q31_t)0x4bef092d, (q31_t)0x98f4bbbc, (q31_t)0x4b9e0390, (q31_t)0x98b93828, (q31_t)0x4b4ccf4d, (q31_t)0x987df449,
	(q31_t)0x4afb6c98, (q31_t)0x9842f043, (q31_t)0x4aa9dba2, (q31_t)0x98082c3b, (q31_t)0x4a581c9e, (q31_t)0x97cda855, (q31_t)0x4a062fbd, (q31_t)0x979364b5,
	(q31_t)0x49b41533, (q31_t)0x9759617f, (q31_t)0x4961cd33, (q31_t)0x971f9ed7, (q31_t)0x490f57ee, (q31_t)0x96e61ce0, (q31_t)0x48bcb599, (q31_t)0x96acdbbe,
	(q31_t)0x4869e665, (q31_t)0x9673db94, (q31_t)0x4816ea86, (q31_t)0x963b1c86, (q31_t)0x47c3c22f, (q31_t)0x96029eb6, (q31_t)0x47706d93, (q31_t)0x95ca6247,
	(q31_t)0x471cece7, (q31_t)0x9592675c, (q31_t)0x46c9405c, (q31_t)0x955aae17, (q31_t)0x46756828, (q31_t)0x9523369c, (q31_t)0x4621647d, (q31_t)0x94ec010b,
	(q31_t)0x45cd358f, (q31_t)0x94b50d87, (q31_t)0x4578db93, (q31_t)0x947e5c33, (q31_t)0x452456bd, (q31_t)0x9447ed2f, (q31_t)0x44cfa740, (q31_t)0x9411c09e,
	(q31_t)0x447acd50, (q31_t)0x93dbd6a0, (q31_t)0x4425c923, (q31_t)0x93a62f57, (q31_t)0x43d09aed, (q31_t)0x9370cae4, (q31_t)0x437b42e1, (q31_t)0x933ba968,
	(q31_t)0x4325c135, (q31_t)0x9306cb04, (q31_t)0x42d0161e, (q31_t)0x92d22fd9, (q31_t)0x427a41d0, (q31_t)0x929dd806, (q31_t)0x42244481, (q31_t)0x9269c3ac,
	(q31_t)0x41ce1e65, (q31_t)0x9235f2ec, (q31_t)0x4177cfb1, (q31_t)0x920265e4, (q31_t)0x4121589b, (q31_t)0x91cf1cb6, (q31_t)0x40cab958, (q31_t)0x919c1781,
	(q31_t)0x4073f21d, (q31_t)0x91695663, (q31_t)0x401d0321, (q31_t)0x9136d97d, (q31_t)0x3fc5ec98, (q31_t)0x9104a0ee, (q31_t)0x3f6eaeb8, (q31_t)0x90d2acd4,
	(q31_t)0x3f1749b8, (q31_t)0x90a0fd4e, (q31_t)0x3ebfbdcd, (q31_t)0x906f927c, (q31_t)0x3e680b2c, (q31_t)0x903e6c7b, (q31_t)0x3e10320d, (q31_t)0x900d8b69,
	(q31_t)0x3db832a6, (q31_t)0x8fdcef66, (q31_t)0x3d600d2c, (q31_t)0x8fac988f, (q31_t)0x3d07c1d6, (q31_t)0x8f7c8701, (q31_t)0x3caf50da, (q31_t)0x8f4cbadb,
	(q31_t)0x3c56ba70, (q31_t)0x8f1d343a, (q31_t)0x3bfdfecd, (q31_t)0x8eedf33b, (q31_t)0x3ba51e29, (q31_t)0x8ebef7fb, (q31_t)0x3b4c18ba, (q31_t)0x8e904298,
	(q31_t)0x3af2eeb7, (q31_t)0x8e61d32e, (q31_t)0x3a99a057, (q31_t)0x8e33a9da, (q31_t)0x3a402dd2, (q31_t)0x8e05c6b7, (q31_t)0x39e6975e, (q31_t)0x8dd829e4,
	(q31_t)0x398cdd32, (q31_t)0x8daad37b, (q31_t)0x3932ff87, (q31_t)0x8d7dc399, (q31_t)0x38d8fe93, (q31_t)0x8d50fa59, (q31_t)0x387eda8e, (q31_t)0x8d2477d8,
	(q31_t)0x382493b0, (q31_t)0x8cf83c30, (q31_t)0x37ca2a30, (q31_t)0x8ccc477d, (q31_t)0x376f9e46, (q31_t)0x8ca099da, (q31_t)0x3714f02a, (q31_t)0x8c753362,
	(q31_t)0x36ba2014, (q31_t)0x8c4a142f, (q31_t)0x365f2e3b, (q31_t)0x8c1f3c5d, (q31_t)0x36041ad9, (q31_t)0x8bf4ac05, (q31_t)0x35a8e625, (q31_t)0x8bca6343,
	(q31_t)0x354d9057, (q31_t)0x8ba0622f, (q31_t)0x34f219a8, (q31_t)0x8b76a8e4, (q31_t)0x34968250, (q31_t)0x8b4d377c, (q31_t)0x343aca87, (q31_t)0x8b240e11,
	(q31_t)0x33def287, (q31_t)0x8afb2cbb, (q31_t)0x3382fa88, (q31_t)0x8ad29394, (q31_t)0x3326e2c3, (q31_t)0x8aaa42b4, (q31_t)0x32caab6f, (q31_t)0x8a823a36,
	(q31_t)0x326e54c7, (q31_t)0x8a5a7a31, (q31_t)0x3211df04, (q31_t)0x8a3302be, (q31_t)0x31b54a5e, (q31_t)0x8a0bd3f5, (q31_t)0x3158970e, (q31_t)0x89e4edef,
	(q31_t)0x30fbc54d, (q31_t)0x89be50c3, (q31_t)0x309ed556, (q31_t)0x8997fc8a, (q31_t)0x3041c761, (q31_t)0x8971f15a, (q31_t)0x2fe49ba7, (q31_t)0x894c2f4c,
	(q31_t)0x2f875262, (q31_t)0x8926b677, (q31_t)0x2f29ebcc, (q31_t)0x890186f2, (q31_t)0x2ecc681e, (q31_t)0x88dca0d3, (q31_t)0x2e6ec792, (q31_t)0x88b80432,
	(q31_t)0x2e110a62, (q31_t)0x8893b125, (q31_t)0x2db330c7, (q31_t)0x886fa7c2, (q31_t)0x2d553afc, (q31_t)0x884be821, (q31_t)0x2cf72939, (q31_t)0x88287256,
	(q31_t)0x2c98fbba, (q31_t)0x88054677, (q31_t)0x2c3ab2b9, (q31_t)0x87e2649b, (q31_t)0x2bdc4e6f, (q31_t)0x87bfccd7, (q31_t)0x2b7dcf17, (q31_t)0x879d7f41,
	(q31_t)0x2b1f34eb, (q31_t)0x877b7bec, (q31_t)0x2ac08026, (q31_t)0x8759c2ef, (q31_t)0x2a61b101, (q31_t)0x8738545e, (q31_t)0x2a02c7b8, (q31_t)0x8717304e,
	(q31_t)0x29a3c485, (q31_t)0x86f656d3, (q31_t)0x2944a7a2, (q31_t)0x86d5c802, (q31_t)0x28e5714b, (q31_t)0x86b583ee, (q31_t)0x288621b9, (q31_t)0x86958aac,
	(q31_t)0x2826b928, (q31_t)0x8675dc4f, (q31_t)0x27c737d3, (q31_t)0x865678eb, (q31_t)0x27679df4, (q31_t)0x86376092, (q31_t)0x2707ebc7, (q31_t)0x86189359,
	(q31_t)0x26a82186, (q31_t)0x85fa1153, (q31_t)0x26483f6c, (q31_t)0x85dbda91, (q31_t)0x25e845b6, (q31_t)0x85bdef28, (q31_t)0x2588349d, (q31_t)0x85a04f28,
	(q31_t)0x25280c5e, (q31_t)0x8582faa5, (q31_t)0x24c7cd33, (q31_t)0x8565f1b0, (q31_t)0x24677758, (q31_t)0x8549345c, (q31_t)0x24070b08, (q31_t)0x852cc2bb,
	(q31_t)0x23a6887f, (q31_t)0x85109cdd, (q31_t)0x2345eff8, (q31_t)0x84f4c2d4, (q31_t)0x22e541af, (q31_t)0x84d934b1, (q31_t)0x22847de0, (q31_t)0x84bdf286,
	(q31_t)0x2223a4c5, (q31_t)0x84a2fc62, (q31_t)0x21c2b69c, (q31_t)0x84885258, (q31_t)0x2161b3a0, (q31_t)0x846df477, (q31_t)0x21009c0c, (q31_t)0x8453e2cf,
	(q31_t)0x209f701c, (q31_t)0x843a1d70, (q31_t)0x203e300d, (q31_t)0x8420a46c, (q31_t)0x1fdcdc1b, (q31_t)0x840777d0, (q31_t)0x1f7b7481, (q31_t)0x83ee97ad,
	(q31_t)0x1f19f97b, (q31_t)0x83d60412, (q31_t)0x1eb86b46, (q31_t)0x83bdbd0e, (q31_t)0x1e56ca1e, (q31_t)0x83a5c2b0, (q31_t)0x1df5163f, (q31_t)0x838e1507,
	(q31_t)0x1d934fe5, (q31_t)0x8376b422, (q31_t)0x1d31774d, (q31_t)0x835fa00f, (q31_t)0x1ccf8cb3, (q31_t)0x8348d8dc, (q31_t)0x1c6d9053, (q31_t)0x83325e97,
	(q31_t)0x1c0b826a, (q31_t)0x831c314e, (q31_t)0x1ba96335, (q31_t)0x83065110, (q31_t)0x1b4732ef, (q31_t)0x82f0bde8, (q31_t)0x1ae4f1d6, (q31_t)0x82db77e5,
	(q31_t)0x1a82a026, (q31_t)0x82c67f14, (q31_t)0x1a203e1b, (q31_t)0x82b1d381, (q31_t)0x19bdcbf3, (q31_t)0x829d753a, (q31_t)0x195b49ea, (q31_t)0x8289644b,
	(q31_t)0x18f8b83c, (q31_t)0x8275a0c0, (q31_t)0x18961728, (q31_t)0x82622aa6, (q31_t)0x183366e9, (q31_t)0x824f0208, (q31_t)0x17d0a7bc, (q31_t)0x823c26f3,
	(q31_t)0x176dd9de, (q31_t)0x82299971, (q31_t)0x170afd8d, (q31_t)0x82175990, (q31_t)0x16a81305, (q31_t)0x82056758, (q31_t)0x16451a83, (q31_t)0x81f3c2d7,
	(q31_t)0x15e21445, (q31_t)0x81e26c16, (q31_t)0x157f0086, (q31_t)0x81d16321, (q31_t)0x151bdf86, (q31_t)0x81c0a801, (q31_t)0x14b8b17f, (q31_t)0x81b03ac2,
	(q31_t)0x145576b1, (q31_t)0x81a01b6d, (q31_t)0x13f22f58, (q31_t)0x81904a0c, (q31_t)0x138edbb1, (q31_t)0x8180c6a9, (q31_t)0x132b7bf9, (q31_t)0x8171914e,
	(q31_t)0x12c8106f, (q31_t)0x8162aa04, (q31_t)0x1264994e, (q31_t)0x815410d4, (q31_t)0x120116d5, (q31_t)0x8145c5c7, (q31_t)0x119d8941, (q31_t)0x8137c8e6,
	(q31_t)0x1139f0cf, (q31_t)0x812a1a3a, (q31_t)0x10d64dbd, (q31_t)0x811cb9ca, (q31_t)0x1072a048, (q31_t)0x810fa7a0, (q31_t)0x100ee8ad, (q31_t)0x8102e3c4,
	(q31_t)0xfab272b, (q31_t)0x80f66e3c, (q31_t)0xf475bff, (q31_t)0x80ea4712, (q31_t)0xee38766, (q31_t)0x80de6e4c, (q31_t)0xe7fa99e, (q31_t)0x80d2e3f2,
	(q31_t)0xe1bc2e4, (q31_t)0x80c7a80a, (q31_t)0xdb7d376, (q31_t)0x80bcba9d, (q31_t)0xd53db92, (q31_t)0x80b21baf, (q31_t)0xcefdb76, (q31_t)0x80a7cb49,
	(q31_t)0xc8bd35e, (q31_t)0x809dc971, (q31_t)0xc27c389, (q31_t)0x8094162c, (q31_t)0xbc3ac35, (q31_t)0x808ab180, (q31_t)0xb5f8d9f, (q31_t)0x80819b74,
	(q31_t)0xafb6805, (q31_t)0x8078d40d, (q31_t)0xa973ba5, (q31_t)0x80705b50, (q31_t)0xa3308bd, (q31_t)0x80683143, (q31_t)0x9cecf89, (q31_t)0x806055eb,
	(q31_t)0x96a9049, (q31_t)0x8058c94c, (q31_t)0x9064b3a, (q31_t)0x80518b6b, (q31_t)0x8a2009a, (q31_t)0x804a9c4d, (q31_t)0x83db0a7, (q31_t)0x8043fbf6,
	(q31_t)0x7d95b9e, (q31_t)0x803daa6a, (q31_t)0x77501be, (q31_t)0x8037a7ac, (q31_t)0x710a345, (q31_t)0x8031f3c2, (q31_t)0x6ac406f, (q31_t)0x802c8ead,
	(q31_t)0x647d97c, (q31_t)0x80277872, (q31_t)0x5e36ea9, (q31_t)0x8022b114, (q31_t)0x57f0035, (q31_t)0x801e3895, (q31_t)0x51a8e5c, (q31_t)0x801a0ef8,
	(q31_t)0x4b6195d, (q31_t)0x80163440, (q31_t)0x451a177, (q31_t)0x8012a86f, (q31_t)0x3ed26e6, (q31_t)0x800f6b88, (q31_t)0x388a9ea, (q31_t)0x800c7d8c,
	(q31_t)0x3242abf, (q31_t)0x8009de7e, (q31_t)0x2bfa9a4, (q31_t)0x80078e5e, (q31_t)0x25b26d7, (q31_t)0x80058d2f, (q31_t)0x1f6a297, (q31_t)0x8003daf1,
	(q31_t)0x1921d20, (q31_t)0x800277a6, (q31_t)0x12d96b1, (q31_t)0x8001634e, (q31_t)0xc90f88, (q31_t)0x80009dea, (q31_t)0x6487e3, (q31_t)0x8000277a
};

static const q31_t WeightsQ31_2048[4096] = {
	(q31_t)0x7fffffff, (q31_t)0x00000000, (q31_t)0x7ffffd88, (q31_t)0xffe6de05, (q31_t)0x7ffff621, (q31_t)0xffcdbc0b, (q31_t)0x7fffe9cb, (q31_t)0xffb49a12,
	(q31_t)0x7fffd886, (q31_t)0xff9b781d, (q31_t)0x7fffc251, (q31_t)0xff82562c, (q31_t)0x7fffa72c, (q31_t)0xff69343f, (q31_t)0x7fff8719, (q31_t)0xff501258,
	(q31_t)0x7fff6216, (q31_t)0xff36f078, (q31_t)0x7fff3824, (q31_t)0xff1dcea0, (q31_t)0x7fff0943, (q31_t)0xff04acd0, (q31_t)0x7ffed572, (q31_t)0xfeeb8b0a,
	(q31_t)0x7ffe9cb2, (q31_t)0xfed2694f, (q31_t)0x7ffe5f03, (q31_t)0xfeb947a0, (q31_t)0x7ffe1c65, (q31_t)0xfea025fd, (q31_t)0x7ffdd4d7, (q31_t)0xfe870467,
	(q31_t)0x7ffd885a, (q31_t)0xfe6de2e0, (q31_t)0x7ffd36ee, (q31_t)0xfe54c169, (q31_t)0x7ffce093, (q31_t)0xfe3ba002, (q31_t)0x7ffc8549, (q31_t)0xfe227eac,
	(q31_t)0x7ffc250f, (q31_t)0xfe095d69, (q31_t)0x7ffbbfe6, (q31_t)0xfdf03c3a, (q31_t)0x7ffb55ce, (q31_t)0xfdd71b1e, (q31_t)0x7ffae6c7, (q31_t)0xfdbdfa18,
	(q31_t)0x7ffa72d1, (q31_t)0xfda4d929, (q31_t)0x7ff9f9ec, (q31_t)0xfd8bb850, (q31_t)0x7ff97c18, (q31_t)0xfd729790, (q31_t)0x7ff8f954, (q31_t)0xfd5976e9,
	(q31_t)0x7ff871a2, (q31_t)0xfd40565c, (q31_t)0x7ff7e500, (q31_t)0xfd2735ea, (q31_t)0x7ff75370, (q31_t)0xfd0e1594, (q31_t)0x7ff6bcf0, (q31_t)0xfcf4f55c,
	(q31_t)0x7ff62182, (q31_t)0xfcdbd541, (q31_t)0x7ff58125, (q31_t)0xfcc2b545, (q31_t)0x7ff4dbd9, (q31_t)0xfca9956a, (q31_t)0x7ff4319d, (q31_t)0xfc9075af,
	(q31_t)0x7ff38274, (q31_t)0xfc775616, (q31_t)0x7ff2ce5b, (q31_t)0xfc5e36a0, (q31_t)0x7ff21553, (q31_t)0xfc45174e, (q31_t)0x7ff1575d, (q31_t)0xfc2bf821,
	(q31_t)0x7ff09478, (q31_t)0xfc12d91a, (q31_t)0x7fefcca4, (q31_t)0xfbf9ba39, (q31_t)0x7feeffe1, (q31_t)0xfbe09b80, (q31_t)0x7fee2e30, (q31_t)0xfbc77cf0,
	(q31_t)0x7fed5791, (q31_t)0xfbae5e89, (q31_t)0x7fec7c02, (q31_t)0xfb95404d, (q31_t)0x7feb9b85, (q31_t)0xfb7c223d, (q31_t)0x7feab61a, (q31_t)0xfb630459,
	(q31_t)0x7fe9cbc0, (q31_t)0xfb49e6a3, (q31_t)0x7fe8dc78, (q31_t)0xfb30c91b, (q31_t)0x7fe7e841, (q31_t)0xfb17abc2, (q31_t)0x7fe6ef1c, (q31_t)0xfafe8e9b,
	(q31_t)0x7fe5f108, (q31_t)0xfae571a4, (q31_t)0x7fe4ee06, (q31_t)0xfacc54e0, (q31_t)0x7fe3e616, (q31_t)0xfab3384f, (q31_t)0x7fe2d938, (q31_t)0xfa9a1bf3,
	(q31_t)0x7fe1c76b, (q31_t)0xfa80ffcb, (q31_t)0x7fe0b0b1, (q31_t)0xfa67e3da, (q31_t)0x7fdf9508, (q31_t)0xfa4ec821, (q31_t)0x7fde7471, (q31_t)0xfa35ac9f,
	(q31_t)0x7fdd4eec, (q31_t)0xfa1c9157, (q31_t)0x7fdc247a, (q31_t)0xfa037648, (q31_t)0x7fdaf519, (q31_t)0xf9ea5b75, (q31_t)0x7fd9c0ca, (q31_t)0xf9d140de,
	(q31_t)0x7fd8878e, (q31_t)0xf9b82684, (q31_t)0x7fd74964, (q31_t)0xf99f0c68, (q31_t)0x7fd6064c, (q31_t)0xf985f28a, (q31_t)0x7fd4be46, (q31_t)0xf96cd8ed,
	(q31_t)0x7fd37153, (q31_t)0xf953bf91, (q31_t)0x7fd21f72, (q31_t)0xf93aa676, (q31_t)0x7fd0c8a3, (q31_t)0xf9218d9e, (q31_t)0x7fcf6ce8, (q31_t)0xf908750a,
	(q31_t)0x7fce0c3e, (q31_t)0xf8ef5cbb, (q31_t)0x7fcca6a7, (q31_t)0xf8d644b2, (q31_t)0x7fcb3c23, (q31_t)0xf8bd2cef, (q31_t)0x7fc9ccb2, (q31_t)0xf8a41574,
	(q31_t)0x7fc85854, (q31_t)0xf88afe42, (q31_t)0x7fc6df08, (q31_t)0xf871e759, (q31_t)0x7fc560cf, (q31_t)0xf858d0bb, (q31_t)0x7fc3dda9, (q31_t)0xf83fba68,
	(q31_t)0x7fc25596, (q31_t)0xf826a462, (q31_t)0x7fc0c896, (q31_t)0xf80d8ea9, (q31_t)0x7fbf36aa, (q31_t)0xf7f4793e, (q31_t)0x7fbd9fd0, (q31_t)0xf7db6423,
	(q31_t)0x7fbc040a, (q31_t)0xf7c24f59, (q31_t)0x7fba6357, (q31_t)0xf7a93ae0, (q31_t)0x7fb8bdb8, (q31_t)0xf79026b9, (q31_t)0x7fb7132b, (q31_t)0xf77712e5,
	(q31_t)0x7fb563b3, (q31_t)0xf75dff66, (q31_t)0x7fb3af4e, (q31_t)0xf744ec3b, (q31_t)0x7fb1f5fc, (q31_t)0xf72bd967, (q31_t)0x7fb037bf, (q31_t)0xf712c6ea,
	(q31_t)0x7fae7495, (q31_t)0xf6f9b4c6, (q31_t)0x7facac7f, (q31_t)0xf6e0a2fa, (q31_t)0x7faadf7c, (q31_t)0xf6c79188, (q31_t)0x7fa90d8e, (q31_t)0xf6ae8071,
	(q31_t)0x7fa736b4, (q31_t)0xf6956fb7, (q31_t)0x7fa55aee, (q31_t)0xf67c5f59, (q31_t)0x7fa37a3c, (q31_t)0xf6634f59, (q31_t)0x7fa1949e, (q31_t)0xf64a3fb8,
	(q31_t)0x7f9faa15, (q31_t)0xf6313077, (q31_t)0x7f9dbaa0, (q31_t)0xf6182196, (q31_t)0x7f9bc640, (q31_t)0xf5ff1318, (q31_t)0x7f99ccf4, (q31_t)0xf5e604fc,
	(q31_t)0x7f97cebd, (q31_t)0xf5ccf743, (q31_t)0x7f95cb9a, (q31_t)0xf5b3e9f0, (q31_t)0x7f93c38c, (q31_t)0xf59add02, (q31_t)0x7f91b694, (q31_t)0xf581d07b,
	(q31_t)0x7f8fa4b0, (q31_t)0xf568c45b, (q31_t)0x7f8d8de1, (q31_t)0xf54fb8a4, (q31_t)0x7f8b7227, (q31_t)0xf536ad56, (q31_t)0x7f895182, (q31_t)0xf51da273,
	(q31_t)0x7f872bf3, (q31_t)0xf50497fb, (q31_t)0x7f850179, (q31_t)0xf4eb8def, (q31_t)0x7f82d214, (q31_t)0xf4d28451, (q31_t)0x7f809dc5, (q31_t)0xf4b97b21,
	(q31_t)0x7f7e648c, (q31_t)0xf4a07261, (q31_t)0x7f7c2668, (q31_t)0xf4876a10, (q31_t)0x7f79e35a, (q31_t)0xf46e6231, (q31_t)0x7f779b62, (q31_t)0xf4555ac5,
	(q31_t)0x7f754e80, (q31_t)0xf43c53cb, (q31_t)0x7f72fcb4, (q31_t)0xf4234d45, (q31_t)0x7f70a5fe, (q31_t)0xf40a4735, (q31_t)0x7f6e4a5e, (q31_t)0xf3f1419a,
	(q31_t)0x7f6be9d4, (q31_t)0xf3d83c77, (q31_t)0x7f698461, (q31_t)0xf3bf37cb, (q31_t)0x7f671a05, (q31_t)0xf3a63398, (q31_t)0x7f64aabf, (q31_t)0xf38d2fe0,
	(q31_t)0x7f62368f, (q31_t)0xf3742ca2, (q31_t)0x7f5fbd77, (q31_t)0xf35b29e0, (q31_t)0x7f5d3f75, (q31_t)0xf342279b, (q31_t)0x7f5abc8a, (q31_t)0xf32925d3,
	(q31_t)0x7f5834b7, (q31_t)0xf310248a, (q31_t)0x7f55a7fa, (q31_t)0xf2f723c1, (q31_t)0x7f531655, (q31_t)0xf2de2379, (q31_t)0x7f507fc7, (q31_t)0xf2c523b2,
	(q31_t)0x7f4de451, (q31_t)0xf2ac246e, (q31_t)0x7f4b43f2, (q31_t)0xf29325ad, (q31_t)0x7f489eaa, (q31_t)0xf27a2771, (q31_t)0x7f45f47b, (q31_t)0xf26129ba,
	(q31_t)0x7f434563, (q31_t)0xf2482c8a, (q31_t)0x7f409164, (q31_t)0xf22f2fe1, (q31_t)0x7f3dd87c, (q31_t)0xf21633c0, (q31_t)0x7f3b1aad, (q31_t)0xf1fd3829,
	(q31_t)0x7f3857f6, (q31_t)0xf1e43d1c, (q31_t)0x7f359057, (q31_t)0xf1cb429a, (q31_t)0x7f32c3d1, (q31_t)0xf1b248a5, (q31_t)0x7f2ff263, (q31_t)0xf1994f3d,
	(q31_t)0x7f2d1c0e, (q31_t)0xf1805662, (q31_t)0x7f2a40d2, (q31_t)0xf1675e17, (q31_t)0x7f2760af, (q31_t)0xf14e665c, (q31_t)0x7f247ba5, (q31_t)0xf1356f32,
	(q31_t)0x7f2191b4, (q31_t)0xf11c789a, (q31_t)0x7f1ea2dc, (q31_t)0xf1038295, (q31_t)0x7f1baf1e, (q31_t)0xf0ea8d24, (q31_t)0x7f18b679, (q31_t)0xf0d19848,
	(q31_t)0x7f15b8ee, (q31_t)0xf0b8a401, (q31_t)0x7f12b67c, (q31_t)0xf09fb051, (q31_t)0x7f0faf25, (q31_t)0xf086bd39, (q31_t)0x7f0ca2e7, (q31_t)0xf06dcaba,
	(q31_t)0x7f0991c4, (q31_t)0xf054d8d5, (q31_t)0x7f067bba, (q31_t)0xf03be78a, (q31_t)0x7f0360cb, (q31_t)0xf022f6da, (q31_t)0x7f0040f6, (q31_t)0xf00a06c8,
	(q31_t)0x7efd1c3c, (q31_t)0xeff11753, (q31_t)0x7ef9f29d, (q31_t)0xefd8287c, (q31_t)0x7ef6c418, (q31_t)0xefbf3a45, (q31_t)0x7ef390ae, (q31_t)0xefa64cae,
	(q31_t)0x7ef05860, (q31_t)0xef8d5fb8, (q31_t)0x7eed1b2c, (q31_t)0xef747365, (q31_t)0x7ee9d914, (q31_t)0xef5b87b5, (q31_t)0x7ee69217, (q31_t)0xef429caa,
	(q31_t)0x7ee34636, (q31_t)0xef29b243, (q31_t)0x7edff570, (q31_t)0xef10c883, (q31_t)0x7edc9fc6, (q31_t)0xeef7df6a, (q31_t)0x7ed94538, (q31_t)0xeedef6f9,
	(q31_t)0x7ed5e5c6, (q31_t)0xeec60f31, (q31_t)0x7ed28171, (q31_t)0xeead2813, (q31_t)0x7ecf1837, (q31_t)0xee9441a0, (q31_t)0x7ecbaa1a, (q31_t)0xee7b5bd9,
	(q31_t)0x7ec8371a, (q31_t)0xee6276bf, (q31_t)0x7ec4bf36, (q31_t)0xee499253, (q31_t)0x7ec14270, (q31_t)0xee30ae96, (q31_t)0x7ebdc0c6, (q31_t)0xee17cb88,
	(q31_t)0x7eba3a39, (q31_t)0xedfee92b, (q31_t)0x7eb6aeca, (q31_t)0xede60780, (q31_t)0x7eb31e78, (q31_t)0xedcd2687, (q31_t)0x7eaf8943, (q31_t)0xedb44642,
	(q31_t)0x7eabef2c, (q31_t)0xed9b66b2, (q31_t)0x7ea85033, (q31_t)0xed8287d7, (q31_t)0x7ea4ac58, (q31_t)0xed69a9b3, (q31_t)0x7ea1039b, (q31_t)0xed50cc46,
	(q31_t)0x7e9d55fc, (q31_t)0xed37ef91, (q31_t)0x7e99a37c, (q31_t)0xed1f1396, (q31_t)0x7e95ec1a, (q31_t)0xed063856, (q31_t)0x7e922fd6, (q31_t)0xeced5dd0,
	(q31_t)0x7e8e6eb2, (q31_t)0xecd48407, (q31_t)0x7e8aa8ac, (q31_t)0xecbbaafb, (q31_t)0x7e86ddc6, (q31_t)0xeca2d2ad, (q31_t)0x7e830dff, (q31_t)0xec89fb1e,
	(q31_t)0x7e7f3957, (q31_t)0xec71244f, (q31_t)0x7e7b5fce, (q31_t)0xec584e41, (q31_t)0x7e778166, (q31_t)0xec3f78f6, (q31_t)0x7e739e1d, (q31_t)0xec26a46d,
	(q31_t)0x7e6fb5f4, (q31_t)0xec0dd0a8, (q31_t)0x7e6bc8eb, (q31_t)0xebf4fda8, (q31_t)0x7e67d703, (q31_t)0xebdc2b6e, (q31_t)0x7e63e03b, (q31_t)0xebc359fb,
	(q31_t)0x7e5fe493, (q31_t)0xebaa894f, (q31_t)0x7e5be40c, (q31_t)0xeb91b96c, (q31_t)0x7e57dea7, (q31_t)0xeb78ea52, (q31_t)0x7e53d462, (q31_t)0xeb601c04,
	(q31_t)0x7e4fc53e, (q31_t)0xeb474e81, (q31_t)0x7e4bb13c, (q31_t)0xeb2e81ca, (q31_t)0x7e47985b, (q31_t)0xeb15b5e1, (q31_t)0x7e437a9c, (q31_t)0xeafceac6,
	(q31_t)0x7e3f57ff, (q31_t)0xeae4207a, (q31_t)0x7e3b3083, (q31_t)0xeacb56ff, (q31_t)0x7e37042a, (q31_t)0xeab28e56, (q31_t)0x7e32d2f4, (q31_t)0xea99c67e,
	(q31_t)0x7e2e9cdf, (q31_t)0xea80ff7a, (q31_t)0x7e2a61ed, (q31_t)0xea683949, (q31_t)0x7e26221f, (q31_t)0xea4f73ee, (q31_t)0x7e21dd73, (q31_t)0xea36af69,
	(q31_t)0x7e1d93ea, (q31_t)0xea1debbb, (q31_t)0x7e194584, (q31_t)0xea0528e5, (q31_t)0x7e14f242, (q31_t)0xe9ec66e8, (q31_t)0x7e109a24, (q31_t)0xe9d3a5c5,
	(q31_t)0x7e0c3d29, (q31_t)0xe9bae57d, (q31_t)0x7e07db52, (q31_t)0xe9a22610, (q31_t)0x7e0374a0, (q31_t)0xe9896781, (q31_t)0x7dff0911, (q31_t)0xe970a9ce,
	(q31_t)0x7dfa98a8, (q31_t)0xe957ecfb, (q31_t)0x7df62362, (q31_t)0xe93f3107, (q31_t)0x7df1a942, (q31_t)0xe92675f4, (q31_t)0x7ded2a47, (q31_t)0xe90dbbc2,
	(q31_t)0x7de8a670, (q31_t)0xe8f50273, (q31_t)0x7de41dc0, (q31_t)0xe8dc4a07, (q31_t)0x7ddf9034, (q31_t)0xe8c39280, (q31_t)0x7ddafdce, (q31_t)0xe8aadbde,
	(q31_t)0x7dd6668f, (q31_t)0xe8922622, (q31_t)0x7dd1ca75, (q31_t)0xe879714d, (q31_t)0x7dcd2981, (q31_t)0xe860bd61, (q31_t)0x7dc883b4, (q31_t)0xe8480a5d,
	(q31_t)0x7dc3d90d, (q31_t)0xe82f5844, (q31_t)0x7dbf298d, (q31_t)0xe816a716, (q31_t)0x7dba7534, (q31_t)0xe7fdf6d4, (q31_t)0x7db5bc02, (q31_t)0xe7e5477f,
	(q31_t)0x7db0fdf8, (q31_t)0xe7cc9917, (q31_t)0x7dac3b15, (q31_t)0xe7b3eb9f, (q31_t)0x7da77359, (q31_t)0xe79b3f16, (q31_t)0x7da2a6c6, (q31_t)0xe782937e,
	(q31_t)0x7d9dd55a, (q31_t)0xe769e8d8, (q31_t)0x7d98ff17, (q31_t)0xe7513f25, (q31_t)0x7d9423fc, (q31_t)0xe7389665, (q31_t)0x7d8f4409, (q31_t)0xe71fee99,
	(q31_t)0x7d8a5f40, (q31_t)0xe70747c4, (q31_t)0x7d85759f, (q31_t)0xe6eea1e4, (q31_t)0x7d808728, (q31_t)0xe6d5fcfc, (q31_t)0x7d7b93da, (q31_t)0xe6bd590d,
	(q31_t)0x7d769bb5, (q31_t)0xe6a4b616, (q31_t)0x7d719eba, (q31_t)0xe68c141a, (q31_t)0x7d6c9ce9, (q31_t)0xe6737319, (q31_t)0x7d679642, (q31_t)0xe65ad315,
	(q31_t)0x7d628ac6, (q31_t)0xe642340d, (q31_t)0x7d5d7a74, (q31_t)0xe6299604, (q31_t)0x7d58654d, (q31_t)0xe610f8f9, (q31_t)0x7d534b50, (q31_t)0xe5f85cef,
	(q31_t)0x7d4e2c7f, (q31_t)0xe5dfc1e5, (q31_t)0x7d4908d9, (q31_t)0xe5c727dd, (q31_t)0x7d43e05e, (q31_t)0xe5ae8ed8, (q31_t)0x7d3eb30f, (q31_t)0xe595f6d7,
	(q31_t)0x7d3980ec, (q31_t)0xe57d5fda, (q31_t)0x7d3449f5, (q31_t)0xe564c9e3, (q31_t)0x7d2f0e2b, (q31_t)0xe54c34f3, (q31_t)0x7d29cd8c, (q31_t)0xe533a10a,
	(q31_t)0x7d24881b, (q31_t)0xe51b0e2a, (q31_t)0x7d1f3dd6, (q31_t)0xe5027c53, (q31_t)0x7d19eebf, (q31_t)0xe4e9eb87, (q31_t)0x7d149ad5, (q31_t)0xe4d15bc6,
	(q31_t)0x7d0f4218, (q31_t)0xe4b8cd11, (q31_t)0x7d09e489, (q31_t)0xe4a03f69, (q31_t)0x7d048228, (q31_t)0xe487b2d0, (q31_t)0x7cff1af5, (q31_t)0xe46f2745,
	(q31_t)0x7cf9aef0, (q31_t)0xe4569ccb, (q31_t)0x7cf43e1a, (q31_t)0xe43e1362, (q31_t)0x7ceec873, (q31_t)0xe4258b0a, (q31_t)0x7ce94dfb, (q31_t)0xe40d03c6,
	(q31_t)0x7ce3ceb2, (q31_t)0xe3f47d96, (q31_t)0x7cde4a98, (q31_t)0xe3dbf87a, (q31_t)0x7cd8c1ae, (q31_t)0xe3c37474, (q31_t)0x7cd333f3, (q31_t)0xe3aaf184,
	(q31_t)0x7ccda169, (q31_t)0xe3926fad, (q31_t)0x7cc80a0f, (q31_t)0xe379eeed, (q31_t)0x7cc26de5, (q31_t)0xe3616f48, (q31_t)0x7cbcccec, (q31_t)0xe348f0bd,
	(q31_t)0x7cb72724, (q31_t)0xe330734d, (q31_t)0x7cb17c8d, (q31_t)0xe317f6fa, (q31_t)0x7cabcd28, (q31_t)0xe2ff7bc3, (q31_t)0x7ca618f3, (q31_t)0xe2e701ac,
	(q31_t)0x7ca05ff1, (q31_t)0xe2ce88b3, (q31_t)0x7c9aa221, (q31_t)0xe2b610da, (q31_t)0x7c94df83, (q31_t)0xe29d9a23, (q31_t)0x7c8f1817, (q31_t)0xe285248d,
	(q31_t)0x7c894bde, (q31_t)0xe26cb01b, (q31_t)0x7c837ad8, (q31_t)0xe2543ccc, (q31_t)0x7c7da505, (q31_t)0xe23bcaa2, (q31_t)0x7c77ca65, (q31_t)0xe223599e,
	(q31_t)0x7c71eaf9, (q31_t)0xe20ae9c1, (q31_t)0x7c6c06c0, (q31_t)0xe1f27b0b, (q31_t)0x7c661dbc, (q31_t)0xe1da0d7e, (q31_t)0x7c602fec, (q31_t)0xe1c1a11b,
	(q31_t)0x7c5a3d50, (q31_t)0xe1a935e2, (q31_t)0x7c5445e9, (q31_t)0xe190cbd4, (q31_t)0x7c4e49b7, (q31_t)0xe17862f3, (q31_t)0x7c4848ba, (q31_t)0xe15ffb3f,
	(q31_t)0x7c4242f2, (q31_t)0xe14794ba, (q31_t)0x7c3c3860, (q31_t)0xe12f2f63, (q31_t)0x7c362904, (q31_t)0xe116cb3d, (q31_t)0x7c3014de, (q31_t)0xe0fe6848,
	(q31_t)0x7c29fbee, (q31_t)0xe0e60685, (q31_t)0x7c23de35, (q31_t)0xe0cda5f5, (q31_t)0x7c1dbbb3, (q31_t)0xe0b54698, (q31_t)0x7c179467, (q31_t)0xe09ce871,
	(q31_t)0x7c116853, (q31_t)0xe0848b7f, (q31_t)0x7c0b3777, (q31_t)0xe06c2fc4, (q31_t)0x7c0501d2, (q31_t)0xe053d541, (q31_t)0x7bfec765, (q31_t)0xe03b7bf6,
	(q31_t)0x7bf88830, (q31_t)0xe02323e5, (q31_t)0x7bf24434, (q31_t)0xe00acd0e, (q31_t)0x7bebfb70, (q31_t)0xdff27773, (q31_t)0x7be5ade6, (q31_t)0xdfda2314,
	(q31_t)0x7bdf5b94, (q31_t)0xdfc1cff3, (q31_t)0x7bd9047c, (q31_t)0xdfa97e0f, (q31_t)0x7bd2a89e, (q31_t)0xdf912d6b, (q31_t)0x7bcc47fa, (q31_t)0xdf78de07,
	(q31_t)0x7bc5e290, (q31_t)0xdf608fe4, (q31_t)0x7bbf7860, (q31_t)0xdf484302, (q31_t)0x7bb9096b, (q31_t)0xdf2ff764, (q31_t)0x7bb295b0, (q31_t)0xdf17ad0a,
	(q31_t)0x7bac1d31, (q31_t)0xdeff63f4, (q31_t)0x7ba59fee, (q31_t)0xdee71c24, (q31_t)0x7b9f1de6, (q31_t)0xdeced59b, (q31_t)0x7b989719, (q31_t)0xdeb69059,
	(q31_t)0x7b920b89, (q31_t)0xde9e4c60, (q31_t)0x7b8b7b36, (q31_t)0xde8609b1, (q31_t)0x7b84e61f, (q31_t)0xde6dc84b, (q31_t)0x7b7e4c45, (q31_t)0xde558831,
	(q31_t)0x7b77ada8, (q31_t)0xde3d4964, (q31_t)0x7b710a49, (q31_t)0xde250be3, (q31_t)0x7b6a6227, (q31_t)0xde0ccfb1, (q31_t)0x7b63b543, (q31_t)0xddf494ce,
	(q31_t)0x7b5d039e, (q31_t)0xdddc5b3b, (q31_t)0x7b564d36, (q31_t)0xddc422f8, (q31_t)0x7b4f920e, (q31_t)0xddabec08, (q31_t)0x7b48d225, (q31_t)0xdd93b66a,
	(q31_t)0x7b420d7a, (q31_t)0xdd7b8220, (q31_t)0x7b3b4410, (q31_t)0xdd634f2b, (q31_t)0x7b3475e5, (q31_t)0xdd4b1d8c, (q31_t)0x7b2da2fa, (q31_t)0xdd32ed43,
	(q31_t)0x7b26cb4f, (q31_t)0xdd1abe51, (q31_t)0x7b1feee5, (q31_t)0xdd0290b8, (q31_t)0x7b190dbc, (q31_t)0xdcea6478, (q31_t)0x7b1227d3, (q31_t)0xdcd23993,
	(q31_t)0x7b0b3d2c, (q31_t)0xdcba1008, (q31_t)0x7b044dc7, (q31_t)0xdca1e7da, (q31_t)0x7afd59a4, (q31_t)0xdc89c109, (q31_t)0x7af660c2, (q31_t)0xdc719b96,
	(q31_t)0x7aef6323, (q31_t)0xdc597781, (q31_t)0x7ae860c7, (q31_t)0xdc4154cd, (q31_t)0x7ae159ae, (q31_t)0xdc293379, (q31_t)0x7ada4dd8, (q31_t)0xdc111388,
	(q31_t)0x7ad33d45, (q31_t)0xdbf8f4f8, (q31_t)0x7acc27f7, (q31_t)0xdbe0d7cd, (q31_t)0x7ac50dec, (q31_t)0xdbc8bc06, (q31_t)0x7abdef25, (q31_t)0xdbb0a1a4,
	(q31_t)0x7ab6cba4, (q31_t)0xdb9888a8, (q31_t)0x7aafa367, (q31_t)0xdb807114, (q31_t)0x7aa8766f, (q31_t)0xdb685ae9, (q31_t)0x7aa144bc, (q31_t)0xdb504626,
	(q31_t)0x7a9a0e50, (q31_t)0xdb3832cd, (q31_t)0x7a92d329, (q31_t)0xdb2020e0, (q31_t)0x7a8b9348, (q31_t)0xdb08105e, (q31_t)0x7a844eae, (q31_t)0xdaf00149,
	(q31_t)0x7a7d055b, (q31_t)0xdad7f3a2, (q31_t)0x7a75b74f, (q31_t)0xdabfe76a, (q31_t)0x7a6e648a, (q31_t)0xdaa7dca1, (q31_t)0x7a670d0d, (q31_t)0xda8fd349,
	(q31_t)0x7a5fb0d8, (q31_t)0xda77cb63, (q31_t)0x7a584feb, (q31_t)0xda5fc4ef, (q31_t)0x7a50ea47, (q31_t)0xda47bfee, (q31_t)0x7a497feb, (q31_t)0xda2fbc61,
	(q31_t)0x7a4210d8, (q31_t)0xda17ba4a, (q31_t)0x7a3a9d0f, (q31_t)0xd9ffb9a9, (q31_t)0x7a332490, (q31_t)0xd9e7ba7f, (q31_t)0x7a2ba75a, (q31_t)0xd9cfbccd,
	(q31_t)0x7a24256f, (q31_t)0xd9b7c094, (q31_t)0x7a1c9ece, (q31_t)0xd99fc5d4, (q31_t)0x7a151378, (q31_t)0xd987cc90, (q31_t)0x7a0d836d, (q31_t)0xd96fd4c7,
	(q31_t)0x7a05eead, (q31_t)0xd957de7a, (q31_t)0x79fe5539, (q31_t)0xd93fe9ab, (q31_t)0x79f6b711, (q31_t)0xd927f65b, (q31_t)0x79ef1436, (q31_t)0xd910048a,
	(q31_t)0x79e76ca7, (q31_t)0xd8f81439, (q31_t)0x79dfc064, (q31_t)0xd8e0256a, (q31_t)0x79d80f6f, (q31_t)0xd8c8381d, (q31_t)0x79d059c8, (q31_t)0xd8b04c52,
	(q31_t)0x79c89f6e, (q31_t)0xd898620c, (q31_t)0x79c0e062, (q31_t)0xd880794b, (q31_t)0x79b91ca4, (q31_t)0xd868920f, (q31_t)0x79b15435, (q31_t)0xd850ac5a,
	(q31_t)0x79a98715, (q31_t)0xd838c82d, (q31_t)0x79a1b545, (q31_t)0xd820e589, (q31_t)0x7999dec4, (q31_t)0xd809046e, (q31_t)0x79920392, (q31_t)0xd7f124dd,
	(q31_t)0x798a23b1, (q31_t)0xd7d946d8, (q31_t)0x79823f20, (q31_t)0xd7c16a5f, (q31_t)0x797a55e0, (q31_t)0xd7a98f73, (q31_t)0x797267f2, (q31_t)0xd791b616,
	(q31_t)0x796a7554, (q31_t)0xd779de47, (q31_t)0x79627e08, (q31_t)0xd7620808, (q31_t)0x795a820e, (q31_t)0xd74a335b, (q31_t)0x79528167, (q31_t)0xd732603f,
	(q31_t)0x794a7c12, (q31_t)0xd71a8eb5, (q31_t)0x79427210, (q31_t)0xd702bec0, (q31_t)0x793a6361, (q31_t)0xd6eaf05f, (q31_t)0x79325006, (q31_t)0xd6d32393,
	(q31_t)0x792a37fe, (q31_t)0xd6bb585e, (q31_t)0x79221b4b, (q31_t)0xd6a38ec0, (q31_t)0x7919f9ec, (q31_t)0xd68bc6ba, (q31_t)0x7911d3e2, (q31_t)0xd674004e,
	(q31_t)0x7909a92d, (q31_t)0xd65c3b7b, (q31_t)0x790179cd, (q31_t)0xd6447844, (q31_t)0x78f945c3, (q31_t)0xd62cb6a8, (q31_t)0x78f10d0f, (q31_t)0xd614f6a9,
	(q31_t)0x78e8cfb2, (q31_t)0xd5fd3848, (q31_t)0x78e08dab, (q31_t)0xd5e57b85, (q31_t)0x78d846fb, (q31_t)0xd5cdc062, (q31_t)0x78cffba3, (q31_t)0xd5b606e0,
	(q31_t)0x78c7aba2, (q31_t)0xd59e4eff, (q31_t)0x78bf56f9, (q31_t)0xd58698c0, (q31_t)0x78b6fda8, (q31_t)0xd56ee424, (q31_t)0x78ae9fb0, (q31_t)0xd557312d,
	(q31_t)0x78a63d11, (q31_t)0xd53f7fda, (q31_t)0x789dd5cb, (q31_t)0xd527d02e, (q31_t)0x789569df, (q31_t)0xd5102228, (q31_t)0x788cf94c, (q31_t)0xd4f875ca,
	(q31_t)0x78848414, (q31_t)0xd4e0cb15, (q31_t)0x787c0a36, (q31_t)0xd4c92209, (q31_t)0x78738bb3, (q31_t)0xd4b17aa8, (q31_t)0x786b088c, (q31_t)0xd499d4f2,
	(q31_t)0x786280bf, (q31_t)0xd48230e9, (q31_t)0x7859f44f, (q31_t)0xd46a8e8d, (q31_t)0x7851633b, (q31_t)0xd452eddf, (q31_t)0x7848cd83, (q31_t)0xd43b4ee0,
	(q31_t)0x78403329, (q31_t)0xd423b191, (q31_t)0x7837942b, (q31_t)0xd40c15f3, (q31_t)0x782ef08b, (q31_t)0xd3f47c06, (q31_t)0x78264849, (q31_t)0xd3dce3cd,
	(q31_t)0x781d9b65, (q31_t)0xd3c54d47, (q31_t)0x7814e9df, (q31_t)0xd3adb876, (q31_t)0x780c33b8, (q31_t)0xd396255a, (q31_t)0x780378f1, (q31_t)0xd37e93f4,
	(q31_t)0x77fab989, (q31_t)0xd3670446, (q31_t)0x77f1f581, (q31_t)0xd34f764f, (q31_t)0x77e92cd9, (q31_t)0xd337ea12, (q31_t)0x77e05f91, (q31_t)0xd3205f8f,
	(q31_t)0x77d78daa, (q31_t)0xd308d6c7, (q31_t)0x77ceb725, (q31_t)0xd2f14fba, (q31_t)0x77c5dc01, (q31_t)0xd2d9ca6a, (q31_t)0x77bcfc3f, (q31_t)0xd2c246d8,
	(q31_t)0x77b417df, (q31_t)0xd2aac504, (q31_t)0x77ab2ee2, (q31_t)0xd29344f0, (q31_t)0x77a24148, (q31_t)0xd27bc69c, (q31_t)0x77994f11, (q31_t)0xd2644a0a,
	(q31_t)0x7790583e, (q31_t)0xd24ccf39, (q31_t)0x77875cce, (q31_t)0xd235562b, (q31_t)0x777e5cc3, (q31_t)0xd21ddee2, (q31_t)0x7775581d, (q31_t)0xd206695d,
	(q31_t)0x776c4edb, (q31_t)0xd1eef59e, (q31_t)0x776340ff, (q31_t)0xd1d783a6, (q31_t)0x775a2e89, (q31_t)0xd1c01375, (q31_t)0x77511778, (q31_t)0xd1a8a50d,
	(q31_t)0x7747fbce, (q31_t)0xd191386e, (q31_t)0x773edb8b, (q31_t)0xd179cd99, (q31_t)0x7735b6af, (q31_t)0xd1626490, (q31_t)0x772c8d3a, (q31_t)0xd14afd52,
	(q31_t)0x77235f2d, (q31_t)0xd13397e2, (q31_t)0x771a2c88, (q31_t)0xd11c343f, (q31_t)0x7710f54c, (q31_t)0xd104d26b, (q31_t)0x7707b979, (q31_t)0xd0ed7267,
	(q31_t)0x76fe790e, (q31_t)0xd0d61434, (q31_t)0x76f5340e, (q31_t)0xd0beb7d2, (q31_t)0x76ebea77, (q31_t)0xd0a75d42, (q31_t)0x76e29c4b, (q31_t)0xd0900486,
	(q31_t)0x76d94989, (q31_t)0xd078ad9e, (q31_t)0x76cff232, (q31_t)0xd061588b, (q31_t)0x76c69647, (q31_t)0xd04a054e, (q31_t)0x76bd35c7, (q31_t)0xd032b3e7,
	(q31_t)0x76b3d0b4, (q31_t)0xd01b6459, (q31_t)0x76aa670d, (q31_t)0xd00416a3, (q31_t)0x76a0f8d2, (q31_t)0xcfeccac7, (q31_t)0x76978605, (q31_t)0xcfd580c6,
	(q31_t)0x768e0ea6, (q31_t)0xcfbe389f, (q31_t)0x768492b4, (q31_t)0xcfa6f255, (q31_t)0x767b1231, (q31_t)0xcf8fade9, (q31_t)0x76718d1c, (q31_t)0xcf786b5a,
	(q31_t)0x76680376, (q31_t)0xcf612aaa, (q31_t)0x765e7540, (q31_t)0xcf49ebda, (q31_t)0x7654e279, (q31_t)0xcf32aeeb, (q31_t)0x764b4b23, (q31_t)0xcf1b73de,
	(q31_t)0x7641af3d, (q31_t)0xcf043ab3, (q31_t)0x76380ec8, (q31_t)0xceed036b, (q31_t)0x762e69c4, (q31_t)0xced5ce08, (q31_t)0x7624c031, (q31_t)0xcebe9a8a,
	(q31_t)0x761b1211, (q31_t)0xcea768f2, (q31_t)0x76115f63, (q31_t)0xce903942, (q31_t)0x7607a828, (q31_t)0xce790b79, (q31_t)0x75fdec60, (q31_t)0xce61df99,
	(q31_t)0x75f42c0b, (q31_t)0xce4ab5a2, (q31_t)0x75ea672a, (q31_t)0xce338d97, (q31_t)0x75e09dbd, (q31_t)0xce1c6777, (q31_t)0x75d6cfc5, (q31_t)0xce054343,
	(q31_t)0x75ccfd42, (q31_t)0xcdee20fc, (q31_t)0x75c32634, (q31_t)0xcdd700a4, (q31_t)0x75b94a9c, (q31_t)0xcdbfe23a, (q31_t)0x75af6a7b, (q31_t)0xcda8c5c1,
	(q31_t)0x75a585cf, (q31_t)0xcd91ab39, (q31_t)0x759b9c9b, (q31_t)0xcd7a92a2, (q31_t)0x7591aedd, (q31_t)0xcd637bfe, (q31_t)0x7587bc98, (q31_t)0xcd4c674d,
	(q31_t)0x757dc5ca, (q31_t)0xcd355491, (q31_t)0x7573ca75, (q31_t)0xcd1e43ca, (q31_t)0x7569ca99, (q31_t)0xcd0734f9, (q31_t)0x755fc635, (q31_t)0xccf0281f,
	(q31_t)0x7555bd4c, (q31_t)0xccd91d3d, (q31_t)0x754bafdc, (q31_t)0xccc21455, (q31_t)0x75419de7, (q31_t)0xccab0d65, (q31_t)0x7537876c, (q31_t)0xcc940871,
	(q31_t)0x752d6c6c, (q31_t)0xcc7d0578, (q31_t)0x75234ce8, (q31_t)0xcc66047b, (q31_t)0x751928e0, (q31_t)0xcc4f057c, (q31_t)0x750f0054, (q31_t)0xcc38087b,
	(q31_t)0x7504d345, (q31_t)0xcc210d79, (q31_t)0x74faa1b3, (q31_t)0xcc0a1477, (q31_t)0x74f06b9e, (q31_t)0xcbf31d75, (q31_t)0x74e63108, (q31_t)0xcbdc2876,
	(q31_t)0x74dbf1ef, (q31_t)0xcbc53579, (q31_t)0x74d1ae55, (q31_t)0xcbae447f, (q31_t)0x74c7663a, (q31_t)0xcb97558a, (q31_t)0x74bd199f, (q31_t)0xcb80689a,
	(q31_t)0x74b2c884, (q31_t)0xcb697db0, (q31_t)0x74a872e8, (q31_t)0xcb5294ce, (q31_t)0x749e18cd, (q31_t)0xcb3badf3, (q31_t)0x7493ba34, (q31_t)0xcb24c921,
	(q31_t)0x7489571c, (q31_t)0xcb0de658, (q31_t)0x747eef85, (q31_t)0xcaf7059a, (q31_t)0x74748371, (q31_t)0xcae026e8, (q31_t)0x746a12df, (q31_t)0xcac94a42,
	(q31_t)0x745f9dd1, (q31_t)0xcab26fa9, (q31_t)0x74552446, (q31_t)0xca9b971e, (q31_t)0x744aa63f, (q31_t)0xca84c0a3, (q31_t)0x744023bc, (q31_t)0xca6dec37,
	(q31_t)0x74359cbd, (q31_t)0xca5719db, (q31_t)0x742b1144, (q31_t)0xca404992, (q31_t)0x74208150, (q31_t)0xca297b5a, (q31_t)0x7415ece2, (q31_t)0xca12af37,
	(q31_t)0x740b53fb, (q31_t)0xc9fbe527, (q31_t)0x7400b69a, (q31_t)0xc9e51d2d, (q31_t)0x73f614c0, (q31_t)0xc9ce5748, (q31_t)0x73eb6e6e, (q31_t)0xc9b7937a,
	(q31_t)0x73e0c3a3, (q31_t)0xc9a0d1c5, (q31_t)0x73d61461, (q31_t)0xc98a1227, (q31_t)0x73cb60a8, (q31_t)0xc97354a4, (q31_t)0x73c0a878, (q31_t)0xc95c993a,
	(q31_t)0x73b5ebd1, (q31_t)0xc945dfec, (q31_t)0x73ab2ab4, (q31_t)0xc92f28ba, (q31_t)0x73a06522, (q31_t)0xc91873a5, (q31_t)0x73959b1b, (q31_t)0xc901c0ae,
	(q31_t)0x738acc9e, (q31_t)0xc8eb0fd6, (q31_t)0x737ff9ae, (q31_t)0xc8d4611d, (q31_t)0x73752249, (q31_t)0xc8bdb485, (q31_t)0x736a4671, (q31_t)0xc8a70a0e,
	(q31_t)0x735f6626, (q31_t)0xc89061ba, (q31_t)0x73548168, (q31_t)0xc879bb89, (q31_t)0x73499838, (q31_t)0xc863177b, (q31_t)0x733eaa96, (q31_t)0xc84c7593,
	(q31_t)0x7333b883, (q31_t)0xc835d5d0, (q31_t)0x7328c1ff, (q31_t)0xc81f3834, (q31_t)0x731dc70a, (q31_t)0xc8089cbf, (q31_t)0x7312c7a5, (q31_t)0xc7f20373,
	(q31_t)0x7307c3d0, (q31_t)0xc7db6c50, (q31_t)0x72fcbb8c, (q31_t)0xc7c4d757, (q31_t)0x72f1aed9, (q31_t)0xc7ae4489, (q31_t)0x72e69db7, (q31_t)0xc797b3e7,
	(q31_t)0x72db8828, (q31_t)0xc7812572, (q31_t)0x72d06e2b, (q31_t)0xc76a992a, (q31_t)0x72c54fc1, (q31_t)0xc7540f11, (q31_t)0x72ba2cea, (q31_t)0xc73d8727,
	(q31_t)0x72af05a7, (q31_t)0xc727016d, (q31_t)0x72a3d9f7, (q31_t)0xc7107de4, (q31_t)0x7298a9dd, (q31_t)0xc6f9fc8d, (q31_t)0x728d7557, (q31_t)0xc6e37d69,
	(q31_t)0x72823c67, (q31_t)0xc6cd0079, (q31_t)0x7276ff0d, (q31_t)0xc6b685bd, (q31_t)0x726bbd48, (q31_t)0xc6a00d37, (q31_t)0x7260771b, (q31_t)0xc68996e7,
	(q31_t)0x72552c85, (q31_t)0xc67322ce, (q31_t)0x7249dd86, (q31_t)0xc65cb0ed, (q31_t)0x723e8a20, (q31_t)0xc6464144, (q31_t)0x72333251, (q31_t)0xc62fd3d6,
	(q31_t)0x7227d61c, (q31_t)0xc61968a2, (q31_t)0x721c7580, (q31_t)0xc602ffaa, (q31_t)0x7211107e, (q31_t)0xc5ec98ee, (q31_t)0x7205a716, (q31_t)0xc5d6346f,
	(q31_t)0x71fa3949, (q31_t)0xc5bfd22e, (q31_t)0x71eec716, (q31_t)0xc5a9722c, (q31_t)0x71e35080, (q31_t)0xc593146a, (q31_t)0x71d7d585, (q31_t)0xc57cb8e9,
	(q31_t)0x71cc5626, (q31_t)0xc5665fa9, (q31_t)0x71c0d265, (q31_t)0xc55008ab, (q31_t)0x71b54a41, (q31_t)0xc539b3f1, (q31_t)0x71a9bdba, (q31_t)0xc523617a,
	(q31_t)0x719e2cd2, (q31_t)0xc50d1149, (q31_t)0x71929789, (q31_t)0xc4f6c35d, (q31_t)0x7186fdde, (q31_t)0xc4e077b8, (q31_t)0x717b5fd3, (q31_t)0xc4ca2e5b,
	(q31_t)0x716fbd68, (q31_t)0xc4b3e746, (q31_t)0x7164169d, (q31_t)0xc49da27a, (q31_t)0x71586b74, (q31_t)0xc4875ff9, (q31_t)0x714cbbeb, (q31_t)0xc4711fc2,
	(q31_t)0x71410805, (q31_t)0xc45ae1d7, (q31_t)0x71354fc0, (q31_t)0xc444a639, (q31_t)0x7129931f, (q31_t)0xc42e6ce8, (q31_t)0x711dd220, (q31_t)0xc41835e6,
	(q31_t)0x71120cc5, (q31_t)0xc4020133, (q31_t)0x7106430e, (q31_t)0xc3ebced0, (q31_t)0x70fa74fc, (q31_t)0xc3d59ebe, (q31_t)0x70eea28e, (q31_t)0xc3bf70fd,
	(q31_t)0x70e2cbc6, (q31_t)0xc3a94590, (q31_t)0x70d6f0a4, (q31_t)0xc3931c76, (q31_t)0x70cb1128, (q31_t)0xc37cf5b0, (q31_t)0x70bf2d53, (q31_t)0xc366d140,
	(q31_t)0x70b34525, (q31_t)0xc350af26, (q31_t)0x70a7589f, (q31_t)0xc33a8f62, (q31_t)0x709b67c0, (q31_t)0xc32471f7, (q31_t)0x708f728b, (q31_t)0xc30e56e4,
	(q31_t)0x708378ff, (q31_t)0xc2f83e2a, (q31_t)0x70777b1c, (q31_t)0xc2e227cb, (q31_t)0x706b78e3, (q31_t)0xc2cc13c7, (q31_t)0x705f7255, (q31_t)0xc2b6021f,
	(q31_t)0x70536771, (q31_t)0xc29ff2d4, (q31_t)0x70475839, (q31_t)0xc289e5e7, (q31_t)0x703b44ad, (q31_t)0xc273db58, (q31_t)0x702f2ccd, (q31_t)0xc25dd329,
	(q31_t)0x7023109a, (q31_t)0xc247cd5a, (q31_t)0x7016f014, (q31_t)0xc231c9ec, (q31_t)0x700acb3c, (q31_t)0xc21bc8e1, (q31_t)0x6ffea212, (q31_t)0xc205ca38,
	(q31_t)0x6ff27497, (q31_t)0xc1efcdf3, (q31_t)0x6fe642ca, (q31_t)0xc1d9d412, (q31_t)0x6fda0cae, (q31_t)0xc1c3dc97, (q31_t)0x6fcdd241, (q31_t)0xc1ade781,
	(q31_t)0x6fc19385, (q31_t)0xc197f4d4, (q31_t)0x6fb5507a, (q31_t)0xc182048d, (q31_t)0x6fa90921, (q31_t)0xc16c16b0, (q31_t)0x6f9cbd79, (q31_t)0xc1562b3d,
	(q31_t)0x6f906d84, (q31_t)0xc1404233, (q31_t)0x6f841942, (q31_t)0xc12a5b95, (q31_t)0x6f77c0b3, (q31_t)0xc1147764, (q31_t)0x6f6b63d8, (q31_t)0xc0fe959f,
	(q31_t)0x6f5f02b2, (q31_t)0xc0e8b648, (q31_t)0x6f529d40, (q31_t)0xc0d2d960, (q31_t)0x6f463383, (q31_t)0xc0bcfee7, (q31_t)0x6f39c57d, (q31_t)0xc0a726df,
	(q31_t)0x6f2d532c, (q31_t)0xc0915148, (q31_t)0x6f20dc92, (q31_t)0xc07b7e23, (q31_t)0x6f1461b0, (q31_t)0xc065ad70, (q31_t)0x6f07e285, (q31_t)0xc04fdf32,
	(q31_t)0x6efb5f12, (q31_t)0xc03a1368, (q31_t)0x6eeed758, (q31_t)0xc0244a14, (q31_t)0x6ee24b57, (q31_t)0xc00e8336, (q31_t)0x6ed5bb10, (q31_t)0xbff8bece,
	(q31_t)0x6ec92683, (q31_t)0xbfe2fcdf, (q31_t)0x6ebc8db0, (q31_t)0xbfcd3d69, (q31_t)0x6eaff099, (q31_t)0xbfb7806c, (q31_t)0x6ea34f3d, (q31_t)0xbfa1c5ea,
	(q31_t)0x6e96a99d, (q31_t)0xbf8c0de3, (q31_t)0x6e89ffb9, (q31_t)0xbf765858, (q31_t)0x6e7d5193, (q31_t)0xbf60a54a, (q31_t)0x6e709f2a, (q31_t)0xbf4af4ba,
	(q31_t)0x6e63e87f, (q31_t)0xbf3546a8, (q31_t)0x6e572d93, (q31_t)0xbf1f9b16, (q31_t)0x6e4a6e66, (q31_t)0xbf09f205, (q31_t)0x6e3daaf8, (q31_t)0xbef44b74,
	(q31_t)0x6e30e34a, (q31_t)0xbedea765, (q31_t)0x6e24175c, (q31_t)0xbec905d9, (q31_t)0x6e174730, (q31_t)0xbeb366d1, (q31_t)0x6e0a72c5, (q31_t)0xbe9dca4e,
	(q31_t)0x6dfd9a1c, (q31_t)0xbe88304f, (q31_t)0x6df0bd35, (q31_t)0xbe7298d7, (q31_t)0x6de3dc11, (q31_t)0xbe5d03e6, (q31_t)0x6dd6f6b1, (q31_t)0xbe47717c,
	(q31_t)0x6dca0d14, (q31_t)0xbe31e19b, (q31_t)0x6dbd1f3c, (q31_t)0xbe1c5444, (q31_t)0x6db02d29, (q31_t)0xbe06c977, (q31_t)0x6da336dc, (q31_t)0xbdf14135,
	(q31_t)0x6d963c54, (q31_t)0xbddbbb7f, (q31_t)0x6d893d93, (q31_t)0xbdc63856, (q31_t)0x6d7c3a98, (q31_t)0xbdb0b7bb, (q31_t)0x6d6f3365, (q31_t)0xbd9b39ad,
	(q31_t)0x6d6227fa, (q31_t)0xbd85be30, (q31_t)0x6d551858, (q31_t)0xbd704542, (q31_t)0x6d48047e, (q31_t)0xbd5acee5, (q31_t)0x6d3aec6e, (q31_t)0xbd455b1a,
	(q31_t)0x6d2dd027, (q31_t)0xbd2fe9e2, (q31_t)0x6d20afac, (q31_t)0xbd1a7b3d, (q31_t)0x6d138afb, (q31_t)0xbd050f2c, (q31_t)0x6d066215, (q31_t)0xbcefa5b0,
	(q31_t)0x6cf934fc, (q31_t)0xbcda3ecb, (q31_t)0x6cec03af, (q31_t)0xbcc4da7b, (q31_t)0x6cdece2f, (q31_t)0xbcaf78c4, (q31_t)0x6cd1947c, (q31_t)0xbc9a19a5,
	(q31_t)0x6cc45698, (q31_t)0xbc84bd1f, (q31_t)0x6cb71482, (q31_t)0xbc6f6333, (q31_t)0x6ca9ce3b, (q31_t)0xbc5a0be2, (q31_t)0x6c9c83c3, (q31_t)0xbc44b72c,
	(q31_t)0x6c8f351c, (q31_t)0xbc2f6513, (q31_t)0x6c81e245, (q31_t)0xbc1a1598, (q31_t)0x6c748b3f, (q31_t)0xbc04c8ba, (q31_t)0x6c67300b, (q31_t)0xbbef7e7c,
	(q31_t)0x6c59d0a9, (q31_t)0xbbda36dd, (q31_t)0x6c4c6d1a, (q31_t)0xbbc4f1df, (q31_t)0x6c3f055d, (q31_t)0xbbafaf82, (q31_t)0x6c319975, (q31_t)0xbb9a6fc7,
	(q31_t)0x6c242960, (q31_t)0xbb8532b0, (q31_t)0x6c16b521, (q31_t)0xbb6ff83c, (q31_t)0x6c093cb6, (q31_t)0xbb5ac06d, (q31_t)0x6bfbc021, (q31_t)0xbb458b43,
	(q31_t)0x6bee3f62, (q31_t)0xbb3058c0, (q31_t)0x6be0ba7b, (q31_t)0xbb1b28e4, (q31_t)0x6bd3316a, (q31_t)0xbb05fbb0, (q31_t)0x6bc5a431, (q31_t)0xbaf0d125,
	(q31_t)0x6bb812d1, (q31_t)0xbadba943, (q31_t)0x6baa7d49, (q31_t)0xbac6840c, (q31_t)0x6b9ce39b, (q31_t)0xbab16180, (q31_t)0x6b8f45c7, (q31_t)0xba9c41a0,
	(q31_t)0x6b81a3cd, (q31_t)0xba87246d, (q31_t)0x6b73fdae, (q31_t)0xba7209e7, (q31_t)0x6b66536b, (q31_t)0xba5cf210, (q31_t)0x6b58a503, (q31_t)0xba47dce8,
	(q31_t)0x6b4af279, (q31_t)0xba32ca71, (q31_t)0x6b3d3bcb, (q31_t)0xba1dbaaa, (q31_t)0x6b2f80fb, (q31_t)0xba08ad95, (q31_t)0x6b21c208, (q31_t)0xb9f3a332,
	(q31_t)0x6b13fef5, (q31_t)0xb9de9b83, (q31_t)0x6b0637c1, (q31_t)0xb9c99688, (q31_t)0x6af86c6c, (q31_t)0xb9b49442, (q31_t)0x6aea9cf8, (q31_t)0xb99f94b2,
	(q31_t)0x6adcc964, (q31_t)0xb98a97d8, (q31_t)0x6acef1b2, (q31_t)0xb9759db6, (q31_t)0x6ac115e2, (q31_t)0xb960a64c, (q31_t)0x6ab335f4, (q31_t)0xb94bb19b,
	(q31_t)0x6aa551e9, (q31_t)0xb936bfa4, (q31_t)0x6a9769c1, (q31_t)0xb921d067, (q31_t)0x6a897d7d, (q31_t)0xb90ce3e6, (q31_t)0x6a7b8d1e, (q31_t)0xb8f7fa21,
	(q31_t)0x6a6d98a4, (q31_t)0xb8e31319, (q31_t)0x6a5fa010, (q31_t)0xb8ce2ecf, (q31_t)0x6a51a361, (q31_t)0xb8b94d44, (q31_t)0x6a43a29a, (q31_t)0xb8a46e78,
	(q31_t)0x6a359db9, (q31_t)0xb88f926d, (q31_t)0x6a2794c1, (q31_t)0xb87ab922, (q31_t)0x6a1987b0, (q31_t)0xb865e299, (q31_t)0x6a0b7689, (q31_t)0xb8510ed4,
	(q31_t)0x69fd614a, (q31_t)0xb83c3dd1, (q31_t)0x69ef47f6, (q31_t)0xb8276f93, (q31_t)0x69e12a8c, (q31_t)0xb812a41a, (q31_t)0x69d3090e, (q31_t)0xb7fddb67,
	(q31_t)0x69c4e37a, (q31_t)0xb7e9157a, (q31_t)0x69b6b9d3, (q31_t)0xb7d45255, (q31_t)0x69a88c19, (q31_t)0xb7bf91f8, (q31_t)0x699a5a4c, (q31_t)0xb7aad465,
	(q31_t)0x698c246c, (q31_t)0xb796199b, (q31_t)0x697dea7b, (q31_t)0xb781619c, (q31_t)0x696fac78, (q31_t)0xb76cac69, (q31_t)0x69616a65, (q31_t)0xb757fa01,
	(q31_t)0x69532442, (q31_t)0xb7434a67, (q31_t)0x6944da10, (q31_t)0xb72e9d9b, (q31_t)0x69368bce, (q31_t)0xb719f39e, (q31_t)0x6928397e, (q31_t)0xb7054c6f,
	(q31_t)0x6919e320, (q31_t)0xb6f0a812, (q31_t)0x690b88b5, (q31_t)0xb6dc0685, (q31_t)0x68fd2a3d, (q31_t)0xb6c767ca, (q31_t)0x68eec7b9, (q31_t)0xb6b2cbe2,
	(q31_t)0x68e06129, (q31_t)0xb69e32cd, (q31_t)0x68d1f68f, (q31_t)0xb6899c8d, (q31_t)0x68c387e9, (q31_t)0xb6750921, (q31_t)0x68b5153a, (q31_t)0xb660788c,
	(q31_t)0x68a69e81, (q31_t)0xb64beacd, (q31_t)0x689823bf, (q31_t)0xb6375fe5, (q31_t)0x6889a4f6, (q31_t)0xb622d7d6, (q31_t)0x687b2224, (q31_t)0xb60e529f,
	(q31_t)0x686c9b4b, (q31_t)0xb5f9d043, (q31_t)0x685e106c, (q31_t)0xb5e550c1, (q31_t)0x684f8186, (q31_t)0xb5d0d41a, (q31_t)0x6840ee9b, (q31_t)0xb5bc5a50,
	(q31_t)0x683257ab, (q31_t)0xb5a7e362, (q31_t)0x6823bcb7, (q31_t)0xb5936f53, (q31_t)0x68151dbe, (q31_t)0xb57efe22, (q31_t)0x68067ac3, (q31_t)0xb56a8fd0,
	(q31_t)0x67f7d3c5, (q31_t)0xb556245e, (q31_t)0x67e928c5, (q31_t)0xb541bbcd, (q31_t)0x67da79c3, (q31_t)0xb52d561e, (q31_t)0x67cbc6c0, (q31_t)0xb518f351,
	(q31_t)0x67bd0fbd, (q31_t)0xb5049368, (q31_t)0x67ae54ba, (q31_t)0xb4f03663, (q31_t)0x679f95b7, (q31_t)0xb4dbdc42, (q31_t)0x6790d2b6, (q31_t)0xb4c78507,
	(q31_t)0x67820bb7, (q31_t)0xb4b330b3, (q31_t)0x677340ba, (q31_t)0xb49edf45, (q31_t)0x676471c0, (q31_t)0xb48a90c0, (q31_t)0x67559eca, (q31_t)0xb4764523,
	(q31_t)0x6746c7d8, (q31_t)0xb461fc70, (q31_t)0x6737ecea, (q31_t)0xb44db6a8, (q31_t)0x67290e02, (q31_t)0xb43973ca, (q31_t)0x671a2b20, (q31_t)0xb42533d8,
	(q31_t)0x670b4444, (q31_t)0xb410f6d3, (q31_t)0x66fc596f, (q31_t)0xb3fcbcbb, (q31_t)0x66ed6aa1, (q31_t)0xb3e88592, (q31_t)0x66de77dc, (q31_t)0xb3d45157,
	(q31_t)0x66cf8120, (q31_t)0xb3c0200c, (q31_t)0x66c0866d, (q31_t)0xb3abf1b2, (q31_t)0x66b187c3, (q31_t)0xb397c649, (q31_t)0x66a28524, (q31_t)0xb3839dd3,
	(q31_t)0x66937e91, (q31_t)0xb36f784f, (q31_t)0x66847408, (q31_t)0xb35b55bf, (q31_t)0x6675658c, (q31_t)0xb3473623, (q31_t)0x6666531d, (q31_t)0xb333197c,
	(q31_t)0x66573cbb, (q31_t)0xb31effcc, (q31_t)0x66482267, (q31_t)0xb30ae912, (q31_t)0x66390422, (q31_t)0xb2f6d550, (q31_t)0x6629e1ec, (q31_t)0xb2e2c486,
	(q31_t)0x661abbc5, (q31_t)0xb2ceb6b5, (q31_t)0x660b91af, (q31_t)0xb2baabde, (q31_t)0x65fc63a9, (q31_t)0xb2a6a402, (q31_t)0x65ed31b5, (q31_t)0xb2929f21,
	(q31_t)0x65ddfbd3, (q31_t)0xb27e9d3c, (q31_t)0x65cec204, (q31_t)0xb26a9e54, (q31_t)0x65bf8447, (q31_t)0xb256a26a, (q31_t)0x65b0429f, (q31_t)0xb242a97e,
	(q31_t)0x65a0fd0b, (q31_t)0xb22eb392, (q31_t)0x6591b38c, (q31_t)0xb21ac0a6, (q31_t)0x65826622, (q31_t)0xb206d0ba, (q31_t)0x657314cf, (q31_t)0xb1f2e3d0,
	(q31_t)0x6563bf92, (q31_t)0xb1def9e9, (q31_t)0x6554666d, (q31_t)0xb1cb1304, (q31_t)0x6545095f, (q31_t)0xb1b72f23, (q31_t)0x6535a86b, (q31_t)0xb1a34e47,
	(q31_t)0x6526438f, (q31_t)0xb18f7071, (q31_t)0x6516dacd, (q31_t)0xb17b95a0, (q31_t)0x65076e25, (q31_t)0xb167bdd7, (q31_t)0x64f7fd98, (q31_t)0xb153e915,
	(q31_t)0x64e88926, (q31_t)0xb140175b, (q31_t)0x64d910d1, (q31_t)0xb12c48ab, (q31_t)0x64c99498, (q31_t)0xb1187d05, (q31_t)0x64ba147d, (q31_t)0xb104b46a,
	(q31_t)0x64aa907f, (q31_t)0xb0f0eeda, (q31_t)0x649b08a0, (q31_t)0xb0dd2c56, (q31_t)0x648b7ce0, (q31_t)0xb0c96ce0, (q31_t)0x647bed3f, (q31_t)0xb0b5b077,
	(q31_t)0x646c59bf, (q31_t)0xb0a1f71d, (q31_t)0x645cc260, (q31_t)0xb08e40d2, (q31_t)0x644d2722, (q31_t)0xb07a8d97, (q31_t)0x643d8806, (q31_t)0xb066dd6d,
	(q31_t)0x642de50d, (q31_t)0xb0533055, (q31_t)0x641e3e38, (q31_t)0xb03f864f, (q31_t)0x640e9386, (q31_t)0xb02bdf5c, (q31_t)0x63fee4f8, (q31_t)0xb0183b7d,
	(q31_t)0x63ef3290, (q31_t)0xb0049ab3, (q31_t)0x63df7c4d, (q31_t)0xaff0fcfe, (q31_t)0x63cfc231, (q31_t)0xafdd625f, (q31_t)0x63c0043b, (q31_t)0xafc9cad7,
	(q31_t)0x63b0426d, (q31_t)0xafb63667, (q31_t)0x63a07cc7, (q31_t)0xafa2a50f, (q31_t)0x6390b34a, (q31_t)0xaf8f16d1, (q31_t)0x6380e5f6, (q31_t)0xaf7b8bac,
	(q31_t)0x637114cc, (q31_t)0xaf6803a2, (q31_t)0x63613fcd, (q31_t)0xaf547eb3, (q31_t)0x635166f9, (q31_t)0xaf40fce1, (q31_t)0x63418a50, (q31_t)0xaf2d7e2b,
	(q31_t)0x6331a9d4, (q31_t)0xaf1a0293, (q31_t)0x6321c585, (q31_t)0xaf068a1a, (q31_t)0x6311dd64, (q31_t)0xaef314c0, (q31_t)0x6301f171, (q31_t)0xaedfa285,
	(q31_t)0x62f201ac, (q31_t)0xaecc336c, (q31_t)0x62e20e17, (q31_t)0xaeb8c774, (q31_t)0x62d216b3, (q31_t)0xaea55e9e, (q31_t)0x62c21b7e, (q31_t)0xae91f8eb,
	(q31_t)0x62b21c7b, (q31_t)0xae7e965b, (q31_t)0x62a219aa, (q31_t)0xae6b36f0, (q31_t)0x6292130c, (q31_t)0xae57daab, (q31_t)0x628208a1, (q31_t)0xae44818b,
	(q31_t)0x6271fa69, (q31_t)0xae312b92, (q31_t)0x6261e866, (q31_t)0xae1dd8c0, (q31_t)0x6251d298, (q31_t)0xae0a8916, (q31_t)0x6241b8ff, (q31_t)0xadf73c96,
	(q31_t)0x62319b9d, (q31_t)0xade3f33e, (q31_t)0x62217a72, (q31_t)0xadd0ad12, (q31_t)0x6211557e, (q31_t)0xadbd6a10, (q31_t)0x62012cc2, (q31_t)0xadaa2a3b,
	(q31_t)0x61f1003f, (q31_t)0xad96ed92, (q31_t)0x61e0cff5, (q31_t)0xad83b416, (q31_t)0x61d09be5, (q31_t)0xad707dc8, (q31_t)0x61c06410, (q31_t)0xad5d4aaa,
	(q31_t)0x61b02876, (q31_t)0xad4a1aba, (q31_t)0x619fe918, (q31_t)0xad36edfc, (q31_t)0x618fa5f7, (q31_t)0xad23c46e, (q31_t)0x617f5f12, (q31_t)0xad109e12,
	(q31_t)0x616f146c, (q31_t)0xacfd7ae8, (q31_t)0x615ec603, (q31_t)0xacea5af2, (q31_t)0x614e73da, (q31_t)0xacd73e30, (q31_t)0x613e1df0, (q31_t)0xacc424a3,
	(q31_t)0x612dc447, (q31_t)0xacb10e4b, (q31_t)0x611d66de, (q31_t)0xac9dfb29, (q31_t)0x610d05b7, (q31_t)0xac8aeb3e, (q31_t)0x60fca0d2, (q31_t)0xac77de8b,
	(q31_t)0x60ec3830, (q31_t)0xac64d510, (q31_t)0x60dbcbd1, (q31_t)0xac51cecf, (q31_t)0x60cb5bb7, (q31_t)0xac3ecbc7, (q31_t)0x60bae7e1, (q31_t)0xac2bcbfa,
	(q31_t)0x60aa7050, (q31_t)0xac18cf69, (q31_t)0x6099f505, (q31_t)0xac05d613, (q31_t)0x60897601, (q31_t)0xabf2dffb, (q31_t)0x6078f344, (q31_t)0xabdfed1f,
	(q31_t)0x60686ccf, (q31_t)0xabccfd83, (q31_t)0x6057e2a2, (q31_t)0xabba1125, (q31_t)0x604754bf, (q31_t)0xaba72807, (q31_t)0x6036c325, (q31_t)0xab944229,
	(q31_t)0x60262dd6, (q31_t)0xab815f8d, (q31_t)0x601594d1, (q31_t)0xab6e8032, (q31_t)0x6004f819, (q31_t)0xab5ba41a, (q31_t)0x5ff457ad, (q31_t)0xab48cb46,
	(q31_t)0x5fe3b38d, (q31_t)0xab35f5b5, (q31_t)0x5fd30bbc, (q31_t)0xab23236a, (q31_t)0x5fc26038, (q31_t)0xab105464, (q31_t)0x5fb1b104, (q31_t)0xaafd88a4,
	(q31_t)0x5fa0fe1f, (q31_t)0xaaeac02c, (q31_t)0x5f90478a, (q31_t)0xaad7fafb, (q31_t)0x5f7f8d46, (q31_t)0xaac53912, (q31_t)0x5f6ecf53, (q31_t)0xaab27a73,
	(q31_t)0x5f5e0db3, (q31_t)0xaa9fbf1e, (q31_t)0x5f4d4865, (q31_t)0xaa8d0713, (q31_t)0x5f3c7f6b, (q31_t)0xaa7a5253, (q31_t)0x5f2bb2c5, (q31_t)0xaa67a0e0,
	(q31_t)0x5f1ae274, (q31_t)0xaa54f2ba, (q31_t)0x5f0a0e77, (q31_t)0xaa4247e1, (q31_t)0x5ef936d1, (q31_t)0xaa2fa056, (q31_t)0x5ee85b82, (q31_t)0xaa1cfc1a,
	(q31_t)0x5ed77c8a, (q31_t)0xaa0a5b2e, (q31_t)0x5ec699e9, (q31_t)0xa9f7bd92, (q31_t)0x5eb5b3a2, (q31_t)0xa9e52347, (q31_t)0x5ea4c9b3, (q31_t)0xa9d28c4e,
	(q31_t)0x5e93dc1f, (q31_t)0xa9bff8a8, (q31_t)0x5e82eae5, (q31_t)0xa9ad6855, (q31_t)0x5e71f606, (q31_t)0xa99adb56, (q31_t)0x5e60fd84, (q31_t)0xa98851ac,
	(q31_t)0x5e50015d, (q31_t)0xa975cb57, (q31_t)0x5e3f0194, (q31_t)0xa9634858, (q31_t)0x5e2dfe29, (q31_t)0xa950c8b0, (q31_t)0x5e1cf71c, (q31_t)0xa93e4c5f,
	(q31_t)0x5e0bec6e, (q31_t)0xa92bd367, (q31_t)0x5dfade20, (q31_t)0xa9195dc7, (q31_t)0x5de9cc33, (q31_t)0xa906eb82, (q31_t)0x5dd8b6a7, (q31_t)0xa8f47c97,
	(q31_t)0x5dc79d7c, (q31_t)0xa8e21106, (q31_t)0x5db680b4, (q31_t)0xa8cfa8d2, (q31_t)0x5da5604f, (q31_t)0xa8bd43fa, (q31_t)0x5d943c4e, (q31_t)0xa8aae280,
	(q31_t)0x5d8314b1, (q31_t)0xa8988463, (q31_t)0x5d71e979, (q31_t)0xa88629a5, (q31_t)0x5d60baa7, (q31_t)0xa873d246, (q31_t)0x5d4f883b, (q31_t)0xa8617e48,
	(q31_t)0x5d3e5237, (q31_t)0xa84f2daa, (q31_t)0x5d2d189a, (q31_t)0xa83ce06e, (q31_t)0x5d1bdb65, (q31_t)0xa82a9693, (q31_t)0x5d0a9a9a, (q31_t)0xa818501c,
	(q31_t)0x5cf95638, (q31_t)0xa8060d08, (q31_t)0x5ce80e41, (q31_t)0xa7f3cd59, (q31_t)0x5cd6c2b5, (q31_t)0xa7e1910f, (q31_t)0x5cc57394, (q31_t)0xa7cf582a,
	(q31_t)0x5cb420e0, (q31_t)0xa7bd22ac, (q31_t)0x5ca2ca99, (q31_t)0xa7aaf094, (q31_t)0x5c9170bf, (q31_t)0xa798c1e5, (q31_t)0x5c801354, (q31_t)0xa786969e,
	(q31_t)0x5c6eb258, (q31_t)0xa7746ec0, (q31_t)0x5c5d4dcc, (q31_t)0xa7624a4d, (q31_t)0x5c4be5b0, (q31_t)0xa7502943, (q31_t)0x5c3a7a05, (q31_t)0xa73e0ba5,
	(q31_t)0x5c290acc, (q31_t)0xa72bf174, (q31_t)0x5c179806, (q31_t)0xa719daae, (q31_t)0x5c0621b2, (q31_t)0xa707c757, (q31_t)0x5bf4a7d2, (q31_t)0xa6f5b76d,
	(q31_t)0x5be32a67, (q31_t)0xa6e3aaf2, (q31_t)0x5bd1a971, (q31_t)0xa6d1a1e7, (q31_t)0x5bc024f0, (q31_t)0xa6bf9c4b, (q31_t)0x5bae9ce7, (q31_t)0xa6ad9a21,
	(q31_t)0x5b9d1154, (q31_t)0xa69b9b68, (q31_t)0x5b8b8239, (q31_t)0xa689a022, (q31_t)0x5b79ef96, (q31_t)0xa677a84e, (q31_t)0x5b68596d, (q31_t)0xa665b3ee,
	(q31_t)0x5b56bfbd, (q31_t)0xa653c303, (q31_t)0x5b452288, (q31_t)0xa641d58c, (q31_t)0x5b3381ce, (q31_t)0xa62feb8b, (q31_t)0x5b21dd90, (q31_t)0xa61e0501,
	(q31_t)0x5b1035cf, (q31_t)0xa60c21ee, (q31_t)0x5afe8a8b, (q31_t)0xa5fa4252, (q31_t)0x5aecdbc5, (q31_t)0xa5e8662f, (q31_t)0x5adb297d, (q31_t)0xa5d68d85,
	(q31_t)0x5ac973b5, (q31_t)0xa5c4b855, (q31_t)0x5ab7ba6c, (q31_t)0xa5b2e6a0, (q31_t)0x5aa5fda5, (q31_t)0xa5a11866, (q31_t)0x5a943d5e, (q31_t)0xa58f4da8,
	(q31_t)0x5a82799a, (q31_t)0xa57d8666, (q31_t)0x5a70b258, (q31_t)0xa56bc2a2, (q31_t)0x5a5ee79a, (q31_t)0xa55a025b, (q31_t)0x5a4d1960, (q31_t)0xa5484594,
	(q31_t)0x5a3b47ab, (q31_t)0xa5368c4b, (q31_t)0x5a29727b, (q31_t)0xa524d683, (q31_t)0x5a1799d1, (q31_t)0xa513243b, (q31_t)0x5a05bdae, (q31_t)0xa5017575,
	(q31_t)0x59f3de12, (q31_t)0xa4efca31, (q31_t)0x59e1faff, (q31_t)0xa4de2270, (q31_t)0x59d01475, (q31_t)0xa4cc7e32, (q31_t)0x59be2a74, (q31_t)0xa4badd78,
	(q31_t)0x59ac3cfd, (q31_t)0xa4a94043, (q31_t)0x599a4c12, (q31_t)0xa497a693, (q31_t)0x598857b2, (q31_t)0xa486106a, (q31_t)0x59765fde, (q31_t)0xa4747dc7,
	(q31_t)0x59646498, (q31_t)0xa462eeac, (q31_t)0x595265df, (q31_t)0xa4516319, (q31_t)0x594063b5, (q31_t)0xa43fdb10, (q31_t)0x592e5e19, (q31_t)0xa42e568f,
	(q31_t)0x591c550e, (q31_t)0xa41cd599, (q31_t)0x590a4893, (q31_t)0xa40b582e, (q31_t)0x58f838a9, (q31_t)0xa3f9de4e, (q31_t)0x58e62552, (q31_t)0xa3e867fa,
	(q31_t)0x58d40e8c, (q31_t)0xa3d6f534, (q31_t)0x58c1f45b, (q31_t)0xa3c585fb, (q31_t)0x58afd6bd, (q31_t)0xa3b41a50, (q31_t)0x589db5b3, (q31_t)0xa3a2b234,
	(q31_t)0x588b9140, (q31_t)0xa3914da8, (q31_t)0x58796962, (q31_t)0xa37fecac, (q31_t)0x58673e1b, (q31_t)0xa36e8f41, (q31_t)0x58550f6c, (q31_t)0xa35d3567,
	(q31_t)0x5842dd54, (q31_t)0xa34bdf20, (q31_t)0x5830a7d6, (q31_t)0xa33a8c6c, (q31_t)0x581e6ef1, (q31_t)0xa3293d4b, (q31_t)0x580c32a7, (q31_t)0xa317f1bf,
	(q31_t)0x57f9f2f8, (q31_t)0xa306a9c8, (q31_t)0x57e7afe4, (q31_t)0xa2f56566, (q31_t)0x57d5696d, (q31_t)0xa2e4249b, (q31_t)0x57c31f92, (q31_t)0xa2d2e766,
	(q31_t)0x57b0d256, (q31_t)0xa2c1adc9, (q31_t)0x579e81b8, (q31_t)0xa2b077c5, (q31_t)0x578c2dba, (q31_t)0xa29f4559, (q31_t)0x5779d65b, (q31_t)0xa28e1687,
	(q31_t)0x57677b9d, (q31_t)0xa27ceb4f, (q31_t)0x57551d80, (q31_t)0xa26bc3b2, (q31_t)0x5742bc06, (q31_t)0xa25a9fb1, (q31_t)0x5730572e, (q31_t)0xa2497f4c,
	(q31_t)0x571deefa, (q31_t)0xa2386284, (q31_t)0x570b8369, (q31_t)0xa2274959, (q31_t)0x56f9147e, (q31_t)0xa21633cd, (q31_t)0x56e6a239, (q31_t)0xa20521e0,
	(q31_t)0x56d42c99, (q31_t)0xa1f41392, (q31_t)0x56c1b3a1, (q31_t)0xa1e308e4, (q31_t)0x56af3750, (q31_t)0xa1d201d7, (q31_t)0x569cb7a8, (q31_t)0xa1c0fe6c,
	(q31_t)0x568a34a9, (q31_t)0xa1affea3, (q31_t)0x5677ae54, (q31_t)0xa19f027c, (q31_t)0x566524aa, (q31_t)0xa18e09fa, (q31_t)0x565297ab, (q31_t)0xa17d151b,
	(q31_t)0x56400758, (q31_t)0xa16c23e1, (q31_t)0x562d73b2, (q31_t)0xa15b364d, (q31_t)0x561adcb9, (q31_t)0xa14a4c5e, (q31_t)0x5608426e, (q31_t)0xa1396617,
	(q31_t)0x55f5a4d2, (q31_t)0xa1288376, (q31_t)0x55e303e6, (q31_t)0xa117a47e, (q31_t)0x55d05faa, (q31_t)0xa106c92f, (q31_t)0x55bdb81f, (q31_t)0xa0f5f189,
	(q31_t)0x55ab0d46, (q31_t)0xa0e51d8c, (q31_t)0x55985f20, (q31_t)0xa0d44d3b, (q31_t)0x5585adad, (q31_t)0xa0c38095, (q31_t)0x5572f8ed, (q31_t)0xa0b2b79b,
	(q31_t)0x556040e2, (q31_t)0xa0a1f24d, (q31_t)0x554d858d, (q31_t)0xa09130ad, (q31_t)0x553ac6ee, (q31_t)0xa08072ba, (q31_t)0x55280505, (q31_t)0xa06fb876,
	(q31_t)0x55153fd4, (q31_t)0xa05f01e1, (q31_t)0x5502775c, (q31_t)0xa04e4efc, (q31_t)0x54efab9c, (q31_t)0xa03d9fc8, (q31_t)0x54dcdc96, (q31_t)0xa02cf444,
	(q31_t)0x54ca0a4b, (q31_t)0xa01c4c73, (q31_t)0x54b734ba, (q31_t)0xa00ba853, (q31_t)0x54a45be6, (q31_t)0x9ffb07e7, (q31_t)0x54917fce, (q31_t)0x9fea6b2f,
	(q31_t)0x547ea073, (q31_t)0x9fd9d22a, (q31_t)0x546bbdd7, (q31_t)0x9fc93cdb, (q31_t)0x5458d7f9, (q31_t)0x9fb8ab41, (q31_t)0x5445eedb, (q31_t)0x9fa81d5e,
	(q31_t)0x5433027d, (q31_t)0x9f979331, (q31_t)0x542012e1, (q31_t)0x9f870cbc, (q31_t)0x540d2005, (q31_t)0x9f7689ff, (q31_t)0x53fa29ed, (q31_t)0x9f660afb,
	(q31_t)0x53e73097, (q31_t)0x9f558fb0, (q31_t)0x53d43406, (q31_t)0x9f45181f, (q31_t)0x53c13439, (q31_t)0x9f34a449, (q31_t)0x53ae3131, (q31_t)0x9f24342f,
	(q31_t)0x539b2af0, (q31_t)0x9f13c7d0, (q31_t)0x53882175, (q31_t)0x9f035f2e, (q31_t)0x537514c2, (q31_t)0x9ef2fa49, (q31_t)0x536204d7, (q31_t)0x9ee29922,
	(q31_t)0x534ef1b5, (q31_t)0x9ed23bb9, (q31_t)0x533bdb5d, (q31_t)0x9ec1e210, (q31_t)0x5328c1d0, (q31_t)0x9eb18c26, (q31_t)0x5315a50e, (q31_t)0x9ea139fd,
	(q31_t)0x53028518, (q31_t)0x9e90eb94, (q31_t)0x52ef61ee, (q31_t)0x9e80a0ee, (q31_t)0x52dc3b92, (q31_t)0x9e705a09, (q31_t)0x52c91204, (q31_t)0x9e6016e8,
	(q31_t)0x52b5e546, (q31_t)0x9e4fd78a, (q31_t)0x52a2b556, (q31_t)0x9e3f9bf0, (q31_t)0x528f8238, (q31_t)0x9e2f641b, (q31_t)0x527c4bea, (q31_t)0x9e1f300b,
	(q31_t)0x5269126e, (q31_t)0x9e0effc1, (q31_t)0x5255d5c5, (q31_t)0x9dfed33e, (q31_t)0x524295f0, (q31_t)0x9deeaa82, (q31_t)0x522f52ee, (q31_t)0x9dde858e,
	(q31_t)0x521c0cc2, (q31_t)0x9dce6463, (q31_t)0x5208c36a, (q31_t)0x9dbe4701, (q31_t)0x51f576ea, (q31_t)0x9dae2d68, (q31_t)0x51e22740, (q31_t)0x9d9e179a,
	(q31_t)0x51ced46e, (q31_t)0x9d8e0597, (q31_t)0x51bb7e75, (q31_t)0x9d7df75f, (q31_t)0x51a82555, (q31_t)0x9d6decf4, (q31_t)0x5194c910, (q31_t)0x9d5de656,
	(q31_t)0x518169a5, (q31_t)0x9d4de385, (q31_t)0x516e0715, (q31_t)0x9d3de482, (q31_t)0x515aa162, (q31_t)0x9d2de94d, (q31_t)0x5147388c, (q31_t)0x9d1df1e9,
	(q31_t)0x5133cc94, (q31_t)0x9d0dfe54, (q31_t)0x51205d7b, (q31_t)0x9cfe0e8f, (q31_t)0x510ceb40, (q31_t)0x9cee229c, (q31_t)0x50f975e6, (q31_t)0x9cde3a7b,
	(q31_t)0x50e5fd6d, (q31_t)0x9cce562c, (q31_t)0x50d281d5, (q31_t)0x9cbe75b0, (q31_t)0x50bf031f, (q31_t)0x9cae9907, (q31_t)0x50ab814d, (q31_t)0x9c9ec033,
	(q31_t)0x5097fc5e, (q31_t)0x9c8eeb34, (q31_t)0x50847454, (q31_t)0x9c7f1a0a, (q31_t)0x5070e92f, (q31_t)0x9c6f4cb6, (q31_t)0x505d5af1, (q31_t)0x9c5f8339,
	(q31_t)0x5049c999, (q31_t)0x9c4fbd93, (q31_t)0x50363529, (q31_t)0x9c3ffbc5, (q31_t)0x50229da1, (q31_t)0x9c303dcf, (q31_t)0x500f0302, (q31_t)0x9c2083b3,
	(q31_t)0x4ffb654d, (q31_t)0x9c10cd70, (q31_t)0x4fe7c483, (q31_t)0x9c011b08, (q31_t)0x4fd420a4, (q31_t)0x9bf16c7a, (q31_t)0x4fc079b1, (q31_t)0x9be1c1c8,
	(q31_t)0x4faccfab, (q31_t)0x9bd21af3, (q31_t)0x4f992293, (q31_t)0x9bc277fa, (q31_t)0x4f857269, (q31_t)0x9bb2d8de, (q31_t)0x4f71bf2e, (q31_t)0x9ba33da0,
	(q31_t)0x4f5e08e3, (q31_t)0x9b93a641, (q31_t)0x4f4a4f89, (q31_t)0x9b8412c1, (q31_t)0x4f369320, (q31_t)0x9b748320, (q31_t)0x4f22d3aa, (q31_t)0x9b64f760,
	(q31_t)0x4f0f1126, (q31_t)0x9b556f81, (q31_t)0x4efb4b96, (q31_t)0x9b45eb83, (q31_t)0x4ee782fb, (q31_t)0x9b366b68, (q31_t)0x4ed3b755, (q31_t)0x9b26ef2f,
	(q31_t)0x4ebfe8a5, (q31_t)0x9b1776da, (q31_t)0x4eac16eb, (q31_t)0x9b080268, (q31_t)0x4e984229, (q31_t)0x9af891db, (q31_t)0x4e846a60, (q31_t)0x9ae92533,
	(q31_t)0x4e708f8f, (q31_t)0x9ad9bc71, (q31_t)0x4e5cb1b9, (q31_t)0x9aca5795, (q31_t)0x4e48d0dd, (q31_t)0x9abaf6a1, (q31_t)0x4e34ecfc, (q31_t)0x9aab9993,
	(q31_t)0x4e210617, (q31_t)0x9a9c406e, (q31_t)0x4e0d1c30, (q31_t)0x9a8ceb31, (q31_t)0x4df92f46, (q31_t)0x9a7d99de, (q31_t)0x4de53f5a, (q31_t)0x9a6e4c74,
	(q31_t)0x4dd14c6e, (q31_t)0x9a5f02f5, (q31_t)0x4dbd5682, (q31_t)0x9a4fbd61, (q31_t)0x4da95d96, (q31_t)0x9a407bb9, (q31_t)0x4d9561ac, (q31_t)0x9a313dfc,
	(q31_t)0x4d8162c4, (q31_t)0x9a22042d, (q31_t)0x4d6d60df, (q31_t)0x9a12ce4b, (q31_t)0x4d595bfe, (q31_t)0x9a039c57, (q31_t)0x4d455422, (q31_t)0x99f46e51,
	(q31_t)0x4d31494b, (q31_t)0x99e5443b, (q31_t)0x4d1d3b7a, (q31_t)0x99d61e14, (q31_t)0x4d092ab0, (q31_t)0x99c6fbde, (q31_t)0x4cf516ee, (q31_t)0x99b7dd99,
	(q31_t)0x4ce10034, (q31_t)0x99a8c345, (q31_t)0x4ccce684, (q31_t)0x9999ace3, (q31_t)0x4cb8c9dd, (q31_t)0x998a9a74, (q31_t)0x4ca4aa41, (q31_t)0x997b8bf8,
	(q31_t)0x4c9087b1, (q31_t)0x996c816f, (q31_t)0x4c7c622d, (q31_t)0x995d7adc, (q31_t)0x4c6839b7, (q31_t)0x994e783d, (q31_t)0x4c540e4e, (q31_t)0x993f7993,
	(q31_t)0x4c3fdff4, (q31_t)0x99307ee0, (q31_t)0x4c2baea9, (q31_t)0x99218824, (q31_t)0x4c177a6e, (q31_t)0x9912955f, (q31_t)0x4c034345, (q31_t)0x9903a691,
	(q31_t)0x4bef092d, (q31_t)0x98f4bbbc, (q31_t)0x4bdacc28, (q31_t)0x98e5d4e0, (q31_t)0x4bc68c36, (q31_t)0x98d6f1fe, (q31_t)0x4bb24958, (q31_t)0x98c81316,
	(q31_t)0x4b9e0390, (q31_t)0x98b93828, (q31_t)0x4b89badd, (q31_t)0x98aa6136, (q31_t)0x4b756f40, (q31_t)0x989b8e40, (q31_t)0x4b6120bb, (q31_t)0x988cbf46,
	(q31_t)0x4b4ccf4d, (q31_t)0x987df449, (q31_t)0x4b387af9, (q31_t)0x986f2d4a, (q31_t)0x4b2423be, (q31_t)0x98606a49, (q31_t)0x4b0fc99d, (q31_t)0x9851ab46,
	(q31_t)0x4afb6c98, (q31_t)0x9842f043, (q31_t)0x4ae70caf, (q31_t)0x98343940, (q31_t)0x4ad2a9e2, (q31_t)0x9825863d, (q31_t)0x4abe4433, (q31_t)0x9816d73b,
	(q31_t)0x4aa9dba2, (q31_t)0x98082c3b, (q31_t)0x4a957030, (q31_t)0x97f9853d, (q31_t)0x4a8101de, (q31_t)0x97eae242, (q31_t)0x4a6c90ad, (q31_t)0x97dc4349,
	(q31_t)0x4a581c9e, (q31_t)0x97cda855, (q31_t)0x4a43a5b0, (q31_t)0x97bf1165, (q31_t)0x4a2f2be6, (q31_t)0x97b07e7a, (q31_t)0x4a1aaf3f, (q31_t)0x97a1ef94,
	(q31_t)0x4a062fbd, (q31_t)0x979364b5, (q31_t)0x49f1ad61, (q31_t)0x9784dddc, (q31_t)0x49dd282a, (q31_t)0x97765b0a, (q31_t)0x49c8a01b, (q31_t)0x9767dc41,
	(q31_t)0x49b41533, (q31_t)0x9759617f, (q31_t)0x499f8774, (q31_t)0x974aeac6, (q31_t)0x498af6df, (q31_t)0x973c7817, (q31_t)0x49766373, (q31_t)0x972e0971,
	(q31_t)0x4961cd33, (q31_t)0x971f9ed7, (q31_t)0x494d341e, (q31_t)0x97113847, (q31_t)0x49389836, (q31_t)0x9702d5c3, (q31_t)0x4923f97b, (q31_t)0x96f4774b,
	(q31_t)0x490f57ee, (q31_t)0x96e61ce0, (q31_t)0x48fab391, (q31_t)0x96d7c682, (q31_t)0x48e60c62, (q31_t)0x96c97432, (q31_t)0x48d16265, (q31_t)0x96bb25f0,
	(q31_t)0x48bcb599, (q31_t)0x96acdbbe, (q31_t)0x48a805ff, (q31_t)0x969e959b, (q31_t)0x48935397, (q31_t)0x96905388, (q31_t)0x487e9e64, (q31_t)0x96821585,
	(q31_t)0x4869e665, (q31_t)0x9673db94, (q31_t)0x48552b9b, (q31_t)0x9665a5b4, (q31_t)0x48406e08, (q31_t)0x965773e7, (q31_t)0x482badab, (q31_t)0x9649462d,
	(q31_t)0x4816ea86, (q31_t)0x963b1c86, (q31_t)0x48022499, (q31_t)0x962cf6f2, (q31_t)0x47ed5be6, (q31_t)0x961ed574, (q31_t)0x47d8906d, (q31_t)0x9610b80a,
	(q31_t)0x47c3c22f, (q31_t)0x96029eb6, (q31_t)0x47aef12c, (q31_t)0x95f48977, (q31_t)0x479a1d67, (q31_t)0x95e67850, (q31_t)0x478546de, (q31_t)0x95d86b3f,
	(q31_t)0x47706d93, (q31_t)0x95ca6247, (q31_t)0x475b9188, (q31_t)0x95bc5d66, (q31_t)0x4746b2bc, (q31_t)0x95ae5c9f, (q31_t)0x4731d131, (q31_t)0x95a05ff0,
	(q31_t)0x471cece7, (q31_t)0x9592675c, (q31_t)0x470805df, (q31_t)0x958472e2, (q31_t)0x46f31c1a, (q31_t)0x95768283, (q31_t)0x46de2f99, (q31_t)0x9568963f,
	(q31_t)0x46c9405c, (q31_t)0x955aae17, (q31_t)0x46b44e65, (q31_t)0x954cca0c, (q31_t)0x469f59b4, (q31_t)0x953eea1e, (q31_t)0x468a624a, (q31_t)0x95310e4e,
	(q31_t)0x46756828, (q31_t)0x9523369c, (q31_t)0x46606b4e, (q31_t)0x95156308, (q31_t)0x464b6bbe, (q31_t)0x95079394, (q31_t)0x46366978, (q31_t)0x94f9c83f,
	(q31_t)0x4621647d, (q31_t)0x94ec010b, (q31_t)0x460c5cce, (q31_t)0x94de3df8, (q31_t)0x45f7526b, (q31_t)0x94d07f05, (q31_t)0x45e24556, (q31_t)0x94c2c435,
	(q31_t)0x45cd358f, (q31_t)0x94b50d87, (q31_t)0x45b82318, (q31_t)0x94a75afd, (q31_t)0x45a30df0, (q31_t)0x9499ac95, (q31_t)0x458df619, (q31_t)0x948c0252,
	(q31_t)0x4578db93, (q31_t)0x947e5c33, (q31_t)0x4563be60, (q31_t)0x9470ba39, (q31_t)0x454e9e80, (q31_t)0x94631c65, (q31_t)0x45397bf4, (q31_t)0x945582b7,
	(q31_t)0x452456bd, (q31_t)0x9447ed2f, (q31_t)0x450f2edb, (q31_t)0x943a5bcf, (q31_t)0x44fa0450, (q31_t)0x942cce96, (q31_t)0x44e4d71c, (q31_t)0x941f4585,
	(q31_t)0x44cfa740, (q31_t)0x9411c09e, (q31_t)0x44ba74bd, (q31_t)0x94043fdf, (q31_t)0x44a53f93, (q31_t)0x93f6c34a, (q31_t)0x449007c4, (q31_t)0x93e94adf,
	(q31_t)0x447acd50, (q31_t)0x93dbd6a0, (q31_t)0x44659039, (q31_t)0x93ce668b, (q31_t)0x4450507e, (q31_t)0x93c0faa3, (q31_t)0x443b0e21, (q31_t)0x93b392e6,
	(q31_t)0x4425c923, (q31_t)0x93a62f57, (q31_t)0x44108184, (q31_t)0x9398cff5, (q31_t)0x43fb3746, (q31_t)0x938b74c1, (q31_t)0x43e5ea68, (q31_t)0x937e1dbb,
	(q31_t)0x43d09aed, (q31_t)0x9370cae4, (q31_t)0x43bb48d4, (q31_t)0x93637c3d, (q31_t)0x43a5f41e, (q31_t)0x935631c5, (q31_t)0x43909ccd, (q31_t)0x9348eb7e,
	(q31_t)0x437b42e1, (q31_t)0x933ba968, (q31_t)0x4365e65b, (q31_t)0x932e6b84, (q31_t)0x4350873c, (q31_t)0x932131d1, (q31_t)0x433b2585, (q31_t)0x9313fc51,
	(q31_t)0x4325c135, (q31_t)0x9306cb04, (q31_t)0x43105a50, (q31_t)0x92f99deb, (q31_t)0x42faf0d4, (q31_t)0x92ec7505, (q31_t)0x42e584c3, (q31_t)0x92df5054,
	(q31_t)0x42d0161e, (q31_t)0x92d22fd9, (q31_t)0x42baa4e6, (q31_t)0x92c51392, (q31_t)0x42a5311b, (q31_t)0x92b7fb82, (q31_t)0x428fbabe, (q31_t)0x92aae7a8,
	(q31_t)0x427a41d0, (q31_t)0x929dd806, (q31_t)0x4264c653, (q31_t)0x9290cc9b, (q31_t)0x424f4845, (q31_t)0x9283c568, (q31_t)0x4239c7aa, (q31_t)0x9276c26d,
	(q31_t)0x42244481, (q31_t)0x9269c3ac, (q31_t)0x420ebecb, (q31_t)0x925cc924, (q31_t)0x41f93689, (q31_t)0x924fd2d7, (q31_t)0x41e3abbc, (q31_t)0x9242e0c4,
	(q31_t)0x41ce1e65, (q31_t)0x9235f2ec, (q31_t)0x41b88e84, (q31_t)0x9229094f, (q31_t)0x41a2fc1a, (q31_t)0x921c23ef, (q31_t)0x418d6729, (q31_t)0x920f42cb,
	(q31_t)0x4177cfb1, (q31_t)0x920265e4, (q31_t)0x416235b2, (q31_t)0x91f58d3b, (q31_t)0x414c992f, (q31_t)0x91e8b8d0, (q31_t)0x4136fa27, (q31_t)0x91dbe8a4,
	(q31_t)0x4121589b, (q31_t)0x91cf1cb6, (q31_t)0x410bb48c, (q31_t)0x91c25508, (q31_t)0x40f60dfb, (q31_t)0x91b5919a, (q31_t)0x40e064ea, (q31_t)0x91a8d26d,
	(q31_t)0x40cab958, (q31_t)0x919c1781, (q31_t)0x40b50b46, (q31_t)0x918f60d6, (q31_t)0x409f5ab6, (q31_t)0x9182ae6d, (q31_t)0x4089a7a8, (q31_t)0x91760047,
	(q31_t)0x4073f21d, (q31_t)0x91695663, (q31_t)0x405e3a16, (q31_t)0x915cb0c3, (q31_t)0x40487f94, (q31_t)0x91500f67, (q31_t)0x4032c297, (q31_t)0x91437250,
	(q31_t)0x401d0321, (q31_t)0x9136d97d, (q31_t)0x40074132, (q31_t)0x912a44f0, (q31_t)0x3ff17cca, (q31_t)0x911db4a9, (q31_t)0x3fdbb5ec, (q31_t)0x911128a8,
	(q31_t)0x3fc5ec98, (q31_t)0x9104a0ee, (q31_t)0x3fb020ce, (q31_t)0x90f81d7b, (q31_t)0x3f9a5290, (q31_t)0x90eb9e50, (q31_t)0x3f8481dd, (q31_t)0x90df236e,
	(q31_t)0x3f6eaeb8, (q31_t)0x90d2acd4, (q31_t)0x3f58d921, (q31_t)0x90c63a83, (q31_t)0x3f430119, (q31_t)0x90b9cc7d, (q31_t)0x3f2d26a0, (q31_t)0x90ad62c0,
	(q31_t)0x3f1749b8, (q31_t)0x90a0fd4e, (q31_t)0x3f016a61, (q31_t)0x90949c28, (q31_t)0x3eeb889c, (q31_t)0x90883f4d, (q31_t)0x3ed5a46b, (q31_t)0x907be6be,
	(q31_t)0x3ebfbdcd, (q31_t)0x906f927c, (q31_t)0x3ea9d4c3, (q31_t)0x90634287, (q31_t)0x3e93e950, (q31_t)0x9056f6df, (q31_t)0x3e7dfb73, (q31_t)0x904aaf86,
	(q31_t)0x3e680b2c, (q31_t)0x903e6c7b, (q31_t)0x3e52187f, (q31_t)0x90322dbf, (q31_t)0x3e3c2369, (q31_t)0x9025f352, (q31_t)0x3e262bee, (q31_t)0x9019bd36,
	(q31_t)0x3e10320d, (q31_t)0x900d8b69, (q31_t)0x3dfa35c8, (q31_t)0x90015dee, (q31_t)0x3de4371f, (q31_t)0x8ff534c4, (q31_t)0x3dce3614, (q31_t)0x8fe90fec,
	(q31_t)0x3db832a6, (q31_t)0x8fdcef66, (q31_t)0x3da22cd7, (q31_t)0x8fd0d333, (q31_t)0x3d8c24a8, (q31_t)0x8fc4bb53, (q31_t)0x3d761a19, (q31_t)0x8fb8a7c7,
	(q31_t)0x3d600d2c, (q31_t)0x8fac988f, (q31_t)0x3d49fde1, (q31_t)0x8fa08dab, (q31_t)0x3d33ec39, (q31_t)0x8f94871d, (q31_t)0x3d1dd835, (q31_t)0x8f8884e4,
	(q31_t)0x3d07c1d6, (q31_t)0x8f7c8701, (q31_t)0x3cf1a91c, (q31_t)0x8f708d75, (q31_t)0x3cdb8e09, (q31_t)0x8f649840, (q31_t)0x3cc5709e, (q31_t)0x8f58a761,
	(q31_t)0x3caf50da, (q31_t)0x8f4cbadb, (q31_t)0x3c992ec0, (q31_t)0x8f40d2ad, (q31_t)0x3c830a50, (q31_t)0x8f34eed8, (q31_t)0x3c6ce38a, (q31_t)0x8f290f5c,
	(q31_t)0x3c56ba70, (q31_t)0x8f1d343a, (q31_t)0x3c408f03, (q31_t)0x8f115d72, (q31_t)0x3c2a6142, (q31_t)0x8f058b04, (q31_t)0x3c143130, (q31_t)0x8ef9bcf2,
	(q31_t)0x3bfdfecd, (q31_t)0x8eedf33b, (q31_t)0x3be7ca1a, (q31_t)0x8ee22de0, (q31_t)0x3bd19318, (q31_t)0x8ed66ce1, (q31_t)0x3bbb59c7, (q31_t)0x8ecab040,
	(q31_t)0x3ba51e29, (q31_t)0x8ebef7fb, (q31_t)0x3b8ee03e, (q31_t)0x8eb34415, (q31_t)0x3b78a007, (q31_t)0x8ea7948c, (q31_t)0x3b625d86, (q31_t)0x8e9be963,
	(q31_t)0x3b4c18ba, (q31_t)0x8e904298, (q31_t)0x3b35d1a5, (q31_t)0x8e84a02d, (q31_t)0x3b1f8848, (q31_t)0x8e790222, (q31_t)0x3b093ca3, (q31_t)0x8e6d6877,
	(q31_t)0x3af2eeb7, (q31_t)0x8e61d32e, (q31_t)0x3adc9e86, (q31_t)0x8e564246, (q31_t)0x3ac64c0f, (q31_t)0x8e4ab5bf, (q31_t)0x3aaff755, (q31_t)0x8e3f2d9b,
	(q31_t)0x3a99a057, (q31_t)0x8e33a9da, (q31_t)0x3a834717, (q31_t)0x8e282a7b, (q31_t)0x3a6ceb96, (q31_t)0x8e1caf80, (q31_t)0x3a568dd4, (q31_t)0x8e1138ea,
	(q31_t)0x3a402dd2, (q31_t)0x8e05c6b7, (q31_t)0x3a29cb91, (q31_t)0x8dfa58ea, (q31_t)0x3a136712, (q31_t)0x8deeef82, (q31_t)0x39fd0056, (q31_t)0x8de38a80,
	(q31_t)0x39e6975e, (q31_t)0x8dd829e4, (q31_t)0x39d02c2a, (q31_t)0x8dcccdaf, (q31_t)0x39b9bebc, (q31_t)0x8dc175e0, (q31_t)0x39a34f13, (q31_t)0x8db6227a,
	(q31_t)0x398cdd32, (q31_t)0x8daad37b, (q31_t)0x39766919, (q31_t)0x8d9f88e5, (q31_t)0x395ff2c9, (q31_t)0x8d9442b8, (q31_t)0x39497a43, (q31_t)0x8d8900f3,
	(q31_t)0x3932ff87, (q31_t)0x8d7dc399, (q31_t)0x391c8297, (q31_t)0x8d728aa9, (q31_t)0x39060373, (q31_t)0x8d675623, (q31_t)0x38ef821c, (q31_t)0x8d5c2609,
	(q31_t)0x38d8fe93, (q31_t)0x8d50fa59, (q31_t)0x38c278d9, (q31_t)0x8d45d316, (q31_t)0x38abf0ef, (q31_t)0x8d3ab03f, (q31_t)0x389566d6, (q31_t)0x8d2f91d5,
	(q31_t)0x387eda8e, (q31_t)0x8d2477d8, (q31_t)0x38684c19, (q31_t)0x8d196249, (q31_t)0x3851bb77, (q31_t)0x8d0e5127, (q31_t)0x383b28a9, (q31_t)0x8d034474,
	(q31_t)0x382493b0, (q31_t)0x8cf83c30, (q31_t)0x380dfc8d, (q31_t)0x8ced385b, (q31_t)0x37f76341, (q31_t)0x8ce238f6, (q31_t)0x37e0c7cc, (q31_t)0x8cd73e01,
	(q31_t)0x37ca2a30, (q31_t)0x8ccc477d, (q31_t)0x37b38a6d, (q31_t)0x8cc1556a, (q31_t)0x379ce885, (q31_t)0x8cb667c8, (q31_t)0x37864477, (q31_t)0x8cab7e98,
	(q31_t)0x376f9e46, (q31_t)0x8ca099da, (q31_t)0x3758f5f2, (q31_t)0x8c95b98f, (q31_t)0x37424b7b, (q31_t)0x8c8addb7, (q31_t)0x372b9ee3, (q31_t)0x8c800652,
	(q31_t)0x3714f02a, (q31_t)0x8c753362, (q31_t)0x36fe3f52, (q31_t)0x8c6a64e5, (q31_t)0x36e78c5b, (q31_t)0x8c5f9ade, (q31_t)0x36d0d746, (q31_t)0x8c54d54c,
	(q31_t)0x36ba2014, (q31_t)0x8c4a142f, (q31_t)0x36a366c6, (q31_t)0x8c3f5788, (q31_t)0x368cab5c, (q31_t)0x8c349f58, (q31_t)0x3675edd9, (q31_t)0x8c29eb9f,
	(q31_t)0x365f2e3b, (q31_t)0x8c1f3c5d, (q31_t)0x36486c86, (q31_t)0x8c149192, (q31_t)0x3631a8b8, (q31_t)0x8c09eb40, (q31_t)0x361ae2d3, (q31_t)0x8bff4966,
	(q31_t)0x36041ad9, (q31_t)0x8bf4ac05, (q31_t)0x35ed50c9, (q31_t)0x8bea131e, (q31_t)0x35d684a6, (q31_t)0x8bdf7eb0, (q31_t)0x35bfb66e, (q31_t)0x8bd4eebc,
	(q31_t)0x35a8e625, (q31_t)0x8bca6343, (q31_t)0x359213c9, (q31_t)0x8bbfdc44, (q31_t)0x357b3f5d, (q31_t)0x8bb559c1, (q31_t)0x356468e2, (q31_t)0x8baadbba,
	(q31_t)0x354d9057, (q31_t)0x8ba0622f, (q31_t)0x3536b5be, (q31_t)0x8b95ed21, (q31_t)0x351fd918, (q31_t)0x8b8b7c8f, (q31_t)0x3508fa66, (q31_t)0x8b81107b,
	(q31_t)0x34f219a8, (q31_t)0x8b76a8e4, (q31_t)0x34db36df, (q31_t)0x8b6c45cc, (q31_t)0x34c4520d, (q31_t)0x8b61e733, (q31_t)0x34ad6b32, (q31_t)0x8b578d18,
	(q31_t)0x34968250, (q31_t)0x8b4d377c, (q31_t)0x347f9766, (q31_t)0x8b42e661, (q31_t)0x3468aa76, (q31_t)0x8b3899c6, (q31_t)0x3451bb81, (q31_t)0x8b2e51ab,
	(q31_t)0x343aca87, (q31_t)0x8b240e11, (q31_t)0x3423d78a, (q31_t)0x8b19cef8, (q31_t)0x340ce28b, (q31_t)0x8b0f9462, (q31_t)0x33f5eb89, (q31_t)0x8b055e4d,
	(q31_t)0x33def287, (q31_t)0x8afb2cbb, (q31_t)0x33c7f785, (q31_t)0x8af0ffac, (q31_t)0x33b0fa84, (q31_t)0x8ae6d720, (q31_t)0x3399fb85, (q31_t)0x8adcb318,
	(q31_t)0x3382fa88, (q31_t)0x8ad29394, (q31_t)0x336bf78f, (q31_t)0x8ac87894, (q31_t)0x3354f29b, (q31_t)0x8abe6219, (q31_t)0x333debab, (q31_t)0x8ab45024,
	(q31_t)0x3326e2c3, (q31_t)0x8aaa42b4, (q31_t)0x330fd7e1, (q31_t)0x8aa039cb, (q31_t)0x32f8cb07, (q31_t)0x8a963567, (q31_t)0x32e1bc36, (q31_t)0x8a8c358b,
	(q31_t)0x32caab6f, (q31_t)0x8a823a36, (q31_t)0x32b398b3, (q31_t)0x8a784368, (q31_t)0x329c8402, (q31_t)0x8a6e5123, (q31_t)0x32856d5e, (q31_t)0x8a646365,
	(q31_t)0x326e54c7, (q31_t)0x8a5a7a31, (q31_t)0x32573a3f, (q31_t)0x8a509585, (q31_t)0x32401dc6, (q31_t)0x8a46b564, (q31_t)0x3228ff5c, (q31_t)0x8a3cd9cc,
	(q31_t)0x3211df04, (q31_t)0x8a3302be, (q31_t)0x31fabcbd, (q31_t)0x8a29303b, (q31_t)0x31e39889, (q31_t)0x8a1f6243, (q31_t)0x31cc7269, (q31_t)0x8a1598d6,
	(q31_t)0x31b54a5e, (q31_t)0x8a0bd3f5, (q31_t)0x319e2067, (q31_t)0x8a0213a0, (q31_t)0x3186f487, (q31_t)0x89f857d8, (q31_t)0x316fc6be, (q31_t)0x89eea09d,
	(q31_t)0x3158970e, (q31_t)0x89e4edef, (q31_t)0x31416576, (q31_t)0x89db3fcf, (q31_t)0x312a31f8, (q31_t)0x89d1963c, (q31_t)0x3112fc95, (q31_t)0x89c7f138,
	(q31_t)0x30fbc54d, (q31_t)0x89be50c3, (q31_t)0x30e48c22, (q31_t)0x89b4b4dd, (q31_t)0x30cd5115, (q31_t)0x89ab1d87, (q31_t)0x30b61426, (q31_t)0x89a18ac0,
	(q31_t)0x309ed556, (q31_t)0x8997fc8a, (q31_t)0x308794a6, (q31_t)0x898e72e4, (q31_t)0x30705217, (q31_t)0x8984edcf, (q31_t)0x30590dab, (q31_t)0x897b6d4c,
	(q31_t)0x3041c761, (q31_t)0x8971f15a, (q31_t)0x302a7f3a, (q31_t)0x896879fb, (q31_t)0x30133539, (q31_t)0x895f072e, (q31_t)0x2ffbe95d, (q31_t)0x895598f3,
	(q31_t)0x2fe49ba7, (q31_t)0x894c2f4c, (q31_t)0x2fcd4c19, (q31_t)0x8942ca39, (q31_t)0x2fb5fab2, (q31_t)0x893969b9, (q31_t)0x2f9ea775, (q31_t)0x89300dce,
	(q31_t)0x2f875262, (q31_t)0x8926b677, (q31_t)0x2f6ffb7a, (q31_t)0x891d63b5, (q31_t)0x2f58a2be, (q31_t)0x89141589, (q31_t)0x2f41482e, (q31_t)0x890acbf2,
	(q31_t)0x2f29ebcc, (q31_t)0x890186f2, (q31_t)0x2f128d99, (q31_t)0x88f84687, (q31_t)0x2efb2d95, (q31_t)0x88ef0ab4, (q31_t)0x2ee3cbc1, (q31_t)0x88e5d378,
	(q31_t)0x2ecc681e, (q31_t)0x88dca0d3, (q31_t)0x2eb502ae, (q31_t)0x88d372c6, (q31_t)0x2e9d9b70, (q31_t)0x88ca4951, (q31_t)0x2e863267, (q31_t)0x88c12475,
	(q31_t)0x2e6ec792, (q31_t)0x88b80432, (q31_t)0x2e575af3, (q31_t)0x88aee888, (q31_t)0x2e3fec8b, (q31_t)0x88a5d177, (q31_t)0x2e287c5a, (q31_t)0x889cbf01,
	(q31_t)0x2e110a62, (q31_t)0x8893b125, (q31_t)0x2df996a3, (q31_t)0x888aa7e3, (q31_t)0x2de2211e, (q31_t)0x8881a33d, (q31_t)0x2dcaa9d5, (q31_t)0x8878a332,
	(q31_t)0x2db330c7, (q31_t)0x886fa7c2, (q31_t)0x2d9bb5f6, (q31_t)0x8866b0ef, (q31_t)0x2d843964, (q31_t)0x885dbeb8, (q31_t)0x2d6cbb10, (q31_t)0x8854d11e,
	(q31_t)0x2d553afc, (q31_t)0x884be821, (q31_t)0x2d3db928, (q31_t)0x884303c1, (q31_t)0x2d263596, (q31_t)0x883a23ff, (q31_t)0x2d0eb046, (q31_t)0x883148db,
	(q31_t)0x2cf72939, (q31_t)0x88287256, (q31_t)0x2cdfa071, (q31_t)0x881fa06f, (q31_t)0x2cc815ee, (q31_t)0x8816d327, (q31_t)0x2cb089b1, (q31_t)0x880e0a7f,
	(q31_t)0x2c98fbba, (q31_t)0x88054677, (q31_t)0x2c816c0c, (q31_t)0x87fc870f, (q31_t)0x2c69daa6, (q31_t)0x87f3cc48, (q31_t)0x2c52478a, (q31_t)0x87eb1621,
	(q31_t)0x2c3ab2b9, (q31_t)0x87e2649b, (q31_t)0x2c231c33, (q31_t)0x87d9b7b7, (q31_t)0x2c0b83fa, (q31_t)0x87d10f75, (q31_t)0x2bf3ea0d, (q31_t)0x87c86bd5,
	(q31_t)0x2bdc4e6f, (q31_t)0x87bfccd7, (q31_t)0x2bc4b120, (q31_t)0x87b7327d, (q31_t)0x2bad1221, (q31_t)0x87ae9cc5, (q31_t)0x2b957173, (q31_t)0x87a60bb1,
	(q31_t)0x2b7dcf17, (q31_t)0x879d7f41, (q31_t)0x2b662b0e, (q31_t)0x8794f774, (q31_t)0x2b4e8558, (q31_t)0x878c744d, (q31_t)0x2b36ddf7, (q31_t)0x8783f5ca,
	(q31_t)0x2b1f34eb, (q31_t)0x877b7bec, (q31_t)0x2b078a36, (q31_t)0x877306b4, (q31_t)0x2aefddd8, (q31_t)0x876a9621, (q31_t)0x2ad82fd2, (q31_t)0x87622a35,
	(q31_t)0x2ac08026, (q31_t)0x8759c2ef, (q31_t)0x2aa8ced3, (q31_t)0x87516050, (q31_t)0x2a911bdc, (q31_t)0x87490258, (q31_t)0x2a796740, (q31_t)0x8740a907,
	(q31_t)0x2a61b101, (q31_t)0x8738545e, (q31_t)0x2a49f920, (q31_t)0x8730045d, (q31_t)0x2a323f9e, (q31_t)0x8727b905, (q31_t)0x2a1a847b, (q31_t)0x871f7255,
	(q31_t)0x2a02c7b8, (q31_t)0x8717304e, (q31_t)0x29eb0957, (q31_t)0x870ef2f1, (q31_t)0x29d34958, (q31_t)0x8706ba3d, (q31_t)0x29bb87bc, (q31_t)0x86fe8633,
	(q31_t)0x29a3c485, (q31_t)0x86f656d3, (q31_t)0x298bffb2, (q31_t)0x86ee2c1e, (q31_t)0x29743946, (q31_t)0x86e60614, (q31_t)0x295c7140, (q31_t)0x86dde4b5,
	(q31_t)0x2944a7a2, (q31_t)0x86d5c802, (q31_t)0x292cdc6d, (q31_t)0x86cdaffa, (q31_t)0x29150fa1, (q31_t)0x86c59c9f, (q31_t)0x28fd4140, (q31_t)0x86bd8df0,
	(q31_t)0x28e5714b, (q31_t)0x86b583ee, (q31_t)0x28cd9fc1, (q31_t)0x86ad7e99, (q31_t)0x28b5cca5, (q31_t)0x86a57df2, (q31_t)0x289df7f8, (q31_t)0x869d81f8,
	(q31_t)0x288621b9, (q31_t)0x86958aac, (q31_t)0x286e49ea, (q31_t)0x868d980e, (q31_t)0x2856708d, (q31_t)0x8685aa20, (q31_t)0x283e95a1, (q31_t)0x867dc0e0,
	(q31_t)0x2826b928, (q31_t)0x8675dc4f, (q31_t)0x280edb23, (q31_t)0x866dfc6e, (q31_t)0x27f6fb92, (q31_t)0x8666213c, (q31_t)0x27df1a77, (q31_t)0x865e4abb,
	(q31_t)0x27c737d3, (q31_t)0x865678eb, (q31_t)0x27af53a6, (q31_t)0x864eabcb, (q31_t)0x27976df1, (q31_t)0x8646e35c, (q31_t)0x277f86b5, (q31_t)0x863f1f9e,
	(q31_t)0x27679df4, (q31_t)0x86376092, (q31_t)0x274fb3ae, (q31_t)0x862fa638, (q31_t)0x2737c7e3, (q31_t)0x8627f091, (q31_t)0x271fda96, (q31_t)0x86203f9c,
	(q31_t)0x2707ebc7, (q31_t)0x86189359, (q31_t)0x26effb76, (q31_t)0x8610ebca, (q31_t)0x26d809a5, (q31_t)0x860948ef, (q31_t)0x26c01655, (q31_t)0x8601aac7,
	(q31_t)0x26a82186, (q31_t)0x85fa1153, (q31_t)0x26902b39, (q31_t)0x85f27c93, (q31_t)0x26783370, (q31_t)0x85eaec88, (q31_t)0x26603a2c, (q31_t)0x85e36132,
	(q31_t)0x26483f6c, (q31_t)0x85dbda91, (q31_t)0x26304333, (q31_t)0x85d458a6, (q31_t)0x26184581, (q31_t)0x85ccdb70, (q31_t)0x26004657, (q31_t)0x85c562f1,
	(q31_t)0x25e845b6, (q31_t)0x85bdef28, (q31_t)0x25d0439f, (q31_t)0x85b68015, (q31_t)0x25b84012, (q31_t)0x85af15b9, (q31_t)0x25a03b11, (q31_t)0x85a7b015,
	(q31_t)0x2588349d, (q31_t)0x85a04f28, (q31_t)0x25702cb7, (q31_t)0x8598f2f3, (q31_t)0x2558235f, (q31_t)0x85919b76, (q31_t)0x25401896, (q31_t)0x858a48b1,
	(q31_t)0x25280c5e, (q31_t)0x8582faa5, (q31_t)0x250ffeb7, (q31_t)0x857bb152, (q31_t)0x24f7efa2, (q31_t)0x85746cb8, (q31_t)0x24dfdf20, (q31_t)0x856d2cd7,
	(q31_t)0x24c7cd33, (q31_t)0x8565f1b0, (q31_t)0x24afb9da, (q31_t)0x855ebb44, (q31_t)0x2497a517, (q31_t)0x85578991, (q31_t)0x247f8eec, (q31_t)0x85505c99,
	(q31_t)0x24677758, (q31_t)0x8549345c, (q31_t)0x244f5e5c, (q31_t)0x854210db, (q31_t)0x243743fa, (q31_t)0x853af214, (q31_t)0x241f2833, (q31_t)0x8533d809,
	(q31_t)0x24070b08, (q31_t)0x852cc2bb, (q31_t)0x23eeec78, (q31_t)0x8525b228, (q31_t)0x23d6cc87, (q31_t)0x851ea652, (q31_t)0x23beab33, (q31_t)0x85179f39,
	(q31_t)0x23a6887f, (q31_t)0x85109cdd, (q31_t)0x238e646a, (q31_t)0x85099f3e, (q31_t)0x23763ef7, (q31_t)0x8502a65c, (q31_t)0x235e1826, (q31_t)0x84fbb239,
	(q31_t)0x2345eff8, (q31_t)0x84f4c2d4, (q31_t)0x232dc66d, (q31_t)0x84edd82d, (q31_t)0x23159b88, (q31_t)0x84e6f244, (q31_t)0x22fd6f48, (q31_t)0x84e0111b,
	(q31_t)0x22e541af, (q31_t)0x84d934b1, (q31_t)0x22cd12bd, (q31_t)0x84d25d06, (q31_t)0x22b4e274, (q31_t)0x84cb8a1b, (q31_t)0x229cb0d5, (q31_t)0x84c4bbf0,
	(q31_t)0x22847de0, (q31_t)0x84bdf286, (q31_t)0x226c4996, (q31_t)0x84b72ddb, (q31_t)0x225413f8, (q31_t)0x84b06df2, (q31_t)0x223bdd08, (q31_t)0x84a9b2ca,
	(q31_t)0x2223a4c5, (q31_t)0x84a2fc62, (q31_t)0x220b6b32, (q31_t)0x849c4abd, (q31_t)0x21f3304f, (q31_t)0x84959dd9, (q31_t)0x21daf41d, (q31_t)0x848ef5b7,
	(q31_t)0x21c2b69c, (q31_t)0x84885258, (q31_t)0x21aa77cf, (q31_t)0x8481b3bb, (q31_t)0x219237b5, (q31_t)0x847b19e1, (q31_t)0x2179f64f, (q31_t)0x847484ca,
	(q31_t)0x2161b3a0, (q31_t)0x846df477, (q31_t)0x21496fa7, (q31_t)0x846768e7, (q31_t)0x21312a65, (q31_t)0x8460e21a, (q31_t)0x2118e3dc, (q31_t)0x845a6012,
	(q31_t)0x21009c0c, (q31_t)0x8453e2cf, (q31_t)0x20e852f6, (q31_t)0x844d6a50, (q31_t)0x20d0089c, (q31_t)0x8446f695, (q31_t)0x20b7bcfe, (q31_t)0x844087a0,
	(q31_t)0x209f701c, (q31_t)0x843a1d70, (q31_t)0x208721f9, (q31_t)0x8433b806, (q31_t)0x206ed295, (q31_t)0x842d5762, (q31_t)0x205681f1, (q31_t)0x8426fb84,
	(q31_t)0x203e300d, (q31_t)0x8420a46c, (q31_t)0x2025dcec, (q31_t)0x841a521a, (q31_t)0x200d888d, (q31_t)0x84140490, (q31_t)0x1ff532f2, (q31_t)0x840dbbcc,
	(q31_t)0x1fdcdc1b, (q31_t)0x840777d0, (q31_t)0x1fc4840a, (q31_t)0x8401389b, (q31_t)0x1fac2abf, (q31_t)0x83fafe2e, (q31_t)0x1f93d03c, (q31_t)0x83f4c889,
	(q31_t)0x1f7b7481, (q31_t)0x83ee97ad, (q31_t)0x1f63178f, (q31_t)0x83e86b99, (q31_t)0x1f4ab968, (q31_t)0x83e2444d, (q31_t)0x1f325a0b, (q31_t)0x83dc21cb,
	(q31_t)0x1f19f97b, (q31_t)0x83d60412, (q31_t)0x1f0197b8, (q31_t)0x83cfeb22, (q31_t)0x1ee934c3, (q31_t)0x83c9d6fc, (q31_t)0x1ed0d09d, (q31_t)0x83c3c7a0,
	(q31_t)0x1eb86b46, (q31_t)0x83bdbd0e, (q31_t)0x1ea004c1, (q31_t)0x83b7b746, (q31_t)0x1e879d0d, (q31_t)0x83b1b649, (q31_t)0x1e6f342c, (q31_t)0x83abba17,
	(q31_t)0x1e56ca1e, (q31_t)0x83a5c2b0, (q31_t)0x1e3e5ee5, (q31_t)0x839fd014, (q31_t)0x1e25f282, (q31_t)0x8399e244, (q31_t)0x1e0d84f5, (q31_t)0x8393f940,
	(q31_t)0x1df5163f, (q31_t)0x838e1507, (q31_t)0x1ddca662, (q31_t)0x8388359b, (q31_t)0x1dc4355e, (q31_t)0x83825afb, (q31_t)0x1dabc334, (q31_t)0x837c8528,
	(q31_t)0x1d934fe5, (q31_t)0x8376b422, (q31_t)0x1d7adb73, (q31_t)0x8370e7e9, (q31_t)0x1d6265dd, (q31_t)0x836b207d, (q31_t)0x1d49ef26, (q31_t)0x83655ddf,
	(q31_t)0x1d31774d, (q31_t)0x835fa00f, (q31_t)0x1d18fe54, (q31_t)0x8359e70d, (q31_t)0x1d00843d, (q31_t)0x835432d8, (q31_t)0x1ce80906, (q31_t)0x834e8373,
	(q31_t)0x1ccf8cb3, (q31_t)0x8348d8dc, (q31_t)0x1cb70f43, (q31_t)0x83433314, (q31_t)0x1c9e90b8, (q31_t)0x833d921b, (q31_t)0x1c861113, (q31_t)0x8337f5f1,
	(q31_t)0x1c6d9053, (q31_t)0x83325e97, (q31_t)0x1c550e7c, (q31_t)0x832ccc0d, (q31_t)0x1c3c8b8c, (q31_t)0x83273e52, (q31_t)0x1c240786, (q31_t)0x8321b568,
	(q31_t)0x1c0b826a, (q31_t)0x831c314e, (q31_t)0x1bf2fc3a, (q31_t)0x8316b205, (q31_t)0x1bda74f6, (q31_t)0x8311378d, (q31_t)0x1bc1ec9e, (q31_t)0x830bc1e6,
	(q31_t)0x1ba96335, (q31_t)0x83065110, (q31_t)0x1b90d8bb, (q31_t)0x8300e50b, (q31_t)0x1b784d30, (q31_t)0x82fb7dd8, (q31_t)0x1b5fc097, (q31_t)0x82f61b77,
	(q31_t)0x1b4732ef, (q31_t)0x82f0bde8, (q31_t)0x1b2ea43a, (q31_t)0x82eb652b, (q31_t)0x1b161479, (q31_t)0x82e61141, (q31_t)0x1afd83ad, (q31_t)0x82e0c22a,
	(q31_t)0x1ae4f1d6, (q31_t)0x82db77e5, (q31_t)0x1acc5ef6, (q31_t)0x82d63274, (q31_t)0x1ab3cb0d, (q31_t)0x82d0f1d5, (q31_t)0x1a9b361d, (q31_t)0x82cbb60b,
	(q31_t)0x1a82a026, (q31_t)0x82c67f14, (q31_t)0x1a6a0929, (q31_t)0x82c14cf1, (q31_t)0x1a517128, (q31_t)0x82bc1fa2, (q31_t)0x1a38d823, (q31_t)0x82b6f727,
	(q31_t)0x1a203e1b, (q31_t)0x82b1d381, (q31_t)0x1a07a311, (q31_t)0x82acb4b0, (q31_t)0x19ef0707, (q31_t)0x82a79ab3, (q31_t)0x19d669fc, (q31_t)0x82a2858c,
	(q31_t)0x19bdcbf3, (q31_t)0x829d753a, (q31_t)0x19a52ceb, (q31_t)0x829869be, (q31_t)0x198c8ce7, (q31_t)0x82936317, (q31_t)0x1973ebe6, (q31_t)0x828e6146,
	(q31_t)0x195b49ea, (q31_t)0x8289644b, (q31_t)0x1942a6f3, (q31_t)0x82846c26, (q31_t)0x192a0304, (q31_t)0x827f78d8, (q31_t)0x19115e1c, (q31_t)0x827a8a61,
	(q31_t)0x18f8b83c, (q31_t)0x8275a0c0, (q31_t)0x18e01167, (q31_t)0x8270bbf7, (q31_t)0x18c7699b, (q31_t)0x826bdc04, (q31_t)0x18aec0db, (q31_t)0x826700e9,
	(q31_t)0x18961728, (q31_t)0x82622aa6, (q31_t)0x187d6c82, (q31_t)0x825d593a, (q31_t)0x1864c0ea, (q31_t)0x82588ca7, (q31_t)0x184c1461, (q31_t)0x8253c4eb,
	(q31_t)0x183366e9, (q31_t)0x824f0208, (q31_t)0x181ab881, (q31_t)0x824a43fe, (q31_t)0x1802092c, (q31_t)0x82458acc, (q31_t)0x17e958ea, (q31_t)0x8240d673,
	(q31_t)0x17d0a7bc, (q31_t)0x823c26f3, (q31_t)0x17b7f5a3, (q31_t)0x82377c4c, (q31_t)0x179f429f, (q31_t)0x8232d67f, (q31_t)0x17868eb3, (q31_t)0x822e358b,
	(q31_t)0x176dd9de, (q31_t)0x82299971, (q31_t)0x17552422, (q31_t)0x82250232, (q31_t)0x173c6d80, (q31_t)0x82206fcc, (q31_t)0x1723b5f9, (q31_t)0x821be240,
	(q31_t)0x170afd8d, (q31_t)0x82175990, (q31_t)0x16f2443e, (q31_t)0x8212d5b9, (q31_t)0x16d98a0c, (q31_t)0x820e56be, (q31_t)0x16c0cef9, (q31_t)0x8209dc9e,
	(q31_t)0x16a81305, (q31_t)0x82056758, (q31_t)0x168f5632, (q31_t)0x8200f6ef, (q31_t)0x1676987f, (q31_t)0x81fc8b60, (q31_t)0x165dd9f0, (q31_t)0x81f824ae,
	(q31_t)0x16451a83, (q31_t)0x81f3c2d7, (q31_t)0x162c5a3b, (q31_t)0x81ef65dc, (q31_t)0x16139918, (q31_t)0x81eb0dbe, (q31_t)0x15fad71b, (q31_t)0x81e6ba7c,
	(q31_t)0x15e21445, (q31_t)0x81e26c16, (q31_t)0x15c95097, (q31_t)0x81de228d, (q31_t)0x15b08c12, (q31_t)0x81d9dde1, (q31_t)0x1597c6b7, (q31_t)0x81d59e13,
	(q31_t)0x157f0086, (q31_t)0x81d16321, (q31_t)0x15663982, (q31_t)0x81cd2d0c, (q31_t)0x154d71aa, (q31_t)0x81c8fbd6, (q31_t)0x1534a901, (q31_t)0x81c4cf7d,
	(q31_t)0x151bdf86, (q31_t)0x81c0a801, (q31_t)0x1503153a, (q31_t)0x81bc8564, (q31_t)0x14ea4a1f, (q31_t)0x81b867a5, (q31_t)0x14d17e36, (q31_t)0x81b44ec4,
	(q31_t)0x14b8b17f, (q31_t)0x81b03ac2, (q31_t)0x149fe3fc, (q31_t)0x81ac2b9e, (q31_t)0x148715ae, (q31_t)0x81a82159, (q31_t)0x146e4694, (q31_t)0x81a41bf4,
	(q31_t)0x145576b1, (q31_t)0x81a01b6d, (q31_t)0x143ca605, (q31_t)0x819c1fc5, (q31_t)0x1423d492, (q31_t)0x819828fd, (q31_t)0x140b0258, (q31_t)0x81943715,
	(q31_t)0x13f22f58, (q31_t)0x81904a0c, (q31_t)0x13d95b93, (q31_t)0x818c61e3, (q31_t)0x13c0870a, (q31_t)0x81887e9a, (q31_t)0x13a7b1bf, (q31_t)0x8184a032,
	(q31_t)0x138edbb1, (q31_t)0x8180c6a9, (q31_t)0x137604e2, (q31_t)0x817cf201, (q31_t)0x135d2d53, (q31_t)0x8179223a, (q31_t)0x13445505, (q31_t)0x81755754,
	(q31_t)0x132b7bf9, (q31_t)0x8171914e, (q31_t)0x1312a230, (q31_t)0x816dd02a, (q31_t)0x12f9c7aa, (q31_t)0x816a13e6, (q31_t)0x12e0ec6a, (q31_t)0x81665c84,
	(q31_t)0x12c8106f, (q31_t)0x8162aa04, (q31_t)0x12af33ba, (q31_t)0x815efc65, (q31_t)0x1296564d, (q31_t)0x815b53a8, (q31_t)0x127d7829, (q31_t)0x8157afcd,
	(q31_t)0x1264994e, (q31_t)0x815410d4, (q31_t)0x124bb9be, (q31_t)0x815076bd, (q31_t)0x1232d979, (q31_t)0x814ce188, (q31_t)0x1219f880, (q31_t)0x81495136,
	(q31_t)0x120116d5, (q31_t)0x8145c5c7, (q31_t)0x11e83478, (q31_t)0x81423f3a, (q31_t)0x11cf516a, (q31_t)0x813ebd90, (q31_t)0x11b66dad, (q31_t)0x813b40ca,
	(q31_t)0x119d8941, (q31_t)0x8137c8e6, (q31_t)0x1184a427, (q31_t)0x813455e6, (q31_t)0x116bbe60, (q31_t)0x8130e7c9, (q31_t)0x1152d7ed, (q31_t)0x812d7e8f,
	(q31_t)0x1139f0cf, (q31_t)0x812a1a3a, (q31_t)0x11210907, (q31_t)0x8126bac8, (q31_t)0x11082096, (q31_t)0x8123603a, (q31_t)0x10ef377d, (q31_t)0x81200a90,
	(q31_t)0x10d64dbd, (q31_t)0x811cb9ca, (q31_t)0x10bd6356, (q31_t)0x81196de9, (q31_t)0x10a4784b, (q31_t)0x811626ec, (q31_t)0x108b8c9b, (q31_t)0x8112e4d4,
	(q31_t)0x1072a048, (q31_t)0x810fa7a0, (q31_t)0x1059b352, (q31_t)0x810c6f52, (q31_t)0x1040c5bb, (q31_t)0x81093be8, (q31_t)0x1027d784, (q31_t)0x81060d63,
	(q31_t)0x100ee8ad, (q31_t)0x8102e3c4, (q31_t)0xff5f938, (q31_t)0x80ffbf0a, (q31_t)0xfdd0926, (q31_t)0x80fc9f35, (q31_t)0xfc41876, (q31_t)0x80f98446,
	(q31_t)0xfab272b, (q31_t)0x80f66e3c, (q31_t)0xf923546, (q31_t)0x80f35d19, (q31_t)0xf7942c7, (q31_t)0x80f050db, (q31_t)0xf604faf, (q31_t)0x80ed4984,
	(q31_t)0xf475bff, (q31_t)0x80ea4712, (q31_t)0xf2e67b8, (q31_t)0x80e74987, (q31_t)0xf1572dc, (q31_t)0x80e450e2, (q31_t)0xefc7d6b, (q31_t)0x80e15d24,
	(q31_t)0xee38766, (q31_t)0x80de6e4c, (q31_t)0xeca90ce, (q31_t)0x80db845b, (q31_t)0xeb199a4, (q31_t)0x80d89f51, (q31_t)0xe98a1e9, (q31_t)0x80d5bf2e,
	(q31_t)0xe7fa99e, (q31_t)0x80d2e3f2, (q31_t)0xe66b0c3, (q31_t)0x80d00d9d, (q31_t)0xe4db75b, (q31_t)0x80cd3c2f, (q31_t)0xe34bd66, (q31_t)0x80ca6fa9,
	(q31_t)0xe1bc2e4, (q31_t)0x80c7a80a, (q31_t)0xe02c7d7, (q31_t)0x80c4e553, (q31_t)0xde9cc40, (q31_t)0x80c22784, (q31_t)0xdd0d01f, (q31_t)0x80bf6e9c,
	(q31_t)0xdb7d376, (q31_t)0x80bcba9d, (q31_t)0xd9ed646, (q31_t)0x80ba0b85, (q31_t)0xd85d88f, (q31_t)0x80b76156, (q31_t)0xd6cda53, (q31_t)0x80b4bc0e,
	(q31_t)0xd53db92, (q31_t)0x80b21baf, (q31_t)0xd3adc4e, (q31_t)0x80af8039, (q31_t)0xd21dc87, (q31_t)0x80ace9ab, (q31_t)0xd08dc3f, (q31_t)0x80aa5806,
	(q31_t)0xcefdb76, (q31_t)0x80a7cb49, (q31_t)0xcd6da2d, (q31_t)0x80a54376, (q31_t)0xcbdd865, (q31_t)0x80a2c08b, (q31_t)0xca4d620, (q31_t)0x80a04289,
	(q31_t)0xc8bd35e, (q31_t)0x809dc971, (q31_t)0xc72d020, (q31_t)0x809b5541, (q31_t)0xc59cc68, (q31_t)0x8098e5fb, (q31_t)0xc40c835, (q31_t)0x80967b9f,
	(q31_t)0xc27c389, (q31_t)0x8094162c, (q31_t)0xc0ebe66, (q31_t)0x8091b5a2, (q31_t)0xbf5b8cb, (q31_t)0x808f5a02, (q31_t)0xbdcb2bb, (q31_t)0x808d034c,
	(q31_t)0xbc3ac35, (q31_t)0x808ab180, (q31_t)0xbaaa53b, (q31_t)0x8088649e, (q31_t)0xb919dcf, (q31_t)0x80861ca6, (q31_t)0xb7895f0, (q31_t)0x8083d998,
	(q31_t)0xb5f8d9f, (q31_t)0x80819b74, (q31_t)0xb4684df, (q31_t)0x807f623b, (q31_t)0xb2d7baf, (q31_t)0x807d2dec, (q31_t)0xb147211, (q31_t)0x807afe87,
	(q31_t)0xafb6805, (q31_t)0x8078d40d, (q31_t)0xae25d8d, (q31_t)0x8076ae7e, (q31_t)0xac952aa, (q31_t)0x80748dd9, (q31_t)0xab0475c, (q31_t)0x8072721f,
	(q31_t)0xa973ba5, (q31_t)0x80705b50, (q31_t)0xa7e2f85, (q31_t)0x806e496c, (q31_t)0xa6522fe, (q31_t)0x806c3c74, (q31_t)0xa4c1610, (q31_t)0x806a3466,
	(q31_t)0xa3308bd, (q31_t)0x80683143, (q31_t)0xa19fb04, (q31_t)0x8066330c, (q31_t)0xa00ece8, (q31_t)0x806439c0, (q31_t)0x9e7de6a, (q31_t)0x80624560,
	(q31_t)0x9cecf89, (q31_t)0x806055eb, (q31_t)0x9b5c048, (q31_t)0x805e6b62, (q31_t)0x99cb0a7, (q31_t)0x805c85c4, (q31_t)0x983a0a7, (q31_t)0x805aa512,
	(q31_t)0x96a9049, (q31_t)0x8058c94c, (q31_t)0x9517f8f, (q31_t)0x8056f272, (q31_t)0x9386e78, (q31_t)0x80552084, (q31_t)0x91f5d06, (q31_t)0x80535381,
	(q31_t)0x9064b3a, (q31_t)0x80518b6b, (q31_t)0x8ed3916, (q31_t)0x804fc841, (q31_t)0x8d42699, (q31_t)0x804e0a04, (q31_t)0x8bb13c5, (q31_t)0x804c50b2,
	(q31_t)0x8a2009a, (q31_t)0x804a9c4d, (q31_t)0x888ed1b, (q31_t)0x8048ecd5, (q31_t)0x86fd947, (q31_t)0x80474248, (q31_t)0x856c520, (q31_t)0x80459ca9,
	(q31_t)0x83db0a7, (q31_t)0x8043fbf6, (q31_t)0x8249bdd, (q31_t)0x80426030, (q31_t)0x80b86c2, (q31_t)0x8040c956, (q31_t)0x7f27157, (q31_t)0x803f376a,
	(q31_t)0x7d95b9e, (q31_t)0x803daa6a, (q31_t)0x7c04598, (q31_t)0x803c2257, (q31_t)0x7a72f45, (q31_t)0x803a9f31, (q31_t)0x78e18a7, (q31_t)0x803920f8,
	(q31_t)0x77501be, (q31_t)0x8037a7ac, (q31_t)0x75bea8c, (q31_t)0x8036334e, (q31_t)0x742d311, (q31_t)0x8034c3dd, (q31_t)0x729bb4e, (q31_t)0x80335959,
	(q31_t)0x710a345, (q31_t)0x8031f3c2, (q31_t)0x6f78af6, (q31_t)0x80309318, (q31_t)0x6de7262, (q31_t)0x802f375d, (q31_t)0x6c5598a, (q31_t)0x802de08e,
	(q31_t)0x6ac406f, (q31_t)0x802c8ead, (q31_t)0x6932713, (q31_t)0x802b41ba, (q31_t)0x67a0d76, (q31_t)0x8029f9b4, (q31_t)0x660f398, (q31_t)0x8028b69c,
	(q31_t)0x647d97c, (q31_t)0x80277872, (q31_t)0x62ebf22, (q31_t)0x80263f36, (q31_t)0x615a48b, (q31_t)0x80250ae7, (q31_t)0x5fc89b8, (q31_t)0x8023db86,
	(q31_t)0x5e36ea9, (q31_t)0x8022b114, (q31_t)0x5ca5361, (q31_t)0x80218b8f, (q31_t)0x5b137df, (q31_t)0x80206af8, (q31_t)0x5981c26, (q31_t)0x801f4f4f,
	(q31_t)0x57f0035, (q31_t)0x801e3895, (q31_t)0x565e40d, (q31_t)0x801d26c8, (q31_t)0x54cc7b1, (q31_t)0x801c19ea, (q31_t)0x533ab20, (q31_t)0x801b11fa,
	(q31_t)0x51a8e5c, (q31_t)0x801a0ef8, (q31_t)0x5017165, (q31_t)0x801910e4, (q31_t)0x4e8543e, (q31_t)0x801817bf, (q31_t)0x4cf36e5, (q31_t)0x80172388,
	(q31_t)0x4b6195d, (q31_t)0x80163440, (q31_t)0x49cfba7, (q31_t)0x801549e6, (q31_t)0x483ddc3, (q31_t)0x8014647b, (q31_t)0x46abfb3, (q31_t)0x801383fe,
	(q31_t)0x451a177, (q31_t)0x8012a86f, (q31_t)0x4388310, (q31_t)0x8011d1d0, (q31_t)0x41f6480, (q31_t)0x8011001f, (q31_t)0x40645c7, (q31_t)0x8010335c,
	(q31_t)0x3ed26e6, (q31_t)0x800f6b88, (q31_t)0x3d407df, (q31_t)0x800ea8a3, (q31_t)0x3bae8b2, (q31_t)0x800deaad, (q31_t)0x3a1c960, (q31_t)0x800d31a5,
	(q31_t)0x388a9ea, (q31_t)0x800c7d8c, (q31_t)0x36f8a51, (q31_t)0x800bce63, (q31_t)0x3566a96, (q31_t)0x800b2427, (q31_t)0x33d4abb, (q31_t)0x800a7edb,
	(q31_t)0x3242abf, (q31_t)0x8009de7e, (q31_t)0x30b0aa4, (q31_t)0x80094310, (q31_t)0x2f1ea6c, (q31_t)0x8008ac90, (q31_t)0x2d8ca16, (q31_t)0x80081b00,
	(q31_t)0x2bfa9a4, (q31_t)0x80078e5e, (q31_t)0x2a68917, (q31_t)0x800706ac, (q31_t)0x28d6870, (q31_t)0x800683e8, (q31_t)0x27447b0, (q31_t)0x80060614,
	(q31_t)0x25b26d7, (q31_t)0x80058d2f, (q31_t)0x24205e8, (q31_t)0x80051939, (q31_t)0x228e4e2, (q31_t)0x8004aa32, (q31_t)0x20fc3c6, (q31_t)0x8004401a,
	(q31_t)0x1f6a297, (q31_t)0x8003daf1, (q31_t)0x1dd8154, (q31_t)0x80037ab7, (q31_t)0x1c45ffe, (q31_t)0x80031f6d, (q31_t)0x1ab3e97, (q31_t)0x8002c912,
	(q31_t)0x1921d20, (q31_t)0x800277a6, (q31_t)0x178fb99, (q31_t)0x80022b29, (q31_t)0x15fda03, (q31_t)0x8001e39b, (q31_t)0x146b860, (q31_t)0x8001a0fd,
	(q31_t)0x12d96b1, (q31_t)0x8001634e, (q31_t)0x11474f6, (q31_t)0x80012a8e, (q31_t)0xfb5330, (q31_t)0x8000f6bd, (q31_t)0xe23160, (q31_t)0x8000c7dc,
	(q31_t)0xc90f88, (q31_t)0x80009dea, (q31_t)0xafeda8, (q31_t)0x800078e7, (q31_t)0x96cbc1, (q31_t)0x800058d4, (q31_t)0x7da9d4, (q31_t)0x80003daf,
	(q31_t)0x6487e3, (q31_t)0x8000277a, (q31_t)0x4b65ee, (q31_t)0x80001635, (q31_t)0x3243f5, (q31_t)0x800009df, (q31_t)0x1921fb, (q31_t)0x80000278
};

static const q31_t WeightsQ31_8192[16384] = {
	(q31_t)0x7fffffff, (q31_t)0x0, (q31_t)0x7fffffd9, (q31_t)0xfff9b781, (q31_t)0x7fffff62, (q31_t)0xfff36f02, (q31_t)0x7ffffe9d,
	(q31_t)0xffed2684,
	(q31_t)0x7ffffd88, (q31_t)0xffe6de05, (q31_t)0x7ffffc25, (q31_t)0xffe09586, (q31_t)0x7ffffa73, (q31_t)0xffda4d08,
	(q31_t)0x7ffff872, (q31_t)0xffd40489,
	(q31_t)0x7ffff621, (q31_t)0xffcdbc0b, (q31_t)0x7ffff382, (q31_t)0xffc7738c, (q31_t)0x7ffff094, (q31_t)0xffc12b0e,
	(q31_t)0x7fffed57, (q31_t)0xffbae290,
	(q31_t)0x7fffe9cb, (q31_t)0xffb49a12, (q31_t)0x7fffe5f0, (q31_t)0xffae5195, (q31_t)0x7fffe1c6, (q31_t)0xffa80917,
	(q31_t)0x7fffdd4d, (q31_t)0xffa1c09a,
	(q31_t)0x7fffd886, (q31_t)0xff9b781d, (q31_t)0x7fffd36f, (q31_t)0xff952fa0, (q31_t)0x7fffce09, (q31_t)0xff8ee724,
	(q31_t)0x7fffc854, (q31_t)0xff889ea7,
	(q31_t)0x7fffc251, (q31_t)0xff82562c, (q31_t)0x7fffbbfe, (q31_t)0xff7c0db0, (q31_t)0x7fffb55c, (q31_t)0xff75c535,
	(q31_t)0x7fffae6c, (q31_t)0xff6f7cba,
	(q31_t)0x7fffa72c, (q31_t)0xff69343f, (q31_t)0x7fff9f9e, (q31_t)0xff62ebc5, (q31_t)0x7fff97c1, (q31_t)0xff5ca34b,
	(q31_t)0x7fff8f94, (q31_t)0xff565ad1,
	(q31_t)0x7fff8719, (q31_t)0xff501258, (q31_t)0x7fff7e4f, (q31_t)0xff49c9df, (q31_t)0x7fff7536, (q31_t)0xff438167,
	(q31_t)0x7fff6bcd, (q31_t)0xff3d38ef,
	(q31_t)0x7fff6216, (q31_t)0xff36f078, (q31_t)0x7fff5810, (q31_t)0xff30a801, (q31_t)0x7fff4dbb, (q31_t)0xff2a5f8b,
	(q31_t)0x7fff4317, (q31_t)0xff241715,
	(q31_t)0x7fff3824, (q31_t)0xff1dcea0, (q31_t)0x7fff2ce2, (q31_t)0xff17862b, (q31_t)0x7fff2151, (q31_t)0xff113db7,
	(q31_t)0x7fff1572, (q31_t)0xff0af543,
	(q31_t)0x7fff0943, (q31_t)0xff04acd0, (q31_t)0x7ffefcc5, (q31_t)0xfefe645e, (q31_t)0x7ffeeff8, (q31_t)0xfef81bec,
	(q31_t)0x7ffee2dd, (q31_t)0xfef1d37b,
	(q31_t)0x7ffed572, (q31_t)0xfeeb8b0a, (q31_t)0x7ffec7b9, (q31_t)0xfee5429a, (q31_t)0x7ffeb9b0, (q31_t)0xfedefa2b,
	(q31_t)0x7ffeab59, (q31_t)0xfed8b1bd,
	(q31_t)0x7ffe9cb2, (q31_t)0xfed2694f, (q31_t)0x7ffe8dbd, (q31_t)0xfecc20e2, (q31_t)0x7ffe7e79, (q31_t)0xfec5d876,
	(q31_t)0x7ffe6ee5, (q31_t)0xfebf900a,
	(q31_t)0x7ffe5f03, (q31_t)0xfeb947a0, (q31_t)0x7ffe4ed2, (q31_t)0xfeb2ff36, (q31_t)0x7ffe3e52, (q31_t)0xfeacb6cc,
	(q31_t)0x7ffe2d83, (q31_t)0xfea66e64,
	(q31_t)0x7ffe1c65, (q31_t)0xfea025fd, (q31_t)0x7ffe0af8, (q31_t)0xfe99dd96, (q31_t)0x7ffdf93c, (q31_t)0xfe939530,
	(q31_t)0x7ffde731, (q31_t)0xfe8d4ccb,
	(q31_t)0x7ffdd4d7, (q31_t)0xfe870467, (q31_t)0x7ffdc22e, (q31_t)0xfe80bc04, (q31_t)0x7ffdaf37, (q31_t)0xfe7a73a2,
	(q31_t)0x7ffd9bf0, (q31_t)0xfe742b41,
	(q31_t)0x7ffd885a, (q31_t)0xfe6de2e0, (q31_t)0x7ffd7476, (q31_t)0xfe679a81, (q31_t)0x7ffd6042, (q31_t)0xfe615223,
	(q31_t)0x7ffd4bc0, (q31_t)0xfe5b09c5,
	(q31_t)0x7ffd36ee, (q31_t)0xfe54c169, (q31_t)0x7ffd21ce, (q31_t)0xfe4e790d, (q31_t)0x7ffd0c5f, (q31_t)0xfe4830b3,
	(q31_t)0x7ffcf6a0, (q31_t)0xfe41e85a,
	(q31_t)0x7ffce093, (q31_t)0xfe3ba002, (q31_t)0x7ffcca37, (q31_t)0xfe3557ab, (q31_t)0x7ffcb38c, (q31_t)0xfe2f0f55,
	(q31_t)0x7ffc9c92, (q31_t)0xfe28c700,
	(q31_t)0x7ffc8549, (q31_t)0xfe227eac, (q31_t)0x7ffc6db1, (q31_t)0xfe1c365a, (q31_t)0x7ffc55ca, (q31_t)0xfe15ee09,
	(q31_t)0x7ffc3d94, (q31_t)0xfe0fa5b8,
	(q31_t)0x7ffc250f, (q31_t)0xfe095d69, (q31_t)0x7ffc0c3b, (q31_t)0xfe03151c, (q31_t)0x7ffbf319, (q31_t)0xfdfccccf,
	(q31_t)0x7ffbd9a7, (q31_t)0xfdf68484,
	(q31_t)0x7ffbbfe6, (q31_t)0xfdf03c3a, (q31_t)0x7ffba5d7, (q31_t)0xfde9f3f1, (q31_t)0x7ffb8b78, (q31_t)0xfde3aba9,
	(q31_t)0x7ffb70cb, (q31_t)0xfddd6363,
	(q31_t)0x7ffb55ce, (q31_t)0xfdd71b1e, (q31_t)0x7ffb3a83, (q31_t)0xfdd0d2db, (q31_t)0x7ffb1ee9, (q31_t)0xfdca8a99,
	(q31_t)0x7ffb0300, (q31_t)0xfdc44258,
	(q31_t)0x7ffae6c7, (q31_t)0xfdbdfa18, (q31_t)0x7ffaca40, (q31_t)0xfdb7b1da, (q31_t)0x7ffaad6a, (q31_t)0xfdb1699e,
	(q31_t)0x7ffa9045, (q31_t)0xfdab2162,
	(q31_t)0x7ffa72d1, (q31_t)0xfda4d929, (q31_t)0x7ffa550e, (q31_t)0xfd9e90f0, (q31_t)0x7ffa36fc, (q31_t)0xfd9848b9,
	(q31_t)0x7ffa189c, (q31_t)0xfd920084,
	(q31_t)0x7ff9f9ec, (q31_t)0xfd8bb850, (q31_t)0x7ff9daed, (q31_t)0xfd85701e, (q31_t)0x7ff9bba0, (q31_t)0xfd7f27ed,
	(q31_t)0x7ff99c03, (q31_t)0xfd78dfbd,
	(q31_t)0x7ff97c18, (q31_t)0xfd729790, (q31_t)0x7ff95bdd, (q31_t)0xfd6c4f64, (q31_t)0x7ff93b54, (q31_t)0xfd660739,
	(q31_t)0x7ff91a7b, (q31_t)0xfd5fbf10,
	(q31_t)0x7ff8f954, (q31_t)0xfd5976e9, (q31_t)0x7ff8d7de, (q31_t)0xfd532ec3, (q31_t)0x7ff8b619, (q31_t)0xfd4ce69f,
	(q31_t)0x7ff89405, (q31_t)0xfd469e7c,
	(q31_t)0x7ff871a2, (q31_t)0xfd40565c, (q31_t)0x7ff84ef0, (q31_t)0xfd3a0e3d, (q31_t)0x7ff82bef, (q31_t)0xfd33c61f,
	(q31_t)0x7ff8089f, (q31_t)0xfd2d7e04,
	(q31_t)0x7ff7e500, (q31_t)0xfd2735ea, (q31_t)0x7ff7c113, (q31_t)0xfd20edd2, (q31_t)0x7ff79cd6, (q31_t)0xfd1aa5bc,
	(q31_t)0x7ff7784a, (q31_t)0xfd145da7,
	(q31_t)0x7ff75370, (q31_t)0xfd0e1594, (q31_t)0x7ff72e46, (q31_t)0xfd07cd83, (q31_t)0x7ff708ce, (q31_t)0xfd018574,
	(q31_t)0x7ff6e307, (q31_t)0xfcfb3d67,
	(q31_t)0x7ff6bcf0, (q31_t)0xfcf4f55c, (q31_t)0x7ff6968b, (q31_t)0xfceead52, (q31_t)0x7ff66fd7, (q31_t)0xfce8654b,
	(q31_t)0x7ff648d4, (q31_t)0xfce21d45,
	(q31_t)0x7ff62182, (q31_t)0xfcdbd541, (q31_t)0x7ff5f9e1, (q31_t)0xfcd58d3f, (q31_t)0x7ff5d1f1, (q31_t)0xfccf453f,
	(q31_t)0x7ff5a9b2, (q31_t)0xfcc8fd41,
	(q31_t)0x7ff58125, (q31_t)0xfcc2b545, (q31_t)0x7ff55848, (q31_t)0xfcbc6d4c, (q31_t)0x7ff52f1d, (q31_t)0xfcb62554,
	(q31_t)0x7ff505a2, (q31_t)0xfcafdd5e,
	(q31_t)0x7ff4dbd9, (q31_t)0xfca9956a, (q31_t)0x7ff4b1c0, (q31_t)0xfca34d78, (q31_t)0x7ff48759, (q31_t)0xfc9d0588,
	(q31_t)0x7ff45ca3, (q31_t)0xfc96bd9b,
	(q31_t)0x7ff4319d, (q31_t)0xfc9075af, (q31_t)0x7ff40649, (q31_t)0xfc8a2dc6, (q31_t)0x7ff3daa6, (q31_t)0xfc83e5de,
	(q31_t)0x7ff3aeb4, (q31_t)0xfc7d9df9,
	(q31_t)0x7ff38274, (q31_t)0xfc775616, (q31_t)0x7ff355e4, (q31_t)0xfc710e36, (q31_t)0x7ff32905, (q31_t)0xfc6ac657,
	(q31_t)0x7ff2fbd7, (q31_t)0xfc647e7b,
	(q31_t)0x7ff2ce5b, (q31_t)0xfc5e36a0, (q31_t)0x7ff2a08f, (q31_t)0xfc57eec9, (q31_t)0x7ff27275, (q31_t)0xfc51a6f3,
	(q31_t)0x7ff2440b, (q31_t)0xfc4b5f20,
	(q31_t)0x7ff21553, (q31_t)0xfc45174e, (q31_t)0x7ff1e64c, (q31_t)0xfc3ecf80, (q31_t)0x7ff1b6f6, (q31_t)0xfc3887b3,
	(q31_t)0x7ff18751, (q31_t)0xfc323fe9,
	(q31_t)0x7ff1575d, (q31_t)0xfc2bf821, (q31_t)0x7ff1271a, (q31_t)0xfc25b05c, (q31_t)0x7ff0f688, (q31_t)0xfc1f6899,
	(q31_t)0x7ff0c5a7, (q31_t)0xfc1920d8,
	(q31_t)0x7ff09478, (q31_t)0xfc12d91a, (q31_t)0x7ff062f9, (q31_t)0xfc0c915e, (q31_t)0x7ff0312c, (q31_t)0xfc0649a5,
	(q31_t)0x7fefff0f, (q31_t)0xfc0001ee,
	(q31_t)0x7fefcca4, (q31_t)0xfbf9ba39, (q31_t)0x7fef99ea, (q31_t)0xfbf37287, (q31_t)0x7fef66e1, (q31_t)0xfbed2ad8,
	(q31_t)0x7fef3388, (q31_t)0xfbe6e32b,
	(q31_t)0x7feeffe1, (q31_t)0xfbe09b80, (q31_t)0x7feecbec, (q31_t)0xfbda53d8, (q31_t)0x7fee97a7, (q31_t)0xfbd40c33,
	(q31_t)0x7fee6313, (q31_t)0xfbcdc490,
	(q31_t)0x7fee2e30, (q31_t)0xfbc77cf0, (q31_t)0x7fedf8ff, (q31_t)0xfbc13552, (q31_t)0x7fedc37e, (q31_t)0xfbbaedb7,
	(q31_t)0x7fed8daf, (q31_t)0xfbb4a61f,
	(q31_t)0x7fed5791, (q31_t)0xfbae5e89, (q31_t)0x7fed2123, (q31_t)0xfba816f6, (q31_t)0x7fecea67, (q31_t)0xfba1cf66,
	(q31_t)0x7fecb35c, (q31_t)0xfb9b87d8,
	(q31_t)0x7fec7c02, (q31_t)0xfb95404d, (q31_t)0x7fec4459, (q31_t)0xfb8ef8c5, (q31_t)0x7fec0c62, (q31_t)0xfb88b13f,
	(q31_t)0x7febd41b, (q31_t)0xfb8269bd,
	(q31_t)0x7feb9b85, (q31_t)0xfb7c223d, (q31_t)0x7feb62a1, (q31_t)0xfb75dac0, (q31_t)0x7feb296d, (q31_t)0xfb6f9345,
	(q31_t)0x7feaefeb, (q31_t)0xfb694bce,
	(q31_t)0x7feab61a, (q31_t)0xfb630459, (q31_t)0x7fea7bfa, (q31_t)0xfb5cbce7, (q31_t)0x7fea418b, (q31_t)0xfb567578,
	(q31_t)0x7fea06cd, (q31_t)0xfb502e0c,
	(q31_t)0x7fe9cbc0, (q31_t)0xfb49e6a3, (q31_t)0x7fe99064, (q31_t)0xfb439f3c, (q31_t)0x7fe954ba, (q31_t)0xfb3d57d9,
	(q31_t)0x7fe918c0, (q31_t)0xfb371078,
	(q31_t)0x7fe8dc78, (q31_t)0xfb30c91b, (q31_t)0x7fe89fe0, (q31_t)0xfb2a81c0, (q31_t)0x7fe862fa, (q31_t)0xfb243a69,
	(q31_t)0x7fe825c5, (q31_t)0xfb1df314,
	(q31_t)0x7fe7e841, (q31_t)0xfb17abc2, (q31_t)0x7fe7aa6e, (q31_t)0xfb116474, (q31_t)0x7fe76c4c, (q31_t)0xfb0b1d28,
	(q31_t)0x7fe72ddb, (q31_t)0xfb04d5e0,
	(q31_t)0x7fe6ef1c, (q31_t)0xfafe8e9b, (q31_t)0x7fe6b00d, (q31_t)0xfaf84758, (q31_t)0x7fe670b0, (q31_t)0xfaf20019,
	(q31_t)0x7fe63103, (q31_t)0xfaebb8dd,
	(q31_t)0x7fe5f108, (q31_t)0xfae571a4, (q31_t)0x7fe5b0be, (q31_t)0xfadf2a6e, (q31_t)0x7fe57025, (q31_t)0xfad8e33c,
	(q31_t)0x7fe52f3d, (q31_t)0xfad29c0c,
	(q31_t)0x7fe4ee06, (q31_t)0xfacc54e0, (q31_t)0x7fe4ac81, (q31_t)0xfac60db7, (q31_t)0x7fe46aac, (q31_t)0xfabfc691,
	(q31_t)0x7fe42889, (q31_t)0xfab97f6e,
	(q31_t)0x7fe3e616, (q31_t)0xfab3384f, (q31_t)0x7fe3a355, (q31_t)0xfaacf133, (q31_t)0x7fe36045, (q31_t)0xfaa6aa1a,
	(q31_t)0x7fe31ce6, (q31_t)0xfaa06305,
	(q31_t)0x7fe2d938, (q31_t)0xfa9a1bf3, (q31_t)0x7fe2953b, (q31_t)0xfa93d4e4, (q31_t)0x7fe250ef, (q31_t)0xfa8d8dd8,
	(q31_t)0x7fe20c55, (q31_t)0xfa8746d0,
	(q31_t)0x7fe1c76b, (q31_t)0xfa80ffcb, (q31_t)0x7fe18233, (q31_t)0xfa7ab8ca, (q31_t)0x7fe13cac, (q31_t)0xfa7471cc,
	(q31_t)0x7fe0f6d6, (q31_t)0xfa6e2ad1,
	(q31_t)0x7fe0b0b1, (q31_t)0xfa67e3da, (q31_t)0x7fe06a3d, (q31_t)0xfa619ce7, (q31_t)0x7fe0237a, (q31_t)0xfa5b55f7,
	(q31_t)0x7fdfdc69, (q31_t)0xfa550f0a,
	(q31_t)0x7fdf9508, (q31_t)0xfa4ec821, (q31_t)0x7fdf4d59, (q31_t)0xfa48813b, (q31_t)0x7fdf055a, (q31_t)0xfa423a59,
	(q31_t)0x7fdebd0d, (q31_t)0xfa3bf37a,
	(q31_t)0x7fde7471, (q31_t)0xfa35ac9f, (q31_t)0x7fde2b86, (q31_t)0xfa2f65c8, (q31_t)0x7fdde24d, (q31_t)0xfa291ef4,
	(q31_t)0x7fdd98c4, (q31_t)0xfa22d823,
	(q31_t)0x7fdd4eec, (q31_t)0xfa1c9157, (q31_t)0x7fdd04c6, (q31_t)0xfa164a8e, (q31_t)0x7fdcba51, (q31_t)0xfa1003c8,
	(q31_t)0x7fdc6f8d, (q31_t)0xfa09bd06,
	(q31_t)0x7fdc247a, (q31_t)0xfa037648, (q31_t)0x7fdbd918, (q31_t)0xf9fd2f8e, (q31_t)0x7fdb8d67, (q31_t)0xf9f6e8d7,
	(q31_t)0x7fdb4167, (q31_t)0xf9f0a224,
	(q31_t)0x7fdaf519, (q31_t)0xf9ea5b75, (q31_t)0x7fdaa87c, (q31_t)0xf9e414ca, (q31_t)0x7fda5b8f, (q31_t)0xf9ddce22,
	(q31_t)0x7fda0e54, (q31_t)0xf9d7877e,
	(q31_t)0x7fd9c0ca, (q31_t)0xf9d140de, (q31_t)0x7fd972f2, (q31_t)0xf9cafa42, (q31_t)0x7fd924ca, (q31_t)0xf9c4b3a9,
	(q31_t)0x7fd8d653, (q31_t)0xf9be6d15,
	(q31_t)0x7fd8878e, (q31_t)0xf9b82684, (q31_t)0x7fd8387a, (q31_t)0xf9b1dff7, (q31_t)0x7fd7e917, (q31_t)0xf9ab996e,
	(q31_t)0x7fd79965, (q31_t)0xf9a552e9,
	(q31_t)0x7fd74964, (q31_t)0xf99f0c68, (q31_t)0x7fd6f914, (q31_t)0xf998c5ea, (q31_t)0x7fd6a875, (q31_t)0xf9927f71,
	(q31_t)0x7fd65788, (q31_t)0xf98c38fc,
	(q31_t)0x7fd6064c, (q31_t)0xf985f28a, (q31_t)0x7fd5b4c1, (q31_t)0xf97fac1d, (q31_t)0x7fd562e7, (q31_t)0xf97965b4,
	(q31_t)0x7fd510be, (q31_t)0xf9731f4e,
	(q31_t)0x7fd4be46, (q31_t)0xf96cd8ed, (q31_t)0x7fd46b80, (q31_t)0xf9669290, (q31_t)0x7fd4186a, (q31_t)0xf9604c37,
	(q31_t)0x7fd3c506, (q31_t)0xf95a05e2,
	(q31_t)0x7fd37153, (q31_t)0xf953bf91, (q31_t)0x7fd31d51, (q31_t)0xf94d7944, (q31_t)0x7fd2c900, (q31_t)0xf94732fb,
	(q31_t)0x7fd27460, (q31_t)0xf940ecb7,
	(q31_t)0x7fd21f72, (q31_t)0xf93aa676, (q31_t)0x7fd1ca35, (q31_t)0xf934603a, (q31_t)0x7fd174a8, (q31_t)0xf92e1a02,
	(q31_t)0x7fd11ecd, (q31_t)0xf927d3ce,
	(q31_t)0x7fd0c8a3, (q31_t)0xf9218d9e, (q31_t)0x7fd0722b, (q31_t)0xf91b4773, (q31_t)0x7fd01b63, (q31_t)0xf915014c,
	(q31_t)0x7fcfc44d, (q31_t)0xf90ebb29,
	(q31_t)0x7fcf6ce8, (q31_t)0xf908750a, (q31_t)0x7fcf1533, (q31_t)0xf9022ef0, (q31_t)0x7fcebd31, (q31_t)0xf8fbe8da,
	(q31_t)0x7fce64df, (q31_t)0xf8f5a2c9,
	(q31_t)0x7fce0c3e, (q31_t)0xf8ef5cbb, (q31_t)0x7fcdb34f, (q31_t)0xf8e916b2, (q31_t)0x7fcd5a11, (q31_t)0xf8e2d0ae,
	(q31_t)0x7fcd0083, (q31_t)0xf8dc8aae,
	(q31_t)0x7fcca6a7, (q31_t)0xf8d644b2, (q31_t)0x7fcc4c7d, (q31_t)0xf8cffebb, (q31_t)0x7fcbf203, (q31_t)0xf8c9b8c8,
	(q31_t)0x7fcb973b, (q31_t)0xf8c372d9,
	(q31_t)0x7fcb3c23, (q31_t)0xf8bd2cef, (q31_t)0x7fcae0bd, (q31_t)0xf8b6e70a, (q31_t)0x7fca8508, (q31_t)0xf8b0a129,
	(q31_t)0x7fca2905, (q31_t)0xf8aa5b4c,
	(q31_t)0x7fc9ccb2, (q31_t)0xf8a41574, (q31_t)0x7fc97011, (q31_t)0xf89dcfa1, (q31_t)0x7fc91320, (q31_t)0xf89789d2,
	(q31_t)0x7fc8b5e1, (q31_t)0xf8914407,
	(q31_t)0x7fc85854, (q31_t)0xf88afe42, (q31_t)0x7fc7fa77, (q31_t)0xf884b880, (q31_t)0x7fc79c4b, (q31_t)0xf87e72c4,
	(q31_t)0x7fc73dd1, (q31_t)0xf8782d0c,
	(q31_t)0x7fc6df08, (q31_t)0xf871e759, (q31_t)0x7fc67ff0, (q31_t)0xf86ba1aa, (q31_t)0x7fc62089, (q31_t)0xf8655c00,
	(q31_t)0x7fc5c0d3, (q31_t)0xf85f165b,
	(q31_t)0x7fc560cf, (q31_t)0xf858d0bb, (q31_t)0x7fc5007c, (q31_t)0xf8528b1f, (q31_t)0x7fc49fda, (q31_t)0xf84c4588,
	(q31_t)0x7fc43ee9, (q31_t)0xf845fff5,
	(q31_t)0x7fc3dda9, (q31_t)0xf83fba68, (q31_t)0x7fc37c1b, (q31_t)0xf83974df, (q31_t)0x7fc31a3d, (q31_t)0xf8332f5b,
	(q31_t)0x7fc2b811, (q31_t)0xf82ce9dc,
	(q31_t)0x7fc25596, (q31_t)0xf826a462, (q31_t)0x7fc1f2cc, (q31_t)0xf8205eec, (q31_t)0x7fc18fb4, (q31_t)0xf81a197b,
	(q31_t)0x7fc12c4d, (q31_t)0xf813d410,
	(q31_t)0x7fc0c896, (q31_t)0xf80d8ea9, (q31_t)0x7fc06491, (q31_t)0xf8074947, (q31_t)0x7fc0003e, (q31_t)0xf80103ea,
	(q31_t)0x7fbf9b9b, (q31_t)0xf7fabe92,
	(q31_t)0x7fbf36aa, (q31_t)0xf7f4793e, (q31_t)0x7fbed16a, (q31_t)0xf7ee33f0, (q31_t)0x7fbe6bdb, (q31_t)0xf7e7eea7,
	(q31_t)0x7fbe05fd, (q31_t)0xf7e1a963,
	(q31_t)0x7fbd9fd0, (q31_t)0xf7db6423, (q31_t)0x7fbd3955, (q31_t)0xf7d51ee9, (q31_t)0x7fbcd28b, (q31_t)0xf7ced9b4,
	(q31_t)0x7fbc6b72, (q31_t)0xf7c89484,
	(q31_t)0x7fbc040a, (q31_t)0xf7c24f59, (q31_t)0x7fbb9c53, (q31_t)0xf7bc0a33, (q31_t)0x7fbb344e, (q31_t)0xf7b5c512,
	(q31_t)0x7fbacbfa, (q31_t)0xf7af7ff6,
	(q31_t)0x7fba6357, (q31_t)0xf7a93ae0, (q31_t)0x7fb9fa65, (q31_t)0xf7a2f5ce, (q31_t)0x7fb99125, (q31_t)0xf79cb0c2,
	(q31_t)0x7fb92796, (q31_t)0xf7966bbb,
	(q31_t)0x7fb8bdb8, (q31_t)0xf79026b9, (q31_t)0x7fb8538b, (q31_t)0xf789e1bc, (q31_t)0x7fb7e90f, (q31_t)0xf7839cc4,
	(q31_t)0x7fb77e45, (q31_t)0xf77d57d2,
	(q31_t)0x7fb7132b, (q31_t)0xf77712e5, (q31_t)0x7fb6a7c3, (q31_t)0xf770cdfd, (q31_t)0x7fb63c0d, (q31_t)0xf76a891b,
	(q31_t)0x7fb5d007, (q31_t)0xf764443d,
	(q31_t)0x7fb563b3, (q31_t)0xf75dff66, (q31_t)0x7fb4f710, (q31_t)0xf757ba93, (q31_t)0x7fb48a1e, (q31_t)0xf75175c6,
	(q31_t)0x7fb41cdd, (q31_t)0xf74b30fe,
	(q31_t)0x7fb3af4e, (q31_t)0xf744ec3b, (q31_t)0x7fb34170, (q31_t)0xf73ea77e, (q31_t)0x7fb2d343, (q31_t)0xf73862c6,
	(q31_t)0x7fb264c7, (q31_t)0xf7321e14,
	(q31_t)0x7fb1f5fc, (q31_t)0xf72bd967, (q31_t)0x7fb186e3, (q31_t)0xf72594c0, (q31_t)0x7fb1177b, (q31_t)0xf71f501e,
	(q31_t)0x7fb0a7c4, (q31_t)0xf7190b81,
	(q31_t)0x7fb037bf, (q31_t)0xf712c6ea, (q31_t)0x7fafc76a, (q31_t)0xf70c8259, (q31_t)0x7faf56c7, (q31_t)0xf7063dcd,
	(q31_t)0x7faee5d5, (q31_t)0xf6fff946,
	(q31_t)0x7fae7495, (q31_t)0xf6f9b4c6, (q31_t)0x7fae0305, (q31_t)0xf6f3704a, (q31_t)0x7fad9127, (q31_t)0xf6ed2bd4,
	(q31_t)0x7fad1efa, (q31_t)0xf6e6e764,
	(q31_t)0x7facac7f, (q31_t)0xf6e0a2fa, (q31_t)0x7fac39b4, (q31_t)0xf6da5e95, (q31_t)0x7fabc69b, (q31_t)0xf6d41a36,
	(q31_t)0x7fab5333, (q31_t)0xf6cdd5dc,
	(q31_t)0x7faadf7c, (q31_t)0xf6c79188, (q31_t)0x7faa6b77, (q31_t)0xf6c14d3a, (q31_t)0x7fa9f723, (q31_t)0xf6bb08f1,
	(q31_t)0x7fa98280, (q31_t)0xf6b4c4ae,
	(q31_t)0x7fa90d8e, (q31_t)0xf6ae8071, (q31_t)0x7fa8984e, (q31_t)0xf6a83c3a, (q31_t)0x7fa822bf, (q31_t)0xf6a1f808,
	(q31_t)0x7fa7ace1, (q31_t)0xf69bb3dd,
	(q31_t)0x7fa736b4, (q31_t)0xf6956fb7, (q31_t)0x7fa6c039, (q31_t)0xf68f2b96, (q31_t)0x7fa6496e, (q31_t)0xf688e77c,
	(q31_t)0x7fa5d256, (q31_t)0xf682a367,
	(q31_t)0x7fa55aee, (q31_t)0xf67c5f59, (q31_t)0x7fa4e338, (q31_t)0xf6761b50, (q31_t)0x7fa46b32, (q31_t)0xf66fd74d,
	(q31_t)0x7fa3f2df, (q31_t)0xf6699350,
	(q31_t)0x7fa37a3c, (q31_t)0xf6634f59, (q31_t)0x7fa3014b, (q31_t)0xf65d0b68, (q31_t)0x7fa2880b, (q31_t)0xf656c77c,
	(q31_t)0x7fa20e7c, (q31_t)0xf6508397,
	(q31_t)0x7fa1949e, (q31_t)0xf64a3fb8, (q31_t)0x7fa11a72, (q31_t)0xf643fbdf, (q31_t)0x7fa09ff7, (q31_t)0xf63db80b,
	(q31_t)0x7fa0252e, (q31_t)0xf637743e,
	(q31_t)0x7f9faa15, (q31_t)0xf6313077, (q31_t)0x7f9f2eae, (q31_t)0xf62aecb5, (q31_t)0x7f9eb2f8, (q31_t)0xf624a8fa,
	(q31_t)0x7f9e36f4, (q31_t)0xf61e6545,
	(q31_t)0x7f9dbaa0, (q31_t)0xf6182196, (q31_t)0x7f9d3dfe, (q31_t)0xf611dded, (q31_t)0x7f9cc10d, (q31_t)0xf60b9a4b,
	(q31_t)0x7f9c43ce, (q31_t)0xf60556ae,
	(q31_t)0x7f9bc640, (q31_t)0xf5ff1318, (q31_t)0x7f9b4863, (q31_t)0xf5f8cf87, (q31_t)0x7f9aca37, (q31_t)0xf5f28bfd,
	(q31_t)0x7f9a4bbd, (q31_t)0xf5ec4879,
	(q31_t)0x7f99ccf4, (q31_t)0xf5e604fc, (q31_t)0x7f994ddc, (q31_t)0xf5dfc184, (q31_t)0x7f98ce76, (q31_t)0xf5d97e13,
	(q31_t)0x7f984ec1, (q31_t)0xf5d33aa8,
	(q31_t)0x7f97cebd, (q31_t)0xf5ccf743, (q31_t)0x7f974e6a, (q31_t)0xf5c6b3e5, (q31_t)0x7f96cdc9, (q31_t)0xf5c0708d,
	(q31_t)0x7f964cd9, (q31_t)0xf5ba2d3b,
	(q31_t)0x7f95cb9a, (q31_t)0xf5b3e9f0, (q31_t)0x7f954a0d, (q31_t)0xf5ada6ab, (q31_t)0x7f94c831, (q31_t)0xf5a7636c,
	(q31_t)0x7f944606, (q31_t)0xf5a12034,
	(q31_t)0x7f93c38c, (q31_t)0xf59add02, (q31_t)0x7f9340c4, (q31_t)0xf59499d6, (q31_t)0x7f92bdad, (q31_t)0xf58e56b1,
	(q31_t)0x7f923a48, (q31_t)0xf5881393,
	(q31_t)0x7f91b694, (q31_t)0xf581d07b, (q31_t)0x7f913291, (q31_t)0xf57b8d69, (q31_t)0x7f90ae3f, (q31_t)0xf5754a5e,
	(q31_t)0x7f90299f, (q31_t)0xf56f0759,
	(q31_t)0x7f8fa4b0, (q31_t)0xf568c45b, (q31_t)0x7f8f1f72, (q31_t)0xf5628163, (q31_t)0x7f8e99e6, (q31_t)0xf55c3e72,
	(q31_t)0x7f8e140a, (q31_t)0xf555fb88,
	(q31_t)0x7f8d8de1, (q31_t)0xf54fb8a4, (q31_t)0x7f8d0768, (q31_t)0xf54975c6, (q31_t)0x7f8c80a1, (q31_t)0xf54332ef,
	(q31_t)0x7f8bf98b, (q31_t)0xf53cf01f,
	(q31_t)0x7f8b7227, (q31_t)0xf536ad56, (q31_t)0x7f8aea74, (q31_t)0xf5306a93, (q31_t)0x7f8a6272, (q31_t)0xf52a27d7,
	(q31_t)0x7f89da21, (q31_t)0xf523e521,
	(q31_t)0x7f895182, (q31_t)0xf51da273, (q31_t)0x7f88c894, (q31_t)0xf5175fca, (q31_t)0x7f883f58, (q31_t)0xf5111d29,
	(q31_t)0x7f87b5cd, (q31_t)0xf50ada8f,
	(q31_t)0x7f872bf3, (q31_t)0xf50497fb, (q31_t)0x7f86a1ca, (q31_t)0xf4fe556e, (q31_t)0x7f861753, (q31_t)0xf4f812e7,
	(q31_t)0x7f858c8d, (q31_t)0xf4f1d068,
	(q31_t)0x7f850179, (q31_t)0xf4eb8def, (q31_t)0x7f847616, (q31_t)0xf4e54b7d, (q31_t)0x7f83ea64, (q31_t)0xf4df0912,
	(q31_t)0x7f835e64, (q31_t)0xf4d8c6ae,
	(q31_t)0x7f82d214, (q31_t)0xf4d28451, (q31_t)0x7f824577, (q31_t)0xf4cc41fb, (q31_t)0x7f81b88a, (q31_t)0xf4c5ffab,
	(q31_t)0x7f812b4f, (q31_t)0xf4bfbd63,
	(q31_t)0x7f809dc5, (q31_t)0xf4b97b21, (q31_t)0x7f800fed, (q31_t)0xf4b338e7, (q31_t)0x7f7f81c6, (q31_t)0xf4acf6b3,
	(q31_t)0x7f7ef350, (q31_t)0xf4a6b486,
	(q31_t)0x7f7e648c, (q31_t)0xf4a07261, (q31_t)0x7f7dd579, (q31_t)0xf49a3042, (q31_t)0x7f7d4617, (q31_t)0xf493ee2b,
	(q31_t)0x7f7cb667, (q31_t)0xf48dac1a,
	(q31_t)0x7f7c2668, (q31_t)0xf4876a10, (q31_t)0x7f7b961b, (q31_t)0xf481280e, (q31_t)0x7f7b057e, (q31_t)0xf47ae613,
	(q31_t)0x7f7a7494, (q31_t)0xf474a41f,
	(q31_t)0x7f79e35a, (q31_t)0xf46e6231, (q31_t)0x7f7951d2, (q31_t)0xf468204b, (q31_t)0x7f78bffb, (q31_t)0xf461de6d,
	(q31_t)0x7f782dd6, (q31_t)0xf45b9c95,
	(q31_t)0x7f779b62, (q31_t)0xf4555ac5, (q31_t)0x7f77089f, (q31_t)0xf44f18fb, (q31_t)0x7f76758e, (q31_t)0xf448d739,
	(q31_t)0x7f75e22e, (q31_t)0xf442957e,
	(q31_t)0x7f754e80, (q31_t)0xf43c53cb, (q31_t)0x7f74ba83, (q31_t)0xf436121e, (q31_t)0x7f742637, (q31_t)0xf42fd079,
	(q31_t)0x7f73919d, (q31_t)0xf4298edc,
	(q31_t)0x7f72fcb4, (q31_t)0xf4234d45, (q31_t)0x7f72677c, (q31_t)0xf41d0bb6, (q31_t)0x7f71d1f6, (q31_t)0xf416ca2e,
	(q31_t)0x7f713c21, (q31_t)0xf41088ae,
	(q31_t)0x7f70a5fe, (q31_t)0xf40a4735, (q31_t)0x7f700f8c, (q31_t)0xf40405c3, (q31_t)0x7f6f78cb, (q31_t)0xf3fdc459,
	(q31_t)0x7f6ee1bc, (q31_t)0xf3f782f6,
	(q31_t)0x7f6e4a5e, (q31_t)0xf3f1419a, (q31_t)0x7f6db2b1, (q31_t)0xf3eb0046, (q31_t)0x7f6d1ab6, (q31_t)0xf3e4bef9,
	(q31_t)0x7f6c826d, (q31_t)0xf3de7db4,
	(q31_t)0x7f6be9d4, (q31_t)0xf3d83c77, (q31_t)0x7f6b50ed, (q31_t)0xf3d1fb40, (q31_t)0x7f6ab7b8, (q31_t)0xf3cbba12,
	(q31_t)0x7f6a1e34, (q31_t)0xf3c578eb,
	(q31_t)0x7f698461, (q31_t)0xf3bf37cb, (q31_t)0x7f68ea40, (q31_t)0xf3b8f6b3, (q31_t)0x7f684fd0, (q31_t)0xf3b2b5a3,
	(q31_t)0x7f67b512, (q31_t)0xf3ac749a,
	(q31_t)0x7f671a05, (q31_t)0xf3a63398, (q31_t)0x7f667ea9, (q31_t)0xf39ff29f, (q31_t)0x7f65e2ff, (q31_t)0xf399b1ad,
	(q31_t)0x7f654706, (q31_t)0xf39370c2,
	(q31_t)0x7f64aabf, (q31_t)0xf38d2fe0, (q31_t)0x7f640e29, (q31_t)0xf386ef05, (q31_t)0x7f637144, (q31_t)0xf380ae31,
	(q31_t)0x7f62d411, (q31_t)0xf37a6d66,
	(q31_t)0x7f62368f, (q31_t)0xf3742ca2, (q31_t)0x7f6198bf, (q31_t)0xf36debe6, (q31_t)0x7f60faa0, (q31_t)0xf367ab31,
	(q31_t)0x7f605c33, (q31_t)0xf3616a85,
	(q31_t)0x7f5fbd77, (q31_t)0xf35b29e0, (q31_t)0x7f5f1e6c, (q31_t)0xf354e943, (q31_t)0x7f5e7f13, (q31_t)0xf34ea8ae,
	(q31_t)0x7f5ddf6b, (q31_t)0xf3486820,
	(q31_t)0x7f5d3f75, (q31_t)0xf342279b, (q31_t)0x7f5c9f30, (q31_t)0xf33be71d, (q31_t)0x7f5bfe9d, (q31_t)0xf335a6a7,
	(q31_t)0x7f5b5dbb, (q31_t)0xf32f6639,
	(q31_t)0x7f5abc8a, (q31_t)0xf32925d3, (q31_t)0x7f5a1b0b, (q31_t)0xf322e575, (q31_t)0x7f59793e, (q31_t)0xf31ca51f,
	(q31_t)0x7f58d721, (q31_t)0xf31664d1,
	(q31_t)0x7f5834b7, (q31_t)0xf310248a, (q31_t)0x7f5791fd, (q31_t)0xf309e44c, (q31_t)0x7f56eef5, (q31_t)0xf303a416,
	(q31_t)0x7f564b9f, (q31_t)0xf2fd63e8,
	(q31_t)0x7f55a7fa, (q31_t)0xf2f723c1, (q31_t)0x7f550407, (q31_t)0xf2f0e3a3, (q31_t)0x7f545fc5, (q31_t)0xf2eaa38d,
	(q31_t)0x7f53bb34, (q31_t)0xf2e4637f,
	(q31_t)0x7f531655, (q31_t)0xf2de2379, (q31_t)0x7f527127, (q31_t)0xf2d7e37b, (q31_t)0x7f51cbab, (q31_t)0xf2d1a385,
	(q31_t)0x7f5125e0, (q31_t)0xf2cb6398,
	(q31_t)0x7f507fc7, (q31_t)0xf2c523b2, (q31_t)0x7f4fd95f, (q31_t)0xf2bee3d5, (q31_t)0x7f4f32a9, (q31_t)0xf2b8a400,
	(q31_t)0x7f4e8ba4, (q31_t)0xf2b26433,
	(q31_t)0x7f4de451, (q31_t)0xf2ac246e, (q31_t)0x7f4d3caf, (q31_t)0xf2a5e4b1, (q31_t)0x7f4c94be, (q31_t)0xf29fa4fd,
	(q31_t)0x7f4bec7f, (q31_t)0xf2996551,
	(q31_t)0x7f4b43f2, (q31_t)0xf29325ad, (q31_t)0x7f4a9b16, (q31_t)0xf28ce612, (q31_t)0x7f49f1eb, (q31_t)0xf286a67e,
	(q31_t)0x7f494872, (q31_t)0xf28066f4,
	(q31_t)0x7f489eaa, (q31_t)0xf27a2771, (q31_t)0x7f47f494, (q31_t)0xf273e7f7, (q31_t)0x7f474a30, (q31_t)0xf26da885,
	(q31_t)0x7f469f7d, (q31_t)0xf267691b,
	(q31_t)0x7f45f47b, (q31_t)0xf26129ba, (q31_t)0x7f45492b, (q31_t)0xf25aea61, (q31_t)0x7f449d8c, (q31_t)0xf254ab11,
	(q31_t)0x7f43f19f, (q31_t)0xf24e6bc9,
	(q31_t)0x7f434563, (q31_t)0xf2482c8a, (q31_t)0x7f4298d9, (q31_t)0xf241ed53, (q31_t)0x7f41ec01, (q31_t)0xf23bae24,
	(q31_t)0x7f413ed9, (q31_t)0xf2356efe,
	(q31_t)0x7f409164, (q31_t)0xf22f2fe1, (q31_t)0x7f3fe3a0, (q31_t)0xf228f0cc, (q31_t)0x7f3f358d, (q31_t)0xf222b1c0,
	(q31_t)0x7f3e872c, (q31_t)0xf21c72bc,
	(q31_t)0x7f3dd87c, (q31_t)0xf21633c0, (q31_t)0x7f3d297e, (q31_t)0xf20ff4ce, (q31_t)0x7f3c7a31, (q31_t)0xf209b5e4,
	(q31_t)0x7f3bca96, (q31_t)0xf2037702,
	(q31_t)0x7f3b1aad, (q31_t)0xf1fd3829, (q31_t)0x7f3a6a75, (q31_t)0xf1f6f959, (q31_t)0x7f39b9ee, (q31_t)0xf1f0ba91,
	(q31_t)0x7f390919, (q31_t)0xf1ea7bd2,
	(q31_t)0x7f3857f6, (q31_t)0xf1e43d1c, (q31_t)0x7f37a684, (q31_t)0xf1ddfe6f, (q31_t)0x7f36f4c3, (q31_t)0xf1d7bfca,
	(q31_t)0x7f3642b4, (q31_t)0xf1d1812e,
	(q31_t)0x7f359057, (q31_t)0xf1cb429a, (q31_t)0x7f34ddab, (q31_t)0xf1c50410, (q31_t)0x7f342ab1, (q31_t)0xf1bec58e,
	(q31_t)0x7f337768, (q31_t)0xf1b88715,
	(q31_t)0x7f32c3d1, (q31_t)0xf1b248a5, (q31_t)0x7f320feb, (q31_t)0xf1ac0a3e, (q31_t)0x7f315bb7, (q31_t)0xf1a5cbdf,
	(q31_t)0x7f30a734, (q31_t)0xf19f8d89,
	(q31_t)0x7f2ff263, (q31_t)0xf1994f3d, (q31_t)0x7f2f3d44, (q31_t)0xf19310f9, (q31_t)0x7f2e87d6, (q31_t)0xf18cd2be,
	(q31_t)0x7f2dd219, (q31_t)0xf186948c,
	(q31_t)0x7f2d1c0e, (q31_t)0xf1805662, (q31_t)0x7f2c65b5, (q31_t)0xf17a1842, (q31_t)0x7f2baf0d, (q31_t)0xf173da2b,
	(q31_t)0x7f2af817, (q31_t)0xf16d9c1d,
	(q31_t)0x7f2a40d2, (q31_t)0xf1675e17, (q31_t)0x7f29893f, (q31_t)0xf161201b, (q31_t)0x7f28d15d, (q31_t)0xf15ae228,
	(q31_t)0x7f28192d, (q31_t)0xf154a43d,
	(q31_t)0x7f2760af, (q31_t)0xf14e665c, (q31_t)0x7f26a7e2, (q31_t)0xf1482884, (q31_t)0x7f25eec7, (q31_t)0xf141eab5,
	(q31_t)0x7f25355d, (q31_t)0xf13bacef,
	(q31_t)0x7f247ba5, (q31_t)0xf1356f32, (q31_t)0x7f23c19e, (q31_t)0xf12f317e, (q31_t)0x7f230749, (q31_t)0xf128f3d4,
	(q31_t)0x7f224ca6, (q31_t)0xf122b632,
	(q31_t)0x7f2191b4, (q31_t)0xf11c789a, (q31_t)0x7f20d674, (q31_t)0xf1163b0b, (q31_t)0x7f201ae5, (q31_t)0xf10ffd85,
	(q31_t)0x7f1f5f08, (q31_t)0xf109c009,
	(q31_t)0x7f1ea2dc, (q31_t)0xf1038295, (q31_t)0x7f1de662, (q31_t)0xf0fd452b, (q31_t)0x7f1d299a, (q31_t)0xf0f707ca,
	(q31_t)0x7f1c6c83, (q31_t)0xf0f0ca72,
	(q31_t)0x7f1baf1e, (q31_t)0xf0ea8d24, (q31_t)0x7f1af16a, (q31_t)0xf0e44fdf, (q31_t)0x7f1a3368, (q31_t)0xf0de12a3,
	(q31_t)0x7f197518, (q31_t)0xf0d7d571,
	(q31_t)0x7f18b679, (q31_t)0xf0d19848, (q31_t)0x7f17f78c, (q31_t)0xf0cb5b28, (q31_t)0x7f173850, (q31_t)0xf0c51e12,
	(q31_t)0x7f1678c6, (q31_t)0xf0bee105,
	(q31_t)0x7f15b8ee, (q31_t)0xf0b8a401, (q31_t)0x7f14f8c7, (q31_t)0xf0b26707, (q31_t)0x7f143852, (q31_t)0xf0ac2a16,
	(q31_t)0x7f13778e, (q31_t)0xf0a5ed2f,
	(q31_t)0x7f12b67c, (q31_t)0xf09fb051, (q31_t)0x7f11f51c, (q31_t)0xf099737d, (q31_t)0x7f11336d, (q31_t)0xf09336b2,
	(q31_t)0x7f107170, (q31_t)0xf08cf9f1,
	(q31_t)0x7f0faf25, (q31_t)0xf086bd39, (q31_t)0x7f0eec8b, (q31_t)0xf080808b, (q31_t)0x7f0e29a3, (q31_t)0xf07a43e7,
	(q31_t)0x7f0d666c, (q31_t)0xf074074c,
	(q31_t)0x7f0ca2e7, (q31_t)0xf06dcaba, (q31_t)0x7f0bdf14, (q31_t)0xf0678e32, (q31_t)0x7f0b1af2, (q31_t)0xf06151b4,
	(q31_t)0x7f0a5682, (q31_t)0xf05b1540,
	(q31_t)0x7f0991c4, (q31_t)0xf054d8d5, (q31_t)0x7f08ccb7, (q31_t)0xf04e9c73, (q31_t)0x7f08075c, (q31_t)0xf048601c,
	(q31_t)0x7f0741b2, (q31_t)0xf04223ce,
	(q31_t)0x7f067bba, (q31_t)0xf03be78a, (q31_t)0x7f05b574, (q31_t)0xf035ab4f, (q31_t)0x7f04eedf, (q31_t)0xf02f6f1f,
	(q31_t)0x7f0427fc, (q31_t)0xf02932f8,
	(q31_t)0x7f0360cb, (q31_t)0xf022f6da, (q31_t)0x7f02994b, (q31_t)0xf01cbac7, (q31_t)0x7f01d17d, (q31_t)0xf0167ebd,
	(q31_t)0x7f010961, (q31_t)0xf01042be,
	(q31_t)0x7f0040f6, (q31_t)0xf00a06c8, (q31_t)0x7eff783d, (q31_t)0xf003cadc, (q31_t)0x7efeaf36, (q31_t)0xeffd8ef9,
	(q31_t)0x7efde5e0, (q31_t)0xeff75321,
	(q31_t)0x7efd1c3c, (q31_t)0xeff11753, (q31_t)0x7efc524a, (q31_t)0xefeadb8e, (q31_t)0x7efb8809, (q31_t)0xefe49fd3,
	(q31_t)0x7efabd7a, (q31_t)0xefde6423,
	(q31_t)0x7ef9f29d, (q31_t)0xefd8287c, (q31_t)0x7ef92771, (q31_t)0xefd1ecdf, (q31_t)0x7ef85bf7, (q31_t)0xefcbb14c,
	(q31_t)0x7ef7902f, (q31_t)0xefc575c3,
	(q31_t)0x7ef6c418, (q31_t)0xefbf3a45, (q31_t)0x7ef5f7b3, (q31_t)0xefb8fed0, (q31_t)0x7ef52b00, (q31_t)0xefb2c365,
	(q31_t)0x7ef45dfe, (q31_t)0xefac8804,
	(q31_t)0x7ef390ae, (q31_t)0xefa64cae, (q31_t)0x7ef2c310, (q31_t)0xefa01161, (q31_t)0x7ef1f524, (q31_t)0xef99d61f,
	(q31_t)0x7ef126e9, (q31_t)0xef939ae6,
	(q31_t)0x7ef05860, (q31_t)0xef8d5fb8, (q31_t)0x7eef8988, (q31_t)0xef872494, (q31_t)0x7eeeba62, (q31_t)0xef80e97a,
	(q31_t)0x7eedeaee, (q31_t)0xef7aae6b,
	(q31_t)0x7eed1b2c, (q31_t)0xef747365, (q31_t)0x7eec4b1b, (q31_t)0xef6e386a, (q31_t)0x7eeb7abc, (q31_t)0xef67fd79,
	(q31_t)0x7eeaaa0f, (q31_t)0xef61c292,
	(q31_t)0x7ee9d914, (q31_t)0xef5b87b5, (q31_t)0x7ee907ca, (q31_t)0xef554ce3, (q31_t)0x7ee83632, (q31_t)0xef4f121b,
	(q31_t)0x7ee7644c, (q31_t)0xef48d75d,
	(q31_t)0x7ee69217, (q31_t)0xef429caa, (q31_t)0x7ee5bf94, (q31_t)0xef3c6201, (q31_t)0x7ee4ecc3, (q31_t)0xef362762,
	(q31_t)0x7ee419a3, (q31_t)0xef2feccd,
	(q31_t)0x7ee34636, (q31_t)0xef29b243, (q31_t)0x7ee2727a, (q31_t)0xef2377c4, (q31_t)0x7ee19e6f, (q31_t)0xef1d3d4e,
	(q31_t)0x7ee0ca17, (q31_t)0xef1702e4,
	(q31_t)0x7edff570, (q31_t)0xef10c883, (q31_t)0x7edf207b, (q31_t)0xef0a8e2d, (q31_t)0x7ede4b38, (q31_t)0xef0453e2,
	(q31_t)0x7edd75a6, (q31_t)0xeefe19a1,
	(q31_t)0x7edc9fc6, (q31_t)0xeef7df6a, (q31_t)0x7edbc998, (q31_t)0xeef1a53e, (q31_t)0x7edaf31c, (q31_t)0xeeeb6b1c,
	(q31_t)0x7eda1c51, (q31_t)0xeee53105,
	(q31_t)0x7ed94538, (q31_t)0xeedef6f9, (q31_t)0x7ed86dd1, (q31_t)0xeed8bcf7, (q31_t)0x7ed7961c, (q31_t)0xeed28300,
	(q31_t)0x7ed6be18, (q31_t)0xeecc4913,
	(q31_t)0x7ed5e5c6, (q31_t)0xeec60f31, (q31_t)0x7ed50d26, (q31_t)0xeebfd55a, (q31_t)0x7ed43438, (q31_t)0xeeb99b8d,
	(q31_t)0x7ed35afb, (q31_t)0xeeb361cb,
	(q31_t)0x7ed28171, (q31_t)0xeead2813, (q31_t)0x7ed1a798, (q31_t)0xeea6ee66, (q31_t)0x7ed0cd70, (q31_t)0xeea0b4c4,
	(q31_t)0x7ecff2fb, (q31_t)0xee9a7b2d,
	(q31_t)0x7ecf1837, (q31_t)0xee9441a0, (q31_t)0x7ece3d25, (q31_t)0xee8e081e, (q31_t)0x7ecd61c5, (q31_t)0xee87cea7,
	(q31_t)0x7ecc8617, (q31_t)0xee81953b,
	(q31_t)0x7ecbaa1a, (q31_t)0xee7b5bd9, (q31_t)0x7ecacdd0, (q31_t)0xee752283, (q31_t)0x7ec9f137, (q31_t)0xee6ee937,
	(q31_t)0x7ec9144f, (q31_t)0xee68aff6,
	(q31_t)0x7ec8371a, (q31_t)0xee6276bf, (q31_t)0x7ec75996, (q31_t)0xee5c3d94, (q31_t)0x7ec67bc5, (q31_t)0xee560473,
	(q31_t)0x7ec59da5, (q31_t)0xee4fcb5e,
	(q31_t)0x7ec4bf36, (q31_t)0xee499253, (q31_t)0x7ec3e07a, (q31_t)0xee435953, (q31_t)0x7ec3016f, (q31_t)0xee3d205e,
	(q31_t)0x7ec22217, (q31_t)0xee36e775,
	(q31_t)0x7ec14270, (q31_t)0xee30ae96, (q31_t)0x7ec0627a, (q31_t)0xee2a75c2, (q31_t)0x7ebf8237, (q31_t)0xee243cf9,
	(q31_t)0x7ebea1a6, (q31_t)0xee1e043b,
	(q31_t)0x7ebdc0c6, (q31_t)0xee17cb88, (q31_t)0x7ebcdf98, (q31_t)0xee1192e0, (q31_t)0x7ebbfe1c, (q31_t)0xee0b5a43,
	(q31_t)0x7ebb1c52, (q31_t)0xee0521b2,
	(q31_t)0x7eba3a39, (q31_t)0xedfee92b, (q31_t)0x7eb957d2, (q31_t)0xedf8b0b0, (q31_t)0x7eb8751e, (q31_t)0xedf2783f,
	(q31_t)0x7eb7921b, (q31_t)0xedec3fda,
	(q31_t)0x7eb6aeca, (q31_t)0xede60780, (q31_t)0x7eb5cb2a, (q31_t)0xeddfcf31, (q31_t)0x7eb4e73d, (q31_t)0xedd996ed,
	(q31_t)0x7eb40301, (q31_t)0xedd35eb5,
	(q31_t)0x7eb31e78, (q31_t)0xedcd2687, (q31_t)0x7eb239a0, (q31_t)0xedc6ee65, (q31_t)0x7eb1547a, (q31_t)0xedc0b64e,
	(q31_t)0x7eb06f05, (q31_t)0xedba7e43,
	(q31_t)0x7eaf8943, (q31_t)0xedb44642, (q31_t)0x7eaea333, (q31_t)0xedae0e4d, (q31_t)0x7eadbcd4, (q31_t)0xeda7d664,
	(q31_t)0x7eacd627, (q31_t)0xeda19e85,
	(q31_t)0x7eabef2c, (q31_t)0xed9b66b2, (q31_t)0x7eab07e3, (q31_t)0xed952eea, (q31_t)0x7eaa204c, (q31_t)0xed8ef72e,
	(q31_t)0x7ea93867, (q31_t)0xed88bf7d,
	(q31_t)0x7ea85033, (q31_t)0xed8287d7, (q31_t)0x7ea767b2, (q31_t)0xed7c503d, (q31_t)0x7ea67ee2, (q31_t)0xed7618ae,
	(q31_t)0x7ea595c4, (q31_t)0xed6fe12b,
	(q31_t)0x7ea4ac58, (q31_t)0xed69a9b3, (q31_t)0x7ea3c29e, (q31_t)0xed637246, (q31_t)0x7ea2d896, (q31_t)0xed5d3ae5,
	(q31_t)0x7ea1ee3f, (q31_t)0xed570390,
	(q31_t)0x7ea1039b, (q31_t)0xed50cc46, (q31_t)0x7ea018a8, (q31_t)0xed4a9507, (q31_t)0x7e9f2d68, (q31_t)0xed445dd5,
	(q31_t)0x7e9e41d9, (q31_t)0xed3e26ad,
	(q31_t)0x7e9d55fc, (q31_t)0xed37ef91, (q31_t)0x7e9c69d1, (q31_t)0xed31b881, (q31_t)0x7e9b7d58, (q31_t)0xed2b817d,
	(q31_t)0x7e9a9091, (q31_t)0xed254a84,
	(q31_t)0x7e99a37c, (q31_t)0xed1f1396, (q31_t)0x7e98b618, (q31_t)0xed18dcb5, (q31_t)0x7e97c867, (q31_t)0xed12a5df,
	(q31_t)0x7e96da67, (q31_t)0xed0c6f14,
	(q31_t)0x7e95ec1a, (q31_t)0xed063856, (q31_t)0x7e94fd7e, (q31_t)0xed0001a3, (q31_t)0x7e940e94, (q31_t)0xecf9cafb,
	(q31_t)0x7e931f5c, (q31_t)0xecf39460,
	(q31_t)0x7e922fd6, (q31_t)0xeced5dd0, (q31_t)0x7e914002, (q31_t)0xece7274c, (q31_t)0x7e904fe0, (q31_t)0xece0f0d4,
	(q31_t)0x7e8f5f70, (q31_t)0xecdaba67,
	(q31_t)0x7e8e6eb2, (q31_t)0xecd48407, (q31_t)0x7e8d7da6, (q31_t)0xecce4db2, (q31_t)0x7e8c8c4b, (q31_t)0xecc81769,
	(q31_t)0x7e8b9aa3, (q31_t)0xecc1e12c,
	(q31_t)0x7e8aa8ac, (q31_t)0xecbbaafb, (q31_t)0x7e89b668, (q31_t)0xecb574d5, (q31_t)0x7e88c3d5, (q31_t)0xecaf3ebc,
	(q31_t)0x7e87d0f5, (q31_t)0xeca908ae,
	(q31_t)0x7e86ddc6, (q31_t)0xeca2d2ad, (q31_t)0x7e85ea49, (q31_t)0xec9c9cb7, (q31_t)0x7e84f67e, (q31_t)0xec9666cd,
	(q31_t)0x7e840265, (q31_t)0xec9030f0,
	(q31_t)0x7e830dff, (q31_t)0xec89fb1e, (q31_t)0x7e82194a, (q31_t)0xec83c558, (q31_t)0x7e812447, (q31_t)0xec7d8f9e,
	(q31_t)0x7e802ef6, (q31_t)0xec7759f1,
	(q31_t)0x7e7f3957, (q31_t)0xec71244f, (q31_t)0x7e7e436a, (q31_t)0xec6aeeba, (q31_t)0x7e7d4d2f, (q31_t)0xec64b930,
	(q31_t)0x7e7c56a5, (q31_t)0xec5e83b3,
	(q31_t)0x7e7b5fce, (q31_t)0xec584e41, (q31_t)0x7e7a68a9, (q31_t)0xec5218dc, (q31_t)0x7e797136, (q31_t)0xec4be383,
	(q31_t)0x7e787975, (q31_t)0xec45ae36,
	(q31_t)0x7e778166, (q31_t)0xec3f78f6, (q31_t)0x7e768908, (q31_t)0xec3943c1, (q31_t)0x7e75905d, (q31_t)0xec330e99,
	(q31_t)0x7e749764, (q31_t)0xec2cd97d,
	(q31_t)0x7e739e1d, (q31_t)0xec26a46d, (q31_t)0x7e72a488, (q31_t)0xec206f69, (q31_t)0x7e71aaa4, (q31_t)0xec1a3a72,
	(q31_t)0x7e70b073, (q31_t)0xec140587,
	(q31_t)0x7e6fb5f4, (q31_t)0xec0dd0a8, (q31_t)0x7e6ebb27, (q31_t)0xec079bd6, (q31_t)0x7e6dc00c, (q31_t)0xec01670f,
	(q31_t)0x7e6cc4a2, (q31_t)0xebfb3256,
	(q31_t)0x7e6bc8eb, (q31_t)0xebf4fda8, (q31_t)0x7e6acce6, (q31_t)0xebeec907, (q31_t)0x7e69d093, (q31_t)0xebe89472,
	(q31_t)0x7e68d3f2, (q31_t)0xebe25fea,
	(q31_t)0x7e67d703, (q31_t)0xebdc2b6e, (q31_t)0x7e66d9c6, (q31_t)0xebd5f6fe, (q31_t)0x7e65dc3b, (q31_t)0xebcfc29b,
	(q31_t)0x7e64de62, (q31_t)0xebc98e45,
	(q31_t)0x7e63e03b, (q31_t)0xebc359fb, (q31_t)0x7e62e1c6, (q31_t)0xebbd25bd, (q31_t)0x7e61e303, (q31_t)0xebb6f18c,
	(q31_t)0x7e60e3f2, (q31_t)0xebb0bd67,
	(q31_t)0x7e5fe493, (q31_t)0xebaa894f, (q31_t)0x7e5ee4e6, (q31_t)0xeba45543, (q31_t)0x7e5de4ec, (q31_t)0xeb9e2144,
	(q31_t)0x7e5ce4a3, (q31_t)0xeb97ed52,
	(q31_t)0x7e5be40c, (q31_t)0xeb91b96c, (q31_t)0x7e5ae328, (q31_t)0xeb8b8593, (q31_t)0x7e59e1f5, (q31_t)0xeb8551c6,
	(q31_t)0x7e58e075, (q31_t)0xeb7f1e06,
	(q31_t)0x7e57dea7, (q31_t)0xeb78ea52, (q31_t)0x7e56dc8a, (q31_t)0xeb72b6ac, (q31_t)0x7e55da20, (q31_t)0xeb6c8312,
	(q31_t)0x7e54d768, (q31_t)0xeb664f84,
	(q31_t)0x7e53d462, (q31_t)0xeb601c04, (q31_t)0x7e52d10e, (q31_t)0xeb59e890, (q31_t)0x7e51cd6c, (q31_t)0xeb53b529,
	(q31_t)0x7e50c97c, (q31_t)0xeb4d81ce,
	(q31_t)0x7e4fc53e, (q31_t)0xeb474e81, (q31_t)0x7e4ec0b2, (q31_t)0xeb411b40, (q31_t)0x7e4dbbd9, (q31_t)0xeb3ae80c,
	(q31_t)0x7e4cb6b1, (q31_t)0xeb34b4e4,
	(q31_t)0x7e4bb13c, (q31_t)0xeb2e81ca, (q31_t)0x7e4aab78, (q31_t)0xeb284ebc, (q31_t)0x7e49a567, (q31_t)0xeb221bbb,
	(q31_t)0x7e489f08, (q31_t)0xeb1be8c8,
	(q31_t)0x7e47985b, (q31_t)0xeb15b5e1, (q31_t)0x7e469160, (q31_t)0xeb0f8307, (q31_t)0x7e458a17, (q31_t)0xeb095039,
	(q31_t)0x7e448281, (q31_t)0xeb031d79,
	(q31_t)0x7e437a9c, (q31_t)0xeafceac6, (q31_t)0x7e427269, (q31_t)0xeaf6b81f, (q31_t)0x7e4169e9, (q31_t)0xeaf08586,
	(q31_t)0x7e40611b, (q31_t)0xeaea52fa,
	(q31_t)0x7e3f57ff, (q31_t)0xeae4207a, (q31_t)0x7e3e4e95, (q31_t)0xeaddee08, (q31_t)0x7e3d44dd, (q31_t)0xead7bba3,
	(q31_t)0x7e3c3ad7, (q31_t)0xead1894b,
	(q31_t)0x7e3b3083, (q31_t)0xeacb56ff, (q31_t)0x7e3a25e2, (q31_t)0xeac524c1, (q31_t)0x7e391af3, (q31_t)0xeabef290,
	(q31_t)0x7e380fb5, (q31_t)0xeab8c06c,
	(q31_t)0x7e37042a, (q31_t)0xeab28e56, (q31_t)0x7e35f851, (q31_t)0xeaac5c4c, (q31_t)0x7e34ec2b, (q31_t)0xeaa62a4f,
	(q31_t)0x7e33dfb6, (q31_t)0xea9ff860,
	(q31_t)0x7e32d2f4, (q31_t)0xea99c67e, (q31_t)0x7e31c5e3, (q31_t)0xea9394a9, (q31_t)0x7e30b885, (q31_t)0xea8d62e1,
	(q31_t)0x7e2faad9, (q31_t)0xea873127,
	(q31_t)0x7e2e9cdf, (q31_t)0xea80ff7a, (q31_t)0x7e2d8e97, (q31_t)0xea7acdda, (q31_t)0x7e2c8002, (q31_t)0xea749c47,
	(q31_t)0x7e2b711f, (q31_t)0xea6e6ac2,
	(q31_t)0x7e2a61ed, (q31_t)0xea683949, (q31_t)0x7e29526e, (q31_t)0xea6207df, (q31_t)0x7e2842a2, (q31_t)0xea5bd681,
	(q31_t)0x7e273287, (q31_t)0xea55a531,
	(q31_t)0x7e26221f, (q31_t)0xea4f73ee, (q31_t)0x7e251168, (q31_t)0xea4942b9, (q31_t)0x7e240064, (q31_t)0xea431191,
	(q31_t)0x7e22ef12, (q31_t)0xea3ce077,
	(q31_t)0x7e21dd73, (q31_t)0xea36af69, (q31_t)0x7e20cb85, (q31_t)0xea307e6a, (q31_t)0x7e1fb94a, (q31_t)0xea2a4d78,
	(q31_t)0x7e1ea6c1, (q31_t)0xea241c93,
	(q31_t)0x7e1d93ea, (q31_t)0xea1debbb, (q31_t)0x7e1c80c5, (q31_t)0xea17baf2, (q31_t)0x7e1b6d53, (q31_t)0xea118a35,
	(q31_t)0x7e1a5992, (q31_t)0xea0b5987,
	(q31_t)0x7e194584, (q31_t)0xea0528e5, (q31_t)0x7e183128, (q31_t)0xe9fef852, (q31_t)0x7e171c7f, (q31_t)0xe9f8c7cc,
	(q31_t)0x7e160787, (q31_t)0xe9f29753,
	(q31_t)0x7e14f242, (q31_t)0xe9ec66e8, (q31_t)0x7e13dcaf, (q31_t)0xe9e6368b, (q31_t)0x7e12c6ce, (q31_t)0xe9e0063c,
	(q31_t)0x7e11b0a0, (q31_t)0xe9d9d5fa,
	(q31_t)0x7e109a24, (q31_t)0xe9d3a5c5, (q31_t)0x7e0f835a, (q31_t)0xe9cd759f, (q31_t)0x7e0e6c42, (q31_t)0xe9c74586,
	(q31_t)0x7e0d54dc, (q31_t)0xe9c1157a,
	(q31_t)0x7e0c3d29, (q31_t)0xe9bae57d, (q31_t)0x7e0b2528, (q31_t)0xe9b4b58d, (q31_t)0x7e0a0cd9, (q31_t)0xe9ae85ab,
	(q31_t)0x7e08f43d, (q31_t)0xe9a855d7,
	(q31_t)0x7e07db52, (q31_t)0xe9a22610, (q31_t)0x7e06c21a, (q31_t)0xe99bf658, (q31_t)0x7e05a894, (q31_t)0xe995c6ad,
	(q31_t)0x7e048ec1, (q31_t)0xe98f9710,
	(q31_t)0x7e0374a0, (q31_t)0xe9896781, (q31_t)0x7e025a31, (q31_t)0xe98337ff, (q31_t)0x7e013f74, (q31_t)0xe97d088c,
	(q31_t)0x7e00246a, (q31_t)0xe976d926,
	(q31_t)0x7dff0911, (q31_t)0xe970a9ce, (q31_t)0x7dfded6c, (q31_t)0xe96a7a85, (q31_t)0x7dfcd178, (q31_t)0xe9644b49,
	(q31_t)0x7dfbb537, (q31_t)0xe95e1c1b,
	(q31_t)0x7dfa98a8, (q31_t)0xe957ecfb, (q31_t)0x7df97bcb, (q31_t)0xe951bde9, (q31_t)0x7df85ea0, (q31_t)0xe94b8ee5,
	(q31_t)0x7df74128, (q31_t)0xe9455fef,
	(q31_t)0x7df62362, (q31_t)0xe93f3107, (q31_t)0x7df5054f, (q31_t)0xe939022d, (q31_t)0x7df3e6ee, (q31_t)0xe932d361,
	(q31_t)0x7df2c83f, (q31_t)0xe92ca4a4,
	(q31_t)0x7df1a942, (q31_t)0xe92675f4, (q31_t)0x7df089f8, (q31_t)0xe9204752, (q31_t)0x7def6a60, (q31_t)0xe91a18bf,
	(q31_t)0x7dee4a7a, (q31_t)0xe913ea39,
	(q31_t)0x7ded2a47, (q31_t)0xe90dbbc2, (q31_t)0x7dec09c6, (q31_t)0xe9078d59, (q31_t)0x7deae8f7, (q31_t)0xe9015efe,
	(q31_t)0x7de9c7da, (q31_t)0xe8fb30b1,
	(q31_t)0x7de8a670, (q31_t)0xe8f50273, (q31_t)0x7de784b9, (q31_t)0xe8eed443, (q31_t)0x7de662b3, (q31_t)0xe8e8a621,
	(q31_t)0x7de54060, (q31_t)0xe8e2780d,
	(q31_t)0x7de41dc0, (q31_t)0xe8dc4a07, (q31_t)0x7de2fad1, (q31_t)0xe8d61c10, (q31_t)0x7de1d795, (q31_t)0xe8cfee27,
	(q31_t)0x7de0b40b, (q31_t)0xe8c9c04c,
	(q31_t)0x7ddf9034, (q31_t)0xe8c39280, (q31_t)0x7dde6c0f, (q31_t)0xe8bd64c2, (q31_t)0x7ddd479d, (q31_t)0xe8b73712,
	(q31_t)0x7ddc22dc, (q31_t)0xe8b10971,
	(q31_t)0x7ddafdce, (q31_t)0xe8aadbde, (q31_t)0x7dd9d873, (q31_t)0xe8a4ae59, (q31_t)0x7dd8b2ca, (q31_t)0xe89e80e3,
	(q31_t)0x7dd78cd3, (q31_t)0xe898537b,
	(q31_t)0x7dd6668f, (q31_t)0xe8922622, (q31_t)0x7dd53ffc, (q31_t)0xe88bf8d7, (q31_t)0x7dd4191d, (q31_t)0xe885cb9a,
	(q31_t)0x7dd2f1f0, (q31_t)0xe87f9e6c,
	(q31_t)0x7dd1ca75, (q31_t)0xe879714d, (q31_t)0x7dd0a2ac, (q31_t)0xe873443c, (q31_t)0x7dcf7a96, (q31_t)0xe86d173a,
	(q31_t)0x7dce5232, (q31_t)0xe866ea46,
	(q31_t)0x7dcd2981, (q31_t)0xe860bd61, (q31_t)0x7dcc0082, (q31_t)0xe85a908a, (q31_t)0x7dcad736, (q31_t)0xe85463c2,
	(q31_t)0x7dc9ad9c, (q31_t)0xe84e3708,
	(q31_t)0x7dc883b4, (q31_t)0xe8480a5d, (q31_t)0x7dc7597f, (q31_t)0xe841ddc1, (q31_t)0x7dc62efc, (q31_t)0xe83bb133,
	(q31_t)0x7dc5042b, (q31_t)0xe83584b4,
	(q31_t)0x7dc3d90d, (q31_t)0xe82f5844, (q31_t)0x7dc2ada2, (q31_t)0xe8292be3, (q31_t)0x7dc181e8, (q31_t)0xe822ff90,
	(q31_t)0x7dc055e2, (q31_t)0xe81cd34b,
	(q31_t)0x7dbf298d, (q31_t)0xe816a716, (q31_t)0x7dbdfceb, (q31_t)0xe8107aef, (q31_t)0x7dbccffc, (q31_t)0xe80a4ed7,
	(q31_t)0x7dbba2bf, (q31_t)0xe80422ce,
	(q31_t)0x7dba7534, (q31_t)0xe7fdf6d4, (q31_t)0x7db9475c, (q31_t)0xe7f7cae8, (q31_t)0x7db81936, (q31_t)0xe7f19f0c,
	(q31_t)0x7db6eac3, (q31_t)0xe7eb733e,
	(q31_t)0x7db5bc02, (q31_t)0xe7e5477f, (q31_t)0x7db48cf4, (q31_t)0xe7df1bcf, (q31_t)0x7db35d98, (q31_t)0xe7d8f02d,
	(q31_t)0x7db22def, (q31_t)0xe7d2c49b,
	(q31_t)0x7db0fdf8, (q31_t)0xe7cc9917, (q31_t)0x7dafcdb3, (q31_t)0xe7c66da3, (q31_t)0x7dae9d21, (q31_t)0xe7c0423d,
	(q31_t)0x7dad6c42, (q31_t)0xe7ba16e7,
	(q31_t)0x7dac3b15, (q31_t)0xe7b3eb9f, (q31_t)0x7dab099a, (q31_t)0xe7adc066, (q31_t)0x7da9d7d2, (q31_t)0xe7a7953d,
	(q31_t)0x7da8a5bc, (q31_t)0xe7a16a22,
	(q31_t)0x7da77359, (q31_t)0xe79b3f16, (q31_t)0x7da640a9, (q31_t)0xe795141a, (q31_t)0x7da50dab, (q31_t)0xe78ee92c,
	(q31_t)0x7da3da5f, (q31_t)0xe788be4e,
	(q31_t)0x7da2a6c6, (q31_t)0xe782937e, (q31_t)0x7da172df, (q31_t)0xe77c68be, (q31_t)0x7da03eab, (q31_t)0xe7763e0d,
	(q31_t)0x7d9f0a29, (q31_t)0xe770136b,
	(q31_t)0x7d9dd55a, (q31_t)0xe769e8d8, (q31_t)0x7d9ca03e, (q31_t)0xe763be55, (q31_t)0x7d9b6ad3, (q31_t)0xe75d93e0,
	(q31_t)0x7d9a351c, (q31_t)0xe757697b,
	(q31_t)0x7d98ff17, (q31_t)0xe7513f25, (q31_t)0x7d97c8c4, (q31_t)0xe74b14de, (q31_t)0x7d969224, (q31_t)0xe744eaa6,
	(q31_t)0x7d955b37, (q31_t)0xe73ec07e,
	(q31_t)0x7d9423fc, (q31_t)0xe7389665, (q31_t)0x7d92ec73, (q31_t)0xe7326c5b, (q31_t)0x7d91b49e, (q31_t)0xe72c4260,
	(q31_t)0x7d907c7a, (q31_t)0xe7261875,
	(q31_t)0x7d8f4409, (q31_t)0xe71fee99, (q31_t)0x7d8e0b4b, (q31_t)0xe719c4cd, (q31_t)0x7d8cd240, (q31_t)0xe7139b10,
	(q31_t)0x7d8b98e6, (q31_t)0xe70d7162,
	(q31_t)0x7d8a5f40, (q31_t)0xe70747c4, (q31_t)0x7d89254c, (q31_t)0xe7011e35, (q31_t)0x7d87eb0a, (q31_t)0xe6faf4b5,
	(q31_t)0x7d86b07c, (q31_t)0xe6f4cb45,
	(q31_t)0x7d85759f, (q31_t)0xe6eea1e4, (q31_t)0x7d843a76, (q31_t)0xe6e87893, (q31_t)0x7d82fefe, (q31_t)0xe6e24f51,
	(q31_t)0x7d81c33a, (q31_t)0xe6dc261f,
	(q31_t)0x7d808728, (q31_t)0xe6d5fcfc, (q31_t)0x7d7f4ac8, (q31_t)0xe6cfd3e9, (q31_t)0x7d7e0e1c, (q31_t)0xe6c9aae5,
	(q31_t)0x7d7cd121, (q31_t)0xe6c381f1,
	(q31_t)0x7d7b93da, (q31_t)0xe6bd590d, (q31_t)0x7d7a5645, (q31_t)0xe6b73038, (q31_t)0x7d791862, (q31_t)0xe6b10772,
	(q31_t)0x7d77da32, (q31_t)0xe6aadebc,
	(q31_t)0x7d769bb5, (q31_t)0xe6a4b616, (q31_t)0x7d755cea, (q31_t)0xe69e8d80, (q31_t)0x7d741dd2, (q31_t)0xe69864f9,
	(q31_t)0x7d72de6d, (q31_t)0xe6923c82,
	(q31_t)0x7d719eba, (q31_t)0xe68c141a, (q31_t)0x7d705eba, (q31_t)0xe685ebc2, (q31_t)0x7d6f1e6c, (q31_t)0xe67fc37a,
	(q31_t)0x7d6dddd2, (q31_t)0xe6799b42,
	(q31_t)0x7d6c9ce9, (q31_t)0xe6737319, (q31_t)0x7d6b5bb4, (q31_t)0xe66d4b01, (q31_t)0x7d6a1a31, (q31_t)0xe66722f7,
	(q31_t)0x7d68d860, (q31_t)0xe660fafe,
	(q31_t)0x7d679642, (q31_t)0xe65ad315, (q31_t)0x7d6653d7, (q31_t)0xe654ab3b, (q31_t)0x7d65111f, (q31_t)0xe64e8371,
	(q31_t)0x7d63ce19, (q31_t)0xe6485bb7,
	(q31_t)0x7d628ac6, (q31_t)0xe642340d, (q31_t)0x7d614725, (q31_t)0xe63c0c73, (q31_t)0x7d600338, (q31_t)0xe635e4e9,
	(q31_t)0x7d5ebefc, (q31_t)0xe62fbd6e,
	(q31_t)0x7d5d7a74, (q31_t)0xe6299604, (q31_t)0x7d5c359e, (q31_t)0xe6236ea9, (q31_t)0x7d5af07b, (q31_t)0xe61d475e,
	(q31_t)0x7d59ab0a, (q31_t)0xe6172024,
	(q31_t)0x7d58654d, (q31_t)0xe610f8f9, (q31_t)0x7d571f41, (q31_t)0xe60ad1de, (q31_t)0x7d55d8e9, (q31_t)0xe604aad4,
	(q31_t)0x7d549243, (q31_t)0xe5fe83d9,
	(q31_t)0x7d534b50, (q31_t)0xe5f85cef, (q31_t)0x7d520410, (q31_t)0xe5f23614, (q31_t)0x7d50bc82, (q31_t)0xe5ec0f4a,
	(q31_t)0x7d4f74a7, (q31_t)0xe5e5e88f,
	(q31_t)0x7d4e2c7f, (q31_t)0xe5dfc1e5, (q31_t)0x7d4ce409, (q31_t)0xe5d99b4b, (q31_t)0x7d4b9b46, (q31_t)0xe5d374c1,
	(q31_t)0x7d4a5236, (q31_t)0xe5cd4e47,
	(q31_t)0x7d4908d9, (q31_t)0xe5c727dd, (q31_t)0x7d47bf2e, (q31_t)0xe5c10184, (q31_t)0x7d467536, (q31_t)0xe5badb3a,
	(q31_t)0x7d452af1, (q31_t)0xe5b4b501,
	(q31_t)0x7d43e05e, (q31_t)0xe5ae8ed8, (q31_t)0x7d42957e, (q31_t)0xe5a868bf, (q31_t)0x7d414a51, (q31_t)0xe5a242b7,
	(q31_t)0x7d3ffed7, (q31_t)0xe59c1cbf,
	(q31_t)0x7d3eb30f, (q31_t)0xe595f6d7, (q31_t)0x7d3d66fa, (q31_t)0xe58fd0ff, (q31_t)0x7d3c1a98, (q31_t)0xe589ab38,
	(q31_t)0x7d3acde9, (q31_t)0xe5838581,
	(q31_t)0x7d3980ec, (q31_t)0xe57d5fda, (q31_t)0x7d3833a2, (q31_t)0xe5773a44, (q31_t)0x7d36e60b, (q31_t)0xe57114be,
	(q31_t)0x7d359827, (q31_t)0xe56aef49,
	(q31_t)0x7d3449f5, (q31_t)0xe564c9e3, (q31_t)0x7d32fb76, (q31_t)0xe55ea48f, (q31_t)0x7d31acaa, (q31_t)0xe5587f4a,
	(q31_t)0x7d305d91, (q31_t)0xe5525a17,
	(q31_t)0x7d2f0e2b, (q31_t)0xe54c34f3, (q31_t)0x7d2dbe77, (q31_t)0xe5460fe0, (q31_t)0x7d2c6e76, (q31_t)0xe53feade,
	(q31_t)0x7d2b1e28, (q31_t)0xe539c5ec,
	(q31_t)0x7d29cd8c, (q31_t)0xe533a10a, (q31_t)0x7d287ca4, (q31_t)0xe52d7c39, (q31_t)0x7d272b6e, (q31_t)0xe5275779,
	(q31_t)0x7d25d9eb, (q31_t)0xe52132c9,
	(q31_t)0x7d24881b, (q31_t)0xe51b0e2a, (q31_t)0x7d2335fe, (q31_t)0xe514e99b, (q31_t)0x7d21e393, (q31_t)0xe50ec51d,
	(q31_t)0x7d2090db, (q31_t)0xe508a0b0,
	(q31_t)0x7d1f3dd6, (q31_t)0xe5027c53, (q31_t)0x7d1dea84, (q31_t)0xe4fc5807, (q31_t)0x7d1c96e5, (q31_t)0xe4f633cc,
	(q31_t)0x7d1b42f9, (q31_t)0xe4f00fa1,
	(q31_t)0x7d19eebf, (q31_t)0xe4e9eb87, (q31_t)0x7d189a38, (q31_t)0xe4e3c77d, (q31_t)0x7d174564, (q31_t)0xe4dda385,
	(q31_t)0x7d15f043, (q31_t)0xe4d77f9d,
	(q31_t)0x7d149ad5, (q31_t)0xe4d15bc6, (q31_t)0x7d134519, (q31_t)0xe4cb37ff, (q31_t)0x7d11ef11, (q31_t)0xe4c5144a,
	(q31_t)0x7d1098bb, (q31_t)0xe4bef0a5,
	(q31_t)0x7d0f4218, (q31_t)0xe4b8cd11, (q31_t)0x7d0deb28, (q31_t)0xe4b2a98e, (q31_t)0x7d0c93eb, (q31_t)0xe4ac861b,
	(q31_t)0x7d0b3c60, (q31_t)0xe4a662ba,
	(q31_t)0x7d09e489, (q31_t)0xe4a03f69, (q31_t)0x7d088c64, (q31_t)0xe49a1c29, (q31_t)0x7d0733f3, (q31_t)0xe493f8fb,
	(q31_t)0x7d05db34, (q31_t)0xe48dd5dd,
	(q31_t)0x7d048228, (q31_t)0xe487b2d0, (q31_t)0x7d0328cf, (q31_t)0xe4818fd4, (q31_t)0x7d01cf29, (q31_t)0xe47b6ce9,
	(q31_t)0x7d007535, (q31_t)0xe4754a0e,
	(q31_t)0x7cff1af5, (q31_t)0xe46f2745, (q31_t)0x7cfdc068, (q31_t)0xe469048d, (q31_t)0x7cfc658d, (q31_t)0xe462e1e6,
	(q31_t)0x7cfb0a65, (q31_t)0xe45cbf50,
	(q31_t)0x7cf9aef0, (q31_t)0xe4569ccb, (q31_t)0x7cf8532f, (q31_t)0xe4507a57, (q31_t)0x7cf6f720, (q31_t)0xe44a57f4,
	(q31_t)0x7cf59ac4, (q31_t)0xe44435a2,
	(q31_t)0x7cf43e1a, (q31_t)0xe43e1362, (q31_t)0x7cf2e124, (q31_t)0xe437f132, (q31_t)0x7cf183e1, (q31_t)0xe431cf14,
	(q31_t)0x7cf02651, (q31_t)0xe42bad07,
	(q31_t)0x7ceec873, (q31_t)0xe4258b0a, (q31_t)0x7ced6a49, (q31_t)0xe41f6920, (q31_t)0x7cec0bd1, (q31_t)0xe4194746,
	(q31_t)0x7ceaad0c, (q31_t)0xe413257d,
	(q31_t)0x7ce94dfb, (q31_t)0xe40d03c6, (q31_t)0x7ce7ee9c, (q31_t)0xe406e220, (q31_t)0x7ce68ef0, (q31_t)0xe400c08b,
	(q31_t)0x7ce52ef7, (q31_t)0xe3fa9f08,
	(q31_t)0x7ce3ceb2, (q31_t)0xe3f47d96, (q31_t)0x7ce26e1f, (q31_t)0xe3ee5c35, (q31_t)0x7ce10d3f, (q31_t)0xe3e83ae5,
	(q31_t)0x7cdfac12, (q31_t)0xe3e219a7,
	(q31_t)0x7cde4a98, (q31_t)0xe3dbf87a, (q31_t)0x7cdce8d1, (q31_t)0xe3d5d75e, (q31_t)0x7cdb86bd, (q31_t)0xe3cfb654,
	(q31_t)0x7cda245c, (q31_t)0xe3c9955b,
	(q31_t)0x7cd8c1ae, (q31_t)0xe3c37474, (q31_t)0x7cd75eb3, (q31_t)0xe3bd539e, (q31_t)0x7cd5fb6a, (q31_t)0xe3b732d9,
	(q31_t)0x7cd497d5, (q31_t)0xe3b11226,
	(q31_t)0x7cd333f3, (q31_t)0xe3aaf184, (q31_t)0x7cd1cfc4, (q31_t)0xe3a4d0f4, (q31_t)0x7cd06b48, (q31_t)0xe39eb075,
	(q31_t)0x7ccf067f, (q31_t)0xe3989008,
	(q31_t)0x7ccda169, (q31_t)0xe3926fad, (q31_t)0x7ccc3c06, (q31_t)0xe38c4f63, (q31_t)0x7ccad656, (q31_t)0xe3862f2a,
	(q31_t)0x7cc97059, (q31_t)0xe3800f03,
	(q31_t)0x7cc80a0f, (q31_t)0xe379eeed, (q31_t)0x7cc6a378, (q31_t)0xe373ceea, (q31_t)0x7cc53c94, (q31_t)0xe36daef7,
	(q31_t)0x7cc3d563, (q31_t)0xe3678f17,
	(q31_t)0x7cc26de5, (q31_t)0xe3616f48, (q31_t)0x7cc1061a, (q31_t)0xe35b4f8b, (q31_t)0x7cbf9e03, (q31_t)0xe3552fdf,
	(q31_t)0x7cbe359e, (q31_t)0xe34f1045,
	(q31_t)0x7cbcccec, (q31_t)0xe348f0bd, (q31_t)0x7cbb63ee, (q31_t)0xe342d146, (q31_t)0x7cb9faa2, (q31_t)0xe33cb1e1,
	(q31_t)0x7cb8910a, (q31_t)0xe336928e,
	(q31_t)0x7cb72724, (q31_t)0xe330734d, (q31_t)0x7cb5bcf2, (q31_t)0xe32a541d, (q31_t)0x7cb45272, (q31_t)0xe3243500,
	(q31_t)0x7cb2e7a6, (q31_t)0xe31e15f4,
	(q31_t)0x7cb17c8d, (q31_t)0xe317f6fa, (q31_t)0x7cb01127, (q31_t)0xe311d811, (q31_t)0x7caea574, (q31_t)0xe30bb93b,
	(q31_t)0x7cad3974, (q31_t)0xe3059a76,
	(q31_t)0x7cabcd28, (q31_t)0xe2ff7bc3, (q31_t)0x7caa608e, (q31_t)0xe2f95d23, (q31_t)0x7ca8f3a7, (q31_t)0xe2f33e94,
	(q31_t)0x7ca78674, (q31_t)0xe2ed2017,
	(q31_t)0x7ca618f3, (q31_t)0xe2e701ac, (q31_t)0x7ca4ab26, (q31_t)0xe2e0e352, (q31_t)0x7ca33d0c, (q31_t)0xe2dac50b,
	(q31_t)0x7ca1cea5, (q31_t)0xe2d4a6d6,
	(q31_t)0x7ca05ff1, (q31_t)0xe2ce88b3, (q31_t)0x7c9ef0f0, (q31_t)0xe2c86aa2, (q31_t)0x7c9d81a3, (q31_t)0xe2c24ca2,
	(q31_t)0x7c9c1208, (q31_t)0xe2bc2eb5,
	(q31_t)0x7c9aa221, (q31_t)0xe2b610da, (q31_t)0x7c9931ec, (q31_t)0xe2aff311, (q31_t)0x7c97c16b, (q31_t)0xe2a9d55a,
	(q31_t)0x7c96509d, (q31_t)0xe2a3b7b5,
	(q31_t)0x7c94df83, (q31_t)0xe29d9a23, (q31_t)0x7c936e1b, (q31_t)0xe2977ca2, (q31_t)0x7c91fc66, (q31_t)0xe2915f34,
	(q31_t)0x7c908a65, (q31_t)0xe28b41d7,
	(q31_t)0x7c8f1817, (q31_t)0xe285248d, (q31_t)0x7c8da57c, (q31_t)0xe27f0755, (q31_t)0x7c8c3294, (q31_t)0xe278ea30,
	(q31_t)0x7c8abf5f, (q31_t)0xe272cd1c,
	(q31_t)0x7c894bde, (q31_t)0xe26cb01b, (q31_t)0x7c87d810, (q31_t)0xe266932c, (q31_t)0x7c8663f4, (q31_t)0xe260764f,
	(q31_t)0x7c84ef8c, (q31_t)0xe25a5984,
	(q31_t)0x7c837ad8, (q31_t)0xe2543ccc, (q31_t)0x7c8205d6, (q31_t)0xe24e2026, (q31_t)0x7c809088, (q31_t)0xe2480393,
	(q31_t)0x7c7f1aed, (q31_t)0xe241e711,
	(q31_t)0x7c7da505, (q31_t)0xe23bcaa2, (q31_t)0x7c7c2ed0, (q31_t)0xe235ae46, (q31_t)0x7c7ab84e, (q31_t)0xe22f91fc,
	(q31_t)0x7c794180, (q31_t)0xe22975c4,
	(q31_t)0x7c77ca65, (q31_t)0xe223599e, (q31_t)0x7c7652fd, (q31_t)0xe21d3d8b, (q31_t)0x7c74db48, (q31_t)0xe217218b,
	(q31_t)0x7c736347, (q31_t)0xe211059d,
	(q31_t)0x7c71eaf9, (q31_t)0xe20ae9c1, (q31_t)0x7c70725e, (q31_t)0xe204cdf8, (q31_t)0x7c6ef976, (q31_t)0xe1feb241,
	(q31_t)0x7c6d8041, (q31_t)0xe1f8969d,
	(q31_t)0x7c6c06c0, (q31_t)0xe1f27b0b, (q31_t)0x7c6a8cf2, (q31_t)0xe1ec5f8c, (q31_t)0x7c6912d7, (q31_t)0xe1e64420,
	(q31_t)0x7c679870, (q31_t)0xe1e028c6,
	(q31_t)0x7c661dbc, (q31_t)0xe1da0d7e, (q31_t)0x7c64a2bb, (q31_t)0xe1d3f24a, (q31_t)0x7c63276d, (q31_t)0xe1cdd727,
	(q31_t)0x7c61abd3, (q31_t)0xe1c7bc18,
	(q31_t)0x7c602fec, (q31_t)0xe1c1a11b, (q31_t)0x7c5eb3b8, (q31_t)0xe1bb8631, (q31_t)0x7c5d3737, (q31_t)0xe1b56b59,
	(q31_t)0x7c5bba6a, (q31_t)0xe1af5094,
	(q31_t)0x7c5a3d50, (q31_t)0xe1a935e2, (q31_t)0x7c58bfe9, (q31_t)0xe1a31b42, (q31_t)0x7c574236, (q31_t)0xe19d00b6,
	(q31_t)0x7c55c436, (q31_t)0xe196e63c,
	(q31_t)0x7c5445e9, (q31_t)0xe190cbd4, (q31_t)0x7c52c74f, (q31_t)0xe18ab180, (q31_t)0x7c514869, (q31_t)0xe184973e,
	(q31_t)0x7c4fc936, (q31_t)0xe17e7d0f,
	(q31_t)0x7c4e49b7, (q31_t)0xe17862f3, (q31_t)0x7c4cc9ea, (q31_t)0xe17248ea, (q31_t)0x7c4b49d2, (q31_t)0xe16c2ef4,
	(q31_t)0x7c49c96c, (q31_t)0xe1661510,
	(q31_t)0x7c4848ba, (q31_t)0xe15ffb3f, (q31_t)0x7c46c7bb, (q31_t)0xe159e182, (q31_t)0x7c45466f, (q31_t)0xe153c7d7,
	(q31_t)0x7c43c4d7, (q31_t)0xe14dae3f,
	(q31_t)0x7c4242f2, (q31_t)0xe14794ba, (q31_t)0x7c40c0c1, (q31_t)0xe1417b48, (q31_t)0x7c3f3e42, (q31_t)0xe13b61e9,
	(q31_t)0x7c3dbb78, (q31_t)0xe135489d,
	(q31_t)0x7c3c3860, (q31_t)0xe12f2f63, (q31_t)0x7c3ab4fc, (q31_t)0xe129163d, (q31_t)0x7c39314b, (q31_t)0xe122fd2a,
	(q31_t)0x7c37ad4e, (q31_t)0xe11ce42a,
	(q31_t)0x7c362904, (q31_t)0xe116cb3d, (q31_t)0x7c34a46d, (q31_t)0xe110b263, (q31_t)0x7c331f8a, (q31_t)0xe10a999c,
	(q31_t)0x7c319a5a, (q31_t)0xe10480e9,
	(q31_t)0x7c3014de, (q31_t)0xe0fe6848, (q31_t)0x7c2e8f15, (q31_t)0xe0f84fbb, (q31_t)0x7c2d08ff, (q31_t)0xe0f23740,
	(q31_t)0x7c2b829d, (q31_t)0xe0ec1ed9,
	(q31_t)0x7c29fbee, (q31_t)0xe0e60685, (q31_t)0x7c2874f3, (q31_t)0xe0dfee44, (q31_t)0x7c26edab, (q31_t)0xe0d9d616,
	(q31_t)0x7c256616, (q31_t)0xe0d3bdfc,
	(q31_t)0x7c23de35, (q31_t)0xe0cda5f5, (q31_t)0x7c225607, (q31_t)0xe0c78e01, (q31_t)0x7c20cd8d, (q31_t)0xe0c17620,
	(q31_t)0x7c1f44c6, (q31_t)0xe0bb5e53,
	(q31_t)0x7c1dbbb3, (q31_t)0xe0b54698, (q31_t)0x7c1c3253, (q31_t)0xe0af2ef2, (q31_t)0x7c1aa8a6, (q31_t)0xe0a9175e,
	(q31_t)0x7c191ead, (q31_t)0xe0a2ffde,
	(q31_t)0x7c179467, (q31_t)0xe09ce871, (q31_t)0x7c1609d5, (q31_t)0xe096d117, (q31_t)0x7c147ef6, (q31_t)0xe090b9d1,
	(q31_t)0x7c12f3cb, (q31_t)0xe08aa29f,
	(q31_t)0x7c116853, (q31_t)0xe0848b7f, (q31_t)0x7c0fdc8f, (q31_t)0xe07e7473, (q31_t)0x7c0e507e, (q31_t)0xe0785d7b,
	(q31_t)0x7c0cc421, (q31_t)0xe0724696,
	(q31_t)0x7c0b3777, (q31_t)0xe06c2fc4, (q31_t)0x7c09aa80, (q31_t)0xe0661906, (q31_t)0x7c081d3d, (q31_t)0xe060025c,
	(q31_t)0x7c068fae, (q31_t)0xe059ebc5,
	(q31_t)0x7c0501d2, (q31_t)0xe053d541, (q31_t)0x7c0373a9, (q31_t)0xe04dbed1, (q31_t)0x7c01e534, (q31_t)0xe047a875,
	(q31_t)0x7c005673, (q31_t)0xe041922c,
	(q31_t)0x7bfec765, (q31_t)0xe03b7bf6, (q31_t)0x7bfd380a, (q31_t)0xe03565d5, (q31_t)0x7bfba863, (q31_t)0xe02f4fc6,
	(q31_t)0x7bfa1870, (q31_t)0xe02939cc,
	(q31_t)0x7bf88830, (q31_t)0xe02323e5, (q31_t)0x7bf6f7a4, (q31_t)0xe01d0e12, (q31_t)0x7bf566cb, (q31_t)0xe016f852,
	(q31_t)0x7bf3d5a6, (q31_t)0xe010e2a7,
	(q31_t)0x7bf24434, (q31_t)0xe00acd0e, (q31_t)0x7bf0b276, (q31_t)0xe004b78a, (q31_t)0x7bef206b, (q31_t)0xdffea219,
	(q31_t)0x7bed8e14, (q31_t)0xdff88cbc,
	(q31_t)0x7bebfb70, (q31_t)0xdff27773, (q31_t)0x7bea6880, (q31_t)0xdfec623e, (q31_t)0x7be8d544, (q31_t)0xdfe64d1c,
	(q31_t)0x7be741bb, (q31_t)0xdfe0380e,
	(q31_t)0x7be5ade6, (q31_t)0xdfda2314, (q31_t)0x7be419c4, (q31_t)0xdfd40e2e, (q31_t)0x7be28556, (q31_t)0xdfcdf95c,
	(q31_t)0x7be0f09b, (q31_t)0xdfc7e49d,
	(q31_t)0x7bdf5b94, (q31_t)0xdfc1cff3, (q31_t)0x7bddc641, (q31_t)0xdfbbbb5c, (q31_t)0x7bdc30a1, (q31_t)0xdfb5a6d9,
	(q31_t)0x7bda9ab5, (q31_t)0xdfaf926a,
	(q31_t)0x7bd9047c, (q31_t)0xdfa97e0f, (q31_t)0x7bd76df7, (q31_t)0xdfa369c8, (q31_t)0x7bd5d726, (q31_t)0xdf9d5595,
	(q31_t)0x7bd44008, (q31_t)0xdf974176,
	(q31_t)0x7bd2a89e, (q31_t)0xdf912d6b, (q31_t)0x7bd110e8, (q31_t)0xdf8b1974, (q31_t)0x7bcf78e5, (q31_t)0xdf850591,
	(q31_t)0x7bcde095, (q31_t)0xdf7ef1c2,
	(q31_t)0x7bcc47fa, (q31_t)0xdf78de07, (q31_t)0x7bcaaf12, (q31_t)0xdf72ca60, (q31_t)0x7bc915dd, (q31_t)0xdf6cb6cd,
	(q31_t)0x7bc77c5d, (q31_t)0xdf66a34e,
	(q31_t)0x7bc5e290, (q31_t)0xdf608fe4, (q31_t)0x7bc44876, (q31_t)0xdf5a7c8d, (q31_t)0x7bc2ae10, (q31_t)0xdf54694b,
	(q31_t)0x7bc1135e, (q31_t)0xdf4e561c,
	(q31_t)0x7bbf7860, (q31_t)0xdf484302, (q31_t)0x7bbddd15, (q31_t)0xdf422ffd, (q31_t)0x7bbc417e, (q31_t)0xdf3c1d0b,
	(q31_t)0x7bbaa59a, (q31_t)0xdf360a2d,
	(q31_t)0x7bb9096b, (q31_t)0xdf2ff764, (q31_t)0x7bb76cef, (q31_t)0xdf29e4af, (q31_t)0x7bb5d026, (q31_t)0xdf23d20e,
	(q31_t)0x7bb43311, (q31_t)0xdf1dbf82,
	(q31_t)0x7bb295b0, (q31_t)0xdf17ad0a, (q31_t)0x7bb0f803, (q31_t)0xdf119aa6, (q31_t)0x7baf5a09, (q31_t)0xdf0b8856,
	(q31_t)0x7badbbc3, (q31_t)0xdf05761b,
	(q31_t)0x7bac1d31, (q31_t)0xdeff63f4, (q31_t)0x7baa7e53, (q31_t)0xdef951e2, (q31_t)0x7ba8df28, (q31_t)0xdef33fe3,
	(q31_t)0x7ba73fb1, (q31_t)0xdeed2dfa,
	(q31_t)0x7ba59fee, (q31_t)0xdee71c24, (q31_t)0x7ba3ffde, (q31_t)0xdee10a63, (q31_t)0x7ba25f82, (q31_t)0xdedaf8b7,
	(q31_t)0x7ba0beda, (q31_t)0xded4e71f,
	(q31_t)0x7b9f1de6, (q31_t)0xdeced59b, (q31_t)0x7b9d7ca5, (q31_t)0xdec8c42c, (q31_t)0x7b9bdb18, (q31_t)0xdec2b2d1,
	(q31_t)0x7b9a393f, (q31_t)0xdebca18b,
	(q31_t)0x7b989719, (q31_t)0xdeb69059, (q31_t)0x7b96f4a8, (q31_t)0xdeb07f3c, (q31_t)0x7b9551ea, (q31_t)0xdeaa6e34,
	(q31_t)0x7b93aee0, (q31_t)0xdea45d40,
	(q31_t)0x7b920b89, (q31_t)0xde9e4c60, (q31_t)0x7b9067e7, (q31_t)0xde983b95, (q31_t)0x7b8ec3f8, (q31_t)0xde922adf,
	(q31_t)0x7b8d1fbd, (q31_t)0xde8c1a3e,
	(q31_t)0x7b8b7b36, (q31_t)0xde8609b1, (q31_t)0x7b89d662, (q31_t)0xde7ff938, (q31_t)0x7b883143, (q31_t)0xde79e8d5,
	(q31_t)0x7b868bd7, (q31_t)0xde73d886,
	(q31_t)0x7b84e61f, (q31_t)0xde6dc84b, (q31_t)0x7b83401b, (q31_t)0xde67b826, (q31_t)0x7b8199ca, (q31_t)0xde61a815,
	(q31_t)0x7b7ff32e, (q31_t)0xde5b9819,
	(q31_t)0x7b7e4c45, (q31_t)0xde558831, (q31_t)0x7b7ca510, (q31_t)0xde4f785f, (q31_t)0x7b7afd8f, (q31_t)0xde4968a1,
	(q31_t)0x7b7955c2, (q31_t)0xde4358f8,
	(q31_t)0x7b77ada8, (q31_t)0xde3d4964, (q31_t)0x7b760542, (q31_t)0xde3739e4, (q31_t)0x7b745c91, (q31_t)0xde312a7a,
	(q31_t)0x7b72b393, (q31_t)0xde2b1b24,
	(q31_t)0x7b710a49, (q31_t)0xde250be3, (q31_t)0x7b6f60b2, (q31_t)0xde1efcb7, (q31_t)0x7b6db6d0, (q31_t)0xde18eda0,
	(q31_t)0x7b6c0ca2, (q31_t)0xde12de9e,
	(q31_t)0x7b6a6227, (q31_t)0xde0ccfb1, (q31_t)0x7b68b760, (q31_t)0xde06c0d9, (q31_t)0x7b670c4d, (q31_t)0xde00b216,
	(q31_t)0x7b6560ee, (q31_t)0xddfaa367,
	(q31_t)0x7b63b543, (q31_t)0xddf494ce, (q31_t)0x7b62094c, (q31_t)0xddee8649, (q31_t)0x7b605d09, (q31_t)0xdde877da,
	(q31_t)0x7b5eb079, (q31_t)0xdde26980,
	(q31_t)0x7b5d039e, (q31_t)0xdddc5b3b, (q31_t)0x7b5b5676, (q31_t)0xddd64d0a, (q31_t)0x7b59a902, (q31_t)0xddd03eef,
	(q31_t)0x7b57fb42, (q31_t)0xddca30e9,
	(q31_t)0x7b564d36, (q31_t)0xddc422f8, (q31_t)0x7b549ede, (q31_t)0xddbe151d, (q31_t)0x7b52f03a, (q31_t)0xddb80756,
	(q31_t)0x7b51414a, (q31_t)0xddb1f9a4,
	(q31_t)0x7b4f920e, (q31_t)0xddabec08, (q31_t)0x7b4de286, (q31_t)0xdda5de81, (q31_t)0x7b4c32b1, (q31_t)0xdd9fd10f,
	(q31_t)0x7b4a8291, (q31_t)0xdd99c3b2,
	(q31_t)0x7b48d225, (q31_t)0xdd93b66a, (q31_t)0x7b47216c, (q31_t)0xdd8da938, (q31_t)0x7b457068, (q31_t)0xdd879c1b,
	(q31_t)0x7b43bf17, (q31_t)0xdd818f13,
	(q31_t)0x7b420d7a, (q31_t)0xdd7b8220, (q31_t)0x7b405b92, (q31_t)0xdd757543, (q31_t)0x7b3ea95d, (q31_t)0xdd6f687b,
	(q31_t)0x7b3cf6dc, (q31_t)0xdd695bc9,
	(q31_t)0x7b3b4410, (q31_t)0xdd634f2b, (q31_t)0x7b3990f7, (q31_t)0xdd5d42a3, (q31_t)0x7b37dd92, (q31_t)0xdd573631,
	(q31_t)0x7b3629e1, (q31_t)0xdd5129d4,
	(q31_t)0x7b3475e5, (q31_t)0xdd4b1d8c, (q31_t)0x7b32c19c, (q31_t)0xdd451159, (q31_t)0x7b310d07, (q31_t)0xdd3f053c,
	(q31_t)0x7b2f5826, (q31_t)0xdd38f935,
	(q31_t)0x7b2da2fa, (q31_t)0xdd32ed43, (q31_t)0x7b2bed81, (q31_t)0xdd2ce166, (q31_t)0x7b2a37bc, (q31_t)0xdd26d59f,
	(q31_t)0x7b2881ac, (q31_t)0xdd20c9ed,
	(q31_t)0x7b26cb4f, (q31_t)0xdd1abe51, (q31_t)0x7b2514a6, (q31_t)0xdd14b2ca, (q31_t)0x7b235db2, (q31_t)0xdd0ea759,
	(q31_t)0x7b21a671, (q31_t)0xdd089bfe,
	(q31_t)0x7b1feee5, (q31_t)0xdd0290b8, (q31_t)0x7b1e370d, (q31_t)0xdcfc8588, (q31_t)0x7b1c7ee8, (q31_t)0xdcf67a6d,
	(q31_t)0x7b1ac678, (q31_t)0xdcf06f68,
	(q31_t)0x7b190dbc, (q31_t)0xdcea6478, (q31_t)0x7b1754b3, (q31_t)0xdce4599e, (q31_t)0x7b159b5f, (q31_t)0xdcde4eda,
	(q31_t)0x7b13e1bf, (q31_t)0xdcd8442b,
	(q31_t)0x7b1227d3, (q31_t)0xdcd23993, (q31_t)0x7b106d9b, (q31_t)0xdccc2f0f, (q31_t)0x7b0eb318, (q31_t)0xdcc624a2,
	(q31_t)0x7b0cf848, (q31_t)0xdcc01a4a,
	(q31_t)0x7b0b3d2c, (q31_t)0xdcba1008, (q31_t)0x7b0981c5, (q31_t)0xdcb405dc, (q31_t)0x7b07c612, (q31_t)0xdcadfbc5,
	(q31_t)0x7b060a12, (q31_t)0xdca7f1c5,
	(q31_t)0x7b044dc7, (q31_t)0xdca1e7da, (q31_t)0x7b029130, (q31_t)0xdc9bde05, (q31_t)0x7b00d44d, (q31_t)0xdc95d446,
	(q31_t)0x7aff171e, (q31_t)0xdc8fca9c,
	(q31_t)0x7afd59a4, (q31_t)0xdc89c109, (q31_t)0x7afb9bdd, (q31_t)0xdc83b78b, (q31_t)0x7af9ddcb, (q31_t)0xdc7dae23,
	(q31_t)0x7af81f6c, (q31_t)0xdc77a4d2,
	(q31_t)0x7af660c2, (q31_t)0xdc719b96, (q31_t)0x7af4a1cc, (q31_t)0xdc6b9270, (q31_t)0x7af2e28b, (q31_t)0xdc658960,
	(q31_t)0x7af122fd, (q31_t)0xdc5f8066,
	(q31_t)0x7aef6323, (q31_t)0xdc597781, (q31_t)0x7aeda2fe, (q31_t)0xdc536eb3, (q31_t)0x7aebe28d, (q31_t)0xdc4d65fb,
	(q31_t)0x7aea21d0, (q31_t)0xdc475d59,
	(q31_t)0x7ae860c7, (q31_t)0xdc4154cd, (q31_t)0x7ae69f73, (q31_t)0xdc3b4c57, (q31_t)0x7ae4ddd2, (q31_t)0xdc3543f7,
	(q31_t)0x7ae31be6, (q31_t)0xdc2f3bad,
	(q31_t)0x7ae159ae, (q31_t)0xdc293379, (q31_t)0x7adf972a, (q31_t)0xdc232b5c, (q31_t)0x7addd45b, (q31_t)0xdc1d2354,
	(q31_t)0x7adc113f, (q31_t)0xdc171b63,
	(q31_t)0x7ada4dd8, (q31_t)0xdc111388, (q31_t)0x7ad88a25, (q31_t)0xdc0b0bc2, (q31_t)0x7ad6c626, (q31_t)0xdc050414,
	(q31_t)0x7ad501dc, (q31_t)0xdbfefc7b,
	(q31_t)0x7ad33d45, (q31_t)0xdbf8f4f8, (q31_t)0x7ad17863, (q31_t)0xdbf2ed8c, (q31_t)0x7acfb336, (q31_t)0xdbece636,
	(q31_t)0x7acdedbc, (q31_t)0xdbe6def6,
	(q31_t)0x7acc27f7, (q31_t)0xdbe0d7cd, (q31_t)0x7aca61e6, (q31_t)0xdbdad0b9, (q31_t)0x7ac89b89, (q31_t)0xdbd4c9bc,
	(q31_t)0x7ac6d4e0, (q31_t)0xdbcec2d6,
	(q31_t)0x7ac50dec, (q31_t)0xdbc8bc06, (q31_t)0x7ac346ac, (q31_t)0xdbc2b54c, (q31_t)0x7ac17f20, (q31_t)0xdbbcaea8,
	(q31_t)0x7abfb749, (q31_t)0xdbb6a81b,
	(q31_t)0x7abdef25, (q31_t)0xdbb0a1a4, (q31_t)0x7abc26b7, (q31_t)0xdbaa9b43, (q31_t)0x7aba5dfc, (q31_t)0xdba494f9,
	(q31_t)0x7ab894f6, (q31_t)0xdb9e8ec6,
	(q31_t)0x7ab6cba4, (q31_t)0xdb9888a8, (q31_t)0x7ab50206, (q31_t)0xdb9282a2, (q31_t)0x7ab3381d, (q31_t)0xdb8c7cb1,
	(q31_t)0x7ab16de7, (q31_t)0xdb8676d8,
	(q31_t)0x7aafa367, (q31_t)0xdb807114, (q31_t)0x7aadd89a, (q31_t)0xdb7a6b68, (q31_t)0x7aac0d82, (q31_t)0xdb7465d1,
	(q31_t)0x7aaa421e, (q31_t)0xdb6e6052,
	(q31_t)0x7aa8766f, (q31_t)0xdb685ae9, (q31_t)0x7aa6aa74, (q31_t)0xdb625596, (q31_t)0x7aa4de2d, (q31_t)0xdb5c505a,
	(q31_t)0x7aa3119a, (q31_t)0xdb564b35,
	(q31_t)0x7aa144bc, (q31_t)0xdb504626, (q31_t)0x7a9f7793, (q31_t)0xdb4a412e, (q31_t)0x7a9daa1d, (q31_t)0xdb443c4c,
	(q31_t)0x7a9bdc5c, (q31_t)0xdb3e3781,
	(q31_t)0x7a9a0e50, (q31_t)0xdb3832cd, (q31_t)0x7a983ff7, (q31_t)0xdb322e30, (q31_t)0x7a967153, (q31_t)0xdb2c29a9,
	(q31_t)0x7a94a264, (q31_t)0xdb262539,
	(q31_t)0x7a92d329, (q31_t)0xdb2020e0, (q31_t)0x7a9103a2, (q31_t)0xdb1a1c9d, (q31_t)0x7a8f33d0, (q31_t)0xdb141871,
	(q31_t)0x7a8d63b2, (q31_t)0xdb0e145c,
	(q31_t)0x7a8b9348, (q31_t)0xdb08105e, (q31_t)0x7a89c293, (q31_t)0xdb020c77, (q31_t)0x7a87f192, (q31_t)0xdafc08a6,
	(q31_t)0x7a862046, (q31_t)0xdaf604ec,
	(q31_t)0x7a844eae, (q31_t)0xdaf00149, (q31_t)0x7a827ccb, (q31_t)0xdae9fdbd, (q31_t)0x7a80aa9c, (q31_t)0xdae3fa48,
	(q31_t)0x7a7ed821, (q31_t)0xdaddf6ea,
	(q31_t)0x7a7d055b, (q31_t)0xdad7f3a2, (q31_t)0x7a7b3249, (q31_t)0xdad1f072, (q31_t)0x7a795eec, (q31_t)0xdacbed58,
	(q31_t)0x7a778b43, (q31_t)0xdac5ea56,
	(q31_t)0x7a75b74f, (q31_t)0xdabfe76a, (q31_t)0x7a73e30f, (q31_t)0xdab9e495, (q31_t)0x7a720e84, (q31_t)0xdab3e1d8,
	(q31_t)0x7a7039ad, (q31_t)0xdaaddf31,
	(q31_t)0x7a6e648a, (q31_t)0xdaa7dca1, (q31_t)0x7a6c8f1c, (q31_t)0xdaa1da29, (q31_t)0x7a6ab963, (q31_t)0xda9bd7c7,
	(q31_t)0x7a68e35e, (q31_t)0xda95d57d,
	(q31_t)0x7a670d0d, (q31_t)0xda8fd349, (q31_t)0x7a653671, (q31_t)0xda89d12d, (q31_t)0x7a635f8a, (q31_t)0xda83cf28,
	(q31_t)0x7a618857, (q31_t)0xda7dcd3a,
	(q31_t)0x7a5fb0d8, (q31_t)0xda77cb63, (q31_t)0x7a5dd90e, (q31_t)0xda71c9a3, (q31_t)0x7a5c00f9, (q31_t)0xda6bc7fa,
	(q31_t)0x7a5a2898, (q31_t)0xda65c669,
	(q31_t)0x7a584feb, (q31_t)0xda5fc4ef, (q31_t)0x7a5676f3, (q31_t)0xda59c38c, (q31_t)0x7a549db0, (q31_t)0xda53c240,
	(q31_t)0x7a52c421, (q31_t)0xda4dc10b,
	(q31_t)0x7a50ea47, (q31_t)0xda47bfee, (q31_t)0x7a4f1021, (q31_t)0xda41bee8, (q31_t)0x7a4d35b0, (q31_t)0xda3bbdf9,
	(q31_t)0x7a4b5af3, (q31_t)0xda35bd22,
	(q31_t)0x7a497feb, (q31_t)0xda2fbc61, (q31_t)0x7a47a498, (q31_t)0xda29bbb9, (q31_t)0x7a45c8f9, (q31_t)0xda23bb27,
	(q31_t)0x7a43ed0e, (q31_t)0xda1dbaad,
	(q31_t)0x7a4210d8, (q31_t)0xda17ba4a, (q31_t)0x7a403457, (q31_t)0xda11b9ff, (q31_t)0x7a3e578b, (q31_t)0xda0bb9cb,
	(q31_t)0x7a3c7a73, (q31_t)0xda05b9ae,
	(q31_t)0x7a3a9d0f, (q31_t)0xd9ffb9a9, (q31_t)0x7a38bf60, (q31_t)0xd9f9b9bb, (q31_t)0x7a36e166, (q31_t)0xd9f3b9e5,
	(q31_t)0x7a350321, (q31_t)0xd9edba26,
	(q31_t)0x7a332490, (q31_t)0xd9e7ba7f, (q31_t)0x7a3145b3, (q31_t)0xd9e1baef, (q31_t)0x7a2f668c, (q31_t)0xd9dbbb77,
	(q31_t)0x7a2d8719, (q31_t)0xd9d5bc16,
	(q31_t)0x7a2ba75a, (q31_t)0xd9cfbccd, (q31_t)0x7a29c750, (q31_t)0xd9c9bd9b, (q31_t)0x7a27e6fb, (q31_t)0xd9c3be81,
	(q31_t)0x7a26065b, (q31_t)0xd9bdbf7e,
	(q31_t)0x7a24256f, (q31_t)0xd9b7c094, (q31_t)0x7a224437, (q31_t)0xd9b1c1c0, (q31_t)0x7a2062b5, (q31_t)0xd9abc305,
	(q31_t)0x7a1e80e7, (q31_t)0xd9a5c461,
	(q31_t)0x7a1c9ece, (q31_t)0xd99fc5d4, (q31_t)0x7a1abc69, (q31_t)0xd999c75f, (q31_t)0x7a18d9b9, (q31_t)0xd993c902,
	(q31_t)0x7a16f6be, (q31_t)0xd98dcabd,
	(q31_t)0x7a151378, (q31_t)0xd987cc90, (q31_t)0x7a132fe6, (q31_t)0xd981ce7a, (q31_t)0x7a114c09, (q31_t)0xd97bd07c,
	(q31_t)0x7a0f67e0, (q31_t)0xd975d295,
	(q31_t)0x7a0d836d, (q31_t)0xd96fd4c7, (q31_t)0x7a0b9eae, (q31_t)0xd969d710, (q31_t)0x7a09b9a4, (q31_t)0xd963d971,
	(q31_t)0x7a07d44e, (q31_t)0xd95ddbea,
	(q31_t)0x7a05eead, (q31_t)0xd957de7a, (q31_t)0x7a0408c1, (q31_t)0xd951e123, (q31_t)0x7a02228a, (q31_t)0xd94be3e3,
	(q31_t)0x7a003c07, (q31_t)0xd945e6bb,
	(q31_t)0x79fe5539, (q31_t)0xd93fe9ab, (q31_t)0x79fc6e20, (q31_t)0xd939ecb3, (q31_t)0x79fa86bc, (q31_t)0xd933efd3,
	(q31_t)0x79f89f0c, (q31_t)0xd92df30b,
	(q31_t)0x79f6b711, (q31_t)0xd927f65b, (q31_t)0x79f4cecb, (q31_t)0xd921f9c3, (q31_t)0x79f2e63a, (q31_t)0xd91bfd43,
	(q31_t)0x79f0fd5d, (q31_t)0xd91600da,
	(q31_t)0x79ef1436, (q31_t)0xd910048a, (q31_t)0x79ed2ac3, (q31_t)0xd90a0852, (q31_t)0x79eb4105, (q31_t)0xd9040c32,
	(q31_t)0x79e956fb, (q31_t)0xd8fe1029,
	(q31_t)0x79e76ca7, (q31_t)0xd8f81439, (q31_t)0x79e58207, (q31_t)0xd8f21861, (q31_t)0x79e3971c, (q31_t)0xd8ec1ca1,
	(q31_t)0x79e1abe6, (q31_t)0xd8e620fa,
	(q31_t)0x79dfc064, (q31_t)0xd8e0256a, (q31_t)0x79ddd498, (q31_t)0xd8da29f2, (q31_t)0x79dbe880, (q31_t)0xd8d42e93,
	(q31_t)0x79d9fc1d, (q31_t)0xd8ce334c,
	(q31_t)0x79d80f6f, (q31_t)0xd8c8381d, (q31_t)0x79d62276, (q31_t)0xd8c23d06, (q31_t)0x79d43532, (q31_t)0xd8bc4207,
	(q31_t)0x79d247a2, (q31_t)0xd8b64720,
	(q31_t)0x79d059c8, (q31_t)0xd8b04c52, (q31_t)0x79ce6ba2, (q31_t)0xd8aa519c, (q31_t)0x79cc7d31, (q31_t)0xd8a456ff,
	(q31_t)0x79ca8e75, (q31_t)0xd89e5c79,
	(q31_t)0x79c89f6e, (q31_t)0xd898620c, (q31_t)0x79c6b01b, (q31_t)0xd89267b7, (q31_t)0x79c4c07e, (q31_t)0xd88c6d7b,
	(q31_t)0x79c2d095, (q31_t)0xd8867356,
	(q31_t)0x79c0e062, (q31_t)0xd880794b, (q31_t)0x79beefe3, (q31_t)0xd87a7f57, (q31_t)0x79bcff19, (q31_t)0xd874857c,
	(q31_t)0x79bb0e04, (q31_t)0xd86e8bb9,
	(q31_t)0x79b91ca4, (q31_t)0xd868920f, (q31_t)0x79b72af9, (q31_t)0xd862987d, (q31_t)0x79b53903, (q31_t)0xd85c9f04,
	(q31_t)0x79b346c2, (q31_t)0xd856a5a3,
	(q31_t)0x79b15435, (q31_t)0xd850ac5a, (q31_t)0x79af615e, (q31_t)0xd84ab32a, (q31_t)0x79ad6e3c, (q31_t)0xd844ba13,
	(q31_t)0x79ab7ace, (q31_t)0xd83ec114,
	(q31_t)0x79a98715, (q31_t)0xd838c82d, (q31_t)0x79a79312, (q31_t)0xd832cf5f, (q31_t)0x79a59ec3, (q31_t)0xd82cd6aa,
	(q31_t)0x79a3aa29, (q31_t)0xd826de0d,
	(q31_t)0x79a1b545, (q31_t)0xd820e589, (q31_t)0x799fc015, (q31_t)0xd81aed1d, (q31_t)0x799dca9a, (q31_t)0xd814f4ca,
	(q31_t)0x799bd4d4, (q31_t)0xd80efc8f,
	(q31_t)0x7999dec4, (q31_t)0xd809046e, (q31_t)0x7997e868, (q31_t)0xd8030c64, (q31_t)0x7995f1c1, (q31_t)0xd7fd1474,
	(q31_t)0x7993facf, (q31_t)0xd7f71c9c,
	(q31_t)0x79920392, (q31_t)0xd7f124dd, (q31_t)0x79900c0a, (q31_t)0xd7eb2d37, (q31_t)0x798e1438, (q31_t)0xd7e535a9,
	(q31_t)0x798c1c1a, (q31_t)0xd7df3e34,
	(q31_t)0x798a23b1, (q31_t)0xd7d946d8, (q31_t)0x79882afd, (q31_t)0xd7d34f94, (q31_t)0x798631ff, (q31_t)0xd7cd586a,
	(q31_t)0x798438b5, (q31_t)0xd7c76158,
	(q31_t)0x79823f20, (q31_t)0xd7c16a5f, (q31_t)0x79804541, (q31_t)0xd7bb737f, (q31_t)0x797e4b16, (q31_t)0xd7b57cb7,
	(q31_t)0x797c50a1, (q31_t)0xd7af8609,
	(q31_t)0x797a55e0, (q31_t)0xd7a98f73, (q31_t)0x79785ad5, (q31_t)0xd7a398f6, (q31_t)0x79765f7f, (q31_t)0xd79da293,
	(q31_t)0x797463de, (q31_t)0xd797ac48,
	(q31_t)0x797267f2, (q31_t)0xd791b616, (q31_t)0x79706bbb, (q31_t)0xd78bbffc, (q31_t)0x796e6f39, (q31_t)0xd785c9fc,
	(q31_t)0x796c726c, (q31_t)0xd77fd415,
	(q31_t)0x796a7554, (q31_t)0xd779de47, (q31_t)0x796877f1, (q31_t)0xd773e892, (q31_t)0x79667a44, (q31_t)0xd76df2f6,
	(q31_t)0x79647c4c, (q31_t)0xd767fd72,
	(q31_t)0x79627e08, (q31_t)0xd7620808, (q31_t)0x79607f7a, (q31_t)0xd75c12b7, (q31_t)0x795e80a1, (q31_t)0xd7561d7f,
	(q31_t)0x795c817d, (q31_t)0xd7502860,
	(q31_t)0x795a820e, (q31_t)0xd74a335b, (q31_t)0x79588255, (q31_t)0xd7443e6e, (q31_t)0x79568250, (q31_t)0xd73e499a,
	(q31_t)0x79548201, (q31_t)0xd73854e0,
	(q31_t)0x79528167, (q31_t)0xd732603f, (q31_t)0x79508082, (q31_t)0xd72c6bb6, (q31_t)0x794e7f52, (q31_t)0xd7267748,
	(q31_t)0x794c7dd7, (q31_t)0xd72082f2,
	(q31_t)0x794a7c12, (q31_t)0xd71a8eb5, (q31_t)0x79487a01, (q31_t)0xd7149a92, (q31_t)0x794677a6, (q31_t)0xd70ea688,
	(q31_t)0x79447500, (q31_t)0xd708b297,
	(q31_t)0x79427210, (q31_t)0xd702bec0, (q31_t)0x79406ed4, (q31_t)0xd6fccb01, (q31_t)0x793e6b4e, (q31_t)0xd6f6d75d,
	(q31_t)0x793c677d, (q31_t)0xd6f0e3d1,
	(q31_t)0x793a6361, (q31_t)0xd6eaf05f, (q31_t)0x79385efa, (q31_t)0xd6e4fd06, (q31_t)0x79365a49, (q31_t)0xd6df09c6,
	(q31_t)0x7934554d, (q31_t)0xd6d916a0,
	(q31_t)0x79325006, (q31_t)0xd6d32393, (q31_t)0x79304a74, (q31_t)0xd6cd30a0, (q31_t)0x792e4497, (q31_t)0xd6c73dc6,
	(q31_t)0x792c3e70, (q31_t)0xd6c14b05,
	(q31_t)0x792a37fe, (q31_t)0xd6bb585e, (q31_t)0x79283141, (q31_t)0xd6b565d0, (q31_t)0x79262a3a, (q31_t)0xd6af735c,
	(q31_t)0x792422e8, (q31_t)0xd6a98101,
	(q31_t)0x79221b4b, (q31_t)0xd6a38ec0, (q31_t)0x79201363, (q31_t)0xd69d9c98, (q31_t)0x791e0b31, (q31_t)0xd697aa8a,
	(q31_t)0x791c02b4, (q31_t)0xd691b895,
	(q31_t)0x7919f9ec, (q31_t)0xd68bc6ba, (q31_t)0x7917f0d9, (q31_t)0xd685d4f9, (q31_t)0x7915e77c, (q31_t)0xd67fe351,
	(q31_t)0x7913ddd4, (q31_t)0xd679f1c2,
	(q31_t)0x7911d3e2, (q31_t)0xd674004e, (q31_t)0x790fc9a4, (q31_t)0xd66e0ef2, (q31_t)0x790dbf1d, (q31_t)0xd6681db1,
	(q31_t)0x790bb44a, (q31_t)0xd6622c89,
	(q31_t)0x7909a92d, (q31_t)0xd65c3b7b, (q31_t)0x79079dc5, (q31_t)0xd6564a87, (q31_t)0x79059212, (q31_t)0xd65059ac,
	(q31_t)0x79038615, (q31_t)0xd64a68eb,
	(q31_t)0x790179cd, (q31_t)0xd6447844, (q31_t)0x78ff6d3b, (q31_t)0xd63e87b6, (q31_t)0x78fd605d, (q31_t)0xd6389742,
	(q31_t)0x78fb5336, (q31_t)0xd632a6e8,
	(q31_t)0x78f945c3, (q31_t)0xd62cb6a8, (q31_t)0x78f73806, (q31_t)0xd626c681, (q31_t)0x78f529fe, (q31_t)0xd620d675,
	(q31_t)0x78f31bac, (q31_t)0xd61ae682,
	(q31_t)0x78f10d0f, (q31_t)0xd614f6a9, (q31_t)0x78eefe28, (q31_t)0xd60f06ea, (q31_t)0x78eceef6, (q31_t)0xd6091745,
	(q31_t)0x78eadf79, (q31_t)0xd60327b9,
	(q31_t)0x78e8cfb2, (q31_t)0xd5fd3848, (q31_t)0x78e6bfa0, (q31_t)0xd5f748f0, (q31_t)0x78e4af44, (q31_t)0xd5f159b3,
	(q31_t)0x78e29e9d, (q31_t)0xd5eb6a8f,
	(q31_t)0x78e08dab, (q31_t)0xd5e57b85, (q31_t)0x78de7c6f, (q31_t)0xd5df8c96, (q31_t)0x78dc6ae8, (q31_t)0xd5d99dc0,
	(q31_t)0x78da5917, (q31_t)0xd5d3af04,
	(q31_t)0x78d846fb, (q31_t)0xd5cdc062, (q31_t)0x78d63495, (q31_t)0xd5c7d1db, (q31_t)0x78d421e4, (q31_t)0xd5c1e36d,
	(q31_t)0x78d20ee9, (q31_t)0xd5bbf519,
	(q31_t)0x78cffba3, (q31_t)0xd5b606e0, (q31_t)0x78cde812, (q31_t)0xd5b018c0, (q31_t)0x78cbd437, (q31_t)0xd5aa2abb,
	(q31_t)0x78c9c012, (q31_t)0xd5a43cd0,
	(q31_t)0x78c7aba2, (q31_t)0xd59e4eff, (q31_t)0x78c596e7, (q31_t)0xd5986148, (q31_t)0x78c381e2, (q31_t)0xd59273ab,
	(q31_t)0x78c16c93, (q31_t)0xd58c8628,
	(q31_t)0x78bf56f9, (q31_t)0xd58698c0, (q31_t)0x78bd4114, (q31_t)0xd580ab72, (q31_t)0x78bb2ae5, (q31_t)0xd57abe3d,
	(q31_t)0x78b9146c, (q31_t)0xd574d124,
	(q31_t)0x78b6fda8, (q31_t)0xd56ee424, (q31_t)0x78b4e69a, (q31_t)0xd568f73f, (q31_t)0x78b2cf41, (q31_t)0xd5630a74,
	(q31_t)0x78b0b79e, (q31_t)0xd55d1dc3,
	(q31_t)0x78ae9fb0, (q31_t)0xd557312d, (q31_t)0x78ac8778, (q31_t)0xd55144b0, (q31_t)0x78aa6ef5, (q31_t)0xd54b584f,
	(q31_t)0x78a85628, (q31_t)0xd5456c07,
	(q31_t)0x78a63d11, (q31_t)0xd53f7fda, (q31_t)0x78a423af, (q31_t)0xd53993c7, (q31_t)0x78a20a03, (q31_t)0xd533a7cf,
	(q31_t)0x789ff00c, (q31_t)0xd52dbbf1,
	(q31_t)0x789dd5cb, (q31_t)0xd527d02e, (q31_t)0x789bbb3f, (q31_t)0xd521e484, (q31_t)0x7899a06a, (q31_t)0xd51bf8f6,
	(q31_t)0x78978549, (q31_t)0xd5160d82,
	(q31_t)0x789569df, (q31_t)0xd5102228, (q31_t)0x78934e2a, (q31_t)0xd50a36e9, (q31_t)0x7891322a, (q31_t)0xd5044bc4,
	(q31_t)0x788f15e0, (q31_t)0xd4fe60ba,
	(q31_t)0x788cf94c, (q31_t)0xd4f875ca, (q31_t)0x788adc6e, (q31_t)0xd4f28af5, (q31_t)0x7888bf45, (q31_t)0xd4eca03a,
	(q31_t)0x7886a1d1, (q31_t)0xd4e6b59a,
	(q31_t)0x78848414, (q31_t)0xd4e0cb15, (q31_t)0x7882660c, (q31_t)0xd4dae0aa, (q31_t)0x788047ba, (q31_t)0xd4d4f65a,
	(q31_t)0x787e291d, (q31_t)0xd4cf0c24,
	(q31_t)0x787c0a36, (q31_t)0xd4c92209, (q31_t)0x7879eb05, (q31_t)0xd4c33809, (q31_t)0x7877cb89, (q31_t)0xd4bd4e23,
	(q31_t)0x7875abc3, (q31_t)0xd4b76458,
	(q31_t)0x78738bb3, (q31_t)0xd4b17aa8, (q31_t)0x78716b59, (q31_t)0xd4ab9112, (q31_t)0x786f4ab4, (q31_t)0xd4a5a798,
	(q31_t)0x786d29c5, (q31_t)0xd49fbe37,
	(q31_t)0x786b088c, (q31_t)0xd499d4f2, (q31_t)0x7868e708, (q31_t)0xd493ebc8, (q31_t)0x7866c53a, (q31_t)0xd48e02b8,
	(q31_t)0x7864a322, (q31_t)0xd48819c3,
	(q31_t)0x786280bf, (q31_t)0xd48230e9, (q31_t)0x78605e13, (q31_t)0xd47c4829, (q31_t)0x785e3b1c, (q31_t)0xd4765f85,
	(q31_t)0x785c17db, (q31_t)0xd47076fb,
	(q31_t)0x7859f44f, (q31_t)0xd46a8e8d, (q31_t)0x7857d079, (q31_t)0xd464a639, (q31_t)0x7855ac5a, (q31_t)0xd45ebe00,
	(q31_t)0x785387ef, (q31_t)0xd458d5e2,
	(q31_t)0x7851633b, (q31_t)0xd452eddf, (q31_t)0x784f3e3c, (q31_t)0xd44d05f6, (q31_t)0x784d18f4, (q31_t)0xd4471e29,
	(q31_t)0x784af361, (q31_t)0xd4413677,
	(q31_t)0x7848cd83, (q31_t)0xd43b4ee0, (q31_t)0x7846a75c, (q31_t)0xd4356763, (q31_t)0x784480ea, (q31_t)0xd42f8002,
	(q31_t)0x78425a2f, (q31_t)0xd42998bc,
	(q31_t)0x78403329, (q31_t)0xd423b191, (q31_t)0x783e0bd9, (q31_t)0xd41dca81, (q31_t)0x783be43e, (q31_t)0xd417e38c,
	(q31_t)0x7839bc5a, (q31_t)0xd411fcb2,
	(q31_t)0x7837942b, (q31_t)0xd40c15f3, (q31_t)0x78356bb2, (q31_t)0xd4062f4f, (q31_t)0x783342ef, (q31_t)0xd40048c6,
	(q31_t)0x783119e2, (q31_t)0xd3fa6259,
	(q31_t)0x782ef08b, (q31_t)0xd3f47c06, (q31_t)0x782cc6ea, (q31_t)0xd3ee95cf, (q31_t)0x782a9cfe, (q31_t)0xd3e8afb3,
	(q31_t)0x782872c8, (q31_t)0xd3e2c9b2,
	(q31_t)0x78264849, (q31_t)0xd3dce3cd, (q31_t)0x78241d7f, (q31_t)0xd3d6fe03, (q31_t)0x7821f26b, (q31_t)0xd3d11853,
	(q31_t)0x781fc70d, (q31_t)0xd3cb32c0,
	(q31_t)0x781d9b65, (q31_t)0xd3c54d47, (q31_t)0x781b6f72, (q31_t)0xd3bf67ea, (q31_t)0x78194336, (q31_t)0xd3b982a8,
	(q31_t)0x781716b0, (q31_t)0xd3b39d81,
	(q31_t)0x7814e9df, (q31_t)0xd3adb876, (q31_t)0x7812bcc4, (q31_t)0xd3a7d385, (q31_t)0x78108f60, (q31_t)0xd3a1eeb1,
	(q31_t)0x780e61b1, (q31_t)0xd39c09f7,
	(q31_t)0x780c33b8, (q31_t)0xd396255a, (q31_t)0x780a0575, (q31_t)0xd39040d7, (q31_t)0x7807d6e9, (q31_t)0xd38a5c70,
	(q31_t)0x7805a812, (q31_t)0xd3847824,
	(q31_t)0x780378f1, (q31_t)0xd37e93f4, (q31_t)0x78014986, (q31_t)0xd378afdf, (q31_t)0x77ff19d1, (q31_t)0xd372cbe6,
	(q31_t)0x77fce9d2, (q31_t)0xd36ce808,
	(q31_t)0x77fab989, (q31_t)0xd3670446, (q31_t)0x77f888f6, (q31_t)0xd361209f, (q31_t)0x77f65819, (q31_t)0xd35b3d13,
	(q31_t)0x77f426f2, (q31_t)0xd35559a4,
	(q31_t)0x77f1f581, (q31_t)0xd34f764f, (q31_t)0x77efc3c5, (q31_t)0xd3499317, (q31_t)0x77ed91c0, (q31_t)0xd343affa,
	(q31_t)0x77eb5f71, (q31_t)0xd33dccf8,
	(q31_t)0x77e92cd9, (q31_t)0xd337ea12, (q31_t)0x77e6f9f6, (q31_t)0xd3320748, (q31_t)0x77e4c6c9, (q31_t)0xd32c2499,
	(q31_t)0x77e29352, (q31_t)0xd3264206,
	(q31_t)0x77e05f91, (q31_t)0xd3205f8f, (q31_t)0x77de2b86, (q31_t)0xd31a7d33, (q31_t)0x77dbf732, (q31_t)0xd3149af3,
	(q31_t)0x77d9c293, (q31_t)0xd30eb8cf,
	(q31_t)0x77d78daa, (q31_t)0xd308d6c7, (q31_t)0x77d55878, (q31_t)0xd302f4da, (q31_t)0x77d322fc, (q31_t)0xd2fd1309,
	(q31_t)0x77d0ed35, (q31_t)0xd2f73154,
	(q31_t)0x77ceb725, (q31_t)0xd2f14fba, (q31_t)0x77cc80cb, (q31_t)0xd2eb6e3c, (q31_t)0x77ca4a27, (q31_t)0xd2e58cdb,
	(q31_t)0x77c81339, (q31_t)0xd2dfab95,
	(q31_t)0x77c5dc01, (q31_t)0xd2d9ca6a, (q31_t)0x77c3a47f, (q31_t)0xd2d3e95c, (q31_t)0x77c16cb4, (q31_t)0xd2ce0869,
	(q31_t)0x77bf349f, (q31_t)0xd2c82793,
	(q31_t)0x77bcfc3f, (q31_t)0xd2c246d8, (q31_t)0x77bac396, (q31_t)0xd2bc6639, (q31_t)0x77b88aa3, (q31_t)0xd2b685b6,
	(q31_t)0x77b65166, (q31_t)0xd2b0a54f,
	(q31_t)0x77b417df, (q31_t)0xd2aac504, (q31_t)0x77b1de0f, (q31_t)0xd2a4e4d5, (q31_t)0x77afa3f5, (q31_t)0xd29f04c2,
	(q31_t)0x77ad6990, (q31_t)0xd29924cb,
	(q31_t)0x77ab2ee2, (q31_t)0xd29344f0, (q31_t)0x77a8f3ea, (q31_t)0xd28d6531, (q31_t)0x77a6b8a9, (q31_t)0xd287858e,
	(q31_t)0x77a47d1d, (q31_t)0xd281a607,
	(q31_t)0x77a24148, (q31_t)0xd27bc69c, (q31_t)0x77a00529, (q31_t)0xd275e74d, (q31_t)0x779dc8c0, (q31_t)0xd270081b,
	(q31_t)0x779b8c0e, (q31_t)0xd26a2904,
	(q31_t)0x77994f11, (q31_t)0xd2644a0a, (q31_t)0x779711cb, (q31_t)0xd25e6b2b, (q31_t)0x7794d43b, (q31_t)0xd2588c69,
	(q31_t)0x77929661, (q31_t)0xd252adc3,
	(q31_t)0x7790583e, (q31_t)0xd24ccf39, (q31_t)0x778e19d0, (q31_t)0xd246f0cb, (q31_t)0x778bdb19, (q31_t)0xd241127a,
	(q31_t)0x77899c19, (q31_t)0xd23b3444,
	(q31_t)0x77875cce, (q31_t)0xd235562b, (q31_t)0x77851d3a, (q31_t)0xd22f782f, (q31_t)0x7782dd5c, (q31_t)0xd2299a4e,
	(q31_t)0x77809d35, (q31_t)0xd223bc8a,
	(q31_t)0x777e5cc3, (q31_t)0xd21ddee2, (q31_t)0x777c1c08, (q31_t)0xd2180156, (q31_t)0x7779db03, (q31_t)0xd21223e7,
	(q31_t)0x777799b5, (q31_t)0xd20c4694,
	(q31_t)0x7775581d, (q31_t)0xd206695d, (q31_t)0x7773163b, (q31_t)0xd2008c43, (q31_t)0x7770d40f, (q31_t)0xd1faaf45,
	(q31_t)0x776e919a, (q31_t)0xd1f4d263,
	(q31_t)0x776c4edb, (q31_t)0xd1eef59e, (q31_t)0x776a0bd3, (q31_t)0xd1e918f5, (q31_t)0x7767c880, (q31_t)0xd1e33c69,
	(q31_t)0x776584e5, (q31_t)0xd1dd5ff9,
	(q31_t)0x776340ff, (q31_t)0xd1d783a6, (q31_t)0x7760fcd0, (q31_t)0xd1d1a76f, (q31_t)0x775eb857, (q31_t)0xd1cbcb54,
	(q31_t)0x775c7395, (q31_t)0xd1c5ef56,
	(q31_t)0x775a2e89, (q31_t)0xd1c01375, (q31_t)0x7757e933, (q31_t)0xd1ba37b0, (q31_t)0x7755a394, (q31_t)0xd1b45c08,
	(q31_t)0x77535dab, (q31_t)0xd1ae807c,
	(q31_t)0x77511778, (q31_t)0xd1a8a50d, (q31_t)0x774ed0fc, (q31_t)0xd1a2c9ba, (q31_t)0x774c8a36, (q31_t)0xd19cee84,
	(q31_t)0x774a4327, (q31_t)0xd197136b,
	(q31_t)0x7747fbce, (q31_t)0xd191386e, (q31_t)0x7745b42c, (q31_t)0xd18b5d8e, (q31_t)0x77436c40, (q31_t)0xd18582ca,
	(q31_t)0x7741240a, (q31_t)0xd17fa823,
	(q31_t)0x773edb8b, (q31_t)0xd179cd99, (q31_t)0x773c92c2, (q31_t)0xd173f32c, (q31_t)0x773a49b0, (q31_t)0xd16e18db,
	(q31_t)0x77380054, (q31_t)0xd1683ea7,
	(q31_t)0x7735b6af, (q31_t)0xd1626490, (q31_t)0x77336cc0, (q31_t)0xd15c8a95, (q31_t)0x77312287, (q31_t)0xd156b0b7,
	(q31_t)0x772ed805, (q31_t)0xd150d6f6,
	(q31_t)0x772c8d3a, (q31_t)0xd14afd52, (q31_t)0x772a4225, (q31_t)0xd14523cb, (q31_t)0x7727f6c6, (q31_t)0xd13f4a60,
	(q31_t)0x7725ab1f, (q31_t)0xd1397113,
	(q31_t)0x77235f2d, (q31_t)0xd13397e2, (q31_t)0x772112f2, (q31_t)0xd12dbece, (q31_t)0x771ec66e, (q31_t)0xd127e5d7,
	(q31_t)0x771c79a0, (q31_t)0xd1220cfc,
	(q31_t)0x771a2c88, (q31_t)0xd11c343f, (q31_t)0x7717df27, (q31_t)0xd1165b9f, (q31_t)0x7715917d, (q31_t)0xd110831b,
	(q31_t)0x77134389, (q31_t)0xd10aaab5,
	(q31_t)0x7710f54c, (q31_t)0xd104d26b, (q31_t)0x770ea6c5, (q31_t)0xd0fefa3f, (q31_t)0x770c57f5, (q31_t)0xd0f9222f,
	(q31_t)0x770a08dc, (q31_t)0xd0f34a3d,
	(q31_t)0x7707b979, (q31_t)0xd0ed7267, (q31_t)0x770569cc, (q31_t)0xd0e79aaf, (q31_t)0x770319d6, (q31_t)0xd0e1c313,
	(q31_t)0x7700c997, (q31_t)0xd0dbeb95,
	(q31_t)0x76fe790e, (q31_t)0xd0d61434, (q31_t)0x76fc283c, (q31_t)0xd0d03cf0, (q31_t)0x76f9d721, (q31_t)0xd0ca65c9,
	(q31_t)0x76f785bc, (q31_t)0xd0c48ebf,
	(q31_t)0x76f5340e, (q31_t)0xd0beb7d2, (q31_t)0x76f2e216, (q31_t)0xd0b8e102, (q31_t)0x76f08fd5, (q31_t)0xd0b30a50,
	(q31_t)0x76ee3d4b, (q31_t)0xd0ad33ba,
	(q31_t)0x76ebea77, (q31_t)0xd0a75d42, (q31_t)0x76e9975a, (q31_t)0xd0a186e7, (q31_t)0x76e743f4, (q31_t)0xd09bb0aa,
	(q31_t)0x76e4f044, (q31_t)0xd095da89,
	(q31_t)0x76e29c4b, (q31_t)0xd0900486, (q31_t)0x76e04808, (q31_t)0xd08a2ea0, (q31_t)0x76ddf37c, (q31_t)0xd08458d7,
	(q31_t)0x76db9ea7, (q31_t)0xd07e832c,
	(q31_t)0x76d94989, (q31_t)0xd078ad9e, (q31_t)0x76d6f421, (q31_t)0xd072d82d, (q31_t)0x76d49e70, (q31_t)0xd06d02da,
	(q31_t)0x76d24876, (q31_t)0xd0672da3,
	(q31_t)0x76cff232, (q31_t)0xd061588b, (q31_t)0x76cd9ba5, (q31_t)0xd05b838f, (q31_t)0x76cb44cf, (q31_t)0xd055aeb1,
	(q31_t)0x76c8edb0, (q31_t)0xd04fd9f1,
	(q31_t)0x76c69647, (q31_t)0xd04a054e, (q31_t)0x76c43e95, (q31_t)0xd04430c8, (q31_t)0x76c1e699, (q31_t)0xd03e5c60,
	(q31_t)0x76bf8e55, (q31_t)0xd0388815,
	(q31_t)0x76bd35c7, (q31_t)0xd032b3e7, (q31_t)0x76badcf0, (q31_t)0xd02cdfd8, (q31_t)0x76b883d0, (q31_t)0xd0270be5,
	(q31_t)0x76b62a66, (q31_t)0xd0213810,
	(q31_t)0x76b3d0b4, (q31_t)0xd01b6459, (q31_t)0x76b176b8, (q31_t)0xd01590bf, (q31_t)0x76af1c72, (q31_t)0xd00fbd43,
	(q31_t)0x76acc1e4, (q31_t)0xd009e9e4,
	(q31_t)0x76aa670d, (q31_t)0xd00416a3, (q31_t)0x76a80bec, (q31_t)0xcffe4380, (q31_t)0x76a5b082, (q31_t)0xcff8707a,
	(q31_t)0x76a354cf, (q31_t)0xcff29d92,
	(q31_t)0x76a0f8d2, (q31_t)0xcfeccac7, (q31_t)0x769e9c8d, (q31_t)0xcfe6f81a, (q31_t)0x769c3ffe, (q31_t)0xcfe1258b,
	(q31_t)0x7699e326, (q31_t)0xcfdb531a,
	(q31_t)0x76978605, (q31_t)0xcfd580c6, (q31_t)0x7695289b, (q31_t)0xcfcfae8f, (q31_t)0x7692cae8, (q31_t)0xcfc9dc77,
	(q31_t)0x76906ceb, (q31_t)0xcfc40a7c,
	(q31_t)0x768e0ea6, (q31_t)0xcfbe389f, (q31_t)0x768bb017, (q31_t)0xcfb866e0, (q31_t)0x7689513f, (q31_t)0xcfb2953f,
	(q31_t)0x7686f21e, (q31_t)0xcfacc3bb,
	(q31_t)0x768492b4, (q31_t)0xcfa6f255, (q31_t)0x76823301, (q31_t)0xcfa1210d, (q31_t)0x767fd304, (q31_t)0xcf9b4fe3,
	(q31_t)0x767d72bf, (q31_t)0xcf957ed7,
	(q31_t)0x767b1231, (q31_t)0xcf8fade9, (q31_t)0x7678b159, (q31_t)0xcf89dd18, (q31_t)0x76765038, (q31_t)0xcf840c65,
	(q31_t)0x7673eecf, (q31_t)0xcf7e3bd1,
	(q31_t)0x76718d1c, (q31_t)0xcf786b5a, (q31_t)0x766f2b20, (q31_t)0xcf729b01, (q31_t)0x766cc8db, (q31_t)0xcf6ccac6,
	(q31_t)0x766a664d, (q31_t)0xcf66faa9,
	(q31_t)0x76680376, (q31_t)0xcf612aaa, (q31_t)0x7665a056, (q31_t)0xcf5b5ac9, (q31_t)0x76633ced, (q31_t)0xcf558b06,
	(q31_t)0x7660d93b, (q31_t)0xcf4fbb61,
	(q31_t)0x765e7540, (q31_t)0xcf49ebda, (q31_t)0x765c10fc, (q31_t)0xcf441c71, (q31_t)0x7659ac6f, (q31_t)0xcf3e4d26,
	(q31_t)0x76574798, (q31_t)0xcf387dfa,
	(q31_t)0x7654e279, (q31_t)0xcf32aeeb, (q31_t)0x76527d11, (q31_t)0xcf2cdffa, (q31_t)0x76501760, (q31_t)0xcf271128,
	(q31_t)0x764db166, (q31_t)0xcf214274,
	(q31_t)0x764b4b23, (q31_t)0xcf1b73de, (q31_t)0x7648e497, (q31_t)0xcf15a566, (q31_t)0x76467dc2, (q31_t)0xcf0fd70c,
	(q31_t)0x764416a4, (q31_t)0xcf0a08d0,
	(q31_t)0x7641af3d, (q31_t)0xcf043ab3, (q31_t)0x763f478d, (q31_t)0xcefe6cb3, (q31_t)0x763cdf94, (q31_t)0xcef89ed2,
	(q31_t)0x763a7752, (q31_t)0xcef2d110,
	(q31_t)0x76380ec8, (q31_t)0xceed036b, (q31_t)0x7635a5f4, (q31_t)0xcee735e5, (q31_t)0x76333cd8, (q31_t)0xcee1687d,
	(q31_t)0x7630d372, (q31_t)0xcedb9b33,
	(q31_t)0x762e69c4, (q31_t)0xced5ce08, (q31_t)0x762bffcd, (q31_t)0xced000fb, (q31_t)0x7629958c, (q31_t)0xceca340c,
	(q31_t)0x76272b03, (q31_t)0xcec4673c,
	(q31_t)0x7624c031, (q31_t)0xcebe9a8a, (q31_t)0x76225517, (q31_t)0xceb8cdf7, (q31_t)0x761fe9b3, (q31_t)0xceb30181,
	(q31_t)0x761d7e06, (q31_t)0xcead352b,
	(q31_t)0x761b1211, (q31_t)0xcea768f2, (q31_t)0x7618a5d3, (q31_t)0xcea19cd8, (q31_t)0x7616394c, (q31_t)0xce9bd0dd,
	(q31_t)0x7613cc7c, (q31_t)0xce960500,
	(q31_t)0x76115f63, (q31_t)0xce903942, (q31_t)0x760ef201, (q31_t)0xce8a6da2, (q31_t)0x760c8457, (q31_t)0xce84a220,
	(q31_t)0x760a1664, (q31_t)0xce7ed6bd,
	(q31_t)0x7607a828, (q31_t)0xce790b79, (q31_t)0x760539a3, (q31_t)0xce734053, (q31_t)0x7602cad5, (q31_t)0xce6d754c,
	(q31_t)0x76005bbf, (q31_t)0xce67aa63,
	(q31_t)0x75fdec60, (q31_t)0xce61df99, (q31_t)0x75fb7cb8, (q31_t)0xce5c14ed, (q31_t)0x75f90cc7, (q31_t)0xce564a60,
	(q31_t)0x75f69c8d, (q31_t)0xce507ff2,
	(q31_t)0x75f42c0b, (q31_t)0xce4ab5a2, (q31_t)0x75f1bb40, (q31_t)0xce44eb71, (q31_t)0x75ef4a2c, (q31_t)0xce3f215f,
	(q31_t)0x75ecd8cf, (q31_t)0xce39576c,
	(q31_t)0x75ea672a, (q31_t)0xce338d97, (q31_t)0x75e7f53c, (q31_t)0xce2dc3e1, (q31_t)0x75e58305, (q31_t)0xce27fa49,
	(q31_t)0x75e31086, (q31_t)0xce2230d0,
	(q31_t)0x75e09dbd, (q31_t)0xce1c6777, (q31_t)0x75de2aac, (q31_t)0xce169e3b, (q31_t)0x75dbb753, (q31_t)0xce10d51f,
	(q31_t)0x75d943b0, (q31_t)0xce0b0c21,
	(q31_t)0x75d6cfc5, (q31_t)0xce054343, (q31_t)0x75d45b92, (q31_t)0xcdff7a83, (q31_t)0x75d1e715, (q31_t)0xcdf9b1e2,
	(q31_t)0x75cf7250, (q31_t)0xcdf3e95f,
	(q31_t)0x75ccfd42, (q31_t)0xcdee20fc, (q31_t)0x75ca87ec, (q31_t)0xcde858b8, (q31_t)0x75c8124d, (q31_t)0xcde29092,
	(q31_t)0x75c59c65, (q31_t)0xcddcc88b,
	(q31_t)0x75c32634, (q31_t)0xcdd700a4, (q31_t)0x75c0afbb, (q31_t)0xcdd138db, (q31_t)0x75be38fa, (q31_t)0xcdcb7131,
	(q31_t)0x75bbc1ef, (q31_t)0xcdc5a9a6,
	(q31_t)0x75b94a9c, (q31_t)0xcdbfe23a, (q31_t)0x75b6d301, (q31_t)0xcdba1aee, (q31_t)0x75b45b1d, (q31_t)0xcdb453c0,
	(q31_t)0x75b1e2f0, (q31_t)0xcdae8cb1,
	(q31_t)0x75af6a7b, (q31_t)0xcda8c5c1, (q31_t)0x75acf1bd, (q31_t)0xcda2fef0, (q31_t)0x75aa78b6, (q31_t)0xcd9d383f,
	(q31_t)0x75a7ff67, (q31_t)0xcd9771ac,
	(q31_t)0x75a585cf, (q31_t)0xcd91ab39, (q31_t)0x75a30bef, (q31_t)0xcd8be4e4, (q31_t)0x75a091c6, (q31_t)0xcd861eaf,
	(q31_t)0x759e1755, (q31_t)0xcd805899,
	(q31_t)0x759b9c9b, (q31_t)0xcd7a92a2, (q31_t)0x75992198, (q31_t)0xcd74ccca, (q31_t)0x7596a64d, (q31_t)0xcd6f0711,
	(q31_t)0x75942ab9, (q31_t)0xcd694178,
	(q31_t)0x7591aedd, (q31_t)0xcd637bfe, (q31_t)0x758f32b9, (q31_t)0xcd5db6a3, (q31_t)0x758cb64c, (q31_t)0xcd57f167,
	(q31_t)0x758a3996, (q31_t)0xcd522c4a,
	(q31_t)0x7587bc98, (q31_t)0xcd4c674d, (q31_t)0x75853f51, (q31_t)0xcd46a26f, (q31_t)0x7582c1c2, (q31_t)0xcd40ddb0,
	(q31_t)0x758043ea, (q31_t)0xcd3b1911,
	(q31_t)0x757dc5ca, (q31_t)0xcd355491, (q31_t)0x757b4762, (q31_t)0xcd2f9030, (q31_t)0x7578c8b0, (q31_t)0xcd29cbee,
	(q31_t)0x757649b7, (q31_t)0xcd2407cc,
	(q31_t)0x7573ca75, (q31_t)0xcd1e43ca, (q31_t)0x75714aea, (q31_t)0xcd187fe6, (q31_t)0x756ecb18, (q31_t)0xcd12bc22,
	(q31_t)0x756c4afc, (q31_t)0xcd0cf87e,
	(q31_t)0x7569ca99, (q31_t)0xcd0734f9, (q31_t)0x756749ec, (q31_t)0xcd017193, (q31_t)0x7564c8f8, (q31_t)0xccfbae4d,
	(q31_t)0x756247bb, (q31_t)0xccf5eb26,
	(q31_t)0x755fc635, (q31_t)0xccf0281f, (q31_t)0x755d4467, (q31_t)0xccea6538, (q31_t)0x755ac251, (q31_t)0xcce4a26f,
	(q31_t)0x75583ff3, (q31_t)0xccdedfc7,
	(q31_t)0x7555bd4c, (q31_t)0xccd91d3d, (q31_t)0x75533a5c, (q31_t)0xccd35ad4, (q31_t)0x7550b725, (q31_t)0xcccd988a,
	(q31_t)0x754e33a4, (q31_t)0xccc7d65f,
	(q31_t)0x754bafdc, (q31_t)0xccc21455, (q31_t)0x75492bcb, (q31_t)0xccbc5269, (q31_t)0x7546a772, (q31_t)0xccb6909e,
	(q31_t)0x754422d0, (q31_t)0xccb0cef2,
	(q31_t)0x75419de7, (q31_t)0xccab0d65, (q31_t)0x753f18b4, (q31_t)0xcca54bf9, (q31_t)0x753c933a, (q31_t)0xcc9f8aac,
	(q31_t)0x753a0d77, (q31_t)0xcc99c97e,
	(q31_t)0x7537876c, (q31_t)0xcc940871, (q31_t)0x75350118, (q31_t)0xcc8e4783, (q31_t)0x75327a7d, (q31_t)0xcc8886b5,
	(q31_t)0x752ff399, (q31_t)0xcc82c607,
	(q31_t)0x752d6c6c, (q31_t)0xcc7d0578, (q31_t)0x752ae4f8, (q31_t)0xcc774509, (q31_t)0x75285d3b, (q31_t)0xcc7184ba,
	(q31_t)0x7525d536, (q31_t)0xcc6bc48b,
	(q31_t)0x75234ce8, (q31_t)0xcc66047b, (q31_t)0x7520c453, (q31_t)0xcc60448c, (q31_t)0x751e3b75, (q31_t)0xcc5a84bc,
	(q31_t)0x751bb24f, (q31_t)0xcc54c50c,
	(q31_t)0x751928e0, (q31_t)0xcc4f057c, (q31_t)0x75169f2a, (q31_t)0xcc49460c, (q31_t)0x7514152b, (q31_t)0xcc4386bc,
	(q31_t)0x75118ae4, (q31_t)0xcc3dc78b,
	(q31_t)0x750f0054, (q31_t)0xcc38087b, (q31_t)0x750c757d, (q31_t)0xcc32498a, (q31_t)0x7509ea5d, (q31_t)0xcc2c8aba,
	(q31_t)0x75075ef5, (q31_t)0xcc26cc09,
	(q31_t)0x7504d345, (q31_t)0xcc210d79, (q31_t)0x7502474d, (q31_t)0xcc1b4f08, (q31_t)0x74ffbb0d, (q31_t)0xcc1590b8,
	(q31_t)0x74fd2e84, (q31_t)0xcc0fd287,
	(q31_t)0x74faa1b3, (q31_t)0xcc0a1477, (q31_t)0x74f8149a, (q31_t)0xcc045686, (q31_t)0x74f58739, (q31_t)0xcbfe98b6,
	(q31_t)0x74f2f990, (q31_t)0xcbf8db05,
	(q31_t)0x74f06b9e, (q31_t)0xcbf31d75, (q31_t)0x74eddd65, (q31_t)0xcbed6005, (q31_t)0x74eb4ee3, (q31_t)0xcbe7a2b5,
	(q31_t)0x74e8c01a, (q31_t)0xcbe1e585,
	(q31_t)0x74e63108, (q31_t)0xcbdc2876, (q31_t)0x74e3a1ae, (q31_t)0xcbd66b86, (q31_t)0x74e1120c, (q31_t)0xcbd0aeb7,
	(q31_t)0x74de8221, (q31_t)0xcbcaf208,
	(q31_t)0x74dbf1ef, (q31_t)0xcbc53579, (q31_t)0x74d96175, (q31_t)0xcbbf790a, (q31_t)0x74d6d0b2, (q31_t)0xcbb9bcbb,
	(q31_t)0x74d43fa8, (q31_t)0xcbb4008d,
	(q31_t)0x74d1ae55, (q31_t)0xcbae447f, (q31_t)0x74cf1cbb, (q31_t)0xcba88891, (q31_t)0x74cc8ad8, (q31_t)0xcba2ccc4,
	(q31_t)0x74c9f8ad, (q31_t)0xcb9d1117,
	(q31_t)0x74c7663a, (q31_t)0xcb97558a, (q31_t)0x74c4d380, (q31_t)0xcb919a1d, (q31_t)0x74c2407d, (q31_t)0xcb8bded1,
	(q31_t)0x74bfad32, (q31_t)0xcb8623a5,
	(q31_t)0x74bd199f, (q31_t)0xcb80689a, (q31_t)0x74ba85c4, (q31_t)0xcb7aadaf, (q31_t)0x74b7f1a1, (q31_t)0xcb74f2e4,
	(q31_t)0x74b55d36, (q31_t)0xcb6f383a,
	(q31_t)0x74b2c884, (q31_t)0xcb697db0, (q31_t)0x74b03389, (q31_t)0xcb63c347, (q31_t)0x74ad9e46, (q31_t)0xcb5e08fe,
	(q31_t)0x74ab08bb, (q31_t)0xcb584ed6,
	(q31_t)0x74a872e8, (q31_t)0xcb5294ce, (q31_t)0x74a5dccd, (q31_t)0xcb4cdae6, (q31_t)0x74a3466b, (q31_t)0xcb47211f,
	(q31_t)0x74a0afc0, (q31_t)0xcb416779,
	(q31_t)0x749e18cd, (q31_t)0xcb3badf3, (q31_t)0x749b8193, (q31_t)0xcb35f48d, (q31_t)0x7498ea11, (q31_t)0xcb303b49,
	(q31_t)0x74965246, (q31_t)0xcb2a8224,
	(q31_t)0x7493ba34, (q31_t)0xcb24c921, (q31_t)0x749121da, (q31_t)0xcb1f103e, (q31_t)0x748e8938, (q31_t)0xcb19577b,
	(q31_t)0x748bf04d, (q31_t)0xcb139ed9,
	(q31_t)0x7489571c, (q31_t)0xcb0de658, (q31_t)0x7486bda2, (q31_t)0xcb082df8, (q31_t)0x748423e0, (q31_t)0xcb0275b8,
	(q31_t)0x748189d7, (q31_t)0xcafcbd99,
	(q31_t)0x747eef85, (q31_t)0xcaf7059a, (q31_t)0x747c54ec, (q31_t)0xcaf14dbd, (q31_t)0x7479ba0b, (q31_t)0xcaeb9600,
	(q31_t)0x74771ee2, (q31_t)0xcae5de64,
	(q31_t)0x74748371, (q31_t)0xcae026e8, (q31_t)0x7471e7b8, (q31_t)0xcada6f8d, (q31_t)0x746f4bb8, (q31_t)0xcad4b853,
	(q31_t)0x746caf70, (q31_t)0xcacf013a,
	(q31_t)0x746a12df, (q31_t)0xcac94a42, (q31_t)0x74677608, (q31_t)0xcac3936b, (q31_t)0x7464d8e8, (q31_t)0xcabddcb4,
	(q31_t)0x74623b80, (q31_t)0xcab8261e,
	(q31_t)0x745f9dd1, (q31_t)0xcab26fa9, (q31_t)0x745cffda, (q31_t)0xcaacb955, (q31_t)0x745a619b, (q31_t)0xcaa70322,
	(q31_t)0x7457c314, (q31_t)0xcaa14d10,
	(q31_t)0x74552446, (q31_t)0xca9b971e, (q31_t)0x74528530, (q31_t)0xca95e14e, (q31_t)0x744fe5d2, (q31_t)0xca902b9f,
	(q31_t)0x744d462c, (q31_t)0xca8a7610,
	(q31_t)0x744aa63f, (q31_t)0xca84c0a3, (q31_t)0x7448060a, (q31_t)0xca7f0b56, (q31_t)0x7445658d, (q31_t)0xca79562b,
	(q31_t)0x7442c4c8, (q31_t)0xca73a120,
	(q31_t)0x744023bc, (q31_t)0xca6dec37, (q31_t)0x743d8268, (q31_t)0xca68376e, (q31_t)0x743ae0cc, (q31_t)0xca6282c7,
	(q31_t)0x74383ee9, (q31_t)0xca5cce40,
	(q31_t)0x74359cbd, (q31_t)0xca5719db, (q31_t)0x7432fa4b, (q31_t)0xca516597, (q31_t)0x74305790, (q31_t)0xca4bb174,
	(q31_t)0x742db48e, (q31_t)0xca45fd72,
	(q31_t)0x742b1144, (q31_t)0xca404992, (q31_t)0x74286db3, (q31_t)0xca3a95d2, (q31_t)0x7425c9da, (q31_t)0xca34e234,
	(q31_t)0x742325b9, (q31_t)0xca2f2eb6,
	(q31_t)0x74208150, (q31_t)0xca297b5a, (q31_t)0x741ddca0, (q31_t)0xca23c820, (q31_t)0x741b37a9, (q31_t)0xca1e1506,
	(q31_t)0x74189269, (q31_t)0xca18620e,
	(q31_t)0x7415ece2, (q31_t)0xca12af37, (q31_t)0x74134714, (q31_t)0xca0cfc81, (q31_t)0x7410a0fe, (q31_t)0xca0749ec,
	(q31_t)0x740dfaa0, (q31_t)0xca019779,
	(q31_t)0x740b53fb, (q31_t)0xc9fbe527, (q31_t)0x7408ad0e, (q31_t)0xc9f632f6, (q31_t)0x740605d9, (q31_t)0xc9f080e7,
	(q31_t)0x74035e5d, (q31_t)0xc9eacef9,
	(q31_t)0x7400b69a, (q31_t)0xc9e51d2d, (q31_t)0x73fe0e8f, (q31_t)0xc9df6b81, (q31_t)0x73fb663c, (q31_t)0xc9d9b9f7,
	(q31_t)0x73f8bda2, (q31_t)0xc9d4088f,
	(q31_t)0x73f614c0, (q31_t)0xc9ce5748, (q31_t)0x73f36b97, (q31_t)0xc9c8a622, (q31_t)0x73f0c226, (q31_t)0xc9c2f51e,
	(q31_t)0x73ee186e, (q31_t)0xc9bd443c,
	(q31_t)0x73eb6e6e, (q31_t)0xc9b7937a, (q31_t)0x73e8c426, (q31_t)0xc9b1e2db, (q31_t)0x73e61997, (q31_t)0xc9ac325d,
	(q31_t)0x73e36ec1, (q31_t)0xc9a68200,
	(q31_t)0x73e0c3a3, (q31_t)0xc9a0d1c5, (q31_t)0x73de183e, (q31_t)0xc99b21ab, (q31_t)0x73db6c91, (q31_t)0xc99571b3,
	(q31_t)0x73d8c09d, (q31_t)0xc98fc1dc,
	(q31_t)0x73d61461, (q31_t)0xc98a1227, (q31_t)0x73d367de, (q31_t)0xc9846294, (q31_t)0x73d0bb13, (q31_t)0xc97eb322,
	(q31_t)0x73ce0e01, (q31_t)0xc97903d2,
	(q31_t)0x73cb60a8, (q31_t)0xc97354a4, (q31_t)0x73c8b307, (q31_t)0xc96da597, (q31_t)0x73c6051f, (q31_t)0xc967f6ac,
	(q31_t)0x73c356ef, (q31_t)0xc96247e2,
	(q31_t)0x73c0a878, (q31_t)0xc95c993a, (q31_t)0x73bdf9b9, (q31_t)0xc956eab4, (q31_t)0x73bb4ab3, (q31_t)0xc9513c50,
	(q31_t)0x73b89b66, (q31_t)0xc94b8e0d,
	(q31_t)0x73b5ebd1, (q31_t)0xc945dfec, (q31_t)0x73b33bf5, (q31_t)0xc94031ed, (q31_t)0x73b08bd1, (q31_t)0xc93a8410,
	(q31_t)0x73addb67, (q31_t)0xc934d654,
	(q31_t)0x73ab2ab4, (q31_t)0xc92f28ba, (q31_t)0x73a879bb, (q31_t)0xc9297b42, (q31_t)0x73a5c87a, (q31_t)0xc923cdec,
	(q31_t)0x73a316f2, (q31_t)0xc91e20b8,
	(q31_t)0x73a06522, (q31_t)0xc91873a5, (q31_t)0x739db30b, (q31_t)0xc912c6b5, (q31_t)0x739b00ad, (q31_t)0xc90d19e6,
	(q31_t)0x73984e07, (q31_t)0xc9076d39,
	(q31_t)0x73959b1b, (q31_t)0xc901c0ae, (q31_t)0x7392e7e6, (q31_t)0xc8fc1445, (q31_t)0x7390346b, (q31_t)0xc8f667fe,
	(q31_t)0x738d80a8, (q31_t)0xc8f0bbd9,
	(q31_t)0x738acc9e, (q31_t)0xc8eb0fd6, (q31_t)0x7388184d, (q31_t)0xc8e563f5, (q31_t)0x738563b5, (q31_t)0xc8dfb836,
	(q31_t)0x7382aed5, (q31_t)0xc8da0c99,
	(q31_t)0x737ff9ae, (q31_t)0xc8d4611d, (q31_t)0x737d4440, (q31_t)0xc8ceb5c4, (q31_t)0x737a8e8a, (q31_t)0xc8c90a8d,
	(q31_t)0x7377d88d, (q31_t)0xc8c35f78,
	(q31_t)0x73752249, (q31_t)0xc8bdb485, (q31_t)0x73726bbe, (q31_t)0xc8b809b4, (q31_t)0x736fb4ec, (q31_t)0xc8b25f06,
	(q31_t)0x736cfdd2, (q31_t)0xc8acb479,
	(q31_t)0x736a4671, (q31_t)0xc8a70a0e, (q31_t)0x73678ec9, (q31_t)0xc8a15fc6, (q31_t)0x7364d6da, (q31_t)0xc89bb5a0,
	(q31_t)0x73621ea4, (q31_t)0xc8960b9c,
	(q31_t)0x735f6626, (q31_t)0xc89061ba, (q31_t)0x735cad61, (q31_t)0xc88ab7fa, (q31_t)0x7359f456, (q31_t)0xc8850e5d,
	(q31_t)0x73573b03, (q31_t)0xc87f64e2,
	(q31_t)0x73548168, (q31_t)0xc879bb89, (q31_t)0x7351c787, (q31_t)0xc8741252, (q31_t)0x734f0d5f, (q31_t)0xc86e693d,
	(q31_t)0x734c52ef, (q31_t)0xc868c04b,
	(q31_t)0x73499838, (q31_t)0xc863177b, (q31_t)0x7346dd3a, (q31_t)0xc85d6ece, (q31_t)0x734421f6, (q31_t)0xc857c642,
	(q31_t)0x7341666a, (q31_t)0xc8521dd9,
	(q31_t)0x733eaa96, (q31_t)0xc84c7593, (q31_t)0x733bee7c, (q31_t)0xc846cd6e, (q31_t)0x7339321b, (q31_t)0xc841256d,
	(q31_t)0x73367572, (q31_t)0xc83b7d8d,
	(q31_t)0x7333b883, (q31_t)0xc835d5d0, (q31_t)0x7330fb4d, (q31_t)0xc8302e35, (q31_t)0x732e3dcf, (q31_t)0xc82a86bd,
	(q31_t)0x732b800a, (q31_t)0xc824df67,
	(q31_t)0x7328c1ff, (q31_t)0xc81f3834, (q31_t)0x732603ac, (q31_t)0xc8199123, (q31_t)0x73234512, (q31_t)0xc813ea35,
	(q31_t)0x73208632, (q31_t)0xc80e4369,
	(q31_t)0x731dc70a, (q31_t)0xc8089cbf, (q31_t)0x731b079b, (q31_t)0xc802f638, (q31_t)0x731847e5, (q31_t)0xc7fd4fd4,
	(q31_t)0x731587e8, (q31_t)0xc7f7a992,
	(q31_t)0x7312c7a5, (q31_t)0xc7f20373, (q31_t)0x7310071a, (q31_t)0xc7ec5d76, (q31_t)0x730d4648, (q31_t)0xc7e6b79c,
	(q31_t)0x730a8530, (q31_t)0xc7e111e5,
	(q31_t)0x7307c3d0, (q31_t)0xc7db6c50, (q31_t)0x73050229, (q31_t)0xc7d5c6de, (q31_t)0x7302403c, (q31_t)0xc7d0218e,
	(q31_t)0x72ff7e07, (q31_t)0xc7ca7c61,
	(q31_t)0x72fcbb8c, (q31_t)0xc7c4d757, (q31_t)0x72f9f8c9, (q31_t)0xc7bf3270, (q31_t)0x72f735c0, (q31_t)0xc7b98dab,
	(q31_t)0x72f47270, (q31_t)0xc7b3e909,
	(q31_t)0x72f1aed9, (q31_t)0xc7ae4489, (q31_t)0x72eeeafb, (q31_t)0xc7a8a02c, (q31_t)0x72ec26d6, (q31_t)0xc7a2fbf3,
	(q31_t)0x72e9626a, (q31_t)0xc79d57db,
	(q31_t)0x72e69db7, (q31_t)0xc797b3e7, (q31_t)0x72e3d8be, (q31_t)0xc7921015, (q31_t)0x72e1137d, (q31_t)0xc78c6c67,
	(q31_t)0x72de4df6, (q31_t)0xc786c8db,
	(q31_t)0x72db8828, (q31_t)0xc7812572, (q31_t)0x72d8c213, (q31_t)0xc77b822b, (q31_t)0x72d5fbb7, (q31_t)0xc775df08,
	(q31_t)0x72d33514, (q31_t)0xc7703c08,
	(q31_t)0x72d06e2b, (q31_t)0xc76a992a, (q31_t)0x72cda6fb, (q31_t)0xc764f66f, (q31_t)0x72cadf83, (q31_t)0xc75f53d7,
	(q31_t)0x72c817c6, (q31_t)0xc759b163,
	(q31_t)0x72c54fc1, (q31_t)0xc7540f11, (q31_t)0x72c28775, (q31_t)0xc74e6ce2, (q31_t)0x72bfbee3, (q31_t)0xc748cad6,
	(q31_t)0x72bcf60a, (q31_t)0xc74328ed,
	(q31_t)0x72ba2cea, (q31_t)0xc73d8727, (q31_t)0x72b76383, (q31_t)0xc737e584, (q31_t)0x72b499d6, (q31_t)0xc7324404,
	(q31_t)0x72b1cfe1, (q31_t)0xc72ca2a7,
	(q31_t)0x72af05a7, (q31_t)0xc727016d, (q31_t)0x72ac3b25, (q31_t)0xc7216056, (q31_t)0x72a9705c, (q31_t)0xc71bbf62,
	(q31_t)0x72a6a54d, (q31_t)0xc7161e92,
	(q31_t)0x72a3d9f7, (q31_t)0xc7107de4, (q31_t)0x72a10e5b, (q31_t)0xc70add5a, (q31_t)0x729e4277, (q31_t)0xc7053cf2,
	(q31_t)0x729b764d, (q31_t)0xc6ff9cae,
	(q31_t)0x7298a9dd, (q31_t)0xc6f9fc8d, (q31_t)0x7295dd25, (q31_t)0xc6f45c8f, (q31_t)0x72931027, (q31_t)0xc6eebcb5,
	(q31_t)0x729042e3, (q31_t)0xc6e91cfd,
	(q31_t)0x728d7557, (q31_t)0xc6e37d69, (q31_t)0x728aa785, (q31_t)0xc6ddddf8, (q31_t)0x7287d96c, (q31_t)0xc6d83eab,
	(q31_t)0x72850b0d, (q31_t)0xc6d29f80,
	(q31_t)0x72823c67, (q31_t)0xc6cd0079, (q31_t)0x727f6d7a, (q31_t)0xc6c76195, (q31_t)0x727c9e47, (q31_t)0xc6c1c2d4,
	(q31_t)0x7279cecd, (q31_t)0xc6bc2437,
	(q31_t)0x7276ff0d, (q31_t)0xc6b685bd, (q31_t)0x72742f05, (q31_t)0xc6b0e767, (q31_t)0x72715eb8, (q31_t)0xc6ab4933,
	(q31_t)0x726e8e23, (q31_t)0xc6a5ab23,
	(q31_t)0x726bbd48, (q31_t)0xc6a00d37, (q31_t)0x7268ec27, (q31_t)0xc69a6f6e, (q31_t)0x72661abf, (q31_t)0xc694d1c8,
	(q31_t)0x72634910, (q31_t)0xc68f3446,
	(q31_t)0x7260771b, (q31_t)0xc68996e7, (q31_t)0x725da4df, (q31_t)0xc683f9ab, (q31_t)0x725ad25d, (q31_t)0xc67e5c93,
	(q31_t)0x7257ff94, (q31_t)0xc678bf9f,
	(q31_t)0x72552c85, (q31_t)0xc67322ce, (q31_t)0x7252592f, (q31_t)0xc66d8620, (q31_t)0x724f8593, (q31_t)0xc667e996,
	(q31_t)0x724cb1b0, (q31_t)0xc6624d30,
	(q31_t)0x7249dd86, (q31_t)0xc65cb0ed, (q31_t)0x72470916, (q31_t)0xc65714cd, (q31_t)0x72443460, (q31_t)0xc65178d1,
	(q31_t)0x72415f63, (q31_t)0xc64bdcf9,
	(q31_t)0x723e8a20, (q31_t)0xc6464144, (q31_t)0x723bb496, (q31_t)0xc640a5b3, (q31_t)0x7238dec5, (q31_t)0xc63b0a46,
	(q31_t)0x723608af, (q31_t)0xc6356efc,
	(q31_t)0x72333251, (q31_t)0xc62fd3d6, (q31_t)0x72305bae, (q31_t)0xc62a38d4, (q31_t)0x722d84c4, (q31_t)0xc6249df5,
	(q31_t)0x722aad93, (q31_t)0xc61f033a,
	(q31_t)0x7227d61c, (q31_t)0xc61968a2, (q31_t)0x7224fe5f, (q31_t)0xc613ce2f, (q31_t)0x7222265b, (q31_t)0xc60e33df,
	(q31_t)0x721f4e11, (q31_t)0xc60899b2,
	(q31_t)0x721c7580, (q31_t)0xc602ffaa, (q31_t)0x72199ca9, (q31_t)0xc5fd65c5, (q31_t)0x7216c38c, (q31_t)0xc5f7cc04,
	(q31_t)0x7213ea28, (q31_t)0xc5f23267,
	(q31_t)0x7211107e, (q31_t)0xc5ec98ee, (q31_t)0x720e368d, (q31_t)0xc5e6ff98, (q31_t)0x720b5c57, (q31_t)0xc5e16667,
	(q31_t)0x720881d9, (q31_t)0xc5dbcd59,
	(q31_t)0x7205a716, (q31_t)0xc5d6346f, (q31_t)0x7202cc0c, (q31_t)0xc5d09ba9, (q31_t)0x71fff0bc, (q31_t)0xc5cb0307,
	(q31_t)0x71fd1525, (q31_t)0xc5c56a89,
	(q31_t)0x71fa3949, (q31_t)0xc5bfd22e, (q31_t)0x71f75d25, (q31_t)0xc5ba39f8, (q31_t)0x71f480bc, (q31_t)0xc5b4a1e5,
	(q31_t)0x71f1a40c, (q31_t)0xc5af09f7,
	(q31_t)0x71eec716, (q31_t)0xc5a9722c, (q31_t)0x71ebe9da, (q31_t)0xc5a3da86, (q31_t)0x71e90c57, (q31_t)0xc59e4303,
	(q31_t)0x71e62e8f, (q31_t)0xc598aba5,
	(q31_t)0x71e35080, (q31_t)0xc593146a, (q31_t)0x71e0722a, (q31_t)0xc58d7d54, (q31_t)0x71dd938f, (q31_t)0xc587e661,
	(q31_t)0x71dab4ad, (q31_t)0xc5824f93,
	(q31_t)0x71d7d585, (q31_t)0xc57cb8e9, (q31_t)0x71d4f617, (q31_t)0xc5772263, (q31_t)0x71d21662, (q31_t)0xc5718c00,
	(q31_t)0x71cf3667, (q31_t)0xc56bf5c2,
	(q31_t)0x71cc5626, (q31_t)0xc5665fa9, (q31_t)0x71c9759f, (q31_t)0xc560c9b3, (q31_t)0x71c694d2, (q31_t)0xc55b33e2,
	(q31_t)0x71c3b3bf, (q31_t)0xc5559e34,
	(q31_t)0x71c0d265, (q31_t)0xc55008ab, (q31_t)0x71bdf0c5, (q31_t)0xc54a7346, (q31_t)0x71bb0edf, (q31_t)0xc544de05,
	(q31_t)0x71b82cb3, (q31_t)0xc53f48e9,
	(q31_t)0x71b54a41, (q31_t)0xc539b3f1, (q31_t)0x71b26788, (q31_t)0xc5341f1d, (q31_t)0x71af848a, (q31_t)0xc52e8a6d,
	(q31_t)0x71aca145, (q31_t)0xc528f5e1,
	(q31_t)0x71a9bdba, (q31_t)0xc523617a, (q31_t)0x71a6d9e9, (q31_t)0xc51dcd37, (q31_t)0x71a3f5d2, (q31_t)0xc5183919,
	(q31_t)0x71a11175, (q31_t)0xc512a51f,
	(q31_t)0x719e2cd2, (q31_t)0xc50d1149, (q31_t)0x719b47e9, (q31_t)0xc5077d97, (q31_t)0x719862b9, (q31_t)0xc501ea0a,
	(q31_t)0x71957d44, (q31_t)0xc4fc56a2,
	(q31_t)0x71929789, (q31_t)0xc4f6c35d, (q31_t)0x718fb187, (q31_t)0xc4f1303d, (q31_t)0x718ccb3f, (q31_t)0xc4eb9d42,
	(q31_t)0x7189e4b2, (q31_t)0xc4e60a6b,
	(q31_t)0x7186fdde, (q31_t)0xc4e077b8, (q31_t)0x718416c4, (q31_t)0xc4dae52a, (q31_t)0x71812f65, (q31_t)0xc4d552c1,
	(q31_t)0x717e47bf, (q31_t)0xc4cfc07c,
	(q31_t)0x717b5fd3, (q31_t)0xc4ca2e5b, (q31_t)0x717877a1, (q31_t)0xc4c49c5f, (q31_t)0x71758f29, (q31_t)0xc4bf0a87,
	(q31_t)0x7172a66c, (q31_t)0xc4b978d4,
	(q31_t)0x716fbd68, (q31_t)0xc4b3e746, (q31_t)0x716cd41e, (q31_t)0xc4ae55dc, (q31_t)0x7169ea8f, (q31_t)0xc4a8c497,
	(q31_t)0x716700b9, (q31_t)0xc4a33376,
	(q31_t)0x7164169d, (q31_t)0xc49da27a, (q31_t)0x71612c3c, (q31_t)0xc49811a3, (q31_t)0x715e4194, (q31_t)0xc49280f0,
	(q31_t)0x715b56a7, (q31_t)0xc48cf062,
	(q31_t)0x71586b74, (q31_t)0xc4875ff9, (q31_t)0x71557ffa, (q31_t)0xc481cfb4, (q31_t)0x7152943b, (q31_t)0xc47c3f94,
	(q31_t)0x714fa836, (q31_t)0xc476af98,
	(q31_t)0x714cbbeb, (q31_t)0xc4711fc2, (q31_t)0x7149cf5a, (q31_t)0xc46b9010, (q31_t)0x7146e284, (q31_t)0xc4660083,
	(q31_t)0x7143f567, (q31_t)0xc460711b,
	(q31_t)0x71410805, (q31_t)0xc45ae1d7, (q31_t)0x713e1a5c, (q31_t)0xc45552b8, (q31_t)0x713b2c6e, (q31_t)0xc44fc3be,
	(q31_t)0x71383e3a, (q31_t)0xc44a34e9,
	(q31_t)0x71354fc0, (q31_t)0xc444a639, (q31_t)0x71326101, (q31_t)0xc43f17ad, (q31_t)0x712f71fb, (q31_t)0xc4398947,
	(q31_t)0x712c82b0, (q31_t)0xc433fb05,
	(q31_t)0x7129931f, (q31_t)0xc42e6ce8, (q31_t)0x7126a348, (q31_t)0xc428def0, (q31_t)0x7123b32b, (q31_t)0xc423511d,
	(q31_t)0x7120c2c8, (q31_t)0xc41dc36f,
	(q31_t)0x711dd220, (q31_t)0xc41835e6, (q31_t)0x711ae132, (q31_t)0xc412a882, (q31_t)0x7117effe, (q31_t)0xc40d1b42,
	(q31_t)0x7114fe84, (q31_t)0xc4078e28,
	(q31_t)0x71120cc5, (q31_t)0xc4020133, (q31_t)0x710f1ac0, (q31_t)0xc3fc7462, (q31_t)0x710c2875, (q31_t)0xc3f6e7b7,
	(q31_t)0x710935e4, (q31_t)0xc3f15b31,
	(q31_t)0x7106430e, (q31_t)0xc3ebced0, (q31_t)0x71034ff2, (q31_t)0xc3e64294, (q31_t)0x71005c90, (q31_t)0xc3e0b67d,
	(q31_t)0x70fd68e9, (q31_t)0xc3db2a8b,
	(q31_t)0x70fa74fc, (q31_t)0xc3d59ebe, (q31_t)0x70f780c9, (q31_t)0xc3d01316, (q31_t)0x70f48c50, (q31_t)0xc3ca8793,
	(q31_t)0x70f19792, (q31_t)0xc3c4fc36,
	(q31_t)0x70eea28e, (q31_t)0xc3bf70fd, (q31_t)0x70ebad45, (q31_t)0xc3b9e5ea, (q31_t)0x70e8b7b5, (q31_t)0xc3b45afc,
	(q31_t)0x70e5c1e1, (q31_t)0xc3aed034,
	(q31_t)0x70e2cbc6, (q31_t)0xc3a94590, (q31_t)0x70dfd566, (q31_t)0xc3a3bb12, (q31_t)0x70dcdec0, (q31_t)0xc39e30b8,
	(q31_t)0x70d9e7d5, (q31_t)0xc398a685,
	(q31_t)0x70d6f0a4, (q31_t)0xc3931c76, (q31_t)0x70d3f92d, (q31_t)0xc38d928d, (q31_t)0x70d10171, (q31_t)0xc38808c9,
	(q31_t)0x70ce096f, (q31_t)0xc3827f2a,
	(q31_t)0x70cb1128, (q31_t)0xc37cf5b0, (q31_t)0x70c8189b, (q31_t)0xc3776c5c, (q31_t)0x70c51fc8, (q31_t)0xc371e32d,
	(q31_t)0x70c226b0, (q31_t)0xc36c5a24,
	(q31_t)0x70bf2d53, (q31_t)0xc366d140, (q31_t)0x70bc33b0, (q31_t)0xc3614881, (q31_t)0x70b939c7, (q31_t)0xc35bbfe8,
	(q31_t)0x70b63f99, (q31_t)0xc3563774,
	(q31_t)0x70b34525, (q31_t)0xc350af26, (q31_t)0x70b04a6b, (q31_t)0xc34b26fc, (q31_t)0x70ad4f6d, (q31_t)0xc3459ef9,
	(q31_t)0x70aa5428, (q31_t)0xc340171b,
	(q31_t)0x70a7589f, (q31_t)0xc33a8f62, (q31_t)0x70a45ccf, (q31_t)0xc33507cf, (q31_t)0x70a160ba, (q31_t)0xc32f8061,
	(q31_t)0x709e6460, (q31_t)0xc329f919,
	(q31_t)0x709b67c0, (q31_t)0xc32471f7, (q31_t)0x70986adb, (q31_t)0xc31eeaf9, (q31_t)0x70956db1, (q31_t)0xc3196422,
	(q31_t)0x70927041, (q31_t)0xc313dd70,
	(q31_t)0x708f728b, (q31_t)0xc30e56e4, (q31_t)0x708c7490, (q31_t)0xc308d07d, (q31_t)0x70897650, (q31_t)0xc3034a3c,
	(q31_t)0x708677ca, (q31_t)0xc2fdc420,
	(q31_t)0x708378ff, (q31_t)0xc2f83e2a, (q31_t)0x708079ee, (q31_t)0xc2f2b85a, (q31_t)0x707d7a98, (q31_t)0xc2ed32af,
	(q31_t)0x707a7afd, (q31_t)0xc2e7ad2a,
	(q31_t)0x70777b1c, (q31_t)0xc2e227cb, (q31_t)0x70747af6, (q31_t)0xc2dca291, (q31_t)0x70717a8a, (q31_t)0xc2d71d7e,
	(q31_t)0x706e79d9, (q31_t)0xc2d1988f,
	(q31_t)0x706b78e3, (q31_t)0xc2cc13c7, (q31_t)0x706877a7, (q31_t)0xc2c68f24, (q31_t)0x70657626, (q31_t)0xc2c10aa7,
	(q31_t)0x70627460, (q31_t)0xc2bb8650,
	(q31_t)0x705f7255, (q31_t)0xc2b6021f, (q31_t)0x705c7004, (q31_t)0xc2b07e14, (q31_t)0x70596d6d, (q31_t)0xc2aafa2e,
	(q31_t)0x70566a92, (q31_t)0xc2a5766e,
	(q31_t)0x70536771, (q31_t)0xc29ff2d4, (q31_t)0x7050640b, (q31_t)0xc29a6f60, (q31_t)0x704d6060, (q31_t)0xc294ec12,
	(q31_t)0x704a5c6f, (q31_t)0xc28f68e9,
	(q31_t)0x70475839, (q31_t)0xc289e5e7, (q31_t)0x704453be, (q31_t)0xc284630a, (q31_t)0x70414efd, (q31_t)0xc27ee054,
	(q31_t)0x703e49f8, (q31_t)0xc2795dc3,
	(q31_t)0x703b44ad, (q31_t)0xc273db58, (q31_t)0x70383f1d, (q31_t)0xc26e5913, (q31_t)0x70353947, (q31_t)0xc268d6f5,
	(q31_t)0x7032332d, (q31_t)0xc26354fc,
	(q31_t)0x702f2ccd, (q31_t)0xc25dd329, (q31_t)0x702c2628, (q31_t)0xc258517c, (q31_t)0x70291f3e, (q31_t)0xc252cff5,
	(q31_t)0x7026180e, (q31_t)0xc24d4e95,
	(q31_t)0x7023109a, (q31_t)0xc247cd5a, (q31_t)0x702008e0, (q31_t)0xc2424c46, (q31_t)0x701d00e1, (q31_t)0xc23ccb57,
	(q31_t)0x7019f89d, (q31_t)0xc2374a8f,
	(q31_t)0x7016f014, (q31_t)0xc231c9ec, (q31_t)0x7013e746, (q31_t)0xc22c4970, (q31_t)0x7010de32, (q31_t)0xc226c91a,
	(q31_t)0x700dd4da, (q31_t)0xc22148ea,
	(q31_t)0x700acb3c, (q31_t)0xc21bc8e1, (q31_t)0x7007c159, (q31_t)0xc21648fd, (q31_t)0x7004b731, (q31_t)0xc210c940,
	(q31_t)0x7001acc4, (q31_t)0xc20b49a9,
	(q31_t)0x6ffea212, (q31_t)0xc205ca38, (q31_t)0x6ffb971b, (q31_t)0xc2004aed, (q31_t)0x6ff88bde, (q31_t)0xc1facbc9,
	(q31_t)0x6ff5805d, (q31_t)0xc1f54cca,
	(q31_t)0x6ff27497, (q31_t)0xc1efcdf3, (q31_t)0x6fef688b, (q31_t)0xc1ea4f41, (q31_t)0x6fec5c3b, (q31_t)0xc1e4d0b6,
	(q31_t)0x6fe94fa5, (q31_t)0xc1df5251,
	(q31_t)0x6fe642ca, (q31_t)0xc1d9d412, (q31_t)0x6fe335ab, (q31_t)0xc1d455f9, (q31_t)0x6fe02846, (q31_t)0xc1ced807,
	(q31_t)0x6fdd1a9c, (q31_t)0xc1c95a3c,
	(q31_t)0x6fda0cae, (q31_t)0xc1c3dc97, (q31_t)0x6fd6fe7a, (q31_t)0xc1be5f18, (q31_t)0x6fd3f001, (q31_t)0xc1b8e1bf,
	(q31_t)0x6fd0e144, (q31_t)0xc1b3648d,
	(q31_t)0x6fcdd241, (q31_t)0xc1ade781, (q31_t)0x6fcac2fa, (q31_t)0xc1a86a9c, (q31_t)0x6fc7b36d, (q31_t)0xc1a2edde,
	(q31_t)0x6fc4a39c, (q31_t)0xc19d7145,
	(q31_t)0x6fc19385, (q31_t)0xc197f4d4, (q31_t)0x6fbe832a, (q31_t)0xc1927888, (q31_t)0x6fbb728a, (q31_t)0xc18cfc63,
	(q31_t)0x6fb861a4, (q31_t)0xc1878065,
	(q31_t)0x6fb5507a, (q31_t)0xc182048d, (q31_t)0x6fb23f0b, (q31_t)0xc17c88dc, (q31_t)0x6faf2d57, (q31_t)0xc1770d52,
	(q31_t)0x6fac1b5f, (q31_t)0xc17191ee,
	(q31_t)0x6fa90921, (q31_t)0xc16c16b0, (q31_t)0x6fa5f69e, (q31_t)0xc1669b99, (q31_t)0x6fa2e3d7, (q31_t)0xc16120a9,
	(q31_t)0x6f9fd0cb, (q31_t)0xc15ba5df,
	(q31_t)0x6f9cbd79, (q31_t)0xc1562b3d, (q31_t)0x6f99a9e3, (q31_t)0xc150b0c0, (q31_t)0x6f969608, (q31_t)0xc14b366b,
	(q31_t)0x6f9381e9, (q31_t)0xc145bc3c,
	(q31_t)0x6f906d84, (q31_t)0xc1404233, (q31_t)0x6f8d58db, (q31_t)0xc13ac852, (q31_t)0x6f8a43ed, (q31_t)0xc1354e97,
	(q31_t)0x6f872eba, (q31_t)0xc12fd503,
	(q31_t)0x6f841942, (q31_t)0xc12a5b95, (q31_t)0x6f810386, (q31_t)0xc124e24f, (q31_t)0x6f7ded84, (q31_t)0xc11f692f,
	(q31_t)0x6f7ad73e, (q31_t)0xc119f036,
	(q31_t)0x6f77c0b3, (q31_t)0xc1147764, (q31_t)0x6f74a9e4, (q31_t)0xc10efeb8, (q31_t)0x6f7192cf, (q31_t)0xc1098634,
	(q31_t)0x6f6e7b76, (q31_t)0xc1040dd6,
	(q31_t)0x6f6b63d8, (q31_t)0xc0fe959f, (q31_t)0x6f684bf6, (q31_t)0xc0f91d8f, (q31_t)0x6f6533ce, (q31_t)0xc0f3a5a6,
	(q31_t)0x6f621b62, (q31_t)0xc0ee2de3,
	(q31_t)0x6f5f02b2, (q31_t)0xc0e8b648, (q31_t)0x6f5be9bc, (q31_t)0xc0e33ed4, (q31_t)0x6f58d082, (q31_t)0xc0ddc786,
	(q31_t)0x6f55b703, (q31_t)0xc0d8505f,
	(q31_t)0x6f529d40, (q31_t)0xc0d2d960, (q31_t)0x6f4f8338, (q31_t)0xc0cd6287, (q31_t)0x6f4c68eb, (q31_t)0xc0c7ebd6,
	(q31_t)0x6f494e5a, (q31_t)0xc0c2754b,
	(q31_t)0x6f463383, (q31_t)0xc0bcfee7, (q31_t)0x6f431869, (q31_t)0xc0b788ab, (q31_t)0x6f3ffd09, (q31_t)0xc0b21295,
	(q31_t)0x6f3ce165, (q31_t)0xc0ac9ca6,
	(q31_t)0x6f39c57d, (q31_t)0xc0a726df, (q31_t)0x6f36a94f, (q31_t)0xc0a1b13e, (q31_t)0x6f338cde, (q31_t)0xc09c3bc5,
	(q31_t)0x6f307027, (q31_t)0xc096c673,
	(q31_t)0x6f2d532c, (q31_t)0xc0915148, (q31_t)0x6f2a35ed, (q31_t)0xc08bdc44, (q31_t)0x6f271868, (q31_t)0xc0866767,
	(q31_t)0x6f23faa0, (q31_t)0xc080f2b1,
	(q31_t)0x6f20dc92, (q31_t)0xc07b7e23, (q31_t)0x6f1dbe41, (q31_t)0xc07609bb, (q31_t)0x6f1a9faa, (q31_t)0xc070957b,
	(q31_t)0x6f1780cf, (q31_t)0xc06b2162,
	(q31_t)0x6f1461b0, (q31_t)0xc065ad70, (q31_t)0x6f11424c, (q31_t)0xc06039a6, (q31_t)0x6f0e22a3, (q31_t)0xc05ac603,
	(q31_t)0x6f0b02b6, (q31_t)0xc0555287,
	(q31_t)0x6f07e285, (q31_t)0xc04fdf32, (q31_t)0x6f04c20f, (q31_t)0xc04a6c05, (q31_t)0x6f01a155, (q31_t)0xc044f8fe,
	(q31_t)0x6efe8056, (q31_t)0xc03f8620,
	(q31_t)0x6efb5f12, (q31_t)0xc03a1368, (q31_t)0x6ef83d8a, (q31_t)0xc034a0d8, (q31_t)0x6ef51bbe, (q31_t)0xc02f2e6f,
	(q31_t)0x6ef1f9ad, (q31_t)0xc029bc2e,
	(q31_t)0x6eeed758, (q31_t)0xc0244a14, (q31_t)0x6eebb4bf, (q31_t)0xc01ed821, (q31_t)0x6ee891e1, (q31_t)0xc0196656,
	(q31_t)0x6ee56ebe, (q31_t)0xc013f4b2,
	(q31_t)0x6ee24b57, (q31_t)0xc00e8336, (q31_t)0x6edf27ac, (q31_t)0xc00911e1, (q31_t)0x6edc03bc, (q31_t)0xc003a0b3,
	(q31_t)0x6ed8df88, (q31_t)0xbffe2fad,
	(q31_t)0x6ed5bb10, (q31_t)0xbff8bece, (q31_t)0x6ed29653, (q31_t)0xbff34e17, (q31_t)0x6ecf7152, (q31_t)0xbfeddd88,
	(q31_t)0x6ecc4c0d, (q31_t)0xbfe86d20,
	(q31_t)0x6ec92683, (q31_t)0xbfe2fcdf, (q31_t)0x6ec600b5, (q31_t)0xbfdd8cc6, (q31_t)0x6ec2daa2, (q31_t)0xbfd81cd5,
	(q31_t)0x6ebfb44b, (q31_t)0xbfd2ad0b,
	(q31_t)0x6ebc8db0, (q31_t)0xbfcd3d69, (q31_t)0x6eb966d1, (q31_t)0xbfc7cdee, (q31_t)0x6eb63fad, (q31_t)0xbfc25e9b,
	(q31_t)0x6eb31845, (q31_t)0xbfbcef70,
	(q31_t)0x6eaff099, (q31_t)0xbfb7806c, (q31_t)0x6eacc8a8, (q31_t)0xbfb21190, (q31_t)0x6ea9a073, (q31_t)0xbfaca2dc,
	(q31_t)0x6ea677fa, (q31_t)0xbfa7344f,
	(q31_t)0x6ea34f3d, (q31_t)0xbfa1c5ea, (q31_t)0x6ea0263b, (q31_t)0xbf9c57ac, (q31_t)0x6e9cfcf5, (q31_t)0xbf96e997,
	(q31_t)0x6e99d36b, (q31_t)0xbf917ba9,
	(q31_t)0x6e96a99d, (q31_t)0xbf8c0de3, (q31_t)0x6e937f8a, (q31_t)0xbf86a044, (q31_t)0x6e905534, (q31_t)0xbf8132ce,
	(q31_t)0x6e8d2a99, (q31_t)0xbf7bc57f,
	(q31_t)0x6e89ffb9, (q31_t)0xbf765858, (q31_t)0x6e86d496, (q31_t)0xbf70eb59, (q31_t)0x6e83a92f, (q31_t)0xbf6b7e81,
	(q31_t)0x6e807d83, (q31_t)0xbf6611d2,
	(q31_t)0x6e7d5193, (q31_t)0xbf60a54a, (q31_t)0x6e7a255f, (q31_t)0xbf5b38ea, (q31_t)0x6e76f8e7, (q31_t)0xbf55ccb2,
	(q31_t)0x6e73cc2b, (q31_t)0xbf5060a2,
	(q31_t)0x6e709f2a, (q31_t)0xbf4af4ba, (q31_t)0x6e6d71e6, (q31_t)0xbf4588fa, (q31_t)0x6e6a445d, (q31_t)0xbf401d61,
	(q31_t)0x6e671690, (q31_t)0xbf3ab1f1,
	(q31_t)0x6e63e87f, (q31_t)0xbf3546a8, (q31_t)0x6e60ba2a, (q31_t)0xbf2fdb88, (q31_t)0x6e5d8b91, (q31_t)0xbf2a708f,
	(q31_t)0x6e5a5cb4, (q31_t)0xbf2505bf,
	(q31_t)0x6e572d93, (q31_t)0xbf1f9b16, (q31_t)0x6e53fe2e, (q31_t)0xbf1a3096, (q31_t)0x6e50ce84, (q31_t)0xbf14c63d,
	(q31_t)0x6e4d9e97, (q31_t)0xbf0f5c0d,
	(q31_t)0x6e4a6e66, (q31_t)0xbf09f205, (q31_t)0x6e473df0, (q31_t)0xbf048824, (q31_t)0x6e440d37, (q31_t)0xbeff1e6c,
	(q31_t)0x6e40dc39, (q31_t)0xbef9b4dc,
	(q31_t)0x6e3daaf8, (q31_t)0xbef44b74, (q31_t)0x6e3a7972, (q31_t)0xbeeee234, (q31_t)0x6e3747a9, (q31_t)0xbee9791c,
	(q31_t)0x6e34159b, (q31_t)0xbee4102d,
	(q31_t)0x6e30e34a, (q31_t)0xbedea765, (q31_t)0x6e2db0b4, (q31_t)0xbed93ec6, (q31_t)0x6e2a7ddb, (q31_t)0xbed3d64f,
	(q31_t)0x6e274abe, (q31_t)0xbece6e00,
	(q31_t)0x6e24175c, (q31_t)0xbec905d9, (q31_t)0x6e20e3b7, (q31_t)0xbec39ddb, (q31_t)0x6e1dafce, (q31_t)0xbebe3605,
	(q31_t)0x6e1a7ba1, (q31_t)0xbeb8ce57,
	(q31_t)0x6e174730, (q31_t)0xbeb366d1, (q31_t)0x6e14127b, (q31_t)0xbeadff74, (q31_t)0x6e10dd82, (q31_t)0xbea8983f,
	(q31_t)0x6e0da845, (q31_t)0xbea33132,
	(q31_t)0x6e0a72c5, (q31_t)0xbe9dca4e, (q31_t)0x6e073d00, (q31_t)0xbe986391, (q31_t)0x6e0406f8, (q31_t)0xbe92fcfe,
	(q31_t)0x6e00d0ac, (q31_t)0xbe8d9692,
	(q31_t)0x6dfd9a1c, (q31_t)0xbe88304f, (q31_t)0x6dfa6348, (q31_t)0xbe82ca35, (q31_t)0x6df72c30, (q31_t)0xbe7d6442,
	(q31_t)0x6df3f4d4, (q31_t)0xbe77fe78,
	(q31_t)0x6df0bd35, (q31_t)0xbe7298d7, (q31_t)0x6ded8552, (q31_t)0xbe6d335e, (q31_t)0x6dea4d2b, (q31_t)0xbe67ce0d,
	(q31_t)0x6de714c0, (q31_t)0xbe6268e5,
	(q31_t)0x6de3dc11, (q31_t)0xbe5d03e6, (q31_t)0x6de0a31f, (q31_t)0xbe579f0f, (q31_t)0x6ddd69e9, (q31_t)0xbe523a60,
	(q31_t)0x6dda306f, (q31_t)0xbe4cd5da,
	(q31_t)0x6dd6f6b1, (q31_t)0xbe47717c, (q31_t)0x6dd3bcaf, (q31_t)0xbe420d47, (q31_t)0x6dd0826a, (q31_t)0xbe3ca93b,
	(q31_t)0x6dcd47e1, (q31_t)0xbe374557,
	(q31_t)0x6dca0d14, (q31_t)0xbe31e19b, (q31_t)0x6dc6d204, (q31_t)0xbe2c7e09, (q31_t)0x6dc396b0, (q31_t)0xbe271a9f,
	(q31_t)0x6dc05b18, (q31_t)0xbe21b75d,
	(q31_t)0x6dbd1f3c, (q31_t)0xbe1c5444, (q31_t)0x6db9e31d, (q31_t)0xbe16f154, (q31_t)0x6db6a6ba, (q31_t)0xbe118e8c,
	(q31_t)0x6db36a14, (q31_t)0xbe0c2bed,
	(q31_t)0x6db02d29, (q31_t)0xbe06c977, (q31_t)0x6daceffb, (q31_t)0xbe01672a, (q31_t)0x6da9b28a, (q31_t)0xbdfc0505,
	(q31_t)0x6da674d5, (q31_t)0xbdf6a309,
	(q31_t)0x6da336dc, (q31_t)0xbdf14135, (q31_t)0x6d9ff89f, (q31_t)0xbdebdf8b, (q31_t)0x6d9cba1f, (q31_t)0xbde67e09,
	(q31_t)0x6d997b5b, (q31_t)0xbde11cb0,
	(q31_t)0x6d963c54, (q31_t)0xbddbbb7f, (q31_t)0x6d92fd09, (q31_t)0xbdd65a78, (q31_t)0x6d8fbd7a, (q31_t)0xbdd0f999,
	(q31_t)0x6d8c7da8, (q31_t)0xbdcb98e3,
	(q31_t)0x6d893d93, (q31_t)0xbdc63856, (q31_t)0x6d85fd39, (q31_t)0xbdc0d7f2, (q31_t)0x6d82bc9d, (q31_t)0xbdbb77b7,
	(q31_t)0x6d7f7bbc, (q31_t)0xbdb617a4,
	(q31_t)0x6d7c3a98, (q31_t)0xbdb0b7bb, (q31_t)0x6d78f931, (q31_t)0xbdab57fa, (q31_t)0x6d75b786, (q31_t)0xbda5f862,
	(q31_t)0x6d727597, (q31_t)0xbda098f3,
	(q31_t)0x6d6f3365, (q31_t)0xbd9b39ad, (q31_t)0x6d6bf0f0, (q31_t)0xbd95da91, (q31_t)0x6d68ae37, (q31_t)0xbd907b9d,
	(q31_t)0x6d656b3a, (q31_t)0xbd8b1cd2,
	(q31_t)0x6d6227fa, (q31_t)0xbd85be30, (q31_t)0x6d5ee477, (q31_t)0xbd805fb7, (q31_t)0x6d5ba0b0, (q31_t)0xbd7b0167,
	(q31_t)0x6d585ca6, (q31_t)0xbd75a340,
	(q31_t)0x6d551858, (q31_t)0xbd704542, (q31_t)0x6d51d3c6, (q31_t)0xbd6ae76d, (q31_t)0x6d4e8ef2, (q31_t)0xbd6589c1,
	(q31_t)0x6d4b49da, (q31_t)0xbd602c3f,
	(q31_t)0x6d48047e, (q31_t)0xbd5acee5, (q31_t)0x6d44bedf, (q31_t)0xbd5571b5, (q31_t)0x6d4178fd, (q31_t)0xbd5014ad,
	(q31_t)0x6d3e32d7, (q31_t)0xbd4ab7cf,
	(q31_t)0x6d3aec6e, (q31_t)0xbd455b1a, (q31_t)0x6d37a5c1, (q31_t)0xbd3ffe8e, (q31_t)0x6d345ed1, (q31_t)0xbd3aa22c,
	(q31_t)0x6d31179e, (q31_t)0xbd3545f2,
	(q31_t)0x6d2dd027, (q31_t)0xbd2fe9e2, (q31_t)0x6d2a886e, (q31_t)0xbd2a8dfb, (q31_t)0x6d274070, (q31_t)0xbd25323d,
	(q31_t)0x6d23f830, (q31_t)0xbd1fd6a8,
	(q31_t)0x6d20afac, (q31_t)0xbd1a7b3d, (q31_t)0x6d1d66e4, (q31_t)0xbd151ffb, (q31_t)0x6d1a1dda, (q31_t)0xbd0fc4e2,
	(q31_t)0x6d16d48c, (q31_t)0xbd0a69f2,
	(q31_t)0x6d138afb, (q31_t)0xbd050f2c, (q31_t)0x6d104126, (q31_t)0xbcffb48f, (q31_t)0x6d0cf70f, (q31_t)0xbcfa5a1b,
	(q31_t)0x6d09acb4, (q31_t)0xbcf4ffd1,
	(q31_t)0x6d066215, (q31_t)0xbcefa5b0, (q31_t)0x6d031734, (q31_t)0xbcea4bb9, (q31_t)0x6cffcc0f, (q31_t)0xbce4f1eb,
	(q31_t)0x6cfc80a7, (q31_t)0xbcdf9846,
	(q31_t)0x6cf934fc, (q31_t)0xbcda3ecb, (q31_t)0x6cf5e90d, (q31_t)0xbcd4e579, (q31_t)0x6cf29cdc, (q31_t)0xbccf8c50,
	(q31_t)0x6cef5067, (q31_t)0xbcca3351,
	(q31_t)0x6cec03af, (q31_t)0xbcc4da7b, (q31_t)0x6ce8b6b4, (q31_t)0xbcbf81cf, (q31_t)0x6ce56975, (q31_t)0xbcba294d,
	(q31_t)0x6ce21bf4, (q31_t)0xbcb4d0f4,
	(q31_t)0x6cdece2f, (q31_t)0xbcaf78c4, (q31_t)0x6cdb8027, (q31_t)0xbcaa20be, (q31_t)0x6cd831dc, (q31_t)0xbca4c8e1,
	(q31_t)0x6cd4e34e, (q31_t)0xbc9f712e,
	(q31_t)0x6cd1947c, (q31_t)0xbc9a19a5, (q31_t)0x6cce4568, (q31_t)0xbc94c245, (q31_t)0x6ccaf610, (q31_t)0xbc8f6b0f,
	(q31_t)0x6cc7a676, (q31_t)0xbc8a1402,
	(q31_t)0x6cc45698, (q31_t)0xbc84bd1f, (q31_t)0x6cc10677, (q31_t)0xbc7f6665, (q31_t)0x6cbdb613, (q31_t)0xbc7a0fd6,
	(q31_t)0x6cba656c, (q31_t)0xbc74b96f,
	(q31_t)0x6cb71482, (q31_t)0xbc6f6333, (q31_t)0x6cb3c355, (q31_t)0xbc6a0d20, (q31_t)0x6cb071e4, (q31_t)0xbc64b737,
	(q31_t)0x6cad2031, (q31_t)0xbc5f6177,
	(q31_t)0x6ca9ce3b, (q31_t)0xbc5a0be2, (q31_t)0x6ca67c01, (q31_t)0xbc54b676, (q31_t)0x6ca32985, (q31_t)0xbc4f6134,
	(q31_t)0x6c9fd6c6, (q31_t)0xbc4a0c1b,
	(q31_t)0x6c9c83c3, (q31_t)0xbc44b72c, (q31_t)0x6c99307e, (q31_t)0xbc3f6267, (q31_t)0x6c95dcf6, (q31_t)0xbc3a0dcc,
	(q31_t)0x6c92892a, (q31_t)0xbc34b95b,
	(q31_t)0x6c8f351c, (q31_t)0xbc2f6513, (q31_t)0x6c8be0cb, (q31_t)0xbc2a10f6, (q31_t)0x6c888c36, (q31_t)0xbc24bd02,
	(q31_t)0x6c85375f, (q31_t)0xbc1f6938,
	(q31_t)0x6c81e245, (q31_t)0xbc1a1598, (q31_t)0x6c7e8ce8, (q31_t)0xbc14c221, (q31_t)0x6c7b3748, (q31_t)0xbc0f6ed5,
	(q31_t)0x6c77e165, (q31_t)0xbc0a1bb3,
	(q31_t)0x6c748b3f, (q31_t)0xbc04c8ba, (q31_t)0x6c7134d7, (q31_t)0xbbff75ec, (q31_t)0x6c6dde2b, (q31_t)0xbbfa2347,
	(q31_t)0x6c6a873d, (q31_t)0xbbf4d0cc,
	(q31_t)0x6c67300b, (q31_t)0xbbef7e7c, (q31_t)0x6c63d897, (q31_t)0xbbea2c55, (q31_t)0x6c6080e0, (q31_t)0xbbe4da58,
	(q31_t)0x6c5d28e6, (q31_t)0xbbdf8885,
	(q31_t)0x6c59d0a9, (q31_t)0xbbda36dd, (q31_t)0x6c56782a, (q31_t)0xbbd4e55e, (q31_t)0x6c531f67, (q31_t)0xbbcf940a,
	(q31_t)0x6c4fc662, (q31_t)0xbbca42df,
	(q31_t)0x6c4c6d1a, (q31_t)0xbbc4f1df, (q31_t)0x6c49138f, (q31_t)0xbbbfa108, (q31_t)0x6c45b9c1, (q31_t)0xbbba505c,
	(q31_t)0x6c425fb1, (q31_t)0xbbb4ffda,
	(q31_t)0x6c3f055d, (q31_t)0xbbafaf82, (q31_t)0x6c3baac7, (q31_t)0xbbaa5f54, (q31_t)0x6c384fef, (q31_t)0xbba50f50,
	(q31_t)0x6c34f4d3, (q31_t)0xbb9fbf77,
	(q31_t)0x6c319975, (q31_t)0xbb9a6fc7, (q31_t)0x6c2e3dd4, (q31_t)0xbb952042, (q31_t)0x6c2ae1f0, (q31_t)0xbb8fd0e7,
	(q31_t)0x6c2785ca, (q31_t)0xbb8a81b6,
	(q31_t)0x6c242960, (q31_t)0xbb8532b0, (q31_t)0x6c20ccb4, (q31_t)0xbb7fe3d3, (q31_t)0x6c1d6fc6, (q31_t)0xbb7a9521,
	(q31_t)0x6c1a1295, (q31_t)0xbb754699,
	(q31_t)0x6c16b521, (q31_t)0xbb6ff83c, (q31_t)0x6c13576a, (q31_t)0xbb6aaa09, (q31_t)0x6c0ff971, (q31_t)0xbb655c00,
	(q31_t)0x6c0c9b35, (q31_t)0xbb600e21,
	(q31_t)0x6c093cb6, (q31_t)0xbb5ac06d, (q31_t)0x6c05ddf5, (q31_t)0xbb5572e3, (q31_t)0x6c027ef1, (q31_t)0xbb502583,
	(q31_t)0x6bff1faa, (q31_t)0xbb4ad84e,
	(q31_t)0x6bfbc021, (q31_t)0xbb458b43, (q31_t)0x6bf86055, (q31_t)0xbb403e63, (q31_t)0x6bf50047, (q31_t)0xbb3af1ad,
	(q31_t)0x6bf19ff6, (q31_t)0xbb35a521,
	(q31_t)0x6bee3f62, (q31_t)0xbb3058c0, (q31_t)0x6beade8c, (q31_t)0xbb2b0c8a, (q31_t)0x6be77d74, (q31_t)0xbb25c07d,
	(q31_t)0x6be41c18, (q31_t)0xbb20749c,
	(q31_t)0x6be0ba7b, (q31_t)0xbb1b28e4, (q31_t)0x6bdd589a, (q31_t)0xbb15dd57, (q31_t)0x6bd9f677, (q31_t)0xbb1091f5,
	(q31_t)0x6bd69412, (q31_t)0xbb0b46bd,
	(q31_t)0x6bd3316a, (q31_t)0xbb05fbb0, (q31_t)0x6bcfce80, (q31_t)0xbb00b0ce, (q31_t)0x6bcc6b53, (q31_t)0xbafb6615,
	(q31_t)0x6bc907e3, (q31_t)0xbaf61b88,
	(q31_t)0x6bc5a431, (q31_t)0xbaf0d125, (q31_t)0x6bc2403d, (q31_t)0xbaeb86ed, (q31_t)0x6bbedc06, (q31_t)0xbae63cdf,
	(q31_t)0x6bbb778d, (q31_t)0xbae0f2fc,
	(q31_t)0x6bb812d1, (q31_t)0xbadba943, (q31_t)0x6bb4add3, (q31_t)0xbad65fb5, (q31_t)0x6bb14892, (q31_t)0xbad11652,
	(q31_t)0x6bade30f, (q31_t)0xbacbcd1a,
	(q31_t)0x6baa7d49, (q31_t)0xbac6840c, (q31_t)0x6ba71741, (q31_t)0xbac13b29, (q31_t)0x6ba3b0f7, (q31_t)0xbabbf270,
	(q31_t)0x6ba04a6a, (q31_t)0xbab6a9e3,
	(q31_t)0x6b9ce39b, (q31_t)0xbab16180, (q31_t)0x6b997c8a, (q31_t)0xbaac1948, (q31_t)0x6b961536, (q31_t)0xbaa6d13a,
	(q31_t)0x6b92ada0, (q31_t)0xbaa18958,
	(q31_t)0x6b8f45c7, (q31_t)0xba9c41a0, (q31_t)0x6b8bddac, (q31_t)0xba96fa13, (q31_t)0x6b88754f, (q31_t)0xba91b2b1,
	(q31_t)0x6b850caf, (q31_t)0xba8c6b79,
	(q31_t)0x6b81a3cd, (q31_t)0xba87246d, (q31_t)0x6b7e3aa9, (q31_t)0xba81dd8b, (q31_t)0x6b7ad142, (q31_t)0xba7c96d4,
	(q31_t)0x6b776799, (q31_t)0xba775048,
	(q31_t)0x6b73fdae, (q31_t)0xba7209e7, (q31_t)0x6b709381, (q31_t)0xba6cc3b1, (q31_t)0x6b6d2911, (q31_t)0xba677da6,
	(q31_t)0x6b69be5f, (q31_t)0xba6237c5,
	(q31_t)0x6b66536b, (q31_t)0xba5cf210, (q31_t)0x6b62e834, (q31_t)0xba57ac86, (q31_t)0x6b5f7cbc, (q31_t)0xba526726,
	(q31_t)0x6b5c1101, (q31_t)0xba4d21f2,
	(q31_t)0x6b58a503, (q31_t)0xba47dce8, (q31_t)0x6b5538c4, (q31_t)0xba42980a, (q31_t)0x6b51cc42, (q31_t)0xba3d5356,
	(q31_t)0x6b4e5f7f, (q31_t)0xba380ece,
	(q31_t)0x6b4af279, (q31_t)0xba32ca71, (q31_t)0x6b478530, (q31_t)0xba2d863e, (q31_t)0x6b4417a6, (q31_t)0xba284237,
	(q31_t)0x6b40a9d9, (q31_t)0xba22fe5b,
	(q31_t)0x6b3d3bcb, (q31_t)0xba1dbaaa, (q31_t)0x6b39cd7a, (q31_t)0xba187724, (q31_t)0x6b365ee7, (q31_t)0xba1333c9,
	(q31_t)0x6b32f012, (q31_t)0xba0df099,
	(q31_t)0x6b2f80fb, (q31_t)0xba08ad95, (q31_t)0x6b2c11a1, (q31_t)0xba036abb, (q31_t)0x6b28a206, (q31_t)0xb9fe280d,
	(q31_t)0x6b253228, (q31_t)0xb9f8e58a,
	(q31_t)0x6b21c208, (q31_t)0xb9f3a332, (q31_t)0x6b1e51a7, (q31_t)0xb9ee6106, (q31_t)0x6b1ae103, (q31_t)0xb9e91f04,
	(q31_t)0x6b17701d, (q31_t)0xb9e3dd2e,
	(q31_t)0x6b13fef5, (q31_t)0xb9de9b83, (q31_t)0x6b108d8b, (q31_t)0xb9d95a03, (q31_t)0x6b0d1bdf, (q31_t)0xb9d418af,
	(q31_t)0x6b09a9f1, (q31_t)0xb9ced786,
	(q31_t)0x6b0637c1, (q31_t)0xb9c99688, (q31_t)0x6b02c54f, (q31_t)0xb9c455b6, (q31_t)0x6aff529a, (q31_t)0xb9bf150e,
	(q31_t)0x6afbdfa4, (q31_t)0xb9b9d493,
	(q31_t)0x6af86c6c, (q31_t)0xb9b49442, (q31_t)0x6af4f8f2, (q31_t)0xb9af541d, (q31_t)0x6af18536, (q31_t)0xb9aa1423,
	(q31_t)0x6aee1138, (q31_t)0xb9a4d455,
	(q31_t)0x6aea9cf8, (q31_t)0xb99f94b2, (q31_t)0x6ae72876, (q31_t)0xb99a553a, (q31_t)0x6ae3b3b2, (q31_t)0xb99515ee,
	(q31_t)0x6ae03eac, (q31_t)0xb98fd6cd,
	(q31_t)0x6adcc964, (q31_t)0xb98a97d8, (q31_t)0x6ad953db, (q31_t)0xb985590e, (q31_t)0x6ad5de0f, (q31_t)0xb9801a70,
	(q31_t)0x6ad26802, (q31_t)0xb97adbfd,
	(q31_t)0x6acef1b2, (q31_t)0xb9759db6, (q31_t)0x6acb7b21, (q31_t)0xb9705f9a, (q31_t)0x6ac8044e, (q31_t)0xb96b21aa,
	(q31_t)0x6ac48d39, (q31_t)0xb965e3e5,
	(q31_t)0x6ac115e2, (q31_t)0xb960a64c, (q31_t)0x6abd9e49, (q31_t)0xb95b68de, (q31_t)0x6aba266e, (q31_t)0xb9562b9c,
	(q31_t)0x6ab6ae52, (q31_t)0xb950ee86,
	(q31_t)0x6ab335f4, (q31_t)0xb94bb19b, (q31_t)0x6aafbd54, (q31_t)0xb94674dc, (q31_t)0x6aac4472, (q31_t)0xb9413848,
	(q31_t)0x6aa8cb4e, (q31_t)0xb93bfbe0,
	(q31_t)0x6aa551e9, (q31_t)0xb936bfa4, (q31_t)0x6aa1d841, (q31_t)0xb9318393, (q31_t)0x6a9e5e58, (q31_t)0xb92c47ae,
	(q31_t)0x6a9ae42e, (q31_t)0xb9270bf5,
	(q31_t)0x6a9769c1, (q31_t)0xb921d067, (q31_t)0x6a93ef13, (q31_t)0xb91c9505, (q31_t)0x6a907423, (q31_t)0xb91759cf,
	(q31_t)0x6a8cf8f1, (q31_t)0xb9121ec5,
	(q31_t)0x6a897d7d, (q31_t)0xb90ce3e6, (q31_t)0x6a8601c8, (q31_t)0xb907a933, (q31_t)0x6a8285d1, (q31_t)0xb9026eac,
	(q31_t)0x6a7f0999, (q31_t)0xb8fd3451,
	(q31_t)0x6a7b8d1e, (q31_t)0xb8f7fa21, (q31_t)0x6a781062, (q31_t)0xb8f2c01d, (q31_t)0x6a749365, (q31_t)0xb8ed8646,
	(q31_t)0x6a711625, (q31_t)0xb8e84c99,
	(q31_t)0x6a6d98a4, (q31_t)0xb8e31319, (q31_t)0x6a6a1ae2, (q31_t)0xb8ddd9c5, (q31_t)0x6a669cdd, (q31_t)0xb8d8a09d,
	(q31_t)0x6a631e97, (q31_t)0xb8d367a0,
	(q31_t)0x6a5fa010, (q31_t)0xb8ce2ecf, (q31_t)0x6a5c2147, (q31_t)0xb8c8f62b, (q31_t)0x6a58a23c, (q31_t)0xb8c3bdb2,
	(q31_t)0x6a5522ef, (q31_t)0xb8be8565,
	(q31_t)0x6a51a361, (q31_t)0xb8b94d44, (q31_t)0x6a4e2392, (q31_t)0xb8b4154f, (q31_t)0x6a4aa381, (q31_t)0xb8aedd86,
	(q31_t)0x6a47232e, (q31_t)0xb8a9a5e9,
	(q31_t)0x6a43a29a, (q31_t)0xb8a46e78, (q31_t)0x6a4021c4, (q31_t)0xb89f3733, (q31_t)0x6a3ca0ad, (q31_t)0xb89a001a,
	(q31_t)0x6a391f54, (q31_t)0xb894c92d,
	(q31_t)0x6a359db9, (q31_t)0xb88f926d, (q31_t)0x6a321bdd, (q31_t)0xb88a5bd8, (q31_t)0x6a2e99c0, (q31_t)0xb885256f,
	(q31_t)0x6a2b1761, (q31_t)0xb87fef33,
	(q31_t)0x6a2794c1, (q31_t)0xb87ab922, (q31_t)0x6a2411df, (q31_t)0xb875833e, (q31_t)0x6a208ebb, (q31_t)0xb8704d85,
	(q31_t)0x6a1d0b57, (q31_t)0xb86b17f9,
	(q31_t)0x6a1987b0, (q31_t)0xb865e299, (q31_t)0x6a1603c8, (q31_t)0xb860ad66, (q31_t)0x6a127f9f, (q31_t)0xb85b785e,
	(q31_t)0x6a0efb35, (q31_t)0xb8564383,
	(q31_t)0x6a0b7689, (q31_t)0xb8510ed4, (q31_t)0x6a07f19b, (q31_t)0xb84bda51, (q31_t)0x6a046c6c, (q31_t)0xb846a5fa,
	(q31_t)0x6a00e6fc, (q31_t)0xb84171cf,
	(q31_t)0x69fd614a, (q31_t)0xb83c3dd1, (q31_t)0x69f9db57, (q31_t)0xb83709ff, (q31_t)0x69f65523, (q31_t)0xb831d659,
	(q31_t)0x69f2cead, (q31_t)0xb82ca2e0,
	(q31_t)0x69ef47f6, (q31_t)0xb8276f93, (q31_t)0x69ebc0fe, (q31_t)0xb8223c72, (q31_t)0x69e839c4, (q31_t)0xb81d097e,
	(q31_t)0x69e4b249, (q31_t)0xb817d6b6,
	(q31_t)0x69e12a8c, (q31_t)0xb812a41a, (q31_t)0x69dda28f, (q31_t)0xb80d71aa, (q31_t)0x69da1a50, (q31_t)0xb8083f67,
	(q31_t)0x69d691cf, (q31_t)0xb8030d51,
	(q31_t)0x69d3090e, (q31_t)0xb7fddb67, (q31_t)0x69cf800b, (q31_t)0xb7f8a9a9, (q31_t)0x69cbf6c7, (q31_t)0xb7f37818,
	(q31_t)0x69c86d41, (q31_t)0xb7ee46b3,
	(q31_t)0x69c4e37a, (q31_t)0xb7e9157a, (q31_t)0x69c15973, (q31_t)0xb7e3e46e, (q31_t)0x69bdcf29, (q31_t)0xb7deb38f,
	(q31_t)0x69ba449f, (q31_t)0xb7d982dc,
	(q31_t)0x69b6b9d3, (q31_t)0xb7d45255, (q31_t)0x69b32ec7, (q31_t)0xb7cf21fb, (q31_t)0x69afa378, (q31_t)0xb7c9f1ce,
	(q31_t)0x69ac17e9, (q31_t)0xb7c4c1cd,
	(q31_t)0x69a88c19, (q31_t)0xb7bf91f8, (q31_t)0x69a50007, (q31_t)0xb7ba6251, (q31_t)0x69a173b5, (q31_t)0xb7b532d6,
	(q31_t)0x699de721, (q31_t)0xb7b00387,
	(q31_t)0x699a5a4c, (q31_t)0xb7aad465, (q31_t)0x6996cd35, (q31_t)0xb7a5a570, (q31_t)0x69933fde, (q31_t)0xb7a076a7,
	(q31_t)0x698fb246, (q31_t)0xb79b480b,
	(q31_t)0x698c246c, (q31_t)0xb796199b, (q31_t)0x69889651, (q31_t)0xb790eb58, (q31_t)0x698507f6, (q31_t)0xb78bbd42,
	(q31_t)0x69817959, (q31_t)0xb7868f59,
	(q31_t)0x697dea7b, (q31_t)0xb781619c, (q31_t)0x697a5b5c, (q31_t)0xb77c340c, (q31_t)0x6976cbfc, (q31_t)0xb77706a9,
	(q31_t)0x69733c5b, (q31_t)0xb771d972,
	(q31_t)0x696fac78, (q31_t)0xb76cac69, (q31_t)0x696c1c55, (q31_t)0xb7677f8c, (q31_t)0x69688bf1, (q31_t)0xb76252db,
	(q31_t)0x6964fb4c, (q31_t)0xb75d2658,
	(q31_t)0x69616a65, (q31_t)0xb757fa01, (q31_t)0x695dd93e, (q31_t)0xb752cdd8, (q31_t)0x695a47d6, (q31_t)0xb74da1db,
	(q31_t)0x6956b62d, (q31_t)0xb748760b,
	(q31_t)0x69532442, (q31_t)0xb7434a67, (q31_t)0x694f9217, (q31_t)0xb73e1ef1, (q31_t)0x694bffab, (q31_t)0xb738f3a7,
	(q31_t)0x69486cfe, (q31_t)0xb733c88b,
	(q31_t)0x6944da10, (q31_t)0xb72e9d9b, (q31_t)0x694146e1, (q31_t)0xb72972d8, (q31_t)0x693db371, (q31_t)0xb7244842,
	(q31_t)0x693a1fc0, (q31_t)0xb71f1dd9,
	(q31_t)0x69368bce, (q31_t)0xb719f39e, (q31_t)0x6932f79b, (q31_t)0xb714c98e, (q31_t)0x692f6328, (q31_t)0xb70f9fac,
	(q31_t)0x692bce73, (q31_t)0xb70a75f7,
	(q31_t)0x6928397e, (q31_t)0xb7054c6f, (q31_t)0x6924a448, (q31_t)0xb7002314, (q31_t)0x69210ed1, (q31_t)0xb6faf9e6,
	(q31_t)0x691d7919, (q31_t)0xb6f5d0e5,
	(q31_t)0x6919e320, (q31_t)0xb6f0a812, (q31_t)0x69164ce7, (q31_t)0xb6eb7f6b, (q31_t)0x6912b66c, (q31_t)0xb6e656f1,
	(q31_t)0x690f1fb1, (q31_t)0xb6e12ea4,
	(q31_t)0x690b88b5, (q31_t)0xb6dc0685, (q31_t)0x6907f178, (q31_t)0xb6d6de92, (q31_t)0x690459fb, (q31_t)0xb6d1b6cd,
	(q31_t)0x6900c23c, (q31_t)0xb6cc8f35,
	(q31_t)0x68fd2a3d, (q31_t)0xb6c767ca, (q31_t)0x68f991fd, (q31_t)0xb6c2408c, (q31_t)0x68f5f97d, (q31_t)0xb6bd197c,
	(q31_t)0x68f260bb, (q31_t)0xb6b7f298,
	(q31_t)0x68eec7b9, (q31_t)0xb6b2cbe2, (q31_t)0x68eb2e76, (q31_t)0xb6ada559, (q31_t)0x68e794f3, (q31_t)0xb6a87efd,
	(q31_t)0x68e3fb2e, (q31_t)0xb6a358ce,
	(q31_t)0x68e06129, (q31_t)0xb69e32cd, (q31_t)0x68dcc6e4, (q31_t)0xb6990cf9, (q31_t)0x68d92c5d, (q31_t)0xb693e752,
	(q31_t)0x68d59196, (q31_t)0xb68ec1d9,
	(q31_t)0x68d1f68f, (q31_t)0xb6899c8d, (q31_t)0x68ce5b46, (q31_t)0xb684776e, (q31_t)0x68cabfbd, (q31_t)0xb67f527c,
	(q31_t)0x68c723f3, (q31_t)0xb67a2db8,
	(q31_t)0x68c387e9, (q31_t)0xb6750921, (q31_t)0x68bfeb9e, (q31_t)0xb66fe4b8, (q31_t)0x68bc4f13, (q31_t)0xb66ac07c,
	(q31_t)0x68b8b247, (q31_t)0xb6659c6d,
	(q31_t)0x68b5153a, (q31_t)0xb660788c, (q31_t)0x68b177ed, (q31_t)0xb65b54d8, (q31_t)0x68adda5f, (q31_t)0xb6563151,
	(q31_t)0x68aa3c90, (q31_t)0xb6510df8,
	(q31_t)0x68a69e81, (q31_t)0xb64beacd, (q31_t)0x68a30031, (q31_t)0xb646c7ce, (q31_t)0x689f61a1, (q31_t)0xb641a4fe,
	(q31_t)0x689bc2d1, (q31_t)0xb63c825b,
	(q31_t)0x689823bf, (q31_t)0xb6375fe5, (q31_t)0x6894846e, (q31_t)0xb6323d9d, (q31_t)0x6890e4dc, (q31_t)0xb62d1b82,
	(q31_t)0x688d4509, (q31_t)0xb627f995,
	(q31_t)0x6889a4f6, (q31_t)0xb622d7d6, (q31_t)0x688604a2, (q31_t)0xb61db644, (q31_t)0x6882640e, (q31_t)0xb61894df,
	(q31_t)0x687ec339, (q31_t)0xb61373a9,
	(q31_t)0x687b2224, (q31_t)0xb60e529f, (q31_t)0x687780ce, (q31_t)0xb60931c4, (q31_t)0x6873df38, (q31_t)0xb6041116,
	(q31_t)0x68703d62, (q31_t)0xb5fef095,
	(q31_t)0x686c9b4b, (q31_t)0xb5f9d043, (q31_t)0x6868f8f4, (q31_t)0xb5f4b01e, (q31_t)0x6865565c, (q31_t)0xb5ef9026,
	(q31_t)0x6861b384, (q31_t)0xb5ea705d,
	(q31_t)0x685e106c, (q31_t)0xb5e550c1, (q31_t)0x685a6d13, (q31_t)0xb5e03153, (q31_t)0x6856c979, (q31_t)0xb5db1212,
	(q31_t)0x685325a0, (q31_t)0xb5d5f2ff,
	(q31_t)0x684f8186, (q31_t)0xb5d0d41a, (q31_t)0x684bdd2c, (q31_t)0xb5cbb563, (q31_t)0x68483891, (q31_t)0xb5c696da,
	(q31_t)0x684493b6, (q31_t)0xb5c1787e,
	(q31_t)0x6840ee9b, (q31_t)0xb5bc5a50, (q31_t)0x683d493f, (q31_t)0xb5b73c50, (q31_t)0x6839a3a4, (q31_t)0xb5b21e7e,
	(q31_t)0x6835fdc7, (q31_t)0xb5ad00d9,
	(q31_t)0x683257ab, (q31_t)0xb5a7e362, (q31_t)0x682eb14e, (q31_t)0xb5a2c61a, (q31_t)0x682b0ab1, (q31_t)0xb59da8ff,
	(q31_t)0x682763d4, (q31_t)0xb5988c12,
	(q31_t)0x6823bcb7, (q31_t)0xb5936f53, (q31_t)0x68201559, (q31_t)0xb58e52c2, (q31_t)0x681c6dbb, (q31_t)0xb589365e,
	(q31_t)0x6818c5dd, (q31_t)0xb5841a29,
	(q31_t)0x68151dbe, (q31_t)0xb57efe22, (q31_t)0x68117560, (q31_t)0xb579e248, (q31_t)0x680dccc1, (q31_t)0xb574c69d,
	(q31_t)0x680a23e2, (q31_t)0xb56fab1f,
	(q31_t)0x68067ac3, (q31_t)0xb56a8fd0, (q31_t)0x6802d164, (q31_t)0xb56574ae, (q31_t)0x67ff27c4, (q31_t)0xb56059bb,
	(q31_t)0x67fb7de5, (q31_t)0xb55b3ef5,
	(q31_t)0x67f7d3c5, (q31_t)0xb556245e, (q31_t)0x67f42965, (q31_t)0xb55109f5, (q31_t)0x67f07ec5, (q31_t)0xb54befba,
	(q31_t)0x67ecd3e5, (q31_t)0xb546d5ac,
	(q31_t)0x67e928c5, (q31_t)0xb541bbcd, (q31_t)0x67e57d64, (q31_t)0xb53ca21c, (q31_t)0x67e1d1c4, (q31_t)0xb5378899,
	(q31_t)0x67de25e3, (q31_t)0xb5326f45,
	(q31_t)0x67da79c3, (q31_t)0xb52d561e, (q31_t)0x67d6cd62, (q31_t)0xb5283d26, (q31_t)0x67d320c1, (q31_t)0xb523245b,
	(q31_t)0x67cf73e1, (q31_t)0xb51e0bbf,
	(q31_t)0x67cbc6c0, (q31_t)0xb518f351, (q31_t)0x67c8195f, (q31_t)0xb513db12, (q31_t)0x67c46bbe, (q31_t)0xb50ec300,
	(q31_t)0x67c0bddd, (q31_t)0xb509ab1d,
	(q31_t)0x67bd0fbd, (q31_t)0xb5049368, (q31_t)0x67b9615c, (q31_t)0xb4ff7be1, (q31_t)0x67b5b2bb, (q31_t)0xb4fa6489,
	(q31_t)0x67b203da, (q31_t)0xb4f54d5f,
	(q31_t)0x67ae54ba, (q31_t)0xb4f03663, (q31_t)0x67aaa559, (q31_t)0xb4eb1f95, (q31_t)0x67a6f5b8, (q31_t)0xb4e608f6,
	(q31_t)0x67a345d8, (q31_t)0xb4e0f285,
	(q31_t)0x679f95b7, (q31_t)0xb4dbdc42, (q31_t)0x679be557, (q31_t)0xb4d6c62e, (q31_t)0x679834b6, (q31_t)0xb4d1b048,
	(q31_t)0x679483d6, (q31_t)0xb4cc9a90,
	(q31_t)0x6790d2b6, (q31_t)0xb4c78507, (q31_t)0x678d2156, (q31_t)0xb4c26fad, (q31_t)0x67896fb6, (q31_t)0xb4bd5a80,
	(q31_t)0x6785bdd6, (q31_t)0xb4b84582,
	(q31_t)0x67820bb7, (q31_t)0xb4b330b3, (q31_t)0x677e5957, (q31_t)0xb4ae1c12, (q31_t)0x677aa6b8, (q31_t)0xb4a9079f,
	(q31_t)0x6776f3d9, (q31_t)0xb4a3f35b,
	(q31_t)0x677340ba, (q31_t)0xb49edf45, (q31_t)0x676f8d5b, (q31_t)0xb499cb5e, (q31_t)0x676bd9bd, (q31_t)0xb494b7a6,
	(q31_t)0x676825de, (q31_t)0xb48fa41c,
	(q31_t)0x676471c0, (q31_t)0xb48a90c0, (q31_t)0x6760bd62, (q31_t)0xb4857d93, (q31_t)0x675d08c4, (q31_t)0xb4806a95,
	(q31_t)0x675953e7, (q31_t)0xb47b57c5,
	(q31_t)0x67559eca, (q31_t)0xb4764523, (q31_t)0x6751e96d, (q31_t)0xb47132b1, (q31_t)0x674e33d0, (q31_t)0xb46c206d,
	(q31_t)0x674a7df4, (q31_t)0xb4670e57,
	(q31_t)0x6746c7d8, (q31_t)0xb461fc70, (q31_t)0x6743117c, (q31_t)0xb45ceab8, (q31_t)0x673f5ae0, (q31_t)0xb457d92f,
	(q31_t)0x673ba405, (q31_t)0xb452c7d4,
	(q31_t)0x6737ecea, (q31_t)0xb44db6a8, (q31_t)0x67343590, (q31_t)0xb448a5aa, (q31_t)0x67307df5, (q31_t)0xb44394db,
	(q31_t)0x672cc61c, (q31_t)0xb43e843b,
	(q31_t)0x67290e02, (q31_t)0xb43973ca, (q31_t)0x672555a9, (q31_t)0xb4346387, (q31_t)0x67219d10, (q31_t)0xb42f5373,
	(q31_t)0x671de438, (q31_t)0xb42a438e,
	(q31_t)0x671a2b20, (q31_t)0xb42533d8, (q31_t)0x671671c8, (q31_t)0xb4202451, (q31_t)0x6712b831, (q31_t)0xb41b14f8,
	(q31_t)0x670efe5a, (q31_t)0xb41605ce,
	(q31_t)0x670b4444, (q31_t)0xb410f6d3, (q31_t)0x670789ee, (q31_t)0xb40be807, (q31_t)0x6703cf58, (q31_t)0xb406d969,
	(q31_t)0x67001483, (q31_t)0xb401cafb,
	(q31_t)0x66fc596f, (q31_t)0xb3fcbcbb, (q31_t)0x66f89e1b, (q31_t)0xb3f7aeaa, (q31_t)0x66f4e287, (q31_t)0xb3f2a0c9,
	(q31_t)0x66f126b4, (q31_t)0xb3ed9316,
	(q31_t)0x66ed6aa1, (q31_t)0xb3e88592, (q31_t)0x66e9ae4f, (q31_t)0xb3e3783d, (q31_t)0x66e5f1be, (q31_t)0xb3de6b17,
	(q31_t)0x66e234ed, (q31_t)0xb3d95e1f,
	(q31_t)0x66de77dc, (q31_t)0xb3d45157, (q31_t)0x66daba8c, (q31_t)0xb3cf44be, (q31_t)0x66d6fcfd, (q31_t)0xb3ca3854,
	(q31_t)0x66d33f2e, (q31_t)0xb3c52c19,
	(q31_t)0x66cf8120, (q31_t)0xb3c0200c, (q31_t)0x66cbc2d2, (q31_t)0xb3bb142f, (q31_t)0x66c80445, (q31_t)0xb3b60881,
	(q31_t)0x66c44579, (q31_t)0xb3b0fd02,
	(q31_t)0x66c0866d, (q31_t)0xb3abf1b2, (q31_t)0x66bcc721, (q31_t)0xb3a6e691, (q31_t)0x66b90797, (q31_t)0xb3a1dba0,
	(q31_t)0x66b547cd, (q31_t)0xb39cd0dd,
	(q31_t)0x66b187c3, (q31_t)0xb397c649, (q31_t)0x66adc77b, (q31_t)0xb392bbe5, (q31_t)0x66aa06f3, (q31_t)0xb38db1b0,
	(q31_t)0x66a6462b, (q31_t)0xb388a7aa,
	(q31_t)0x66a28524, (q31_t)0xb3839dd3, (q31_t)0x669ec3de, (q31_t)0xb37e942b, (q31_t)0x669b0259, (q31_t)0xb3798ab2,
	(q31_t)0x66974095, (q31_t)0xb3748169,
	(q31_t)0x66937e91, (q31_t)0xb36f784f, (q31_t)0x668fbc4e, (q31_t)0xb36a6f64, (q31_t)0x668bf9cb, (q31_t)0xb36566a8,
	(q31_t)0x66883709, (q31_t)0xb3605e1c,
	(q31_t)0x66847408, (q31_t)0xb35b55bf, (q31_t)0x6680b0c8, (q31_t)0xb3564d91, (q31_t)0x667ced49, (q31_t)0xb3514592,
	(q31_t)0x6679298a, (q31_t)0xb34c3dc3,
	(q31_t)0x6675658c, (q31_t)0xb3473623, (q31_t)0x6671a14f, (q31_t)0xb3422eb2, (q31_t)0x666ddcd3, (q31_t)0xb33d2771,
	(q31_t)0x666a1818, (q31_t)0xb338205f,
	(q31_t)0x6666531d, (q31_t)0xb333197c, (q31_t)0x66628de4, (q31_t)0xb32e12c9, (q31_t)0x665ec86b, (q31_t)0xb3290c45,
	(q31_t)0x665b02b3, (q31_t)0xb32405f1,
	(q31_t)0x66573cbb, (q31_t)0xb31effcc, (q31_t)0x66537685, (q31_t)0xb319f9d6, (q31_t)0x664fb010, (q31_t)0xb314f410,
	(q31_t)0x664be95b, (q31_t)0xb30fee79,
	(q31_t)0x66482267, (q31_t)0xb30ae912, (q31_t)0x66445b35, (q31_t)0xb305e3da, (q31_t)0x664093c3, (q31_t)0xb300ded2,
	(q31_t)0x663ccc12, (q31_t)0xb2fbd9f9,
	(q31_t)0x66390422, (q31_t)0xb2f6d550, (q31_t)0x66353bf3, (q31_t)0xb2f1d0d6, (q31_t)0x66317385, (q31_t)0xb2eccc8c,
	(q31_t)0x662daad8, (q31_t)0xb2e7c871,
	(q31_t)0x6629e1ec, (q31_t)0xb2e2c486, (q31_t)0x662618c1, (q31_t)0xb2ddc0ca, (q31_t)0x66224f56, (q31_t)0xb2d8bd3e,
	(q31_t)0x661e85ad, (q31_t)0xb2d3b9e2,
	(q31_t)0x661abbc5, (q31_t)0xb2ceb6b5, (q31_t)0x6616f19e, (q31_t)0xb2c9b3b8, (q31_t)0x66132738, (q31_t)0xb2c4b0ea,
	(q31_t)0x660f5c93, (q31_t)0xb2bfae4c,
	(q31_t)0x660b91af, (q31_t)0xb2baabde, (q31_t)0x6607c68c, (q31_t)0xb2b5a99f, (q31_t)0x6603fb2a, (q31_t)0xb2b0a790,
	(q31_t)0x66002f89, (q31_t)0xb2aba5b1,
	(q31_t)0x65fc63a9, (q31_t)0xb2a6a402, (q31_t)0x65f8978b, (q31_t)0xb2a1a282, (q31_t)0x65f4cb2d, (q31_t)0xb29ca132,
	(q31_t)0x65f0fe91, (q31_t)0xb297a011,
	(q31_t)0x65ed31b5, (q31_t)0xb2929f21, (q31_t)0x65e9649b, (q31_t)0xb28d9e60, (q31_t)0x65e59742, (q31_t)0xb2889dcf,
	(q31_t)0x65e1c9aa, (q31_t)0xb2839d6d,
	(q31_t)0x65ddfbd3, (q31_t)0xb27e9d3c, (q31_t)0x65da2dbd, (q31_t)0xb2799d3a, (q31_t)0x65d65f69, (q31_t)0xb2749d68,
	(q31_t)0x65d290d6, (q31_t)0xb26f9dc6,
	(q31_t)0x65cec204, (q31_t)0xb26a9e54, (q31_t)0x65caf2f3, (q31_t)0xb2659f12, (q31_t)0x65c723a3, (q31_t)0xb2609fff,
	(q31_t)0x65c35415, (q31_t)0xb25ba11d,
	(q31_t)0x65bf8447, (q31_t)0xb256a26a, (q31_t)0x65bbb43b, (q31_t)0xb251a3e7, (q31_t)0x65b7e3f1, (q31_t)0xb24ca594,
	(q31_t)0x65b41367, (q31_t)0xb247a771,
	(q31_t)0x65b0429f, (q31_t)0xb242a97e, (q31_t)0x65ac7198, (q31_t)0xb23dabbb, (q31_t)0x65a8a052, (q31_t)0xb238ae28,
	(q31_t)0x65a4cece, (q31_t)0xb233b0c5,
	(q31_t)0x65a0fd0b, (q31_t)0xb22eb392, (q31_t)0x659d2b09, (q31_t)0xb229b68f, (q31_t)0x659958c9, (q31_t)0xb224b9bc,
	(q31_t)0x6595864a, (q31_t)0xb21fbd19,
	(q31_t)0x6591b38c, (q31_t)0xb21ac0a6, (q31_t)0x658de08f, (q31_t)0xb215c463, (q31_t)0x658a0d54, (q31_t)0xb210c850,
	(q31_t)0x658639db, (q31_t)0xb20bcc6d,
	(q31_t)0x65826622, (q31_t)0xb206d0ba, (q31_t)0x657e922b, (q31_t)0xb201d537, (q31_t)0x657abdf6, (q31_t)0xb1fcd9e5,
	(q31_t)0x6576e982, (q31_t)0xb1f7dec2,
	(q31_t)0x657314cf, (q31_t)0xb1f2e3d0, (q31_t)0x656f3fde, (q31_t)0xb1ede90e, (q31_t)0x656b6aae, (q31_t)0xb1e8ee7c,
	(q31_t)0x6567953f, (q31_t)0xb1e3f41a,
	(q31_t)0x6563bf92, (q31_t)0xb1def9e9, (q31_t)0x655fe9a7, (q31_t)0xb1d9ffe7, (q31_t)0x655c137d, (q31_t)0xb1d50616,
	(q31_t)0x65583d14, (q31_t)0xb1d00c75,
	(q31_t)0x6554666d, (q31_t)0xb1cb1304, (q31_t)0x65508f87, (q31_t)0xb1c619c3, (q31_t)0x654cb863, (q31_t)0xb1c120b3,
	(q31_t)0x6548e101, (q31_t)0xb1bc27d3,
	(q31_t)0x6545095f, (q31_t)0xb1b72f23, (q31_t)0x65413180, (q31_t)0xb1b236a4, (q31_t)0x653d5962, (q31_t)0xb1ad3e55,
	(q31_t)0x65398105, (q31_t)0xb1a84636,
	(q31_t)0x6535a86b, (q31_t)0xb1a34e47, (q31_t)0x6531cf91, (q31_t)0xb19e5689, (q31_t)0x652df679, (q31_t)0xb1995efb,
	(q31_t)0x652a1d23, (q31_t)0xb194679e,
	(q31_t)0x6526438f, (q31_t)0xb18f7071, (q31_t)0x652269bc, (q31_t)0xb18a7974, (q31_t)0x651e8faa, (q31_t)0xb18582a8,
	(q31_t)0x651ab55b, (q31_t)0xb1808c0c,
	(q31_t)0x6516dacd, (q31_t)0xb17b95a0, (q31_t)0x65130000, (q31_t)0xb1769f65, (q31_t)0x650f24f5, (q31_t)0xb171a95b,
	(q31_t)0x650b49ac, (q31_t)0xb16cb380,
	(q31_t)0x65076e25, (q31_t)0xb167bdd7, (q31_t)0x6503925f, (q31_t)0xb162c85d, (q31_t)0x64ffb65b, (q31_t)0xb15dd315,
	(q31_t)0x64fbda18, (q31_t)0xb158ddfd,
	(q31_t)0x64f7fd98, (q31_t)0xb153e915, (q31_t)0x64f420d9, (q31_t)0xb14ef45e, (q31_t)0x64f043dc, (q31_t)0xb149ffd7,
	(q31_t)0x64ec66a0, (q31_t)0xb1450b81,
	(q31_t)0x64e88926, (q31_t)0xb140175b, (q31_t)0x64e4ab6e, (q31_t)0xb13b2367, (q31_t)0x64e0cd78, (q31_t)0xb1362fa2,
	(q31_t)0x64dcef44, (q31_t)0xb1313c0e,
	(q31_t)0x64d910d1, (q31_t)0xb12c48ab, (q31_t)0x64d53220, (q31_t)0xb1275579, (q31_t)0x64d15331, (q31_t)0xb1226277,
	(q31_t)0x64cd7404, (q31_t)0xb11d6fa6,
	(q31_t)0x64c99498, (q31_t)0xb1187d05, (q31_t)0x64c5b4ef, (q31_t)0xb1138a95, (q31_t)0x64c1d507, (q31_t)0xb10e9856,
	(q31_t)0x64bdf4e1, (q31_t)0xb109a648,
	(q31_t)0x64ba147d, (q31_t)0xb104b46a, (q31_t)0x64b633da, (q31_t)0xb0ffc2bd, (q31_t)0x64b252fa, (q31_t)0xb0fad140,
	(q31_t)0x64ae71dc, (q31_t)0xb0f5dff5,
	(q31_t)0x64aa907f, (q31_t)0xb0f0eeda, (q31_t)0x64a6aee4, (q31_t)0xb0ebfdf0, (q31_t)0x64a2cd0c, (q31_t)0xb0e70d37,
	(q31_t)0x649eeaf5, (q31_t)0xb0e21cae,
	(q31_t)0x649b08a0, (q31_t)0xb0dd2c56, (q31_t)0x6497260d, (q31_t)0xb0d83c2f, (q31_t)0x6493433c, (q31_t)0xb0d34c39,
	(q31_t)0x648f602d, (q31_t)0xb0ce5c74,
	(q31_t)0x648b7ce0, (q31_t)0xb0c96ce0, (q31_t)0x64879955, (q31_t)0xb0c47d7c, (q31_t)0x6483b58c, (q31_t)0xb0bf8e4a,
	(q31_t)0x647fd185, (q31_t)0xb0ba9f48,
	(q31_t)0x647bed3f, (q31_t)0xb0b5b077, (q31_t)0x647808bc, (q31_t)0xb0b0c1d7, (q31_t)0x647423fb, (q31_t)0xb0abd368,
	(q31_t)0x64703efc, (q31_t)0xb0a6e52a,
	(q31_t)0x646c59bf, (q31_t)0xb0a1f71d, (q31_t)0x64687444, (q31_t)0xb09d0941, (q31_t)0x64648e8c, (q31_t)0xb0981b96,
	(q31_t)0x6460a895, (q31_t)0xb0932e1b,
	(q31_t)0x645cc260, (q31_t)0xb08e40d2, (q31_t)0x6458dbed, (q31_t)0xb08953ba, (q31_t)0x6454f53d, (q31_t)0xb08466d3,
	(q31_t)0x64510e4e, (q31_t)0xb07f7a1c,
	(q31_t)0x644d2722, (q31_t)0xb07a8d97, (q31_t)0x64493fb8, (q31_t)0xb075a143, (q31_t)0x64455810, (q31_t)0xb070b520,
	(q31_t)0x6441702a, (q31_t)0xb06bc92e,
	(q31_t)0x643d8806, (q31_t)0xb066dd6d, (q31_t)0x64399fa5, (q31_t)0xb061f1de, (q31_t)0x6435b706, (q31_t)0xb05d067f,
	(q31_t)0x6431ce28, (q31_t)0xb0581b51,
	(q31_t)0x642de50d, (q31_t)0xb0533055, (q31_t)0x6429fbb5, (q31_t)0xb04e458a, (q31_t)0x6426121e, (q31_t)0xb0495af0,
	(q31_t)0x6422284a, (q31_t)0xb0447087,
	(q31_t)0x641e3e38, (q31_t)0xb03f864f, (q31_t)0x641a53e8, (q31_t)0xb03a9c49, (q31_t)0x6416695a, (q31_t)0xb035b273,
	(q31_t)0x64127e8f, (q31_t)0xb030c8cf,
	(q31_t)0x640e9386, (q31_t)0xb02bdf5c, (q31_t)0x640aa83f, (q31_t)0xb026f61b, (q31_t)0x6406bcba, (q31_t)0xb0220d0a,
	(q31_t)0x6402d0f8, (q31_t)0xb01d242b,
	(q31_t)0x63fee4f8, (q31_t)0xb0183b7d, (q31_t)0x63faf8bb, (q31_t)0xb0135301, (q31_t)0x63f70c3f, (q31_t)0xb00e6ab5,
	(q31_t)0x63f31f86, (q31_t)0xb009829c,
	(q31_t)0x63ef3290, (q31_t)0xb0049ab3, (q31_t)0x63eb455c, (q31_t)0xafffb2fc, (q31_t)0x63e757ea, (q31_t)0xaffacb76,
	(q31_t)0x63e36a3a, (q31_t)0xaff5e421,
	(q31_t)0x63df7c4d, (q31_t)0xaff0fcfe, (q31_t)0x63db8e22, (q31_t)0xafec160c, (q31_t)0x63d79fba, (q31_t)0xafe72f4c,
	(q31_t)0x63d3b114, (q31_t)0xafe248bd,
	(q31_t)0x63cfc231, (q31_t)0xafdd625f, (q31_t)0x63cbd310, (q31_t)0xafd87c33, (q31_t)0x63c7e3b1, (q31_t)0xafd39638,
	(q31_t)0x63c3f415, (q31_t)0xafceb06f,
	(q31_t)0x63c0043b, (q31_t)0xafc9cad7, (q31_t)0x63bc1424, (q31_t)0xafc4e571, (q31_t)0x63b823cf, (q31_t)0xafc0003c,
	(q31_t)0x63b4333d, (q31_t)0xafbb1b39,
	(q31_t)0x63b0426d, (q31_t)0xafb63667, (q31_t)0x63ac5160, (q31_t)0xafb151c7, (q31_t)0x63a86015, (q31_t)0xafac6d58,
	(q31_t)0x63a46e8d, (q31_t)0xafa7891b,
	(q31_t)0x63a07cc7, (q31_t)0xafa2a50f, (q31_t)0x639c8ac4, (q31_t)0xaf9dc135, (q31_t)0x63989884, (q31_t)0xaf98dd8d,
	(q31_t)0x6394a606, (q31_t)0xaf93fa16,
	(q31_t)0x6390b34a, (q31_t)0xaf8f16d1, (q31_t)0x638cc051, (q31_t)0xaf8a33bd, (q31_t)0x6388cd1b, (q31_t)0xaf8550db,
	(q31_t)0x6384d9a7, (q31_t)0xaf806e2b,
	(q31_t)0x6380e5f6, (q31_t)0xaf7b8bac, (q31_t)0x637cf208, (q31_t)0xaf76a95f, (q31_t)0x6378fddc, (q31_t)0xaf71c743,
	(q31_t)0x63750973, (q31_t)0xaf6ce55a,
	(q31_t)0x637114cc, (q31_t)0xaf6803a2, (q31_t)0x636d1fe9, (q31_t)0xaf63221c, (q31_t)0x63692ac7, (q31_t)0xaf5e40c7,
	(q31_t)0x63653569, (q31_t)0xaf595fa4,
	(q31_t)0x63613fcd, (q31_t)0xaf547eb3, (q31_t)0x635d49f4, (q31_t)0xaf4f9df4, (q31_t)0x635953dd, (q31_t)0xaf4abd66,
	(q31_t)0x63555d8a, (q31_t)0xaf45dd0b,
	(q31_t)0x635166f9, (q31_t)0xaf40fce1, (q31_t)0x634d702b, (q31_t)0xaf3c1ce9, (q31_t)0x6349791f, (q31_t)0xaf373d22,
	(q31_t)0x634581d6, (q31_t)0xaf325d8e,
	(q31_t)0x63418a50, (q31_t)0xaf2d7e2b, (q31_t)0x633d928d, (q31_t)0xaf289efa, (q31_t)0x63399a8d, (q31_t)0xaf23bffb,
	(q31_t)0x6335a24f, (q31_t)0xaf1ee12e,
	(q31_t)0x6331a9d4, (q31_t)0xaf1a0293, (q31_t)0x632db11c, (q31_t)0xaf15242a, (q31_t)0x6329b827, (q31_t)0xaf1045f3,
	(q31_t)0x6325bef5, (q31_t)0xaf0b67ed,
	(q31_t)0x6321c585, (q31_t)0xaf068a1a, (q31_t)0x631dcbd9, (q31_t)0xaf01ac78, (q31_t)0x6319d1ef, (q31_t)0xaefccf09,
	(q31_t)0x6315d7c8, (q31_t)0xaef7f1cb,
	(q31_t)0x6311dd64, (q31_t)0xaef314c0, (q31_t)0x630de2c3, (q31_t)0xaeee37e6, (q31_t)0x6309e7e4, (q31_t)0xaee95b3f,
	(q31_t)0x6305ecc9, (q31_t)0xaee47ec9,
	(q31_t)0x6301f171, (q31_t)0xaedfa285, (q31_t)0x62fdf5db, (q31_t)0xaedac674, (q31_t)0x62f9fa09, (q31_t)0xaed5ea95,
	(q31_t)0x62f5fdf9, (q31_t)0xaed10ee7,
	(q31_t)0x62f201ac, (q31_t)0xaecc336c, (q31_t)0x62ee0523, (q31_t)0xaec75823, (q31_t)0x62ea085c, (q31_t)0xaec27d0c,
	(q31_t)0x62e60b58, (q31_t)0xaebda227,
	(q31_t)0x62e20e17, (q31_t)0xaeb8c774, (q31_t)0x62de109a, (q31_t)0xaeb3ecf3, (q31_t)0x62da12df, (q31_t)0xaeaf12a4,
	(q31_t)0x62d614e7, (q31_t)0xaeaa3888,
	(q31_t)0x62d216b3, (q31_t)0xaea55e9e, (q31_t)0x62ce1841, (q31_t)0xaea084e6, (q31_t)0x62ca1992, (q31_t)0xae9bab60,
	(q31_t)0x62c61aa7, (q31_t)0xae96d20c,
	(q31_t)0x62c21b7e, (q31_t)0xae91f8eb, (q31_t)0x62be1c19, (q31_t)0xae8d1ffb, (q31_t)0x62ba1c77, (q31_t)0xae88473e,
	(q31_t)0x62b61c98, (q31_t)0xae836eb4,
	(q31_t)0x62b21c7b, (q31_t)0xae7e965b, (q31_t)0x62ae1c23, (q31_t)0xae79be35, (q31_t)0x62aa1b8d, (q31_t)0xae74e641,
	(q31_t)0x62a61aba, (q31_t)0xae700e80,
	(q31_t)0x62a219aa, (q31_t)0xae6b36f0, (q31_t)0x629e185e, (q31_t)0xae665f93, (q31_t)0x629a16d5, (q31_t)0xae618869,
	(q31_t)0x6296150f, (q31_t)0xae5cb171,
	(q31_t)0x6292130c, (q31_t)0xae57daab, (q31_t)0x628e10cc, (q31_t)0xae530417, (q31_t)0x628a0e50, (q31_t)0xae4e2db6,
	(q31_t)0x62860b97, (q31_t)0xae495787,
	(q31_t)0x628208a1, (q31_t)0xae44818b, (q31_t)0x627e056e, (q31_t)0xae3fabc1, (q31_t)0x627a01fe, (q31_t)0xae3ad629,
	(q31_t)0x6275fe52, (q31_t)0xae3600c4,
	(q31_t)0x6271fa69, (q31_t)0xae312b92, (q31_t)0x626df643, (q31_t)0xae2c5691, (q31_t)0x6269f1e1, (q31_t)0xae2781c4,
	(q31_t)0x6265ed42, (q31_t)0xae22ad29,
	(q31_t)0x6261e866, (q31_t)0xae1dd8c0, (q31_t)0x625de34e, (q31_t)0xae19048a, (q31_t)0x6259ddf8, (q31_t)0xae143086,
	(q31_t)0x6255d866, (q31_t)0xae0f5cb5,
	(q31_t)0x6251d298, (q31_t)0xae0a8916, (q31_t)0x624dcc8d, (q31_t)0xae05b5aa, (q31_t)0x6249c645, (q31_t)0xae00e271,
	(q31_t)0x6245bfc0, (q31_t)0xadfc0f6a,
	(q31_t)0x6241b8ff, (q31_t)0xadf73c96, (q31_t)0x623db202, (q31_t)0xadf269f4, (q31_t)0x6239aac7, (q31_t)0xaded9785,
	(q31_t)0x6235a351, (q31_t)0xade8c548,
	(q31_t)0x62319b9d, (q31_t)0xade3f33e, (q31_t)0x622d93ad, (q31_t)0xaddf2167, (q31_t)0x62298b81, (q31_t)0xadda4fc3,
	(q31_t)0x62258317, (q31_t)0xadd57e51,
	(q31_t)0x62217a72, (q31_t)0xadd0ad12, (q31_t)0x621d7190, (q31_t)0xadcbdc05, (q31_t)0x62196871, (q31_t)0xadc70b2c,
	(q31_t)0x62155f16, (q31_t)0xadc23a85,
	(q31_t)0x6211557e, (q31_t)0xadbd6a10, (q31_t)0x620d4baa, (q31_t)0xadb899cf, (q31_t)0x62094199, (q31_t)0xadb3c9c0,
	(q31_t)0x6205374c, (q31_t)0xadaef9e4,
	(q31_t)0x62012cc2, (q31_t)0xadaa2a3b, (q31_t)0x61fd21fc, (q31_t)0xada55ac4, (q31_t)0x61f916f9, (q31_t)0xada08b80,
	(q31_t)0x61f50bba, (q31_t)0xad9bbc70,
	(q31_t)0x61f1003f, (q31_t)0xad96ed92, (q31_t)0x61ecf487, (q31_t)0xad921ee6, (q31_t)0x61e8e893, (q31_t)0xad8d506e,
	(q31_t)0x61e4dc62, (q31_t)0xad888229,
	(q31_t)0x61e0cff5, (q31_t)0xad83b416, (q31_t)0x61dcc34c, (q31_t)0xad7ee636, (q31_t)0x61d8b666, (q31_t)0xad7a1889,
	(q31_t)0x61d4a944, (q31_t)0xad754b0f,
	(q31_t)0x61d09be5, (q31_t)0xad707dc8, (q31_t)0x61cc8e4b, (q31_t)0xad6bb0b4, (q31_t)0x61c88074, (q31_t)0xad66e3d3,
	(q31_t)0x61c47260, (q31_t)0xad621725,
	(q31_t)0x61c06410, (q31_t)0xad5d4aaa, (q31_t)0x61bc5584, (q31_t)0xad587e61, (q31_t)0x61b846bc, (q31_t)0xad53b24c,
	(q31_t)0x61b437b7, (q31_t)0xad4ee66a,
	(q31_t)0x61b02876, (q31_t)0xad4a1aba, (q31_t)0x61ac18f9, (q31_t)0xad454f3e, (q31_t)0x61a80940, (q31_t)0xad4083f5,
	(q31_t)0x61a3f94a, (q31_t)0xad3bb8df,
	(q31_t)0x619fe918, (q31_t)0xad36edfc, (q31_t)0x619bd8aa, (q31_t)0xad32234b, (q31_t)0x6197c800, (q31_t)0xad2d58ce,
	(q31_t)0x6193b719, (q31_t)0xad288e85,
	(q31_t)0x618fa5f7, (q31_t)0xad23c46e, (q31_t)0x618b9498, (q31_t)0xad1efa8a, (q31_t)0x618782fd, (q31_t)0xad1a30d9,
	(q31_t)0x61837126, (q31_t)0xad15675c,
	(q31_t)0x617f5f12, (q31_t)0xad109e12, (q31_t)0x617b4cc3, (q31_t)0xad0bd4fb, (q31_t)0x61773a37, (q31_t)0xad070c17,
	(q31_t)0x61732770, (q31_t)0xad024366,
	(q31_t)0x616f146c, (q31_t)0xacfd7ae8, (q31_t)0x616b012c, (q31_t)0xacf8b29e, (q31_t)0x6166edb0, (q31_t)0xacf3ea87,
	(q31_t)0x6162d9f8, (q31_t)0xacef22a3,
	(q31_t)0x615ec603, (q31_t)0xacea5af2, (q31_t)0x615ab1d3, (q31_t)0xace59375, (q31_t)0x61569d67, (q31_t)0xace0cc2b,
	(q31_t)0x615288be, (q31_t)0xacdc0514,
	(q31_t)0x614e73da, (q31_t)0xacd73e30, (q31_t)0x614a5eba, (q31_t)0xacd27780, (q31_t)0x6146495d, (q31_t)0xaccdb103,
	(q31_t)0x614233c5, (q31_t)0xacc8eab9,
	(q31_t)0x613e1df0, (q31_t)0xacc424a3, (q31_t)0x613a07e0, (q31_t)0xacbf5ec0, (q31_t)0x6135f193, (q31_t)0xacba9910,
	(q31_t)0x6131db0b, (q31_t)0xacb5d394,
	(q31_t)0x612dc447, (q31_t)0xacb10e4b, (q31_t)0x6129ad46, (q31_t)0xacac4935, (q31_t)0x6125960a, (q31_t)0xaca78453,
	(q31_t)0x61217e92, (q31_t)0xaca2bfa4,
	(q31_t)0x611d66de, (q31_t)0xac9dfb29, (q31_t)0x61194eee, (q31_t)0xac9936e1, (q31_t)0x611536c2, (q31_t)0xac9472cd,
	(q31_t)0x61111e5b, (q31_t)0xac8faeec,
	(q31_t)0x610d05b7, (q31_t)0xac8aeb3e, (q31_t)0x6108ecd8, (q31_t)0xac8627c4, (q31_t)0x6104d3bc, (q31_t)0xac81647e,
	(q31_t)0x6100ba65, (q31_t)0xac7ca16b,
	(q31_t)0x60fca0d2, (q31_t)0xac77de8b, (q31_t)0x60f88703, (q31_t)0xac731bdf, (q31_t)0x60f46cf9, (q31_t)0xac6e5967,
	(q31_t)0x60f052b2, (q31_t)0xac699722,
	(q31_t)0x60ec3830, (q31_t)0xac64d510, (q31_t)0x60e81d72, (q31_t)0xac601333, (q31_t)0x60e40278, (q31_t)0xac5b5189,
	(q31_t)0x60dfe743, (q31_t)0xac569012,
	(q31_t)0x60dbcbd1, (q31_t)0xac51cecf, (q31_t)0x60d7b024, (q31_t)0xac4d0dc0, (q31_t)0x60d3943b, (q31_t)0xac484ce4,
	(q31_t)0x60cf7817, (q31_t)0xac438c3c,
	(q31_t)0x60cb5bb7, (q31_t)0xac3ecbc7, (q31_t)0x60c73f1b, (q31_t)0xac3a0b87, (q31_t)0x60c32243, (q31_t)0xac354b7a,
	(q31_t)0x60bf0530, (q31_t)0xac308ba0,
	(q31_t)0x60bae7e1, (q31_t)0xac2bcbfa, (q31_t)0x60b6ca56, (q31_t)0xac270c88, (q31_t)0x60b2ac8f, (q31_t)0xac224d4a,
	(q31_t)0x60ae8e8d, (q31_t)0xac1d8e40,
	(q31_t)0x60aa7050, (q31_t)0xac18cf69, (q31_t)0x60a651d7, (q31_t)0xac1410c6, (q31_t)0x60a23322, (q31_t)0xac0f5256,
	(q31_t)0x609e1431, (q31_t)0xac0a941b,
	(q31_t)0x6099f505, (q31_t)0xac05d613, (q31_t)0x6095d59d, (q31_t)0xac01183f, (q31_t)0x6091b5fa, (q31_t)0xabfc5a9f,
	(q31_t)0x608d961b, (q31_t)0xabf79d33,
	(q31_t)0x60897601, (q31_t)0xabf2dffb, (q31_t)0x608555ab, (q31_t)0xabee22f6, (q31_t)0x60813519, (q31_t)0xabe96625,
	(q31_t)0x607d144c, (q31_t)0xabe4a988,
	(q31_t)0x6078f344, (q31_t)0xabdfed1f, (q31_t)0x6074d200, (q31_t)0xabdb30ea, (q31_t)0x6070b080, (q31_t)0xabd674e9,
	(q31_t)0x606c8ec5, (q31_t)0xabd1b91c,
	(q31_t)0x60686ccf, (q31_t)0xabccfd83, (q31_t)0x60644a9d, (q31_t)0xabc8421d, (q31_t)0x6060282f, (q31_t)0xabc386ec,
	(q31_t)0x605c0587, (q31_t)0xabbecbee,
	(q31_t)0x6057e2a2, (q31_t)0xabba1125, (q31_t)0x6053bf82, (q31_t)0xabb5568f, (q31_t)0x604f9c27, (q31_t)0xabb09c2e,
	(q31_t)0x604b7891, (q31_t)0xababe200,
	(q31_t)0x604754bf, (q31_t)0xaba72807, (q31_t)0x604330b1, (q31_t)0xaba26e41, (q31_t)0x603f0c69, (q31_t)0xab9db4b0,
	(q31_t)0x603ae7e5, (q31_t)0xab98fb52,
	(q31_t)0x6036c325, (q31_t)0xab944229, (q31_t)0x60329e2a, (q31_t)0xab8f8934, (q31_t)0x602e78f4, (q31_t)0xab8ad073,
	(q31_t)0x602a5383, (q31_t)0xab8617e6,
	(q31_t)0x60262dd6, (q31_t)0xab815f8d, (q31_t)0x602207ee, (q31_t)0xab7ca768, (q31_t)0x601de1ca, (q31_t)0xab77ef77,
	(q31_t)0x6019bb6b, (q31_t)0xab7337bb,
	(q31_t)0x601594d1, (q31_t)0xab6e8032, (q31_t)0x60116dfc, (q31_t)0xab69c8de, (q31_t)0x600d46ec, (q31_t)0xab6511be,
	(q31_t)0x60091fa0, (q31_t)0xab605ad2,
	(q31_t)0x6004f819, (q31_t)0xab5ba41a, (q31_t)0x6000d057, (q31_t)0xab56ed97, (q31_t)0x5ffca859, (q31_t)0xab523748,
	(q31_t)0x5ff88021, (q31_t)0xab4d812d,
	(q31_t)0x5ff457ad, (q31_t)0xab48cb46, (q31_t)0x5ff02efe, (q31_t)0xab441593, (q31_t)0x5fec0613, (q31_t)0xab3f6015,
	(q31_t)0x5fe7dcee, (q31_t)0xab3aaacb,
	(q31_t)0x5fe3b38d, (q31_t)0xab35f5b5, (q31_t)0x5fdf89f2, (q31_t)0xab3140d4, (q31_t)0x5fdb601b, (q31_t)0xab2c8c27,
	(q31_t)0x5fd73609, (q31_t)0xab27d7ae,
	(q31_t)0x5fd30bbc, (q31_t)0xab23236a, (q31_t)0x5fcee133, (q31_t)0xab1e6f5a, (q31_t)0x5fcab670, (q31_t)0xab19bb7e,
	(q31_t)0x5fc68b72, (q31_t)0xab1507d7,
	(q31_t)0x5fc26038, (q31_t)0xab105464, (q31_t)0x5fbe34c4, (q31_t)0xab0ba125, (q31_t)0x5fba0914, (q31_t)0xab06ee1b,
	(q31_t)0x5fb5dd29, (q31_t)0xab023b46,
	(q31_t)0x5fb1b104, (q31_t)0xaafd88a4, (q31_t)0x5fad84a3, (q31_t)0xaaf8d637, (q31_t)0x5fa95807, (q31_t)0xaaf423ff,
	(q31_t)0x5fa52b31, (q31_t)0xaaef71fb,
	(q31_t)0x5fa0fe1f, (q31_t)0xaaeac02c, (q31_t)0x5f9cd0d2, (q31_t)0xaae60e91, (q31_t)0x5f98a34a, (q31_t)0xaae15d2a,
	(q31_t)0x5f947588, (q31_t)0xaadcabf8,
	(q31_t)0x5f90478a, (q31_t)0xaad7fafb, (q31_t)0x5f8c1951, (q31_t)0xaad34a32, (q31_t)0x5f87eade, (q31_t)0xaace999d,
	(q31_t)0x5f83bc2f, (q31_t)0xaac9e93e,
	(q31_t)0x5f7f8d46, (q31_t)0xaac53912, (q31_t)0x5f7b5e22, (q31_t)0xaac0891c, (q31_t)0x5f772ec2, (q31_t)0xaabbd959,
	(q31_t)0x5f72ff28, (q31_t)0xaab729cc,
	(q31_t)0x5f6ecf53, (q31_t)0xaab27a73, (q31_t)0x5f6a9f44, (q31_t)0xaaadcb4f, (q31_t)0x5f666ef9, (q31_t)0xaaa91c5f,
	(q31_t)0x5f623e73, (q31_t)0xaaa46da4,
	(q31_t)0x5f5e0db3, (q31_t)0xaa9fbf1e, (q31_t)0x5f59dcb8, (q31_t)0xaa9b10cc, (q31_t)0x5f55ab82, (q31_t)0xaa9662af,
	(q31_t)0x5f517a11, (q31_t)0xaa91b4c7,
	(q31_t)0x5f4d4865, (q31_t)0xaa8d0713, (q31_t)0x5f49167f, (q31_t)0xaa885994, (q31_t)0x5f44e45e, (q31_t)0xaa83ac4a,
	(q31_t)0x5f40b202, (q31_t)0xaa7eff34,
	(q31_t)0x5f3c7f6b, (q31_t)0xaa7a5253, (q31_t)0x5f384c9a, (q31_t)0xaa75a5a8, (q31_t)0x5f34198e, (q31_t)0xaa70f930,
	(q31_t)0x5f2fe647, (q31_t)0xaa6c4cee,
	(q31_t)0x5f2bb2c5, (q31_t)0xaa67a0e0, (q31_t)0x5f277f09, (q31_t)0xaa62f507, (q31_t)0x5f234b12, (q31_t)0xaa5e4963,
	(q31_t)0x5f1f16e0, (q31_t)0xaa599df4,
	(q31_t)0x5f1ae274, (q31_t)0xaa54f2ba, (q31_t)0x5f16adcc, (q31_t)0xaa5047b4, (q31_t)0x5f1278eb, (q31_t)0xaa4b9ce3,
	(q31_t)0x5f0e43ce, (q31_t)0xaa46f248,
	(q31_t)0x5f0a0e77, (q31_t)0xaa4247e1, (q31_t)0x5f05d8e6, (q31_t)0xaa3d9daf, (q31_t)0x5f01a31a, (q31_t)0xaa38f3b1,
	(q31_t)0x5efd6d13, (q31_t)0xaa3449e9,
	(q31_t)0x5ef936d1, (q31_t)0xaa2fa056, (q31_t)0x5ef50055, (q31_t)0xaa2af6f7, (q31_t)0x5ef0c99f, (q31_t)0xaa264dce,
	(q31_t)0x5eec92ae, (q31_t)0xaa21a4d9,
	(q31_t)0x5ee85b82, (q31_t)0xaa1cfc1a, (q31_t)0x5ee4241c, (q31_t)0xaa18538f, (q31_t)0x5edfec7b, (q31_t)0xaa13ab3a,
	(q31_t)0x5edbb49f, (q31_t)0xaa0f0319,
	(q31_t)0x5ed77c8a, (q31_t)0xaa0a5b2e, (q31_t)0x5ed34439, (q31_t)0xaa05b377, (q31_t)0x5ecf0baf, (q31_t)0xaa010bf6,
	(q31_t)0x5ecad2e9, (q31_t)0xa9fc64a9,
	(q31_t)0x5ec699e9, (q31_t)0xa9f7bd92, (q31_t)0x5ec260af, (q31_t)0xa9f316b0, (q31_t)0x5ebe273b, (q31_t)0xa9ee7002,
	(q31_t)0x5eb9ed8b, (q31_t)0xa9e9c98a,
	(q31_t)0x5eb5b3a2, (q31_t)0xa9e52347, (q31_t)0x5eb1797e, (q31_t)0xa9e07d39, (q31_t)0x5ead3f1f, (q31_t)0xa9dbd761,
	(q31_t)0x5ea90487, (q31_t)0xa9d731bd,
	(q31_t)0x5ea4c9b3, (q31_t)0xa9d28c4e, (q31_t)0x5ea08ea6, (q31_t)0xa9cde715, (q31_t)0x5e9c535e, (q31_t)0xa9c94211,
	(q31_t)0x5e9817dc, (q31_t)0xa9c49d42,
	(q31_t)0x5e93dc1f, (q31_t)0xa9bff8a8, (q31_t)0x5e8fa028, (q31_t)0xa9bb5444, (q31_t)0x5e8b63f7, (q31_t)0xa9b6b014,
	(q31_t)0x5e87278b, (q31_t)0xa9b20c1a,
	(q31_t)0x5e82eae5, (q31_t)0xa9ad6855, (q31_t)0x5e7eae05, (q31_t)0xa9a8c4c5, (q31_t)0x5e7a70ea, (q31_t)0xa9a4216b,
	(q31_t)0x5e763395, (q31_t)0xa99f7e46,
	(q31_t)0x5e71f606, (q31_t)0xa99adb56, (q31_t)0x5e6db83d, (q31_t)0xa996389b, (q31_t)0x5e697a39, (q31_t)0xa9919616,
	(q31_t)0x5e653bfc, (q31_t)0xa98cf3c6,
	(q31_t)0x5e60fd84, (q31_t)0xa98851ac, (q31_t)0x5e5cbed1, (q31_t)0xa983afc6, (q31_t)0x5e587fe5, (q31_t)0xa97f0e16,
	(q31_t)0x5e5440be, (q31_t)0xa97a6c9c,
	(q31_t)0x5e50015d, (q31_t)0xa975cb57, (q31_t)0x5e4bc1c2, (q31_t)0xa9712a47, (q31_t)0x5e4781ed, (q31_t)0xa96c896c,
	(q31_t)0x5e4341de, (q31_t)0xa967e8c7,
	(q31_t)0x5e3f0194, (q31_t)0xa9634858, (q31_t)0x5e3ac110, (q31_t)0xa95ea81d, (q31_t)0x5e368053, (q31_t)0xa95a0819,
	(q31_t)0x5e323f5b, (q31_t)0xa9556849,
	(q31_t)0x5e2dfe29, (q31_t)0xa950c8b0, (q31_t)0x5e29bcbd, (q31_t)0xa94c294b, (q31_t)0x5e257b17, (q31_t)0xa9478a1c,
	(q31_t)0x5e213936, (q31_t)0xa942eb23,
	(q31_t)0x5e1cf71c, (q31_t)0xa93e4c5f, (q31_t)0x5e18b4c8, (q31_t)0xa939add1, (q31_t)0x5e147239, (q31_t)0xa9350f78,
	(q31_t)0x5e102f71, (q31_t)0xa9307155,
	(q31_t)0x5e0bec6e, (q31_t)0xa92bd367, (q31_t)0x5e07a932, (q31_t)0xa92735af, (q31_t)0x5e0365bb, (q31_t)0xa922982c,
	(q31_t)0x5dff220b, (q31_t)0xa91dfadf,
	(q31_t)0x5dfade20, (q31_t)0xa9195dc7, (q31_t)0x5df699fc, (q31_t)0xa914c0e6, (q31_t)0x5df2559e, (q31_t)0xa9102439,
	(q31_t)0x5dee1105, (q31_t)0xa90b87c3,
	(q31_t)0x5de9cc33, (q31_t)0xa906eb82, (q31_t)0x5de58727, (q31_t)0xa9024f76, (q31_t)0x5de141e1, (q31_t)0xa8fdb3a1,
	(q31_t)0x5ddcfc61, (q31_t)0xa8f91801,
	(q31_t)0x5dd8b6a7, (q31_t)0xa8f47c97, (q31_t)0x5dd470b3, (q31_t)0xa8efe162, (q31_t)0x5dd02a85, (q31_t)0xa8eb4663,
	(q31_t)0x5dcbe41d, (q31_t)0xa8e6ab9a,
	(q31_t)0x5dc79d7c, (q31_t)0xa8e21106, (q31_t)0x5dc356a1, (q31_t)0xa8dd76a9, (q31_t)0x5dbf0f8c, (q31_t)0xa8d8dc81,
	(q31_t)0x5dbac83d, (q31_t)0xa8d4428f,
	(q31_t)0x5db680b4, (q31_t)0xa8cfa8d2, (q31_t)0x5db238f1, (q31_t)0xa8cb0f4b, (q31_t)0x5dadf0f5, (q31_t)0xa8c675fb,
	(q31_t)0x5da9a8bf, (q31_t)0xa8c1dce0,
	(q31_t)0x5da5604f, (q31_t)0xa8bd43fa, (q31_t)0x5da117a5, (q31_t)0xa8b8ab4b, (q31_t)0x5d9ccec2, (q31_t)0xa8b412d1,
	(q31_t)0x5d9885a5, (q31_t)0xa8af7a8e,
	(q31_t)0x5d943c4e, (q31_t)0xa8aae280, (q31_t)0x5d8ff2bd, (q31_t)0xa8a64aa8, (q31_t)0x5d8ba8f3, (q31_t)0xa8a1b306,
	(q31_t)0x5d875eef, (q31_t)0xa89d1b99,
	(q31_t)0x5d8314b1, (q31_t)0xa8988463, (q31_t)0x5d7eca39, (q31_t)0xa893ed63, (q31_t)0x5d7a7f88, (q31_t)0xa88f5698,
	(q31_t)0x5d76349d, (q31_t)0xa88ac004,
	(q31_t)0x5d71e979, (q31_t)0xa88629a5, (q31_t)0x5d6d9e1b, (q31_t)0xa881937c, (q31_t)0x5d695283, (q31_t)0xa87cfd8a,
	(q31_t)0x5d6506b2, (q31_t)0xa87867cd,
	(q31_t)0x5d60baa7, (q31_t)0xa873d246, (q31_t)0x5d5c6e62, (q31_t)0xa86f3cf6, (q31_t)0x5d5821e4, (q31_t)0xa86aa7db,
	(q31_t)0x5d53d52d, (q31_t)0xa86612f6,
	(q31_t)0x5d4f883b, (q31_t)0xa8617e48, (q31_t)0x5d4b3b10, (q31_t)0xa85ce9cf, (q31_t)0x5d46edac, (q31_t)0xa858558d,
	(q31_t)0x5d42a00e, (q31_t)0xa853c180,
	(q31_t)0x5d3e5237, (q31_t)0xa84f2daa, (q31_t)0x5d3a0426, (q31_t)0xa84a9a0a, (q31_t)0x5d35b5db, (q31_t)0xa84606a0,
	(q31_t)0x5d316757, (q31_t)0xa841736c,
	(q31_t)0x5d2d189a, (q31_t)0xa83ce06e, (q31_t)0x5d28c9a3, (q31_t)0xa8384da6, (q31_t)0x5d247a72, (q31_t)0xa833bb14,
	(q31_t)0x5d202b09, (q31_t)0xa82f28b9,
	(q31_t)0x5d1bdb65, (q31_t)0xa82a9693, (q31_t)0x5d178b89, (q31_t)0xa82604a4, (q31_t)0x5d133b72, (q31_t)0xa82172eb,
	(q31_t)0x5d0eeb23, (q31_t)0xa81ce169,
	(q31_t)0x5d0a9a9a, (q31_t)0xa818501c, (q31_t)0x5d0649d7, (q31_t)0xa813bf06, (q31_t)0x5d01f8dc, (q31_t)0xa80f2e26,
	(q31_t)0x5cfda7a7, (q31_t)0xa80a9d7c,
	(q31_t)0x5cf95638, (q31_t)0xa8060d08, (q31_t)0x5cf50490, (q31_t)0xa8017ccb, (q31_t)0x5cf0b2af, (q31_t)0xa7fcecc4,
	(q31_t)0x5cec6095, (q31_t)0xa7f85cf3,
	(q31_t)0x5ce80e41, (q31_t)0xa7f3cd59, (q31_t)0x5ce3bbb4, (q31_t)0xa7ef3df5, (q31_t)0x5cdf68ed, (q31_t)0xa7eaaec7,
	(q31_t)0x5cdb15ed, (q31_t)0xa7e61fd0,
	(q31_t)0x5cd6c2b5, (q31_t)0xa7e1910f, (q31_t)0x5cd26f42, (q31_t)0xa7dd0284, (q31_t)0x5cce1b97, (q31_t)0xa7d8742f,
	(q31_t)0x5cc9c7b2, (q31_t)0xa7d3e611,
	(q31_t)0x5cc57394, (q31_t)0xa7cf582a, (q31_t)0x5cc11f3d, (q31_t)0xa7caca79, (q31_t)0x5cbccaac, (q31_t)0xa7c63cfe,
	(q31_t)0x5cb875e3, (q31_t)0xa7c1afb9,
	(q31_t)0x5cb420e0, (q31_t)0xa7bd22ac, (q31_t)0x5cafcba4, (q31_t)0xa7b895d4, (q31_t)0x5cab762f, (q31_t)0xa7b40933,
	(q31_t)0x5ca72080, (q31_t)0xa7af7cc8,
	(q31_t)0x5ca2ca99, (q31_t)0xa7aaf094, (q31_t)0x5c9e7478, (q31_t)0xa7a66497, (q31_t)0x5c9a1e1e, (q31_t)0xa7a1d8d0,
	(q31_t)0x5c95c78b, (q31_t)0xa79d4d3f,
	(q31_t)0x5c9170bf, (q31_t)0xa798c1e5, (q31_t)0x5c8d19ba, (q31_t)0xa79436c1, (q31_t)0x5c88c27c, (q31_t)0xa78fabd4,
	(q31_t)0x5c846b05, (q31_t)0xa78b211e,
	(q31_t)0x5c801354, (q31_t)0xa786969e, (q31_t)0x5c7bbb6b, (q31_t)0xa7820c55, (q31_t)0x5c776348, (q31_t)0xa77d8242,
	(q31_t)0x5c730aed, (q31_t)0xa778f866,
	(q31_t)0x5c6eb258, (q31_t)0xa7746ec0, (q31_t)0x5c6a598b, (q31_t)0xa76fe551, (q31_t)0x5c660084, (q31_t)0xa76b5c19,
	(q31_t)0x5c61a745, (q31_t)0xa766d317,
	(q31_t)0x5c5d4dcc, (q31_t)0xa7624a4d, (q31_t)0x5c58f41a, (q31_t)0xa75dc1b8, (q31_t)0x5c549a30, (q31_t)0xa759395b,
	(q31_t)0x5c50400d, (q31_t)0xa754b134,
	(q31_t)0x5c4be5b0, (q31_t)0xa7502943, (q31_t)0x5c478b1b, (q31_t)0xa74ba18a, (q31_t)0x5c43304d, (q31_t)0xa7471a07,
	(q31_t)0x5c3ed545, (q31_t)0xa74292bb,
	(q31_t)0x5c3a7a05, (q31_t)0xa73e0ba5, (q31_t)0x5c361e8c, (q31_t)0xa73984c7, (q31_t)0x5c31c2db, (q31_t)0xa734fe1f,
	(q31_t)0x5c2d66f0, (q31_t)0xa73077ae,
	(q31_t)0x5c290acc, (q31_t)0xa72bf174, (q31_t)0x5c24ae70, (q31_t)0xa7276b70, (q31_t)0x5c2051db, (q31_t)0xa722e5a3,
	(q31_t)0x5c1bf50d, (q31_t)0xa71e600d,
	(q31_t)0x5c179806, (q31_t)0xa719daae, (q31_t)0x5c133ac6, (q31_t)0xa7155586, (q31_t)0x5c0edd4e, (q31_t)0xa710d095,
	(q31_t)0x5c0a7f9c, (q31_t)0xa70c4bda,
	(q31_t)0x5c0621b2, (q31_t)0xa707c757, (q31_t)0x5c01c38f, (q31_t)0xa703430a, (q31_t)0x5bfd6534, (q31_t)0xa6febef4,
	(q31_t)0x5bf906a0, (q31_t)0xa6fa3b15,
	(q31_t)0x5bf4a7d2, (q31_t)0xa6f5b76d, (q31_t)0x5bf048cd, (q31_t)0xa6f133fc, (q31_t)0x5bebe98e, (q31_t)0xa6ecb0c2,
	(q31_t)0x5be78a17, (q31_t)0xa6e82dbe,
	(q31_t)0x5be32a67, (q31_t)0xa6e3aaf2, (q31_t)0x5bdeca7f, (q31_t)0xa6df285d, (q31_t)0x5bda6a5d, (q31_t)0xa6daa5fe,
	(q31_t)0x5bd60a03, (q31_t)0xa6d623d7,
	(q31_t)0x5bd1a971, (q31_t)0xa6d1a1e7, (q31_t)0x5bcd48a6, (q31_t)0xa6cd202d, (q31_t)0x5bc8e7a2, (q31_t)0xa6c89eab,
	(q31_t)0x5bc48666, (q31_t)0xa6c41d60,
	(q31_t)0x5bc024f0, (q31_t)0xa6bf9c4b, (q31_t)0x5bbbc343, (q31_t)0xa6bb1b6e, (q31_t)0x5bb7615d, (q31_t)0xa6b69ac8,
	(q31_t)0x5bb2ff3e, (q31_t)0xa6b21a59,
	(q31_t)0x5bae9ce7, (q31_t)0xa6ad9a21, (q31_t)0x5baa3a57, (q31_t)0xa6a91a20, (q31_t)0x5ba5d78e, (q31_t)0xa6a49a56,
	(q31_t)0x5ba1748d, (q31_t)0xa6a01ac4,
	(q31_t)0x5b9d1154, (q31_t)0xa69b9b68, (q31_t)0x5b98ade2, (q31_t)0xa6971c44, (q31_t)0x5b944a37, (q31_t)0xa6929d57,
	(q31_t)0x5b8fe654, (q31_t)0xa68e1ea1,
	(q31_t)0x5b8b8239, (q31_t)0xa689a022, (q31_t)0x5b871de5, (q31_t)0xa68521da, (q31_t)0x5b82b958, (q31_t)0xa680a3ca,
	(q31_t)0x5b7e5493, (q31_t)0xa67c25f0,
	(q31_t)0x5b79ef96, (q31_t)0xa677a84e, (q31_t)0x5b758a60, (q31_t)0xa6732ae3, (q31_t)0x5b7124f2, (q31_t)0xa66eadb0,
	(q31_t)0x5b6cbf4c, (q31_t)0xa66a30b3,
	(q31_t)0x5b68596d, (q31_t)0xa665b3ee, (q31_t)0x5b63f355, (q31_t)0xa6613760, (q31_t)0x5b5f8d06, (q31_t)0xa65cbb0a,
	(q31_t)0x5b5b267e, (q31_t)0xa6583eeb,
	(q31_t)0x5b56bfbd, (q31_t)0xa653c303, (q31_t)0x5b5258c4, (q31_t)0xa64f4752, (q31_t)0x5b4df193, (q31_t)0xa64acbd9,
	(q31_t)0x5b498a2a, (q31_t)0xa6465097,
	(q31_t)0x5b452288, (q31_t)0xa641d58c, (q31_t)0x5b40baae, (q31_t)0xa63d5ab9, (q31_t)0x5b3c529c, (q31_t)0xa638e01d,
	(q31_t)0x5b37ea51, (q31_t)0xa63465b9,
	(q31_t)0x5b3381ce, (q31_t)0xa62feb8b, (q31_t)0x5b2f1913, (q31_t)0xa62b7196, (q31_t)0x5b2ab020, (q31_t)0xa626f7d7,
	(q31_t)0x5b2646f4, (q31_t)0xa6227e50,
	(q31_t)0x5b21dd90, (q31_t)0xa61e0501, (q31_t)0x5b1d73f4, (q31_t)0xa6198be9, (q31_t)0x5b190a20, (q31_t)0xa6151308,
	(q31_t)0x5b14a014, (q31_t)0xa6109a5f,
	(q31_t)0x5b1035cf, (q31_t)0xa60c21ee, (q31_t)0x5b0bcb52, (q31_t)0xa607a9b4, (q31_t)0x5b07609d, (q31_t)0xa60331b1,
	(q31_t)0x5b02f5b0, (q31_t)0xa5feb9e6,
	(q31_t)0x5afe8a8b, (q31_t)0xa5fa4252, (q31_t)0x5afa1f2e, (q31_t)0xa5f5caf6, (q31_t)0x5af5b398, (q31_t)0xa5f153d2,
	(q31_t)0x5af147ca, (q31_t)0xa5ecdce5,
	(q31_t)0x5aecdbc5, (q31_t)0xa5e8662f, (q31_t)0x5ae86f87, (q31_t)0xa5e3efb1, (q31_t)0x5ae40311, (q31_t)0xa5df796b,
	(q31_t)0x5adf9663, (q31_t)0xa5db035c,
	(q31_t)0x5adb297d, (q31_t)0xa5d68d85, (q31_t)0x5ad6bc5f, (q31_t)0xa5d217e6, (q31_t)0x5ad24f09, (q31_t)0xa5cda27e,
	(q31_t)0x5acde17b, (q31_t)0xa5c92d4e,
	(q31_t)0x5ac973b5, (q31_t)0xa5c4b855, (q31_t)0x5ac505b7, (q31_t)0xa5c04395, (q31_t)0x5ac09781, (q31_t)0xa5bbcf0b,
	(q31_t)0x5abc2912, (q31_t)0xa5b75aba,
	(q31_t)0x5ab7ba6c, (q31_t)0xa5b2e6a0, (q31_t)0x5ab34b8e, (q31_t)0xa5ae72be, (q31_t)0x5aaedc78, (q31_t)0xa5a9ff14,
	(q31_t)0x5aaa6d2b, (q31_t)0xa5a58ba1,
	(q31_t)0x5aa5fda5, (q31_t)0xa5a11866, (q31_t)0x5aa18de7, (q31_t)0xa59ca563, (q31_t)0x5a9d1df1, (q31_t)0xa5983297,
	(q31_t)0x5a98adc4, (q31_t)0xa593c004,
	(q31_t)0x5a943d5e, (q31_t)0xa58f4da8, (q31_t)0x5a8fccc1, (q31_t)0xa58adb84, (q31_t)0x5a8b5bec, (q31_t)0xa5866997,
	(q31_t)0x5a86eadf, (q31_t)0xa581f7e3,
	(q31_t)0x5a82799a, (q31_t)0xa57d8666, (q31_t)0x5a7e081d, (q31_t)0xa5791521, (q31_t)0x5a799669, (q31_t)0xa574a414,
	(q31_t)0x5a75247c, (q31_t)0xa570333f,
	(q31_t)0x5a70b258, (q31_t)0xa56bc2a2, (q31_t)0x5a6c3ffc, (q31_t)0xa567523c, (q31_t)0x5a67cd69, (q31_t)0xa562e20f,
	(q31_t)0x5a635a9d, (q31_t)0xa55e7219,
	(q31_t)0x5a5ee79a, (q31_t)0xa55a025b, (q31_t)0x5a5a745f, (q31_t)0xa55592d5, (q31_t)0x5a5600ec, (q31_t)0xa5512388,
	(q31_t)0x5a518d42, (q31_t)0xa54cb472,
	(q31_t)0x5a4d1960, (q31_t)0xa5484594, (q31_t)0x5a48a546, (q31_t)0xa543d6ee, (q31_t)0x5a4430f5, (q31_t)0xa53f687f,
	(q31_t)0x5a3fbc6b, (q31_t)0xa53afa49,
	(q31_t)0x5a3b47ab, (q31_t)0xa5368c4b, (q31_t)0x5a36d2b2, (q31_t)0xa5321e85, (q31_t)0x5a325d82, (q31_t)0xa52db0f7,
	(q31_t)0x5a2de81a, (q31_t)0xa52943a1,
	(q31_t)0x5a29727b, (q31_t)0xa524d683, (q31_t)0x5a24fca4, (q31_t)0xa520699d, (q31_t)0x5a208695, (q31_t)0xa51bfcef,
	(q31_t)0x5a1c104f, (q31_t)0xa5179079,
	(q31_t)0x5a1799d1, (q31_t)0xa513243b, (q31_t)0x5a13231b, (q31_t)0xa50eb836, (q31_t)0x5a0eac2e, (q31_t)0xa50a4c68,
	(q31_t)0x5a0a350a, (q31_t)0xa505e0d2,
	(q31_t)0x5a05bdae, (q31_t)0xa5017575, (q31_t)0x5a01461a, (q31_t)0xa4fd0a50, (q31_t)0x59fcce4f, (q31_t)0xa4f89f63,
	(q31_t)0x59f8564c, (q31_t)0xa4f434ae,
	(q31_t)0x59f3de12, (q31_t)0xa4efca31, (q31_t)0x59ef65a1, (q31_t)0xa4eb5fec, (q31_t)0x59eaecf8, (q31_t)0xa4e6f5e0,
	(q31_t)0x59e67417, (q31_t)0xa4e28c0c,
	(q31_t)0x59e1faff, (q31_t)0xa4de2270, (q31_t)0x59dd81b0, (q31_t)0xa4d9b90c, (q31_t)0x59d90829, (q31_t)0xa4d54fe0,
	(q31_t)0x59d48e6a, (q31_t)0xa4d0e6ed,
	(q31_t)0x59d01475, (q31_t)0xa4cc7e32, (q31_t)0x59cb9a47, (q31_t)0xa4c815af, (q31_t)0x59c71fe3, (q31_t)0xa4c3ad64,
	(q31_t)0x59c2a547, (q31_t)0xa4bf4552,
	(q31_t)0x59be2a74, (q31_t)0xa4badd78, (q31_t)0x59b9af69, (q31_t)0xa4b675d6, (q31_t)0x59b53427, (q31_t)0xa4b20e6d,
	(q31_t)0x59b0b8ae, (q31_t)0xa4ada73c,
	(q31_t)0x59ac3cfd, (q31_t)0xa4a94043, (q31_t)0x59a7c115, (q31_t)0xa4a4d982, (q31_t)0x59a344f6, (q31_t)0xa4a072fa,
	(q31_t)0x599ec8a0, (q31_t)0xa49c0cab,
	(q31_t)0x599a4c12, (q31_t)0xa497a693, (q31_t)0x5995cf4d, (q31_t)0xa49340b4, (q31_t)0x59915250, (q31_t)0xa48edb0e,
	(q31_t)0x598cd51d, (q31_t)0xa48a75a0,
	(q31_t)0x598857b2, (q31_t)0xa486106a, (q31_t)0x5983da10, (q31_t)0xa481ab6d, (q31_t)0x597f5c36, (q31_t)0xa47d46a8,
	(q31_t)0x597ade26, (q31_t)0xa478e21b,
	(q31_t)0x59765fde, (q31_t)0xa4747dc7, (q31_t)0x5971e15f, (q31_t)0xa47019ac, (q31_t)0x596d62a9, (q31_t)0xa46bb5c9,
	(q31_t)0x5968e3bc, (q31_t)0xa467521e,
	(q31_t)0x59646498, (q31_t)0xa462eeac, (q31_t)0x595fe53c, (q31_t)0xa45e8b73, (q31_t)0x595b65aa, (q31_t)0xa45a2872,
	(q31_t)0x5956e5e0, (q31_t)0xa455c5a9,
	(q31_t)0x595265df, (q31_t)0xa4516319, (q31_t)0x594de5a7, (q31_t)0xa44d00c2, (q31_t)0x59496538, (q31_t)0xa4489ea3,
	(q31_t)0x5944e492, (q31_t)0xa4443cbd,
	(q31_t)0x594063b5, (q31_t)0xa43fdb10, (q31_t)0x593be2a0, (q31_t)0xa43b799a, (q31_t)0x59376155, (q31_t)0xa437185e,
	(q31_t)0x5932dfd3, (q31_t)0xa432b75a,
	(q31_t)0x592e5e19, (q31_t)0xa42e568f, (q31_t)0x5929dc29, (q31_t)0xa429f5fd, (q31_t)0x59255a02, (q31_t)0xa42595a3,
	(q31_t)0x5920d7a3, (q31_t)0xa4213581,
	(q31_t)0x591c550e, (q31_t)0xa41cd599, (q31_t)0x5917d242, (q31_t)0xa41875e9, (q31_t)0x59134f3e, (q31_t)0xa4141672,
	(q31_t)0x590ecc04, (q31_t)0xa40fb733,
	(q31_t)0x590a4893, (q31_t)0xa40b582e, (q31_t)0x5905c4eb, (q31_t)0xa406f960, (q31_t)0x5901410c, (q31_t)0xa4029acc,
	(q31_t)0x58fcbcf6, (q31_t)0xa3fe3c71,
	(q31_t)0x58f838a9, (q31_t)0xa3f9de4e, (q31_t)0x58f3b426, (q31_t)0xa3f58064, (q31_t)0x58ef2f6b, (q31_t)0xa3f122b2,
	(q31_t)0x58eaaa7a, (q31_t)0xa3ecc53a,
	(q31_t)0x58e62552, (q31_t)0xa3e867fa, (q31_t)0x58e19ff3, (q31_t)0xa3e40af3, (q31_t)0x58dd1a5d, (q31_t)0xa3dfae25,
	(q31_t)0x58d89490, (q31_t)0xa3db5190,
	(q31_t)0x58d40e8c, (q31_t)0xa3d6f534, (q31_t)0x58cf8852, (q31_t)0xa3d29910, (q31_t)0x58cb01e1, (q31_t)0xa3ce3d25,
	(q31_t)0x58c67b39, (q31_t)0xa3c9e174,
	(q31_t)0x58c1f45b, (q31_t)0xa3c585fb, (q31_t)0x58bd6d45, (q31_t)0xa3c12abb, (q31_t)0x58b8e5f9, (q31_t)0xa3bccfb3,
	(q31_t)0x58b45e76, (q31_t)0xa3b874e5,
	(q31_t)0x58afd6bd, (q31_t)0xa3b41a50, (q31_t)0x58ab4ecc, (q31_t)0xa3afbff3, (q31_t)0x58a6c6a5, (q31_t)0xa3ab65d0,
	(q31_t)0x58a23e48, (q31_t)0xa3a70be6,
	(q31_t)0x589db5b3, (q31_t)0xa3a2b234, (q31_t)0x58992ce9, (q31_t)0xa39e58bb, (q31_t)0x5894a3e7, (q31_t)0xa399ff7c,
	(q31_t)0x58901aaf, (q31_t)0xa395a675,
	(q31_t)0x588b9140, (q31_t)0xa3914da8, (q31_t)0x5887079a, (q31_t)0xa38cf513, (q31_t)0x58827dbe, (q31_t)0xa3889cb8,
	(q31_t)0x587df3ab, (q31_t)0xa3844495,
	(q31_t)0x58796962, (q31_t)0xa37fecac, (q31_t)0x5874dee2, (q31_t)0xa37b94fb, (q31_t)0x5870542c, (q31_t)0xa3773d84,
	(q31_t)0x586bc93f, (q31_t)0xa372e646,
	(q31_t)0x58673e1b, (q31_t)0xa36e8f41, (q31_t)0x5862b2c1, (q31_t)0xa36a3875, (q31_t)0x585e2730, (q31_t)0xa365e1e2,
	(q31_t)0x58599b69, (q31_t)0xa3618b88,
	(q31_t)0x58550f6c, (q31_t)0xa35d3567, (q31_t)0x58508338, (q31_t)0xa358df80, (q31_t)0x584bf6cd, (q31_t)0xa35489d1,
	(q31_t)0x58476a2c, (q31_t)0xa350345c,
	(q31_t)0x5842dd54, (q31_t)0xa34bdf20, (q31_t)0x583e5047, (q31_t)0xa3478a1d, (q31_t)0x5839c302, (q31_t)0xa3433554,
	(q31_t)0x58353587, (q31_t)0xa33ee0c3,
	(q31_t)0x5830a7d6, (q31_t)0xa33a8c6c, (q31_t)0x582c19ef, (q31_t)0xa336384e, (q31_t)0x58278bd1, (q31_t)0xa331e469,
	(q31_t)0x5822fd7c, (q31_t)0xa32d90be,
	(q31_t)0x581e6ef1, (q31_t)0xa3293d4b, (q31_t)0x5819e030, (q31_t)0xa324ea13, (q31_t)0x58155139, (q31_t)0xa3209713,
	(q31_t)0x5810c20b, (q31_t)0xa31c444c,
	(q31_t)0x580c32a7, (q31_t)0xa317f1bf, (q31_t)0x5807a30d, (q31_t)0xa3139f6b, (q31_t)0x5803133c, (q31_t)0xa30f4d51,
	(q31_t)0x57fe8335, (q31_t)0xa30afb70,
	(q31_t)0x57f9f2f8, (q31_t)0xa306a9c8, (q31_t)0x57f56284, (q31_t)0xa3025859, (q31_t)0x57f0d1da, (q31_t)0xa2fe0724,
	(q31_t)0x57ec40fa, (q31_t)0xa2f9b629,
	(q31_t)0x57e7afe4, (q31_t)0xa2f56566, (q31_t)0x57e31e97, (q31_t)0xa2f114dd, (q31_t)0x57de8d15, (q31_t)0xa2ecc48e,
	(q31_t)0x57d9fb5c, (q31_t)0xa2e87477,
	(q31_t)0x57d5696d, (q31_t)0xa2e4249b, (q31_t)0x57d0d747, (q31_t)0xa2dfd4f7, (q31_t)0x57cc44ec, (q31_t)0xa2db858e,
	(q31_t)0x57c7b25a, (q31_t)0xa2d7365d,
	(q31_t)0x57c31f92, (q31_t)0xa2d2e766, (q31_t)0x57be8c94, (q31_t)0xa2ce98a9, (q31_t)0x57b9f960, (q31_t)0xa2ca4a25,
	(q31_t)0x57b565f6, (q31_t)0xa2c5fbda,
	(q31_t)0x57b0d256, (q31_t)0xa2c1adc9, (q31_t)0x57ac3e80, (q31_t)0xa2bd5ff2, (q31_t)0x57a7aa73, (q31_t)0xa2b91254,
	(q31_t)0x57a31631, (q31_t)0xa2b4c4f0,
	(q31_t)0x579e81b8, (q31_t)0xa2b077c5, (q31_t)0x5799ed0a, (q31_t)0xa2ac2ad3, (q31_t)0x57955825, (q31_t)0xa2a7de1c,
	(q31_t)0x5790c30a, (q31_t)0xa2a3919e,
	(q31_t)0x578c2dba, (q31_t)0xa29f4559, (q31_t)0x57879833, (q31_t)0xa29af94e, (q31_t)0x57830276, (q31_t)0xa296ad7d,
	(q31_t)0x577e6c84, (q31_t)0xa29261e5,
	(q31_t)0x5779d65b, (q31_t)0xa28e1687, (q31_t)0x57753ffc, (q31_t)0xa289cb63, (q31_t)0x5770a968, (q31_t)0xa2858078,
	(q31_t)0x576c129d, (q31_t)0xa28135c7,
	(q31_t)0x57677b9d, (q31_t)0xa27ceb4f, (q31_t)0x5762e467, (q31_t)0xa278a111, (q31_t)0x575e4cfa, (q31_t)0xa274570d,
	(q31_t)0x5759b558, (q31_t)0xa2700d43,
	(q31_t)0x57551d80, (q31_t)0xa26bc3b2, (q31_t)0x57508572, (q31_t)0xa2677a5b, (q31_t)0x574bed2f, (q31_t)0xa263313e,
	(q31_t)0x574754b5, (q31_t)0xa25ee85b,
	(q31_t)0x5742bc06, (q31_t)0xa25a9fb1, (q31_t)0x573e2320, (q31_t)0xa2565741, (q31_t)0x57398a05, (q31_t)0xa2520f0b,
	(q31_t)0x5734f0b5, (q31_t)0xa24dc70f,
	(q31_t)0x5730572e, (q31_t)0xa2497f4c, (q31_t)0x572bbd71, (q31_t)0xa24537c3, (q31_t)0x5727237f, (q31_t)0xa240f074,
	(q31_t)0x57228957, (q31_t)0xa23ca95f,
	(q31_t)0x571deefa, (q31_t)0xa2386284, (q31_t)0x57195466, (q31_t)0xa2341be3, (q31_t)0x5714b99d, (q31_t)0xa22fd57b,
	(q31_t)0x57101e9e, (q31_t)0xa22b8f4d,
	(q31_t)0x570b8369, (q31_t)0xa2274959, (q31_t)0x5706e7ff, (q31_t)0xa223039f, (q31_t)0x57024c5f, (q31_t)0xa21ebe1f,
	(q31_t)0x56fdb08a, (q31_t)0xa21a78d9,
	(q31_t)0x56f9147e, (q31_t)0xa21633cd, (q31_t)0x56f4783d, (q31_t)0xa211eefb, (q31_t)0x56efdbc7, (q31_t)0xa20daa62,
	(q31_t)0x56eb3f1a, (q31_t)0xa2096604,
	(q31_t)0x56e6a239, (q31_t)0xa20521e0, (q31_t)0x56e20521, (q31_t)0xa200ddf5, (q31_t)0x56dd67d4, (q31_t)0xa1fc9a45,
	(q31_t)0x56d8ca51, (q31_t)0xa1f856ce,
	(q31_t)0x56d42c99, (q31_t)0xa1f41392, (q31_t)0x56cf8eab, (q31_t)0xa1efd08f, (q31_t)0x56caf088, (q31_t)0xa1eb8dc7,
	(q31_t)0x56c6522f, (q31_t)0xa1e74b38,
	(q31_t)0x56c1b3a1, (q31_t)0xa1e308e4, (q31_t)0x56bd14dd, (q31_t)0xa1dec6ca, (q31_t)0x56b875e4, (q31_t)0xa1da84e9,
	(q31_t)0x56b3d6b5, (q31_t)0xa1d64343,
	(q31_t)0x56af3750, (q31_t)0xa1d201d7, (q31_t)0x56aa97b7, (q31_t)0xa1cdc0a5, (q31_t)0x56a5f7e7, (q31_t)0xa1c97fad,
	(q31_t)0x56a157e3, (q31_t)0xa1c53ef0,
	(q31_t)0x569cb7a8, (q31_t)0xa1c0fe6c, (q31_t)0x56981739, (q31_t)0xa1bcbe22, (q31_t)0x56937694, (q31_t)0xa1b87e13,
	(q31_t)0x568ed5b9, (q31_t)0xa1b43e3e,
	(q31_t)0x568a34a9, (q31_t)0xa1affea3, (q31_t)0x56859364, (q31_t)0xa1abbf42, (q31_t)0x5680f1ea, (q31_t)0xa1a7801b,
	(q31_t)0x567c503a, (q31_t)0xa1a3412f,
	(q31_t)0x5677ae54, (q31_t)0xa19f027c, (q31_t)0x56730c3a, (q31_t)0xa19ac404, (q31_t)0x566e69ea, (q31_t)0xa19685c7,
	(q31_t)0x5669c765, (q31_t)0xa19247c3,
	(q31_t)0x566524aa, (q31_t)0xa18e09fa, (q31_t)0x566081ba, (q31_t)0xa189cc6b, (q31_t)0x565bde95, (q31_t)0xa1858f16,
	(q31_t)0x56573b3b, (q31_t)0xa18151fb,
	(q31_t)0x565297ab, (q31_t)0xa17d151b, (q31_t)0x564df3e6, (q31_t)0xa178d875, (q31_t)0x56494fec, (q31_t)0xa1749c09,
	(q31_t)0x5644abbc, (q31_t)0xa1705fd8,
	(q31_t)0x56400758, (q31_t)0xa16c23e1, (q31_t)0x563b62be, (q31_t)0xa167e824, (q31_t)0x5636bdef, (q31_t)0xa163aca2,
	(q31_t)0x563218eb, (q31_t)0xa15f715a,
	(q31_t)0x562d73b2, (q31_t)0xa15b364d, (q31_t)0x5628ce43, (q31_t)0xa156fb79, (q31_t)0x5624289f, (q31_t)0xa152c0e1,
	(q31_t)0x561f82c7, (q31_t)0xa14e8682,
	(q31_t)0x561adcb9, (q31_t)0xa14a4c5e, (q31_t)0x56163676, (q31_t)0xa1461275, (q31_t)0x56118ffe, (q31_t)0xa141d8c5,
	(q31_t)0x560ce950, (q31_t)0xa13d9f51,
	(q31_t)0x5608426e, (q31_t)0xa1396617, (q31_t)0x56039b57, (q31_t)0xa1352d17, (q31_t)0x55fef40a, (q31_t)0xa130f451,
	(q31_t)0x55fa4c89, (q31_t)0xa12cbbc7,
	(q31_t)0x55f5a4d2, (q31_t)0xa1288376, (q31_t)0x55f0fce7, (q31_t)0xa1244b61, (q31_t)0x55ec54c6, (q31_t)0xa1201385,
	(q31_t)0x55e7ac71, (q31_t)0xa11bdbe4,
	(q31_t)0x55e303e6, (q31_t)0xa117a47e, (q31_t)0x55de5b27, (q31_t)0xa1136d52, (q31_t)0x55d9b232, (q31_t)0xa10f3661,
	(q31_t)0x55d50909, (q31_t)0xa10affab,
	(q31_t)0x55d05faa, (q31_t)0xa106c92f, (q31_t)0x55cbb617, (q31_t)0xa10292ed, (q31_t)0x55c70c4f, (q31_t)0xa0fe5ce6,
	(q31_t)0x55c26251, (q31_t)0xa0fa271a,
	(q31_t)0x55bdb81f, (q31_t)0xa0f5f189, (q31_t)0x55b90db8, (q31_t)0xa0f1bc32, (q31_t)0x55b4631d, (q31_t)0xa0ed8715,
	(q31_t)0x55afb84c, (q31_t)0xa0e95234,
	(q31_t)0x55ab0d46, (q31_t)0xa0e51d8c, (q31_t)0x55a6620c, (q31_t)0xa0e0e920, (q31_t)0x55a1b69d, (q31_t)0xa0dcb4ee,
	(q31_t)0x559d0af9, (q31_t)0xa0d880f7,
	(q31_t)0x55985f20, (q31_t)0xa0d44d3b, (q31_t)0x5593b312, (q31_t)0xa0d019b9, (q31_t)0x558f06d0, (q31_t)0xa0cbe672,
	(q31_t)0x558a5a58, (q31_t)0xa0c7b366,
	(q31_t)0x5585adad, (q31_t)0xa0c38095, (q31_t)0x558100cc, (q31_t)0xa0bf4dfe, (q31_t)0x557c53b6, (q31_t)0xa0bb1ba2,
	(q31_t)0x5577a66c, (q31_t)0xa0b6e981,
	(q31_t)0x5572f8ed, (q31_t)0xa0b2b79b, (q31_t)0x556e4b39, (q31_t)0xa0ae85ef, (q31_t)0x55699d51, (q31_t)0xa0aa547e,
	(q31_t)0x5564ef34, (q31_t)0xa0a62348,
	(q31_t)0x556040e2, (q31_t)0xa0a1f24d, (q31_t)0x555b925c, (q31_t)0xa09dc18d, (q31_t)0x5556e3a1, (q31_t)0xa0999107,
	(q31_t)0x555234b1, (q31_t)0xa09560bc,
	(q31_t)0x554d858d, (q31_t)0xa09130ad, (q31_t)0x5548d634, (q31_t)0xa08d00d8, (q31_t)0x554426a7, (q31_t)0xa088d13e,
	(q31_t)0x553f76e4, (q31_t)0xa084a1de,
	(q31_t)0x553ac6ee, (q31_t)0xa08072ba, (q31_t)0x553616c2, (q31_t)0xa07c43d1, (q31_t)0x55316663, (q31_t)0xa0781522,
	(q31_t)0x552cb5ce, (q31_t)0xa073e6af,
	(q31_t)0x55280505, (q31_t)0xa06fb876, (q31_t)0x55235408, (q31_t)0xa06b8a78, (q31_t)0x551ea2d6, (q31_t)0xa0675cb6,
	(q31_t)0x5519f16f, (q31_t)0xa0632f2e,
	(q31_t)0x55153fd4, (q31_t)0xa05f01e1, (q31_t)0x55108e05, (q31_t)0xa05ad4cf, (q31_t)0x550bdc01, (q31_t)0xa056a7f9,
	(q31_t)0x550729c9, (q31_t)0xa0527b5d,
	(q31_t)0x5502775c, (q31_t)0xa04e4efc, (q31_t)0x54fdc4ba, (q31_t)0xa04a22d7, (q31_t)0x54f911e5, (q31_t)0xa045f6ec,
	(q31_t)0x54f45edb, (q31_t)0xa041cb3c,
	(q31_t)0x54efab9c, (q31_t)0xa03d9fc8, (q31_t)0x54eaf829, (q31_t)0xa039748e, (q31_t)0x54e64482, (q31_t)0xa0354990,
	(q31_t)0x54e190a6, (q31_t)0xa0311ecd,
	(q31_t)0x54dcdc96, (q31_t)0xa02cf444, (q31_t)0x54d82852, (q31_t)0xa028c9f7, (q31_t)0x54d373d9, (q31_t)0xa0249fe5,
	(q31_t)0x54cebf2c, (q31_t)0xa020760e,
	(q31_t)0x54ca0a4b, (q31_t)0xa01c4c73, (q31_t)0x54c55535, (q31_t)0xa0182312, (q31_t)0x54c09feb, (q31_t)0xa013f9ed,
	(q31_t)0x54bbea6d, (q31_t)0xa00fd102,
	(q31_t)0x54b734ba, (q31_t)0xa00ba853, (q31_t)0x54b27ed3, (q31_t)0xa0077fdf, (q31_t)0x54adc8b8, (q31_t)0xa00357a7,
	(q31_t)0x54a91269, (q31_t)0x9fff2fa9,
	(q31_t)0x54a45be6, (q31_t)0x9ffb07e7, (q31_t)0x549fa52e, (q31_t)0x9ff6e060, (q31_t)0x549aee42, (q31_t)0x9ff2b914,
	(q31_t)0x54963722, (q31_t)0x9fee9204,
	(q31_t)0x54917fce, (q31_t)0x9fea6b2f, (q31_t)0x548cc845, (q31_t)0x9fe64495, (q31_t)0x54881089, (q31_t)0x9fe21e36,
	(q31_t)0x54835898, (q31_t)0x9fddf812,
	(q31_t)0x547ea073, (q31_t)0x9fd9d22a, (q31_t)0x5479e81a, (q31_t)0x9fd5ac7d, (q31_t)0x54752f8d, (q31_t)0x9fd1870c,
	(q31_t)0x547076cc, (q31_t)0x9fcd61d6,
	(q31_t)0x546bbdd7, (q31_t)0x9fc93cdb, (q31_t)0x546704ae, (q31_t)0x9fc5181b, (q31_t)0x54624b50, (q31_t)0x9fc0f397,
	(q31_t)0x545d91bf, (q31_t)0x9fbccf4f,
	(q31_t)0x5458d7f9, (q31_t)0x9fb8ab41, (q31_t)0x54541e00, (q31_t)0x9fb4876f, (q31_t)0x544f63d2, (q31_t)0x9fb063d9,
	(q31_t)0x544aa971, (q31_t)0x9fac407e,
	(q31_t)0x5445eedb, (q31_t)0x9fa81d5e, (q31_t)0x54413412, (q31_t)0x9fa3fa79, (q31_t)0x543c7914, (q31_t)0x9f9fd7d1,
	(q31_t)0x5437bde3, (q31_t)0x9f9bb563,
	(q31_t)0x5433027d, (q31_t)0x9f979331, (q31_t)0x542e46e4, (q31_t)0x9f93713b, (q31_t)0x54298b17, (q31_t)0x9f8f4f80,
	(q31_t)0x5424cf16, (q31_t)0x9f8b2e00,
	(q31_t)0x542012e1, (q31_t)0x9f870cbc, (q31_t)0x541b5678, (q31_t)0x9f82ebb4, (q31_t)0x541699db, (q31_t)0x9f7ecae7,
	(q31_t)0x5411dd0a, (q31_t)0x9f7aaa55,
	(q31_t)0x540d2005, (q31_t)0x9f7689ff, (q31_t)0x540862cd, (q31_t)0x9f7269e5, (q31_t)0x5403a561, (q31_t)0x9f6e4a06,
	(q31_t)0x53fee7c1, (q31_t)0x9f6a2a63,
	(q31_t)0x53fa29ed, (q31_t)0x9f660afb, (q31_t)0x53f56be5, (q31_t)0x9f61ebcf, (q31_t)0x53f0adaa, (q31_t)0x9f5dccde,
	(q31_t)0x53ebef3a, (q31_t)0x9f59ae29,
	(q31_t)0x53e73097, (q31_t)0x9f558fb0, (q31_t)0x53e271c0, (q31_t)0x9f517173, (q31_t)0x53ddb2b6, (q31_t)0x9f4d5371,
	(q31_t)0x53d8f378, (q31_t)0x9f4935aa,
	(q31_t)0x53d43406, (q31_t)0x9f45181f, (q31_t)0x53cf7460, (q31_t)0x9f40fad0, (q31_t)0x53cab486, (q31_t)0x9f3cddbd,
	(q31_t)0x53c5f479, (q31_t)0x9f38c0e5,
	(q31_t)0x53c13439, (q31_t)0x9f34a449, (q31_t)0x53bc73c4, (q31_t)0x9f3087e9, (q31_t)0x53b7b31c, (q31_t)0x9f2c6bc5,
	(q31_t)0x53b2f240, (q31_t)0x9f284fdc,
	(q31_t)0x53ae3131, (q31_t)0x9f24342f, (q31_t)0x53a96fee, (q31_t)0x9f2018bd, (q31_t)0x53a4ae77, (q31_t)0x9f1bfd88,
	(q31_t)0x539feccd, (q31_t)0x9f17e28e,
	(q31_t)0x539b2af0, (q31_t)0x9f13c7d0, (q31_t)0x539668de, (q31_t)0x9f0fad4e, (q31_t)0x5391a699, (q31_t)0x9f0b9307,
	(q31_t)0x538ce421, (q31_t)0x9f0778fd,
	(q31_t)0x53882175, (q31_t)0x9f035f2e, (q31_t)0x53835e95, (q31_t)0x9eff459b, (q31_t)0x537e9b82, (q31_t)0x9efb2c44,
	(q31_t)0x5379d83c, (q31_t)0x9ef71328,
	(q31_t)0x537514c2, (q31_t)0x9ef2fa49, (q31_t)0x53705114, (q31_t)0x9eeee1a5, (q31_t)0x536b8d33, (q31_t)0x9eeac93e,
	(q31_t)0x5366c91f, (q31_t)0x9ee6b112,
	(q31_t)0x536204d7, (q31_t)0x9ee29922, (q31_t)0x535d405c, (q31_t)0x9ede816e, (q31_t)0x53587bad, (q31_t)0x9eda69f6,
	(q31_t)0x5353b6cb, (q31_t)0x9ed652ba,
	(q31_t)0x534ef1b5, (q31_t)0x9ed23bb9, (q31_t)0x534a2c6c, (q31_t)0x9ece24f5, (q31_t)0x534566f0, (q31_t)0x9eca0e6d,
	(q31_t)0x5340a140, (q31_t)0x9ec5f820,
	(q31_t)0x533bdb5d, (q31_t)0x9ec1e210, (q31_t)0x53371547, (q31_t)0x9ebdcc3b, (q31_t)0x53324efd, (q31_t)0x9eb9b6a3,
	(q31_t)0x532d8880, (q31_t)0x9eb5a146,
	(q31_t)0x5328c1d0, (q31_t)0x9eb18c26, (q31_t)0x5323faec, (q31_t)0x9ead7742, (q31_t)0x531f33d5, (q31_t)0x9ea96299,
	(q31_t)0x531a6c8b, (q31_t)0x9ea54e2d,
	(q31_t)0x5315a50e, (q31_t)0x9ea139fd, (q31_t)0x5310dd5d, (q31_t)0x9e9d2608, (q31_t)0x530c1579, (q31_t)0x9e991250,
	(q31_t)0x53074d62, (q31_t)0x9e94fed4,
	(q31_t)0x53028518, (q31_t)0x9e90eb94, (q31_t)0x52fdbc9a, (q31_t)0x9e8cd890, (q31_t)0x52f8f3e9, (q31_t)0x9e88c5c9,
	(q31_t)0x52f42b05, (q31_t)0x9e84b33d,
	(q31_t)0x52ef61ee, (q31_t)0x9e80a0ee, (q31_t)0x52ea98a4, (q31_t)0x9e7c8eda, (q31_t)0x52e5cf27, (q31_t)0x9e787d03,
	(q31_t)0x52e10576, (q31_t)0x9e746b68,
	(q31_t)0x52dc3b92, (q31_t)0x9e705a09, (q31_t)0x52d7717b, (q31_t)0x9e6c48e7, (q31_t)0x52d2a732, (q31_t)0x9e683800,
	(q31_t)0x52cddcb5, (q31_t)0x9e642756,
	(q31_t)0x52c91204, (q31_t)0x9e6016e8, (q31_t)0x52c44721, (q31_t)0x9e5c06b6, (q31_t)0x52bf7c0b, (q31_t)0x9e57f6c0,
	(q31_t)0x52bab0c2, (q31_t)0x9e53e707,
	(q31_t)0x52b5e546, (q31_t)0x9e4fd78a, (q31_t)0x52b11996, (q31_t)0x9e4bc849, (q31_t)0x52ac4db4, (q31_t)0x9e47b944,
	(q31_t)0x52a7819f, (q31_t)0x9e43aa7c,
	(q31_t)0x52a2b556, (q31_t)0x9e3f9bf0, (q31_t)0x529de8db, (q31_t)0x9e3b8da0, (q31_t)0x52991c2d, (q31_t)0x9e377f8c,
	(q31_t)0x52944f4c, (q31_t)0x9e3371b5,
	(q31_t)0x528f8238, (q31_t)0x9e2f641b, (q31_t)0x528ab4f1, (q31_t)0x9e2b56bc, (q31_t)0x5285e777, (q31_t)0x9e27499a,
	(q31_t)0x528119ca, (q31_t)0x9e233cb4,
	(q31_t)0x527c4bea, (q31_t)0x9e1f300b, (q31_t)0x52777dd7, (q31_t)0x9e1b239e, (q31_t)0x5272af92, (q31_t)0x9e17176d,
	(q31_t)0x526de11a, (q31_t)0x9e130b79,
	(q31_t)0x5269126e, (q31_t)0x9e0effc1, (q31_t)0x52644390, (q31_t)0x9e0af446, (q31_t)0x525f7480, (q31_t)0x9e06e907,
	(q31_t)0x525aa53c, (q31_t)0x9e02de04,
	(q31_t)0x5255d5c5, (q31_t)0x9dfed33e, (q31_t)0x5251061c, (q31_t)0x9dfac8b4, (q31_t)0x524c3640, (q31_t)0x9df6be67,
	(q31_t)0x52476631, (q31_t)0x9df2b456,
	(q31_t)0x524295f0, (q31_t)0x9deeaa82, (q31_t)0x523dc57b, (q31_t)0x9deaa0ea, (q31_t)0x5238f4d4, (q31_t)0x9de6978f,
	(q31_t)0x523423fb, (q31_t)0x9de28e70,
	(q31_t)0x522f52ee, (q31_t)0x9dde858e, (q31_t)0x522a81af, (q31_t)0x9dda7ce9, (q31_t)0x5225b03d, (q31_t)0x9dd6747f,
	(q31_t)0x5220de99, (q31_t)0x9dd26c53,
	(q31_t)0x521c0cc2, (q31_t)0x9dce6463, (q31_t)0x52173ab8, (q31_t)0x9dca5caf, (q31_t)0x5212687b, (q31_t)0x9dc65539,
	(q31_t)0x520d960c, (q31_t)0x9dc24dfe,
	(q31_t)0x5208c36a, (q31_t)0x9dbe4701, (q31_t)0x5203f096, (q31_t)0x9dba4040, (q31_t)0x51ff1d8f, (q31_t)0x9db639bb,
	(q31_t)0x51fa4a56, (q31_t)0x9db23373,
	(q31_t)0x51f576ea, (q31_t)0x9dae2d68, (q31_t)0x51f0a34b, (q31_t)0x9daa279a, (q31_t)0x51ebcf7a, (q31_t)0x9da62208,
	(q31_t)0x51e6fb76, (q31_t)0x9da21cb2,
	(q31_t)0x51e22740, (q31_t)0x9d9e179a, (q31_t)0x51dd52d7, (q31_t)0x9d9a12be, (q31_t)0x51d87e3c, (q31_t)0x9d960e1f,
	(q31_t)0x51d3a96f, (q31_t)0x9d9209bd,
	(q31_t)0x51ced46e, (q31_t)0x9d8e0597, (q31_t)0x51c9ff3c, (q31_t)0x9d8a01ae, (q31_t)0x51c529d7, (q31_t)0x9d85fe02,
	(q31_t)0x51c0543f, (q31_t)0x9d81fa92,
	(q31_t)0x51bb7e75, (q31_t)0x9d7df75f, (q31_t)0x51b6a879, (q31_t)0x9d79f469, (q31_t)0x51b1d24a, (q31_t)0x9d75f1b0,
	(q31_t)0x51acfbe9, (q31_t)0x9d71ef34,
	(q31_t)0x51a82555, (q31_t)0x9d6decf4, (q31_t)0x51a34e8f, (q31_t)0x9d69eaf1, (q31_t)0x519e7797, (q31_t)0x9d65e92b,
	(q31_t)0x5199a06d, (q31_t)0x9d61e7a2,
	(q31_t)0x5194c910, (q31_t)0x9d5de656, (q31_t)0x518ff180, (q31_t)0x9d59e546, (q31_t)0x518b19bf, (q31_t)0x9d55e473,
	(q31_t)0x518641cb, (q31_t)0x9d51e3dd,
	(q31_t)0x518169a5, (q31_t)0x9d4de385, (q31_t)0x517c914c, (q31_t)0x9d49e368, (q31_t)0x5177b8c2, (q31_t)0x9d45e389,
	(q31_t)0x5172e005, (q31_t)0x9d41e3e7,
	(q31_t)0x516e0715, (q31_t)0x9d3de482, (q31_t)0x51692df4, (q31_t)0x9d39e559, (q31_t)0x516454a0, (q31_t)0x9d35e66e,
	(q31_t)0x515f7b1a, (q31_t)0x9d31e7bf,
	(q31_t)0x515aa162, (q31_t)0x9d2de94d, (q31_t)0x5155c778, (q31_t)0x9d29eb19, (q31_t)0x5150ed5c, (q31_t)0x9d25ed21,
	(q31_t)0x514c130d, (q31_t)0x9d21ef66,
	(q31_t)0x5147388c, (q31_t)0x9d1df1e9, (q31_t)0x51425dd9, (q31_t)0x9d19f4a8, (q31_t)0x513d82f4, (q31_t)0x9d15f7a4,
	(q31_t)0x5138a7dd, (q31_t)0x9d11fadd,
	(q31_t)0x5133cc94, (q31_t)0x9d0dfe54, (q31_t)0x512ef119, (q31_t)0x9d0a0207, (q31_t)0x512a156b, (q31_t)0x9d0605f7,
	(q31_t)0x5125398c, (q31_t)0x9d020a25,
	(q31_t)0x51205d7b, (q31_t)0x9cfe0e8f, (q31_t)0x511b8137, (q31_t)0x9cfa1337, (q31_t)0x5116a4c1, (q31_t)0x9cf6181c,
	(q31_t)0x5111c81a, (q31_t)0x9cf21d3d,
	(q31_t)0x510ceb40, (q31_t)0x9cee229c, (q31_t)0x51080e35, (q31_t)0x9cea2838, (q31_t)0x510330f7, (q31_t)0x9ce62e11,
	(q31_t)0x50fe5388, (q31_t)0x9ce23427,
	(q31_t)0x50f975e6, (q31_t)0x9cde3a7b, (q31_t)0x50f49813, (q31_t)0x9cda410b, (q31_t)0x50efba0d, (q31_t)0x9cd647d9,
	(q31_t)0x50eadbd6, (q31_t)0x9cd24ee4,
	(q31_t)0x50e5fd6d, (q31_t)0x9cce562c, (q31_t)0x50e11ed2, (q31_t)0x9cca5db1, (q31_t)0x50dc4005, (q31_t)0x9cc66573,
	(q31_t)0x50d76106, (q31_t)0x9cc26d73,
	(q31_t)0x50d281d5, (q31_t)0x9cbe75b0, (q31_t)0x50cda272, (q31_t)0x9cba7e2a, (q31_t)0x50c8c2de, (q31_t)0x9cb686e1,
	(q31_t)0x50c3e317, (q31_t)0x9cb28fd5,
	(q31_t)0x50bf031f, (q31_t)0x9cae9907, (q31_t)0x50ba22f5, (q31_t)0x9caaa276, (q31_t)0x50b5429a, (q31_t)0x9ca6ac23,
	(q31_t)0x50b0620c, (q31_t)0x9ca2b60c,
	(q31_t)0x50ab814d, (q31_t)0x9c9ec033, (q31_t)0x50a6a05c, (q31_t)0x9c9aca97, (q31_t)0x50a1bf39, (q31_t)0x9c96d539,
	(q31_t)0x509cdde4, (q31_t)0x9c92e017,
	(q31_t)0x5097fc5e, (q31_t)0x9c8eeb34, (q31_t)0x50931aa6, (q31_t)0x9c8af68d, (q31_t)0x508e38bd, (q31_t)0x9c870224,
	(q31_t)0x508956a1, (q31_t)0x9c830df8,
	(q31_t)0x50847454, (q31_t)0x9c7f1a0a, (q31_t)0x507f91d5, (q31_t)0x9c7b2659, (q31_t)0x507aaf25, (q31_t)0x9c7732e5,
	(q31_t)0x5075cc43, (q31_t)0x9c733faf,
	(q31_t)0x5070e92f, (q31_t)0x9c6f4cb6, (q31_t)0x506c05ea, (q31_t)0x9c6b59fa, (q31_t)0x50672273, (q31_t)0x9c67677c,
	(q31_t)0x50623ecb, (q31_t)0x9c63753c,
	(q31_t)0x505d5af1, (q31_t)0x9c5f8339, (q31_t)0x505876e5, (q31_t)0x9c5b9173, (q31_t)0x505392a8, (q31_t)0x9c579feb,
	(q31_t)0x504eae39, (q31_t)0x9c53aea0,
	(q31_t)0x5049c999, (q31_t)0x9c4fbd93, (q31_t)0x5044e4c7, (q31_t)0x9c4bccc3, (q31_t)0x503fffc4, (q31_t)0x9c47dc31,
	(q31_t)0x503b1a8f, (q31_t)0x9c43ebdc,
	(q31_t)0x50363529, (q31_t)0x9c3ffbc5, (q31_t)0x50314f91, (q31_t)0x9c3c0beb, (q31_t)0x502c69c8, (q31_t)0x9c381c4f,
	(q31_t)0x502783cd, (q31_t)0x9c342cf0,
	(q31_t)0x50229da1, (q31_t)0x9c303dcf, (q31_t)0x501db743, (q31_t)0x9c2c4eec, (q31_t)0x5018d0b4, (q31_t)0x9c286046,
	(q31_t)0x5013e9f4, (q31_t)0x9c2471de,
	(q31_t)0x500f0302, (q31_t)0x9c2083b3, (q31_t)0x500a1bdf, (q31_t)0x9c1c95c6, (q31_t)0x5005348a, (q31_t)0x9c18a816,
	(q31_t)0x50004d04, (q31_t)0x9c14baa4,
	(q31_t)0x4ffb654d, (q31_t)0x9c10cd70, (q31_t)0x4ff67d64, (q31_t)0x9c0ce07a, (q31_t)0x4ff1954b, (q31_t)0x9c08f3c1,
	(q31_t)0x4fecacff, (q31_t)0x9c050745,
	(q31_t)0x4fe7c483, (q31_t)0x9c011b08, (q31_t)0x4fe2dbd5, (q31_t)0x9bfd2f08, (q31_t)0x4fddf2f6, (q31_t)0x9bf94346,
	(q31_t)0x4fd909e5, (q31_t)0x9bf557c1,
	(q31_t)0x4fd420a4, (q31_t)0x9bf16c7a, (q31_t)0x4fcf3731, (q31_t)0x9bed8171, (q31_t)0x4fca4d8d, (q31_t)0x9be996a6,
	(q31_t)0x4fc563b7, (q31_t)0x9be5ac18,
	(q31_t)0x4fc079b1, (q31_t)0x9be1c1c8, (q31_t)0x4fbb8f79, (q31_t)0x9bddd7b6, (q31_t)0x4fb6a510, (q31_t)0x9bd9ede2,
	(q31_t)0x4fb1ba76, (q31_t)0x9bd6044b,
	(q31_t)0x4faccfab, (q31_t)0x9bd21af3, (q31_t)0x4fa7e4af, (q31_t)0x9bce31d8, (q31_t)0x4fa2f981, (q31_t)0x9bca48fa,
	(q31_t)0x4f9e0e22, (q31_t)0x9bc6605b,
	(q31_t)0x4f992293, (q31_t)0x9bc277fa, (q31_t)0x4f9436d2, (q31_t)0x9bbe8fd6, (q31_t)0x4f8f4ae0, (q31_t)0x9bbaa7f0,
	(q31_t)0x4f8a5ebd, (q31_t)0x9bb6c048,
	(q31_t)0x4f857269, (q31_t)0x9bb2d8de, (q31_t)0x4f8085e4, (q31_t)0x9baef1b2, (q31_t)0x4f7b992d, (q31_t)0x9bab0ac3,
	(q31_t)0x4f76ac46, (q31_t)0x9ba72413,
	(q31_t)0x4f71bf2e, (q31_t)0x9ba33da0, (q31_t)0x4f6cd1e5, (q31_t)0x9b9f576b, (q31_t)0x4f67e46a, (q31_t)0x9b9b7174,
	(q31_t)0x4f62f6bf, (q31_t)0x9b978bbc,
	(q31_t)0x4f5e08e3, (q31_t)0x9b93a641, (q31_t)0x4f591ad6, (q31_t)0x9b8fc104, (q31_t)0x4f542c98, (q31_t)0x9b8bdc05,
	(q31_t)0x4f4f3e29, (q31_t)0x9b87f744,
	(q31_t)0x4f4a4f89, (q31_t)0x9b8412c1, (q31_t)0x4f4560b8, (q31_t)0x9b802e7b, (q31_t)0x4f4071b6, (q31_t)0x9b7c4a74,
	(q31_t)0x4f3b8284, (q31_t)0x9b7866ab,
	(q31_t)0x4f369320, (q31_t)0x9b748320, (q31_t)0x4f31a38c, (q31_t)0x9b709fd3, (q31_t)0x4f2cb3c7, (q31_t)0x9b6cbcc4,
	(q31_t)0x4f27c3d1, (q31_t)0x9b68d9f3,
	(q31_t)0x4f22d3aa, (q31_t)0x9b64f760, (q31_t)0x4f1de352, (q31_t)0x9b61150b, (q31_t)0x4f18f2c9, (q31_t)0x9b5d32f4,
	(q31_t)0x4f140210, (q31_t)0x9b59511c,
	(q31_t)0x4f0f1126, (q31_t)0x9b556f81, (q31_t)0x4f0a200b, (q31_t)0x9b518e24, (q31_t)0x4f052ec0, (q31_t)0x9b4dad06,
	(q31_t)0x4f003d43, (q31_t)0x9b49cc26,
	(q31_t)0x4efb4b96, (q31_t)0x9b45eb83, (q31_t)0x4ef659b8, (q31_t)0x9b420b1f, (q31_t)0x4ef167aa, (q31_t)0x9b3e2af9,
	(q31_t)0x4eec756b, (q31_t)0x9b3a4b11,
	(q31_t)0x4ee782fb, (q31_t)0x9b366b68, (q31_t)0x4ee2905a, (q31_t)0x9b328bfc, (q31_t)0x4edd9d89, (q31_t)0x9b2eaccf,
	(q31_t)0x4ed8aa87, (q31_t)0x9b2acde0,
	(q31_t)0x4ed3b755, (q31_t)0x9b26ef2f, (q31_t)0x4ecec3f2, (q31_t)0x9b2310bc, (q31_t)0x4ec9d05e, (q31_t)0x9b1f3288,
	(q31_t)0x4ec4dc99, (q31_t)0x9b1b5492,
	(q31_t)0x4ebfe8a5, (q31_t)0x9b1776da, (q31_t)0x4ebaf47f, (q31_t)0x9b139960, (q31_t)0x4eb60029, (q31_t)0x9b0fbc24,
	(q31_t)0x4eb10ba2, (q31_t)0x9b0bdf27,
	(q31_t)0x4eac16eb, (q31_t)0x9b080268, (q31_t)0x4ea72203, (q31_t)0x9b0425e8, (q31_t)0x4ea22ceb, (q31_t)0x9b0049a5,
	(q31_t)0x4e9d37a3, (q31_t)0x9afc6da1,
	(q31_t)0x4e984229, (q31_t)0x9af891db, (q31_t)0x4e934c80, (q31_t)0x9af4b654, (q31_t)0x4e8e56a5, (q31_t)0x9af0db0b,
	(q31_t)0x4e89609b, (q31_t)0x9aed0000,
	(q31_t)0x4e846a60, (q31_t)0x9ae92533, (q31_t)0x4e7f73f4, (q31_t)0x9ae54aa5, (q31_t)0x4e7a7d58, (q31_t)0x9ae17056,
	(q31_t)0x4e75868c, (q31_t)0x9add9644,
	(q31_t)0x4e708f8f, (q31_t)0x9ad9bc71, (q31_t)0x4e6b9862, (q31_t)0x9ad5e2dd, (q31_t)0x4e66a105, (q31_t)0x9ad20987,
	(q31_t)0x4e61a977, (q31_t)0x9ace306f,
	(q31_t)0x4e5cb1b9, (q31_t)0x9aca5795, (q31_t)0x4e57b9ca, (q31_t)0x9ac67efb, (q31_t)0x4e52c1ab, (q31_t)0x9ac2a69e,
	(q31_t)0x4e4dc95c, (q31_t)0x9abece80,
	(q31_t)0x4e48d0dd, (q31_t)0x9abaf6a1, (q31_t)0x4e43d82d, (q31_t)0x9ab71eff, (q31_t)0x4e3edf4d, (q31_t)0x9ab3479d,
	(q31_t)0x4e39e63d, (q31_t)0x9aaf7079,
	(q31_t)0x4e34ecfc, (q31_t)0x9aab9993, (q31_t)0x4e2ff38b, (q31_t)0x9aa7c2ec, (q31_t)0x4e2af9ea, (q31_t)0x9aa3ec83,
	(q31_t)0x4e260019, (q31_t)0x9aa01659,
	(q31_t)0x4e210617, (q31_t)0x9a9c406e, (q31_t)0x4e1c0be6, (q31_t)0x9a986ac1, (q31_t)0x4e171184, (q31_t)0x9a949552,
	(q31_t)0x4e1216f2, (q31_t)0x9a90c022,
	(q31_t)0x4e0d1c30, (q31_t)0x9a8ceb31, (q31_t)0x4e08213e, (q31_t)0x9a89167e, (q31_t)0x4e03261b, (q31_t)0x9a85420a,
	(q31_t)0x4dfe2ac9, (q31_t)0x9a816dd5,
	(q31_t)0x4df92f46, (q31_t)0x9a7d99de, (q31_t)0x4df43393, (q31_t)0x9a79c625, (q31_t)0x4def37b0, (q31_t)0x9a75f2ac,
	(q31_t)0x4dea3b9d, (q31_t)0x9a721f71,
	(q31_t)0x4de53f5a, (q31_t)0x9a6e4c74, (q31_t)0x4de042e7, (q31_t)0x9a6a79b6, (q31_t)0x4ddb4644, (q31_t)0x9a66a737,
	(q31_t)0x4dd64971, (q31_t)0x9a62d4f7,
	(q31_t)0x4dd14c6e, (q31_t)0x9a5f02f5, (q31_t)0x4dcc4f3b, (q31_t)0x9a5b3132, (q31_t)0x4dc751d8, (q31_t)0x9a575fae,
	(q31_t)0x4dc25445, (q31_t)0x9a538e68,
	(q31_t)0x4dbd5682, (q31_t)0x9a4fbd61, (q31_t)0x4db8588f, (q31_t)0x9a4bec99, (q31_t)0x4db35a6c, (q31_t)0x9a481c0f,
	(q31_t)0x4dae5c19, (q31_t)0x9a444bc5,
	(q31_t)0x4da95d96, (q31_t)0x9a407bb9, (q31_t)0x4da45ee3, (q31_t)0x9a3cabeb, (q31_t)0x4d9f6001, (q31_t)0x9a38dc5d,
	(q31_t)0x4d9a60ee, (q31_t)0x9a350d0d,
	(q31_t)0x4d9561ac, (q31_t)0x9a313dfc, (q31_t)0x4d90623a, (q31_t)0x9a2d6f2a, (q31_t)0x4d8b6298, (q31_t)0x9a29a097,
	(q31_t)0x4d8662c6, (q31_t)0x9a25d243,
	(q31_t)0x4d8162c4, (q31_t)0x9a22042d, (q31_t)0x4d7c6293, (q31_t)0x9a1e3656, (q31_t)0x4d776231, (q31_t)0x9a1a68be,
	(q31_t)0x4d7261a0, (q31_t)0x9a169b65,
	(q31_t)0x4d6d60df, (q31_t)0x9a12ce4b, (q31_t)0x4d685fef, (q31_t)0x9a0f016f, (q31_t)0x4d635ece, (q31_t)0x9a0b34d3,
	(q31_t)0x4d5e5d7e, (q31_t)0x9a076875,
	(q31_t)0x4d595bfe, (q31_t)0x9a039c57, (q31_t)0x4d545a4f, (q31_t)0x99ffd077, (q31_t)0x4d4f5870, (q31_t)0x99fc04d6,
	(q31_t)0x4d4a5661, (q31_t)0x99f83974,
	(q31_t)0x4d455422, (q31_t)0x99f46e51, (q31_t)0x4d4051b4, (q31_t)0x99f0a36d, (q31_t)0x4d3b4f16, (q31_t)0x99ecd8c8,
	(q31_t)0x4d364c48, (q31_t)0x99e90e62,
	(q31_t)0x4d31494b, (q31_t)0x99e5443b, (q31_t)0x4d2c461e, (q31_t)0x99e17a53, (q31_t)0x4d2742c2, (q31_t)0x99ddb0aa,
	(q31_t)0x4d223f36, (q31_t)0x99d9e73f,
	(q31_t)0x4d1d3b7a, (q31_t)0x99d61e14, (q31_t)0x4d18378f, (q31_t)0x99d25528, (q31_t)0x4d133374, (q31_t)0x99ce8c7b,
	(q31_t)0x4d0e2f2a, (q31_t)0x99cac40d,
	(q31_t)0x4d092ab0, (q31_t)0x99c6fbde, (q31_t)0x4d042607, (q31_t)0x99c333ee, (q31_t)0x4cff212e, (q31_t)0x99bf6c3d,
	(q31_t)0x4cfa1c26, (q31_t)0x99bba4cb,
	(q31_t)0x4cf516ee, (q31_t)0x99b7dd99, (q31_t)0x4cf01187, (q31_t)0x99b416a5, (q31_t)0x4ceb0bf0, (q31_t)0x99b04ff0,
	(q31_t)0x4ce6062a, (q31_t)0x99ac897b,
	(q31_t)0x4ce10034, (q31_t)0x99a8c345, (q31_t)0x4cdbfa0f, (q31_t)0x99a4fd4d, (q31_t)0x4cd6f3bb, (q31_t)0x99a13795,
	(q31_t)0x4cd1ed37, (q31_t)0x999d721c,
	(q31_t)0x4ccce684, (q31_t)0x9999ace3, (q31_t)0x4cc7dfa1, (q31_t)0x9995e7e8, (q31_t)0x4cc2d88f, (q31_t)0x9992232d,
	(q31_t)0x4cbdd14e, (q31_t)0x998e5eb1,
	(q31_t)0x4cb8c9dd, (q31_t)0x998a9a74, (q31_t)0x4cb3c23d, (q31_t)0x9986d676, (q31_t)0x4caeba6e, (q31_t)0x998312b7,
	(q31_t)0x4ca9b26f, (q31_t)0x997f4f38,
	(q31_t)0x4ca4aa41, (q31_t)0x997b8bf8, (q31_t)0x4c9fa1e4, (q31_t)0x9977c8f7, (q31_t)0x4c9a9958, (q31_t)0x99740635,
	(q31_t)0x4c95909c, (q31_t)0x997043b2,
	(q31_t)0x4c9087b1, (q31_t)0x996c816f, (q31_t)0x4c8b7e97, (q31_t)0x9968bf6b, (q31_t)0x4c86754e, (q31_t)0x9964fda7,
	(q31_t)0x4c816bd5, (q31_t)0x99613c22,
	(q31_t)0x4c7c622d, (q31_t)0x995d7adc, (q31_t)0x4c775856, (q31_t)0x9959b9d5, (q31_t)0x4c724e50, (q31_t)0x9955f90d,
	(q31_t)0x4c6d441b, (q31_t)0x99523885,
	(q31_t)0x4c6839b7, (q31_t)0x994e783d, (q31_t)0x4c632f23, (q31_t)0x994ab833, (q31_t)0x4c5e2460, (q31_t)0x9946f869,
	(q31_t)0x4c59196f, (q31_t)0x994338df,
	(q31_t)0x4c540e4e, (q31_t)0x993f7993, (q31_t)0x4c4f02fe, (q31_t)0x993bba87, (q31_t)0x4c49f77f, (q31_t)0x9937fbbb,
	(q31_t)0x4c44ebd1, (q31_t)0x99343d2e,
	(q31_t)0x4c3fdff4, (q31_t)0x99307ee0, (q31_t)0x4c3ad3e7, (q31_t)0x992cc0d2, (q31_t)0x4c35c7ac, (q31_t)0x99290303,
	(q31_t)0x4c30bb42, (q31_t)0x99254574,
	(q31_t)0x4c2baea9, (q31_t)0x99218824, (q31_t)0x4c26a1e1, (q31_t)0x991dcb13, (q31_t)0x4c2194e9, (q31_t)0x991a0e42,
	(q31_t)0x4c1c87c3, (q31_t)0x991651b1,
	(q31_t)0x4c177a6e, (q31_t)0x9912955f, (q31_t)0x4c126cea, (q31_t)0x990ed94c, (q31_t)0x4c0d5f37, (q31_t)0x990b1d79,
	(q31_t)0x4c085156, (q31_t)0x990761e5,
	(q31_t)0x4c034345, (q31_t)0x9903a691, (q31_t)0x4bfe3505, (q31_t)0x98ffeb7d, (q31_t)0x4bf92697, (q31_t)0x98fc30a8,
	(q31_t)0x4bf417f9, (q31_t)0x98f87612,
	(q31_t)0x4bef092d, (q31_t)0x98f4bbbc, (q31_t)0x4be9fa32, (q31_t)0x98f101a6, (q31_t)0x4be4eb08, (q31_t)0x98ed47cf,
	(q31_t)0x4bdfdbaf, (q31_t)0x98e98e38,
	(q31_t)0x4bdacc28, (q31_t)0x98e5d4e0, (q31_t)0x4bd5bc72, (q31_t)0x98e21bc8, (q31_t)0x4bd0ac8d, (q31_t)0x98de62f0,
	(q31_t)0x4bcb9c79, (q31_t)0x98daaa57,
	(q31_t)0x4bc68c36, (q31_t)0x98d6f1fe, (q31_t)0x4bc17bc5, (q31_t)0x98d339e4, (q31_t)0x4bbc6b25, (q31_t)0x98cf820b,
	(q31_t)0x4bb75a56, (q31_t)0x98cbca70,
	(q31_t)0x4bb24958, (q31_t)0x98c81316, (q31_t)0x4bad382c, (q31_t)0x98c45bfb, (q31_t)0x4ba826d1, (q31_t)0x98c0a520,
	(q31_t)0x4ba31548, (q31_t)0x98bcee84,
	(q31_t)0x4b9e0390, (q31_t)0x98b93828, (q31_t)0x4b98f1a9, (q31_t)0x98b5820c, (q31_t)0x4b93df93, (q31_t)0x98b1cc30,
	(q31_t)0x4b8ecd4f, (q31_t)0x98ae1693,
	(q31_t)0x4b89badd, (q31_t)0x98aa6136, (q31_t)0x4b84a83b, (q31_t)0x98a6ac19, (q31_t)0x4b7f956b, (q31_t)0x98a2f73c,
	(q31_t)0x4b7a826d, (q31_t)0x989f429e,
	(q31_t)0x4b756f40, (q31_t)0x989b8e40, (q31_t)0x4b705be4, (q31_t)0x9897da22, (q31_t)0x4b6b485a, (q31_t)0x98942643,
	(q31_t)0x4b6634a2, (q31_t)0x989072a5,
	(q31_t)0x4b6120bb, (q31_t)0x988cbf46, (q31_t)0x4b5c0ca5, (q31_t)0x98890c27, (q31_t)0x4b56f861, (q31_t)0x98855948,
	(q31_t)0x4b51e3ee, (q31_t)0x9881a6a9,
	(q31_t)0x4b4ccf4d, (q31_t)0x987df449, (q31_t)0x4b47ba7e, (q31_t)0x987a422a, (q31_t)0x4b42a580, (q31_t)0x9876904a,
	(q31_t)0x4b3d9053, (q31_t)0x9872deaa,
	(q31_t)0x4b387af9, (q31_t)0x986f2d4a, (q31_t)0x4b336570, (q31_t)0x986b7c2a, (q31_t)0x4b2e4fb8, (q31_t)0x9867cb4a,
	(q31_t)0x4b2939d2, (q31_t)0x98641aa9,
	(q31_t)0x4b2423be, (q31_t)0x98606a49, (q31_t)0x4b1f0d7b, (q31_t)0x985cba28, (q31_t)0x4b19f70a, (q31_t)0x98590a48,
	(q31_t)0x4b14e06b, (q31_t)0x98555aa7,
	(q31_t)0x4b0fc99d, (q31_t)0x9851ab46, (q31_t)0x4b0ab2a1, (q31_t)0x984dfc26, (q31_t)0x4b059b77, (q31_t)0x984a4d45,
	(q31_t)0x4b00841f, (q31_t)0x98469ea4,
	(q31_t)0x4afb6c98, (q31_t)0x9842f043, (q31_t)0x4af654e3, (q31_t)0x983f4223, (q31_t)0x4af13d00, (q31_t)0x983b9442,
	(q31_t)0x4aec24ee, (q31_t)0x9837e6a1,
	(q31_t)0x4ae70caf, (q31_t)0x98343940, (q31_t)0x4ae1f441, (q31_t)0x98308c1f, (q31_t)0x4adcdba5, (q31_t)0x982cdf3f,
	(q31_t)0x4ad7c2da, (q31_t)0x9829329e,
	(q31_t)0x4ad2a9e2, (q31_t)0x9825863d, (q31_t)0x4acd90bb, (q31_t)0x9821da1d, (q31_t)0x4ac87767, (q31_t)0x981e2e3c,
	(q31_t)0x4ac35de4, (q31_t)0x981a829c,
	(q31_t)0x4abe4433, (q31_t)0x9816d73b, (q31_t)0x4ab92a54, (q31_t)0x98132c1b, (q31_t)0x4ab41046, (q31_t)0x980f813b,
	(q31_t)0x4aaef60b, (q31_t)0x980bd69b,
	(q31_t)0x4aa9dba2, (q31_t)0x98082c3b, (q31_t)0x4aa4c10b, (q31_t)0x9804821b, (q31_t)0x4a9fa645, (q31_t)0x9800d83c,
	(q31_t)0x4a9a8b52, (q31_t)0x97fd2e9c,
	(q31_t)0x4a957030, (q31_t)0x97f9853d, (q31_t)0x4a9054e1, (q31_t)0x97f5dc1e, (q31_t)0x4a8b3963, (q31_t)0x97f2333f,
	(q31_t)0x4a861db8, (q31_t)0x97ee8aa0,
	(q31_t)0x4a8101de, (q31_t)0x97eae242, (q31_t)0x4a7be5d7, (q31_t)0x97e73a23, (q31_t)0x4a76c9a2, (q31_t)0x97e39245,
	(q31_t)0x4a71ad3e, (q31_t)0x97dfeaa7,
	(q31_t)0x4a6c90ad, (q31_t)0x97dc4349, (q31_t)0x4a6773ee, (q31_t)0x97d89c2c, (q31_t)0x4a625701, (q31_t)0x97d4f54f,
	(q31_t)0x4a5d39e6, (q31_t)0x97d14eb2,
	(q31_t)0x4a581c9e, (q31_t)0x97cda855, (q31_t)0x4a52ff27, (q31_t)0x97ca0239, (q31_t)0x4a4de182, (q31_t)0x97c65c5c,
	(q31_t)0x4a48c3b0, (q31_t)0x97c2b6c1,
	(q31_t)0x4a43a5b0, (q31_t)0x97bf1165, (q31_t)0x4a3e8782, (q31_t)0x97bb6c4a, (q31_t)0x4a396926, (q31_t)0x97b7c76f,
	(q31_t)0x4a344a9d, (q31_t)0x97b422d4,
	(q31_t)0x4a2f2be6, (q31_t)0x97b07e7a, (q31_t)0x4a2a0d01, (q31_t)0x97acda60, (q31_t)0x4a24edee, (q31_t)0x97a93687,
	(q31_t)0x4a1fcead, (q31_t)0x97a592ed,
	(q31_t)0x4a1aaf3f, (q31_t)0x97a1ef94, (q31_t)0x4a158fa3, (q31_t)0x979e4c7c, (q31_t)0x4a106fda, (q31_t)0x979aa9a4,
	(q31_t)0x4a0b4fe2, (q31_t)0x9797070c,
	(q31_t)0x4a062fbd, (q31_t)0x979364b5, (q31_t)0x4a010f6b, (q31_t)0x978fc29e, (q31_t)0x49fbeeea, (q31_t)0x978c20c8,
	(q31_t)0x49f6ce3c, (q31_t)0x97887f32,
	(q31_t)0x49f1ad61, (q31_t)0x9784dddc, (q31_t)0x49ec8c57, (q31_t)0x97813cc7, (q31_t)0x49e76b21, (q31_t)0x977d9bf2,
	(q31_t)0x49e249bc, (q31_t)0x9779fb5e,
	(q31_t)0x49dd282a, (q31_t)0x97765b0a, (q31_t)0x49d8066b, (q31_t)0x9772baf7, (q31_t)0x49d2e47e, (q31_t)0x976f1b24,
	(q31_t)0x49cdc263, (q31_t)0x976b7b92,
	(q31_t)0x49c8a01b, (q31_t)0x9767dc41, (q31_t)0x49c37da5, (q31_t)0x97643d2f, (q31_t)0x49be5b02, (q31_t)0x97609e5f,
	(q31_t)0x49b93832, (q31_t)0x975cffcf,
	(q31_t)0x49b41533, (q31_t)0x9759617f, (q31_t)0x49aef208, (q31_t)0x9755c370, (q31_t)0x49a9ceaf, (q31_t)0x975225a1,
	(q31_t)0x49a4ab28, (q31_t)0x974e8813,
	(q31_t)0x499f8774, (q31_t)0x974aeac6, (q31_t)0x499a6393, (q31_t)0x97474db9, (q31_t)0x49953f84, (q31_t)0x9743b0ed,
	(q31_t)0x49901b48, (q31_t)0x97401462,
	(q31_t)0x498af6df, (q31_t)0x973c7817, (q31_t)0x4985d248, (q31_t)0x9738dc0d, (q31_t)0x4980ad84, (q31_t)0x97354043,
	(q31_t)0x497b8892, (q31_t)0x9731a4ba,
	(q31_t)0x49766373, (q31_t)0x972e0971, (q31_t)0x49713e27, (q31_t)0x972a6e6a, (q31_t)0x496c18ae, (q31_t)0x9726d3a3,
	(q31_t)0x4966f307, (q31_t)0x9723391c,
	(q31_t)0x4961cd33, (q31_t)0x971f9ed7, (q31_t)0x495ca732, (q31_t)0x971c04d2, (q31_t)0x49578103, (q31_t)0x97186b0d,
	(q31_t)0x49525aa7, (q31_t)0x9714d18a,
	(q31_t)0x494d341e, (q31_t)0x97113847, (q31_t)0x49480d68, (q31_t)0x970d9f45, (q31_t)0x4942e684, (q31_t)0x970a0683,
	(q31_t)0x493dbf74, (q31_t)0x97066e03,
	(q31_t)0x49389836, (q31_t)0x9702d5c3, (q31_t)0x493370cb, (q31_t)0x96ff3dc4, (q31_t)0x492e4933, (q31_t)0x96fba605,
	(q31_t)0x4929216e, (q31_t)0x96f80e88,
	(q31_t)0x4923f97b, (q31_t)0x96f4774b, (q31_t)0x491ed15c, (q31_t)0x96f0e04f, (q31_t)0x4919a90f, (q31_t)0x96ed4994,
	(q31_t)0x49148095, (q31_t)0x96e9b319,
	(q31_t)0x490f57ee, (q31_t)0x96e61ce0, (q31_t)0x490a2f1b, (q31_t)0x96e286e7, (q31_t)0x4905061a, (q31_t)0x96def12f,
	(q31_t)0x48ffdcec, (q31_t)0x96db5bb8,
	(q31_t)0x48fab391, (q31_t)0x96d7c682, (q31_t)0x48f58a09, (q31_t)0x96d4318d, (q31_t)0x48f06054, (q31_t)0x96d09cd8,
	(q31_t)0x48eb3672, (q31_t)0x96cd0865,
	(q31_t)0x48e60c62, (q31_t)0x96c97432, (q31_t)0x48e0e227, (q31_t)0x96c5e040, (q31_t)0x48dbb7be, (q31_t)0x96c24c8f,
	(q31_t)0x48d68d28, (q31_t)0x96beb91f,
	(q31_t)0x48d16265, (q31_t)0x96bb25f0, (q31_t)0x48cc3775, (q31_t)0x96b79302, (q31_t)0x48c70c59, (q31_t)0x96b40055,
	(q31_t)0x48c1e10f, (q31_t)0x96b06de9,
	(q31_t)0x48bcb599, (q31_t)0x96acdbbe, (q31_t)0x48b789f5, (q31_t)0x96a949d3, (q31_t)0x48b25e25, (q31_t)0x96a5b82a,
	(q31_t)0x48ad3228, (q31_t)0x96a226c2,
	(q31_t)0x48a805ff, (q31_t)0x969e959b, (q31_t)0x48a2d9a8, (q31_t)0x969b04b4, (q31_t)0x489dad25, (q31_t)0x9697740f,
	(q31_t)0x48988074, (q31_t)0x9693e3ab,
	(q31_t)0x48935397, (q31_t)0x96905388, (q31_t)0x488e268e, (q31_t)0x968cc3a5, (q31_t)0x4888f957, (q31_t)0x96893404,
	(q31_t)0x4883cbf4, (q31_t)0x9685a4a4,
	(q31_t)0x487e9e64, (q31_t)0x96821585, (q31_t)0x487970a7, (q31_t)0x967e86a7, (q31_t)0x487442be, (q31_t)0x967af80a,
	(q31_t)0x486f14a8, (q31_t)0x967769af,
	(q31_t)0x4869e665, (q31_t)0x9673db94, (q31_t)0x4864b7f5, (q31_t)0x96704dba, (q31_t)0x485f8959, (q31_t)0x966cc022,
	(q31_t)0x485a5a90, (q31_t)0x966932cb,
	(q31_t)0x48552b9b, (q31_t)0x9665a5b4, (q31_t)0x484ffc79, (q31_t)0x966218df, (q31_t)0x484acd2a, (q31_t)0x965e8c4b,
	(q31_t)0x48459daf, (q31_t)0x965afff9,
	(q31_t)0x48406e08, (q31_t)0x965773e7, (q31_t)0x483b3e33, (q31_t)0x9653e817, (q31_t)0x48360e32, (q31_t)0x96505c88,
	(q31_t)0x4830de05, (q31_t)0x964cd139,
	(q31_t)0x482badab, (q31_t)0x9649462d, (q31_t)0x48267d24, (q31_t)0x9645bb61, (q31_t)0x48214c71, (q31_t)0x964230d7,
	(q31_t)0x481c1b92, (q31_t)0x963ea68d,
	(q31_t)0x4816ea86, (q31_t)0x963b1c86, (q31_t)0x4811b94d, (q31_t)0x963792bf, (q31_t)0x480c87e8, (q31_t)0x96340939,
	(q31_t)0x48075657, (q31_t)0x96307ff5,
	(q31_t)0x48022499, (q31_t)0x962cf6f2, (q31_t)0x47fcf2af, (q31_t)0x96296e31, (q31_t)0x47f7c099, (q31_t)0x9625e5b0,
	(q31_t)0x47f28e56, (q31_t)0x96225d71,
	(q31_t)0x47ed5be6, (q31_t)0x961ed574, (q31_t)0x47e8294a, (q31_t)0x961b4db7, (q31_t)0x47e2f682, (q31_t)0x9617c63c,
	(q31_t)0x47ddc38e, (q31_t)0x96143f02,
	(q31_t)0x47d8906d, (q31_t)0x9610b80a, (q31_t)0x47d35d20, (q31_t)0x960d3153, (q31_t)0x47ce29a7, (q31_t)0x9609aadd,
	(q31_t)0x47c8f601, (q31_t)0x960624a9,
	(q31_t)0x47c3c22f, (q31_t)0x96029eb6, (q31_t)0x47be8e31, (q31_t)0x95ff1904, (q31_t)0x47b95a06, (q31_t)0x95fb9394,
	(q31_t)0x47b425af, (q31_t)0x95f80e65,
	(q31_t)0x47aef12c, (q31_t)0x95f48977, (q31_t)0x47a9bc7d, (q31_t)0x95f104cb, (q31_t)0x47a487a2, (q31_t)0x95ed8061,
	(q31_t)0x479f529a, (q31_t)0x95e9fc38,
	(q31_t)0x479a1d67, (q31_t)0x95e67850, (q31_t)0x4794e807, (q31_t)0x95e2f4a9, (q31_t)0x478fb27b, (q31_t)0x95df7145,
	(q31_t)0x478a7cc2, (q31_t)0x95dbee21,
	(q31_t)0x478546de, (q31_t)0x95d86b3f, (q31_t)0x478010cd, (q31_t)0x95d4e89f, (q31_t)0x477ada91, (q31_t)0x95d16640,
	(q31_t)0x4775a428, (q31_t)0x95cde423,
	(q31_t)0x47706d93, (q31_t)0x95ca6247, (q31_t)0x476b36d3, (q31_t)0x95c6e0ac, (q31_t)0x4765ffe6, (q31_t)0x95c35f53,
	(q31_t)0x4760c8cd, (q31_t)0x95bfde3c,
	(q31_t)0x475b9188, (q31_t)0x95bc5d66, (q31_t)0x47565a17, (q31_t)0x95b8dcd2, (q31_t)0x4751227a, (q31_t)0x95b55c7f,
	(q31_t)0x474beab1, (q31_t)0x95b1dc6e,
	(q31_t)0x4746b2bc, (q31_t)0x95ae5c9f, (q31_t)0x47417a9b, (q31_t)0x95aadd11, (q31_t)0x473c424e, (q31_t)0x95a75dc4,
	(q31_t)0x473709d5, (q31_t)0x95a3deb9,
	(q31_t)0x4731d131, (q31_t)0x95a05ff0, (q31_t)0x472c9860, (q31_t)0x959ce169, (q31_t)0x47275f63, (q31_t)0x95996323,
	(q31_t)0x4722263b, (q31_t)0x9595e51e,
	(q31_t)0x471cece7, (q31_t)0x9592675c, (q31_t)0x4717b367, (q31_t)0x958ee9db, (q31_t)0x471279ba, (q31_t)0x958b6c9b,
	(q31_t)0x470d3fe3, (q31_t)0x9587ef9e,
	(q31_t)0x470805df, (q31_t)0x958472e2, (q31_t)0x4702cbaf, (q31_t)0x9580f667, (q31_t)0x46fd9154, (q31_t)0x957d7a2f,
	(q31_t)0x46f856cd, (q31_t)0x9579fe38,
	(q31_t)0x46f31c1a, (q31_t)0x95768283, (q31_t)0x46ede13b, (q31_t)0x9573070f, (q31_t)0x46e8a631, (q31_t)0x956f8bdd,
	(q31_t)0x46e36afb, (q31_t)0x956c10ed,
	(q31_t)0x46de2f99, (q31_t)0x9568963f, (q31_t)0x46d8f40b, (q31_t)0x95651bd2, (q31_t)0x46d3b852, (q31_t)0x9561a1a8,
	(q31_t)0x46ce7c6d, (q31_t)0x955e27bf,
	(q31_t)0x46c9405c, (q31_t)0x955aae17, (q31_t)0x46c40420, (q31_t)0x955734b2, (q31_t)0x46bec7b8, (q31_t)0x9553bb8e,
	(q31_t)0x46b98b24, (q31_t)0x955042ac,
	(q31_t)0x46b44e65, (q31_t)0x954cca0c, (q31_t)0x46af117a, (q31_t)0x954951ae, (q31_t)0x46a9d464, (q31_t)0x9545d992,
	(q31_t)0x46a49722, (q31_t)0x954261b7,
	(q31_t)0x469f59b4, (q31_t)0x953eea1e, (q31_t)0x469a1c1b, (q31_t)0x953b72c7, (q31_t)0x4694de56, (q31_t)0x9537fbb2,
	(q31_t)0x468fa066, (q31_t)0x953484df,
	(q31_t)0x468a624a, (q31_t)0x95310e4e, (q31_t)0x46852403, (q31_t)0x952d97fe, (q31_t)0x467fe590, (q31_t)0x952a21f1,
	(q31_t)0x467aa6f2, (q31_t)0x9526ac25,
	(q31_t)0x46756828, (q31_t)0x9523369c, (q31_t)0x46702933, (q31_t)0x951fc154, (q31_t)0x466aea12, (q31_t)0x951c4c4e,
	(q31_t)0x4665aac6, (q31_t)0x9518d78a,
	(q31_t)0x46606b4e, (q31_t)0x95156308, (q31_t)0x465b2bab, (q31_t)0x9511eec8, (q31_t)0x4655ebdd, (q31_t)0x950e7aca,
	(q31_t)0x4650abe3, (q31_t)0x950b070e,
	(q31_t)0x464b6bbe, (q31_t)0x95079394, (q31_t)0x46462b6d, (q31_t)0x9504205c, (q31_t)0x4640eaf2, (q31_t)0x9500ad66,
	(q31_t)0x463baa4a, (q31_t)0x94fd3ab1,
	(q31_t)0x46366978, (q31_t)0x94f9c83f, (q31_t)0x4631287a, (q31_t)0x94f6560f, (q31_t)0x462be751, (q31_t)0x94f2e421,
	(q31_t)0x4626a5fd, (q31_t)0x94ef7275,
	(q31_t)0x4621647d, (q31_t)0x94ec010b, (q31_t)0x461c22d2, (q31_t)0x94e88fe3, (q31_t)0x4616e0fc, (q31_t)0x94e51efd,
	(q31_t)0x46119efa, (q31_t)0x94e1ae59,
	(q31_t)0x460c5cce, (q31_t)0x94de3df8, (q31_t)0x46071a76, (q31_t)0x94dacdd8, (q31_t)0x4601d7f3, (q31_t)0x94d75dfa,
	(q31_t)0x45fc9545, (q31_t)0x94d3ee5f,
	(q31_t)0x45f7526b, (q31_t)0x94d07f05, (q31_t)0x45f20f67, (q31_t)0x94cd0fee, (q31_t)0x45eccc37, (q31_t)0x94c9a119,
	(q31_t)0x45e788dc, (q31_t)0x94c63286,
	(q31_t)0x45e24556, (q31_t)0x94c2c435, (q31_t)0x45dd01a5, (q31_t)0x94bf5627, (q31_t)0x45d7bdc9, (q31_t)0x94bbe85a,
	(q31_t)0x45d279c2, (q31_t)0x94b87ad0,
	(q31_t)0x45cd358f, (q31_t)0x94b50d87, (q31_t)0x45c7f132, (q31_t)0x94b1a081, (q31_t)0x45c2acaa, (q31_t)0x94ae33be,
	(q31_t)0x45bd67f6, (q31_t)0x94aac73c,
	(q31_t)0x45b82318, (q31_t)0x94a75afd, (q31_t)0x45b2de0e, (q31_t)0x94a3eeff, (q31_t)0x45ad98da, (q31_t)0x94a08344,
	(q31_t)0x45a8537a, (q31_t)0x949d17cc,
	(q31_t)0x45a30df0, (q31_t)0x9499ac95, (q31_t)0x459dc83b, (q31_t)0x949641a1, (q31_t)0x4598825a, (q31_t)0x9492d6ef,
	(q31_t)0x45933c4f, (q31_t)0x948f6c7f,
	(q31_t)0x458df619, (q31_t)0x948c0252, (q31_t)0x4588afb8, (q31_t)0x94889867, (q31_t)0x4583692c, (q31_t)0x94852ebe,
	(q31_t)0x457e2275, (q31_t)0x9481c557,
	(q31_t)0x4578db93, (q31_t)0x947e5c33, (q31_t)0x45739487, (q31_t)0x947af351, (q31_t)0x456e4d4f, (q31_t)0x94778ab1,
	(q31_t)0x456905ed, (q31_t)0x94742254,
	(q31_t)0x4563be60, (q31_t)0x9470ba39, (q31_t)0x455e76a8, (q31_t)0x946d5260, (q31_t)0x45592ec6, (q31_t)0x9469eaca,
	(q31_t)0x4553e6b8, (q31_t)0x94668376,
	(q31_t)0x454e9e80, (q31_t)0x94631c65, (q31_t)0x4549561d, (q31_t)0x945fb596, (q31_t)0x45440d90, (q31_t)0x945c4f09,
	(q31_t)0x453ec4d7, (q31_t)0x9458e8bf,
	(q31_t)0x45397bf4, (q31_t)0x945582b7, (q31_t)0x453432e6, (q31_t)0x94521cf1, (q31_t)0x452ee9ae, (q31_t)0x944eb76e,
	(q31_t)0x4529a04b, (q31_t)0x944b522d,
	(q31_t)0x452456bd, (q31_t)0x9447ed2f, (q31_t)0x451f0d04, (q31_t)0x94448873, (q31_t)0x4519c321, (q31_t)0x944123fa,
	(q31_t)0x45147913, (q31_t)0x943dbfc3,
	(q31_t)0x450f2edb, (q31_t)0x943a5bcf, (q31_t)0x4509e478, (q31_t)0x9436f81d, (q31_t)0x450499eb, (q31_t)0x943394ad,
	(q31_t)0x44ff4f32, (q31_t)0x94303180,
	(q31_t)0x44fa0450, (q31_t)0x942cce96, (q31_t)0x44f4b943, (q31_t)0x94296bee, (q31_t)0x44ef6e0b, (q31_t)0x94260989,
	(q31_t)0x44ea22a9, (q31_t)0x9422a766,
	(q31_t)0x44e4d71c, (q31_t)0x941f4585, (q31_t)0x44df8b64, (q31_t)0x941be3e8, (q31_t)0x44da3f83, (q31_t)0x9418828c,
	(q31_t)0x44d4f376, (q31_t)0x94152174,
	(q31_t)0x44cfa740, (q31_t)0x9411c09e, (q31_t)0x44ca5adf, (q31_t)0x940e600a, (q31_t)0x44c50e53, (q31_t)0x940affb9,
	(q31_t)0x44bfc19d, (q31_t)0x94079fab,
	(q31_t)0x44ba74bd, (q31_t)0x94043fdf, (q31_t)0x44b527b2, (q31_t)0x9400e056, (q31_t)0x44afda7d, (q31_t)0x93fd810f,
	(q31_t)0x44aa8d1d, (q31_t)0x93fa220b,
	(q31_t)0x44a53f93, (q31_t)0x93f6c34a, (q31_t)0x449ff1df, (q31_t)0x93f364cb, (q31_t)0x449aa400, (q31_t)0x93f0068f,
	(q31_t)0x449555f7, (q31_t)0x93eca896,
	(q31_t)0x449007c4, (q31_t)0x93e94adf, (q31_t)0x448ab967, (q31_t)0x93e5ed6b, (q31_t)0x44856adf, (q31_t)0x93e2903a,
	(q31_t)0x44801c2d, (q31_t)0x93df334c,
	(q31_t)0x447acd50, (q31_t)0x93dbd6a0, (q31_t)0x44757e4a, (q31_t)0x93d87a36, (q31_t)0x44702f19, (q31_t)0x93d51e10,
	(q31_t)0x446adfbe, (q31_t)0x93d1c22c,
	(q31_t)0x44659039, (q31_t)0x93ce668b, (q31_t)0x44604089, (q31_t)0x93cb0b2d, (q31_t)0x445af0b0, (q31_t)0x93c7b011,
	(q31_t)0x4455a0ac, (q31_t)0x93c45539,
	(q31_t)0x4450507e, (q31_t)0x93c0faa3, (q31_t)0x444b0026, (q31_t)0x93bda04f, (q31_t)0x4445afa4, (q31_t)0x93ba463f,
	(q31_t)0x44405ef8, (q31_t)0x93b6ec71,
	(q31_t)0x443b0e21, (q31_t)0x93b392e6, (q31_t)0x4435bd21, (q31_t)0x93b0399e, (q31_t)0x44306bf6, (q31_t)0x93ace099,
	(q31_t)0x442b1aa2, (q31_t)0x93a987d6,
	(q31_t)0x4425c923, (q31_t)0x93a62f57, (q31_t)0x4420777b, (q31_t)0x93a2d71a, (q31_t)0x441b25a8, (q31_t)0x939f7f20,
	(q31_t)0x4415d3ab, (q31_t)0x939c2769,
	(q31_t)0x44108184, (q31_t)0x9398cff5, (q31_t)0x440b2f34, (q31_t)0x939578c3, (q31_t)0x4405dcb9, (q31_t)0x939221d5,
	(q31_t)0x44008a14, (q31_t)0x938ecb29,
	(q31_t)0x43fb3746, (q31_t)0x938b74c1, (q31_t)0x43f5e44d, (q31_t)0x93881e9b, (q31_t)0x43f0912b, (q31_t)0x9384c8b8,
	(q31_t)0x43eb3ddf, (q31_t)0x93817318,
	(q31_t)0x43e5ea68, (q31_t)0x937e1dbb, (q31_t)0x43e096c8, (q31_t)0x937ac8a1, (q31_t)0x43db42fe, (q31_t)0x937773ca,
	(q31_t)0x43d5ef0a, (q31_t)0x93741f35,
	(q31_t)0x43d09aed, (q31_t)0x9370cae4, (q31_t)0x43cb46a5, (q31_t)0x936d76d6, (q31_t)0x43c5f234, (q31_t)0x936a230a,
	(q31_t)0x43c09d99, (q31_t)0x9366cf82,
	(q31_t)0x43bb48d4, (q31_t)0x93637c3d, (q31_t)0x43b5f3e5, (q31_t)0x9360293a, (q31_t)0x43b09ecc, (q31_t)0x935cd67b,
	(q31_t)0x43ab498a, (q31_t)0x935983ff,
	(q31_t)0x43a5f41e, (q31_t)0x935631c5, (q31_t)0x43a09e89, (q31_t)0x9352dfcf, (q31_t)0x439b48c9, (q31_t)0x934f8e1c,
	(q31_t)0x4395f2e0, (q31_t)0x934c3cab,
	(q31_t)0x43909ccd, (q31_t)0x9348eb7e, (q31_t)0x438b4691, (q31_t)0x93459a94, (q31_t)0x4385f02a, (q31_t)0x934249ed,
	(q31_t)0x4380999b, (q31_t)0x933ef989,
	(q31_t)0x437b42e1, (q31_t)0x933ba968, (q31_t)0x4375ebfe, (q31_t)0x9338598a, (q31_t)0x437094f1, (q31_t)0x933509f0,
	(q31_t)0x436b3dbb, (q31_t)0x9331ba98,
	(q31_t)0x4365e65b, (q31_t)0x932e6b84, (q31_t)0x43608ed2, (q31_t)0x932b1cb2, (q31_t)0x435b371f, (q31_t)0x9327ce24,
	(q31_t)0x4355df42, (q31_t)0x93247fd9,
	(q31_t)0x4350873c, (q31_t)0x932131d1, (q31_t)0x434b2f0c, (q31_t)0x931de40c, (q31_t)0x4345d6b3, (q31_t)0x931a968b,
	(q31_t)0x43407e31, (q31_t)0x9317494c,
	(q31_t)0x433b2585, (q31_t)0x9313fc51, (q31_t)0x4335ccaf, (q31_t)0x9310af99, (q31_t)0x433073b0, (q31_t)0x930d6324,
	(q31_t)0x432b1a87, (q31_t)0x930a16f3,
	(q31_t)0x4325c135, (q31_t)0x9306cb04, (q31_t)0x432067ba, (q31_t)0x93037f59, (q31_t)0x431b0e15, (q31_t)0x930033f1,
	(q31_t)0x4315b447, (q31_t)0x92fce8cc,
	(q31_t)0x43105a50, (q31_t)0x92f99deb, (q31_t)0x430b002f, (q31_t)0x92f6534c, (q31_t)0x4305a5e5, (q31_t)0x92f308f1,
	(q31_t)0x43004b71, (q31_t)0x92efbeda,
	(q31_t)0x42faf0d4, (q31_t)0x92ec7505, (q31_t)0x42f5960e, (q31_t)0x92e92b74, (q31_t)0x42f03b1e, (q31_t)0x92e5e226,
	(q31_t)0x42eae005, (q31_t)0x92e2991c,
	(q31_t)0x42e584c3, (q31_t)0x92df5054, (q31_t)0x42e02958, (q31_t)0x92dc07d0, (q31_t)0x42dacdc3, (q31_t)0x92d8bf90,
	(q31_t)0x42d57205, (q31_t)0x92d57792,
	(q31_t)0x42d0161e, (q31_t)0x92d22fd9, (q31_t)0x42caba0e, (q31_t)0x92cee862, (q31_t)0x42c55dd4, (q31_t)0x92cba12f,
	(q31_t)0x42c00172, (q31_t)0x92c85a3f,
	(q31_t)0x42baa4e6, (q31_t)0x92c51392, (q31_t)0x42b54831, (q31_t)0x92c1cd29, (q31_t)0x42afeb53, (q31_t)0x92be8703,
	(q31_t)0x42aa8e4b, (q31_t)0x92bb4121,
	(q31_t)0x42a5311b, (q31_t)0x92b7fb82, (q31_t)0x429fd3c1, (q31_t)0x92b4b626, (q31_t)0x429a763f, (q31_t)0x92b1710e,
	(q31_t)0x42951893, (q31_t)0x92ae2c3a,
	(q31_t)0x428fbabe, (q31_t)0x92aae7a8, (q31_t)0x428a5cc0, (q31_t)0x92a7a35a, (q31_t)0x4284fe99, (q31_t)0x92a45f50,
	(q31_t)0x427fa049, (q31_t)0x92a11b89,
	(q31_t)0x427a41d0, (q31_t)0x929dd806, (q31_t)0x4274e32e, (q31_t)0x929a94c6, (q31_t)0x426f8463, (q31_t)0x929751c9,
	(q31_t)0x426a256f, (q31_t)0x92940f10,
	(q31_t)0x4264c653, (q31_t)0x9290cc9b, (q31_t)0x425f670d, (q31_t)0x928d8a69, (q31_t)0x425a079e, (q31_t)0x928a487a,
	(q31_t)0x4254a806, (q31_t)0x928706cf,
	(q31_t)0x424f4845, (q31_t)0x9283c568, (q31_t)0x4249e85c, (q31_t)0x92808444, (q31_t)0x42448849, (q31_t)0x927d4363,
	(q31_t)0x423f280e, (q31_t)0x927a02c7,
	(q31_t)0x4239c7aa, (q31_t)0x9276c26d, (q31_t)0x4234671d, (q31_t)0x92738258, (q31_t)0x422f0667, (q31_t)0x92704286,
	(q31_t)0x4229a588, (q31_t)0x926d02f7,
	(q31_t)0x42244481, (q31_t)0x9269c3ac, (q31_t)0x421ee350, (q31_t)0x926684a5, (q31_t)0x421981f7, (q31_t)0x926345e1,
	(q31_t)0x42142075, (q31_t)0x92600761,
	(q31_t)0x420ebecb, (q31_t)0x925cc924, (q31_t)0x42095cf7, (q31_t)0x92598b2b, (q31_t)0x4203fafb, (q31_t)0x92564d76,
	(q31_t)0x41fe98d6, (q31_t)0x92531005,
	(q31_t)0x41f93689, (q31_t)0x924fd2d7, (q31_t)0x41f3d413, (q31_t)0x924c95ec, (q31_t)0x41ee7174, (q31_t)0x92495946,
	(q31_t)0x41e90eac, (q31_t)0x92461ce3,
	(q31_t)0x41e3abbc, (q31_t)0x9242e0c4, (q31_t)0x41de48a3, (q31_t)0x923fa4e8, (q31_t)0x41d8e561, (q31_t)0x923c6950,
	(q31_t)0x41d381f7, (q31_t)0x92392dfc,
	(q31_t)0x41ce1e65, (q31_t)0x9235f2ec, (q31_t)0x41c8baa9, (q31_t)0x9232b81f, (q31_t)0x41c356c5, (q31_t)0x922f7d96,
	(q31_t)0x41bdf2b9, (q31_t)0x922c4351,
	(q31_t)0x41b88e84, (q31_t)0x9229094f, (q31_t)0x41b32a26, (q31_t)0x9225cf91, (q31_t)0x41adc5a0, (q31_t)0x92229617,
	(q31_t)0x41a860f1, (q31_t)0x921f5ce1,
	(q31_t)0x41a2fc1a, (q31_t)0x921c23ef, (q31_t)0x419d971b, (q31_t)0x9218eb40, (q31_t)0x419831f3, (q31_t)0x9215b2d5,
	(q31_t)0x4192cca2, (q31_t)0x92127aae,
	(q31_t)0x418d6729, (q31_t)0x920f42cb, (q31_t)0x41880188, (q31_t)0x920c0b2c, (q31_t)0x41829bbe, (q31_t)0x9208d3d0,
	(q31_t)0x417d35cb, (q31_t)0x92059cb8,
	(q31_t)0x4177cfb1, (q31_t)0x920265e4, (q31_t)0x4172696e, (q31_t)0x91ff2f54, (q31_t)0x416d0302, (q31_t)0x91fbf908,
	(q31_t)0x41679c6f, (q31_t)0x91f8c300,
	(q31_t)0x416235b2, (q31_t)0x91f58d3b, (q31_t)0x415ccece, (q31_t)0x91f257bb, (q31_t)0x415767c1, (q31_t)0x91ef227e,
	(q31_t)0x4152008c, (q31_t)0x91ebed85,
	(q31_t)0x414c992f, (q31_t)0x91e8b8d0, (q31_t)0x414731a9, (q31_t)0x91e5845f, (q31_t)0x4141c9fb, (q31_t)0x91e25032,
	(q31_t)0x413c6225, (q31_t)0x91df1c49,
	(q31_t)0x4136fa27, (q31_t)0x91dbe8a4, (q31_t)0x41319200, (q31_t)0x91d8b542, (q31_t)0x412c29b1, (q31_t)0x91d58225,
	(q31_t)0x4126c13a, (q31_t)0x91d24f4c,
	(q31_t)0x4121589b, (q31_t)0x91cf1cb6, (q31_t)0x411befd3, (q31_t)0x91cbea65, (q31_t)0x411686e4, (q31_t)0x91c8b857,
	(q31_t)0x41111dcc, (q31_t)0x91c5868e,
	(q31_t)0x410bb48c, (q31_t)0x91c25508, (q31_t)0x41064b24, (q31_t)0x91bf23c7, (q31_t)0x4100e194, (q31_t)0x91bbf2c9,
	(q31_t)0x40fb77dc, (q31_t)0x91b8c210,
	(q31_t)0x40f60dfb, (q31_t)0x91b5919a, (q31_t)0x40f0a3f3, (q31_t)0x91b26169, (q31_t)0x40eb39c3, (q31_t)0x91af317c,
	(q31_t)0x40e5cf6a, (q31_t)0x91ac01d2,
	(q31_t)0x40e064ea, (q31_t)0x91a8d26d, (q31_t)0x40dafa41, (q31_t)0x91a5a34c, (q31_t)0x40d58f71, (q31_t)0x91a2746f,
	(q31_t)0x40d02478, (q31_t)0x919f45d6,
	(q31_t)0x40cab958, (q31_t)0x919c1781, (q31_t)0x40c54e0f, (q31_t)0x9198e970, (q31_t)0x40bfe29f, (q31_t)0x9195bba3,
	(q31_t)0x40ba7706, (q31_t)0x91928e1a,
	(q31_t)0x40b50b46, (q31_t)0x918f60d6, (q31_t)0x40af9f5e, (q31_t)0x918c33d5, (q31_t)0x40aa334e, (q31_t)0x91890719,
	(q31_t)0x40a4c716, (q31_t)0x9185daa1,
	(q31_t)0x409f5ab6, (q31_t)0x9182ae6d, (q31_t)0x4099ee2e, (q31_t)0x917f827d, (q31_t)0x4094817f, (q31_t)0x917c56d1,
	(q31_t)0x408f14a7, (q31_t)0x91792b6a,
	(q31_t)0x4089a7a8, (q31_t)0x91760047, (q31_t)0x40843a81, (q31_t)0x9172d567, (q31_t)0x407ecd32, (q31_t)0x916faacc,
	(q31_t)0x40795fbc, (q31_t)0x916c8076,
	(q31_t)0x4073f21d, (q31_t)0x91695663, (q31_t)0x406e8457, (q31_t)0x91662c95, (q31_t)0x40691669, (q31_t)0x9163030b,
	(q31_t)0x4063a854, (q31_t)0x915fd9c5,
	(q31_t)0x405e3a16, (q31_t)0x915cb0c3, (q31_t)0x4058cbb1, (q31_t)0x91598806, (q31_t)0x40535d24, (q31_t)0x91565f8d,
	(q31_t)0x404dee70, (q31_t)0x91533758,
	(q31_t)0x40487f94, (q31_t)0x91500f67, (q31_t)0x40431090, (q31_t)0x914ce7bb, (q31_t)0x403da165, (q31_t)0x9149c053,
	(q31_t)0x40383212, (q31_t)0x9146992f,
	(q31_t)0x4032c297, (q31_t)0x91437250, (q31_t)0x402d52f5, (q31_t)0x91404bb5, (q31_t)0x4027e32b, (q31_t)0x913d255e,
	(q31_t)0x4022733a, (q31_t)0x9139ff4b,
	(q31_t)0x401d0321, (q31_t)0x9136d97d, (q31_t)0x401792e0, (q31_t)0x9133b3f3, (q31_t)0x40122278, (q31_t)0x91308eae,
	(q31_t)0x400cb1e9, (q31_t)0x912d69ad,
	(q31_t)0x40074132, (q31_t)0x912a44f0, (q31_t)0x4001d053, (q31_t)0x91272078, (q31_t)0x3ffc5f4d, (q31_t)0x9123fc44,
	(q31_t)0x3ff6ee1f, (q31_t)0x9120d854,
	(q31_t)0x3ff17cca, (q31_t)0x911db4a9, (q31_t)0x3fec0b4e, (q31_t)0x911a9142, (q31_t)0x3fe699aa, (q31_t)0x91176e1f,
	(q31_t)0x3fe127df, (q31_t)0x91144b41,
	(q31_t)0x3fdbb5ec, (q31_t)0x911128a8, (q31_t)0x3fd643d2, (q31_t)0x910e0653, (q31_t)0x3fd0d191, (q31_t)0x910ae442,
	(q31_t)0x3fcb5f28, (q31_t)0x9107c276,
	(q31_t)0x3fc5ec98, (q31_t)0x9104a0ee, (q31_t)0x3fc079e0, (q31_t)0x91017faa, (q31_t)0x3fbb0702, (q31_t)0x90fe5eab,
	(q31_t)0x3fb593fb, (q31_t)0x90fb3df1,
	(q31_t)0x3fb020ce, (q31_t)0x90f81d7b, (q31_t)0x3faaad79, (q31_t)0x90f4fd4a, (q31_t)0x3fa539fd, (q31_t)0x90f1dd5d,
	(q31_t)0x3f9fc65a, (q31_t)0x90eebdb4,
	(q31_t)0x3f9a5290, (q31_t)0x90eb9e50, (q31_t)0x3f94de9e, (q31_t)0x90e87f31, (q31_t)0x3f8f6a85, (q31_t)0x90e56056,
	(q31_t)0x3f89f645, (q31_t)0x90e241bf,
	(q31_t)0x3f8481dd, (q31_t)0x90df236e, (q31_t)0x3f7f0d4f, (q31_t)0x90dc0560, (q31_t)0x3f799899, (q31_t)0x90d8e798,
	(q31_t)0x3f7423bc, (q31_t)0x90d5ca13,
	(q31_t)0x3f6eaeb8, (q31_t)0x90d2acd4, (q31_t)0x3f69398d, (q31_t)0x90cf8fd9, (q31_t)0x3f63c43b, (q31_t)0x90cc7322,
	(q31_t)0x3f5e4ec2, (q31_t)0x90c956b1,
	(q31_t)0x3f58d921, (q31_t)0x90c63a83, (q31_t)0x3f53635a, (q31_t)0x90c31e9b, (q31_t)0x3f4ded6b, (q31_t)0x90c002f7,
	(q31_t)0x3f487755, (q31_t)0x90bce797,
	(q31_t)0x3f430119, (q31_t)0x90b9cc7d, (q31_t)0x3f3d8ab5, (q31_t)0x90b6b1a6, (q31_t)0x3f38142a, (q31_t)0x90b39715,
	(q31_t)0x3f329d79, (q31_t)0x90b07cc8,
	(q31_t)0x3f2d26a0, (q31_t)0x90ad62c0, (q31_t)0x3f27afa1, (q31_t)0x90aa48fd, (q31_t)0x3f22387a, (q31_t)0x90a72f7e,
	(q31_t)0x3f1cc12c, (q31_t)0x90a41644,
	(q31_t)0x3f1749b8, (q31_t)0x90a0fd4e, (q31_t)0x3f11d21d, (q31_t)0x909de49e, (q31_t)0x3f0c5a5a, (q31_t)0x909acc32,
	(q31_t)0x3f06e271, (q31_t)0x9097b40a,
	(q31_t)0x3f016a61, (q31_t)0x90949c28, (q31_t)0x3efbf22a, (q31_t)0x9091848a, (q31_t)0x3ef679cc, (q31_t)0x908e6d31,
	(q31_t)0x3ef10148, (q31_t)0x908b561c,
	(q31_t)0x3eeb889c, (q31_t)0x90883f4d, (q31_t)0x3ee60fca, (q31_t)0x908528c2, (q31_t)0x3ee096d1, (q31_t)0x9082127c,
	(q31_t)0x3edb1db1, (q31_t)0x907efc7a,
	(q31_t)0x3ed5a46b, (q31_t)0x907be6be, (q31_t)0x3ed02afd, (q31_t)0x9078d146, (q31_t)0x3ecab169, (q31_t)0x9075bc13,
	(q31_t)0x3ec537ae, (q31_t)0x9072a725,
	(q31_t)0x3ebfbdcd, (q31_t)0x906f927c, (q31_t)0x3eba43c4, (q31_t)0x906c7e17, (q31_t)0x3eb4c995, (q31_t)0x906969f8,
	(q31_t)0x3eaf4f40, (q31_t)0x9066561d,
	(q31_t)0x3ea9d4c3, (q31_t)0x90634287, (q31_t)0x3ea45a21, (q31_t)0x90602f35, (q31_t)0x3e9edf57, (q31_t)0x905d1c29,
	(q31_t)0x3e996467, (q31_t)0x905a0962,
	(q31_t)0x3e93e950, (q31_t)0x9056f6df, (q31_t)0x3e8e6e12, (q31_t)0x9053e4a1, (q31_t)0x3e88f2ae, (q31_t)0x9050d2a9,
	(q31_t)0x3e837724, (q31_t)0x904dc0f5,
	(q31_t)0x3e7dfb73, (q31_t)0x904aaf86, (q31_t)0x3e787f9b, (q31_t)0x90479e5c, (q31_t)0x3e73039d, (q31_t)0x90448d76,
	(q31_t)0x3e6d8778, (q31_t)0x90417cd6,
	(q31_t)0x3e680b2c, (q31_t)0x903e6c7b, (q31_t)0x3e628ebb, (q31_t)0x903b5c64, (q31_t)0x3e5d1222, (q31_t)0x90384c93,
	(q31_t)0x3e579564, (q31_t)0x90353d06,
	(q31_t)0x3e52187f, (q31_t)0x90322dbf, (q31_t)0x3e4c9b73, (q31_t)0x902f1ebc, (q31_t)0x3e471e41, (q31_t)0x902c0fff,
	(q31_t)0x3e41a0e8, (q31_t)0x90290186,
	(q31_t)0x3e3c2369, (q31_t)0x9025f352, (q31_t)0x3e36a5c4, (q31_t)0x9022e564, (q31_t)0x3e3127f9, (q31_t)0x901fd7ba,
	(q31_t)0x3e2baa07, (q31_t)0x901cca55,
	(q31_t)0x3e262bee, (q31_t)0x9019bd36, (q31_t)0x3e20adaf, (q31_t)0x9016b05b, (q31_t)0x3e1b2f4a, (q31_t)0x9013a3c5,
	(q31_t)0x3e15b0bf, (q31_t)0x90109775,
	(q31_t)0x3e10320d, (q31_t)0x900d8b69, (q31_t)0x3e0ab336, (q31_t)0x900a7fa3, (q31_t)0x3e053437, (q31_t)0x90077422,
	(q31_t)0x3dffb513, (q31_t)0x900468e5,
	(q31_t)0x3dfa35c8, (q31_t)0x90015dee, (q31_t)0x3df4b657, (q31_t)0x8ffe533c, (q31_t)0x3def36c0, (q31_t)0x8ffb48cf,
	(q31_t)0x3de9b703, (q31_t)0x8ff83ea7,
	(q31_t)0x3de4371f, (q31_t)0x8ff534c4, (q31_t)0x3ddeb716, (q31_t)0x8ff22b26, (q31_t)0x3dd936e6, (q31_t)0x8fef21ce,
	(q31_t)0x3dd3b690, (q31_t)0x8fec18ba,
	(q31_t)0x3dce3614, (q31_t)0x8fe90fec, (q31_t)0x3dc8b571, (q31_t)0x8fe60763, (q31_t)0x3dc334a9, (q31_t)0x8fe2ff1f,
	(q31_t)0x3dbdb3ba, (q31_t)0x8fdff720,
	(q31_t)0x3db832a6, (q31_t)0x8fdcef66, (q31_t)0x3db2b16b, (q31_t)0x8fd9e7f2, (q31_t)0x3dad300b, (q31_t)0x8fd6e0c2,
	(q31_t)0x3da7ae84, (q31_t)0x8fd3d9d8,
	(q31_t)0x3da22cd7, (q31_t)0x8fd0d333, (q31_t)0x3d9cab04, (q31_t)0x8fcdccd3, (q31_t)0x3d97290b, (q31_t)0x8fcac6b9,
	(q31_t)0x3d91a6ed, (q31_t)0x8fc7c0e3,
	(q31_t)0x3d8c24a8, (q31_t)0x8fc4bb53, (q31_t)0x3d86a23d, (q31_t)0x8fc1b608, (q31_t)0x3d811fac, (q31_t)0x8fbeb103,
	(q31_t)0x3d7b9cf6, (q31_t)0x8fbbac42,
	(q31_t)0x3d761a19, (q31_t)0x8fb8a7c7, (q31_t)0x3d709717, (q31_t)0x8fb5a391, (q31_t)0x3d6b13ee, (q31_t)0x8fb29fa0,
	(q31_t)0x3d6590a0, (q31_t)0x8faf9bf5,
	(q31_t)0x3d600d2c, (q31_t)0x8fac988f, (q31_t)0x3d5a8992, (q31_t)0x8fa9956e, (q31_t)0x3d5505d2, (q31_t)0x8fa69293,
	(q31_t)0x3d4f81ec, (q31_t)0x8fa38ffc,
	(q31_t)0x3d49fde1, (q31_t)0x8fa08dab, (q31_t)0x3d4479b0, (q31_t)0x8f9d8ba0, (q31_t)0x3d3ef559, (q31_t)0x8f9a89da,
	(q31_t)0x3d3970dc, (q31_t)0x8f978859,
	(q31_t)0x3d33ec39, (q31_t)0x8f94871d, (q31_t)0x3d2e6771, (q31_t)0x8f918627, (q31_t)0x3d28e282, (q31_t)0x8f8e8576,
	(q31_t)0x3d235d6f, (q31_t)0x8f8b850a,
	(q31_t)0x3d1dd835, (q31_t)0x8f8884e4, (q31_t)0x3d1852d6, (q31_t)0x8f858503, (q31_t)0x3d12cd51, (q31_t)0x8f828568,
	(q31_t)0x3d0d47a6, (q31_t)0x8f7f8612,
	(q31_t)0x3d07c1d6, (q31_t)0x8f7c8701, (q31_t)0x3d023be0, (q31_t)0x8f798836, (q31_t)0x3cfcb5c4, (q31_t)0x8f7689b0,
	(q31_t)0x3cf72f83, (q31_t)0x8f738b70,
	(q31_t)0x3cf1a91c, (q31_t)0x8f708d75, (q31_t)0x3cec2290, (q31_t)0x8f6d8fbf, (q31_t)0x3ce69bde, (q31_t)0x8f6a924f,
	(q31_t)0x3ce11507, (q31_t)0x8f679525,
	(q31_t)0x3cdb8e09, (q31_t)0x8f649840, (q31_t)0x3cd606e7, (q31_t)0x8f619ba0, (q31_t)0x3cd07f9f, (q31_t)0x8f5e9f46,
	(q31_t)0x3ccaf831, (q31_t)0x8f5ba331,
	(q31_t)0x3cc5709e, (q31_t)0x8f58a761, (q31_t)0x3cbfe8e5, (q31_t)0x8f55abd8, (q31_t)0x3cba6107, (q31_t)0x8f52b093,
	(q31_t)0x3cb4d904, (q31_t)0x8f4fb595,
	(q31_t)0x3caf50da, (q31_t)0x8f4cbadb, (q31_t)0x3ca9c88c, (q31_t)0x8f49c067, (q31_t)0x3ca44018, (q31_t)0x8f46c639,
	(q31_t)0x3c9eb77f, (q31_t)0x8f43cc50,
	(q31_t)0x3c992ec0, (q31_t)0x8f40d2ad, (q31_t)0x3c93a5dc, (q31_t)0x8f3dd950, (q31_t)0x3c8e1cd3, (q31_t)0x8f3ae038,
	(q31_t)0x3c8893a4, (q31_t)0x8f37e765,
	(q31_t)0x3c830a50, (q31_t)0x8f34eed8, (q31_t)0x3c7d80d6, (q31_t)0x8f31f691, (q31_t)0x3c77f737, (q31_t)0x8f2efe8f,
	(q31_t)0x3c726d73, (q31_t)0x8f2c06d3,
	(q31_t)0x3c6ce38a, (q31_t)0x8f290f5c, (q31_t)0x3c67597b, (q31_t)0x8f26182b, (q31_t)0x3c61cf48, (q31_t)0x8f232140,
	(q31_t)0x3c5c44ee, (q31_t)0x8f202a9a,
	(q31_t)0x3c56ba70, (q31_t)0x8f1d343a, (q31_t)0x3c512fcc, (q31_t)0x8f1a3e1f, (q31_t)0x3c4ba504, (q31_t)0x8f17484b,
	(q31_t)0x3c461a16, (q31_t)0x8f1452bb,
	(q31_t)0x3c408f03, (q31_t)0x8f115d72, (q31_t)0x3c3b03ca, (q31_t)0x8f0e686e, (q31_t)0x3c35786d, (q31_t)0x8f0b73b0,
	(q31_t)0x3c2fecea, (q31_t)0x8f087f37,
	(q31_t)0x3c2a6142, (q31_t)0x8f058b04, (q31_t)0x3c24d575, (q31_t)0x8f029717, (q31_t)0x3c1f4983, (q31_t)0x8effa370,
	(q31_t)0x3c19bd6c, (q31_t)0x8efcb00e,
	(q31_t)0x3c143130, (q31_t)0x8ef9bcf2, (q31_t)0x3c0ea4cf, (q31_t)0x8ef6ca1c, (q31_t)0x3c091849, (q31_t)0x8ef3d78b,
	(q31_t)0x3c038b9e, (q31_t)0x8ef0e540,
	(q31_t)0x3bfdfecd, (q31_t)0x8eedf33b, (q31_t)0x3bf871d8, (q31_t)0x8eeb017c, (q31_t)0x3bf2e4be, (q31_t)0x8ee81002,
	(q31_t)0x3bed577e, (q31_t)0x8ee51ece,
	(q31_t)0x3be7ca1a, (q31_t)0x8ee22de0, (q31_t)0x3be23c91, (q31_t)0x8edf3d38, (q31_t)0x3bdcaee3, (q31_t)0x8edc4cd5,
	(q31_t)0x3bd72110, (q31_t)0x8ed95cb8,
	(q31_t)0x3bd19318, (q31_t)0x8ed66ce1, (q31_t)0x3bcc04fb, (q31_t)0x8ed37d50, (q31_t)0x3bc676b9, (q31_t)0x8ed08e05,
	(q31_t)0x3bc0e853, (q31_t)0x8ecd9eff,
	(q31_t)0x3bbb59c7, (q31_t)0x8ecab040, (q31_t)0x3bb5cb17, (q31_t)0x8ec7c1c6, (q31_t)0x3bb03c42, (q31_t)0x8ec4d392,
	(q31_t)0x3baaad48, (q31_t)0x8ec1e5a4,
	(q31_t)0x3ba51e29, (q31_t)0x8ebef7fb, (q31_t)0x3b9f8ee5, (q31_t)0x8ebc0a99, (q31_t)0x3b99ff7d, (q31_t)0x8eb91d7c,
	(q31_t)0x3b946ff0, (q31_t)0x8eb630a6,
	(q31_t)0x3b8ee03e, (q31_t)0x8eb34415, (q31_t)0x3b895068, (q31_t)0x8eb057ca, (q31_t)0x3b83c06c, (q31_t)0x8ead6bc5,
	(q31_t)0x3b7e304c, (q31_t)0x8eaa8006,
	(q31_t)0x3b78a007, (q31_t)0x8ea7948c, (q31_t)0x3b730f9e, (q31_t)0x8ea4a959, (q31_t)0x3b6d7f10, (q31_t)0x8ea1be6c,
	(q31_t)0x3b67ee5d, (q31_t)0x8e9ed3c4,
	(q31_t)0x3b625d86, (q31_t)0x8e9be963, (q31_t)0x3b5ccc8a, (q31_t)0x8e98ff47, (q31_t)0x3b573b69, (q31_t)0x8e961571,
	(q31_t)0x3b51aa24, (q31_t)0x8e932be2,
	(q31_t)0x3b4c18ba, (q31_t)0x8e904298, (q31_t)0x3b46872c, (q31_t)0x8e8d5994, (q31_t)0x3b40f579, (q31_t)0x8e8a70d7,
	(q31_t)0x3b3b63a1, (q31_t)0x8e87885f,
	(q31_t)0x3b35d1a5, (q31_t)0x8e84a02d, (q31_t)0x3b303f84, (q31_t)0x8e81b841, (q31_t)0x3b2aad3f, (q31_t)0x8e7ed09b,
	(q31_t)0x3b251ad6, (q31_t)0x8e7be93c,
	(q31_t)0x3b1f8848, (q31_t)0x8e790222, (q31_t)0x3b19f595, (q31_t)0x8e761b4e, (q31_t)0x3b1462be, (q31_t)0x8e7334c1,
	(q31_t)0x3b0ecfc3, (q31_t)0x8e704e79,
	(q31_t)0x3b093ca3, (q31_t)0x8e6d6877, (q31_t)0x3b03a95e, (q31_t)0x8e6a82bc, (q31_t)0x3afe15f6, (q31_t)0x8e679d47,
	(q31_t)0x3af88269, (q31_t)0x8e64b817,
	(q31_t)0x3af2eeb7, (q31_t)0x8e61d32e, (q31_t)0x3aed5ae1, (q31_t)0x8e5eee8b, (q31_t)0x3ae7c6e7, (q31_t)0x8e5c0a2e,
	(q31_t)0x3ae232c9, (q31_t)0x8e592617,
	(q31_t)0x3adc9e86, (q31_t)0x8e564246, (q31_t)0x3ad70a1f, (q31_t)0x8e535ebb, (q31_t)0x3ad17593, (q31_t)0x8e507b76,
	(q31_t)0x3acbe0e3, (q31_t)0x8e4d9878,
	(q31_t)0x3ac64c0f, (q31_t)0x8e4ab5bf, (q31_t)0x3ac0b717, (q31_t)0x8e47d34d, (q31_t)0x3abb21fb, (q31_t)0x8e44f121,
	(q31_t)0x3ab58cba, (q31_t)0x8e420f3b,
	(q31_t)0x3aaff755, (q31_t)0x8e3f2d9b, (q31_t)0x3aaa61cc, (q31_t)0x8e3c4c41, (q31_t)0x3aa4cc1e, (q31_t)0x8e396b2e,
	(q31_t)0x3a9f364d, (q31_t)0x8e368a61,
	(q31_t)0x3a99a057, (q31_t)0x8e33a9da, (q31_t)0x3a940a3e, (q31_t)0x8e30c999, (q31_t)0x3a8e7400, (q31_t)0x8e2de99e,
	(q31_t)0x3a88dd9d, (q31_t)0x8e2b09e9,
	(q31_t)0x3a834717, (q31_t)0x8e282a7b, (q31_t)0x3a7db06d, (q31_t)0x8e254b53, (q31_t)0x3a78199f, (q31_t)0x8e226c71,
	(q31_t)0x3a7282ac, (q31_t)0x8e1f8dd6,
	(q31_t)0x3a6ceb96, (q31_t)0x8e1caf80, (q31_t)0x3a67545b, (q31_t)0x8e19d171, (q31_t)0x3a61bcfd, (q31_t)0x8e16f3a9,
	(q31_t)0x3a5c257a, (q31_t)0x8e141626,
	(q31_t)0x3a568dd4, (q31_t)0x8e1138ea, (q31_t)0x3a50f609, (q31_t)0x8e0e5bf4, (q31_t)0x3a4b5e1b, (q31_t)0x8e0b7f44,
	(q31_t)0x3a45c608, (q31_t)0x8e08a2db,
	(q31_t)0x3a402dd2, (q31_t)0x8e05c6b7, (q31_t)0x3a3a9577, (q31_t)0x8e02eadb, (q31_t)0x3a34fcf9, (q31_t)0x8e000f44,
	(q31_t)0x3a2f6457, (q31_t)0x8dfd33f4,
	(q31_t)0x3a29cb91, (q31_t)0x8dfa58ea, (q31_t)0x3a2432a7, (q31_t)0x8df77e27, (q31_t)0x3a1e9999, (q31_t)0x8df4a3a9,
	(q31_t)0x3a190068, (q31_t)0x8df1c973,
	(q31_t)0x3a136712, (q31_t)0x8deeef82, (q31_t)0x3a0dcd99, (q31_t)0x8dec15d8, (q31_t)0x3a0833fc, (q31_t)0x8de93c74,
	(q31_t)0x3a029a3b, (q31_t)0x8de66357,
	(q31_t)0x39fd0056, (q31_t)0x8de38a80, (q31_t)0x39f7664e, (q31_t)0x8de0b1ef, (q31_t)0x39f1cc21, (q31_t)0x8dddd9a5,
	(q31_t)0x39ec31d1, (q31_t)0x8ddb01a1,
	(q31_t)0x39e6975e, (q31_t)0x8dd829e4, (q31_t)0x39e0fcc6, (q31_t)0x8dd5526d, (q31_t)0x39db620b, (q31_t)0x8dd27b3c,
	(q31_t)0x39d5c72c, (q31_t)0x8dcfa452,
	(q31_t)0x39d02c2a, (q31_t)0x8dcccdaf, (q31_t)0x39ca9104, (q31_t)0x8dc9f751, (q31_t)0x39c4f5ba, (q31_t)0x8dc7213b,
	(q31_t)0x39bf5a4d, (q31_t)0x8dc44b6a,
	(q31_t)0x39b9bebc, (q31_t)0x8dc175e0, (q31_t)0x39b42307, (q31_t)0x8dbea09d, (q31_t)0x39ae872f, (q31_t)0x8dbbcba0,
	(q31_t)0x39a8eb33, (q31_t)0x8db8f6ea,
	(q31_t)0x39a34f13, (q31_t)0x8db6227a, (q31_t)0x399db2d0, (q31_t)0x8db34e50, (q31_t)0x3998166a, (q31_t)0x8db07a6d,
	(q31_t)0x399279e0, (q31_t)0x8dada6d1,
	(q31_t)0x398cdd32, (q31_t)0x8daad37b, (q31_t)0x39874061, (q31_t)0x8da8006c, (q31_t)0x3981a36d, (q31_t)0x8da52da3,
	(q31_t)0x397c0655, (q31_t)0x8da25b21,
	(q31_t)0x39766919, (q31_t)0x8d9f88e5, (q31_t)0x3970cbba, (q31_t)0x8d9cb6f0, (q31_t)0x396b2e38, (q31_t)0x8d99e541,
	(q31_t)0x39659092, (q31_t)0x8d9713d9,
	(q31_t)0x395ff2c9, (q31_t)0x8d9442b8, (q31_t)0x395a54dd, (q31_t)0x8d9171dd, (q31_t)0x3954b6cd, (q31_t)0x8d8ea148,
	(q31_t)0x394f1899, (q31_t)0x8d8bd0fb,
	(q31_t)0x39497a43, (q31_t)0x8d8900f3, (q31_t)0x3943dbc9, (q31_t)0x8d863133, (q31_t)0x393e3d2c, (q31_t)0x8d8361b9,
	(q31_t)0x39389e6b, (q31_t)0x8d809286,
	(q31_t)0x3932ff87, (q31_t)0x8d7dc399, (q31_t)0x392d6080, (q31_t)0x8d7af4f3, (q31_t)0x3927c155, (q31_t)0x8d782694,
	(q31_t)0x39222208, (q31_t)0x8d75587b,
	(q31_t)0x391c8297, (q31_t)0x8d728aa9, (q31_t)0x3916e303, (q31_t)0x8d6fbd1d, (q31_t)0x3911434b, (q31_t)0x8d6cefd9,
	(q31_t)0x390ba371, (q31_t)0x8d6a22db,
	(q31_t)0x39060373, (q31_t)0x8d675623, (q31_t)0x39006352, (q31_t)0x8d6489b3, (q31_t)0x38fac30e, (q31_t)0x8d61bd89,
	(q31_t)0x38f522a6, (q31_t)0x8d5ef1a5,
	(q31_t)0x38ef821c, (q31_t)0x8d5c2609, (q31_t)0x38e9e16e, (q31_t)0x8d595ab3, (q31_t)0x38e4409e, (q31_t)0x8d568fa4,
	(q31_t)0x38de9faa, (q31_t)0x8d53c4db,
	(q31_t)0x38d8fe93, (q31_t)0x8d50fa59, (q31_t)0x38d35d59, (q31_t)0x8d4e301f, (q31_t)0x38cdbbfc, (q31_t)0x8d4b662a,
	(q31_t)0x38c81a7c, (q31_t)0x8d489c7d,
	(q31_t)0x38c278d9, (q31_t)0x8d45d316, (q31_t)0x38bcd713, (q31_t)0x8d4309f6, (q31_t)0x38b7352a, (q31_t)0x8d40411d,
	(q31_t)0x38b1931e, (q31_t)0x8d3d788b,
	(q31_t)0x38abf0ef, (q31_t)0x8d3ab03f, (q31_t)0x38a64e9d, (q31_t)0x8d37e83a, (q31_t)0x38a0ac29, (q31_t)0x8d35207d,
	(q31_t)0x389b0991, (q31_t)0x8d325905,
	(q31_t)0x389566d6, (q31_t)0x8d2f91d5, (q31_t)0x388fc3f8, (q31_t)0x8d2ccaec, (q31_t)0x388a20f8, (q31_t)0x8d2a0449,
	(q31_t)0x38847dd5, (q31_t)0x8d273ded,
	(q31_t)0x387eda8e, (q31_t)0x8d2477d8, (q31_t)0x38793725, (q31_t)0x8d21b20a, (q31_t)0x38739399, (q31_t)0x8d1eec83,
	(q31_t)0x386defeb, (q31_t)0x8d1c2742,
	(q31_t)0x38684c19, (q31_t)0x8d196249, (q31_t)0x3862a825, (q31_t)0x8d169d96, (q31_t)0x385d040d, (q31_t)0x8d13d92a,
	(q31_t)0x38575fd4, (q31_t)0x8d111505,
	(q31_t)0x3851bb77, (q31_t)0x8d0e5127, (q31_t)0x384c16f7, (q31_t)0x8d0b8d90, (q31_t)0x38467255, (q31_t)0x8d08ca40,
	(q31_t)0x3840cd90, (q31_t)0x8d060737,
	(q31_t)0x383b28a9, (q31_t)0x8d034474, (q31_t)0x3835839f, (q31_t)0x8d0081f9, (q31_t)0x382fde72, (q31_t)0x8cfdbfc4,
	(q31_t)0x382a3922, (q31_t)0x8cfafdd7,
	(q31_t)0x382493b0, (q31_t)0x8cf83c30, (q31_t)0x381eee1b, (q31_t)0x8cf57ad0, (q31_t)0x38194864, (q31_t)0x8cf2b9b8,
	(q31_t)0x3813a28a, (q31_t)0x8ceff8e6,
	(q31_t)0x380dfc8d, (q31_t)0x8ced385b, (q31_t)0x3808566e, (q31_t)0x8cea7818, (q31_t)0x3802b02c, (q31_t)0x8ce7b81b,
	(q31_t)0x37fd09c8, (q31_t)0x8ce4f865,
	(q31_t)0x37f76341, (q31_t)0x8ce238f6, (q31_t)0x37f1bc97, (q31_t)0x8cdf79ce, (q31_t)0x37ec15cb, (q31_t)0x8cdcbaee,
	(q31_t)0x37e66edd, (q31_t)0x8cd9fc54,
	(q31_t)0x37e0c7cc, (q31_t)0x8cd73e01, (q31_t)0x37db2099, (q31_t)0x8cd47ff6, (q31_t)0x37d57943, (q31_t)0x8cd1c231,
	(q31_t)0x37cfd1cb, (q31_t)0x8ccf04b3,
	(q31_t)0x37ca2a30, (q31_t)0x8ccc477d, (q31_t)0x37c48273, (q31_t)0x8cc98a8e, (q31_t)0x37beda93, (q31_t)0x8cc6cde5,
	(q31_t)0x37b93292, (q31_t)0x8cc41184,
	(q31_t)0x37b38a6d, (q31_t)0x8cc1556a, (q31_t)0x37ade227, (q31_t)0x8cbe9996, (q31_t)0x37a839be, (q31_t)0x8cbbde0a,
	(q31_t)0x37a29132, (q31_t)0x8cb922c6,
	(q31_t)0x379ce885, (q31_t)0x8cb667c8, (q31_t)0x37973fb5, (q31_t)0x8cb3ad11, (q31_t)0x379196c3, (q31_t)0x8cb0f2a1,
	(q31_t)0x378bedae, (q31_t)0x8cae3879,
	(q31_t)0x37864477, (q31_t)0x8cab7e98, (q31_t)0x37809b1e, (q31_t)0x8ca8c4fd, (q31_t)0x377af1a3, (q31_t)0x8ca60baa,
	(q31_t)0x37754806, (q31_t)0x8ca3529f,
	(q31_t)0x376f9e46, (q31_t)0x8ca099da, (q31_t)0x3769f464, (q31_t)0x8c9de15c, (q31_t)0x37644a60, (q31_t)0x8c9b2926,
	(q31_t)0x375ea03a, (q31_t)0x8c987137,
	(q31_t)0x3758f5f2, (q31_t)0x8c95b98f, (q31_t)0x37534b87, (q31_t)0x8c93022e, (q31_t)0x374da0fa, (q31_t)0x8c904b14,
	(q31_t)0x3747f64c, (q31_t)0x8c8d9442,
	(q31_t)0x37424b7b, (q31_t)0x8c8addb7, (q31_t)0x373ca088, (q31_t)0x8c882773, (q31_t)0x3736f573, (q31_t)0x8c857176,
	(q31_t)0x37314a3c, (q31_t)0x8c82bbc0,
	(q31_t)0x372b9ee3, (q31_t)0x8c800652, (q31_t)0x3725f367, (q31_t)0x8c7d512b, (q31_t)0x372047ca, (q31_t)0x8c7a9c4b,
	(q31_t)0x371a9c0b, (q31_t)0x8c77e7b3,
	(q31_t)0x3714f02a, (q31_t)0x8c753362, (q31_t)0x370f4427, (q31_t)0x8c727f58, (q31_t)0x37099802, (q31_t)0x8c6fcb95,
	(q31_t)0x3703ebbb, (q31_t)0x8c6d181a,
	(q31_t)0x36fe3f52, (q31_t)0x8c6a64e5, (q31_t)0x36f892c7, (q31_t)0x8c67b1f9, (q31_t)0x36f2e61a, (q31_t)0x8c64ff53,
	(q31_t)0x36ed394b, (q31_t)0x8c624cf5,
	(q31_t)0x36e78c5b, (q31_t)0x8c5f9ade, (q31_t)0x36e1df48, (q31_t)0x8c5ce90e, (q31_t)0x36dc3214, (q31_t)0x8c5a3786,
	(q31_t)0x36d684be, (q31_t)0x8c578645,
	(q31_t)0x36d0d746, (q31_t)0x8c54d54c, (q31_t)0x36cb29ac, (q31_t)0x8c522499, (q31_t)0x36c57bf0, (q31_t)0x8c4f742f,
	(q31_t)0x36bfce13, (q31_t)0x8c4cc40b,
	(q31_t)0x36ba2014, (q31_t)0x8c4a142f, (q31_t)0x36b471f3, (q31_t)0x8c47649a, (q31_t)0x36aec3b0, (q31_t)0x8c44b54d,
	(q31_t)0x36a9154c, (q31_t)0x8c420647,
	(q31_t)0x36a366c6, (q31_t)0x8c3f5788, (q31_t)0x369db81e, (q31_t)0x8c3ca911, (q31_t)0x36980954, (q31_t)0x8c39fae1,
	(q31_t)0x36925a69, (q31_t)0x8c374cf9,
	(q31_t)0x368cab5c, (q31_t)0x8c349f58, (q31_t)0x3686fc2e, (q31_t)0x8c31f1ff, (q31_t)0x36814cde, (q31_t)0x8c2f44ed,
	(q31_t)0x367b9d6c, (q31_t)0x8c2c9822,
	(q31_t)0x3675edd9, (q31_t)0x8c29eb9f, (q31_t)0x36703e24, (q31_t)0x8c273f63, (q31_t)0x366a8e4d, (q31_t)0x8c24936f,
	(q31_t)0x3664de55, (q31_t)0x8c21e7c2,
	(q31_t)0x365f2e3b, (q31_t)0x8c1f3c5d, (q31_t)0x36597e00, (q31_t)0x8c1c913f, (q31_t)0x3653cda3, (q31_t)0x8c19e669,
	(q31_t)0x364e1d25, (q31_t)0x8c173bda,
	(q31_t)0x36486c86, (q31_t)0x8c149192, (q31_t)0x3642bbc4, (q31_t)0x8c11e792, (q31_t)0x363d0ae2, (q31_t)0x8c0f3dda,
	(q31_t)0x363759de, (q31_t)0x8c0c9469,
	(q31_t)0x3631a8b8, (q31_t)0x8c09eb40, (q31_t)0x362bf771, (q31_t)0x8c07425e, (q31_t)0x36264609, (q31_t)0x8c0499c4,
	(q31_t)0x3620947f, (q31_t)0x8c01f171,
	(q31_t)0x361ae2d3, (q31_t)0x8bff4966, (q31_t)0x36153107, (q31_t)0x8bfca1a3, (q31_t)0x360f7f19, (q31_t)0x8bf9fa27,
	(q31_t)0x3609cd0a, (q31_t)0x8bf752f2,
	(q31_t)0x36041ad9, (q31_t)0x8bf4ac05, (q31_t)0x35fe6887, (q31_t)0x8bf20560, (q31_t)0x35f8b614, (q31_t)0x8bef5f02,
	(q31_t)0x35f3037f, (q31_t)0x8becb8ec,
	(q31_t)0x35ed50c9, (q31_t)0x8bea131e, (q31_t)0x35e79df2, (q31_t)0x8be76d97, (q31_t)0x35e1eafa, (q31_t)0x8be4c857,
	(q31_t)0x35dc37e0, (q31_t)0x8be22360,
	(q31_t)0x35d684a6, (q31_t)0x8bdf7eb0, (q31_t)0x35d0d14a, (q31_t)0x8bdcda47, (q31_t)0x35cb1dcc, (q31_t)0x8bda3626,
	(q31_t)0x35c56a2e, (q31_t)0x8bd7924d,
	(q31_t)0x35bfb66e, (q31_t)0x8bd4eebc, (q31_t)0x35ba028e, (q31_t)0x8bd24b72, (q31_t)0x35b44e8c, (q31_t)0x8bcfa870,
	(q31_t)0x35ae9a69, (q31_t)0x8bcd05b5,
	(q31_t)0x35a8e625, (q31_t)0x8bca6343, (q31_t)0x35a331c0, (q31_t)0x8bc7c117, (q31_t)0x359d7d39, (q31_t)0x8bc51f34,
	(q31_t)0x3597c892, (q31_t)0x8bc27d98,
	(q31_t)0x359213c9, (q31_t)0x8bbfdc44, (q31_t)0x358c5ee0, (q31_t)0x8bbd3b38, (q31_t)0x3586a9d5, (q31_t)0x8bba9a73,
	(q31_t)0x3580f4aa, (q31_t)0x8bb7f9f6,
	(q31_t)0x357b3f5d, (q31_t)0x8bb559c1, (q31_t)0x357589f0, (q31_t)0x8bb2b9d4, (q31_t)0x356fd461, (q31_t)0x8bb01a2e,
	(q31_t)0x356a1eb2, (q31_t)0x8bad7ad0,
	(q31_t)0x356468e2, (q31_t)0x8baadbba, (q31_t)0x355eb2f0, (q31_t)0x8ba83cec, (q31_t)0x3558fcde, (q31_t)0x8ba59e65,
	(q31_t)0x355346ab, (q31_t)0x8ba30026,
	(q31_t)0x354d9057, (q31_t)0x8ba0622f, (q31_t)0x3547d9e2, (q31_t)0x8b9dc480, (q31_t)0x3542234c, (q31_t)0x8b9b2718,
	(q31_t)0x353c6c95, (q31_t)0x8b9889f8,
	(q31_t)0x3536b5be, (q31_t)0x8b95ed21, (q31_t)0x3530fec6, (q31_t)0x8b935090, (q31_t)0x352b47ad, (q31_t)0x8b90b448,
	(q31_t)0x35259073, (q31_t)0x8b8e1848,
	(q31_t)0x351fd918, (q31_t)0x8b8b7c8f, (q31_t)0x351a219c, (q31_t)0x8b88e11e, (q31_t)0x35146a00, (q31_t)0x8b8645f5,
	(q31_t)0x350eb243, (q31_t)0x8b83ab14,
	(q31_t)0x3508fa66, (q31_t)0x8b81107b, (q31_t)0x35034267, (q31_t)0x8b7e7629, (q31_t)0x34fd8a48, (q31_t)0x8b7bdc20,
	(q31_t)0x34f7d208, (q31_t)0x8b79425e,
	(q31_t)0x34f219a8, (q31_t)0x8b76a8e4, (q31_t)0x34ec6127, (q31_t)0x8b740fb3, (q31_t)0x34e6a885, (q31_t)0x8b7176c8,
	(q31_t)0x34e0efc2, (q31_t)0x8b6ede26,
	(q31_t)0x34db36df, (q31_t)0x8b6c45cc, (q31_t)0x34d57ddc, (q31_t)0x8b69adba, (q31_t)0x34cfc4b7, (q31_t)0x8b6715ef,
	(q31_t)0x34ca0b73, (q31_t)0x8b647e6d,
	(q31_t)0x34c4520d, (q31_t)0x8b61e733, (q31_t)0x34be9887, (q31_t)0x8b5f5040, (q31_t)0x34b8dee1, (q31_t)0x8b5cb995,
	(q31_t)0x34b3251a, (q31_t)0x8b5a2333,
	(q31_t)0x34ad6b32, (q31_t)0x8b578d18, (q31_t)0x34a7b12a, (q31_t)0x8b54f745, (q31_t)0x34a1f702, (q31_t)0x8b5261ba,
	(q31_t)0x349c3cb9, (q31_t)0x8b4fcc77,
	(q31_t)0x34968250, (q31_t)0x8b4d377c, (q31_t)0x3490c7c6, (q31_t)0x8b4aa2ca, (q31_t)0x348b0d1c, (q31_t)0x8b480e5f,
	(q31_t)0x34855251, (q31_t)0x8b457a3c,
	(q31_t)0x347f9766, (q31_t)0x8b42e661, (q31_t)0x3479dc5b, (q31_t)0x8b4052ce, (q31_t)0x3474212f, (q31_t)0x8b3dbf83,
	(q31_t)0x346e65e3, (q31_t)0x8b3b2c80,
	(q31_t)0x3468aa76, (q31_t)0x8b3899c6, (q31_t)0x3462eee9, (q31_t)0x8b360753, (q31_t)0x345d333c, (q31_t)0x8b337528,
	(q31_t)0x3457776f, (q31_t)0x8b30e345,
	(q31_t)0x3451bb81, (q31_t)0x8b2e51ab, (q31_t)0x344bff73, (q31_t)0x8b2bc058, (q31_t)0x34464345, (q31_t)0x8b292f4e,
	(q31_t)0x344086f6, (q31_t)0x8b269e8b,
	(q31_t)0x343aca87, (q31_t)0x8b240e11, (q31_t)0x34350df8, (q31_t)0x8b217ddf, (q31_t)0x342f5149, (q31_t)0x8b1eedf4,
	(q31_t)0x3429947a, (q31_t)0x8b1c5e52,
	(q31_t)0x3423d78a, (q31_t)0x8b19cef8, (q31_t)0x341e1a7b, (q31_t)0x8b173fe6, (q31_t)0x34185d4b, (q31_t)0x8b14b11d,
	(q31_t)0x34129ffb, (q31_t)0x8b12229b,
	(q31_t)0x340ce28b, (q31_t)0x8b0f9462, (q31_t)0x340724fb, (q31_t)0x8b0d0670, (q31_t)0x3401674a, (q31_t)0x8b0a78c7,
	(q31_t)0x33fba97a, (q31_t)0x8b07eb66,
	(q31_t)0x33f5eb89, (q31_t)0x8b055e4d, (q31_t)0x33f02d79, (q31_t)0x8b02d17c, (q31_t)0x33ea6f48, (q31_t)0x8b0044f3,
	(q31_t)0x33e4b0f8, (q31_t)0x8afdb8b3,
	(q31_t)0x33def287, (q31_t)0x8afb2cbb, (q31_t)0x33d933f7, (q31_t)0x8af8a10b, (q31_t)0x33d37546, (q31_t)0x8af615a3,
	(q31_t)0x33cdb676, (q31_t)0x8af38a83,
	(q31_t)0x33c7f785, (q31_t)0x8af0ffac, (q31_t)0x33c23875, (q31_t)0x8aee751c, (q31_t)0x33bc7944, (q31_t)0x8aebead5,
	(q31_t)0x33b6b9f4, (q31_t)0x8ae960d6,
	(q31_t)0x33b0fa84, (q31_t)0x8ae6d720, (q31_t)0x33ab3af4, (q31_t)0x8ae44db1, (q31_t)0x33a57b44, (q31_t)0x8ae1c48b,
	(q31_t)0x339fbb74, (q31_t)0x8adf3bad,
	(q31_t)0x3399fb85, (q31_t)0x8adcb318, (q31_t)0x33943b75, (q31_t)0x8ada2aca, (q31_t)0x338e7b46, (q31_t)0x8ad7a2c5,
	(q31_t)0x3388baf7, (q31_t)0x8ad51b08,
	(q31_t)0x3382fa88, (q31_t)0x8ad29394, (q31_t)0x337d39f9, (q31_t)0x8ad00c67, (q31_t)0x3377794b, (q31_t)0x8acd8583,
	(q31_t)0x3371b87d, (q31_t)0x8acafee8,
	(q31_t)0x336bf78f, (q31_t)0x8ac87894, (q31_t)0x33663682, (q31_t)0x8ac5f289, (q31_t)0x33607554, (q31_t)0x8ac36cc6,
	(q31_t)0x335ab407, (q31_t)0x8ac0e74c,
	(q31_t)0x3354f29b, (q31_t)0x8abe6219, (q31_t)0x334f310e, (q31_t)0x8abbdd30, (q31_t)0x33496f62, (q31_t)0x8ab9588e,
	(q31_t)0x3343ad97, (q31_t)0x8ab6d435,
	(q31_t)0x333debab, (q31_t)0x8ab45024, (q31_t)0x333829a1, (q31_t)0x8ab1cc5c, (q31_t)0x33326776, (q31_t)0x8aaf48db,
	(q31_t)0x332ca52c, (q31_t)0x8aacc5a4,
	(q31_t)0x3326e2c3, (q31_t)0x8aaa42b4, (q31_t)0x33212039, (q31_t)0x8aa7c00d, (q31_t)0x331b5d91, (q31_t)0x8aa53daf,
	(q31_t)0x33159ac8, (q31_t)0x8aa2bb99,
	(q31_t)0x330fd7e1, (q31_t)0x8aa039cb, (q31_t)0x330a14da, (q31_t)0x8a9db845, (q31_t)0x330451b3, (q31_t)0x8a9b3708,
	(q31_t)0x32fe8e6d, (q31_t)0x8a98b614,
	(q31_t)0x32f8cb07, (q31_t)0x8a963567, (q31_t)0x32f30782, (q31_t)0x8a93b504, (q31_t)0x32ed43de, (q31_t)0x8a9134e8,
	(q31_t)0x32e7801a, (q31_t)0x8a8eb516,
	(q31_t)0x32e1bc36, (q31_t)0x8a8c358b, (q31_t)0x32dbf834, (q31_t)0x8a89b649, (q31_t)0x32d63412, (q31_t)0x8a873750,
	(q31_t)0x32d06fd0, (q31_t)0x8a84b89e,
	(q31_t)0x32caab6f, (q31_t)0x8a823a36, (q31_t)0x32c4e6ef, (q31_t)0x8a7fbc16, (q31_t)0x32bf2250, (q31_t)0x8a7d3e3e,
	(q31_t)0x32b95d91, (q31_t)0x8a7ac0af,
	(q31_t)0x32b398b3, (q31_t)0x8a784368, (q31_t)0x32add3b6, (q31_t)0x8a75c66a, (q31_t)0x32a80e99, (q31_t)0x8a7349b4,
	(q31_t)0x32a2495d, (q31_t)0x8a70cd47,
	(q31_t)0x329c8402, (q31_t)0x8a6e5123, (q31_t)0x3296be88, (q31_t)0x8a6bd547, (q31_t)0x3290f8ef, (q31_t)0x8a6959b3,
	(q31_t)0x328b3336, (q31_t)0x8a66de68,
	(q31_t)0x32856d5e, (q31_t)0x8a646365, (q31_t)0x327fa767, (q31_t)0x8a61e8ab, (q31_t)0x3279e151, (q31_t)0x8a5f6e3a,
	(q31_t)0x32741b1c, (q31_t)0x8a5cf411,
	(q31_t)0x326e54c7, (q31_t)0x8a5a7a31, (q31_t)0x32688e54, (q31_t)0x8a580099, (q31_t)0x3262c7c1, (q31_t)0x8a55874a,
	(q31_t)0x325d0110, (q31_t)0x8a530e43,
	(q31_t)0x32573a3f, (q31_t)0x8a509585, (q31_t)0x3251734f, (q31_t)0x8a4e1d10, (q31_t)0x324bac40, (q31_t)0x8a4ba4e3,
	(q31_t)0x3245e512, (q31_t)0x8a492cff,
	(q31_t)0x32401dc6, (q31_t)0x8a46b564, (q31_t)0x323a565a, (q31_t)0x8a443e11, (q31_t)0x32348ecf, (q31_t)0x8a41c706,
	(q31_t)0x322ec725, (q31_t)0x8a3f5045,
	(q31_t)0x3228ff5c, (q31_t)0x8a3cd9cc, (q31_t)0x32233775, (q31_t)0x8a3a639b, (q31_t)0x321d6f6e, (q31_t)0x8a37edb3,
	(q31_t)0x3217a748, (q31_t)0x8a357814,
	(q31_t)0x3211df04, (q31_t)0x8a3302be, (q31_t)0x320c16a1, (q31_t)0x8a308db0, (q31_t)0x32064e1e, (q31_t)0x8a2e18eb,
	(q31_t)0x3200857d, (q31_t)0x8a2ba46e,
	(q31_t)0x31fabcbd, (q31_t)0x8a29303b, (q31_t)0x31f4f3df, (q31_t)0x8a26bc50, (q31_t)0x31ef2ae1, (q31_t)0x8a2448ad,
	(q31_t)0x31e961c5, (q31_t)0x8a21d554,
	(q31_t)0x31e39889, (q31_t)0x8a1f6243, (q31_t)0x31ddcf30, (q31_t)0x8a1cef7a, (q31_t)0x31d805b7, (q31_t)0x8a1a7cfb,
	(q31_t)0x31d23c1f, (q31_t)0x8a180ac4,
	(q31_t)0x31cc7269, (q31_t)0x8a1598d6, (q31_t)0x31c6a894, (q31_t)0x8a132731, (q31_t)0x31c0dea1, (q31_t)0x8a10b5d4,
	(q31_t)0x31bb148f, (q31_t)0x8a0e44c0,
	(q31_t)0x31b54a5e, (q31_t)0x8a0bd3f5, (q31_t)0x31af800e, (q31_t)0x8a096373, (q31_t)0x31a9b5a0, (q31_t)0x8a06f339,
	(q31_t)0x31a3eb13, (q31_t)0x8a048348,
	(q31_t)0x319e2067, (q31_t)0x8a0213a0, (q31_t)0x3198559d, (q31_t)0x89ffa441, (q31_t)0x31928ab4, (q31_t)0x89fd352b,
	(q31_t)0x318cbfad, (q31_t)0x89fac65d,
	(q31_t)0x3186f487, (q31_t)0x89f857d8, (q31_t)0x31812943, (q31_t)0x89f5e99c, (q31_t)0x317b5de0, (q31_t)0x89f37ba9,
	(q31_t)0x3175925e, (q31_t)0x89f10dff,
	(q31_t)0x316fc6be, (q31_t)0x89eea09d, (q31_t)0x3169fb00, (q31_t)0x89ec3384, (q31_t)0x31642f23, (q31_t)0x89e9c6b4,
	(q31_t)0x315e6328, (q31_t)0x89e75a2d,
	(q31_t)0x3158970e, (q31_t)0x89e4edef, (q31_t)0x3152cad5, (q31_t)0x89e281fa, (q31_t)0x314cfe7f, (q31_t)0x89e0164d,
	(q31_t)0x31473209, (q31_t)0x89ddaae9,
	(q31_t)0x31416576, (q31_t)0x89db3fcf, (q31_t)0x313b98c4, (q31_t)0x89d8d4fd, (q31_t)0x3135cbf4, (q31_t)0x89d66a74,
	(q31_t)0x312fff05, (q31_t)0x89d40033,
	(q31_t)0x312a31f8, (q31_t)0x89d1963c, (q31_t)0x312464cd, (q31_t)0x89cf2c8e, (q31_t)0x311e9783, (q31_t)0x89ccc328,
	(q31_t)0x3118ca1b, (q31_t)0x89ca5a0c,
	(q31_t)0x3112fc95, (q31_t)0x89c7f138, (q31_t)0x310d2ef0, (q31_t)0x89c588ae, (q31_t)0x3107612e, (q31_t)0x89c3206c,
	(q31_t)0x3101934d, (q31_t)0x89c0b873,
	(q31_t)0x30fbc54d, (q31_t)0x89be50c3, (q31_t)0x30f5f730, (q31_t)0x89bbe95c, (q31_t)0x30f028f4, (q31_t)0x89b9823e,
	(q31_t)0x30ea5a9a, (q31_t)0x89b71b69,
	(q31_t)0x30e48c22, (q31_t)0x89b4b4dd, (q31_t)0x30debd8c, (q31_t)0x89b24e9a, (q31_t)0x30d8eed8, (q31_t)0x89afe8a0,
	(q31_t)0x30d32006, (q31_t)0x89ad82ef,
	(q31_t)0x30cd5115, (q31_t)0x89ab1d87, (q31_t)0x30c78206, (q31_t)0x89a8b868, (q31_t)0x30c1b2da, (q31_t)0x89a65391,
	(q31_t)0x30bbe38f, (q31_t)0x89a3ef04,
	(q31_t)0x30b61426, (q31_t)0x89a18ac0, (q31_t)0x30b0449f, (q31_t)0x899f26c5, (q31_t)0x30aa74fa, (q31_t)0x899cc313,
	(q31_t)0x30a4a537, (q31_t)0x899a5faa,
	(q31_t)0x309ed556, (q31_t)0x8997fc8a, (q31_t)0x30990557, (q31_t)0x899599b3, (q31_t)0x3093353a, (q31_t)0x89933725,
	(q31_t)0x308d64ff, (q31_t)0x8990d4e0,
	(q31_t)0x308794a6, (q31_t)0x898e72e4, (q31_t)0x3081c42f, (q31_t)0x898c1131, (q31_t)0x307bf39b, (q31_t)0x8989afc8,
	(q31_t)0x307622e8, (q31_t)0x89874ea7,
	(q31_t)0x30705217, (q31_t)0x8984edcf, (q31_t)0x306a8129, (q31_t)0x89828d41, (q31_t)0x3064b01d, (q31_t)0x89802cfc,
	(q31_t)0x305edef3, (q31_t)0x897dccff,
	(q31_t)0x30590dab, (q31_t)0x897b6d4c, (q31_t)0x30533c45, (q31_t)0x89790de2, (q31_t)0x304d6ac1, (q31_t)0x8976aec1,
	(q31_t)0x30479920, (q31_t)0x89744fe9,
	(q31_t)0x3041c761, (q31_t)0x8971f15a, (q31_t)0x303bf584, (q31_t)0x896f9315, (q31_t)0x30362389, (q31_t)0x896d3518,
	(q31_t)0x30305171, (q31_t)0x896ad765,
	(q31_t)0x302a7f3a, (q31_t)0x896879fb, (q31_t)0x3024ace6, (q31_t)0x89661cda, (q31_t)0x301eda75, (q31_t)0x8963c002,
	(q31_t)0x301907e6, (q31_t)0x89616373,
	(q31_t)0x30133539, (q31_t)0x895f072e, (q31_t)0x300d626e, (q31_t)0x895cab31, (q31_t)0x30078f86, (q31_t)0x895a4f7e,
	(q31_t)0x3001bc80, (q31_t)0x8957f414,
	(q31_t)0x2ffbe95d, (q31_t)0x895598f3, (q31_t)0x2ff6161c, (q31_t)0x89533e1c, (q31_t)0x2ff042bd, (q31_t)0x8950e38e,
	(q31_t)0x2fea6f41, (q31_t)0x894e8948,
	(q31_t)0x2fe49ba7, (q31_t)0x894c2f4c, (q31_t)0x2fdec7f0, (q31_t)0x8949d59a, (q31_t)0x2fd8f41b, (q31_t)0x89477c30,
	(q31_t)0x2fd32028, (q31_t)0x89452310,
	(q31_t)0x2fcd4c19, (q31_t)0x8942ca39, (q31_t)0x2fc777eb, (q31_t)0x894071ab, (q31_t)0x2fc1a3a0, (q31_t)0x893e1967,
	(q31_t)0x2fbbcf38, (q31_t)0x893bc16b,
	(q31_t)0x2fb5fab2, (q31_t)0x893969b9, (q31_t)0x2fb0260f, (q31_t)0x89371250, (q31_t)0x2faa514f, (q31_t)0x8934bb31,
	(q31_t)0x2fa47c71, (q31_t)0x8932645b,
	(q31_t)0x2f9ea775, (q31_t)0x89300dce, (q31_t)0x2f98d25d, (q31_t)0x892db78a, (q31_t)0x2f92fd26, (q31_t)0x892b6190,
	(q31_t)0x2f8d27d3, (q31_t)0x89290bdf,
	(q31_t)0x2f875262, (q31_t)0x8926b677, (q31_t)0x2f817cd4, (q31_t)0x89246159, (q31_t)0x2f7ba729, (q31_t)0x89220c84,
	(q31_t)0x2f75d160, (q31_t)0x891fb7f8,
	(q31_t)0x2f6ffb7a, (q31_t)0x891d63b5, (q31_t)0x2f6a2577, (q31_t)0x891b0fbc, (q31_t)0x2f644f56, (q31_t)0x8918bc0c,
	(q31_t)0x2f5e7919, (q31_t)0x891668a6,
	(q31_t)0x2f58a2be, (q31_t)0x89141589, (q31_t)0x2f52cc46, (q31_t)0x8911c2b5, (q31_t)0x2f4cf5b0, (q31_t)0x890f702b,
	(q31_t)0x2f471efe, (q31_t)0x890d1dea,
	(q31_t)0x2f41482e, (q31_t)0x890acbf2, (q31_t)0x2f3b7141, (q31_t)0x89087a44, (q31_t)0x2f359a37, (q31_t)0x890628df,
	(q31_t)0x2f2fc310, (q31_t)0x8903d7c4,
	(q31_t)0x2f29ebcc, (q31_t)0x890186f2, (q31_t)0x2f24146b, (q31_t)0x88ff3669, (q31_t)0x2f1e3ced, (q31_t)0x88fce62a,
	(q31_t)0x2f186551, (q31_t)0x88fa9634,
	(q31_t)0x2f128d99, (q31_t)0x88f84687, (q31_t)0x2f0cb5c3, (q31_t)0x88f5f724, (q31_t)0x2f06ddd1, (q31_t)0x88f3a80b,
	(q31_t)0x2f0105c1, (q31_t)0x88f1593b,
	(q31_t)0x2efb2d95, (q31_t)0x88ef0ab4, (q31_t)0x2ef5554b, (q31_t)0x88ecbc77, (q31_t)0x2eef7ce5, (q31_t)0x88ea6e83,
	(q31_t)0x2ee9a461, (q31_t)0x88e820d9,
	(q31_t)0x2ee3cbc1, (q31_t)0x88e5d378, (q31_t)0x2eddf304, (q31_t)0x88e38660, (q31_t)0x2ed81a29, (q31_t)0x88e13992,
	(q31_t)0x2ed24132, (q31_t)0x88deed0e,
	(q31_t)0x2ecc681e, (q31_t)0x88dca0d3, (q31_t)0x2ec68eed, (q31_t)0x88da54e1, (q31_t)0x2ec0b5a0, (q31_t)0x88d8093a,
	(q31_t)0x2ebadc35, (q31_t)0x88d5bddb,
	(q31_t)0x2eb502ae, (q31_t)0x88d372c6, (q31_t)0x2eaf290a, (q31_t)0x88d127fb, (q31_t)0x2ea94f49, (q31_t)0x88cedd79,
	(q31_t)0x2ea3756b, (q31_t)0x88cc9340,
	(q31_t)0x2e9d9b70, (q31_t)0x88ca4951, (q31_t)0x2e97c159, (q31_t)0x88c7ffac, (q31_t)0x2e91e725, (q31_t)0x88c5b650,
	(q31_t)0x2e8c0cd4, (q31_t)0x88c36d3e,
	(q31_t)0x2e863267, (q31_t)0x88c12475, (q31_t)0x2e8057dd, (q31_t)0x88bedbf6, (q31_t)0x2e7a7d36, (q31_t)0x88bc93c0,
	(q31_t)0x2e74a272, (q31_t)0x88ba4bd4,
	(q31_t)0x2e6ec792, (q31_t)0x88b80432, (q31_t)0x2e68ec95, (q31_t)0x88b5bcd9, (q31_t)0x2e63117c, (q31_t)0x88b375ca,
	(q31_t)0x2e5d3646, (q31_t)0x88b12f04,
	(q31_t)0x2e575af3, (q31_t)0x88aee888, (q31_t)0x2e517f84, (q31_t)0x88aca255, (q31_t)0x2e4ba3f8, (q31_t)0x88aa5c6c,
	(q31_t)0x2e45c850, (q31_t)0x88a816cd,
	(q31_t)0x2e3fec8b, (q31_t)0x88a5d177, (q31_t)0x2e3a10aa, (q31_t)0x88a38c6b, (q31_t)0x2e3434ac, (q31_t)0x88a147a9,
	(q31_t)0x2e2e5891, (q31_t)0x889f0330,
	(q31_t)0x2e287c5a, (q31_t)0x889cbf01, (q31_t)0x2e22a007, (q31_t)0x889a7b1b, (q31_t)0x2e1cc397, (q31_t)0x88983780,
	(q31_t)0x2e16e70b, (q31_t)0x8895f42d,
	(q31_t)0x2e110a62, (q31_t)0x8893b125, (q31_t)0x2e0b2d9d, (q31_t)0x88916e66, (q31_t)0x2e0550bb, (q31_t)0x888f2bf1,
	(q31_t)0x2dff73bd, (q31_t)0x888ce9c5,
	(q31_t)0x2df996a3, (q31_t)0x888aa7e3, (q31_t)0x2df3b96c, (q31_t)0x8888664b, (q31_t)0x2deddc19, (q31_t)0x888624fd,
	(q31_t)0x2de7feaa, (q31_t)0x8883e3f8,
	(q31_t)0x2de2211e, (q31_t)0x8881a33d, (q31_t)0x2ddc4376, (q31_t)0x887f62cb, (q31_t)0x2dd665b2, (q31_t)0x887d22a4,
	(q31_t)0x2dd087d1, (q31_t)0x887ae2c6,
	(q31_t)0x2dcaa9d5, (q31_t)0x8878a332, (q31_t)0x2dc4cbbc, (q31_t)0x887663e7, (q31_t)0x2dbeed86, (q31_t)0x887424e7,
	(q31_t)0x2db90f35, (q31_t)0x8871e630,
	(q31_t)0x2db330c7, (q31_t)0x886fa7c2, (q31_t)0x2dad523d, (q31_t)0x886d699f, (q31_t)0x2da77397, (q31_t)0x886b2bc5,
	(q31_t)0x2da194d5, (q31_t)0x8868ee35,
	(q31_t)0x2d9bb5f6, (q31_t)0x8866b0ef, (q31_t)0x2d95d6fc, (q31_t)0x886473f2, (q31_t)0x2d8ff7e5, (q31_t)0x88623740,
	(q31_t)0x2d8a18b3, (q31_t)0x885ffad7,
	(q31_t)0x2d843964, (q31_t)0x885dbeb8, (q31_t)0x2d7e59f9, (q31_t)0x885b82e3, (q31_t)0x2d787a72, (q31_t)0x88594757,
	(q31_t)0x2d729acf, (q31_t)0x88570c16,
	(q31_t)0x2d6cbb10, (q31_t)0x8854d11e, (q31_t)0x2d66db35, (q31_t)0x88529670, (q31_t)0x2d60fb3e, (q31_t)0x88505c0b,
	(q31_t)0x2d5b1b2b, (q31_t)0x884e21f1,
	(q31_t)0x2d553afc, (q31_t)0x884be821, (q31_t)0x2d4f5ab1, (q31_t)0x8849ae9a, (q31_t)0x2d497a4a, (q31_t)0x8847755d,
	(q31_t)0x2d4399c7, (q31_t)0x88453c6a,
	(q31_t)0x2d3db928, (q31_t)0x884303c1, (q31_t)0x2d37d86d, (q31_t)0x8840cb61, (q31_t)0x2d31f797, (q31_t)0x883e934c,
	(q31_t)0x2d2c16a4, (q31_t)0x883c5b81,
	(q31_t)0x2d263596, (q31_t)0x883a23ff, (q31_t)0x2d20546b, (q31_t)0x8837ecc7, (q31_t)0x2d1a7325, (q31_t)0x8835b5d9,
	(q31_t)0x2d1491c4, (q31_t)0x88337f35,
	(q31_t)0x2d0eb046, (q31_t)0x883148db, (q31_t)0x2d08ceac, (q31_t)0x882f12cb, (q31_t)0x2d02ecf7, (q31_t)0x882cdd04,
	(q31_t)0x2cfd0b26, (q31_t)0x882aa788,
	(q31_t)0x2cf72939, (q31_t)0x88287256, (q31_t)0x2cf14731, (q31_t)0x88263d6d, (q31_t)0x2ceb650d, (q31_t)0x882408ce,
	(q31_t)0x2ce582cd, (q31_t)0x8821d47a,
	(q31_t)0x2cdfa071, (q31_t)0x881fa06f, (q31_t)0x2cd9bdfa, (q31_t)0x881d6cae, (q31_t)0x2cd3db67, (q31_t)0x881b3937,
	(q31_t)0x2ccdf8b8, (q31_t)0x8819060a,
	(q31_t)0x2cc815ee, (q31_t)0x8816d327, (q31_t)0x2cc23308, (q31_t)0x8814a08f, (q31_t)0x2cbc5006, (q31_t)0x88126e40,
	(q31_t)0x2cb66ce9, (q31_t)0x88103c3b,
	(q31_t)0x2cb089b1, (q31_t)0x880e0a7f, (q31_t)0x2caaa65c, (q31_t)0x880bd90e, (q31_t)0x2ca4c2ed, (q31_t)0x8809a7e7,
	(q31_t)0x2c9edf61, (q31_t)0x8807770a,
	(q31_t)0x2c98fbba, (q31_t)0x88054677, (q31_t)0x2c9317f8, (q31_t)0x8803162e, (q31_t)0x2c8d341a, (q31_t)0x8800e62f,
	(q31_t)0x2c875021, (q31_t)0x87feb67a,
	(q31_t)0x2c816c0c, (q31_t)0x87fc870f, (q31_t)0x2c7b87dc, (q31_t)0x87fa57ee, (q31_t)0x2c75a390, (q31_t)0x87f82917,
	(q31_t)0x2c6fbf29, (q31_t)0x87f5fa8b,
	(q31_t)0x2c69daa6, (q31_t)0x87f3cc48, (q31_t)0x2c63f609, (q31_t)0x87f19e4f, (q31_t)0x2c5e114f, (q31_t)0x87ef70a0,
	(q31_t)0x2c582c7b, (q31_t)0x87ed433c,
	(q31_t)0x2c52478a, (q31_t)0x87eb1621, (q31_t)0x2c4c627f, (q31_t)0x87e8e950, (q31_t)0x2c467d58, (q31_t)0x87e6bcca,
	(q31_t)0x2c409816, (q31_t)0x87e4908e,
	(q31_t)0x2c3ab2b9, (q31_t)0x87e2649b, (q31_t)0x2c34cd40, (q31_t)0x87e038f3, (q31_t)0x2c2ee7ad, (q31_t)0x87de0d95,
	(q31_t)0x2c2901fd, (q31_t)0x87dbe281,
	(q31_t)0x2c231c33, (q31_t)0x87d9b7b7, (q31_t)0x2c1d364e, (q31_t)0x87d78d38, (q31_t)0x2c17504d, (q31_t)0x87d56302,
	(q31_t)0x2c116a31, (q31_t)0x87d33916,
	(q31_t)0x2c0b83fa, (q31_t)0x87d10f75, (q31_t)0x2c059da7, (q31_t)0x87cee61e, (q31_t)0x2bffb73a, (q31_t)0x87ccbd11,
	(q31_t)0x2bf9d0b1, (q31_t)0x87ca944e,
	(q31_t)0x2bf3ea0d, (q31_t)0x87c86bd5, (q31_t)0x2bee034e, (q31_t)0x87c643a6, (q31_t)0x2be81c74, (q31_t)0x87c41bc2,
	(q31_t)0x2be2357f, (q31_t)0x87c1f427,
	(q31_t)0x2bdc4e6f, (q31_t)0x87bfccd7, (q31_t)0x2bd66744, (q31_t)0x87bda5d1, (q31_t)0x2bd07ffe, (q31_t)0x87bb7f16,
	(q31_t)0x2bca989d, (q31_t)0x87b958a4,
	(q31_t)0x2bc4b120, (q31_t)0x87b7327d, (q31_t)0x2bbec989, (q31_t)0x87b50c9f, (q31_t)0x2bb8e1d7, (q31_t)0x87b2e70c,
	(q31_t)0x2bb2fa0a, (q31_t)0x87b0c1c4,
	(q31_t)0x2bad1221, (q31_t)0x87ae9cc5, (q31_t)0x2ba72a1e, (q31_t)0x87ac7811, (q31_t)0x2ba14200, (q31_t)0x87aa53a6,
	(q31_t)0x2b9b59c7, (q31_t)0x87a82f87,
	(q31_t)0x2b957173, (q31_t)0x87a60bb1, (q31_t)0x2b8f8905, (q31_t)0x87a3e825, (q31_t)0x2b89a07b, (q31_t)0x87a1c4e4,
	(q31_t)0x2b83b7d7, (q31_t)0x879fa1ed,
	(q31_t)0x2b7dcf17, (q31_t)0x879d7f41, (q31_t)0x2b77e63d, (q31_t)0x879b5cde, (q31_t)0x2b71fd48, (q31_t)0x87993ac6,
	(q31_t)0x2b6c1438, (q31_t)0x879718f8,
	(q31_t)0x2b662b0e, (q31_t)0x8794f774, (q31_t)0x2b6041c9, (q31_t)0x8792d63b, (q31_t)0x2b5a5868, (q31_t)0x8790b54c,
	(q31_t)0x2b546eee, (q31_t)0x878e94a7,
	(q31_t)0x2b4e8558, (q31_t)0x878c744d, (q31_t)0x2b489ba8, (q31_t)0x878a543d, (q31_t)0x2b42b1dd, (q31_t)0x87883477,
	(q31_t)0x2b3cc7f7, (q31_t)0x878614fb,
	(q31_t)0x2b36ddf7, (q31_t)0x8783f5ca, (q31_t)0x2b30f3dc, (q31_t)0x8781d6e3, (q31_t)0x2b2b09a6, (q31_t)0x877fb846,
	(q31_t)0x2b251f56, (q31_t)0x877d99f4,
	(q31_t)0x2b1f34eb, (q31_t)0x877b7bec, (q31_t)0x2b194a66, (q31_t)0x87795e2f, (q31_t)0x2b135fc6, (q31_t)0x877740bb,
	(q31_t)0x2b0d750b, (q31_t)0x87752392,
	(q31_t)0x2b078a36, (q31_t)0x877306b4, (q31_t)0x2b019f46, (q31_t)0x8770ea20, (q31_t)0x2afbb43c, (q31_t)0x876ecdd6,
	(q31_t)0x2af5c917, (q31_t)0x876cb1d6,
	(q31_t)0x2aefddd8, (q31_t)0x876a9621, (q31_t)0x2ae9f27e, (q31_t)0x87687ab7, (q31_t)0x2ae4070a, (q31_t)0x87665f96,
	(q31_t)0x2ade1b7c, (q31_t)0x876444c1,
	(q31_t)0x2ad82fd2, (q31_t)0x87622a35, (q31_t)0x2ad2440f, (q31_t)0x87600ff4, (q31_t)0x2acc5831, (q31_t)0x875df5fd,
	(q31_t)0x2ac66c39, (q31_t)0x875bdc51,
	(q31_t)0x2ac08026, (q31_t)0x8759c2ef, (q31_t)0x2aba93f9, (q31_t)0x8757a9d8, (q31_t)0x2ab4a7b1, (q31_t)0x8755910b,
	(q31_t)0x2aaebb50, (q31_t)0x87537888,
	(q31_t)0x2aa8ced3, (q31_t)0x87516050, (q31_t)0x2aa2e23d, (q31_t)0x874f4862, (q31_t)0x2a9cf58c, (q31_t)0x874d30bf,
	(q31_t)0x2a9708c1, (q31_t)0x874b1966,
	(q31_t)0x2a911bdc, (q31_t)0x87490258, (q31_t)0x2a8b2edc, (q31_t)0x8746eb94, (q31_t)0x2a8541c3, (q31_t)0x8744d51b,
	(q31_t)0x2a7f548e, (q31_t)0x8742beec,
	(q31_t)0x2a796740, (q31_t)0x8740a907, (q31_t)0x2a7379d8, (q31_t)0x873e936d, (q31_t)0x2a6d8c55, (q31_t)0x873c7e1e,
	(q31_t)0x2a679eb8, (q31_t)0x873a6919,
	(q31_t)0x2a61b101, (q31_t)0x8738545e, (q31_t)0x2a5bc330, (q31_t)0x87363fee, (q31_t)0x2a55d545, (q31_t)0x87342bc9,
	(q31_t)0x2a4fe740, (q31_t)0x873217ee,
	(q31_t)0x2a49f920, (q31_t)0x8730045d, (q31_t)0x2a440ae7, (q31_t)0x872df117, (q31_t)0x2a3e1c93, (q31_t)0x872bde1c,
	(q31_t)0x2a382e25, (q31_t)0x8729cb6b,
	(q31_t)0x2a323f9e, (q31_t)0x8727b905, (q31_t)0x2a2c50fc, (q31_t)0x8725a6e9, (q31_t)0x2a266240, (q31_t)0x87239518,
	(q31_t)0x2a20736a, (q31_t)0x87218391,
	(q31_t)0x2a1a847b, (q31_t)0x871f7255, (q31_t)0x2a149571, (q31_t)0x871d6163, (q31_t)0x2a0ea64d, (q31_t)0x871b50bc,
	(q31_t)0x2a08b710, (q31_t)0x87194060,
	(q31_t)0x2a02c7b8, (q31_t)0x8717304e, (q31_t)0x29fcd847, (q31_t)0x87152087, (q31_t)0x29f6e8bb, (q31_t)0x8713110a,
	(q31_t)0x29f0f916, (q31_t)0x871101d8,
	(q31_t)0x29eb0957, (q31_t)0x870ef2f1, (q31_t)0x29e5197e, (q31_t)0x870ce454, (q31_t)0x29df298b, (q31_t)0x870ad602,
	(q31_t)0x29d9397f, (q31_t)0x8708c7fa,
	(q31_t)0x29d34958, (q31_t)0x8706ba3d, (q31_t)0x29cd5918, (q31_t)0x8704acca, (q31_t)0x29c768be, (q31_t)0x87029fa3,
	(q31_t)0x29c1784a, (q31_t)0x870092c5,
	(q31_t)0x29bb87bc, (q31_t)0x86fe8633, (q31_t)0x29b59715, (q31_t)0x86fc79eb, (q31_t)0x29afa654, (q31_t)0x86fa6dee,
	(q31_t)0x29a9b579, (q31_t)0x86f8623b,
	(q31_t)0x29a3c485, (q31_t)0x86f656d3, (q31_t)0x299dd377, (q31_t)0x86f44bb6, (q31_t)0x2997e24f, (q31_t)0x86f240e3,
	(q31_t)0x2991f10e, (q31_t)0x86f0365c,
	(q31_t)0x298bffb2, (q31_t)0x86ee2c1e, (q31_t)0x29860e3e, (q31_t)0x86ec222c, (q31_t)0x29801caf, (q31_t)0x86ea1884,
	(q31_t)0x297a2b07, (q31_t)0x86e80f27,
	(q31_t)0x29743946, (q31_t)0x86e60614, (q31_t)0x296e476b, (q31_t)0x86e3fd4c, (q31_t)0x29685576, (q31_t)0x86e1f4cf,
	(q31_t)0x29626368, (q31_t)0x86dfec9d,
	(q31_t)0x295c7140, (q31_t)0x86dde4b5, (q31_t)0x29567eff, (q31_t)0x86dbdd18, (q31_t)0x29508ca4, (q31_t)0x86d9d5c6,
	(q31_t)0x294a9a30, (q31_t)0x86d7cebf,
	(q31_t)0x2944a7a2, (q31_t)0x86d5c802, (q31_t)0x293eb4fb, (q31_t)0x86d3c190, (q31_t)0x2938c23a, (q31_t)0x86d1bb69,
	(q31_t)0x2932cf60, (q31_t)0x86cfb58c,
	(q31_t)0x292cdc6d, (q31_t)0x86cdaffa, (q31_t)0x2926e960, (q31_t)0x86cbaab3, (q31_t)0x2920f63a, (q31_t)0x86c9a5b7,
	(q31_t)0x291b02fa, (q31_t)0x86c7a106,
	(q31_t)0x29150fa1, (q31_t)0x86c59c9f, (q31_t)0x290f1c2f, (q31_t)0x86c39883, (q31_t)0x290928a3, (q31_t)0x86c194b2,
	(q31_t)0x290334ff, (q31_t)0x86bf912c,
	(q31_t)0x28fd4140, (q31_t)0x86bd8df0, (q31_t)0x28f74d69, (q31_t)0x86bb8b00, (q31_t)0x28f15978, (q31_t)0x86b9885a,
	(q31_t)0x28eb656e, (q31_t)0x86b785ff,
	(q31_t)0x28e5714b, (q31_t)0x86b583ee, (q31_t)0x28df7d0e, (q31_t)0x86b38229, (q31_t)0x28d988b8, (q31_t)0x86b180ae,
	(q31_t)0x28d3944a, (q31_t)0x86af7f7e,
	(q31_t)0x28cd9fc1, (q31_t)0x86ad7e99, (q31_t)0x28c7ab20, (q31_t)0x86ab7dff, (q31_t)0x28c1b666, (q31_t)0x86a97db0,
	(q31_t)0x28bbc192, (q31_t)0x86a77dab,
	(q31_t)0x28b5cca5, (q31_t)0x86a57df2, (q31_t)0x28afd7a0, (q31_t)0x86a37e83, (q31_t)0x28a9e281, (q31_t)0x86a17f5f,
	(q31_t)0x28a3ed49, (q31_t)0x869f8086,
	(q31_t)0x289df7f8, (q31_t)0x869d81f8, (q31_t)0x2898028e, (q31_t)0x869b83b4, (q31_t)0x28920d0a, (q31_t)0x869985bc,
	(q31_t)0x288c176e, (q31_t)0x8697880f,
	(q31_t)0x288621b9, (q31_t)0x86958aac, (q31_t)0x28802beb, (q31_t)0x86938d94, (q31_t)0x287a3604, (q31_t)0x869190c7,
	(q31_t)0x28744004, (q31_t)0x868f9445,
	(q31_t)0x286e49ea, (q31_t)0x868d980e, (q31_t)0x286853b8, (q31_t)0x868b9c22, (q31_t)0x28625d6d, (q31_t)0x8689a081,
	(q31_t)0x285c670a, (q31_t)0x8687a52b,
	(q31_t)0x2856708d, (q31_t)0x8685aa20, (q31_t)0x285079f7, (q31_t)0x8683af5f, (q31_t)0x284a8349, (q31_t)0x8681b4ea,
	(q31_t)0x28448c81, (q31_t)0x867fbabf,
	(q31_t)0x283e95a1, (q31_t)0x867dc0e0, (q31_t)0x28389ea8, (q31_t)0x867bc74b, (q31_t)0x2832a796, (q31_t)0x8679ce01,
	(q31_t)0x282cb06c, (q31_t)0x8677d503,
	(q31_t)0x2826b928, (q31_t)0x8675dc4f, (q31_t)0x2820c1cc, (q31_t)0x8673e3e6, (q31_t)0x281aca57, (q31_t)0x8671ebc8,
	(q31_t)0x2814d2c9, (q31_t)0x866ff3f6,
	(q31_t)0x280edb23, (q31_t)0x866dfc6e, (q31_t)0x2808e364, (q31_t)0x866c0531, (q31_t)0x2802eb8c, (q31_t)0x866a0e3f,
	(q31_t)0x27fcf39c, (q31_t)0x86681798,
	(q31_t)0x27f6fb92, (q31_t)0x8666213c, (q31_t)0x27f10371, (q31_t)0x86642b2c, (q31_t)0x27eb0b36, (q31_t)0x86623566,
	(q31_t)0x27e512e3, (q31_t)0x86603feb,
	(q31_t)0x27df1a77, (q31_t)0x865e4abb, (q31_t)0x27d921f3, (q31_t)0x865c55d7, (q31_t)0x27d32956, (q31_t)0x865a613d,
	(q31_t)0x27cd30a1, (q31_t)0x86586cee,
	(q31_t)0x27c737d3, (q31_t)0x865678eb, (q31_t)0x27c13eec, (q31_t)0x86548532, (q31_t)0x27bb45ed, (q31_t)0x865291c4,
	(q31_t)0x27b54cd6, (q31_t)0x86509ea2,
	(q31_t)0x27af53a6, (q31_t)0x864eabcb, (q31_t)0x27a95a5d, (q31_t)0x864cb93e, (q31_t)0x27a360fc, (q31_t)0x864ac6fd,
	(q31_t)0x279d6783, (q31_t)0x8648d507,
	(q31_t)0x27976df1, (q31_t)0x8646e35c, (q31_t)0x27917447, (q31_t)0x8644f1fc, (q31_t)0x278b7a84, (q31_t)0x864300e7,
	(q31_t)0x278580a9, (q31_t)0x8641101d,
	(q31_t)0x277f86b5, (q31_t)0x863f1f9e, (q31_t)0x27798caa, (q31_t)0x863d2f6b, (q31_t)0x27739285, (q31_t)0x863b3f82,
	(q31_t)0x276d9849, (q31_t)0x86394fe5,
	(q31_t)0x27679df4, (q31_t)0x86376092, (q31_t)0x2761a387, (q31_t)0x8635718b, (q31_t)0x275ba901, (q31_t)0x863382cf,
	(q31_t)0x2755ae64, (q31_t)0x8631945e,
	(q31_t)0x274fb3ae, (q31_t)0x862fa638, (q31_t)0x2749b8e0, (q31_t)0x862db85e, (q31_t)0x2743bdf9, (q31_t)0x862bcace,
	(q31_t)0x273dc2fa, (q31_t)0x8629dd8a,
	(q31_t)0x2737c7e3, (q31_t)0x8627f091, (q31_t)0x2731ccb4, (q31_t)0x862603e3, (q31_t)0x272bd16d, (q31_t)0x86241780,
	(q31_t)0x2725d60e, (q31_t)0x86222b68,
	(q31_t)0x271fda96, (q31_t)0x86203f9c, (q31_t)0x2719df06, (q31_t)0x861e541a, (q31_t)0x2713e35f, (q31_t)0x861c68e4,
	(q31_t)0x270de79f, (q31_t)0x861a7df9,
	(q31_t)0x2707ebc7, (q31_t)0x86189359, (q31_t)0x2701efd7, (q31_t)0x8616a905, (q31_t)0x26fbf3ce, (q31_t)0x8614befb,
	(q31_t)0x26f5f7ae, (q31_t)0x8612d53d,
	(q31_t)0x26effb76, (q31_t)0x8610ebca, (q31_t)0x26e9ff26, (q31_t)0x860f02a3, (q31_t)0x26e402bd, (q31_t)0x860d19c6,
	(q31_t)0x26de063d, (q31_t)0x860b3135,
	(q31_t)0x26d809a5, (q31_t)0x860948ef, (q31_t)0x26d20cf5, (q31_t)0x860760f4, (q31_t)0x26cc102d, (q31_t)0x86057944,
	(q31_t)0x26c6134d, (q31_t)0x860391e0,
	(q31_t)0x26c01655, (q31_t)0x8601aac7, (q31_t)0x26ba1945, (q31_t)0x85ffc3f9, (q31_t)0x26b41c1d, (q31_t)0x85fddd76,
	(q31_t)0x26ae1edd, (q31_t)0x85fbf73f,
	(q31_t)0x26a82186, (q31_t)0x85fa1153, (q31_t)0x26a22416, (q31_t)0x85f82bb2, (q31_t)0x269c268f, (q31_t)0x85f6465c,
	(q31_t)0x269628f0, (q31_t)0x85f46152,
	(q31_t)0x26902b39, (q31_t)0x85f27c93, (q31_t)0x268a2d6b, (q31_t)0x85f09820, (q31_t)0x26842f84, (q31_t)0x85eeb3f7,
	(q31_t)0x267e3186, (q31_t)0x85ecd01a,
	(q31_t)0x26783370, (q31_t)0x85eaec88, (q31_t)0x26723543, (q31_t)0x85e90942, (q31_t)0x266c36fe, (q31_t)0x85e72647,
	(q31_t)0x266638a1, (q31_t)0x85e54397,
	(q31_t)0x26603a2c, (q31_t)0x85e36132, (q31_t)0x265a3b9f, (q31_t)0x85e17f19, (q31_t)0x26543cfb, (q31_t)0x85df9d4b,
	(q31_t)0x264e3e40, (q31_t)0x85ddbbc9,
	(q31_t)0x26483f6c, (q31_t)0x85dbda91, (q31_t)0x26424082, (q31_t)0x85d9f9a5, (q31_t)0x263c417f, (q31_t)0x85d81905,
	(q31_t)0x26364265, (q31_t)0x85d638b0,
	(q31_t)0x26304333, (q31_t)0x85d458a6, (q31_t)0x262a43ea, (q31_t)0x85d278e7, (q31_t)0x26244489, (q31_t)0x85d09974,
	(q31_t)0x261e4511, (q31_t)0x85ceba4d,
	(q31_t)0x26184581, (q31_t)0x85ccdb70, (q31_t)0x261245da, (q31_t)0x85cafcdf, (q31_t)0x260c461b, (q31_t)0x85c91e9a,
	(q31_t)0x26064645, (q31_t)0x85c740a0,
	(q31_t)0x26004657, (q31_t)0x85c562f1, (q31_t)0x25fa4652, (q31_t)0x85c3858d, (q31_t)0x25f44635, (q31_t)0x85c1a875,
	(q31_t)0x25ee4601, (q31_t)0x85bfcba9,
	(q31_t)0x25e845b6, (q31_t)0x85bdef28, (q31_t)0x25e24553, (q31_t)0x85bc12f2, (q31_t)0x25dc44d9, (q31_t)0x85ba3707,
	(q31_t)0x25d64447, (q31_t)0x85b85b68,
	(q31_t)0x25d0439f, (q31_t)0x85b68015, (q31_t)0x25ca42de, (q31_t)0x85b4a50d, (q31_t)0x25c44207, (q31_t)0x85b2ca50,
	(q31_t)0x25be4118, (q31_t)0x85b0efdf,
	(q31_t)0x25b84012, (q31_t)0x85af15b9, (q31_t)0x25b23ef5, (q31_t)0x85ad3bdf, (q31_t)0x25ac3dc0, (q31_t)0x85ab6250,
	(q31_t)0x25a63c74, (q31_t)0x85a9890d,
	(q31_t)0x25a03b11, (q31_t)0x85a7b015, (q31_t)0x259a3997, (q31_t)0x85a5d768, (q31_t)0x25943806, (q31_t)0x85a3ff07,
	(q31_t)0x258e365d, (q31_t)0x85a226f2,
	(q31_t)0x2588349d, (q31_t)0x85a04f28, (q31_t)0x258232c6, (q31_t)0x859e77a9, (q31_t)0x257c30d8, (q31_t)0x859ca076,
	(q31_t)0x25762ed3, (q31_t)0x859ac98f,
	(q31_t)0x25702cb7, (q31_t)0x8598f2f3, (q31_t)0x256a2a83, (q31_t)0x85971ca2, (q31_t)0x25642839, (q31_t)0x8595469d,
	(q31_t)0x255e25d7, (q31_t)0x859370e4,
	(q31_t)0x2558235f, (q31_t)0x85919b76, (q31_t)0x255220cf, (q31_t)0x858fc653, (q31_t)0x254c1e28, (q31_t)0x858df17c,
	(q31_t)0x25461b6b, (q31_t)0x858c1cf1,
	(q31_t)0x25401896, (q31_t)0x858a48b1, (q31_t)0x253a15aa, (q31_t)0x858874bd, (q31_t)0x253412a8, (q31_t)0x8586a114,
	(q31_t)0x252e0f8e, (q31_t)0x8584cdb7,
	(q31_t)0x25280c5e, (q31_t)0x8582faa5, (q31_t)0x25220916, (q31_t)0x858127df, (q31_t)0x251c05b8, (q31_t)0x857f5564,
	(q31_t)0x25160243, (q31_t)0x857d8335,
	(q31_t)0x250ffeb7, (q31_t)0x857bb152, (q31_t)0x2509fb14, (q31_t)0x8579dfba, (q31_t)0x2503f75a, (q31_t)0x85780e6e,
	(q31_t)0x24fdf389, (q31_t)0x85763d6d,
	(q31_t)0x24f7efa2, (q31_t)0x85746cb8, (q31_t)0x24f1eba4, (q31_t)0x85729c4e, (q31_t)0x24ebe78f, (q31_t)0x8570cc30,
	(q31_t)0x24e5e363, (q31_t)0x856efc5e,
	(q31_t)0x24dfdf20, (q31_t)0x856d2cd7, (q31_t)0x24d9dac7, (q31_t)0x856b5d9c, (q31_t)0x24d3d657, (q31_t)0x85698ead,
	(q31_t)0x24cdd1d0, (q31_t)0x8567c009,
	(q31_t)0x24c7cd33, (q31_t)0x8565f1b0, (q31_t)0x24c1c87f, (q31_t)0x856423a4, (q31_t)0x24bbc3b4, (q31_t)0x856255e3,
	(q31_t)0x24b5bed2, (q31_t)0x8560886d,
	(q31_t)0x24afb9da, (q31_t)0x855ebb44, (q31_t)0x24a9b4cb, (q31_t)0x855cee66, (q31_t)0x24a3afa6, (q31_t)0x855b21d3,
	(q31_t)0x249daa6a, (q31_t)0x8559558c,
	(q31_t)0x2497a517, (q31_t)0x85578991, (q31_t)0x24919fae, (q31_t)0x8555bde2, (q31_t)0x248b9a2f, (q31_t)0x8553f27e,
	(q31_t)0x24859498, (q31_t)0x85522766,
	(q31_t)0x247f8eec, (q31_t)0x85505c99, (q31_t)0x24798928, (q31_t)0x854e9219, (q31_t)0x2473834f, (q31_t)0x854cc7e3,
	(q31_t)0x246d7d5e, (q31_t)0x854afdfa,
	(q31_t)0x24677758, (q31_t)0x8549345c, (q31_t)0x2461713a, (q31_t)0x85476b0a, (q31_t)0x245b6b07, (q31_t)0x8545a204,
	(q31_t)0x245564bd, (q31_t)0x8543d949,
	(q31_t)0x244f5e5c, (q31_t)0x854210db, (q31_t)0x244957e5, (q31_t)0x854048b7, (q31_t)0x24435158, (q31_t)0x853e80e0,
	(q31_t)0x243d4ab4, (q31_t)0x853cb954,
	(q31_t)0x243743fa, (q31_t)0x853af214, (q31_t)0x24313d2a, (q31_t)0x85392b20, (q31_t)0x242b3644, (q31_t)0x85376477,
	(q31_t)0x24252f47, (q31_t)0x85359e1a,
	(q31_t)0x241f2833, (q31_t)0x8533d809, (q31_t)0x2419210a, (q31_t)0x85321244, (q31_t)0x241319ca, (q31_t)0x85304cca,
	(q31_t)0x240d1274, (q31_t)0x852e879d,
	(q31_t)0x24070b08, (q31_t)0x852cc2bb, (q31_t)0x24010385, (q31_t)0x852afe24, (q31_t)0x23fafbec, (q31_t)0x852939da,
	(q31_t)0x23f4f43e, (q31_t)0x852775db,
	(q31_t)0x23eeec78, (q31_t)0x8525b228, (q31_t)0x23e8e49d, (q31_t)0x8523eec1, (q31_t)0x23e2dcac, (q31_t)0x85222ba5,
	(q31_t)0x23dcd4a4, (q31_t)0x852068d6,
	(q31_t)0x23d6cc87, (q31_t)0x851ea652, (q31_t)0x23d0c453, (q31_t)0x851ce41a, (q31_t)0x23cabc09, (q31_t)0x851b222e,
	(q31_t)0x23c4b3a9, (q31_t)0x8519608d,
	(q31_t)0x23beab33, (q31_t)0x85179f39, (q31_t)0x23b8a2a7, (q31_t)0x8515de30, (q31_t)0x23b29a05, (q31_t)0x85141d73,
	(q31_t)0x23ac914d, (q31_t)0x85125d02,
	(q31_t)0x23a6887f, (q31_t)0x85109cdd, (q31_t)0x23a07f9a, (q31_t)0x850edd03, (q31_t)0x239a76a0, (q31_t)0x850d1d75,
	(q31_t)0x23946d90, (q31_t)0x850b5e34,
	(q31_t)0x238e646a, (q31_t)0x85099f3e, (q31_t)0x23885b2e, (q31_t)0x8507e094, (q31_t)0x238251dd, (q31_t)0x85062235,
	(q31_t)0x237c4875, (q31_t)0x85046423,
	(q31_t)0x23763ef7, (q31_t)0x8502a65c, (q31_t)0x23703564, (q31_t)0x8500e8e2, (q31_t)0x236a2bba, (q31_t)0x84ff2bb3,
	(q31_t)0x236421fb, (q31_t)0x84fd6ed0,
	(q31_t)0x235e1826, (q31_t)0x84fbb239, (q31_t)0x23580e3b, (q31_t)0x84f9f5ee, (q31_t)0x2352043b, (q31_t)0x84f839ee,
	(q31_t)0x234bfa24, (q31_t)0x84f67e3b,
	(q31_t)0x2345eff8, (q31_t)0x84f4c2d4, (q31_t)0x233fe5b6, (q31_t)0x84f307b8, (q31_t)0x2339db5e, (q31_t)0x84f14ce8,
	(q31_t)0x2333d0f1, (q31_t)0x84ef9265,
	(q31_t)0x232dc66d, (q31_t)0x84edd82d, (q31_t)0x2327bbd5, (q31_t)0x84ec1e41, (q31_t)0x2321b126, (q31_t)0x84ea64a1,
	(q31_t)0x231ba662, (q31_t)0x84e8ab4d,
	(q31_t)0x23159b88, (q31_t)0x84e6f244, (q31_t)0x230f9098, (q31_t)0x84e53988, (q31_t)0x23098593, (q31_t)0x84e38118,
	(q31_t)0x23037a78, (q31_t)0x84e1c8f3,
	(q31_t)0x22fd6f48, (q31_t)0x84e0111b, (q31_t)0x22f76402, (q31_t)0x84de598f, (q31_t)0x22f158a7, (q31_t)0x84dca24e,
	(q31_t)0x22eb4d36, (q31_t)0x84daeb5a,
	(q31_t)0x22e541af, (q31_t)0x84d934b1, (q31_t)0x22df3613, (q31_t)0x84d77e54, (q31_t)0x22d92a61, (q31_t)0x84d5c844,
	(q31_t)0x22d31e9a, (q31_t)0x84d4127f,
	(q31_t)0x22cd12bd, (q31_t)0x84d25d06, (q31_t)0x22c706cb, (q31_t)0x84d0a7da, (q31_t)0x22c0fac4, (q31_t)0x84cef2f9,
	(q31_t)0x22baeea7, (q31_t)0x84cd3e64,
	(q31_t)0x22b4e274, (q31_t)0x84cb8a1b, (q31_t)0x22aed62c, (q31_t)0x84c9d61f, (q31_t)0x22a8c9cf, (q31_t)0x84c8226e,
	(q31_t)0x22a2bd5d, (q31_t)0x84c66f09,
	(q31_t)0x229cb0d5, (q31_t)0x84c4bbf0, (q31_t)0x2296a437, (q31_t)0x84c30924, (q31_t)0x22909785, (q31_t)0x84c156a3,
	(q31_t)0x228a8abd, (q31_t)0x84bfa46e,
	(q31_t)0x22847de0, (q31_t)0x84bdf286, (q31_t)0x227e70ed, (q31_t)0x84bc40e9, (q31_t)0x227863e5, (q31_t)0x84ba8f98,
	(q31_t)0x227256c8, (q31_t)0x84b8de94,
	(q31_t)0x226c4996, (q31_t)0x84b72ddb, (q31_t)0x22663c4e, (q31_t)0x84b57d6f, (q31_t)0x22602ef1, (q31_t)0x84b3cd4f,
	(q31_t)0x225a217f, (q31_t)0x84b21d7a,
	(q31_t)0x225413f8, (q31_t)0x84b06df2, (q31_t)0x224e065c, (q31_t)0x84aebeb6, (q31_t)0x2247f8aa, (q31_t)0x84ad0fc6,
	(q31_t)0x2241eae3, (q31_t)0x84ab6122,
	(q31_t)0x223bdd08, (q31_t)0x84a9b2ca, (q31_t)0x2235cf17, (q31_t)0x84a804be, (q31_t)0x222fc111, (q31_t)0x84a656fe,
	(q31_t)0x2229b2f6, (q31_t)0x84a4a98a,
	(q31_t)0x2223a4c5, (q31_t)0x84a2fc62, (q31_t)0x221d9680, (q31_t)0x84a14f87, (q31_t)0x22178826, (q31_t)0x849fa2f7,
	(q31_t)0x221179b7, (q31_t)0x849df6b4,
	(q31_t)0x220b6b32, (q31_t)0x849c4abd, (q31_t)0x22055c99, (q31_t)0x849a9f12, (q31_t)0x21ff4dea, (q31_t)0x8498f3b3,
	(q31_t)0x21f93f27, (q31_t)0x849748a0,
	(q31_t)0x21f3304f, (q31_t)0x84959dd9, (q31_t)0x21ed2162, (q31_t)0x8493f35e, (q31_t)0x21e71260, (q31_t)0x84924930,
	(q31_t)0x21e10349, (q31_t)0x84909f4e,
	(q31_t)0x21daf41d, (q31_t)0x848ef5b7, (q31_t)0x21d4e4dc, (q31_t)0x848d4c6d, (q31_t)0x21ced586, (q31_t)0x848ba36f,
	(q31_t)0x21c8c61c, (q31_t)0x8489fabe,
	(q31_t)0x21c2b69c, (q31_t)0x84885258, (q31_t)0x21bca708, (q31_t)0x8486aa3e, (q31_t)0x21b6975f, (q31_t)0x84850271,
	(q31_t)0x21b087a1, (q31_t)0x84835af0,
	(q31_t)0x21aa77cf, (q31_t)0x8481b3bb, (q31_t)0x21a467e7, (q31_t)0x84800cd2, (q31_t)0x219e57eb, (q31_t)0x847e6636,
	(q31_t)0x219847da, (q31_t)0x847cbfe5,
	(q31_t)0x219237b5, (q31_t)0x847b19e1, (q31_t)0x218c277a, (q31_t)0x84797429, (q31_t)0x2186172b, (q31_t)0x8477cebd,
	(q31_t)0x218006c8, (q31_t)0x8476299e,
	(q31_t)0x2179f64f, (q31_t)0x847484ca, (q31_t)0x2173e5c2, (q31_t)0x8472e043, (q31_t)0x216dd521, (q31_t)0x84713c08,
	(q31_t)0x2167c46b, (q31_t)0x846f9819,
	(q31_t)0x2161b3a0, (q31_t)0x846df477, (q31_t)0x215ba2c0, (q31_t)0x846c5120, (q31_t)0x215591cc, (q31_t)0x846aae16,
	(q31_t)0x214f80c4, (q31_t)0x84690b58,
	(q31_t)0x21496fa7, (q31_t)0x846768e7, (q31_t)0x21435e75, (q31_t)0x8465c6c1, (q31_t)0x213d4d2f, (q31_t)0x846424e8,
	(q31_t)0x21373bd4, (q31_t)0x8462835b,
	(q31_t)0x21312a65, (q31_t)0x8460e21a, (q31_t)0x212b18e1, (q31_t)0x845f4126, (q31_t)0x21250749, (q31_t)0x845da07e,
	(q31_t)0x211ef59d, (q31_t)0x845c0022,
	(q31_t)0x2118e3dc, (q31_t)0x845a6012, (q31_t)0x2112d206, (q31_t)0x8458c04f, (q31_t)0x210cc01d, (q31_t)0x845720d8,
	(q31_t)0x2106ae1e, (q31_t)0x845581ad,
	(q31_t)0x21009c0c, (q31_t)0x8453e2cf, (q31_t)0x20fa89e5, (q31_t)0x8452443d, (q31_t)0x20f477aa, (q31_t)0x8450a5f7,
	(q31_t)0x20ee655a, (q31_t)0x844f07fd,
	(q31_t)0x20e852f6, (q31_t)0x844d6a50, (q31_t)0x20e2407e, (q31_t)0x844bccef, (q31_t)0x20dc2df2, (q31_t)0x844a2fda,
	(q31_t)0x20d61b51, (q31_t)0x84489311,
	(q31_t)0x20d0089c, (q31_t)0x8446f695, (q31_t)0x20c9f5d3, (q31_t)0x84455a66, (q31_t)0x20c3e2f5, (q31_t)0x8443be82,
	(q31_t)0x20bdd003, (q31_t)0x844222eb,
	(q31_t)0x20b7bcfe, (q31_t)0x844087a0, (q31_t)0x20b1a9e4, (q31_t)0x843eeca2, (q31_t)0x20ab96b5, (q31_t)0x843d51f0,
	(q31_t)0x20a58373, (q31_t)0x843bb78a,
	(q31_t)0x209f701c, (q31_t)0x843a1d70, (q31_t)0x20995cb2, (q31_t)0x843883a3, (q31_t)0x20934933, (q31_t)0x8436ea23,
	(q31_t)0x208d35a0, (q31_t)0x843550ee,
	(q31_t)0x208721f9, (q31_t)0x8433b806, (q31_t)0x20810e3e, (q31_t)0x84321f6b, (q31_t)0x207afa6f, (q31_t)0x8430871b,
	(q31_t)0x2074e68c, (q31_t)0x842eef18,
	(q31_t)0x206ed295, (q31_t)0x842d5762, (q31_t)0x2068be8a, (q31_t)0x842bbff8, (q31_t)0x2062aa6b, (q31_t)0x842a28da,
	(q31_t)0x205c9638, (q31_t)0x84289209,
	(q31_t)0x205681f1, (q31_t)0x8426fb84, (q31_t)0x20506d96, (q31_t)0x8425654b, (q31_t)0x204a5927, (q31_t)0x8423cf5f,
	(q31_t)0x204444a4, (q31_t)0x842239bf,
	(q31_t)0x203e300d, (q31_t)0x8420a46c, (q31_t)0x20381b63, (q31_t)0x841f0f65, (q31_t)0x203206a4, (q31_t)0x841d7aaa,
	(q31_t)0x202bf1d2, (q31_t)0x841be63c,
	(q31_t)0x2025dcec, (q31_t)0x841a521a, (q31_t)0x201fc7f2, (q31_t)0x8418be45, (q31_t)0x2019b2e4, (q31_t)0x84172abc,
	(q31_t)0x20139dc2, (q31_t)0x84159780,
	(q31_t)0x200d888d, (q31_t)0x84140490, (q31_t)0x20077344, (q31_t)0x841271ec, (q31_t)0x20015de7, (q31_t)0x8410df95,
	(q31_t)0x1ffb4876, (q31_t)0x840f4d8a,
	(q31_t)0x1ff532f2, (q31_t)0x840dbbcc, (q31_t)0x1fef1d59, (q31_t)0x840c2a5a, (q31_t)0x1fe907ae, (q31_t)0x840a9935,
	(q31_t)0x1fe2f1ee, (q31_t)0x8409085c,
	(q31_t)0x1fdcdc1b, (q31_t)0x840777d0, (q31_t)0x1fd6c634, (q31_t)0x8405e790, (q31_t)0x1fd0b03a, (q31_t)0x8404579d,
	(q31_t)0x1fca9a2b, (q31_t)0x8402c7f6,
	(q31_t)0x1fc4840a, (q31_t)0x8401389b, (q31_t)0x1fbe6dd4, (q31_t)0x83ffa98d, (q31_t)0x1fb8578b, (q31_t)0x83fe1acc,
	(q31_t)0x1fb2412f, (q31_t)0x83fc8c57,
	(q31_t)0x1fac2abf, (q31_t)0x83fafe2e, (q31_t)0x1fa6143b, (q31_t)0x83f97052, (q31_t)0x1f9ffda4, (q31_t)0x83f7e2c3,
	(q31_t)0x1f99e6fa, (q31_t)0x83f65580,
	(q31_t)0x1f93d03c, (q31_t)0x83f4c889, (q31_t)0x1f8db96a, (q31_t)0x83f33bdf, (q31_t)0x1f87a285, (q31_t)0x83f1af82,
	(q31_t)0x1f818b8d, (q31_t)0x83f02371,
	(q31_t)0x1f7b7481, (q31_t)0x83ee97ad, (q31_t)0x1f755d61, (q31_t)0x83ed0c35, (q31_t)0x1f6f462f, (q31_t)0x83eb810a,
	(q31_t)0x1f692ee9, (q31_t)0x83e9f62b,
	(q31_t)0x1f63178f, (q31_t)0x83e86b99, (q31_t)0x1f5d0022, (q31_t)0x83e6e153, (q31_t)0x1f56e8a2, (q31_t)0x83e5575a,
	(q31_t)0x1f50d10e, (q31_t)0x83e3cdad,
	(q31_t)0x1f4ab968, (q31_t)0x83e2444d, (q31_t)0x1f44a1ad, (q31_t)0x83e0bb3a, (q31_t)0x1f3e89e0, (q31_t)0x83df3273,
	(q31_t)0x1f3871ff, (q31_t)0x83dda9f9,
	(q31_t)0x1f325a0b, (q31_t)0x83dc21cb, (q31_t)0x1f2c4204, (q31_t)0x83da99ea, (q31_t)0x1f2629ea, (q31_t)0x83d91255,
	(q31_t)0x1f2011bc, (q31_t)0x83d78b0d,
	(q31_t)0x1f19f97b, (q31_t)0x83d60412, (q31_t)0x1f13e127, (q31_t)0x83d47d63, (q31_t)0x1f0dc8c0, (q31_t)0x83d2f701,
	(q31_t)0x1f07b045, (q31_t)0x83d170eb,
	(q31_t)0x1f0197b8, (q31_t)0x83cfeb22, (q31_t)0x1efb7f17, (q31_t)0x83ce65a6, (q31_t)0x1ef56664, (q31_t)0x83cce076,
	(q31_t)0x1eef4d9d, (q31_t)0x83cb5b93,
	(q31_t)0x1ee934c3, (q31_t)0x83c9d6fc, (q31_t)0x1ee31bd6, (q31_t)0x83c852b2, (q31_t)0x1edd02d6, (q31_t)0x83c6ceb5,
	(q31_t)0x1ed6e9c3, (q31_t)0x83c54b04,
	(q31_t)0x1ed0d09d, (q31_t)0x83c3c7a0, (q31_t)0x1ecab763, (q31_t)0x83c24488, (q31_t)0x1ec49e17, (q31_t)0x83c0c1be,
	(q31_t)0x1ebe84b8, (q31_t)0x83bf3f3f,
	(q31_t)0x1eb86b46, (q31_t)0x83bdbd0e, (q31_t)0x1eb251c1, (q31_t)0x83bc3b29, (q31_t)0x1eac3829, (q31_t)0x83bab991,
	(q31_t)0x1ea61e7e, (q31_t)0x83b93845,
	(q31_t)0x1ea004c1, (q31_t)0x83b7b746, (q31_t)0x1e99eaf0, (q31_t)0x83b63694, (q31_t)0x1e93d10c, (q31_t)0x83b4b62e,
	(q31_t)0x1e8db716, (q31_t)0x83b33616,
	(q31_t)0x1e879d0d, (q31_t)0x83b1b649, (q31_t)0x1e8182f1, (q31_t)0x83b036ca, (q31_t)0x1e7b68c2, (q31_t)0x83aeb797,
	(q31_t)0x1e754e80, (q31_t)0x83ad38b1,
	(q31_t)0x1e6f342c, (q31_t)0x83abba17, (q31_t)0x1e6919c4, (q31_t)0x83aa3bca, (q31_t)0x1e62ff4a, (q31_t)0x83a8bdca,
	(q31_t)0x1e5ce4be, (q31_t)0x83a74017,
	(q31_t)0x1e56ca1e, (q31_t)0x83a5c2b0, (q31_t)0x1e50af6c, (q31_t)0x83a44596, (q31_t)0x1e4a94a7, (q31_t)0x83a2c8c9,
	(q31_t)0x1e4479cf, (q31_t)0x83a14c48,
	(q31_t)0x1e3e5ee5, (q31_t)0x839fd014, (q31_t)0x1e3843e8, (q31_t)0x839e542d, (q31_t)0x1e3228d9, (q31_t)0x839cd893,
	(q31_t)0x1e2c0db6, (q31_t)0x839b5d45,
	(q31_t)0x1e25f282, (q31_t)0x8399e244, (q31_t)0x1e1fd73a, (q31_t)0x83986790, (q31_t)0x1e19bbe0, (q31_t)0x8396ed29,
	(q31_t)0x1e13a074, (q31_t)0x8395730e,
	(q31_t)0x1e0d84f5, (q31_t)0x8393f940, (q31_t)0x1e076963, (q31_t)0x83927fbf, (q31_t)0x1e014dbf, (q31_t)0x8391068a,
	(q31_t)0x1dfb3208, (q31_t)0x838f8da2,
	(q31_t)0x1df5163f, (q31_t)0x838e1507, (q31_t)0x1deefa63, (q31_t)0x838c9cb9, (q31_t)0x1de8de75, (q31_t)0x838b24b8,
	(q31_t)0x1de2c275, (q31_t)0x8389ad03,
	(q31_t)0x1ddca662, (q31_t)0x8388359b, (q31_t)0x1dd68a3c, (q31_t)0x8386be80, (q31_t)0x1dd06e04, (q31_t)0x838547b2,
	(q31_t)0x1dca51ba, (q31_t)0x8383d130,
	(q31_t)0x1dc4355e, (q31_t)0x83825afb, (q31_t)0x1dbe18ef, (q31_t)0x8380e513, (q31_t)0x1db7fc6d, (q31_t)0x837f6f78,
	(q31_t)0x1db1dfda, (q31_t)0x837dfa2a,
	(q31_t)0x1dabc334, (q31_t)0x837c8528, (q31_t)0x1da5a67c, (q31_t)0x837b1074, (q31_t)0x1d9f89b1, (q31_t)0x83799c0c,
	(q31_t)0x1d996cd4, (q31_t)0x837827f0,
	(q31_t)0x1d934fe5, (q31_t)0x8376b422, (q31_t)0x1d8d32e4, (q31_t)0x837540a1, (q31_t)0x1d8715d0, (q31_t)0x8373cd6c,
	(q31_t)0x1d80f8ab, (q31_t)0x83725a84,
	(q31_t)0x1d7adb73, (q31_t)0x8370e7e9, (q31_t)0x1d74be29, (q31_t)0x836f759b, (q31_t)0x1d6ea0cc, (q31_t)0x836e039a,
	(q31_t)0x1d68835e, (q31_t)0x836c91e5,
	(q31_t)0x1d6265dd, (q31_t)0x836b207d, (q31_t)0x1d5c484b, (q31_t)0x8369af63, (q31_t)0x1d562aa6, (q31_t)0x83683e95,
	(q31_t)0x1d500cef, (q31_t)0x8366ce14,
	(q31_t)0x1d49ef26, (q31_t)0x83655ddf, (q31_t)0x1d43d14b, (q31_t)0x8363edf8, (q31_t)0x1d3db35e, (q31_t)0x83627e5d,
	(q31_t)0x1d37955e, (q31_t)0x83610f10,
	(q31_t)0x1d31774d, (q31_t)0x835fa00f, (q31_t)0x1d2b592a, (q31_t)0x835e315b, (q31_t)0x1d253af5, (q31_t)0x835cc2f4,
	(q31_t)0x1d1f1cae, (q31_t)0x835b54da,
	(q31_t)0x1d18fe54, (q31_t)0x8359e70d, (q31_t)0x1d12dfe9, (q31_t)0x8358798c, (q31_t)0x1d0cc16c, (q31_t)0x83570c59,
	(q31_t)0x1d06a2dd, (q31_t)0x83559f72,
	(q31_t)0x1d00843d, (q31_t)0x835432d8, (q31_t)0x1cfa658a, (q31_t)0x8352c68c, (q31_t)0x1cf446c5, (q31_t)0x83515a8c,
	(q31_t)0x1cee27ef, (q31_t)0x834feed9,
	(q31_t)0x1ce80906, (q31_t)0x834e8373, (q31_t)0x1ce1ea0c, (q31_t)0x834d185a, (q31_t)0x1cdbcb00, (q31_t)0x834bad8e,
	(q31_t)0x1cd5abe3, (q31_t)0x834a430e,
	(q31_t)0x1ccf8cb3, (q31_t)0x8348d8dc, (q31_t)0x1cc96d72, (q31_t)0x83476ef6, (q31_t)0x1cc34e1f, (q31_t)0x8346055e,
	(q31_t)0x1cbd2eba, (q31_t)0x83449c12,
	(q31_t)0x1cb70f43, (q31_t)0x83433314, (q31_t)0x1cb0efbb, (q31_t)0x8341ca62, (q31_t)0x1caad021, (q31_t)0x834061fd,
	(q31_t)0x1ca4b075, (q31_t)0x833ef9e6,
	(q31_t)0x1c9e90b8, (q31_t)0x833d921b, (q31_t)0x1c9870e9, (q31_t)0x833c2a9d, (q31_t)0x1c925109, (q31_t)0x833ac36c,
	(q31_t)0x1c8c3116, (q31_t)0x83395c88,
	(q31_t)0x1c861113, (q31_t)0x8337f5f1, (q31_t)0x1c7ff0fd, (q31_t)0x83368fa7, (q31_t)0x1c79d0d6, (q31_t)0x833529aa,
	(q31_t)0x1c73b09d, (q31_t)0x8333c3fa,
	(q31_t)0x1c6d9053, (q31_t)0x83325e97, (q31_t)0x1c676ff8, (q31_t)0x8330f981, (q31_t)0x1c614f8b, (q31_t)0x832f94b8,
	(q31_t)0x1c5b2f0c, (q31_t)0x832e303c,
	(q31_t)0x1c550e7c, (q31_t)0x832ccc0d, (q31_t)0x1c4eedda, (q31_t)0x832b682b, (q31_t)0x1c48cd27, (q31_t)0x832a0496,
	(q31_t)0x1c42ac62, (q31_t)0x8328a14d,
	(q31_t)0x1c3c8b8c, (q31_t)0x83273e52, (q31_t)0x1c366aa5, (q31_t)0x8325dba4, (q31_t)0x1c3049ac, (q31_t)0x83247943,
	(q31_t)0x1c2a28a2, (q31_t)0x8323172f,
	(q31_t)0x1c240786, (q31_t)0x8321b568, (q31_t)0x1c1de659, (q31_t)0x832053ee, (q31_t)0x1c17c51b, (q31_t)0x831ef2c1,
	(q31_t)0x1c11a3cb, (q31_t)0x831d91e1,
	(q31_t)0x1c0b826a, (q31_t)0x831c314e, (q31_t)0x1c0560f8, (q31_t)0x831ad109, (q31_t)0x1bff3f75, (q31_t)0x83197110,
	(q31_t)0x1bf91de0, (q31_t)0x83181164,
	(q31_t)0x1bf2fc3a, (q31_t)0x8316b205, (q31_t)0x1becda83, (q31_t)0x831552f4, (q31_t)0x1be6b8ba, (q31_t)0x8313f42f,
	(q31_t)0x1be096e0, (q31_t)0x831295b7,
	(q31_t)0x1bda74f6, (q31_t)0x8311378d, (q31_t)0x1bd452f9, (q31_t)0x830fd9af, (q31_t)0x1bce30ec, (q31_t)0x830e7c1f,
	(q31_t)0x1bc80ece, (q31_t)0x830d1edc,
	(q31_t)0x1bc1ec9e, (q31_t)0x830bc1e6, (q31_t)0x1bbbca5e, (q31_t)0x830a653c, (q31_t)0x1bb5a80c, (q31_t)0x830908e0,
	(q31_t)0x1baf85a9, (q31_t)0x8307acd1,
	(q31_t)0x1ba96335, (q31_t)0x83065110, (q31_t)0x1ba340b0, (q31_t)0x8304f59b, (q31_t)0x1b9d1e1a, (q31_t)0x83039a73,
	(q31_t)0x1b96fb73, (q31_t)0x83023f98,
	(q31_t)0x1b90d8bb, (q31_t)0x8300e50b, (q31_t)0x1b8ab5f2, (q31_t)0x82ff8acb, (q31_t)0x1b849317, (q31_t)0x82fe30d7,
	(q31_t)0x1b7e702c, (q31_t)0x82fcd731,
	(q31_t)0x1b784d30, (q31_t)0x82fb7dd8, (q31_t)0x1b722a23, (q31_t)0x82fa24cc, (q31_t)0x1b6c0705, (q31_t)0x82f8cc0d,
	(q31_t)0x1b65e3d7, (q31_t)0x82f7739c,
	(q31_t)0x1b5fc097, (q31_t)0x82f61b77, (q31_t)0x1b599d46, (q31_t)0x82f4c3a0, (q31_t)0x1b5379e5, (q31_t)0x82f36c15,
	(q31_t)0x1b4d5672, (q31_t)0x82f214d8,
	(q31_t)0x1b4732ef, (q31_t)0x82f0bde8, (q31_t)0x1b410f5b, (q31_t)0x82ef6745, (q31_t)0x1b3aebb6, (q31_t)0x82ee10ef,
	(q31_t)0x1b34c801, (q31_t)0x82ecbae7,
	(q31_t)0x1b2ea43a, (q31_t)0x82eb652b, (q31_t)0x1b288063, (q31_t)0x82ea0fbd, (q31_t)0x1b225c7b, (q31_t)0x82e8ba9c,
	(q31_t)0x1b1c3883, (q31_t)0x82e765c8,
	(q31_t)0x1b161479, (q31_t)0x82e61141, (q31_t)0x1b0ff05f, (q31_t)0x82e4bd07, (q31_t)0x1b09cc34, (q31_t)0x82e3691b,
	(q31_t)0x1b03a7f9, (q31_t)0x82e2157c,
	(q31_t)0x1afd83ad, (q31_t)0x82e0c22a, (q31_t)0x1af75f50, (q31_t)0x82df6f25, (q31_t)0x1af13ae3, (q31_t)0x82de1c6d,
	(q31_t)0x1aeb1665, (q31_t)0x82dcca02,
	(q31_t)0x1ae4f1d6, (q31_t)0x82db77e5, (q31_t)0x1adecd37, (q31_t)0x82da2615, (q31_t)0x1ad8a887, (q31_t)0x82d8d492,
	(q31_t)0x1ad283c7, (q31_t)0x82d7835c,
	(q31_t)0x1acc5ef6, (q31_t)0x82d63274, (q31_t)0x1ac63a14, (q31_t)0x82d4e1d8, (q31_t)0x1ac01522, (q31_t)0x82d3918a,
	(q31_t)0x1ab9f020, (q31_t)0x82d24189,
	(q31_t)0x1ab3cb0d, (q31_t)0x82d0f1d5, (q31_t)0x1aada5e9, (q31_t)0x82cfa26f, (q31_t)0x1aa780b6, (q31_t)0x82ce5356,
	(q31_t)0x1aa15b71, (q31_t)0x82cd048a,
	(q31_t)0x1a9b361d, (q31_t)0x82cbb60b, (q31_t)0x1a9510b7, (q31_t)0x82ca67d9, (q31_t)0x1a8eeb42, (q31_t)0x82c919f5,
	(q31_t)0x1a88c5bc, (q31_t)0x82c7cc5e,
	(q31_t)0x1a82a026, (q31_t)0x82c67f14, (q31_t)0x1a7c7a7f, (q31_t)0x82c53217, (q31_t)0x1a7654c8, (q31_t)0x82c3e568,
	(q31_t)0x1a702f01, (q31_t)0x82c29906,
	(q31_t)0x1a6a0929, (q31_t)0x82c14cf1, (q31_t)0x1a63e341, (q31_t)0x82c00129, (q31_t)0x1a5dbd49, (q31_t)0x82beb5af,
	(q31_t)0x1a579741, (q31_t)0x82bd6a82,
	(q31_t)0x1a517128, (q31_t)0x82bc1fa2, (q31_t)0x1a4b4aff, (q31_t)0x82bad50f, (q31_t)0x1a4524c6, (q31_t)0x82b98aca,
	(q31_t)0x1a3efe7c, (q31_t)0x82b840d2,
	(q31_t)0x1a38d823, (q31_t)0x82b6f727, (q31_t)0x1a32b1b9, (q31_t)0x82b5adca, (q31_t)0x1a2c8b3f, (q31_t)0x82b464ba,
	(q31_t)0x1a2664b5, (q31_t)0x82b31bf7,
	(q31_t)0x1a203e1b, (q31_t)0x82b1d381, (q31_t)0x1a1a1771, (q31_t)0x82b08b59, (q31_t)0x1a13f0b6, (q31_t)0x82af437e,
	(q31_t)0x1a0dc9ec, (q31_t)0x82adfbf0,
	(q31_t)0x1a07a311, (q31_t)0x82acb4b0, (q31_t)0x1a017c27, (q31_t)0x82ab6dbd, (q31_t)0x19fb552c, (q31_t)0x82aa2717,
	(q31_t)0x19f52e22, (q31_t)0x82a8e0bf,
	(q31_t)0x19ef0707, (q31_t)0x82a79ab3, (q31_t)0x19e8dfdc, (q31_t)0x82a654f6, (q31_t)0x19e2b8a2, (q31_t)0x82a50f85,
	(q31_t)0x19dc9157, (q31_t)0x82a3ca62,
	(q31_t)0x19d669fc, (q31_t)0x82a2858c, (q31_t)0x19d04292, (q31_t)0x82a14104, (q31_t)0x19ca1b17, (q31_t)0x829ffcc8,
	(q31_t)0x19c3f38d, (q31_t)0x829eb8db,
	(q31_t)0x19bdcbf3, (q31_t)0x829d753a, (q31_t)0x19b7a449, (q31_t)0x829c31e7, (q31_t)0x19b17c8f, (q31_t)0x829aeee1,
	(q31_t)0x19ab54c5, (q31_t)0x8299ac29,
	(q31_t)0x19a52ceb, (q31_t)0x829869be, (q31_t)0x199f0502, (q31_t)0x829727a0, (q31_t)0x1998dd09, (q31_t)0x8295e5cf,
	(q31_t)0x1992b4ff, (q31_t)0x8294a44c,
	(q31_t)0x198c8ce7, (q31_t)0x82936317, (q31_t)0x198664be, (q31_t)0x8292222e, (q31_t)0x19803c86, (q31_t)0x8290e194,
	(q31_t)0x197a143e, (q31_t)0x828fa146,
	(q31_t)0x1973ebe6, (q31_t)0x828e6146, (q31_t)0x196dc37e, (q31_t)0x828d2193, (q31_t)0x19679b07, (q31_t)0x828be22e,
	(q31_t)0x19617280, (q31_t)0x828aa316,
	(q31_t)0x195b49ea, (q31_t)0x8289644b, (q31_t)0x19552144, (q31_t)0x828825ce, (q31_t)0x194ef88e, (q31_t)0x8286e79e,
	(q31_t)0x1948cfc8, (q31_t)0x8285a9bb,
	(q31_t)0x1942a6f3, (q31_t)0x82846c26, (q31_t)0x193c7e0f, (q31_t)0x82832edf, (q31_t)0x1936551b, (q31_t)0x8281f1e4,
	(q31_t)0x19302c17, (q31_t)0x8280b538,
	(q31_t)0x192a0304, (q31_t)0x827f78d8, (q31_t)0x1923d9e1, (q31_t)0x827e3cc6, (q31_t)0x191db0af, (q31_t)0x827d0102,
	(q31_t)0x1917876d, (q31_t)0x827bc58a,
	(q31_t)0x19115e1c, (q31_t)0x827a8a61, (q31_t)0x190b34bb, (q31_t)0x82794f84, (q31_t)0x19050b4b, (q31_t)0x827814f6,
	(q31_t)0x18fee1cb, (q31_t)0x8276dab4,
	(q31_t)0x18f8b83c, (q31_t)0x8275a0c0, (q31_t)0x18f28e9e, (q31_t)0x8274671a, (q31_t)0x18ec64f0, (q31_t)0x82732dc0,
	(q31_t)0x18e63b33, (q31_t)0x8271f4b5,
	(q31_t)0x18e01167, (q31_t)0x8270bbf7, (q31_t)0x18d9e78b, (q31_t)0x826f8386, (q31_t)0x18d3bda0, (q31_t)0x826e4b62,
	(q31_t)0x18cd93a5, (q31_t)0x826d138d,
	(q31_t)0x18c7699b, (q31_t)0x826bdc04, (q31_t)0x18c13f82, (q31_t)0x826aa4c9, (q31_t)0x18bb155a, (q31_t)0x82696ddc,
	(q31_t)0x18b4eb22, (q31_t)0x8268373c,
	(q31_t)0x18aec0db, (q31_t)0x826700e9, (q31_t)0x18a89685, (q31_t)0x8265cae4, (q31_t)0x18a26c20, (q31_t)0x8264952d,
	(q31_t)0x189c41ab, (q31_t)0x82635fc2,
	(q31_t)0x18961728, (q31_t)0x82622aa6, (q31_t)0x188fec95, (q31_t)0x8260f5d7, (q31_t)0x1889c1f3, (q31_t)0x825fc155,
	(q31_t)0x18839742, (q31_t)0x825e8d21,
	(q31_t)0x187d6c82, (q31_t)0x825d593a, (q31_t)0x187741b2, (q31_t)0x825c25a1, (q31_t)0x187116d4, (q31_t)0x825af255,
	(q31_t)0x186aebe6, (q31_t)0x8259bf57,
	(q31_t)0x1864c0ea, (q31_t)0x82588ca7, (q31_t)0x185e95de, (q31_t)0x82575a44, (q31_t)0x18586ac3, (q31_t)0x8256282e,
	(q31_t)0x18523f9a, (q31_t)0x8254f666,
	(q31_t)0x184c1461, (q31_t)0x8253c4eb, (q31_t)0x1845e919, (q31_t)0x825293be, (q31_t)0x183fbdc3, (q31_t)0x825162df,
	(q31_t)0x1839925d, (q31_t)0x8250324d,
	(q31_t)0x183366e9, (q31_t)0x824f0208, (q31_t)0x182d3b65, (q31_t)0x824dd211, (q31_t)0x18270fd3, (q31_t)0x824ca268,
	(q31_t)0x1820e431, (q31_t)0x824b730c,
	(q31_t)0x181ab881, (q31_t)0x824a43fe, (q31_t)0x18148cc2, (q31_t)0x8249153d, (q31_t)0x180e60f4, (q31_t)0x8247e6ca,
	(q31_t)0x18083518, (q31_t)0x8246b8a4,
	(q31_t)0x1802092c, (q31_t)0x82458acc, (q31_t)0x17fbdd32, (q31_t)0x82445d41, (q31_t)0x17f5b129, (q31_t)0x82433004,
	(q31_t)0x17ef8511, (q31_t)0x82420315,
	(q31_t)0x17e958ea, (q31_t)0x8240d673, (q31_t)0x17e32cb5, (q31_t)0x823faa1e, (q31_t)0x17dd0070, (q31_t)0x823e7e18,
	(q31_t)0x17d6d41d, (q31_t)0x823d525e,
	(q31_t)0x17d0a7bc, (q31_t)0x823c26f3, (q31_t)0x17ca7b4c, (q31_t)0x823afbd5, (q31_t)0x17c44ecd, (q31_t)0x8239d104,
	(q31_t)0x17be223f, (q31_t)0x8238a681,
	(q31_t)0x17b7f5a3, (q31_t)0x82377c4c, (q31_t)0x17b1c8f8, (q31_t)0x82365264, (q31_t)0x17ab9c3e, (q31_t)0x823528ca,
	(q31_t)0x17a56f76, (q31_t)0x8233ff7e,
	(q31_t)0x179f429f, (q31_t)0x8232d67f, (q31_t)0x179915ba, (q31_t)0x8231adce, (q31_t)0x1792e8c6, (q31_t)0x8230856a,
	(q31_t)0x178cbbc4, (q31_t)0x822f5d54,
	(q31_t)0x17868eb3, (q31_t)0x822e358b, (q31_t)0x17806194, (q31_t)0x822d0e10, (q31_t)0x177a3466, (q31_t)0x822be6e3,
	(q31_t)0x17740729, (q31_t)0x822ac004,
	(q31_t)0x176dd9de, (q31_t)0x82299971, (q31_t)0x1767ac85, (q31_t)0x8228732d, (q31_t)0x17617f1d, (q31_t)0x82274d36,
	(q31_t)0x175b51a7, (q31_t)0x8226278d,
	(q31_t)0x17552422, (q31_t)0x82250232, (q31_t)0x174ef68f, (q31_t)0x8223dd24, (q31_t)0x1748c8ee, (q31_t)0x8222b863,
	(q31_t)0x17429b3e, (q31_t)0x822193f1,
	(q31_t)0x173c6d80, (q31_t)0x82206fcc, (q31_t)0x17363fb4, (q31_t)0x821f4bf5, (q31_t)0x173011d9, (q31_t)0x821e286b,
	(q31_t)0x1729e3f0, (q31_t)0x821d052f,
	(q31_t)0x1723b5f9, (q31_t)0x821be240, (q31_t)0x171d87f3, (q31_t)0x821abfa0, (q31_t)0x171759df, (q31_t)0x82199d4d,
	(q31_t)0x17112bbd, (q31_t)0x82187b47,
	(q31_t)0x170afd8d, (q31_t)0x82175990, (q31_t)0x1704cf4f, (q31_t)0x82163826, (q31_t)0x16fea102, (q31_t)0x82151709,
	(q31_t)0x16f872a7, (q31_t)0x8213f63a,
	(q31_t)0x16f2443e, (q31_t)0x8212d5b9, (q31_t)0x16ec15c7, (q31_t)0x8211b586, (q31_t)0x16e5e741, (q31_t)0x821095a0,
	(q31_t)0x16dfb8ae, (q31_t)0x820f7608,
	(q31_t)0x16d98a0c, (q31_t)0x820e56be, (q31_t)0x16d35b5c, (q31_t)0x820d37c1, (q31_t)0x16cd2c9f, (q31_t)0x820c1912,
	(q31_t)0x16c6fdd3, (q31_t)0x820afab1,
	(q31_t)0x16c0cef9, (q31_t)0x8209dc9e, (q31_t)0x16baa011, (q31_t)0x8208bed8, (q31_t)0x16b4711b, (q31_t)0x8207a160,
	(q31_t)0x16ae4217, (q31_t)0x82068435,
	(q31_t)0x16a81305, (q31_t)0x82056758, (q31_t)0x16a1e3e5, (q31_t)0x82044ac9, (q31_t)0x169bb4b7, (q31_t)0x82032e88,
	(q31_t)0x1695857b, (q31_t)0x82021294,
	(q31_t)0x168f5632, (q31_t)0x8200f6ef, (q31_t)0x168926da, (q31_t)0x81ffdb96, (q31_t)0x1682f774, (q31_t)0x81fec08c,
	(q31_t)0x167cc801, (q31_t)0x81fda5cf,
	(q31_t)0x1676987f, (q31_t)0x81fc8b60, (q31_t)0x167068f0, (q31_t)0x81fb713f, (q31_t)0x166a3953, (q31_t)0x81fa576c,
	(q31_t)0x166409a8, (q31_t)0x81f93de6,
	(q31_t)0x165dd9f0, (q31_t)0x81f824ae, (q31_t)0x1657aa29, (q31_t)0x81f70bc3, (q31_t)0x16517a55, (q31_t)0x81f5f327,
	(q31_t)0x164b4a73, (q31_t)0x81f4dad8,
	(q31_t)0x16451a83, (q31_t)0x81f3c2d7, (q31_t)0x163eea86, (q31_t)0x81f2ab24, (q31_t)0x1638ba7a, (q31_t)0x81f193be,
	(q31_t)0x16328a61, (q31_t)0x81f07ca6,
	(q31_t)0x162c5a3b, (q31_t)0x81ef65dc, (q31_t)0x16262a06, (q31_t)0x81ee4f60, (q31_t)0x161ff9c4, (q31_t)0x81ed3932,
	(q31_t)0x1619c975, (q31_t)0x81ec2351,
	(q31_t)0x16139918, (q31_t)0x81eb0dbe, (q31_t)0x160d68ad, (q31_t)0x81e9f879, (q31_t)0x16073834, (q31_t)0x81e8e381,
	(q31_t)0x160107ae, (q31_t)0x81e7ced8,
	(q31_t)0x15fad71b, (q31_t)0x81e6ba7c, (q31_t)0x15f4a679, (q31_t)0x81e5a66e, (q31_t)0x15ee75cb, (q31_t)0x81e492ad,
	(q31_t)0x15e8450e, (q31_t)0x81e37f3b,
	(q31_t)0x15e21445, (q31_t)0x81e26c16, (q31_t)0x15dbe36d, (q31_t)0x81e1593f, (q31_t)0x15d5b288, (q31_t)0x81e046b6,
	(q31_t)0x15cf8196, (q31_t)0x81df347b,
	(q31_t)0x15c95097, (q31_t)0x81de228d, (q31_t)0x15c31f89, (q31_t)0x81dd10ee, (q31_t)0x15bcee6f, (q31_t)0x81dbff9c,
	(q31_t)0x15b6bd47, (q31_t)0x81daee98,
	(q31_t)0x15b08c12, (q31_t)0x81d9dde1, (q31_t)0x15aa5acf, (q31_t)0x81d8cd79, (q31_t)0x15a4297f, (q31_t)0x81d7bd5e,
	(q31_t)0x159df821, (q31_t)0x81d6ad92,
	(q31_t)0x1597c6b7, (q31_t)0x81d59e13, (q31_t)0x1591953e, (q31_t)0x81d48ee1, (q31_t)0x158b63b9, (q31_t)0x81d37ffe,
	(q31_t)0x15853226, (q31_t)0x81d27169,
	(q31_t)0x157f0086, (q31_t)0x81d16321, (q31_t)0x1578ced9, (q31_t)0x81d05527, (q31_t)0x15729d1f, (q31_t)0x81cf477b,
	(q31_t)0x156c6b57, (q31_t)0x81ce3a1d,
	(q31_t)0x15663982, (q31_t)0x81cd2d0c, (q31_t)0x156007a0, (q31_t)0x81cc204a, (q31_t)0x1559d5b1, (q31_t)0x81cb13d5,
	(q31_t)0x1553a3b4, (q31_t)0x81ca07af,
	(q31_t)0x154d71aa, (q31_t)0x81c8fbd6, (q31_t)0x15473f94, (q31_t)0x81c7f04b, (q31_t)0x15410d70, (q31_t)0x81c6e50d,
	(q31_t)0x153adb3f, (q31_t)0x81c5da1e,
	(q31_t)0x1534a901, (q31_t)0x81c4cf7d, (q31_t)0x152e76b5, (q31_t)0x81c3c529, (q31_t)0x1528445d, (q31_t)0x81c2bb23,
	(q31_t)0x152211f8, (q31_t)0x81c1b16b,
	(q31_t)0x151bdf86, (q31_t)0x81c0a801, (q31_t)0x1515ad06, (q31_t)0x81bf9ee5, (q31_t)0x150f7a7a, (q31_t)0x81be9617,
	(q31_t)0x150947e1, (q31_t)0x81bd8d97,
	(q31_t)0x1503153a, (q31_t)0x81bc8564, (q31_t)0x14fce287, (q31_t)0x81bb7d7f, (q31_t)0x14f6afc7, (q31_t)0x81ba75e9,
	(q31_t)0x14f07cf9, (q31_t)0x81b96ea0,
	(q31_t)0x14ea4a1f, (q31_t)0x81b867a5, (q31_t)0x14e41738, (q31_t)0x81b760f8, (q31_t)0x14dde445, (q31_t)0x81b65a99,
	(q31_t)0x14d7b144, (q31_t)0x81b55488,
	(q31_t)0x14d17e36, (q31_t)0x81b44ec4, (q31_t)0x14cb4b1c, (q31_t)0x81b3494f, (q31_t)0x14c517f4, (q31_t)0x81b24427,
	(q31_t)0x14bee4c0, (q31_t)0x81b13f4e,
	(q31_t)0x14b8b17f, (q31_t)0x81b03ac2, (q31_t)0x14b27e32, (q31_t)0x81af3684, (q31_t)0x14ac4ad7, (q31_t)0x81ae3294,
	(q31_t)0x14a61770, (q31_t)0x81ad2ef2,
	(q31_t)0x149fe3fc, (q31_t)0x81ac2b9e, (q31_t)0x1499b07c, (q31_t)0x81ab2898, (q31_t)0x14937cee, (q31_t)0x81aa25e0,
	(q31_t)0x148d4954, (q31_t)0x81a92376,
	(q31_t)0x148715ae, (q31_t)0x81a82159, (q31_t)0x1480e1fa, (q31_t)0x81a71f8b, (q31_t)0x147aae3a, (q31_t)0x81a61e0b,
	(q31_t)0x14747a6d, (q31_t)0x81a51cd8,
	(q31_t)0x146e4694, (q31_t)0x81a41bf4, (q31_t)0x146812ae, (q31_t)0x81a31b5d, (q31_t)0x1461debc, (q31_t)0x81a21b14,
	(q31_t)0x145baabd, (q31_t)0x81a11b1a,
	(q31_t)0x145576b1, (q31_t)0x81a01b6d, (q31_t)0x144f4299, (q31_t)0x819f1c0e, (q31_t)0x14490e74, (q31_t)0x819e1cfd,
	(q31_t)0x1442da43, (q31_t)0x819d1e3a,
	(q31_t)0x143ca605, (q31_t)0x819c1fc5, (q31_t)0x143671bb, (q31_t)0x819b219e, (q31_t)0x14303d65, (q31_t)0x819a23c5,
	(q31_t)0x142a0902, (q31_t)0x8199263a,
	(q31_t)0x1423d492, (q31_t)0x819828fd, (q31_t)0x141da016, (q31_t)0x81972c0e, (q31_t)0x14176b8e, (q31_t)0x81962f6d,
	(q31_t)0x141136f9, (q31_t)0x8195331a,
	(q31_t)0x140b0258, (q31_t)0x81943715, (q31_t)0x1404cdaa, (q31_t)0x81933b5e, (q31_t)0x13fe98f1, (q31_t)0x81923ff4,
	(q31_t)0x13f8642a, (q31_t)0x819144d9,
	(q31_t)0x13f22f58, (q31_t)0x81904a0c, (q31_t)0x13ebfa79, (q31_t)0x818f4f8d, (q31_t)0x13e5c58e, (q31_t)0x818e555c,
	(q31_t)0x13df9097, (q31_t)0x818d5b78,
	(q31_t)0x13d95b93, (q31_t)0x818c61e3, (q31_t)0x13d32683, (q31_t)0x818b689c, (q31_t)0x13ccf167, (q31_t)0x818a6fa3,
	(q31_t)0x13c6bc3f, (q31_t)0x818976f8,
	(q31_t)0x13c0870a, (q31_t)0x81887e9a, (q31_t)0x13ba51ca, (q31_t)0x8187868b, (q31_t)0x13b41c7d, (q31_t)0x81868eca,
	(q31_t)0x13ade724, (q31_t)0x81859757,
	(q31_t)0x13a7b1bf, (q31_t)0x8184a032, (q31_t)0x13a17c4d, (q31_t)0x8183a95b, (q31_t)0x139b46d0, (q31_t)0x8182b2d1,
	(q31_t)0x13951146, (q31_t)0x8181bc96,
	(q31_t)0x138edbb1, (q31_t)0x8180c6a9, (q31_t)0x1388a60f, (q31_t)0x817fd10a, (q31_t)0x13827062, (q31_t)0x817edbb9,
	(q31_t)0x137c3aa8, (q31_t)0x817de6b6,
	(q31_t)0x137604e2, (q31_t)0x817cf201, (q31_t)0x136fcf10, (q31_t)0x817bfd9b, (q31_t)0x13699933, (q31_t)0x817b0982,
	(q31_t)0x13636349, (q31_t)0x817a15b7,
	(q31_t)0x135d2d53, (q31_t)0x8179223a, (q31_t)0x1356f752, (q31_t)0x81782f0b, (q31_t)0x1350c144, (q31_t)0x81773c2b,
	(q31_t)0x134a8b2b, (q31_t)0x81764998,
	(q31_t)0x13445505, (q31_t)0x81755754, (q31_t)0x133e1ed4, (q31_t)0x8174655d, (q31_t)0x1337e897, (q31_t)0x817373b5,
	(q31_t)0x1331b24e, (q31_t)0x8172825a,
	(q31_t)0x132b7bf9, (q31_t)0x8171914e, (q31_t)0x13254599, (q31_t)0x8170a090, (q31_t)0x131f0f2c, (q31_t)0x816fb020,
	(q31_t)0x1318d8b4, (q31_t)0x816ebffe,
	(q31_t)0x1312a230, (q31_t)0x816dd02a, (q31_t)0x130c6ba0, (q31_t)0x816ce0a4, (q31_t)0x13063505, (q31_t)0x816bf16c,
	(q31_t)0x12fffe5d, (q31_t)0x816b0282,
	(q31_t)0x12f9c7aa, (q31_t)0x816a13e6, (q31_t)0x12f390ec, (q31_t)0x81692599, (q31_t)0x12ed5a21, (q31_t)0x81683799,
	(q31_t)0x12e7234b, (q31_t)0x816749e8,
	(q31_t)0x12e0ec6a, (q31_t)0x81665c84, (q31_t)0x12dab57c, (q31_t)0x81656f6f, (q31_t)0x12d47e83, (q31_t)0x816482a8,
	(q31_t)0x12ce477f, (q31_t)0x8163962f,
	(q31_t)0x12c8106f, (q31_t)0x8162aa04, (q31_t)0x12c1d953, (q31_t)0x8161be27, (q31_t)0x12bba22b, (q31_t)0x8160d298,
	(q31_t)0x12b56af9, (q31_t)0x815fe758,
	(q31_t)0x12af33ba, (q31_t)0x815efc65, (q31_t)0x12a8fc70, (q31_t)0x815e11c1, (q31_t)0x12a2c51b, (q31_t)0x815d276a,
	(q31_t)0x129c8dba, (q31_t)0x815c3d62,
	(q31_t)0x1296564d, (q31_t)0x815b53a8, (q31_t)0x12901ed5, (q31_t)0x815a6a3c, (q31_t)0x1289e752, (q31_t)0x8159811e,
	(q31_t)0x1283afc3, (q31_t)0x8158984e,
	(q31_t)0x127d7829, (q31_t)0x8157afcd, (q31_t)0x12774083, (q31_t)0x8156c799, (q31_t)0x127108d2, (q31_t)0x8155dfb4,
	(q31_t)0x126ad116, (q31_t)0x8154f81d,
	(q31_t)0x1264994e, (q31_t)0x815410d4, (q31_t)0x125e617b, (q31_t)0x815329d9, (q31_t)0x1258299c, (q31_t)0x8152432c,
	(q31_t)0x1251f1b3, (q31_t)0x81515ccd,
	(q31_t)0x124bb9be, (q31_t)0x815076bd, (q31_t)0x124581bd, (q31_t)0x814f90fb, (q31_t)0x123f49b2, (q31_t)0x814eab86,
	(q31_t)0x1239119b, (q31_t)0x814dc660,
	(q31_t)0x1232d979, (q31_t)0x814ce188, (q31_t)0x122ca14b, (q31_t)0x814bfcff, (q31_t)0x12266913, (q31_t)0x814b18c3,
	(q31_t)0x122030cf, (q31_t)0x814a34d6,
	(q31_t)0x1219f880, (q31_t)0x81495136, (q31_t)0x1213c026, (q31_t)0x81486de5, (q31_t)0x120d87c1, (q31_t)0x81478ae2,
	(q31_t)0x12074f50, (q31_t)0x8146a82e,
	(q31_t)0x120116d5, (q31_t)0x8145c5c7, (q31_t)0x11fade4e, (q31_t)0x8144e3ae, (q31_t)0x11f4a5bd, (q31_t)0x814401e4,
	(q31_t)0x11ee6d20, (q31_t)0x81432068,
	(q31_t)0x11e83478, (q31_t)0x81423f3a, (q31_t)0x11e1fbc5, (q31_t)0x81415e5a, (q31_t)0x11dbc307, (q31_t)0x81407dc9,
	(q31_t)0x11d58a3e, (q31_t)0x813f9d86,
	(q31_t)0x11cf516a, (q31_t)0x813ebd90, (q31_t)0x11c9188b, (q31_t)0x813ddde9, (q31_t)0x11c2dfa2, (q31_t)0x813cfe91,
	(q31_t)0x11bca6ad, (q31_t)0x813c1f86,
	(q31_t)0x11b66dad, (q31_t)0x813b40ca, (q31_t)0x11b034a2, (q31_t)0x813a625b, (q31_t)0x11a9fb8d, (q31_t)0x8139843b,
	(q31_t)0x11a3c26c, (q31_t)0x8138a66a,
	(q31_t)0x119d8941, (q31_t)0x8137c8e6, (q31_t)0x1197500a, (q31_t)0x8136ebb1, (q31_t)0x119116c9, (q31_t)0x81360ec9,
	(q31_t)0x118add7d, (q31_t)0x81353230,
	(q31_t)0x1184a427, (q31_t)0x813455e6, (q31_t)0x117e6ac5, (q31_t)0x813379e9, (q31_t)0x11783159, (q31_t)0x81329e3b,
	(q31_t)0x1171f7e2, (q31_t)0x8131c2db,
	(q31_t)0x116bbe60, (q31_t)0x8130e7c9, (q31_t)0x116584d3, (q31_t)0x81300d05, (q31_t)0x115f4b3c, (q31_t)0x812f3290,
	(q31_t)0x1159119a, (q31_t)0x812e5868,
	(q31_t)0x1152d7ed, (q31_t)0x812d7e8f, (q31_t)0x114c9e35, (q31_t)0x812ca505, (q31_t)0x11466473, (q31_t)0x812bcbc8,
	(q31_t)0x11402aa6, (q31_t)0x812af2da,
	(q31_t)0x1139f0cf, (q31_t)0x812a1a3a, (q31_t)0x1133b6ed, (q31_t)0x812941e8, (q31_t)0x112d7d00, (q31_t)0x812869e4,
	(q31_t)0x11274309, (q31_t)0x8127922f,
	(q31_t)0x11210907, (q31_t)0x8126bac8, (q31_t)0x111acefb, (q31_t)0x8125e3af, (q31_t)0x111494e4, (q31_t)0x81250ce4,
	(q31_t)0x110e5ac2, (q31_t)0x81243668,
	(q31_t)0x11082096, (q31_t)0x8123603a, (q31_t)0x1101e65f, (q31_t)0x81228a5a, (q31_t)0x10fbac1e, (q31_t)0x8121b4c8,
	(q31_t)0x10f571d3, (q31_t)0x8120df85,
	(q31_t)0x10ef377d, (q31_t)0x81200a90, (q31_t)0x10e8fd1c, (q31_t)0x811f35e9, (q31_t)0x10e2c2b2, (q31_t)0x811e6191,
	(q31_t)0x10dc883c, (q31_t)0x811d8d86,
	(q31_t)0x10d64dbd, (q31_t)0x811cb9ca, (q31_t)0x10d01333, (q31_t)0x811be65d, (q31_t)0x10c9d89e, (q31_t)0x811b133d,
	(q31_t)0x10c39dff, (q31_t)0x811a406c,
	(q31_t)0x10bd6356, (q31_t)0x81196de9, (q31_t)0x10b728a3, (q31_t)0x81189bb4, (q31_t)0x10b0ede5, (q31_t)0x8117c9ce,
	(q31_t)0x10aab31d, (q31_t)0x8116f836,
	(q31_t)0x10a4784b, (q31_t)0x811626ec, (q31_t)0x109e3d6e, (q31_t)0x811555f1, (q31_t)0x10980287, (q31_t)0x81148544,
	(q31_t)0x1091c796, (q31_t)0x8113b4e5,
	(q31_t)0x108b8c9b, (q31_t)0x8112e4d4, (q31_t)0x10855195, (q31_t)0x81121512, (q31_t)0x107f1686, (q31_t)0x8111459e,
	(q31_t)0x1078db6c, (q31_t)0x81107678,
	(q31_t)0x1072a048, (q31_t)0x810fa7a0, (q31_t)0x106c651a, (q31_t)0x810ed917, (q31_t)0x106629e1, (q31_t)0x810e0adc,
	(q31_t)0x105fee9f, (q31_t)0x810d3cf0,
	(q31_t)0x1059b352, (q31_t)0x810c6f52, (q31_t)0x105377fc, (q31_t)0x810ba202, (q31_t)0x104d3c9b, (q31_t)0x810ad500,
	(q31_t)0x10470130, (q31_t)0x810a084d,
	(q31_t)0x1040c5bb, (q31_t)0x81093be8, (q31_t)0x103a8a3d, (q31_t)0x81086fd1, (q31_t)0x10344eb4, (q31_t)0x8107a409,
	(q31_t)0x102e1321, (q31_t)0x8106d88f,
	(q31_t)0x1027d784, (q31_t)0x81060d63, (q31_t)0x10219bdd, (q31_t)0x81054286, (q31_t)0x101b602d, (q31_t)0x810477f7,
	(q31_t)0x10152472, (q31_t)0x8103adb6,
	(q31_t)0x100ee8ad, (q31_t)0x8102e3c4, (q31_t)0x1008acdf, (q31_t)0x81021a20, (q31_t)0x10027107, (q31_t)0x810150ca,
	(q31_t)0xffc3524, (q31_t)0x810087c3,
	(q31_t)0xff5f938, (q31_t)0x80ffbf0a, (q31_t)0xfefbd42, (q31_t)0x80fef69f, (q31_t)0xfe98143, (q31_t)0x80fe2e83,
	(q31_t)0xfe34539, (q31_t)0x80fd66b5,
	(q31_t)0xfdd0926, (q31_t)0x80fc9f35, (q31_t)0xfd6cd08, (q31_t)0x80fbd804, (q31_t)0xfd090e1, (q31_t)0x80fb1121,
	(q31_t)0xfca54b1, (q31_t)0x80fa4a8c,
	(q31_t)0xfc41876, (q31_t)0x80f98446, (q31_t)0xfbddc32, (q31_t)0x80f8be4e, (q31_t)0xfb79fe4, (q31_t)0x80f7f8a4,
	(q31_t)0xfb1638d, (q31_t)0x80f73349,
	(q31_t)0xfab272b, (q31_t)0x80f66e3c, (q31_t)0xfa4eac0, (q31_t)0x80f5a97e, (q31_t)0xf9eae4c, (q31_t)0x80f4e50e,
	(q31_t)0xf9871ce, (q31_t)0x80f420ec,
	(q31_t)0xf923546, (q31_t)0x80f35d19, (q31_t)0xf8bf8b4, (q31_t)0x80f29994, (q31_t)0xf85bc19, (q31_t)0x80f1d65d,
	(q31_t)0xf7f7f75, (q31_t)0x80f11375,
	(q31_t)0xf7942c7, (q31_t)0x80f050db, (q31_t)0xf73060f, (q31_t)0x80ef8e90, (q31_t)0xf6cc94e, (q31_t)0x80eecc93,
	(q31_t)0xf668c83, (q31_t)0x80ee0ae4,
	(q31_t)0xf604faf, (q31_t)0x80ed4984, (q31_t)0xf5a12d1, (q31_t)0x80ec8872, (q31_t)0xf53d5ea, (q31_t)0x80ebc7ae,
	(q31_t)0xf4d98f9, (q31_t)0x80eb0739,
	(q31_t)0xf475bff, (q31_t)0x80ea4712, (q31_t)0xf411efb, (q31_t)0x80e9873a, (q31_t)0xf3ae1ee, (q31_t)0x80e8c7b0,
	(q31_t)0xf34a4d8, (q31_t)0x80e80874,
	(q31_t)0xf2e67b8, (q31_t)0x80e74987, (q31_t)0xf282a8f, (q31_t)0x80e68ae8, (q31_t)0xf21ed5d, (q31_t)0x80e5cc98,
	(q31_t)0xf1bb021, (q31_t)0x80e50e96,
	(q31_t)0xf1572dc, (q31_t)0x80e450e2, (q31_t)0xf0f358e, (q31_t)0x80e3937d, (q31_t)0xf08f836, (q31_t)0x80e2d666,
	(q31_t)0xf02bad5, (q31_t)0x80e2199e,
	(q31_t)0xefc7d6b, (q31_t)0x80e15d24, (q31_t)0xef63ff7, (q31_t)0x80e0a0f8, (q31_t)0xef0027b, (q31_t)0x80dfe51b,
	(q31_t)0xee9c4f5, (q31_t)0x80df298c,
	(q31_t)0xee38766, (q31_t)0x80de6e4c, (q31_t)0xedd49ce, (q31_t)0x80ddb35a, (q31_t)0xed70c2c, (q31_t)0x80dcf8b7,
	(q31_t)0xed0ce82, (q31_t)0x80dc3e62,
	(q31_t)0xeca90ce, (q31_t)0x80db845b, (q31_t)0xec45311, (q31_t)0x80dacaa3, (q31_t)0xebe154b, (q31_t)0x80da1139,
	(q31_t)0xeb7d77c, (q31_t)0x80d9581e,
	(q31_t)0xeb199a4, (q31_t)0x80d89f51, (q31_t)0xeab5bc3, (q31_t)0x80d7e6d3, (q31_t)0xea51dd8, (q31_t)0x80d72ea3,
	(q31_t)0xe9edfe5, (q31_t)0x80d676c1,
	(q31_t)0xe98a1e9, (q31_t)0x80d5bf2e, (q31_t)0xe9263e3, (q31_t)0x80d507e9, (q31_t)0xe8c25d5, (q31_t)0x80d450f3,
	(q31_t)0xe85e7be, (q31_t)0x80d39a4b,
	(q31_t)0xe7fa99e, (q31_t)0x80d2e3f2, (q31_t)0xe796b74, (q31_t)0x80d22de7, (q31_t)0xe732d42, (q31_t)0x80d1782a,
	(q31_t)0xe6cef07, (q31_t)0x80d0c2bc,
	(q31_t)0xe66b0c3, (q31_t)0x80d00d9d, (q31_t)0xe607277, (q31_t)0x80cf58cc, (q31_t)0xe5a3421, (q31_t)0x80cea449,
	(q31_t)0xe53f5c2, (q31_t)0x80cdf015,
	(q31_t)0xe4db75b, (q31_t)0x80cd3c2f, (q31_t)0xe4778eb, (q31_t)0x80cc8898, (q31_t)0xe413a72, (q31_t)0x80cbd54f,
	(q31_t)0xe3afbf0, (q31_t)0x80cb2255,
	(q31_t)0xe34bd66, (q31_t)0x80ca6fa9, (q31_t)0xe2e7ed2, (q31_t)0x80c9bd4c, (q31_t)0xe284036, (q31_t)0x80c90b3d,
	(q31_t)0xe220191, (q31_t)0x80c8597c,
	(q31_t)0xe1bc2e4, (q31_t)0x80c7a80a, (q31_t)0xe15842e, (q31_t)0x80c6f6e7, (q31_t)0xe0f456f, (q31_t)0x80c64612,
	(q31_t)0xe0906a7, (q31_t)0x80c5958b,
	(q31_t)0xe02c7d7, (q31_t)0x80c4e553, (q31_t)0xdfc88fe, (q31_t)0x80c4356a, (q31_t)0xdf64a1c, (q31_t)0x80c385cf,
	(q31_t)0xdf00b32, (q31_t)0x80c2d682,
	(q31_t)0xde9cc40, (q31_t)0x80c22784, (q31_t)0xde38d44, (q31_t)0x80c178d4, (q31_t)0xddd4e40, (q31_t)0x80c0ca73,
	(q31_t)0xdd70f34, (q31_t)0x80c01c60,
	(q31_t)0xdd0d01f, (q31_t)0x80bf6e9c, (q31_t)0xdca9102, (q31_t)0x80bec127, (q31_t)0xdc451dc, (q31_t)0x80be13ff,
	(q31_t)0xdbe12ad, (q31_t)0x80bd6727,
	(q31_t)0xdb7d376, (q31_t)0x80bcba9d, (q31_t)0xdb19437, (q31_t)0x80bc0e61, (q31_t)0xdab54ef, (q31_t)0x80bb6274,
	(q31_t)0xda5159f, (q31_t)0x80bab6d5,
	(q31_t)0xd9ed646, (q31_t)0x80ba0b85, (q31_t)0xd9896e5, (q31_t)0x80b96083, (q31_t)0xd92577b, (q31_t)0x80b8b5d0,
	(q31_t)0xd8c1809, (q31_t)0x80b80b6c,
	(q31_t)0xd85d88f, (q31_t)0x80b76156, (q31_t)0xd7f990c, (q31_t)0x80b6b78e, (q31_t)0xd795982, (q31_t)0x80b60e15,
	(q31_t)0xd7319ee, (q31_t)0x80b564ea,
	(q31_t)0xd6cda53, (q31_t)0x80b4bc0e, (q31_t)0xd669aaf, (q31_t)0x80b41381, (q31_t)0xd605b03, (q31_t)0x80b36b42,
	(q31_t)0xd5a1b4f, (q31_t)0x80b2c351,
	(q31_t)0xd53db92, (q31_t)0x80b21baf, (q31_t)0xd4d9bcd, (q31_t)0x80b1745c, (q31_t)0xd475c00, (q31_t)0x80b0cd57,
	(q31_t)0xd411c2b, (q31_t)0x80b026a1,
	(q31_t)0xd3adc4e, (q31_t)0x80af8039, (q31_t)0xd349c68, (q31_t)0x80aeda20, (q31_t)0xd2e5c7b, (q31_t)0x80ae3455,
	(q31_t)0xd281c85, (q31_t)0x80ad8ed9,
	(q31_t)0xd21dc87, (q31_t)0x80ace9ab, (q31_t)0xd1b9c81, (q31_t)0x80ac44cc, (q31_t)0xd155c73, (q31_t)0x80aba03b,
	(q31_t)0xd0f1c5d, (q31_t)0x80aafbf9,
	(q31_t)0xd08dc3f, (q31_t)0x80aa5806, (q31_t)0xd029c18, (q31_t)0x80a9b461, (q31_t)0xcfc5bea, (q31_t)0x80a9110b,
	(q31_t)0xcf61bb4, (q31_t)0x80a86e03,
	(q31_t)0xcefdb76, (q31_t)0x80a7cb49, (q31_t)0xce99b2f, (q31_t)0x80a728df, (q31_t)0xce35ae1, (q31_t)0x80a686c2,
	(q31_t)0xcdd1a8b, (q31_t)0x80a5e4f5,
	(q31_t)0xcd6da2d, (q31_t)0x80a54376, (q31_t)0xcd099c7, (q31_t)0x80a4a245, (q31_t)0xcca5959, (q31_t)0x80a40163,
	(q31_t)0xcc418e3, (q31_t)0x80a360d0,
	(q31_t)0xcbdd865, (q31_t)0x80a2c08b, (q31_t)0xcb797e0, (q31_t)0x80a22095, (q31_t)0xcb15752, (q31_t)0x80a180ed,
	(q31_t)0xcab16bd, (q31_t)0x80a0e194,
	(q31_t)0xca4d620, (q31_t)0x80a04289, (q31_t)0xc9e957b, (q31_t)0x809fa3cd, (q31_t)0xc9854cf, (q31_t)0x809f0560,
	(q31_t)0xc92141a, (q31_t)0x809e6741,
	(q31_t)0xc8bd35e, (q31_t)0x809dc971, (q31_t)0xc85929a, (q31_t)0x809d2bef, (q31_t)0xc7f51cf, (q31_t)0x809c8ebc,
	(q31_t)0xc7910fb, (q31_t)0x809bf1d7,
	(q31_t)0xc72d020, (q31_t)0x809b5541, (q31_t)0xc6c8f3e, (q31_t)0x809ab8fa, (q31_t)0xc664e53, (q31_t)0x809a1d01,
	(q31_t)0xc600d61, (q31_t)0x80998157,
	(q31_t)0xc59cc68, (q31_t)0x8098e5fb, (q31_t)0xc538b66, (q31_t)0x80984aee, (q31_t)0xc4d4a5d, (q31_t)0x8097b030,
	(q31_t)0xc47094d, (q31_t)0x809715c0,
	(q31_t)0xc40c835, (q31_t)0x80967b9f, (q31_t)0xc3a8715, (q31_t)0x8095e1cc, (q31_t)0xc3445ee, (q31_t)0x80954848,
	(q31_t)0xc2e04c0, (q31_t)0x8094af13,
	(q31_t)0xc27c389, (q31_t)0x8094162c, (q31_t)0xc21824c, (q31_t)0x80937d93, (q31_t)0xc1b4107, (q31_t)0x8092e54a,
	(q31_t)0xc14ffba, (q31_t)0x80924d4f,
	(q31_t)0xc0ebe66, (q31_t)0x8091b5a2, (q31_t)0xc087d0a, (q31_t)0x80911e44, (q31_t)0xc023ba7, (q31_t)0x80908735,
	(q31_t)0xbfbfa3d, (q31_t)0x808ff074,
	(q31_t)0xbf5b8cb, (q31_t)0x808f5a02, (q31_t)0xbef7752, (q31_t)0x808ec3df, (q31_t)0xbe935d2, (q31_t)0x808e2e0a,
	(q31_t)0xbe2f44a, (q31_t)0x808d9884,
	(q31_t)0xbdcb2bb, (q31_t)0x808d034c, (q31_t)0xbd67124, (q31_t)0x808c6e63, (q31_t)0xbd02f87, (q31_t)0x808bd9c9,
	(q31_t)0xbc9ede2, (q31_t)0x808b457d,
	(q31_t)0xbc3ac35, (q31_t)0x808ab180, (q31_t)0xbbd6a82, (q31_t)0x808a1dd2, (q31_t)0xbb728c7, (q31_t)0x80898a72,
	(q31_t)0xbb0e705, (q31_t)0x8088f761,
	(q31_t)0xbaaa53b, (q31_t)0x8088649e, (q31_t)0xba4636b, (q31_t)0x8087d22a, (q31_t)0xb9e2193, (q31_t)0x80874005,
	(q31_t)0xb97dfb5, (q31_t)0x8086ae2e,
	(q31_t)0xb919dcf, (q31_t)0x80861ca6, (q31_t)0xb8b5be1, (q31_t)0x80858b6c, (q31_t)0xb8519ed, (q31_t)0x8084fa82,
	(q31_t)0xb7ed7f2, (q31_t)0x808469e5,
	(q31_t)0xb7895f0, (q31_t)0x8083d998, (q31_t)0xb7253e6, (q31_t)0x80834999, (q31_t)0xb6c11d5, (q31_t)0x8082b9e9,
	(q31_t)0xb65cfbe, (q31_t)0x80822a87,
	(q31_t)0xb5f8d9f, (q31_t)0x80819b74, (q31_t)0xb594b7a, (q31_t)0x80810cb0, (q31_t)0xb53094d, (q31_t)0x80807e3a,
	(q31_t)0xb4cc719, (q31_t)0x807ff013,
	(q31_t)0xb4684df, (q31_t)0x807f623b, (q31_t)0xb40429d, (q31_t)0x807ed4b1, (q31_t)0xb3a0055, (q31_t)0x807e4776,
	(q31_t)0xb33be05, (q31_t)0x807dba89,
	(q31_t)0xb2d7baf, (q31_t)0x807d2dec, (q31_t)0xb273952, (q31_t)0x807ca19c, (q31_t)0xb20f6ee, (q31_t)0x807c159c,
	(q31_t)0xb1ab483, (q31_t)0x807b89ea,
	(q31_t)0xb147211, (q31_t)0x807afe87, (q31_t)0xb0e2f98, (q31_t)0x807a7373, (q31_t)0xb07ed19, (q31_t)0x8079e8ad,
	(q31_t)0xb01aa92, (q31_t)0x80795e36,
	(q31_t)0xafb6805, (q31_t)0x8078d40d, (q31_t)0xaf52571, (q31_t)0x80784a33, (q31_t)0xaeee2d7, (q31_t)0x8077c0a8,
	(q31_t)0xae8a036, (q31_t)0x8077376c,
	(q31_t)0xae25d8d, (q31_t)0x8076ae7e, (q31_t)0xadc1adf, (q31_t)0x807625df, (q31_t)0xad5d829, (q31_t)0x80759d8e,
	(q31_t)0xacf956d, (q31_t)0x8075158c,
	(q31_t)0xac952aa, (q31_t)0x80748dd9, (q31_t)0xac30fe1, (q31_t)0x80740675, (q31_t)0xabccd11, (q31_t)0x80737f5f,
	(q31_t)0xab68a3a, (q31_t)0x8072f898,
	(q31_t)0xab0475c, (q31_t)0x8072721f, (q31_t)0xaaa0478, (q31_t)0x8071ebf6, (q31_t)0xaa3c18e, (q31_t)0x8071661a,
	(q31_t)0xa9d7e9d, (q31_t)0x8070e08e,
	(q31_t)0xa973ba5, (q31_t)0x80705b50, (q31_t)0xa90f8a7, (q31_t)0x806fd661, (q31_t)0xa8ab5a2, (q31_t)0x806f51c1,
	(q31_t)0xa847297, (q31_t)0x806ecd6f,
	(q31_t)0xa7e2f85, (q31_t)0x806e496c, (q31_t)0xa77ec6d, (q31_t)0x806dc5b8, (q31_t)0xa71a94f, (q31_t)0x806d4253,
	(q31_t)0xa6b662a, (q31_t)0x806cbf3c,
	(q31_t)0xa6522fe, (q31_t)0x806c3c74, (q31_t)0xa5edfcc, (q31_t)0x806bb9fa, (q31_t)0xa589c94, (q31_t)0x806b37cf,
	(q31_t)0xa525955, (q31_t)0x806ab5f3,
	(q31_t)0xa4c1610, (q31_t)0x806a3466, (q31_t)0xa45d2c5, (q31_t)0x8069b327, (q31_t)0xa3f8f73, (q31_t)0x80693237,
	(q31_t)0xa394c1b, (q31_t)0x8068b196,
	(q31_t)0xa3308bd, (q31_t)0x80683143, (q31_t)0xa2cc558, (q31_t)0x8067b13f, (q31_t)0xa2681ed, (q31_t)0x8067318a,
	(q31_t)0xa203e7c, (q31_t)0x8066b224,
	(q31_t)0xa19fb04, (q31_t)0x8066330c, (q31_t)0xa13b787, (q31_t)0x8065b443, (q31_t)0xa0d7403, (q31_t)0x806535c9,
	(q31_t)0xa073079, (q31_t)0x8064b79d,
	(q31_t)0xa00ece8, (q31_t)0x806439c0, (q31_t)0x9faa952, (q31_t)0x8063bc32, (q31_t)0x9f465b5, (q31_t)0x80633ef3,
	(q31_t)0x9ee2213, (q31_t)0x8062c202,
	(q31_t)0x9e7de6a, (q31_t)0x80624560, (q31_t)0x9e19abb, (q31_t)0x8061c90c, (q31_t)0x9db5706, (q31_t)0x80614d08,
	(q31_t)0x9d5134b, (q31_t)0x8060d152,
	(q31_t)0x9cecf89, (q31_t)0x806055eb, (q31_t)0x9c88bc2, (q31_t)0x805fdad2, (q31_t)0x9c247f5, (q31_t)0x805f6009,
	(q31_t)0x9bc0421, (q31_t)0x805ee58e,
	(q31_t)0x9b5c048, (q31_t)0x805e6b62, (q31_t)0x9af7c69, (q31_t)0x805df184, (q31_t)0x9a93884, (q31_t)0x805d77f5,
	(q31_t)0x9a2f498, (q31_t)0x805cfeb5,
	(q31_t)0x99cb0a7, (q31_t)0x805c85c4, (q31_t)0x9966cb0, (q31_t)0x805c0d21, (q31_t)0x99028b3, (q31_t)0x805b94ce,
	(q31_t)0x989e4b0, (q31_t)0x805b1cc8,
	(q31_t)0x983a0a7, (q31_t)0x805aa512, (q31_t)0x97d5c99, (q31_t)0x805a2daa, (q31_t)0x9771884, (q31_t)0x8059b692,
	(q31_t)0x970d46a, (q31_t)0x80593fc7,
	(q31_t)0x96a9049, (q31_t)0x8058c94c, (q31_t)0x9644c23, (q31_t)0x8058531f, (q31_t)0x95e07f8, (q31_t)0x8057dd41,
	(q31_t)0x957c3c6, (q31_t)0x805767b2,
	(q31_t)0x9517f8f, (q31_t)0x8056f272, (q31_t)0x94b3b52, (q31_t)0x80567d80, (q31_t)0x944f70f, (q31_t)0x805608dd,
	(q31_t)0x93eb2c6, (q31_t)0x80559489,
	(q31_t)0x9386e78, (q31_t)0x80552084, (q31_t)0x9322a24, (q31_t)0x8054accd, (q31_t)0x92be5ca, (q31_t)0x80543965,
	(q31_t)0x925a16b, (q31_t)0x8053c64c,
	(q31_t)0x91f5d06, (q31_t)0x80535381, (q31_t)0x919189c, (q31_t)0x8052e106, (q31_t)0x912d42c, (q31_t)0x80526ed9,
	(q31_t)0x90c8fb6, (q31_t)0x8051fcfb,
	(q31_t)0x9064b3a, (q31_t)0x80518b6b, (q31_t)0x90006ba, (q31_t)0x80511a2b, (q31_t)0x8f9c233, (q31_t)0x8050a939,
	(q31_t)0x8f37da7, (q31_t)0x80503896,
	(q31_t)0x8ed3916, (q31_t)0x804fc841, (q31_t)0x8e6f47f, (q31_t)0x804f583c, (q31_t)0x8e0afe2, (q31_t)0x804ee885,
	(q31_t)0x8da6b40, (q31_t)0x804e791d,
	(q31_t)0x8d42699, (q31_t)0x804e0a04, (q31_t)0x8cde1ec, (q31_t)0x804d9b39, (q31_t)0x8c79d3a, (q31_t)0x804d2cbd,
	(q31_t)0x8c15882, (q31_t)0x804cbe90,
	(q31_t)0x8bb13c5, (q31_t)0x804c50b2, (q31_t)0x8b4cf02, (q31_t)0x804be323, (q31_t)0x8ae8a3a, (q31_t)0x804b75e2,
	(q31_t)0x8a8456d, (q31_t)0x804b08f0,
	(q31_t)0x8a2009a, (q31_t)0x804a9c4d, (q31_t)0x89bbbc3, (q31_t)0x804a2ff9, (q31_t)0x89576e5, (q31_t)0x8049c3f3,
	(q31_t)0x88f3203, (q31_t)0x8049583d,
	(q31_t)0x888ed1b, (q31_t)0x8048ecd5, (q31_t)0x882a82e, (q31_t)0x804881bb, (q31_t)0x87c633c, (q31_t)0x804816f1,
	(q31_t)0x8761e44, (q31_t)0x8047ac75,
	(q31_t)0x86fd947, (q31_t)0x80474248, (q31_t)0x8699445, (q31_t)0x8046d86a, (q31_t)0x8634f3e, (q31_t)0x80466edb,
	(q31_t)0x85d0a32, (q31_t)0x8046059b,
	(q31_t)0x856c520, (q31_t)0x80459ca9, (q31_t)0x850800a, (q31_t)0x80453406, (q31_t)0x84a3aee, (q31_t)0x8044cbb2,
	(q31_t)0x843f5cd, (q31_t)0x804463ad,
	(q31_t)0x83db0a7, (q31_t)0x8043fbf6, (q31_t)0x8376b7c, (q31_t)0x8043948e, (q31_t)0x831264c, (q31_t)0x80432d75,
	(q31_t)0x82ae117, (q31_t)0x8042c6ab,
	(q31_t)0x8249bdd, (q31_t)0x80426030, (q31_t)0x81e569d, (q31_t)0x8041fa03, (q31_t)0x8181159, (q31_t)0x80419425,
	(q31_t)0x811cc10, (q31_t)0x80412e96,
	(q31_t)0x80b86c2, (q31_t)0x8040c956, (q31_t)0x805416e, (q31_t)0x80406465, (q31_t)0x7fefc16, (q31_t)0x803fffc2,
	(q31_t)0x7f8b6b9, (q31_t)0x803f9b6f,
	(q31_t)0x7f27157, (q31_t)0x803f376a, (q31_t)0x7ec2bf0, (q31_t)0x803ed3b3, (q31_t)0x7e5e685, (q31_t)0x803e704c,
	(q31_t)0x7dfa114, (q31_t)0x803e0d34,
	(q31_t)0x7d95b9e, (q31_t)0x803daa6a, (q31_t)0x7d31624, (q31_t)0x803d47ef, (q31_t)0x7ccd0a5, (q31_t)0x803ce5c3,
	(q31_t)0x7c68b21, (q31_t)0x803c83e5,
	(q31_t)0x7c04598, (q31_t)0x803c2257, (q31_t)0x7ba000b, (q31_t)0x803bc117, (q31_t)0x7b3ba78, (q31_t)0x803b6026,
	(q31_t)0x7ad74e1, (q31_t)0x803aff84,
	(q31_t)0x7a72f45, (q31_t)0x803a9f31, (q31_t)0x7a0e9a5, (q31_t)0x803a3f2d, (q31_t)0x79aa400, (q31_t)0x8039df77,
	(q31_t)0x7945e56, (q31_t)0x80398010,
	(q31_t)0x78e18a7, (q31_t)0x803920f8, (q31_t)0x787d2f4, (q31_t)0x8038c22f, (q31_t)0x7818d3c, (q31_t)0x803863b5,
	(q31_t)0x77b4780, (q31_t)0x80380589,
	(q31_t)0x77501be, (q31_t)0x8037a7ac, (q31_t)0x76ebbf9, (q31_t)0x80374a1f, (q31_t)0x768762e, (q31_t)0x8036ece0,
	(q31_t)0x762305f, (q31_t)0x80368fef,
	(q31_t)0x75bea8c, (q31_t)0x8036334e, (q31_t)0x755a4b4, (q31_t)0x8035d6fb, (q31_t)0x74f5ed7, (q31_t)0x80357af8,
	(q31_t)0x74918f6, (q31_t)0x80351f43,
	(q31_t)0x742d311, (q31_t)0x8034c3dd, (q31_t)0x73c8d27, (q31_t)0x803468c5, (q31_t)0x7364738, (q31_t)0x80340dfd,
	(q31_t)0x7300145, (q31_t)0x8033b383,
	(q31_t)0x729bb4e, (q31_t)0x80335959, (q31_t)0x7237552, (q31_t)0x8032ff7d, (q31_t)0x71d2f52, (q31_t)0x8032a5ef,
	(q31_t)0x716e94e, (q31_t)0x80324cb1,
	(q31_t)0x710a345, (q31_t)0x8031f3c2, (q31_t)0x70a5d37, (q31_t)0x80319b21, (q31_t)0x7041726, (q31_t)0x803142cf,
	(q31_t)0x6fdd110, (q31_t)0x8030eacd,
	(q31_t)0x6f78af6, (q31_t)0x80309318, (q31_t)0x6f144d7, (q31_t)0x80303bb3, (q31_t)0x6eafeb4, (q31_t)0x802fe49d,
	(q31_t)0x6e4b88d, (q31_t)0x802f8dd5,
	(q31_t)0x6de7262, (q31_t)0x802f375d, (q31_t)0x6d82c32, (q31_t)0x802ee133, (q31_t)0x6d1e5fe, (q31_t)0x802e8b58,
	(q31_t)0x6cb9fc6, (q31_t)0x802e35cb,
	(q31_t)0x6c5598a, (q31_t)0x802de08e, (q31_t)0x6bf1349, (q31_t)0x802d8ba0, (q31_t)0x6b8cd05, (q31_t)0x802d3700,
	(q31_t)0x6b286bc, (q31_t)0x802ce2af,
	(q31_t)0x6ac406f, (q31_t)0x802c8ead, (q31_t)0x6a5fa1e, (q31_t)0x802c3afa, (q31_t)0x69fb3c9, (q31_t)0x802be796,
	(q31_t)0x6996d70, (q31_t)0x802b9480,
	(q31_t)0x6932713, (q31_t)0x802b41ba, (q31_t)0x68ce0b2, (q31_t)0x802aef42, (q31_t)0x6869a4c, (q31_t)0x802a9d19,
	(q31_t)0x68053e3, (q31_t)0x802a4b3f,
	(q31_t)0x67a0d76, (q31_t)0x8029f9b4, (q31_t)0x673c704, (q31_t)0x8029a878, (q31_t)0x66d808f, (q31_t)0x8029578b,
	(q31_t)0x6673a16, (q31_t)0x802906ec,
	(q31_t)0x660f398, (q31_t)0x8028b69c, (q31_t)0x65aad17, (q31_t)0x8028669b, (q31_t)0x6546692, (q31_t)0x802816e9,
	(q31_t)0x64e2009, (q31_t)0x8027c786,
	(q31_t)0x647d97c, (q31_t)0x80277872, (q31_t)0x64192eb, (q31_t)0x802729ad, (q31_t)0x63b4c57, (q31_t)0x8026db36,
	(q31_t)0x63505be, (q31_t)0x80268d0e,
	(q31_t)0x62ebf22, (q31_t)0x80263f36, (q31_t)0x6287882, (q31_t)0x8025f1ac, (q31_t)0x62231de, (q31_t)0x8025a471,
	(q31_t)0x61beb36, (q31_t)0x80255784,
	(q31_t)0x615a48b, (q31_t)0x80250ae7, (q31_t)0x60f5ddc, (q31_t)0x8024be99, (q31_t)0x6091729, (q31_t)0x80247299,
	(q31_t)0x602d072, (q31_t)0x802426e8,
	(q31_t)0x5fc89b8, (q31_t)0x8023db86, (q31_t)0x5f642fa, (q31_t)0x80239073, (q31_t)0x5effc38, (q31_t)0x802345af,
	(q31_t)0x5e9b572, (q31_t)0x8022fb3a,
	(q31_t)0x5e36ea9, (q31_t)0x8022b114, (q31_t)0x5dd27dd, (q31_t)0x8022673c, (q31_t)0x5d6e10c, (q31_t)0x80221db3,
	(q31_t)0x5d09a38, (q31_t)0x8021d47a,
	(q31_t)0x5ca5361, (q31_t)0x80218b8f, (q31_t)0x5c40c86, (q31_t)0x802142f3, (q31_t)0x5bdc5a7, (q31_t)0x8020faa6,
	(q31_t)0x5b77ec5, (q31_t)0x8020b2a7,
	(q31_t)0x5b137df, (q31_t)0x80206af8, (q31_t)0x5aaf0f6, (q31_t)0x80202397, (q31_t)0x5a4aa09, (q31_t)0x801fdc86,
	(q31_t)0x59e6319, (q31_t)0x801f95c3,
	(q31_t)0x5981c26, (q31_t)0x801f4f4f, (q31_t)0x591d52f, (q31_t)0x801f092a, (q31_t)0x58b8e34, (q31_t)0x801ec354,
	(q31_t)0x5854736, (q31_t)0x801e7dcd,
	(q31_t)0x57f0035, (q31_t)0x801e3895, (q31_t)0x578b930, (q31_t)0x801df3ab, (q31_t)0x5727228, (q31_t)0x801daf11,
	(q31_t)0x56c2b1c, (q31_t)0x801d6ac5,
	(q31_t)0x565e40d, (q31_t)0x801d26c8, (q31_t)0x55f9cfb, (q31_t)0x801ce31a, (q31_t)0x55955e6, (q31_t)0x801c9fbb,
	(q31_t)0x5530ecd, (q31_t)0x801c5cab,
	(q31_t)0x54cc7b1, (q31_t)0x801c19ea, (q31_t)0x5468092, (q31_t)0x801bd777, (q31_t)0x540396f, (q31_t)0x801b9554,
	(q31_t)0x539f249, (q31_t)0x801b537f,
	(q31_t)0x533ab20, (q31_t)0x801b11fa, (q31_t)0x52d63f4, (q31_t)0x801ad0c3, (q31_t)0x5271cc4, (q31_t)0x801a8fdb,
	(q31_t)0x520d592, (q31_t)0x801a4f42,
	(q31_t)0x51a8e5c, (q31_t)0x801a0ef8, (q31_t)0x5144723, (q31_t)0x8019cefd, (q31_t)0x50dffe7, (q31_t)0x80198f50,
	(q31_t)0x507b8a8, (q31_t)0x80194ff3,
	(q31_t)0x5017165, (q31_t)0x801910e4, (q31_t)0x4fb2a20, (q31_t)0x8018d225, (q31_t)0x4f4e2d8, (q31_t)0x801893b4,
	(q31_t)0x4ee9b8c, (q31_t)0x80185592,
	(q31_t)0x4e8543e, (q31_t)0x801817bf, (q31_t)0x4e20cec, (q31_t)0x8017da3b, (q31_t)0x4dbc597, (q31_t)0x80179d06,
	(q31_t)0x4d57e40, (q31_t)0x80176020,
	(q31_t)0x4cf36e5, (q31_t)0x80172388, (q31_t)0x4c8ef88, (q31_t)0x8016e740, (q31_t)0x4c2a827, (q31_t)0x8016ab46,
	(q31_t)0x4bc60c4, (q31_t)0x80166f9c,
	(q31_t)0x4b6195d, (q31_t)0x80163440, (q31_t)0x4afd1f4, (q31_t)0x8015f933, (q31_t)0x4a98a88, (q31_t)0x8015be75,
	(q31_t)0x4a34319, (q31_t)0x80158406,
	(q31_t)0x49cfba7, (q31_t)0x801549e6, (q31_t)0x496b432, (q31_t)0x80151015, (q31_t)0x4906cbb, (q31_t)0x8014d693,
	(q31_t)0x48a2540, (q31_t)0x80149d5f,
	(q31_t)0x483ddc3, (q31_t)0x8014647b, (q31_t)0x47d9643, (q31_t)0x80142be5, (q31_t)0x4774ec1, (q31_t)0x8013f39e,
	(q31_t)0x471073b, (q31_t)0x8013bba7,
	(q31_t)0x46abfb3, (q31_t)0x801383fe, (q31_t)0x4647828, (q31_t)0x80134ca4, (q31_t)0x45e309a, (q31_t)0x80131599,
	(q31_t)0x457e90a, (q31_t)0x8012dedd,
	(q31_t)0x451a177, (q31_t)0x8012a86f, (q31_t)0x44b59e1, (q31_t)0x80127251, (q31_t)0x4451249, (q31_t)0x80123c82,
	(q31_t)0x43ecaae, (q31_t)0x80120701,
	(q31_t)0x4388310, (q31_t)0x8011d1d0, (q31_t)0x4323b70, (q31_t)0x80119ced, (q31_t)0x42bf3cd, (q31_t)0x80116859,
	(q31_t)0x425ac28, (q31_t)0x80113414,
	(q31_t)0x41f6480, (q31_t)0x8011001f, (q31_t)0x4191cd5, (q31_t)0x8010cc78, (q31_t)0x412d528, (q31_t)0x8010991f,
	(q31_t)0x40c8d79, (q31_t)0x80106616,
	(q31_t)0x40645c7, (q31_t)0x8010335c, (q31_t)0x3fffe12, (q31_t)0x801000f1, (q31_t)0x3f9b65b, (q31_t)0x800fced4,
	(q31_t)0x3f36ea2, (q31_t)0x800f9d07,
	(q31_t)0x3ed26e6, (q31_t)0x800f6b88, (q31_t)0x3e6df28, (q31_t)0x800f3a59, (q31_t)0x3e09767, (q31_t)0x800f0978,
	(q31_t)0x3da4fa4, (q31_t)0x800ed8e6,
	(q31_t)0x3d407df, (q31_t)0x800ea8a3, (q31_t)0x3cdc017, (q31_t)0x800e78af, (q31_t)0x3c7784d, (q31_t)0x800e490a,
	(q31_t)0x3c13080, (q31_t)0x800e19b4,
	(q31_t)0x3bae8b2, (q31_t)0x800deaad, (q31_t)0x3b4a0e0, (q31_t)0x800dbbf5, (q31_t)0x3ae590d, (q31_t)0x800d8d8b,
	(q31_t)0x3a81137, (q31_t)0x800d5f71,
	(q31_t)0x3a1c960, (q31_t)0x800d31a5, (q31_t)0x39b8185, (q31_t)0x800d0429, (q31_t)0x39539a9, (q31_t)0x800cd6fb,
	(q31_t)0x38ef1ca, (q31_t)0x800caa1c,
	(q31_t)0x388a9ea, (q31_t)0x800c7d8c, (q31_t)0x3826207, (q31_t)0x800c514c, (q31_t)0x37c1a22, (q31_t)0x800c255a,
	(q31_t)0x375d23a, (q31_t)0x800bf9b7,
	(q31_t)0x36f8a51, (q31_t)0x800bce63, (q31_t)0x3694265, (q31_t)0x800ba35d, (q31_t)0x362fa78, (q31_t)0x800b78a7,
	(q31_t)0x35cb288, (q31_t)0x800b4e40,
	(q31_t)0x3566a96, (q31_t)0x800b2427, (q31_t)0x35022a2, (q31_t)0x800afa5e, (q31_t)0x349daac, (q31_t)0x800ad0e3,
	(q31_t)0x34392b4, (q31_t)0x800aa7b8,
	(q31_t)0x33d4abb, (q31_t)0x800a7edb, (q31_t)0x33702bf, (q31_t)0x800a564e, (q31_t)0x330bac1, (q31_t)0x800a2e0f,
	(q31_t)0x32a72c1, (q31_t)0x800a061f,
	(q31_t)0x3242abf, (q31_t)0x8009de7e, (q31_t)0x31de2bb, (q31_t)0x8009b72c, (q31_t)0x3179ab5, (q31_t)0x80099029,
	(q31_t)0x31152ae, (q31_t)0x80096975,
	(q31_t)0x30b0aa4, (q31_t)0x80094310, (q31_t)0x304c299, (q31_t)0x80091cf9, (q31_t)0x2fe7a8c, (q31_t)0x8008f732,
	(q31_t)0x2f8327d, (q31_t)0x8008d1ba,
	(q31_t)0x2f1ea6c, (q31_t)0x8008ac90, (q31_t)0x2eba259, (q31_t)0x800887b6, (q31_t)0x2e55a44, (q31_t)0x8008632a,
	(q31_t)0x2df122e, (q31_t)0x80083eed,
	(q31_t)0x2d8ca16, (q31_t)0x80081b00, (q31_t)0x2d281fc, (q31_t)0x8007f761, (q31_t)0x2cc39e1, (q31_t)0x8007d411,
	(q31_t)0x2c5f1c3, (q31_t)0x8007b110,
	(q31_t)0x2bfa9a4, (q31_t)0x80078e5e, (q31_t)0x2b96184, (q31_t)0x80076bfb, (q31_t)0x2b31961, (q31_t)0x800749e7,
	(q31_t)0x2acd13d, (q31_t)0x80072822,
	(q31_t)0x2a68917, (q31_t)0x800706ac, (q31_t)0x2a040f0, (q31_t)0x8006e585, (q31_t)0x299f8c7, (q31_t)0x8006c4ac,
	(q31_t)0x293b09c, (q31_t)0x8006a423,
	(q31_t)0x28d6870, (q31_t)0x800683e8, (q31_t)0x2872043, (q31_t)0x800663fd, (q31_t)0x280d813, (q31_t)0x80064460,
	(q31_t)0x27a8fe2, (q31_t)0x80062513,
	(q31_t)0x27447b0, (q31_t)0x80060614, (q31_t)0x26dff7c, (q31_t)0x8005e764, (q31_t)0x267b747, (q31_t)0x8005c904,
	(q31_t)0x2616f10, (q31_t)0x8005aaf2,
	(q31_t)0x25b26d7, (q31_t)0x80058d2f, (q31_t)0x254de9e, (q31_t)0x80056fbb, (q31_t)0x24e9662, (q31_t)0x80055296,
	(q31_t)0x2484e26, (q31_t)0x800535c0,
	(q31_t)0x24205e8, (q31_t)0x80051939, (q31_t)0x23bbda8, (q31_t)0x8004fd00, (q31_t)0x2357567, (q31_t)0x8004e117,
	(q31_t)0x22f2d25, (q31_t)0x8004c57d,
	(q31_t)0x228e4e2, (q31_t)0x8004aa32, (q31_t)0x2229c9d, (q31_t)0x80048f35, (q31_t)0x21c5457, (q31_t)0x80047488,
	(q31_t)0x2160c0f, (q31_t)0x80045a29,
	(q31_t)0x20fc3c6, (q31_t)0x8004401a, (q31_t)0x2097b7c, (q31_t)0x80042659, (q31_t)0x2033331, (q31_t)0x80040ce7,
	(q31_t)0x1fceae4, (q31_t)0x8003f3c5,
	(q31_t)0x1f6a297, (q31_t)0x8003daf1, (q31_t)0x1f05a48, (q31_t)0x8003c26c, (q31_t)0x1ea11f7, (q31_t)0x8003aa36,
	(q31_t)0x1e3c9a6, (q31_t)0x8003924f,
	(q31_t)0x1dd8154, (q31_t)0x80037ab7, (q31_t)0x1d73900, (q31_t)0x8003636e, (q31_t)0x1d0f0ab, (q31_t)0x80034c74,
	(q31_t)0x1caa855, (q31_t)0x800335c9,
	(q31_t)0x1c45ffe, (q31_t)0x80031f6d, (q31_t)0x1be17a6, (q31_t)0x80030960, (q31_t)0x1b7cf4d, (q31_t)0x8002f3a1,
	(q31_t)0x1b186f3, (q31_t)0x8002de32,
	(q31_t)0x1ab3e97, (q31_t)0x8002c912, (q31_t)0x1a4f63b, (q31_t)0x8002b440, (q31_t)0x19eaddd, (q31_t)0x80029fbe,
	(q31_t)0x198657f, (q31_t)0x80028b8a,
	(q31_t)0x1921d20, (q31_t)0x800277a6, (q31_t)0x18bd4bf, (q31_t)0x80026410, (q31_t)0x1858c5e, (q31_t)0x800250c9,
	(q31_t)0x17f43fc, (q31_t)0x80023dd2,
	(q31_t)0x178fb99, (q31_t)0x80022b29, (q31_t)0x172b335, (q31_t)0x800218cf, (q31_t)0x16c6ad0, (q31_t)0x800206c4,
	(q31_t)0x166226a, (q31_t)0x8001f508,
	(q31_t)0x15fda03, (q31_t)0x8001e39b, (q31_t)0x159919c, (q31_t)0x8001d27d, (q31_t)0x1534934, (q31_t)0x8001c1ae,
	(q31_t)0x14d00ca, (q31_t)0x8001b12e,
	(q31_t)0x146b860, (q31_t)0x8001a0fd, (q31_t)0x1406ff6, (q31_t)0x8001911b, (q31_t)0x13a278a, (q31_t)0x80018187,
	(q31_t)0x133df1e, (q31_t)0x80017243,
	(q31_t)0x12d96b1, (q31_t)0x8001634e, (q31_t)0x1274e43, (q31_t)0x800154a7, (q31_t)0x12105d5, (q31_t)0x80014650,
	(q31_t)0x11abd66, (q31_t)0x80013847,
	(q31_t)0x11474f6, (q31_t)0x80012a8e, (q31_t)0x10e2c85, (q31_t)0x80011d23, (q31_t)0x107e414, (q31_t)0x80011008,
	(q31_t)0x1019ba2, (q31_t)0x8001033b,
	(q31_t)0xfb5330, (q31_t)0x8000f6bd, (q31_t)0xf50abd, (q31_t)0x8000ea8e, (q31_t)0xeec249, (q31_t)0x8000deaf, (q31_t)0xe879d5,
	(q31_t)0x8000d31e,
	(q31_t)0xe23160, (q31_t)0x8000c7dc, (q31_t)0xdbe8eb, (q31_t)0x8000bce9, (q31_t)0xd5a075, (q31_t)0x8000b245, (q31_t)0xcf57ff,
	(q31_t)0x8000a7f0,
	(q31_t)0xc90f88, (q31_t)0x80009dea, (q31_t)0xc2c711, (q31_t)0x80009433, (q31_t)0xbc7e99, (q31_t)0x80008aca, (q31_t)0xb63621,
	(q31_t)0x800081b1,
	(q31_t)0xafeda8, (q31_t)0x800078e7, (q31_t)0xa9a52f, (q31_t)0x8000706c, (q31_t)0xa35cb5, (q31_t)0x8000683f, (q31_t)0x9d143b,
	(q31_t)0x80006062,
	(q31_t)0x96cbc1, (q31_t)0x800058d4, (q31_t)0x908346, (q31_t)0x80005194, (q31_t)0x8a3acb, (q31_t)0x80004aa4, (q31_t)0x83f250,
	(q31_t)0x80004402,
	(q31_t)0x7da9d4, (q31_t)0x80003daf, (q31_t)0x776159, (q31_t)0x800037ac, (q31_t)0x7118dc, (q31_t)0x800031f7, (q31_t)0x6ad060,
	(q31_t)0x80002c91,
	(q31_t)0x6487e3, (q31_t)0x8000277a, (q31_t)0x5e3f66, (q31_t)0x800022b3, (q31_t)0x57f6e9, (q31_t)0x80001e3a, (q31_t)0x51ae6b,
	(q31_t)0x80001a10,
	(q31_t)0x4b65ee, (q31_t)0x80001635, (q31_t)0x451d70, (q31_t)0x800012a9, (q31_t)0x3ed4f2, (q31_t)0x80000f6c, (q31_t)0x388c74,
	(q31_t)0x80000c7e,
	(q31_t)0x3243f5, (q31_t)0x800009df, (q31_t)0x2bfb77, (q31_t)0x8000078e, (q31_t)0x25b2f8, (q31_t)0x8000058d, (q31_t)0x1f6a7a,
	(q31_t)0x800003db,
	(q31_t)0x1921fb, (q31_t)0x80000278, (q31_t)0x12d97c, (q31_t)0x80000163, (q31_t)0xc90fe, (q31_t)0x8000009e, (q31_t)0x6487f,
	(q31_t)0x80000027

};

/**
* \par
* cosFactor tables are generated using the formula : <pre>cos_factors[n] = 2 * cos((2n+1)*pi/(4*N))</pre>
* \par
* C command to generate the table
* <pre>
* for(i = 0; i< N; i++)
* {
*   cos_factors[i]= 2 * cos((2*i+1)*c/2);
* } </pre>
* \par
* where <code>N</code> is the number of factors to generate and <code>c</code> is <code>pi/(2*N)</code>
* \par
* Then converted to q31 format by multiplying with 2^31 and saturated if required.
*/


static const q31_t cos_factorsQ31_128[128] = {
	(q31_t)0x7fff6216, (q31_t)0x7ffa72d1, (q31_t)0x7ff09478, (q31_t)0x7fe1c76b, (q31_t)0x7fce0c3e, (q31_t)0x7fb563b3,
	(q31_t)0x7f97cebd, (q31_t)0x7f754e80,
	(q31_t)0x7f4de451, (q31_t)0x7f2191b4, (q31_t)0x7ef05860, (q31_t)0x7eba3a39, (q31_t)0x7e7f3957, (q31_t)0x7e3f57ff,
	(q31_t)0x7dfa98a8, (q31_t)0x7db0fdf8,
	(q31_t)0x7d628ac6, (q31_t)0x7d0f4218, (q31_t)0x7cb72724, (q31_t)0x7c5a3d50, (q31_t)0x7bf88830, (q31_t)0x7b920b89,
	(q31_t)0x7b26cb4f, (q31_t)0x7ab6cba4,
	(q31_t)0x7a4210d8, (q31_t)0x79c89f6e, (q31_t)0x794a7c12, (q31_t)0x78c7aba2, (q31_t)0x78403329, (q31_t)0x77b417df,
	(q31_t)0x77235f2d, (q31_t)0x768e0ea6,
	(q31_t)0x75f42c0b, (q31_t)0x7555bd4c, (q31_t)0x74b2c884, (q31_t)0x740b53fb, (q31_t)0x735f6626, (q31_t)0x72af05a7,
	(q31_t)0x71fa3949, (q31_t)0x71410805,
	(q31_t)0x708378ff, (q31_t)0x6fc19385, (q31_t)0x6efb5f12, (q31_t)0x6e30e34a, (q31_t)0x6d6227fa, (q31_t)0x6c8f351c,
	(q31_t)0x6bb812d1, (q31_t)0x6adcc964,
	(q31_t)0x69fd614a, (q31_t)0x6919e320, (q31_t)0x683257ab, (q31_t)0x6746c7d8, (q31_t)0x66573cbb, (q31_t)0x6563bf92,
	(q31_t)0x646c59bf, (q31_t)0x637114cc,
	(q31_t)0x6271fa69, (q31_t)0x616f146c, (q31_t)0x60686ccf, (q31_t)0x5f5e0db3, (q31_t)0x5e50015d, (q31_t)0x5d3e5237,
	(q31_t)0x5c290acc, (q31_t)0x5b1035cf,
	(q31_t)0x59f3de12, (q31_t)0x58d40e8c, (q31_t)0x57b0d256, (q31_t)0x568a34a9, (q31_t)0x556040e2, (q31_t)0x5433027d,
	(q31_t)0x53028518, (q31_t)0x51ced46e,
	(q31_t)0x5097fc5e, (q31_t)0x4f5e08e3, (q31_t)0x4e210617, (q31_t)0x4ce10034, (q31_t)0x4b9e0390, (q31_t)0x4a581c9e,
	(q31_t)0x490f57ee, (q31_t)0x47c3c22f,
	(q31_t)0x46756828, (q31_t)0x452456bd, (q31_t)0x43d09aed, (q31_t)0x427a41d0, (q31_t)0x4121589b, (q31_t)0x3fc5ec98,
	(q31_t)0x3e680b2c, (q31_t)0x3d07c1d6,
	(q31_t)0x3ba51e29, (q31_t)0x3a402dd2, (q31_t)0x38d8fe93, (q31_t)0x376f9e46, (q31_t)0x36041ad9, (q31_t)0x34968250,
	(q31_t)0x3326e2c3, (q31_t)0x31b54a5e,
	(q31_t)0x3041c761, (q31_t)0x2ecc681e, (q31_t)0x2d553afc, (q31_t)0x2bdc4e6f, (q31_t)0x2a61b101, (q31_t)0x28e5714b,
	(q31_t)0x27679df4, (q31_t)0x25e845b6,
	(q31_t)0x24677758, (q31_t)0x22e541af, (q31_t)0x2161b3a0, (q31_t)0x1fdcdc1b, (q31_t)0x1e56ca1e, (q31_t)0x1ccf8cb3,
	(q31_t)0x1b4732ef, (q31_t)0x19bdcbf3,
	(q31_t)0x183366e9, (q31_t)0x16a81305, (q31_t)0x151bdf86, (q31_t)0x138edbb1, (q31_t)0x120116d5, (q31_t)0x1072a048,
	(q31_t)0xee38766, (q31_t)0xd53db92,
	(q31_t)0xbc3ac35, (q31_t)0xa3308bd, (q31_t)0x8a2009a, (q31_t)0x710a345, (q31_t)0x57f0035, (q31_t)0x3ed26e6, (q31_t)0x25b26d7,
	(q31_t)0xc90f88
};

static const q31_t cos_factorsQ31_512[512] = {
	(q31_t)0x7ffff621, (q31_t)0x7fffa72c, (q31_t)0x7fff0943, (q31_t)0x7ffe1c65, (q31_t)0x7ffce093, (q31_t)0x7ffb55ce,
	(q31_t)0x7ff97c18, (q31_t)0x7ff75370,
	(q31_t)0x7ff4dbd9, (q31_t)0x7ff21553, (q31_t)0x7feeffe1, (q31_t)0x7feb9b85, (q31_t)0x7fe7e841, (q31_t)0x7fe3e616,
	(q31_t)0x7fdf9508, (q31_t)0x7fdaf519,
	(q31_t)0x7fd6064c, (q31_t)0x7fd0c8a3, (q31_t)0x7fcb3c23, (q31_t)0x7fc560cf, (q31_t)0x7fbf36aa, (q31_t)0x7fb8bdb8,
	(q31_t)0x7fb1f5fc, (q31_t)0x7faadf7c,
	(q31_t)0x7fa37a3c, (q31_t)0x7f9bc640, (q31_t)0x7f93c38c, (q31_t)0x7f8b7227, (q31_t)0x7f82d214, (q31_t)0x7f79e35a,
	(q31_t)0x7f70a5fe, (q31_t)0x7f671a05,
	(q31_t)0x7f5d3f75, (q31_t)0x7f531655, (q31_t)0x7f489eaa, (q31_t)0x7f3dd87c, (q31_t)0x7f32c3d1, (q31_t)0x7f2760af,
	(q31_t)0x7f1baf1e, (q31_t)0x7f0faf25,
	(q31_t)0x7f0360cb, (q31_t)0x7ef6c418, (q31_t)0x7ee9d914, (q31_t)0x7edc9fc6, (q31_t)0x7ecf1837, (q31_t)0x7ec14270,
	(q31_t)0x7eb31e78, (q31_t)0x7ea4ac58,
	(q31_t)0x7e95ec1a, (q31_t)0x7e86ddc6, (q31_t)0x7e778166, (q31_t)0x7e67d703, (q31_t)0x7e57dea7, (q31_t)0x7e47985b,
	(q31_t)0x7e37042a, (q31_t)0x7e26221f,
	(q31_t)0x7e14f242, (q31_t)0x7e0374a0, (q31_t)0x7df1a942, (q31_t)0x7ddf9034, (q31_t)0x7dcd2981, (q31_t)0x7dba7534,
	(q31_t)0x7da77359, (q31_t)0x7d9423fc,
	(q31_t)0x7d808728, (q31_t)0x7d6c9ce9, (q31_t)0x7d58654d, (q31_t)0x7d43e05e, (q31_t)0x7d2f0e2b, (q31_t)0x7d19eebf,
	(q31_t)0x7d048228, (q31_t)0x7ceec873,
	(q31_t)0x7cd8c1ae, (q31_t)0x7cc26de5, (q31_t)0x7cabcd28, (q31_t)0x7c94df83, (q31_t)0x7c7da505, (q31_t)0x7c661dbc,
	(q31_t)0x7c4e49b7, (q31_t)0x7c362904,
	(q31_t)0x7c1dbbb3, (q31_t)0x7c0501d2, (q31_t)0x7bebfb70, (q31_t)0x7bd2a89e, (q31_t)0x7bb9096b, (q31_t)0x7b9f1de6,
	(q31_t)0x7b84e61f, (q31_t)0x7b6a6227,
	(q31_t)0x7b4f920e, (q31_t)0x7b3475e5, (q31_t)0x7b190dbc, (q31_t)0x7afd59a4, (q31_t)0x7ae159ae, (q31_t)0x7ac50dec,
	(q31_t)0x7aa8766f, (q31_t)0x7a8b9348,
	(q31_t)0x7a6e648a, (q31_t)0x7a50ea47, (q31_t)0x7a332490, (q31_t)0x7a151378, (q31_t)0x79f6b711, (q31_t)0x79d80f6f,
	(q31_t)0x79b91ca4, (q31_t)0x7999dec4,
	(q31_t)0x797a55e0, (q31_t)0x795a820e, (q31_t)0x793a6361, (q31_t)0x7919f9ec, (q31_t)0x78f945c3, (q31_t)0x78d846fb,
	(q31_t)0x78b6fda8, (q31_t)0x789569df,
	(q31_t)0x78738bb3, (q31_t)0x7851633b, (q31_t)0x782ef08b, (q31_t)0x780c33b8, (q31_t)0x77e92cd9, (q31_t)0x77c5dc01,
	(q31_t)0x77a24148, (q31_t)0x777e5cc3,
	(q31_t)0x775a2e89, (q31_t)0x7735b6af, (q31_t)0x7710f54c, (q31_t)0x76ebea77, (q31_t)0x76c69647, (q31_t)0x76a0f8d2,
	(q31_t)0x767b1231, (q31_t)0x7654e279,
	(q31_t)0x762e69c4, (q31_t)0x7607a828, (q31_t)0x75e09dbd, (q31_t)0x75b94a9c, (q31_t)0x7591aedd, (q31_t)0x7569ca99,
	(q31_t)0x75419de7, (q31_t)0x751928e0,
	(q31_t)0x74f06b9e, (q31_t)0x74c7663a, (q31_t)0x749e18cd, (q31_t)0x74748371, (q31_t)0x744aa63f, (q31_t)0x74208150,
	(q31_t)0x73f614c0, (q31_t)0x73cb60a8,
	(q31_t)0x73a06522, (q31_t)0x73752249, (q31_t)0x73499838, (q31_t)0x731dc70a, (q31_t)0x72f1aed9, (q31_t)0x72c54fc1,
	(q31_t)0x7298a9dd, (q31_t)0x726bbd48,
	(q31_t)0x723e8a20, (q31_t)0x7211107e, (q31_t)0x71e35080, (q31_t)0x71b54a41, (q31_t)0x7186fdde, (q31_t)0x71586b74,
	(q31_t)0x7129931f, (q31_t)0x70fa74fc,
	(q31_t)0x70cb1128, (q31_t)0x709b67c0, (q31_t)0x706b78e3, (q31_t)0x703b44ad, (q31_t)0x700acb3c, (q31_t)0x6fda0cae,
	(q31_t)0x6fa90921, (q31_t)0x6f77c0b3,
	(q31_t)0x6f463383, (q31_t)0x6f1461b0, (q31_t)0x6ee24b57, (q31_t)0x6eaff099, (q31_t)0x6e7d5193, (q31_t)0x6e4a6e66,
	(q31_t)0x6e174730, (q31_t)0x6de3dc11,
	(q31_t)0x6db02d29, (q31_t)0x6d7c3a98, (q31_t)0x6d48047e, (q31_t)0x6d138afb, (q31_t)0x6cdece2f, (q31_t)0x6ca9ce3b,
	(q31_t)0x6c748b3f, (q31_t)0x6c3f055d,
	(q31_t)0x6c093cb6, (q31_t)0x6bd3316a, (q31_t)0x6b9ce39b, (q31_t)0x6b66536b, (q31_t)0x6b2f80fb, (q31_t)0x6af86c6c,
	(q31_t)0x6ac115e2, (q31_t)0x6a897d7d,
	(q31_t)0x6a51a361, (q31_t)0x6a1987b0, (q31_t)0x69e12a8c, (q31_t)0x69a88c19, (q31_t)0x696fac78, (q31_t)0x69368bce,
	(q31_t)0x68fd2a3d, (q31_t)0x68c387e9,
	(q31_t)0x6889a4f6, (q31_t)0x684f8186, (q31_t)0x68151dbe, (q31_t)0x67da79c3, (q31_t)0x679f95b7, (q31_t)0x676471c0,
	(q31_t)0x67290e02, (q31_t)0x66ed6aa1,
	(q31_t)0x66b187c3, (q31_t)0x6675658c, (q31_t)0x66390422, (q31_t)0x65fc63a9, (q31_t)0x65bf8447, (q31_t)0x65826622,
	(q31_t)0x6545095f, (q31_t)0x65076e25,
	(q31_t)0x64c99498, (q31_t)0x648b7ce0, (q31_t)0x644d2722, (q31_t)0x640e9386, (q31_t)0x63cfc231, (q31_t)0x6390b34a,
	(q31_t)0x635166f9, (q31_t)0x6311dd64,
	(q31_t)0x62d216b3, (q31_t)0x6292130c, (q31_t)0x6251d298, (q31_t)0x6211557e, (q31_t)0x61d09be5, (q31_t)0x618fa5f7,
	(q31_t)0x614e73da, (q31_t)0x610d05b7,
	(q31_t)0x60cb5bb7, (q31_t)0x60897601, (q31_t)0x604754bf, (q31_t)0x6004f819, (q31_t)0x5fc26038, (q31_t)0x5f7f8d46,
	(q31_t)0x5f3c7f6b, (q31_t)0x5ef936d1,
	(q31_t)0x5eb5b3a2, (q31_t)0x5e71f606, (q31_t)0x5e2dfe29, (q31_t)0x5de9cc33, (q31_t)0x5da5604f, (q31_t)0x5d60baa7,
	(q31_t)0x5d1bdb65, (q31_t)0x5cd6c2b5,
	(q31_t)0x5c9170bf, (q31_t)0x5c4be5b0, (q31_t)0x5c0621b2, (q31_t)0x5bc024f0, (q31_t)0x5b79ef96, (q31_t)0x5b3381ce,
	(q31_t)0x5aecdbc5, (q31_t)0x5aa5fda5,
	(q31_t)0x5a5ee79a, (q31_t)0x5a1799d1, (q31_t)0x59d01475, (q31_t)0x598857b2, (q31_t)0x594063b5, (q31_t)0x58f838a9,
	(q31_t)0x58afd6bd, (q31_t)0x58673e1b,
	(q31_t)0x581e6ef1, (q31_t)0x57d5696d, (q31_t)0x578c2dba, (q31_t)0x5742bc06, (q31_t)0x56f9147e, (q31_t)0x56af3750,
	(q31_t)0x566524aa, (q31_t)0x561adcb9,
	(q31_t)0x55d05faa, (q31_t)0x5585adad, (q31_t)0x553ac6ee, (q31_t)0x54efab9c, (q31_t)0x54a45be6, (q31_t)0x5458d7f9,
	(q31_t)0x540d2005, (q31_t)0x53c13439,
	(q31_t)0x537514c2, (q31_t)0x5328c1d0, (q31_t)0x52dc3b92, (q31_t)0x528f8238, (q31_t)0x524295f0, (q31_t)0x51f576ea,
	(q31_t)0x51a82555, (q31_t)0x515aa162,
	(q31_t)0x510ceb40, (q31_t)0x50bf031f, (q31_t)0x5070e92f, (q31_t)0x50229da1, (q31_t)0x4fd420a4, (q31_t)0x4f857269,
	(q31_t)0x4f369320, (q31_t)0x4ee782fb,
	(q31_t)0x4e984229, (q31_t)0x4e48d0dd, (q31_t)0x4df92f46, (q31_t)0x4da95d96, (q31_t)0x4d595bfe, (q31_t)0x4d092ab0,
	(q31_t)0x4cb8c9dd, (q31_t)0x4c6839b7,
	(q31_t)0x4c177a6e, (q31_t)0x4bc68c36, (q31_t)0x4b756f40, (q31_t)0x4b2423be, (q31_t)0x4ad2a9e2, (q31_t)0x4a8101de,
	(q31_t)0x4a2f2be6, (q31_t)0x49dd282a,
	(q31_t)0x498af6df, (q31_t)0x49389836, (q31_t)0x48e60c62, (q31_t)0x48935397, (q31_t)0x48406e08, (q31_t)0x47ed5be6,
	(q31_t)0x479a1d67, (q31_t)0x4746b2bc,
	(q31_t)0x46f31c1a, (q31_t)0x469f59b4, (q31_t)0x464b6bbe, (q31_t)0x45f7526b, (q31_t)0x45a30df0, (q31_t)0x454e9e80,
	(q31_t)0x44fa0450, (q31_t)0x44a53f93,
	(q31_t)0x4450507e, (q31_t)0x43fb3746, (q31_t)0x43a5f41e, (q31_t)0x4350873c, (q31_t)0x42faf0d4, (q31_t)0x42a5311b,
	(q31_t)0x424f4845, (q31_t)0x41f93689,
	(q31_t)0x41a2fc1a, (q31_t)0x414c992f, (q31_t)0x40f60dfb, (q31_t)0x409f5ab6, (q31_t)0x40487f94, (q31_t)0x3ff17cca,
	(q31_t)0x3f9a5290, (q31_t)0x3f430119,
	(q31_t)0x3eeb889c, (q31_t)0x3e93e950, (q31_t)0x3e3c2369, (q31_t)0x3de4371f, (q31_t)0x3d8c24a8, (q31_t)0x3d33ec39,
	(q31_t)0x3cdb8e09, (q31_t)0x3c830a50,
	(q31_t)0x3c2a6142, (q31_t)0x3bd19318, (q31_t)0x3b78a007, (q31_t)0x3b1f8848, (q31_t)0x3ac64c0f, (q31_t)0x3a6ceb96,
	(q31_t)0x3a136712, (q31_t)0x39b9bebc,
	(q31_t)0x395ff2c9, (q31_t)0x39060373, (q31_t)0x38abf0ef, (q31_t)0x3851bb77, (q31_t)0x37f76341, (q31_t)0x379ce885,
	(q31_t)0x37424b7b, (q31_t)0x36e78c5b,
	(q31_t)0x368cab5c, (q31_t)0x3631a8b8, (q31_t)0x35d684a6, (q31_t)0x357b3f5d, (q31_t)0x351fd918, (q31_t)0x34c4520d,
	(q31_t)0x3468aa76, (q31_t)0x340ce28b,
	(q31_t)0x33b0fa84, (q31_t)0x3354f29b, (q31_t)0x32f8cb07, (q31_t)0x329c8402, (q31_t)0x32401dc6, (q31_t)0x31e39889,
	(q31_t)0x3186f487, (q31_t)0x312a31f8,
	(q31_t)0x30cd5115, (q31_t)0x30705217, (q31_t)0x30133539, (q31_t)0x2fb5fab2, (q31_t)0x2f58a2be, (q31_t)0x2efb2d95,
	(q31_t)0x2e9d9b70, (q31_t)0x2e3fec8b,
	(q31_t)0x2de2211e, (q31_t)0x2d843964, (q31_t)0x2d263596, (q31_t)0x2cc815ee, (q31_t)0x2c69daa6, (q31_t)0x2c0b83fa,
	(q31_t)0x2bad1221, (q31_t)0x2b4e8558,
	(q31_t)0x2aefddd8, (q31_t)0x2a911bdc, (q31_t)0x2a323f9e, (q31_t)0x29d34958, (q31_t)0x29743946, (q31_t)0x29150fa1,
	(q31_t)0x28b5cca5, (q31_t)0x2856708d,
	(q31_t)0x27f6fb92, (q31_t)0x27976df1, (q31_t)0x2737c7e3, (q31_t)0x26d809a5, (q31_t)0x26783370, (q31_t)0x26184581,
	(q31_t)0x25b84012, (q31_t)0x2558235f,
	(q31_t)0x24f7efa2, (q31_t)0x2497a517, (q31_t)0x243743fa, (q31_t)0x23d6cc87, (q31_t)0x23763ef7, (q31_t)0x23159b88,
	(q31_t)0x22b4e274, (q31_t)0x225413f8,
	(q31_t)0x21f3304f, (q31_t)0x219237b5, (q31_t)0x21312a65, (q31_t)0x20d0089c, (q31_t)0x206ed295, (q31_t)0x200d888d,
	(q31_t)0x1fac2abf, (q31_t)0x1f4ab968,
	(q31_t)0x1ee934c3, (q31_t)0x1e879d0d, (q31_t)0x1e25f282, (q31_t)0x1dc4355e, (q31_t)0x1d6265dd, (q31_t)0x1d00843d,
	(q31_t)0x1c9e90b8, (q31_t)0x1c3c8b8c,
	(q31_t)0x1bda74f6, (q31_t)0x1b784d30, (q31_t)0x1b161479, (q31_t)0x1ab3cb0d, (q31_t)0x1a517128, (q31_t)0x19ef0707,
	(q31_t)0x198c8ce7, (q31_t)0x192a0304,
	(q31_t)0x18c7699b, (q31_t)0x1864c0ea, (q31_t)0x1802092c, (q31_t)0x179f429f, (q31_t)0x173c6d80, (q31_t)0x16d98a0c,
	(q31_t)0x1676987f, (q31_t)0x16139918,
	(q31_t)0x15b08c12, (q31_t)0x154d71aa, (q31_t)0x14ea4a1f, (q31_t)0x148715ae, (q31_t)0x1423d492, (q31_t)0x13c0870a,
	(q31_t)0x135d2d53, (q31_t)0x12f9c7aa,
	(q31_t)0x1296564d, (q31_t)0x1232d979, (q31_t)0x11cf516a, (q31_t)0x116bbe60, (q31_t)0x11082096, (q31_t)0x10a4784b,
	(q31_t)0x1040c5bb, (q31_t)0xfdd0926,
	(q31_t)0xf7942c7, (q31_t)0xf1572dc, (q31_t)0xeb199a4, (q31_t)0xe4db75b, (q31_t)0xde9cc40, (q31_t)0xd85d88f, (q31_t)0xd21dc87,
	(q31_t)0xcbdd865,
	(q31_t)0xc59cc68, (q31_t)0xbf5b8cb, (q31_t)0xb919dcf, (q31_t)0xb2d7baf, (q31_t)0xac952aa, (q31_t)0xa6522fe, (q31_t)0xa00ece8,
	(q31_t)0x99cb0a7,
	(q31_t)0x9386e78, (q31_t)0x8d42699, (q31_t)0x86fd947, (q31_t)0x80b86c2, (q31_t)0x7a72f45, (q31_t)0x742d311, (q31_t)0x6de7262,
	(q31_t)0x67a0d76,
	(q31_t)0x615a48b, (q31_t)0x5b137df, (q31_t)0x54cc7b1, (q31_t)0x4e8543e, (q31_t)0x483ddc3, (q31_t)0x41f6480, (q31_t)0x3bae8b2,
	(q31_t)0x3566a96,
	(q31_t)0x2f1ea6c, (q31_t)0x28d6870, (q31_t)0x228e4e2, (q31_t)0x1c45ffe, (q31_t)0x15fda03, (q31_t)0xfb5330, (q31_t)0x96cbc1,
	(q31_t)0x3243f5
};

static const q31_t cos_factorsQ31_2048[2048] = {
	(q31_t)0x7fffff62, (q31_t)0x7ffffa73, (q31_t)0x7ffff094, (q31_t)0x7fffe1c6, (q31_t)0x7fffce09, (q31_t)0x7fffb55c,
	(q31_t)0x7fff97c1, (q31_t)0x7fff7536,
	(q31_t)0x7fff4dbb, (q31_t)0x7fff2151, (q31_t)0x7ffeeff8, (q31_t)0x7ffeb9b0, (q31_t)0x7ffe7e79, (q31_t)0x7ffe3e52,
	(q31_t)0x7ffdf93c, (q31_t)0x7ffdaf37,
	(q31_t)0x7ffd6042, (q31_t)0x7ffd0c5f, (q31_t)0x7ffcb38c, (q31_t)0x7ffc55ca, (q31_t)0x7ffbf319, (q31_t)0x7ffb8b78,
	(q31_t)0x7ffb1ee9, (q31_t)0x7ffaad6a,
	(q31_t)0x7ffa36fc, (q31_t)0x7ff9bba0, (q31_t)0x7ff93b54, (q31_t)0x7ff8b619, (q31_t)0x7ff82bef, (q31_t)0x7ff79cd6,
	(q31_t)0x7ff708ce, (q31_t)0x7ff66fd7,
	(q31_t)0x7ff5d1f1, (q31_t)0x7ff52f1d, (q31_t)0x7ff48759, (q31_t)0x7ff3daa6, (q31_t)0x7ff32905, (q31_t)0x7ff27275,
	(q31_t)0x7ff1b6f6, (q31_t)0x7ff0f688,
	(q31_t)0x7ff0312c, (q31_t)0x7fef66e1, (q31_t)0x7fee97a7, (q31_t)0x7fedc37e, (q31_t)0x7fecea67, (q31_t)0x7fec0c62,
	(q31_t)0x7feb296d, (q31_t)0x7fea418b,
	(q31_t)0x7fe954ba, (q31_t)0x7fe862fa, (q31_t)0x7fe76c4c, (q31_t)0x7fe670b0, (q31_t)0x7fe57025, (q31_t)0x7fe46aac,
	(q31_t)0x7fe36045, (q31_t)0x7fe250ef,
	(q31_t)0x7fe13cac, (q31_t)0x7fe0237a, (q31_t)0x7fdf055a, (q31_t)0x7fdde24d, (q31_t)0x7fdcba51, (q31_t)0x7fdb8d67,
	(q31_t)0x7fda5b8f, (q31_t)0x7fd924ca,
	(q31_t)0x7fd7e917, (q31_t)0x7fd6a875, (q31_t)0x7fd562e7, (q31_t)0x7fd4186a, (q31_t)0x7fd2c900, (q31_t)0x7fd174a8,
	(q31_t)0x7fd01b63, (q31_t)0x7fcebd31,
	(q31_t)0x7fcd5a11, (q31_t)0x7fcbf203, (q31_t)0x7fca8508, (q31_t)0x7fc91320, (q31_t)0x7fc79c4b, (q31_t)0x7fc62089,
	(q31_t)0x7fc49fda, (q31_t)0x7fc31a3d,
	(q31_t)0x7fc18fb4, (q31_t)0x7fc0003e, (q31_t)0x7fbe6bdb, (q31_t)0x7fbcd28b, (q31_t)0x7fbb344e, (q31_t)0x7fb99125,
	(q31_t)0x7fb7e90f, (q31_t)0x7fb63c0d,
	(q31_t)0x7fb48a1e, (q31_t)0x7fb2d343, (q31_t)0x7fb1177b, (q31_t)0x7faf56c7, (q31_t)0x7fad9127, (q31_t)0x7fabc69b,
	(q31_t)0x7fa9f723, (q31_t)0x7fa822bf,
	(q31_t)0x7fa6496e, (q31_t)0x7fa46b32, (q31_t)0x7fa2880b, (q31_t)0x7fa09ff7, (q31_t)0x7f9eb2f8, (q31_t)0x7f9cc10d,
	(q31_t)0x7f9aca37, (q31_t)0x7f98ce76,
	(q31_t)0x7f96cdc9, (q31_t)0x7f94c831, (q31_t)0x7f92bdad, (q31_t)0x7f90ae3f, (q31_t)0x7f8e99e6, (q31_t)0x7f8c80a1,
	(q31_t)0x7f8a6272, (q31_t)0x7f883f58,
	(q31_t)0x7f861753, (q31_t)0x7f83ea64, (q31_t)0x7f81b88a, (q31_t)0x7f7f81c6, (q31_t)0x7f7d4617, (q31_t)0x7f7b057e,
	(q31_t)0x7f78bffb, (q31_t)0x7f76758e,
	(q31_t)0x7f742637, (q31_t)0x7f71d1f6, (q31_t)0x7f6f78cb, (q31_t)0x7f6d1ab6, (q31_t)0x7f6ab7b8, (q31_t)0x7f684fd0,
	(q31_t)0x7f65e2ff, (q31_t)0x7f637144,
	(q31_t)0x7f60faa0, (q31_t)0x7f5e7f13, (q31_t)0x7f5bfe9d, (q31_t)0x7f59793e, (q31_t)0x7f56eef5, (q31_t)0x7f545fc5,
	(q31_t)0x7f51cbab, (q31_t)0x7f4f32a9,
	(q31_t)0x7f4c94be, (q31_t)0x7f49f1eb, (q31_t)0x7f474a30, (q31_t)0x7f449d8c, (q31_t)0x7f41ec01, (q31_t)0x7f3f358d,
	(q31_t)0x7f3c7a31, (q31_t)0x7f39b9ee,
	(q31_t)0x7f36f4c3, (q31_t)0x7f342ab1, (q31_t)0x7f315bb7, (q31_t)0x7f2e87d6, (q31_t)0x7f2baf0d, (q31_t)0x7f28d15d,
	(q31_t)0x7f25eec7, (q31_t)0x7f230749,
	(q31_t)0x7f201ae5, (q31_t)0x7f1d299a, (q31_t)0x7f1a3368, (q31_t)0x7f173850, (q31_t)0x7f143852, (q31_t)0x7f11336d,
	(q31_t)0x7f0e29a3, (q31_t)0x7f0b1af2,
	(q31_t)0x7f08075c, (q31_t)0x7f04eedf, (q31_t)0x7f01d17d, (q31_t)0x7efeaf36, (q31_t)0x7efb8809, (q31_t)0x7ef85bf7,
	(q31_t)0x7ef52b00, (q31_t)0x7ef1f524,
	(q31_t)0x7eeeba62, (q31_t)0x7eeb7abc, (q31_t)0x7ee83632, (q31_t)0x7ee4ecc3, (q31_t)0x7ee19e6f, (q31_t)0x7ede4b38,
	(q31_t)0x7edaf31c, (q31_t)0x7ed7961c,
	(q31_t)0x7ed43438, (q31_t)0x7ed0cd70, (q31_t)0x7ecd61c5, (q31_t)0x7ec9f137, (q31_t)0x7ec67bc5, (q31_t)0x7ec3016f,
	(q31_t)0x7ebf8237, (q31_t)0x7ebbfe1c,
	(q31_t)0x7eb8751e, (q31_t)0x7eb4e73d, (q31_t)0x7eb1547a, (q31_t)0x7eadbcd4, (q31_t)0x7eaa204c, (q31_t)0x7ea67ee2,
	(q31_t)0x7ea2d896, (q31_t)0x7e9f2d68,
	(q31_t)0x7e9b7d58, (q31_t)0x7e97c867, (q31_t)0x7e940e94, (q31_t)0x7e904fe0, (q31_t)0x7e8c8c4b, (q31_t)0x7e88c3d5,
	(q31_t)0x7e84f67e, (q31_t)0x7e812447,
	(q31_t)0x7e7d4d2f, (q31_t)0x7e797136, (q31_t)0x7e75905d, (q31_t)0x7e71aaa4, (q31_t)0x7e6dc00c, (q31_t)0x7e69d093,
	(q31_t)0x7e65dc3b, (q31_t)0x7e61e303,
	(q31_t)0x7e5de4ec, (q31_t)0x7e59e1f5, (q31_t)0x7e55da20, (q31_t)0x7e51cd6c, (q31_t)0x7e4dbbd9, (q31_t)0x7e49a567,
	(q31_t)0x7e458a17, (q31_t)0x7e4169e9,
	(q31_t)0x7e3d44dd, (q31_t)0x7e391af3, (q31_t)0x7e34ec2b, (q31_t)0x7e30b885, (q31_t)0x7e2c8002, (q31_t)0x7e2842a2,
	(q31_t)0x7e240064, (q31_t)0x7e1fb94a,
	(q31_t)0x7e1b6d53, (q31_t)0x7e171c7f, (q31_t)0x7e12c6ce, (q31_t)0x7e0e6c42, (q31_t)0x7e0a0cd9, (q31_t)0x7e05a894,
	(q31_t)0x7e013f74, (q31_t)0x7dfcd178,
	(q31_t)0x7df85ea0, (q31_t)0x7df3e6ee, (q31_t)0x7def6a60, (q31_t)0x7deae8f7, (q31_t)0x7de662b3, (q31_t)0x7de1d795,
	(q31_t)0x7ddd479d, (q31_t)0x7dd8b2ca,
	(q31_t)0x7dd4191d, (q31_t)0x7dcf7a96, (q31_t)0x7dcad736, (q31_t)0x7dc62efc, (q31_t)0x7dc181e8, (q31_t)0x7dbccffc,
	(q31_t)0x7db81936, (q31_t)0x7db35d98,
	(q31_t)0x7dae9d21, (q31_t)0x7da9d7d2, (q31_t)0x7da50dab, (q31_t)0x7da03eab, (q31_t)0x7d9b6ad3, (q31_t)0x7d969224,
	(q31_t)0x7d91b49e, (q31_t)0x7d8cd240,
	(q31_t)0x7d87eb0a, (q31_t)0x7d82fefe, (q31_t)0x7d7e0e1c, (q31_t)0x7d791862, (q31_t)0x7d741dd2, (q31_t)0x7d6f1e6c,
	(q31_t)0x7d6a1a31, (q31_t)0x7d65111f,
	(q31_t)0x7d600338, (q31_t)0x7d5af07b, (q31_t)0x7d55d8e9, (q31_t)0x7d50bc82, (q31_t)0x7d4b9b46, (q31_t)0x7d467536,
	(q31_t)0x7d414a51, (q31_t)0x7d3c1a98,
	(q31_t)0x7d36e60b, (q31_t)0x7d31acaa, (q31_t)0x7d2c6e76, (q31_t)0x7d272b6e, (q31_t)0x7d21e393, (q31_t)0x7d1c96e5,
	(q31_t)0x7d174564, (q31_t)0x7d11ef11,
	(q31_t)0x7d0c93eb, (q31_t)0x7d0733f3, (q31_t)0x7d01cf29, (q31_t)0x7cfc658d, (q31_t)0x7cf6f720, (q31_t)0x7cf183e1,
	(q31_t)0x7cec0bd1, (q31_t)0x7ce68ef0,
	(q31_t)0x7ce10d3f, (q31_t)0x7cdb86bd, (q31_t)0x7cd5fb6a, (q31_t)0x7cd06b48, (q31_t)0x7ccad656, (q31_t)0x7cc53c94,
	(q31_t)0x7cbf9e03, (q31_t)0x7cb9faa2,
	(q31_t)0x7cb45272, (q31_t)0x7caea574, (q31_t)0x7ca8f3a7, (q31_t)0x7ca33d0c, (q31_t)0x7c9d81a3, (q31_t)0x7c97c16b,
	(q31_t)0x7c91fc66, (q31_t)0x7c8c3294,
	(q31_t)0x7c8663f4, (q31_t)0x7c809088, (q31_t)0x7c7ab84e, (q31_t)0x7c74db48, (q31_t)0x7c6ef976, (q31_t)0x7c6912d7,
	(q31_t)0x7c63276d, (q31_t)0x7c5d3737,
	(q31_t)0x7c574236, (q31_t)0x7c514869, (q31_t)0x7c4b49d2, (q31_t)0x7c45466f, (q31_t)0x7c3f3e42, (q31_t)0x7c39314b,
	(q31_t)0x7c331f8a, (q31_t)0x7c2d08ff,
	(q31_t)0x7c26edab, (q31_t)0x7c20cd8d, (q31_t)0x7c1aa8a6, (q31_t)0x7c147ef6, (q31_t)0x7c0e507e, (q31_t)0x7c081d3d,
	(q31_t)0x7c01e534, (q31_t)0x7bfba863,
	(q31_t)0x7bf566cb, (q31_t)0x7bef206b, (q31_t)0x7be8d544, (q31_t)0x7be28556, (q31_t)0x7bdc30a1, (q31_t)0x7bd5d726,
	(q31_t)0x7bcf78e5, (q31_t)0x7bc915dd,
	(q31_t)0x7bc2ae10, (q31_t)0x7bbc417e, (q31_t)0x7bb5d026, (q31_t)0x7baf5a09, (q31_t)0x7ba8df28, (q31_t)0x7ba25f82,
	(q31_t)0x7b9bdb18, (q31_t)0x7b9551ea,
	(q31_t)0x7b8ec3f8, (q31_t)0x7b883143, (q31_t)0x7b8199ca, (q31_t)0x7b7afd8f, (q31_t)0x7b745c91, (q31_t)0x7b6db6d0,
	(q31_t)0x7b670c4d, (q31_t)0x7b605d09,
	(q31_t)0x7b59a902, (q31_t)0x7b52f03a, (q31_t)0x7b4c32b1, (q31_t)0x7b457068, (q31_t)0x7b3ea95d, (q31_t)0x7b37dd92,
	(q31_t)0x7b310d07, (q31_t)0x7b2a37bc,
	(q31_t)0x7b235db2, (q31_t)0x7b1c7ee8, (q31_t)0x7b159b5f, (q31_t)0x7b0eb318, (q31_t)0x7b07c612, (q31_t)0x7b00d44d,
	(q31_t)0x7af9ddcb, (q31_t)0x7af2e28b,
	(q31_t)0x7aebe28d, (q31_t)0x7ae4ddd2, (q31_t)0x7addd45b, (q31_t)0x7ad6c626, (q31_t)0x7acfb336, (q31_t)0x7ac89b89,
	(q31_t)0x7ac17f20, (q31_t)0x7aba5dfc,
	(q31_t)0x7ab3381d, (q31_t)0x7aac0d82, (q31_t)0x7aa4de2d, (q31_t)0x7a9daa1d, (q31_t)0x7a967153, (q31_t)0x7a8f33d0,
	(q31_t)0x7a87f192, (q31_t)0x7a80aa9c,
	(q31_t)0x7a795eec, (q31_t)0x7a720e84, (q31_t)0x7a6ab963, (q31_t)0x7a635f8a, (q31_t)0x7a5c00f9, (q31_t)0x7a549db0,
	(q31_t)0x7a4d35b0, (q31_t)0x7a45c8f9,
	(q31_t)0x7a3e578b, (q31_t)0x7a36e166, (q31_t)0x7a2f668c, (q31_t)0x7a27e6fb, (q31_t)0x7a2062b5, (q31_t)0x7a18d9b9,
	(q31_t)0x7a114c09, (q31_t)0x7a09b9a4,
	(q31_t)0x7a02228a, (q31_t)0x79fa86bc, (q31_t)0x79f2e63a, (q31_t)0x79eb4105, (q31_t)0x79e3971c, (q31_t)0x79dbe880,
	(q31_t)0x79d43532, (q31_t)0x79cc7d31,
	(q31_t)0x79c4c07e, (q31_t)0x79bcff19, (q31_t)0x79b53903, (q31_t)0x79ad6e3c, (q31_t)0x79a59ec3, (q31_t)0x799dca9a,
	(q31_t)0x7995f1c1, (q31_t)0x798e1438,
	(q31_t)0x798631ff, (q31_t)0x797e4b16, (q31_t)0x79765f7f, (q31_t)0x796e6f39, (q31_t)0x79667a44, (q31_t)0x795e80a1,
	(q31_t)0x79568250, (q31_t)0x794e7f52,
	(q31_t)0x794677a6, (q31_t)0x793e6b4e, (q31_t)0x79365a49, (q31_t)0x792e4497, (q31_t)0x79262a3a, (q31_t)0x791e0b31,
	(q31_t)0x7915e77c, (q31_t)0x790dbf1d,
	(q31_t)0x79059212, (q31_t)0x78fd605d, (q31_t)0x78f529fe, (q31_t)0x78eceef6, (q31_t)0x78e4af44, (q31_t)0x78dc6ae8,
	(q31_t)0x78d421e4, (q31_t)0x78cbd437,
	(q31_t)0x78c381e2, (q31_t)0x78bb2ae5, (q31_t)0x78b2cf41, (q31_t)0x78aa6ef5, (q31_t)0x78a20a03, (q31_t)0x7899a06a,
	(q31_t)0x7891322a, (q31_t)0x7888bf45,
	(q31_t)0x788047ba, (q31_t)0x7877cb89, (q31_t)0x786f4ab4, (q31_t)0x7866c53a, (q31_t)0x785e3b1c, (q31_t)0x7855ac5a,
	(q31_t)0x784d18f4, (q31_t)0x784480ea,
	(q31_t)0x783be43e, (q31_t)0x783342ef, (q31_t)0x782a9cfe, (q31_t)0x7821f26b, (q31_t)0x78194336, (q31_t)0x78108f60,
	(q31_t)0x7807d6e9, (q31_t)0x77ff19d1,
	(q31_t)0x77f65819, (q31_t)0x77ed91c0, (q31_t)0x77e4c6c9, (q31_t)0x77dbf732, (q31_t)0x77d322fc, (q31_t)0x77ca4a27,
	(q31_t)0x77c16cb4, (q31_t)0x77b88aa3,
	(q31_t)0x77afa3f5, (q31_t)0x77a6b8a9, (q31_t)0x779dc8c0, (q31_t)0x7794d43b, (q31_t)0x778bdb19, (q31_t)0x7782dd5c,
	(q31_t)0x7779db03, (q31_t)0x7770d40f,
	(q31_t)0x7767c880, (q31_t)0x775eb857, (q31_t)0x7755a394, (q31_t)0x774c8a36, (q31_t)0x77436c40, (q31_t)0x773a49b0,
	(q31_t)0x77312287, (q31_t)0x7727f6c6,
	(q31_t)0x771ec66e, (q31_t)0x7715917d, (q31_t)0x770c57f5, (q31_t)0x770319d6, (q31_t)0x76f9d721, (q31_t)0x76f08fd5,
	(q31_t)0x76e743f4, (q31_t)0x76ddf37c,
	(q31_t)0x76d49e70, (q31_t)0x76cb44cf, (q31_t)0x76c1e699, (q31_t)0x76b883d0, (q31_t)0x76af1c72, (q31_t)0x76a5b082,
	(q31_t)0x769c3ffe, (q31_t)0x7692cae8,
	(q31_t)0x7689513f, (q31_t)0x767fd304, (q31_t)0x76765038, (q31_t)0x766cc8db, (q31_t)0x76633ced, (q31_t)0x7659ac6f,
	(q31_t)0x76501760, (q31_t)0x76467dc2,
	(q31_t)0x763cdf94, (q31_t)0x76333cd8, (q31_t)0x7629958c, (q31_t)0x761fe9b3, (q31_t)0x7616394c, (q31_t)0x760c8457,
	(q31_t)0x7602cad5, (q31_t)0x75f90cc7,
	(q31_t)0x75ef4a2c, (q31_t)0x75e58305, (q31_t)0x75dbb753, (q31_t)0x75d1e715, (q31_t)0x75c8124d, (q31_t)0x75be38fa,
	(q31_t)0x75b45b1d, (q31_t)0x75aa78b6,
	(q31_t)0x75a091c6, (q31_t)0x7596a64d, (q31_t)0x758cb64c, (q31_t)0x7582c1c2, (q31_t)0x7578c8b0, (q31_t)0x756ecb18,
	(q31_t)0x7564c8f8, (q31_t)0x755ac251,
	(q31_t)0x7550b725, (q31_t)0x7546a772, (q31_t)0x753c933a, (q31_t)0x75327a7d, (q31_t)0x75285d3b, (q31_t)0x751e3b75,
	(q31_t)0x7514152b, (q31_t)0x7509ea5d,
	(q31_t)0x74ffbb0d, (q31_t)0x74f58739, (q31_t)0x74eb4ee3, (q31_t)0x74e1120c, (q31_t)0x74d6d0b2, (q31_t)0x74cc8ad8,
	(q31_t)0x74c2407d, (q31_t)0x74b7f1a1,
	(q31_t)0x74ad9e46, (q31_t)0x74a3466b, (q31_t)0x7498ea11, (q31_t)0x748e8938, (q31_t)0x748423e0, (q31_t)0x7479ba0b,
	(q31_t)0x746f4bb8, (q31_t)0x7464d8e8,
	(q31_t)0x745a619b, (q31_t)0x744fe5d2, (q31_t)0x7445658d, (q31_t)0x743ae0cc, (q31_t)0x74305790, (q31_t)0x7425c9da,
	(q31_t)0x741b37a9, (q31_t)0x7410a0fe,
	(q31_t)0x740605d9, (q31_t)0x73fb663c, (q31_t)0x73f0c226, (q31_t)0x73e61997, (q31_t)0x73db6c91, (q31_t)0x73d0bb13,
	(q31_t)0x73c6051f, (q31_t)0x73bb4ab3,
	(q31_t)0x73b08bd1, (q31_t)0x73a5c87a, (q31_t)0x739b00ad, (q31_t)0x7390346b, (q31_t)0x738563b5, (q31_t)0x737a8e8a,
	(q31_t)0x736fb4ec, (q31_t)0x7364d6da,
	(q31_t)0x7359f456, (q31_t)0x734f0d5f, (q31_t)0x734421f6, (q31_t)0x7339321b, (q31_t)0x732e3dcf, (q31_t)0x73234512,
	(q31_t)0x731847e5, (q31_t)0x730d4648,
	(q31_t)0x7302403c, (q31_t)0x72f735c0, (q31_t)0x72ec26d6, (q31_t)0x72e1137d, (q31_t)0x72d5fbb7, (q31_t)0x72cadf83,
	(q31_t)0x72bfbee3, (q31_t)0x72b499d6,
	(q31_t)0x72a9705c, (q31_t)0x729e4277, (q31_t)0x72931027, (q31_t)0x7287d96c, (q31_t)0x727c9e47, (q31_t)0x72715eb8,
	(q31_t)0x72661abf, (q31_t)0x725ad25d,
	(q31_t)0x724f8593, (q31_t)0x72443460, (q31_t)0x7238dec5, (q31_t)0x722d84c4, (q31_t)0x7222265b, (q31_t)0x7216c38c,
	(q31_t)0x720b5c57, (q31_t)0x71fff0bc,
	(q31_t)0x71f480bc, (q31_t)0x71e90c57, (q31_t)0x71dd938f, (q31_t)0x71d21662, (q31_t)0x71c694d2, (q31_t)0x71bb0edf,
	(q31_t)0x71af848a, (q31_t)0x71a3f5d2,
	(q31_t)0x719862b9, (q31_t)0x718ccb3f, (q31_t)0x71812f65, (q31_t)0x71758f29, (q31_t)0x7169ea8f, (q31_t)0x715e4194,
	(q31_t)0x7152943b, (q31_t)0x7146e284,
	(q31_t)0x713b2c6e, (q31_t)0x712f71fb, (q31_t)0x7123b32b, (q31_t)0x7117effe, (q31_t)0x710c2875, (q31_t)0x71005c90,
	(q31_t)0x70f48c50, (q31_t)0x70e8b7b5,
	(q31_t)0x70dcdec0, (q31_t)0x70d10171, (q31_t)0x70c51fc8, (q31_t)0x70b939c7, (q31_t)0x70ad4f6d, (q31_t)0x70a160ba,
	(q31_t)0x70956db1, (q31_t)0x70897650,
	(q31_t)0x707d7a98, (q31_t)0x70717a8a, (q31_t)0x70657626, (q31_t)0x70596d6d, (q31_t)0x704d6060, (q31_t)0x70414efd,
	(q31_t)0x70353947, (q31_t)0x70291f3e,
	(q31_t)0x701d00e1, (q31_t)0x7010de32, (q31_t)0x7004b731, (q31_t)0x6ff88bde, (q31_t)0x6fec5c3b, (q31_t)0x6fe02846,
	(q31_t)0x6fd3f001, (q31_t)0x6fc7b36d,
	(q31_t)0x6fbb728a, (q31_t)0x6faf2d57, (q31_t)0x6fa2e3d7, (q31_t)0x6f969608, (q31_t)0x6f8a43ed, (q31_t)0x6f7ded84,
	(q31_t)0x6f7192cf, (q31_t)0x6f6533ce,
	(q31_t)0x6f58d082, (q31_t)0x6f4c68eb, (q31_t)0x6f3ffd09, (q31_t)0x6f338cde, (q31_t)0x6f271868, (q31_t)0x6f1a9faa,
	(q31_t)0x6f0e22a3, (q31_t)0x6f01a155,
	(q31_t)0x6ef51bbe, (q31_t)0x6ee891e1, (q31_t)0x6edc03bc, (q31_t)0x6ecf7152, (q31_t)0x6ec2daa2, (q31_t)0x6eb63fad,
	(q31_t)0x6ea9a073, (q31_t)0x6e9cfcf5,
	(q31_t)0x6e905534, (q31_t)0x6e83a92f, (q31_t)0x6e76f8e7, (q31_t)0x6e6a445d, (q31_t)0x6e5d8b91, (q31_t)0x6e50ce84,
	(q31_t)0x6e440d37, (q31_t)0x6e3747a9,
	(q31_t)0x6e2a7ddb, (q31_t)0x6e1dafce, (q31_t)0x6e10dd82, (q31_t)0x6e0406f8, (q31_t)0x6df72c30, (q31_t)0x6dea4d2b,
	(q31_t)0x6ddd69e9, (q31_t)0x6dd0826a,
	(q31_t)0x6dc396b0, (q31_t)0x6db6a6ba, (q31_t)0x6da9b28a, (q31_t)0x6d9cba1f, (q31_t)0x6d8fbd7a, (q31_t)0x6d82bc9d,
	(q31_t)0x6d75b786, (q31_t)0x6d68ae37,
	(q31_t)0x6d5ba0b0, (q31_t)0x6d4e8ef2, (q31_t)0x6d4178fd, (q31_t)0x6d345ed1, (q31_t)0x6d274070, (q31_t)0x6d1a1dda,
	(q31_t)0x6d0cf70f, (q31_t)0x6cffcc0f,
	(q31_t)0x6cf29cdc, (q31_t)0x6ce56975, (q31_t)0x6cd831dc, (q31_t)0x6ccaf610, (q31_t)0x6cbdb613, (q31_t)0x6cb071e4,
	(q31_t)0x6ca32985, (q31_t)0x6c95dcf6,
	(q31_t)0x6c888c36, (q31_t)0x6c7b3748, (q31_t)0x6c6dde2b, (q31_t)0x6c6080e0, (q31_t)0x6c531f67, (q31_t)0x6c45b9c1,
	(q31_t)0x6c384fef, (q31_t)0x6c2ae1f0,
	(q31_t)0x6c1d6fc6, (q31_t)0x6c0ff971, (q31_t)0x6c027ef1, (q31_t)0x6bf50047, (q31_t)0x6be77d74, (q31_t)0x6bd9f677,
	(q31_t)0x6bcc6b53, (q31_t)0x6bbedc06,
	(q31_t)0x6bb14892, (q31_t)0x6ba3b0f7, (q31_t)0x6b961536, (q31_t)0x6b88754f, (q31_t)0x6b7ad142, (q31_t)0x6b6d2911,
	(q31_t)0x6b5f7cbc, (q31_t)0x6b51cc42,
	(q31_t)0x6b4417a6, (q31_t)0x6b365ee7, (q31_t)0x6b28a206, (q31_t)0x6b1ae103, (q31_t)0x6b0d1bdf, (q31_t)0x6aff529a,
	(q31_t)0x6af18536, (q31_t)0x6ae3b3b2,
	(q31_t)0x6ad5de0f, (q31_t)0x6ac8044e, (q31_t)0x6aba266e, (q31_t)0x6aac4472, (q31_t)0x6a9e5e58, (q31_t)0x6a907423,
	(q31_t)0x6a8285d1, (q31_t)0x6a749365,
	(q31_t)0x6a669cdd, (q31_t)0x6a58a23c, (q31_t)0x6a4aa381, (q31_t)0x6a3ca0ad, (q31_t)0x6a2e99c0, (q31_t)0x6a208ebb,
	(q31_t)0x6a127f9f, (q31_t)0x6a046c6c,
	(q31_t)0x69f65523, (q31_t)0x69e839c4, (q31_t)0x69da1a50, (q31_t)0x69cbf6c7, (q31_t)0x69bdcf29, (q31_t)0x69afa378,
	(q31_t)0x69a173b5, (q31_t)0x69933fde,
	(q31_t)0x698507f6, (q31_t)0x6976cbfc, (q31_t)0x69688bf1, (q31_t)0x695a47d6, (q31_t)0x694bffab, (q31_t)0x693db371,
	(q31_t)0x692f6328, (q31_t)0x69210ed1,
	(q31_t)0x6912b66c, (q31_t)0x690459fb, (q31_t)0x68f5f97d, (q31_t)0x68e794f3, (q31_t)0x68d92c5d, (q31_t)0x68cabfbd,
	(q31_t)0x68bc4f13, (q31_t)0x68adda5f,
	(q31_t)0x689f61a1, (q31_t)0x6890e4dc, (q31_t)0x6882640e, (q31_t)0x6873df38, (q31_t)0x6865565c, (q31_t)0x6856c979,
	(q31_t)0x68483891, (q31_t)0x6839a3a4,
	(q31_t)0x682b0ab1, (q31_t)0x681c6dbb, (q31_t)0x680dccc1, (q31_t)0x67ff27c4, (q31_t)0x67f07ec5, (q31_t)0x67e1d1c4,
	(q31_t)0x67d320c1, (q31_t)0x67c46bbe,
	(q31_t)0x67b5b2bb, (q31_t)0x67a6f5b8, (q31_t)0x679834b6, (q31_t)0x67896fb6, (q31_t)0x677aa6b8, (q31_t)0x676bd9bd,
	(q31_t)0x675d08c4, (q31_t)0x674e33d0,
	(q31_t)0x673f5ae0, (q31_t)0x67307df5, (q31_t)0x67219d10, (q31_t)0x6712b831, (q31_t)0x6703cf58, (q31_t)0x66f4e287,
	(q31_t)0x66e5f1be, (q31_t)0x66d6fcfd,
	(q31_t)0x66c80445, (q31_t)0x66b90797, (q31_t)0x66aa06f3, (q31_t)0x669b0259, (q31_t)0x668bf9cb, (q31_t)0x667ced49,
	(q31_t)0x666ddcd3, (q31_t)0x665ec86b,
	(q31_t)0x664fb010, (q31_t)0x664093c3, (q31_t)0x66317385, (q31_t)0x66224f56, (q31_t)0x66132738, (q31_t)0x6603fb2a,
	(q31_t)0x65f4cb2d, (q31_t)0x65e59742,
	(q31_t)0x65d65f69, (q31_t)0x65c723a3, (q31_t)0x65b7e3f1, (q31_t)0x65a8a052, (q31_t)0x659958c9, (q31_t)0x658a0d54,
	(q31_t)0x657abdf6, (q31_t)0x656b6aae,
	(q31_t)0x655c137d, (q31_t)0x654cb863, (q31_t)0x653d5962, (q31_t)0x652df679, (q31_t)0x651e8faa, (q31_t)0x650f24f5,
	(q31_t)0x64ffb65b, (q31_t)0x64f043dc,
	(q31_t)0x64e0cd78, (q31_t)0x64d15331, (q31_t)0x64c1d507, (q31_t)0x64b252fa, (q31_t)0x64a2cd0c, (q31_t)0x6493433c,
	(q31_t)0x6483b58c, (q31_t)0x647423fb,
	(q31_t)0x64648e8c, (q31_t)0x6454f53d, (q31_t)0x64455810, (q31_t)0x6435b706, (q31_t)0x6426121e, (q31_t)0x6416695a,
	(q31_t)0x6406bcba, (q31_t)0x63f70c3f,
	(q31_t)0x63e757ea, (q31_t)0x63d79fba, (q31_t)0x63c7e3b1, (q31_t)0x63b823cf, (q31_t)0x63a86015, (q31_t)0x63989884,
	(q31_t)0x6388cd1b, (q31_t)0x6378fddc,
	(q31_t)0x63692ac7, (q31_t)0x635953dd, (q31_t)0x6349791f, (q31_t)0x63399a8d, (q31_t)0x6329b827, (q31_t)0x6319d1ef,
	(q31_t)0x6309e7e4, (q31_t)0x62f9fa09,
	(q31_t)0x62ea085c, (q31_t)0x62da12df, (q31_t)0x62ca1992, (q31_t)0x62ba1c77, (q31_t)0x62aa1b8d, (q31_t)0x629a16d5,
	(q31_t)0x628a0e50, (q31_t)0x627a01fe,
	(q31_t)0x6269f1e1, (q31_t)0x6259ddf8, (q31_t)0x6249c645, (q31_t)0x6239aac7, (q31_t)0x62298b81, (q31_t)0x62196871,
	(q31_t)0x62094199, (q31_t)0x61f916f9,
	(q31_t)0x61e8e893, (q31_t)0x61d8b666, (q31_t)0x61c88074, (q31_t)0x61b846bc, (q31_t)0x61a80940, (q31_t)0x6197c800,
	(q31_t)0x618782fd, (q31_t)0x61773a37,
	(q31_t)0x6166edb0, (q31_t)0x61569d67, (q31_t)0x6146495d, (q31_t)0x6135f193, (q31_t)0x6125960a, (q31_t)0x611536c2,
	(q31_t)0x6104d3bc, (q31_t)0x60f46cf9,
	(q31_t)0x60e40278, (q31_t)0x60d3943b, (q31_t)0x60c32243, (q31_t)0x60b2ac8f, (q31_t)0x60a23322, (q31_t)0x6091b5fa,
	(q31_t)0x60813519, (q31_t)0x6070b080,
	(q31_t)0x6060282f, (q31_t)0x604f9c27, (q31_t)0x603f0c69, (q31_t)0x602e78f4, (q31_t)0x601de1ca, (q31_t)0x600d46ec,
	(q31_t)0x5ffca859, (q31_t)0x5fec0613,
	(q31_t)0x5fdb601b, (q31_t)0x5fcab670, (q31_t)0x5fba0914, (q31_t)0x5fa95807, (q31_t)0x5f98a34a, (q31_t)0x5f87eade,
	(q31_t)0x5f772ec2, (q31_t)0x5f666ef9,
	(q31_t)0x5f55ab82, (q31_t)0x5f44e45e, (q31_t)0x5f34198e, (q31_t)0x5f234b12, (q31_t)0x5f1278eb, (q31_t)0x5f01a31a,
	(q31_t)0x5ef0c99f, (q31_t)0x5edfec7b,
	(q31_t)0x5ecf0baf, (q31_t)0x5ebe273b, (q31_t)0x5ead3f1f, (q31_t)0x5e9c535e, (q31_t)0x5e8b63f7, (q31_t)0x5e7a70ea,
	(q31_t)0x5e697a39, (q31_t)0x5e587fe5,
	(q31_t)0x5e4781ed, (q31_t)0x5e368053, (q31_t)0x5e257b17, (q31_t)0x5e147239, (q31_t)0x5e0365bb, (q31_t)0x5df2559e,
	(q31_t)0x5de141e1, (q31_t)0x5dd02a85,
	(q31_t)0x5dbf0f8c, (q31_t)0x5dadf0f5, (q31_t)0x5d9ccec2, (q31_t)0x5d8ba8f3, (q31_t)0x5d7a7f88, (q31_t)0x5d695283,
	(q31_t)0x5d5821e4, (q31_t)0x5d46edac,
	(q31_t)0x5d35b5db, (q31_t)0x5d247a72, (q31_t)0x5d133b72, (q31_t)0x5d01f8dc, (q31_t)0x5cf0b2af, (q31_t)0x5cdf68ed,
	(q31_t)0x5cce1b97, (q31_t)0x5cbccaac,
	(q31_t)0x5cab762f, (q31_t)0x5c9a1e1e, (q31_t)0x5c88c27c, (q31_t)0x5c776348, (q31_t)0x5c660084, (q31_t)0x5c549a30,
	(q31_t)0x5c43304d, (q31_t)0x5c31c2db,
	(q31_t)0x5c2051db, (q31_t)0x5c0edd4e, (q31_t)0x5bfd6534, (q31_t)0x5bebe98e, (q31_t)0x5bda6a5d, (q31_t)0x5bc8e7a2,
	(q31_t)0x5bb7615d, (q31_t)0x5ba5d78e,
	(q31_t)0x5b944a37, (q31_t)0x5b82b958, (q31_t)0x5b7124f2, (q31_t)0x5b5f8d06, (q31_t)0x5b4df193, (q31_t)0x5b3c529c,
	(q31_t)0x5b2ab020, (q31_t)0x5b190a20,
	(q31_t)0x5b07609d, (q31_t)0x5af5b398, (q31_t)0x5ae40311, (q31_t)0x5ad24f09, (q31_t)0x5ac09781, (q31_t)0x5aaedc78,
	(q31_t)0x5a9d1df1, (q31_t)0x5a8b5bec,
	(q31_t)0x5a799669, (q31_t)0x5a67cd69, (q31_t)0x5a5600ec, (q31_t)0x5a4430f5, (q31_t)0x5a325d82, (q31_t)0x5a208695,
	(q31_t)0x5a0eac2e, (q31_t)0x59fcce4f,
	(q31_t)0x59eaecf8, (q31_t)0x59d90829, (q31_t)0x59c71fe3, (q31_t)0x59b53427, (q31_t)0x59a344f6, (q31_t)0x59915250,
	(q31_t)0x597f5c36, (q31_t)0x596d62a9,
	(q31_t)0x595b65aa, (q31_t)0x59496538, (q31_t)0x59376155, (q31_t)0x59255a02, (q31_t)0x59134f3e, (q31_t)0x5901410c,
	(q31_t)0x58ef2f6b, (q31_t)0x58dd1a5d,
	(q31_t)0x58cb01e1, (q31_t)0x58b8e5f9, (q31_t)0x58a6c6a5, (q31_t)0x5894a3e7, (q31_t)0x58827dbe, (q31_t)0x5870542c,
	(q31_t)0x585e2730, (q31_t)0x584bf6cd,
	(q31_t)0x5839c302, (q31_t)0x58278bd1, (q31_t)0x58155139, (q31_t)0x5803133c, (q31_t)0x57f0d1da, (q31_t)0x57de8d15,
	(q31_t)0x57cc44ec, (q31_t)0x57b9f960,
	(q31_t)0x57a7aa73, (q31_t)0x57955825, (q31_t)0x57830276, (q31_t)0x5770a968, (q31_t)0x575e4cfa, (q31_t)0x574bed2f,
	(q31_t)0x57398a05, (q31_t)0x5727237f,
	(q31_t)0x5714b99d, (q31_t)0x57024c5f, (q31_t)0x56efdbc7, (q31_t)0x56dd67d4, (q31_t)0x56caf088, (q31_t)0x56b875e4,
	(q31_t)0x56a5f7e7, (q31_t)0x56937694,
	(q31_t)0x5680f1ea, (q31_t)0x566e69ea, (q31_t)0x565bde95, (q31_t)0x56494fec, (q31_t)0x5636bdef, (q31_t)0x5624289f,
	(q31_t)0x56118ffe, (q31_t)0x55fef40a,
	(q31_t)0x55ec54c6, (q31_t)0x55d9b232, (q31_t)0x55c70c4f, (q31_t)0x55b4631d, (q31_t)0x55a1b69d, (q31_t)0x558f06d0,
	(q31_t)0x557c53b6, (q31_t)0x55699d51,
	(q31_t)0x5556e3a1, (q31_t)0x554426a7, (q31_t)0x55316663, (q31_t)0x551ea2d6, (q31_t)0x550bdc01, (q31_t)0x54f911e5,
	(q31_t)0x54e64482, (q31_t)0x54d373d9,
	(q31_t)0x54c09feb, (q31_t)0x54adc8b8, (q31_t)0x549aee42, (q31_t)0x54881089, (q31_t)0x54752f8d, (q31_t)0x54624b50,
	(q31_t)0x544f63d2, (q31_t)0x543c7914,
	(q31_t)0x54298b17, (q31_t)0x541699db, (q31_t)0x5403a561, (q31_t)0x53f0adaa, (q31_t)0x53ddb2b6, (q31_t)0x53cab486,
	(q31_t)0x53b7b31c, (q31_t)0x53a4ae77,
	(q31_t)0x5391a699, (q31_t)0x537e9b82, (q31_t)0x536b8d33, (q31_t)0x53587bad, (q31_t)0x534566f0, (q31_t)0x53324efd,
	(q31_t)0x531f33d5, (q31_t)0x530c1579,
	(q31_t)0x52f8f3e9, (q31_t)0x52e5cf27, (q31_t)0x52d2a732, (q31_t)0x52bf7c0b, (q31_t)0x52ac4db4, (q31_t)0x52991c2d,
	(q31_t)0x5285e777, (q31_t)0x5272af92,
	(q31_t)0x525f7480, (q31_t)0x524c3640, (q31_t)0x5238f4d4, (q31_t)0x5225b03d, (q31_t)0x5212687b, (q31_t)0x51ff1d8f,
	(q31_t)0x51ebcf7a, (q31_t)0x51d87e3c,
	(q31_t)0x51c529d7, (q31_t)0x51b1d24a, (q31_t)0x519e7797, (q31_t)0x518b19bf, (q31_t)0x5177b8c2, (q31_t)0x516454a0,
	(q31_t)0x5150ed5c, (q31_t)0x513d82f4,
	(q31_t)0x512a156b, (q31_t)0x5116a4c1, (q31_t)0x510330f7, (q31_t)0x50efba0d, (q31_t)0x50dc4005, (q31_t)0x50c8c2de,
	(q31_t)0x50b5429a, (q31_t)0x50a1bf39,
	(q31_t)0x508e38bd, (q31_t)0x507aaf25, (q31_t)0x50672273, (q31_t)0x505392a8, (q31_t)0x503fffc4, (q31_t)0x502c69c8,
	(q31_t)0x5018d0b4, (q31_t)0x5005348a,
	(q31_t)0x4ff1954b, (q31_t)0x4fddf2f6, (q31_t)0x4fca4d8d, (q31_t)0x4fb6a510, (q31_t)0x4fa2f981, (q31_t)0x4f8f4ae0,
	(q31_t)0x4f7b992d, (q31_t)0x4f67e46a,
	(q31_t)0x4f542c98, (q31_t)0x4f4071b6, (q31_t)0x4f2cb3c7, (q31_t)0x4f18f2c9, (q31_t)0x4f052ec0, (q31_t)0x4ef167aa,
	(q31_t)0x4edd9d89, (q31_t)0x4ec9d05e,
	(q31_t)0x4eb60029, (q31_t)0x4ea22ceb, (q31_t)0x4e8e56a5, (q31_t)0x4e7a7d58, (q31_t)0x4e66a105, (q31_t)0x4e52c1ab,
	(q31_t)0x4e3edf4d, (q31_t)0x4e2af9ea,
	(q31_t)0x4e171184, (q31_t)0x4e03261b, (q31_t)0x4def37b0, (q31_t)0x4ddb4644, (q31_t)0x4dc751d8, (q31_t)0x4db35a6c,
	(q31_t)0x4d9f6001, (q31_t)0x4d8b6298,
	(q31_t)0x4d776231, (q31_t)0x4d635ece, (q31_t)0x4d4f5870, (q31_t)0x4d3b4f16, (q31_t)0x4d2742c2, (q31_t)0x4d133374,
	(q31_t)0x4cff212e, (q31_t)0x4ceb0bf0,
	(q31_t)0x4cd6f3bb, (q31_t)0x4cc2d88f, (q31_t)0x4caeba6e, (q31_t)0x4c9a9958, (q31_t)0x4c86754e, (q31_t)0x4c724e50,
	(q31_t)0x4c5e2460, (q31_t)0x4c49f77f,
	(q31_t)0x4c35c7ac, (q31_t)0x4c2194e9, (q31_t)0x4c0d5f37, (q31_t)0x4bf92697, (q31_t)0x4be4eb08, (q31_t)0x4bd0ac8d,
	(q31_t)0x4bbc6b25, (q31_t)0x4ba826d1,
	(q31_t)0x4b93df93, (q31_t)0x4b7f956b, (q31_t)0x4b6b485a, (q31_t)0x4b56f861, (q31_t)0x4b42a580, (q31_t)0x4b2e4fb8,
	(q31_t)0x4b19f70a, (q31_t)0x4b059b77,
	(q31_t)0x4af13d00, (q31_t)0x4adcdba5, (q31_t)0x4ac87767, (q31_t)0x4ab41046, (q31_t)0x4a9fa645, (q31_t)0x4a8b3963,
	(q31_t)0x4a76c9a2, (q31_t)0x4a625701,
	(q31_t)0x4a4de182, (q31_t)0x4a396926, (q31_t)0x4a24edee, (q31_t)0x4a106fda, (q31_t)0x49fbeeea, (q31_t)0x49e76b21,
	(q31_t)0x49d2e47e, (q31_t)0x49be5b02,
	(q31_t)0x49a9ceaf, (q31_t)0x49953f84, (q31_t)0x4980ad84, (q31_t)0x496c18ae, (q31_t)0x49578103, (q31_t)0x4942e684,
	(q31_t)0x492e4933, (q31_t)0x4919a90f,
	(q31_t)0x4905061a, (q31_t)0x48f06054, (q31_t)0x48dbb7be, (q31_t)0x48c70c59, (q31_t)0x48b25e25, (q31_t)0x489dad25,
	(q31_t)0x4888f957, (q31_t)0x487442be,
	(q31_t)0x485f8959, (q31_t)0x484acd2a, (q31_t)0x48360e32, (q31_t)0x48214c71, (q31_t)0x480c87e8, (q31_t)0x47f7c099,
	(q31_t)0x47e2f682, (q31_t)0x47ce29a7,
	(q31_t)0x47b95a06, (q31_t)0x47a487a2, (q31_t)0x478fb27b, (q31_t)0x477ada91, (q31_t)0x4765ffe6, (q31_t)0x4751227a,
	(q31_t)0x473c424e, (q31_t)0x47275f63,
	(q31_t)0x471279ba, (q31_t)0x46fd9154, (q31_t)0x46e8a631, (q31_t)0x46d3b852, (q31_t)0x46bec7b8, (q31_t)0x46a9d464,
	(q31_t)0x4694de56, (q31_t)0x467fe590,
	(q31_t)0x466aea12, (q31_t)0x4655ebdd, (q31_t)0x4640eaf2, (q31_t)0x462be751, (q31_t)0x4616e0fc, (q31_t)0x4601d7f3,
	(q31_t)0x45eccc37, (q31_t)0x45d7bdc9,
	(q31_t)0x45c2acaa, (q31_t)0x45ad98da, (q31_t)0x4598825a, (q31_t)0x4583692c, (q31_t)0x456e4d4f, (q31_t)0x45592ec6,
	(q31_t)0x45440d90, (q31_t)0x452ee9ae,
	(q31_t)0x4519c321, (q31_t)0x450499eb, (q31_t)0x44ef6e0b, (q31_t)0x44da3f83, (q31_t)0x44c50e53, (q31_t)0x44afda7d,
	(q31_t)0x449aa400, (q31_t)0x44856adf,
	(q31_t)0x44702f19, (q31_t)0x445af0b0, (q31_t)0x4445afa4, (q31_t)0x44306bf6, (q31_t)0x441b25a8, (q31_t)0x4405dcb9,
	(q31_t)0x43f0912b, (q31_t)0x43db42fe,
	(q31_t)0x43c5f234, (q31_t)0x43b09ecc, (q31_t)0x439b48c9, (q31_t)0x4385f02a, (q31_t)0x437094f1, (q31_t)0x435b371f,
	(q31_t)0x4345d6b3, (q31_t)0x433073b0,
	(q31_t)0x431b0e15, (q31_t)0x4305a5e5, (q31_t)0x42f03b1e, (q31_t)0x42dacdc3, (q31_t)0x42c55dd4, (q31_t)0x42afeb53,
	(q31_t)0x429a763f, (q31_t)0x4284fe99,
	(q31_t)0x426f8463, (q31_t)0x425a079e, (q31_t)0x42448849, (q31_t)0x422f0667, (q31_t)0x421981f7, (q31_t)0x4203fafb,
	(q31_t)0x41ee7174, (q31_t)0x41d8e561,
	(q31_t)0x41c356c5, (q31_t)0x41adc5a0, (q31_t)0x419831f3, (q31_t)0x41829bbe, (q31_t)0x416d0302, (q31_t)0x415767c1,
	(q31_t)0x4141c9fb, (q31_t)0x412c29b1,
	(q31_t)0x411686e4, (q31_t)0x4100e194, (q31_t)0x40eb39c3, (q31_t)0x40d58f71, (q31_t)0x40bfe29f, (q31_t)0x40aa334e,
	(q31_t)0x4094817f, (q31_t)0x407ecd32,
	(q31_t)0x40691669, (q31_t)0x40535d24, (q31_t)0x403da165, (q31_t)0x4027e32b, (q31_t)0x40122278, (q31_t)0x3ffc5f4d,
	(q31_t)0x3fe699aa, (q31_t)0x3fd0d191,
	(q31_t)0x3fbb0702, (q31_t)0x3fa539fd, (q31_t)0x3f8f6a85, (q31_t)0x3f799899, (q31_t)0x3f63c43b, (q31_t)0x3f4ded6b,
	(q31_t)0x3f38142a, (q31_t)0x3f22387a,
	(q31_t)0x3f0c5a5a, (q31_t)0x3ef679cc, (q31_t)0x3ee096d1, (q31_t)0x3ecab169, (q31_t)0x3eb4c995, (q31_t)0x3e9edf57,
	(q31_t)0x3e88f2ae, (q31_t)0x3e73039d,
	(q31_t)0x3e5d1222, (q31_t)0x3e471e41, (q31_t)0x3e3127f9, (q31_t)0x3e1b2f4a, (q31_t)0x3e053437, (q31_t)0x3def36c0,
	(q31_t)0x3dd936e6, (q31_t)0x3dc334a9,
	(q31_t)0x3dad300b, (q31_t)0x3d97290b, (q31_t)0x3d811fac, (q31_t)0x3d6b13ee, (q31_t)0x3d5505d2, (q31_t)0x3d3ef559,
	(q31_t)0x3d28e282, (q31_t)0x3d12cd51,
	(q31_t)0x3cfcb5c4, (q31_t)0x3ce69bde, (q31_t)0x3cd07f9f, (q31_t)0x3cba6107, (q31_t)0x3ca44018, (q31_t)0x3c8e1cd3,
	(q31_t)0x3c77f737, (q31_t)0x3c61cf48,
	(q31_t)0x3c4ba504, (q31_t)0x3c35786d, (q31_t)0x3c1f4983, (q31_t)0x3c091849, (q31_t)0x3bf2e4be, (q31_t)0x3bdcaee3,
	(q31_t)0x3bc676b9, (q31_t)0x3bb03c42,
	(q31_t)0x3b99ff7d, (q31_t)0x3b83c06c, (q31_t)0x3b6d7f10, (q31_t)0x3b573b69, (q31_t)0x3b40f579, (q31_t)0x3b2aad3f,
	(q31_t)0x3b1462be, (q31_t)0x3afe15f6,
	(q31_t)0x3ae7c6e7, (q31_t)0x3ad17593, (q31_t)0x3abb21fb, (q31_t)0x3aa4cc1e, (q31_t)0x3a8e7400, (q31_t)0x3a78199f,
	(q31_t)0x3a61bcfd, (q31_t)0x3a4b5e1b,
	(q31_t)0x3a34fcf9, (q31_t)0x3a1e9999, (q31_t)0x3a0833fc, (q31_t)0x39f1cc21, (q31_t)0x39db620b, (q31_t)0x39c4f5ba,
	(q31_t)0x39ae872f, (q31_t)0x3998166a,
	(q31_t)0x3981a36d, (q31_t)0x396b2e38, (q31_t)0x3954b6cd, (q31_t)0x393e3d2c, (q31_t)0x3927c155, (q31_t)0x3911434b,
	(q31_t)0x38fac30e, (q31_t)0x38e4409e,
	(q31_t)0x38cdbbfc, (q31_t)0x38b7352a, (q31_t)0x38a0ac29, (q31_t)0x388a20f8, (q31_t)0x38739399, (q31_t)0x385d040d,
	(q31_t)0x38467255, (q31_t)0x382fde72,
	(q31_t)0x38194864, (q31_t)0x3802b02c, (q31_t)0x37ec15cb, (q31_t)0x37d57943, (q31_t)0x37beda93, (q31_t)0x37a839be,
	(q31_t)0x379196c3, (q31_t)0x377af1a3,
	(q31_t)0x37644a60, (q31_t)0x374da0fa, (q31_t)0x3736f573, (q31_t)0x372047ca, (q31_t)0x37099802, (q31_t)0x36f2e61a,
	(q31_t)0x36dc3214, (q31_t)0x36c57bf0,
	(q31_t)0x36aec3b0, (q31_t)0x36980954, (q31_t)0x36814cde, (q31_t)0x366a8e4d, (q31_t)0x3653cda3, (q31_t)0x363d0ae2,
	(q31_t)0x36264609, (q31_t)0x360f7f19,
	(q31_t)0x35f8b614, (q31_t)0x35e1eafa, (q31_t)0x35cb1dcc, (q31_t)0x35b44e8c, (q31_t)0x359d7d39, (q31_t)0x3586a9d5,
	(q31_t)0x356fd461, (q31_t)0x3558fcde,
	(q31_t)0x3542234c, (q31_t)0x352b47ad, (q31_t)0x35146a00, (q31_t)0x34fd8a48, (q31_t)0x34e6a885, (q31_t)0x34cfc4b7,
	(q31_t)0x34b8dee1, (q31_t)0x34a1f702,
	(q31_t)0x348b0d1c, (q31_t)0x3474212f, (q31_t)0x345d333c, (q31_t)0x34464345, (q31_t)0x342f5149, (q31_t)0x34185d4b,
	(q31_t)0x3401674a, (q31_t)0x33ea6f48,
	(q31_t)0x33d37546, (q31_t)0x33bc7944, (q31_t)0x33a57b44, (q31_t)0x338e7b46, (q31_t)0x3377794b, (q31_t)0x33607554,
	(q31_t)0x33496f62, (q31_t)0x33326776,
	(q31_t)0x331b5d91, (q31_t)0x330451b3, (q31_t)0x32ed43de, (q31_t)0x32d63412, (q31_t)0x32bf2250, (q31_t)0x32a80e99,
	(q31_t)0x3290f8ef, (q31_t)0x3279e151,
	(q31_t)0x3262c7c1, (q31_t)0x324bac40, (q31_t)0x32348ecf, (q31_t)0x321d6f6e, (q31_t)0x32064e1e, (q31_t)0x31ef2ae1,
	(q31_t)0x31d805b7, (q31_t)0x31c0dea1,
	(q31_t)0x31a9b5a0, (q31_t)0x31928ab4, (q31_t)0x317b5de0, (q31_t)0x31642f23, (q31_t)0x314cfe7f, (q31_t)0x3135cbf4,
	(q31_t)0x311e9783, (q31_t)0x3107612e,
	(q31_t)0x30f028f4, (q31_t)0x30d8eed8, (q31_t)0x30c1b2da, (q31_t)0x30aa74fa, (q31_t)0x3093353a, (q31_t)0x307bf39b,
	(q31_t)0x3064b01d, (q31_t)0x304d6ac1,
	(q31_t)0x30362389, (q31_t)0x301eda75, (q31_t)0x30078f86, (q31_t)0x2ff042bd, (q31_t)0x2fd8f41b, (q31_t)0x2fc1a3a0,
	(q31_t)0x2faa514f, (q31_t)0x2f92fd26,
	(q31_t)0x2f7ba729, (q31_t)0x2f644f56, (q31_t)0x2f4cf5b0, (q31_t)0x2f359a37, (q31_t)0x2f1e3ced, (q31_t)0x2f06ddd1,
	(q31_t)0x2eef7ce5, (q31_t)0x2ed81a29,
	(q31_t)0x2ec0b5a0, (q31_t)0x2ea94f49, (q31_t)0x2e91e725, (q31_t)0x2e7a7d36, (q31_t)0x2e63117c, (q31_t)0x2e4ba3f8,
	(q31_t)0x2e3434ac, (q31_t)0x2e1cc397,
	(q31_t)0x2e0550bb, (q31_t)0x2deddc19, (q31_t)0x2dd665b2, (q31_t)0x2dbeed86, (q31_t)0x2da77397, (q31_t)0x2d8ff7e5,
	(q31_t)0x2d787a72, (q31_t)0x2d60fb3e,
	(q31_t)0x2d497a4a, (q31_t)0x2d31f797, (q31_t)0x2d1a7325, (q31_t)0x2d02ecf7, (q31_t)0x2ceb650d, (q31_t)0x2cd3db67,
	(q31_t)0x2cbc5006, (q31_t)0x2ca4c2ed,
	(q31_t)0x2c8d341a, (q31_t)0x2c75a390, (q31_t)0x2c5e114f, (q31_t)0x2c467d58, (q31_t)0x2c2ee7ad, (q31_t)0x2c17504d,
	(q31_t)0x2bffb73a, (q31_t)0x2be81c74,
	(q31_t)0x2bd07ffe, (q31_t)0x2bb8e1d7, (q31_t)0x2ba14200, (q31_t)0x2b89a07b, (q31_t)0x2b71fd48, (q31_t)0x2b5a5868,
	(q31_t)0x2b42b1dd, (q31_t)0x2b2b09a6,
	(q31_t)0x2b135fc6, (q31_t)0x2afbb43c, (q31_t)0x2ae4070a, (q31_t)0x2acc5831, (q31_t)0x2ab4a7b1, (q31_t)0x2a9cf58c,
	(q31_t)0x2a8541c3, (q31_t)0x2a6d8c55,
	(q31_t)0x2a55d545, (q31_t)0x2a3e1c93, (q31_t)0x2a266240, (q31_t)0x2a0ea64d, (q31_t)0x29f6e8bb, (q31_t)0x29df298b,
	(q31_t)0x29c768be, (q31_t)0x29afa654,
	(q31_t)0x2997e24f, (q31_t)0x29801caf, (q31_t)0x29685576, (q31_t)0x29508ca4, (q31_t)0x2938c23a, (q31_t)0x2920f63a,
	(q31_t)0x290928a3, (q31_t)0x28f15978,
	(q31_t)0x28d988b8, (q31_t)0x28c1b666, (q31_t)0x28a9e281, (q31_t)0x28920d0a, (q31_t)0x287a3604, (q31_t)0x28625d6d,
	(q31_t)0x284a8349, (q31_t)0x2832a796,
	(q31_t)0x281aca57, (q31_t)0x2802eb8c, (q31_t)0x27eb0b36, (q31_t)0x27d32956, (q31_t)0x27bb45ed, (q31_t)0x27a360fc,
	(q31_t)0x278b7a84, (q31_t)0x27739285,
	(q31_t)0x275ba901, (q31_t)0x2743bdf9, (q31_t)0x272bd16d, (q31_t)0x2713e35f, (q31_t)0x26fbf3ce, (q31_t)0x26e402bd,
	(q31_t)0x26cc102d, (q31_t)0x26b41c1d,
	(q31_t)0x269c268f, (q31_t)0x26842f84, (q31_t)0x266c36fe, (q31_t)0x26543cfb, (q31_t)0x263c417f, (q31_t)0x26244489,
	(q31_t)0x260c461b, (q31_t)0x25f44635,
	(q31_t)0x25dc44d9, (q31_t)0x25c44207, (q31_t)0x25ac3dc0, (q31_t)0x25943806, (q31_t)0x257c30d8, (q31_t)0x25642839,
	(q31_t)0x254c1e28, (q31_t)0x253412a8,
	(q31_t)0x251c05b8, (q31_t)0x2503f75a, (q31_t)0x24ebe78f, (q31_t)0x24d3d657, (q31_t)0x24bbc3b4, (q31_t)0x24a3afa6,
	(q31_t)0x248b9a2f, (q31_t)0x2473834f,
	(q31_t)0x245b6b07, (q31_t)0x24435158, (q31_t)0x242b3644, (q31_t)0x241319ca, (q31_t)0x23fafbec, (q31_t)0x23e2dcac,
	(q31_t)0x23cabc09, (q31_t)0x23b29a05,
	(q31_t)0x239a76a0, (q31_t)0x238251dd, (q31_t)0x236a2bba, (q31_t)0x2352043b, (q31_t)0x2339db5e, (q31_t)0x2321b126,
	(q31_t)0x23098593, (q31_t)0x22f158a7,
	(q31_t)0x22d92a61, (q31_t)0x22c0fac4, (q31_t)0x22a8c9cf, (q31_t)0x22909785, (q31_t)0x227863e5, (q31_t)0x22602ef1,
	(q31_t)0x2247f8aa, (q31_t)0x222fc111,
	(q31_t)0x22178826, (q31_t)0x21ff4dea, (q31_t)0x21e71260, (q31_t)0x21ced586, (q31_t)0x21b6975f, (q31_t)0x219e57eb,
	(q31_t)0x2186172b, (q31_t)0x216dd521,
	(q31_t)0x215591cc, (q31_t)0x213d4d2f, (q31_t)0x21250749, (q31_t)0x210cc01d, (q31_t)0x20f477aa, (q31_t)0x20dc2df2,
	(q31_t)0x20c3e2f5, (q31_t)0x20ab96b5,
	(q31_t)0x20934933, (q31_t)0x207afa6f, (q31_t)0x2062aa6b, (q31_t)0x204a5927, (q31_t)0x203206a4, (q31_t)0x2019b2e4,
	(q31_t)0x20015de7, (q31_t)0x1fe907ae,
	(q31_t)0x1fd0b03a, (q31_t)0x1fb8578b, (q31_t)0x1f9ffda4, (q31_t)0x1f87a285, (q31_t)0x1f6f462f, (q31_t)0x1f56e8a2,
	(q31_t)0x1f3e89e0, (q31_t)0x1f2629ea,
	(q31_t)0x1f0dc8c0, (q31_t)0x1ef56664, (q31_t)0x1edd02d6, (q31_t)0x1ec49e17, (q31_t)0x1eac3829, (q31_t)0x1e93d10c,
	(q31_t)0x1e7b68c2, (q31_t)0x1e62ff4a,
	(q31_t)0x1e4a94a7, (q31_t)0x1e3228d9, (q31_t)0x1e19bbe0, (q31_t)0x1e014dbf, (q31_t)0x1de8de75, (q31_t)0x1dd06e04,
	(q31_t)0x1db7fc6d, (q31_t)0x1d9f89b1,
	(q31_t)0x1d8715d0, (q31_t)0x1d6ea0cc, (q31_t)0x1d562aa6, (q31_t)0x1d3db35e, (q31_t)0x1d253af5, (q31_t)0x1d0cc16c,
	(q31_t)0x1cf446c5, (q31_t)0x1cdbcb00,
	(q31_t)0x1cc34e1f, (q31_t)0x1caad021, (q31_t)0x1c925109, (q31_t)0x1c79d0d6, (q31_t)0x1c614f8b, (q31_t)0x1c48cd27,
	(q31_t)0x1c3049ac, (q31_t)0x1c17c51b,
	(q31_t)0x1bff3f75, (q31_t)0x1be6b8ba, (q31_t)0x1bce30ec, (q31_t)0x1bb5a80c, (q31_t)0x1b9d1e1a, (q31_t)0x1b849317,
	(q31_t)0x1b6c0705, (q31_t)0x1b5379e5,
	(q31_t)0x1b3aebb6, (q31_t)0x1b225c7b, (q31_t)0x1b09cc34, (q31_t)0x1af13ae3, (q31_t)0x1ad8a887, (q31_t)0x1ac01522,
	(q31_t)0x1aa780b6, (q31_t)0x1a8eeb42,
	(q31_t)0x1a7654c8, (q31_t)0x1a5dbd49, (q31_t)0x1a4524c6, (q31_t)0x1a2c8b3f, (q31_t)0x1a13f0b6, (q31_t)0x19fb552c,
	(q31_t)0x19e2b8a2, (q31_t)0x19ca1b17,
	(q31_t)0x19b17c8f, (q31_t)0x1998dd09, (q31_t)0x19803c86, (q31_t)0x19679b07, (q31_t)0x194ef88e, (q31_t)0x1936551b,
	(q31_t)0x191db0af, (q31_t)0x19050b4b,
	(q31_t)0x18ec64f0, (q31_t)0x18d3bda0, (q31_t)0x18bb155a, (q31_t)0x18a26c20, (q31_t)0x1889c1f3, (q31_t)0x187116d4,
	(q31_t)0x18586ac3, (q31_t)0x183fbdc3,
	(q31_t)0x18270fd3, (q31_t)0x180e60f4, (q31_t)0x17f5b129, (q31_t)0x17dd0070, (q31_t)0x17c44ecd, (q31_t)0x17ab9c3e,
	(q31_t)0x1792e8c6, (q31_t)0x177a3466,
	(q31_t)0x17617f1d, (q31_t)0x1748c8ee, (q31_t)0x173011d9, (q31_t)0x171759df, (q31_t)0x16fea102, (q31_t)0x16e5e741,
	(q31_t)0x16cd2c9f, (q31_t)0x16b4711b,
	(q31_t)0x169bb4b7, (q31_t)0x1682f774, (q31_t)0x166a3953, (q31_t)0x16517a55, (q31_t)0x1638ba7a, (q31_t)0x161ff9c4,
	(q31_t)0x16073834, (q31_t)0x15ee75cb,
	(q31_t)0x15d5b288, (q31_t)0x15bcee6f, (q31_t)0x15a4297f, (q31_t)0x158b63b9, (q31_t)0x15729d1f, (q31_t)0x1559d5b1,
	(q31_t)0x15410d70, (q31_t)0x1528445d,
	(q31_t)0x150f7a7a, (q31_t)0x14f6afc7, (q31_t)0x14dde445, (q31_t)0x14c517f4, (q31_t)0x14ac4ad7, (q31_t)0x14937cee,
	(q31_t)0x147aae3a, (q31_t)0x1461debc,
	(q31_t)0x14490e74, (q31_t)0x14303d65, (q31_t)0x14176b8e, (q31_t)0x13fe98f1, (q31_t)0x13e5c58e, (q31_t)0x13ccf167,
	(q31_t)0x13b41c7d, (q31_t)0x139b46d0,
	(q31_t)0x13827062, (q31_t)0x13699933, (q31_t)0x1350c144, (q31_t)0x1337e897, (q31_t)0x131f0f2c, (q31_t)0x13063505,
	(q31_t)0x12ed5a21, (q31_t)0x12d47e83,
	(q31_t)0x12bba22b, (q31_t)0x12a2c51b, (q31_t)0x1289e752, (q31_t)0x127108d2, (q31_t)0x1258299c, (q31_t)0x123f49b2,
	(q31_t)0x12266913, (q31_t)0x120d87c1,
	(q31_t)0x11f4a5bd, (q31_t)0x11dbc307, (q31_t)0x11c2dfa2, (q31_t)0x11a9fb8d, (q31_t)0x119116c9, (q31_t)0x11783159,
	(q31_t)0x115f4b3c, (q31_t)0x11466473,
	(q31_t)0x112d7d00, (q31_t)0x111494e4, (q31_t)0x10fbac1e, (q31_t)0x10e2c2b2, (q31_t)0x10c9d89e, (q31_t)0x10b0ede5,
	(q31_t)0x10980287, (q31_t)0x107f1686,
	(q31_t)0x106629e1, (q31_t)0x104d3c9b, (q31_t)0x10344eb4, (q31_t)0x101b602d, (q31_t)0x10027107, (q31_t)0xfe98143,
	(q31_t)0xfd090e1, (q31_t)0xfb79fe4,
	(q31_t)0xf9eae4c, (q31_t)0xf85bc19, (q31_t)0xf6cc94e, (q31_t)0xf53d5ea, (q31_t)0xf3ae1ee, (q31_t)0xf21ed5d, (q31_t)0xf08f836,
	(q31_t)0xef0027b,
	(q31_t)0xed70c2c, (q31_t)0xebe154b, (q31_t)0xea51dd8, (q31_t)0xe8c25d5, (q31_t)0xe732d42, (q31_t)0xe5a3421, (q31_t)0xe413a72,
	(q31_t)0xe284036,
	(q31_t)0xe0f456f, (q31_t)0xdf64a1c, (q31_t)0xddd4e40, (q31_t)0xdc451dc, (q31_t)0xdab54ef, (q31_t)0xd92577b, (q31_t)0xd795982,
	(q31_t)0xd605b03,
	(q31_t)0xd475c00, (q31_t)0xd2e5c7b, (q31_t)0xd155c73, (q31_t)0xcfc5bea, (q31_t)0xce35ae1, (q31_t)0xcca5959, (q31_t)0xcb15752,
	(q31_t)0xc9854cf,
	(q31_t)0xc7f51cf, (q31_t)0xc664e53, (q31_t)0xc4d4a5d, (q31_t)0xc3445ee, (q31_t)0xc1b4107, (q31_t)0xc023ba7, (q31_t)0xbe935d2,
	(q31_t)0xbd02f87,
	(q31_t)0xbb728c7, (q31_t)0xb9e2193, (q31_t)0xb8519ed, (q31_t)0xb6c11d5, (q31_t)0xb53094d, (q31_t)0xb3a0055, (q31_t)0xb20f6ee,
	(q31_t)0xb07ed19,
	(q31_t)0xaeee2d7, (q31_t)0xad5d829, (q31_t)0xabccd11, (q31_t)0xaa3c18e, (q31_t)0xa8ab5a2, (q31_t)0xa71a94f, (q31_t)0xa589c94,
	(q31_t)0xa3f8f73,
	(q31_t)0xa2681ed, (q31_t)0xa0d7403, (q31_t)0x9f465b5, (q31_t)0x9db5706, (q31_t)0x9c247f5, (q31_t)0x9a93884, (q31_t)0x99028b3,
	(q31_t)0x9771884,
	(q31_t)0x95e07f8, (q31_t)0x944f70f, (q31_t)0x92be5ca, (q31_t)0x912d42c, (q31_t)0x8f9c233, (q31_t)0x8e0afe2, (q31_t)0x8c79d3a,
	(q31_t)0x8ae8a3a,
	(q31_t)0x89576e5, (q31_t)0x87c633c, (q31_t)0x8634f3e, (q31_t)0x84a3aee, (q31_t)0x831264c, (q31_t)0x8181159, (q31_t)0x7fefc16,
	(q31_t)0x7e5e685,
	(q31_t)0x7ccd0a5, (q31_t)0x7b3ba78, (q31_t)0x79aa400, (q31_t)0x7818d3c, (q31_t)0x768762e, (q31_t)0x74f5ed7, (q31_t)0x7364738,
	(q31_t)0x71d2f52,
	(q31_t)0x7041726, (q31_t)0x6eafeb4, (q31_t)0x6d1e5fe, (q31_t)0x6b8cd05, (q31_t)0x69fb3c9, (q31_t)0x6869a4c, (q31_t)0x66d808f,
	(q31_t)0x6546692,
	(q31_t)0x63b4c57, (q31_t)0x62231de, (q31_t)0x6091729, (q31_t)0x5effc38, (q31_t)0x5d6e10c, (q31_t)0x5bdc5a7, (q31_t)0x5a4aa09,
	(q31_t)0x58b8e34,
	(q31_t)0x5727228, (q31_t)0x55955e6, (q31_t)0x540396f, (q31_t)0x5271cc4, (q31_t)0x50dffe7, (q31_t)0x4f4e2d8, (q31_t)0x4dbc597,
	(q31_t)0x4c2a827,
	(q31_t)0x4a98a88, (q31_t)0x4906cbb, (q31_t)0x4774ec1, (q31_t)0x45e309a, (q31_t)0x4451249, (q31_t)0x42bf3cd, (q31_t)0x412d528,
	(q31_t)0x3f9b65b,
	(q31_t)0x3e09767, (q31_t)0x3c7784d, (q31_t)0x3ae590d, (q31_t)0x39539a9, (q31_t)0x37c1a22, (q31_t)0x362fa78, (q31_t)0x349daac,
	(q31_t)0x330bac1,
	(q31_t)0x3179ab5, (q31_t)0x2fe7a8c, (q31_t)0x2e55a44, (q31_t)0x2cc39e1, (q31_t)0x2b31961, (q31_t)0x299f8c7, (q31_t)0x280d813,
	(q31_t)0x267b747,
	(q31_t)0x24e9662, (q31_t)0x2357567, (q31_t)0x21c5457, (q31_t)0x2033331, (q31_t)0x1ea11f7, (q31_t)0x1d0f0ab, (q31_t)0x1b7cf4d,
	(q31_t)0x19eaddd,
	(q31_t)0x1858c5e, (q31_t)0x16c6ad0, (q31_t)0x1534934, (q31_t)0x13a278a, (q31_t)0x12105d5, (q31_t)0x107e414, (q31_t)0xeec249,
	(q31_t)0xd5a075,
	(q31_t)0xbc7e99, (q31_t)0xa35cb5, (q31_t)0x8a3acb, (q31_t)0x7118dc, (q31_t)0x57f6e9, (q31_t)0x3ed4f2, (q31_t)0x25b2f8,
	(q31_t)0xc90fe
};

static const q31_t cos_factorsQ31_8192[8192] = {
	(q31_t)0x7ffffff6, (q31_t)0x7fffffa7, (q31_t)0x7fffff09, (q31_t)0x7ffffe1c, (q31_t)0x7ffffce1, (q31_t)0x7ffffb56,
	(q31_t)0x7ffff97c, (q31_t)0x7ffff753,
	(q31_t)0x7ffff4dc, (q31_t)0x7ffff215, (q31_t)0x7fffef00, (q31_t)0x7fffeb9b, (q31_t)0x7fffe7e8, (q31_t)0x7fffe3e5,
	(q31_t)0x7fffdf94, (q31_t)0x7fffdaf3,
	(q31_t)0x7fffd604, (q31_t)0x7fffd0c6, (q31_t)0x7fffcb39, (q31_t)0x7fffc55c, (q31_t)0x7fffbf31, (q31_t)0x7fffb8b7,
	(q31_t)0x7fffb1ee, (q31_t)0x7fffaad6,
	(q31_t)0x7fffa36f, (q31_t)0x7fff9bb9, (q31_t)0x7fff93b4, (q31_t)0x7fff8b61, (q31_t)0x7fff82be, (q31_t)0x7fff79cc,
	(q31_t)0x7fff708b, (q31_t)0x7fff66fc,
	(q31_t)0x7fff5d1d, (q31_t)0x7fff52ef, (q31_t)0x7fff4873, (q31_t)0x7fff3da8, (q31_t)0x7fff328d, (q31_t)0x7fff2724,
	(q31_t)0x7fff1b6b, (q31_t)0x7fff0f64,
	(q31_t)0x7fff030e, (q31_t)0x7ffef669, (q31_t)0x7ffee975, (q31_t)0x7ffedc31, (q31_t)0x7ffece9f, (q31_t)0x7ffec0be,
	(q31_t)0x7ffeb28e, (q31_t)0x7ffea40f,
	(q31_t)0x7ffe9542, (q31_t)0x7ffe8625, (q31_t)0x7ffe76b9, (q31_t)0x7ffe66fe, (q31_t)0x7ffe56f5, (q31_t)0x7ffe469c,
	(q31_t)0x7ffe35f4, (q31_t)0x7ffe24fe,
	(q31_t)0x7ffe13b8, (q31_t)0x7ffe0224, (q31_t)0x7ffdf040, (q31_t)0x7ffdde0e, (q31_t)0x7ffdcb8d, (q31_t)0x7ffdb8bc,
	(q31_t)0x7ffda59d, (q31_t)0x7ffd922f,
	(q31_t)0x7ffd7e72, (q31_t)0x7ffd6a66, (q31_t)0x7ffd560b, (q31_t)0x7ffd4161, (q31_t)0x7ffd2c68, (q31_t)0x7ffd1720,
	(q31_t)0x7ffd0189, (q31_t)0x7ffceba4,
	(q31_t)0x7ffcd56f, (q31_t)0x7ffcbeeb, (q31_t)0x7ffca819, (q31_t)0x7ffc90f7, (q31_t)0x7ffc7987, (q31_t)0x7ffc61c7,
	(q31_t)0x7ffc49b9, (q31_t)0x7ffc315b,
	(q31_t)0x7ffc18af, (q31_t)0x7ffbffb4, (q31_t)0x7ffbe66a, (q31_t)0x7ffbccd0, (q31_t)0x7ffbb2e8, (q31_t)0x7ffb98b1,
	(q31_t)0x7ffb7e2b, (q31_t)0x7ffb6356,
	(q31_t)0x7ffb4833, (q31_t)0x7ffb2cc0, (q31_t)0x7ffb10fe, (q31_t)0x7ffaf4ed, (q31_t)0x7ffad88e, (q31_t)0x7ffabbdf,
	(q31_t)0x7ffa9ee2, (q31_t)0x7ffa8195,
	(q31_t)0x7ffa63fa, (q31_t)0x7ffa460f, (q31_t)0x7ffa27d6, (q31_t)0x7ffa094e, (q31_t)0x7ff9ea76, (q31_t)0x7ff9cb50,
	(q31_t)0x7ff9abdb, (q31_t)0x7ff98c17,
	(q31_t)0x7ff96c04, (q31_t)0x7ff94ba2, (q31_t)0x7ff92af1, (q31_t)0x7ff909f2, (q31_t)0x7ff8e8a3, (q31_t)0x7ff8c705,
	(q31_t)0x7ff8a519, (q31_t)0x7ff882dd,
	(q31_t)0x7ff86053, (q31_t)0x7ff83d79, (q31_t)0x7ff81a51, (q31_t)0x7ff7f6da, (q31_t)0x7ff7d313, (q31_t)0x7ff7aefe,
	(q31_t)0x7ff78a9a, (q31_t)0x7ff765e7,
	(q31_t)0x7ff740e5, (q31_t)0x7ff71b94, (q31_t)0x7ff6f5f4, (q31_t)0x7ff6d005, (q31_t)0x7ff6a9c8, (q31_t)0x7ff6833b,
	(q31_t)0x7ff65c5f, (q31_t)0x7ff63535,
	(q31_t)0x7ff60dbb, (q31_t)0x7ff5e5f3, (q31_t)0x7ff5bddc, (q31_t)0x7ff59576, (q31_t)0x7ff56cc0, (q31_t)0x7ff543bc,
	(q31_t)0x7ff51a69, (q31_t)0x7ff4f0c7,
	(q31_t)0x7ff4c6d6, (q31_t)0x7ff49c96, (q31_t)0x7ff47208, (q31_t)0x7ff4472a, (q31_t)0x7ff41bfd, (q31_t)0x7ff3f082,
	(q31_t)0x7ff3c4b7, (q31_t)0x7ff3989e,
	(q31_t)0x7ff36c36, (q31_t)0x7ff33f7e, (q31_t)0x7ff31278, (q31_t)0x7ff2e523, (q31_t)0x7ff2b77f, (q31_t)0x7ff2898c,
	(q31_t)0x7ff25b4a, (q31_t)0x7ff22cb9,
	(q31_t)0x7ff1fdd9, (q31_t)0x7ff1ceab, (q31_t)0x7ff19f2d, (q31_t)0x7ff16f61, (q31_t)0x7ff13f45, (q31_t)0x7ff10edb,
	(q31_t)0x7ff0de22, (q31_t)0x7ff0ad19,
	(q31_t)0x7ff07bc2, (q31_t)0x7ff04a1c, (q31_t)0x7ff01827, (q31_t)0x7fefe5e4, (q31_t)0x7fefb351, (q31_t)0x7fef806f,
	(q31_t)0x7fef4d3e, (q31_t)0x7fef19bf,
	(q31_t)0x7feee5f0, (q31_t)0x7feeb1d3, (q31_t)0x7fee7d67, (q31_t)0x7fee48ac, (q31_t)0x7fee13a1, (q31_t)0x7fedde48,
	(q31_t)0x7feda8a0, (q31_t)0x7fed72aa,
	(q31_t)0x7fed3c64, (q31_t)0x7fed05cf, (q31_t)0x7fecceec, (q31_t)0x7fec97b9, (q31_t)0x7fec6038, (q31_t)0x7fec2867,
	(q31_t)0x7febf048, (q31_t)0x7febb7da,
	(q31_t)0x7feb7f1d, (q31_t)0x7feb4611, (q31_t)0x7feb0cb6, (q31_t)0x7fead30c, (q31_t)0x7fea9914, (q31_t)0x7fea5ecc,
	(q31_t)0x7fea2436, (q31_t)0x7fe9e950,
	(q31_t)0x7fe9ae1c, (q31_t)0x7fe97299, (q31_t)0x7fe936c7, (q31_t)0x7fe8faa6, (q31_t)0x7fe8be36, (q31_t)0x7fe88177,
	(q31_t)0x7fe84469, (q31_t)0x7fe8070d,
	(q31_t)0x7fe7c961, (q31_t)0x7fe78b67, (q31_t)0x7fe74d1e, (q31_t)0x7fe70e85, (q31_t)0x7fe6cf9e, (q31_t)0x7fe69068,
	(q31_t)0x7fe650e3, (q31_t)0x7fe61110,
	(q31_t)0x7fe5d0ed, (q31_t)0x7fe5907b, (q31_t)0x7fe54fbb, (q31_t)0x7fe50eac, (q31_t)0x7fe4cd4d, (q31_t)0x7fe48ba0,
	(q31_t)0x7fe449a4, (q31_t)0x7fe40759,
	(q31_t)0x7fe3c4bf, (q31_t)0x7fe381d7, (q31_t)0x7fe33e9f, (q31_t)0x7fe2fb19, (q31_t)0x7fe2b743, (q31_t)0x7fe2731f,
	(q31_t)0x7fe22eac, (q31_t)0x7fe1e9ea,
	(q31_t)0x7fe1a4d9, (q31_t)0x7fe15f79, (q31_t)0x7fe119cb, (q31_t)0x7fe0d3cd, (q31_t)0x7fe08d81, (q31_t)0x7fe046e5,
	(q31_t)0x7fdffffb, (q31_t)0x7fdfb8c2,
	(q31_t)0x7fdf713a, (q31_t)0x7fdf2963, (q31_t)0x7fdee13e, (q31_t)0x7fde98c9, (q31_t)0x7fde5006, (q31_t)0x7fde06f3,
	(q31_t)0x7fddbd92, (q31_t)0x7fdd73e2,
	(q31_t)0x7fdd29e3, (q31_t)0x7fdcdf95, (q31_t)0x7fdc94f9, (q31_t)0x7fdc4a0d, (q31_t)0x7fdbfed3, (q31_t)0x7fdbb349,
	(q31_t)0x7fdb6771, (q31_t)0x7fdb1b4a,
	(q31_t)0x7fdaced4, (q31_t)0x7fda820f, (q31_t)0x7fda34fc, (q31_t)0x7fd9e799, (q31_t)0x7fd999e8, (q31_t)0x7fd94be8,
	(q31_t)0x7fd8fd98, (q31_t)0x7fd8aefa,
	(q31_t)0x7fd8600e, (q31_t)0x7fd810d2, (q31_t)0x7fd7c147, (q31_t)0x7fd7716e, (q31_t)0x7fd72146, (q31_t)0x7fd6d0cf,
	(q31_t)0x7fd68009, (q31_t)0x7fd62ef4,
	(q31_t)0x7fd5dd90, (q31_t)0x7fd58bdd, (q31_t)0x7fd539dc, (q31_t)0x7fd4e78c, (q31_t)0x7fd494ed, (q31_t)0x7fd441ff,
	(q31_t)0x7fd3eec2, (q31_t)0x7fd39b36,
	(q31_t)0x7fd3475c, (q31_t)0x7fd2f332, (q31_t)0x7fd29eba, (q31_t)0x7fd249f3, (q31_t)0x7fd1f4dd, (q31_t)0x7fd19f78,
	(q31_t)0x7fd149c5, (q31_t)0x7fd0f3c2,
	(q31_t)0x7fd09d71, (q31_t)0x7fd046d1, (q31_t)0x7fcfefe2, (q31_t)0x7fcf98a4, (q31_t)0x7fcf4117, (q31_t)0x7fcee93c,
	(q31_t)0x7fce9112, (q31_t)0x7fce3898,
	(q31_t)0x7fcddfd0, (q31_t)0x7fcd86b9, (q31_t)0x7fcd2d54, (q31_t)0x7fccd39f, (q31_t)0x7fcc799c, (q31_t)0x7fcc1f4a,
	(q31_t)0x7fcbc4a9, (q31_t)0x7fcb69b9,
	(q31_t)0x7fcb0e7a, (q31_t)0x7fcab2ed, (q31_t)0x7fca5710, (q31_t)0x7fc9fae5, (q31_t)0x7fc99e6b, (q31_t)0x7fc941a2,
	(q31_t)0x7fc8e48b, (q31_t)0x7fc88724,
	(q31_t)0x7fc8296f, (q31_t)0x7fc7cb6b, (q31_t)0x7fc76d18, (q31_t)0x7fc70e76, (q31_t)0x7fc6af86, (q31_t)0x7fc65046,
	(q31_t)0x7fc5f0b8, (q31_t)0x7fc590db,
	(q31_t)0x7fc530af, (q31_t)0x7fc4d035, (q31_t)0x7fc46f6b, (q31_t)0x7fc40e53, (q31_t)0x7fc3acec, (q31_t)0x7fc34b36,
	(q31_t)0x7fc2e931, (q31_t)0x7fc286de,
	(q31_t)0x7fc2243b, (q31_t)0x7fc1c14a, (q31_t)0x7fc15e0a, (q31_t)0x7fc0fa7b, (q31_t)0x7fc0969e, (q31_t)0x7fc03271,
	(q31_t)0x7fbfcdf6, (q31_t)0x7fbf692c,
	(q31_t)0x7fbf0414, (q31_t)0x7fbe9eac, (q31_t)0x7fbe38f6, (q31_t)0x7fbdd2f0, (q31_t)0x7fbd6c9c, (q31_t)0x7fbd05fa,
	(q31_t)0x7fbc9f08, (q31_t)0x7fbc37c8,
	(q31_t)0x7fbbd039, (q31_t)0x7fbb685b, (q31_t)0x7fbb002e, (q31_t)0x7fba97b2, (q31_t)0x7fba2ee8, (q31_t)0x7fb9c5cf,
	(q31_t)0x7fb95c67, (q31_t)0x7fb8f2b0,
	(q31_t)0x7fb888ab, (q31_t)0x7fb81e57, (q31_t)0x7fb7b3b4, (q31_t)0x7fb748c2, (q31_t)0x7fb6dd81, (q31_t)0x7fb671f2,
	(q31_t)0x7fb60614, (q31_t)0x7fb599e7,
	(q31_t)0x7fb52d6b, (q31_t)0x7fb4c0a1, (q31_t)0x7fb45387, (q31_t)0x7fb3e61f, (q31_t)0x7fb37869, (q31_t)0x7fb30a63,
	(q31_t)0x7fb29c0f, (q31_t)0x7fb22d6c,
	(q31_t)0x7fb1be7a, (q31_t)0x7fb14f39, (q31_t)0x7fb0dfaa, (q31_t)0x7fb06fcb, (q31_t)0x7fafff9e, (q31_t)0x7faf8f23,
	(q31_t)0x7faf1e58, (q31_t)0x7faead3f,
	(q31_t)0x7fae3bd7, (q31_t)0x7fadca20, (q31_t)0x7fad581b, (q31_t)0x7face5c6, (q31_t)0x7fac7323, (q31_t)0x7fac0031,
	(q31_t)0x7fab8cf1, (q31_t)0x7fab1962,
	(q31_t)0x7faaa584, (q31_t)0x7faa3157, (q31_t)0x7fa9bcdb, (q31_t)0x7fa94811, (q31_t)0x7fa8d2f8, (q31_t)0x7fa85d90,
	(q31_t)0x7fa7e7d9, (q31_t)0x7fa771d4,
	(q31_t)0x7fa6fb80, (q31_t)0x7fa684dd, (q31_t)0x7fa60dec, (q31_t)0x7fa596ac, (q31_t)0x7fa51f1d, (q31_t)0x7fa4a73f,
	(q31_t)0x7fa42f12, (q31_t)0x7fa3b697,
	(q31_t)0x7fa33dcd, (q31_t)0x7fa2c4b5, (q31_t)0x7fa24b4d, (q31_t)0x7fa1d197, (q31_t)0x7fa15792, (q31_t)0x7fa0dd3f,
	(q31_t)0x7fa0629c, (q31_t)0x7f9fe7ab,
	(q31_t)0x7f9f6c6b, (q31_t)0x7f9ef0dd, (q31_t)0x7f9e7500, (q31_t)0x7f9df8d4, (q31_t)0x7f9d7c59, (q31_t)0x7f9cff90,
	(q31_t)0x7f9c8278, (q31_t)0x7f9c0511,
	(q31_t)0x7f9b875b, (q31_t)0x7f9b0957, (q31_t)0x7f9a8b04, (q31_t)0x7f9a0c62, (q31_t)0x7f998d72, (q31_t)0x7f990e33,
	(q31_t)0x7f988ea5, (q31_t)0x7f980ec8,
	(q31_t)0x7f978e9d, (q31_t)0x7f970e23, (q31_t)0x7f968d5b, (q31_t)0x7f960c43, (q31_t)0x7f958add, (q31_t)0x7f950929,
	(q31_t)0x7f948725, (q31_t)0x7f9404d3,
	(q31_t)0x7f938232, (q31_t)0x7f92ff43, (q31_t)0x7f927c04, (q31_t)0x7f91f878, (q31_t)0x7f91749c, (q31_t)0x7f90f072,
	(q31_t)0x7f906bf9, (q31_t)0x7f8fe731,
	(q31_t)0x7f8f621b, (q31_t)0x7f8edcb6, (q31_t)0x7f8e5702, (q31_t)0x7f8dd0ff, (q31_t)0x7f8d4aae, (q31_t)0x7f8cc40f,
	(q31_t)0x7f8c3d20, (q31_t)0x7f8bb5e3,
	(q31_t)0x7f8b2e57, (q31_t)0x7f8aa67d, (q31_t)0x7f8a1e54, (q31_t)0x7f8995dc, (q31_t)0x7f890d15, (q31_t)0x7f888400,
	(q31_t)0x7f87fa9c, (q31_t)0x7f8770ea,
	(q31_t)0x7f86e6e9, (q31_t)0x7f865c99, (q31_t)0x7f85d1fa, (q31_t)0x7f85470d, (q31_t)0x7f84bbd1, (q31_t)0x7f843047,
	(q31_t)0x7f83a46e, (q31_t)0x7f831846,
	(q31_t)0x7f828bcf, (q31_t)0x7f81ff0a, (q31_t)0x7f8171f6, (q31_t)0x7f80e494, (q31_t)0x7f8056e3, (q31_t)0x7f7fc8e3,
	(q31_t)0x7f7f3a95, (q31_t)0x7f7eabf8,
	(q31_t)0x7f7e1d0c, (q31_t)0x7f7d8dd2, (q31_t)0x7f7cfe49, (q31_t)0x7f7c6e71, (q31_t)0x7f7bde4b, (q31_t)0x7f7b4dd6,
	(q31_t)0x7f7abd13, (q31_t)0x7f7a2c01,
	(q31_t)0x7f799aa0, (q31_t)0x7f7908f0, (q31_t)0x7f7876f2, (q31_t)0x7f77e4a6, (q31_t)0x7f77520a, (q31_t)0x7f76bf21,
	(q31_t)0x7f762be8, (q31_t)0x7f759861,
	(q31_t)0x7f75048b, (q31_t)0x7f747067, (q31_t)0x7f73dbf4, (q31_t)0x7f734732, (q31_t)0x7f72b222, (q31_t)0x7f721cc3,
	(q31_t)0x7f718715, (q31_t)0x7f70f119,
	(q31_t)0x7f705ace, (q31_t)0x7f6fc435, (q31_t)0x7f6f2d4d, (q31_t)0x7f6e9617, (q31_t)0x7f6dfe91, (q31_t)0x7f6d66be,
	(q31_t)0x7f6cce9b, (q31_t)0x7f6c362a,
	(q31_t)0x7f6b9d6b, (q31_t)0x7f6b045d, (q31_t)0x7f6a6b00, (q31_t)0x7f69d154, (q31_t)0x7f69375a, (q31_t)0x7f689d12,
	(q31_t)0x7f68027b, (q31_t)0x7f676795,
	(q31_t)0x7f66cc61, (q31_t)0x7f6630de, (q31_t)0x7f65950c, (q31_t)0x7f64f8ec, (q31_t)0x7f645c7d, (q31_t)0x7f63bfc0,
	(q31_t)0x7f6322b4, (q31_t)0x7f62855a,
	(q31_t)0x7f61e7b1, (q31_t)0x7f6149b9, (q31_t)0x7f60ab73, (q31_t)0x7f600cdf, (q31_t)0x7f5f6dfb, (q31_t)0x7f5ecec9,
	(q31_t)0x7f5e2f49, (q31_t)0x7f5d8f7a,
	(q31_t)0x7f5cef5c, (q31_t)0x7f5c4ef0, (q31_t)0x7f5bae36, (q31_t)0x7f5b0d2c, (q31_t)0x7f5a6bd5, (q31_t)0x7f59ca2e,
	(q31_t)0x7f592839, (q31_t)0x7f5885f6,
	(q31_t)0x7f57e364, (q31_t)0x7f574083, (q31_t)0x7f569d54, (q31_t)0x7f55f9d6, (q31_t)0x7f55560a, (q31_t)0x7f54b1ef,
	(q31_t)0x7f540d86, (q31_t)0x7f5368ce,
	(q31_t)0x7f52c3c8, (q31_t)0x7f521e73, (q31_t)0x7f5178cf, (q31_t)0x7f50d2dd, (q31_t)0x7f502c9d, (q31_t)0x7f4f860e,
	(q31_t)0x7f4edf30, (q31_t)0x7f4e3804,
	(q31_t)0x7f4d9089, (q31_t)0x7f4ce8c0, (q31_t)0x7f4c40a8, (q31_t)0x7f4b9842, (q31_t)0x7f4aef8d, (q31_t)0x7f4a468a,
	(q31_t)0x7f499d38, (q31_t)0x7f48f398,
	(q31_t)0x7f4849a9, (q31_t)0x7f479f6c, (q31_t)0x7f46f4e0, (q31_t)0x7f464a06, (q31_t)0x7f459edd, (q31_t)0x7f44f365,
	(q31_t)0x7f44479f, (q31_t)0x7f439b8b,
	(q31_t)0x7f42ef28, (q31_t)0x7f424277, (q31_t)0x7f419577, (q31_t)0x7f40e828, (q31_t)0x7f403a8b, (q31_t)0x7f3f8ca0,
	(q31_t)0x7f3ede66, (q31_t)0x7f3e2fde,
	(q31_t)0x7f3d8107, (q31_t)0x7f3cd1e2, (q31_t)0x7f3c226e, (q31_t)0x7f3b72ab, (q31_t)0x7f3ac29b, (q31_t)0x7f3a123b,
	(q31_t)0x7f39618e, (q31_t)0x7f38b091,
	(q31_t)0x7f37ff47, (q31_t)0x7f374dad, (q31_t)0x7f369bc6, (q31_t)0x7f35e990, (q31_t)0x7f35370b, (q31_t)0x7f348438,
	(q31_t)0x7f33d116, (q31_t)0x7f331da6,
	(q31_t)0x7f3269e8, (q31_t)0x7f31b5db, (q31_t)0x7f31017f, (q31_t)0x7f304cd6, (q31_t)0x7f2f97dd, (q31_t)0x7f2ee296,
	(q31_t)0x7f2e2d01, (q31_t)0x7f2d771e,
	(q31_t)0x7f2cc0eb, (q31_t)0x7f2c0a6b, (q31_t)0x7f2b539c, (q31_t)0x7f2a9c7e, (q31_t)0x7f29e512, (q31_t)0x7f292d58,
	(q31_t)0x7f28754f, (q31_t)0x7f27bcf8,
	(q31_t)0x7f270452, (q31_t)0x7f264b5e, (q31_t)0x7f25921c, (q31_t)0x7f24d88b, (q31_t)0x7f241eab, (q31_t)0x7f23647e,
	(q31_t)0x7f22aa01, (q31_t)0x7f21ef37,
	(q31_t)0x7f21341e, (q31_t)0x7f2078b6, (q31_t)0x7f1fbd00, (q31_t)0x7f1f00fc, (q31_t)0x7f1e44a9, (q31_t)0x7f1d8808,
	(q31_t)0x7f1ccb18, (q31_t)0x7f1c0dda,
	(q31_t)0x7f1b504e, (q31_t)0x7f1a9273, (q31_t)0x7f19d44a, (q31_t)0x7f1915d2, (q31_t)0x7f18570c, (q31_t)0x7f1797f8,
	(q31_t)0x7f16d895, (q31_t)0x7f1618e4,
	(q31_t)0x7f1558e4, (q31_t)0x7f149896, (q31_t)0x7f13d7fa, (q31_t)0x7f13170f, (q31_t)0x7f1255d6, (q31_t)0x7f11944f,
	(q31_t)0x7f10d279, (q31_t)0x7f101054,
	(q31_t)0x7f0f4de2, (q31_t)0x7f0e8b21, (q31_t)0x7f0dc811, (q31_t)0x7f0d04b3, (q31_t)0x7f0c4107, (q31_t)0x7f0b7d0d,
	(q31_t)0x7f0ab8c4, (q31_t)0x7f09f42d,
	(q31_t)0x7f092f47, (q31_t)0x7f086a13, (q31_t)0x7f07a491, (q31_t)0x7f06dec0, (q31_t)0x7f0618a1, (q31_t)0x7f055233,
	(q31_t)0x7f048b78, (q31_t)0x7f03c46d,
	(q31_t)0x7f02fd15, (q31_t)0x7f02356e, (q31_t)0x7f016d79, (q31_t)0x7f00a535, (q31_t)0x7effdca4, (q31_t)0x7eff13c3,
	(q31_t)0x7efe4a95, (q31_t)0x7efd8118,
	(q31_t)0x7efcb74d, (q31_t)0x7efbed33, (q31_t)0x7efb22cb, (q31_t)0x7efa5815, (q31_t)0x7ef98d11, (q31_t)0x7ef8c1be,
	(q31_t)0x7ef7f61d, (q31_t)0x7ef72a2d,
	(q31_t)0x7ef65def, (q31_t)0x7ef59163, (q31_t)0x7ef4c489, (q31_t)0x7ef3f760, (q31_t)0x7ef329e9, (q31_t)0x7ef25c24,
	(q31_t)0x7ef18e10, (q31_t)0x7ef0bfae,
	(q31_t)0x7eeff0fe, (q31_t)0x7eef21ff, (q31_t)0x7eee52b2, (q31_t)0x7eed8317, (q31_t)0x7eecb32d, (q31_t)0x7eebe2f6,
	(q31_t)0x7eeb1270, (q31_t)0x7eea419b,
	(q31_t)0x7ee97079, (q31_t)0x7ee89f08, (q31_t)0x7ee7cd49, (q31_t)0x7ee6fb3b, (q31_t)0x7ee628df, (q31_t)0x7ee55635,
	(q31_t)0x7ee4833d, (q31_t)0x7ee3aff6,
	(q31_t)0x7ee2dc61, (q31_t)0x7ee2087e, (q31_t)0x7ee1344d, (q31_t)0x7ee05fcd, (q31_t)0x7edf8aff, (q31_t)0x7edeb5e3,
	(q31_t)0x7edde079, (q31_t)0x7edd0ac0,
	(q31_t)0x7edc34b9, (q31_t)0x7edb5e64, (q31_t)0x7eda87c0, (q31_t)0x7ed9b0ce, (q31_t)0x7ed8d98e, (q31_t)0x7ed80200,
	(q31_t)0x7ed72a24, (q31_t)0x7ed651f9,
	(q31_t)0x7ed57980, (q31_t)0x7ed4a0b9, (q31_t)0x7ed3c7a3, (q31_t)0x7ed2ee40, (q31_t)0x7ed2148e, (q31_t)0x7ed13a8e,
	(q31_t)0x7ed0603f, (q31_t)0x7ecf85a3,
	(q31_t)0x7eceaab8, (q31_t)0x7ecdcf7f, (q31_t)0x7eccf3f8, (q31_t)0x7ecc1822, (q31_t)0x7ecb3bff, (q31_t)0x7eca5f8d,
	(q31_t)0x7ec982cd, (q31_t)0x7ec8a5bf,
	(q31_t)0x7ec7c862, (q31_t)0x7ec6eab7, (q31_t)0x7ec60cbe, (q31_t)0x7ec52e77, (q31_t)0x7ec44fe2, (q31_t)0x7ec370fe,
	(q31_t)0x7ec291cd, (q31_t)0x7ec1b24d,
	(q31_t)0x7ec0d27f, (q31_t)0x7ebff263, (q31_t)0x7ebf11f8, (q31_t)0x7ebe313f, (q31_t)0x7ebd5039, (q31_t)0x7ebc6ee4,
	(q31_t)0x7ebb8d40, (q31_t)0x7ebaab4f,
	(q31_t)0x7eb9c910, (q31_t)0x7eb8e682, (q31_t)0x7eb803a6, (q31_t)0x7eb7207c, (q31_t)0x7eb63d04, (q31_t)0x7eb5593d,
	(q31_t)0x7eb47529, (q31_t)0x7eb390c6,
	(q31_t)0x7eb2ac15, (q31_t)0x7eb1c716, (q31_t)0x7eb0e1c9, (q31_t)0x7eaffc2e, (q31_t)0x7eaf1645, (q31_t)0x7eae300d,
	(q31_t)0x7ead4987, (q31_t)0x7eac62b3,
	(q31_t)0x7eab7b91, (q31_t)0x7eaa9421, (q31_t)0x7ea9ac63, (q31_t)0x7ea8c457, (q31_t)0x7ea7dbfc, (q31_t)0x7ea6f353,
	(q31_t)0x7ea60a5d, (q31_t)0x7ea52118,
	(q31_t)0x7ea43785, (q31_t)0x7ea34da4, (q31_t)0x7ea26374, (q31_t)0x7ea178f7, (q31_t)0x7ea08e2b, (q31_t)0x7e9fa312,
	(q31_t)0x7e9eb7aa, (q31_t)0x7e9dcbf4,
	(q31_t)0x7e9cdff0, (q31_t)0x7e9bf39e, (q31_t)0x7e9b06fe, (q31_t)0x7e9a1a10, (q31_t)0x7e992cd4, (q31_t)0x7e983f49,
	(q31_t)0x7e975171, (q31_t)0x7e96634a,
	(q31_t)0x7e9574d6, (q31_t)0x7e948613, (q31_t)0x7e939702, (q31_t)0x7e92a7a3, (q31_t)0x7e91b7f6, (q31_t)0x7e90c7fb,
	(q31_t)0x7e8fd7b2, (q31_t)0x7e8ee71b,
	(q31_t)0x7e8df636, (q31_t)0x7e8d0502, (q31_t)0x7e8c1381, (q31_t)0x7e8b21b1, (q31_t)0x7e8a2f94, (q31_t)0x7e893d28,
	(q31_t)0x7e884a6f, (q31_t)0x7e875767,
	(q31_t)0x7e866411, (q31_t)0x7e85706d, (q31_t)0x7e847c7c, (q31_t)0x7e83883c, (q31_t)0x7e8293ae, (q31_t)0x7e819ed2,
	(q31_t)0x7e80a9a8, (q31_t)0x7e7fb430,
	(q31_t)0x7e7ebe6a, (q31_t)0x7e7dc856, (q31_t)0x7e7cd1f4, (q31_t)0x7e7bdb44, (q31_t)0x7e7ae446, (q31_t)0x7e79ecf9,
	(q31_t)0x7e78f55f, (q31_t)0x7e77fd77,
	(q31_t)0x7e770541, (q31_t)0x7e760cbd, (q31_t)0x7e7513ea, (q31_t)0x7e741aca, (q31_t)0x7e73215c, (q31_t)0x7e7227a0,
	(q31_t)0x7e712d96, (q31_t)0x7e70333d,
	(q31_t)0x7e6f3897, (q31_t)0x7e6e3da3, (q31_t)0x7e6d4261, (q31_t)0x7e6c46d1, (q31_t)0x7e6b4af2, (q31_t)0x7e6a4ec6,
	(q31_t)0x7e69524c, (q31_t)0x7e685584,
	(q31_t)0x7e67586e, (q31_t)0x7e665b0a, (q31_t)0x7e655d58, (q31_t)0x7e645f58, (q31_t)0x7e63610a, (q31_t)0x7e62626e,
	(q31_t)0x7e616384, (q31_t)0x7e60644c,
	(q31_t)0x7e5f64c7, (q31_t)0x7e5e64f3, (q31_t)0x7e5d64d1, (q31_t)0x7e5c6461, (q31_t)0x7e5b63a4, (q31_t)0x7e5a6298,
	(q31_t)0x7e59613f, (q31_t)0x7e585f97,
	(q31_t)0x7e575da2, (q31_t)0x7e565b5f, (q31_t)0x7e5558ce, (q31_t)0x7e5455ef, (q31_t)0x7e5352c1, (q31_t)0x7e524f46,
	(q31_t)0x7e514b7e, (q31_t)0x7e504767,
	(q31_t)0x7e4f4302, (q31_t)0x7e4e3e4f, (q31_t)0x7e4d394f, (q31_t)0x7e4c3400, (q31_t)0x7e4b2e64, (q31_t)0x7e4a287a,
	(q31_t)0x7e492241, (q31_t)0x7e481bbb,
	(q31_t)0x7e4714e7, (q31_t)0x7e460dc5, (q31_t)0x7e450656, (q31_t)0x7e43fe98, (q31_t)0x7e42f68c, (q31_t)0x7e41ee33,
	(q31_t)0x7e40e58c, (q31_t)0x7e3fdc97,
	(q31_t)0x7e3ed353, (q31_t)0x7e3dc9c3, (q31_t)0x7e3cbfe4, (q31_t)0x7e3bb5b7, (q31_t)0x7e3aab3c, (q31_t)0x7e39a074,
	(q31_t)0x7e38955e, (q31_t)0x7e3789fa,
	(q31_t)0x7e367e48, (q31_t)0x7e357248, (q31_t)0x7e3465fa, (q31_t)0x7e33595e, (q31_t)0x7e324c75, (q31_t)0x7e313f3e,
	(q31_t)0x7e3031b9, (q31_t)0x7e2f23e6,
	(q31_t)0x7e2e15c5, (q31_t)0x7e2d0756, (q31_t)0x7e2bf89a, (q31_t)0x7e2ae990, (q31_t)0x7e29da38, (q31_t)0x7e28ca92,
	(q31_t)0x7e27ba9e, (q31_t)0x7e26aa5d,
	(q31_t)0x7e2599cd, (q31_t)0x7e2488f0, (q31_t)0x7e2377c5, (q31_t)0x7e22664c, (q31_t)0x7e215486, (q31_t)0x7e204271,
	(q31_t)0x7e1f300f, (q31_t)0x7e1e1d5f,
	(q31_t)0x7e1d0a61, (q31_t)0x7e1bf716, (q31_t)0x7e1ae37c, (q31_t)0x7e19cf95, (q31_t)0x7e18bb60, (q31_t)0x7e17a6dd,
	(q31_t)0x7e16920d, (q31_t)0x7e157cee,
	(q31_t)0x7e146782, (q31_t)0x7e1351c9, (q31_t)0x7e123bc1, (q31_t)0x7e11256c, (q31_t)0x7e100ec8, (q31_t)0x7e0ef7d7,
	(q31_t)0x7e0de099, (q31_t)0x7e0cc90c,
	(q31_t)0x7e0bb132, (q31_t)0x7e0a990a, (q31_t)0x7e098095, (q31_t)0x7e0867d1, (q31_t)0x7e074ec0, (q31_t)0x7e063561,
	(q31_t)0x7e051bb4, (q31_t)0x7e0401ba,
	(q31_t)0x7e02e772, (q31_t)0x7e01ccdc, (q31_t)0x7e00b1f9, (q31_t)0x7dff96c7, (q31_t)0x7dfe7b48, (q31_t)0x7dfd5f7b,
	(q31_t)0x7dfc4361, (q31_t)0x7dfb26f9,
	(q31_t)0x7dfa0a43, (q31_t)0x7df8ed3f, (q31_t)0x7df7cfee, (q31_t)0x7df6b24f, (q31_t)0x7df59462, (q31_t)0x7df47628,
	(q31_t)0x7df357a0, (q31_t)0x7df238ca,
	(q31_t)0x7df119a7, (q31_t)0x7deffa35, (q31_t)0x7deeda77, (q31_t)0x7dedba6a, (q31_t)0x7dec9a10, (q31_t)0x7deb7968,
	(q31_t)0x7dea5872, (q31_t)0x7de9372f,
	(q31_t)0x7de8159e, (q31_t)0x7de6f3c0, (q31_t)0x7de5d193, (q31_t)0x7de4af1a, (q31_t)0x7de38c52, (q31_t)0x7de2693d,
	(q31_t)0x7de145da, (q31_t)0x7de02229,
	(q31_t)0x7ddefe2b, (q31_t)0x7dddd9e0, (q31_t)0x7ddcb546, (q31_t)0x7ddb905f, (q31_t)0x7dda6b2a, (q31_t)0x7dd945a8,
	(q31_t)0x7dd81fd8, (q31_t)0x7dd6f9ba,
	(q31_t)0x7dd5d34f, (q31_t)0x7dd4ac96, (q31_t)0x7dd38590, (q31_t)0x7dd25e3c, (q31_t)0x7dd1369a, (q31_t)0x7dd00eab,
	(q31_t)0x7dcee66e, (q31_t)0x7dcdbde3,
	(q31_t)0x7dcc950b, (q31_t)0x7dcb6be6, (q31_t)0x7dca4272, (q31_t)0x7dc918b1, (q31_t)0x7dc7eea3, (q31_t)0x7dc6c447,
	(q31_t)0x7dc5999d, (q31_t)0x7dc46ea6,
	(q31_t)0x7dc34361, (q31_t)0x7dc217cf, (q31_t)0x7dc0ebef, (q31_t)0x7dbfbfc1, (q31_t)0x7dbe9346, (q31_t)0x7dbd667d,
	(q31_t)0x7dbc3967, (q31_t)0x7dbb0c03,
	(q31_t)0x7db9de52, (q31_t)0x7db8b053, (q31_t)0x7db78207, (q31_t)0x7db6536d, (q31_t)0x7db52485, (q31_t)0x7db3f550,
	(q31_t)0x7db2c5cd, (q31_t)0x7db195fd,
	(q31_t)0x7db065df, (q31_t)0x7daf3574, (q31_t)0x7dae04bb, (q31_t)0x7dacd3b5, (q31_t)0x7daba261, (q31_t)0x7daa70c0,
	(q31_t)0x7da93ed1, (q31_t)0x7da80c95,
	(q31_t)0x7da6da0b, (q31_t)0x7da5a733, (q31_t)0x7da4740e, (q31_t)0x7da3409c, (q31_t)0x7da20cdc, (q31_t)0x7da0d8cf,
	(q31_t)0x7d9fa474, (q31_t)0x7d9e6fcb,
	(q31_t)0x7d9d3ad6, (q31_t)0x7d9c0592, (q31_t)0x7d9ad001, (q31_t)0x7d999a23, (q31_t)0x7d9863f7, (q31_t)0x7d972d7e,
	(q31_t)0x7d95f6b7, (q31_t)0x7d94bfa3,
	(q31_t)0x7d938841, (q31_t)0x7d925092, (q31_t)0x7d911896, (q31_t)0x7d8fe04c, (q31_t)0x7d8ea7b4, (q31_t)0x7d8d6ecf,
	(q31_t)0x7d8c359d, (q31_t)0x7d8afc1d,
	(q31_t)0x7d89c250, (q31_t)0x7d888835, (q31_t)0x7d874dcd, (q31_t)0x7d861317, (q31_t)0x7d84d814, (q31_t)0x7d839cc4,
	(q31_t)0x7d826126, (q31_t)0x7d81253a,
	(q31_t)0x7d7fe902, (q31_t)0x7d7eac7c, (q31_t)0x7d7d6fa8, (q31_t)0x7d7c3287, (q31_t)0x7d7af519, (q31_t)0x7d79b75d,
	(q31_t)0x7d787954, (q31_t)0x7d773afd,
	(q31_t)0x7d75fc59, (q31_t)0x7d74bd68, (q31_t)0x7d737e29, (q31_t)0x7d723e9d, (q31_t)0x7d70fec4, (q31_t)0x7d6fbe9d,
	(q31_t)0x7d6e7e29, (q31_t)0x7d6d3d67,
	(q31_t)0x7d6bfc58, (q31_t)0x7d6abafc, (q31_t)0x7d697952, (q31_t)0x7d68375b, (q31_t)0x7d66f517, (q31_t)0x7d65b285,
	(q31_t)0x7d646fa6, (q31_t)0x7d632c79,
	(q31_t)0x7d61e8ff, (q31_t)0x7d60a538, (q31_t)0x7d5f6124, (q31_t)0x7d5e1cc2, (q31_t)0x7d5cd813, (q31_t)0x7d5b9316,
	(q31_t)0x7d5a4dcc, (q31_t)0x7d590835,
	(q31_t)0x7d57c251, (q31_t)0x7d567c1f, (q31_t)0x7d5535a0, (q31_t)0x7d53eed3, (q31_t)0x7d52a7ba, (q31_t)0x7d516053,
	(q31_t)0x7d50189e, (q31_t)0x7d4ed09d,
	(q31_t)0x7d4d884e, (q31_t)0x7d4c3fb1, (q31_t)0x7d4af6c8, (q31_t)0x7d49ad91, (q31_t)0x7d48640d, (q31_t)0x7d471a3c,
	(q31_t)0x7d45d01d, (q31_t)0x7d4485b1,
	(q31_t)0x7d433af8, (q31_t)0x7d41eff1, (q31_t)0x7d40a49e, (q31_t)0x7d3f58fd, (q31_t)0x7d3e0d0e, (q31_t)0x7d3cc0d3,
	(q31_t)0x7d3b744a, (q31_t)0x7d3a2774,
	(q31_t)0x7d38da51, (q31_t)0x7d378ce0, (q31_t)0x7d363f23, (q31_t)0x7d34f118, (q31_t)0x7d33a2bf, (q31_t)0x7d32541a,
	(q31_t)0x7d310527, (q31_t)0x7d2fb5e7,
	(q31_t)0x7d2e665a, (q31_t)0x7d2d1680, (q31_t)0x7d2bc659, (q31_t)0x7d2a75e4, (q31_t)0x7d292522, (q31_t)0x7d27d413,
	(q31_t)0x7d2682b6, (q31_t)0x7d25310d,
	(q31_t)0x7d23df16, (q31_t)0x7d228cd2, (q31_t)0x7d213a41, (q31_t)0x7d1fe762, (q31_t)0x7d1e9437, (q31_t)0x7d1d40be,
	(q31_t)0x7d1becf8, (q31_t)0x7d1a98e5,
	(q31_t)0x7d194485, (q31_t)0x7d17efd8, (q31_t)0x7d169add, (q31_t)0x7d154595, (q31_t)0x7d13f001, (q31_t)0x7d129a1f,
	(q31_t)0x7d1143ef, (q31_t)0x7d0fed73,
	(q31_t)0x7d0e96aa, (q31_t)0x7d0d3f93, (q31_t)0x7d0be82f, (q31_t)0x7d0a907e, (q31_t)0x7d093880, (q31_t)0x7d07e035,
	(q31_t)0x7d06879d, (q31_t)0x7d052eb8,
	(q31_t)0x7d03d585, (q31_t)0x7d027c05, (q31_t)0x7d012239, (q31_t)0x7cffc81f, (q31_t)0x7cfe6db8, (q31_t)0x7cfd1304,
	(q31_t)0x7cfbb803, (q31_t)0x7cfa5cb4,
	(q31_t)0x7cf90119, (q31_t)0x7cf7a531, (q31_t)0x7cf648fb, (q31_t)0x7cf4ec79, (q31_t)0x7cf38fa9, (q31_t)0x7cf2328c,
	(q31_t)0x7cf0d522, (q31_t)0x7cef776b,
	(q31_t)0x7cee1967, (q31_t)0x7cecbb16, (q31_t)0x7ceb5c78, (q31_t)0x7ce9fd8d, (q31_t)0x7ce89e55, (q31_t)0x7ce73ed0,
	(q31_t)0x7ce5defd, (q31_t)0x7ce47ede,
	(q31_t)0x7ce31e72, (q31_t)0x7ce1bdb8, (q31_t)0x7ce05cb2, (q31_t)0x7cdefb5e, (q31_t)0x7cdd99be, (q31_t)0x7cdc37d0,
	(q31_t)0x7cdad596, (q31_t)0x7cd9730e,
	(q31_t)0x7cd8103a, (q31_t)0x7cd6ad18, (q31_t)0x7cd549aa, (q31_t)0x7cd3e5ee, (q31_t)0x7cd281e5, (q31_t)0x7cd11d90,
	(q31_t)0x7ccfb8ed, (q31_t)0x7cce53fe,
	(q31_t)0x7ccceec1, (q31_t)0x7ccb8937, (q31_t)0x7cca2361, (q31_t)0x7cc8bd3d, (q31_t)0x7cc756cd, (q31_t)0x7cc5f010,
	(q31_t)0x7cc48905, (q31_t)0x7cc321ae,
	(q31_t)0x7cc1ba09, (q31_t)0x7cc05218, (q31_t)0x7cbee9da, (q31_t)0x7cbd814f, (q31_t)0x7cbc1877, (q31_t)0x7cbaaf51,
	(q31_t)0x7cb945df, (q31_t)0x7cb7dc20,
	(q31_t)0x7cb67215, (q31_t)0x7cb507bc, (q31_t)0x7cb39d16, (q31_t)0x7cb23223, (q31_t)0x7cb0c6e4, (q31_t)0x7caf5b57,
	(q31_t)0x7cadef7e, (q31_t)0x7cac8358,
	(q31_t)0x7cab16e4, (q31_t)0x7ca9aa24, (q31_t)0x7ca83d17, (q31_t)0x7ca6cfbd, (q31_t)0x7ca56216, (q31_t)0x7ca3f423,
	(q31_t)0x7ca285e2, (q31_t)0x7ca11755,
	(q31_t)0x7c9fa87a, (q31_t)0x7c9e3953, (q31_t)0x7c9cc9df, (q31_t)0x7c9b5a1e, (q31_t)0x7c99ea10, (q31_t)0x7c9879b6,
	(q31_t)0x7c97090e, (q31_t)0x7c95981a,
	(q31_t)0x7c9426d8, (q31_t)0x7c92b54a, (q31_t)0x7c91436f, (q31_t)0x7c8fd148, (q31_t)0x7c8e5ed3, (q31_t)0x7c8cec12,
	(q31_t)0x7c8b7903, (q31_t)0x7c8a05a8,
	(q31_t)0x7c889200, (q31_t)0x7c871e0c, (q31_t)0x7c85a9ca, (q31_t)0x7c84353c, (q31_t)0x7c82c060, (q31_t)0x7c814b39,
	(q31_t)0x7c7fd5c4, (q31_t)0x7c7e6002,
	(q31_t)0x7c7ce9f4, (q31_t)0x7c7b7399, (q31_t)0x7c79fcf1, (q31_t)0x7c7885fc, (q31_t)0x7c770eba, (q31_t)0x7c75972c,
	(q31_t)0x7c741f51, (q31_t)0x7c72a729,
	(q31_t)0x7c712eb5, (q31_t)0x7c6fb5f3, (q31_t)0x7c6e3ce5, (q31_t)0x7c6cc38a, (q31_t)0x7c6b49e3, (q31_t)0x7c69cfee,
	(q31_t)0x7c6855ad, (q31_t)0x7c66db1f,
	(q31_t)0x7c656045, (q31_t)0x7c63e51e, (q31_t)0x7c6269aa, (q31_t)0x7c60ede9, (q31_t)0x7c5f71db, (q31_t)0x7c5df581,
	(q31_t)0x7c5c78da, (q31_t)0x7c5afbe6,
	(q31_t)0x7c597ea6, (q31_t)0x7c580119, (q31_t)0x7c56833f, (q31_t)0x7c550519, (q31_t)0x7c5386a6, (q31_t)0x7c5207e6,
	(q31_t)0x7c5088d9, (q31_t)0x7c4f0980,
	(q31_t)0x7c4d89da, (q31_t)0x7c4c09e8, (q31_t)0x7c4a89a8, (q31_t)0x7c49091c, (q31_t)0x7c478844, (q31_t)0x7c46071f,
	(q31_t)0x7c4485ad, (q31_t)0x7c4303ee,
	(q31_t)0x7c4181e3, (q31_t)0x7c3fff8b, (q31_t)0x7c3e7ce7, (q31_t)0x7c3cf9f5, (q31_t)0x7c3b76b8, (q31_t)0x7c39f32d,
	(q31_t)0x7c386f56, (q31_t)0x7c36eb33,
	(q31_t)0x7c3566c2, (q31_t)0x7c33e205, (q31_t)0x7c325cfc, (q31_t)0x7c30d7a6, (q31_t)0x7c2f5203, (q31_t)0x7c2dcc14,
	(q31_t)0x7c2c45d8, (q31_t)0x7c2abf4f,
	(q31_t)0x7c29387a, (q31_t)0x7c27b158, (q31_t)0x7c2629ea, (q31_t)0x7c24a22f, (q31_t)0x7c231a28, (q31_t)0x7c2191d4,
	(q31_t)0x7c200933, (q31_t)0x7c1e8046,
	(q31_t)0x7c1cf70c, (q31_t)0x7c1b6d86, (q31_t)0x7c19e3b3, (q31_t)0x7c185994, (q31_t)0x7c16cf28, (q31_t)0x7c15446f,
	(q31_t)0x7c13b96a, (q31_t)0x7c122e19,
	(q31_t)0x7c10a27b, (q31_t)0x7c0f1690, (q31_t)0x7c0d8a59, (q31_t)0x7c0bfdd5, (q31_t)0x7c0a7105, (q31_t)0x7c08e3e8,
	(q31_t)0x7c07567f, (q31_t)0x7c05c8c9,
	(q31_t)0x7c043ac7, (q31_t)0x7c02ac78, (q31_t)0x7c011ddd, (q31_t)0x7bff8ef5, (q31_t)0x7bfdffc1, (q31_t)0x7bfc7041,
	(q31_t)0x7bfae073, (q31_t)0x7bf9505a,
	(q31_t)0x7bf7bff4, (q31_t)0x7bf62f41, (q31_t)0x7bf49e42, (q31_t)0x7bf30cf6, (q31_t)0x7bf17b5e, (q31_t)0x7befe97a,
	(q31_t)0x7bee5749, (q31_t)0x7becc4cc,
	(q31_t)0x7beb3202, (q31_t)0x7be99eec, (q31_t)0x7be80b89, (q31_t)0x7be677da, (q31_t)0x7be4e3df, (q31_t)0x7be34f97,
	(q31_t)0x7be1bb02, (q31_t)0x7be02621,
	(q31_t)0x7bde90f4, (q31_t)0x7bdcfb7b, (q31_t)0x7bdb65b5, (q31_t)0x7bd9cfa2, (q31_t)0x7bd83944, (q31_t)0x7bd6a298,
	(q31_t)0x7bd50ba1, (q31_t)0x7bd3745d,
	(q31_t)0x7bd1dccc, (q31_t)0x7bd044f0, (q31_t)0x7bceacc7, (q31_t)0x7bcd1451, (q31_t)0x7bcb7b8f, (q31_t)0x7bc9e281,
	(q31_t)0x7bc84927, (q31_t)0x7bc6af80,
	(q31_t)0x7bc5158c, (q31_t)0x7bc37b4d, (q31_t)0x7bc1e0c1, (q31_t)0x7bc045e9, (q31_t)0x7bbeaac4, (q31_t)0x7bbd0f53,
	(q31_t)0x7bbb7396, (q31_t)0x7bb9d78c,
	(q31_t)0x7bb83b36, (q31_t)0x7bb69e94, (q31_t)0x7bb501a5, (q31_t)0x7bb3646a, (q31_t)0x7bb1c6e3, (q31_t)0x7bb02910,
	(q31_t)0x7bae8af0, (q31_t)0x7bacec84,
	(q31_t)0x7bab4dcc, (q31_t)0x7ba9aec7, (q31_t)0x7ba80f76, (q31_t)0x7ba66fd9, (q31_t)0x7ba4cfef, (q31_t)0x7ba32fba,
	(q31_t)0x7ba18f38, (q31_t)0x7b9fee69,
	(q31_t)0x7b9e4d4f, (q31_t)0x7b9cabe8, (q31_t)0x7b9b0a35, (q31_t)0x7b996836, (q31_t)0x7b97c5ea, (q31_t)0x7b962352,
	(q31_t)0x7b94806e, (q31_t)0x7b92dd3e,
	(q31_t)0x7b9139c2, (q31_t)0x7b8f95f9, (q31_t)0x7b8df1e4, (q31_t)0x7b8c4d83, (q31_t)0x7b8aa8d6, (q31_t)0x7b8903dc,
	(q31_t)0x7b875e96, (q31_t)0x7b85b904,
	(q31_t)0x7b841326, (q31_t)0x7b826cfc, (q31_t)0x7b80c686, (q31_t)0x7b7f1fc3, (q31_t)0x7b7d78b4, (q31_t)0x7b7bd159,
	(q31_t)0x7b7a29b2, (q31_t)0x7b7881be,
	(q31_t)0x7b76d97f, (q31_t)0x7b7530f3, (q31_t)0x7b73881b, (q31_t)0x7b71def7, (q31_t)0x7b703587, (q31_t)0x7b6e8bcb,
	(q31_t)0x7b6ce1c2, (q31_t)0x7b6b376e,
	(q31_t)0x7b698ccd, (q31_t)0x7b67e1e0, (q31_t)0x7b6636a7, (q31_t)0x7b648b22, (q31_t)0x7b62df51, (q31_t)0x7b613334,
	(q31_t)0x7b5f86ca, (q31_t)0x7b5dda15,
	(q31_t)0x7b5c2d13, (q31_t)0x7b5a7fc6, (q31_t)0x7b58d22c, (q31_t)0x7b572446, (q31_t)0x7b557614, (q31_t)0x7b53c796,
	(q31_t)0x7b5218cc, (q31_t)0x7b5069b6,
	(q31_t)0x7b4eba53, (q31_t)0x7b4d0aa5, (q31_t)0x7b4b5aab, (q31_t)0x7b49aa64, (q31_t)0x7b47f9d2, (q31_t)0x7b4648f3,
	(q31_t)0x7b4497c9, (q31_t)0x7b42e652,
	(q31_t)0x7b413490, (q31_t)0x7b3f8281, (q31_t)0x7b3dd026, (q31_t)0x7b3c1d80, (q31_t)0x7b3a6a8d, (q31_t)0x7b38b74e,
	(q31_t)0x7b3703c3, (q31_t)0x7b354fed,
	(q31_t)0x7b339bca, (q31_t)0x7b31e75b, (q31_t)0x7b3032a0, (q31_t)0x7b2e7d9a, (q31_t)0x7b2cc847, (q31_t)0x7b2b12a8,
	(q31_t)0x7b295cbe, (q31_t)0x7b27a687,
	(q31_t)0x7b25f004, (q31_t)0x7b243936, (q31_t)0x7b22821b, (q31_t)0x7b20cab5, (q31_t)0x7b1f1302, (q31_t)0x7b1d5b04,
	(q31_t)0x7b1ba2b9, (q31_t)0x7b19ea23,
	(q31_t)0x7b183141, (q31_t)0x7b167813, (q31_t)0x7b14be99, (q31_t)0x7b1304d3, (q31_t)0x7b114ac1, (q31_t)0x7b0f9063,
	(q31_t)0x7b0dd5b9, (q31_t)0x7b0c1ac4,
	(q31_t)0x7b0a5f82, (q31_t)0x7b08a3f5, (q31_t)0x7b06e81b, (q31_t)0x7b052bf6, (q31_t)0x7b036f85, (q31_t)0x7b01b2c8,
	(q31_t)0x7afff5bf, (q31_t)0x7afe386a,
	(q31_t)0x7afc7aca, (q31_t)0x7afabcdd, (q31_t)0x7af8fea5, (q31_t)0x7af74021, (q31_t)0x7af58151, (q31_t)0x7af3c235,
	(q31_t)0x7af202cd, (q31_t)0x7af0431a,
	(q31_t)0x7aee831a, (q31_t)0x7aecc2cf, (q31_t)0x7aeb0238, (q31_t)0x7ae94155, (q31_t)0x7ae78026, (q31_t)0x7ae5beac,
	(q31_t)0x7ae3fce6, (q31_t)0x7ae23ad4,
	(q31_t)0x7ae07876, (q31_t)0x7adeb5cc, (q31_t)0x7adcf2d6, (q31_t)0x7adb2f95, (q31_t)0x7ad96c08, (q31_t)0x7ad7a82f,
	(q31_t)0x7ad5e40a, (q31_t)0x7ad41f9a,
	(q31_t)0x7ad25ade, (q31_t)0x7ad095d6, (q31_t)0x7aced082, (q31_t)0x7acd0ae3, (q31_t)0x7acb44f8, (q31_t)0x7ac97ec1,
	(q31_t)0x7ac7b83e, (q31_t)0x7ac5f170,
	(q31_t)0x7ac42a55, (q31_t)0x7ac262ef, (q31_t)0x7ac09b3e, (q31_t)0x7abed341, (q31_t)0x7abd0af7, (q31_t)0x7abb4263,
	(q31_t)0x7ab97982, (q31_t)0x7ab7b056,
	(q31_t)0x7ab5e6de, (q31_t)0x7ab41d1b, (q31_t)0x7ab2530b, (q31_t)0x7ab088b0, (q31_t)0x7aaebe0a, (q31_t)0x7aacf318,
	(q31_t)0x7aab27da, (q31_t)0x7aa95c50,
	(q31_t)0x7aa7907b, (q31_t)0x7aa5c45a, (q31_t)0x7aa3f7ed, (q31_t)0x7aa22b35, (q31_t)0x7aa05e31, (q31_t)0x7a9e90e1,
	(q31_t)0x7a9cc346, (q31_t)0x7a9af55f,
	(q31_t)0x7a99272d, (q31_t)0x7a9758af, (q31_t)0x7a9589e5, (q31_t)0x7a93bad0, (q31_t)0x7a91eb6f, (q31_t)0x7a901bc2,
	(q31_t)0x7a8e4bca, (q31_t)0x7a8c7b87,
	(q31_t)0x7a8aaaf7, (q31_t)0x7a88da1c, (q31_t)0x7a8708f6, (q31_t)0x7a853784, (q31_t)0x7a8365c6, (q31_t)0x7a8193bd,
	(q31_t)0x7a7fc168, (q31_t)0x7a7deec8,
	(q31_t)0x7a7c1bdc, (q31_t)0x7a7a48a4, (q31_t)0x7a787521, (q31_t)0x7a76a153, (q31_t)0x7a74cd38, (q31_t)0x7a72f8d3,
	(q31_t)0x7a712422, (q31_t)0x7a6f4f25,
	(q31_t)0x7a6d79dd, (q31_t)0x7a6ba449, (q31_t)0x7a69ce6a, (q31_t)0x7a67f83f, (q31_t)0x7a6621c9, (q31_t)0x7a644b07,
	(q31_t)0x7a6273fa, (q31_t)0x7a609ca1,
	(q31_t)0x7a5ec4fc, (q31_t)0x7a5ced0d, (q31_t)0x7a5b14d1, (q31_t)0x7a593c4b, (q31_t)0x7a576379, (q31_t)0x7a558a5b,
	(q31_t)0x7a53b0f2, (q31_t)0x7a51d73d,
	(q31_t)0x7a4ffd3d, (q31_t)0x7a4e22f2, (q31_t)0x7a4c485b, (q31_t)0x7a4a6d78, (q31_t)0x7a48924b, (q31_t)0x7a46b6d1,
	(q31_t)0x7a44db0d, (q31_t)0x7a42fefd,
	(q31_t)0x7a4122a1, (q31_t)0x7a3f45fa, (q31_t)0x7a3d6908, (q31_t)0x7a3b8bca, (q31_t)0x7a39ae41, (q31_t)0x7a37d06d,
	(q31_t)0x7a35f24d, (q31_t)0x7a3413e2,
	(q31_t)0x7a32352b, (q31_t)0x7a305629, (q31_t)0x7a2e76dc, (q31_t)0x7a2c9743, (q31_t)0x7a2ab75f, (q31_t)0x7a28d72f,
	(q31_t)0x7a26f6b4, (q31_t)0x7a2515ee,
	(q31_t)0x7a2334dd, (q31_t)0x7a215380, (q31_t)0x7a1f71d7, (q31_t)0x7a1d8fe4, (q31_t)0x7a1bada5, (q31_t)0x7a19cb1b,
	(q31_t)0x7a17e845, (q31_t)0x7a160524,
	(q31_t)0x7a1421b8, (q31_t)0x7a123e01, (q31_t)0x7a1059fe, (q31_t)0x7a0e75b0, (q31_t)0x7a0c9117, (q31_t)0x7a0aac32,
	(q31_t)0x7a08c702, (q31_t)0x7a06e187,
	(q31_t)0x7a04fbc1, (q31_t)0x7a0315af, (q31_t)0x7a012f52, (q31_t)0x79ff48aa, (q31_t)0x79fd61b6, (q31_t)0x79fb7a77,
	(q31_t)0x79f992ed, (q31_t)0x79f7ab18,
	(q31_t)0x79f5c2f8, (q31_t)0x79f3da8c, (q31_t)0x79f1f1d5, (q31_t)0x79f008d3, (q31_t)0x79ee1f86, (q31_t)0x79ec35ed,
	(q31_t)0x79ea4c09, (q31_t)0x79e861da,
	(q31_t)0x79e67760, (q31_t)0x79e48c9b, (q31_t)0x79e2a18a, (q31_t)0x79e0b62e, (q31_t)0x79deca87, (q31_t)0x79dcde95,
	(q31_t)0x79daf258, (q31_t)0x79d905d0,
	(q31_t)0x79d718fc, (q31_t)0x79d52bdd, (q31_t)0x79d33e73, (q31_t)0x79d150be, (q31_t)0x79cf62be, (q31_t)0x79cd7473,
	(q31_t)0x79cb85dc, (q31_t)0x79c996fb,
	(q31_t)0x79c7a7ce, (q31_t)0x79c5b856, (q31_t)0x79c3c893, (q31_t)0x79c1d885, (q31_t)0x79bfe82c, (q31_t)0x79bdf788,
	(q31_t)0x79bc0698, (q31_t)0x79ba155e,
	(q31_t)0x79b823d8, (q31_t)0x79b63207, (q31_t)0x79b43fec, (q31_t)0x79b24d85, (q31_t)0x79b05ad3, (q31_t)0x79ae67d6,
	(q31_t)0x79ac748e, (q31_t)0x79aa80fb,
	(q31_t)0x79a88d1d, (q31_t)0x79a698f4, (q31_t)0x79a4a480, (q31_t)0x79a2afc1, (q31_t)0x79a0bab6, (q31_t)0x799ec561,
	(q31_t)0x799ccfc1, (q31_t)0x799ad9d5,
	(q31_t)0x7998e39f, (q31_t)0x7996ed1e, (q31_t)0x7994f651, (q31_t)0x7992ff3a, (q31_t)0x799107d8, (q31_t)0x798f102a,
	(q31_t)0x798d1832, (q31_t)0x798b1fef,
	(q31_t)0x79892761, (q31_t)0x79872e87, (q31_t)0x79853563, (q31_t)0x79833bf4, (q31_t)0x7981423a, (q31_t)0x797f4835,
	(q31_t)0x797d4de5, (q31_t)0x797b534a,
	(q31_t)0x79795864, (q31_t)0x79775d33, (q31_t)0x797561b8, (q31_t)0x797365f1, (q31_t)0x797169df, (q31_t)0x796f6d83,
	(q31_t)0x796d70dc, (q31_t)0x796b73e9,
	(q31_t)0x796976ac, (q31_t)0x79677924, (q31_t)0x79657b51, (q31_t)0x79637d33, (q31_t)0x79617eca, (q31_t)0x795f8017,
	(q31_t)0x795d8118, (q31_t)0x795b81cf,
	(q31_t)0x7959823b, (q31_t)0x7957825c, (q31_t)0x79558232, (q31_t)0x795381bd, (q31_t)0x795180fe, (q31_t)0x794f7ff3,
	(q31_t)0x794d7e9e, (q31_t)0x794b7cfe,
	(q31_t)0x79497b13, (q31_t)0x794778dd, (q31_t)0x7945765d, (q31_t)0x79437391, (q31_t)0x7941707b, (q31_t)0x793f6d1a,
	(q31_t)0x793d696f, (q31_t)0x793b6578,
	(q31_t)0x79396137, (q31_t)0x79375cab, (q31_t)0x793557d4, (q31_t)0x793352b2, (q31_t)0x79314d46, (q31_t)0x792f478f,
	(q31_t)0x792d418d, (q31_t)0x792b3b40,
	(q31_t)0x792934a9, (q31_t)0x79272dc7, (q31_t)0x7925269a, (q31_t)0x79231f22, (q31_t)0x79211760, (q31_t)0x791f0f53,
	(q31_t)0x791d06fb, (q31_t)0x791afe59,
	(q31_t)0x7918f56c, (q31_t)0x7916ec34, (q31_t)0x7914e2b2, (q31_t)0x7912d8e4, (q31_t)0x7910cecc, (q31_t)0x790ec46a,
	(q31_t)0x790cb9bd, (q31_t)0x790aaec5,
	(q31_t)0x7908a382, (q31_t)0x790697f5, (q31_t)0x79048c1d, (q31_t)0x79027ffa, (q31_t)0x7900738d, (q31_t)0x78fe66d5,
	(q31_t)0x78fc59d3, (q31_t)0x78fa4c86,
	(q31_t)0x78f83eee, (q31_t)0x78f6310c, (q31_t)0x78f422df, (q31_t)0x78f21467, (q31_t)0x78f005a5, (q31_t)0x78edf698,
	(q31_t)0x78ebe741, (q31_t)0x78e9d79f,
	(q31_t)0x78e7c7b2, (q31_t)0x78e5b77b, (q31_t)0x78e3a6f9, (q31_t)0x78e1962d, (q31_t)0x78df8516, (q31_t)0x78dd73b5,
	(q31_t)0x78db6209, (q31_t)0x78d95012,
	(q31_t)0x78d73dd1, (q31_t)0x78d52b46, (q31_t)0x78d31870, (q31_t)0x78d1054f, (q31_t)0x78cef1e4, (q31_t)0x78ccde2e,
	(q31_t)0x78caca2e, (q31_t)0x78c8b5e3,
	(q31_t)0x78c6a14e, (q31_t)0x78c48c6e, (q31_t)0x78c27744, (q31_t)0x78c061cf, (q31_t)0x78be4c10, (q31_t)0x78bc3606,
	(q31_t)0x78ba1fb2, (q31_t)0x78b80913,
	(q31_t)0x78b5f22a, (q31_t)0x78b3daf7, (q31_t)0x78b1c379, (q31_t)0x78afabb0, (q31_t)0x78ad939d, (q31_t)0x78ab7b40,
	(q31_t)0x78a96298, (q31_t)0x78a749a6,
	(q31_t)0x78a53069, (q31_t)0x78a316e2, (q31_t)0x78a0fd11, (q31_t)0x789ee2f5, (q31_t)0x789cc88f, (q31_t)0x789aadde,
	(q31_t)0x789892e3, (q31_t)0x7896779d,
	(q31_t)0x78945c0d, (q31_t)0x78924033, (q31_t)0x7890240e, (q31_t)0x788e07a0, (q31_t)0x788beae6, (q31_t)0x7889cde2,
	(q31_t)0x7887b094, (q31_t)0x788592fc,
	(q31_t)0x78837519, (q31_t)0x788156ec, (q31_t)0x787f3875, (q31_t)0x787d19b3, (q31_t)0x787afaa7, (q31_t)0x7878db50,
	(q31_t)0x7876bbb0, (q31_t)0x78749bc5,
	(q31_t)0x78727b8f, (q31_t)0x78705b10, (q31_t)0x786e3a46, (q31_t)0x786c1932, (q31_t)0x7869f7d3, (q31_t)0x7867d62a,
	(q31_t)0x7865b437, (q31_t)0x786391fa,
	(q31_t)0x78616f72, (q31_t)0x785f4ca1, (q31_t)0x785d2984, (q31_t)0x785b061e, (q31_t)0x7858e26e, (q31_t)0x7856be73,
	(q31_t)0x78549a2e, (q31_t)0x7852759e,
	(q31_t)0x785050c5, (q31_t)0x784e2ba1, (q31_t)0x784c0633, (q31_t)0x7849e07b, (q31_t)0x7847ba79, (q31_t)0x7845942c,
	(q31_t)0x78436d96, (q31_t)0x784146b5,
	(q31_t)0x783f1f8a, (q31_t)0x783cf815, (q31_t)0x783ad055, (q31_t)0x7838a84c, (q31_t)0x78367ff8, (q31_t)0x7834575a,
	(q31_t)0x78322e72, (q31_t)0x78300540,
	(q31_t)0x782ddbc4, (q31_t)0x782bb1fd, (q31_t)0x782987ed, (q31_t)0x78275d92, (q31_t)0x782532ed, (q31_t)0x782307fe,
	(q31_t)0x7820dcc5, (q31_t)0x781eb142,
	(q31_t)0x781c8575, (q31_t)0x781a595d, (q31_t)0x78182cfc, (q31_t)0x78160051, (q31_t)0x7813d35b, (q31_t)0x7811a61b,
	(q31_t)0x780f7892, (q31_t)0x780d4abe,
	(q31_t)0x780b1ca0, (q31_t)0x7808ee38, (q31_t)0x7806bf86, (q31_t)0x7804908a, (q31_t)0x78026145, (q31_t)0x780031b5,
	(q31_t)0x77fe01db, (q31_t)0x77fbd1b6,
	(q31_t)0x77f9a148, (q31_t)0x77f77090, (q31_t)0x77f53f8e, (q31_t)0x77f30e42, (q31_t)0x77f0dcac, (q31_t)0x77eeaacc,
	(q31_t)0x77ec78a2, (q31_t)0x77ea462e,
	(q31_t)0x77e81370, (q31_t)0x77e5e068, (q31_t)0x77e3ad17, (q31_t)0x77e1797b, (q31_t)0x77df4595, (q31_t)0x77dd1165,
	(q31_t)0x77dadcec, (q31_t)0x77d8a828,
	(q31_t)0x77d6731a, (q31_t)0x77d43dc3, (q31_t)0x77d20822, (q31_t)0x77cfd236, (q31_t)0x77cd9c01, (q31_t)0x77cb6582,
	(q31_t)0x77c92eb9, (q31_t)0x77c6f7a6,
	(q31_t)0x77c4c04a, (q31_t)0x77c288a3, (q31_t)0x77c050b2, (q31_t)0x77be1878, (q31_t)0x77bbdff4, (q31_t)0x77b9a726,
	(q31_t)0x77b76e0e, (q31_t)0x77b534ac,
	(q31_t)0x77b2fb00, (q31_t)0x77b0c10b, (q31_t)0x77ae86cc, (q31_t)0x77ac4c43, (q31_t)0x77aa1170, (q31_t)0x77a7d653,
	(q31_t)0x77a59aec, (q31_t)0x77a35f3c,
	(q31_t)0x77a12342, (q31_t)0x779ee6fe, (q31_t)0x779caa70, (q31_t)0x779a6d99, (q31_t)0x77983077, (q31_t)0x7795f30c,
	(q31_t)0x7793b557, (q31_t)0x77917759,
	(q31_t)0x778f3910, (q31_t)0x778cfa7e, (q31_t)0x778abba2, (q31_t)0x77887c7d, (q31_t)0x77863d0d, (q31_t)0x7783fd54,
	(q31_t)0x7781bd52, (q31_t)0x777f7d05,
	(q31_t)0x777d3c6f, (q31_t)0x777afb8f, (q31_t)0x7778ba65, (q31_t)0x777678f2, (q31_t)0x77743735, (q31_t)0x7771f52e,
	(q31_t)0x776fb2de, (q31_t)0x776d7044,
	(q31_t)0x776b2d60, (q31_t)0x7768ea33, (q31_t)0x7766a6bc, (q31_t)0x776462fb, (q31_t)0x77621ef1, (q31_t)0x775fda9d,
	(q31_t)0x775d95ff, (q31_t)0x775b5118,
	(q31_t)0x77590be7, (q31_t)0x7756c66c, (q31_t)0x775480a8, (q31_t)0x77523a9b, (q31_t)0x774ff443, (q31_t)0x774dada2,
	(q31_t)0x774b66b8, (q31_t)0x77491f84,
	(q31_t)0x7746d806, (q31_t)0x7744903f, (q31_t)0x7742482e, (q31_t)0x773fffd4, (q31_t)0x773db730, (q31_t)0x773b6e42,
	(q31_t)0x7739250b, (q31_t)0x7736db8b,
	(q31_t)0x773491c0, (q31_t)0x773247ad, (q31_t)0x772ffd50, (q31_t)0x772db2a9, (q31_t)0x772b67b9, (q31_t)0x77291c7f,
	(q31_t)0x7726d0fc, (q31_t)0x7724852f,
	(q31_t)0x77223919, (q31_t)0x771fecb9, (q31_t)0x771da010, (q31_t)0x771b531d, (q31_t)0x771905e1, (q31_t)0x7716b85b,
	(q31_t)0x77146a8c, (q31_t)0x77121c74,
	(q31_t)0x770fce12, (q31_t)0x770d7f66, (q31_t)0x770b3072, (q31_t)0x7708e133, (q31_t)0x770691ab, (q31_t)0x770441da,
	(q31_t)0x7701f1c0, (q31_t)0x76ffa15c,
	(q31_t)0x76fd50ae, (q31_t)0x76faffb8, (q31_t)0x76f8ae78, (q31_t)0x76f65cee, (q31_t)0x76f40b1b, (q31_t)0x76f1b8ff,
	(q31_t)0x76ef6699, (q31_t)0x76ed13ea,
	(q31_t)0x76eac0f2, (q31_t)0x76e86db0, (q31_t)0x76e61a25, (q31_t)0x76e3c650, (q31_t)0x76e17233, (q31_t)0x76df1dcb,
	(q31_t)0x76dcc91b, (q31_t)0x76da7421,
	(q31_t)0x76d81ede, (q31_t)0x76d5c952, (q31_t)0x76d3737c, (q31_t)0x76d11d5d, (q31_t)0x76cec6f5, (q31_t)0x76cc7043,
	(q31_t)0x76ca1948, (q31_t)0x76c7c204,
	(q31_t)0x76c56a77, (q31_t)0x76c312a0, (q31_t)0x76c0ba80, (q31_t)0x76be6217, (q31_t)0x76bc0965, (q31_t)0x76b9b069,
	(q31_t)0x76b75724, (q31_t)0x76b4fd96,
	(q31_t)0x76b2a3bf, (q31_t)0x76b0499e, (q31_t)0x76adef34, (q31_t)0x76ab9481, (q31_t)0x76a93985, (q31_t)0x76a6de40,
	(q31_t)0x76a482b1, (q31_t)0x76a226da,
	(q31_t)0x769fcab9, (q31_t)0x769d6e4f, (q31_t)0x769b119b, (q31_t)0x7698b49f, (q31_t)0x76965759, (q31_t)0x7693f9ca,
	(q31_t)0x76919bf3, (q31_t)0x768f3dd2,
	(q31_t)0x768cdf67, (q31_t)0x768a80b4, (q31_t)0x768821b8, (q31_t)0x7685c272, (q31_t)0x768362e4, (q31_t)0x7681030c,
	(q31_t)0x767ea2eb, (q31_t)0x767c4281,
	(q31_t)0x7679e1ce, (q31_t)0x767780d2, (q31_t)0x76751f8d, (q31_t)0x7672bdfe, (q31_t)0x76705c27, (q31_t)0x766dfa07,
	(q31_t)0x766b979d, (q31_t)0x766934eb,
	(q31_t)0x7666d1ef, (q31_t)0x76646eab, (q31_t)0x76620b1d, (q31_t)0x765fa747, (q31_t)0x765d4327, (q31_t)0x765adebe,
	(q31_t)0x76587a0d, (q31_t)0x76561512,
	(q31_t)0x7653afce, (q31_t)0x76514a42, (q31_t)0x764ee46c, (q31_t)0x764c7e4d, (q31_t)0x764a17e6, (q31_t)0x7647b135,
	(q31_t)0x76454a3c, (q31_t)0x7642e2f9,
	(q31_t)0x76407b6e, (q31_t)0x763e139a, (q31_t)0x763bab7c, (q31_t)0x76394316, (q31_t)0x7636da67, (q31_t)0x7634716f,
	(q31_t)0x7632082e, (q31_t)0x762f9ea4,
	(q31_t)0x762d34d1, (q31_t)0x762acab6, (q31_t)0x76286051, (q31_t)0x7625f5a3, (q31_t)0x76238aad, (q31_t)0x76211f6e,
	(q31_t)0x761eb3e6, (q31_t)0x761c4815,
	(q31_t)0x7619dbfb, (q31_t)0x76176f98, (q31_t)0x761502ed, (q31_t)0x761295f9, (q31_t)0x761028bb, (q31_t)0x760dbb35,
	(q31_t)0x760b4d67, (q31_t)0x7608df4f,
	(q31_t)0x760670ee, (q31_t)0x76040245, (q31_t)0x76019353, (q31_t)0x75ff2418, (q31_t)0x75fcb495, (q31_t)0x75fa44c8,
	(q31_t)0x75f7d4b3, (q31_t)0x75f56455,
	(q31_t)0x75f2f3ae, (q31_t)0x75f082bf, (q31_t)0x75ee1187, (q31_t)0x75eba006, (q31_t)0x75e92e3c, (q31_t)0x75e6bc2a,
	(q31_t)0x75e449ce, (q31_t)0x75e1d72b,
	(q31_t)0x75df643e, (q31_t)0x75dcf109, (q31_t)0x75da7d8b, (q31_t)0x75d809c4, (q31_t)0x75d595b4, (q31_t)0x75d3215c,
	(q31_t)0x75d0acbc, (q31_t)0x75ce37d2,
	(q31_t)0x75cbc2a0, (q31_t)0x75c94d25, (q31_t)0x75c6d762, (q31_t)0x75c46156, (q31_t)0x75c1eb01, (q31_t)0x75bf7464,
	(q31_t)0x75bcfd7e, (q31_t)0x75ba864f,
	(q31_t)0x75b80ed8, (q31_t)0x75b59718, (q31_t)0x75b31f0f, (q31_t)0x75b0a6be, (q31_t)0x75ae2e25, (q31_t)0x75abb542,
	(q31_t)0x75a93c18, (q31_t)0x75a6c2a4,
	(q31_t)0x75a448e8, (q31_t)0x75a1cee4, (q31_t)0x759f5496, (q31_t)0x759cda01, (q31_t)0x759a5f22, (q31_t)0x7597e3fc,
	(q31_t)0x7595688c, (q31_t)0x7592ecd4,
	(q31_t)0x759070d4, (q31_t)0x758df48b, (q31_t)0x758b77fa, (q31_t)0x7588fb20, (q31_t)0x75867dfd, (q31_t)0x75840093,
	(q31_t)0x758182df, (q31_t)0x757f04e3,
	(q31_t)0x757c869f, (q31_t)0x757a0812, (q31_t)0x7577893d, (q31_t)0x75750a1f, (q31_t)0x75728ab9, (q31_t)0x75700b0a,
	(q31_t)0x756d8b13, (q31_t)0x756b0ad3,
	(q31_t)0x75688a4b, (q31_t)0x7566097b, (q31_t)0x75638862, (q31_t)0x75610701, (q31_t)0x755e8557, (q31_t)0x755c0365,
	(q31_t)0x7559812b, (q31_t)0x7556fea8,
	(q31_t)0x75547bdd, (q31_t)0x7551f8c9, (q31_t)0x754f756e, (q31_t)0x754cf1c9, (q31_t)0x754a6ddd, (q31_t)0x7547e9a8,
	(q31_t)0x7545652a, (q31_t)0x7542e065,
	(q31_t)0x75405b57, (q31_t)0x753dd600, (q31_t)0x753b5061, (q31_t)0x7538ca7b, (q31_t)0x7536444b, (q31_t)0x7533bdd4,
	(q31_t)0x75313714, (q31_t)0x752eb00c,
	(q31_t)0x752c28bb, (q31_t)0x7529a122, (q31_t)0x75271941, (q31_t)0x75249118, (q31_t)0x752208a7, (q31_t)0x751f7fed,
	(q31_t)0x751cf6eb, (q31_t)0x751a6da0,
	(q31_t)0x7517e40e, (q31_t)0x75155a33, (q31_t)0x7512d010, (q31_t)0x751045a5, (q31_t)0x750dbaf2, (q31_t)0x750b2ff6,
	(q31_t)0x7508a4b2, (q31_t)0x75061926,
	(q31_t)0x75038d52, (q31_t)0x75010136, (q31_t)0x74fe74d1, (q31_t)0x74fbe825, (q31_t)0x74f95b30, (q31_t)0x74f6cdf3,
	(q31_t)0x74f4406d, (q31_t)0x74f1b2a0,
	(q31_t)0x74ef248b, (q31_t)0x74ec962d, (q31_t)0x74ea0787, (q31_t)0x74e7789a, (q31_t)0x74e4e964, (q31_t)0x74e259e6,
	(q31_t)0x74dfca20, (q31_t)0x74dd3a11,
	(q31_t)0x74daa9bb, (q31_t)0x74d8191d, (q31_t)0x74d58836, (q31_t)0x74d2f708, (q31_t)0x74d06591, (q31_t)0x74cdd3d2,
	(q31_t)0x74cb41cc, (q31_t)0x74c8af7d,
	(q31_t)0x74c61ce6, (q31_t)0x74c38a07, (q31_t)0x74c0f6e0, (q31_t)0x74be6372, (q31_t)0x74bbcfbb, (q31_t)0x74b93bbc,
	(q31_t)0x74b6a775, (q31_t)0x74b412e6,
	(q31_t)0x74b17e0f, (q31_t)0x74aee8f0, (q31_t)0x74ac5389, (q31_t)0x74a9bddb, (q31_t)0x74a727e4, (q31_t)0x74a491a5,
	(q31_t)0x74a1fb1e, (q31_t)0x749f6450,
	(q31_t)0x749ccd39, (q31_t)0x749a35db, (q31_t)0x74979e34, (q31_t)0x74950646, (q31_t)0x74926e10, (q31_t)0x748fd592,
	(q31_t)0x748d3ccb, (q31_t)0x748aa3be,
	(q31_t)0x74880a68, (q31_t)0x748570ca, (q31_t)0x7482d6e4, (q31_t)0x74803cb7, (q31_t)0x747da242, (q31_t)0x747b0784,
	(q31_t)0x74786c7f, (q31_t)0x7475d132,
	(q31_t)0x7473359e, (q31_t)0x747099c1, (q31_t)0x746dfd9d, (q31_t)0x746b6131, (q31_t)0x7468c47c, (q31_t)0x74662781,
	(q31_t)0x74638a3d, (q31_t)0x7460ecb2,
	(q31_t)0x745e4ede, (q31_t)0x745bb0c3, (q31_t)0x74591261, (q31_t)0x745673b6, (q31_t)0x7453d4c4, (q31_t)0x7451358a,
	(q31_t)0x744e9608, (q31_t)0x744bf63e,
	(q31_t)0x7449562d, (q31_t)0x7446b5d4, (q31_t)0x74441533, (q31_t)0x7441744b, (q31_t)0x743ed31b, (q31_t)0x743c31a3,
	(q31_t)0x74398fe3, (q31_t)0x7436eddc,
	(q31_t)0x74344b8d, (q31_t)0x7431a8f6, (q31_t)0x742f0618, (q31_t)0x742c62f2, (q31_t)0x7429bf84, (q31_t)0x74271bcf,
	(q31_t)0x742477d2, (q31_t)0x7421d38e,
	(q31_t)0x741f2f01, (q31_t)0x741c8a2d, (q31_t)0x7419e512, (q31_t)0x74173faf, (q31_t)0x74149a04, (q31_t)0x7411f412,
	(q31_t)0x740f4dd8, (q31_t)0x740ca756,
	(q31_t)0x740a008d, (q31_t)0x7407597d, (q31_t)0x7404b224, (q31_t)0x74020a85, (q31_t)0x73ff629d, (q31_t)0x73fcba6e,
	(q31_t)0x73fa11f8, (q31_t)0x73f7693a,
	(q31_t)0x73f4c034, (q31_t)0x73f216e7, (q31_t)0x73ef6d53, (q31_t)0x73ecc377, (q31_t)0x73ea1953, (q31_t)0x73e76ee8,
	(q31_t)0x73e4c435, (q31_t)0x73e2193b,
	(q31_t)0x73df6df9, (q31_t)0x73dcc270, (q31_t)0x73da16a0, (q31_t)0x73d76a88, (q31_t)0x73d4be28, (q31_t)0x73d21182,
	(q31_t)0x73cf6493, (q31_t)0x73ccb75d,
	(q31_t)0x73ca09e0, (q31_t)0x73c75c1c, (q31_t)0x73c4ae10, (q31_t)0x73c1ffbc, (q31_t)0x73bf5121, (q31_t)0x73bca23f,
	(q31_t)0x73b9f315, (q31_t)0x73b743a4,
	(q31_t)0x73b493ec, (q31_t)0x73b1e3ec, (q31_t)0x73af33a5, (q31_t)0x73ac8316, (q31_t)0x73a9d240, (q31_t)0x73a72123,
	(q31_t)0x73a46fbf, (q31_t)0x73a1be13,
	(q31_t)0x739f0c20, (q31_t)0x739c59e5, (q31_t)0x7399a763, (q31_t)0x7396f49a, (q31_t)0x73944189, (q31_t)0x73918e32,
	(q31_t)0x738eda93, (q31_t)0x738c26ac,
	(q31_t)0x7389727f, (q31_t)0x7386be0a, (q31_t)0x7384094e, (q31_t)0x7381544a, (q31_t)0x737e9f00, (q31_t)0x737be96e,
	(q31_t)0x73793395, (q31_t)0x73767d74,
	(q31_t)0x7373c70d, (q31_t)0x7371105e, (q31_t)0x736e5968, (q31_t)0x736ba22b, (q31_t)0x7368eaa6, (q31_t)0x736632db,
	(q31_t)0x73637ac8, (q31_t)0x7360c26e,
	(q31_t)0x735e09cd, (q31_t)0x735b50e4, (q31_t)0x735897b5, (q31_t)0x7355de3e, (q31_t)0x73532481, (q31_t)0x73506a7c,
	(q31_t)0x734db030, (q31_t)0x734af59d,
	(q31_t)0x73483ac2, (q31_t)0x73457fa1, (q31_t)0x7342c438, (q31_t)0x73400889, (q31_t)0x733d4c92, (q31_t)0x733a9054,
	(q31_t)0x7337d3d0, (q31_t)0x73351704,
	(q31_t)0x733259f1, (q31_t)0x732f9c97, (q31_t)0x732cdef6, (q31_t)0x732a210d, (q31_t)0x732762de, (q31_t)0x7324a468,
	(q31_t)0x7321e5ab, (q31_t)0x731f26a7,
	(q31_t)0x731c675b, (q31_t)0x7319a7c9, (q31_t)0x7316e7f0, (q31_t)0x731427cf, (q31_t)0x73116768, (q31_t)0x730ea6ba,
	(q31_t)0x730be5c5, (q31_t)0x73092489,
	(q31_t)0x73066306, (q31_t)0x7303a13b, (q31_t)0x7300df2a, (q31_t)0x72fe1cd2, (q31_t)0x72fb5a34, (q31_t)0x72f8974e,
	(q31_t)0x72f5d421, (q31_t)0x72f310ad,
	(q31_t)0x72f04cf3, (q31_t)0x72ed88f1, (q31_t)0x72eac4a9, (q31_t)0x72e8001a, (q31_t)0x72e53b44, (q31_t)0x72e27627,
	(q31_t)0x72dfb0c3, (q31_t)0x72dceb18,
	(q31_t)0x72da2526, (q31_t)0x72d75eee, (q31_t)0x72d4986f, (q31_t)0x72d1d1a9, (q31_t)0x72cf0a9c, (q31_t)0x72cc4348,
	(q31_t)0x72c97bad, (q31_t)0x72c6b3cc,
	(q31_t)0x72c3eba4, (q31_t)0x72c12335, (q31_t)0x72be5a7f, (q31_t)0x72bb9183, (q31_t)0x72b8c83f, (q31_t)0x72b5feb5,
	(q31_t)0x72b334e4, (q31_t)0x72b06acd,
	(q31_t)0x72ada06f, (q31_t)0x72aad5c9, (q31_t)0x72a80ade, (q31_t)0x72a53fab, (q31_t)0x72a27432, (q31_t)0x729fa872,
	(q31_t)0x729cdc6b, (q31_t)0x729a101e,
	(q31_t)0x7297438a, (q31_t)0x729476af, (q31_t)0x7291a98e, (q31_t)0x728edc26, (q31_t)0x728c0e77, (q31_t)0x72894082,
	(q31_t)0x72867245, (q31_t)0x7283a3c3,
	(q31_t)0x7280d4f9, (q31_t)0x727e05e9, (q31_t)0x727b3693, (q31_t)0x727866f6, (q31_t)0x72759712, (q31_t)0x7272c6e7,
	(q31_t)0x726ff676, (q31_t)0x726d25bf,
	(q31_t)0x726a54c1, (q31_t)0x7267837c, (q31_t)0x7264b1f0, (q31_t)0x7261e01e, (q31_t)0x725f0e06, (q31_t)0x725c3ba7,
	(q31_t)0x72596901, (q31_t)0x72569615,
	(q31_t)0x7253c2e3, (q31_t)0x7250ef6a, (q31_t)0x724e1baa, (q31_t)0x724b47a4, (q31_t)0x72487357, (q31_t)0x72459ec4,
	(q31_t)0x7242c9ea, (q31_t)0x723ff4ca,
	(q31_t)0x723d1f63, (q31_t)0x723a49b6, (q31_t)0x723773c3, (q31_t)0x72349d89, (q31_t)0x7231c708, (q31_t)0x722ef041,
	(q31_t)0x722c1934, (q31_t)0x722941e0,
	(q31_t)0x72266a46, (q31_t)0x72239266, (q31_t)0x7220ba3f, (q31_t)0x721de1d1, (q31_t)0x721b091d, (q31_t)0x72183023,
	(q31_t)0x721556e3, (q31_t)0x72127d5c,
	(q31_t)0x720fa38e, (q31_t)0x720cc97b, (q31_t)0x7209ef21, (q31_t)0x72071480, (q31_t)0x7204399a, (q31_t)0x72015e6d,
	(q31_t)0x71fe82f9, (q31_t)0x71fba740,
	(q31_t)0x71f8cb40, (q31_t)0x71f5eefa, (q31_t)0x71f3126d, (q31_t)0x71f0359a, (q31_t)0x71ed5881, (q31_t)0x71ea7b22,
	(q31_t)0x71e79d7c, (q31_t)0x71e4bf90,
	(q31_t)0x71e1e15e, (q31_t)0x71df02e5, (q31_t)0x71dc2427, (q31_t)0x71d94522, (q31_t)0x71d665d6, (q31_t)0x71d38645,
	(q31_t)0x71d0a66d, (q31_t)0x71cdc650,
	(q31_t)0x71cae5ec, (q31_t)0x71c80542, (q31_t)0x71c52451, (q31_t)0x71c2431b, (q31_t)0x71bf619e, (q31_t)0x71bc7fdb,
	(q31_t)0x71b99dd2, (q31_t)0x71b6bb83,
	(q31_t)0x71b3d8ed, (q31_t)0x71b0f612, (q31_t)0x71ae12f0, (q31_t)0x71ab2f89, (q31_t)0x71a84bdb, (q31_t)0x71a567e7,
	(q31_t)0x71a283ad, (q31_t)0x719f9f2c,
	(q31_t)0x719cba66, (q31_t)0x7199d55a, (q31_t)0x7196f008, (q31_t)0x71940a6f, (q31_t)0x71912490, (q31_t)0x718e3e6c,
	(q31_t)0x718b5801, (q31_t)0x71887151,
	(q31_t)0x71858a5a, (q31_t)0x7182a31d, (q31_t)0x717fbb9a, (q31_t)0x717cd3d2, (q31_t)0x7179ebc3, (q31_t)0x7177036e,
	(q31_t)0x71741ad3, (q31_t)0x717131f3,
	(q31_t)0x716e48cc, (q31_t)0x716b5f5f, (q31_t)0x716875ad, (q31_t)0x71658bb4, (q31_t)0x7162a175, (q31_t)0x715fb6f1,
	(q31_t)0x715ccc26, (q31_t)0x7159e116,
	(q31_t)0x7156f5c0, (q31_t)0x71540a24, (q31_t)0x71511e42, (q31_t)0x714e321a, (q31_t)0x714b45ac, (q31_t)0x714858f8,
	(q31_t)0x71456bfe, (q31_t)0x71427ebf,
	(q31_t)0x713f9139, (q31_t)0x713ca36e, (q31_t)0x7139b55d, (q31_t)0x7136c706, (q31_t)0x7133d869, (q31_t)0x7130e987,
	(q31_t)0x712dfa5e, (q31_t)0x712b0af0,
	(q31_t)0x71281b3c, (q31_t)0x71252b42, (q31_t)0x71223b02, (q31_t)0x711f4a7d, (q31_t)0x711c59b2, (q31_t)0x711968a1,
	(q31_t)0x7116774a, (q31_t)0x711385ad,
	(q31_t)0x711093cb, (q31_t)0x710da1a3, (q31_t)0x710aaf35, (q31_t)0x7107bc82, (q31_t)0x7104c989, (q31_t)0x7101d64a,
	(q31_t)0x70fee2c5, (q31_t)0x70fbeefb,
	(q31_t)0x70f8faeb, (q31_t)0x70f60695, (q31_t)0x70f311fa, (q31_t)0x70f01d19, (q31_t)0x70ed27f2, (q31_t)0x70ea3286,
	(q31_t)0x70e73cd4, (q31_t)0x70e446dc,
	(q31_t)0x70e1509f, (q31_t)0x70de5a1c, (q31_t)0x70db6353, (q31_t)0x70d86c45, (q31_t)0x70d574f1, (q31_t)0x70d27d58,
	(q31_t)0x70cf8579, (q31_t)0x70cc8d54,
	(q31_t)0x70c994ea, (q31_t)0x70c69c3a, (q31_t)0x70c3a345, (q31_t)0x70c0aa0a, (q31_t)0x70bdb08a, (q31_t)0x70bab6c4,
	(q31_t)0x70b7bcb8, (q31_t)0x70b4c267,
	(q31_t)0x70b1c7d1, (q31_t)0x70aeccf5, (q31_t)0x70abd1d3, (q31_t)0x70a8d66c, (q31_t)0x70a5dac0, (q31_t)0x70a2dece,
	(q31_t)0x709fe296, (q31_t)0x709ce619,
	(q31_t)0x7099e957, (q31_t)0x7096ec4f, (q31_t)0x7093ef01, (q31_t)0x7090f16e, (q31_t)0x708df396, (q31_t)0x708af579,
	(q31_t)0x7087f715, (q31_t)0x7084f86d,
	(q31_t)0x7081f97f, (q31_t)0x707efa4c, (q31_t)0x707bfad3, (q31_t)0x7078fb15, (q31_t)0x7075fb11, (q31_t)0x7072fac9,
	(q31_t)0x706ffa3a, (q31_t)0x706cf967,
	(q31_t)0x7069f84e, (q31_t)0x7066f6f0, (q31_t)0x7063f54c, (q31_t)0x7060f363, (q31_t)0x705df135, (q31_t)0x705aeec1,
	(q31_t)0x7057ec08, (q31_t)0x7054e90a,
	(q31_t)0x7051e5c7, (q31_t)0x704ee23e, (q31_t)0x704bde70, (q31_t)0x7048da5d, (q31_t)0x7045d604, (q31_t)0x7042d166,
	(q31_t)0x703fcc83, (q31_t)0x703cc75b,
	(q31_t)0x7039c1ed, (q31_t)0x7036bc3b, (q31_t)0x7033b643, (q31_t)0x7030b005, (q31_t)0x702da983, (q31_t)0x702aa2bb,
	(q31_t)0x70279baf, (q31_t)0x7024945d,
	(q31_t)0x70218cc6, (q31_t)0x701e84e9, (q31_t)0x701b7cc8, (q31_t)0x70187461, (q31_t)0x70156bb5, (q31_t)0x701262c4,
	(q31_t)0x700f598e, (q31_t)0x700c5013,
	(q31_t)0x70094653, (q31_t)0x70063c4e, (q31_t)0x70033203, (q31_t)0x70002774, (q31_t)0x6ffd1c9f, (q31_t)0x6ffa1185,
	(q31_t)0x6ff70626, (q31_t)0x6ff3fa82,
	(q31_t)0x6ff0ee99, (q31_t)0x6fede26b, (q31_t)0x6fead5f8, (q31_t)0x6fe7c940, (q31_t)0x6fe4bc43, (q31_t)0x6fe1af01,
	(q31_t)0x6fdea17a, (q31_t)0x6fdb93ae,
	(q31_t)0x6fd8859d, (q31_t)0x6fd57746, (q31_t)0x6fd268ab, (q31_t)0x6fcf59cb, (q31_t)0x6fcc4aa6, (q31_t)0x6fc93b3c,
	(q31_t)0x6fc62b8d, (q31_t)0x6fc31b99,
	(q31_t)0x6fc00b60, (q31_t)0x6fbcfae2, (q31_t)0x6fb9ea20, (q31_t)0x6fb6d918, (q31_t)0x6fb3c7cb, (q31_t)0x6fb0b63a,
	(q31_t)0x6fada464, (q31_t)0x6faa9248,
	(q31_t)0x6fa77fe8, (q31_t)0x6fa46d43, (q31_t)0x6fa15a59, (q31_t)0x6f9e472b, (q31_t)0x6f9b33b7, (q31_t)0x6f981fff,
	(q31_t)0x6f950c01, (q31_t)0x6f91f7bf,
	(q31_t)0x6f8ee338, (q31_t)0x6f8bce6c, (q31_t)0x6f88b95c, (q31_t)0x6f85a407, (q31_t)0x6f828e6c, (q31_t)0x6f7f788d,
	(q31_t)0x6f7c626a, (q31_t)0x6f794c01,
	(q31_t)0x6f763554, (q31_t)0x6f731e62, (q31_t)0x6f70072b, (q31_t)0x6f6cefb0, (q31_t)0x6f69d7f0, (q31_t)0x6f66bfeb,
	(q31_t)0x6f63a7a1, (q31_t)0x6f608f13,
	(q31_t)0x6f5d7640, (q31_t)0x6f5a5d28, (q31_t)0x6f5743cb, (q31_t)0x6f542a2a, (q31_t)0x6f511044, (q31_t)0x6f4df61a,
	(q31_t)0x6f4adbab, (q31_t)0x6f47c0f7,
	(q31_t)0x6f44a5ff, (q31_t)0x6f418ac2, (q31_t)0x6f3e6f40, (q31_t)0x6f3b537a, (q31_t)0x6f38376f, (q31_t)0x6f351b1f,
	(q31_t)0x6f31fe8b, (q31_t)0x6f2ee1b2,
	(q31_t)0x6f2bc495, (q31_t)0x6f28a733, (q31_t)0x6f25898d, (q31_t)0x6f226ba2, (q31_t)0x6f1f4d72, (q31_t)0x6f1c2efe,
	(q31_t)0x6f191045, (q31_t)0x6f15f148,
	(q31_t)0x6f12d206, (q31_t)0x6f0fb280, (q31_t)0x6f0c92b6, (q31_t)0x6f0972a6, (q31_t)0x6f065253, (q31_t)0x6f0331ba,
	(q31_t)0x6f0010de, (q31_t)0x6efcefbd,
	(q31_t)0x6ef9ce57, (q31_t)0x6ef6acad, (q31_t)0x6ef38abe, (q31_t)0x6ef0688b, (q31_t)0x6eed4614, (q31_t)0x6eea2358,
	(q31_t)0x6ee70058, (q31_t)0x6ee3dd13,
	(q31_t)0x6ee0b98a, (q31_t)0x6edd95bd, (q31_t)0x6eda71ab, (q31_t)0x6ed74d55, (q31_t)0x6ed428ba, (q31_t)0x6ed103db,
	(q31_t)0x6ecddeb8, (q31_t)0x6ecab950,
	(q31_t)0x6ec793a4, (q31_t)0x6ec46db4, (q31_t)0x6ec1477f, (q31_t)0x6ebe2106, (q31_t)0x6ebafa49, (q31_t)0x6eb7d347,
	(q31_t)0x6eb4ac02, (q31_t)0x6eb18477,
	(q31_t)0x6eae5ca9, (q31_t)0x6eab3496, (q31_t)0x6ea80c3f, (q31_t)0x6ea4e3a4, (q31_t)0x6ea1bac4, (q31_t)0x6e9e91a1,
	(q31_t)0x6e9b6839, (q31_t)0x6e983e8d,
	(q31_t)0x6e95149c, (q31_t)0x6e91ea67, (q31_t)0x6e8ebfef, (q31_t)0x6e8b9532, (q31_t)0x6e886a30, (q31_t)0x6e853eeb,
	(q31_t)0x6e821361, (q31_t)0x6e7ee794,
	(q31_t)0x6e7bbb82, (q31_t)0x6e788f2c, (q31_t)0x6e756291, (q31_t)0x6e7235b3, (q31_t)0x6e6f0890, (q31_t)0x6e6bdb2a,
	(q31_t)0x6e68ad7f, (q31_t)0x6e657f90,
	(q31_t)0x6e62515d, (q31_t)0x6e5f22e6, (q31_t)0x6e5bf42b, (q31_t)0x6e58c52c, (q31_t)0x6e5595e9, (q31_t)0x6e526662,
	(q31_t)0x6e4f3696, (q31_t)0x6e4c0687,
	(q31_t)0x6e48d633, (q31_t)0x6e45a59c, (q31_t)0x6e4274c1, (q31_t)0x6e3f43a1, (q31_t)0x6e3c123e, (q31_t)0x6e38e096,
	(q31_t)0x6e35aeab, (q31_t)0x6e327c7b,
	(q31_t)0x6e2f4a08, (q31_t)0x6e2c1750, (q31_t)0x6e28e455, (q31_t)0x6e25b115, (q31_t)0x6e227d92, (q31_t)0x6e1f49cb,
	(q31_t)0x6e1c15c0, (q31_t)0x6e18e171,
	(q31_t)0x6e15acde, (q31_t)0x6e127807, (q31_t)0x6e0f42ec, (q31_t)0x6e0c0d8e, (q31_t)0x6e08d7eb, (q31_t)0x6e05a205,
	(q31_t)0x6e026bda, (q31_t)0x6dff356c,
	(q31_t)0x6dfbfeba, (q31_t)0x6df8c7c4, (q31_t)0x6df5908b, (q31_t)0x6df2590d, (q31_t)0x6def214c, (q31_t)0x6debe947,
	(q31_t)0x6de8b0fe, (q31_t)0x6de57871,
	(q31_t)0x6de23fa0, (q31_t)0x6ddf068c, (q31_t)0x6ddbcd34, (q31_t)0x6dd89398, (q31_t)0x6dd559b9, (q31_t)0x6dd21f95,
	(q31_t)0x6dcee52e, (q31_t)0x6dcbaa83,
	(q31_t)0x6dc86f95, (q31_t)0x6dc53462, (q31_t)0x6dc1f8ec, (q31_t)0x6dbebd33, (q31_t)0x6dbb8135, (q31_t)0x6db844f4,
	(q31_t)0x6db5086f, (q31_t)0x6db1cba7,
	(q31_t)0x6dae8e9b, (q31_t)0x6dab514b, (q31_t)0x6da813b8, (q31_t)0x6da4d5e1, (q31_t)0x6da197c6, (q31_t)0x6d9e5968,
	(q31_t)0x6d9b1ac6, (q31_t)0x6d97dbe0,
	(q31_t)0x6d949cb7, (q31_t)0x6d915d4a, (q31_t)0x6d8e1d9a, (q31_t)0x6d8adda6, (q31_t)0x6d879d6e, (q31_t)0x6d845cf3,
	(q31_t)0x6d811c35, (q31_t)0x6d7ddb33,
	(q31_t)0x6d7a99ed, (q31_t)0x6d775864, (q31_t)0x6d741697, (q31_t)0x6d70d487, (q31_t)0x6d6d9233, (q31_t)0x6d6a4f9c,
	(q31_t)0x6d670cc1, (q31_t)0x6d63c9a3,
	(q31_t)0x6d608641, (q31_t)0x6d5d429c, (q31_t)0x6d59feb3, (q31_t)0x6d56ba87, (q31_t)0x6d537617, (q31_t)0x6d503164,
	(q31_t)0x6d4cec6e, (q31_t)0x6d49a734,
	(q31_t)0x6d4661b7, (q31_t)0x6d431bf6, (q31_t)0x6d3fd5f2, (q31_t)0x6d3c8fab, (q31_t)0x6d394920, (q31_t)0x6d360252,
	(q31_t)0x6d32bb40, (q31_t)0x6d2f73eb,
	(q31_t)0x6d2c2c53, (q31_t)0x6d28e477, (q31_t)0x6d259c58, (q31_t)0x6d2253f6, (q31_t)0x6d1f0b50, (q31_t)0x6d1bc267,
	(q31_t)0x6d18793b, (q31_t)0x6d152fcc,
	(q31_t)0x6d11e619, (q31_t)0x6d0e9c23, (q31_t)0x6d0b51e9, (q31_t)0x6d08076d, (q31_t)0x6d04bcad, (q31_t)0x6d0171aa,
	(q31_t)0x6cfe2663, (q31_t)0x6cfadada,
	(q31_t)0x6cf78f0d, (q31_t)0x6cf442fd, (q31_t)0x6cf0f6aa, (q31_t)0x6cedaa13, (q31_t)0x6cea5d3a, (q31_t)0x6ce7101d,
	(q31_t)0x6ce3c2bd, (q31_t)0x6ce0751a,
	(q31_t)0x6cdd2733, (q31_t)0x6cd9d90a, (q31_t)0x6cd68a9d, (q31_t)0x6cd33bed, (q31_t)0x6ccfecfa, (q31_t)0x6ccc9dc4,
	(q31_t)0x6cc94e4b, (q31_t)0x6cc5fe8f,
	(q31_t)0x6cc2ae90, (q31_t)0x6cbf5e4d, (q31_t)0x6cbc0dc8, (q31_t)0x6cb8bcff, (q31_t)0x6cb56bf4, (q31_t)0x6cb21aa5,
	(q31_t)0x6caec913, (q31_t)0x6cab773e,
	(q31_t)0x6ca82527, (q31_t)0x6ca4d2cc, (q31_t)0x6ca1802e, (q31_t)0x6c9e2d4d, (q31_t)0x6c9ada29, (q31_t)0x6c9786c2,
	(q31_t)0x6c943318, (q31_t)0x6c90df2c,
	(q31_t)0x6c8d8afc, (q31_t)0x6c8a3689, (q31_t)0x6c86e1d3, (q31_t)0x6c838cdb, (q31_t)0x6c80379f, (q31_t)0x6c7ce220,
	(q31_t)0x6c798c5f, (q31_t)0x6c76365b,
	(q31_t)0x6c72e013, (q31_t)0x6c6f8989, (q31_t)0x6c6c32bc, (q31_t)0x6c68dbac, (q31_t)0x6c658459, (q31_t)0x6c622cc4,
	(q31_t)0x6c5ed4eb, (q31_t)0x6c5b7cd0,
	(q31_t)0x6c582472, (q31_t)0x6c54cbd1, (q31_t)0x6c5172ed, (q31_t)0x6c4e19c6, (q31_t)0x6c4ac05d, (q31_t)0x6c4766b0,
	(q31_t)0x6c440cc1, (q31_t)0x6c40b28f,
	(q31_t)0x6c3d581b, (q31_t)0x6c39fd63, (q31_t)0x6c36a269, (q31_t)0x6c33472c, (q31_t)0x6c2febad, (q31_t)0x6c2c8fea,
	(q31_t)0x6c2933e5, (q31_t)0x6c25d79d,
	(q31_t)0x6c227b13, (q31_t)0x6c1f1e45, (q31_t)0x6c1bc136, (q31_t)0x6c1863e3, (q31_t)0x6c15064e, (q31_t)0x6c11a876,
	(q31_t)0x6c0e4a5b, (q31_t)0x6c0aebfe,
	(q31_t)0x6c078d5e, (q31_t)0x6c042e7b, (q31_t)0x6c00cf56, (q31_t)0x6bfd6fee, (q31_t)0x6bfa1044, (q31_t)0x6bf6b056,
	(q31_t)0x6bf35027, (q31_t)0x6befefb5,
	(q31_t)0x6bec8f00, (q31_t)0x6be92e08, (q31_t)0x6be5ccce, (q31_t)0x6be26b52, (q31_t)0x6bdf0993, (q31_t)0x6bdba791,
	(q31_t)0x6bd8454d, (q31_t)0x6bd4e2c6,
	(q31_t)0x6bd17ffd, (q31_t)0x6bce1cf1, (q31_t)0x6bcab9a3, (q31_t)0x6bc75613, (q31_t)0x6bc3f23f, (q31_t)0x6bc08e2a,
	(q31_t)0x6bbd29d2, (q31_t)0x6bb9c537,
	(q31_t)0x6bb6605a, (q31_t)0x6bb2fb3b, (q31_t)0x6baf95d9, (q31_t)0x6bac3034, (q31_t)0x6ba8ca4e, (q31_t)0x6ba56425,
	(q31_t)0x6ba1fdb9, (q31_t)0x6b9e970b,
	(q31_t)0x6b9b301b, (q31_t)0x6b97c8e8, (q31_t)0x6b946173, (q31_t)0x6b90f9bc, (q31_t)0x6b8d91c2, (q31_t)0x6b8a2986,
	(q31_t)0x6b86c107, (q31_t)0x6b835846,
	(q31_t)0x6b7fef43, (q31_t)0x6b7c85fe, (q31_t)0x6b791c76, (q31_t)0x6b75b2ac, (q31_t)0x6b7248a0, (q31_t)0x6b6ede51,
	(q31_t)0x6b6b73c0, (q31_t)0x6b6808ed,
	(q31_t)0x6b649dd8, (q31_t)0x6b613280, (q31_t)0x6b5dc6e6, (q31_t)0x6b5a5b0a, (q31_t)0x6b56eeec, (q31_t)0x6b53828b,
	(q31_t)0x6b5015e9, (q31_t)0x6b4ca904,
	(q31_t)0x6b493bdd, (q31_t)0x6b45ce73, (q31_t)0x6b4260c8, (q31_t)0x6b3ef2da, (q31_t)0x6b3b84ab, (q31_t)0x6b381639,
	(q31_t)0x6b34a785, (q31_t)0x6b31388e,
	(q31_t)0x6b2dc956, (q31_t)0x6b2a59dc, (q31_t)0x6b26ea1f, (q31_t)0x6b237a21, (q31_t)0x6b2009e0, (q31_t)0x6b1c995d,
	(q31_t)0x6b192898, (q31_t)0x6b15b791,
	(q31_t)0x6b124648, (q31_t)0x6b0ed4bd, (q31_t)0x6b0b62f0, (q31_t)0x6b07f0e1, (q31_t)0x6b047e90, (q31_t)0x6b010bfd,
	(q31_t)0x6afd9928, (q31_t)0x6afa2610,
	(q31_t)0x6af6b2b7, (q31_t)0x6af33f1c, (q31_t)0x6aefcb3f, (q31_t)0x6aec5720, (q31_t)0x6ae8e2bf, (q31_t)0x6ae56e1c,
	(q31_t)0x6ae1f937, (q31_t)0x6ade8410,
	(q31_t)0x6adb0ea8, (q31_t)0x6ad798fd, (q31_t)0x6ad42311, (q31_t)0x6ad0ace2, (q31_t)0x6acd3672, (q31_t)0x6ac9bfc0,
	(q31_t)0x6ac648cb, (q31_t)0x6ac2d195,
	(q31_t)0x6abf5a1e, (q31_t)0x6abbe264, (q31_t)0x6ab86a68, (q31_t)0x6ab4f22b, (q31_t)0x6ab179ac, (q31_t)0x6aae00eb,
	(q31_t)0x6aaa87e8, (q31_t)0x6aa70ea4,
	(q31_t)0x6aa3951d, (q31_t)0x6aa01b55, (q31_t)0x6a9ca14b, (q31_t)0x6a992700, (q31_t)0x6a95ac72, (q31_t)0x6a9231a3,
	(q31_t)0x6a8eb692, (q31_t)0x6a8b3b3f,
	(q31_t)0x6a87bfab, (q31_t)0x6a8443d5, (q31_t)0x6a80c7bd, (q31_t)0x6a7d4b64, (q31_t)0x6a79cec8, (q31_t)0x6a7651ec,
	(q31_t)0x6a72d4cd, (q31_t)0x6a6f576d,
	(q31_t)0x6a6bd9cb, (q31_t)0x6a685be8, (q31_t)0x6a64ddc2, (q31_t)0x6a615f5c, (q31_t)0x6a5de0b3, (q31_t)0x6a5a61c9,
	(q31_t)0x6a56e29e, (q31_t)0x6a536331,
	(q31_t)0x6a4fe382, (q31_t)0x6a4c6391, (q31_t)0x6a48e360, (q31_t)0x6a4562ec, (q31_t)0x6a41e237, (q31_t)0x6a3e6140,
	(q31_t)0x6a3ae008, (q31_t)0x6a375e8f,
	(q31_t)0x6a33dcd4, (q31_t)0x6a305ad7, (q31_t)0x6a2cd899, (q31_t)0x6a295619, (q31_t)0x6a25d358, (q31_t)0x6a225055,
	(q31_t)0x6a1ecd11, (q31_t)0x6a1b498c,
	(q31_t)0x6a17c5c5, (q31_t)0x6a1441bc, (q31_t)0x6a10bd72, (q31_t)0x6a0d38e7, (q31_t)0x6a09b41a, (q31_t)0x6a062f0c,
	(q31_t)0x6a02a9bc, (q31_t)0x69ff242b,
	(q31_t)0x69fb9e59, (q31_t)0x69f81845, (q31_t)0x69f491f0, (q31_t)0x69f10b5a, (q31_t)0x69ed8482, (q31_t)0x69e9fd69,
	(q31_t)0x69e6760f, (q31_t)0x69e2ee73,
	(q31_t)0x69df6696, (q31_t)0x69dbde77, (q31_t)0x69d85618, (q31_t)0x69d4cd77, (q31_t)0x69d14494, (q31_t)0x69cdbb71,
	(q31_t)0x69ca320c, (q31_t)0x69c6a866,
	(q31_t)0x69c31e7f, (q31_t)0x69bf9456, (q31_t)0x69bc09ec, (q31_t)0x69b87f41, (q31_t)0x69b4f455, (q31_t)0x69b16928,
	(q31_t)0x69adddb9, (q31_t)0x69aa5209,
	(q31_t)0x69a6c618, (q31_t)0x69a339e6, (q31_t)0x699fad73, (q31_t)0x699c20be, (q31_t)0x699893c9, (q31_t)0x69950692,
	(q31_t)0x6991791a, (q31_t)0x698deb61,
	(q31_t)0x698a5d67, (q31_t)0x6986cf2c, (q31_t)0x698340af, (q31_t)0x697fb1f2, (q31_t)0x697c22f3, (q31_t)0x697893b4,
	(q31_t)0x69750433, (q31_t)0x69717472,
	(q31_t)0x696de46f, (q31_t)0x696a542b, (q31_t)0x6966c3a6, (q31_t)0x696332e1, (q31_t)0x695fa1da, (q31_t)0x695c1092,
	(q31_t)0x69587f09, (q31_t)0x6954ed40,
	(q31_t)0x69515b35, (q31_t)0x694dc8e9, (q31_t)0x694a365c, (q31_t)0x6946a38f, (q31_t)0x69431080, (q31_t)0x693f7d31,
	(q31_t)0x693be9a0, (q31_t)0x693855cf,
	(q31_t)0x6934c1bd, (q31_t)0x69312d6a, (q31_t)0x692d98d6, (q31_t)0x692a0401, (q31_t)0x69266eeb, (q31_t)0x6922d995,
	(q31_t)0x691f43fd, (q31_t)0x691bae25,
	(q31_t)0x6918180c, (q31_t)0x691481b2, (q31_t)0x6910eb17, (q31_t)0x690d543b, (q31_t)0x6909bd1f, (q31_t)0x690625c2,
	(q31_t)0x69028e24, (q31_t)0x68fef645,
	(q31_t)0x68fb5e25, (q31_t)0x68f7c5c5, (q31_t)0x68f42d24, (q31_t)0x68f09442, (q31_t)0x68ecfb20, (q31_t)0x68e961bd,
	(q31_t)0x68e5c819, (q31_t)0x68e22e34,
	(q31_t)0x68de940f, (q31_t)0x68daf9a9, (q31_t)0x68d75f02, (q31_t)0x68d3c41b, (q31_t)0x68d028f2, (q31_t)0x68cc8d8a,
	(q31_t)0x68c8f1e0, (q31_t)0x68c555f6,
	(q31_t)0x68c1b9cc, (q31_t)0x68be1d61, (q31_t)0x68ba80b5, (q31_t)0x68b6e3c8, (q31_t)0x68b3469b, (q31_t)0x68afa92e,
	(q31_t)0x68ac0b7f, (q31_t)0x68a86d91,
	(q31_t)0x68a4cf61, (q31_t)0x68a130f1, (q31_t)0x689d9241, (q31_t)0x6899f350, (q31_t)0x6896541f, (q31_t)0x6892b4ad,
	(q31_t)0x688f14fa, (q31_t)0x688b7507,
	(q31_t)0x6887d4d4, (q31_t)0x68843460, (q31_t)0x688093ab, (q31_t)0x687cf2b6, (q31_t)0x68795181, (q31_t)0x6875b00b,
	(q31_t)0x68720e55, (q31_t)0x686e6c5e,
	(q31_t)0x686aca27, (q31_t)0x686727b0, (q31_t)0x686384f8, (q31_t)0x685fe200, (q31_t)0x685c3ec7, (q31_t)0x68589b4e,
	(q31_t)0x6854f795, (q31_t)0x6851539b,
	(q31_t)0x684daf61, (q31_t)0x684a0ae6, (q31_t)0x6846662c, (q31_t)0x6842c131, (q31_t)0x683f1bf5, (q31_t)0x683b7679,
	(q31_t)0x6837d0bd, (q31_t)0x68342ac1,
	(q31_t)0x68308485, (q31_t)0x682cde08, (q31_t)0x6829374b, (q31_t)0x6825904d, (q31_t)0x6821e910, (q31_t)0x681e4192,
	(q31_t)0x681a99d4, (q31_t)0x6816f1d6,
	(q31_t)0x68134997, (q31_t)0x680fa118, (q31_t)0x680bf85a, (q31_t)0x68084f5a, (q31_t)0x6804a61b, (q31_t)0x6800fc9c,
	(q31_t)0x67fd52dc, (q31_t)0x67f9a8dd,
	(q31_t)0x67f5fe9d, (q31_t)0x67f2541d, (q31_t)0x67eea95d, (q31_t)0x67eafe5d, (q31_t)0x67e7531c, (q31_t)0x67e3a79c,
	(q31_t)0x67dffbdc, (q31_t)0x67dc4fdb,
	(q31_t)0x67d8a39a, (q31_t)0x67d4f71a, (q31_t)0x67d14a59, (q31_t)0x67cd9d58, (q31_t)0x67c9f017, (q31_t)0x67c64297,
	(q31_t)0x67c294d6, (q31_t)0x67bee6d5,
	(q31_t)0x67bb3894, (q31_t)0x67b78a13, (q31_t)0x67b3db53, (q31_t)0x67b02c52, (q31_t)0x67ac7d11, (q31_t)0x67a8cd91,
	(q31_t)0x67a51dd0, (q31_t)0x67a16dcf,
	(q31_t)0x679dbd8f, (q31_t)0x679a0d0f, (q31_t)0x67965c4e, (q31_t)0x6792ab4e, (q31_t)0x678efa0e, (q31_t)0x678b488e,
	(q31_t)0x678796ce, (q31_t)0x6783e4cf,
	(q31_t)0x6780328f, (q31_t)0x677c8010, (q31_t)0x6778cd50, (q31_t)0x67751a51, (q31_t)0x67716713, (q31_t)0x676db394,
	(q31_t)0x6769ffd5, (q31_t)0x67664bd7,
	(q31_t)0x67629799, (q31_t)0x675ee31b, (q31_t)0x675b2e5e, (q31_t)0x67577960, (q31_t)0x6753c423, (q31_t)0x67500ea7,
	(q31_t)0x674c58ea, (q31_t)0x6748a2ee,
	(q31_t)0x6744ecb2, (q31_t)0x67413636, (q31_t)0x673d7f7b, (q31_t)0x6739c880, (q31_t)0x67361145, (q31_t)0x673259ca,
	(q31_t)0x672ea210, (q31_t)0x672aea17,
	(q31_t)0x672731dd, (q31_t)0x67237964, (q31_t)0x671fc0ac, (q31_t)0x671c07b4, (q31_t)0x67184e7c, (q31_t)0x67149504,
	(q31_t)0x6710db4d, (q31_t)0x670d2157,
	(q31_t)0x67096721, (q31_t)0x6705acab, (q31_t)0x6701f1f6, (q31_t)0x66fe3701, (q31_t)0x66fa7bcd, (q31_t)0x66f6c059,
	(q31_t)0x66f304a6, (q31_t)0x66ef48b3,
	(q31_t)0x66eb8c80, (q31_t)0x66e7d00f, (q31_t)0x66e4135d, (q31_t)0x66e0566c, (q31_t)0x66dc993c, (q31_t)0x66d8dbcd,
	(q31_t)0x66d51e1d, (q31_t)0x66d1602f,
	(q31_t)0x66cda201, (q31_t)0x66c9e393, (q31_t)0x66c624e7, (q31_t)0x66c265fa, (q31_t)0x66bea6cf, (q31_t)0x66bae764,
	(q31_t)0x66b727ba, (q31_t)0x66b367d0,
	(q31_t)0x66afa7a7, (q31_t)0x66abe73f, (q31_t)0x66a82697, (q31_t)0x66a465b0, (q31_t)0x66a0a489, (q31_t)0x669ce324,
	(q31_t)0x6699217f, (q31_t)0x66955f9b,
	(q31_t)0x66919d77, (q31_t)0x668ddb14, (q31_t)0x668a1872, (q31_t)0x66865591, (q31_t)0x66829270, (q31_t)0x667ecf11,
	(q31_t)0x667b0b72, (q31_t)0x66774793,
	(q31_t)0x66738376, (q31_t)0x666fbf19, (q31_t)0x666bfa7d, (q31_t)0x666835a2, (q31_t)0x66647088, (q31_t)0x6660ab2f,
	(q31_t)0x665ce596, (q31_t)0x66591fbf,
	(q31_t)0x665559a8, (q31_t)0x66519352, (q31_t)0x664dccbd, (q31_t)0x664a05e9, (q31_t)0x66463ed6, (q31_t)0x66427784,
	(q31_t)0x663eaff2, (q31_t)0x663ae822,
	(q31_t)0x66372012, (q31_t)0x663357c4, (q31_t)0x662f8f36, (q31_t)0x662bc66a, (q31_t)0x6627fd5e, (q31_t)0x66243413,
	(q31_t)0x66206a8a, (q31_t)0x661ca0c1,
	(q31_t)0x6618d6b9, (q31_t)0x66150c73, (q31_t)0x661141ed, (q31_t)0x660d7729, (q31_t)0x6609ac25, (q31_t)0x6605e0e3,
	(q31_t)0x66021561, (q31_t)0x65fe49a1,
	(q31_t)0x65fa7da2, (q31_t)0x65f6b164, (q31_t)0x65f2e4e7, (q31_t)0x65ef182b, (q31_t)0x65eb4b30, (q31_t)0x65e77df6,
	(q31_t)0x65e3b07e, (q31_t)0x65dfe2c6,
	(q31_t)0x65dc14d0, (q31_t)0x65d8469b, (q31_t)0x65d47827, (q31_t)0x65d0a975, (q31_t)0x65ccda83, (q31_t)0x65c90b53,
	(q31_t)0x65c53be4, (q31_t)0x65c16c36,
	(q31_t)0x65bd9c49, (q31_t)0x65b9cc1e, (q31_t)0x65b5fbb4, (q31_t)0x65b22b0b, (q31_t)0x65ae5a23, (q31_t)0x65aa88fd,
	(q31_t)0x65a6b798, (q31_t)0x65a2e5f4,
	(q31_t)0x659f1412, (q31_t)0x659b41f1, (q31_t)0x65976f91, (q31_t)0x65939cf3, (q31_t)0x658fca15, (q31_t)0x658bf6fa,
	(q31_t)0x6588239f, (q31_t)0x65845006,
	(q31_t)0x65807c2f, (q31_t)0x657ca818, (q31_t)0x6578d3c4, (q31_t)0x6574ff30, (q31_t)0x65712a5e, (q31_t)0x656d554d,
	(q31_t)0x65697ffe, (q31_t)0x6565aa71,
	(q31_t)0x6561d4a4, (q31_t)0x655dfe99, (q31_t)0x655a2850, (q31_t)0x655651c8, (q31_t)0x65527b02, (q31_t)0x654ea3fd,
	(q31_t)0x654accba, (q31_t)0x6546f538,
	(q31_t)0x65431d77, (q31_t)0x653f4579, (q31_t)0x653b6d3b, (q31_t)0x653794c0, (q31_t)0x6533bc06, (q31_t)0x652fe30d,
	(q31_t)0x652c09d6, (q31_t)0x65283061,
	(q31_t)0x652456ad, (q31_t)0x65207cbb, (q31_t)0x651ca28a, (q31_t)0x6518c81b, (q31_t)0x6514ed6e, (q31_t)0x65111283,
	(q31_t)0x650d3759, (q31_t)0x65095bf0,
	(q31_t)0x6505804a, (q31_t)0x6501a465, (q31_t)0x64fdc841, (q31_t)0x64f9ebe0, (q31_t)0x64f60f40, (q31_t)0x64f23262,
	(q31_t)0x64ee5546, (q31_t)0x64ea77eb,
	(q31_t)0x64e69a52, (q31_t)0x64e2bc7b, (q31_t)0x64dede66, (q31_t)0x64db0012, (q31_t)0x64d72180, (q31_t)0x64d342b0,
	(q31_t)0x64cf63a2, (q31_t)0x64cb8456,
	(q31_t)0x64c7a4cb, (q31_t)0x64c3c502, (q31_t)0x64bfe4fc, (q31_t)0x64bc04b6, (q31_t)0x64b82433, (q31_t)0x64b44372,
	(q31_t)0x64b06273, (q31_t)0x64ac8135,
	(q31_t)0x64a89fba, (q31_t)0x64a4be00, (q31_t)0x64a0dc08, (q31_t)0x649cf9d2, (q31_t)0x6499175e, (q31_t)0x649534ac,
	(q31_t)0x649151bc, (q31_t)0x648d6e8e,
	(q31_t)0x64898b22, (q31_t)0x6485a778, (q31_t)0x6481c390, (q31_t)0x647ddf6a, (q31_t)0x6479fb06, (q31_t)0x64761664,
	(q31_t)0x64723184, (q31_t)0x646e4c66,
	(q31_t)0x646a670a, (q31_t)0x64668170, (q31_t)0x64629b98, (q31_t)0x645eb582, (q31_t)0x645acf2e, (q31_t)0x6456e89d,
	(q31_t)0x645301cd, (q31_t)0x644f1ac0,
	(q31_t)0x644b3375, (q31_t)0x64474bec, (q31_t)0x64436425, (q31_t)0x643f7c20, (q31_t)0x643b93dd, (q31_t)0x6437ab5d,
	(q31_t)0x6433c29f, (q31_t)0x642fd9a3,
	(q31_t)0x642bf069, (q31_t)0x642806f1, (q31_t)0x64241d3c, (q31_t)0x64203348, (q31_t)0x641c4917, (q31_t)0x64185ea9,
	(q31_t)0x641473fc, (q31_t)0x64108912,
	(q31_t)0x640c9dea, (q31_t)0x6408b284, (q31_t)0x6404c6e1, (q31_t)0x6400db00, (q31_t)0x63fceee1, (q31_t)0x63f90285,
	(q31_t)0x63f515eb, (q31_t)0x63f12913,
	(q31_t)0x63ed3bfd, (q31_t)0x63e94eaa, (q31_t)0x63e5611a, (q31_t)0x63e1734b, (q31_t)0x63dd853f, (q31_t)0x63d996f6,
	(q31_t)0x63d5a86f, (q31_t)0x63d1b9aa,
	(q31_t)0x63cdcaa8, (q31_t)0x63c9db68, (q31_t)0x63c5ebeb, (q31_t)0x63c1fc30, (q31_t)0x63be0c37, (q31_t)0x63ba1c01,
	(q31_t)0x63b62b8e, (q31_t)0x63b23add,
	(q31_t)0x63ae49ee, (q31_t)0x63aa58c2, (q31_t)0x63a66759, (q31_t)0x63a275b2, (q31_t)0x639e83cd, (q31_t)0x639a91ac,
	(q31_t)0x63969f4c, (q31_t)0x6392acaf,
	(q31_t)0x638eb9d5, (q31_t)0x638ac6be, (q31_t)0x6386d369, (q31_t)0x6382dfd6, (q31_t)0x637eec07, (q31_t)0x637af7fa,
	(q31_t)0x637703af, (q31_t)0x63730f27,
	(q31_t)0x636f1a62, (q31_t)0x636b2560, (q31_t)0x63673020, (q31_t)0x63633aa3, (q31_t)0x635f44e8, (q31_t)0x635b4ef0,
	(q31_t)0x635758bb, (q31_t)0x63536249,
	(q31_t)0x634f6b99, (q31_t)0x634b74ad, (q31_t)0x63477d82, (q31_t)0x6343861b, (q31_t)0x633f8e76, (q31_t)0x633b9695,
	(q31_t)0x63379e76, (q31_t)0x6333a619,
	(q31_t)0x632fad80, (q31_t)0x632bb4a9, (q31_t)0x6327bb96, (q31_t)0x6323c245, (q31_t)0x631fc8b7, (q31_t)0x631bceeb,
	(q31_t)0x6317d4e3, (q31_t)0x6313da9e,
	(q31_t)0x630fe01b, (q31_t)0x630be55b, (q31_t)0x6307ea5e, (q31_t)0x6303ef25, (q31_t)0x62fff3ae, (q31_t)0x62fbf7fa,
	(q31_t)0x62f7fc08, (q31_t)0x62f3ffda,
	(q31_t)0x62f0036f, (q31_t)0x62ec06c7, (q31_t)0x62e809e2, (q31_t)0x62e40cbf, (q31_t)0x62e00f60, (q31_t)0x62dc11c4,
	(q31_t)0x62d813eb, (q31_t)0x62d415d4,
	(q31_t)0x62d01781, (q31_t)0x62cc18f1, (q31_t)0x62c81a24, (q31_t)0x62c41b1a, (q31_t)0x62c01bd3, (q31_t)0x62bc1c4f,
	(q31_t)0x62b81c8f, (q31_t)0x62b41c91,
	(q31_t)0x62b01c57, (q31_t)0x62ac1bdf, (q31_t)0x62a81b2b, (q31_t)0x62a41a3a, (q31_t)0x62a0190c, (q31_t)0x629c17a1,
	(q31_t)0x629815fa, (q31_t)0x62941415,
	(q31_t)0x629011f4, (q31_t)0x628c0f96, (q31_t)0x62880cfb, (q31_t)0x62840a23, (q31_t)0x6280070f, (q31_t)0x627c03be,
	(q31_t)0x62780030, (q31_t)0x6273fc65,
	(q31_t)0x626ff85e, (q31_t)0x626bf41a, (q31_t)0x6267ef99, (q31_t)0x6263eadc, (q31_t)0x625fe5e1, (q31_t)0x625be0ab,
	(q31_t)0x6257db37, (q31_t)0x6253d587,
	(q31_t)0x624fcf9a, (q31_t)0x624bc970, (q31_t)0x6247c30a, (q31_t)0x6243bc68, (q31_t)0x623fb588, (q31_t)0x623bae6c,
	(q31_t)0x6237a714, (q31_t)0x62339f7e,
	(q31_t)0x622f97ad, (q31_t)0x622b8f9e, (q31_t)0x62278754, (q31_t)0x62237ecc, (q31_t)0x621f7608, (q31_t)0x621b6d08,
	(q31_t)0x621763cb, (q31_t)0x62135a51,
	(q31_t)0x620f509b, (q31_t)0x620b46a9, (q31_t)0x62073c7a, (q31_t)0x6203320e, (q31_t)0x61ff2766, (q31_t)0x61fb1c82,
	(q31_t)0x61f71161, (q31_t)0x61f30604,
	(q31_t)0x61eefa6b, (q31_t)0x61eaee95, (q31_t)0x61e6e282, (q31_t)0x61e2d633, (q31_t)0x61dec9a8, (q31_t)0x61dabce0,
	(q31_t)0x61d6afdd, (q31_t)0x61d2a29c,
	(q31_t)0x61ce9520, (q31_t)0x61ca8767, (q31_t)0x61c67971, (q31_t)0x61c26b40, (q31_t)0x61be5cd2, (q31_t)0x61ba4e28,
	(q31_t)0x61b63f41, (q31_t)0x61b2301e,
	(q31_t)0x61ae20bf, (q31_t)0x61aa1124, (q31_t)0x61a6014d, (q31_t)0x61a1f139, (q31_t)0x619de0e9, (q31_t)0x6199d05d,
	(q31_t)0x6195bf94, (q31_t)0x6191ae90,
	(q31_t)0x618d9d4f, (q31_t)0x61898bd2, (q31_t)0x61857a19, (q31_t)0x61816824, (q31_t)0x617d55f2, (q31_t)0x61794385,
	(q31_t)0x617530db, (q31_t)0x61711df5,
	(q31_t)0x616d0ad3, (q31_t)0x6168f775, (q31_t)0x6164e3db, (q31_t)0x6160d005, (q31_t)0x615cbbf3, (q31_t)0x6158a7a4,
	(q31_t)0x6154931a, (q31_t)0x61507e54,
	(q31_t)0x614c6951, (q31_t)0x61485413, (q31_t)0x61443e98, (q31_t)0x614028e2, (q31_t)0x613c12f0, (q31_t)0x6137fcc1,
	(q31_t)0x6133e657, (q31_t)0x612fcfb0,
	(q31_t)0x612bb8ce, (q31_t)0x6127a1b0, (q31_t)0x61238a56, (q31_t)0x611f72c0, (q31_t)0x611b5aee, (q31_t)0x611742e0,
	(q31_t)0x61132a96, (q31_t)0x610f1210,
	(q31_t)0x610af94f, (q31_t)0x6106e051, (q31_t)0x6102c718, (q31_t)0x60feada3, (q31_t)0x60fa93f2, (q31_t)0x60f67a05,
	(q31_t)0x60f25fdd, (q31_t)0x60ee4579,
	(q31_t)0x60ea2ad8, (q31_t)0x60e60ffd, (q31_t)0x60e1f4e5, (q31_t)0x60ddd991, (q31_t)0x60d9be02, (q31_t)0x60d5a237,
	(q31_t)0x60d18631, (q31_t)0x60cd69ee,
	(q31_t)0x60c94d70, (q31_t)0x60c530b6, (q31_t)0x60c113c1, (q31_t)0x60bcf690, (q31_t)0x60b8d923, (q31_t)0x60b4bb7a,
	(q31_t)0x60b09d96, (q31_t)0x60ac7f76,
	(q31_t)0x60a8611b, (q31_t)0x60a44284, (q31_t)0x60a023b1, (q31_t)0x609c04a3, (q31_t)0x6097e559, (q31_t)0x6093c5d3,
	(q31_t)0x608fa612, (q31_t)0x608b8616,
	(q31_t)0x608765dd, (q31_t)0x6083456a, (q31_t)0x607f24ba, (q31_t)0x607b03d0, (q31_t)0x6076e2a9, (q31_t)0x6072c148,
	(q31_t)0x606e9faa, (q31_t)0x606a7dd2,
	(q31_t)0x60665bbd, (q31_t)0x6062396e, (q31_t)0x605e16e2, (q31_t)0x6059f41c, (q31_t)0x6055d11a, (q31_t)0x6051addc,
	(q31_t)0x604d8a63, (q31_t)0x604966af,
	(q31_t)0x604542bf, (q31_t)0x60411e94, (q31_t)0x603cfa2e, (q31_t)0x6038d58c, (q31_t)0x6034b0af, (q31_t)0x60308b97,
	(q31_t)0x602c6643, (q31_t)0x602840b4,
	(q31_t)0x60241ae9, (q31_t)0x601ff4e3, (q31_t)0x601bcea2, (q31_t)0x6017a826, (q31_t)0x6013816e, (q31_t)0x600f5a7b,
	(q31_t)0x600b334d, (q31_t)0x60070be4,
	(q31_t)0x6002e43f, (q31_t)0x5ffebc5f, (q31_t)0x5ffa9444, (q31_t)0x5ff66bee, (q31_t)0x5ff2435d, (q31_t)0x5fee1a90,
	(q31_t)0x5fe9f188, (q31_t)0x5fe5c845,
	(q31_t)0x5fe19ec7, (q31_t)0x5fdd750e, (q31_t)0x5fd94b19, (q31_t)0x5fd520ea, (q31_t)0x5fd0f67f, (q31_t)0x5fcccbd9,
	(q31_t)0x5fc8a0f8, (q31_t)0x5fc475dc,
	(q31_t)0x5fc04a85, (q31_t)0x5fbc1ef3, (q31_t)0x5fb7f326, (q31_t)0x5fb3c71e, (q31_t)0x5faf9adb, (q31_t)0x5fab6e5d,
	(q31_t)0x5fa741a3, (q31_t)0x5fa314af,
	(q31_t)0x5f9ee780, (q31_t)0x5f9aba16, (q31_t)0x5f968c70, (q31_t)0x5f925e90, (q31_t)0x5f8e3075, (q31_t)0x5f8a021f,
	(q31_t)0x5f85d38e, (q31_t)0x5f81a4c2,
	(q31_t)0x5f7d75bb, (q31_t)0x5f794679, (q31_t)0x5f7516fd, (q31_t)0x5f70e745, (q31_t)0x5f6cb753, (q31_t)0x5f688726,
	(q31_t)0x5f6456be, (q31_t)0x5f60261b,
	(q31_t)0x5f5bf53d, (q31_t)0x5f57c424, (q31_t)0x5f5392d1, (q31_t)0x5f4f6143, (q31_t)0x5f4b2f7a, (q31_t)0x5f46fd76,
	(q31_t)0x5f42cb37, (q31_t)0x5f3e98be,
	(q31_t)0x5f3a660a, (q31_t)0x5f36331b, (q31_t)0x5f31fff1, (q31_t)0x5f2dcc8d, (q31_t)0x5f2998ee, (q31_t)0x5f256515,
	(q31_t)0x5f213100, (q31_t)0x5f1cfcb1,
	(q31_t)0x5f18c827, (q31_t)0x5f149363, (q31_t)0x5f105e64, (q31_t)0x5f0c292a, (q31_t)0x5f07f3b6, (q31_t)0x5f03be07,
	(q31_t)0x5eff881d, (q31_t)0x5efb51f9,
	(q31_t)0x5ef71b9b, (q31_t)0x5ef2e501, (q31_t)0x5eeeae2d, (q31_t)0x5eea771f, (q31_t)0x5ee63fd6, (q31_t)0x5ee20853,
	(q31_t)0x5eddd094, (q31_t)0x5ed9989c,
	(q31_t)0x5ed56069, (q31_t)0x5ed127fb, (q31_t)0x5eccef53, (q31_t)0x5ec8b671, (q31_t)0x5ec47d54, (q31_t)0x5ec043fc,
	(q31_t)0x5ebc0a6a, (q31_t)0x5eb7d09e,
	(q31_t)0x5eb39697, (q31_t)0x5eaf5c56, (q31_t)0x5eab21da, (q31_t)0x5ea6e724, (q31_t)0x5ea2ac34, (q31_t)0x5e9e7109,
	(q31_t)0x5e9a35a4, (q31_t)0x5e95fa05,
	(q31_t)0x5e91be2b, (q31_t)0x5e8d8217, (q31_t)0x5e8945c8, (q31_t)0x5e85093f, (q31_t)0x5e80cc7c, (q31_t)0x5e7c8f7f,
	(q31_t)0x5e785247, (q31_t)0x5e7414d5,
	(q31_t)0x5e6fd729, (q31_t)0x5e6b9943, (q31_t)0x5e675b22, (q31_t)0x5e631cc7, (q31_t)0x5e5ede32, (q31_t)0x5e5a9f62,
	(q31_t)0x5e566059, (q31_t)0x5e522115,
	(q31_t)0x5e4de197, (q31_t)0x5e49a1df, (q31_t)0x5e4561ed, (q31_t)0x5e4121c0, (q31_t)0x5e3ce15a, (q31_t)0x5e38a0b9,
	(q31_t)0x5e345fde, (q31_t)0x5e301ec9,
	(q31_t)0x5e2bdd7a, (q31_t)0x5e279bf1, (q31_t)0x5e235a2e, (q31_t)0x5e1f1830, (q31_t)0x5e1ad5f9, (q31_t)0x5e169388,
	(q31_t)0x5e1250dc, (q31_t)0x5e0e0df7,
	(q31_t)0x5e09cad7, (q31_t)0x5e05877e, (q31_t)0x5e0143ea, (q31_t)0x5dfd001d, (q31_t)0x5df8bc15, (q31_t)0x5df477d4,
	(q31_t)0x5df03359, (q31_t)0x5debeea3,
	(q31_t)0x5de7a9b4, (q31_t)0x5de3648b, (q31_t)0x5ddf1f28, (q31_t)0x5ddad98b, (q31_t)0x5dd693b4, (q31_t)0x5dd24da3,
	(q31_t)0x5dce0759, (q31_t)0x5dc9c0d4,
	(q31_t)0x5dc57a16, (q31_t)0x5dc1331d, (q31_t)0x5dbcebeb, (q31_t)0x5db8a480, (q31_t)0x5db45cda, (q31_t)0x5db014fa,
	(q31_t)0x5dabcce1, (q31_t)0x5da7848e,
	(q31_t)0x5da33c01, (q31_t)0x5d9ef33b, (q31_t)0x5d9aaa3a, (q31_t)0x5d966100, (q31_t)0x5d92178d, (q31_t)0x5d8dcddf,
	(q31_t)0x5d8983f8, (q31_t)0x5d8539d7,
	(q31_t)0x5d80ef7c, (q31_t)0x5d7ca4e8, (q31_t)0x5d785a1a, (q31_t)0x5d740f12, (q31_t)0x5d6fc3d1, (q31_t)0x5d6b7856,
	(q31_t)0x5d672ca2, (q31_t)0x5d62e0b4,
	(q31_t)0x5d5e948c, (q31_t)0x5d5a482a, (q31_t)0x5d55fb90, (q31_t)0x5d51aebb, (q31_t)0x5d4d61ad, (q31_t)0x5d491465,
	(q31_t)0x5d44c6e4, (q31_t)0x5d40792a,
	(q31_t)0x5d3c2b35, (q31_t)0x5d37dd08, (q31_t)0x5d338ea0, (q31_t)0x5d2f4000, (q31_t)0x5d2af125, (q31_t)0x5d26a212,
	(q31_t)0x5d2252c5, (q31_t)0x5d1e033e,
	(q31_t)0x5d19b37e, (q31_t)0x5d156385, (q31_t)0x5d111352, (q31_t)0x5d0cc2e5, (q31_t)0x5d087240, (q31_t)0x5d042161,
	(q31_t)0x5cffd048, (q31_t)0x5cfb7ef7,
	(q31_t)0x5cf72d6b, (q31_t)0x5cf2dba7, (q31_t)0x5cee89a9, (q31_t)0x5cea3772, (q31_t)0x5ce5e501, (q31_t)0x5ce19258,
	(q31_t)0x5cdd3f75, (q31_t)0x5cd8ec58,
	(q31_t)0x5cd49903, (q31_t)0x5cd04574, (q31_t)0x5ccbf1ab, (q31_t)0x5cc79daa, (q31_t)0x5cc3496f, (q31_t)0x5cbef4fc,
	(q31_t)0x5cbaa04f, (q31_t)0x5cb64b68,
	(q31_t)0x5cb1f649, (q31_t)0x5cada0f0, (q31_t)0x5ca94b5e, (q31_t)0x5ca4f594, (q31_t)0x5ca09f8f, (q31_t)0x5c9c4952,
	(q31_t)0x5c97f2dc, (q31_t)0x5c939c2c,
	(q31_t)0x5c8f4544, (q31_t)0x5c8aee22, (q31_t)0x5c8696c7, (q31_t)0x5c823f34, (q31_t)0x5c7de767, (q31_t)0x5c798f61,
	(q31_t)0x5c753722, (q31_t)0x5c70deaa,
	(q31_t)0x5c6c85f9, (q31_t)0x5c682d0f, (q31_t)0x5c63d3eb, (q31_t)0x5c5f7a8f, (q31_t)0x5c5b20fa, (q31_t)0x5c56c72c,
	(q31_t)0x5c526d25, (q31_t)0x5c4e12e5,
	(q31_t)0x5c49b86d, (q31_t)0x5c455dbb, (q31_t)0x5c4102d0, (q31_t)0x5c3ca7ad, (q31_t)0x5c384c50, (q31_t)0x5c33f0bb,
	(q31_t)0x5c2f94ec, (q31_t)0x5c2b38e5,
	(q31_t)0x5c26dca5, (q31_t)0x5c22802c, (q31_t)0x5c1e237b, (q31_t)0x5c19c690, (q31_t)0x5c15696d, (q31_t)0x5c110c11,
	(q31_t)0x5c0cae7c, (q31_t)0x5c0850ae,
	(q31_t)0x5c03f2a8, (q31_t)0x5bff9469, (q31_t)0x5bfb35f1, (q31_t)0x5bf6d740, (q31_t)0x5bf27857, (q31_t)0x5bee1935,
	(q31_t)0x5be9b9da, (q31_t)0x5be55a46,
	(q31_t)0x5be0fa7a, (q31_t)0x5bdc9a75, (q31_t)0x5bd83a37, (q31_t)0x5bd3d9c1, (q31_t)0x5bcf7912, (q31_t)0x5bcb182b,
	(q31_t)0x5bc6b70b, (q31_t)0x5bc255b2,
	(q31_t)0x5bbdf421, (q31_t)0x5bb99257, (q31_t)0x5bb53054, (q31_t)0x5bb0ce19, (q31_t)0x5bac6ba6, (q31_t)0x5ba808f9,
	(q31_t)0x5ba3a615, (q31_t)0x5b9f42f7,
	(q31_t)0x5b9adfa2, (q31_t)0x5b967c13, (q31_t)0x5b92184d, (q31_t)0x5b8db44d, (q31_t)0x5b895016, (q31_t)0x5b84eba6,
	(q31_t)0x5b8086fd, (q31_t)0x5b7c221c,
	(q31_t)0x5b77bd02, (q31_t)0x5b7357b0, (q31_t)0x5b6ef226, (q31_t)0x5b6a8c63, (q31_t)0x5b662668, (q31_t)0x5b61c035,
	(q31_t)0x5b5d59c9, (q31_t)0x5b58f324,
	(q31_t)0x5b548c48, (q31_t)0x5b502533, (q31_t)0x5b4bbde6, (q31_t)0x5b475660, (q31_t)0x5b42eea2, (q31_t)0x5b3e86ac,
	(q31_t)0x5b3a1e7e, (q31_t)0x5b35b617,
	(q31_t)0x5b314d78, (q31_t)0x5b2ce4a1, (q31_t)0x5b287b91, (q31_t)0x5b241249, (q31_t)0x5b1fa8c9, (q31_t)0x5b1b3f11,
	(q31_t)0x5b16d521, (q31_t)0x5b126af8,
	(q31_t)0x5b0e0098, (q31_t)0x5b0995ff, (q31_t)0x5b052b2e, (q31_t)0x5b00c025, (q31_t)0x5afc54e3, (q31_t)0x5af7e96a,
	(q31_t)0x5af37db8, (q31_t)0x5aef11cf,
	(q31_t)0x5aeaa5ad, (q31_t)0x5ae63953, (q31_t)0x5ae1ccc1, (q31_t)0x5add5ff7, (q31_t)0x5ad8f2f5, (q31_t)0x5ad485bb,
	(q31_t)0x5ad01849, (q31_t)0x5acbaa9f,
	(q31_t)0x5ac73cbd, (q31_t)0x5ac2cea3, (q31_t)0x5abe6050, (q31_t)0x5ab9f1c6, (q31_t)0x5ab58304, (q31_t)0x5ab1140a,
	(q31_t)0x5aaca4d8, (q31_t)0x5aa8356f,
	(q31_t)0x5aa3c5cd, (q31_t)0x5a9f55f3, (q31_t)0x5a9ae5e2, (q31_t)0x5a967598, (q31_t)0x5a920517, (q31_t)0x5a8d945d,
	(q31_t)0x5a89236c, (q31_t)0x5a84b243,
	(q31_t)0x5a8040e3, (q31_t)0x5a7bcf4a, (q31_t)0x5a775d7a, (q31_t)0x5a72eb71, (q31_t)0x5a6e7931, (q31_t)0x5a6a06ba,
	(q31_t)0x5a65940a, (q31_t)0x5a612123,
	(q31_t)0x5a5cae04, (q31_t)0x5a583aad, (q31_t)0x5a53c71e, (q31_t)0x5a4f5358, (q31_t)0x5a4adf5a, (q31_t)0x5a466b24,
	(q31_t)0x5a41f6b7, (q31_t)0x5a3d8212,
	(q31_t)0x5a390d35, (q31_t)0x5a349821, (q31_t)0x5a3022d5, (q31_t)0x5a2bad51, (q31_t)0x5a273796, (q31_t)0x5a22c1a3,
	(q31_t)0x5a1e4b79, (q31_t)0x5a19d517,
	(q31_t)0x5a155e7d, (q31_t)0x5a10e7ac, (q31_t)0x5a0c70a3, (q31_t)0x5a07f963, (q31_t)0x5a0381eb, (q31_t)0x59ff0a3c,
	(q31_t)0x59fa9255, (q31_t)0x59f61a36,
	(q31_t)0x59f1a1e0, (q31_t)0x59ed2953, (q31_t)0x59e8b08e, (q31_t)0x59e43792, (q31_t)0x59dfbe5e, (q31_t)0x59db44f3,
	(q31_t)0x59d6cb50, (q31_t)0x59d25176,
	(q31_t)0x59cdd765, (q31_t)0x59c95d1c, (q31_t)0x59c4e29c, (q31_t)0x59c067e4, (q31_t)0x59bbecf5, (q31_t)0x59b771cf,
	(q31_t)0x59b2f671, (q31_t)0x59ae7add,
	(q31_t)0x59a9ff10, (q31_t)0x59a5830d, (q31_t)0x59a106d2, (q31_t)0x599c8a60, (q31_t)0x59980db6, (q31_t)0x599390d5,
	(q31_t)0x598f13bd, (q31_t)0x598a966e,
	(q31_t)0x598618e8, (q31_t)0x59819b2a, (q31_t)0x597d1d35, (q31_t)0x59789f09, (q31_t)0x597420a6, (q31_t)0x596fa20b,
	(q31_t)0x596b233a, (q31_t)0x5966a431,
	(q31_t)0x596224f1, (q31_t)0x595da57a, (q31_t)0x595925cc, (q31_t)0x5954a5e6, (q31_t)0x595025ca, (q31_t)0x594ba576,
	(q31_t)0x594724ec, (q31_t)0x5942a42a,
	(q31_t)0x593e2331, (q31_t)0x5939a202, (q31_t)0x5935209b, (q31_t)0x59309efd, (q31_t)0x592c1d28, (q31_t)0x59279b1c,
	(q31_t)0x592318d9, (q31_t)0x591e9660,
	(q31_t)0x591a13af, (q31_t)0x591590c7, (q31_t)0x59110da8, (q31_t)0x590c8a53, (q31_t)0x590806c6, (q31_t)0x59038302,
	(q31_t)0x58feff08, (q31_t)0x58fa7ad7,
	(q31_t)0x58f5f66e, (q31_t)0x58f171cf, (q31_t)0x58ececf9, (q31_t)0x58e867ed, (q31_t)0x58e3e2a9, (q31_t)0x58df5d2e,
	(q31_t)0x58dad77d, (q31_t)0x58d65195,
	(q31_t)0x58d1cb76, (q31_t)0x58cd4520, (q31_t)0x58c8be94, (q31_t)0x58c437d1, (q31_t)0x58bfb0d7, (q31_t)0x58bb29a6,
	(q31_t)0x58b6a23e, (q31_t)0x58b21aa0,
	(q31_t)0x58ad92cb, (q31_t)0x58a90ac0, (q31_t)0x58a4827d, (q31_t)0x589ffa04, (q31_t)0x589b7155, (q31_t)0x5896e86f,
	(q31_t)0x58925f52, (q31_t)0x588dd5fe,
	(q31_t)0x58894c74, (q31_t)0x5884c2b3, (q31_t)0x588038bb, (q31_t)0x587bae8d, (q31_t)0x58772429, (q31_t)0x5872998e,
	(q31_t)0x586e0ebc, (q31_t)0x586983b4,
	(q31_t)0x5864f875, (q31_t)0x58606d00, (q31_t)0x585be154, (q31_t)0x58575571, (q31_t)0x5852c958, (q31_t)0x584e3d09,
	(q31_t)0x5849b083, (q31_t)0x584523c7,
	(q31_t)0x584096d4, (q31_t)0x583c09ab, (q31_t)0x58377c4c, (q31_t)0x5832eeb6, (q31_t)0x582e60e9, (q31_t)0x5829d2e6,
	(q31_t)0x582544ad, (q31_t)0x5820b63e,
	(q31_t)0x581c2798, (q31_t)0x581798bb, (q31_t)0x581309a9, (q31_t)0x580e7a60, (q31_t)0x5809eae1, (q31_t)0x58055b2b,
	(q31_t)0x5800cb3f, (q31_t)0x57fc3b1d,
	(q31_t)0x57f7aac5, (q31_t)0x57f31a36, (q31_t)0x57ee8971, (q31_t)0x57e9f876, (q31_t)0x57e56744, (q31_t)0x57e0d5dd,
	(q31_t)0x57dc443f, (q31_t)0x57d7b26b,
	(q31_t)0x57d32061, (q31_t)0x57ce8e20, (q31_t)0x57c9fbaa, (q31_t)0x57c568fd, (q31_t)0x57c0d61a, (q31_t)0x57bc4301,
	(q31_t)0x57b7afb2, (q31_t)0x57b31c2d,
	(q31_t)0x57ae8872, (q31_t)0x57a9f480, (q31_t)0x57a56059, (q31_t)0x57a0cbfb, (q31_t)0x579c3768, (q31_t)0x5797a29e,
	(q31_t)0x57930d9e, (q31_t)0x578e7869,
	(q31_t)0x5789e2fd, (q31_t)0x57854d5b, (q31_t)0x5780b784, (q31_t)0x577c2176, (q31_t)0x57778b32, (q31_t)0x5772f4b9,
	(q31_t)0x576e5e09, (q31_t)0x5769c724,
	(q31_t)0x57653009, (q31_t)0x576098b7, (q31_t)0x575c0130, (q31_t)0x57576973, (q31_t)0x5752d180, (q31_t)0x574e3957,
	(q31_t)0x5749a0f9, (q31_t)0x57450864,
	(q31_t)0x57406f9a, (q31_t)0x573bd69a, (q31_t)0x57373d64, (q31_t)0x5732a3f8, (q31_t)0x572e0a56, (q31_t)0x5729707f,
	(q31_t)0x5724d672, (q31_t)0x57203c2f,
	(q31_t)0x571ba1b7, (q31_t)0x57170708, (q31_t)0x57126c24, (q31_t)0x570dd10a, (q31_t)0x570935bb, (q31_t)0x57049a36,
	(q31_t)0x56fffe7b, (q31_t)0x56fb628b,
	(q31_t)0x56f6c664, (q31_t)0x56f22a09, (q31_t)0x56ed8d77, (q31_t)0x56e8f0b0, (q31_t)0x56e453b4, (q31_t)0x56dfb681,
	(q31_t)0x56db1919, (q31_t)0x56d67b7c,
	(q31_t)0x56d1dda9, (q31_t)0x56cd3fa1, (q31_t)0x56c8a162, (q31_t)0x56c402ef, (q31_t)0x56bf6446, (q31_t)0x56bac567,
	(q31_t)0x56b62653, (q31_t)0x56b18709,
	(q31_t)0x56ace78a, (q31_t)0x56a847d6, (q31_t)0x56a3a7ec, (q31_t)0x569f07cc, (q31_t)0x569a6777, (q31_t)0x5695c6ed,
	(q31_t)0x5691262d, (q31_t)0x568c8538,
	(q31_t)0x5687e40e, (q31_t)0x568342ae, (q31_t)0x567ea118, (q31_t)0x5679ff4e, (q31_t)0x56755d4e, (q31_t)0x5670bb19,
	(q31_t)0x566c18ae, (q31_t)0x5667760e,
	(q31_t)0x5662d339, (q31_t)0x565e302e, (q31_t)0x56598cee, (q31_t)0x5654e979, (q31_t)0x565045cf, (q31_t)0x564ba1f0,
	(q31_t)0x5646fddb, (q31_t)0x56425991,
	(q31_t)0x563db512, (q31_t)0x5639105d, (q31_t)0x56346b74, (q31_t)0x562fc655, (q31_t)0x562b2101, (q31_t)0x56267b78,
	(q31_t)0x5621d5ba, (q31_t)0x561d2fc6,
	(q31_t)0x5618899e, (q31_t)0x5613e340, (q31_t)0x560f3cae, (q31_t)0x560a95e6, (q31_t)0x5605eee9, (q31_t)0x560147b7,
	(q31_t)0x55fca050, (q31_t)0x55f7f8b4,
	(q31_t)0x55f350e3, (q31_t)0x55eea8dd, (q31_t)0x55ea00a2, (q31_t)0x55e55832, (q31_t)0x55e0af8d, (q31_t)0x55dc06b3,
	(q31_t)0x55d75da4, (q31_t)0x55d2b460,
	(q31_t)0x55ce0ae7, (q31_t)0x55c96139, (q31_t)0x55c4b757, (q31_t)0x55c00d3f, (q31_t)0x55bb62f3, (q31_t)0x55b6b871,
	(q31_t)0x55b20dbb, (q31_t)0x55ad62d0,
	(q31_t)0x55a8b7b0, (q31_t)0x55a40c5b, (q31_t)0x559f60d1, (q31_t)0x559ab513, (q31_t)0x55960920, (q31_t)0x55915cf8,
	(q31_t)0x558cb09b, (q31_t)0x55880409,
	(q31_t)0x55835743, (q31_t)0x557eaa48, (q31_t)0x5579fd18, (q31_t)0x55754fb3, (q31_t)0x5570a21a, (q31_t)0x556bf44c,
	(q31_t)0x55674649, (q31_t)0x55629812,
	(q31_t)0x555de9a6, (q31_t)0x55593b05, (q31_t)0x55548c30, (q31_t)0x554fdd26, (q31_t)0x554b2de7, (q31_t)0x55467e74,
	(q31_t)0x5541cecc, (q31_t)0x553d1ef0,
	(q31_t)0x55386edf, (q31_t)0x5533be99, (q31_t)0x552f0e1f, (q31_t)0x552a5d70, (q31_t)0x5525ac8d, (q31_t)0x5520fb75,
	(q31_t)0x551c4a29, (q31_t)0x551798a8,
	(q31_t)0x5512e6f3, (q31_t)0x550e3509, (q31_t)0x550982eb, (q31_t)0x5504d099, (q31_t)0x55001e12, (q31_t)0x54fb6b56,
	(q31_t)0x54f6b866, (q31_t)0x54f20542,
	(q31_t)0x54ed51e9, (q31_t)0x54e89e5c, (q31_t)0x54e3ea9a, (q31_t)0x54df36a5, (q31_t)0x54da827a, (q31_t)0x54d5ce1c,
	(q31_t)0x54d11989, (q31_t)0x54cc64c2,
	(q31_t)0x54c7afc6, (q31_t)0x54c2fa96, (q31_t)0x54be4532, (q31_t)0x54b98f9a, (q31_t)0x54b4d9cd, (q31_t)0x54b023cc,
	(q31_t)0x54ab6d97, (q31_t)0x54a6b72e,
	(q31_t)0x54a20090, (q31_t)0x549d49bf, (q31_t)0x549892b9, (q31_t)0x5493db7f, (q31_t)0x548f2410, (q31_t)0x548a6c6e,
	(q31_t)0x5485b497, (q31_t)0x5480fc8c,
	(q31_t)0x547c444d, (q31_t)0x54778bda, (q31_t)0x5472d333, (q31_t)0x546e1a58, (q31_t)0x54696149, (q31_t)0x5464a805,
	(q31_t)0x545fee8e, (q31_t)0x545b34e3,
	(q31_t)0x54567b03, (q31_t)0x5451c0f0, (q31_t)0x544d06a8, (q31_t)0x54484c2d, (q31_t)0x5443917d, (q31_t)0x543ed699,
	(q31_t)0x543a1b82, (q31_t)0x54356037,
	(q31_t)0x5430a4b7, (q31_t)0x542be904, (q31_t)0x54272d1d, (q31_t)0x54227102, (q31_t)0x541db4b3, (q31_t)0x5418f830,
	(q31_t)0x54143b79, (q31_t)0x540f7e8e,
	(q31_t)0x540ac170, (q31_t)0x5406041d, (q31_t)0x54014697, (q31_t)0x53fc88dd, (q31_t)0x53f7caef, (q31_t)0x53f30cce,
	(q31_t)0x53ee4e78, (q31_t)0x53e98fef,
	(q31_t)0x53e4d132, (q31_t)0x53e01242, (q31_t)0x53db531d, (q31_t)0x53d693c5, (q31_t)0x53d1d439, (q31_t)0x53cd147a,
	(q31_t)0x53c85486, (q31_t)0x53c3945f,
	(q31_t)0x53bed405, (q31_t)0x53ba1377, (q31_t)0x53b552b5, (q31_t)0x53b091bf, (q31_t)0x53abd096, (q31_t)0x53a70f39,
	(q31_t)0x53a24da9, (q31_t)0x539d8be5,
	(q31_t)0x5398c9ed, (q31_t)0x539407c2, (q31_t)0x538f4564, (q31_t)0x538a82d1, (q31_t)0x5385c00c, (q31_t)0x5380fd12,
	(q31_t)0x537c39e6, (q31_t)0x53777685,
	(q31_t)0x5372b2f2, (q31_t)0x536def2a, (q31_t)0x53692b30, (q31_t)0x53646701, (q31_t)0x535fa2a0, (q31_t)0x535ade0b,
	(q31_t)0x53561942, (q31_t)0x53515447,
	(q31_t)0x534c8f17, (q31_t)0x5347c9b5, (q31_t)0x5343041f, (q31_t)0x533e3e55, (q31_t)0x53397859, (q31_t)0x5334b229,
	(q31_t)0x532febc5, (q31_t)0x532b252f,
	(q31_t)0x53265e65, (q31_t)0x53219767, (q31_t)0x531cd037, (q31_t)0x531808d3, (q31_t)0x5313413c, (q31_t)0x530e7972,
	(q31_t)0x5309b174, (q31_t)0x5304e943,
	(q31_t)0x530020df, (q31_t)0x52fb5848, (q31_t)0x52f68f7e, (q31_t)0x52f1c680, (q31_t)0x52ecfd4f, (q31_t)0x52e833ec,
	(q31_t)0x52e36a55, (q31_t)0x52dea08a,
	(q31_t)0x52d9d68d, (q31_t)0x52d50c5d, (q31_t)0x52d041f9, (q31_t)0x52cb7763, (q31_t)0x52c6ac99, (q31_t)0x52c1e19d,
	(q31_t)0x52bd166d, (q31_t)0x52b84b0a,
	(q31_t)0x52b37f74, (q31_t)0x52aeb3ac, (q31_t)0x52a9e7b0, (q31_t)0x52a51b81, (q31_t)0x52a04f1f, (q31_t)0x529b828a,
	(q31_t)0x5296b5c3, (q31_t)0x5291e8c8,
	(q31_t)0x528d1b9b, (q31_t)0x52884e3a, (q31_t)0x528380a7, (q31_t)0x527eb2e0, (q31_t)0x5279e4e7, (q31_t)0x527516bb,
	(q31_t)0x5270485c, (q31_t)0x526b79ca,
	(q31_t)0x5266ab06, (q31_t)0x5261dc0e, (q31_t)0x525d0ce4, (q31_t)0x52583d87, (q31_t)0x52536df7, (q31_t)0x524e9e34,
	(q31_t)0x5249ce3f, (q31_t)0x5244fe17,
	(q31_t)0x52402dbc, (q31_t)0x523b5d2e, (q31_t)0x52368c6e, (q31_t)0x5231bb7b, (q31_t)0x522cea55, (q31_t)0x522818fc,
	(q31_t)0x52234771, (q31_t)0x521e75b3,
	(q31_t)0x5219a3c3, (q31_t)0x5214d1a0, (q31_t)0x520fff4a, (q31_t)0x520b2cc2, (q31_t)0x52065a07, (q31_t)0x52018719,
	(q31_t)0x51fcb3f9, (q31_t)0x51f7e0a6,
	(q31_t)0x51f30d21, (q31_t)0x51ee3969, (q31_t)0x51e9657e, (q31_t)0x51e49162, (q31_t)0x51dfbd12, (q31_t)0x51dae890,
	(q31_t)0x51d613dc, (q31_t)0x51d13ef5,
	(q31_t)0x51cc69db, (q31_t)0x51c79490, (q31_t)0x51c2bf11, (q31_t)0x51bde960, (q31_t)0x51b9137d, (q31_t)0x51b43d68,
	(q31_t)0x51af6720, (q31_t)0x51aa90a5,
	(q31_t)0x51a5b9f9, (q31_t)0x51a0e31a, (q31_t)0x519c0c08, (q31_t)0x519734c4, (q31_t)0x51925d4e, (q31_t)0x518d85a6,
	(q31_t)0x5188adcb, (q31_t)0x5183d5be,
	(q31_t)0x517efd7f, (q31_t)0x517a250d, (q31_t)0x51754c69, (q31_t)0x51707393, (q31_t)0x516b9a8b, (q31_t)0x5166c150,
	(q31_t)0x5161e7e4, (q31_t)0x515d0e45,
	(q31_t)0x51583473, (q31_t)0x51535a70, (q31_t)0x514e803b, (q31_t)0x5149a5d3, (q31_t)0x5144cb39, (q31_t)0x513ff06d,
	(q31_t)0x513b156f, (q31_t)0x51363a3f,
	(q31_t)0x51315edd, (q31_t)0x512c8348, (q31_t)0x5127a782, (q31_t)0x5122cb8a, (q31_t)0x511def5f, (q31_t)0x51191302,
	(q31_t)0x51143674, (q31_t)0x510f59b3,
	(q31_t)0x510a7cc1, (q31_t)0x51059f9c, (q31_t)0x5100c246, (q31_t)0x50fbe4bd, (q31_t)0x50f70703, (q31_t)0x50f22916,
	(q31_t)0x50ed4af8, (q31_t)0x50e86ca8,
	(q31_t)0x50e38e25, (q31_t)0x50deaf71, (q31_t)0x50d9d08b, (q31_t)0x50d4f173, (q31_t)0x50d0122a, (q31_t)0x50cb32ae,
	(q31_t)0x50c65301, (q31_t)0x50c17322,
	(q31_t)0x50bc9311, (q31_t)0x50b7b2ce, (q31_t)0x50b2d259, (q31_t)0x50adf1b3, (q31_t)0x50a910db, (q31_t)0x50a42fd1,
	(q31_t)0x509f4e95, (q31_t)0x509a6d28,
	(q31_t)0x50958b88, (q31_t)0x5090a9b8, (q31_t)0x508bc7b5, (q31_t)0x5086e581, (q31_t)0x5082031b, (q31_t)0x507d2083,
	(q31_t)0x50783dba, (q31_t)0x50735abf,
	(q31_t)0x506e7793, (q31_t)0x50699435, (q31_t)0x5064b0a5, (q31_t)0x505fcce4, (q31_t)0x505ae8f1, (q31_t)0x505604cd,
	(q31_t)0x50512077, (q31_t)0x504c3bef,
	(q31_t)0x50475736, (q31_t)0x5042724c, (q31_t)0x503d8d30, (q31_t)0x5038a7e2, (q31_t)0x5033c263, (q31_t)0x502edcb2,
	(q31_t)0x5029f6d1, (q31_t)0x502510bd,
	(q31_t)0x50202a78, (q31_t)0x501b4402, (q31_t)0x50165d5a, (q31_t)0x50117681, (q31_t)0x500c8f77, (q31_t)0x5007a83b,
	(q31_t)0x5002c0cd, (q31_t)0x4ffdd92f,
	(q31_t)0x4ff8f15f, (q31_t)0x4ff4095e, (q31_t)0x4fef212b, (q31_t)0x4fea38c7, (q31_t)0x4fe55032, (q31_t)0x4fe0676c,
	(q31_t)0x4fdb7e74, (q31_t)0x4fd6954b,
	(q31_t)0x4fd1abf0, (q31_t)0x4fccc265, (q31_t)0x4fc7d8a8, (q31_t)0x4fc2eeba, (q31_t)0x4fbe049b, (q31_t)0x4fb91a4b,
	(q31_t)0x4fb42fc9, (q31_t)0x4faf4517,
	(q31_t)0x4faa5a33, (q31_t)0x4fa56f1e, (q31_t)0x4fa083d8, (q31_t)0x4f9b9861, (q31_t)0x4f96acb8, (q31_t)0x4f91c0df,
	(q31_t)0x4f8cd4d4, (q31_t)0x4f87e899,
	(q31_t)0x4f82fc2c, (q31_t)0x4f7e0f8f, (q31_t)0x4f7922c0, (q31_t)0x4f7435c0, (q31_t)0x4f6f488f, (q31_t)0x4f6a5b2e,
	(q31_t)0x4f656d9b, (q31_t)0x4f607fd7,
	(q31_t)0x4f5b91e3, (q31_t)0x4f56a3bd, (q31_t)0x4f51b566, (q31_t)0x4f4cc6df, (q31_t)0x4f47d827, (q31_t)0x4f42e93d,
	(q31_t)0x4f3dfa23, (q31_t)0x4f390ad8,
	(q31_t)0x4f341b5c, (q31_t)0x4f2f2baf, (q31_t)0x4f2a3bd2, (q31_t)0x4f254bc3, (q31_t)0x4f205b84, (q31_t)0x4f1b6b14,
	(q31_t)0x4f167a73, (q31_t)0x4f1189a1,
	(q31_t)0x4f0c989f, (q31_t)0x4f07a76b, (q31_t)0x4f02b608, (q31_t)0x4efdc473, (q31_t)0x4ef8d2ad, (q31_t)0x4ef3e0b7,
	(q31_t)0x4eeeee90, (q31_t)0x4ee9fc39,
	(q31_t)0x4ee509b1, (q31_t)0x4ee016f8, (q31_t)0x4edb240e, (q31_t)0x4ed630f4, (q31_t)0x4ed13da9, (q31_t)0x4ecc4a2e,
	(q31_t)0x4ec75682, (q31_t)0x4ec262a5,
	(q31_t)0x4ebd6e98, (q31_t)0x4eb87a5a, (q31_t)0x4eb385ec, (q31_t)0x4eae914d, (q31_t)0x4ea99c7d, (q31_t)0x4ea4a77d,
	(q31_t)0x4e9fb24d, (q31_t)0x4e9abcec,
	(q31_t)0x4e95c75b, (q31_t)0x4e90d199, (q31_t)0x4e8bdba6, (q31_t)0x4e86e583, (q31_t)0x4e81ef30, (q31_t)0x4e7cf8ac,
	(q31_t)0x4e7801f8, (q31_t)0x4e730b14,
	(q31_t)0x4e6e13ff, (q31_t)0x4e691cba, (q31_t)0x4e642544, (q31_t)0x4e5f2d9e, (q31_t)0x4e5a35c7, (q31_t)0x4e553dc1,
	(q31_t)0x4e50458a, (q31_t)0x4e4b4d22,
	(q31_t)0x4e46548b, (q31_t)0x4e415bc3, (q31_t)0x4e3c62cb, (q31_t)0x4e3769a2, (q31_t)0x4e32704a, (q31_t)0x4e2d76c1,
	(q31_t)0x4e287d08, (q31_t)0x4e23831e,
	(q31_t)0x4e1e8905, (q31_t)0x4e198ebb, (q31_t)0x4e149441, (q31_t)0x4e0f9997, (q31_t)0x4e0a9ebd, (q31_t)0x4e05a3b2,
	(q31_t)0x4e00a878, (q31_t)0x4dfbad0d,
	(q31_t)0x4df6b173, (q31_t)0x4df1b5a8, (q31_t)0x4decb9ad, (q31_t)0x4de7bd82, (q31_t)0x4de2c127, (q31_t)0x4dddc49c,
	(q31_t)0x4dd8c7e1, (q31_t)0x4dd3caf6,
	(q31_t)0x4dcecdda, (q31_t)0x4dc9d08f, (q31_t)0x4dc4d314, (q31_t)0x4dbfd569, (q31_t)0x4dbad78e, (q31_t)0x4db5d983,
	(q31_t)0x4db0db48, (q31_t)0x4dabdcdd,
	(q31_t)0x4da6de43, (q31_t)0x4da1df78, (q31_t)0x4d9ce07d, (q31_t)0x4d97e153, (q31_t)0x4d92e1f9, (q31_t)0x4d8de26f,
	(q31_t)0x4d88e2b5, (q31_t)0x4d83e2cb,
	(q31_t)0x4d7ee2b1, (q31_t)0x4d79e268, (q31_t)0x4d74e1ef, (q31_t)0x4d6fe146, (q31_t)0x4d6ae06d, (q31_t)0x4d65df64,
	(q31_t)0x4d60de2c, (q31_t)0x4d5bdcc4,
	(q31_t)0x4d56db2d, (q31_t)0x4d51d965, (q31_t)0x4d4cd76e, (q31_t)0x4d47d547, (q31_t)0x4d42d2f1, (q31_t)0x4d3dd06b,
	(q31_t)0x4d38cdb5, (q31_t)0x4d33cad0,
	(q31_t)0x4d2ec7bb, (q31_t)0x4d29c476, (q31_t)0x4d24c102, (q31_t)0x4d1fbd5e, (q31_t)0x4d1ab98b, (q31_t)0x4d15b588,
	(q31_t)0x4d10b155, (q31_t)0x4d0bacf3,
	(q31_t)0x4d06a862, (q31_t)0x4d01a3a0, (q31_t)0x4cfc9eb0, (q31_t)0x4cf79990, (q31_t)0x4cf29440, (q31_t)0x4ced8ec1,
	(q31_t)0x4ce88913, (q31_t)0x4ce38335,
	(q31_t)0x4cde7d28, (q31_t)0x4cd976eb, (q31_t)0x4cd4707f, (q31_t)0x4ccf69e3, (q31_t)0x4cca6318, (q31_t)0x4cc55c1e,
	(q31_t)0x4cc054f4, (q31_t)0x4cbb4d9b,
	(q31_t)0x4cb64613, (q31_t)0x4cb13e5b, (q31_t)0x4cac3674, (q31_t)0x4ca72e5e, (q31_t)0x4ca22619, (q31_t)0x4c9d1da4,
	(q31_t)0x4c981500, (q31_t)0x4c930c2d,
	(q31_t)0x4c8e032a, (q31_t)0x4c88f9f8, (q31_t)0x4c83f097, (q31_t)0x4c7ee707, (q31_t)0x4c79dd48, (q31_t)0x4c74d359,
	(q31_t)0x4c6fc93b, (q31_t)0x4c6abeef,
	(q31_t)0x4c65b473, (q31_t)0x4c60a9c8, (q31_t)0x4c5b9eed, (q31_t)0x4c5693e4, (q31_t)0x4c5188ac, (q31_t)0x4c4c7d44,
	(q31_t)0x4c4771ae, (q31_t)0x4c4265e8,
	(q31_t)0x4c3d59f3, (q31_t)0x4c384dd0, (q31_t)0x4c33417d, (q31_t)0x4c2e34fb, (q31_t)0x4c29284b, (q31_t)0x4c241b6b,
	(q31_t)0x4c1f0e5c, (q31_t)0x4c1a011f,
	(q31_t)0x4c14f3b2, (q31_t)0x4c0fe617, (q31_t)0x4c0ad84c, (q31_t)0x4c05ca53, (q31_t)0x4c00bc2b, (q31_t)0x4bfbadd4,
	(q31_t)0x4bf69f4e, (q31_t)0x4bf19099,
	(q31_t)0x4bec81b5, (q31_t)0x4be772a3, (q31_t)0x4be26362, (q31_t)0x4bdd53f2, (q31_t)0x4bd84453, (q31_t)0x4bd33485,
	(q31_t)0x4bce2488, (q31_t)0x4bc9145d,
	(q31_t)0x4bc40403, (q31_t)0x4bbef37b, (q31_t)0x4bb9e2c3, (q31_t)0x4bb4d1dd, (q31_t)0x4bafc0c8, (q31_t)0x4baaaf85,
	(q31_t)0x4ba59e12, (q31_t)0x4ba08c72,
	(q31_t)0x4b9b7aa2, (q31_t)0x4b9668a4, (q31_t)0x4b915677, (q31_t)0x4b8c441c, (q31_t)0x4b873192, (q31_t)0x4b821ed9,
	(q31_t)0x4b7d0bf2, (q31_t)0x4b77f8dc,
	(q31_t)0x4b72e598, (q31_t)0x4b6dd225, (q31_t)0x4b68be84, (q31_t)0x4b63aab4, (q31_t)0x4b5e96b6, (q31_t)0x4b598289,
	(q31_t)0x4b546e2d, (q31_t)0x4b4f59a4,
	(q31_t)0x4b4a44eb, (q31_t)0x4b453005, (q31_t)0x4b401aef, (q31_t)0x4b3b05ac, (q31_t)0x4b35f03a, (q31_t)0x4b30da9a,
	(q31_t)0x4b2bc4cb, (q31_t)0x4b26aece,
	(q31_t)0x4b2198a2, (q31_t)0x4b1c8248, (q31_t)0x4b176bc0, (q31_t)0x4b12550a, (q31_t)0x4b0d3e25, (q31_t)0x4b082712,
	(q31_t)0x4b030fd1, (q31_t)0x4afdf861,
	(q31_t)0x4af8e0c3, (q31_t)0x4af3c8f7, (q31_t)0x4aeeb0fd, (q31_t)0x4ae998d4, (q31_t)0x4ae4807d, (q31_t)0x4adf67f8,
	(q31_t)0x4ada4f45, (q31_t)0x4ad53664,
	(q31_t)0x4ad01d54, (q31_t)0x4acb0417, (q31_t)0x4ac5eaab, (q31_t)0x4ac0d111, (q31_t)0x4abbb749, (q31_t)0x4ab69d53,
	(q31_t)0x4ab1832f, (q31_t)0x4aac68dc,
	(q31_t)0x4aa74e5c, (q31_t)0x4aa233ae, (q31_t)0x4a9d18d1, (q31_t)0x4a97fdc7, (q31_t)0x4a92e28e, (q31_t)0x4a8dc728,
	(q31_t)0x4a88ab93, (q31_t)0x4a838fd1,
	(q31_t)0x4a7e73e0, (q31_t)0x4a7957c2, (q31_t)0x4a743b76, (q31_t)0x4a6f1efc, (q31_t)0x4a6a0253, (q31_t)0x4a64e57d,
	(q31_t)0x4a5fc879, (q31_t)0x4a5aab48,
	(q31_t)0x4a558de8, (q31_t)0x4a50705a, (q31_t)0x4a4b529f, (q31_t)0x4a4634b6, (q31_t)0x4a41169f, (q31_t)0x4a3bf85a,
	(q31_t)0x4a36d9e7, (q31_t)0x4a31bb47,
	(q31_t)0x4a2c9c79, (q31_t)0x4a277d7d, (q31_t)0x4a225e53, (q31_t)0x4a1d3efc, (q31_t)0x4a181f77, (q31_t)0x4a12ffc4,
	(q31_t)0x4a0ddfe4, (q31_t)0x4a08bfd5,
	(q31_t)0x4a039f9a, (q31_t)0x49fe7f30, (q31_t)0x49f95e99, (q31_t)0x49f43dd4, (q31_t)0x49ef1ce2, (q31_t)0x49e9fbc2,
	(q31_t)0x49e4da74, (q31_t)0x49dfb8f9,
	(q31_t)0x49da9750, (q31_t)0x49d5757a, (q31_t)0x49d05376, (q31_t)0x49cb3145, (q31_t)0x49c60ee6, (q31_t)0x49c0ec59,
	(q31_t)0x49bbc9a0, (q31_t)0x49b6a6b8,
	(q31_t)0x49b183a3, (q31_t)0x49ac6061, (q31_t)0x49a73cf1, (q31_t)0x49a21954, (q31_t)0x499cf589, (q31_t)0x4997d191,
	(q31_t)0x4992ad6c, (q31_t)0x498d8919,
	(q31_t)0x49886499, (q31_t)0x49833fec, (q31_t)0x497e1b11, (q31_t)0x4978f609, (q31_t)0x4973d0d3, (q31_t)0x496eab70,
	(q31_t)0x496985e0, (q31_t)0x49646023,
	(q31_t)0x495f3a38, (q31_t)0x495a1420, (q31_t)0x4954eddb, (q31_t)0x494fc768, (q31_t)0x494aa0c9, (q31_t)0x494579fc,
	(q31_t)0x49405302, (q31_t)0x493b2bdb,
	(q31_t)0x49360486, (q31_t)0x4930dd05, (q31_t)0x492bb556, (q31_t)0x49268d7a, (q31_t)0x49216571, (q31_t)0x491c3d3b,
	(q31_t)0x491714d8, (q31_t)0x4911ec47,
	(q31_t)0x490cc38a, (q31_t)0x49079aa0, (q31_t)0x49027188, (q31_t)0x48fd4844, (q31_t)0x48f81ed2, (q31_t)0x48f2f534,
	(q31_t)0x48edcb68, (q31_t)0x48e8a170,
	(q31_t)0x48e3774a, (q31_t)0x48de4cf8, (q31_t)0x48d92278, (q31_t)0x48d3f7cc, (q31_t)0x48ceccf3, (q31_t)0x48c9a1ed,
	(q31_t)0x48c476b9, (q31_t)0x48bf4b59,
	(q31_t)0x48ba1fcd, (q31_t)0x48b4f413, (q31_t)0x48afc82c, (q31_t)0x48aa9c19, (q31_t)0x48a56fd9, (q31_t)0x48a0436c,
	(q31_t)0x489b16d2, (q31_t)0x4895ea0b,
	(q31_t)0x4890bd18, (q31_t)0x488b8ff8, (q31_t)0x488662ab, (q31_t)0x48813531, (q31_t)0x487c078b, (q31_t)0x4876d9b8,
	(q31_t)0x4871abb8, (q31_t)0x486c7d8c,
	(q31_t)0x48674f33, (q31_t)0x486220ad, (q31_t)0x485cf1fa, (q31_t)0x4857c31b, (q31_t)0x48529410, (q31_t)0x484d64d7,
	(q31_t)0x48483572, (q31_t)0x484305e1,
	(q31_t)0x483dd623, (q31_t)0x4838a638, (q31_t)0x48337621, (q31_t)0x482e45dd, (q31_t)0x4829156d, (q31_t)0x4823e4d0,
	(q31_t)0x481eb407, (q31_t)0x48198311,
	(q31_t)0x481451ef, (q31_t)0x480f20a0, (q31_t)0x4809ef25, (q31_t)0x4804bd7e, (q31_t)0x47ff8baa, (q31_t)0x47fa59a9,
	(q31_t)0x47f5277d, (q31_t)0x47eff523,
	(q31_t)0x47eac29e, (q31_t)0x47e58fec, (q31_t)0x47e05d0e, (q31_t)0x47db2a03, (q31_t)0x47d5f6cc, (q31_t)0x47d0c369,
	(q31_t)0x47cb8fd9, (q31_t)0x47c65c1d,
	(q31_t)0x47c12835, (q31_t)0x47bbf421, (q31_t)0x47b6bfe0, (q31_t)0x47b18b74, (q31_t)0x47ac56da, (q31_t)0x47a72215,
	(q31_t)0x47a1ed24, (q31_t)0x479cb806,
	(q31_t)0x479782bc, (q31_t)0x47924d46, (q31_t)0x478d17a4, (q31_t)0x4787e1d6, (q31_t)0x4782abdb, (q31_t)0x477d75b5,
	(q31_t)0x47783f62, (q31_t)0x477308e3,
	(q31_t)0x476dd239, (q31_t)0x47689b62, (q31_t)0x4763645f, (q31_t)0x475e2d30, (q31_t)0x4758f5d5, (q31_t)0x4753be4e,
	(q31_t)0x474e869b, (q31_t)0x47494ebc,
	(q31_t)0x474416b1, (q31_t)0x473ede7a, (q31_t)0x4739a617, (q31_t)0x47346d89, (q31_t)0x472f34ce, (q31_t)0x4729fbe7,
	(q31_t)0x4724c2d5, (q31_t)0x471f8996,
	(q31_t)0x471a502c, (q31_t)0x47151696, (q31_t)0x470fdcd4, (q31_t)0x470aa2e6, (q31_t)0x470568cd, (q31_t)0x47002e87,
	(q31_t)0x46faf416, (q31_t)0x46f5b979,
	(q31_t)0x46f07eb0, (q31_t)0x46eb43bc, (q31_t)0x46e6089b, (q31_t)0x46e0cd4f, (q31_t)0x46db91d8, (q31_t)0x46d65634,
	(q31_t)0x46d11a65, (q31_t)0x46cbde6a,
	(q31_t)0x46c6a244, (q31_t)0x46c165f1, (q31_t)0x46bc2974, (q31_t)0x46b6ecca, (q31_t)0x46b1aff5, (q31_t)0x46ac72f4,
	(q31_t)0x46a735c8, (q31_t)0x46a1f870,
	(q31_t)0x469cbaed, (q31_t)0x46977d3e, (q31_t)0x46923f63, (q31_t)0x468d015d, (q31_t)0x4687c32c, (q31_t)0x468284cf,
	(q31_t)0x467d4646, (q31_t)0x46780792,
	(q31_t)0x4672c8b3, (q31_t)0x466d89a8, (q31_t)0x46684a71, (q31_t)0x46630b0f, (q31_t)0x465dcb82, (q31_t)0x46588bc9,
	(q31_t)0x46534be5, (q31_t)0x464e0bd6,
	(q31_t)0x4648cb9b, (q31_t)0x46438b35, (q31_t)0x463e4aa3, (q31_t)0x463909e7, (q31_t)0x4633c8fe, (q31_t)0x462e87eb,
	(q31_t)0x462946ac, (q31_t)0x46240542,
	(q31_t)0x461ec3ad, (q31_t)0x461981ec, (q31_t)0x46144001, (q31_t)0x460efde9, (q31_t)0x4609bba7, (q31_t)0x4604793a,
	(q31_t)0x45ff36a1, (q31_t)0x45f9f3dd,
	(q31_t)0x45f4b0ee, (q31_t)0x45ef6dd4, (q31_t)0x45ea2a8f, (q31_t)0x45e4e71f, (q31_t)0x45dfa383, (q31_t)0x45da5fbc,
	(q31_t)0x45d51bcb, (q31_t)0x45cfd7ae,
	(q31_t)0x45ca9366, (q31_t)0x45c54ef3, (q31_t)0x45c00a55, (q31_t)0x45bac58c, (q31_t)0x45b58098, (q31_t)0x45b03b79,
	(q31_t)0x45aaf630, (q31_t)0x45a5b0bb,
	(q31_t)0x45a06b1b, (q31_t)0x459b2550, (q31_t)0x4595df5a, (q31_t)0x45909939, (q31_t)0x458b52ee, (q31_t)0x45860c77,
	(q31_t)0x4580c5d6, (q31_t)0x457b7f0a,
	(q31_t)0x45763813, (q31_t)0x4570f0f1, (q31_t)0x456ba9a4, (q31_t)0x4566622c, (q31_t)0x45611a8a, (q31_t)0x455bd2bc,
	(q31_t)0x45568ac4, (q31_t)0x455142a2,
	(q31_t)0x454bfa54, (q31_t)0x4546b1dc, (q31_t)0x45416939, (q31_t)0x453c206b, (q31_t)0x4536d773, (q31_t)0x45318e4f,
	(q31_t)0x452c4502, (q31_t)0x4526fb89,
	(q31_t)0x4521b1e6, (q31_t)0x451c6818, (q31_t)0x45171e20, (q31_t)0x4511d3fd, (q31_t)0x450c89af, (q31_t)0x45073f37,
	(q31_t)0x4501f494, (q31_t)0x44fca9c6,
	(q31_t)0x44f75ecf, (q31_t)0x44f213ac, (q31_t)0x44ecc85f, (q31_t)0x44e77ce7, (q31_t)0x44e23145, (q31_t)0x44dce579,
	(q31_t)0x44d79982, (q31_t)0x44d24d60,
	(q31_t)0x44cd0114, (q31_t)0x44c7b49e, (q31_t)0x44c267fd, (q31_t)0x44bd1b32, (q31_t)0x44b7ce3c, (q31_t)0x44b2811c,
	(q31_t)0x44ad33d2, (q31_t)0x44a7e65d,
	(q31_t)0x44a298be, (q31_t)0x449d4af5, (q31_t)0x4497fd01, (q31_t)0x4492aee3, (q31_t)0x448d609b, (q31_t)0x44881228,
	(q31_t)0x4482c38b, (q31_t)0x447d74c4,
	(q31_t)0x447825d2, (q31_t)0x4472d6b7, (q31_t)0x446d8771, (q31_t)0x44683801, (q31_t)0x4462e866, (q31_t)0x445d98a2,
	(q31_t)0x445848b3, (q31_t)0x4452f89b,
	(q31_t)0x444da858, (q31_t)0x444857ea, (q31_t)0x44430753, (q31_t)0x443db692, (q31_t)0x443865a7, (q31_t)0x44331491,
	(q31_t)0x442dc351, (q31_t)0x442871e8,
	(q31_t)0x44232054, (q31_t)0x441dce96, (q31_t)0x44187caf, (q31_t)0x44132a9d, (q31_t)0x440dd861, (q31_t)0x440885fc,
	(q31_t)0x4403336c, (q31_t)0x43fde0b2,
	(q31_t)0x43f88dcf, (q31_t)0x43f33ac1, (q31_t)0x43ede78a, (q31_t)0x43e89429, (q31_t)0x43e3409d, (q31_t)0x43ddece8,
	(q31_t)0x43d8990a, (q31_t)0x43d34501,
	(q31_t)0x43cdf0ce, (q31_t)0x43c89c72, (q31_t)0x43c347eb, (q31_t)0x43bdf33b, (q31_t)0x43b89e62, (q31_t)0x43b3495e,
	(q31_t)0x43adf431, (q31_t)0x43a89ed9,
	(q31_t)0x43a34959, (q31_t)0x439df3ae, (q31_t)0x43989dda, (q31_t)0x439347dc, (q31_t)0x438df1b4, (q31_t)0x43889b63,
	(q31_t)0x438344e8, (q31_t)0x437dee43,
	(q31_t)0x43789775, (q31_t)0x4373407d, (q31_t)0x436de95b, (q31_t)0x43689210, (q31_t)0x43633a9c, (q31_t)0x435de2fd,
	(q31_t)0x43588b36, (q31_t)0x43533344,
	(q31_t)0x434ddb29, (q31_t)0x434882e5, (q31_t)0x43432a77, (q31_t)0x433dd1e0, (q31_t)0x4338791f, (q31_t)0x43332035,
	(q31_t)0x432dc721, (q31_t)0x43286de4,
	(q31_t)0x4323147d, (q31_t)0x431dbaed, (q31_t)0x43186133, (q31_t)0x43130751, (q31_t)0x430dad44, (q31_t)0x4308530f,
	(q31_t)0x4302f8b0, (q31_t)0x42fd9e28,
	(q31_t)0x42f84376, (q31_t)0x42f2e89b, (q31_t)0x42ed8d97, (q31_t)0x42e83269, (q31_t)0x42e2d713, (q31_t)0x42dd7b93,
	(q31_t)0x42d81fe9, (q31_t)0x42d2c417,
	(q31_t)0x42cd681b, (q31_t)0x42c80bf6, (q31_t)0x42c2afa8, (q31_t)0x42bd5331, (q31_t)0x42b7f690, (q31_t)0x42b299c7,
	(q31_t)0x42ad3cd4, (q31_t)0x42a7dfb8,
	(q31_t)0x42a28273, (q31_t)0x429d2505, (q31_t)0x4297c76e, (q31_t)0x429269ae, (q31_t)0x428d0bc4, (q31_t)0x4287adb2,
	(q31_t)0x42824f76, (q31_t)0x427cf112,
	(q31_t)0x42779285, (q31_t)0x427233ce, (q31_t)0x426cd4ef, (q31_t)0x426775e6, (q31_t)0x426216b5, (q31_t)0x425cb75a,
	(q31_t)0x425757d7, (q31_t)0x4251f82b,
	(q31_t)0x424c9856, (q31_t)0x42473858, (q31_t)0x4241d831, (q31_t)0x423c77e1, (q31_t)0x42371769, (q31_t)0x4231b6c7,
	(q31_t)0x422c55fd, (q31_t)0x4226f50a,
	(q31_t)0x422193ee, (q31_t)0x421c32a9, (q31_t)0x4216d13c, (q31_t)0x42116fa5, (q31_t)0x420c0de6, (q31_t)0x4206abfe,
	(q31_t)0x420149ee, (q31_t)0x41fbe7b5,
	(q31_t)0x41f68553, (q31_t)0x41f122c8, (q31_t)0x41ebc015, (q31_t)0x41e65d39, (q31_t)0x41e0fa35, (q31_t)0x41db9707,
	(q31_t)0x41d633b1, (q31_t)0x41d0d033,
	(q31_t)0x41cb6c8c, (q31_t)0x41c608bc, (q31_t)0x41c0a4c4, (q31_t)0x41bb40a3, (q31_t)0x41b5dc5a, (q31_t)0x41b077e8,
	(q31_t)0x41ab134e, (q31_t)0x41a5ae8b,
	(q31_t)0x41a049a0, (q31_t)0x419ae48c, (q31_t)0x41957f4f, (q31_t)0x419019eb, (q31_t)0x418ab45d, (q31_t)0x41854ea8,
	(q31_t)0x417fe8ca, (q31_t)0x417a82c3,
	(q31_t)0x41751c94, (q31_t)0x416fb63d, (q31_t)0x416a4fbd, (q31_t)0x4164e916, (q31_t)0x415f8245, (q31_t)0x415a1b4d,
	(q31_t)0x4154b42c, (q31_t)0x414f4ce2,
	(q31_t)0x4149e571, (q31_t)0x41447dd7, (q31_t)0x413f1615, (q31_t)0x4139ae2b, (q31_t)0x41344618, (q31_t)0x412edddd,
	(q31_t)0x4129757b, (q31_t)0x41240cef,
	(q31_t)0x411ea43c, (q31_t)0x41193b61, (q31_t)0x4113d25d, (q31_t)0x410e6931, (q31_t)0x4108ffdd, (q31_t)0x41039661,
	(q31_t)0x40fe2cbd, (q31_t)0x40f8c2f1,
	(q31_t)0x40f358fc, (q31_t)0x40edeee0, (q31_t)0x40e8849b, (q31_t)0x40e31a2f, (q31_t)0x40ddaf9b, (q31_t)0x40d844de,
	(q31_t)0x40d2d9f9, (q31_t)0x40cd6eed,
	(q31_t)0x40c803b8, (q31_t)0x40c2985c, (q31_t)0x40bd2cd8, (q31_t)0x40b7c12b, (q31_t)0x40b25557, (q31_t)0x40ace95b,
	(q31_t)0x40a77d37, (q31_t)0x40a210eb,
	(q31_t)0x409ca477, (q31_t)0x409737dc, (q31_t)0x4091cb18, (q31_t)0x408c5e2d, (q31_t)0x4086f11a, (q31_t)0x408183df,
	(q31_t)0x407c167c, (q31_t)0x4076a8f1,
	(q31_t)0x40713b3f, (q31_t)0x406bcd65, (q31_t)0x40665f63, (q31_t)0x4060f13a, (q31_t)0x405b82e9, (q31_t)0x40561470,
	(q31_t)0x4050a5cf, (q31_t)0x404b3707,
	(q31_t)0x4045c817, (q31_t)0x404058ff, (q31_t)0x403ae9c0, (q31_t)0x40357a59, (q31_t)0x40300acb, (q31_t)0x402a9b15,
	(q31_t)0x40252b37, (q31_t)0x401fbb32,
	(q31_t)0x401a4b05, (q31_t)0x4014dab1, (q31_t)0x400f6a35, (q31_t)0x4009f992, (q31_t)0x400488c7, (q31_t)0x3fff17d5,
	(q31_t)0x3ff9a6bb, (q31_t)0x3ff4357a,
	(q31_t)0x3feec411, (q31_t)0x3fe95281, (q31_t)0x3fe3e0c9, (q31_t)0x3fde6eeb, (q31_t)0x3fd8fce4, (q31_t)0x3fd38ab6,
	(q31_t)0x3fce1861, (q31_t)0x3fc8a5e5,
	(q31_t)0x3fc33341, (q31_t)0x3fbdc076, (q31_t)0x3fb84d83, (q31_t)0x3fb2da6a, (q31_t)0x3fad6729, (q31_t)0x3fa7f3c0,
	(q31_t)0x3fa28031, (q31_t)0x3f9d0c7a,
	(q31_t)0x3f97989c, (q31_t)0x3f922496, (q31_t)0x3f8cb06a, (q31_t)0x3f873c16, (q31_t)0x3f81c79b, (q31_t)0x3f7c52f9,
	(q31_t)0x3f76de30, (q31_t)0x3f71693f,
	(q31_t)0x3f6bf428, (q31_t)0x3f667ee9, (q31_t)0x3f610983, (q31_t)0x3f5b93f6, (q31_t)0x3f561e42, (q31_t)0x3f50a867,
	(q31_t)0x3f4b3265, (q31_t)0x3f45bc3c,
	(q31_t)0x3f4045ec, (q31_t)0x3f3acf75, (q31_t)0x3f3558d7, (q31_t)0x3f2fe211, (q31_t)0x3f2a6b25, (q31_t)0x3f24f412,
	(q31_t)0x3f1f7cd8, (q31_t)0x3f1a0577,
	(q31_t)0x3f148def, (q31_t)0x3f0f1640, (q31_t)0x3f099e6b, (q31_t)0x3f04266e, (q31_t)0x3efeae4a, (q31_t)0x3ef93600,
	(q31_t)0x3ef3bd8f, (q31_t)0x3eee44f7,
	(q31_t)0x3ee8cc38, (q31_t)0x3ee35352, (q31_t)0x3eddda46, (q31_t)0x3ed86113, (q31_t)0x3ed2e7b9, (q31_t)0x3ecd6e38,
	(q31_t)0x3ec7f491, (q31_t)0x3ec27ac2,
	(q31_t)0x3ebd00cd, (q31_t)0x3eb786b2, (q31_t)0x3eb20c6f, (q31_t)0x3eac9206, (q31_t)0x3ea71777, (q31_t)0x3ea19cc1,
	(q31_t)0x3e9c21e4, (q31_t)0x3e96a6e0,
	(q31_t)0x3e912bb6, (q31_t)0x3e8bb065, (q31_t)0x3e8634ee, (q31_t)0x3e80b950, (q31_t)0x3e7b3d8c, (q31_t)0x3e75c1a1,
	(q31_t)0x3e70458f, (q31_t)0x3e6ac957,
	(q31_t)0x3e654cf8, (q31_t)0x3e5fd073, (q31_t)0x3e5a53c8, (q31_t)0x3e54d6f6, (q31_t)0x3e4f59fe, (q31_t)0x3e49dcdf,
	(q31_t)0x3e445f99, (q31_t)0x3e3ee22e,
	(q31_t)0x3e39649c, (q31_t)0x3e33e6e3, (q31_t)0x3e2e6904, (q31_t)0x3e28eaff, (q31_t)0x3e236cd4, (q31_t)0x3e1dee82,
	(q31_t)0x3e18700a, (q31_t)0x3e12f16b,
	(q31_t)0x3e0d72a6, (q31_t)0x3e07f3bb, (q31_t)0x3e0274aa, (q31_t)0x3dfcf572, (q31_t)0x3df77615, (q31_t)0x3df1f691,
	(q31_t)0x3dec76e6, (q31_t)0x3de6f716,
	(q31_t)0x3de1771f, (q31_t)0x3ddbf703, (q31_t)0x3dd676c0, (q31_t)0x3dd0f656, (q31_t)0x3dcb75c7, (q31_t)0x3dc5f512,
	(q31_t)0x3dc07436, (q31_t)0x3dbaf335,
	(q31_t)0x3db5720d, (q31_t)0x3daff0c0, (q31_t)0x3daa6f4c, (q31_t)0x3da4edb2, (q31_t)0x3d9f6bf2, (q31_t)0x3d99ea0d,
	(q31_t)0x3d946801, (q31_t)0x3d8ee5cf,
	(q31_t)0x3d896377, (q31_t)0x3d83e0f9, (q31_t)0x3d7e5e56, (q31_t)0x3d78db8c, (q31_t)0x3d73589d, (q31_t)0x3d6dd587,
	(q31_t)0x3d68524c, (q31_t)0x3d62ceeb,
	(q31_t)0x3d5d4b64, (q31_t)0x3d57c7b7, (q31_t)0x3d5243e4, (q31_t)0x3d4cbfeb, (q31_t)0x3d473bcd, (q31_t)0x3d41b789,
	(q31_t)0x3d3c331f, (q31_t)0x3d36ae8f,
	(q31_t)0x3d3129da, (q31_t)0x3d2ba4fe, (q31_t)0x3d261ffd, (q31_t)0x3d209ad7, (q31_t)0x3d1b158a, (q31_t)0x3d159018,
	(q31_t)0x3d100a80, (q31_t)0x3d0a84c3,
	(q31_t)0x3d04fee0, (q31_t)0x3cff78d7, (q31_t)0x3cf9f2a9, (q31_t)0x3cf46c55, (q31_t)0x3ceee5db, (q31_t)0x3ce95f3c,
	(q31_t)0x3ce3d877, (q31_t)0x3cde518d,
	(q31_t)0x3cd8ca7d, (q31_t)0x3cd34347, (q31_t)0x3ccdbbed, (q31_t)0x3cc8346c, (q31_t)0x3cc2acc6, (q31_t)0x3cbd24fb,
	(q31_t)0x3cb79d0a, (q31_t)0x3cb214f4,
	(q31_t)0x3cac8cb8, (q31_t)0x3ca70457, (q31_t)0x3ca17bd0, (q31_t)0x3c9bf324, (q31_t)0x3c966a53, (q31_t)0x3c90e15c,
	(q31_t)0x3c8b5840, (q31_t)0x3c85cefe,
	(q31_t)0x3c804598, (q31_t)0x3c7abc0c, (q31_t)0x3c75325a, (q31_t)0x3c6fa883, (q31_t)0x3c6a1e87, (q31_t)0x3c649466,
	(q31_t)0x3c5f0a20, (q31_t)0x3c597fb4,
	(q31_t)0x3c53f523, (q31_t)0x3c4e6a6d, (q31_t)0x3c48df91, (q31_t)0x3c435491, (q31_t)0x3c3dc96b, (q31_t)0x3c383e20,
	(q31_t)0x3c32b2b0, (q31_t)0x3c2d271b,
	(q31_t)0x3c279b61, (q31_t)0x3c220f81, (q31_t)0x3c1c837d, (q31_t)0x3c16f753, (q31_t)0x3c116b04, (q31_t)0x3c0bde91,
	(q31_t)0x3c0651f8, (q31_t)0x3c00c53a,
	(q31_t)0x3bfb3857, (q31_t)0x3bf5ab50, (q31_t)0x3bf01e23, (q31_t)0x3bea90d1, (q31_t)0x3be5035a, (q31_t)0x3bdf75bf,
	(q31_t)0x3bd9e7fe, (q31_t)0x3bd45a19,
	(q31_t)0x3bcecc0e, (q31_t)0x3bc93ddf, (q31_t)0x3bc3af8b, (q31_t)0x3bbe2112, (q31_t)0x3bb89274, (q31_t)0x3bb303b1,
	(q31_t)0x3bad74c9, (q31_t)0x3ba7e5bd,
	(q31_t)0x3ba2568c, (q31_t)0x3b9cc736, (q31_t)0x3b9737bb, (q31_t)0x3b91a81c, (q31_t)0x3b8c1857, (q31_t)0x3b86886e,
	(q31_t)0x3b80f861, (q31_t)0x3b7b682e,
	(q31_t)0x3b75d7d7, (q31_t)0x3b70475c, (q31_t)0x3b6ab6bb, (q31_t)0x3b6525f6, (q31_t)0x3b5f950c, (q31_t)0x3b5a03fe,
	(q31_t)0x3b5472cb, (q31_t)0x3b4ee173,
	(q31_t)0x3b494ff7, (q31_t)0x3b43be57, (q31_t)0x3b3e2c91, (q31_t)0x3b389aa8, (q31_t)0x3b330899, (q31_t)0x3b2d7666,
	(q31_t)0x3b27e40f, (q31_t)0x3b225193,
	(q31_t)0x3b1cbef3, (q31_t)0x3b172c2e, (q31_t)0x3b119945, (q31_t)0x3b0c0637, (q31_t)0x3b067305, (q31_t)0x3b00dfaf,
	(q31_t)0x3afb4c34, (q31_t)0x3af5b894,
	(q31_t)0x3af024d1, (q31_t)0x3aea90e9, (q31_t)0x3ae4fcdc, (q31_t)0x3adf68ac, (q31_t)0x3ad9d457, (q31_t)0x3ad43fdd,
	(q31_t)0x3aceab40, (q31_t)0x3ac9167e,
	(q31_t)0x3ac38198, (q31_t)0x3abdec8d, (q31_t)0x3ab8575f, (q31_t)0x3ab2c20c, (q31_t)0x3aad2c95, (q31_t)0x3aa796fa,
	(q31_t)0x3aa2013a, (q31_t)0x3a9c6b57,
	(q31_t)0x3a96d54f, (q31_t)0x3a913f23, (q31_t)0x3a8ba8d3, (q31_t)0x3a86125f, (q31_t)0x3a807bc7, (q31_t)0x3a7ae50a,
	(q31_t)0x3a754e2a, (q31_t)0x3a6fb726,
	(q31_t)0x3a6a1ffd, (q31_t)0x3a6488b1, (q31_t)0x3a5ef140, (q31_t)0x3a5959ab, (q31_t)0x3a53c1f3, (q31_t)0x3a4e2a16,
	(q31_t)0x3a489216, (q31_t)0x3a42f9f2,
	(q31_t)0x3a3d61a9, (q31_t)0x3a37c93d, (q31_t)0x3a3230ad, (q31_t)0x3a2c97f9, (q31_t)0x3a26ff21, (q31_t)0x3a216625,
	(q31_t)0x3a1bcd05, (q31_t)0x3a1633c1,
	(q31_t)0x3a109a5a, (q31_t)0x3a0b00cf, (q31_t)0x3a056720, (q31_t)0x39ffcd4d, (q31_t)0x39fa3356, (q31_t)0x39f4993c,
	(q31_t)0x39eefefe, (q31_t)0x39e9649c,
	(q31_t)0x39e3ca17, (q31_t)0x39de2f6d, (q31_t)0x39d894a0, (q31_t)0x39d2f9b0, (q31_t)0x39cd5e9b, (q31_t)0x39c7c363,
	(q31_t)0x39c22808, (q31_t)0x39bc8c89,
	(q31_t)0x39b6f0e6, (q31_t)0x39b1551f, (q31_t)0x39abb935, (q31_t)0x39a61d28, (q31_t)0x39a080f6, (q31_t)0x399ae4a2,
	(q31_t)0x39954829, (q31_t)0x398fab8e,
	(q31_t)0x398a0ece, (q31_t)0x398471ec, (q31_t)0x397ed4e5, (q31_t)0x397937bc, (q31_t)0x39739a6e, (q31_t)0x396dfcfe,
	(q31_t)0x39685f6a, (q31_t)0x3962c1b2,
	(q31_t)0x395d23d7, (q31_t)0x395785d9, (q31_t)0x3951e7b8, (q31_t)0x394c4973, (q31_t)0x3946ab0a, (q31_t)0x39410c7f,
	(q31_t)0x393b6dd0, (q31_t)0x3935cefd,
	(q31_t)0x39303008, (q31_t)0x392a90ef, (q31_t)0x3924f1b3, (q31_t)0x391f5254, (q31_t)0x3919b2d1, (q31_t)0x3914132b,
	(q31_t)0x390e7362, (q31_t)0x3908d376,
	(q31_t)0x39033367, (q31_t)0x38fd9334, (q31_t)0x38f7f2de, (q31_t)0x38f25266, (q31_t)0x38ecb1ca, (q31_t)0x38e7110a,
	(q31_t)0x38e17028, (q31_t)0x38dbcf23,
	(q31_t)0x38d62dfb, (q31_t)0x38d08caf, (q31_t)0x38caeb41, (q31_t)0x38c549af, (q31_t)0x38bfa7fb, (q31_t)0x38ba0623,
	(q31_t)0x38b46429, (q31_t)0x38aec20b,
	(q31_t)0x38a91fcb, (q31_t)0x38a37d67, (q31_t)0x389ddae1, (q31_t)0x38983838, (q31_t)0x3892956c, (q31_t)0x388cf27d,
	(q31_t)0x38874f6b, (q31_t)0x3881ac36,
	(q31_t)0x387c08de, (q31_t)0x38766564, (q31_t)0x3870c1c6, (q31_t)0x386b1e06, (q31_t)0x38657a23, (q31_t)0x385fd61d,
	(q31_t)0x385a31f5, (q31_t)0x38548daa,
	(q31_t)0x384ee93b, (q31_t)0x384944ab, (q31_t)0x38439ff7, (q31_t)0x383dfb21, (q31_t)0x38385628, (q31_t)0x3832b10d,
	(q31_t)0x382d0bce, (q31_t)0x3827666d,
	(q31_t)0x3821c0ea, (q31_t)0x381c1b44, (q31_t)0x3816757b, (q31_t)0x3810cf90, (q31_t)0x380b2982, (q31_t)0x38058351,
	(q31_t)0x37ffdcfe, (q31_t)0x37fa3688,
	(q31_t)0x37f48ff0, (q31_t)0x37eee936, (q31_t)0x37e94259, (q31_t)0x37e39b59, (q31_t)0x37ddf437, (q31_t)0x37d84cf2,
	(q31_t)0x37d2a58b, (q31_t)0x37ccfe02,
	(q31_t)0x37c75656, (q31_t)0x37c1ae87, (q31_t)0x37bc0697, (q31_t)0x37b65e84, (q31_t)0x37b0b64e, (q31_t)0x37ab0df6,
	(q31_t)0x37a5657c, (q31_t)0x379fbce0,
	(q31_t)0x379a1421, (q31_t)0x37946b40, (q31_t)0x378ec23d, (q31_t)0x37891917, (q31_t)0x37836fcf, (q31_t)0x377dc665,
	(q31_t)0x37781cd9, (q31_t)0x3772732a,
	(q31_t)0x376cc959, (q31_t)0x37671f66, (q31_t)0x37617551, (q31_t)0x375bcb1a, (q31_t)0x375620c1, (q31_t)0x37507645,
	(q31_t)0x374acba7, (q31_t)0x374520e7,
	(q31_t)0x373f7606, (q31_t)0x3739cb02, (q31_t)0x37341fdc, (q31_t)0x372e7493, (q31_t)0x3728c929, (q31_t)0x37231d9d,
	(q31_t)0x371d71ef, (q31_t)0x3717c61f,
	(q31_t)0x37121a2d, (q31_t)0x370c6e19, (q31_t)0x3706c1e2, (q31_t)0x3701158a, (q31_t)0x36fb6910, (q31_t)0x36f5bc75,
	(q31_t)0x36f00fb7, (q31_t)0x36ea62d7,
	(q31_t)0x36e4b5d6, (q31_t)0x36df08b2, (q31_t)0x36d95b6d, (q31_t)0x36d3ae06, (q31_t)0x36ce007d, (q31_t)0x36c852d2,
	(q31_t)0x36c2a506, (q31_t)0x36bcf718,
	(q31_t)0x36b74908, (q31_t)0x36b19ad6, (q31_t)0x36abec82, (q31_t)0x36a63e0d, (q31_t)0x36a08f76, (q31_t)0x369ae0bd,
	(q31_t)0x369531e3, (q31_t)0x368f82e7,
	(q31_t)0x3689d3c9, (q31_t)0x3684248a, (q31_t)0x367e7529, (q31_t)0x3678c5a7, (q31_t)0x36731602, (q31_t)0x366d663d,
	(q31_t)0x3667b655, (q31_t)0x3662064c,
	(q31_t)0x365c5622, (q31_t)0x3656a5d6, (q31_t)0x3650f569, (q31_t)0x364b44da, (q31_t)0x36459429, (q31_t)0x363fe357,
	(q31_t)0x363a3264, (q31_t)0x3634814f,
	(q31_t)0x362ed019, (q31_t)0x36291ec1, (q31_t)0x36236d48, (q31_t)0x361dbbad, (q31_t)0x361809f1, (q31_t)0x36125814,
	(q31_t)0x360ca615, (q31_t)0x3606f3f5,
	(q31_t)0x360141b4, (q31_t)0x35fb8f52, (q31_t)0x35f5dcce, (q31_t)0x35f02a28, (q31_t)0x35ea7762, (q31_t)0x35e4c47a,
	(q31_t)0x35df1171, (q31_t)0x35d95e47,
	(q31_t)0x35d3aafc, (q31_t)0x35cdf78f, (q31_t)0x35c84401, (q31_t)0x35c29052, (q31_t)0x35bcdc82, (q31_t)0x35b72891,
	(q31_t)0x35b1747e, (q31_t)0x35abc04b,
	(q31_t)0x35a60bf6, (q31_t)0x35a05781, (q31_t)0x359aa2ea, (q31_t)0x3594ee32, (q31_t)0x358f3959, (q31_t)0x3589845f,
	(q31_t)0x3583cf44, (q31_t)0x357e1a08,
	(q31_t)0x357864ab, (q31_t)0x3572af2d, (q31_t)0x356cf98e, (q31_t)0x356743ce, (q31_t)0x35618ded, (q31_t)0x355bd7eb,
	(q31_t)0x355621c9, (q31_t)0x35506b85,
	(q31_t)0x354ab520, (q31_t)0x3544fe9b, (q31_t)0x353f47f5, (q31_t)0x3539912e, (q31_t)0x3533da46, (q31_t)0x352e233d,
	(q31_t)0x35286c14, (q31_t)0x3522b4c9,
	(q31_t)0x351cfd5e, (q31_t)0x351745d2, (q31_t)0x35118e26, (q31_t)0x350bd658, (q31_t)0x35061e6a, (q31_t)0x3500665c,
	(q31_t)0x34faae2c, (q31_t)0x34f4f5dc,
	(q31_t)0x34ef3d6b, (q31_t)0x34e984da, (q31_t)0x34e3cc28, (q31_t)0x34de1355, (q31_t)0x34d85a62, (q31_t)0x34d2a14e,
	(q31_t)0x34cce819, (q31_t)0x34c72ec4,
	(q31_t)0x34c1754e, (q31_t)0x34bbbbb8, (q31_t)0x34b60202, (q31_t)0x34b0482a, (q31_t)0x34aa8e33, (q31_t)0x34a4d41a,
	(q31_t)0x349f19e2, (q31_t)0x34995f88,
	(q31_t)0x3493a50f, (q31_t)0x348dea75, (q31_t)0x34882fba, (q31_t)0x348274e0, (q31_t)0x347cb9e4, (q31_t)0x3476fec9,
	(q31_t)0x3471438d, (q31_t)0x346b8830,
	(q31_t)0x3465ccb4, (q31_t)0x34601117, (q31_t)0x345a5559, (q31_t)0x3454997c, (q31_t)0x344edd7e, (q31_t)0x34492160,
	(q31_t)0x34436521, (q31_t)0x343da8c3,
	(q31_t)0x3437ec44, (q31_t)0x34322fa5, (q31_t)0x342c72e6, (q31_t)0x3426b606, (q31_t)0x3420f907, (q31_t)0x341b3be7,
	(q31_t)0x34157ea7, (q31_t)0x340fc147,
	(q31_t)0x340a03c7, (q31_t)0x34044626, (q31_t)0x33fe8866, (q31_t)0x33f8ca86, (q31_t)0x33f30c85, (q31_t)0x33ed4e65,
	(q31_t)0x33e79024, (q31_t)0x33e1d1c4,
	(q31_t)0x33dc1343, (q31_t)0x33d654a2, (q31_t)0x33d095e2, (q31_t)0x33cad701, (q31_t)0x33c51801, (q31_t)0x33bf58e1,
	(q31_t)0x33b999a0, (q31_t)0x33b3da40,
	(q31_t)0x33ae1ac0, (q31_t)0x33a85b20, (q31_t)0x33a29b60, (q31_t)0x339cdb81, (q31_t)0x33971b81, (q31_t)0x33915b62,
	(q31_t)0x338b9b22, (q31_t)0x3385dac4,
	(q31_t)0x33801a45, (q31_t)0x337a59a6, (q31_t)0x337498e8, (q31_t)0x336ed80a, (q31_t)0x3369170c, (q31_t)0x336355ef,
	(q31_t)0x335d94b2, (q31_t)0x3357d355,
	(q31_t)0x335211d8, (q31_t)0x334c503c, (q31_t)0x33468e80, (q31_t)0x3340cca5, (q31_t)0x333b0aaa, (q31_t)0x3335488f,
	(q31_t)0x332f8655, (q31_t)0x3329c3fb,
	(q31_t)0x33240182, (q31_t)0x331e3ee9, (q31_t)0x33187c31, (q31_t)0x3312b959, (q31_t)0x330cf661, (q31_t)0x3307334a,
	(q31_t)0x33017014, (q31_t)0x32fbacbe,
	(q31_t)0x32f5e948, (q31_t)0x32f025b4, (q31_t)0x32ea61ff, (q31_t)0x32e49e2c, (q31_t)0x32deda39, (q31_t)0x32d91626,
	(q31_t)0x32d351f5, (q31_t)0x32cd8da4,
	(q31_t)0x32c7c933, (q31_t)0x32c204a3, (q31_t)0x32bc3ff4, (q31_t)0x32b67b26, (q31_t)0x32b0b638, (q31_t)0x32aaf12b,
	(q31_t)0x32a52bff, (q31_t)0x329f66b4,
	(q31_t)0x3299a149, (q31_t)0x3293dbbf, (q31_t)0x328e1616, (q31_t)0x3288504e, (q31_t)0x32828a67, (q31_t)0x327cc460,
	(q31_t)0x3276fe3a, (q31_t)0x327137f6,
	(q31_t)0x326b7192, (q31_t)0x3265ab0f, (q31_t)0x325fe46c, (q31_t)0x325a1dab, (q31_t)0x325456cb, (q31_t)0x324e8fcc,
	(q31_t)0x3248c8ad, (q31_t)0x32430170,
	(q31_t)0x323d3a14, (q31_t)0x32377298, (q31_t)0x3231aafe, (q31_t)0x322be345, (q31_t)0x32261b6c, (q31_t)0x32205375,
	(q31_t)0x321a8b5f, (q31_t)0x3214c32a,
	(q31_t)0x320efad6, (q31_t)0x32093263, (q31_t)0x320369d2, (q31_t)0x31fda121, (q31_t)0x31f7d852, (q31_t)0x31f20f64,
	(q31_t)0x31ec4657, (q31_t)0x31e67d2b,
	(q31_t)0x31e0b3e0, (q31_t)0x31daea77, (q31_t)0x31d520ef, (q31_t)0x31cf5748, (q31_t)0x31c98d83, (q31_t)0x31c3c39e,
	(q31_t)0x31bdf99b, (q31_t)0x31b82f7a,
	(q31_t)0x31b2653a, (q31_t)0x31ac9adb, (q31_t)0x31a6d05d, (q31_t)0x31a105c1, (q31_t)0x319b3b06, (q31_t)0x3195702d,
	(q31_t)0x318fa535, (q31_t)0x3189da1e,
	(q31_t)0x31840ee9, (q31_t)0x317e4395, (q31_t)0x31787823, (q31_t)0x3172ac92, (q31_t)0x316ce0e3, (q31_t)0x31671515,
	(q31_t)0x31614929, (q31_t)0x315b7d1e,
	(q31_t)0x3155b0f5, (q31_t)0x314fe4ae, (q31_t)0x314a1848, (q31_t)0x31444bc3, (q31_t)0x313e7f21, (q31_t)0x3138b260,
	(q31_t)0x3132e580, (q31_t)0x312d1882,
	(q31_t)0x31274b66, (q31_t)0x31217e2c, (q31_t)0x311bb0d3, (q31_t)0x3115e35c, (q31_t)0x311015c6, (q31_t)0x310a4813,
	(q31_t)0x31047a41, (q31_t)0x30feac51,
	(q31_t)0x30f8de42, (q31_t)0x30f31016, (q31_t)0x30ed41cb, (q31_t)0x30e77362, (q31_t)0x30e1a4db, (q31_t)0x30dbd636,
	(q31_t)0x30d60772, (q31_t)0x30d03891,
	(q31_t)0x30ca6991, (q31_t)0x30c49a74, (q31_t)0x30becb38, (q31_t)0x30b8fbde, (q31_t)0x30b32c66, (q31_t)0x30ad5cd0,
	(q31_t)0x30a78d1c, (q31_t)0x30a1bd4a,
	(q31_t)0x309bed5a, (q31_t)0x30961d4c, (q31_t)0x30904d20, (q31_t)0x308a7cd6, (q31_t)0x3084ac6e, (q31_t)0x307edbe9,
	(q31_t)0x30790b45, (q31_t)0x30733a83,
	(q31_t)0x306d69a4, (q31_t)0x306798a7, (q31_t)0x3061c78b, (q31_t)0x305bf652, (q31_t)0x305624fb, (q31_t)0x30505387,
	(q31_t)0x304a81f4, (q31_t)0x3044b044,
	(q31_t)0x303ede76, (q31_t)0x30390c8a, (q31_t)0x30333a80, (q31_t)0x302d6859, (q31_t)0x30279614, (q31_t)0x3021c3b1,
	(q31_t)0x301bf131, (q31_t)0x30161e93,
	(q31_t)0x30104bd7, (q31_t)0x300a78fe, (q31_t)0x3004a607, (q31_t)0x2ffed2f2, (q31_t)0x2ff8ffc0, (q31_t)0x2ff32c70,
	(q31_t)0x2fed5902, (q31_t)0x2fe78577,
	(q31_t)0x2fe1b1cf, (q31_t)0x2fdbde09, (q31_t)0x2fd60a25, (q31_t)0x2fd03624, (q31_t)0x2fca6206, (q31_t)0x2fc48dc9,
	(q31_t)0x2fbeb970, (q31_t)0x2fb8e4f9,
	(q31_t)0x2fb31064, (q31_t)0x2fad3bb3, (q31_t)0x2fa766e3, (q31_t)0x2fa191f7, (q31_t)0x2f9bbced, (q31_t)0x2f95e7c5,
	(q31_t)0x2f901280, (q31_t)0x2f8a3d1e,
	(q31_t)0x2f84679f, (q31_t)0x2f7e9202, (q31_t)0x2f78bc48, (q31_t)0x2f72e671, (q31_t)0x2f6d107c, (q31_t)0x2f673a6a,
	(q31_t)0x2f61643b, (q31_t)0x2f5b8def,
	(q31_t)0x2f55b785, (q31_t)0x2f4fe0ff, (q31_t)0x2f4a0a5b, (q31_t)0x2f44339a, (q31_t)0x2f3e5cbb, (q31_t)0x2f3885c0,
	(q31_t)0x2f32aea8, (q31_t)0x2f2cd772,
	(q31_t)0x2f27001f, (q31_t)0x2f2128af, (q31_t)0x2f1b5122, (q31_t)0x2f157979, (q31_t)0x2f0fa1b2, (q31_t)0x2f09c9ce,
	(q31_t)0x2f03f1cd, (q31_t)0x2efe19ae,
	(q31_t)0x2ef84173, (q31_t)0x2ef2691b, (q31_t)0x2eec90a7, (q31_t)0x2ee6b815, (q31_t)0x2ee0df66, (q31_t)0x2edb069a,
	(q31_t)0x2ed52db1, (q31_t)0x2ecf54ac,
	(q31_t)0x2ec97b89, (q31_t)0x2ec3a24a, (q31_t)0x2ebdc8ee, (q31_t)0x2eb7ef75, (q31_t)0x2eb215df, (q31_t)0x2eac3c2d,
	(q31_t)0x2ea6625d, (q31_t)0x2ea08871,
	(q31_t)0x2e9aae68, (q31_t)0x2e94d443, (q31_t)0x2e8efa00, (q31_t)0x2e891fa1, (q31_t)0x2e834525, (q31_t)0x2e7d6a8d,
	(q31_t)0x2e778fd8, (q31_t)0x2e71b506,
	(q31_t)0x2e6bda17, (q31_t)0x2e65ff0c, (q31_t)0x2e6023e5, (q31_t)0x2e5a48a0, (q31_t)0x2e546d3f, (q31_t)0x2e4e91c2,
	(q31_t)0x2e48b628, (q31_t)0x2e42da71,
	(q31_t)0x2e3cfe9e, (q31_t)0x2e3722ae, (q31_t)0x2e3146a2, (q31_t)0x2e2b6a79, (q31_t)0x2e258e34, (q31_t)0x2e1fb1d3,
	(q31_t)0x2e19d554, (q31_t)0x2e13f8ba,
	(q31_t)0x2e0e1c03, (q31_t)0x2e083f30, (q31_t)0x2e026240, (q31_t)0x2dfc8534, (q31_t)0x2df6a80b, (q31_t)0x2df0cac6,
	(q31_t)0x2deaed65, (q31_t)0x2de50fe8,
	(q31_t)0x2ddf324e, (q31_t)0x2dd95498, (q31_t)0x2dd376c5, (q31_t)0x2dcd98d7, (q31_t)0x2dc7bacc, (q31_t)0x2dc1dca4,
	(q31_t)0x2dbbfe61, (q31_t)0x2db62001,
	(q31_t)0x2db04186, (q31_t)0x2daa62ee, (q31_t)0x2da4843a, (q31_t)0x2d9ea569, (q31_t)0x2d98c67d, (q31_t)0x2d92e774,
	(q31_t)0x2d8d084f, (q31_t)0x2d87290f,
	(q31_t)0x2d8149b2, (q31_t)0x2d7b6a39, (q31_t)0x2d758aa4, (q31_t)0x2d6faaf3, (q31_t)0x2d69cb26, (q31_t)0x2d63eb3d,
	(q31_t)0x2d5e0b38, (q31_t)0x2d582b17,
	(q31_t)0x2d524ada, (q31_t)0x2d4c6a81, (q31_t)0x2d468a0c, (q31_t)0x2d40a97b, (q31_t)0x2d3ac8ce, (q31_t)0x2d34e805,
	(q31_t)0x2d2f0721, (q31_t)0x2d292620,
	(q31_t)0x2d234504, (q31_t)0x2d1d63cc, (q31_t)0x2d178278, (q31_t)0x2d11a108, (q31_t)0x2d0bbf7d, (q31_t)0x2d05ddd5,
	(q31_t)0x2cfffc12, (q31_t)0x2cfa1a33,
	(q31_t)0x2cf43839, (q31_t)0x2cee5622, (q31_t)0x2ce873f0, (q31_t)0x2ce291a2, (q31_t)0x2cdcaf39, (q31_t)0x2cd6ccb4,
	(q31_t)0x2cd0ea13, (q31_t)0x2ccb0756,
	(q31_t)0x2cc5247e, (q31_t)0x2cbf418b, (q31_t)0x2cb95e7b, (q31_t)0x2cb37b51, (q31_t)0x2cad980a, (q31_t)0x2ca7b4a8,
	(q31_t)0x2ca1d12a, (q31_t)0x2c9bed91,
	(q31_t)0x2c9609dd, (q31_t)0x2c90260d, (q31_t)0x2c8a4221, (q31_t)0x2c845e1a, (q31_t)0x2c7e79f7, (q31_t)0x2c7895b9,
	(q31_t)0x2c72b160, (q31_t)0x2c6ccceb,
	(q31_t)0x2c66e85b, (q31_t)0x2c6103af, (q31_t)0x2c5b1ee8, (q31_t)0x2c553a06, (q31_t)0x2c4f5508, (q31_t)0x2c496fef,
	(q31_t)0x2c438abb, (q31_t)0x2c3da56b,
	(q31_t)0x2c37c000, (q31_t)0x2c31da7a, (q31_t)0x2c2bf4d8, (q31_t)0x2c260f1c, (q31_t)0x2c202944, (q31_t)0x2c1a4351,
	(q31_t)0x2c145d42, (q31_t)0x2c0e7719,
	(q31_t)0x2c0890d4, (q31_t)0x2c02aa74, (q31_t)0x2bfcc3f9, (q31_t)0x2bf6dd63, (q31_t)0x2bf0f6b1, (q31_t)0x2beb0fe5,
	(q31_t)0x2be528fd, (q31_t)0x2bdf41fb,
	(q31_t)0x2bd95add, (q31_t)0x2bd373a4, (q31_t)0x2bcd8c51, (q31_t)0x2bc7a4e2, (q31_t)0x2bc1bd58, (q31_t)0x2bbbd5b3,
	(q31_t)0x2bb5edf4, (q31_t)0x2bb00619,
	(q31_t)0x2baa1e23, (q31_t)0x2ba43613, (q31_t)0x2b9e4de7, (q31_t)0x2b9865a1, (q31_t)0x2b927d3f, (q31_t)0x2b8c94c3,
	(q31_t)0x2b86ac2c, (q31_t)0x2b80c37a,
	(q31_t)0x2b7adaae, (q31_t)0x2b74f1c6, (q31_t)0x2b6f08c4, (q31_t)0x2b691fa6, (q31_t)0x2b63366f, (q31_t)0x2b5d4d1c,
	(q31_t)0x2b5763ae, (q31_t)0x2b517a26,
	(q31_t)0x2b4b9083, (q31_t)0x2b45a6c6, (q31_t)0x2b3fbced, (q31_t)0x2b39d2fa, (q31_t)0x2b33e8ed, (q31_t)0x2b2dfec5,
	(q31_t)0x2b281482, (q31_t)0x2b222a24,
	(q31_t)0x2b1c3fac, (q31_t)0x2b165519, (q31_t)0x2b106a6c, (q31_t)0x2b0a7fa4, (q31_t)0x2b0494c2, (q31_t)0x2afea9c5,
	(q31_t)0x2af8bead, (q31_t)0x2af2d37b,
	(q31_t)0x2aece82f, (q31_t)0x2ae6fcc8, (q31_t)0x2ae11146, (q31_t)0x2adb25aa, (q31_t)0x2ad539f4, (q31_t)0x2acf4e23,
	(q31_t)0x2ac96238, (q31_t)0x2ac37633,
	(q31_t)0x2abd8a13, (q31_t)0x2ab79dd8, (q31_t)0x2ab1b184, (q31_t)0x2aabc515, (q31_t)0x2aa5d88b, (q31_t)0x2a9febe8,
	(q31_t)0x2a99ff2a, (q31_t)0x2a941252,
	(q31_t)0x2a8e255f, (q31_t)0x2a883853, (q31_t)0x2a824b2c, (q31_t)0x2a7c5deb, (q31_t)0x2a76708f, (q31_t)0x2a70831a,
	(q31_t)0x2a6a958a, (q31_t)0x2a64a7e0,
	(q31_t)0x2a5eba1c, (q31_t)0x2a58cc3e, (q31_t)0x2a52de46, (q31_t)0x2a4cf033, (q31_t)0x2a470207, (q31_t)0x2a4113c0,
	(q31_t)0x2a3b2560, (q31_t)0x2a3536e5,
	(q31_t)0x2a2f4850, (q31_t)0x2a2959a1, (q31_t)0x2a236ad9, (q31_t)0x2a1d7bf6, (q31_t)0x2a178cf9, (q31_t)0x2a119de2,
	(q31_t)0x2a0baeb2, (q31_t)0x2a05bf67,
	(q31_t)0x29ffd003, (q31_t)0x29f9e084, (q31_t)0x29f3f0ec, (q31_t)0x29ee013a, (q31_t)0x29e8116e, (q31_t)0x29e22188,
	(q31_t)0x29dc3188, (q31_t)0x29d6416f,
	(q31_t)0x29d0513b, (q31_t)0x29ca60ee, (q31_t)0x29c47087, (q31_t)0x29be8007, (q31_t)0x29b88f6c, (q31_t)0x29b29eb8,
	(q31_t)0x29acadea, (q31_t)0x29a6bd02,
	(q31_t)0x29a0cc01, (q31_t)0x299adae6, (q31_t)0x2994e9b1, (q31_t)0x298ef863, (q31_t)0x298906fb, (q31_t)0x2983157a,
	(q31_t)0x297d23df, (q31_t)0x2977322a,
	(q31_t)0x2971405b, (q31_t)0x296b4e74, (q31_t)0x29655c72, (q31_t)0x295f6a57, (q31_t)0x29597823, (q31_t)0x295385d5,
	(q31_t)0x294d936d, (q31_t)0x2947a0ec,
	(q31_t)0x2941ae52, (q31_t)0x293bbb9e, (q31_t)0x2935c8d1, (q31_t)0x292fd5ea, (q31_t)0x2929e2ea, (q31_t)0x2923efd0,
	(q31_t)0x291dfc9d, (q31_t)0x29180951,
	(q31_t)0x291215eb, (q31_t)0x290c226c, (q31_t)0x29062ed4, (q31_t)0x29003b23, (q31_t)0x28fa4758, (q31_t)0x28f45374,
	(q31_t)0x28ee5f76, (q31_t)0x28e86b5f,
	(q31_t)0x28e27730, (q31_t)0x28dc82e6, (q31_t)0x28d68e84, (q31_t)0x28d09a09, (q31_t)0x28caa574, (q31_t)0x28c4b0c6,
	(q31_t)0x28bebbff, (q31_t)0x28b8c71f,
	(q31_t)0x28b2d226, (q31_t)0x28acdd13, (q31_t)0x28a6e7e8, (q31_t)0x28a0f2a3, (q31_t)0x289afd46, (q31_t)0x289507cf,
	(q31_t)0x288f123f, (q31_t)0x28891c97,
	(q31_t)0x288326d5, (q31_t)0x287d30fa, (q31_t)0x28773b07, (q31_t)0x287144fa, (q31_t)0x286b4ed5, (q31_t)0x28655896,
	(q31_t)0x285f623f, (q31_t)0x28596bce,
	(q31_t)0x28537545, (q31_t)0x284d7ea3, (q31_t)0x284787e8, (q31_t)0x28419114, (q31_t)0x283b9a28, (q31_t)0x2835a322,
	(q31_t)0x282fac04, (q31_t)0x2829b4cd,
	(q31_t)0x2823bd7d, (q31_t)0x281dc615, (q31_t)0x2817ce93, (q31_t)0x2811d6f9, (q31_t)0x280bdf46, (q31_t)0x2805e77b,
	(q31_t)0x27ffef97, (q31_t)0x27f9f79a,
	(q31_t)0x27f3ff85, (q31_t)0x27ee0756, (q31_t)0x27e80f10, (q31_t)0x27e216b0, (q31_t)0x27dc1e38, (q31_t)0x27d625a8,
	(q31_t)0x27d02cff, (q31_t)0x27ca343d,
	(q31_t)0x27c43b63, (q31_t)0x27be4270, (q31_t)0x27b84965, (q31_t)0x27b25041, (q31_t)0x27ac5705, (q31_t)0x27a65db0,
	(q31_t)0x27a06443, (q31_t)0x279a6abd,
	(q31_t)0x2794711f, (q31_t)0x278e7768, (q31_t)0x27887d99, (q31_t)0x278283b2, (q31_t)0x277c89b3, (q31_t)0x27768f9b,
	(q31_t)0x2770956a, (q31_t)0x276a9b21,
	(q31_t)0x2764a0c0, (q31_t)0x275ea647, (q31_t)0x2758abb6, (q31_t)0x2752b10c, (q31_t)0x274cb64a, (q31_t)0x2746bb6f,
	(q31_t)0x2740c07d, (q31_t)0x273ac572,
	(q31_t)0x2734ca4f, (q31_t)0x272ecf14, (q31_t)0x2728d3c0, (q31_t)0x2722d855, (q31_t)0x271cdcd1, (q31_t)0x2716e136,
	(q31_t)0x2710e582, (q31_t)0x270ae9b6,
	(q31_t)0x2704edd2, (q31_t)0x26fef1d5, (q31_t)0x26f8f5c1, (q31_t)0x26f2f995, (q31_t)0x26ecfd51, (q31_t)0x26e700f5,
	(q31_t)0x26e10480, (q31_t)0x26db07f4,
	(q31_t)0x26d50b50, (q31_t)0x26cf0e94, (q31_t)0x26c911c0, (q31_t)0x26c314d4, (q31_t)0x26bd17d0, (q31_t)0x26b71ab4,
	(q31_t)0x26b11d80, (q31_t)0x26ab2034,
	(q31_t)0x26a522d1, (q31_t)0x269f2556, (q31_t)0x269927c3, (q31_t)0x26932a18, (q31_t)0x268d2c55, (q31_t)0x26872e7b,
	(q31_t)0x26813088, (q31_t)0x267b327e,
	(q31_t)0x2675345d, (q31_t)0x266f3623, (q31_t)0x266937d2, (q31_t)0x26633969, (q31_t)0x265d3ae9, (q31_t)0x26573c50,
	(q31_t)0x26513da1, (q31_t)0x264b3ed9,
	(q31_t)0x26453ffa, (q31_t)0x263f4103, (q31_t)0x263941f5, (q31_t)0x263342cf, (q31_t)0x262d4392, (q31_t)0x2627443d,
	(q31_t)0x262144d0, (q31_t)0x261b454c,
	(q31_t)0x261545b0, (q31_t)0x260f45fd, (q31_t)0x26094633, (q31_t)0x26034651, (q31_t)0x25fd4657, (q31_t)0x25f74646,
	(q31_t)0x25f1461e, (q31_t)0x25eb45de,
	(q31_t)0x25e54587, (q31_t)0x25df4519, (q31_t)0x25d94493, (q31_t)0x25d343f6, (q31_t)0x25cd4341, (q31_t)0x25c74276,
	(q31_t)0x25c14192, (q31_t)0x25bb4098,
	(q31_t)0x25b53f86, (q31_t)0x25af3e5d, (q31_t)0x25a93d1d, (q31_t)0x25a33bc6, (q31_t)0x259d3a57, (q31_t)0x259738d1,
	(q31_t)0x25913734, (q31_t)0x258b3580,
	(q31_t)0x258533b5, (q31_t)0x257f31d2, (q31_t)0x25792fd8, (q31_t)0x25732dc8, (q31_t)0x256d2ba0, (q31_t)0x25672961,
	(q31_t)0x2561270b, (q31_t)0x255b249e,
	(q31_t)0x2555221a, (q31_t)0x254f1f7e, (q31_t)0x25491ccc, (q31_t)0x25431a03, (q31_t)0x253d1723, (q31_t)0x2537142c,
	(q31_t)0x2531111e, (q31_t)0x252b0df9,
	(q31_t)0x25250abd, (q31_t)0x251f076a, (q31_t)0x25190400, (q31_t)0x25130080, (q31_t)0x250cfce8, (q31_t)0x2506f93a,
	(q31_t)0x2500f574, (q31_t)0x24faf198,
	(q31_t)0x24f4eda6, (q31_t)0x24eee99c, (q31_t)0x24e8e57c, (q31_t)0x24e2e144, (q31_t)0x24dcdcf6, (q31_t)0x24d6d892,
	(q31_t)0x24d0d416, (q31_t)0x24cacf84,
	(q31_t)0x24c4cadb, (q31_t)0x24bec61c, (q31_t)0x24b8c146, (q31_t)0x24b2bc59, (q31_t)0x24acb756, (q31_t)0x24a6b23b,
	(q31_t)0x24a0ad0b, (q31_t)0x249aa7c4,
	(q31_t)0x2494a266, (q31_t)0x248e9cf1, (q31_t)0x24889766, (q31_t)0x248291c5, (q31_t)0x247c8c0d, (q31_t)0x2476863e,
	(q31_t)0x24708059, (q31_t)0x246a7a5e,
	(q31_t)0x2464744c, (q31_t)0x245e6e23, (q31_t)0x245867e4, (q31_t)0x2452618f, (q31_t)0x244c5b24, (q31_t)0x244654a1,
	(q31_t)0x24404e09, (q31_t)0x243a475a,
	(q31_t)0x24344095, (q31_t)0x242e39ba, (q31_t)0x242832c8, (q31_t)0x24222bc0, (q31_t)0x241c24a1, (q31_t)0x24161d6d,
	(q31_t)0x24101622, (q31_t)0x240a0ec1,
	(q31_t)0x24040749, (q31_t)0x23fdffbc, (q31_t)0x23f7f818, (q31_t)0x23f1f05e, (q31_t)0x23ebe88e, (q31_t)0x23e5e0a7,
	(q31_t)0x23dfd8ab, (q31_t)0x23d9d098,
	(q31_t)0x23d3c86f, (q31_t)0x23cdc031, (q31_t)0x23c7b7dc, (q31_t)0x23c1af71, (q31_t)0x23bba6f0, (q31_t)0x23b59e59,
	(q31_t)0x23af95ac, (q31_t)0x23a98ce8,
	(q31_t)0x23a3840f, (q31_t)0x239d7b20, (q31_t)0x2397721b, (q31_t)0x23916900, (q31_t)0x238b5fcf, (q31_t)0x23855688,
	(q31_t)0x237f4d2b, (q31_t)0x237943b9,
	(q31_t)0x23733a30, (q31_t)0x236d3092, (q31_t)0x236726dd, (q31_t)0x23611d13, (q31_t)0x235b1333, (q31_t)0x2355093e,
	(q31_t)0x234eff32, (q31_t)0x2348f511,
	(q31_t)0x2342eada, (q31_t)0x233ce08d, (q31_t)0x2336d62a, (q31_t)0x2330cbb2, (q31_t)0x232ac124, (q31_t)0x2324b680,
	(q31_t)0x231eabc7, (q31_t)0x2318a0f8,
	(q31_t)0x23129613, (q31_t)0x230c8b19, (q31_t)0x23068009, (q31_t)0x230074e3, (q31_t)0x22fa69a8, (q31_t)0x22f45e57,
	(q31_t)0x22ee52f1, (q31_t)0x22e84775,
	(q31_t)0x22e23be4, (q31_t)0x22dc303d, (q31_t)0x22d62480, (q31_t)0x22d018ae, (q31_t)0x22ca0cc7, (q31_t)0x22c400ca,
	(q31_t)0x22bdf4b8, (q31_t)0x22b7e890,
	(q31_t)0x22b1dc53, (q31_t)0x22abd001, (q31_t)0x22a5c399, (q31_t)0x229fb71b, (q31_t)0x2299aa89, (q31_t)0x22939de1,
	(q31_t)0x228d9123, (q31_t)0x22878451,
	(q31_t)0x22817769, (q31_t)0x227b6a6c, (q31_t)0x22755d59, (q31_t)0x226f5032, (q31_t)0x226942f5, (q31_t)0x226335a2,
	(q31_t)0x225d283b, (q31_t)0x22571abe,
	(q31_t)0x22510d2d, (q31_t)0x224aff86, (q31_t)0x2244f1c9, (q31_t)0x223ee3f8, (q31_t)0x2238d612, (q31_t)0x2232c816,
	(q31_t)0x222cba06, (q31_t)0x2226abe0,
	(q31_t)0x22209da5, (q31_t)0x221a8f56, (q31_t)0x221480f1, (q31_t)0x220e7277, (q31_t)0x220863e8, (q31_t)0x22025544,
	(q31_t)0x21fc468b, (q31_t)0x21f637be,
	(q31_t)0x21f028db, (q31_t)0x21ea19e3, (q31_t)0x21e40ad7, (q31_t)0x21ddfbb5, (q31_t)0x21d7ec7f, (q31_t)0x21d1dd34,
	(q31_t)0x21cbcdd3, (q31_t)0x21c5be5e,
	(q31_t)0x21bfaed5, (q31_t)0x21b99f36, (q31_t)0x21b38f83, (q31_t)0x21ad7fba, (q31_t)0x21a76fdd, (q31_t)0x21a15fec,
	(q31_t)0x219b4fe5, (q31_t)0x21953fca,
	(q31_t)0x218f2f9a, (q31_t)0x21891f55, (q31_t)0x21830efc, (q31_t)0x217cfe8e, (q31_t)0x2176ee0b, (q31_t)0x2170dd74,
	(q31_t)0x216accc8, (q31_t)0x2164bc08,
	(q31_t)0x215eab33, (q31_t)0x21589a49, (q31_t)0x2152894b, (q31_t)0x214c7838, (q31_t)0x21466710, (q31_t)0x214055d4,
	(q31_t)0x213a4484, (q31_t)0x2134331f,
	(q31_t)0x212e21a6, (q31_t)0x21281018, (q31_t)0x2121fe76, (q31_t)0x211becbf, (q31_t)0x2115daf4, (q31_t)0x210fc914,
	(q31_t)0x2109b720, (q31_t)0x2103a518,
	(q31_t)0x20fd92fb, (q31_t)0x20f780ca, (q31_t)0x20f16e84, (q31_t)0x20eb5c2b, (q31_t)0x20e549bd, (q31_t)0x20df373a,
	(q31_t)0x20d924a4, (q31_t)0x20d311f9,
	(q31_t)0x20ccff3a, (q31_t)0x20c6ec66, (q31_t)0x20c0d97f, (q31_t)0x20bac683, (q31_t)0x20b4b373, (q31_t)0x20aea04f,
	(q31_t)0x20a88d17, (q31_t)0x20a279ca,
	(q31_t)0x209c666a, (q31_t)0x209652f5, (q31_t)0x20903f6c, (q31_t)0x208a2bcf, (q31_t)0x2084181e, (q31_t)0x207e0459,
	(q31_t)0x2077f080, (q31_t)0x2071dc93,
	(q31_t)0x206bc892, (q31_t)0x2065b47d, (q31_t)0x205fa054, (q31_t)0x20598c17, (q31_t)0x205377c6, (q31_t)0x204d6361,
	(q31_t)0x20474ee8, (q31_t)0x20413a5b,
	(q31_t)0x203b25bb, (q31_t)0x20351106, (q31_t)0x202efc3e, (q31_t)0x2028e761, (q31_t)0x2022d271, (q31_t)0x201cbd6d,
	(q31_t)0x2016a856, (q31_t)0x2010932a,
	(q31_t)0x200a7deb, (q31_t)0x20046898, (q31_t)0x1ffe5331, (q31_t)0x1ff83db6, (q31_t)0x1ff22828, (q31_t)0x1fec1286,
	(q31_t)0x1fe5fcd0, (q31_t)0x1fdfe707,
	(q31_t)0x1fd9d12a, (q31_t)0x1fd3bb39, (q31_t)0x1fcda535, (q31_t)0x1fc78f1d, (q31_t)0x1fc178f1, (q31_t)0x1fbb62b2,
	(q31_t)0x1fb54c60, (q31_t)0x1faf35f9,
	(q31_t)0x1fa91f80, (q31_t)0x1fa308f2, (q31_t)0x1f9cf252, (q31_t)0x1f96db9d, (q31_t)0x1f90c4d5, (q31_t)0x1f8aadfa,
	(q31_t)0x1f84970b, (q31_t)0x1f7e8009,
	(q31_t)0x1f7868f4, (q31_t)0x1f7251ca, (q31_t)0x1f6c3a8e, (q31_t)0x1f66233e, (q31_t)0x1f600bdb, (q31_t)0x1f59f465,
	(q31_t)0x1f53dcdb, (q31_t)0x1f4dc53d,
	(q31_t)0x1f47ad8d, (q31_t)0x1f4195c9, (q31_t)0x1f3b7df2, (q31_t)0x1f356608, (q31_t)0x1f2f4e0a, (q31_t)0x1f2935f9,
	(q31_t)0x1f231dd5, (q31_t)0x1f1d059e,
	(q31_t)0x1f16ed54, (q31_t)0x1f10d4f6, (q31_t)0x1f0abc85, (q31_t)0x1f04a401, (q31_t)0x1efe8b6a, (q31_t)0x1ef872c0,
	(q31_t)0x1ef25a03, (q31_t)0x1eec4132,
	(q31_t)0x1ee6284f, (q31_t)0x1ee00f58, (q31_t)0x1ed9f64f, (q31_t)0x1ed3dd32, (q31_t)0x1ecdc402, (q31_t)0x1ec7aac0,
	(q31_t)0x1ec1916a, (q31_t)0x1ebb7802,
	(q31_t)0x1eb55e86, (q31_t)0x1eaf44f8, (q31_t)0x1ea92b56, (q31_t)0x1ea311a2, (q31_t)0x1e9cf7db, (q31_t)0x1e96de01,
	(q31_t)0x1e90c414, (q31_t)0x1e8aaa14,
	(q31_t)0x1e849001, (q31_t)0x1e7e75dc, (q31_t)0x1e785ba3, (q31_t)0x1e724158, (q31_t)0x1e6c26fa, (q31_t)0x1e660c8a,
	(q31_t)0x1e5ff206, (q31_t)0x1e59d770,
	(q31_t)0x1e53bcc7, (q31_t)0x1e4da20c, (q31_t)0x1e47873d, (q31_t)0x1e416c5d, (q31_t)0x1e3b5169, (q31_t)0x1e353663,
	(q31_t)0x1e2f1b4a, (q31_t)0x1e29001e,
	(q31_t)0x1e22e4e0, (q31_t)0x1e1cc990, (q31_t)0x1e16ae2c, (q31_t)0x1e1092b6, (q31_t)0x1e0a772e, (q31_t)0x1e045b93,
	(q31_t)0x1dfe3fe6, (q31_t)0x1df82426,
	(q31_t)0x1df20853, (q31_t)0x1debec6f, (q31_t)0x1de5d077, (q31_t)0x1ddfb46e, (q31_t)0x1dd99851, (q31_t)0x1dd37c23,
	(q31_t)0x1dcd5fe2, (q31_t)0x1dc7438e,
	(q31_t)0x1dc12729, (q31_t)0x1dbb0ab0, (q31_t)0x1db4ee26, (q31_t)0x1daed189, (q31_t)0x1da8b4da, (q31_t)0x1da29819,
	(q31_t)0x1d9c7b45, (q31_t)0x1d965e5f,
	(q31_t)0x1d904167, (q31_t)0x1d8a245c, (q31_t)0x1d840740, (q31_t)0x1d7dea11, (q31_t)0x1d77ccd0, (q31_t)0x1d71af7d,
	(q31_t)0x1d6b9217, (q31_t)0x1d6574a0,
	(q31_t)0x1d5f5716, (q31_t)0x1d59397a, (q31_t)0x1d531bcc, (q31_t)0x1d4cfe0d, (q31_t)0x1d46e03a, (q31_t)0x1d40c256,
	(q31_t)0x1d3aa460, (q31_t)0x1d348658,
	(q31_t)0x1d2e683e, (q31_t)0x1d284a12, (q31_t)0x1d222bd3, (q31_t)0x1d1c0d83, (q31_t)0x1d15ef21, (q31_t)0x1d0fd0ad,
	(q31_t)0x1d09b227, (q31_t)0x1d03938f,
	(q31_t)0x1cfd74e5, (q31_t)0x1cf7562a, (q31_t)0x1cf1375c, (q31_t)0x1ceb187d, (q31_t)0x1ce4f98c, (q31_t)0x1cdeda89,
	(q31_t)0x1cd8bb74, (q31_t)0x1cd29c4d,
	(q31_t)0x1ccc7d15, (q31_t)0x1cc65dca, (q31_t)0x1cc03e6e, (q31_t)0x1cba1f01, (q31_t)0x1cb3ff81, (q31_t)0x1caddff0,
	(q31_t)0x1ca7c04d, (q31_t)0x1ca1a099,
	(q31_t)0x1c9b80d3, (q31_t)0x1c9560fb, (q31_t)0x1c8f4112, (q31_t)0x1c892117, (q31_t)0x1c83010a, (q31_t)0x1c7ce0ec,
	(q31_t)0x1c76c0bc, (q31_t)0x1c70a07b,
	(q31_t)0x1c6a8028, (q31_t)0x1c645fc3, (q31_t)0x1c5e3f4d, (q31_t)0x1c581ec6, (q31_t)0x1c51fe2d, (q31_t)0x1c4bdd83,
	(q31_t)0x1c45bcc7, (q31_t)0x1c3f9bf9,
	(q31_t)0x1c397b1b, (q31_t)0x1c335a2b, (q31_t)0x1c2d3929, (q31_t)0x1c271816, (q31_t)0x1c20f6f2, (q31_t)0x1c1ad5bc,
	(q31_t)0x1c14b475, (q31_t)0x1c0e931d,
	(q31_t)0x1c0871b4, (q31_t)0x1c025039, (q31_t)0x1bfc2ead, (q31_t)0x1bf60d0f, (q31_t)0x1befeb60, (q31_t)0x1be9c9a1,
	(q31_t)0x1be3a7cf, (q31_t)0x1bdd85ed,
	(q31_t)0x1bd763fa, (q31_t)0x1bd141f5, (q31_t)0x1bcb1fdf, (q31_t)0x1bc4fdb8, (q31_t)0x1bbedb80, (q31_t)0x1bb8b937,
	(q31_t)0x1bb296dc, (q31_t)0x1bac7471,
	(q31_t)0x1ba651f5, (q31_t)0x1ba02f67, (q31_t)0x1b9a0cc8, (q31_t)0x1b93ea19, (q31_t)0x1b8dc758, (q31_t)0x1b87a487,
	(q31_t)0x1b8181a4, (q31_t)0x1b7b5eb0,
	(q31_t)0x1b753bac, (q31_t)0x1b6f1897, (q31_t)0x1b68f570, (q31_t)0x1b62d239, (q31_t)0x1b5caef1, (q31_t)0x1b568b98,
	(q31_t)0x1b50682e, (q31_t)0x1b4a44b3,
	(q31_t)0x1b442127, (q31_t)0x1b3dfd8b, (q31_t)0x1b37d9de, (q31_t)0x1b31b620, (q31_t)0x1b2b9251, (q31_t)0x1b256e71,
	(q31_t)0x1b1f4a81, (q31_t)0x1b192680,
	(q31_t)0x1b13026e, (q31_t)0x1b0cde4c, (q31_t)0x1b06ba19, (q31_t)0x1b0095d5, (q31_t)0x1afa7180, (q31_t)0x1af44d1b,
	(q31_t)0x1aee28a6, (q31_t)0x1ae8041f,
	(q31_t)0x1ae1df88, (q31_t)0x1adbbae1, (q31_t)0x1ad59629, (q31_t)0x1acf7160, (q31_t)0x1ac94c87, (q31_t)0x1ac3279d,
	(q31_t)0x1abd02a3, (q31_t)0x1ab6dd98,
	(q31_t)0x1ab0b87d, (q31_t)0x1aaa9352, (q31_t)0x1aa46e16, (q31_t)0x1a9e48c9, (q31_t)0x1a98236c, (q31_t)0x1a91fdff,
	(q31_t)0x1a8bd881, (q31_t)0x1a85b2f3,
	(q31_t)0x1a7f8d54, (q31_t)0x1a7967a6, (q31_t)0x1a7341e6, (q31_t)0x1a6d1c17, (q31_t)0x1a66f637, (q31_t)0x1a60d047,
	(q31_t)0x1a5aaa47, (q31_t)0x1a548436,
	(q31_t)0x1a4e5e15, (q31_t)0x1a4837e4, (q31_t)0x1a4211a3, (q31_t)0x1a3beb52, (q31_t)0x1a35c4f0, (q31_t)0x1a2f9e7e,
	(q31_t)0x1a2977fc, (q31_t)0x1a23516a,
	(q31_t)0x1a1d2ac8, (q31_t)0x1a170416, (q31_t)0x1a10dd53, (q31_t)0x1a0ab681, (q31_t)0x1a048f9e, (q31_t)0x19fe68ac,
	(q31_t)0x19f841a9, (q31_t)0x19f21a96,
	(q31_t)0x19ebf374, (q31_t)0x19e5cc41, (q31_t)0x19dfa4fe, (q31_t)0x19d97dac, (q31_t)0x19d35649, (q31_t)0x19cd2ed7,
	(q31_t)0x19c70754, (q31_t)0x19c0dfc2,
	(q31_t)0x19bab820, (q31_t)0x19b4906e, (q31_t)0x19ae68ac, (q31_t)0x19a840da, (q31_t)0x19a218f9, (q31_t)0x199bf107,
	(q31_t)0x1995c906, (q31_t)0x198fa0f5,
	(q31_t)0x198978d4, (q31_t)0x198350a4, (q31_t)0x197d2864, (q31_t)0x19770014, (q31_t)0x1970d7b4, (q31_t)0x196aaf45,
	(q31_t)0x196486c6, (q31_t)0x195e5e37,
	(q31_t)0x19583599, (q31_t)0x19520ceb, (q31_t)0x194be42d, (q31_t)0x1945bb60, (q31_t)0x193f9283, (q31_t)0x19396997,
	(q31_t)0x1933409b, (q31_t)0x192d178f,
	(q31_t)0x1926ee74, (q31_t)0x1920c54a, (q31_t)0x191a9c10, (q31_t)0x191472c6, (q31_t)0x190e496d, (q31_t)0x19082005,
	(q31_t)0x1901f68d, (q31_t)0x18fbcd06,
	(q31_t)0x18f5a36f, (q31_t)0x18ef79c9, (q31_t)0x18e95014, (q31_t)0x18e3264f, (q31_t)0x18dcfc7b, (q31_t)0x18d6d297,
	(q31_t)0x18d0a8a4, (q31_t)0x18ca7ea2,
	(q31_t)0x18c45491, (q31_t)0x18be2a70, (q31_t)0x18b80040, (q31_t)0x18b1d601, (q31_t)0x18ababb2, (q31_t)0x18a58154,
	(q31_t)0x189f56e8, (q31_t)0x18992c6b,
	(q31_t)0x189301e0, (q31_t)0x188cd746, (q31_t)0x1886ac9c, (q31_t)0x188081e4, (q31_t)0x187a571c, (q31_t)0x18742c45,
	(q31_t)0x186e015f, (q31_t)0x1867d66a,
	(q31_t)0x1861ab66, (q31_t)0x185b8053, (q31_t)0x18555530, (q31_t)0x184f29ff, (q31_t)0x1848febf, (q31_t)0x1842d370,
	(q31_t)0x183ca812, (q31_t)0x18367ca5,
	(q31_t)0x18305129, (q31_t)0x182a259e, (q31_t)0x1823fa04, (q31_t)0x181dce5b, (q31_t)0x1817a2a4, (q31_t)0x181176dd,
	(q31_t)0x180b4b08, (q31_t)0x18051f24,
	(q31_t)0x17fef331, (q31_t)0x17f8c72f, (q31_t)0x17f29b1e, (q31_t)0x17ec6eff, (q31_t)0x17e642d1, (q31_t)0x17e01694,
	(q31_t)0x17d9ea49, (q31_t)0x17d3bdee,
	(q31_t)0x17cd9186, (q31_t)0x17c7650e, (q31_t)0x17c13888, (q31_t)0x17bb0bf3, (q31_t)0x17b4df4f, (q31_t)0x17aeb29d,
	(q31_t)0x17a885dc, (q31_t)0x17a2590d,
	(q31_t)0x179c2c2f, (q31_t)0x1795ff42, (q31_t)0x178fd247, (q31_t)0x1789a53d, (q31_t)0x17837825, (q31_t)0x177d4afe,
	(q31_t)0x17771dc9, (q31_t)0x1770f086,
	(q31_t)0x176ac333, (q31_t)0x176495d3, (q31_t)0x175e6864, (q31_t)0x17583ae7, (q31_t)0x17520d5b, (q31_t)0x174bdfc1,
	(q31_t)0x1745b218, (q31_t)0x173f8461,
	(q31_t)0x1739569c, (q31_t)0x173328c8, (q31_t)0x172cfae6, (q31_t)0x1726ccf6, (q31_t)0x17209ef8, (q31_t)0x171a70eb,
	(q31_t)0x171442d0, (q31_t)0x170e14a7,
	(q31_t)0x1707e670, (q31_t)0x1701b82a, (q31_t)0x16fb89d6, (q31_t)0x16f55b74, (q31_t)0x16ef2d04, (q31_t)0x16e8fe86,
	(q31_t)0x16e2cff9, (q31_t)0x16dca15f,
	(q31_t)0x16d672b6, (q31_t)0x16d043ff, (q31_t)0x16ca153a, (q31_t)0x16c3e667, (q31_t)0x16bdb787, (q31_t)0x16b78898,
	(q31_t)0x16b1599b, (q31_t)0x16ab2a90,
	(q31_t)0x16a4fb77, (q31_t)0x169ecc50, (q31_t)0x16989d1b, (q31_t)0x16926dd8, (q31_t)0x168c3e87, (q31_t)0x16860f29,
	(q31_t)0x167fdfbc, (q31_t)0x1679b042,
	(q31_t)0x167380ba, (q31_t)0x166d5123, (q31_t)0x1667217f, (q31_t)0x1660f1ce, (q31_t)0x165ac20e, (q31_t)0x16549241,
	(q31_t)0x164e6266, (q31_t)0x1648327d,
	(q31_t)0x16420286, (q31_t)0x163bd282, (q31_t)0x1635a270, (q31_t)0x162f7250, (q31_t)0x16294222, (q31_t)0x162311e7,
	(q31_t)0x161ce19e, (q31_t)0x1616b148,
	(q31_t)0x161080e4, (q31_t)0x160a5072, (q31_t)0x16041ff3, (q31_t)0x15fdef66, (q31_t)0x15f7becc, (q31_t)0x15f18e24,
	(q31_t)0x15eb5d6e, (q31_t)0x15e52cab,
	(q31_t)0x15defbdb, (q31_t)0x15d8cafd, (q31_t)0x15d29a11, (q31_t)0x15cc6918, (q31_t)0x15c63812, (q31_t)0x15c006fe,
	(q31_t)0x15b9d5dd, (q31_t)0x15b3a4ae,
	(q31_t)0x15ad7372, (q31_t)0x15a74228, (q31_t)0x15a110d2, (q31_t)0x159adf6e, (q31_t)0x1594adfc, (q31_t)0x158e7c7d,
	(q31_t)0x15884af1, (q31_t)0x15821958,
	(q31_t)0x157be7b1, (q31_t)0x1575b5fe, (q31_t)0x156f843c, (q31_t)0x1569526e, (q31_t)0x15632093, (q31_t)0x155ceeaa,
	(q31_t)0x1556bcb4, (q31_t)0x15508ab1,
	(q31_t)0x154a58a1, (q31_t)0x15442683, (q31_t)0x153df459, (q31_t)0x1537c221, (q31_t)0x15318fdd, (q31_t)0x152b5d8b,
	(q31_t)0x15252b2c, (q31_t)0x151ef8c0,
	(q31_t)0x1518c648, (q31_t)0x151293c2, (q31_t)0x150c612f, (q31_t)0x15062e8f, (q31_t)0x14fffbe2, (q31_t)0x14f9c928,
	(q31_t)0x14f39662, (q31_t)0x14ed638e,
	(q31_t)0x14e730ae, (q31_t)0x14e0fdc0, (q31_t)0x14dacac6, (q31_t)0x14d497bf, (q31_t)0x14ce64ab, (q31_t)0x14c8318a,
	(q31_t)0x14c1fe5c, (q31_t)0x14bbcb22,
	(q31_t)0x14b597da, (q31_t)0x14af6486, (q31_t)0x14a93125, (q31_t)0x14a2fdb8, (q31_t)0x149cca3e, (q31_t)0x149696b7,
	(q31_t)0x14906323, (q31_t)0x148a2f82,
	(q31_t)0x1483fbd5, (q31_t)0x147dc81c, (q31_t)0x14779455, (q31_t)0x14716082, (q31_t)0x146b2ca3, (q31_t)0x1464f8b7,
	(q31_t)0x145ec4be, (q31_t)0x145890b9,
	(q31_t)0x14525ca7, (q31_t)0x144c2888, (q31_t)0x1445f45d, (q31_t)0x143fc026, (q31_t)0x14398be2, (q31_t)0x14335792,
	(q31_t)0x142d2335, (q31_t)0x1426eecb,
	(q31_t)0x1420ba56, (q31_t)0x141a85d3, (q31_t)0x14145145, (q31_t)0x140e1caa, (q31_t)0x1407e803, (q31_t)0x1401b34f,
	(q31_t)0x13fb7e8f, (q31_t)0x13f549c3,
	(q31_t)0x13ef14ea, (q31_t)0x13e8e005, (q31_t)0x13e2ab14, (q31_t)0x13dc7616, (q31_t)0x13d6410d, (q31_t)0x13d00bf7,
	(q31_t)0x13c9d6d4, (q31_t)0x13c3a1a6,
	(q31_t)0x13bd6c6b, (q31_t)0x13b73725, (q31_t)0x13b101d2, (q31_t)0x13aacc73, (q31_t)0x13a49707, (q31_t)0x139e6190,
	(q31_t)0x13982c0d, (q31_t)0x1391f67d,
	(q31_t)0x138bc0e1, (q31_t)0x13858b3a, (q31_t)0x137f5586, (q31_t)0x13791fc6, (q31_t)0x1372e9fb, (q31_t)0x136cb423,
	(q31_t)0x13667e3f, (q31_t)0x13604850,
	(q31_t)0x135a1254, (q31_t)0x1353dc4c, (q31_t)0x134da639, (q31_t)0x1347701a, (q31_t)0x134139ee, (q31_t)0x133b03b7,
	(q31_t)0x1334cd74, (q31_t)0x132e9725,
	(q31_t)0x132860ca, (q31_t)0x13222a64, (q31_t)0x131bf3f2, (q31_t)0x1315bd73, (q31_t)0x130f86ea, (q31_t)0x13095054,
	(q31_t)0x130319b3, (q31_t)0x12fce305,
	(q31_t)0x12f6ac4d, (q31_t)0x12f07588, (q31_t)0x12ea3eb8, (q31_t)0x12e407dc, (q31_t)0x12ddd0f4, (q31_t)0x12d79a01,
	(q31_t)0x12d16303, (q31_t)0x12cb2bf8,
	(q31_t)0x12c4f4e2, (q31_t)0x12bebdc1, (q31_t)0x12b88693, (q31_t)0x12b24f5b, (q31_t)0x12ac1817, (q31_t)0x12a5e0c7,
	(q31_t)0x129fa96c, (q31_t)0x12997205,
	(q31_t)0x12933a93, (q31_t)0x128d0315, (q31_t)0x1286cb8c, (q31_t)0x128093f7, (q31_t)0x127a5c57, (q31_t)0x127424ac,
	(q31_t)0x126decf5, (q31_t)0x1267b533,
	(q31_t)0x12617d66, (q31_t)0x125b458d, (q31_t)0x12550da9, (q31_t)0x124ed5ba, (q31_t)0x12489dbf, (q31_t)0x124265b9,
	(q31_t)0x123c2da8, (q31_t)0x1235f58b,
	(q31_t)0x122fbd63, (q31_t)0x12298530, (q31_t)0x12234cf2, (q31_t)0x121d14a9, (q31_t)0x1216dc54, (q31_t)0x1210a3f5,
	(q31_t)0x120a6b8a, (q31_t)0x12043314,
	(q31_t)0x11fdfa93, (q31_t)0x11f7c207, (q31_t)0x11f18970, (q31_t)0x11eb50cd, (q31_t)0x11e51820, (q31_t)0x11dedf68,
	(q31_t)0x11d8a6a4, (q31_t)0x11d26dd6,
	(q31_t)0x11cc34fc, (q31_t)0x11c5fc18, (q31_t)0x11bfc329, (q31_t)0x11b98a2e, (q31_t)0x11b35129, (q31_t)0x11ad1819,
	(q31_t)0x11a6defe, (q31_t)0x11a0a5d8,
	(q31_t)0x119a6ca7, (q31_t)0x1194336b, (q31_t)0x118dfa25, (q31_t)0x1187c0d3, (q31_t)0x11818777, (q31_t)0x117b4e10,
	(q31_t)0x1175149e, (q31_t)0x116edb22,
	(q31_t)0x1168a19b, (q31_t)0x11626809, (q31_t)0x115c2e6c, (q31_t)0x1155f4c4, (q31_t)0x114fbb12, (q31_t)0x11498156,
	(q31_t)0x1143478e, (q31_t)0x113d0dbc,
	(q31_t)0x1136d3df, (q31_t)0x113099f8, (q31_t)0x112a6006, (q31_t)0x11242609, (q31_t)0x111dec02, (q31_t)0x1117b1f0,
	(q31_t)0x111177d4, (q31_t)0x110b3dad,
	(q31_t)0x1105037c, (q31_t)0x10fec940, (q31_t)0x10f88efa, (q31_t)0x10f254a9, (q31_t)0x10ec1a4e, (q31_t)0x10e5dfe8,
	(q31_t)0x10dfa578, (q31_t)0x10d96afe,
	(q31_t)0x10d33079, (q31_t)0x10ccf5ea, (q31_t)0x10c6bb50, (q31_t)0x10c080ac, (q31_t)0x10ba45fe, (q31_t)0x10b40b45,
	(q31_t)0x10add082, (q31_t)0x10a795b5,
	(q31_t)0x10a15ade, (q31_t)0x109b1ffc, (q31_t)0x1094e510, (q31_t)0x108eaa1a, (q31_t)0x10886f19, (q31_t)0x1082340f,
	(q31_t)0x107bf8fa, (q31_t)0x1075bddb,
	(q31_t)0x106f82b2, (q31_t)0x1069477f, (q31_t)0x10630c41, (q31_t)0x105cd0fa, (q31_t)0x105695a8, (q31_t)0x10505a4d,
	(q31_t)0x104a1ee7, (q31_t)0x1043e377,
	(q31_t)0x103da7fd, (q31_t)0x10376c79, (q31_t)0x103130ec, (q31_t)0x102af554, (q31_t)0x1024b9b2, (q31_t)0x101e7e06,
	(q31_t)0x10184251, (q31_t)0x10120691,
	(q31_t)0x100bcac7, (q31_t)0x10058ef4, (q31_t)0xfff5317, (q31_t)0xff91730, (q31_t)0xff2db3e, (q31_t)0xfec9f44,
	(q31_t)0xfe6633f, (q31_t)0xfe02730,
	(q31_t)0xfd9eb18, (q31_t)0xfd3aef6, (q31_t)0xfcd72ca, (q31_t)0xfc73695, (q31_t)0xfc0fa55, (q31_t)0xfbabe0c, (q31_t)0xfb481ba,
	(q31_t)0xfae455d,
	(q31_t)0xfa808f7, (q31_t)0xfa1cc87, (q31_t)0xf9b900e, (q31_t)0xf95538b, (q31_t)0xf8f16fe, (q31_t)0xf88da68, (q31_t)0xf829dc8,
	(q31_t)0xf7c611f,
	(q31_t)0xf76246c, (q31_t)0xf6fe7af, (q31_t)0xf69aae9, (q31_t)0xf636e1a, (q31_t)0xf5d3141, (q31_t)0xf56f45e, (q31_t)0xf50b773,
	(q31_t)0xf4a7a7d,
	(q31_t)0xf443d7e, (q31_t)0xf3e0076, (q31_t)0xf37c365, (q31_t)0xf318649, (q31_t)0xf2b4925, (q31_t)0xf250bf7, (q31_t)0xf1ecec0,
	(q31_t)0xf189180,
	(q31_t)0xf125436, (q31_t)0xf0c16e3, (q31_t)0xf05d987, (q31_t)0xeff9c21, (q31_t)0xef95eb2, (q31_t)0xef3213a, (q31_t)0xeece3b9,
	(q31_t)0xee6a62f,
	(q31_t)0xee0689b, (q31_t)0xeda2afe, (q31_t)0xed3ed58, (q31_t)0xecdafa9, (q31_t)0xec771f1, (q31_t)0xec1342f, (q31_t)0xebaf665,
	(q31_t)0xeb4b891,
	(q31_t)0xeae7ab4, (q31_t)0xea83ccf, (q31_t)0xea1fee0, (q31_t)0xe9bc0e8, (q31_t)0xe9582e7, (q31_t)0xe8f44dd, (q31_t)0xe8906cb,
	(q31_t)0xe82c8af,
	(q31_t)0xe7c8a8a, (q31_t)0xe764c5c, (q31_t)0xe700e26, (q31_t)0xe69cfe6, (q31_t)0xe63919e, (q31_t)0xe5d534d, (q31_t)0xe5714f3,
	(q31_t)0xe50d690,
	(q31_t)0xe4a9824, (q31_t)0xe4459af, (q31_t)0xe3e1b32, (q31_t)0xe37dcac, (q31_t)0xe319e1d, (q31_t)0xe2b5f85, (q31_t)0xe2520e5,
	(q31_t)0xe1ee23c,
	(q31_t)0xe18a38a, (q31_t)0xe1264cf, (q31_t)0xe0c260c, (q31_t)0xe05e740, (q31_t)0xdffa86b, (q31_t)0xdf9698e, (q31_t)0xdf32aa8,
	(q31_t)0xdecebba,
	(q31_t)0xde6acc3, (q31_t)0xde06dc3, (q31_t)0xdda2ebb, (q31_t)0xdd3efab, (q31_t)0xdcdb091, (q31_t)0xdc77170, (q31_t)0xdc13245,
	(q31_t)0xdbaf313,
	(q31_t)0xdb4b3d7, (q31_t)0xdae7494, (q31_t)0xda83548, (q31_t)0xda1f5f3, (q31_t)0xd9bb696, (q31_t)0xd957731, (q31_t)0xd8f37c3,
	(q31_t)0xd88f84d,
	(q31_t)0xd82b8cf, (q31_t)0xd7c7948, (q31_t)0xd7639b9, (q31_t)0xd6ffa22, (q31_t)0xd69ba82, (q31_t)0xd637ada, (q31_t)0xd5d3b2a,
	(q31_t)0xd56fb71,
	(q31_t)0xd50bbb1, (q31_t)0xd4a7be8, (q31_t)0xd443c17, (q31_t)0xd3dfc3e, (q31_t)0xd37bc5c, (q31_t)0xd317c73, (q31_t)0xd2b3c81,
	(q31_t)0xd24fc87,
	(q31_t)0xd1ebc85, (q31_t)0xd187c7b, (q31_t)0xd123c69, (q31_t)0xd0bfc4f, (q31_t)0xd05bc2d, (q31_t)0xcff7c02, (q31_t)0xcf93bd0,
	(q31_t)0xcf2fb96,
	(q31_t)0xcecbb53, (q31_t)0xce67b09, (q31_t)0xce03ab7, (q31_t)0xcd9fa5d, (q31_t)0xcd3b9fb, (q31_t)0xccd7991, (q31_t)0xcc7391f,
	(q31_t)0xcc0f8a5,
	(q31_t)0xcbab824, (q31_t)0xcb4779a, (q31_t)0xcae3709, (q31_t)0xca7f670, (q31_t)0xca1b5cf, (q31_t)0xc9b7526, (q31_t)0xc953475,
	(q31_t)0xc8ef3bd,
	(q31_t)0xc88b2fd, (q31_t)0xc827235, (q31_t)0xc7c3166, (q31_t)0xc75f08f, (q31_t)0xc6fafb0, (q31_t)0xc696ec9, (q31_t)0xc632ddb,
	(q31_t)0xc5cece5,
	(q31_t)0xc56abe8, (q31_t)0xc506ae3, (q31_t)0xc4a29d6, (q31_t)0xc43e8c2, (q31_t)0xc3da7a6, (q31_t)0xc376683, (q31_t)0xc312558,
	(q31_t)0xc2ae425,
	(q31_t)0xc24a2eb, (q31_t)0xc1e61aa, (q31_t)0xc182061, (q31_t)0xc11df11, (q31_t)0xc0b9db9, (q31_t)0xc055c5a, (q31_t)0xbff1af3,
	(q31_t)0xbf8d985,
	(q31_t)0xbf29810, (q31_t)0xbec5693, (q31_t)0xbe6150f, (q31_t)0xbdfd383, (q31_t)0xbd991f0, (q31_t)0xbd35056, (q31_t)0xbcd0eb5,
	(q31_t)0xbc6cd0c,
	(q31_t)0xbc08b5c, (q31_t)0xbba49a5, (q31_t)0xbb407e7, (q31_t)0xbadc621, (q31_t)0xba78454, (q31_t)0xba14280, (q31_t)0xb9b00a5,
	(q31_t)0xb94bec2,
	(q31_t)0xb8e7cd9, (q31_t)0xb883ae8, (q31_t)0xb81f8f0, (q31_t)0xb7bb6f2, (q31_t)0xb7574ec, (q31_t)0xb6f32df, (q31_t)0xb68f0cb,
	(q31_t)0xb62aeaf,
	(q31_t)0xb5c6c8d, (q31_t)0xb562a64, (q31_t)0xb4fe834, (q31_t)0xb49a5fd, (q31_t)0xb4363bf, (q31_t)0xb3d217a, (q31_t)0xb36df2e,
	(q31_t)0xb309cdb,
	(q31_t)0xb2a5a81, (q31_t)0xb241820, (q31_t)0xb1dd5b9, (q31_t)0xb17934b, (q31_t)0xb1150d5, (q31_t)0xb0b0e59, (q31_t)0xb04cbd6,
	(q31_t)0xafe894d,
	(q31_t)0xaf846bc, (q31_t)0xaf20425, (q31_t)0xaebc187, (q31_t)0xae57ee2, (q31_t)0xadf3c37, (q31_t)0xad8f985, (q31_t)0xad2b6cc,
	(q31_t)0xacc740c,
	(q31_t)0xac63146, (q31_t)0xabfee79, (q31_t)0xab9aba6, (q31_t)0xab368cc, (q31_t)0xaad25eb, (q31_t)0xaa6e304, (q31_t)0xaa0a016,
	(q31_t)0xa9a5d22,
	(q31_t)0xa941a27, (q31_t)0xa8dd725, (q31_t)0xa87941d, (q31_t)0xa81510f, (q31_t)0xa7b0dfa, (q31_t)0xa74cadf, (q31_t)0xa6e87bd,
	(q31_t)0xa684495,
	(q31_t)0xa620166, (q31_t)0xa5bbe31, (q31_t)0xa557af5, (q31_t)0xa4f37b3, (q31_t)0xa48f46b, (q31_t)0xa42b11d, (q31_t)0xa3c6dc8,
	(q31_t)0xa362a6d,
	(q31_t)0xa2fe70b, (q31_t)0xa29a3a3, (q31_t)0xa236035, (q31_t)0xa1d1cc1, (q31_t)0xa16d946, (q31_t)0xa1095c6, (q31_t)0xa0a523f,
	(q31_t)0xa040eb1,
	(q31_t)0x9fdcb1e, (q31_t)0x9f78784, (q31_t)0x9f143e5, (q31_t)0x9eb003f, (q31_t)0x9e4bc93, (q31_t)0x9de78e1, (q31_t)0x9d83529,
	(q31_t)0x9d1f16b,
	(q31_t)0x9cbada7, (q31_t)0x9c569dc, (q31_t)0x9bf260c, (q31_t)0x9b8e236, (q31_t)0x9b29e59, (q31_t)0x9ac5a77, (q31_t)0x9a6168f,
	(q31_t)0x99fd2a0,
	(q31_t)0x9998eac, (q31_t)0x9934ab2, (q31_t)0x98d06b2, (q31_t)0x986c2ac, (q31_t)0x9807ea1, (q31_t)0x97a3a8f, (q31_t)0x973f678,
	(q31_t)0x96db25a,
	(q31_t)0x9676e37, (q31_t)0x9612a0e, (q31_t)0x95ae5e0, (q31_t)0x954a1ab, (q31_t)0x94e5d71, (q31_t)0x9481931, (q31_t)0x941d4eb,
	(q31_t)0x93b90a0,
	(q31_t)0x9354c4f, (q31_t)0x92f07f8, (q31_t)0x928c39b, (q31_t)0x9227f39, (q31_t)0x91c3ad2, (q31_t)0x915f664, (q31_t)0x90fb1f1,
	(q31_t)0x9096d79,
	(q31_t)0x90328fb, (q31_t)0x8fce477, (q31_t)0x8f69fee, (q31_t)0x8f05b5f, (q31_t)0x8ea16cb, (q31_t)0x8e3d231, (q31_t)0x8dd8d92,
	(q31_t)0x8d748ed,
	(q31_t)0x8d10443, (q31_t)0x8cabf93, (q31_t)0x8c47ade, (q31_t)0x8be3624, (q31_t)0x8b7f164, (q31_t)0x8b1ac9f, (q31_t)0x8ab67d4,
	(q31_t)0x8a52304,
	(q31_t)0x89ede2f, (q31_t)0x8989955, (q31_t)0x8925475, (q31_t)0x88c0f90, (q31_t)0x885caa5, (q31_t)0x87f85b5, (q31_t)0x87940c1,
	(q31_t)0x872fbc6,
	(q31_t)0x86cb6c7, (q31_t)0x86671c2, (q31_t)0x8602cb9, (q31_t)0x859e7aa, (q31_t)0x853a296, (q31_t)0x84d5d7d, (q31_t)0x847185e,
	(q31_t)0x840d33b,
	(q31_t)0x83a8e12, (q31_t)0x83448e5, (q31_t)0x82e03b2, (q31_t)0x827be7a, (q31_t)0x821793e, (q31_t)0x81b33fc, (q31_t)0x814eeb5,
	(q31_t)0x80ea969,
	(q31_t)0x8086419, (q31_t)0x8021ec3, (q31_t)0x7fbd968, (q31_t)0x7f59409, (q31_t)0x7ef4ea4, (q31_t)0x7e9093b, (q31_t)0x7e2c3cd,
	(q31_t)0x7dc7e5a,
	(q31_t)0x7d638e2, (q31_t)0x7cff365, (q31_t)0x7c9ade4, (q31_t)0x7c3685d, (q31_t)0x7bd22d2, (q31_t)0x7b6dd42, (q31_t)0x7b097ad,
	(q31_t)0x7aa5214,
	(q31_t)0x7a40c76, (q31_t)0x79dc6d3, (q31_t)0x797812b, (q31_t)0x7913b7f, (q31_t)0x78af5ce, (q31_t)0x784b019, (q31_t)0x77e6a5e,
	(q31_t)0x77824a0,
	(q31_t)0x771dedc, (q31_t)0x76b9914, (q31_t)0x7655347, (q31_t)0x75f0d76, (q31_t)0x758c7a1, (q31_t)0x75281c6, (q31_t)0x74c3be7,
	(q31_t)0x745f604,
	(q31_t)0x73fb01c, (q31_t)0x7396a30, (q31_t)0x733243f, (q31_t)0x72cde4a, (q31_t)0x7269851, (q31_t)0x7205253, (q31_t)0x71a0c50,
	(q31_t)0x713c64a,
	(q31_t)0x70d803f, (q31_t)0x7073a2f, (q31_t)0x700f41b, (q31_t)0x6faae03, (q31_t)0x6f467e7, (q31_t)0x6ee21c6, (q31_t)0x6e7dba1,
	(q31_t)0x6e19578,
	(q31_t)0x6db4f4a, (q31_t)0x6d50919, (q31_t)0x6cec2e3, (q31_t)0x6c87ca9, (q31_t)0x6c2366a, (q31_t)0x6bbf028, (q31_t)0x6b5a9e1,
	(q31_t)0x6af6396,
	(q31_t)0x6a91d47, (q31_t)0x6a2d6f4, (q31_t)0x69c909d, (q31_t)0x6964a42, (q31_t)0x69003e3, (q31_t)0x689bd80, (q31_t)0x6837718,
	(q31_t)0x67d30ad,
	(q31_t)0x676ea3d, (q31_t)0x670a3ca, (q31_t)0x66a5d53, (q31_t)0x66416d8, (q31_t)0x65dd058, (q31_t)0x65789d5, (q31_t)0x651434e,
	(q31_t)0x64afcc3,
	(q31_t)0x644b634, (q31_t)0x63e6fa2, (q31_t)0x638290b, (q31_t)0x631e271, (q31_t)0x62b9bd3, (q31_t)0x6255531, (q31_t)0x61f0e8b,
	(q31_t)0x618c7e1,
	(q31_t)0x6128134, (q31_t)0x60c3a83, (q31_t)0x605f3ce, (q31_t)0x5ffad15, (q31_t)0x5f96659, (q31_t)0x5f31f99, (q31_t)0x5ecd8d6,
	(q31_t)0x5e6920e,
	(q31_t)0x5e04b43, (q31_t)0x5da0475, (q31_t)0x5d3bda3, (q31_t)0x5cd76cd, (q31_t)0x5c72ff4, (q31_t)0x5c0e917, (q31_t)0x5baa237,
	(q31_t)0x5b45b53,
	(q31_t)0x5ae146b, (q31_t)0x5a7cd80, (q31_t)0x5a18692, (q31_t)0x59b3fa0, (q31_t)0x594f8aa, (q31_t)0x58eb1b2, (q31_t)0x5886ab5,
	(q31_t)0x58223b6,
	(q31_t)0x57bdcb3, (q31_t)0x57595ac, (q31_t)0x56f4ea2, (q31_t)0x5690795, (q31_t)0x562c085, (q31_t)0x55c7971, (q31_t)0x556325a,
	(q31_t)0x54feb3f,
	(q31_t)0x549a422, (q31_t)0x5435d01, (q31_t)0x53d15dd, (q31_t)0x536ceb5, (q31_t)0x530878a, (q31_t)0x52a405d, (q31_t)0x523f92c,
	(q31_t)0x51db1f7,
	(q31_t)0x5176ac0, (q31_t)0x5112385, (q31_t)0x50adc48, (q31_t)0x5049507, (q31_t)0x4fe4dc3, (q31_t)0x4f8067c, (q31_t)0x4f1bf32,
	(q31_t)0x4eb77e5,
	(q31_t)0x4e53095, (q31_t)0x4dee942, (q31_t)0x4d8a1ec, (q31_t)0x4d25a93, (q31_t)0x4cc1337, (q31_t)0x4c5cbd8, (q31_t)0x4bf8476,
	(q31_t)0x4b93d11,
	(q31_t)0x4b2f5a9, (q31_t)0x4acae3e, (q31_t)0x4a666d1, (q31_t)0x4a01f60, (q31_t)0x499d7ed, (q31_t)0x4939077, (q31_t)0x48d48fe,
	(q31_t)0x4870182,
	(q31_t)0x480ba04, (q31_t)0x47a7282, (q31_t)0x4742afe, (q31_t)0x46de377, (q31_t)0x4679bee, (q31_t)0x4615461, (q31_t)0x45b0cd2,
	(q31_t)0x454c541,
	(q31_t)0x44e7dac, (q31_t)0x4483615, (q31_t)0x441ee7c, (q31_t)0x43ba6df, (q31_t)0x4355f40, (q31_t)0x42f179f, (q31_t)0x428cffb,
	(q31_t)0x4228854,
	(q31_t)0x41c40ab, (q31_t)0x415f8ff, (q31_t)0x40fb151, (q31_t)0x40969a0, (q31_t)0x40321ed, (q31_t)0x3fcda37, (q31_t)0x3f6927f,
	(q31_t)0x3f04ac4,
	(q31_t)0x3ea0307, (q31_t)0x3e3bb48, (q31_t)0x3dd7386, (q31_t)0x3d72bc2, (q31_t)0x3d0e3fb, (q31_t)0x3ca9c32, (q31_t)0x3c45467,
	(q31_t)0x3be0c99,
	(q31_t)0x3b7c4c9, (q31_t)0x3b17cf7, (q31_t)0x3ab3523, (q31_t)0x3a4ed4c, (q31_t)0x39ea573, (q31_t)0x3985d97, (q31_t)0x39215ba,
	(q31_t)0x38bcdda,
	(q31_t)0x38585f8, (q31_t)0x37f3e14, (q31_t)0x378f62e, (q31_t)0x372ae46, (q31_t)0x36c665b, (q31_t)0x3661e6f, (q31_t)0x35fd680,
	(q31_t)0x3598e8f,
	(q31_t)0x353469c, (q31_t)0x34cfea8, (q31_t)0x346b6b1, (q31_t)0x3406eb8, (q31_t)0x33a26bd, (q31_t)0x333dec0, (q31_t)0x32d96c1,
	(q31_t)0x3274ec0,
	(q31_t)0x32106bd, (q31_t)0x31abeb9, (q31_t)0x31476b2, (q31_t)0x30e2ea9, (q31_t)0x307e69f, (q31_t)0x3019e93, (q31_t)0x2fb5684,
	(q31_t)0x2f50e74,
	(q31_t)0x2eec663, (q31_t)0x2e87e4f, (q31_t)0x2e2363a, (q31_t)0x2dbee22, (q31_t)0x2d5a609, (q31_t)0x2cf5def, (q31_t)0x2c915d2,
	(q31_t)0x2c2cdb4,
	(q31_t)0x2bc8594, (q31_t)0x2b63d73, (q31_t)0x2aff54f, (q31_t)0x2a9ad2a, (q31_t)0x2a36504, (q31_t)0x29d1cdc, (q31_t)0x296d4b2,
	(q31_t)0x2908c87,
	(q31_t)0x28a445a, (q31_t)0x283fc2b, (q31_t)0x27db3fb, (q31_t)0x2776bc9, (q31_t)0x2712396, (q31_t)0x26adb62, (q31_t)0x264932b,
	(q31_t)0x25e4af4,
	(q31_t)0x25802bb, (q31_t)0x251ba80, (q31_t)0x24b7244, (q31_t)0x2452a07, (q31_t)0x23ee1c8, (q31_t)0x2389988, (q31_t)0x2325147,
	(q31_t)0x22c0904,
	(q31_t)0x225c0bf, (q31_t)0x21f787a, (q31_t)0x2193033, (q31_t)0x212e7eb, (q31_t)0x20c9fa1, (q31_t)0x2065757, (q31_t)0x2000f0b,
	(q31_t)0x1f9c6be,
	(q31_t)0x1f37e6f, (q31_t)0x1ed3620, (q31_t)0x1e6edcf, (q31_t)0x1e0a57d, (q31_t)0x1da5d2a, (q31_t)0x1d414d6, (q31_t)0x1cdcc80,
	(q31_t)0x1c7842a,
	(q31_t)0x1c13bd2, (q31_t)0x1baf37a, (q31_t)0x1b4ab20, (q31_t)0x1ae62c5, (q31_t)0x1a81a69, (q31_t)0x1a1d20c, (q31_t)0x19b89ae,
	(q31_t)0x1954150,
	(q31_t)0x18ef8f0, (q31_t)0x188b08f, (q31_t)0x182682d, (q31_t)0x17c1fcb, (q31_t)0x175d767, (q31_t)0x16f8f03, (q31_t)0x169469d,
	(q31_t)0x162fe37,
	(q31_t)0x15cb5d0, (q31_t)0x1566d68, (q31_t)0x15024ff, (q31_t)0x149dc96, (q31_t)0x143942b, (q31_t)0x13d4bc0, (q31_t)0x1370354,
	(q31_t)0x130bae7,
	(q31_t)0x12a727a, (q31_t)0x1242a0c, (q31_t)0x11de19d, (q31_t)0x117992e, (q31_t)0x11150be, (q31_t)0x10b084d, (q31_t)0x104bfdb,
	(q31_t)0xfe7769,
	(q31_t)0xf82ef6, (q31_t)0xf1e683, (q31_t)0xeb9e0f, (q31_t)0xe5559b, (q31_t)0xdf0d26, (q31_t)0xd8c4b0, (q31_t)0xd27c3a,
	(q31_t)0xcc33c3,
	(q31_t)0xc5eb4c, (q31_t)0xbfa2d5, (q31_t)0xb95a5d, (q31_t)0xb311e4, (q31_t)0xacc96b, (q31_t)0xa680f2, (q31_t)0xa03878,
	(q31_t)0x99effe,
	(q31_t)0x93a784, (q31_t)0x8d5f09, (q31_t)0x87168e, (q31_t)0x80ce12, (q31_t)0x7a8597, (q31_t)0x743d1a, (q31_t)0x6df49e,
	(q31_t)0x67ac21,
	(q31_t)0x6163a5, (q31_t)0x5b1b27, (q31_t)0x54d2aa, (q31_t)0x4e8a2c, (q31_t)0x4841af, (q31_t)0x41f931, (q31_t)0x3bb0b3,
	(q31_t)0x356835,
	(q31_t)0x2f1fb6, (q31_t)0x28d738, (q31_t)0x228eb9, (q31_t)0x1c463b, (q31_t)0x15fdbc, (q31_t)0xfb53d, (q31_t)0x96cbe, (q31_t)0x3243f
};

/**
 * @} end of DCT4_IDCT4_Table group
 */

/**
 * @addtogroup DCT4_IDCT4
 * @{
 */

/**
 * @brief  Initialization function for the Q31 DCT4/IDCT4.
 * @param[in,out] *S         points to an instance of Q31 DCT4/IDCT4 structure.
 * @param[in]     *S_RFFT    points to an instance of Q31 RFFT/RIFFT structure
 * @param[in]     *S_CFFT    points to an instance of Q31 CFFT/CIFFT structure
 * @param[in]     N          length of the DCT4.
 * @param[in]     Nby2       half of the length of the DCT4.
 * @param[in]     normalize  normalizing factor.
 * @return		  arm_status function returns ARM_MATH_SUCCESS if initialization is successful or ARM_MATH_ARGUMENT_ERROR if <code>N</code> is not a supported transform length.
 * \par Normalizing factor:
 * The normalizing factor is <code>sqrt(2/N)</code>, which depends on the size of transform <code>N</code>.
 * Normalizing factors in 1.31 format are mentioned in the table below for different DCT sizes:
 * \image html dct4NormalizingQ31Table.gif
 */

arm_status arm_dct4_init_q31(
	arm_dct4_instance_q31 *S,
	arm_rfft_instance_q31 *S_RFFT,
	arm_cfft_radix4_instance_q31 *S_CFFT,
	uint16_t N,
	uint16_t Nby2,
	q31_t normalize)
{
	/*  Initialise the default arm status */
	arm_status status = ARM_MATH_SUCCESS;

	/* Initializing the pointer array with the weight table base addresses of different lengths */
	q31_t *twiddlePtr[4] = { (q31_t *) WeightsQ31_128, (q31_t *) WeightsQ31_512,
							 (q31_t *) WeightsQ31_2048, (q31_t *) WeightsQ31_8192
						   };

	/* Initializing the pointer array with the cos factor table base addresses of different lengths */
	q31_t *pCosFactor[4] = {
		(q31_t *) cos_factorsQ31_128, (q31_t *) cos_factorsQ31_512,
		(q31_t *) cos_factorsQ31_2048, (q31_t *) cos_factorsQ31_8192
	};

	/* Initialize the DCT4 length */
	S->N = N;

	/* Initialize the half of DCT4 length */
	S->Nby2 = Nby2;

	/* Initialize the DCT4 Normalizing factor */
	S->normalize = normalize;

	/* Initialize Real FFT Instance */
	S->pRfft = S_RFFT;

	/* Initialize Complex FFT Instance */
	S->pCfft = S_CFFT;

	switch (N) {
	/* Initialize the table modifier values */
	case 8192U:
		S->pTwiddle = twiddlePtr[3];
		S->pCosFactor = pCosFactor[3];
		break;
	case 2048U:
		S->pTwiddle = twiddlePtr[2];
		S->pCosFactor = pCosFactor[2];
		break;
	case 512U:
		S->pTwiddle = twiddlePtr[1];
		S->pCosFactor = pCosFactor[1];
		break;
	case 128U:
		S->pTwiddle = twiddlePtr[0];
		S->pCosFactor = pCosFactor[0];
		break;
	default:
		status = ARM_MATH_ARGUMENT_ERROR;
	}

	/* Initialize the RFFT/RIFFT Function */
	arm_rfft_init_q31(S->pRfft,  S->N, 0, 1);

	/* return the status of DCT4 Init function */
	return (status);
}

/**
 * @} end of DCT4_IDCT4 group
 */
