/* ----------------------------------------------------------------------
 * Project:      CMSIS DSP Library
 * Title:        arm_cmplx_mag_squared_q31.c
 * Description:  Q31 complex magnitude squared
 *
 * $Date:        27. January 2017
 * $Revision:    V.1.5.1
 *
 * Target Processor: Cortex-M cores
 * -------------------------------------------------------------------- */
/*
 * Copyright (C) 2010-2017 ARM Limited or its affiliates. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "arm_math.h"

/**
 * @ingroup groupCmplxMath
 */

/**
 * @addtogroup cmplx_mag_squared
 * @{
 */


/**
 * @brief  Q31 complex magnitude squared
 * @param  *pSrc points to the complex input vector
 * @param  *pDst points to the real output vector
 * @param  numSamples number of complex samples in the input vector
 * @return none.
 *
 * <b>Scaling and Overflow Behavior:</b>
 * \par
 * The function implements 1.31 by 1.31 multiplications and finally output is converted into 3.29 format.
 * Input down scaling is not required.
 */

void arm_cmplx_mag_squared_q31(
	q31_t *pSrc,
	q31_t *pDst,
	uint32_t numSamples)
{
	q31_t real, imag;                              /* Temporary variables to store real and imaginary values */
	q31_t acc0, acc1;                              /* Accumulators */

#if defined (ARM_MATH_DSP)

	/* Run the below code for Cortex-M4 and Cortex-M3 */
	uint32_t blkCnt;                               /* loop counter */

	/* loop Unrolling */
	blkCnt = numSamples >> 2U;

	/* First part of the processing with loop unrolling.  Compute 4 outputs at a time.
	 ** a second loop below computes the remaining 1 to 3 samples. */
	while (blkCnt > 0U) {
		/* C[0] = (A[0] * A[0] + A[1] * A[1]) */
		real = *pSrc++;
		imag = *pSrc++;
		acc0 = (q31_t)(((q63_t) real * real) >> 33);
		acc1 = (q31_t)(((q63_t) imag * imag) >> 33);
		/* store the result in 3.29 format in the destination buffer. */
		*pDst++ = acc0 + acc1;

		real = *pSrc++;
		imag = *pSrc++;
		acc0 = (q31_t)(((q63_t) real * real) >> 33);
		acc1 = (q31_t)(((q63_t) imag * imag) >> 33);
		/* store the result in 3.29 format in the destination buffer. */
		*pDst++ = acc0 + acc1;

		real = *pSrc++;
		imag = *pSrc++;
		acc0 = (q31_t)(((q63_t) real * real) >> 33);
		acc1 = (q31_t)(((q63_t) imag * imag) >> 33);
		/* store the result in 3.29 format in the destination buffer. */
		*pDst++ = acc0 + acc1;

		real = *pSrc++;
		imag = *pSrc++;
		acc0 = (q31_t)(((q63_t) real * real) >> 33);
		acc1 = (q31_t)(((q63_t) imag * imag) >> 33);
		/* store the result in 3.29 format in the destination buffer. */
		*pDst++ = acc0 + acc1;

		/* Decrement the loop counter */
		blkCnt--;
	}

	/* If the numSamples is not a multiple of 4, compute any remaining output samples here.
	 ** No loop unrolling is used. */
	blkCnt = numSamples % 0x4U;

	while (blkCnt > 0U) {
		/* C[0] = (A[0] * A[0] + A[1] * A[1]) */
		real = *pSrc++;
		imag = *pSrc++;
		acc0 = (q31_t)(((q63_t) real * real) >> 33);
		acc1 = (q31_t)(((q63_t) imag * imag) >> 33);
		/* store the result in 3.29 format in the destination buffer. */
		*pDst++ = acc0 + acc1;

		/* Decrement the loop counter */
		blkCnt--;
	}

#else

	/* Run the below code for Cortex-M0 */

	while (numSamples > 0U) {
		/* out = ((real * real) + (imag * imag)) */
		real = *pSrc++;
		imag = *pSrc++;
		acc0 = (q31_t)(((q63_t) real * real) >> 33);
		acc1 = (q31_t)(((q63_t) imag * imag) >> 33);
		/* store the result in 3.29 format in the destination buffer. */
		*pDst++ = acc0 + acc1;

		/* Decrement the loop counter */
		numSamples--;
	}

#endif /* #if defined (ARM_MATH_DSP) */

}

/**
 * @} end of cmplx_mag_squared group
 */
