import { describe, expect, it } from "vitest";
import type { OpenClawConfig } from "../config/types.js";
import { ensureBrowserControlAuth } from "./control-auth.js";

describe("ensureBrowserControlAuth", () => {
  async function expectNoAutoGeneratedAuth(cfg: OpenClawConfig): Promise<void> {
    const result = await ensureBrowserControlAuth({
      cfg,
      env: { OPENCLAW_BROWSER_AUTO_AUTH: "1" },
    });
    expect(result.generatedToken).toBeUndefined();
    expect(result.auth.token).toBeUndefined();
    expect(result.auth.password).toBeUndefined();
  }

  describe("trusted-proxy mode", () => {
    it("should not auto-generate token when auth mode is trusted-proxy", async () => {
      const cfg: OpenClawConfig = {
        gateway: {
          auth: {
            mode: "trusted-proxy",
            trustedProxy: {
              userHeader: "x-forwarded-user",
            },
          },
          trustedProxies: ["192.168.1.1"],
        },
      };
      await expectNoAutoGeneratedAuth(cfg);
    });
  });

  describe("password mode", () => {
    it("should not auto-generate token when auth mode is password (even if password not set)", async () => {
      const cfg: OpenClawConfig = {
        gateway: {
          auth: {
            mode: "password",
          },
        },
      };
      await expectNoAutoGeneratedAuth(cfg);
    });
  });

  describe("none mode", () => {
    it("should not auto-generate token when auth mode is none", async () => {
      const cfg: OpenClawConfig = {
        gateway: {
          auth: {
            mode: "none",
          },
        },
      };
      await expectNoAutoGeneratedAuth(cfg);
    });
  });

  describe("token mode", () => {
    it("should return existing token if configured", async () => {
      const cfg: OpenClawConfig = {
        gateway: {
          auth: {
            mode: "token",
            token: "existing-token-123",
          },
        },
      };

      const result = await ensureBrowserControlAuth({
        cfg,
        env: { OPENCLAW_BROWSER_AUTO_AUTH: "1" },
      });

      expect(result.generatedToken).toBeUndefined();
      expect(result.auth.token).toBe("existing-token-123");
    });

    it("should skip auto-generation in test environment", async () => {
      const cfg: OpenClawConfig = {
        gateway: {
          auth: {
            mode: "token",
          },
        },
      };

      const result = await ensureBrowserControlAuth({
        cfg,
        env: { NODE_ENV: "test" },
      });

      expect(result.generatedToken).toBeUndefined();
      expect(result.auth.token).toBeUndefined();
    });
  });
});
