#!/usr/bin/env bash
set -euo pipefail

ROOT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)"
IOS_DIR="${ROOT_DIR}/apps/ios"
TEAM_ID_SCRIPT="${ROOT_DIR}/scripts/ios-team-id.sh"
LOCAL_SIGNING_FILE="${IOS_DIR}/.local-signing.xcconfig"

sanitize_identifier_segment() {
  local raw="${1:-}"
  raw="$(printf '%s' "$raw" | tr '[:upper:]' '[:lower:]')"
  raw="$(printf '%s' "$raw" | sed -E 's/[^a-z0-9]+/-/g; s/^-+//; s/-+$//; s/-+/-/g')"
  if [[ -z "$raw" ]]; then
    raw="local"
  fi
  printf '%s\n' "$raw"
}

normalize_bundle_id() {
  local raw="${1:-}"
  raw="$(printf '%s' "$raw" | tr '[:upper:]' '[:lower:]')"
  raw="$(printf '%s' "$raw" | sed -E 's/[^a-z0-9.-]+/-/g; s/\.+/./g; s/^-+//; s/[.-]+$//')"
  if [[ -z "$raw" ]]; then
    raw="ai.openclaw.ios.test.local"
  fi
  printf '%s\n' "$raw"
}

if [[ ! -x "${TEAM_ID_SCRIPT}" ]]; then
  echo "ERROR: Missing team detection helper: ${TEAM_ID_SCRIPT}" >&2
  exit 1
fi

team_id=""
if team_id="$("${TEAM_ID_SCRIPT}" 2>/dev/null)"; then
  :
else
  if [[ "${IOS_SIGNING_REQUIRED:-0}" == "1" ]]; then
    "${TEAM_ID_SCRIPT}"
    exit 1
  fi

  echo "WARN: Unable to detect an Apple Team ID; keeping existing iOS signing override (if any)." >&2
  exit 0
fi

if [[ -n "${OPENCLAW_IOS_BUNDLE_SUFFIX:-}" ]]; then
  identity_source="${OPENCLAW_IOS_BUNDLE_SUFFIX}"
else
  identity_source="${USER:-}"
  if [[ -z "${identity_source}" ]]; then
    identity_source="$(id -un 2>/dev/null || true)"
  fi
  team_segment="$(sanitize_identifier_segment "${team_id}")"
  identity_source="${identity_source}-${team_segment}"
fi
bundle_suffix="$(sanitize_identifier_segment "${identity_source}")"

bundle_base="${OPENCLAW_IOS_APP_BUNDLE_ID:-${OPENCLAW_IOS_BUNDLE_ID_BASE:-}}"
if [[ -z "${bundle_base}" ]]; then
  bundle_base="ai.openclaw.ios.test.${bundle_suffix}"
fi
bundle_base="$(normalize_bundle_id "${bundle_base}")"

share_bundle_id="${OPENCLAW_IOS_SHARE_BUNDLE_ID:-${bundle_base}.share}"
watch_app_bundle_id="${OPENCLAW_IOS_WATCH_APP_BUNDLE_ID:-${bundle_base}.watchkitapp}"
watch_extension_bundle_id="${OPENCLAW_IOS_WATCH_EXTENSION_BUNDLE_ID:-${watch_app_bundle_id}.extension}"

code_sign_style="${OPENCLAW_IOS_CODE_SIGN_STYLE:-Automatic}"
app_profile="${OPENCLAW_IOS_APP_PROFILE:-}"
share_profile="${OPENCLAW_IOS_SHARE_PROFILE:-}"

tmp_file="$(mktemp "${TMPDIR:-/tmp}/openclaw-ios-signing.XXXXXX")"
cat >"${tmp_file}" <<EOF
// Auto-generated by scripts/ios-configure-signing.sh.
// This file is local-only and should not be committed.
// Override values with env vars if needed:
// OPENCLAW_IOS_APP_BUNDLE_ID / OPENCLAW_IOS_BUNDLE_ID_BASE
// OPENCLAW_IOS_SHARE_BUNDLE_ID / OPENCLAW_IOS_WATCH_APP_BUNDLE_ID / OPENCLAW_IOS_WATCH_EXTENSION_BUNDLE_ID
// OPENCLAW_IOS_CODE_SIGN_STYLE / OPENCLAW_IOS_APP_PROFILE / OPENCLAW_IOS_SHARE_PROFILE
OPENCLAW_CODE_SIGN_STYLE = ${code_sign_style}
OPENCLAW_DEVELOPMENT_TEAM = ${team_id}
// Keep legacy key for compatibility with older signing config paths.
OPENCLAW_IOS_SELECTED_TEAM = ${team_id}
OPENCLAW_APP_BUNDLE_ID = ${bundle_base}
OPENCLAW_SHARE_BUNDLE_ID = ${share_bundle_id}
OPENCLAW_WATCH_APP_BUNDLE_ID = ${watch_app_bundle_id}
OPENCLAW_WATCH_EXTENSION_BUNDLE_ID = ${watch_extension_bundle_id}
OPENCLAW_APP_PROFILE = ${app_profile}
OPENCLAW_SHARE_PROFILE = ${share_profile}
EOF

if [[ -f "${LOCAL_SIGNING_FILE}" ]] && cmp -s "${tmp_file}" "${LOCAL_SIGNING_FILE}"; then
  rm -f "${tmp_file}"
  echo "iOS signing config already up to date: team=${team_id} app=${bundle_base}"
  exit 0
fi

mv "${tmp_file}" "${LOCAL_SIGNING_FILE}"
echo "Configured iOS signing: team=${team_id} app=${bundle_base}"
