/*
 * QUANTCONNECT.COM - Democratizing Finance, Empowering Individuals.
 * Lean Algorithmic Trading Engine v2.0. Copyright 2014 QuantConnect Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
*/

using QuantConnect.Data;
using QuantConnect.Interfaces;
using QuantConnect.Securities;
using System.Collections.Generic;
using QuantConnect.Data.Shortable;
using QuantConnect.Securities.Equity;

namespace QuantConnect.Algorithm.CSharp
{
    /// <summary>
    /// Example algorithm showing and asserting the usage of the <see cref="ShortMarginInterestRateModel"/>
    /// paired with a <see cref="IShortableProvider"/> instance, for example <see cref="InteractiveBrokersShortableProvider"/>
    /// </summary>
    public class ShortInterestFeeRegressionAlgorithm : QCAlgorithm, IRegressionAlgorithmDefinition
    {
        private Security _short;
        private Security _long;

        /// <summary>
        /// Initialise the data and resolution required, as well as the cash and start-end dates for your algorithm. All algorithms must initialized.
        /// </summary>
        public override void Initialize()
        {
            SetStartDate(2013, 10, 07);
            SetEndDate(2013, 10, 11);

            _short = AddEquity("SPY", Resolution.Hour);
            _long = AddEquity("AAPL", Resolution.Hour);

            foreach (var security in new[] { _short, _long})
            {
                security.SetShortableProvider(new LocalDiskShortableProvider("testbrokerage"));
                security.MarginInterestRateModel = new ShortMarginInterestRateModel();
            }
        }

        /// <summary>
        /// OnData event is the primary entry point for your algorithm. Each new data point will be pumped in here.
        /// </summary>
        /// <param name="data">Slice object keyed by symbol containing the stock data</param>
        public override void OnData(Slice slice)
        {
            if (!Portfolio.Invested)
            {
                SetHoldings("SPY", -0.5);
                SetHoldings("AAPL", 0.5);
            }
        }

        public override void OnEndOfAlgorithm()
        {
            if (((ShortMarginInterestRateModel)_short.MarginInterestRateModel).Amount >= 0)
            {
                throw new RegressionTestException("Expected short fee interest rate to be charged");
            }

            if (((ShortMarginInterestRateModel)_long.MarginInterestRateModel).Amount <= 0)
            {
                throw new RegressionTestException("Expected short fee interest rate to be earned");
            }
        }

        /// <summary>
        /// This is used by the regression test system to indicate if the open source Lean repository has the required data to run this algorithm.
        /// </summary>
        public bool CanRunLocally { get; } = true;

        /// <summary>
        /// This is used by the regression test system to indicate which languages this algorithm is written in.
        /// </summary>
        public List<Language> Languages { get; } = new() { Language.CSharp, Language.Python };

        /// <summary>
        /// Data Points count of all timeslices of algorithm
        /// </summary>
        public long DataPoints => 113;

        /// <summary>
        /// Data Points count of the algorithm history
        /// </summary>
        public int AlgorithmHistoryDataPoints => 0;

        /// <summary>
        /// Final status of the algorithm
        /// </summary>
        public AlgorithmStatus AlgorithmStatus => AlgorithmStatus.Completed;

        /// <summary>
        /// This is used by the regression test system to indicate what the expected statistics are from running the algorithm
        /// </summary>
        public Dictionary<string, string> ExpectedStatistics => new Dictionary<string, string>
        {
            {"Total Orders", "2"},
            {"Average Win", "0%"},
            {"Average Loss", "0%"},
            {"Compounding Annual Return", "-35.339%"},
            {"Drawdown", "1.000%"},
            {"Expectancy", "0"},
            {"Start Equity", "100000"},
            {"End Equity", "99444.09"},
            {"Net Profit", "-0.556%"},
            {"Sharpe Ratio", "-2.211"},
            {"Sortino Ratio", "-2.634"},
            {"Probabilistic Sharpe Ratio", "35.604%"},
            {"Loss Rate", "0%"},
            {"Win Rate", "0%"},
            {"Profit-Loss Ratio", "0"},
            {"Alpha", "0.256"},
            {"Beta", "-0.22"},
            {"Annual Standard Deviation", "0.081"},
            {"Annual Variance", "0.007"},
            {"Information Ratio", "-7.72"},
            {"Tracking Error", "0.279"},
            {"Treynor Ratio", "0.813"},
            {"Total Fees", "$17.77"},
            {"Estimated Strategy Capacity", "$130000000.00"},
            {"Lowest Capacity Asset", "AAPL R735QTJ8XC9X"},
            {"Portfolio Turnover", "19.97%"},
            {"Drawdown Recovery", "2"},
            {"OrderListHash", "39c20060e6271685d3e48359e9077bfe"}
        };
    }
}
