// AUTO-GENERATED — DO NOT EDIT
// Source: https://github.com/anthropics/skills
import type { SkillTemplateNode } from '../types'

const children: SkillTemplateNode[] = [
  {
    "name": "SKILL.md",
    "node_type": "file",
    "content": "---\nname: web-artifacts-builder\ndescription: Suite of tools for creating elaborate, multi-component claude.ai HTML artifacts using modern frontend web technologies (React, Tailwind CSS, shadcn/ui). Use for complex artifacts requiring state management, routing, or shadcn/ui components - not for simple single-file HTML/JSX artifacts.\nlicense: Complete terms in LICENSE.txt\n---\n\n# Web Artifacts Builder\n\nTo build powerful frontend claude.ai artifacts, follow these steps:\n1. Initialize the frontend repo using `scripts/init-artifact.sh`\n2. Develop your artifact by editing the generated code\n3. Bundle all code into a single HTML file using `scripts/bundle-artifact.sh`\n4. Display artifact to user\n5. (Optional) Test the artifact\n\n**Stack**: React 18 + TypeScript + Vite + Parcel (bundling) + Tailwind CSS + shadcn/ui\n\n## Design & Style Guidelines\n\nVERY IMPORTANT: To avoid what is often referred to as \"AI slop\", avoid using excessive centered layouts, purple gradients, uniform rounded corners, and Inter font.\n\n## Quick Start\n\n### Step 1: Initialize Project\n\nRun the initialization script to create a new React project:\n```bash\nbash scripts/init-artifact.sh <project-name>\ncd <project-name>\n```\n\nThis creates a fully configured project with:\n- ✅ React + TypeScript (via Vite)\n- ✅ Tailwind CSS 3.4.1 with shadcn/ui theming system\n- ✅ Path aliases (`@/`) configured\n- ✅ 40+ shadcn/ui components pre-installed\n- ✅ All Radix UI dependencies included\n- ✅ Parcel configured for bundling (via .parcelrc)\n- ✅ Node 18+ compatibility (auto-detects and pins Vite version)\n\n### Step 2: Develop Your Artifact\n\nTo build the artifact, edit the generated files. See **Common Development Tasks** below for guidance.\n\n### Step 3: Bundle to Single HTML File\n\nTo bundle the React app into a single HTML artifact:\n```bash\nbash scripts/bundle-artifact.sh\n```\n\nThis creates `bundle.html` - a self-contained artifact with all JavaScript, CSS, and dependencies inlined. This file can be directly shared in Claude conversations as an artifact.\n\n**Requirements**: Your project must have an `index.html` in the root directory.\n\n**What the script does**:\n- Installs bundling dependencies (parcel, @parcel/config-default, parcel-resolver-tspaths, html-inline)\n- Creates `.parcelrc` config with path alias support\n- Builds with Parcel (no source maps)\n- Inlines all assets into single HTML using html-inline\n\n### Step 4: Share Artifact with User\n\nFinally, share the bundled HTML file in conversation with the user so they can view it as an artifact.\n\n### Step 5: Testing/Visualizing the Artifact (Optional)\n\nNote: This is a completely optional step. Only perform if necessary or requested.\n\nTo test/visualize the artifact, use available tools (including other Skills or built-in tools like Playwright or Puppeteer). In general, avoid testing the artifact upfront as it adds latency between the request and when the finished artifact can be seen. Test later, after presenting the artifact, if requested or if issues arise.\n\n## Reference\n\n- **shadcn/ui components**: https://ui.shadcn.com/docs/components"
  },
  {
    "name": "scripts",
    "node_type": "folder",
    "children": [
      {
        "name": "bundle-artifact.sh",
        "node_type": "file",
        "content": "#!/bin/bash\nset -e\n\necho \"📦 Bundling React app to single HTML artifact...\"\n\n# Check if we're in a project directory\nif [ ! -f \"package.json\" ]; then\n  echo \"❌ Error: No package.json found. Run this script from your project root.\"\n  exit 1\nfi\n\n# Check if index.html exists\nif [ ! -f \"index.html\" ]; then\n  echo \"❌ Error: No index.html found in project root.\"\n  echo \"   This script requires an index.html entry point.\"\n  exit 1\nfi\n\n# Install bundling dependencies\necho \"📦 Installing bundling dependencies...\"\npnpm add -D parcel @parcel/config-default parcel-resolver-tspaths html-inline\n\n# Create Parcel config with tspaths resolver\nif [ ! -f \".parcelrc\" ]; then\n  echo \"🔧 Creating Parcel configuration with path alias support...\"\n  cat > .parcelrc << 'EOF'\n{\n  \"extends\": \"@parcel/config-default\",\n  \"resolvers\": [\"parcel-resolver-tspaths\", \"...\"]\n}\nEOF\nfi\n\n# Clean previous build\necho \"🧹 Cleaning previous build...\"\nrm -rf dist bundle.html\n\n# Build with Parcel\necho \"🔨 Building with Parcel...\"\npnpm exec parcel build index.html --dist-dir dist --no-source-maps\n\n# Inline everything into single HTML\necho \"🎯 Inlining all assets into single HTML file...\"\npnpm exec html-inline dist/index.html > bundle.html\n\n# Get file size\nFILE_SIZE=$(du -h bundle.html | cut -f1)\n\necho \"\"\necho \"✅ Bundle complete!\"\necho \"📄 Output: bundle.html ($FILE_SIZE)\"\necho \"\"\necho \"You can now use this single HTML file as an artifact in Claude conversations.\"\necho \"To test locally: open bundle.html in your browser\""
      },
      {
        "name": "init-artifact.sh",
        "node_type": "file",
        "content": "#!/bin/bash\n\n# Exit on error\nset -e\n\n# Detect Node version\nNODE_VERSION=$(node -v | cut -d'v' -f2 | cut -d'.' -f1)\n\necho \"🔍 Detected Node.js version: $NODE_VERSION\"\n\nif [ \"$NODE_VERSION\" -lt 18 ]; then\n  echo \"❌ Error: Node.js 18 or higher is required\"\n  echo \"   Current version: $(node -v)\"\n  exit 1\nfi\n\n# Set Vite version based on Node version\nif [ \"$NODE_VERSION\" -ge 20 ]; then\n  VITE_VERSION=\"latest\"\n  echo \"✅ Using Vite latest (Node 20+)\"\nelse\n  VITE_VERSION=\"5.4.11\"\n  echo \"✅ Using Vite $VITE_VERSION (Node 18 compatible)\"\nfi\n\n# Detect OS and set sed syntax\nif [[ \"$OSTYPE\" == \"darwin\"* ]]; then\n  SED_INPLACE=\"sed -i ''\"\nelse\n  SED_INPLACE=\"sed -i\"\nfi\n\n# Check if pnpm is installed\nif ! command -v pnpm &> /dev/null; then\n  echo \"📦 pnpm not found. Installing pnpm...\"\n  npm install -g pnpm\nfi\n\n# Check if project name is provided\nif [ -z \"$1\" ]; then\n  echo \"❌ Usage: ./create-react-shadcn-complete.sh <project-name>\"\n  exit 1\nfi\n\nPROJECT_NAME=\"$1\"\nSCRIPT_DIR=\"$(cd \"$(dirname \"${BASH_SOURCE[0]}\")\" && pwd)\"\nCOMPONENTS_TARBALL=\"$SCRIPT_DIR/shadcn-components.tar.gz\"\n\n# Check if components tarball exists\nif [ ! -f \"$COMPONENTS_TARBALL\" ]; then\n  echo \"❌ Error: shadcn-components.tar.gz not found in script directory\"\n  echo \"   Expected location: $COMPONENTS_TARBALL\"\n  exit 1\nfi\n\necho \"🚀 Creating new React + Vite project: $PROJECT_NAME\"\n\n# Create new Vite project (always use latest create-vite, pin vite version later)\npnpm create vite \"$PROJECT_NAME\" --template react-ts\n\n# Navigate into project directory\ncd \"$PROJECT_NAME\"\n\necho \"🧹 Cleaning up Vite template...\"\n$SED_INPLACE '/<link rel=\"icon\".*vite\\.svg/d' index.html\n$SED_INPLACE 's/<title>.*<\\/title>/<title>'\"$PROJECT_NAME\"'<\\/title>/' index.html\n\necho \"📦 Installing base dependencies...\"\npnpm install\n\n# Pin Vite version for Node 18\nif [ \"$NODE_VERSION\" -lt 20 ]; then\n  echo \"📌 Pinning Vite to $VITE_VERSION for Node 18 compatibility...\"\n  pnpm add -D vite@$VITE_VERSION\nfi\n\necho \"📦 Installing Tailwind CSS and dependencies...\"\npnpm install -D tailwindcss@3.4.1 postcss autoprefixer @types/node tailwindcss-animate\npnpm install class-variance-authority clsx tailwind-merge lucide-react next-themes\n\necho \"⚙️  Creating Tailwind and PostCSS configuration...\"\ncat > postcss.config.js << 'EOF'\nexport default {\n  plugins: {\n    tailwindcss: {},\n    autoprefixer: {},\n  },\n}\nEOF\n\necho \"📝 Configuring Tailwind with shadcn theme...\"\ncat > tailwind.config.js << 'EOF'\n/** @type {import('tailwindcss').Config} */\nmodule.exports = {\n  darkMode: [\"class\"],\n  content: [\n    \"./index.html\",\n    \"./src/**/*.{js,ts,jsx,tsx}\",\n  ],\n  theme: {\n    extend: {\n      colors: {\n        border: \"hsl(var(--border))\",\n        input: \"hsl(var(--input))\",\n        ring: \"hsl(var(--ring))\",\n        background: \"hsl(var(--background))\",\n        foreground: \"hsl(var(--foreground))\",\n        primary: {\n          DEFAULT: \"hsl(var(--primary))\",\n          foreground: \"hsl(var(--primary-foreground))\",\n        },\n        secondary: {\n          DEFAULT: \"hsl(var(--secondary))\",\n          foreground: \"hsl(var(--secondary-foreground))\",\n        },\n        destructive: {\n          DEFAULT: \"hsl(var(--destructive))\",\n          foreground: \"hsl(var(--destructive-foreground))\",\n        },\n        muted: {\n          DEFAULT: \"hsl(var(--muted))\",\n          foreground: \"hsl(var(--muted-foreground))\",\n        },\n        accent: {\n          DEFAULT: \"hsl(var(--accent))\",\n          foreground: \"hsl(var(--accent-foreground))\",\n        },\n        popover: {\n          DEFAULT: \"hsl(var(--popover))\",\n          foreground: \"hsl(var(--popover-foreground))\",\n        },\n        card: {\n          DEFAULT: \"hsl(var(--card))\",\n          foreground: \"hsl(var(--card-foreground))\",\n        },\n      },\n      borderRadius: {\n        lg: \"var(--radius)\",\n        md: \"calc(var(--radius) - 2px)\",\n        sm: \"calc(var(--radius) - 4px)\",\n      },\n      keyframes: {\n        \"accordion-down\": {\n          from: { height: \"0\" },\n          to: { height: \"var(--radix-accordion-content-height)\" },\n        },\n        \"accordion-up\": {\n          from: { height: \"var(--radix-accordion-content-height)\" },\n          to: { height: \"0\" },\n        },\n      },\n      animation: {\n        \"accordion-down\": \"accordion-down 0.2s ease-out\",\n        \"accordion-up\": \"accordion-up 0.2s ease-out\",\n      },\n    },\n  },\n  plugins: [require(\"tailwindcss-animate\")],\n}\nEOF\n\n# Add Tailwind directives and CSS variables to index.css\necho \"🎨 Adding Tailwind directives and CSS variables...\"\ncat > src/index.css << 'EOF'\n@tailwind base;\n@tailwind components;\n@tailwind utilities;\n\n@layer base {\n  :root {\n    --background: 0 0% 100%;\n    --foreground: 0 0% 3.9%;\n    --card: 0 0% 100%;\n    --card-foreground: 0 0% 3.9%;\n    --popover: 0 0% 100%;\n    --popover-foreground: 0 0% 3.9%;\n    --primary: 0 0% 9%;\n    --primary-foreground: 0 0% 98%;\n    --secondary: 0 0% 96.1%;\n    --secondary-foreground: 0 0% 9%;\n    --muted: 0 0% 96.1%;\n    --muted-foreground: 0 0% 45.1%;\n    --accent: 0 0% 96.1%;\n    --accent-foreground: 0 0% 9%;\n    --destructive: 0 84.2% 60.2%;\n    --destructive-foreground: 0 0% 98%;\n    --border: 0 0% 89.8%;\n    --input: 0 0% 89.8%;\n    --ring: 0 0% 3.9%;\n    --radius: 0.5rem;\n  }\n\n  .dark {\n    --background: 0 0% 3.9%;\n    --foreground: 0 0% 98%;\n    --card: 0 0% 3.9%;\n    --card-foreground: 0 0% 98%;\n    --popover: 0 0% 3.9%;\n    --popover-foreground: 0 0% 98%;\n    --primary: 0 0% 98%;\n    --primary-foreground: 0 0% 9%;\n    --secondary: 0 0% 14.9%;\n    --secondary-foreground: 0 0% 98%;\n    --muted: 0 0% 14.9%;\n    --muted-foreground: 0 0% 63.9%;\n    --accent: 0 0% 14.9%;\n    --accent-foreground: 0 0% 98%;\n    --destructive: 0 62.8% 30.6%;\n    --destructive-foreground: 0 0% 98%;\n    --border: 0 0% 14.9%;\n    --input: 0 0% 14.9%;\n    --ring: 0 0% 83.1%;\n  }\n}\n\n@layer base {\n  * {\n    @apply border-border;\n  }\n  body {\n    @apply bg-background text-foreground;\n  }\n}\nEOF\n\n# Add path aliases to tsconfig.json\necho \"🔧 Adding path aliases to tsconfig.json...\"\nnode -e \"\nconst fs = require('fs');\nconst config = JSON.parse(fs.readFileSync('tsconfig.json', 'utf8'));\nconfig.compilerOptions = config.compilerOptions || {};\nconfig.compilerOptions.baseUrl = '.';\nconfig.compilerOptions.paths = { '@/*': ['./src/*'] };\nfs.writeFileSync('tsconfig.json', JSON.stringify(config, null, 2));\n\"\n\n# Add path aliases to tsconfig.app.json\necho \"🔧 Adding path aliases to tsconfig.app.json...\"\nnode -e \"\nconst fs = require('fs');\nconst path = 'tsconfig.app.json';\nconst content = fs.readFileSync(path, 'utf8');\n// Remove comments manually\nconst lines = content.split('\\n').filter(line => !line.trim().startsWith('//'));\nconst jsonContent = lines.join('\\n');\nconst config = JSON.parse(jsonContent.replace(/\\/\\*[\\s\\S]*?\\*\\//g, '').replace(/,(\\s*[}\\]])/g, '\\$1'));\nconfig.compilerOptions = config.compilerOptions || {};\nconfig.compilerOptions.baseUrl = '.';\nconfig.compilerOptions.paths = { '@/*': ['./src/*'] };\nfs.writeFileSync(path, JSON.stringify(config, null, 2));\n\"\n\n# Update vite.config.ts\necho \"⚙️  Updating Vite configuration...\"\ncat > vite.config.ts << 'EOF'\nimport path from \"path\";\nimport react from \"@vitejs/plugin-react\";\nimport { defineConfig } from \"vite\";\n\nexport default defineConfig({\n  plugins: [react()],\n  resolve: {\n    alias: {\n      \"@\": path.resolve(__dirname, \"./src\"),\n    },\n  },\n});\nEOF\n\n# Install all shadcn/ui dependencies\necho \"📦 Installing shadcn/ui dependencies...\"\npnpm install @radix-ui/react-accordion @radix-ui/react-aspect-ratio @radix-ui/react-avatar @radix-ui/react-checkbox @radix-ui/react-collapsible @radix-ui/react-context-menu @radix-ui/react-dialog @radix-ui/react-dropdown-menu @radix-ui/react-hover-card @radix-ui/react-label @radix-ui/react-menubar @radix-ui/react-navigation-menu @radix-ui/react-popover @radix-ui/react-progress @radix-ui/react-radio-group @radix-ui/react-scroll-area @radix-ui/react-select @radix-ui/react-separator @radix-ui/react-slider @radix-ui/react-slot @radix-ui/react-switch @radix-ui/react-tabs @radix-ui/react-toast @radix-ui/react-toggle @radix-ui/react-toggle-group @radix-ui/react-tooltip\npnpm install sonner cmdk vaul embla-carousel-react react-day-picker react-resizable-panels date-fns react-hook-form @hookform/resolvers zod\n\n# Extract shadcn components from tarball\necho \"📦 Extracting shadcn/ui components...\"\ntar -xzf \"$COMPONENTS_TARBALL\" -C src/\n\n# Create components.json for reference\necho \"📝 Creating components.json config...\"\ncat > components.json << 'EOF'\n{\n  \"$schema\": \"https://ui.shadcn.com/schema.json\",\n  \"style\": \"default\",\n  \"rsc\": false,\n  \"tsx\": true,\n  \"tailwind\": {\n    \"config\": \"tailwind.config.js\",\n    \"css\": \"src/index.css\",\n    \"baseColor\": \"slate\",\n    \"cssVariables\": true,\n    \"prefix\": \"\"\n  },\n  \"aliases\": {\n    \"components\": \"@/components\",\n    \"utils\": \"@/lib/utils\",\n    \"ui\": \"@/components/ui\",\n    \"lib\": \"@/lib\",\n    \"hooks\": \"@/hooks\"\n  }\n}\nEOF\n\necho \"✅ Setup complete! You can now use Tailwind CSS and shadcn/ui in your project.\"\necho \"\"\necho \"📦 Included components (40+ total):\"\necho \"  - accordion, alert, aspect-ratio, avatar, badge, breadcrumb\"\necho \"  - button, calendar, card, carousel, checkbox, collapsible\"\necho \"  - command, context-menu, dialog, drawer, dropdown-menu\"\necho \"  - form, hover-card, input, label, menubar, navigation-menu\"\necho \"  - popover, progress, radio-group, resizable, scroll-area\"\necho \"  - select, separator, sheet, skeleton, slider, sonner\"\necho \"  - switch, table, tabs, textarea, toast, toggle, toggle-group, tooltip\"\necho \"\"\necho \"To start developing:\"\necho \"  cd $PROJECT_NAME\"\necho \"  pnpm dev\"\necho \"\"\necho \"📚 Import components like:\"\necho \"  import { Button } from '@/components/ui/button'\"\necho \"  import { Card, CardHeader, CardTitle, CardContent } from '@/components/ui/card'\"\necho \"  import { Dialog, DialogContent, DialogTrigger } from '@/components/ui/dialog'\"\n"
      },
      {
        "name": "shadcn-components.tar.gz",
        "node_type": "file",
        "content": "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",
        "encoding": "base64"
      }
    ]
  }
]

export default children
