import type { TrajectoryStep } from "./trajectory-types";

const TEXT_LIMIT = 500;
const INPUT_LIMIT = 200;
const OUTPUT_LIMIT = 300;

function truncate(s: string | undefined | null, limit: number): string {
  if (!s) return "";
  const cleaned = s.trim();
  if (cleaned.length <= limit) return cleaned;
  return cleaned.slice(0, limit) + "...";
}

/** Parse Claude Code JSONL trajectory content */
export function parseClaudeCode(content: string): TrajectoryStep[] {
  const steps: TrajectoryStep[] = [];
  const lines = content.split("\n").filter(Boolean);
  let idx = 0;

  for (const line of lines) {
    let parsed: any;
    try { parsed = JSON.parse(line); } catch { continue; }

    if (parsed.type === "assistant" && parsed.message?.role === "assistant") {
      const msgContent = parsed.message.content;
      if (!Array.isArray(msgContent)) continue;

      let text = "";
      const toolCalls: { name: string; input_summary: string }[] = [];

      for (const item of msgContent) {
        if (item.type === "text" && item.text) {
          text += item.text;
        } else if (item.type === "tool_use") {
          const inputStr = typeof item.input === "string"
            ? item.input
            : JSON.stringify(item.input || {});
          toolCalls.push({
            name: item.name || "unknown",
            input_summary: truncate(inputStr, INPUT_LIMIT),
          });
        }
      }

      const step: TrajectoryStep = { index: idx++, role: "assistant" };
      if (text) step.text = truncate(text, TEXT_LIMIT);
      if (toolCalls.length > 0) step.toolCalls = toolCalls;
      steps.push(step);
    } else if (parsed.type === "user" && parsed.message?.role === "user") {
      const msgContent = parsed.message.content;
      if (typeof msgContent === "string") {
        steps.push({ index: idx++, role: "user", text: truncate(msgContent, TEXT_LIMIT) });
      } else if (Array.isArray(msgContent)) {
        for (const item of msgContent) {
          if (item.type === "tool_result") {
            const outputText = typeof item.content === "string"
              ? item.content
              : Array.isArray(item.content)
                ? item.content.map((c: any) => c.text || "").join("\n")
                : "";
            steps.push({
              index: idx++,
              role: "tool_result",
              output_summary: truncate(outputText, OUTPUT_LIMIT),
              isError: item.is_error === true,
            });
          } else if (item.type === "text") {
            steps.push({ index: idx++, role: "user", text: truncate(item.text, TEXT_LIMIT) });
          }
        }
      }
    }
  }
  return steps;
}

/** Parse Gemini CLI JSON trajectory content */
export function parseGemini(content: string): TrajectoryStep[] {
  const steps: TrajectoryStep[] = [];
  let data: any;
  try { data = JSON.parse(content); } catch { return steps; }

  const messages = data.messages || [];
  let idx = 0;

  for (const msg of messages) {
    if (msg.type === "user") {
      steps.push({
        index: idx++,
        role: "user",
        timestamp: msg.timestamp,
        text: truncate(msg.content, TEXT_LIMIT),
      });
    } else if (msg.type === "gemini") {
      const step: TrajectoryStep = {
        index: idx++,
        role: "assistant",
        timestamp: msg.timestamp,
      };
      if (msg.content) step.text = truncate(msg.content, TEXT_LIMIT);
      if (msg.toolCalls && msg.toolCalls.length > 0) {
        step.toolCalls = msg.toolCalls.map((tc: any) => ({
          name: tc.name || tc.displayName || "unknown",
          input_summary: truncate(JSON.stringify(tc.args || {}), INPUT_LIMIT),
        }));
      }
      steps.push(step);

      if (msg.toolCalls) {
        for (const tc of msg.toolCalls) {
          if (tc.result !== undefined) {
            steps.push({
              index: idx++,
              role: "tool_result",
              toolName: tc.name || tc.displayName,
              output_summary: truncate(
                typeof tc.result === "string" ? tc.result : JSON.stringify(tc.result),
                OUTPUT_LIMIT
              ),
            });
          }
        }
      }
    }
  }
  return steps;
}

/** Parse Codex JSONL trajectory content */
export function parseCodex(content: string): TrajectoryStep[] {
  const steps: TrajectoryStep[] = [];
  const lines = content.split("\n").filter(Boolean);
  let idx = 0;

  for (const line of lines) {
    let parsed: any;
    try { parsed = JSON.parse(line); } catch { continue; }

    if (parsed.type === "item.completed" || parsed.type === "item.started") {
      const item = parsed.item;
      if (!item) continue;

      if (item.type === "reasoning" && parsed.type === "item.completed") {
        steps.push({
          index: idx++,
          role: "assistant",
          text: truncate(item.text, TEXT_LIMIT),
        });
      } else if (item.type === "command_execution") {
        if (parsed.type === "item.completed") {
          steps.push({
            index: idx++,
            role: "assistant",
            toolCalls: [{
              name: "Bash",
              input_summary: truncate(item.command, INPUT_LIMIT),
            }],
          });
          if (item.aggregated_output) {
            steps.push({
              index: idx++,
              role: "tool_result",
              toolName: "Bash",
              output_summary: truncate(item.aggregated_output, OUTPUT_LIMIT),
              isError: item.exit_code !== 0,
            });
          }
        }
      } else if (item.type === "agent_message" && parsed.type === "item.completed") {
        steps.push({
          index: idx++,
          role: "assistant",
          text: truncate(item.text, TEXT_LIMIT),
        });
      }
    }
  }
  return steps;
}

/** Determine the trajectory format from agent name */
export type TrajectoryFormat = "claude" | "gemini" | "codex";

export function getFormatForAgent(agentName: string): TrajectoryFormat {
  if (agentName === "gemini-cli") return "gemini";
  if (agentName === "codex") return "codex";
  return "claude";
}

/** Parse trajectory content based on format */
export function parseTrajectory(content: string, format: TrajectoryFormat): TrajectoryStep[] {
  switch (format) {
    case "claude": return parseClaudeCode(content);
    case "gemini": return parseGemini(content);
    case "codex": return parseCodex(content);
  }
}

/**
 * Known trajectory file names per agent, in priority order.
 * Used by both the build script (local FS) and API route (GitHub raw).
 */
export function getTrajectoryFileNames(agentName: string): string[] {
  if (agentName === "claude-code" || agentName === "opus-4-6" || agentName.startsWith("claude")) {
    return ["agent/claude-code.txt"];
  }
  if (agentName === "gemini-cli") {
    return ["agent/gemini-cli.trajectory.json"];
  }
  if (agentName === "codex") {
    return ["agent/codex.txt"];
  }
  return [];
}
