export interface Contributor {
  login: string;
  avatar_url: string;
  html_url: string;
  contributions: number;
}

export interface PendingTask {
  taskNames: string[];
  number: number;
  html_url: string;
  user: {
    login: string;
    avatar_url: string;
  };
  labels: string[];
  created_at: string;
}

const GITHUB_API = "https://api.github.com";
const REPO_OWNER = "benchflow-ai";
const REPO_NAME = "skillsbench";

const fetchHeaders = {
  Accept: "application/vnd.github.v3+json",
  ...(process.env.GITHUB_TOKEN && {
    Authorization: `token ${process.env.GITHUB_TOKEN}`,
  }),
};

export async function getContributors(): Promise<Contributor[]> {
  try {
    const response = await fetch(
      `${GITHUB_API}/repos/${REPO_OWNER}/${REPO_NAME}/contributors?per_page=100`,
      {
        headers: fetchHeaders,
        next: { revalidate: 3600 }, // Cache for 1 hour
      }
    );

    if (!response.ok) {
      console.error("Failed to fetch contributors:", response.statusText);
      return [];
    }

    const contributors: Contributor[] = await response.json();
    return contributors;
  } catch (error) {
    console.error("Error fetching contributors:", error);
    return [];
  }
}

async function getTaskNamesFromPR(prNumber: number): Promise<string[]> {
  try {
    const response = await fetch(
      `${GITHUB_API}/repos/${REPO_OWNER}/${REPO_NAME}/pulls/${prNumber}/files?per_page=100`,
      {
        headers: fetchHeaders,
        next: { revalidate: 300 },
      }
    );

    if (!response.ok) {
      return [];
    }

    const files: { filename: string }[] = await response.json();

    // Extract unique task folder names from file paths like "tasks/task-name/…"
    const taskNames = new Set<string>();
    for (const file of files) {
      const match = file.filename.match(/^tasks\/([^/]+)\//);
      if (match) {
        taskNames.add(match[1]);
      }
    }

    return Array.from(taskNames);
  } catch (error) {
    console.error(`Error fetching files for PR #${prNumber}:`, error);
    return [];
  }
}

export async function getPendingTaskPRs(): Promise<PendingTask[]> {
  try {
    // Fetch open PRs with "[Task]" in title or task-related labels
    const response = await fetch(
      `${GITHUB_API}/repos/${REPO_OWNER}/${REPO_NAME}/pulls?state=open&per_page=100`,
      {
        headers: fetchHeaders,
        next: { revalidate: 300 }, // Cache for 5 minutes
      }
    );

    if (!response.ok) {
      console.error("Failed to fetch PRs:", response.statusText);
      return [];
    }

    const prs = await response.json();

    // Filter for task-related PRs (title contains [Task] or has task label)
    const taskPRs = prs.filter(
      (pr: { title: string; labels: { name: string }[] }) => {
        const hasTaskInTitle = pr.title.toLowerCase().includes("[task]");
        const hasTaskLabel = pr.labels?.some(
          (label: { name: string }) =>
            label.name.toLowerCase().includes("task") ||
            label.name.toLowerCase().includes("new-task")
        );
        return hasTaskInTitle || hasTaskLabel;
      }
    );

    // Fetch task names from PR files in parallel with concurrency limit
    const pendingTasks: PendingTask[] = [];
    const BATCH_SIZE = 5;

    for (let i = 0; i < taskPRs.length; i += BATCH_SIZE) {
      const batch = taskPRs.slice(i, i + BATCH_SIZE);
      const batchResults = await Promise.all(
        batch.map(
          async (pr: {
            title: string;
            number: number;
            html_url: string;
            user: { login: string; avatar_url: string };
            labels: { name: string }[];
            created_at: string;
          }) => {
            const taskNames = await getTaskNamesFromPR(pr.number);
            return {
              taskNames:
                taskNames.length > 0
                  ? taskNames
                  : [pr.title.replace(/^\[.*?\]\s*/i, "")], // Fallback to cleaned PR title
              number: pr.number,
              html_url: pr.html_url,
              user: {
                login: pr.user.login,
                avatar_url: pr.user.avatar_url,
              },
              labels: pr.labels.map((l: { name: string }) => l.name),
              created_at: pr.created_at,
            };
          }
        )
      );
      pendingTasks.push(...batchResults);
    }

    // Filter out PRs that don't have any tasks
    return pendingTasks.filter((t) => t.taskNames.length > 0);
  } catch (error) {
    console.error("Error fetching pending task PRs:", error);
    return [];
  }
}
