"use client";

import { useState } from "react";
import { Badge } from "@/components/ui/badge";
import { cn } from "@/lib/utils";

export interface TrajectoryStep {
  timestamp: string;
  type: "thought" | "tool_call" | "tool_result" | "message" | "final_answer";
  content: string;
  toolName?: string;
}

const TOOL_COLORS: Record<string, string> = {
  Bash: "bg-amber-100 text-amber-800 dark:bg-amber-900/30 dark:text-amber-300",
  bash: "bg-amber-100 text-amber-800 dark:bg-amber-900/30 dark:text-amber-300",
  Read: "bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-300",
  read_file: "bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-300",
  Write: "bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300",
  write_file: "bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300",
  Edit: "bg-purple-100 text-purple-800 dark:bg-purple-900/30 dark:text-purple-300",
  Grep: "bg-orange-100 text-orange-800 dark:bg-orange-900/30 dark:text-orange-300",
  Glob: "bg-teal-100 text-teal-800 dark:bg-teal-900/30 dark:text-teal-300",
  WebSearch: "bg-indigo-100 text-indigo-800 dark:bg-indigo-900/30 dark:text-indigo-300",
  WebFetch: "bg-indigo-100 text-indigo-800 dark:bg-indigo-900/30 dark:text-indigo-300",
  run_shell_command: "bg-amber-100 text-amber-800 dark:bg-amber-900/30 dark:text-amber-300",
  shell: "bg-amber-100 text-amber-800 dark:bg-amber-900/30 dark:text-amber-300",
  google_web_search: "bg-indigo-100 text-indigo-800 dark:bg-indigo-900/30 dark:text-indigo-300",
  activate_skill: "bg-violet-100 text-violet-800 dark:bg-violet-900/30 dark:text-violet-300",
};

const DEFAULT_TOOL_COLOR = "bg-muted text-muted-foreground";

function formatTimestamp(ts: string): string {
  if (!ts) return "";
  try {
    const d = new Date(ts);
    return d.toLocaleTimeString("en-US", {
      hour: "2-digit",
      minute: "2-digit",
      second: "2-digit",
      hour12: false,
    });
  } catch {
    return "";
  }
}

function StepContent({ step, expanded }: { step: TrajectoryStep; expanded: boolean }) {
  const lines = step.content.split("\n");
  const preview = lines.slice(0, 3).join("\n");
  const hasMore = lines.length > 3 || step.content.length > 200;
  const text = expanded ? step.content : preview;

  return (
    <pre className="text-[11px] font-mono whitespace-pre-wrap break-words text-muted-foreground/70 mt-0.5 max-w-full overflow-hidden leading-relaxed">
      {text}
      {!expanded && hasMore && (
        <span className="text-primary/50 ml-1">...</span>
      )}
    </pre>
  );
}

export function TrajectoryViewer({ steps }: { steps: TrajectoryStep[] }) {
  const [expandedSteps, setExpandedSteps] = useState<Set<number>>(new Set());

  const toggleStep = (idx: number) => {
    setExpandedSteps((prev) => {
      const next = new Set(prev);
      if (next.has(idx)) next.delete(idx);
      else next.add(idx);
      return next;
    });
  };

  if (!steps || steps.length === 0) {
    return (
      <div className="text-center py-8 text-muted-foreground text-sm">
        No trajectory data available.
      </div>
    );
  }

  return (
    <div className="relative pl-1">
      {steps.map((step, i) => {
        const isExpanded = expandedSteps.has(i);
        const isFinalAnswer = step.type === "final_answer";
        const isLast = i === steps.length - 1;

        return (
          <div
            key={i}
            className={cn(
              "relative flex cursor-pointer transition-colors rounded",
              "hover:bg-muted/30",
              isFinalAnswer && "bg-primary/5 border border-primary/20 rounded-lg",
            )}
            onClick={() => toggleStep(i)}
          >
            {/* Timeline column: dot + line */}
            <div className="flex-shrink-0 w-5 flex flex-col items-center relative">
              {/* Dot */}
              <div
                className={cn(
                  "w-2 h-2 rounded-full mt-[7px] flex-shrink-0 z-10",
                  step.type === "tool_call" && "bg-primary",
                  step.type === "tool_result" && "bg-muted-foreground/25",
                  step.type === "message" && "bg-foreground/40",
                  step.type === "thought" && "bg-muted-foreground/15",
                  step.type === "final_answer" && "bg-primary ring-2 ring-primary/30",
                )}
              />
              {/* Connecting line */}
              {!isLast && (
                <div className="w-px flex-1 bg-border/60" />
              )}
            </div>

            {/* Content */}
            <div className="flex-1 min-w-0 py-1.5 pr-2">
              {/* Type label / tool badge */}
              <div className="flex items-center gap-1.5">
                {step.type === "tool_call" && (
                  <Badge
                    variant="outline"
                    className={cn(
                      "text-[10px] px-1.5 py-0 h-4 font-medium border-0",
                      TOOL_COLORS[step.toolName || ""] || DEFAULT_TOOL_COLOR,
                    )}
                  >
                    {step.toolName || "tool"}
                  </Badge>
                )}

                {step.type === "thought" && (
                  <span className="text-[10px] text-muted-foreground/40 italic">thinking</span>
                )}

                {isFinalAnswer && (
                  <Badge className="text-[10px] px-1.5 py-0 h-4">
                    Final Answer
                  </Badge>
                )}

                {/* Timestamp — shown inline after label */}
                {formatTimestamp(step.timestamp) && (
                  <span className="text-[9px] font-mono text-muted-foreground/30 ml-auto flex-shrink-0">
                    {formatTimestamp(step.timestamp)}
                  </span>
                )}
              </div>

              <StepContent step={step} expanded={isExpanded} />
            </div>
          </div>
        );
      })}
    </div>
  );
}
