"use client";

import { FilterSelect } from "@/components/FilterSelect";
import { TaskCard } from "@/components/TaskCards";
import { Input } from "@/components/ui/input";
import { Task } from "@/utils/tasks";
import { Loader2, Search } from "lucide-react";
import {
  useCallback,
  useEffect,
  useMemo,
  useState,
  useTransition,
} from "react";

const ITEMS_PER_BATCH = 12;

export function TaskExplorer({ tasks }: { tasks: Task[] }) {
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedTag, setSelectedTag] = useState("all");
  const [selectedDifficulty, setSelectedDifficulty] = useState("all");

  // Infinite scroll state
  const [visibleCount, setVisibleCount] = useState(ITEMS_PER_BATCH);
  const [isPending, startTransition] = useTransition();

  const allTags = useMemo(() => {
    const tags = new Set<string>();
    tasks.forEach((task) => task.tags.forEach((tag) => tags.add(tag)));
    return Array.from(tags).sort();
  }, [tasks]);

  const difficultyOptions = [
    { label: "All Difficulties", value: "all" },
    { label: "Easy", value: "easy" },
    { label: "Medium", value: "medium" },
    { label: "Hard", value: "hard" },
  ];

  const tagOptions = useMemo(
    () => [
      { label: "All Tags", value: "all" },
      ...allTags.map((tag) => ({ label: tag, value: tag })),
    ],
    [allTags],
  );

  const filteredTasks = useMemo(() => {
    return tasks.filter((task) => {
      const matchesSearch = task.title
        .toLowerCase()
        .includes(searchQuery.toLowerCase());
      const matchesTag =
        selectedTag === "all" || task.tags.includes(selectedTag);
      const matchesDifficulty =
        selectedDifficulty === "all" ||
        task.difficulty.toLowerCase() === selectedDifficulty.toLowerCase();

      return matchesSearch && matchesTag && matchesDifficulty;
    });
  }, [tasks, searchQuery, selectedTag, selectedDifficulty]);

  const currentTasks = filteredTasks.slice(0, visibleCount);
  const hasMore = visibleCount < filteredTasks.length;

  const loadMore = useCallback(() => {
    if (isPending || !hasMore) return;

    startTransition(() => {
      setVisibleCount((prev) =>
        Math.min(prev + ITEMS_PER_BATCH, filteredTasks.length),
      );
    });
  }, [isPending, hasMore, filteredTasks.length]);

  // Intersection Observer for Infinite Scroll
  useEffect(() => {
    if (!hasMore || isPending) return;

    const observer = new IntersectionObserver(
      (entries) => {
        if (entries[0].isIntersecting) {
          loadMore();
        }
      },
      { threshold: 0.1, rootMargin: "100px" },
    );

    const trigger = document.getElementById("scroll-trigger");
    if (trigger) {
      observer.observe(trigger);
    }

    return () => {
      if (trigger) observer.unobserve(trigger);
    };
  }, [hasMore, isPending, loadMore]);

  return (
    <div className="space-y-8">
      {/* Search and Filters */}
      <div className="flex flex-col md:flex-row gap-4">
        {/* Search Bar */}
        <div className="relative flex-1">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-muted-foreground" />
          <Input
            type="text"
            placeholder="Search tasks by name…"
            value={searchQuery}
            onChange={(e) => {
              setSearchQuery(e.target.value);
              setVisibleCount(ITEMS_PER_BATCH);
            }}
            className="pl-10"
          />
        </div>

        {/* Difficulty Filter */}
        <div className="w-full md:w-48">
          <FilterSelect
            value={selectedDifficulty}
            onChange={(val) => {
              setSelectedDifficulty(val);
              setVisibleCount(ITEMS_PER_BATCH);
            }}
            options={difficultyOptions}
            placeholder="All Difficulties"
            searchPlaceholder="Search difficulty…"
            className="w-full"
          />
        </div>

        {/* Tag Filter */}
        <div className="w-full md:w-64">
          <FilterSelect
            value={selectedTag}
            onChange={(val) => {
              setSelectedTag(val);
              setVisibleCount(ITEMS_PER_BATCH);
            }}
            options={tagOptions}
            placeholder="All Tags"
            searchPlaceholder="Search tag…"
            className="w-full"
          />
        </div>
      </div>

      {/* Grid Layout */}
      {currentTasks.length > 0 ? (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          {currentTasks.map((task) => (
            <TaskCard key={task.title} task={task} />
          ))}
        </div>
      ) : (
        <div className="text-center py-12 border border-dashed border-border rounded-lg">
          <p className="text-muted-foreground">
            No tasks found matching your search.
          </p>
        </div>
      )}

      {/* Loading Indicator / Sentinel */}
      <div id="scroll-trigger" className="flex justify-center py-8">
        {isPending && (
          <div className="flex items-center gap-2 text-muted-foreground">
            <Loader2 className="w-5 h-5 animate-spin" />
            <span>Loading…</span>
          </div>
        )}
        {!hasMore && currentTasks.length > 0 && (
          <div className="text-muted-foreground text-sm">
            All tasks loaded ({currentTasks.length})
          </div>
        )}
      </div>
    </div>
  );
}
