"use client";

import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { cn } from "@/lib/utils";
import { Task } from "@/utils/tasks";
import {
  Code,
  Cpu,
  Database,
  ExternalLink,
  Globe,
  Layers,
  Shield,
  Terminal,
} from "lucide-react";
import Link from "next/link";

const CATEGORY_ICONS: Record<string, React.ElementType> = {
  security: Shield,
  data: Database,
  web: Globe,
  network: Globe,
  system: Terminal,
  os: Terminal,
  ml: Cpu,
  ai: Cpu,
};

const getDifficultyColor = (difficulty: string) => {
  switch (difficulty.toLowerCase()) {
    case "easy":
      return "bg-success/10 text-success border-success/20";
    case "medium":
      return "bg-warning/10 text-warning border-warning/20";
    case "hard":
      return "bg-destructive/10 text-destructive border-destructive/20";
    default:
      return "bg-muted text-muted-foreground border-border";
  }
};

export function CategoryIcon({
  category,
  className,
}: {
  category: string;
  className?: string;
}) {
  const normalized = category.toLowerCase();
  const Icon =
    Object.entries(CATEGORY_ICONS).find(([key]) =>
      normalized.includes(key),
    )?.[1] || Code;
  return <Icon className={className} />;
}

export function TaskCard({ task }: { task: Task }) {
  const githubUrl = `https://github.com/benchflow-ai/skillsbench/tree/main/${task.path}`;
  const taskUrl = `/tasks/${task.title}`;

  return (
    <Card className="group block hover:border-primary/50 transition-colors relative overflow-hidden flex flex-col h-full">
      <div className="absolute top-0 right-0 p-3 opacity-0 group-hover:opacity-100 transition-opacity z-20">
        <ExternalLink
          className="w-4 h-4 text-muted-foreground"
          aria-hidden="true"
        />
      </div>

      <CardHeader className="p-5 pb-2">
        <div className="flex items-start justify-between mb-2">
          <Link href={taskUrl} className="before:absolute before:inset-0 z-10">
            <CardTitle className="text-base font-bold font-sans tracking-tight">
              {task.title}
            </CardTitle>
          </Link>
        </div>

        <div className="flex gap-2 flex-wrap items-center mt-2 relative z-20 pointer-events-none">
          <Badge
            variant="outline"
            className="gap-1.5 font-medium text-muted-foreground"
          >
            <CategoryIcon
              category={task.category}
              className="w-3 h-3"
              aria-hidden="true"
            />
            {task.category}
          </Badge>
          <Badge
            variant="outline"
            className={cn(
              "font-medium border",
              getDifficultyColor(task.difficulty),
            )}
          >
            {task.difficulty}
          </Badge>
        </div>
      </CardHeader>

      <CardContent className="p-5 pt-2 flex-1">
        <CardDescription className="text-sm text-muted-foreground leading-relaxed line-clamp-[12]">
          {task.description}
        </CardDescription>
      </CardContent>

      <CardFooter className="p-5 pt-4 border-t border-border flex items-center justify-between">
        <div className="flex gap-1 flex-wrap relative z-20 pointer-events-none">
          {task.tags.slice(0, 3).map((tag) => (
            <Badge
              key={tag}
              variant="secondary"
              className="text-xs font-normal"
            >
              #{tag}
            </Badge>
          ))}
          {task.tags.length > 3 && (
            <span className="text-xs text-muted-foreground px-1">
              +{task.tags.length - 3}
            </span>
          )}
        </div>

        <Button
          variant="link"
          size="sm"
          asChild
          className="h-auto p-0 text-xs text-muted-foreground hover:text-primary relative z-20"
        >
          <a
            href={githubUrl}
            target="_blank"
            rel="noopener noreferrer"
            aria-label={`View ${task.title} source on GitHub`}
          >
            Source
          </a>
        </Button>
      </CardFooter>
    </Card>
  );
}

export function TaskCards({ tasks }: { tasks: Task[] }) {
  return (
    <section className="mb-16 scroll-mt-20" id="tasks">
      <div className="flex items-center justify-between mb-6">
        <h2 className="text-xl font-bold">Task Registry</h2>
      </div>

      <div className="mb-12">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          {tasks.map((task) => (
            <TaskCard key={task.title} task={task} />
          ))}
        </div>
      </div>

      <div className="text-center mt-12">
        <Button asChild className="group">
          <Link href="/tasks">
            <span>View All Tasks</span>
            <ExternalLink
              className="w-4 h-4 text-primary-foreground/70 group-hover:text-primary-foreground transition-colors"
              aria-hidden="true"
            />
          </Link>
        </Button>
      </div>
    </section>
  );
}
