import { FileTree } from "@/components/FileTree";
import { Badge } from "@/components/ui/badge";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogTitle,
} from "@/components/ui/dialog";
import { Skill } from "@/utils/skills";
import { Code, Folder, Layers, ChevronRight } from "lucide-react";
import { MDXRemote } from "next-mdx-remote";
import Link from "next/link";
import { useMDXComponents } from "@/mdx-components";
import { cn } from "@/lib/utils";
import { useState } from "react";

interface SkillModalProps {
  skill: Skill | null;
  open: boolean;
  onOpenChange: (open: boolean) => void;
}

interface SidebarSectionProps {
  title: string;
  children: React.ReactNode;
  defaultOpen?: boolean;
  className?: string;
  flex?: boolean; // If true, this section will try to fill remaining space when open
}

function SidebarSection({ title, children, defaultOpen = true, className, flex = false }: SidebarSectionProps) {
  const [isOpen, setIsOpen] = useState(defaultOpen);

  return (
    <div className={cn(
      "flex flex-col border-b border-border/40 min-h-[37px] transition-[flex-grow] duration-300 ease-in-out", // 37px is approx header height
      flex && isOpen ? "flex-1 min-h-0" : "flex-grow-0 flex-shrink-0",
      className
    )}>
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="flex items-center gap-1.5 px-4 py-2.5 w-full text-left bg-muted/50 hover:bg-muted/70 transition-colors group select-none flex-shrink-0 border-y border-border/50"
      >
        <ChevronRight className={cn(
          "w-3.5 h-3.5 text-muted-foreground group-hover:text-foreground transition-transform duration-200",
          isOpen && "rotate-90"
        )} aria-hidden="true" />
        <span className="text-[11px] font-bold text-muted-foreground group-hover:text-foreground uppercase tracking-wider">
          {title}
        </span>
      </button>

      <div
        className={cn(
          "grid transition-[grid-template-rows] duration-300 ease-in-out",
          isOpen ? "grid-rows-[1fr]" : "grid-rows-[0fr]",
          flex && isOpen ? "flex-1" : ""
        )}
      >
        <div className="overflow-hidden min-h-0">
          <div className={cn("flex flex-col h-full", flex ? "min-h-0" : "")}>
            {children}
          </div>
        </div>
      </div>
    </div>
  );
}

export function SkillModal({ skill, open, onOpenChange }: SkillModalProps) {
  const components = useMDXComponents({
    // Custom components to match TaskInstruction style if needed
    pre: ({ children, className, ...props }) => (
      <pre
        className={cn(
          "not-prose overflow-x-auto rounded-lg border border-border bg-muted/50 p-4 font-mono text-sm leading-relaxed my-6",
          className
        )}
        {...props}
      >
        {children}
      </pre>
    ),
  });

  if (!skill) return null;

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-6xl h-[85vh] flex flex-col p-0 gap-0 overflow-hidden sm:rounded-xl border-border/50 bg-background dark:bg-zinc-900 shadow-2xl">
        <div className="sr-only">
          <DialogTitle>{skill.name}</DialogTitle>
          <DialogDescription>{skill.description}</DialogDescription>
        </div>

        <div className="flex-1 grid grid-cols-1 md:grid-cols-[280px_1fr] h-full min-h-0">
          {/* Left Sidebar */}
          <div className="hidden md:flex flex-col bg-muted/5 h-full min-h-0 border-r border-border/60">

            {/* Context Section */}
            <SidebarSection title="Context">
              <div className="p-4 space-y-3">
                <div className="flex items-center gap-2">
                  <Badge
                    variant="outline"
                    className="h-5 text-[10px] uppercase tracking-wider font-semibold border-primary/20 text-primary bg-primary/5"
                  >
                    Skill
                  </Badge>
                </div>
                <div className="text-[10px] font-mono text-muted-foreground break-all bg-background border border-border/50 rounded px-2 py-1">
                  {skill.path}
                </div>
              </div>
            </SidebarSection>

            {/* Related Tasks Section */}
            <SidebarSection title="Related Tasks" defaultOpen={true}>
               <div className="p-2 max-h-[200px] overflow-y-auto space-y-1 scrollbar-custom">
                {skill.relatedTasks && skill.relatedTasks.length > 0 ? (
                  skill.relatedTasks.map((task) => (
                    <Link
                      key={task.path}
                      href={task.path}
                      className="group flex items-center gap-2 px-3 py-2 text-xs text-foreground/80 hover:text-primary hover:bg-primary/5 rounded-md transition-colors"
                    >
                      <Code className="w-3.5 h-3.5 text-muted-foreground group-hover:text-primary transition-colors" />
                      <span className="truncate">{task.title}</span>
                    </Link>
                  ))
                ) : (
                  <div className="px-3 py-4 text-xs text-muted-foreground/60 italic text-center">
                    No related tasks found
                  </div>
                )}
              </div>
            </SidebarSection>

            {/* File Structure Section - Flex to fill remaining space */}
            <SidebarSection title="File Structure" flex={true}>
              <div className="flex-1 overflow-y-auto p-2 scrollbar-custom">
                <FileTree data={skill.fileStructure} parentPath={skill.path} />
              </div>
            </SidebarSection>

          </div>

          {/* Right: Markdown Content */}
          <div className="h-full overflow-y-auto bg-background p-8 lg:p-10 scrollbar-custom">
             <div className="max-w-4xl mx-auto mb-8 border-b border-border/40 pb-6">
                <h1 className="text-3xl font-bold tracking-tight text-foreground mb-4">
                  {skill.name}
                </h1>
                <p className="text-lg text-muted-foreground leading-relaxed">
                  {skill.description}
                </p>
             </div>
            <div className="prose prose-neutral dark:prose-invert max-w-4xl mx-auto
              prose-headings:font-bold prose-headings:tracking-tight
              prose-h2:text-2xl prose-h3:text-xl
              prose-p:leading-relaxed prose-p:text-foreground/90
              prose-a:text-primary prose-a:no-underline hover:prose-a:underline
              prose-img:rounded-xl prose-img:border prose-img:border-border/50
              prose-code:bg-muted/50 prose-code:px-1.5 prose-code:py-0.5 prose-code:rounded-md prose-code:font-normal prose-code:before:content-none prose-code:after:content-none
              [&_pre]:!bg-muted/50 [&_pre]:!border [&_pre]:!border-border
            ">
              {/* eslint-disable-next-line @typescript-eslint/no-explicit-any */}
              {(skill as any).mdxSource ? (
                <MDXRemote {...(skill as any).mdxSource} components={components} />
              ) : (
                <div className="whitespace-pre-wrap font-mono text-sm text-muted-foreground">
                  {(skill as any).content}
                </div>
              )}
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}
