import { FileNode } from "@/utils/skills";
import { Folder, FileCode, ChevronRight, ChevronDown } from "lucide-react";
import { useState } from "react";
import { cn } from "@/lib/utils";
import Link from "next/link";

interface FileTreeItemProps {
  node: FileNode;
  level?: number;
  parentPath: string;
}

const REPO_BASE_URL = "https://github.com/benchflow-ai/skillsbench";

export function FileTreeItem({ node, level = 0, parentPath }: FileTreeItemProps) {
  const [isOpen, setIsOpen] = useState(false);
  const isDir = node.type === 'directory';
  const currentPath = `${parentPath}/${node.name}`;

  const content = (
    <div
      className={cn(
        "flex items-center gap-1.5 py-1 px-2 rounded-sm hover:bg-muted/50 transition-colors cursor-default",
        isDir && "cursor-pointer",
        !isDir && "cursor-pointer hover:text-primary"
      )}
      style={{ paddingLeft: `${level * 12 + 8}px` }}
      onClick={(e) => {
        if (isDir) {
          e.stopPropagation();
          setIsOpen(!isOpen);
        }
      }}
    >
      {isDir ? (
        <span className="text-muted-foreground">
          {isOpen ? <ChevronDown className="w-3 h-3" /> : <ChevronRight className="w-3 h-3" />}
        </span>
      ) : (
        <span className="w-3" /> // Spacer
      )}

      {isDir ? (
        <Folder className="w-4 h-4 text-blue-500/80 fill-blue-500/20" />
      ) : (
        <FileCode className="w-4 h-4 text-muted-foreground" />
      )}

      <span className={cn("truncate", isDir ? "font-medium text-foreground" : "text-muted-foreground")}>
        {node.name}
      </span>
    </div>
  );

  if (isDir) {
    return (
      <div className="text-sm">
        {content}
        {isOpen && node.children && (
          <div>
            {node.children.map((child, idx) => (
              <FileTreeItem
                key={`${child.name}-${idx}`}
                node={child}
                level={level + 1}
                parentPath={currentPath}
              />
            ))}
          </div>
        )}
      </div>
    );
  }

  // File
  return (
    <div className="text-sm">
      <Link
        href={`${REPO_BASE_URL}/blob/main/${currentPath}`}
        target="_blank"
        rel="noopener noreferrer"
        className="block"
        onClick={(e) => e.stopPropagation()}
      >
        {content}
      </Link>
    </div>
  );
}

interface FileTreeProps {
  data: FileNode[];
  parentPath: string;
}

export function FileTree({ data, parentPath }: FileTreeProps) {
  return (
    <div className="flex flex-col">
      {data.map((node, idx) => (
        <FileTreeItem
          key={`${node.name}-${idx}`}
          node={node}
          parentPath={parentPath}
        />
      ))}
    </div>
  );
}
