"use client";

import { Badge } from "@/components/ui/badge";
import { Card } from "@/components/ui/card";
import { cn } from "@/lib/utils";
import {
  Activity,
  Brain,
  CheckCircle2,
  Code2,
  Globe,
  Shield,
  Wrench,
  Zap,
} from "lucide-react";

export function BenchmarkDashboard() {
  return (
    <div className="w-full max-w-5xl mx-auto p-4 md:p-6">
      <Card className="relative overflow-hidden border-border/50 bg-background/50 backdrop-blur-sm">
        {/* Header / Top Bar */}
        <div className="flex items-center justify-between border-b border-border/50 px-6 py-4 bg-muted/20">
          <div className="flex items-center gap-3">
            <div className="flex gap-1.5 mr-2">
              <div className="h-3 w-3 rounded-full bg-destructive/80"></div>
              <div className="h-3 w-3 rounded-full bg-warning/80"></div>
              <div className="h-3 w-3 rounded-full bg-success/80"></div>
            </div>
            <Badge variant="outline" className="font-medium bg-background/50">
              Live Evaluation Dashboard
            </Badge>
          </div>
          <div className="flex items-center gap-4 text-xs text-muted-foreground">
            <span className="flex items-center gap-1.5">
              <span className="relative flex h-2 w-2">
                <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-success opacity-75"></span>
                <span className="relative inline-flex rounded-full h-2 w-2 bg-success"></span>
              </span>
              System Online
            </span>
            <Badge variant="secondary" className="text-xxs h-5">
              v2.4.0-beta
            </Badge>
          </div>
        </div>

        {/* Main Content Grid */}
        <div className="grid grid-cols-1 md:grid-cols-12 divide-y md:divide-y-0 md:divide-x divide-border/50">
          {/* Left Column: Radar Chart (Capabilities) */}
          <div className="md:col-span-7 p-6 md:p-8 flex flex-col items-center justify-center bg-gradient-to-b from-background/50 to-muted/5">
            <div className="text-center mb-6">
              <h3 className="text-lg font-semibold text-foreground flex items-center justify-center gap-2">
                <Brain className="w-4 h-4 text-primary" aria-hidden="true" />
                Model Capabilities
              </h3>
              <p className="text-sm text-muted-foreground mt-1">
                Comparing SOTA vs. Average Baseline
              </p>
            </div>

            <div className="relative w-full max-w-xs aspect-square">
              {/* Radar Chart SVG */}
              <svg viewBox="0 0 200 200" className="w-full h-full">
                {/* Background Grid (Pentagons) */}
                <g className="stroke-muted/20 fill-none" strokeWidth="1">
                  {[20, 40, 60, 80, 100].map((r, i) => (
                    <polygon key={i} points={getPentagonPoints(100, 100, r)} />
                  ))}
                </g>

                {/* Axes */}
                <g className="stroke-muted/20" strokeWidth="1">
                  {getPentagonPoints(100, 100, 100)
                    .split(" ")
                    .map((point, i) => {
                      const [x, y] = point.split(",");
                      return <line key={i} x1="100" y1="100" x2={x} y2={y} />;
                    })}
                </g>

                {/* Data: Average (Gray) */}
                <polygon
                  points={getPointsFromValues([60, 50, 70, 45, 65])}
                  className="fill-muted/20 stroke-muted/50"
                  strokeWidth="2"
                />

                {/* Data: SOTA (Primary) */}
                <polygon
                  points={getPointsFromValues([95, 85, 90, 80, 92])}
                  className="fill-primary/20 stroke-primary"
                  strokeWidth="2"
                  strokeLinecap="round"
                  strokeLinejoin="round"
                />

                {/* Axis Labels */}
                <foreignObject
                  x="80"
                  y="5"
                  width="40"
                  height="20"
                  className="text-center"
                >
                  <div className="text-xxs font-bold text-foreground">
                    Code
                  </div>
                </foreignObject>
                <foreignObject
                  x="160"
                  y="65"
                  width="40"
                  height="20"
                  className="text-center"
                >
                  <div className="text-xxs font-bold text-foreground">
                    Sec
                  </div>
                </foreignObject>
                <foreignObject
                  x="130"
                  y="165"
                  width="40"
                  height="20"
                  className="text-center"
                >
                  <div className="text-xxs font-bold text-foreground">
                    Tool
                  </div>
                </foreignObject>
                <foreignObject
                  x="30"
                  y="165"
                  width="40"
                  height="20"
                  className="text-center"
                >
                  <div className="text-xxs font-bold text-foreground">
                    Math
                  </div>
                </foreignObject>
                <foreignObject
                  x="0"
                  y="65"
                  width="40"
                  height="20"
                  className="text-center"
                >
                  <div className="text-xxs font-bold text-foreground">
                    Lang
                  </div>
                </foreignObject>
              </svg>
            </div>

            {/* Legend */}
            <div className="flex gap-6 mt-6 text-xs font-medium">
              <div className="flex items-center gap-2">
                <Badge
                  variant="outline"
                  className="h-3 w-3 p-0 bg-primary/20 border-primary rounded-sm"
                ></Badge>
                <span className="text-foreground">SOTA Model</span>
              </div>
              <div className="flex items-center gap-2">
                <Badge
                  variant="outline"
                  className="h-3 w-3 p-0 bg-muted/20 border-muted/50 rounded-sm"
                ></Badge>
                <span className="text-muted-foreground">Average</span>
              </div>
            </div>
          </div>

          {/* Right Column: Activity Feed */}
          <div className="md:col-span-5 flex flex-col bg-muted/10">
            <div className="p-4 border-b border-border/50 bg-muted/5">
              <h3 className="text-sm font-semibold text-foreground flex items-center gap-2">
                <Activity className="w-4 h-4 text-primary" aria-hidden="true" />
                Recent Evaluations
              </h3>
            </div>

            <div className="flex-1 overflow-hidden relative">
              <div className="absolute inset-0 overflow-y-auto p-4 space-y-3 scrollbar-hide">
                <TaskItem
                  icon={<Shield className="w-3.5 h-3.5 text-chart-5" />}
                  task="CVE-2023-Injection"
                  agent="Claude 3.5"
                  status="success"
                  time="2s ago"
                />
                <TaskItem
                  icon={<Code2 className="w-3.5 h-3.5 text-info" />}
                  task="Refactor Auth.ts"
                  agent="GPT-4o"
                  status="success"
                  time="5s ago"
                />
                <TaskItem
                  icon={<Wrench className="w-3.5 h-3.5 text-primary" />}
                  task="Docker Deploy"
                  agent="Llama 3"
                  status="fail"
                  time="12s ago"
                />
                <TaskItem
                  icon={<Globe className="w-3.5 h-3.5 text-success" />}
                  task="Translate Docs"
                  agent="Mistral Large"
                  status="success"
                  time="18s ago"
                />
                <TaskItem
                  icon={<Zap className="w-3.5 h-3.5 text-warning" />}
                  task="Optimize Query"
                  agent="Gemini Pro"
                  status="success"
                  time="24s ago"
                />
              </div>
              {/* Fade at bottom */}
              <div className="absolute bottom-0 left-0 right-0 h-12 bg-gradient-to-t from-background/80 to-transparent pointer-events-none"></div>
            </div>

            <div className="p-4 border-t border-border/50 bg-muted/5">
              <div className="flex justify-between items-center text-xs">
                <div className="text-muted-foreground">
                  <span className="font-semibold text-foreground">1,248</span>{" "}
                  tasks today
                </div>
                <div className="flex items-center gap-1 text-green-500 font-medium">
                  <CheckCircle2 className="w-3 h-3" aria-hidden="true" />
                  <span>98.2% Uptime</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </Card>
    </div>
  );
}

// Helper components & functions

function TaskItem({
  icon,
  task,
  agent,
  status,
  time,
}: {
  icon: React.ReactNode;
  task: string;
  agent: string;
  status: "success" | "fail";
  time: string;
}) {
  return (
    <Card className="flex items-center gap-3 p-2.5 bg-background/50 border-border/50 transition-colors hover:bg-background/80">
      <div
        className={cn(
          "p-1.5 rounded-md",
          status === "success" ? "bg-success/10" : "bg-destructive/10",
        )}
      >
        {icon}
      </div>
      <div className="flex-1 min-w-0">
        <div className="flex items-center justify-between">
          <span className="text-xs font-medium text-foreground truncate">
            {task}
          </span>
          <Badge
            variant="outline"
            className={cn(
              "text-xxs px-1.5 py-0.5 border-0",
              status === "success"
                ? "bg-success/10 text-success"
                : "bg-destructive/10 text-destructive",
            )}
          >
            {status === "success" ? "PASS" : "FAIL"}
          </Badge>
        </div>
        <div className="flex items-center justify-between mt-1">
          <span className="text-xxs text-muted-foreground">
            Agent: {agent}
          </span>
          <span className="text-xxs text-muted-foreground/60">{time}</span>
        </div>
      </div>
    </Card>
  );
}

// Radar Chart Math Helpers
function getPentagonPoints(cx: number, cy: number, r: number) {
  const points = [];
  for (let i = 0; i < 5; i++) {
    const a = (i * 72 - 90) * (Math.PI / 180);
    points.push(`${cx + r * Math.cos(a)},${cy + r * Math.sin(a)}`);
  }
  return points.join(" ");
}

function getPointsFromValues(values: number[]) {
  // values are 0-100
  const cx = 100;
  const cy = 100;
  const maxR = 100;

  return values
    .map((v, i) => {
      const r = (v / 100) * maxR;
      const a = (i * 72 - 90) * (Math.PI / 180);
      return `${cx + r * Math.cos(a)},${cy + r * Math.sin(a)}`;
    })
    .join(" ");
}
