import { PageLayout } from "@/components/PageLayout";
import { Button } from "@/components/ui/button";
import { getContributors } from "@/utils/github";
import { Github, Users } from "lucide-react";
import { ContributorCard } from "@/components/ContributorCard";
import type { Metadata } from "next";

export const metadata: Metadata = {
  title: "Contributors",
  description:
    "Meet the developers and researchers building SkillsBench — the open benchmark for AI agent skills evaluation.",
  alternates: { canonical: "https://skillsbench.ai/contributors" },
};

export default async function ContributorsPage() {
  const contributors = await getContributors();

  // Sort by contributions just in case API doesn't
  const sortedContributors = [...contributors].sort(
    (a, b) => b.contributions - a.contributions,
  );

  return (
    <PageLayout
      title="Contributors"
      description="The SkillsBench community is building the future of agent evaluation. Thank you to all the developers and researchers who have contributed tasks and code."
      icon={null}
    >
      {sortedContributors.length > 0 ? (
        <>
          <div className="mb-6 flex items-center gap-2 text-sm text-muted-foreground uppercase tracking-wider">
            <span className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></span>
            {sortedContributors.length} Active Contributors
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {sortedContributors.map((contributor, index) => (
              <ContributorCard
                key={contributor.login}
                contributor={contributor}
                rank={index + 1}
              />
            ))}
          </div>
        </>
      ) : (
        <div className="text-center py-24 border border-dashed border-border rounded-xl bg-muted/30">
          <div className="flex justify-center mb-4">
            <Users className="w-12 h-12 text-muted-foreground/50" />
          </div>
          <p className="text-muted-foreground">
            Unable to load contributors at this time.
          </p>
          <p className="text-xs text-muted-foreground mt-2">
            Please check back later.
          </p>
        </div>
      )}

      <div className="text-center mt-16">
        <Button asChild>
          <a
            href="https://github.com/benchflow-ai/skillsbench"
            target="_blank"
            rel="noopener noreferrer"
          >
            <Github className="w-4 h-4" />
            Become a Contributor
          </a>
        </Button>
      </div>
    </PageLayout>
  );
}
