import * as THREE from 'three';

export function createScene() {
  const root = new THREE.Group();
  root.name = 'desk_globe';

  // --- Scale: use centimeters converted to scene units ---
  const CM = 0.4;

  // --- Dimensions (in scene units) ---
  // Base
  const baseDiameterCm = 20;
  const baseThicknessCm = 2;
  const baseRadius = (baseDiameterCm * 0.5) * CM;      // 10cm -> 4.0
  const baseHeight = baseThicknessCm * CM;             // 2cm  -> 0.8

  // Pivot Assembly
  const pivotHeightCm = 5;
  const pivotDiameterCm = 3;
  const pivotHeight = pivotHeightCm * CM;              // 5cm  -> 2.0
  const pivotRadius = (pivotDiameterCm * 0.5) * CM;    // 1.5cm -> 0.6
  const pivotTopY = baseHeight + pivotHeight;          // top of pivot from world origin

  // Globe
  const globeDiameterCm = 30;
  const globeRadius = (globeDiameterCm * 0.5) * CM;     // 15cm -> 6.0

  // Meridian Arch
  const archBandWidthCm = 2;
  const archTubeRadius = (archBandWidthCm * 0.5) * CM;  // 1cm -> 0.4
  const archCenterlineRadius = globeRadius + 5 * CM;    // 6.0 + 2.0 = 8.0
  const archArc = Math.PI * 1.3; // 234 degrees

  // Axis Pins
  const pinDiameterCm = 0.5;
  const pinRadius = (pinDiameterCm * 0.5) * CM;         // 0.25cm -> 0.1
  const archInnerRadius = archCenterlineRadius - archTubeRadius;
  const pinLength = archInnerRadius - globeRadius;

  // Standard Earth tilt
  const earthTilt = 23.5 * Math.PI / 180;

  // --- Top-level Groups (main links) ---
  const base_stand = new THREE.Group();
  base_stand.name = 'base_stand';

  const meridian_arch = new THREE.Group();
  meridian_arch.name = 'meridian_arch';

  const globe_sphere = new THREE.Group();
  globe_sphere.name = 'globe_sphere';

  // --- BASE_STAND: includes base disc and pivot_assembly ---
  const baseDiscGeometry = new THREE.CylinderGeometry(baseRadius, baseRadius, baseHeight, 48);
  const baseDisc = new THREE.Mesh(baseDiscGeometry);
  baseDisc.name = 'base_disc';
  baseDisc.position.set(0, baseHeight / 2, 0);
  base_stand.add(baseDisc);

  const pivotGeometry = new THREE.CylinderGeometry(pivotRadius, pivotRadius, pivotHeight, 32);
  const pivot_assembly = new THREE.Mesh(pivotGeometry);
  pivot_assembly.name = 'pivot_assembly';
  pivot_assembly.position.set(0, baseHeight + (pivotHeight / 2), 0);
  base_stand.add(pivot_assembly);

  // --- NAMEPLATE: non-uniform + mirrored scale with rotation chain ---
  const nameplate_group = new THREE.Group();
  nameplate_group.name = 'nameplate_group';
  nameplate_group.position.set(baseRadius * 0.6, baseHeight * 0.6, -baseRadius * 0.4);
  nameplate_group.rotation.y = Math.PI / 8;
  nameplate_group.scale.set(1.2, 0.5, -0.8); // mirrored Z scale

  const nameplate_tilt = new THREE.Group();
  nameplate_tilt.name = 'nameplate_tilt';
  nameplate_tilt.rotation.x = -Math.PI / 12;
  nameplate_tilt.rotation.z = Math.PI / 18;
  nameplate_tilt.scale.set(0.9, 1.3, 1.1); // additional non-uniform scaling
  nameplate_group.add(nameplate_tilt);

  const nameplateGeometry = new THREE.BoxGeometry(baseRadius * 0.6, baseHeight * 0.25, baseRadius * 0.2);
  const nameplate = new THREE.Mesh(nameplateGeometry, new THREE.MeshBasicMaterial());
  nameplate.name = 'nameplate';
  nameplate.position.set(0, baseHeight * 0.1, 0);
  nameplate_tilt.add(nameplate);
  base_stand.add(nameplate_group);

  // --- MERIDIAN_ARCH: Now a complete assembly ---
  meridian_arch.position.set(0, pivotTopY + archCenterlineRadius, 0);

  const archBandGeometry = new THREE.TorusGeometry(archCenterlineRadius, archTubeRadius, 24, 96, archArc);
  const angleToCenterOpening = -Math.PI / 2 - (archArc - Math.PI) / 2;
  archBandGeometry.rotateZ(angleToCenterOpening);

  const archBand = new THREE.Mesh(archBandGeometry);
  archBand.name = 'arch_band';
  archBand.rotation.y = Math.PI / 2;
  meridian_arch.add(archBand);

  const connectionMountGeom = new THREE.CylinderGeometry(pivotRadius * 1.5, pivotRadius * 1.5, archTubeRadius * 2, 32);
  const connectionMount = new THREE.Mesh(connectionMountGeom);
  connectionMount.name = 'connection_mount';
  connectionMount.position.y = -archCenterlineRadius;
  meridian_arch.add(connectionMount);

  // Axis pins connect the arch and globe.
  const axis_pins = new THREE.Group();
  axis_pins.name = 'axis_pins';
  meridian_arch.add(axis_pins);

  const pinGeometry = new THREE.CylinderGeometry(pinRadius, pinRadius, pinLength, 16);
  const pinPositionRadius = globeRadius + (pinLength / 2);

  const pinMaterial = new THREE.MeshBasicMaterial();
  const axisPins = new THREE.InstancedMesh(pinGeometry, pinMaterial, 2);
  axisPins.name = 'axis_pins_instances';

  const temp = new THREE.Object3D();

  temp.rotation.x = earthTilt;
  temp.position.set(0, pinPositionRadius * Math.cos(earthTilt), pinPositionRadius * Math.sin(earthTilt));
  temp.updateMatrix();
  axisPins.setMatrixAt(0, temp.matrix);

  temp.rotation.x = earthTilt;
  temp.position.set(0, -pinPositionRadius * Math.cos(earthTilt), -pinPositionRadius * Math.sin(earthTilt));
  temp.updateMatrix();
  axisPins.setMatrixAt(1, temp.matrix);

  axisPins.instanceMatrix.needsUpdate = true;
  axis_pins.add(axisPins);

  // --- GLOBE_SPHERE: Centered in arch, parented for proper connection ---
  meridian_arch.add(globe_sphere);

  globe_sphere.position.set(0, 0, 0);
  globe_sphere.rotation.x = earthTilt;

  const globeMapSurfaceGeometry = new THREE.SphereGeometry(globeRadius, 48, 32);
  const globeMapSurface = new THREE.Mesh(globeMapSurfaceGeometry);
  globeMapSurface.name = 'globe_map_surface';
  globe_sphere.add(globeMapSurface);

  // --- Assemble top-level parts ---
  root.add(base_stand);
  root.add(meridian_arch);

  return root;
}
