"""
Time Series Anomaly Detection Skill - Prophet-based Category-Level Trend Analysis

This module provides a comprehensive framework for detecting surge and slump
anomalies in time series data at the category level. It uses Facebook Prophet
for forecasting and calculates anomaly indices based on deviations from predictions.

Key Features:
- Prophet-based time series forecasting per category
- Counterfactual prediction for intervention analysis
- Anomaly index calculation with confidence intervals
- Support for any categorical grouping and metric

Usage:
    from anomaly_detection import TimeSeriesAnomalyDetector

    detector = TimeSeriesAnomalyDetector()
    results = detector.detect_anomalies(
        df=purchases_df,
        date_col='<date column>',
        category_col='<grouping column>',
        value_col='<metric column>',
        cutoff_date='<cutoff date>',
        prediction_end='<end date>'
    )
"""

import pandas as pd
import numpy as np
from typing import Dict, Any, List, Optional, Tuple
from datetime import datetime, timedelta
import warnings
import os
import sys
warnings.filterwarnings('ignore')

# Suppress cmdstan output BEFORE importing prophet
os.environ['CMDSTAN_VERBOSE'] = 'false'

# Completely suppress Prophet and cmdstanpy logging BEFORE import
import logging
logging.getLogger('prophet').setLevel(logging.CRITICAL)
logging.getLogger('cmdstanpy').setLevel(logging.CRITICAL)
logging.getLogger('stan').setLevel(logging.CRITICAL)

# Disable cmdstanpy logger handlers
cmdstanpy_logger = logging.getLogger('cmdstanpy')
cmdstanpy_logger.handlers = []
cmdstanpy_logger.addHandler(logging.NullHandler())
cmdstanpy_logger.propagate = False

try:
    from prophet import Prophet
    # Re-suppress after import (prophet may reset)
    logging.getLogger('cmdstanpy').setLevel(logging.CRITICAL)
    logging.getLogger('cmdstanpy').handlers = []
    logging.getLogger('cmdstanpy').addHandler(logging.NullHandler())
except ImportError:
    raise ImportError("Prophet is required. Install with: pip install prophet")

try:
    from tqdm import tqdm
except ImportError:
    # Fallback if tqdm not installed
    def tqdm(iterable, **kwargs):
        return iterable


class TimeSeriesAnomalyDetector:
    """
    Time series anomaly detection using Prophet forecasting.

    Detects surge (positive) and slump (negative) anomalies by comparing
    actual values against counterfactual predictions.
    """

    def __init__(self,
                 min_training_days: int = 180,
                 confidence_interval: float = 0.68,
                 changepoint_prior_scale: float = 0.05,
                 seasonality_prior_scale: float = 10.0):
        """
        Initialize the anomaly detector.

        Args:
            min_training_days: Minimum days of training data required per category
            confidence_interval: Prophet confidence interval width (0.68 = ±1 std)
            changepoint_prior_scale: Flexibility of trend changes
            seasonality_prior_scale: Strength of seasonality
        """
        self.min_training_days = min_training_days
        self.confidence_interval = confidence_interval
        self.changepoint_prior_scale = changepoint_prior_scale
        self.seasonality_prior_scale = seasonality_prior_scale
        self.models = {}
        self.predictions = {}

    def _aggregate_by_category_date(self,
                                    df: pd.DataFrame,
                                    date_col: str,
                                    category_col: str,
                                    value_col: str,
                                    agg_func: str = 'sum') -> pd.DataFrame:
        """
        Aggregate data by category and date.

        Args:
            df: Input dataframe
            date_col: Date column name
            category_col: Category column name
            value_col: Value column to aggregate
            agg_func: Aggregation function ('sum', 'mean', 'count')

        Returns:
            Aggregated dataframe with columns [category_col, 'date', 'value']
        """
        df = df.copy()
        df[date_col] = pd.to_datetime(df[date_col])

        aggregated = df.groupby([category_col, date_col])[value_col].agg(agg_func).reset_index()
        aggregated.columns = [category_col, 'date', 'value']

        return aggregated

    def _train_prophet_model(self,
                             train_data: pd.DataFrame) -> Optional[Prophet]:
        """
        Train a Prophet model on the given training data.

        Args:
            train_data: DataFrame with 'ds' and 'y' columns

        Returns:
            Fitted Prophet model or None if insufficient data
        """
        # Check unique dates (not just row count)
        if train_data['ds'].nunique() < self.min_training_days:
            return None

        model = Prophet(
            daily_seasonality=True,
            weekly_seasonality=True,
            yearly_seasonality=True,
            interval_width=self.confidence_interval,
            changepoint_prior_scale=self.changepoint_prior_scale,
            seasonality_prior_scale=self.seasonality_prior_scale
        )

        model.fit(train_data)
        return model

    def _calculate_anomaly_index(self,
                                 actual: float,
                                 yhat: float,
                                 yhat_lower: float,
                                 yhat_upper: float) -> float:
        """
        Calculate Anomaly Index for a single observation.

        Anomaly Index Formula:
        - Within CI: 0
        - Above upper: (actual - upper) / std
        - Below lower: (actual - lower) / std

        Where std = (upper - lower) / 2 (from 68% CI = ±1 std)

        Args:
            actual: Actual observed value
            yhat: Predicted value
            yhat_lower: Lower bound of confidence interval
            yhat_upper: Upper bound of confidence interval

        Returns:
            Raw Anomaly Index (unscaled)
        """
        if pd.isna(actual):
            return 0.0

        std = (yhat_upper - yhat_lower) / 2
        if std <= 0:
            return 0.0

        if yhat_lower <= actual <= yhat_upper:
            return 0.0
        elif actual > yhat_upper:
            return (actual - yhat_upper) / std
        else:
            return (actual - yhat_lower) / std

    def _scale_anomaly_index(self, raw_index: float) -> float:
        """
        Scale Anomaly Index to [-100, 100] range using tanh.

        Mapping:
        - index within 1 std → ~76 points
        - index at 2 std → ~96 points
        - index at 3 std → ~99.5 points (approaches ±100)
        - Beyond 3 std → approaches ±100

        Args:
            raw_index: Raw Anomaly Index

        Returns:
            Scaled Anomaly Index in [-100, 100]
        """
        return 100 * np.tanh(raw_index)

    def detect_anomalies(self,
                         df: pd.DataFrame,
                         date_col: str,
                         category_col: str,
                         value_col: str,
                         cutoff_date: str,
                         prediction_start: Optional[str] = None,
                         prediction_end: str = None,
                         agg_func: str = 'sum') -> Dict[str, Any]:
        """
        Detect anomalies across all categories.

        Args:
            df: Input dataframe
            date_col: Date column name
            category_col: Category column name
            value_col: Value column to analyze
            cutoff_date: Last date of training data (format: 'YYYY-MM-DD')
            prediction_start: Start of prediction period (default: cutoff_date)
            prediction_end: End of prediction period (format: 'YYYY-MM-DD')
            agg_func: Aggregation function for daily values

        Returns:
            Dictionary containing:
            - 'anomaly_summary': DataFrame with anomaly indices per category
            - 'predictions': Detailed predictions per category
            - 'models': Trained Prophet models per category
        """
        # Parse dates
        cutoff = pd.Timestamp(cutoff_date)
        pred_start = pd.Timestamp(prediction_start) if prediction_start else cutoff
        pred_end = pd.Timestamp(prediction_end)

        # Aggregate data
        daily_data = self._aggregate_by_category_date(
            df, date_col, category_col, value_col, agg_func
        )

        # Split train/test
        train_data = daily_data[daily_data['date'] < cutoff].copy()
        test_data = daily_data[
            (daily_data['date'] >= pred_start) &
            (daily_data['date'] <= pred_end)
        ].copy()

        categories = train_data[category_col].unique()
        anomaly_results = []

        print(f"Training Prophet models for {len(categories)} categories...")

        # Use tqdm for progress bar
        for category in tqdm(categories, desc="Training models", unit=category_col):
            # Prepare training data for Prophet
            cat_train = train_data[train_data[category_col] == category][['date', 'value']].copy()
            cat_train.columns = ['ds', 'y']

            # Check unique dates (not just row count) for min_training_days
            if cat_train['ds'].nunique() < self.min_training_days:
                continue

            try:
                # Train model
                model = self._train_prophet_model(cat_train)
                if model is None:
                    continue

                self.models[category] = model

                # Create future dataframe
                future = pd.DataFrame({
                    'ds': pd.date_range(start=pred_start, end=pred_end, freq='D')
                })

                # Make predictions
                forecast = model.predict(future)
                self.predictions[category] = forecast[['ds', 'yhat', 'yhat_lower', 'yhat_upper']].copy()

                # Get actual values
                cat_actual = test_data[test_data[category_col] == category].copy()

                if len(cat_actual) == 0:
                    continue

                # Merge predictions with actuals
                merged = forecast.merge(
                    cat_actual[['date', 'value']],
                    left_on='ds',
                    right_on='date',
                    how='left'
                )
                merged = merged.rename(columns={'value': 'actual'})

                # Calculate anomaly indices
                merged['raw_anomaly'] = merged.apply(
                    lambda row: self._calculate_anomaly_index(
                        row['actual'], row['yhat'], row['yhat_lower'], row['yhat_upper']
                    ), axis=1
                )
                merged['scaled_anomaly'] = merged['raw_anomaly'].apply(self._scale_anomaly_index)

                # Aggregate to category-level
                total_anomaly_index = merged['scaled_anomaly'].mean()

                anomaly_results.append({
                    category_col: category,
                    'Anomaly_Index': total_anomaly_index,
                    'days_with_data': merged['actual'].notna().sum(),
                    'avg_daily_anomaly': merged['scaled_anomaly'].mean(),
                    'max_daily_anomaly': merged['scaled_anomaly'].max(),
                    'min_daily_anomaly': merged['scaled_anomaly'].min(),
                    'total_actual': merged['actual'].sum(),
                    'total_predicted': merged['yhat'].sum()
                })

            except Exception as e:
                print(f"  Error processing {category}: {str(e)}")
                continue

        # Create summary DataFrame
        if not anomaly_results:
            raise ValueError("All categories failed anomaly detection. Check Prophet installation and data quality.")

        anomaly_summary = pd.DataFrame(anomaly_results)
        anomaly_summary = anomaly_summary.sort_values('Anomaly_Index', ascending=False)

        print(f"Anomaly detection complete: {len(anomaly_summary)} categories analyzed")

        return {
            'anomaly_summary': anomaly_summary,
            'predictions': self.predictions,
            'models': self.models
        }
