"""
Difference-in-Differences (DiD) Core Analysis Module

Methodology:
- Multivariate Heterogeneous DiD: Y = α + βX + γPost + δ(X × Post) + ε
  - Uses statsmodels OLS regression with interaction terms
  - Provides p-values for statistical inference

- Univariate DiD: (Treat_Post - Treat_Pre) - (Control_Post - Control_Pre)
  - Fallback for small samples
  - No statistical inference
"""

import pandas as pd
import numpy as np
from typing import Dict, Any, List, Optional
import statsmodels.formula.api as smf
import warnings
warnings.filterwarnings('ignore')


class DIDAnalyzer:
    """
    Core DiD analysis engine for pre-filtered data.

    Supports both Multivariate Heterogeneous DiD and Univariate DiD.
    Caller handles filtering, looping, and report building.
    """

    def __init__(self,
                 min_sample_ratio: int = 10,
                 significance_level: float = 0.05,
                 min_group_size: int = 1):
        """
        Initialize DiD analyzer.

        Args:
            min_sample_ratio: Min samples per feature for multivariate DiD
            significance_level: P-value threshold for significance
            min_group_size: Min observations per group for univariate DiD
        """
        self.min_sample_ratio = min_sample_ratio
        self.significance_level = significance_level
        self.min_group_size = min_group_size

    def _clean_feature_name(self, name: str) -> str:
        """Clean feature name for statsmodels formula."""
        return (name.replace('-', '_').replace(',', '').replace('/', '_')
                    .replace('$', 'dollar').replace(' ', '_').replace('(', '')
                    .replace(')', '').replace('+', 'plus').replace('*', 'star')
                    .replace(':', '_').replace('.', '_').replace('%', 'pct')
                    .replace('&', 'and').replace('=', 'eq'))

    def _multivariate_heterogeneous_did(self,
                                        df: pd.DataFrame,
                                        features: List[str],
                                        outcome_col: str) -> Optional[pd.DataFrame]:
        """
        Multivariate Heterogeneous DiD: Y = α + βX + γPost + δ(X × Post) + ε

        Args:
            df: Data with 'Period' column ('baseline'/'treatment')
            features: Feature columns (can be binary or continuous)
            outcome_col: Outcome variable

        Returns:
            DataFrame with DiD estimates, or None if insufficient data
        """
        # Create Post indicator
        data = df.copy()
        data['Post'] = (data['Period'] == 'treatment').astype(int)

        n_samples = len(data)
        n_features = len(features)

        if n_samples / n_features < self.min_sample_ratio:
            return None

        # Filter valid features
        feature_mapping = {}
        valid_features = []

        for feat in features:
            if feat not in data.columns:
                continue
            if data[feat].nunique() < 2:
                continue

            clean_feat = self._clean_feature_name(feat)
            feature_mapping[feat] = clean_feat
            valid_features.append(feat)

        if len(valid_features) == 0:
            return None

        # Prepare regression data
        temp_data = data.copy()
        rename_dict = {feat: feature_mapping[feat] for feat in valid_features}
        rename_dict[outcome_col] = 'Outcome'
        temp_data = temp_data.rename(columns=rename_dict)

        # Build formula: Outcome ~ features + Post + features:Post
        clean_names = [feature_mapping[f] for f in valid_features]
        main_effects = ' + '.join(clean_names)
        interactions = ' + '.join([f'{f}:Post' for f in clean_names])
        formula = f'Outcome ~ {main_effects} + Post + {interactions}'

        try:
            model = smf.ols(formula, data=temp_data).fit()

            results = []
            for feat in valid_features:
                clean_feat = feature_mapping[feat]
                interaction_name = f'{clean_feat}:Post'

                if interaction_name in model.params.index:
                    did = model.params[interaction_name]
                    pval = model.pvalues[interaction_name]
                    se = model.bse[interaction_name]

                    results.append({
                        'Feature': feat,
                        'DiD_Estimate': did,
                        'P_Value': pval,
                        'Std_Error': se,
                        'CI_Lower': did - 1.96 * se,
                        'CI_Upper': did + 1.96 * se,
                        'Significant': pval < self.significance_level
                    })

            return pd.DataFrame(results) if results else None

        except Exception:
            return None

    def _univariate_did(self,
                       df: pd.DataFrame,
                       features: List[str],
                       outcome_col: str) -> Optional[pd.DataFrame]:
        """
        Univariate DiD: Run separate regression for each feature.
        Y = α + βX + γPost + δ(X×Post) for each feature individually.

        Args:
            df: Data with 'Period' column
            features: Feature columns (should be binary)
            outcome_col: Outcome variable

        Returns:
            DataFrame with DiD estimates, or None if insufficient data
        """
        # Create Post indicator
        data = df.copy()
        data['Post'] = (data['Period'] == 'treatment').astype(int)

        if len(data) < 20:
            return None

        results = []

        for feat in features:
            try:
                if feat not in data.columns:
                    continue
                if data[feat].nunique() < 2:
                    continue

                # Check group sizes
                baseline_df = data[data['Period'] == 'baseline']
                treatment_df = data[data['Period'] == 'treatment']

                baseline_treat = baseline_df[baseline_df[feat] == 1]
                treatment_treat = treatment_df[treatment_df[feat] == 1]
                baseline_ctrl = baseline_df[baseline_df[feat] == 0]
                treatment_ctrl = treatment_df[treatment_df[feat] == 0]

                if (len(baseline_treat) < self.min_group_size or
                    len(treatment_treat) < self.min_group_size or
                    len(baseline_ctrl) < self.min_group_size or
                    len(treatment_ctrl) < self.min_group_size):
                    continue

                # Run regression for this single feature
                clean_feat = self._clean_feature_name(feat)
                temp_data = data[[outcome_col, feat, 'Post']].copy()
                temp_data = temp_data.rename(columns={
                    outcome_col: 'Outcome',
                    feat: clean_feat
                })

                formula = f'Outcome ~ {clean_feat} + Post + {clean_feat}:Post'
                model = smf.ols(formula, data=temp_data).fit()

                interaction_name = f'{clean_feat}:Post'
                if interaction_name in model.params.index:
                    did = model.params[interaction_name]
                    pval = model.pvalues[interaction_name]
                    se = model.bse[interaction_name]

                    results.append({
                        'Feature': feat,
                        'DiD_Estimate': did,
                        'P_Value': pval,
                        'Std_Error': se,
                        'CI_Lower': did - 1.96 * se,
                        'CI_Upper': did + 1.96 * se,
                        'Significant': pval < self.significance_level
                    })
            except Exception:
                continue

        return pd.DataFrame(results) if results else None

    def _apply_sorting(self, result_df: pd.DataFrame, sort_by: str, asc: bool) -> pd.DataFrame:
        """Apply sorting to results based on sort_by parameter."""
        if sort_by == 'estimate':
            result_df = result_df.sort_values('DiD_Estimate', ascending=asc)
        elif sort_by == 'p_value':
            result_df['Sort_Key'] = result_df['P_Value'].fillna(1.0) if 'P_Value' in result_df.columns else 1.0
            result_df = result_df.sort_values('Sort_Key', ascending=asc)
        return result_df

    def intensive_margin(self,
                        df: pd.DataFrame,
                        features: List[str],
                        value_col: str,
                        top_n: Optional[int] = None,
                        sort_by: str = 'estimate',
                        asc: bool = False) -> List[Dict[str, Any]]:
        """
        Analyze intensive margin (how much behavior changed).

        Uses Multivariate Heterogeneous DiD when sample size is sufficient,
        otherwise falls back to Univariate DiD.

        Args:
            df: Pre-filtered data with 'Period' column ('baseline'/'treatment')
            features: Feature columns to analyze
            value_col: Continuous outcome variable (e.g., 'Total_Spend')
            top_n: Number of top features to return (None = return all)
            sort_by: 'estimate' or 'p_value'
            asc: True for ascending, False for descending

        Returns:
            List of dicts: [
                {"feature": "age_group_1", "did_estimate": 12.5,
                 "p_value": 0.03, "method": "Multivariate Heterogeneous DiD"},
                {"feature": "income_high", "did_estimate": 8.2,
                 "p_value": None, "method": "Univariate DiD"},
                ...
            ]
        """
        # Try multivariate heterogeneous DiD first
        result_df = self._multivariate_heterogeneous_did(df, features, value_col)
        method = 'Multivariate Heterogeneous DiD'

        # Fallback to univariate DiD if needed
        if result_df is None or len(result_df) == 0:
            result_df = self._univariate_did(df, features, value_col)
            method = 'Univariate DiD'

        if result_df is None or len(result_df) == 0:
            return []

        # Apply sorting
        result_df = self._apply_sorting(result_df, sort_by, asc)

        # Convert to list of dicts
        results = []
        for _, row in result_df.iterrows():
            entry = {
                'feature': row['Feature'],
                'did_estimate': float(row['DiD_Estimate']),
                'method': method
            }
            # Add p_value if available (multivariate only)
            if 'P_Value' in row and pd.notna(row['P_Value']):
                entry['p_value'] = float(row['P_Value'])
            else:
                entry['p_value'] = None

            results.append(entry)

        # Limit to top_n if specified
        if top_n is not None and top_n > 0:
            results = results[:top_n]

        return results

    def extensive_margin(self,
                        df: pd.DataFrame,
                        features: List[str],
                        value_col: str = 'purchased',
                        top_n: Optional[int] = None,
                        sort_by: str = 'estimate',
                        asc: bool = False) -> List[Dict[str, Any]]:
        """
        Analyze extensive margin (how many people changed).

        Expects df to contain a binary outcome column (0/1).
        Uses Multivariate Heterogeneous DiD when sample size is sufficient,
        otherwise falls back to Univariate DiD.

        Args:
            df: Data with 'Period' column and binary outcome (0/1)
            features: Feature columns to analyze
            value_col: Binary outcome column name (default: 'has_purchase')
            top_n: Number of top features to return (None = return all)
            sort_by: 'estimate' or 'p_value'
            asc: True for ascending, False for descending

        Returns:
            List of dicts: [{"feature": str, "did_estimate": float, "p_value": float|None, "method": str}]
        """
        # Check if value_col exists
        if value_col not in df.columns:
            raise ValueError(f"DataFrame must contain '{value_col}' column for extensive margin analysis")

        # Use the data as-is (should already be entity-period level with binary outcome)
        entity_df = df.copy()

        # Try multivariate heterogeneous DiD first
        result_df = self._multivariate_heterogeneous_did(entity_df, features, value_col)
        method = 'Multivariate Heterogeneous DiD'

        # Fallback to univariate DiD if needed
        if result_df is None or len(result_df) == 0:
            result_df = self._univariate_did(entity_df, features, value_col)
            method = 'Univariate DiD'

        if result_df is None or len(result_df) == 0:
            return []

        # Apply sorting
        result_df = self._apply_sorting(result_df, sort_by, asc)

        # Convert to list of dicts
        results = []
        for _, row in result_df.iterrows():
            entry = {
                'feature': row['Feature'],
                'did_estimate': float(row['DiD_Estimate']),
                'method': method
            }
            # Add p_value if available (multivariate only)
            if 'P_Value' in row and pd.notna(row['P_Value']):
                entry['p_value'] = float(row['P_Value'])
            else:
                entry['p_value'] = None

            results.append(entry)

        # Limit to top_n if specified
        if top_n is not None and top_n > 0:
            results = results[:top_n]

        return results
