#!/usr/bin/env node

import minimist from 'minimist';
import {
  listAccounts,
  setDefaultAccount,
  removeAccount,
  getDefaultAccount
} from './auth/auth-utils.js';

function printUsage() {
  console.log(`
📋 Google Calendar Account Manager

Usage:
  node manage-accounts.js --list                    List all configured accounts
  node manage-accounts.js --set-default <id>        Set default account
  node manage-accounts.js --remove <id>             Remove an account
  node manage-accounts.js --info                    Show detailed account info

Examples:
  node manage-accounts.js --list
  node manage-accounts.js --set-default work
  node manage-accounts.js --remove old-account
`);
}

async function main() {
  const args = minimist(process.argv.slice(2));

  try {
    // List accounts
    if (args.list || args.l) {
      const accounts = listAccounts();

      if (accounts.length === 0) {
        console.log('\n❌ No accounts configured.');
        console.log('   Run: npm run setup -- --account <account-id>\n');
        return;
      }

      console.log('\n📋 Configured Google Calendar Accounts:\n');
      accounts.forEach(acc => {
        const marker = acc.isDefault ? '⭐' : '  ';
        console.log(`${marker} ${acc.id}`);
        console.log(`   Email: ${acc.email}`);
        console.log(`   Scopes: ${acc.scope}`);
        console.log('');
      });

      console.log(`Total accounts: ${accounts.length}`);
      console.log(`Default: ${accounts.find(a => a.isDefault)?.id || 'none'}\n`);
      return;
    }

    // Show info
    if (args.info || args.i) {
      const accounts = listAccounts();
      const defaultId = getDefaultAccount();

      console.log('\n📊 Account Information:\n');
      console.log(JSON.stringify({
        totalAccounts: accounts.length,
        defaultAccount: defaultId,
        accounts: accounts
      }, null, 2));
      console.log('');
      return;
    }

    // Set default account
    if (args['set-default']) {
      const accountId = args['set-default'];
      setDefaultAccount(accountId);
      console.log(`\n✅ Default account set to: ${accountId}\n`);
      return;
    }

    // Remove account
    if (args.remove) {
      const accountId = args.remove;

      const accounts = listAccounts();
      const account = accounts.find(a => a.id === accountId);

      if (!account) {
        console.error(`\n❌ Account '${accountId}' not found.\n`);
        process.exit(1);
      }

      if (account.isDefault && accounts.length > 1) {
        console.log(`\n⚠️  Removing default account '${accountId}'.`);
        console.log(`   A new default will be automatically assigned.\n`);
      }

      removeAccount(accountId);
      console.log(`✅ Account '${accountId}' (${account.email}) removed.\n`);

      const remaining = listAccounts();
      if (remaining.length > 0) {
        console.log('Remaining accounts:');
        remaining.forEach(acc => {
          const marker = acc.isDefault ? '⭐' : '  ';
          console.log(`${marker} ${acc.id} (${acc.email})`);
        });
        console.log('');
      }

      return;
    }

    // No recognized action
    printUsage();

  } catch (error) {
    console.error(`\n❌ Error: ${error.message}\n`);
    process.exit(1);
  }
}

main();
