#!/usr/bin/env node

import { google } from 'googleapis';
import minimist from 'minimist';
import { getAuthClient, parseAccountArg } from './auth/auth-utils.js';
import { logAction } from './action-logger.js';

async function listCalendars(args) {
  // Get authenticated client for specified account (or default)
  const accountId = parseAccountArg(args);
  const oauth2Client = await getAuthClient(accountId);

  const calendar = google.calendar({ version: 'v3', auth: oauth2Client });

  // List calendars
  const params = {
    minAccessRole: args.role || 'reader'
  };

  const result = await calendar.calendarList.list(params);

  const calendars = result.data.items.map(cal => ({
    id: cal.id,
    summary: cal.summary,
    description: cal.description,
    timeZone: cal.timeZone,
    primary: cal.primary || false,
    accessRole: cal.accessRole,
    backgroundColor: cal.backgroundColor
  }));

  return {
    success: true,
    count: calendars.length,
    calendars
  };
}

// Main execution
const args = minimist(process.argv.slice(2));
const startTime = Date.now();

listCalendars(args)
  .then(result => {
    // Log the action
    logAction('list_calendars', args, result, {
      account: args.account || 'default',
      duration: Date.now() - startTime
    });

    console.log(JSON.stringify(result, null, 2));
  })
  .catch(error => {
    const errorResult = {
      success: false,
      error: error.message
    };

    // Log the failed action
    logAction('list_calendars', args, errorResult, {
      account: args.account || 'default',
      duration: Date.now() - startTime,
      failed: true
    });

    console.error(JSON.stringify(errorResult, null, 2));
    process.exit(1);
  });
