#!/usr/bin/env node

import { google } from 'googleapis';
import minimist from 'minimist';
import { getAuthClient, parseAccountArg } from './auth/auth-utils.js';
import { logAction } from './action-logger.js';

async function listEvents(args) {
  // Get authenticated client for specified account (or default)
  const accountId = parseAccountArg(args);
  const oauth2Client = await getAuthClient(accountId);

  const calendar = google.calendar({ version: 'v3', auth: oauth2Client });

  // Build query parameters
  const params = {
    calendarId: args.calendar || 'primary',
    singleEvents: true,
    orderBy: 'startTime'
  };

  // Time range
  if (args.timeMin) {
    params.timeMin = args.timeMin;
  } else {
    // Default to now
    params.timeMin = new Date().toISOString();
  }

  if (args.timeMax) {
    params.timeMax = args.timeMax;
  }

  // Search query
  if (args.q || args.query) {
    params.q = args.q || args.query;
  }

  // Limit results
  if (args.limit || args.maxResults) {
    params.maxResults = parseInt(args.limit || args.maxResults);
  } else {
    params.maxResults = 10; // Default limit
  }

  // Show deleted events
  if (args.showDeleted) {
    params.showDeleted = true;
  }

  const result = await calendar.events.list(params);

  const events = (result.data.items || []).map(event => {
    const eventData = {
      id: event.id,
      summary: event.summary,
      description: event.description,
      location: event.location,
      status: event.status,
      htmlLink: event.htmlLink,
      created: event.created,
      updated: event.updated
    };

    // Handle start/end times (could be date or dateTime)
    if (event.start) {
      eventData.start = event.start.dateTime || event.start.date;
      eventData.startTimeZone = event.start.timeZone;
    }

    if (event.end) {
      eventData.end = event.end.dateTime || event.end.date;
      eventData.endTimeZone = event.end.timeZone;
    }

    // Include attendees if present
    if (event.attendees && event.attendees.length > 0) {
      eventData.attendees = event.attendees.map(att => ({
        email: att.email,
        displayName: att.displayName,
        responseStatus: att.responseStatus,
        organizer: att.organizer || false
      }));
    }

    // Include organizer
    if (event.organizer) {
      eventData.organizer = {
        email: event.organizer.email,
        displayName: event.organizer.displayName,
        self: event.organizer.self || false
      };
    }

    // Include recurrence if present
    if (event.recurrence) {
      eventData.recurrence = event.recurrence;
    }

    return eventData;
  });

  return {
    success: true,
    count: events.length,
    calendar: args.calendar || 'primary',
    timeMin: params.timeMin,
    timeMax: params.timeMax,
    events
  };
}

// Main execution
const args = minimist(process.argv.slice(2));
const startTime = Date.now();

listEvents(args)
  .then(result => {
    // Log the action
    logAction('list_events', args, result, {
      account: args.account || 'default',
      duration: Date.now() - startTime
    });

    console.log(JSON.stringify(result, null, 2));
  })
  .catch(error => {
    const errorResult = {
      success: false,
      error: error.message
    };

    // Log the failed action
    logAction('list_events', args, errorResult, {
      account: args.account || 'default',
      duration: Date.now() - startTime,
      failed: true
    });

    console.error(JSON.stringify(errorResult, null, 2));
    process.exit(1);
  });
