#!/usr/bin/env node

import { google } from 'googleapis';
import minimist from 'minimist';
import { getAuthClient, parseAccountArg } from './auth/auth-utils.js';
import { logAction } from './action-logger.js';

async function deleteEvent(args) {
  // Validate required args
  if (!args.id && !args.eventId) {
    throw new Error('Missing required argument: --id or --eventId (event ID)');
  }

  // Get authenticated client for specified account (or default)
  const accountId = parseAccountArg(args);
  const oauth2Client = await getAuthClient(accountId);

  const calendar = google.calendar({ version: 'v3', auth: oauth2Client });

  const eventId = args.id || args.eventId;
  const calendarId = args.calendar || 'primary';

  // Optionally get event details before deleting
  let eventSummary = null;
  if (!args.skipConfirm) {
    try {
      const event = await calendar.events.get({ calendarId, eventId });
      eventSummary = event.data.summary;
    } catch (error) {
      // Event might not exist, proceed with delete anyway
    }
  }

  // Delete event
  await calendar.events.delete({
    calendarId,
    eventId,
    sendUpdates: args.sendUpdates || 'none' // none, all, externalOnly
  });

  return {
    success: true,
    eventId,
    deleted: true,
    eventSummary,
    message: 'Event deleted successfully'
  };
}

// Main execution
const args = minimist(process.argv.slice(2));
const startTime = Date.now();

deleteEvent(args)
  .then(result => {
    // Log the action
    logAction('delete_event', args, result, {
      account: args.account || 'default',
      duration: Date.now() - startTime
    });

    console.log(JSON.stringify(result, null, 2));
  })
  .catch(error => {
    const errorResult = {
      success: false,
      error: error.message
    };

    // Log the failed action
    logAction('delete_event', args, errorResult, {
      account: args.account || 'default',
      duration: Date.now() - startTime,
      failed: true
    });

    console.error(JSON.stringify(errorResult, null, 2));
    process.exit(1);
  });
