---
name: gmail-skill
description: Manage Gmail - send, read, search emails, manage labels and drafts. Use when user wants to interact with their Gmail account for email operations.
---

# Gmail Manager Skill

This skill provides comprehensive Gmail integration through lightweight CLI scripts. All operations are token-efficient and composable.

## First-Time Setup

Before using this skill, you must set up OAuth authentication:

1. **Install dependencies:**
   ```bash
   cd ~/.claude/skills/gmail-skill && npm install
   ```

2. **Set up Google Cloud credentials:**
   - Follow the guide in `docs/google-cloud-setup.md`
   - Download `credentials.json` and save to `scripts/auth/credentials.json`

3. **Authenticate:**
   ```bash
   cd ~/.claude/skills/gmail-skill && npm run setup
   ```

This will open a browser for Google OAuth and save your token locally.

## Multi-Account Support

The Gmail skill supports multiple accounts (e.g., personal and work email):

### Add Additional Accounts

```bash
# Add a second account (from skill directory)
npm run setup -- --account work

# Add a third account
npm run setup -- --account personal
```

Each account needs separate OAuth authentication.

### Manage Accounts

```bash
# List all configured accounts
node scripts/manage-accounts.js --list

# Set default account (used when --account is not specified)
node scripts/manage-accounts.js --set-default work

# Remove an account
node scripts/manage-accounts.js --remove old-account
```

### Using Specific Accounts

All Gmail operations support the `--account` parameter:

```bash
# Send email from work account
node gmail-send.js --account work --to "user@example.com" --subject "..." --body "..."

# Send from personal (or omit --account to use default)
node gmail-send.js --account personal --to "friend@example.com" --subject "..." --body "..."

# Search work emails
node gmail-search.js --account work --query "is:unread"
```

If `--account` is not specified, the default account is used.

## Usage Guidelines

### 1. Read Documentation On-Demand

When first using Gmail operations, read the comprehensive README:
```bash
cat ~/.claude/skills/gmail-skill/README.md
```

This provides detailed usage examples for all operations.

### 2. Execute Scripts via Bash

All scripts are in the `scripts/` directory and output JSON for easy parsing:

```bash
cd ~/.claude/skills/gmail-skill/scripts
```

### 3. Parse JSON Output

All scripts return JSON. Parse the output and present relevant information to the user in a friendly format.

### 4. Chain Operations

Save intermediate results to files when chaining operations:

```bash
# Search for emails
node gmail-search.js --query "from:user@example.com" > /tmp/search-results.json

# Read first message from results
MESSAGE_ID=$(cat /tmp/search-results.json | jq -r '.messages[0].id')
node gmail-read.js --id "$MESSAGE_ID"
```

## Available Operations

### Send Email
```bash
node gmail-send.js --to "user@example.com" --subject "Subject" --body "Body text"
```

### Search Emails
```bash
node gmail-search.js --query "is:unread" --limit 10
```

### Read Message
```bash
node gmail-read.js --id "MESSAGE_ID"
```

### Manage Labels
```bash
# List all labels
node gmail-labels.js --action list

# Add label to message
node gmail-labels.js --action add --id "MESSAGE_ID" --label "Important"
```

### Manage Drafts
```bash
# Create draft
node gmail-drafts.js --action create --to "user@example.com" --subject "Subject" --body "Draft body"

# Send draft
node gmail-drafts.js --action send --id "DRAFT_ID"
```

## Error Handling

If scripts fail:
- Check that `token.json` exists in `scripts/auth/`
- If token is expired, run `npm run setup` again
- Verify the user granted all required Gmail API permissions

## Best Practices

1. **Always change to the scripts directory first** to ensure relative paths work
2. **Parse JSON output** and present user-friendly summaries
3. **Validate user input** before passing to scripts
4. **Handle errors gracefully** and provide helpful error messages
5. **Use search operators** for precise email filtering (see README.md)

## Token Efficiency

This skill is designed for minimal token usage:
- Documentation loaded only when needed
- Scripts are small and focused
- JSON output is compact and parseable
- No persistent server overhead
