#!/usr/bin/env python3
"""
Recalculate Excel formulas with LibreOffice and report Excel-style errors.
"""

import json
import os
import platform
import subprocess
import sys
from pathlib import Path

from openpyxl import load_workbook


def setup_libreoffice_macro() -> bool:
    """Install a lightweight macro that recalculates and saves workbook."""
    if platform.system() == "Darwin":
        macro_dir = os.path.expanduser("~/Library/Application Support/LibreOffice/4/user/basic/Standard")
    else:
        macro_dir = os.path.expanduser("~/.config/libreoffice/4/user/basic/Standard")

    macro_file = os.path.join(macro_dir, "Module1.xba")
    if os.path.exists(macro_file):
        with open(macro_file, "r", encoding="utf-8") as f:
            if "RecalculateAndSave" in f.read():
                return True

    if not os.path.exists(macro_dir):
        subprocess.run(["soffice", "--headless", "--terminate_after_init"], capture_output=True, timeout=10)
        os.makedirs(macro_dir, exist_ok=True)

    macro_content = """<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE script:module PUBLIC "-//OpenOffice.org//DTD OfficeDocument 1.0//EN" "module.dtd">
<script:module xmlns:script="http://openoffice.org/2000/script" script:name="Module1" script:language="StarBasic">
    Sub RecalculateAndSave()
      ThisComponent.calculateAll()
      ThisComponent.store()
      ThisComponent.close(True)
    End Sub
</script:module>"""

    try:
        with open(macro_file, "w", encoding="utf-8") as f:
            f.write(macro_content)
        return True
    except Exception:
        return False


def recalc(filename: str, timeout: int = 30) -> dict:
    """Recalculate formulas and scan all cells for spreadsheet errors."""
    if not Path(filename).exists():
        return {"error": f"File not found: {filename}"}

    if not setup_libreoffice_macro():
        return {"error": "Failed to setup LibreOffice macro"}

    abs_path = str(Path(filename).absolute())
    cmd = [
        "soffice",
        "--headless",
        "--norestore",
        "vnd.sun.star.script:Standard.Module1.RecalculateAndSave?language=Basic&location=application",
        abs_path,
    ]

    if platform.system() == "Linux":
        cmd = ["timeout", str(timeout), *cmd]
    elif platform.system() == "Darwin":
        try:
            subprocess.run(["gtimeout", "--version"], capture_output=True, timeout=1, check=False)
            cmd = ["gtimeout", str(timeout), *cmd]
        except (FileNotFoundError, subprocess.TimeoutExpired):
            pass

    result = subprocess.run(cmd, capture_output=True, text=True)
    if result.returncode not in (0, 124):
        stderr = result.stderr or "Unknown LibreOffice error"
        if "Module1" in stderr or "RecalculateAndSave" not in stderr:
            return {"error": "LibreOffice macro not configured properly"}
        return {"error": stderr}

    try:
        wb_values = load_workbook(filename, data_only=True)
        wb_formulas = load_workbook(filename, data_only=False)

        excel_errors = ["#VALUE!", "#DIV/0!", "#REF!", "#NAME?", "#NULL!", "#NUM!", "#N/A"]
        error_details = {err: [] for err in excel_errors}
        total_errors = 0
        total_formulas = 0

        for sheet_name in wb_values.sheetnames:
            ws_val = wb_values[sheet_name]
            ws_formula = wb_formulas[sheet_name]

            for row in ws_val.iter_rows():
                for cell in row:
                    if isinstance(cell.value, str):
                        for err in excel_errors:
                            if err in cell.value:
                                error_details[err].append(f"{sheet_name}!{cell.coordinate}")
                                total_errors += 1
                                break

            for row in ws_formula.iter_rows():
                for cell in row:
                    if isinstance(cell.value, str) and cell.value.startswith("="):
                        total_formulas += 1

        wb_values.close()
        wb_formulas.close()

        response = {
            "status": "success" if total_errors == 0 else "errors_found",
            "total_errors": total_errors,
            "total_formulas": total_formulas,
            "error_summary": {},
        }
        for err, locs in error_details.items():
            if locs:
                response["error_summary"][err] = {
                    "count": len(locs),
                    "locations": locs[:20],
                }
        return response
    except Exception as exc:
        return {"error": str(exc)}


def main() -> None:
    if len(sys.argv) < 2:
        print("Usage: python recalc.py <excel_file> [timeout_seconds]")
        sys.exit(1)

    filename = sys.argv[1]
    timeout = int(sys.argv[2]) if len(sys.argv) > 2 else 30
    print(json.dumps(recalc(filename, timeout), indent=2))


if __name__ == "__main__":
    main()
