#!/usr/bin/env python3
"""
Speech-to-text transcription using local Whisper model.
No API key required - runs entirely locally.

Usage:
    python transcribe.py video.mp4 -o transcript.txt
    python transcribe.py video.mp4 -o transcript.txt --model tiny
"""

import argparse
import json
import os
import subprocess
import sys
import tempfile

# Import whisper (local model)
import whisper


def extract_audio(video_path: str, output_path: str) -> None:
    """Extract audio from video file as WAV for Whisper."""
    cmd = [
        "ffmpeg",
        "-i",
        video_path,
        "-vn",  # No video
        "-acodec",
        "pcm_s16le",  # WAV format
        "-ar",
        "16000",  # 16kHz sample rate
        "-ac",
        "1",  # Mono
        output_path,
        "-y",  # Overwrite
        "-loglevel",
        "error",
    ]
    subprocess.run(cmd, check=True)


def transcribe_with_local_whisper(audio_path: str, model_name: str = "base") -> list[dict]:
    """Transcribe audio using local Whisper model."""
    print(f"Loading Whisper model '{model_name}'...", file=sys.stderr)
    model = whisper.load_model(model_name)

    print("Transcribing (this may take a few minutes)...", file=sys.stderr)
    result = model.transcribe(audio_path, verbose=False)

    # Convert to structured segments
    segments = []
    for segment in result["segments"]:
        segments.append({"start": segment["start"], "end": segment["end"], "text": segment["text"].strip()})

    return segments


def format_as_text(segments: list[dict]) -> str:
    """Format segments as timestamped text."""
    lines = []
    for seg in segments:
        lines.append(f"[{seg['start']:.1f}s - {seg['end']:.1f}s] {seg['text']}")
    return "\n".join(lines)


def main():
    parser = argparse.ArgumentParser(description="Transcribe video/audio to text (local Whisper)")
    parser.add_argument("input", help="Input video or audio file")
    parser.add_argument("-o", "--output", required=True, help="Output transcript file")
    parser.add_argument("--model", default="base", choices=["tiny", "base", "small", "medium"], help="Whisper model size (default: base)")
    parser.add_argument("--json", action="store_true", help="Output as JSON")
    args = parser.parse_args()

    # Check if input is video or audio
    video_extensions = (".mp4", ".mkv", ".avi", ".mov", ".webm")
    audio_extensions = (".mp3", ".wav", ".flac", ".m4a", ".ogg")

    if args.input.lower().endswith(video_extensions):
        # Extract audio first
        print(f"Extracting audio from {args.input}...", file=sys.stderr)
        audio_path = tempfile.mktemp(suffix=".wav")
        extract_audio(args.input, audio_path)
        cleanup_audio = True
    elif args.input.lower().endswith(audio_extensions):
        audio_path = args.input
        cleanup_audio = False
    else:
        print(f"Unknown file type: {args.input}", file=sys.stderr)
        sys.exit(1)

    try:
        # Transcribe with local model
        segments = transcribe_with_local_whisper(audio_path, args.model)
        print(f"Transcribed {len(segments)} segments", file=sys.stderr)

        # Output
        if args.json:
            with open(args.output, "w") as f:
                json.dump({"segments": segments}, f, indent=2)
        else:
            with open(args.output, "w") as f:
                f.write(format_as_text(segments))

        print(f"Saved to {args.output}", file=sys.stderr)

    finally:
        if cleanup_audio and os.path.exists(audio_path):
            os.remove(audio_path)


if __name__ == "__main__":
    main()
