"""
Feature Engineering Skill - Comprehensive Feature Engineering Framework

This module provides a modular, strategy-based approach to feature engineering
for demographic/survey data and general tabular datasets. It supports numerical
scaling, categorical encoding, feature selection, and pipeline orchestration.

Key Features:
- Multiple scaling methods (standard, minmax, robust)
- Categorical encoding (one-hot, label, frequency, hash)
- Polynomial and interaction features
- Binary feature conversion
- Feature selection (variance, correlation)
- Correlation analysis and visualization

Usage example:
    from feature_engineering import FeatureEngineeringStrategies, FeatureEngineeringPipeline

    pipeline = FeatureEngineeringPipeline(name="<describe your feature set>")
    pipeline.add_step(
        FeatureEngineeringStrategies.convert_to_binary,
        columns=['gender', 'is_subscribed']
    )
    pipeline.add_step(
        FeatureEngineeringStrategies.scale_numerical,
        columns=['height', 'household_income'],
        method='standard'
    )
    df_engineered = pipeline.execute(raw_df)
"""

import pandas as pd
import numpy as np
from typing import Callable, Optional, List, Dict, Any, Union
import hashlib
import warnings
warnings.filterwarnings('ignore')

from sklearn.preprocessing import (
    StandardScaler, MinMaxScaler, RobustScaler,
    LabelEncoder, OneHotEncoder,
    PolynomialFeatures, KBinsDiscretizer
)
from sklearn.feature_selection import (
    SelectKBest, f_classif, f_regression,
    VarianceThreshold, mutual_info_classif, mutual_info_regression
)


class FeatureEngineeringStrategies:
    """Collection of modular feature engineering strategies"""

    # ==================== NUMERICAL FEATURE ENGINEERING ====================

    @staticmethod
    def scale_numerical(df: pd.DataFrame, columns: List[str],
                        method: str = 'standard') -> pd.DataFrame:
        """
        Scale numerical features using various methods.

        Args:
            df: Input dataframe
            columns: Numerical columns to scale
            method: Scaling method
                - 'standard': StandardScaler (mean=0, std=1)
                - 'minmax': MinMaxScaler (range 0-1)
                - 'robust': RobustScaler (median=0, IQR-based, robust to outliers)

        Returns:
            DataFrame with scaled columns
        """
        df = df.copy()

        if method == 'standard':
            scaler = StandardScaler()
        elif method == 'minmax':
            scaler = MinMaxScaler()
        elif method == 'robust':
            scaler = RobustScaler()
        else:
            raise ValueError(f"Unknown scaling method: {method}")

        for col in columns:
            if col in df.columns:
                df[col] = scaler.fit_transform(df[[col]])

        return df

    @staticmethod
    def validate_numeric_features(df: pd.DataFrame, exclude_cols: List[str] = None) -> pd.DataFrame:
        """
        Validate that all feature columns are numeric types.

        This is critical before performing numerical operations like scaling,
        variance analysis, or regression modeling. Non-numeric columns will
        cause errors in downstream analysis.

        Args:
            df: Input dataframe to validate
            exclude_cols: Column names to exclude from validation (e.g., IDs)

        Returns:
            Original dataframe if validation passes

        Raises:
            ValueError: If any non-excluded columns are non-numeric

        Example:
            >>> df = validate_numeric_features(df, exclude_cols=['user_id'])
        """
        if exclude_cols is None:
            exclude_cols = []

        # Check each column that's not in exclude list
        non_numeric_cols = []
        for col in df.columns:
            if col not in exclude_cols:
                if not pd.api.types.is_numeric_dtype(df[col]):
                    non_numeric_cols.append(f"{col} ({df[col].dtype})")

        if non_numeric_cols:
            raise ValueError(
                f"Found non-numeric columns that must be encoded or removed:\n"
                f"  {', '.join(non_numeric_cols)}\n"
                f"Use encode_categorical() or encode_target() to convert categorical columns."
            )

        return df

    @staticmethod
    def validate_no_constants(df: pd.DataFrame, exclude_cols: List[str] = None) -> pd.DataFrame:
        """
        Check for and remove constant columns (all values identical).

        Constant features provide no information for modeling and can cause
        issues in statistical analysis. This validation should be run after
        feature engineering and before model training.

        Args:
            df: Input dataframe
            exclude_cols: Column names to exclude from check (e.g., IDs)

        Returns:
            DataFrame with constant columns removed

        Example:
            >>> df = validate_no_constants(df, exclude_cols=['user_id'])
        """
        if exclude_cols is None:
            exclude_cols = []

        df = df.copy()
        cols_to_check = [c for c in df.columns if c not in exclude_cols]

        # Identify constant columns (nunique <= 1)
        constant_cols = []
        for col in cols_to_check:
            if df[col].nunique() <= 1:
                constant_cols.append(col)

        if constant_cols:
            print(f"WARNING: Removing constant columns with no variance: {constant_cols}")
            df = df.drop(columns=constant_cols)

        return df

    @staticmethod
    def create_bins(df: pd.DataFrame, columns: List[str],
                    n_bins: int = 5, strategy: str = 'quantile') -> pd.DataFrame:
        """
        Create binned/discretized features from continuous numerical features.

        Args:
            df: Input dataframe
            columns: Numerical columns to bin
            n_bins: Number of bins
            strategy: Binning strategy
                - 'uniform': Equal width bins
                - 'quantile': Equal frequency bins
                - 'kmeans': K-means clustering

        Returns:
            DataFrame with new binned columns (original columns preserved)
        """
        df = df.copy()
        discretizer = KBinsDiscretizer(n_bins=n_bins, encode='ordinal', strategy=strategy)

        for col in columns:
            if col in df.columns:
                binned_col = f"{col}_binned"
                df[binned_col] = discretizer.fit_transform(df[[col]]).astype(int)

        return df

    @staticmethod
    def create_polynomial_features(df: pd.DataFrame, columns: List[str],
                                   degree: int = 2,
                                   include_bias: bool = False) -> pd.DataFrame:
        """
        Create polynomial and interaction features.

        Args:
            df: Input dataframe
            columns: Numerical columns for polynomial features
            degree: Polynomial degree (2 = quadratic, 3 = cubic)
            include_bias: Whether to include bias column (all 1s)

        Returns:
            DataFrame with polynomial feature columns added

        Example:
            Input: [x1, x2]
            degree=2 output adds: x1^2, x1*x2, x2^2
        """
        df = df.copy()
        poly = PolynomialFeatures(degree=degree, include_bias=include_bias)

        feature_cols = [c for c in columns if c in df.columns]
        if not feature_cols:
            return df

        poly_features = poly.fit_transform(df[feature_cols])
        poly_feature_names = poly.get_feature_names_out(feature_cols)

        for i, name in enumerate(poly_feature_names):
            if name not in feature_cols:
                df[f"poly_{name}"] = poly_features[:, i]

        return df

    @staticmethod
    def create_interaction_features(df: pd.DataFrame,
                                    column_pairs: List[tuple]) -> pd.DataFrame:
        """
        Create interaction features (multiplication) between column pairs.

        Args:
            df: Input dataframe
            column_pairs: List of tuples [(col1, col2), ...]

        Returns:
            DataFrame with interaction columns added

        Example:
            column_pairs = [('price', 'quantity'), ('age', 'income')]
            Creates: price_x_quantity, age_x_income
        """
        df = df.copy()

        for col1, col2 in column_pairs:
            if col1 in df.columns and col2 in df.columns:
                interaction_name = f"{col1}_x_{col2}"
                df[interaction_name] = df[col1] * df[col2]

        return df

    @staticmethod
    def create_log_features(df: pd.DataFrame, columns: List[str]) -> pd.DataFrame:
        """
        Create log-transformed features (useful for skewed distributions).

        Args:
            df: Input dataframe
            columns: Numerical columns to log-transform

        Returns:
            DataFrame with log-transformed columns added
        """
        df = df.copy()

        for col in columns:
            if col in df.columns:
                df[f"{col}_log"] = np.log1p(df[col].clip(lower=0))

        return df

    # ==================== CATEGORICAL FEATURE ENGINEERING ====================

    @staticmethod
    def encode_categorical(df: pd.DataFrame, columns: List[str],
                           method: str = 'onehot',
                           max_categories: int = 50,
                           fallback: str = 'label') -> pd.DataFrame:
        """
        Encode categorical features using various methods.

        Args:
            df: Input dataframe
            columns: Categorical columns to encode
            method: Encoding method
                - 'onehot': One-hot encoding (binary columns for each category)
                - 'label': Label encoding (integers 0, 1, 2, ...)
                - 'frequency': Frequency encoding (replace with category frequency)
                - 'hash': Hash encoding (for high-cardinality features)
            max_categories: Maximum categories for one-hot encoding
            fallback: Fallback method if categories exceed max ('label' or 'hash')

        Returns:
            DataFrame with encoded columns
        """
        df = df.copy()

        for col in columns:
            if col not in df.columns:
                continue

            if method == 'onehot':
                n_categories = df[col].nunique()

                if n_categories <= max_categories:
                    dummies = pd.get_dummies(df[col], prefix=col, drop_first=True)
                    # Convert bool to int (0/1) for downstream analysis
                    dummies = dummies.astype(int)
                    df = pd.concat([df, dummies], axis=1)
                    df = df.drop(columns=[col])
                else:
                    print(f"Warning: {col} has {n_categories} categories, using {fallback} encoding")
                    if fallback == 'hash':
                        n_bins = min(100, max(10, int(np.sqrt(n_categories))))
                        df[f"{col}_hash"] = df[col].astype(str).apply(
                            lambda x: int(hashlib.md5(x.encode()).hexdigest(), 16) % n_bins
                        )
                        df = df.drop(columns=[col])
                    else:
                        le = LabelEncoder()
                        df[col] = le.fit_transform(df[col].astype(str))

            elif method == 'label':
                le = LabelEncoder()
                df[col] = le.fit_transform(df[col].astype(str))

            elif method == 'frequency':
                freq_map = df[col].value_counts(normalize=True).to_dict()
                df[f"{col}_freq"] = df[col].map(freq_map)

            elif method == 'hash':
                n_categories = df[col].nunique()
                n_bins = min(100, max(10, int(np.sqrt(n_categories))))
                df[f"{col}_hash"] = df[col].astype(str).apply(
                    lambda x: int(hashlib.md5(x.encode()).hexdigest(), 16) % n_bins
                )
                df = df.drop(columns=[col])

        return df

    @staticmethod
    def create_category_aggregations(df: pd.DataFrame,
                                     categorical_col: str,
                                     numerical_cols: List[str],
                                     agg_funcs: List[str] = ['mean', 'std', 'count']) -> pd.DataFrame:
        """
        Create aggregated features by grouping categorical column.

        Args:
            df: Input dataframe
            categorical_col: Categorical column to group by
            numerical_cols: Numerical columns to aggregate
            agg_funcs: Aggregation functions ['mean', 'std', 'sum', 'min', 'max', 'count']

        Returns:
            DataFrame with aggregated feature columns added

        Example:
            Group by 'Category', calculate mean/std of 'Price'
            Creates: Category_Price_mean, Category_Price_std
        """
        df = df.copy()

        if categorical_col not in df.columns:
            return df

        for num_col in numerical_cols:
            if num_col in df.columns:
                for agg_func in agg_funcs:
                    agg_map = df.groupby(categorical_col)[num_col].agg(agg_func).to_dict()
                    feature_name = f"{categorical_col}_{num_col}_{agg_func}"
                    df[feature_name] = df[categorical_col].map(agg_map)

        return df

    # ==================== BINARY FEATURE ENGINEERING ====================

    @staticmethod
    def convert_to_binary(df: pd.DataFrame, columns: List[str]) -> pd.DataFrame:
        """
        Convert binary features to 0/1 encoding.

        Args:
            df: Input dataframe
            columns: Binary columns to convert

        Returns:
            DataFrame with binary columns converted to 0/1

        Handles common binary formats:
            - Yes/No → 1/0
            - True/False → 1/0
            - Male/Female → 1/0
            - Any two-value categorical
        """
        df = df.copy()

        for col in columns:
            if col in df.columns:
                unique_values = df[col].dropna().unique()

                if set(unique_values) <= {0, 1, 0.0, 1.0}:
                    df[col] = df[col].astype(float)
                    continue

                if len(unique_values) == 2:
                    le = LabelEncoder()
                    non_null_mask = df[col].notna()
                    if non_null_mask.sum() > 0:
                        df.loc[non_null_mask, col] = le.fit_transform(
                            df.loc[non_null_mask, col].astype(str)
                        )
                        df[col] = df[col].astype(float)

        return df

    # ==================== FEATURE SELECTION ====================

    @staticmethod
    def select_features_variance(df: pd.DataFrame,
                                 columns: List[str] = None,
                                 threshold: float = 0.0) -> pd.DataFrame:
        """
        Remove low-variance features (constant or near-constant features).

        Args:
            df: Input dataframe
            columns: Columns to check variance (if None, auto-select numerical)
            threshold: Variance threshold (features below this are removed)

        Returns:
            DataFrame with low-variance columns removed
        """
        df = df.copy()

        if columns is None or not columns:
            feature_cols = df.select_dtypes(include=[np.number, 'bool']).columns.tolist()
        else:
            feature_cols = [c for c in columns if c in df.columns and pd.api.types.is_numeric_dtype(df[c])]

        if not feature_cols:
            return df

        selector = VarianceThreshold(threshold=threshold)
        selector.fit(df[feature_cols])

        cols_to_keep = [feature_cols[i] for i in range(len(feature_cols)) if selector.get_support()[i]]
        cols_to_drop = [c for c in feature_cols if c not in cols_to_keep]

        if cols_to_drop:
            print(f"Removing low-variance features: {cols_to_drop}")
            df = df.drop(columns=cols_to_drop)

        return df

    @staticmethod
    def select_features_correlation(df: pd.DataFrame,
                                    columns: List[str],
                                    threshold: float = 0.95) -> pd.DataFrame:
        """
        Remove highly correlated features (reduces multicollinearity).

        Args:
            df: Input dataframe
            columns: Columns to check correlation
            threshold: Correlation threshold (if correlation > threshold, drop one)

        Returns:
            DataFrame with highly correlated columns removed
        """
        df = df.copy()

        feature_cols = [c for c in columns if c in df.columns]
        if len(feature_cols) < 2:
            return df

        corr_matrix = df[feature_cols].corr().abs()
        upper_tri = corr_matrix.where(np.triu(np.ones(corr_matrix.shape), k=1).astype(bool))

        cols_to_drop = [column for column in upper_tri.columns if any(upper_tri[column] > threshold)]

        if cols_to_drop:
            print(f"Removing highly correlated features: {cols_to_drop}")
            df = df.drop(columns=cols_to_drop)

        return df


class FeatureEngineeringPipeline:
    """
    Modular feature engineering pipeline.

    Usage:
        pipeline = FeatureEngineeringPipeline(name="Customer Features")
        pipeline.add_step(
            strategy=FeatureEngineeringStrategies.scale_numerical,
            columns=['age', 'income'],
            method='standard',
            description="Standardize numerical features"
        ).add_step(
            strategy=FeatureEngineeringStrategies.encode_categorical,
            columns=['category', 'region'],
            method='onehot',
            description="One-hot encode categorical features"
        )
        df_engineered = pipeline.execute(df)
    """

    def __init__(self, name: str = "Features"):
        self.name = name
        self.steps = []
        self.execution_log = []

    def add_step(self, strategy: Callable, description: str = "", **kwargs) -> 'FeatureEngineeringPipeline':
        """
        Add a feature engineering step to the pipeline.

        Args:
            strategy: Feature engineering function from FeatureEngineeringStrategies
            description: Human-readable description of the step
            **kwargs: Parameters to pass to the strategy function

        Returns:
            Self for method chaining
        """
        self.steps.append({
            'strategy': strategy,
            'description': description or strategy.__name__,
            'kwargs': kwargs
        })
        return self

    def execute(self, df: pd.DataFrame, verbose: bool = True) -> pd.DataFrame:
        """
        Execute all feature engineering steps in the pipeline.

        Args:
            df: Input dataframe
            verbose: Print progress information

        Returns:
            DataFrame with both original and engineered columns

        Note:
            - Access engineered feature names via pipeline.get_engineered_features()
            - Access execution log via pipeline.get_log()
        """
        df_complete = df.copy()
        original_columns = set(df.columns)
        self.execution_log = []
        self.engineered_features = []  # Track new/modified features

        if verbose:
            print("=" * 80)
            print(f"EXECUTING {self.name.upper()} FEATURE ENGINEERING PIPELINE")
            print("=" * 80)

        for i, step in enumerate(self.steps, 1):
            strategy = step['strategy']
            description = step['description']
            kwargs = step['kwargs']

            cols_before = set(df_complete.columns)

            try:
                df_complete = strategy(df_complete, **kwargs)

                cols_after = set(df_complete.columns)
                new_cols = cols_after - cols_before
                removed_cols = cols_before - cols_after

                # Track engineered features (new or modified columns)
                self.engineered_features.extend(list(new_cols))

                log_entry = {
                    'step': i,
                    'description': description,
                    'cols_before': len(cols_before),
                    'cols_after': len(cols_after),
                    'cols_added': len(new_cols),
                    'cols_removed': len(removed_cols),
                    'new_columns': list(new_cols),
                    'status': 'success'
                }
                self.execution_log.append(log_entry)

                if verbose:
                    print(f"\n[Step {i}/{len(self.steps)}] {description}")
                    print(f"  Columns: {len(cols_before):,} -> {len(cols_after):,} (added: {len(new_cols):+,}, removed: {len(removed_cols)})")
                    if new_cols:
                        print(f"  New features: {', '.join(list(new_cols)[:5])}" +
                              (f" ... (+{len(new_cols)-5} more)" if len(new_cols) > 5 else ""))

            except Exception as e:
                log_entry = {
                    'step': i,
                    'description': description,
                    'error': str(e),
                    'status': 'failed'
                }
                self.execution_log.append(log_entry)

                if verbose:
                    print(f"\n[Step {i}/{len(self.steps)}] {description}")
                    print(f"  ERROR: {str(e)}")

        # Identify all engineered features (including modified original columns)
        all_engineered = list(set(df_complete.columns) - original_columns)
        # Also include original columns that were modified (e.g., scaled, encoded)
        modified_originals = [col for col in original_columns
                             if col in df_complete.columns and
                             not df_complete[col].equals(df[[col]].iloc[:, 0])]
        all_engineered.extend(modified_originals)

        # Update self.engineered_features to reflect final state (after drops)
        self.engineered_features = all_engineered

        if verbose:
            print("\n" + "=" * 80)
            print(f"Pipeline completed: {len(df_complete.columns)} total columns")
            print(f"  Original columns: {len(original_columns)}")
            print(f"  New features: {len(all_engineered)}")
            print("=" * 80)

            # Data quality check: verify all features are numeric
            print("\nDATA QUALITY CHECK:")
            non_numeric_cols = df_complete.select_dtypes(exclude=['number']).columns.tolist()
            if non_numeric_cols:
                print(f"  ⚠️  WARNING: {len(non_numeric_cols)} non-numeric columns found:")
                for col in non_numeric_cols[:10]:  # Show first 10
                    dtype = df_complete[col].dtype
                    sample = df_complete[col].iloc[0] if len(df_complete) > 0 else None
                    print(f"      - {col}: {dtype} (sample: {sample})")
                if len(non_numeric_cols) > 10:
                    print(f"      ... and {len(non_numeric_cols) - 10} more")
                print(f"  ❌ CRITICAL: ALL features must be numeric for downstream analysis!")
                print(f"  💡 TIP: Use encode_categorical() on these columns")
            else:
                print(f"  ✓ All {len(df_complete.columns)} features are numeric")

        return df_complete

    def get_log(self) -> pd.DataFrame:
        """Get execution log as DataFrame"""
        return pd.DataFrame(self.execution_log)

    def get_engineered_features(self) -> List[str]:
        """
        Get list of engineered feature names from last execution.

        Returns:
            List of column names that were created or modified during pipeline execution

        Usage:
            df = pipeline.execute(raw_df)
            feature_names = pipeline.get_engineered_features()
            df_features_only = df[feature_names]
        """
        return self.engineered_features
