"""Utility for searching attractions by city."""

from __future__ import annotations

import argparse
from collections.abc import Callable
from pathlib import Path

import pandas as pd
from pandas import DataFrame


# Default to the bundled attractions CSV.
def _find_data_path() -> Path:
    """Find data file, checking container path first, then relative to script."""
    relative = "attractions/attractions.csv"
    container_path = Path("/app/data") / relative
    if container_path.exists():
        return container_path
    return Path(__file__).resolve().parent.parent.parent / "data" / relative


DEFAULT_DATA_PATH = _find_data_path()

ATTRACTION_COLUMNS = [
    "Name",
    "Latitude",
    "Longitude",
    "Address",
    "Phone",
    "Website",
    "City",
]


class Attractions:
    """Search helper for the attractions dataset."""

    def __init__(
        self,
        path: str | Path = DEFAULT_DATA_PATH,
        city_normalizer: Callable[[str], str] | None = None,
    ) -> None:
        self.path = Path(path)
        self.city_normalizer = city_normalizer or (lambda value: value)
        self.data: DataFrame = DataFrame()
        self.load_db()
        print("Attractions loaded.")

    def load_db(self) -> None:
        """Load and lightly clean the attractions CSV."""
        if not self.path.exists():
            raise FileNotFoundError(f"Attractions CSV not found at {self.path}")

        df = pd.read_csv(self.path)
        existing_columns = [col for col in ATTRACTION_COLUMNS if col in df.columns]
        df = df[existing_columns].dropna()
        df["City"] = df["City"].astype(str).str.strip()
        self.data = df

    def run(self, city: str) -> DataFrame | str:
        """Return attractions for the given city (case-insensitive)."""
        if self.data.empty:
            return "No attractions data is available."

        normalized_city = self.city_normalizer(city).strip()
        mask = self.data["City"].str.lower() == normalized_city.lower()
        results = self.data[mask].reset_index(drop=True)

        if results.empty:
            return "There is no attraction in this city."

        return results


def _build_arg_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(description="Search attractions by city.")
    parser.add_argument("--city", "-c", help="City to search for.")
    parser.add_argument(
        "--path",
        default=str(DEFAULT_DATA_PATH),
        help="Path to attractions CSV (defaults to bundled dataset).",
    )
    return parser


def main() -> None:
    args = _build_arg_parser().parse_args()
    attractions = Attractions(path=args.path)

    if not args.city:
        print("Please provide --city.")
        return

    result = attractions.run(args.city)
    if isinstance(result, str):
        print(result)
    else:
        # Print in a compact, readable format without the pandas index.
        print(result.to_string(index=False))


if __name__ == "__main__":
    main()
