import fs from 'fs';
import path from 'path';
import * as THREE from 'three';
import { OBJExporter } from 'three/examples/jsm/exporters/OBJExporter.js';
import { mergeGeometries } from 'three/examples/jsm/utils/BufferGeometryUtils.js';
import { pathToFileURL } from 'url';

const args = process.argv.slice(2);
const getArg = (flag, fallback = null) => {
    const idx = args.indexOf(flag);
    if (idx === -1 || idx + 1 >= args.length) return fallback;
    return args[idx + 1];
};

const input = getArg('--input');
const outDir = getArg('--out-dir');
const includeRoot = args.includes('--include-root');

if (!input || !outDir) {
    console.error('Usage: node export_link_objs.mjs --input <file.js> --out-dir <dir> [--include-root]');
    process.exit(2);
}

fs.mkdirSync(outDir, { recursive: true });

const sceneModuleURL = pathToFileURL(input).href;
const sceneModule = await import(sceneModuleURL);
const root = sceneModule.createScene();
root.updateMatrixWorld(true);

const linkMap = new Map();
root.traverse((obj) => {
    if (!obj.isGroup || !obj.name) return;
    if (!includeRoot && obj === root) return;
    linkMap.set(obj.name, obj);
});

const linkNames = Array.from(linkMap.keys()).sort();
const linkNameSet = new Set(linkNames);

const addGeometry = (geometries, geometry, matrix) => {
    let geom = geometry.clone();
    geom.applyMatrix4(matrix);
    if (!geom.attributes.normal) {
        geom.computeVertexNormals();
    }
    geometries.push(geom);
};

const collectGeometries = (linkObj) => {
    const geometries = [];
    const tempMatrix = new THREE.Matrix4();
    const instanceMatrix = new THREE.Matrix4();

    const visit = (obj) => {
        if (obj !== linkObj && obj.isGroup && linkNameSet.has(obj.name)) {
            return;
        }
        if (obj.isInstancedMesh) {
            const count = obj.count ?? obj.instanceCount ?? 0;
            for (let i = 0; i < count; i += 1) {
                obj.getMatrixAt(i, instanceMatrix);
                tempMatrix.copy(obj.matrixWorld).multiply(instanceMatrix);
                addGeometry(geometries, obj.geometry, tempMatrix);
            }
        } else if (obj.isMesh) {
            addGeometry(geometries, obj.geometry, obj.matrixWorld);
        }
        for (const child of obj.children) {
            visit(child);
        }
    };

    visit(linkObj);
    return geometries;
};

const exporter = new OBJExporter();
for (const name of linkNames) {
    const linkObj = linkMap.get(name);
    const geometries = collectGeometries(linkObj);
    if (!geometries.length) {
        continue;
    }
    const merged = mergeGeometries(geometries, false);
    const mesh = new THREE.Mesh(merged);
    const objData = exporter.parse(mesh);
    fs.writeFileSync(path.join(outDir, `${name}.obj`), objData);
}

console.log(`Wrote ${linkNames.length} link OBJ files to ${outDir}`);
