"""Step 2: Weighted Embedding Generation

This module generates semantic embeddings for category paths with hierarchical weighting.

Key Considerations:
1. **Hierarchical Weighting**: Different levels get different importance
   - Level 1 (root): 1.0
   - Level 2: 0.6
   - Level 3: 0.36
   - Level 4: 0.216
   - Level 5: 0.1296
   - Exponential decay (X^(n-1)) reflects decreasing importance of deeper levels, X means the discount factor applied from parent to child in terms of their hierarchically semantic importance.

2. **Actual Depth Usage**: Only encode levels that exist in each path
   - No padding for shallow paths (depth < N)
   - Avoids noise from empty/None values

3. **Weight Normalization**: Normalize weights per path
   - Ensures comparable embeddings across different depths
   - Sum of weights = 1.0 for each path

4. **Sentence Transformer**: Uses 'all-MiniLM-L6-v2' for semantic encoding
   - Fast and accurate
   - 384-dimensional embeddings
"""

import numpy as np
import pandas as pd
from sentence_transformers import SentenceTransformer
from tqdm import tqdm


def get_path_embedding_weighted(row, model, weights_dict, max_depth=5):
    """
    Generate weighted embedding for a hierarchical path.
    Uses only the actual depth of the path (no padding).

    Args:
        row: DataFrame row with 'depth' and 'source_level_1', 'source_level_2', ... columns
        model: SentenceTransformer model
        weights_dict: Dict mapping level number to weight
        max_depth: Maximum depth to consider

    Returns:
        Weighted embedding vector (numpy array)
    """
    actual_depth = int(row['depth'])

    # Collect level texts and their weights
    level_texts = []
    level_weights = []

    for i in range(1, actual_depth + 1):
        col = f'source_level_{i}'
        if pd.notna(row[col]) and str(row[col]).strip():
            level_texts.append(str(row[col]).strip())
            level_weights.append(weights_dict[i])
        else:
            level_texts.append('')
            level_weights.append(0.0)

    # Encode each level
    level_embeddings = model.encode(level_texts, show_progress_bar=False)

    # Normalize weights for this specific path
    level_weights = np.array(level_weights)
    if level_weights.sum() > 0:
        level_weights = level_weights / level_weights.sum()

    # Weighted sum
    weighted_embedding = np.sum(
        [emb * w for emb, w in zip(level_embeddings, level_weights)],
        axis=0
    )

    return weighted_embedding


def generate_embeddings(df, target_depth=5, weights=None, model_name='all-MiniLM-L6-v2'):
    """
    Generate weighted embeddings for all records in DataFrame.

    Args:
        df: DataFrame with depth and source_level_1, source_level_2, ... columns
        target_depth: Maximum depth (default 5)
        weights: Dict of {level: weight}
        model_name: SentenceTransformer model name (default 'all-MiniLM-L6-v2')

    Returns:
        numpy array of shape (len(df), embedding_dim)
    """
    if weights is None:
        # Default exponential decay: X^(n-1), default X=0.6 as below suggested
        weights = {1: 1.0, 2: 0.6, 3: 0.36, 4: 0.216, 5: 0.1296}

    print(f"Loading SentenceTransformer model: {model_name}")
    model = SentenceTransformer(model_name)
    print(f"Embedding dimension: {model.get_sentence_embedding_dimension()}")

    print(f"\nGenerating weighted embeddings for {len(df)} records...")
    print(f"Weights: {weights}")

    embeddings_list = []
    for _, row in tqdm(df.iterrows(), total=len(df), desc="Encoding"):
        emb = get_path_embedding_weighted(row, model, weights, target_depth)
        embeddings_list.append(emb)

    embeddings_matrix = np.array(embeddings_list)
    print(f"\n✅ Generated embeddings matrix: {embeddings_matrix.shape}")

    return embeddings_matrix


if __name__ == '__main__':
    print("Step 2: Weighted Embedding Generation")
    print("=" * 60)
    print("\nExample usage:")
    print("""
    from step2_weighted_embedding_generation import generate_embeddings

    # Assuming df has columns: depth, source_level_1, source_level_2, ..., source_level_5
    embeddings = generate_embeddings(df, target_depth=N)
    # Returns: numpy array of shape (len(df), 384)
    """)
