#!/usr/bin/env node

import { google } from 'googleapis';
import minimist from 'minimist';
import { getAuthClient, parseAccountArg } from './auth/auth-utils.js';
import { logAction } from './action-logger.js';

async function updateEventResponse(args) {
  if (!args.id) {
    throw new Error('Missing required argument: --id (event ID)');
  }

  if (!args.response) {
    throw new Error('Missing required argument: --response (accepted|declined|tentative)');
  }

  const validResponses = ['accepted', 'declined', 'tentative', 'needsAction'];
  if (!validResponses.includes(args.response)) {
    throw new Error(`Invalid response. Must be one of: ${validResponses.join(', ')}`);
  }

  // Get authenticated client for specified account (or default)
  const accountId = parseAccountArg(args);
  const oauth2Client = await getAuthClient(accountId);
  const calendar = google.calendar({ version: 'v3', auth: oauth2Client });

  // Get the event first
  const event = await calendar.events.get({
    calendarId: 'primary',
    eventId: args.id
  });

  // Find the current user's attendee entry and update their response
  const attendees = event.data.attendees || [];
  const updatedAttendees = attendees.map(attendee => {
    if (attendee.self) {
      return { ...attendee, responseStatus: args.response };
    }
    return attendee;
  });

  // Update the event
  const result = await calendar.events.patch({
    calendarId: 'primary',
    eventId: args.id,
    requestBody: {
      attendees: updatedAttendees
    }
  });

  return {
    success: true,
    eventId: args.id,
    summary: result.data.summary,
    responseStatus: args.response,
    start: result.data.start,
    account: accountId || 'default'
  };
}

// Main execution
const args = minimist(process.argv.slice(2));
const startTime = Date.now();

updateEventResponse(args)
  .then(result => {
    // Log the action
    logAction('respond_to_event', args, result, {
      account: result.account,
      duration: Date.now() - startTime
    });

    console.log(JSON.stringify(result, null, 2));
  })
  .catch(error => {
    const errorResult = {
      success: false,
      error: error.message
    };

    // Log the failed action
    logAction('respond_to_event', args, errorResult, {
      account: args.account || 'default',
      duration: Date.now() - startTime,
      failed: true
    });

    console.error(JSON.stringify(errorResult, null, 2));
    process.exit(1);
  });
