#!/usr/bin/env node

import { google } from 'googleapis';
import minimist from 'minimist';
import { getAuthClient, parseAccountArg } from './auth/auth-utils.js';
import { logAction } from './action-logger.js';

async function quickAddEvent(args) {
  // Validate required args
  if (!args.text && !args.t) {
    throw new Error('Missing required argument: --text or -t (natural language event description)');
  }

  // Get authenticated client for specified account (or default)
  const accountId = parseAccountArg(args);
  const oauth2Client = await getAuthClient(accountId);

  const calendar = google.calendar({ version: 'v3', auth: oauth2Client });

  // Quick add event using natural language
  const result = await calendar.events.quickAdd({
    calendarId: args.calendar || 'primary',
    text: args.text || args.t
  });

  const event = result.data;

  return {
    success: true,
    eventId: event.id,
    htmlLink: event.htmlLink,
    summary: event.summary,
    start: event.start,
    end: event.end,
    location: event.location,
    description: event.description,
    created: event.created,
    inputText: args.text || args.t
  };
}

// Main execution
const args = minimist(process.argv.slice(2));
const startTime = Date.now();

quickAddEvent(args)
  .then(result => {
    // Log the action
    logAction('quick_add_event', args, result, {
      account: args.account || 'default',
      duration: Date.now() - startTime
    });

    console.log(JSON.stringify(result, null, 2));
  })
  .catch(error => {
    const errorResult = {
      success: false,
      error: error.message
    };

    // Log the failed action
    logAction('quick_add_event', args, errorResult, {
      account: args.account || 'default',
      duration: Date.now() - startTime,
      failed: true
    });

    console.error(JSON.stringify(errorResult, null, 2));
    process.exit(1);
  });
