#!/usr/bin/env node

import { google } from 'googleapis';
import minimist from 'minimist';
import { getAuthClient, parseAccountArg } from './auth/auth-utils.js';
import { logAction } from './action-logger.js';

async function getEvent(args) {
  // Validate required args
  if (!args.id && !args.eventId) {
    throw new Error('Missing required argument: --id or --eventId (event ID)');
  }

  // Get authenticated client for specified account (or default)
  const accountId = parseAccountArg(args);
  const oauth2Client = await getAuthClient(accountId);

  const calendar = google.calendar({ version: 'v3', auth: oauth2Client });

  // Get event
  const result = await calendar.events.get({
    calendarId: args.calendar || 'primary',
    eventId: args.id || args.eventId
  });

  const event = result.data;

  // Format the response
  const eventData = {
    id: event.id,
    summary: event.summary,
    description: event.description,
    location: event.location,
    status: event.status,
    htmlLink: event.htmlLink,
    created: event.created,
    updated: event.updated,
    colorId: event.colorId,
    visibility: event.visibility
  };

  // Handle start/end times
  if (event.start) {
    eventData.start = event.start.dateTime || event.start.date;
    eventData.startTimeZone = event.start.timeZone;
    eventData.isAllDay = !!event.start.date;
  }

  if (event.end) {
    eventData.end = event.end.dateTime || event.end.date;
    eventData.endTimeZone = event.end.timeZone;
  }

  // Include attendees
  if (event.attendees && event.attendees.length > 0) {
    eventData.attendees = event.attendees.map(att => ({
      email: att.email,
      displayName: att.displayName,
      responseStatus: att.responseStatus,
      organizer: att.organizer || false,
      self: att.self || false,
      optional: att.optional || false
    }));
  }

  // Include organizer
  if (event.organizer) {
    eventData.organizer = {
      email: event.organizer.email,
      displayName: event.organizer.displayName,
      self: event.organizer.self || false
    };
  }

  // Include recurrence
  if (event.recurrence) {
    eventData.recurrence = event.recurrence;
  }

  // Include reminders
  if (event.reminders) {
    eventData.reminders = event.reminders;
  }

  // Include conference data (e.g., Google Meet)
  if (event.conferenceData) {
    eventData.conferenceData = {
      entryPoints: event.conferenceData.entryPoints,
      conferenceSolution: event.conferenceData.conferenceSolution,
      conferenceId: event.conferenceData.conferenceId
    };
  }

  return {
    success: true,
    event: eventData
  };
}

// Main execution
const args = minimist(process.argv.slice(2));
const startTime = Date.now();

getEvent(args)
  .then(result => {
    // Log the action
    logAction('get_event', args, result, {
      account: args.account || 'default',
      duration: Date.now() - startTime
    });

    console.log(JSON.stringify(result, null, 2));
  })
  .catch(error => {
    const errorResult = {
      success: false,
      error: error.message
    };

    // Log the failed action
    logAction('get_event', args, errorResult, {
      account: args.account || 'default',
      duration: Date.now() - startTime,
      failed: true
    });

    console.error(JSON.stringify(errorResult, null, 2));
    process.exit(1);
  });
