"""
Skill Finding Agent

A research subagent that finds relevant skills for a task.
Used by the Task Wizard to discover existing skills that could be reused.
"""

from claude_agent_sdk import AgentDefinition

# This agent is designed to be used as a subagent via the Task tool
SKILL_FINDER_AGENT = AgentDefinition(
    description="""Research agent for finding relevant skills in the SkillsBench ecosystem.
Use this agent when you need to:
- Find existing skills that match a task's requirements
- Discover skills from anthropic-skills or other skill repositories
- Analyze what skills are commonly used for similar tasks
- Recommend whether to reuse existing skills or create new ones""",
    prompt="""You are a skill research agent for SkillsBench.

Your job is to find relevant skills that could be used for a given task.

## Skill Sources to Search

1. **SkillsBench task-ideas reference**: Check .claude/skills/skillsbench/references/task-ideas.md for skill categories
2. **Existing task skills**: Look at tasks/*/environment/skills/ for skills already in the repo
3. **Anthropic skills repository**: If available at .local-workspace/anthropic-skills/
4. **Skill patterns**: Identify common skill patterns (xlsx, pdf, docx, etc.)

## What to Report

For each relevant skill found:
1. **Skill name and location**
2. **What it provides** (scripts, references, guidance)
3. **Applicability score** (high/medium/low) for the target task
4. **Reusability notes** - can it be used as-is or needs modification?

## Important Guidelines

- Prefer existing, tested skills over creating new ones
- Skills should be general-purpose, not task-specific
- Note if a skill requires external dependencies (apt packages, pip packages)
- Flag if skills have licensing considerations (check LICENSE.txt)

Return a structured summary of your findings.""",
    tools=["Read", "Glob", "Grep", "WebFetch"],
)


async def find_skills_for_task(task_description: str, cwd: str | None = None):
    """
    Standalone function to find skills for a task description.
    Can be used directly without going through the main agent system.
    """
    from claude_agent_sdk import ClaudeAgentOptions, ResultMessage, query

    options = ClaudeAgentOptions(
        system_prompt=SKILL_FINDER_AGENT.prompt, allowed_tools=SKILL_FINDER_AGENT.tools or [], cwd=cwd, permission_mode="bypassPermissions"
    )

    result = None
    async for message in query(
        prompt=f"""Find relevant skills for this task:

{task_description}

Search the SkillsBench repository for:
1. Existing skills in tasks/*/environment/skills/
2. Skill ideas in .claude/skills/skillsbench/references/task-ideas.md
3. Any skill patterns that would be applicable

Report your findings with applicability scores.""",
        options=options,
    ):
        if isinstance(message, ResultMessage):
            result = message.result

    return result
