"""
Policy Checker Agent

Checks PR content against the PR Review Policy defined in maintainer.md.
Does NOT run experiments - only performs static analysis.

Generates a policy compliance report saved to .local-workspace/
"""

import json
from datetime import datetime
from pathlib import Path
from typing import Optional

from claude_agent_sdk import AssistantMessage, ClaudeAgentOptions, ResultMessage, TextBlock, query


POLICY_CHECKER_SYSTEM_PROMPT = """You are the SkillsBench Policy Checker.

Your job is to check task PRs against the PR Review Policy WITHOUT running any experiments.
This is a static analysis tool - you only read and analyze PR content.

## PR Review Policy Checklist

You must check EVERY item below and report pass/fail/warning for each:

### 1. Instruction & Metadata Verification
- [ ] **AI Detection Check**: Does `instruction.md` appear to be AI-generated? Look for:
  - Overly formal/verbose language
  - Generic phrasing that could apply to any task
  - Perfect grammar with no personality
  - Bullet points that feel templated
- [ ] **AI Detection Check**: Does `task.toml` appear to be AI-generated?
- [ ] **Intentional Grammar Errors**: Are there suspicious grammar errors that seem designed to fool AI detectors?

### 2. Data Quality Assessment
- [ ] **Data Validity**: Is the data real-world or appropriately complex?
- [ ] **Toy Model Check**: Is this a toy/synthetic dataset when real data could be used?
- [ ] **AI-Generated Data**: Does the data appear to be AI-generated?

### 3. Task Validity
- [ ] **Professional Context**: Is the task grounded in a realistic professional scenario?
- [ ] **Artificial Complexity**: Does the author make a simple task artificially difficult?
  - Excessive distractors?
  - Unnecessarily complex solution path?
  - Steps that no human professional would take?

### 4. Oracle Quality
- [ ] **Simplicity Check**: Is the oracle appropriately simple (e.g., Excel-level for data tasks)?
- [ ] **Over-engineering**: Is the oracle unnecessarily complex?

### 5. Test Output Requirements
- [ ] **Test Count**: Are there fewer than 10 tests? If more, is it justified?
- [ ] **Test Quality**: Do tests follow the unit test guidelines?
  - No redundant tests (testing same capability multiple times)
  - Tests verify meaningful behavior
  - Not excessive edge cases

### 6. Multimodal Verification (if applicable)
- [ ] **Output Files Present**: For multimodal tasks, are expected output files present?
  - Audio files for audiobook tasks
  - Images for image generation
  - Videos for video tasks
  - PDFs for PDF editing tasks

### 7. Skill Quality
- [ ] **Valuable Skills**: Do skills provide genuine guidance?
- [ ] **Padding Check**: Are skills just padding/filler content?

## Workflow

1. Fetch PR files using GitHub API (NO git checkout)
2. Read and analyze:
   - instruction.md
   - task.toml
   - solution/solve.sh or solution files
   - tests/test_outputs.py
   - environment/skills/ (all SKILL.md files)
   - environment/workspace/ or data files
3. For EACH policy item, provide:
   - Status: PASS / FAIL / WARNING / N/A
   - Evidence: Specific quotes or observations
   - Recommendation: What to fix if needed

## Report Format

Generate a report with this structure:

```
================================================================================
            PR #[NUMBER] POLICY COMPLIANCE REPORT: [task-name]
================================================================================
Date: YYYY-MM-DD
PR: #[NUMBER] - [PR Title]
Author: [github-username]

================================================================================
                         POLICY COMPLIANCE SUMMARY
================================================================================

Overall: [X/Y] checks passed | [X] warnings | [X] failures

+------------------------------------------+--------+
| Policy Check                             | Status |
+------------------------------------------+--------+
| 1.1 instruction.md AI detection          | [STATUS] |
| 1.2 task.toml AI detection               | [STATUS] |
| 1.3 Intentional grammar errors           | [STATUS] |
| 2.1 Data validity                        | [STATUS] |
| 2.2 Toy model check                      | [STATUS] |
| 2.3 AI-generated data                    | [STATUS] |
| 3.1 Professional context                 | [STATUS] |
| 3.2 Artificial complexity                | [STATUS] |
| 4.1 Oracle simplicity                    | [STATUS] |
| 4.2 Oracle over-engineering              | [STATUS] |
| 5.1 Test count (<10)                     | [STATUS] |
| 5.2 Test quality                         | [STATUS] |
| 6.1 Multimodal outputs (if applicable)   | [STATUS] |
| 7.1 Skill value                          | [STATUS] |
| 7.2 Skill padding check                  | [STATUS] |
+------------------------------------------+--------+

================================================================================
                         DETAILED FINDINGS
================================================================================

### 1. Instruction & Metadata Verification

#### 1.1 instruction.md AI Detection
Status: [PASS/FAIL/WARNING]
Evidence:
[Quote relevant sections or describe observations]
Recommendation:
[What to fix, or "None" if passed]

#### 1.2 task.toml AI Detection
Status: [PASS/FAIL/WARNING]
...

[Continue for all checks]

================================================================================
                         AUTHOR HISTORY CHECK
================================================================================

Author: [username]
Previous flags: [Check if author has been flagged before]
Recommendation: [AUTO-CLOSE if repeat offender, or "Proceed with review"]

================================================================================
                         FINAL RECOMMENDATION
================================================================================

Recommendation: [PROCEED TO BENCHMARK / NEEDS CHANGES / CLOSE PR]

Critical Issues (must fix):
1. [Issue requiring PR closure or major changes]

Warnings (should address):
1. [Minor concerns]

================================================================================
                         END OF REPORT
================================================================================
```

## Important Rules

1. DO NOT run harbor or any experiments
2. DO NOT use git checkout/pull - only fetch files via API
3. Be SPECIFIC with evidence - quote actual content
4. Check ALL policy items - don't skip any
5. For AI detection, look at writing style, not just run a detector
6. For data checks, examine actual data files when possible
"""


def get_timestamp_short() -> str:
    """Generate short timestamp like 0118-1430 for Jan 18, 2:30 PM."""
    return datetime.now().strftime("%m%d-%H%M")


async def run_policy_check(pr_numbers: list[int]):
    """
    Check one or more task PRs against the PR Review Policy.

    Args:
        pr_numbers: List of PR numbers to check
    """
    repo_root = Path(__file__).parent.parent.parent.parent
    workspace = repo_root / ".local-workspace"
    workspace.mkdir(exist_ok=True)

    options = ClaudeAgentOptions(
        system_prompt=POLICY_CHECKER_SYSTEM_PROMPT,
        allowed_tools=["Read", "Write", "Edit", "Bash", "Glob", "Grep"],
        permission_mode="acceptEdits",
        cwd=str(repo_root),
        max_turns=50,
    )

    pr_list = ", ".join(f"#{pr}" for pr in pr_numbers)
    timestamp = get_timestamp_short()

    print("\n📋 Policy Checker\n")
    print(f"PRs to check: {pr_list}")
    print(f"Workspace: {workspace}")
    print(f"Timestamp: {timestamp}")
    print("-" * 60)

    prompt = f"""Check the following PRs against the PR Review Policy: {pr_list}

## Instructions

1. For each PR, fetch the task files using the GitHub API:
   ```bash
   # Get PR info
   gh pr view <number> --json headRefName,title,author,files

   # Get file contents
   gh api repos/benchflow-ai/skillsbench/pulls/<number>/files | jq -r '.[].filename'

   # Download specific files
   gh api repos/benchflow-ai/skillsbench/contents/<path>?ref=<branch> | jq -r '.content' | base64 -d
   ```

2. Read and analyze all relevant files:
   - instruction.md
   - task.toml
   - solution/ files
   - tests/test_outputs.py
   - environment/skills/**/*.md
   - environment/workspace/ or data files

3. Check EVERY policy item and provide specific evidence

4. Generate a report for each PR saved to:
   .local-workspace/pr-{pr_numbers[0]}-policy-{timestamp}.txt

## Policy Reference

Read the PR Review Policy section from maintainer.md for the full policy:
```bash
cat maintainer.md
```

## DO NOT:
- Run any harbor commands
- Run any experiments
- Use git checkout or git pull
- Skip any policy checks

## Output

Save the report to: .local-workspace/pr-<number>-policy-{timestamp}.txt
"""

    try:
        async for message in query(prompt=prompt, options=options):
            if isinstance(message, AssistantMessage):
                for block in message.content:
                    if isinstance(block, TextBlock):
                        print(block.text)
            elif isinstance(message, ResultMessage):
                if message.is_error:
                    print(f"\n❌ Error: {message.result}")
                else:
                    print("\n✅ Policy check complete!")
                    if message.total_cost_usd:
                        print(f"   Cost: ${message.total_cost_usd:.4f}")
    except Exception as e:
        error_msg = str(e)
        if "exit code" in error_msg.lower() or "message reader" in error_msg.lower():
            report_files = list(workspace.glob(f"pr-{pr_numbers[0]}-policy-*.txt"))
            if report_files:
                print(f"\n✅ Policy check complete! (connection closed with minor error)")
                print(f"   Reports generated: {[f.name for f in report_files]}")
            else:
                print(f"\n⚠️  Agent may have completed but connection was lost: {e}")
        else:
            raise
