#!/usr/bin/env python3
"""
NodeMedic-FINE API Wrapper

This script wraps the NodeMedic-FINE tool as a REST API that agents can call.
It runs inside the artifact container and exposes endpoints for analysis.

Endpoints:
    GET  /           - Health check and available endpoints
    GET  /health     - Health check
    POST /analyze    - Run NodeMedic-FINE analysis on a package
    GET  /results/<id> - Get cached results for a previous analysis
"""

import hashlib
import json
import os
import subprocess
from pathlib import Path

from flask import Flask, jsonify, request
from flask_cors import CORS

app = Flask(__name__)
CORS(app)

# Configuration
NODETAINT_DIR = os.environ.get("NODETAINT_DIR", "/nodetaint")
RESULTS_CACHE = {}
ANALYSIS_TIMEOUT = 300  # 5 minutes


def run_nodemedic(package: str, version: str) -> dict:
    """
    Run NodeMedic-FINE analysis on a package.

    This runs the full NodeMedic-FINE pipeline:
    1. Downloads the package from npm
    2. Runs taint analysis
    3. Synthesizes exploits
    4. Validates exploits

    Returns the full analysis result JSON.
    """
    pkg_spec = f"{package}@{version}"
    analysis_id = hashlib.md5(pkg_spec.encode()).hexdigest()[:8]

    # Check cache
    if analysis_id in RESULTS_CACHE:
        return RESULTS_CACHE[analysis_id]

    # Set up environment with Z3 in PATH
    env = os.environ.copy()
    z3_path = Path(NODETAINT_DIR) / "z3" / "bin"
    if z3_path.exists():
        env["PATH"] = f"{z3_path}:{env.get('PATH', '')}"

    # Run the analysis
    # NodeMedic-FINE uses: make run PKG=<package>@<version>
    cmd = ["make", "run", f"PKG={pkg_spec}"]

    try:
        result = subprocess.run(cmd, cwd=NODETAINT_DIR, env=env, capture_output=True, text=True, timeout=ANALYSIS_TIMEOUT)

        # Parse the output - NodeMedic-FINE outputs JSON to stdout
        output = {
            "id": package,
            "version": version,
            "analysis_id": analysis_id,
            "command": " ".join(cmd),
            "exit_code": result.returncode,
            "stdout": result.stdout,
            "stderr": result.stderr,
        }

        # Try to parse JSON from stdout
        try:
            # Look for JSON output in stdout
            lines = result.stdout.strip().split("\n")
            for line in reversed(lines):
                line = line.strip()
                if line.startswith("{") and line.endswith("}"):
                    parsed = json.loads(line)
                    output.update(parsed)
                    break
        except json.JSONDecodeError:
            pass

        # Also check for output files
        output_file = Path(NODETAINT_DIR) / "results" / f"{package}-{version}.json"
        if output_file.exists():
            with open(output_file) as f:
                file_output = json.load(f)
                output.update(file_output)

        RESULTS_CACHE[analysis_id] = output
        return output

    except subprocess.TimeoutExpired:
        return {"id": package, "version": version, "error": "Analysis timed out", "timeout": ANALYSIS_TIMEOUT}
    except Exception as e:
        return {"id": package, "version": version, "error": str(e)}


@app.route("/", methods=["GET"])
def index():
    """Health check and endpoint listing."""
    return jsonify(
        {
            "service": "NodeMedic-FINE API",
            "status": "running",
            "endpoints": {
                "GET /": "This endpoint - health check and info",
                "GET /health": "Health check",
                "POST /analyze": "Run analysis - body: {package, version}",
                "GET /results/<id>": "Get cached analysis results",
            },
            "example": {"method": "POST", "url": "/analyze", "body": {"package": "node-rsync", "version": "1.0.3"}},
        }
    )


@app.route("/health", methods=["GET"])
def health():
    """Health check endpoint."""
    return jsonify({"status": "healthy", "service": "nodemedic-fine"})


@app.route("/analyze", methods=["POST"])
def analyze():
    """
    Run NodeMedic-FINE analysis on a package.

    Request body:
        {
            "package": "node-rsync",
            "version": "1.0.3",
            "mode": "full"  # optional, default is "full"
        }

    Response:
        Full NodeMedic-FINE analysis output including:
        - hasMain: whether package has a main entry point
        - sinks: detected sinks in the code
        - sinksHit: sinks that were executed during analysis
        - sinkType: type of sink (exec, eval, etc.)
        - entryPoints: entry points found
        - treeMetadata: dependency tree info
        - synthesisResult: exploit synthesis results
        - candidateExploit: generated exploit code
        - exploitResults: exploit validation results
    """
    data = request.get_json()

    if not data:
        return jsonify({"error": "No JSON body provided"}), 400

    package = data.get("package")
    version = data.get("version")

    if not package or not version:
        return jsonify({"error": "Missing required fields: package, version"}), 400

    result = run_nodemedic(package, version)
    return jsonify(result)


@app.route("/results/<analysis_id>", methods=["GET"])
def get_results(analysis_id):
    """Get cached results for a previous analysis."""
    if analysis_id in RESULTS_CACHE:
        return jsonify(RESULTS_CACHE[analysis_id])
    return jsonify({"error": "Analysis not found"}), 404


if __name__ == "__main__":
    port = int(os.environ.get("PORT", 3000))
    print(f"Starting NodeMedic-FINE API on port {port}...")
    print(f"NODETAINT_DIR: {NODETAINT_DIR}")
    app.run(host="0.0.0.0", port=port, debug=False)
