# Artifact Runner Design

## Overview

The Artifact Runner enables SkillsBench tasks to interact with paper artifacts (research tool implementations). It handles:
1. Building and running artifact Docker containers
2. Exposing APIs for agent interaction
3. Managing container lifecycle

## Architecture

```
┌─────────────────────────────────────────────────────────────────┐
│                        Host Machine                              │
│                                                                  │
│  ┌──────────────────────┐    ┌────────────────────────────────┐ │
│  │   Artifact Container │    │       Agent Container          │ │
│  │                      │    │       (Harbor)                 │ │
│  │  ┌────────────────┐  │    │                                │ │
│  │  │  API Wrapper   │◄─┼────┼── HTTP requests from agent     │ │
│  │  │  (Flask/etc)   │  │    │                                │ │
│  │  └───────┬────────┘  │    │  Agent reads instructions,     │ │
│  │          │           │    │  calls API, collects results   │ │
│  │  ┌───────▼────────┐  │    │                                │ │
│  │  │  Artifact Tool │  │    │                                │ │
│  │  │  (NodeMedic,   │  │    │                                │ │
│  │  │   etc.)        │  │    │                                │ │
│  │  └────────────────┘  │    │                                │ │
│  │                      │    │                                │ │
│  │  Port: 3000 ─────────┼────┼─► http://172.17.0.1:3000       │ │
│  └──────────────────────┘    └────────────────────────────────┘ │
└─────────────────────────────────────────────────────────────────┘
```

## Why This Approach (Host Port + Gateway IP)

At first glance, it may seem natural to use Harbor's docker-compose support to run the artifact container and the agent container in the same custom Docker network. In practice, we use **host port binding** (e.g., `-p 3000:3000`) and have the agent talk to the artifact via the **Docker host gateway** (typically `http://172.17.0.1:<port>/`) because it is the most reliable option across Harbor environments.

Key reasons:

- **Harbor network configuration is hard to control from tasks.** Harbor's `--ek` mechanism passes kwargs to the environment `__init__.py`, but `network_mode` is computed internally (commonly forced to `bridge` or `none`). This makes it difficult to place both the Harbor agent container and a separate artifact container into the same custom Docker network.

- **Stable addressability from inside the agent container.** The host gateway (`172.17.0.1` on Linux bridge) is a consistent rendezvous point. By binding the artifact service to a host port, the agent can reach it regardless of the agent container name/network, as long as outbound networking is allowed.

- **Avoids Docker-in-Docker / compose-in-container pitfalls.** Having the agent container orchestrate docker-compose usually implies Docker socket mounting or privileged setups. Those add security risk and make tasks less portable.

- **Keeps the task interface uniform.** With an HTTP API wrapper (e.g., `generic-exec`), tasks can assume an interface like `GET /` and `POST /exec`, rather than requiring repo-specific ports, binaries, and docker-compose layouts.

- **Less coupling to artifact repo internals.** Many research artifact repos have nonstandard compose files, multiple services, or assume interactive/manual setup. Running one artifact container with a small injected API wrapper is often simpler than trying to faithfully reproduce full compose orchestration inside Harbor.

## API Wrapper Options

### Option 1: Tool-Specific API (Recommended)

Create a custom API for the specific tool that exposes meaningful endpoints:

```python
# nodemedic-api.py
@app.route("/analyze", methods=["POST"])
def analyze():
    """Run NodeMedic-FINE on a package."""
    data = request.json
    package = data["package"]
    version = data["version"]
    result = run_nodemedic(package, version)
    return jsonify(result)
```

**task.toml configuration:**
```toml
[artifact]
repo = "https://github.com/NodeMedicAnalysis/NodeMedic-FINE"
port = 3000
api_wrapper = "nodemedic"  # Use built-in wrapper
workdir = "/nodetaint"
```

### Option 2: Generic Command Execution API

For artifacts without a custom wrapper, use the generic exec API:

```python
# POST /exec
{
    "command": "make run PKG=node-rsync@1.0.3",
    "workdir": "/nodetaint",
    "timeout": 120
}
```

**task.toml configuration:**
```toml
[artifact]
repo = "https://github.com/some/artifact"
port = 8080
api_wrapper = "generic-exec"
workdir = "/workspace"
```

### Option 3: Custom API Script

Provide your own API wrapper in the task:

```toml
[artifact]
repo = "https://github.com/some/artifact"
port = 3000
api_wrapper = "custom"
api_script = "environment/api-server.py"  # Path relative to task dir
```

## Task Configuration Reference

```toml
[artifact]
# Required: Repository containing the artifact
repo = "https://github.com/user/artifact-repo"

# Required: Port to expose
port = 3000

# API wrapper type (default: "static" for file serving only)
# Options: "static", "generic-exec", "nodemedic", "custom"
api_wrapper = "generic-exec"

# Working directory inside container
workdir = "/workspace"

# Custom command to run instead of default entrypoint
# For static: "python3 -m http.server 3000"
# For API: "python3 /api/server.py"
command = "python3 -m http.server 3000"

# Custom API script (only for api_wrapper = "custom")
api_script = "environment/api-server.py"

# Environment variables for the container
[artifact.env]
DEBUG = "1"
TIMEOUT = "300"

# Files to copy to task environment (PDFs, etc.) - OPTIONAL
files = ["papers/paper.pdf"]
```

## Documentation Options

Tasks can provide reference documentation in several ways:

### Option A: PDF Files (for papers)
Copy PDFs to the task environment:
```toml
[artifact]
files = ["papers/paper.pdf"]
```

### Option B: GitHub URLs (simpler, no files needed)
Just include URLs in `instruction.md`. Agents have internet access and can fetch docs directly:

```markdown
## Reference Documentation
- GitHub repo: https://github.com/user/artifact-repo
- README: https://raw.githubusercontent.com/user/artifact-repo/main/README.md
- Usage docs: https://raw.githubusercontent.com/user/artifact-repo/main/docs/USAGE.md
```

The agent can fetch these with `curl` or the exec API:
```python
# Via exec API
requests.post("http://172.17.0.1:3000/exec", json={
    "command": "curl -s https://raw.githubusercontent.com/user/repo/main/README.md"
})

# Or directly from agent container (has internet)
subprocess.run(["curl", "-s", "https://raw.githubusercontent.com/..."])
```

### Option C: Read from Cloned Repo
The artifact container already has the cloned repo. Use `/files/<path>`:
```python
requests.get("http://172.17.0.1:3000/files/README.md")
```

**Recommendation**: Use Option B (URLs) for simplicity unless you need a paper PDF that isn't publicly accessible online.

## How Agent Interacts with Artifact

### With Generic Exec API

```python
# Agent code to run NodeMedic-FINE
import requests

# 1. Check what's available
response = requests.get("http://172.17.0.1:3000/")
print(response.json())  # Shows available endpoints

# 2. Execute analysis command
result = requests.post("http://172.17.0.1:3000/exec", json={
    "command": "make run PKG=node-rsync@1.0.3",
    "workdir": "/nodetaint",
    "timeout": 120
})
output = result.json()
print(output["stdout"])  # Contains NodeMedic-FINE output

# 3. Read result files if needed
files = requests.get("http://172.17.0.1:3000/files/results/output.json")
```

### With Tool-Specific API

```python
# Agent code - cleaner interface
import requests

# Direct analysis call
result = requests.post("http://172.17.0.1:3000/analyze", json={
    "package": "node-rsync",
    "version": "1.0.3"
})
output = result.json()
# output contains full NodeMedic-FINE results: hasMain, sinks, sinksHit, etc.
```

## Implementation Steps

### For NodeMedic-FINE Task

1. **Update task.toml** to use generic-exec API:
```toml
[artifact]
repo = "https://github.com/NodeMedicAnalysis/NodeMedic-FINE"
port = 3000
api_wrapper = "generic-exec"
workdir = "/nodetaint"
```

2. **Update instruction.md** to explain the API:
```markdown
## Artifact API

The artifact exposes a command execution API:

- `GET /` - List available endpoints
- `POST /exec` - Execute a command
  - Body: `{"command": "...", "workdir": "/nodetaint"}`
  - Returns: `{"stdout": "...", "stderr": "...", "exit_code": 0}`
- `GET /files/<path>` - Read output files
```

3. **Update solve.sh** to use the API instead of hardcoding results

### Adding API Wrapper Support to artifact-runner

The artifact runner needs to:
1. Detect `api_wrapper` config
2. Copy/inject the API script into container
3. Start the API server instead of static file server

This can be done by:
- Building a custom Dockerfile layer
- Or using Docker volumes to mount the API script
- Or using `docker cp` after container start
