#!/usr/bin/env python3
"""
Match TB tasks to skills using BM25 + Vector search + RRF fusion.

Usage:
    python match_task_skills.py --tasks-dir <path> --indexes-dir <path> --output-dir <path>
"""

import argparse
import logging
import pickle
from pathlib import Path
from typing import Any

import numpy as np
import yaml
from sentence_transformers import SentenceTransformer
from tqdm import tqdm

from .utils import SafeTaskParser, TextPreprocessor, build_query_text, compute_rrf_score, load_json, save_json

logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(name)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


class TaskSkillMatcher:
    """
    Match tasks to skills using hybrid retrieval (BM25 + Vector + RRF).
    """

    def __init__(self, config: dict[str, Any], indexes_dir: Path):
        self.config = config
        self.indexes_dir = indexes_dir

        # Initialize components
        self.preprocessor = TextPreprocessor(
            lowercase=config["preprocessing"]["lowercase"], min_token_length=config["preprocessing"]["min_token_length"]
        )
        self.task_parser = SafeTaskParser(verifier_whitelist=config["verifier_whitelist"])

        # Load indexes
        self.load_indexes()

    def load_indexes(self):
        """
        Load pre-built indexes from disk.
        """
        logger.info(f"Loading indexes from {self.indexes_dir}")

        # Load skill documents
        self.skill_docs = load_json(self.indexes_dir / "skill_docs.json")
        logger.info(f"Loaded {len(self.skill_docs)} skill documents")

        # Load BM25 index
        with open(self.indexes_dir / "bm25_index.pkl", "rb") as f:
            self.bm25_index = pickle.load(f)
        logger.info("Loaded BM25 index")

        # Load vector embeddings
        self.embeddings = np.load(self.indexes_dir / "embeddings.npy")
        logger.info(f"Loaded embeddings: shape={self.embeddings.shape}")

        # Load embedding model for query encoding
        logger.info(f"Loading embedding model: {self.config['embedding']['model_name']}")
        self.embedding_model = SentenceTransformer(self.config["embedding"]["model_name"], device=self.config["embedding"]["device"])

        # Load metadata
        metadata = load_json(self.indexes_dir / "index_metadata.json")
        logger.info(f"Index metadata: {metadata}")

    def retrieve_bm25(self, query_text: str, topn: int) -> list[tuple[int, float]]:
        """
        Retrieve top N skills using BM25.

        Returns:
            List of (skill_idx, score) tuples
        """
        # Tokenize query
        query_tokens = self.preprocessor.tokenize(query_text)

        # Get BM25 scores
        scores = self.bm25_index.get_scores(query_tokens)

        # Get top N indices
        top_indices = np.argsort(scores)[::-1][:topn]
        results = [(int(idx), float(scores[idx])) for idx in top_indices]

        return results

    def retrieve_vector(self, query_text: str, topn: int) -> list[tuple[int, float]]:
        """
        Retrieve top N skills using vector similarity.

        Returns:
            List of (skill_idx, score) tuples
        """
        # Encode query
        query_embedding = self.embedding_model.encode(
            [query_text], convert_to_numpy=True, normalize_embeddings=self.config["embedding"]["normalize"]
        )[0]

        # Compute cosine similarity (dot product for normalized vectors)
        similarities = self.embeddings @ query_embedding

        # Get top N indices
        top_indices = np.argsort(similarities)[::-1][:topn]
        results = [(int(idx), float(similarities[idx])) for idx in top_indices]

        return results

    def fuse_with_rrf(self, bm25_results: list[tuple[int, float]], vec_results: list[tuple[int, float]], topk: int) -> list[dict[str, Any]]:
        """
        Fuse BM25 and vector results using Reciprocal Rank Fusion.

        Returns:
            List of candidates with RRF scores
        """
        # Build rank maps (idx -> rank)
        bm25_ranks = {idx: rank + 1 for rank, (idx, _) in enumerate(bm25_results)}
        vec_ranks = {idx: rank + 1 for rank, (idx, _) in enumerate(vec_results)}

        # Build score maps
        bm25_scores = dict(bm25_results)
        vec_scores = dict(vec_results)

        # Collect all unique indices
        all_indices = set(bm25_ranks.keys()) | set(vec_ranks.keys())

        # Compute RRF scores
        rrf_k = self.config["retrieval"]["rrf_k"]
        fused_results = []

        for idx in all_indices:
            ranks = []
            if idx in bm25_ranks:
                ranks.append(bm25_ranks[idx])
            if idx in vec_ranks:
                ranks.append(vec_ranks[idx])

            rrf_score = compute_rrf_score(ranks, k=rrf_k)

            fused_results.append(
                {
                    "skill_idx": idx,
                    "skill_id": self.skill_docs[idx]["skill_id"],
                    "skill_name": self.skill_docs[idx]["skill_name"],
                    "bm25_rank": bm25_ranks.get(idx, -1),
                    "bm25_score": bm25_scores.get(idx, 0.0),
                    "vec_rank": vec_ranks.get(idx, -1),
                    "vec_score": vec_scores.get(idx, 0.0),
                    "rrf_score": rrf_score,
                }
            )

        # Sort by RRF score (desc), then by skill_id (asc) for stable tie-breaking
        fused_results.sort(key=lambda x: (-x["rrf_score"], x["skill_id"]))

        # Take top K
        return fused_results[:topk]

    def add_explainability(self, candidates: list[dict[str, Any]], query_text: str, query_tokens: list[str]) -> list[dict[str, Any]]:
        """
        Add explainability information to candidates.
        """
        for candidate in candidates:
            idx = candidate["skill_idx"]
            skill_text = self.skill_docs[idx]["skill_text"].lower()
            skill_tokens = set(self.preprocessor.tokenize(skill_text))

            # Find matched terms
            matched_terms = list(set(query_tokens) & skill_tokens)

            # Find snippet from skill description where match occurs
            snippet = ""
            if matched_terms and self.skill_docs[idx]["description"]:
                desc = self.skill_docs[idx]["description"]
                # Find first matched term in description
                for term in matched_terms:
                    if term in desc.lower():
                        # Extract snippet around the term
                        pos = desc.lower().find(term)
                        start = max(0, pos - 40)
                        end = min(len(desc), pos + 80)
                        snippet = desc[start:end]
                        if start > 0:
                            snippet = "..." + snippet
                        if end < len(desc):
                            snippet = snippet + "..."
                        break

            candidate["evidence"] = {
                "matched_terms": matched_terms[:10],  # Limit to top 10
                "matched_sources": ["instruction", "tags"],  # Simplified
                "snippet": snippet,
            }

        return candidates

    def check_quality(self, candidates: list[dict[str, Any]]) -> bool:
        """
        Check if manual review is needed based on quality thresholds.
        """
        if not candidates:
            return True  # Needs review if no candidates

        top1_score = candidates[0]["rrf_score"]
        min_top1 = self.config["quality"]["min_top1_score"]

        # Check if top-1 score is too low
        if top1_score < min_top1:
            return True

        # Check if top 10 scores are too flat
        if len(candidates) >= 10:
            top10_scores = [c["rrf_score"] for c in candidates[:10]]
            score_range = max(top10_scores) - min(top10_scores)
            max_flatness = self.config["quality"]["max_score_flatness"]

            if score_range < max_flatness:
                return True

        return False

    def match_task(self, task_dir: Path, output_dir: Path):
        """
        Match a single task to skills and save results.
        """
        task_id = task_dir.name
        logger.info(f"Processing task: {task_id}")

        # Parse task safely
        task_data = self.task_parser.parse_task(task_dir)

        # Extract tokens
        tokens = self.task_parser.extract_tokens(task_data)

        # Build query text
        query_text = build_query_text(task_data, tokens)
        query_tokens = self.preprocessor.tokenize(query_text)

        # Retrieve from BM25
        topn_bm25 = self.config["retrieval"]["topn_bm25"]
        bm25_results = self.retrieve_bm25(query_text, topn_bm25)

        # Retrieve from vector index
        topn_vec = self.config["retrieval"]["topn_vec"]
        vec_results = self.retrieve_vector(query_text, topn_vec)

        # Fuse with RRF
        topk = self.config["retrieval"]["topk_out"]
        candidates = self.fuse_with_rrf(bm25_results, vec_results, topk)

        # Add explainability
        candidates = self.add_explainability(candidates, query_text, query_tokens)

        # Check quality
        needs_review = self.check_quality(candidates)

        # Save query bundle
        query_bundle = {
            "task_id": task_id,
            "instruction_text": task_data["instruction_text"][:500],  # Truncate for brevity
            "tags_text": task_data["tags_text"],
            "verifier_hints": task_data["verifier_hints"],
            "extracted_tokens": {"command_tokens": tokens["command_tokens"], "domain_tokens": tokens["domain_tokens"]},
            "query_text_preview": query_text[:300],  # Preview only
        }

        task_output_dir = output_dir / task_id
        task_output_dir.mkdir(parents=True, exist_ok=True)
        save_json(query_bundle, task_output_dir / "query_bundle.json")

        # Save candidates
        candidates_output = {
            "task_id": task_id,
            "num_candidates": len(candidates),
            "needs_manual_review": needs_review,
            "candidates": candidates,
        }
        save_json(candidates_output, task_output_dir / "candidates.json")

        logger.info(f"✓ {task_id}: {len(candidates)} candidates, needs_review={needs_review}")

    def match_all_tasks(self, tasks_dir: Path, output_dir: Path):
        """
        Match all tasks in directory.
        """
        # Find all task directories (have task.toml)
        task_dirs = [d for d in tasks_dir.iterdir() if d.is_dir() and (d / "task.toml").exists()]

        logger.info(f"Found {len(task_dirs)} tasks to process")

        # Process each task
        for task_dir in tqdm(task_dirs, desc="Matching tasks"):
            try:
                self.match_task(task_dir, output_dir)
            except Exception as e:
                logger.error(f"Failed to process {task_dir.name}: {e}", exc_info=True)

        logger.info(f"✓ All tasks processed! Results saved to {output_dir}")


def main():
    parser = argparse.ArgumentParser(description="Match tasks to skills")
    parser.add_argument("--tasks-dir", type=str, required=True, help="Path to tasks root directory")
    parser.add_argument("--indexes-dir", type=str, default="indexes", help="Path to pre-built indexes directory")
    parser.add_argument("--config", type=str, default="config.yaml", help="Path to config file")
    parser.add_argument("--output-dir", type=str, default="output", help="Output directory for match results")
    parser.add_argument("--task-id", type=str, default=None, help="Process single task (optional)")

    args = parser.parse_args()

    # Load config
    with open(args.config) as f:
        config = yaml.safe_load(f)

    # Initialize matcher
    matcher = TaskSkillMatcher(config, Path(args.indexes_dir))

    # Match tasks
    if args.task_id:
        # Single task
        task_dir = Path(args.tasks_dir) / args.task_id
        matcher.match_task(task_dir, Path(args.output_dir))
    else:
        # All tasks
        matcher.match_all_tasks(Path(args.tasks_dir), Path(args.output_dir))


if __name__ == "__main__":
    main()
