#!/usr/bin/env python3
"""
Build BM25 and vector indexes from skills database.

Usage:
    python build_skill_indexes.py --skills-db <path> --config <path> --output-dir <path>
"""

import argparse
import hashlib
import json
import logging
import pickle
from pathlib import Path
from typing import Any

import numpy as np
import yaml
from rank_bm25 import BM25Okapi
from sentence_transformers import SentenceTransformer
from tqdm import tqdm
try:
    from .utils import TextPreprocessor, save_json
except ImportError:
    from utils import TextPreprocessor, save_json

logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(name)s - %(levelname)s - %(message)s")
logger = logging.getLogger(__name__)


class SkillIndexBuilder:
    """
    Build BM25 and vector indexes for skills database.
    """

    def __init__(self, config: dict[str, Any]):
        self.config = config
        self.preprocessor = TextPreprocessor(
            lowercase=config["preprocessing"]["lowercase"], min_token_length=config["preprocessing"]["min_token_length"]
        )

        # Load embedding model
        logger.info(f"Loading embedding model: {config['embedding']['model_name']}")
        self.embedding_model = SentenceTransformer(config["embedding"]["model_name"], device=config["embedding"]["device"])
        logger.info(f"Embedding dimension: {self.embedding_model.get_sentence_embedding_dimension()}")

    def load_skills_db(self, skills_db_path: Path) -> list[dict[str, Any]]:
        """
        Load skills database from JSON file.
        """
        logger.info(f"Loading skills database from {skills_db_path}")
        with open(skills_db_path, encoding="utf-8") as f:
            skills = json.load(f)

        logger.info(f"Loaded {len(skills)} skills")
        return skills

    def prepare_skill_texts(self, skills: list[dict[str, Any]]) -> list[dict[str, Any]]:
        """
        Prepare skill texts for indexing.

        For each skill: skill_text = name + "\n" + description
        """
        skill_docs = []

        for skill in skills:
            skill_id = skill.get("name", "")  # Use name as ID
            name = skill.get("name", "")
            description = skill.get("description", "")

            # Combine name and description
            skill_text = f"{name}\n{description}"

            skill_docs.append(
                {
                    "skill_id": skill_id,
                    "skill_name": name,
                    "description": description,
                    "skill_text": skill_text,
                    "source": skill.get("source", ""),
                    "path": skill.get("path", ""),
                    "url": skill.get("url", ""),
                }
            )

        return skill_docs

    def build_bm25_index(self, skill_docs: list[dict[str, Any]]) -> BM25Okapi:
        """
        Build BM25 index from skill documents.
        """
        logger.info("Building BM25 index...")

        # Tokenize all skill texts
        tokenized_corpus = []
        for doc in tqdm(skill_docs, desc="Tokenizing"):
            tokens = self.preprocessor.tokenize(doc["skill_text"])
            tokenized_corpus.append(tokens)

        # Build BM25 index
        bm25 = BM25Okapi(tokenized_corpus)
        logger.info(f"BM25 index built with {len(tokenized_corpus)} documents")

        return bm25

    def build_vector_index(self, skill_docs: list[dict[str, Any]]) -> np.ndarray:
        """
        Build vector index using sentence embeddings.

        Returns:
            embeddings: (N, D) numpy array of skill embeddings
        """
        logger.info("Building vector index...")

        # Prepare texts for embedding
        texts = [doc["skill_text"] for doc in skill_docs]

        # Generate embeddings in batches
        batch_size = self.config["embedding"]["batch_size"]
        embeddings = self.embedding_model.encode(
            texts,
            batch_size=batch_size,
            show_progress_bar=True,
            convert_to_numpy=True,
            normalize_embeddings=self.config["embedding"]["normalize"],
        )

        logger.info(f"Generated embeddings: shape={embeddings.shape}")

        return embeddings

    def compute_metadata_hash(self, skill_docs: list[dict[str, Any]]) -> str:
        """
        Compute a hash of the skill documents for reproducibility tracking.
        """
        # Use skill IDs and first few chars of descriptions
        content = "\n".join(
            [
                f"{doc['skill_id']}:{doc['description'][:50]}"
                for doc in skill_docs[:100]  # Sample first 100
            ]
        )
        return hashlib.sha256(content.encode()).hexdigest()[:16]

    def save_indexes(self, output_dir: Path, skill_docs: list[dict[str, Any]], bm25_index: BM25Okapi, embeddings: np.ndarray):
        """
        Save all indexes and metadata to disk.
        """
        output_dir.mkdir(parents=True, exist_ok=True)

        # Save skill documents (metadata)
        save_json(skill_docs, output_dir / "skill_docs.json")

        # Save BM25 index (pickle)
        logger.info("Saving BM25 index...")
        with open(output_dir / "bm25_index.pkl", "wb") as f:
            pickle.dump(bm25_index, f)

        # Save vector embeddings (numpy)
        logger.info("Saving vector embeddings...")
        np.save(output_dir / "embeddings.npy", embeddings)

        # Save metadata for reproducibility
        metadata = {
            "num_skills": len(skill_docs),
            "embedding_model": self.config["embedding"]["model_name"],
            "embedding_dim": embeddings.shape[1],
            "normalize": self.config["embedding"]["normalize"],
            "preprocessing": self.config["preprocessing"],
            "data_hash": self.compute_metadata_hash(skill_docs),
        }
        save_json(metadata, output_dir / "index_metadata.json")

        logger.info(f"All indexes saved to {output_dir}")

    def build(self, skills_db_path: Path, output_dir: Path):
        """
        Main build pipeline.
        """
        # Load skills
        skills = self.load_skills_db(skills_db_path)

        # Prepare skill documents
        skill_docs = self.prepare_skill_texts(skills)

        # Build indexes
        bm25_index = self.build_bm25_index(skill_docs)
        embeddings = self.build_vector_index(skill_docs)

        # Save everything
        self.save_indexes(output_dir, skill_docs, bm25_index, embeddings)

        logger.info("✓ Index building complete!")


def main():
    parser = argparse.ArgumentParser(description="Build skill indexes")
    parser.add_argument("--skills-db", type=str, required=True, help="Path to skills database JSON file")
    parser.add_argument("--config", type=str, default="config.yaml", help="Path to config file")
    parser.add_argument("--output-dir", type=str, default="indexes", help="Output directory for indexes")

    args = parser.parse_args()

    # Load config
    with open(args.config) as f:
        config = yaml.safe_load(f)

    # Build indexes
    builder = SkillIndexBuilder(config)
    builder.build(skills_db_path=Path(args.skills_db), output_dir=Path(args.output_dir))


if __name__ == "__main__":
    main()
