"""
Hybrid skill selector: skill_matcher (retrieval) + LLM (preliminary + quality check)

Stage 1: skill_matcher does BM25 + Vector search -> top 50 candidates
Stage 2: LLM preliminary selection (lenient) -> 10-12 candidates
Stage 3: Load actual skill content (SKILL.md)
Stage 4: LLM quality check (strict) -> 3-6 high-quality skills
"""
import hashlib
import logging
import json
import time
import yaml
import requests
from pathlib import Path
from typing import Dict, List, Any, Tuple
from urllib.parse import urlparse
import anthropic
from skill_matcher.match_task_skills import TaskSkillMatcher
from skill_matcher.build_skill_indexes import SkillIndexBuilder
from skill_matcher.utils import load_json

logger = logging.getLogger("HybridSelector")


class HybridSkillSelector:
    """
    Combines skill_matcher (high recall) with LLM (high precision).
    """

    def __init__(self, skills_db_path: Path, skill_matcher_indexes: Path):
        """
        Initialize both components.

        Args:
            skills_db_path: Path to all_skills_combined.json
            skill_matcher_indexes: Path to skill_matcher/indexes directory
        """
        self.skills_db_path = skills_db_path
        self.skill_matcher_indexes = skill_matcher_indexes

        # Load config for skill_matcher
        config_path = skill_matcher_indexes.parent / "config.yaml"
        with open(config_path, "r") as f:
            self.config = yaml.safe_load(f)

        # Load full skills database for metadata
        with open(skills_db_path, "r") as f:
            self.skills_db = json.load(f)

        # Check if indexes need to be rebuilt
        if self._should_rebuild_indexes():
            self._rebuild_indexes()

        # Initialize skill_matcher
        self.matcher = TaskSkillMatcher(self.config, skill_matcher_indexes)

        # Initialize Anthropic client for final selection
        self.client = anthropic.Anthropic()

        # Build skill name -> full skill mapping
        self.skill_map = {s["name"]: s for s in self.skills_db}

        # Track skills that fail to load (for cleanup)
        self.failed_skills = set()

    def _compute_skills_hash(self) -> str:
        """Compute hash of skills database for change detection."""
        content = json.dumps(
            [{"name": s.get("name"), "desc": s.get("description", "")[:50]}
             for s in self.skills_db[:100]],
            sort_keys=True
        )
        return hashlib.sha256(content.encode()).hexdigest()[:16]

    def _should_rebuild_indexes(self) -> bool:
        """Check if indexes exist and match current skills database."""
        metadata_path = self.skill_matcher_indexes / "index_metadata.json"

        # Check if index files exist
        required_files = ["bm25_index.pkl", "embeddings.npy", "skill_docs.json", "index_metadata.json"]
        for f in required_files:
            if not (self.skill_matcher_indexes / f).exists():
                logger.info(f"Index file missing: {f}, need to rebuild")
                return True

        # Check if skills database has changed
        try:
            with open(metadata_path, "r") as f:
                metadata = json.load(f)

            current_hash = self._compute_skills_hash()
            stored_hash = metadata.get("data_hash", "")

            if current_hash != stored_hash:
                logger.info(f"Skills database changed (hash mismatch), need to rebuild")
                return True

            # Also check count
            if metadata.get("num_skills", 0) != len(self.skills_db):
                logger.info(f"Skills count changed ({metadata.get('num_skills')} -> {len(self.skills_db)}), need to rebuild")
                return True

        except Exception as e:
            logger.warning(f"Error reading index metadata: {e}, need to rebuild")
            return True

        logger.info("Indexes are up to date")
        return False

    def _rebuild_indexes(self):
        """Rebuild skill indexes from current skills database."""
        logger.info("Rebuilding skill indexes...")

        builder = SkillIndexBuilder(self.config)
        builder.build(
            skills_db_path=self.skills_db_path,
            output_dir=self.skill_matcher_indexes
        )

        logger.info("Skill indexes rebuilt successfully")

    def _retrieve_candidates(self, instruction: str, solution_content: str = "") -> List[Dict[str, Any]]:
        """
        Stage 1: Use skill_matcher to get top candidates.

        Returns:
            List of candidate skills with RRF scores
        """
        # Build query text (simplified, similar to skill_matcher)
        query_text = f"{instruction}\n{solution_content[:1000]}"
        query_tokens = self.matcher.preprocessor.tokenize(query_text)

        # Retrieve from BM25
        topn_bm25 = 100  # Adjust as needed
        bm25_results = self.matcher.retrieve_bm25(query_text, topn_bm25)

        # Retrieve from vector index
        topn_vec = 100
        vec_results = self.matcher.retrieve_vector(query_text, topn_vec)

        # Fuse with RRF, take top 30-50
        topk = 50  # Adjust: more candidates = better recall, slower LLM
        candidates = self.matcher.fuse_with_rrf(bm25_results, vec_results, topk)

        # Add explainability
        candidates = self.matcher.add_explainability(candidates, query_text, query_tokens)

        logger.info(f"Retrieved {len(candidates)} candidates from skill_matcher")
        return candidates

    def _llm_preliminary_selection(self, instruction: str,
                            candidates: List[Dict[str, Any]], max_skills: int = 10) -> List[str]:
        """
        Stage 2: Preliminary LLM selection - cast a wider net for quality check.

        This stage is MORE LENIENT - selects 8-10 candidates for further quality review.
        The strict filtering happens in Stage 4 (quality check with actual content).

        Returns:
            List of selected skill names (8-10 candidates)
        """
        # Build candidate list for LLM
        candidate_list_str = ""
        for i, c in enumerate(candidates):
            skill_name = c["skill_name"]
            rrf_score = c["rrf_score"]
            evidence = c.get("evidence", {})
            matched_terms = evidence.get("matched_terms", [])
            snippet = evidence.get("snippet", "")[:150]

            candidate_list_str += f"{i+1}. {skill_name} (score: {rrf_score:.3f})\n"
            if matched_terms:
                candidate_list_str += f"   Matched: {', '.join(matched_terms[:5])}\n"
            if snippet:
                candidate_list_str += f"   Snippet: {snippet}...\n"

        prompt = f"""Task Instruction:
{instruction}

Available Skills (Top {len(candidates)} candidates from hybrid search):
{candidate_list_str}

Select **10-12 potentially relevant skills** for this task. This is a PRELIMINARY selection -
be INCLUSIVE rather than strict. A subsequent quality check will filter based on actual content.

SELECTION CRITERIA (be lenient at this stage):
1. **Relevance**: Skill name/description seems related to the task domain
2. **Domain Coverage**: Include skills that might provide useful domain knowledge
3. **Breadth**: When in doubt, INCLUDE the skill - better to have false positives than miss good skills

EXCLUDE only if:
- Skill is clearly unrelated to the task (e.g., "kubernetes" for a pure text processing task)
- Skill name suggests it's overly task-specific (e.g., "solve-this-exact-problem")

IMPORTANT:
- Select up to {max_skills} skills (or fewer if truly not enough relevant candidates)
- This is preliminary filtering - err on the side of inclusion
- A stricter quality check with actual skill content will follow

Return ONLY a JSON array of skill names, e.g. ["git", "docker", "nginx", "ssl", "bash-scripting"].
"""

        try:
            message = self.client.messages.create(
                model="claude-sonnet-4-20250514",
                max_tokens=1024,
                messages=[{"role": "user", "content": prompt}]
            )
            response_text = message.content[0].text

            # Parse JSON from response
            import re
            json_match = re.search(r'\[.*\]', response_text, re.DOTALL)
            if json_match:
                selected_names = json.loads(json_match.group(0))
                logger.info(f"[Stage 2] Preliminary selection: {len(selected_names)} skills: {selected_names}")
                return selected_names
            else:
                logger.warning("Could not parse JSON from LLM response")
                # Fallback: take top N by RRF score
                return [c["skill_name"] for c in candidates[:max_skills]]

        except Exception as e:
            logger.error(f"LLM preliminary selection failed: {e}")
            # Fallback: take top N by RRF score
            return [c["skill_name"] for c in candidates[:max_skills]]

    def _load_skill_content(self, skill_names: List[str]) -> List[Dict[str, Any]]:
        """
        Stage 3: Load actual skill content (SKILL.md) for quality check.

        Args:
            skill_names: List of skill names from preliminary selection

        Returns:
            List of dicts with skill metadata + actual content
        """
        skills_with_content = []

        for name in skill_names:
            skill_info = self.skill_map.get(name)
            if not skill_info:
                logger.warning(f"Skill '{name}' not found in skill_map, skipping")
                self.failed_skills.add(name)
                continue

            content = None

            # Try local path first
            skill_path = skill_info.get("path")
            if skill_path:
                # Try multiple possible locations
                repo_root = self.skills_db_path.parent.parent
                possible_paths = [
                    repo_root / skill_path,
                    repo_root / "skills" / name / "SKILL.md",
                ]

                for local_file in possible_paths:
                    if local_file.exists():
                        try:
                            content = local_file.read_text(encoding="utf-8")
                            logger.debug(f"Loaded content from local: {local_file}")
                            break
                        except Exception as e:
                            logger.warning(f"Failed to read {local_file}: {e}")

            # Try GitHub if local not found
            if not content and skill_info.get("url"):
                content = self._fetch_skill_from_github(skill_info)

            if content:
                skills_with_content.append({
                    "name": name,
                    "description": skill_info.get("description", ""),
                    "url": skill_info.get("url", ""),
                    "path": skill_info.get("path", ""),
                    "content": content[:3000]  # Truncate for LLM context
                })
            else:
                logger.warning(f"Could not load content for skill '{name}'")
                self.failed_skills.add(name)

        logger.info(f"[Stage 3] Loaded content for {len(skills_with_content)}/{len(skill_names)} skills")
        return skills_with_content

    def _fetch_skill_from_github(self, skill_info: Dict[str, Any], max_retries: int = 3) -> str:
        """
        Fetch SKILL.md content from GitHub with exponential backoff retry.

        Args:
            skill_info: Skill metadata containing url and path
            max_retries: Maximum number of retry attempts (default 3)

        Returns:
            Skill content string or None if fetch failed
        """
        url = skill_info.get("url", "")
        skill_path = skill_info.get("path", "")

        if not url:
            return None

        # Parse GitHub URL
        parsed = urlparse(url)
        path_parts = parsed.path.strip("/").split("/")
        if len(path_parts) < 2:
            return None

        owner = path_parts[0]
        repo = path_parts[1]

        # If skill_path is empty, extract from URL
        # URL format: https://github.com/owner/repo/tree/branch/path/to/skill
        branch = "main"
        if not skill_path:
            if "tree" in path_parts:
                idx = path_parts.index("tree")
                if idx + 1 < len(path_parts):
                    branch = path_parts[idx + 1]
                if idx + 2 < len(path_parts):
                    skill_path = "/".join(path_parts[idx + 2:])
            elif "blob" in path_parts:
                idx = path_parts.index("blob")
                if idx + 1 < len(path_parts):
                    branch = path_parts[idx + 1]
                if idx + 2 < len(path_parts):
                    skill_path = "/".join(path_parts[idx + 2:])

        if not skill_path:
            return None

        # Construct raw content URL for SKILL.md
        raw_url = f"https://raw.githubusercontent.com/{owner}/{repo}/{branch}/{skill_path}/SKILL.md"

        # Retry with exponential backoff for transient errors
        retryable_status_codes = {429, 500, 502, 503, 504}

        for attempt in range(max_retries):
            try:
                response = requests.get(raw_url, timeout=10)

                if response.status_code == 200:
                    return response.text
                elif response.status_code in retryable_status_codes:
                    # Transient error, retry with backoff
                    if attempt < max_retries - 1:
                        wait_time = 2 ** attempt  # 1s, 2s, 4s
                        logger.debug(f"GitHub fetch got {response.status_code}, retrying in {wait_time}s...")
                        time.sleep(wait_time)
                        continue
                    else:
                        logger.debug(f"GitHub fetch failed after {max_retries} retries: HTTP {response.status_code}")
                        return None
                else:
                    # Non-retryable error (404, 403, etc.)
                    logger.debug(f"GitHub fetch failed for {raw_url}: HTTP {response.status_code}")
                    return None

            except requests.exceptions.Timeout:
                if attempt < max_retries - 1:
                    wait_time = 2 ** attempt
                    logger.debug(f"GitHub fetch timeout, retrying in {wait_time}s...")
                    time.sleep(wait_time)
                    continue
                else:
                    logger.debug(f"GitHub fetch failed after {max_retries} retries: timeout")
                    return None
            except requests.exceptions.RequestException as e:
                if attempt < max_retries - 1:
                    wait_time = 2 ** attempt
                    logger.debug(f"GitHub fetch error: {e}, retrying in {wait_time}s...")
                    time.sleep(wait_time)
                    continue
                else:
                    logger.debug(f"GitHub fetch failed after {max_retries} retries: {e}")
                    return None

        return None

    def _check_skill_quality(self, instruction: str,
                            skills_with_content: List[Dict[str, Any]],
                            max_final_skills: int = 6) -> Tuple[List[Dict[str, Any]], List[Dict[str, Any]]]:
        """
        Stage 4: Strict quality check - LLM reviews actual skill content.

        Args:
            instruction: Task instruction
            skills_with_content: Skills with loaded content from Stage 3
            max_final_skills: Maximum number of skills to return (default 6)

        Returns:
            Tuple of (qualified_skills, rejected_skills)
        """
        if not skills_with_content:
            return [], []

        # Build skill content for LLM review
        skills_content_str = ""
        for i, skill in enumerate(skills_with_content):
            content_preview = skill.get("content", "")[:1500]  # Limit per skill
            skills_content_str += f"""
--- Skill {i+1}: {skill['name']} ---
Description: {skill.get('description', 'N/A')}
Content Preview:
{content_preview}
---
"""

        prompt = f"""Task Instruction:
{instruction}

Review the following {len(skills_with_content)} skills and select only HIGH-QUALITY, RELEVANT ones.
This is the FINAL selection - be STRICT.

{skills_content_str}

STRICT QUALITY CRITERIA - A skill must meet ALL of these:

1. **Real Content**: NOT a placeholder, template, or auto-generated stub
   - REJECT if contains: "TODO", "Example skill", "Description here", "placeholder"
   - REJECT if content is too short (<100 chars) or circular

2. **Provides Specialized Knowledge**: Knowledge the AI agent doesn't already have
   - REJECT generic skills like "bash-basics", "git-commands" (agent knows these)
   - KEEP skills with domain-specific knowledge, workflows, or tool configurations

3. **Relevant to Task**: Actually helps complete THIS specific task
   - REJECT if skill domain doesn't match task requirements
   - KEEP if skill provides directly applicable knowledge

4. **No Answer Leakage**: Doesn't give away the solution
   - REJECT if skill basically solves the task step-by-step
   - KEEP if skill provides general domain knowledge

5. **Actionable**: Contains concrete commands, examples, or workflows
   - REJECT vague or theoretical-only content
   - KEEP skills with specific, executable guidance

OUTPUT FORMAT:
Return a JSON object with two arrays:
{{
  "qualified": ["skill1", "skill2", ...],  // Skills that pass ALL criteria (max {max_final_skills})
  "rejected": [
    {{"name": "skill3", "reason": "placeholder content"}},
    {{"name": "skill4", "reason": "too generic, agent already knows this"}}
  ]
}}

Be strict - it's better to have 2-3 high-quality skills than 6 mediocre ones.
"""

        try:
            message = self.client.messages.create(
                model="claude-sonnet-4-20250514",
                max_tokens=2048,
                messages=[{"role": "user", "content": prompt}]
            )
            response_text = message.content[0].text

            # Parse JSON from response
            import re
            json_match = re.search(r'\{.*\}', response_text, re.DOTALL)
            if json_match:
                result = json.loads(json_match.group(0))
                qualified_names = result.get("qualified", [])
                rejected_info = result.get("rejected", [])

                # Map back to full skill objects
                qualified_skills = [s for s in skills_with_content if s["name"] in qualified_names]
                rejected_skills = [{"skill": s, "reason": next((r.get("reason", "unknown") for r in rejected_info if r.get("name") == s["name"]), "unknown")}
                                   for s in skills_with_content if s["name"] not in qualified_names]

                logger.info(f"[Stage 4] Quality check: {len(qualified_skills)} qualified, {len(rejected_skills)} rejected")
                for r in rejected_info:
                    logger.info(f"  Rejected '{r.get('name')}': {r.get('reason')}")

                return qualified_skills[:max_final_skills], rejected_skills
            else:
                logger.warning("Could not parse JSON from quality check response")
                # Quality check failed - return empty to avoid bypassing filter
                return [], []

        except json.JSONDecodeError as e:
            logger.error(f"LLM quality check failed - JSON parse error: {e}")
            # JSON parsing failed, return empty to avoid passing unverified skills
            return [], []
        except Exception as e:
            logger.error(f"LLM quality check failed: {e}")
            # Fail safe: return empty list instead of all skills to avoid bypassing quality filter
            return [], []

    def select_skills(self, instruction: str, solution_content: str = "",
                     max_skills: int = 6) -> Tuple[List[Dict[str, Any]], Dict[str, Any]]:
        """
        Main entry point: 4-stage skill selection pipeline.

        Stage 1: BM25 + Vector retrieval -> 50 candidates
        Stage 2: LLM preliminary selection (lenient) -> 8-10 candidates
        Stage 3: Load actual skill content
        Stage 4: LLM quality check (strict) -> 3-6 final skills

        Args:
            instruction: Task instruction text
            solution_content: Solution script content (should be empty to prevent answer leakage)
            max_skills: Maximum number of skills to select (default 6)

        Returns:
            Tuple of (selected_skills, pipeline_stats)
            - selected_skills: List of selected skill dictionaries
            - pipeline_stats: Dict with counts at each stage for statistics
        """
        # Initialize pipeline statistics
        pipeline_stats = {
            "stage1_retrieved": 0,
            "stage2_preliminary": 0,
            "stage3_content_loaded": 0,
            "stage4_qualified": 0,
            "stage4_rejected": 0,
            "rejected_details": []
        }

        # Stage 1: Retrieve candidates using skill_matcher (BM25 + Vector + RRF)
        candidates = self._retrieve_candidates(instruction, solution_content)
        pipeline_stats["stage1_retrieved"] = len(candidates)

        if not candidates:
            logger.warning("No candidates found by skill_matcher")
            return [], pipeline_stats

        # Stage 2: LLM preliminary selection (lenient, 10-12 candidates)
        preliminary_names = self._llm_preliminary_selection(
            instruction, candidates, max_skills=12
        )
        pipeline_stats["stage2_preliminary"] = len(preliminary_names)

        if not preliminary_names:
            logger.warning("No skills selected in preliminary stage")
            return [], pipeline_stats

        # Stage 3: Load actual skill content (SKILL.md)
        skills_with_content = self._load_skill_content(preliminary_names)
        pipeline_stats["stage3_content_loaded"] = len(skills_with_content)

        if not skills_with_content:
            logger.warning("Could not load content for any skills")
            return [], pipeline_stats

        # Stage 4: LLM quality check (strict, 3-6 final skills)
        qualified_skills, rejected_skills = self._check_skill_quality(
            instruction, skills_with_content, max_final_skills=max_skills
        )
        pipeline_stats["stage4_qualified"] = len(qualified_skills)
        pipeline_stats["stage4_rejected"] = len(rejected_skills)
        pipeline_stats["rejected_details"] = [
            {"name": r["skill"]["name"], "reason": r["reason"]}
            for r in rejected_skills
        ]

        # Remove content field from final output (too large)
        final_skills = []
        for skill in qualified_skills:
            final_skills.append({
                "name": skill["name"],
                "description": skill.get("description", ""),
                "url": skill.get("url", ""),
                "path": skill.get("path", "")
            })

        logger.info(f"[Final] Pipeline complete: {pipeline_stats['stage1_retrieved']} -> "
                   f"{pipeline_stats['stage2_preliminary']} -> {pipeline_stats['stage3_content_loaded']} -> "
                   f"{pipeline_stats['stage4_qualified']} skills")
        logger.info(f"[Final] Selected skills: {[s['name'] for s in final_skills]}")

        return final_skills, pipeline_stats

    def cleanup_failed_skills(self) -> int:
        """
        Remove failed skills from the database file.
        Call this after processing all tasks.

        Returns:
            Number of skills removed
        """
        if not self.failed_skills:
            logger.info("No failed skills to clean up")
            return 0

        logger.info(f"Cleaning up {len(self.failed_skills)} failed skills from database...")

        # Remove from in-memory database
        original_count = len(self.skills_db)
        self.skills_db = [s for s in self.skills_db if s["name"] not in self.failed_skills]
        removed_count = original_count - len(self.skills_db)

        # Save back to file
        with open(self.skills_db_path, "w") as f:
            json.dump(self.skills_db, f, indent=2)

        # Update skill_map
        self.skill_map = {s["name"]: s for s in self.skills_db}

        logger.info(f"Removed {removed_count} invalid skills. New database size: {len(self.skills_db)}")
        logger.info(f"Failed skills: {list(self.failed_skills)}")

        # Clear the set
        self.failed_skills.clear()

        return removed_count
