#!/usr/bin/env python3
"""
Stage 1 only: Collect top-50 skills from embedding retrieval for all tasks,
then validate their GitHub URLs and remove invalid ones.

Usage:
    export GITHUB_TOKEN=xxx
    python collect_and_validate_skills.py terminal-bench-2.0/
"""
import json
import os
import sys
import yaml
import requests
from pathlib import Path
from concurrent.futures import ThreadPoolExecutor, as_completed
from urllib.parse import urlparse

REPO_ROOT = Path(__file__).resolve().parent.parent.parent
SKILLS_DB_PATH = REPO_ROOT / "docs" / "skills-research" / "all_skills_combined.json"
SKILL_MATCHER_INDEXES = Path(__file__).resolve().parent / "skill_matcher" / "indexes"


def load_tasks(source_path: Path) -> list:
    """Find all TB2 task directories."""
    tasks = []
    if source_path.is_dir():
        if (source_path / "task.toml").exists():
            tasks.append(source_path)
        else:
            for item in source_path.iterdir():
                if item.is_dir() and (item / "task.toml").exists():
                    tasks.append(item)
    return tasks


def collect_stage1_skills(tasks: list) -> set:
    """Run Stage 1 (embedding retrieval) for all tasks, collect unique skill names."""
    from skill_matcher.match_task_skills import TaskSkillMatcher

    # Load config
    config_path = SKILL_MATCHER_INDEXES.parent / "config.yaml"
    with open(config_path) as f:
        config = yaml.safe_load(f)

    matcher = TaskSkillMatcher(config, SKILL_MATCHER_INDEXES)

    all_skills = set()

    for i, task_path in enumerate(tasks):
        task_id = task_path.name
        instruction_file = task_path / "instruction.md"

        if not instruction_file.exists():
            print(f"[{i+1}/{len(tasks)}] {task_id}: no instruction.md, skipping")
            continue

        instruction = instruction_file.read_text()

        # Stage 1: BM25 + Vector retrieval
        query_text = instruction[:2000]  # Truncate for efficiency

        bm25_results = matcher.retrieve_bm25(query_text, topn=100)
        vec_results = matcher.retrieve_vector(query_text, topn=100)
        candidates = matcher.fuse_with_rrf(bm25_results, vec_results, topk=50)

        task_skills = {c["skill_name"] for c in candidates}
        all_skills.update(task_skills)

        print(f"[{i+1}/{len(tasks)}] {task_id}: {len(candidates)} candidates, total unique: {len(all_skills)}")

    return all_skills


def build_raw_url(url: str) -> str:
    """Convert GitHub tree URL to raw SKILL.md URL."""
    parsed = urlparse(url)
    parts = parsed.path.strip("/").split("/")

    if len(parts) < 2:
        return None

    owner = parts[0]
    repo = parts[1]

    if "tree" in parts:
        idx = parts.index("tree")
        branch = parts[idx + 1] if idx + 1 < len(parts) else "main"
        skill_path = "/".join(parts[idx + 2:]) if idx + 2 < len(parts) else ""
    elif "blob" in parts:
        idx = parts.index("blob")
        branch = parts[idx + 1] if idx + 1 < len(parts) else "main"
        skill_path = "/".join(parts[idx + 2:]) if idx + 2 < len(parts) else ""
    else:
        branch = "main"
        skill_path = "/".join(parts[2:]) if len(parts) > 2 else ""

    if not skill_path:
        return None

    return f"https://raw.githubusercontent.com/{owner}/{repo}/{branch}/{skill_path}/SKILL.md"


def validate_skill(skill_info: dict, headers: dict) -> dict:
    """Check if skill URL is valid. Returns None if valid, error info if invalid."""
    name = skill_info.get("name", "unknown")
    url = skill_info.get("url", "")

    if not url or "github.com" not in url:
        return {"name": name, "reason": "no github url"}

    raw_url = build_raw_url(url)
    if not raw_url:
        return {"name": name, "url": url, "reason": "could not parse URL"}

    try:
        r = requests.head(raw_url, headers=headers, timeout=10, allow_redirects=True)
        if r.status_code == 404:
            return {"name": name, "url": url, "reason": "404"}
        elif r.status_code == 403:
            return {"name": name, "url": url, "reason": "rate_limited"}
        return None  # Valid
    except requests.Timeout:
        return {"name": name, "url": url, "reason": "timeout"}
    except Exception as e:
        return {"name": name, "url": url, "reason": str(e)[:50]}


def validate_skills(skill_names: set, skills_db: list, headers: dict) -> list:
    """Validate GitHub URLs for given skill names."""
    # Build name -> skill mapping
    skill_map = {}
    for s in skills_db:
        if s["name"] in skill_names:
            skill_map[s["name"]] = s

    print(f"\nValidating {len(skill_map)} skills...")

    invalid = []
    checked = 0

    with ThreadPoolExecutor(max_workers=20) as executor:
        futures = {executor.submit(validate_skill, s, headers): s["name"] for s in skill_map.values()}

        for future in as_completed(futures):
            checked += 1
            result = future.result()
            if result:
                invalid.append(result)
                print(f"  INVALID: {result['name']} - {result['reason']}")
            elif checked % 50 == 0:
                print(f"  [{checked}/{len(skill_map)}] checked... ({len(invalid)} invalid)")

    return invalid


def remove_invalid_skills(skills_db: list, invalid_names: set) -> list:
    """Remove invalid skills from database."""
    return [s for s in skills_db if s["name"] not in invalid_names]


def main():
    if len(sys.argv) < 2:
        print("Usage: python collect_and_validate_skills.py <source_path>")
        print("Example: python collect_and_validate_skills.py terminal-bench-2.0/")
        sys.exit(1)

    source_path = Path(sys.argv[1])
    if not source_path.exists():
        print(f"Error: {source_path} does not exist")
        sys.exit(1)

    # Setup GitHub headers
    headers = {}
    token = os.environ.get("GITHUB_TOKEN")
    if token:
        headers["Authorization"] = f"token {token}"
        print("Using GITHUB_TOKEN (5000 req/hour)")
    else:
        print("WARNING: No GITHUB_TOKEN (60 req/hour limit)")

    # Load skills database
    with open(SKILLS_DB_PATH) as f:
        skills_db = json.load(f)
    print(f"Loaded {len(skills_db)} skills from database")

    # Step 1: Load tasks
    tasks = load_tasks(source_path)
    print(f"Found {len(tasks)} tasks")

    # Step 2: Collect Stage 1 skills
    print("\n" + "="*60)
    print("STAGE 1: Collecting candidate skills from embedding retrieval")
    print("="*60)
    skill_names = collect_stage1_skills(tasks)
    print(f"\nTotal unique skills from Stage 1: {len(skill_names)}")

    # Save collected skills
    output_dir = Path(__file__).parent
    collected_file = output_dir / "collected_skill_names.json"
    with open(collected_file, "w") as f:
        json.dump(list(skill_names), f, indent=2)
    print(f"Saved to: {collected_file}")

    # Step 3: Validate URLs
    print("\n" + "="*60)
    print("STAGE 2: Validating GitHub URLs")
    print("="*60)
    invalid_skills = validate_skills(skill_names, skills_db, headers)

    invalid_names = {s["name"] for s in invalid_skills}
    print(f"\nResults: {len(invalid_skills)} invalid out of {len(skill_names)}")

    # Save invalid list
    invalid_file = output_dir / "invalid_skills.json"
    with open(invalid_file, "w") as f:
        json.dump(invalid_skills, f, indent=2)
    print(f"Invalid skills saved to: {invalid_file}")

    # Step 4: Ask to clean
    if invalid_skills:
        print("\n" + "="*60)
        print(f"Found {len(invalid_skills)} invalid skills. Remove from database?")
        print("="*60)
        response = input("Type 'yes' to remove, anything else to skip: ")

        if response.strip().lower() == "yes":
            cleaned_db = remove_invalid_skills(skills_db, invalid_names)

            # Backup original
            backup_file = SKILLS_DB_PATH.with_suffix(".json.backup")
            with open(backup_file, "w") as f:
                json.dump(skills_db, f)
            print(f"Backup saved to: {backup_file}")

            # Save cleaned
            with open(SKILLS_DB_PATH, "w") as f:
                json.dump(cleaned_db, f, indent=2)
            print(f"Removed {len(invalid_names)} invalid skills")
            print(f"New database size: {len(cleaned_db)}")
        else:
            print("Skipped cleaning. You can manually remove later.")

    print("\nDone!")


if __name__ == "__main__":
    main()
