import json
import logging
import os
import re
from pathlib import Path
from typing import Any
from urllib.parse import urlparse

import anthropic
import requests
from sklearn.feature_extraction.text import TfidfVectorizer
from sklearn.metrics.pairwise import cosine_similarity

logger = logging.getLogger("SkillsSelector")


class SkillsSelector:
    def __init__(self, db_path: Path, enable_low_quality_filter: bool = True):
        self.db_path = db_path
        self.skills_db: list[dict[str, Any]] = []
        self.enable_low_quality_filter = enable_low_quality_filter
        self.low_quality_blacklist: set[tuple[str, str]] = set()  # (skill_name, source) tuples

        self._load_db()
        self._load_low_quality_blacklist()
        self.client = anthropic.Anthropic()

        # Initialize TF-IDF
        self.vectorizer = TfidfVectorizer(stop_words="english", max_features=10000)
        self.skill_vectors = None
        self._prepare_tfidf_index()

    def _load_db(self):
        if self.db_path.exists():
            try:
                with open(self.db_path) as f:
                    self.skills_db = json.load(f)
                logger.info(f"Loaded {len(self.skills_db)} skills from {self.db_path}")
            except Exception as e:
                logger.error(f"Failed to load skills DB: {e}")
        else:
            logger.warning(f"Skills DB not found at {self.db_path}")

    def _load_low_quality_blacklist(self):
        """
        Load low-quality skills blacklist from low_quality_sources.json.
        Creates a set of (skill_name, source) tuples for fast lookup.
        """
        if not self.enable_low_quality_filter:
            logger.info("Low-quality filtering disabled")
            return

        low_quality_file = self.db_path.parent / "low_quality_sources.json"

        if not low_quality_file.exists():
            logger.info("No low-quality sources file found - starting fresh")
            return

        try:
            with open(low_quality_file) as f:
                content = f.read().strip()
                if not content:
                    return

                records = json.loads(content)

                for record in records:
                    skill_name = record.get("skill_name", "")
                    url = record.get("url", "")

                    # Extract source from URL
                    source = "unknown"
                    if url and "github.com" in url:
                        parts = url.split("github.com/")[1].split("/")
                        if len(parts) >= 2:
                            source = f"{parts[0]}/{parts[1]}"

                    if skill_name and source:
                        self.low_quality_blacklist.add((skill_name, source))

                if self.low_quality_blacklist:
                    logger.info(f"Loaded {len(self.low_quality_blacklist)} low-quality skills to filter")
                    logger.debug(f"Blacklisted: {list(self.low_quality_blacklist)[:5]}...")

        except Exception as e:
            logger.warning(f"Failed to load low-quality blacklist: {e}")

    def _prepare_tfidf_index(self):
        """
        Create TF-IDF index for all skills based on name and description.
        """
        if not self.skills_db:
            return

        logger.info("Building TF-IDF index for skills...")
        # Combine name and description for each skill
        corpus = [f"{s.get('name', '')} {s.get('name', '')} {s.get('description', '')}" for s in self.skills_db]
        try:
            self.skill_vectors = self.vectorizer.fit_transform(corpus)
            logger.info("TF-IDF index built successfully.")
        except Exception as e:
            logger.error(f"Failed to build TF-IDF index: {e}")

    def _pre_filter_skills(self, query_text: str, limit: int = 50) -> list[dict[str, Any]]:
        """
        Stage 1: TF-IDF based semantic retrieval to select candidates.
        """
        if self.skill_vectors is None:
            return []

        try:
            # Vectorize query
            query_vec = self.vectorizer.transform([query_text])

            # Calculate similarity
            similarities = cosine_similarity(query_vec, self.skill_vectors).flatten()

            # Get top indices
            top_indices = similarities.argsort()[-limit:][::-1]

            # Filter out zero similarity results and blacklisted skills
            candidates = []
            filtered_count = 0

            for idx in top_indices:
                if similarities[idx] > 0.05:  # Minimal threshold to avoid noise
                    skill = self.skills_db[idx].copy()
                    skill["similarity_score"] = float(similarities[idx])

                    # Check if skill is blacklisted
                    if self.enable_low_quality_filter and self._is_blacklisted(skill):
                        filtered_count += 1
                        logger.debug(f"Filtered out blacklisted skill: {skill['name']}")
                        continue

                    candidates.append(skill)

            if filtered_count > 0:
                logger.info(f"Filtered out {filtered_count} known low-quality skills in Stage 1")

            return candidates
        except Exception as e:
            logger.error(f"TF-IDF retrieval failed: {e}")
            return []

    def _is_blacklisted(self, skill: dict[str, Any]) -> bool:
        """
        Check if a skill is in the low-quality blacklist.
        Matches by (skill_name, source) tuple.
        """
        if not self.low_quality_blacklist:
            return False

        skill_name = skill.get("name", "")
        url = skill.get("url", "")

        # Extract source from URL
        source = "unknown"
        if url and "github.com" in url:
            parts = url.split("github.com/")[1].split("/")
            if len(parts) >= 2:
                source = f"{parts[0]}/{parts[1]}"

        return (skill_name, source) in self.low_quality_blacklist

    def _is_truly_low_quality(self, reasoning: str) -> bool:
        """
        Determine if the reasoning indicates truly low-quality skills (unavailable/broken)
        vs just task-specific irrelevance.

        Returns True if skills should be permanently blacklisted, False otherwise.
        """
        reasoning_lower = reasoning.lower()

        # Indicators of truly unavailable/broken content (should be recorded)
        truly_bad_indicators = [
            "content unavailable",
            "404",
            "[content not available]",
            "download failed",
            "no content available",
            "has no content",
            "truncated",
            "incomplete",
            "cuts off",
            "cut off mid-sentence",
            "broken",
            "empty",
            "placeholder text",
            "auto-generated",
            "minimal content",
        ]

        if any(indicator in reasoning_lower for indicator in truly_bad_indicators):
            return True

        # Indicators of task-specific irrelevance (should NOT be recorded)
        task_specific_indicators = [
            "too generic",
            "more general",
            "not relevant",
            "is irrelevant",
            "focuses on",
            "rather than",
            "more specific",
            "better suited",
            "less specific",
            "not needed",
            "is about",
            "is for",
            "wrong purpose",
            "not applicable",
            "only provides",
            "alternative",
            "replacing with",
            "chose",
            "selected",
            "keeping",
        ]

        if any(indicator in reasoning_lower for indicator in task_specific_indicators):
            return False

        # Default: don't record if unsure (conservative approach)
        return False

    def _record_low_quality_sources(self, skills: list[dict[str, Any]], low_quality_names: list[str], reasoning: str) -> None:
        """
        Record truly low-quality skill sources (404, unavailable, broken) to JSON file for future filtering.
        Does NOT record task-specific irrelevance.
        """
        # Only record if this is truly low-quality (not just task-specific irrelevance)
        if not self._is_truly_low_quality(reasoning):
            logger.debug("Skipping recording - reasoning indicates task-specific rejection, not permanent quality issue")
            return

        low_quality_file = self.db_path.parent / "low_quality_sources.json"

        # Load existing records
        existing_records = []
        if low_quality_file.exists():
            try:
                with open(low_quality_file) as f:
                    content = f.read().strip()
                    if content:  # Check if file is not empty
                        existing_records = json.loads(content)
            except Exception as e:
                logger.warning(f"Failed to load existing low quality records: {e}")
                existing_records = []

        # Extract low-quality skills info
        from datetime import datetime

        timestamp = datetime.now().isoformat()

        for skill in skills:
            if skill["name"] in low_quality_names:
                url = skill.get("url", "")

                # Extract source/repository from URL
                source = "unknown"
                if url and "github.com" in url:
                    parts = url.split("github.com/")[1].split("/")
                    if len(parts) >= 2:
                        source = f"{parts[0]}/{parts[1]}"  # owner/repo

                record = {
                    "skill_name": skill["name"],
                    "source": source,
                    "url": url,
                    "description": skill.get("description", "")[:150],
                    "reason": reasoning[:500],  # Increased from 200 to capture more context
                    "timestamp": timestamp,
                }

                # Avoid duplicates (same skill name from same source)
                is_duplicate = any(
                    r.get("skill_name") == record["skill_name"] and r.get("source") == record["source"] for r in existing_records
                )

                if not is_duplicate:
                    existing_records.append(record)
                    logger.info(f"Recorded truly low-quality source: {skill['name']} from {source}")

        # Save updated records
        try:
            with open(low_quality_file, "w") as f:
                json.dump(existing_records, f, indent=2)
        except Exception as e:
            logger.error(f"Failed to save low quality records: {e}")

    def _fetch_skill_content(self, skill: dict[str, Any]) -> str:
        """
        Fetch the actual skill content (SKILL.md) from GitHub URL.
        Returns the content as a string, or empty string if fetch fails.
        """
        url = skill.get("url", "")
        if not url or "github.com" not in url:
            return ""

        try:
            # Parse GitHub URL to get raw content URL
            parsed = urlparse(url)
            path_parts = parsed.path.strip("/").split("/")

            if len(path_parts) < 2:
                return ""

            owner = path_parts[0]
            repo = path_parts[1]

            # Handle tree/blob URLs
            if "tree" in path_parts or "blob" in path_parts:
                idx = path_parts.index("tree") if "tree" in path_parts else path_parts.index("blob")
                branch = path_parts[idx + 1] if idx + 1 < len(path_parts) else "main"
                skill_path = "/".join(path_parts[idx + 2 :]) if idx + 2 < len(path_parts) else ""
            else:
                branch = "main"
                skill_path = "/".join(path_parts[2:]) if len(path_parts) > 2 else ""

            # Construct raw URL for SKILL.md
            raw_url = f"https://raw.githubusercontent.com/{owner}/{repo}/{branch}/{skill_path}/SKILL.md"

            headers = {}
            github_token = os.environ.get("GITHUB_TOKEN")
            if github_token:
                headers["Authorization"] = f"token {github_token}"

            response = requests.get(raw_url, headers=headers, timeout=10)

            if response.status_code == 200:
                # Limit content to first 3000 chars for quality check
                return response.text[:3000]
            else:
                logger.debug(f"Failed to fetch skill content from {raw_url}: {response.status_code}")
                return ""

        except Exception as e:
            logger.debug(f"Error fetching skill content for {skill.get('name')}: {e}")
            return ""

    def _validate_skill_quality(self, skills: list[dict[str, Any]], candidates: list[dict[str, Any]], instruction: str) -> list[dict[str, Any]]:
        """
        Stage 3: Validate the quality of selected skills by examining their actual content.
        If low-quality skills are detected, replace them with better alternatives from candidates.
        """
        if not skills:
            return []

        # Fetch actual content for selected skills
        skills_with_content = []
        for skill in skills:
            content = self._fetch_skill_content(skill)
            skills_with_content.append(
                {
                    "name": skill["name"],
                    "description": skill.get("description", "")[:200],
                    "content_preview": content[:1000] if content else "[Content not available]",
                }
            )

        # Build validation prompt
        skills_info = ""
        for i, s in enumerate(skills_with_content, 1):
            skills_info += f"\n{i}. **{s['name']}**\n"
            skills_info += f"   Description: {s['description']}\n"
            skills_info += f"   Content Preview:\n   ```\n{s['content_preview']}\n   ```\n"

        # Get remaining candidates (not already selected)
        selected_names = {s["name"] for s in skills}
        remaining_candidates = [c for c in candidates if c["name"] not in selected_names]

        candidates_info = ""
        if remaining_candidates:
            candidates_info = "\n\nAlternative candidates available:\n"
            for i, c in enumerate(remaining_candidates[:10], 1):
                score = f"({c.get('similarity_score', 0):.2f})"
                candidates_info += f"{i}. {c['name']} {score}: {c.get('description', '')[:150]}\n"

        validation_prompt = f"""Task Context:
{instruction[:1500]}

Selected Skills to Validate:
{skills_info}
{candidates_info}

**Quality Validation Task:**

Evaluate each selected skill's quality and usefulness for this specific task. A HIGH-QUALITY skill should:
1. Provide CONCRETE, ACTIONABLE information (specific commands, code examples, configurations)
2. Contain SPECIALIZED knowledge not already known to a general AI (domain-specific tools, frameworks, workflows)
3. Be DIRECTLY relevant to the task at hand
4. Have SUBSTANTIAL content (not just vague descriptions or placeholder text)
5. Have ACTUAL CONTENT available (not "[Content not available]")

LOW-QUALITY indicators (MUST REJECT):
- **[Content not available]** - AUTOMATICALLY reject skills with no content
- Repetitive placeholder text ("helps with X", "provides guidance for X" without specifics)
- Only basic/general knowledge (basic SQL syntax, common CLI commands)
- Empty templates or auto-generated content
- Circular descriptions (skill name repeated without adding information)
- Very short content with no concrete examples

**CRITICAL RULE**: If a skill shows "[Content not available]", it MUST be marked as low-quality and rejected, regardless of how good the description sounds. No content = no value.

**Your Response:**

Respond with a JSON object:
{{
  "validated_skills": ["skill-1", "skill-2"],  // Names of HIGH-quality skills to keep (must have actual content)
  "low_quality_skills": ["bad-skill"],  // Names of LOW-quality skills to remove (including all "[Content not available]")
  "replacements": ["alternative-1"],  // Names from alternative candidates to add instead (if available)
  "reasoning": "Brief explanation of quality assessment"
}}

If ALL selected skills are low-quality AND no good alternatives exist, return empty arrays.
Skills with "[Content not available]" should NEVER appear in validated_skills.
"""

        try:
            message = self.client.messages.create(
                model="claude-sonnet-4-20250514", max_tokens=1024, messages=[{"role": "user", "content": validation_prompt}]
            )
            response_text = message.content[0].text

            # Parse JSON from response
            json_match = re.search(r"\{.*\}", response_text, re.DOTALL)
            if json_match:
                validation_result = json.loads(json_match.group(0))

                validated_names = validation_result.get("validated_skills", [])
                replacement_names = validation_result.get("replacements", [])
                low_quality_names = validation_result.get("low_quality_skills", [])
                reasoning = validation_result.get("reasoning", "")

                logger.info(f"Quality validation: {reasoning}")

                # Record low-quality skill sources
                if low_quality_names:
                    self._record_low_quality_sources(skills, low_quality_names, reasoning)

                # Build final skill list
                final_skills = []

                # Keep validated skills
                for skill in skills:
                    if skill["name"] in validated_names:
                        final_skills.append(skill)

                # Add replacements from candidates
                for candidate in candidates:
                    if candidate["name"] in replacement_names:
                        final_skills.append(candidate)

                # Deduplicate by skill name (keep first occurrence)
                seen_names = set()
                deduplicated_skills = []
                for skill in final_skills:
                    skill_name = skill.get("name")
                    if skill_name not in seen_names:
                        seen_names.add(skill_name)
                        deduplicated_skills.append(skill)

                if len(deduplicated_skills) != len(skills):
                    logger.info(f"Skills after quality validation: {[s['name'] for s in deduplicated_skills]}")

                return deduplicated_skills
            else:
                logger.warning("Could not parse JSON from quality validation response")
                return skills  # Keep original selection if validation fails

        except Exception as e:
            logger.error(f"Quality validation failed: {e}")
            return skills  # Keep original selection if validation fails

    def select_skills(self, instruction: str, solution_content: str = "") -> list[dict[str, Any]]:
        """
        Select best skills using a 3-stage pipeline:
        Stage 1: TF-IDF semantic retrieval
        Stage 2: LLM-based relevance selection
        Stage 3: Quality validation with actual skill content
        """
        # Stage 1: Prepare query text from context
        combined_text = f"{instruction}\n{solution_content}"

        # Stage 2: Get candidates via TF-IDF (Full database coverage)
        candidates = self._pre_filter_skills(combined_text, limit=30)

        if not candidates:
            logger.warning("No relevant skills found via TF-IDF.")
            return []

        # Stage 2: Ask Claude to pick the relevant ones
        candidate_list_str = ""
        for i, s in enumerate(candidates):
            score = f"({s.get('similarity_score', 0):.2f})"
            candidate_list_str += f"{i+1}. {s['name']} {score}: {s.get('description', '')[:200]}...\n"

        prompt = f"""Task Context:
{combined_text[:2000]}

Available Skills Candidates (with similarity scores):
{candidate_list_str}

Select 3-6 skills that would most help an AI agent complete this task successfully.
Aim for 3-6 skills, but you can select fewer (even 0) if no skills would significantly help.

Include a skill if it provides:
- Specialized tools, libraries, or frameworks (e.g. AMUSE, qutip, kubernetes CLI)
- Best practices, common pitfalls, or workflow guidance for complex processes
- Configuration examples or troubleshooting steps

Exclude skills for:
- Basic CLI tools (grep, ls, cat) unless the skill covers advanced usage patterns
- General programming concepts without specific tooling

Return ONLY a JSON array of skill names, e.g. ["skill-name-1", "skill-name-2", "skill-name-3"].
If no skills would significantly help, return [].
"""

        try:
            message = self.client.messages.create(
                model="claude-sonnet-4-20250514", max_tokens=1024, messages=[{"role": "user", "content": prompt}]
            )
            response_text = message.content[0].text
            # Parse JSON from response
            json_match = re.search(r"\[.*\]", response_text, re.DOTALL)
            if json_match:
                selected_names = json.loads(json_match.group(0))
                # Map back to skill objects
                selected_skills = [c for c in candidates if c["name"] in selected_names]
                logger.info(f"Stage 2 - Claude selected skills: {[s['name'] for s in selected_skills]}")

                # Stage 3: Validate skill quality by examining actual content
                if selected_skills:
                    validated_skills = self._validate_skill_quality(selected_skills, candidates, instruction)
                    return validated_skills
                else:
                    return []
            else:
                logger.warning("Could not parse JSON from Claude response")
                return candidates[:1]  # Fallback to top 1 semantic match

        except Exception as e:
            logger.error(f"LLM selection failed: {e}")
            return candidates[:1]  # Fallback
