#!/usr/bin/env python3
"""
Fetch and rank ICLR 2026 papers by average rating.
Uses the directReplies API to efficiently get papers with reviews.
"""

import json
import time

import requests

BASE_URL = "https://api2.openreview.net/notes"


def extract_value(field):
    """Extract value from potentially nested field."""
    if isinstance(field, dict) and "value" in field:
        return field["value"]
    return field


def extract_numeric_rating(rating_str):
    """Extract numeric rating from string like '8: Strong Accept'."""
    if rating_str is None:
        return None
    if isinstance(rating_str, (int, float)):
        return float(rating_str)
    if isinstance(rating_str, str):
        try:
            return float(rating_str.split(":")[0].strip())
        except:
            pass
    return None


def fetch_batch(offset, limit=10):
    """Fetch a batch of papers with their reviews."""
    params = {"content.venueid": "ICLR.cc/2026/Conference/Submission", "limit": limit, "offset": offset, "details": "directReplies"}

    try:
        response = requests.get(BASE_URL, params=params, timeout=60)
        response.raise_for_status()
        return response.json()
    except Exception as e:
        print(f"Error at offset {offset}: {e}")
        return None


def process_paper(paper):
    """Extract paper info and ratings from a paper with direct replies."""
    paper_id = paper.get("id") or paper.get("forum")
    paper_number = paper.get("number")
    content = paper.get("content", {})

    title = extract_value(content.get("title", "Unknown"))

    # Get reviews from directReplies
    direct_replies = paper.get("details", {}).get("directReplies", [])
    reviews = [r for r in direct_replies if any("Official_Review" in inv for inv in r.get("invitations", []))]

    ratings = []
    confidences = []
    soundness_scores = []

    for review in reviews:
        rcontent = review.get("content", {})

        # Extract rating
        rating = extract_value(rcontent.get("rating"))
        numeric_rating = extract_numeric_rating(rating)
        if numeric_rating is not None:
            ratings.append(numeric_rating)

        # Extract confidence
        conf = extract_value(rcontent.get("confidence"))
        conf_num = extract_numeric_rating(conf)
        if conf_num is not None:
            confidences.append(conf_num)

        # Extract soundness
        sound = extract_value(rcontent.get("soundness"))
        sound_num = extract_numeric_rating(sound)
        if sound_num is not None:
            soundness_scores.append(sound_num)

    if ratings:
        return {
            "id": paper_id,
            "number": paper_number,
            "title": title,
            "avg_rating": sum(ratings) / len(ratings),
            "ratings": ratings,
            "num_reviews": len(reviews),
            "avg_confidence": sum(confidences) / len(confidences) if confidences else None,
            "avg_soundness": sum(soundness_scores) / len(soundness_scores) if soundness_scores else None,
            "url": f"https://openreview.net/forum?id={paper_id}",
        }
    return None


def main():
    print("=" * 70)
    print("ICLR 2026 - Top Rated Papers Finder")
    print("=" * 70)

    # First, get total count
    print("\nGetting total submission count...")
    initial = fetch_batch(0, limit=1)
    if not initial:
        print("Failed to fetch initial data")
        return

    total = initial.get("count", 0)
    print(f"Total submissions: {total:,}")

    # Fetch all papers with reviews
    all_scored_papers = []
    batch_size = 10  # Small batches due to directReplies payload size
    offset = 0

    print(f"\nFetching papers with reviews (batch size: {batch_size})...")

    while offset < total:
        data = fetch_batch(offset, limit=batch_size)

        if not data or not data.get("notes"):
            print(f"  [!] Empty response at offset {offset}, retrying...")
            time.sleep(2)
            data = fetch_batch(offset, limit=batch_size)
            if not data or not data.get("notes"):
                offset += batch_size
                continue

        papers = data["notes"]

        for paper in papers:
            result = process_paper(paper)
            if result:
                all_scored_papers.append(result)

        if offset % 500 == 0:
            print(f"  Processed {offset}/{total} papers, {len(all_scored_papers)} with scores...")

        offset += batch_size
        time.sleep(0.3)  # Rate limiting

    print(f"\nTotal papers with scores: {len(all_scored_papers)}")

    # Sort by average rating
    all_scored_papers.sort(key=lambda x: x["avg_rating"], reverse=True)

    # Save results
    results = {
        "total_submissions": total,
        "papers_with_scores": len(all_scored_papers),
        "top_100": all_scored_papers[:100],
        "all_papers": all_scored_papers,
    }

    with open("iclr2026_ranked.json", "w") as f:
        json.dump(results, f, indent=2)

    # Print top 50
    print("\n" + "=" * 70)
    print("TOP 50 HIGHEST RATED PAPERS - ICLR 2026")
    print("=" * 70)

    for i, paper in enumerate(all_scored_papers[:50], 1):
        print(f"\n{i}. [{paper['avg_rating']:.2f}] {paper['title'][:65]}")
        print(f"   Ratings: {paper['ratings']} (n={paper['num_reviews']})")
        print(f"   {paper['url']}")

    # Generate markdown report
    with open("iclr2026_top_papers.md", "w") as f:
        f.write("# ICLR 2026 - Highest Rated Papers\n\n")
        f.write(f"*Data fetched from OpenReview API on {time.strftime('%Y-%m-%d')}*\n\n")
        f.write(f"- **Total Submissions:** {total:,}\n")
        f.write(f"- **Papers with Review Scores:** {len(all_scored_papers):,}\n\n")

        # Stats
        if all_scored_papers:
            avg_all = sum(p["avg_rating"] for p in all_scored_papers) / len(all_scored_papers)
            max_rating = max(p["avg_rating"] for p in all_scored_papers)
            min_rating = min(p["avg_rating"] for p in all_scored_papers)
            f.write("### Statistics\n")
            f.write(f"- Average rating: {avg_all:.2f}\n")
            f.write(f"- Highest average: {max_rating:.2f}\n")
            f.write(f"- Lowest average: {min_rating:.2f}\n\n")

        f.write("## Top 100 Papers by Average Rating\n\n")
        f.write("| Rank | Avg | Ratings | Title | Link |\n")
        f.write("|------|-----|---------|-------|------|\n")

        for i, paper in enumerate(all_scored_papers[:100], 1):
            title = paper["title"][:55].replace("|", "\\|").replace("\n", " ")
            ratings_str = ",".join(str(int(r)) for r in paper["ratings"])
            f.write(f"| {i} | {paper['avg_rating']:.1f} | {ratings_str} | {title}... | [link]({paper['url']}) |\n")

    print("\n" + "=" * 70)
    print("Results saved to:")
    print("  - iclr2026_ranked.json (all data)")
    print("  - iclr2026_top_papers.md (markdown report)")
    print("=" * 70)


if __name__ == "__main__":
    main()
