#!/usr/bin/env python3
"""
Fetch and rank ICML 2025 papers by average rating.
"""

import json
import time

import requests

BASE_URL = "https://api2.openreview.net/notes"


def extract_value(field):
    """Extract value from potentially nested field."""
    if isinstance(field, dict) and "value" in field:
        return field["value"]
    return field


def extract_numeric_rating(rating_str):
    """Extract numeric rating from string like '8: Strong Accept'."""
    if rating_str is None:
        return None
    if isinstance(rating_str, (int, float)):
        return float(rating_str)
    if isinstance(rating_str, str):
        try:
            return float(rating_str.split(":")[0].strip())
        except:
            pass
    return None


def fetch_batch(offset, limit=20):
    """Fetch a batch of papers with reviews."""
    params = {"content.venueid": "ICML.cc/2025/Conference", "limit": limit, "offset": offset, "details": "directReplies"}
    for attempt in range(3):
        try:
            response = requests.get(BASE_URL, params=params, timeout=120)
            if response.status_code == 429:
                time.sleep(5 * (attempt + 1))
                continue
            response.raise_for_status()
            return response.json()
        except Exception:
            if attempt < 2:
                time.sleep(2)
    return None


def process_paper(paper):
    """Extract paper info and ratings."""
    paper_id = paper.get("id") or paper.get("forum")
    paper_number = paper.get("number")
    content = paper.get("content", {})

    title = extract_value(content.get("title", "Unknown"))
    venue = extract_value(content.get("venue", ""))

    direct_replies = paper.get("details", {}).get("directReplies", [])
    reviews = [r for r in direct_replies if any("Official_Review" in inv for inv in r.get("invitations", []))]

    ratings = []
    confidences = []

    for r in reviews:
        rcontent = r.get("content", {})

        # ICML uses 'overall_recommendation' field
        for field in ["overall_recommendation", "rating", "score"]:
            rating = extract_value(rcontent.get(field))
            if rating:
                numeric_rating = extract_numeric_rating(rating)
                if numeric_rating is not None:
                    ratings.append(numeric_rating)
                    break

        conf = extract_value(rcontent.get("confidence"))
        conf_num = extract_numeric_rating(conf)
        if conf_num is not None:
            confidences.append(conf_num)

    return {
        "id": paper_id,
        "number": paper_number,
        "title": title,
        "venue": venue,
        "avg_rating": sum(ratings) / len(ratings) if ratings else None,
        "ratings": ratings,
        "num_reviews": len(reviews),
        "avg_confidence": sum(confidences) / len(confidences) if confidences else None,
        "url": f"https://openreview.net/forum?id={paper_id}",
    }


def fetch_range(start_offset, end_offset, batch_size=20):
    """Fetch papers in a range."""
    results = []
    offset = start_offset
    while offset < end_offset:
        data = fetch_batch(offset, limit=batch_size)
        if data and data.get("notes"):
            for paper in data["notes"]:
                result = process_paper(paper)
                results.append(result)
        else:
            break
        offset += batch_size
        time.sleep(0.2)
    return results


def main():
    print("=" * 70)
    print("ICML 2025 - Paper Fetcher and Ranker")
    print("=" * 70)

    # Fetch all papers - since count is unreliable, fetch until empty
    all_papers = []
    batch_size = 50
    offset = 0

    print("\nFetching papers...")

    while True:
        data = fetch_batch(offset, limit=batch_size)
        if not data or not data.get("notes"):
            break

        papers = data["notes"]
        for paper in papers:
            result = process_paper(paper)
            all_papers.append(result)

        if offset % 500 == 0:
            print(f"  Fetched {len(all_papers)} papers...")

        if len(papers) < batch_size:
            break

        offset += batch_size
        time.sleep(0.3)

    print(f"\nTotal papers fetched: {len(all_papers)}")

    # Separate by venue
    orals = [p for p in all_papers if "oral" in str(p.get("venue", "")).lower()]
    spotlights = [p for p in all_papers if "spotlight" in str(p.get("venue", "")).lower()]
    posters = [p for p in all_papers if "poster" in str(p.get("venue", "")).lower() and "spotlight" not in str(p.get("venue", "")).lower()]
    rejected = [p for p in all_papers if "Submitted to ICML" in str(p.get("venue", ""))]

    print(f"  Orals: {len(orals)}")
    print(f"  Spotlight Posters: {len(spotlights)}")
    print(f"  Posters: {len(posters)}")
    print(f"  Rejected/Other: {len(rejected)}")

    # Papers with scores
    scored_papers = [p for p in all_papers if p.get("avg_rating") is not None]
    scored_papers.sort(key=lambda x: x["avg_rating"], reverse=True)

    print(f"\nPapers with review scores: {len(scored_papers)}")

    # Save results
    with open("icml2025_ranked.json", "w") as f:
        json.dump(
            {
                "total_papers": len(all_papers),
                "papers_with_scores": len(scored_papers),
                "orals": len(orals),
                "spotlights": len(spotlights),
                "posters": len(posters),
                "top_100": scored_papers[:100],
                "all_papers": scored_papers,
            },
            f,
            indent=2,
        )

    # Print top papers
    print("\n" + "=" * 70)
    print("TOP 50 HIGHEST RATED PAPERS - ICML 2025")
    print("=" * 70)

    for i, paper in enumerate(scored_papers[:50], 1):
        venue_tag = ""
        if "oral" in str(paper.get("venue", "")).lower():
            venue_tag = " [ORAL]"
        elif "spotlight" in str(paper.get("venue", "")).lower():
            venue_tag = " [SPOTLIGHT]"
        elif "poster" in str(paper.get("venue", "")).lower():
            venue_tag = " [POSTER]"

        print(f"\n{i}. [{paper['avg_rating']:.2f}]{venue_tag} {paper['title'][:55]}")
        print(f"   Ratings: {paper['ratings']} | {paper['url']}")

    # Generate markdown
    with open("icml2025_top_papers.md", "w") as f:
        f.write("# ICML 2025 - Highest Rated Papers\n\n")
        f.write(f"*Data fetched from OpenReview API on {time.strftime('%Y-%m-%d')}*\n\n")
        f.write("## Summary\n")
        f.write(f"- **Total Papers:** {len(all_papers):,}\n")
        f.write(f"- **Oral Presentations:** {len(orals)}\n")
        f.write(f"- **Spotlight Posters:** {len(spotlights)}\n")
        f.write(f"- **Posters:** {len(posters)}\n")
        f.write(f"- **Papers with Scores:** {len(scored_papers):,}\n\n")

        if scored_papers:
            avg_all = sum(p["avg_rating"] for p in scored_papers) / len(scored_papers)
            f.write("### Rating Statistics\n")
            f.write(f"- Overall average rating: {avg_all:.2f}\n")
            f.write(f"- Highest average: {scored_papers[0]['avg_rating']:.2f}\n")
            f.write(f"- Lowest average: {scored_papers[-1]['avg_rating']:.2f}\n\n")

        f.write("## Top 100 Papers by Average Rating\n\n")
        f.write("| Rank | Avg | Type | Ratings | Title | Link |\n")
        f.write("|------|-----|------|---------|-------|------|\n")

        for i, paper in enumerate(scored_papers[:100], 1):
            title = paper["title"][:45].replace("|", "-").replace("\n", " ")
            ratings_str = ",".join(str(int(r)) for r in paper["ratings"]) if paper["ratings"] else "-"

            ptype = ""
            if "oral" in str(paper.get("venue", "")).lower():
                ptype = "Oral"
            elif "spotlight" in str(paper.get("venue", "")).lower():
                ptype = "Spotlight"
            elif "poster" in str(paper.get("venue", "")).lower():
                ptype = "Poster"

            f.write(f"| {i} | {paper['avg_rating']:.1f} | {ptype} | {ratings_str} | {title}... | [link]({paper['url']}) |\n")

    print("\n" + "=" * 70)
    print("Results saved to:")
    print("  - icml2025_ranked.json")
    print("  - icml2025_top_papers.md")
    print("=" * 70)


if __name__ == "__main__":
    main()
