#!/usr/bin/env python3
"""
Fetch ICLR 2026 papers and their ratings from OpenReview API.
"""

import json
import time
from collections import defaultdict

import requests

BASE_URL = "https://api2.openreview.net"


def fetch_notes(venue_id, limit=1000, offset=0):
    """Fetch notes (papers) from OpenReview API."""
    url = f"{BASE_URL}/notes"
    params = {"content.venueid": venue_id, "limit": limit, "offset": offset}
    response = requests.get(url, params=params)
    if response.status_code == 200:
        return response.json()
    else:
        print(f"Error fetching notes: {response.status_code}")
        return None


def fetch_all_submissions():
    """Fetch all ICLR 2026 submissions."""
    venue_id = "ICLR.cc/2026/Conference/Submission"
    all_papers = []
    offset = 0
    limit = 1000

    while True:
        print(f"Fetching papers offset={offset}...")
        data = fetch_notes(venue_id, limit=limit, offset=offset)
        if data is None or not data.get("notes"):
            break

        papers = data["notes"]
        all_papers.extend(papers)
        print(f"  Got {len(papers)} papers, total: {len(all_papers)}")

        if len(papers) < limit:
            break

        offset += limit
        time.sleep(0.5)  # Rate limiting

    return all_papers


def fetch_reviews_for_paper(forum_id):
    """Fetch reviews for a specific paper."""
    url = f"{BASE_URL}/notes"
    params = {"forum": forum_id, "details": "replies"}
    response = requests.get(url, params=params)
    if response.status_code == 200:
        return response.json()
    else:
        return None


def fetch_reviews_bulk(invitation_pattern="ICLR.cc/2026/Conference/-/Official_Review", limit=1000, offset=0):
    """Fetch reviews in bulk using invitation pattern."""
    url = f"{BASE_URL}/notes"
    params = {"invitation": invitation_pattern, "limit": limit, "offset": offset}
    response = requests.get(url, params=params)
    if response.status_code == 200:
        return response.json()
    else:
        print(f"Error: {response.status_code} - {response.text[:200]}")
        return None


def fetch_all_reviews():
    """Fetch all reviews for ICLR 2026."""
    # Try different invitation patterns
    patterns = [
        "ICLR.cc/2026/Conference/-/Official_Review",
        "ICLR.cc/2026/Conference/Submission.*/-/Official_Review",
    ]

    all_reviews = []

    for pattern in patterns:
        print(f"Trying pattern: {pattern}")
        offset = 0
        limit = 1000

        while True:
            print(f"  Fetching reviews offset={offset}...")
            data = fetch_reviews_bulk(pattern, limit=limit, offset=offset)

            if data is None:
                break

            reviews = data.get("notes", [])
            if not reviews:
                break

            all_reviews.extend(reviews)
            print(f"    Got {len(reviews)} reviews, total: {len(all_reviews)}")

            if len(reviews) < limit:
                break

            offset += limit
            time.sleep(0.5)

        if all_reviews:
            break

    return all_reviews


def extract_rating(review):
    """Extract rating from review content."""
    content = review.get("content", {})

    # Try different rating field names
    for field in ["rating", "overall_assessment", "recommendation", "score"]:
        if field in content:
            val = content[field]
            if isinstance(val, dict) and "value" in val:
                val = val["value"]
            # Extract numeric rating
            if isinstance(val, str):
                # Parse "8: Strong Accept" format
                try:
                    return float(val.split(":")[0].strip())
                except:
                    pass
            elif isinstance(val, (int, float)):
                return float(val)
    return None


def compute_paper_scores(papers, reviews):
    """Compute average scores for each paper."""

    # Group reviews by forum (paper)
    reviews_by_paper = defaultdict(list)
    for review in reviews:
        forum_id = review.get("forum")
        if forum_id:
            rating = extract_rating(review)
            if rating is not None:
                reviews_by_paper[forum_id].append(rating)

    # Calculate average scores
    scores = []
    for paper in papers:
        paper_id = paper["id"]
        ratings = reviews_by_paper.get(paper_id, [])

        if ratings:
            avg_rating = sum(ratings) / len(ratings)
            scores.append(
                {
                    "id": paper_id,
                    "title": paper.get("content", {}).get("title", {}).get("value", "Unknown"),
                    "avg_rating": avg_rating,
                    "num_reviews": len(ratings),
                    "ratings": ratings,
                    "url": f"https://openreview.net/forum?id={paper_id}",
                }
            )

    # Sort by average rating
    scores.sort(key=lambda x: x["avg_rating"], reverse=True)
    return scores


def main():
    print("=" * 60)
    print("Fetching ICLR 2026 Papers from OpenReview")
    print("=" * 60)

    # Fetch all submissions
    print("\n1. Fetching submissions...")
    papers = fetch_all_submissions()
    print(f"Total submissions: {len(papers)}")

    # Fetch all reviews
    print("\n2. Fetching reviews...")
    reviews = fetch_all_reviews()
    print(f"Total reviews: {len(reviews)}")

    # Compute scores
    print("\n3. Computing scores...")
    scored_papers = compute_paper_scores(papers, reviews)
    print(f"Papers with scores: {len(scored_papers)}")

    # Save results
    results = {
        "total_submissions": len(papers),
        "total_reviews": len(reviews),
        "papers_with_scores": len(scored_papers),
        "top_papers": scored_papers[:100],  # Top 100
        "all_scored_papers": scored_papers,
    }

    with open("iclr2026_results.json", "w") as f:
        json.dump(results, f, indent=2)

    # Print top 50
    print("\n" + "=" * 60)
    print("TOP 50 HIGHEST RATED PAPERS")
    print("=" * 60)

    for i, paper in enumerate(scored_papers[:50], 1):
        print(f"\n{i}. [{paper['avg_rating']:.2f}] {paper['title'][:70]}...")
        print(f"   Reviews: {paper['num_reviews']}, Ratings: {paper['ratings']}")
        print(f"   URL: {paper['url']}")

    # Generate markdown report
    with open("iclr2026_top_papers.md", "w") as f:
        f.write("# ICLR 2026 - Highest Rated Papers\n\n")
        f.write("*Data fetched from OpenReview API*\n\n")
        f.write(f"- **Total Submissions:** {len(papers)}\n")
        f.write(f"- **Total Reviews:** {len(reviews)}\n")
        f.write(f"- **Papers with Scores:** {len(scored_papers)}\n\n")
        f.write("## Top 50 Papers by Average Rating\n\n")
        f.write("| Rank | Avg Rating | Reviews | Title | Link |\n")
        f.write("|------|------------|---------|-------|------|\n")

        for i, paper in enumerate(scored_papers[:50], 1):
            title = paper["title"][:60].replace("|", "\\|")
            f.write(f"| {i} | {paper['avg_rating']:.2f} | {paper['num_reviews']} | {title}... | [link]({paper['url']}) |\n")

    print("\n\nResults saved to:")
    print("  - iclr2026_results.json")
    print("  - iclr2026_top_papers.md")


if __name__ == "__main__":
    main()
