#!/usr/bin/env python3
"""
Fast parallel fetcher for ICLR 2026 papers with ratings.
Uses concurrent requests for speed.
"""

import json
import time
from concurrent.futures import ThreadPoolExecutor, as_completed
from threading import Lock

import requests

BASE_URL = "https://api2.openreview.net/notes"
print_lock = Lock()


def fetch_batch(offset, limit=20):
    """Fetch a batch of papers with reviews."""
    params = {"content.venueid": "ICLR.cc/2026/Conference/Submission", "limit": limit, "offset": offset, "details": "directReplies"}
    for attempt in range(3):
        try:
            response = requests.get(BASE_URL, params=params, timeout=120)
            if response.status_code == 429:  # Rate limited
                time.sleep(5 * (attempt + 1))
                continue
            response.raise_for_status()
            return response.json()
        except Exception:
            if attempt < 2:
                time.sleep(2)
            else:
                return None
    return None


def process_paper(paper):
    """Extract paper info and ratings."""
    paper_id = paper.get("id") or paper.get("forum")
    paper_number = paper.get("number")
    content = paper.get("content", {})

    title = content.get("title", {})
    if isinstance(title, dict):
        title = title.get("value", "Unknown")

    direct_replies = paper.get("details", {}).get("directReplies", [])
    reviews = [r for r in direct_replies if any("Official_Review" in inv for inv in r.get("invitations", []))]

    ratings = []
    for r in reviews:
        rating = r.get("content", {}).get("rating", {})
        if isinstance(rating, dict):
            rating = rating.get("value")
        if rating:
            try:
                ratings.append(float(str(rating).split(":")[0]))
            except:
                pass

    if ratings:
        return {
            "id": paper_id,
            "number": paper_number,
            "title": title,
            "avg_rating": sum(ratings) / len(ratings),
            "ratings": ratings,
            "num_reviews": len(reviews),
            "url": f"https://openreview.net/forum?id={paper_id}",
        }
    return None


def fetch_range(start_offset, end_offset, batch_size=20):
    """Fetch papers in a range."""
    results = []
    offset = start_offset
    while offset < end_offset:
        data = fetch_batch(offset, limit=batch_size)
        if data and data.get("notes"):
            for paper in data["notes"]:
                result = process_paper(paper)
                if result:
                    results.append(result)
        offset += batch_size
        time.sleep(0.2)
    return results


def main():
    print("=" * 70)
    print("ICLR 2026 - Fast Parallel Paper Fetcher")
    print("=" * 70)

    # Get total count
    print("\nGetting total count...")
    initial = fetch_batch(0, limit=1)
    if not initial:
        print("Failed to connect")
        return

    total = initial.get("count", 0)
    print(f"Total submissions: {total:,}")

    # Fetch in parallel with multiple workers
    batch_size = 20
    num_workers = 5
    chunk_size = (total // num_workers) + batch_size

    all_papers = []

    print(f"\nFetching with {num_workers} parallel workers...")
    start_time = time.time()

    with ThreadPoolExecutor(max_workers=num_workers) as executor:
        futures = []
        for i in range(num_workers):
            start = i * chunk_size
            end = min((i + 1) * chunk_size, total)
            futures.append(executor.submit(fetch_range, start, end, batch_size))

        for _i, future in enumerate(as_completed(futures)):
            results = future.result()
            all_papers.extend(results)
            print(f"  Worker completed: +{len(results)} papers (total: {len(all_papers)})")

    elapsed = time.time() - start_time
    print(f"\nFetched {len(all_papers)} papers with scores in {elapsed:.1f}s")

    # Sort by rating
    all_papers.sort(key=lambda x: x["avg_rating"], reverse=True)

    # Save results
    with open("iclr2026_ranked.json", "w") as f:
        json.dump(
            {"total_submissions": total, "papers_with_scores": len(all_papers), "top_100": all_papers[:100], "all_papers": all_papers},
            f,
            indent=2,
        )

    # Print top 50
    print("\n" + "=" * 70)
    print("TOP 50 HIGHEST RATED PAPERS - ICLR 2026")
    print("=" * 70)

    for i, paper in enumerate(all_papers[:50], 1):
        print(f"\n{i}. [{paper['avg_rating']:.2f}] {paper['title'][:60]}")
        print(f"   Ratings: {paper['ratings']} | {paper['url']}")

    # Generate markdown
    with open("iclr2026_top_papers.md", "w") as f:
        f.write("# ICLR 2026 - Highest Rated Papers\n\n")
        f.write(f"*Data fetched from OpenReview API on {time.strftime('%Y-%m-%d')}*\n\n")
        f.write(f"- **Total Submissions:** {total:,}\n")
        f.write(f"- **Papers with Scores:** {len(all_papers):,}\n\n")

        if all_papers:
            avg_all = sum(p["avg_rating"] for p in all_papers) / len(all_papers)
            f.write("### Statistics\n")
            f.write(f"- Overall average rating: {avg_all:.2f}\n")
            f.write(f"- Highest average: {all_papers[0]['avg_rating']:.2f}\n")
            f.write(f"- Lowest average: {all_papers[-1]['avg_rating']:.2f}\n\n")

        f.write("## Top 100 Papers by Average Rating\n\n")
        f.write("| Rank | Avg | Ratings | Title | Link |\n")
        f.write("|------|-----|---------|-------|------|\n")

        for i, paper in enumerate(all_papers[:100], 1):
            title = paper["title"][:50].replace("|", "-").replace("\n", " ")
            ratings_str = ",".join(str(int(r)) for r in paper["ratings"])
            f.write(f"| {i} | {paper['avg_rating']:.1f} | {ratings_str} | {title}... | [link]({paper['url']}) |\n")

    print("\n" + "=" * 70)
    print("Results saved to:")
    print("  - iclr2026_ranked.json")
    print("  - iclr2026_top_papers.md")
    print("=" * 70)


if __name__ == "__main__":
    main()
