# OpenReview API Guide for ICLR 2026 Reviews

## Summary

Reviews for ICLR 2026 papers **ARE publicly available** and can be fetched using the OpenReview API v2. The submission deadline was October 2025, and reviews are complete as of December 2025.

**Total Submissions**: 15,735 papers

## Working API Methods

### Method 1: Batch Fetch with Nested Reviews (RECOMMENDED)

Fetch multiple submissions with their reviews in a single request using the `details=directReplies` parameter.

**Endpoint**: `https://api2.openreview.net/notes`

**Parameters**:
- `content.venueid`: `ICLR.cc/2026/Conference/Submission`
- `details`: `directReplies`
- `limit`: Number of papers to fetch (max ~5-10 due to response size)
- `offset`: Pagination offset (0-indexed)

**Example Request**:
```bash
curl "https://api2.openreview.net/notes?content.venueid=ICLR.cc/2026/Conference/Submission&limit=5&offset=0&details=directReplies"
```

**Python Example**:
```python
import requests

url = 'https://api2.openreview.net/notes'
params = {
    'content.venueid': 'ICLR.cc/2026/Conference/Submission',
    'limit': 5,
    'offset': 0,
    'details': 'directReplies'
}

response = requests.get(url, params=params)
data = response.json()

for paper in data['notes']:
    paper_id = paper['forum']
    paper_number = paper['number']

    # Extract reviews from directReplies
    direct_replies = paper.get('details', {}).get('directReplies', [])
    reviews = [
        r for r in direct_replies
        if any('Official_Review' in inv for inv in r.get('invitations', []))
    ]

    for review in reviews:
        content = review['content']
        rating = content.get('rating', {})
        confidence = content.get('confidence', {})
        print(f"Paper {paper_number}: Rating={rating}, Confidence={confidence}")
```

**Response Structure**:
```json
{
  "notes": [
    {
      "id": "FPLNSx1jmL",
      "forum": "FPLNSx1jmL",
      "number": 25649,
      "content": { ... paper content ... },
      "details": {
        "directReplies": [
          {
            "id": "WSdP0QXNPV",
            "invitations": [
              "ICLR.cc/2026/Conference/Submission25649/-/Official_Review",
              "ICLR.cc/2026/Conference/-/Edit"
            ],
            "content": {
              "rating": 4,
              "confidence": 4,
              "soundness": 2,
              "presentation": 2,
              "contribution": 2,
              "summary": "...",
              "strengths": "...",
              "weaknesses": "...",
              "questions": "..."
            }
          }
        ]
      }
    }
  ],
  "count": 15735
}
```

**Pros**:
- Fetch multiple papers with reviews in one request
- Includes all reply types (reviews, comments, author responses)
- Most efficient for batch processing

**Cons**:
- Limited to ~5-10 papers per request due to response size
- Requires pagination to fetch all papers

---

### Method 2: Fetch Reviews by Paper Forum ID

Fetch all replies (reviews, comments) for a specific paper using its forum ID.

**Endpoint**: `https://api2.openreview.net/notes`

**Parameters**:
- `replyto`: Paper forum ID (e.g., `FPLNSx1jmL`)

**Example Request**:
```bash
curl "https://api2.openreview.net/notes?replyto=FPLNSx1jmL"
```

**Python Example**:
```python
import requests

forum_id = 'FPLNSx1jmL'  # Paper's forum ID
url = 'https://api2.openreview.net/notes'
params = {'replyto': forum_id}

response = requests.get(url, params=params)
data = response.json()

# Filter for official reviews
reviews = [
    note for note in data['notes']
    if any('Official_Review' in inv for inv in note.get('invitations', []))
]

for review in reviews:
    content = review['content']
    print(f"Rating: {content.get('rating')}")
    print(f"Confidence: {content.get('confidence')}")
```

**Pros**:
- Get all replies for a specific paper
- Includes reviews, comments, and author responses

**Cons**:
- Requires forum ID (must fetch submissions first)
- One request per paper

---

### Method 3: Paper-Specific Invitation Pattern

Fetch reviews for a specific paper using its submission number.

**Endpoint**: `https://api2.openreview.net/notes`

**Parameters**:
- `invitation`: `ICLR.cc/2026/Conference/Submission{NUMBER}/-/Official_Review`

**Example Request**:
```bash
curl "https://api2.openreview.net/notes?invitation=ICLR.cc/2026/Conference/Submission25649/-/Official_Review"
```

**Python Example**:
```python
import requests

submission_number = 25649
url = 'https://api2.openreview.net/notes'
params = {
    'invitation': f'ICLR.cc/2026/Conference/Submission{submission_number}/-/Official_Review'
}

response = requests.get(url, params=params)
data = response.json()

for review in data['notes']:
    content = review['content']
    print(f"Rating: {content.get('rating')}")
```

**Pros**:
- Direct access to only official reviews
- No need to filter replies

**Cons**:
- Requires submission number
- One request per paper
- Misses other reply types (comments, author responses)

---

## What DOESN'T Work

### Failed Patterns

1. **Wildcard invitation patterns**:
   ```bash
   # Returns 0 results
   https://api2.openreview.net/notes?invitation=ICLR.cc/2026/Conference/Submission.*/-/Official_Review
   ```

2. **Generic invitation pattern**:
   ```bash
   # Returns 0 results
   https://api2.openreview.net/notes?invitation=ICLR.cc/2026/Conference/-/Official_Review
   ```

3. **Regex patterns in invitations endpoint**:
   ```bash
   # Returns 400 error
   https://api2.openreview.net/invitations?regex=ICLR.cc/2026/Conference.*Official_Review
   ```

---

## Review Data Structure

Each review contains the following fields:

### Core Metrics
- **rating**: Overall score (1-10 scale, typically 1-8 used)
- **confidence**: Reviewer's confidence (1-5 scale)
- **soundness**: Technical soundness (1-4 scale)
- **presentation**: Quality of presentation (1-4 scale)
- **contribution**: Novelty/contribution (1-4 scale)

### Text Content
- **summary**: Brief summary of the paper
- **strengths**: Positive aspects
- **weaknesses**: Identified issues
- **questions**: Questions for authors
- **flag_for_ethics_review**: Ethical concerns (if any)
- **code_of_conduct**: Confirmation of adherence

### Metadata
- **id**: Review ID
- **forum**: Paper forum ID
- **replyto**: Parent note ID (the submission)
- **signatures**: Reviewer signature (anonymized)
- **invitations**: List of invitation IDs
- **tcdate**, **cdate**, **tmdate**, **mdate**: Timestamps

---

## Complete Workflow: Fetch All Reviews

To fetch reviews for all 15,735 ICLR 2026 submissions:

```python
import requests
import time

def fetch_all_reviews():
    all_reviews = []
    offset = 0
    limit = 5  # Small batches due to response size
    total = 15735

    while offset < total:
        url = 'https://api2.openreview.net/notes'
        params = {
            'content.venueid': 'ICLR.cc/2026/Conference/Submission',
            'limit': limit,
            'offset': offset,
            'details': 'directReplies'
        }

        response = requests.get(url, params=params)
        data = response.json()

        if not data.get('notes'):
            break

        for paper in data['notes']:
            paper_id = paper['forum']
            paper_number = paper['number']

            # Extract reviews
            direct_replies = paper.get('details', {}).get('directReplies', [])
            reviews = [
                r for r in direct_replies
                if any('Official_Review' in inv for inv in r.get('invitations', []))
            ]

            for review in reviews:
                review_data = {
                    'paper_id': paper_id,
                    'paper_number': paper_number,
                    'review_id': review['id'],
                    'rating': review['content'].get('rating'),
                    'confidence': review['content'].get('confidence'),
                    'soundness': review['content'].get('soundness'),
                    'presentation': review['content'].get('presentation'),
                    'contribution': review['content'].get('contribution'),
                    'summary': review['content'].get('summary'),
                    'strengths': review['content'].get('strengths'),
                    'weaknesses': review['content'].get('weaknesses'),
                }
                all_reviews.append(review_data)

        print(f"Processed {offset + len(data['notes'])}/{total} papers, {len(all_reviews)} reviews so far")
        offset += len(data['notes'])

        # Rate limiting
        time.sleep(0.5)

    return all_reviews

# Execute
reviews = fetch_all_reviews()
print(f"Total reviews collected: {len(reviews)}")
```

---

## Key Findings

1. **Reviews are publicly available** for ICLR 2026 submissions
2. **Most papers have exactly 4 reviews** (standard ICLR protocol)
3. **All reviews contain ratings** and confidence scores
4. **Use `details=directReplies`** for batch fetching (most efficient)
5. **Pagination works** with offset parameter
6. **Response size limits** mean ~5-10 papers per request max
7. **Total of ~60,000+ reviews** expected (15,735 papers × ~4 reviews each)

---

## Invitation Patterns Discovered

- **Official Review**: `ICLR.cc/2026/Conference/Submission{NUMBER}/-/Official_Review`
- **Public Comment**: `ICLR.cc/2026/Conference/Submission{NUMBER}/-/Public_Comment`
- **Edit**: `ICLR.cc/2026/Conference/-/Edit`

Each review note typically has multiple invitations in its `invitations` array.

---

## Timeline Reference

- **Submission Deadline**: October 2025
- **Review Period**: October-December 2025
- **Reviews Available**: December 2025 (confirmed)
- **Conference Date**: April 2026
